/**************************************************************************//**
 * \file       SXMDatabaseHandler.cpp
 *
 * This file is part of the SdsAdapter component.
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "SXMDatabaseHandler.h"
#include "sds_sxm_query_defines.h"
#include "adapter/external/sds2hmi_fi.h"
#include <string.h>


#define XM_STATION_LIST_TABLEX             "XMStationList"
#define XM_PHONETICS_LIST_TABLEX           "XMStationPhonetics"
#define XM_TIME_STAMP_TABLEX               "XMTimestamp"

#define XM_TIME_STAMP_ID        0x01
#define INVALID_TIME_STAMP      0xFFFFFFFF
#define START_TIME_STAMP        0x01


#define QUERY_CREATE_XM_STATION_LIST_TABLE      "CREATE TABLE XMStationList( channel_object_id INTEGER, sid INTEGER PRIMARY KEY, channel_number INTEGER, channel_name CHAR(100));"
#define QUERY_CREATE_XM_TIME_STAMP_TABLE        "CREATE TABLE XMTimestamp( unique_id INTEGER PRIMARY KEY, time_stamp INTEGER);"

#define QUERY_CREATE_XM_PHONETIC_LIST_TABLE_EN     "CREATE TABLE XMStationPhonetics_EN( language_id INTEGER, sid INTEGER PRIMARY KEY, channel_phonemes CHAR(500));"
#define QUERY_CREATE_XM_PHONETIC_LIST_TABLE_SP     "CREATE TABLE XMStationPhonetics_SP( language_id INTEGER, sid INTEGER PRIMARY KEY, channel_phonemes CHAR(500));"
#define QUERY_CREATE_XM_PHONETIC_LIST_TABLE_FR     "CREATE TABLE XMStationPhonetics_FR( language_id INTEGER, sid INTEGER PRIMARY KEY, channel_phonemes CHAR(500));"


#define FORMAT_QUERY_INSERT_XM_TIME_STAMPTBL    "INSERT INTO XMTimestamp VALUES    ('%u', '%u');"
#define FORMAT_QUERY_UPDATE_XM_TIME_STAMP_TBL   "UPDATE XMTimestamp SET time_stamp='%u' WHERE unique_id='%u';"

#define FORMAT_QUERY_UPDATE_XM_STATION_LIST     "INSERT OR REPLACE INTO  XMStationList (sid, channel_object_id, channel_number, channel_name) VALUES ('%u', '%u', '%u', '%s');"

#define FORMAT_QUERY_UPDATE_XM_PHONETIC_LIST    "INSERT OR REPLACE INTO  %s (language_id, sid, channel_phonemes) VALUES ('%u', '%u', '%s');"


#define FORMAT_QUERY_GET_OBJECT_ID_FROM_CHANNEL_NUM_XM_STATION_LIST             "SELECT channel_object_id FROM XMStationList WHERE channel_number='%u';"
#define FORMAT_QUERY_GET_OBJECT_ID_FROM_CHANNEL_NAME_XM_STATION_LIST            "SELECT channel_object_id FROM XMStationList WHERE channel_name='%s';"
#define FORMAT_QUERY_GET_CHANNEL_NAME_FROM_CHANNEL_SID_XM_STATION_LIST          "SELECT channel_name, channel_number FROM XMStationList WHERE sid='%u';"

#define FORMAT_QUERY_GET_PHONEMES_FROM_TABLE          "SELECT channel_phonemes FROM %s WHERE sid='%u';"


#define FORMAT_QUERY_GET_TIME_STAMP_FROM_ID_XM_TIME_STAMP_TBL                   "SELECT time_stamp FROM XMTimestamp WHERE unique_id='%u';"


#define QUERY_DELETE_ALL_DATA_XM_STATION_LIST            "delete from XMStationList;"
#define QUERY_DELETE_ALL_DATA_XM_TIME_STAMPTBL           "delete from XMTimestamp;"


#define QUERY_VIEW_XM_CHANNEL                            "CREATE VIEW XMChannel AS  SELECT channel_name AS XMChannelname, channel_object_id AS XMChannelID FROM XMStationList WHERE channel_name != '';"
#define QUERY_VIEW_XM_CHECK_SUM                          "CREATE VIEW XM_Checksum AS SELECT time_stamp AS Checksum FROM XMTimestamp;"
#define QUERY_VIEW_XM_CHANNEL_PHONEMES                   "CREATE VIEW XM_StationPhonetics AS \
   SELECT language_id, XMStationList.channel_object_id AS channel_id, channel_phonemes, channel_name \
   FROM XMStationPhonetics_EN INNER JOIN XMStationList \
   ON XMStationPhonetics_EN.sid = XMStationList.sid \
   UNION \
   SELECT language_id, XMStationList.channel_object_id AS channel_id, channel_phonemes, channel_name \
   FROM XMStationPhonetics_SP INNER JOIN XMStationList \
   ON XMStationPhonetics_SP.sid = XMStationList.sid \
   UNION \
   SELECT language_id, XMStationList.channel_object_id AS channel_id, channel_phonemes, channel_name \
   FROM XMStationPhonetics_FR INNER JOIN XMStationList \
   ON XMStationPhonetics_FR.sid = XMStationList.sid"


#define QUERY_ROW_COUNT_XM_TABLE                         "SELECT COUNT(*) FROM XMStationList;"

#define MAX_QUERY_LEN       1024

#define SAAL_OBJECTID_NOTFOUND       0xFFFF // If object ID is not found, 0xFFFF will be returned


SXMDatabaseHandler::SXMDatabaseHandler()
   : SqliteDB("/var/opt/bosch/dynamic/speech/radio/speech_XM_tuner_data.db")
{
}


SXMDatabaseHandler::~SXMDatabaseHandler()
{
}


void SXMDatabaseHandler::setup()
{
   setupTablesAndViews();
   setupValidTimestamp();
}


void SXMDatabaseHandler::setupTablesAndViews()
{
   exec(QUERY_CREATE_XM_STATION_LIST_TABLE);
   exec(QUERY_CREATE_XM_PHONETIC_LIST_TABLE_EN);
   exec(QUERY_CREATE_XM_PHONETIC_LIST_TABLE_SP);
   exec(QUERY_CREATE_XM_PHONETIC_LIST_TABLE_FR);
   exec(QUERY_CREATE_XM_TIME_STAMP_TABLE);
   exec(QUERY_VIEW_XM_CHANNEL);
   exec(QUERY_VIEW_XM_CHECK_SUM);
   exec(QUERY_VIEW_XM_CHANNEL_PHONEMES);
}


void SXMDatabaseHandler::setupValidTimestamp()
{
   tU32 u32TimeStamp = u32GetCurrentTimeStamp();
   if (u32TimeStamp == INVALID_TIME_STAMP)
   {
      vDeleteAllFromXMTimeStampTbl();
      vInsertRecordForXMTimeStampTbl(XM_TIME_STAMP_ID, START_TIME_STAMP);
   }
}


tVoid SXMDatabaseHandler::vDeleteAllFromXMStationList()
{
   exec(QUERY_DELETE_ALL_DATA_XM_STATION_LIST);
}


tVoid SXMDatabaseHandler::vDeleteAllFromXMTimeStampTbl()
{
   exec(QUERY_DELETE_ALL_DATA_XM_TIME_STAMPTBL);
}


void SXMDatabaseHandler::vUpdateRecordForXMStationList(const sds_sxm_fi::SdsSxmService::SXMChannelItem& channel)
{
   char szQuery[MAX_QUERY_LEN];
   sprintf(
      szQuery,
      FORMAT_QUERY_UPDATE_XM_STATION_LIST,
      channel.getServiceID(),
      channel.getChannelID(),
      channel.getChannelNumber(),
      sdsa::sqlite::SqliteDB::escape(channel.getChannelName()).c_str());
   exec(szQuery);
}


void SXMDatabaseHandler::vUpdateRecordForXMPhoneticList(const sds_sxm_fi::SdsSxmService::SXMPhoneticData& channelPhoneme)
{
   char szQuery[MAX_QUERY_LEN];
   sprintf(
      szQuery,
      FORMAT_QUERY_UPDATE_XM_PHONETIC_LIST,
      getPhoneticsTableName(channelPhoneme.getLanguageID()).c_str(),
      channelPhoneme.getLanguageID(),
      channelPhoneme.getServiceID(),
      sdsa::sqlite::SqliteDB::escape(channelPhoneme.getPhoneticData()).c_str());
   exec(szQuery);
}


std::string SXMDatabaseHandler::getPhoneticsTableName(tU16 languageId)
{
   switch (languageId)
   {
      case sds2hmi_fi_tcl_e16_ISO639_3_SDSLanguageCode::FI_EN_ISO_639_3_ENG:
         return "XMStationPhonetics_EN";

      case sds2hmi_fi_tcl_e16_ISO639_3_SDSLanguageCode::FI_EN_ISO_639_3_SPA:
         return "XMStationPhonetics_SP";

      case sds2hmi_fi_tcl_e16_ISO639_3_SDSLanguageCode::FI_EN_ISO_639_3_FRA:
         return "XMStationPhonetics_FR";

      default:
         return "XMStationPhonetics_EN";
   }
}


tU32 SXMDatabaseHandler::u32FindXMStationObjectId(tU16 channelNumber)
{
   char szQuery[MAX_QUERY_LEN];
   sprintf(szQuery, FORMAT_QUERY_GET_OBJECT_ID_FROM_CHANNEL_NUM_XM_STATION_LIST, channelNumber);
   return selectRowAndReadInt(szQuery, 0, SAAL_OBJECTID_NOTFOUND);
}


tU32 SXMDatabaseHandler::u32FindXMStationObjectId(const ::std::string& channelName)
{
   char szQuery[MAX_QUERY_LEN];
   ::std::string escapedName = sdsa::sqlite::SqliteDB::escape(channelName);
   sprintf(szQuery, FORMAT_QUERY_GET_OBJECT_ID_FROM_CHANNEL_NAME_XM_STATION_LIST, escapedName.c_str());
   return selectRowAndReadInt(szQuery, 0, SAAL_OBJECTID_NOTFOUND);
}


UpdateType SXMDatabaseHandler::enGetXMStationUpdateType(const sds_sxm_fi::SdsSxmService::SXMChannelItem& channel)
{
   UpdateType updateType = XM_LIST_NO_UPDATE_REQUIRED;
   char szQuery[MAX_QUERY_LEN];
   sprintf(szQuery, FORMAT_QUERY_GET_CHANNEL_NAME_FROM_CHANNEL_SID_XM_STATION_LIST, channel.getServiceID());
   prepare(szQuery);
   step();
   if (columnSelected())
   {
      if ((getColumnInt(1) != channel.getChannelNumber()) ||
            (getColumnText(0) != channel.getChannelName()))
      {
         updateType = XM_LIST_UPDATE_REQUIRED;
      }
   }
   else
   {
      updateType = XM_LIST_INSERT_REQUIRED;
   }
   finalize();

   return updateType;
}


UpdateType SXMDatabaseHandler::enGetXMPhoneticsUpdateType(const sds_sxm_fi::SdsSxmService::SXMPhoneticData& channelPhoneme)
{
   UpdateType updateType = XM_LIST_NO_UPDATE_REQUIRED;
   char szQuery[MAX_QUERY_LEN];
   sprintf(
      szQuery,
      FORMAT_QUERY_GET_PHONEMES_FROM_TABLE,
      getPhoneticsTableName(channelPhoneme.getLanguageID()).c_str(),
      channelPhoneme.getServiceID());
   prepare(szQuery);
   step();
   if (columnSelected())
   {
      if (getColumnText(0) != channelPhoneme.getPhoneticData())
      {
         updateType = XM_LIST_UPDATE_REQUIRED;
      }
   }
   else
   {
      updateType = XM_LIST_INSERT_REQUIRED;
   }
   finalize();

   return updateType;
}


tVoid SXMDatabaseHandler::vUpdateTimeStamp()
{
   tU32 u32TimeStamp = u32GetCurrentTimeStamp();
   if (u32TimeStamp == INVALID_TIME_STAMP)
   {
      vDeleteAllFromXMTimeStampTbl();
      vInsertRecordForXMTimeStampTbl(XM_TIME_STAMP_ID, START_TIME_STAMP);
   }
   else
   {
      u32TimeStamp = u32TimeStamp + 1;
      if (u32TimeStamp == INVALID_TIME_STAMP)
      {
         u32TimeStamp = START_TIME_STAMP;
      }
      vUpdateRecordForXMTimeStampTbl(XM_TIME_STAMP_ID, u32TimeStamp);
   }
}


tU32 SXMDatabaseHandler::u32GetCurrentTimeStamp()
{
   char szQuery[MAX_QUERY_LEN];
   sprintf(szQuery, FORMAT_QUERY_GET_TIME_STAMP_FROM_ID_XM_TIME_STAMP_TBL, XM_TIME_STAMP_ID);
   return selectRowAndReadInt(szQuery, 0, INVALID_TIME_STAMP);
}


void SXMDatabaseHandler::vInsertRecordForXMTimeStampTbl(tU32 u32Id, tU32 u32TimeStamp)
{
   char szQuery[MAX_QUERY_LEN];
   sprintf(szQuery, FORMAT_QUERY_INSERT_XM_TIME_STAMPTBL, u32Id, u32TimeStamp);
   exec(szQuery);
}


void SXMDatabaseHandler::vUpdateRecordForXMTimeStampTbl(tU32 u32Id, tU32 u32TimeStamp)
{
   char szQuery[MAX_QUERY_LEN];
   sprintf(szQuery, FORMAT_QUERY_UPDATE_XM_TIME_STAMP_TBL, u32TimeStamp, u32Id);
   exec(szQuery);
}


int SXMDatabaseHandler::selectRowAndReadInt(const ::std::string& query, int column, int defaultValue)
{
   prepare(query);
   step();
   int value = getColumnInt(column, defaultValue);
   finalize();
   return value;
}
