/**************************************************************************//**
 * \file       clSDS_XMLStringCreation.cpp
 *
 * clSDS_XMLStringCreation.cpp class implementation
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "application/clSDS_XMLStringCreation.h"
#include "application/clSDS_ConfigurationFlags.h"
#include "application/StringUtils.h"
#include "libxml/xmlwriter.h"
#include "SdsAdapter_Trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/clSDS_XMLStringCreation.cpp.trc.h"
#endif

std::vector <std::string> clSDS_XMLStringCreation::_configData;

/**************************************************************************//**

******************************************************************************/
static void removeLineBreaks(std::string& str)
{
   size_t pos = str.find("\n");
   while (pos != std::string::npos)
   {
      str.erase(pos, 1);
      pos = str.find("\n");
   }
}


/**************************************************************************//**

******************************************************************************/
static std::string formatXml(const GroupMap& groups)
{
   xmlBufferPtr bufptr = xmlBufferCreate();
   xmlTextWriterPtr wrtptr = xmlNewTextWriterMemory(bufptr, 0);
   xmlTextWriterStartDocument(wrtptr, NULL, "UTF-8", NULL);

   if (xmlTextWriterStartElement(wrtptr, (const xmlChar*) "SDS_CONFIG_PARAM") < 0)
   {
      ETG_TRACE_USR1(("clSDS_XMLStringCreation:Error at xmlTextWriterStartElement "));
   }
   if (xmlTextWriterWriteAttribute(wrtptr, (const xmlChar*) "project", (const xmlChar*) "AIVI") < 0)
   {
      ETG_TRACE_USR1(("clSDS_XMLStringCreation:Error at xmlTextWriterWriteAttribute "));
   }

   GroupMap::const_iterator groupIter = groups.begin();
   while (groupIter != groups.end())
   {
      if (xmlTextWriterStartElement(wrtptr, (const xmlChar*)"CONFIG_GROUP") < 0)
      {
         ETG_TRACE_USR1(("clSDS_XMLStringCreation:Error at xmlTextWriterStartElement "));
      }
      if (xmlTextWriterWriteAttribute(wrtptr, (const xmlChar*) "name", (const xmlChar*) groupIter->first.c_str()) < 0)
      {
         ETG_TRACE_USR1(("clSDS_XMLStringCreation:Error at xmlTextWriterWriteAttribute "));
      }
      if (xmlTextWriterStartElement(wrtptr, (const xmlChar*) "PARAMETERS") < 0)
      {
         ETG_TRACE_USR1(("clSDS_XMLStringCreation:Error at xmlTextWriterStartElement "));
      }
      ConfigItemMap itemMap = groupIter->second;
      ConfigItemMap::const_iterator itemIter = itemMap.begin();
      while (itemIter != itemMap.end())
      {
         if (xmlTextWriterStartElement(wrtptr, (const xmlChar*) "PARAMETER") < 0)
         {
            ETG_TRACE_USR1(("clSDS_XMLStringCreation:Error at xmlTextWriterStartElement "));
         }
         if (xmlTextWriterWriteElement(wrtptr, (const xmlChar*) "NAME", (const xmlChar*) itemIter->first.c_str()) < 0)
         {
            ETG_TRACE_USR1(("clSDS_XMLStringCreation:Error at xmlTextWriterWriteElement "));
         }
         if (xmlTextWriterWriteFormatElement(wrtptr, (const xmlChar*) "VALUE", "%u", itemIter->second) < 0)
         {
            ETG_TRACE_USR1(("clSDS_XMLStringCreation:Error at xmlTextWriterWriteFormatElement"));
         }
         xmlTextWriterEndElement(wrtptr);
         ++itemIter;
      }
      xmlTextWriterEndElement(wrtptr);
      xmlTextWriterEndElement(wrtptr);
      ++groupIter;
   }
   xmlTextWriterEndElement(wrtptr);
   xmlTextWriterEndDocument(wrtptr);
   xmlTextWriterFlush(wrtptr);
   std::string str = (const char*) bufptr->content;
   xmlFreeTextWriter(wrtptr);
   xmlBufferFree(bufptr);
   removeLineBreaks(str);
   return str;
}


/**************************************************************************//**
*Destructor
******************************************************************************/
clSDS_XMLStringCreation::~clSDS_XMLStringCreation()
{
}


/**************************************************************************//**
*Constructor
******************************************************************************/
clSDS_XMLStringCreation::clSDS_XMLStringCreation()
{
}


/**************************************************************************//**

******************************************************************************/
void clSDS_XMLStringCreation::fillConfigVectorValues()
{
   _configData.clear();
   _configData.push_back("AUD");
   _configData.push_back("PHO");
   _configData.push_back("NAV");
   _configData.push_back("INF");
   _configData.push_back("GRL");
   _configData.push_back("APP");
   _configData.push_back("EVE");
   _configData.push_back("VINF");
}


/**************************************************************************//**

******************************************************************************/
std::string clSDS_XMLStringCreation::getStaticXmlString()
{
   GroupMap staticGroups;
   staticGroups["NAV"]["B_NAV"] = clSDS_ConfigurationFlags::getNavKey();
   staticGroups["NAV"]["B_NAV_MultiLingualVDE"] = clSDS_ConfigurationFlags::getNavMultiLingualVDE();
   staticGroups["NAV"]["B_NAV_ADRS"] = clSDS_ConfigurationFlags::getNavAdrsKey();
   staticGroups["NAV"]["B_NAV_POI"] = clSDS_ConfigurationFlags::getPOIKey();
   staticGroups["NAV"]["B_NAV_POICAT"] = clSDS_ConfigurationFlags::getPOICatKey();
   staticGroups["NAV"]["B_NAV_CHNG_CNTRY"] = clSDS_ConfigurationFlags::getNavCountryAvailable();
   staticGroups["NAV"]["B_NAV_Japanese_Address_Input"] = clSDS_ConfigurationFlags::getNavJapaneseAddressInput();
   staticGroups["NAV"]["B_NAV_Intersection"] = clSDS_ConfigurationFlags::getNavIntersectionKey();
   staticGroups["NAV"]["B_NAV_TaxiMode"] = clSDS_ConfigurationFlags::getTaxiAvailabilityKey();
   staticGroups["INF"]["B_INF_HEV"] = clSDS_ConfigurationFlags::getInfHEVKey();
   staticGroups["INF"]["B_INF_TRA"] = clSDS_ConfigurationFlags::getInfTrafficKey();
   staticGroups["INF"]["B_INF_SXM"] = clSDS_ConfigurationFlags::getInfSXMKey();
   staticGroups["INF"]["E_INF_VASSIST"] = clSDS_ConfigurationFlags::getTcuInformationOnVariantType();
   staticGroups["AUD"]["B_AUD_DAB"] = clSDS_ConfigurationFlags::getAudDABKey();
   staticGroups["AUD"]["B_AUD_SXM"] = clSDS_ConfigurationFlags::getAudSXMKey();
   staticGroups["AUD"]["B_AUD_RDS"] = clSDS_ConfigurationFlags::getAudRDSKey();
   staticGroups["AUD"]["B_AUD_HD"] = clSDS_ConfigurationFlags::getAudHDKey();
   staticGroups["AUD"]["B_AUD_TV"] = clSDS_ConfigurationFlags::getAudTVKey();
   staticGroups["AUD"]["B_AUD_CarPlay"] = clSDS_ConfigurationFlags::getCarplay();
   staticGroups["AUD"]["B_AUD_AndroidAuto"] = clSDS_ConfigurationFlags::getAndroidAuto();
   staticGroups["AUD"]["B_AUD_CD"] = clSDS_ConfigurationFlags::getCDAvailableKey();
   staticGroups["AUD"]["B_AUD_DVD"] = clSDS_ConfigurationFlags::getDVDAvailableKey();
   staticGroups["AUD"]["B_AUD_MusicBox"] = clSDS_ConfigurationFlags::getMusicBoxAvailableKey();
   staticGroups["AUD"]["E_AUD_FREQCY_TYPE"] = clSDS_ConfigurationFlags::getAudFreqTypeKey();
   staticGroups["PHO"]["B_PHO"] = 1; //enable phone functionality independent of "Bluetooth function" config in KDS
   staticGroups["PHO"]["B_PHO_Siri"] = clSDS_ConfigurationFlags::getSIRIStatus();
   staticGroups["PHO"]["B_PHO_VoiceAssistant"] = clSDS_ConfigurationFlags::getVoiceAssistant();
   staticGroups["INF"]["B_INF_FuelEconomy"] = clSDS_ConfigurationFlags::getVINFFuelEconomykey();
   staticGroups["INF"]["B_INF_Maintenance"] = clSDS_ConfigurationFlags::getVINFMaintenancekey();
   staticGroups["INF"]["B_INF_WhereAmI"] = clSDS_ConfigurationFlags::getWhereAmIKey();
   staticGroups["EVE"]["B_EVE"] = clSDS_ConfigurationFlags::getEveKey();
   addCommandFlagsFromConfigFile(staticGroups);
   addRootContextCommandFlags(staticGroups);

   return formatXml(staticGroups);
}


/**************************************************************************//**

******************************************************************************/
std::string clSDS_XMLStringCreation::getDynamicXmlString(const SDSConfigData& cfgData)
{
   GroupMap dynamicGroups;

   dynamicGroups["AUD"]["B_AUD_Nbest"] = cfgData.audioNBestMatch;
   dynamicGroups["GRL"]["B_GRL_BEEPONLY_MODE"] = cfgData.beepOnlyMode;
   dynamicGroups["GRL"]["B_GRL_Voice_BargeIn"] = cfgData.voiceBargeIn;
   dynamicGroups["INF"]["B_INF"] = cfgData.infoAvailable;
   dynamicGroups["INF"]["B_INF_TRAFFIC_AVAILABLE"] = cfgData.trafficServiceAvailable;
   dynamicGroups["INF"]["B_INF_Celsius"] = cfgData.unitInCelsius;
   dynamicGroups["NAV"]["B_NAV"] = cfgData.naviAvailable;
   dynamicGroups["NAV"]["B_NAV_KM"] = cfgData.unitInKm;
   dynamicGroups["PHO"]["B_PHO_Nbest"] = cfgData.phoneBookNBestMatch;
   dynamicGroups["PHO"]["B_PHO_Multipoint_User_Setting"] = true;
   dynamicGroups["PHO"]["B_PHO_Rel_Mom_Asgnd"] = cfgData.phoneRelationshipFlags.momUsed;
   dynamicGroups["PHO"]["B_PHO_Rel_Dad_Asgnd"] = cfgData.phoneRelationshipFlags.dadUsed;
   dynamicGroups["PHO"]["B_PHO_Rel_Parents_Asgnd"] = cfgData.phoneRelationshipFlags.parentsUsed;
   dynamicGroups["PHO"]["B_PHO_Rel_Brother_Asgnd"] = cfgData.phoneRelationshipFlags.brotherUsed;
   dynamicGroups["PHO"]["B_PHO_Rel_Sister_Asgnd"] = cfgData.phoneRelationshipFlags.sisterUsed;
   dynamicGroups["PHO"]["B_PHO_Rel_Child_Asgnd"] = cfgData.phoneRelationshipFlags.childUsed;
   dynamicGroups["PHO"]["B_PHO_Rel_Son_Asgnd"] = cfgData.phoneRelationshipFlags.sonUsed;
   dynamicGroups["PHO"]["B_PHO_Rel_Daughter_Asgnd"] = cfgData.phoneRelationshipFlags.daughterUsed;
   dynamicGroups["PHO"]["B_PHO_Rel_Friend_Asgnd"] = cfgData.phoneRelationshipFlags.friendUsed;
   dynamicGroups["PHO"]["B_PHO_Rel_Partner_Asgnd"] = cfgData.phoneRelationshipFlags.partnerUsed;
   dynamicGroups["PHO"]["B_PHO_Rel_Wife_Asgnd"] = cfgData.phoneRelationshipFlags.wifeUsed;
   dynamicGroups["PHO"]["B_PHO_Rel_Husband_Asgnd"] = cfgData.phoneRelationshipFlags.husbandUsed;
   dynamicGroups["PHO"]["B_PHO_Rel_Home_Asgnd"] = cfgData.phoneRelationshipFlags.homeUsed;
   dynamicGroups["PHO"]["B_PHO_Rel_Office_Asgnd"] = cfgData.phoneRelationshipFlags.officeUsed;
   dynamicGroups["PHO"]["B_PHO_Rel_Assistant_Asgnd"] = cfgData.phoneRelationshipFlags.assistantUsed;
   dynamicGroups["PHO"]["B_PHO_Rel_Manager_Asgnd"] = cfgData.phoneRelationshipFlags.managerUsed;
   dynamicGroups["PHO"]["B_PHO_Rel_Other_Asgnd"] = cfgData.phoneRelationshipFlags.otherUsed;

   return formatXml(dynamicGroups);
}


/**************************************************************************//**
Root Context Phone expert mode tags
******************************************************************************/
void clSDS_XMLStringCreation::addRootContextCommandFlags(GroupMap& staticGroups)
{
   staticGroups["PHO"]["B_PHO_RC_VoiceMail"] = 1;
   staticGroups["PHO"]["B_PHO_RC_VoiceAssistant"] = 1;
   staticGroups["PHO"]["B_PHO_RC_Siri"] = 1;
   staticGroups["PHO"]["B_PHO_RC_SendText"] = 1;
   staticGroups["PHO"]["B_PHO_RC_SelectPhone"] = 1;
   staticGroups["PHO"]["B_PHO_RC_Redial"] = 1;
   staticGroups["PHO"]["B_PHO_RC_RecentCalls"] = 1;
   staticGroups["PHO"]["B_PHO_RC_ReadText"] = 1;
   staticGroups["PHO"]["B_PHO_RC_Quickdial"] = 1;
   staticGroups["PHO"]["B_PHO_RC_Phonebook"] = 1;
   staticGroups["PHO"]["B_PHO_RC_OutgoingCalls"] = 1;
   staticGroups["PHO"]["B_PHO_RC_MissedCalls"] = 1;
   staticGroups["PHO"]["B_PHO_RC_IncomingCalls"] =  1;
   staticGroups["PHO"]["B_PHO_RC_Call"] = 1;

   staticGroups["NAV"]["B_NAV_RC_Zoom"] = 1;
   staticGroups["NAV"]["B_NAV_RC_Work"] = 1;
   staticGroups["NAV"]["B_NAV_RC_VoiceGuidance"] = 1;
   staticGroups["NAV"]["B_NAV_RC_TurnMap"] = 1;
   staticGroups["NAV"]["B_NAV_RC_StreetAddress"] = 1;
   staticGroups["NAV"]["B_NAV_RC_RepeatInstruction"] = 1;
   staticGroups["NAV"]["B_NAV_RC_ReRoute"] = 1;
   staticGroups["NAV"]["B_NAV_RC_PreviousDestinations"] = 1;
   staticGroups["NAV"]["B_NAV_RC_POICategory"] = 1;
   staticGroups["NAV"]["B_NAV_RC_POI"] = 1;
   staticGroups["NAV"]["B_NAV_RC_Nearby"] = 1;
   staticGroups["NAV"]["B_NAV_RC_Map"] = 1;
   staticGroups["NAV"]["B_NAV_RC_Icons"] = 1;
   staticGroups["NAV"]["B_NAV_RC_Home"] = 1;
   staticGroups["NAV"]["B_NAV_RC_CityCenter"] = 1;
   staticGroups["NAV"]["B_NAV_RC_CancelRoute"] = 1;
   staticGroups["NAV"]["B_NAV_RC_AddressBook"] = 1;
   staticGroups["NAV"]["B_NAV_RC_ChangeCountry"] = 1;
   staticGroups["NAV"]["B_NAV_RC_DistanceDetour"] = 1;

   staticGroups["INF"]["B_INF_RC_WhereAmI"] = 1;
   staticGroups["INF"]["B_INF_RC_WeatherMap"] = 1;
   staticGroups["INF"]["B_INF_RC_VoiceMenu"] = 1;
   staticGroups["INF"]["B_INF_RC_Traffic"]  = 1;
   staticGroups["INF"]["B_INF_RC_Stocks"] = 1;
   staticGroups["INF"]["B_INF_RC_Sports"] = 1;
   staticGroups["INF"]["B_INF_RC_Parking"] = 1;
   staticGroups["INF"]["B_INF_RC_MovieTheaters"] = 1;
   staticGroups["INF"]["B_INF_RC_FuelEconomy"] = 1;
   staticGroups["INF"]["B_INF_RC_FuelPrices"] = 1;
   staticGroups["INF"]["B_INF_RC_EnergyFlow"] = 1;
   staticGroups["INF"]["B_INF_RC_CurrentWeather"] = 1;
   staticGroups["INF"]["B_INF_RC_ClimateControl"] = 1;
   staticGroups["INF"]["B_INF_RC_6HourForecast"]  = 1;
   staticGroups["INF"]["B_INF_RC_5DayForecast"] = 1;
   staticGroups["INF"]["B_INF_RC_InformationFeeds"] = 1;
   staticGroups["INF"]["B_INF_RC_Maintenance"] = 1;
   staticGroups["INF"]["B_INF_RC_Temperature"] = 1;
   staticGroups["INF"]["B_INF_RC_TirePressure"] = 1;

   staticGroups["AUD"]["B_AUD_RC_TuneSXM"] = 1;
   staticGroups["AUD"]["B_AUD_RC_TuneFM"] = 1;
   staticGroups["AUD"]["B_AUD_RC_TuneAM"] = 1;
   staticGroups["AUD"]["B_AUD_RC_TuneDAB"] = 1;
   staticGroups["AUD"]["B_AUD_RC_TuneDRM"] = 1;
   staticGroups["AUD"]["B_AUD_RC_Preset"] = 1;
   staticGroups["AUD"]["B_AUD_RC_PlaySource"] = 1;
   staticGroups["AUD"]["B_AUD_RC_MusicSearch"] = 1;
   staticGroups["AUD"]["B_AUD_RC_CDTrack"] = 1;
   staticGroups["AUD"]["B_AUD_RC_TV"] = 1;

   staticGroups["APP"]["B_APP_RC_LaunchApp"] = 1;

   staticGroups["GRL"]["B_GRL_RC_Settings"] = 1;

   staticGroups["EVE"]["B_EVE_RC_DrivingRange"] = 1;
   staticGroups["EVE"]["B_EVE_RC_EnergyUsage"] = 1;
   staticGroups["EVE"]["B_EVE_RC_NearbyStations"] = 1;
}


/**************************************************************************//**
Flags Read from config file
******************************************************************************/
void clSDS_XMLStringCreation::addCommandFlagsFromConfigFile(GroupMap& staticGroups)
{
   std::map< std::string, std::string> configDataMap;
   std::map< std::string, std::string>::iterator it;
   std::vector <std::string>::iterator itr;
   clSDS_ConfigurationFlags::getConfigDataMap(configDataMap);
   fillConfigVectorValues();
   for (it = configDataMap.begin(); it != configDataMap.end(); it++)
   {
      std::string strConfig = "";
      std::string strConfigVal = "";
      strConfig = it->first;
      strConfigVal = it->second;

      for (itr = _configData.begin(); itr != _configData.end(); ++itr)
      {
         if ((strConfig.find(*itr) != std::string::npos))
         {
            {
               staticGroups[*itr][strConfig] = atoi(strConfigVal.c_str());
            }
         }
      }
   }
}
