/**************************************************************************//**
 * \file       clSDS_Userwords.cpp
 *
 * clSDS_Userwords class implementation
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "application/clSDS_Userwords.h"
#include "application/clSDS_SdsControl.h"
#include "application/clSDS_SDSStatus.h"
#include "application/clSDS_QuickDialList.h"
#include "SdsAdapter_Trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/clSDS_Userwords.cpp.trc.h"
#endif


#define SDS_USER_UNKNOWN            0xff
#define PHONE_PROFILE_UNKNOWN       0xff
#define AVAILABLE_PROFILES_UNKNOWN  0xffffffff
#define MAXIMUM_NUMBER_OF_PROFILES  7


using namespace sds_gui_fi::SdsPhoneService;


/**************************************************************************//**
* Destructor
******************************************************************************/
clSDS_Userwords::~clSDS_Userwords()
{
   _pSDSStatus = NULL;
   _pSdsControl = NULL;
   _pQuickDialList = NULL;
}


/**************************************************************************//**
* Constructor
******************************************************************************/
clSDS_Userwords::clSDS_Userwords(
   clSDS_SDSStatus* pSDSStatus,
   clSDS_SdsControl* pSdsControl,
   clSDS_QuickDialList* pQuickDialList)

   : _pSDSStatus(pSDSStatus)
   , _pSdsControl(pSdsControl)
   , _pQuickDialList(pQuickDialList)
   , _u32ActiveProfileNum(PHONE_PROFILE_UNKNOWN)
   , _u32CurrentSDSUser(SDS_USER_UNKNOWN)
   , _u32AvailableProfiles(AVAILABLE_PROFILES_UNKNOWN)
   , _userWordListID(0)
{
   if (_pSDSStatus != NULL)
   {
      _pSDSStatus->vRegisterObserver(this);
   }
}


/**************************************************************************//**
*
******************************************************************************/
tBool clSDS_Userwords::bIsVoiceControlIdle() const
{
   return _pSDSStatus && _pSDSStatus->bIsIdle();
}


/**************************************************************************//**
*
******************************************************************************/
tVoid clSDS_Userwords::vSDSStatusChanged()
{
   bSetSDSUser();
}


/**************************************************************************//**
*
******************************************************************************/
tVoid clSDS_Userwords::vSetActiveProfile(tU32 u32ActiveProfileNum)
{
   _u32ActiveProfileNum = u32ActiveProfileNum - 1 ;
   bSetSDSUser();
}


/**************************************************************************//**
*
******************************************************************************/
tBool clSDS_Userwords::bSetSDSUser()
{
   tU32 u32NewSDSUser = _u32ActiveProfileNum;
   if ((_u32CurrentSDSUser != u32NewSDSUser)
         && (bIsVoiceControlIdle() == TRUE)
         && (PHONE_PROFILE_UNKNOWN != _u32ActiveProfileNum))
   {
      if (_pSdsControl)
      {
         _pSdsControl->vSetUWProfile(u32NewSDSUser, sds2hmi_fi_tcl_e8_Domain ::FI_EN_PHONE);
      }
      _u32CurrentSDSUser = u32NewSDSUser;
      return TRUE;
   }
   return FALSE;
}


/**************************************************************************//**
*
******************************************************************************/
tVoid clSDS_Userwords::vDeleteUserword(tU32 u32UWID, sds2hmi_fi_tcl_e8_Domain::tenType enDomain)
{
   if (bIsVoiceControlIdle() && (_pSdsControl != NULL))
   {
      _pSdsControl->vDeleteUserWord(u32UWID, enDomain);
   }
}


/**************************************************************************//**
*
******************************************************************************/
tVoid clSDS_Userwords::vDeleteAllUserwords(tU32 u32UWProfile, sds2hmi_fi_tcl_e8_Domain::tenType enDomain)
{
   if (bIsVoiceControlIdle() && (_pSdsControl != NULL))
   {
      _pSdsControl->vDeleteAllUserWords(u32UWProfile - 1, enDomain);
   }
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_Userwords::handleUserWordActions(const ::boost::shared_ptr< VoiceTagActionRequest >& request)
{
   _userWordListID = request->getUniqueContactID();
   vSetActiveProfile(request->getPhoneProfileID());
   switch (request->getUserAction())
   {
      case UserWordAction__RECORD:
         recordUserWord(_userWordListID, sds2hmi_fi_tcl_e8_Domain ::FI_EN_PHONE);
         break;

      case UserWordAction__REPLACE:
         replaceUserWord(_userWordListID, sds2hmi_fi_tcl_e8_Domain ::FI_EN_PHONE);
         break;

      case UserWordAction__DELETE:
         vDeleteUserword(_userWordListID, sds2hmi_fi_tcl_e8_Domain ::FI_EN_PHONE);
         break;

      case UserWordAction__PLAY:
         playUserWord(_userWordListID, sds2hmi_fi_tcl_e8_Domain ::FI_EN_PHONE);
         break;
      case UserWordAction__DELETE_ALL:
         vDeleteAllUserwords(request->getPhoneProfileID(), sds2hmi_fi_tcl_e8_Domain ::FI_EN_PHONE);
         break;

      default:
         break;
   }
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_Userwords::recordUserWord(unsigned long u32UWID, sds2hmi_fi_tcl_e8_Domain::tenType enDomain)
{
   if (_pSdsControl != NULL)
   {
      _pSdsControl->recordUserWord(u32UWID, enDomain);
   }
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_Userwords::replaceUserWord(unsigned long u32UWID, sds2hmi_fi_tcl_e8_Domain::tenType enDomain)
{
   if (_pSdsControl != NULL)
   {
      _pSdsControl->replaceUserWord(u32UWID, enDomain);
   }
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_Userwords::playUserWord(unsigned long u32UWID, sds2hmi_fi_tcl_e8_Domain::tenType enDomain)
{
   if (_pSdsControl != NULL)
   {
      _pSdsControl->playUserWord(u32UWID, enDomain);
   }
}


/**************************************************************************//**
*
******************************************************************************/
uint32 clSDS_Userwords::getContactID() const
{
   return _userWordListID;
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_Userwords::phoneProfileChanged(uint8 profileID)
{
   vSetActiveProfile(profileID);
}
