/**************************************************************************//**
 * \file       clSDS_ScreenData.cpp
 *
 * clSDS_ScreenData class implementation
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "application/clSDS_ScreenData.h"
#include "application/clSDS_ListItems.h"
#include "application/clSDS_TagContents.h"
#include "application/clSDS_XMLDoc.h"
#include "application/StringUtils.h"
#include "SdsAdapter_Trace.h"
#include <cstdlib>

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/clSDS_ScreenData.cpp.trc.h"
#endif


/**************************************************************************//**
* Destructor
******************************************************************************/
clSDS_ScreenData::~clSDS_ScreenData()
{
   try
   {
      delete _pXMLDoc;
   }
   catch (...)
   {
      ETG_TRACE_FATAL(("Exception in destructor"));
   }
}


/**************************************************************************//**
* Constructor
******************************************************************************/
clSDS_ScreenData::clSDS_ScreenData(std::string const& oXMLString)
{
   _pXMLDoc = new clSDS_XMLDoc(oXMLString);
}


/***********************************************************************//**
*
***************************************************************************/
std::string clSDS_ScreenData::readScreenId() const
{
   if (NULL != _pXMLDoc)
   {
      std::vector<clSDS_TagContents> oTagContents = _pXMLDoc->oGetElementsOfTag("SCREENID");
      if (oTagContents.size())
      {
         std::string oScreenId(oTagContents.back().oTagValue.c_str());
         return oScreenId;
      }
   }
   return "";
}


/***********************************************************************//**
*
***************************************************************************/
std::string clSDS_ScreenData::readTemplateId()
{
   if (NULL != _pXMLDoc)
   {
      std::vector<clSDS_TagContents> oHeaderTagContents = _pXMLDoc->oGetElementsOfTag("HEADER");
      if (oHeaderTagContents.size())
      {
         std::vector<clSDS_TagContents> oTagContents = oHeaderTagContents.back().voChildrenTags;
         for (tU32 u32Index = 0; u32Index < oTagContents.size(); u32Index++)
         {
            if ((oTagContents[u32Index].oTagName == "TEMPLATE"))
            {
               return readTidTagValue(oTagContents[u32Index].voChildrenTags);
            }
         }
      }
   }
   return "";
}


/***********************************************************************//**
*
***************************************************************************/
std::string clSDS_ScreenData::readTidTagValue(std::vector<clSDS_TagContents>&  oTagContents)
{
   for (tU32 u32Index = 0; u32Index < oTagContents.size(); u32Index++)
   {
      if ((oTagContents[u32Index].oTagName == "TID"))
      {
         std::string oTemplateId(oTagContents[u32Index].oTagValue.c_str());
         ETG_TRACE_USR1(("Template ID = %s", oTemplateId.c_str()));
         if (strcmp(oTemplateId.c_str(), "PhoneNumberDigits") == 0)
         {
            _templateId = oTagContents[u32Index].oTagValue.c_str();
         }
         return oTemplateId;
      }
   }
   ETG_TRACE_USR1(("No Template ID"));
   return "";
}


/***********************************************************************//**
*
***************************************************************************/
tU32 clSDS_ScreenData::u32GetCount(std::string const& oTag) const
{
   if (NULL != _pXMLDoc)
   {
      return _pXMLDoc->oGetElementsOfTag(oTag).size();
   }
   return 0;
}


/***********************************************************************//**
*
***************************************************************************/
tU32 clSDS_ScreenData::u32GetNumberOfElements() const
{
   return u32GetCount("ELEMENT");
}


/***********************************************************************//**
*
***************************************************************************/
tVoid clSDS_ScreenData::vStoreListContents(const std::vector<clSDS_ListItems>& oListContents)
{
   if (!oListContents.empty())
   {
      _oListContents = oListContents;
   }
}


/***********************************************************************//**
*
***************************************************************************/
std::vector<clSDS_ListItems> clSDS_ScreenData::oGetListItemsofView(tVoid) const
{
   if (!_oListContents.empty())
   {
      return _oListContents;
   }

   std::vector<clSDS_ListItems>  oListContents;
   if (NULL != _pXMLDoc)
   {
      std::vector<clSDS_TagContents> oTagContents = _pXMLDoc->oGetElementsOfTag("ELEMENT");
      for (tU32 u32Index = 0; u32Index < oTagContents.size(); u32Index++)
      {
         clSDS_ListItems oListContent;

         std::string oTagValue = oTagContents[u32Index].oTagValue;

         std::vector<uint> localIds = getLocalIds(oTagContents[u32Index].voChildrenTags);

         oListContent.oDescription.szString = formatTagValue(oTagValue, localIds);

         oListContents.push_back(oListContent);
      }

      printListItems(oListContents);
   }
   return oListContents;
}


tVoid clSDS_ScreenData::printListItems(const std::vector<clSDS_ListItems>& listContents) const
{
   std::string oTempTrace;

   for (tU32 u32Index = 0; u32Index < listContents.size(); u32Index++)
   {
      oTempTrace.append(StringUtils::toString(u32Index + 1) + ": " + listContents[u32Index].oDescription.szString);
   }

   if (!oTempTrace.empty())
   {
      oTempTrace.append("$");
      std::string oPart1(oTempTrace.substr(0, 226));
      ETG_TRACE_USR1(("NBestList = %s", oPart1.c_str()));
      if (oTempTrace.length() > 226)
      {
         std::string oPart2(oTempTrace.substr(226, oTempTrace.length() - 226));
         ETG_TRACE_USR1(("%s", oPart2.c_str()));
      }
   }
}


std::vector<uint> clSDS_ScreenData::getLocalIds(std::vector<clSDS_TagContents>& oChildrenTags) const
{
   std::vector<uint> localIds;

   for (uint i = 0; i < oChildrenTags.size(); i++)
   {
      if (oChildrenTags[i].oTagName == "LOCALID")
      {
         uint localId = (uint)atoi(oChildrenTags[i].oTagValue.c_str());
         localIds.push_back(localId);
      }
   }
   return localIds;
}


Sds_TextId clSDS_ScreenData::getTextIDForLocalId(const std::string& localIdString) const
{
   uint localId = (uint)std::atoi(localIdString.c_str());
   return getTextIDForLocalId(localId);
}


Sds_TextId clSDS_ScreenData::getTextIDForLocalId(uint localId) const
{
   const uint LOCATIONS_OFFSET = 100;
   const Sds_TextId locations[] = { SDS_TEXT_HOMENUMBER, SDS_TEXT_WORKNUMBER, SDS_TEXT_MOBILENUMBER, SDS_TEXT_OTHER};
   const uint LOCATIONS_COUNT = sizeof(locations) / sizeof(locations[0]);

   const uint RELATIONSHIPS_OFFSET = 400;
   const Sds_TextId relationships[] =
   {
      SDS_TEXT_MOM, SDS_TEXT_DAD, SDS_TEXT_PARENTS, SDS_TEXT_BROTHER, SDS_TEXT_SISTER,
      SDS_TEXT_CHILD, SDS_TEXT_SON, SDS_TEXT_DAUGHTER,
      SDS_TEXT_FRIEND, SDS_TEXT_PARTNER, SDS_TEXT_WIFE, SDS_TEXT_HUSBAND,
      SDS_TEXT_MYHOME, SDS_TEXT_MYOFFICE, SDS_TEXT_ASSISTANT, SDS_TEXT_MANAGER, SDS_TEXT_OTHER
   };
   const uint RELATIONSHIPS_COUNT = sizeof(relationships) / sizeof(relationships[0]);

   const uint DISTANCE_UNIT_OFFSET = 1;
   const Sds_TextId units[] =
   {
      SDS_TEXT_KM, SDS_TEXT_METER, SDS_TEXT_MILE, SDS_TEXT_YARD, SDS_TEXT_FEET
   };
   const uint DISTANCE_UNIT_COUNT = sizeof(units) / sizeof(units[0]);

   if ((localId >= RELATIONSHIPS_OFFSET) && (localId < RELATIONSHIPS_OFFSET + RELATIONSHIPS_COUNT))
   {
      return relationships[localId - RELATIONSHIPS_OFFSET];
   }

   if ((localId >= LOCATIONS_OFFSET) && (localId < LOCATIONS_OFFSET + LOCATIONS_COUNT))
   {
      return locations[localId - LOCATIONS_OFFSET];
   }

   if ((localId >= DISTANCE_UNIT_OFFSET) && (localId < (DISTANCE_UNIT_OFFSET + DISTANCE_UNIT_COUNT)))
   {
      return units[localId - DISTANCE_UNIT_OFFSET];
   }

   NORMAL_M_ASSERT_ALWAYS();
   return SDS_TEXT_ID_LIMIT;  //lint !e527 Unreachable code - jnd2hi: behind assertion
}


std::string clSDS_ScreenData::formatTagValue(std::string tagValue, std::vector<uint> localIDs) const
{
   std::string oTagValue = tagValue;

   StringUtils::trim(oTagValue);//string may need trimming

   for (uint i = 0; i < localIDs.size(); i++)
   {
      Sds_TextId textID = getTextIDForLocalId(localIDs[i]);
      if (SDS_TEXT_ID_LIMIT != textID)
      {
         oTagValue.append(" (");
         oTagValue.append(Sds_TextDB_vGetText(textID));
         oTagValue.append(")");
      }
   }

   return oTagValue;
}


/***********************************************************************//**
*
***************************************************************************/
std::vector<tU32> clSDS_ScreenData::oGetListOfObjectIds(tVoid) const
{
   std::vector<tU32> listObjectIds;
   if (NULL != _pXMLDoc)
   {
      std::vector<clSDS_TagContents> oTagContents = _pXMLDoc->oGetElementsOfTag("ELEMENT");
      for (tU32 u32Index = 0; u32Index < oTagContents.size(); u32Index++)
      {
         std::vector<tU32> objectIds = getObjectIds(oTagContents[u32Index].voChildrenTags);

         if (!objectIds.empty())
         {
            //ASSERT(objectIds.size==1) //TODO: how to ASSERT
            listObjectIds.push_back(objectIds[0]);
         }
      }
   }
   return listObjectIds;
}


std::vector<tU32> clSDS_ScreenData::getObjectIds(std::vector<clSDS_TagContents>& oChildrenTags) const
{
   std::vector<tU32> objectIds;

   for (tU32 i = 0; i < oChildrenTags.size(); i++)
   {
      ETG_TRACE_USR1(("clSDS_ScreenData::getObjectIds TagName = %s", oChildrenTags[i].oTagName));
      if (oChildrenTags[i].oTagName == "OBJECTID")
      {
         tU32 objectId = (tU32)atoi(oChildrenTags[i].oTagValue.c_str());
         objectIds.push_back(objectId);
         ETG_TRACE_USR1(("clSDS_ScreenData::getObjectIds ObjectID = %i", objectId));
      }
   }
   return objectIds;
}


/***********************************************************************//**
*
***************************************************************************/
std::map<std::string, std::string> clSDS_ScreenData::oGetScreenVariableData(tVoid)
{
   std::vector<std::string> oTagList;
   oTagList.push_back("PARAMETERS");
   oTagList.push_back("PARAMETER");
   return oGetMap(oTagList, "NAME", "VALUE");
}


/***********************************************************************//**
*
***************************************************************************/
std::map<std::string, std::string> clSDS_ScreenData::oGetHeaderVariableData(tVoid)
{
   std::vector<std::string> oTagList;
   oTagList.push_back("HEADER");
   oTagList.push_back("TEMPLATE");
   oTagList.push_back("VARIABLE");
   return oGetMap(oTagList, "TAG", "VALUE");
}


/***********************************************************************//**
*
***************************************************************************/
std::vector<std::string> clSDS_ScreenData::oGetList(std::vector<std::string>& oTagList)
{
   if (oTagList.size() && NULL != _pXMLDoc)
   {
      std::vector<clSDS_TagContents> oTagContents = _pXMLDoc->oGetElementsOfTag(oTagList[0]);
      return oGetListofElements(oTagList, oTagContents);
   }
   std::vector<std::string> oDummyList;
   return oDummyList;
}


/***********************************************************************//**
*
***************************************************************************/
std::map<std::string, std::string> clSDS_ScreenData::oGetMap(std::vector<std::string>& oTagList, std::string const& oKeyId, std::string const& oValueId)
{
   if (oTagList.size()  && NULL != _pXMLDoc)
   {
      std::vector<clSDS_TagContents> oTagContents = _pXMLDoc->oGetElementsOfTag(oTagList[0]);
      return oGetMap(oTagList, oTagContents, oKeyId, oValueId);
   }
   std::map<std::string, std::string> oDummyMap;
   return oDummyMap;
}


/***********************************************************************//**
*
***************************************************************************/
std::map<std::string, std::string> clSDS_ScreenData::oGetMap(std::vector<std::string>& oTagList, std::vector<clSDS_TagContents>& oTagContents, std::string const& oKeyId, std::string const& oValueId)
{
   VariableData oData;
   oData.oMap[""] = ""; // Empty set which is added by default for all calls
   oData.oKeyId = oKeyId;
   oData.oValueId = oValueId;
   vAddDataToMap(0, oTagList, oTagContents, oData);
   return oData.oMap;
}


/***********************************************************************//**
*
***************************************************************************/
std::vector<std::string> clSDS_ScreenData::oGetListofElements(std::vector<std::string>& oTagList, std::vector<clSDS_TagContents>& oTagContents)
{
   std::vector<std::string> oDataList;
   oDataList.push_back(""); // Empty string symbolizes no condition which is added by default for all calls
   vAddDataToList(0, oTagList, oTagContents, oDataList);
   return oDataList;
}


/***********************************************************************//**
*
***************************************************************************/
tVoid clSDS_ScreenData::vAddDataToMap(tU32 u32TagIndex, std::vector<std::string>& oTagList, std::vector<clSDS_TagContents>& oTagContents, VariableData& oData)
{
   for (tU32 u32Index = 0; u32Index < oTagContents.size(); u32Index++)
   {
      if (bIsLastTag(u32TagIndex, oTagList))
      {
         vAddTagDataToMap(u32TagIndex, oTagList, oTagContents[u32Index], oData);
      }
      else
      {
         vGotoNextTag(u32TagIndex, oTagList, oTagContents[u32Index], oData);
      }
   }
}


/***********************************************************************//**
*
***************************************************************************/
tVoid clSDS_ScreenData::vAddDataToList(tU32 u32TagIndex, std::vector<std::string>& oTagList, std::vector<clSDS_TagContents>& oTagContents, std::vector<std::string>& oDataList)
{
   for (tU32 u32Index = 0; u32Index < oTagContents.size(); u32Index++)
   {
      if (bIsLastTag(u32TagIndex, oTagList))
      {
         vAddTagDataToList(u32TagIndex, oTagList, oTagContents[u32Index], oDataList);
      }
      else
      {
         vGotoNextTag(u32TagIndex, oTagList, oTagContents[u32Index], oDataList);
      }
   }
}


/***********************************************************************//**
*
***************************************************************************/
tVoid clSDS_ScreenData::vAddTagDataToList(
   tU32 u32TagIndex,
   std::vector<std::string>& oTagList,
   const clSDS_TagContents& oTagContent,
   std::vector<std::string>& oDataList) const
{
   if ((oTagContent.oTagName == (oTagList[u32TagIndex])))
   {
      oDataList.push_back(oTagContent.oTagValue);
   }
}


/***********************************************************************//**
*
***************************************************************************/
tVoid clSDS_ScreenData::vAddTagDataToMap(
   tU32 u32TagIndex,
   std::vector<std::string>& oTagList,
   const clSDS_TagContents& oTagContent,
   VariableData& oData)
{
   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(u32TagIndex);
   if (oTagContent.oTagName == oTagList.back())
   {
      std::vector<clSDS_TagContents> oTagContents = oTagContent.voChildrenTags;
      const std::string key = oGetKey(oTagContents, oData);
      const std::string value = oGetValue(oTagContents, oData);
      vAddToMap(key, value, oData);
      vGetPhoneNumberDigits(oTagContents, oData);
   }
}


/***********************************************************************//**
*
***************************************************************************/
tVoid clSDS_ScreenData::vAddToMap(std::string const& oKey, std::string const& oValue, VariableData& oData) const
{
   oData.oMap[oKey] = oValue;
}


/***********************************************************************//**
*
***************************************************************************/
std::string clSDS_ScreenData::oGetKey(std::vector<clSDS_TagContents>& oTagContents, const VariableData& oData) const
{
   for (tU32 u32Index = 0; u32Index < oTagContents.size(); u32Index++)
   {
      if ((oTagContents[u32Index].oTagName == (oData.oKeyId)))
      {
         ETG_TRACE_USR1(("Param = %s", oTagContents[u32Index].oTagValue.c_str()));
         return oTagContents[u32Index].oTagValue;
      }
   }
   return "";
}


/***********************************************************************//**
*
***************************************************************************/
std::string clSDS_ScreenData::oGetValue(std::vector<clSDS_TagContents>& oTagContents, const VariableData& oData) const
{
   for (tU32 u32Index = 0; u32Index < oTagContents.size(); u32Index++)
   {
      if ((oTagContents[u32Index].oTagName == (oData.oValueId)))
      {
         std::string value;

         // check for special case <VALUE><LOCALID>xxx</LOCALID></VALUE>
         if (hasLocalIDValue(oTagContents[u32Index].voChildrenTags))
         {
            value = getLocalIDValue(oTagContents[u32Index].voChildrenTags[0].oTagValue);
         }
         //workaround for sds bug NCG3D-80614 <VALUE><OBJECTID>xxx</OBJECTID></VALUE>
         else if (oTagContents[u32Index].voChildrenTags.size() == 1 && oTagContents[u32Index].voChildrenTags[0].oTagName == "OBJECTID")
         {
            value = oTagContents[u32Index].voChildrenTags[0].oTagValue;
         }
         else //regular case: <VALUE>xxx</VALUE>
         {
            value = oTagContents[u32Index].oTagValue;
         }

         ETG_TRACE_USR1(("Value = %s", value.c_str()));

         return value;
      }
   }
   return "";
}


tBool clSDS_ScreenData::hasLocalIDValue(std::vector<clSDS_TagContents> voChildrenTags) const
{
   if (voChildrenTags.size() == 1 &&
         voChildrenTags[0].oTagName == "LOCALID")
   {
      return true;
   }
   return false;
}


std::string clSDS_ScreenData::getLocalIDValue(std::string rawLocalID) const
{
   Sds_TextId textId = getTextIDForLocalId(rawLocalID);
   return Sds_TextDB_vGetText(textId);
}


/***********************************************************************//**
*
***************************************************************************/
tVoid clSDS_ScreenData::vGotoNextTag(tU32 u32TagIndex, std::vector<std::string>& oTagList, clSDS_TagContents& oTagContent, VariableData& oData)
{
   if (u32TagIndex < (oTagList.size() - 1))
   {
      if ((oTagContent.oTagName == (oTagList[u32TagIndex])))
      {
         vAddDataToMap((u32TagIndex + 1), oTagList, oTagContent.voChildrenTags, oData);
      }
   }
}


/***********************************************************************//**
*
***************************************************************************/
tVoid clSDS_ScreenData::vGotoNextTag(tU32 u32TagIndex, std::vector<std::string>& oTagList, clSDS_TagContents& oTagContent, std::vector<std::string>& oDataList)
{
   if (u32TagIndex < (oTagList.size() - 1))
   {
      if ((oTagContent.oTagName == (oTagList[u32TagIndex])))
      {
         vAddDataToList((u32TagIndex + 1), oTagList, oTagContent.voChildrenTags, oDataList);
      }
   }
}


/***********************************************************************//**
*
***************************************************************************/
tBool clSDS_ScreenData::bIsLastTag(tU32 u32TagIndex, const std::vector<std::string>& oTagList) const
{
   if (u32TagIndex == (oTagList.size() - 1))
   {
      return TRUE;
   }
   return FALSE;
}


/***********************************************************************//**
*
***************************************************************************/
tVoid clSDS_ScreenData::vGetPhoneNumberDigits(std::vector<clSDS_TagContents>& oTagContents, const VariableData& oData)
{
   for (tU32 u32Index = 0; u32Index < oTagContents.size(); u32Index++)
   {
      if ((oTagContents[u32Index].oTagName == (oData.oValueId)))
      {
         if (strcmp(_templateId.c_str(), "PhoneNumberDigits") == 0)
         {
            _phoneNumberDigits = oTagContents[u32Index].oTagValue;
            ETG_TRACE_USR1(("PhNumDigits_TagValue = %s", _phoneNumberDigits.c_str()));
         }
      }
   }
}


/***********************************************************************//**
*
***************************************************************************/
std::string clSDS_ScreenData::getPhNumDigit() const
{
   return _phoneNumberDigits;
}


/***********************************************************************//**
*
***************************************************************************/
bool clSDS_ScreenData::getIsSpeakableList()
{
   return (oGetScreenVariableData()["SPEAKABLE_LIST_ITEMS"] == "TRUE");
}
