/**************************************************************************//**
 * \file       clSDS_QuickDialList.cpp
 *
 * This file is part of the SdsAdapter component.
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "application/clSDS_QuickDialList.h"
#include "application/clSDS_StringVarList.h"
#include "application/clSDS_ScreenData.h"
#include "application/StringUtils.h"
#include "application/PhoneUtils.h"
#include "view_db/Sds_TextDB.h"

#include "SdsAdapter_Trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/clSDS_QuickDialList.cpp.trc.h"
#endif


using namespace MOST_PhonBk_FI;
using namespace most_PhonBk_fi_types;
using namespace most_PhonBk_fi_types_Extended;
using namespace sds_gui_fi::SdsPhoneService;


/**************************************************************************//**
 *Destructor
 ******************************************************************************/
clSDS_QuickDialList::~clSDS_QuickDialList()
{
}


/**************************************************************************//**
 *Constructor
 ******************************************************************************/
clSDS_QuickDialList::clSDS_QuickDialList(::boost::shared_ptr< ::MOST_PhonBk_FI::MOST_PhonBk_FIProxy> phonebookProxy)
   : clSDS_QuickDialListCreator(phonebookProxy)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_QuickDialList::onAvailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& stateChange)
{
   if (proxy == _phonebookProxy)
   {
      _phonebookProxy->sendPreferredPhoneBookSortOrderUpReg(*this);
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_QuickDialList::onUnavailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _phonebookProxy)
   {
      _phonebookProxy->sendPreferredPhoneBookSortOrderRelUpRegAll();
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
tU32 clSDS_QuickDialList::u32GetSize()
{
   return _quickDialListForDisplay.size();
}


/**************************************************************************//**
 *
 ******************************************************************************/
std::vector<clSDS_ListItems> clSDS_QuickDialList::oGetItems(tU32 u32StartIndex, tU32 u32EndIndex)
{
   std::vector<clSDS_ListItems> oListItems;
   for (tU32 u32Index = u32StartIndex; u32Index < u32EndIndex; u32Index++)
   {
      clSDS_ListItems oListItem;
      oListItem.oDescription.szString = oGetItem(u32Index);
      oListItems.push_back(oListItem);
   }
   return oListItems;
}


/**************************************************************************//**
 *
 ******************************************************************************/
std::string clSDS_QuickDialList::oGetItem(tU32 u32Index)
{
   if (!oGetName(u32Index).empty())
   {
      return (oGetName(u32Index) + " " + toLanguageString(_quickDialListForDisplay[u32Index].phoneNumberType) + " " + addParanthesis(oGetRelationship(u32Index)));
   }
   else if (oGetPhoneNumber(u32Index) != "")
   {
      return (oGetPhoneNumber(u32Index) + " " + toLanguageString(_quickDialListForDisplay[u32Index].phoneNumberType) + " " + addParanthesis(oGetRelationship(u32Index)));
   }
   else
   {
      return Sds_TextDB_vGetText(SDS_TEXT_PRIVATE_NUMBER);
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
tBool clSDS_QuickDialList::bSelectElement(tU32 u32SelectedIndex)
{
   if (u32SelectedIndex > 0)
   {
      clSDS_StringVarList::vSetVariable("$(ContactName)", oGetName(u32SelectedIndex - 1));
      return TRUE;
   }
   return FALSE;
}


/**************************************************************************//**
 *
 ******************************************************************************/
std::string clSDS_QuickDialList::oGetSelectedItem(tU32 u32Index)
{
   if (u32Index > 0)
   {
      return oGetPhoneNumber(u32Index - 1);
   }
   return "";
}


/**************************************************************************//**
 *
 ******************************************************************************/
std::string clSDS_QuickDialList::oGetName(tU32 u32Index)
{
   std::string contactName = "";
   if (u32Index < _quickDialListForDisplay.size())
   {
      //Remark: currently, the contacts in Vehicle Phonebook are stored only with first name
      //This is due to the fact that phone HMI stores the quick dial contact with first name containing the full name and empty second name.
      if (_phonebookProxy->hasPreferredPhoneBookSortOrder()
            && _phonebookProxy->getPreferredPhoneBookSortOrder().getE8PreferredPhoneBookSortOrder() == T_e8_PhonBkPreferredPhoneBookSortOrder__e8PREFERRED_SORT_ORDER_LASTNAME)
      {
         contactName = _quickDialListForDisplay[u32Index].lastName + " " + _quickDialListForDisplay[u32Index].firstName;
      }
      else
      {
         contactName = _quickDialListForDisplay[u32Index].firstName + " " + _quickDialListForDisplay[u32Index].lastName;
      }
   }
   return StringUtils::trim(contactName);
}


/**************************************************************************//**
 *
 ******************************************************************************/
std::string clSDS_QuickDialList::oGetRelationship(tU32 u32Index)
{
   if (u32Index < _quickDialListForDisplay.size())
   {
      return _quickDialListForDisplay[u32Index].relationship;
   }
   return "";
}


/**************************************************************************//**
 *
 ******************************************************************************/
std::string clSDS_QuickDialList::oGetPhoneNumber(tU32 u32Index)
{
   if (u32Index < _quickDialListForDisplay.size())
   {
      return _quickDialListForDisplay[u32Index].phoneNumber;
   }
   return "";
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_QuickDialList::vGetListInfo(sds2hmi_fi_tcl_e8_HMI_ListType::tenType enListType)
{
   switch (enListType)
   {
      case sds2hmi_fi_tcl_e8_HMI_ListType::FI_EN_LIST_PHONE_USERWORDS:
         requestQuickDialList();
         break;

      default:
         notifyListObserver();
         break;
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_QuickDialList::onPreferredPhoneBookSortOrderError(
   const ::boost::shared_ptr< MOST_PhonBk_FI::MOST_PhonBk_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< MOST_PhonBk_FI::PreferredPhoneBookSortOrderError >& /*error*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_QuickDialList::onPreferredPhoneBookSortOrderStatus(
   const ::boost::shared_ptr< MOST_PhonBk_FI::MOST_PhonBk_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< MOST_PhonBk_FI::PreferredPhoneBookSortOrderStatus >& /*status*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
std::string clSDS_QuickDialList::getRelationship(const std::string& localID) const
{
   clSDS_ScreenData screenData("");
   std::string relationship = screenData.getLocalIDValue(localID);
   return relationship;
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_QuickDialList::onQuickDialListDataChanged()
{
   notifyListObserver();
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_QuickDialList::addQuickDialDataToList(const::boost::shared_ptr<GetContactDetailsExtendedResult>& result)
{
   QuickDialContact quickDialContactItem;
   quickDialContactItem.firstName = result->getOContactDetailsExtended().getSFirstName();
   quickDialContactItem.lastName = result->getOContactDetailsExtended().getSLastName();
   quickDialContactItem.phoneNumber = PhoneUtils::getPhoneNumber(result->getOContactDetailsExtended());
   quickDialContactItem.phoneNumberType = PhoneUtils::getPhoneNumberType(result->getOContactDetailsExtended());
   quickDialContactItem.relationship = getRelationship(result->getOContactDetailsExtended().getSCategory());
   _quickDialListForDisplay.push_back(quickDialContactItem);
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_QuickDialList::clearQuickDialData()
{
   _quickDialListForDisplay.clear();
}


/***********************************************************************//**
 *
 ***************************************************************************/
std::vector<sds2hmi_fi_tcl_HMIElementDescription> clSDS_QuickDialList::getHmiElementDescription(unsigned int index)
{
   std::vector<sds2hmi_fi_tcl_HMIElementDescription> hmiElementDescriptionList;
   sds2hmi_fi_tcl_HMIElementDescription hmiElementDescription = getHmiElementDescriptionItem(index - 1);
   hmiElementDescriptionList.push_back(hmiElementDescription);
   return hmiElementDescriptionList;
}


/***********************************************************************//**
 *
 ***************************************************************************/
std::vector<sds2hmi_fi_tcl_HMIElementDescrptionList> clSDS_QuickDialList::getHmiListDescription()
{
   ETG_TRACE_USR1(("clSDS_QuickDialList::getHmiListDescription"));
   std::vector<sds2hmi_fi_tcl_HMIElementDescrptionList> hmiElementDescriptionList;
   for (size_t index = 0; index < _quickDialListForDisplay.size(); index++)
   {
      sds2hmi_fi_tcl_HMIElementDescrptionList hmiElementList;
      sds2hmi_fi_tcl_HMIElementDescription hmiElement = getHmiElementDescriptionItem(index);
      hmiElementList.DescriptionList.push_back(hmiElement);
      hmiElementDescriptionList.push_back(hmiElementList);
   }
   return hmiElementDescriptionList;
}


/***********************************************************************//**
 *
 ***************************************************************************/
sds2hmi_fi_tcl_HMIElementDescription clSDS_QuickDialList::getHmiElementDescriptionItem(unsigned int index)
{
   sds2hmi_fi_tcl_HMIElementDescription hmiElementDescription;
   std::string descriptorValue;
   if (!oGetName(index).empty())
   {
      hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_CONTACTNAME;
      descriptorValue = oGetName(index);
   }
   else if (!oGetPhoneNumber(index).empty())
   {
      hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_PHONENUMBER;
      descriptorValue = oGetPhoneNumber(index);
   }
   appendNumberTypeAndRelation(descriptorValue, _quickDialListForDisplay[index]);
   hmiElementDescription.DescriptorValue = descriptorValue.c_str();
   return hmiElementDescription;
}


/***********************************************************************//**
 *
 ***************************************************************************/
std::string clSDS_QuickDialList::toLanguageString(sds2hmi_fi_tcl_e8_PHN_NumberType::tenType phoneNumberType)
{
   std::string numTypeString = "";
   switch (phoneNumberType)
   {
      case sds2hmi_fi_tcl_e8_PHN_NumberType::FI_EN_MOBILE1:
      case sds2hmi_fi_tcl_e8_PHN_NumberType::FI_EN_MOBILE2:
         numTypeString = Sds_TextDB_vGetText(SDS_TEXT_MOBILENUMBER);
         break;

      case sds2hmi_fi_tcl_e8_PHN_NumberType::FI_EN_HOME1:
      case sds2hmi_fi_tcl_e8_PHN_NumberType::FI_EN_HOME2:
         numTypeString = Sds_TextDB_vGetText(SDS_TEXT_HOMENUMBER);
         break;

      case sds2hmi_fi_tcl_e8_PHN_NumberType::FI_EN_OFFICE1:
      case sds2hmi_fi_tcl_e8_PHN_NumberType::FI_EN_OFFICE2:
         numTypeString = Sds_TextDB_vGetText(SDS_TEXT_WORKNUMBER);
         break;

      case sds2hmi_fi_tcl_e8_PHN_NumberType::FI_EN_OTHER:
         numTypeString = Sds_TextDB_vGetText(SDS_TEXT_OTHER);
         break;

      default :
         break;
   }
   return numTypeString;;
}


/***********************************************************************//**
 *
 ***************************************************************************/
std::string clSDS_QuickDialList::addParanthesis(const std::string& str) const
{
   std::string strWithParanthesis;
   if (str != "")
   {
      strWithParanthesis = "(" + str + ")"; //apply formating
   }
   return strWithParanthesis;
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_QuickDialList::appendNumberTypeAndRelation(std::string& descriptionString, const QuickDialContact& quickDialListItem)
{
   if (!(toLanguageString(quickDialListItem.phoneNumberType)).empty())
   {
      descriptionString.append(" ");
      descriptionString.append(toLanguageString(quickDialListItem.phoneNumberType));
   }
   if (!quickDialListItem.relationship.empty())
   {
      descriptionString.append(" ");
      descriptionString.append(quickDialListItem.relationship);
   }
}
