/**************************************************************************//**
 * \file       clSDS_PreviousDestinationsList.cpp
 *
 * This file is part of the SdsAdapter component.
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "application/clSDS_PreviousDestinationsList.h"
#include "application/clSDS_StringVarList.h"
#include "application/GuiService.h"

using namespace org::bosch::cm::navigation::NavigationService;
using namespace sds_gui_fi::SdsGuiService;
/**************************************************************************//**
*Destructor
******************************************************************************/
clSDS_PreviousDestinationsList::~clSDS_PreviousDestinationsList()
{
   _pGuiService = NULL;
   oClearPreviousDestinationListItems();
}


/**************************************************************************//**
*Constructor
******************************************************************************/
clSDS_PreviousDestinationsList::clSDS_PreviousDestinationsList(
   ::boost::shared_ptr< NavigationServiceProxy > ponaviProxy
   , NaviAddressFormatter* pNaviAddressFormatter
   , GuiService* pGuiService)
   : _naviProxy(ponaviProxy)
   , _pNaviAddressFormatter(pNaviAddressFormatter)
   , _pGuiService(pGuiService)
{
   _previousDestinationCount = 0;
   _previousDestinationPOIIndexValue = 0;
   _isLastDestinationListChanged = false;
   _isInitialRequest = true;
}


/***********************************************************************//**
*
***************************************************************************/
void clSDS_PreviousDestinationsList::onAvailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _naviProxy)
   {
      _naviProxy->sendIsLastDestinationListChangedRegister(*this);
   }
}


/***********************************************************************//**
*
***************************************************************************/
void clSDS_PreviousDestinationsList::onUnavailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _naviProxy)
   {
      _naviProxy->sendIsLastDestinationListChangedDeregisterAll();
   }
}


/**************************************************************************//**
*
******************************************************************************/

void clSDS_PreviousDestinationsList::onRequestDetailsForSelectedLocationError(
   const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationService::NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationService::RequestDetailsForSelectedLocationError >& /*error*/)
{
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_PreviousDestinationsList::onRequestDetailsForSelectedLocationResponse(
   const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationService::NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationService::RequestDetailsForSelectedLocationResponse >& /*response*/)
{
   if (_pGuiService)
   {
      _pGuiService->sendEventSignal(Event__SPEECH_DIALOG_GADGET_ACTIVATION);
   }
}


/**************************************************************************//**
*
******************************************************************************/
bool clSDS_PreviousDestinationsList::isUserWordAddress(const org::bosch::cm::navigation::NavigationService::DestinationMemoryDetailListElement& address)
{
   std::vector <std::string> currentCountryAddressFormat;
   std::string addressString = "";

   std::map < std::string, std::vector <std::string> >::iterator itr;

   for (itr = _countryAddressFormat.begin(); itr != _countryAddressFormat.end(); ++itr)
   {
      if ((itr->first) == _addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_COUNTRY_CODE])
      {
         currentCountryAddressFormat = itr->second;
      }
   }

   addressString = getFormattedAddress(currentCountryAddressFormat);

   if (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI].empty())
   {
      addressString.append(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI].c_str());
   }

   else if (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI_CATEGORY].empty())
   {
      addressString.append(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI_CATEGORY].c_str());
   }
   else if (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI_BRAND].empty())
   {
      addressString.append(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI_BRAND].c_str());
   }

   if (!addressString.compare(address.getEntry().getName()))
   {
      return false;
   }
   else
   {
      //userword
      return true;
   }
}


std::string clSDS_PreviousDestinationsList::getFormattedAddress(const std::vector <std::string> currentCountryAddressFormat)
{
   std::string formattedAddressString = "";

   for (size_t strIterator = 0; strIterator < currentCountryAddressFormat.size(); strIterator++)
   {
      if (currentCountryAddressFormat[strIterator] == "HNR" && (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_HOUSENO].empty()))
      {
         if (currentCountryAddressFormat[strIterator + 1] == "ROAD")
         {
            formattedAddressString.append(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_HOUSENO].c_str()).append(" ");
         }
         else
         {
            formattedAddressString.append(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_HOUSENO].c_str()).append(", ");
         }
      }
      if (currentCountryAddressFormat[strIterator] == "ROAD" && (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_STREET].empty()))
      {
         if (currentCountryAddressFormat[strIterator + 1] == "HNR")
         {
            formattedAddressString.append(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_STREET].c_str()).append(" ");
         }
         else
         {
            formattedAddressString.append(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_STREET].c_str()).append(", ");
         }
      }
      if (currentCountryAddressFormat[strIterator] == "CROSSROAD" && (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_JUNCTION].empty()))
      {
         formattedAddressString.append(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_JUNCTION].c_str());
      }
      if (currentCountryAddressFormat[strIterator] == "PLACE" && (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_CITY].empty()))
      {
         formattedAddressString.append(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_CITY].c_str()).append(", ");
      }
      if (currentCountryAddressFormat[strIterator] == "SUBCOUNTRY" && (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_STATE].empty()))
      {
         formattedAddressString.append(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_STATE].c_str()).append(", ");
      }
      if (currentCountryAddressFormat[strIterator] == "COUNTRY" && (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_COUNTRY].empty()))
      {
         formattedAddressString.append(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_COUNTRY].c_str());
      }
   }
   return formattedAddressString;
}


std::vector<sds2hmi_fi_tcl_HMIElementDescrptionList> clSDS_PreviousDestinationsList::getHmiListDescription()
{
   _countryAddressFormat = _pNaviAddressFormatter->getAddressFormat();
   std::vector<sds2hmi_fi_tcl_HMIElementDescrptionList> previousDestinationDetails;

   for (size_t i = 0; i < _previousDestinations.size(); ++i)
   {
      sds2hmi_fi_tcl_HMIElementDescription hmiElementDescription;
      sds2hmi_fi_tcl_HMIElementDescrptionList hmiElementDescriptionList;

      if (isUserWordAddress(_previousDestinations[i]))
      {
         hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_PLAIN_TEXT;
         hmiElementDescription.DescriptorValue.bSet(_previousDestinations[i].getEntry().getName().c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);

         hmiElementDescriptionList.DescriptionList.push_back(hmiElementDescription);
      }
      else
      {
         if (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI].empty())
         {
            hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_POINAME;
            hmiElementDescription.DescriptorValue.bSet(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI].c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
            hmiElementDescription.PhonemeList.push_back((sds2hmi_fi_tclString)_addressDetailsItemPhonemeList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI].c_str());

            hmiElementDescriptionList.DescriptionList.push_back(hmiElementDescription);
         }

         else if (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI_CATEGORY].empty())
         {
            hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_POICATEGORY;
            hmiElementDescription.DescriptorValue.bSet(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI_CATEGORY].c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
            hmiElementDescription.PhonemeList.push_back((sds2hmi_fi_tclString)_addressDetailsItemPhonemeList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI_CATEGORY].c_str());

            hmiElementDescriptionList.DescriptionList.push_back(hmiElementDescription);
         }

         else if (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI_BRAND].empty())
         {
            hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_POIBRAND;
            hmiElementDescription.DescriptorValue.bSet(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI_BRAND].c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
            hmiElementDescription.PhonemeList.push_back((sds2hmi_fi_tclString)_addressDetailsItemPhonemeList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI_BRAND].c_str());

            hmiElementDescriptionList.DescriptionList.push_back(hmiElementDescription);
         }
         else
         {
            std::map <std::string, std::vector <std::string> >::iterator itr;

            for (itr = _countryAddressFormat.begin(); itr != _countryAddressFormat.end(); ++itr)
            {
               if (itr->first == _addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_COUNTRY_CODE])
               {
                  std::vector <std::string>::iterator strIterator;

                  for (strIterator = itr->second.begin(); strIterator != itr->second.end(); ++strIterator)
                  {
                     if (*strIterator == "HNR" && (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_HOUSENO].empty()))
                     {
                        hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_HOUSENUM;
                        hmiElementDescription.DescriptorValue.bSet(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_HOUSENO].c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
                        hmiElementDescription.PhonemeList.push_back((sds2hmi_fi_tclString)_addressDetailsItemPhonemeList[DestinationMemoryAddressType__DESTINATION_MEMORY_HOUSENO].c_str());

                        hmiElementDescriptionList.DescriptionList.push_back(hmiElementDescription);
                     }

                     if (*strIterator == "ROAD" && (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_STREET].empty()))
                     {
                        hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_STREET;
                        hmiElementDescription.DescriptorValue.bSet(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_STREET].c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
                        hmiElementDescription.PhonemeList.push_back((sds2hmi_fi_tclString)_addressDetailsItemPhonemeList[DestinationMemoryAddressType__DESTINATION_MEMORY_STREET].c_str());

                        hmiElementDescriptionList.DescriptionList.push_back(hmiElementDescription);
                     }

                     if (*strIterator == "CROSSROAD" && (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_JUNCTION].empty()))
                     {
                        hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_STREET2;
                        hmiElementDescription.DescriptorValue.bSet(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_JUNCTION].c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
                        hmiElementDescription.PhonemeList.push_back((sds2hmi_fi_tclString)_addressDetailsItemPhonemeList[DestinationMemoryAddressType__DESTINATION_MEMORY_JUNCTION].c_str());

                        hmiElementDescriptionList.DescriptionList.push_back(hmiElementDescription);
                     }

                     if (*strIterator == "PLACE" && (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_CITY].empty()))
                     {
                        hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_CITY;
                        hmiElementDescription.DescriptorValue.bSet(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_CITY].c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
                        hmiElementDescription.PhonemeList.push_back((sds2hmi_fi_tclString)_addressDetailsItemPhonemeList[DestinationMemoryAddressType__DESTINATION_MEMORY_CITY].c_str());

                        hmiElementDescriptionList.DescriptionList.push_back(hmiElementDescription);
                     }

                     if (*strIterator == "SUBCOUNTRY" && (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_STATE].empty()))
                     {
                        hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_STATE;
                        hmiElementDescription.DescriptorValue.bSet(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_STATE].c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
                        hmiElementDescription.PhonemeList.push_back((sds2hmi_fi_tclString)_addressDetailsItemPhonemeList[DestinationMemoryAddressType__DESTINATION_MEMORY_STATE].c_str());

                        hmiElementDescriptionList.DescriptionList.push_back(hmiElementDescription);
                     }

                     if (*strIterator == "COUNTRY" && (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_COUNTRY].empty()))
                     {
                        hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_COUNTRY;
                        hmiElementDescription.DescriptorValue.bSet(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_COUNTRY].c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
                        hmiElementDescription.PhonemeList.push_back((sds2hmi_fi_tclString)_addressDetailsItemPhonemeList[DestinationMemoryAddressType__DESTINATION_MEMORY_COUNTRY].c_str());

                        hmiElementDescriptionList.DescriptionList.push_back(hmiElementDescription);
                     }
                  }
               }
            }
         }
      }
      previousDestinationDetails.push_back(hmiElementDescriptionList);
   }

   return previousDestinationDetails;
}


/**************************************************************************//**
*
******************************************************************************/
tBool clSDS_PreviousDestinationsList::bSelectElement(tU32 u32SelectedIndex)
{
   if (u32SelectedIndex > 0)
   {
      clSDS_StringVarList::vSetVariable("$(Address)", oGetItem(u32SelectedIndex - 1));
      oSetPreviousDestinationIndex(u32SelectedIndex - 1);
      sendDestinationMemoryEntryIdToNavi(u32SelectedIndex);
      return TRUE;
   }
   return FALSE;
}


/**************************************************************************//**
*
******************************************************************************/
tVoid clSDS_PreviousDestinationsList::vGetListInfo(sds2hmi_fi_tcl_e8_HMI_ListType::tenType listType)
{
   if (listType == sds2hmi_fi_tcl_e8_HMI_ListType::FI_EN_LIST_NAV_PREV_DESTINATIONS)
   {
      if (_isInitialRequest)
      {
         _naviProxy->sendRequestDestinationMemoryEntriesRequest(*this, DestinationMemoryCategoryType__LAST_DESTINATIONS);
         _isInitialRequest = false;
      }
      else if (_isLastDestinationListChanged)
      {
         _naviProxy->sendRequestDestinationMemoryEntriesRequest(*this, DestinationMemoryCategoryType__LAST_DESTINATIONS);
      }
      else
      {
         notifyListObserver();
      }
   }
   else
   {
      oSetPreviousDestinationListCount(0);
      notifyListObserver();
   }
}


/**************************************************************************//**
*
******************************************************************************/
tU32 clSDS_PreviousDestinationsList::u32GetSize()
{
   return _previousDestinationCount;
}


/**************************************************************************//**
*
******************************************************************************/
std::vector<clSDS_ListItems> clSDS_PreviousDestinationsList:: oGetItems(tU32 u32StartIndex, tU32 u32EndIndex)
{
   std::vector<clSDS_ListItems> oListItems;
   for (tU32 u32Index = u32StartIndex; u32Index < std::min(u32EndIndex, u32GetSize()); u32Index++)
   {
      clSDS_ListItems oListItem;
      oListItem.oDescription.szString = oGetItem(u32Index);
      oListItems.push_back(oListItem);
   }
   return oListItems;
}


/**************************************************************************//**
*
******************************************************************************/
void  clSDS_PreviousDestinationsList::onRequestDestinationMemoryEntriesError(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< RequestDestinationMemoryEntriesError >& /*error*/)
{
   oClearPreviousDestinationListItems();
   oSetPreviousDestinationListCount(0);
   notifyListObserver();
}


/**************************************************************************//**
*
******************************************************************************/
void  clSDS_PreviousDestinationsList::onRequestDestinationMemoryEntriesResponse(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< RequestDestinationMemoryEntriesResponse >& response)
{
   oClearPreviousDestinationListItems();
   if (!response->getDestinationMemoryEntries().empty())
   {
      _previousDestinations = response->getDestinationMemoryEntries();
      oSetPreviousDestinationListCount(response->getDestinationMemoryEntries().size());
   }
   else
   {
      oSetPreviousDestinationListCount(0);
   }
   notifyListObserver();
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_PreviousDestinationsList::oSetPreviousDestinationListCount(unsigned long previousDestinationList)
{
   _previousDestinationCount = previousDestinationList;
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_PreviousDestinationsList::oClearPreviousDestinationListItems()
{
   _previousDestinations.clear();
}


/**************************************************************************//**
*
******************************************************************************/
std::string clSDS_PreviousDestinationsList::oGetItem(tU32 u32Index)
{
   if (u32Index < _previousDestinations.size())
   {
      return _previousDestinations[u32Index].getEntry().getName();
   }
   else
   {
      return "";
   }
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_PreviousDestinationsList::oSetPreviousDestinationIndex(unsigned long previousDestinationPOI)
{
   _previousDestinationPOIIndexValue = previousDestinationPOI;
}


unsigned long clSDS_PreviousDestinationsList::vGetPreviousDestinationIndex() const
{
   return _previousDestinationPOIIndexValue;
}


/**************************************************************************//**
*
******************************************************************************/

void clSDS_PreviousDestinationsList::vStartGuidance()
{
   _naviProxy->sendStartGuidanceRequest(*this);
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_PreviousDestinationsList::onStartGuidanceError(const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
      const ::boost::shared_ptr< StartGuidanceError >& /*error*/)
{
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_PreviousDestinationsList::onStartGuidanceResponse(const ::boost::shared_ptr<NavigationServiceProxy >& /*proxy*/,
      const ::boost::shared_ptr< StartGuidanceResponse >& /*response*/)
{
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_PreviousDestinationsList::onIsLastDestinationListChangedError(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< IsLastDestinationListChangedError >& /*error*/)
{
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_PreviousDestinationsList::onIsLastDestinationListChangedUpdate(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< IsLastDestinationListChangedUpdate >& update)
{
   _isLastDestinationListChanged = update->getIsLastDestinationListChanged();
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_PreviousDestinationsList::onSetLocationWithDestinationMemoryEntryError(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< SetLocationWithDestinationMemoryEntryError >& /*error*/)
{
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_PreviousDestinationsList::onSetLocationWithDestinationMemoryEntryResponse(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< SetLocationWithDestinationMemoryEntryResponse >& /*response*/)
{
   _naviProxy->sendRequestDetailsForSelectedLocationRequest(*this);
}


/**************************************************************************//**
*
******************************************************************************/

void clSDS_PreviousDestinationsList::sendDestinationMemoryEntryIdToNavi(tU32 selectedIndex)
{
   if (selectedIndex > 0 && selectedIndex <= _previousDestinations.size())
   {
      _naviProxy->sendSetLocationWithDestinationMemoryEntryRequest(*this, (uint32)_previousDestinations[selectedIndex - 1].getEntry().getEntryID());
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/

std::vector<sds2hmi_fi_tcl_HMIElementDescription>  clSDS_PreviousDestinationsList::getHmiElementDescription(unsigned int index)
{
   sds2hmi_fi_tcl_HMIElementDescription hmiElementDescription;
   std::vector<sds2hmi_fi_tcl_HMIElementDescription> _hmiElementDescriptionList;
   _hmiElementDescriptionList.clear();
   if ((index <= _previousDestinations.size()) && (index > 0))
   {
      std::vector<DestinationMemoryEntryItem> destinationMemoryEntryItemList = _previousDestinations[index - 1].getDetails().getSdsDestinationMemoryEntries();
      setAddressDetails(destinationMemoryEntryItemList);

      if (isUserWordAddress(_previousDestinations[index - 1]))
      {
         hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_PLAIN_TEXT;
         hmiElementDescription.DescriptorValue.bSet(_previousDestinations[index - 1].getEntry().getName().c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);

         _hmiElementDescriptionList.push_back(hmiElementDescription);
      }
      else
      {
         if (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI].empty())
         {
            hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_POINAME;
            hmiElementDescription.DescriptorValue.bSet(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI].c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
            hmiElementDescription.PhonemeList.push_back((sds2hmi_fi_tclString)_addressDetailsItemPhonemeList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI].c_str());

            _hmiElementDescriptionList.push_back(hmiElementDescription);
         }

         else if (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI_CATEGORY].empty())
         {
            hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_POICATEGORY;
            hmiElementDescription.DescriptorValue.bSet(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI_CATEGORY].c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
            hmiElementDescription.PhonemeList.push_back((sds2hmi_fi_tclString)_addressDetailsItemPhonemeList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI_CATEGORY].c_str());

            _hmiElementDescriptionList.push_back(hmiElementDescription);
         }

         else if (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI_BRAND].empty())
         {
            hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_POIBRAND;
            hmiElementDescription.DescriptorValue.bSet(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI_BRAND].c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
            hmiElementDescription.PhonemeList.push_back((sds2hmi_fi_tclString)_addressDetailsItemPhonemeList[DestinationMemoryAddressType__DESTINATION_MEMORY_POI_BRAND].c_str());

            _hmiElementDescriptionList.push_back(hmiElementDescription);
         }
         else
         {
            std::map <std::string, std::vector <std::string> >::iterator itr;

            for (itr = _countryAddressFormat.begin(); itr != _countryAddressFormat.end(); ++itr)
            {
               if ((itr->first) == _addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_COUNTRY_CODE])
               {
                  std::vector <std::string>::iterator strIterator;

                  for (strIterator = itr->second.begin(); strIterator != itr->second.end(); ++strIterator)
                  {
                     if (*strIterator == "HNR" && (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_HOUSENO].empty()))
                     {
                        hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_HOUSENUM;
                        hmiElementDescription.DescriptorValue.bSet(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_HOUSENO].c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
                        hmiElementDescription.PhonemeList.push_back((sds2hmi_fi_tclString)_addressDetailsItemPhonemeList[DestinationMemoryAddressType__DESTINATION_MEMORY_HOUSENO].c_str());

                        _hmiElementDescriptionList.push_back(hmiElementDescription);
                     }

                     if (*strIterator == "ROAD" && (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_STREET].empty()))
                     {
                        hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_STREET;
                        hmiElementDescription.DescriptorValue.bSet(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_STREET].c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
                        hmiElementDescription.PhonemeList.push_back((sds2hmi_fi_tclString)_addressDetailsItemPhonemeList[DestinationMemoryAddressType__DESTINATION_MEMORY_STREET].c_str());

                        _hmiElementDescriptionList.push_back(hmiElementDescription);
                     }

                     if (*strIterator == "CROSSROAD" && (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_JUNCTION].empty()))
                     {
                        hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_STREET2;
                        hmiElementDescription.DescriptorValue.bSet(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_JUNCTION].c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
                        hmiElementDescription.PhonemeList.push_back((sds2hmi_fi_tclString)_addressDetailsItemPhonemeList[DestinationMemoryAddressType__DESTINATION_MEMORY_JUNCTION].c_str());

                        _hmiElementDescriptionList.push_back(hmiElementDescription);
                     }

                     if (*strIterator == "PLACE" && (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_CITY].empty()))
                     {
                        hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_CITY;
                        hmiElementDescription.DescriptorValue.bSet(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_CITY].c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
                        hmiElementDescription.PhonemeList.push_back((sds2hmi_fi_tclString)_addressDetailsItemPhonemeList[DestinationMemoryAddressType__DESTINATION_MEMORY_CITY].c_str());

                        _hmiElementDescriptionList.push_back(hmiElementDescription);
                     }

                     if (*strIterator == "SUBCOUNTRY" && (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_STATE].empty()))
                     {
                        hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_STATE;
                        hmiElementDescription.DescriptorValue.bSet(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_STATE].c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
                        hmiElementDescription.PhonemeList.push_back((sds2hmi_fi_tclString)_addressDetailsItemPhonemeList[DestinationMemoryAddressType__DESTINATION_MEMORY_STATE].c_str());

                        _hmiElementDescriptionList.push_back(hmiElementDescription);
                     }

                     if (*strIterator == "COUNTRY" && (!_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_COUNTRY].empty()))
                     {
                        hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_COUNTRY;
                        hmiElementDescription.DescriptorValue.bSet(_addressDetailsItemStringList[DestinationMemoryAddressType__DESTINATION_MEMORY_COUNTRY].c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
                        hmiElementDescription.PhonemeList.push_back((sds2hmi_fi_tclString)_addressDetailsItemPhonemeList[DestinationMemoryAddressType__DESTINATION_MEMORY_COUNTRY].c_str());

                        _hmiElementDescriptionList.push_back(hmiElementDescription);
                     }
                  }
               }
            }
         }
      }
   }
   return _hmiElementDescriptionList;
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_PreviousDestinationsList::setAddressDetails(const std::vector<DestinationMemoryEntryItem>& destinationMemoryEntryItemList)
{
   std::vector<DestinationMemoryEntryItem>::const_iterator entryIt;

   for (entryIt = destinationMemoryEntryItemList.begin(); entryIt != destinationMemoryEntryItemList.end(); ++entryIt)
   {
      _addressDetailsItemStringList[entryIt->getAddressType()] = entryIt->getItemString();
      _addressDetailsItemPhonemeList[entryIt->getAddressType()] = entryIt->getItemPhoneme();
   }
}
