/**************************************************************************//**
 * \file       clSDS_ParkingList.cpp
 *
 * clSDS_ParkingList class implementation
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/


#include "application/clSDS_ParkingList.h"
#include "application/clSDS_StringVarList.h"
#include "application/GuiService.h"
#include "application/StringUtils.h"


using namespace org::bosch::cm::navigation::NavigationService;
using namespace org::bosch::cm::navigation::NavigationSDSService;
using namespace sds_gui_fi::SdsGuiService;
using namespace sxm_main_fi_types;
using namespace sxm_parking_main_fi;
using namespace sxm_parking_main_fi_types;
using namespace VEHICLE_MAIN_FI;
using namespace vehicle_main_fi_types;


/**************************************************************************
*
***************************************************************************/
clSDS_ParkingList::~ clSDS_ParkingList()
{
   _pGuiService = NULL;
}


/**************************************************************************
*
***************************************************************************/
clSDS_ParkingList::clSDS_ParkingList(
   boost::shared_ptr< Sxm_parking_main_fiProxy > sxmParkingProxy,
   boost::shared_ptr< NavigationServiceProxy > naviProxy,
   boost::shared_ptr< NavigationSDSServiceProxy > naviSDSProxy,
   boost::shared_ptr< VEHICLE_MAIN_FIProxy > vehicleFiProxy,
   GuiService* pGuiService)
   : _sxmParkingProxy(sxmParkingProxy)
   , _naviProxy(naviProxy)
   , _naviSDSProxy(naviSDSProxy)
   , _vehicleFiProxy(vehicleFiProxy)
   , _pGuiService(pGuiService)
   , _vehicleDistanceUnits(DistanceDirectionUtility::EN_DISTANCE_MILES)
   , _startvehicleheading(0.0)
{
   _naviGeoPosition.setLatitude(0.0);
   _naviGeoPosition.setLongitude(0.0);
}


/**************************************************************************
*
***************************************************************************/
void clSDS_ParkingList::onAvailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _vehicleFiProxy)
   {
      _vehicleFiProxy->sendDistanceUnitUpReg(*this);
      _vehicleFiProxy->sendDistanceUnitGet(*this);
   }
   if (_naviSDSProxy == proxy)
   {
      _naviSDSProxy->sendPositionStatusInfoRegister(*this);
      _naviSDSProxy->sendPositionStatusInfoGet(*this);
   }
}


/**************************************************************************
*
***************************************************************************/
void clSDS_ParkingList::onUnavailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _vehicleFiProxy)
   {
      _vehicleFiProxy->sendDistanceUnitRelUpRegAll();
   }
   if (_naviSDSProxy == proxy)
   {
      _naviSDSProxy->sendPositionStatusInfoDeregisterAll();
   }
}


/**************************************************************************
*
***************************************************************************/
void clSDS_ParkingList::onDistanceUnitError(
   const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< DistanceUnitError >& /*error*/)
{
}


/**************************************************************************
*
***************************************************************************/
void clSDS_ParkingList::onDistanceUnitStatus(
   const ::boost::shared_ptr< VEHICLE_MAIN_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< DistanceUnitStatus >& status)
{
   if (status->hasE8DistanceUnit())
   {
      switch (status->getE8DistanceUnit())
      {
         case::T_e8_Vehicle_DistanceUnit__Unknown:
            _vehicleDistanceUnits = DistanceDirectionUtility::EN_DISTANCE_UNKNOWN;
            break;
         case::T_e8_Vehicle_DistanceUnit__Miles:
            _vehicleDistanceUnits = DistanceDirectionUtility::EN_DISTANCE_MILES;
            break;
         case::T_e8_Vehicle_DistanceUnit__Km:
            _vehicleDistanceUnits = DistanceDirectionUtility::EN_DISTANCE_KM;
            break;
         default:
            _vehicleDistanceUnits = DistanceDirectionUtility::EN_DISTANCE_MILES;
            break;
      }
   }
}


/**************************************************************************
*
***************************************************************************/
void clSDS_ParkingList::onSetLocationWithDetailsError(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< SetLocationWithDetailsError >& /*error*/)
{
}


/**************************************************************************
*
***************************************************************************/
void clSDS_ParkingList::onSetLocationWithDetailsResponse(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< SetLocationWithDetailsResponse >& /*response*/)
{
   _naviProxy->sendRequestDetailsForSelectedLocationRequest(*this);
}


/**************************************************************************
*
***************************************************************************/
void clSDS_ParkingList::onRequestDetailsForSelectedLocationError(
   const ::boost::shared_ptr<NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< RequestDetailsForSelectedLocationError >& /*error*/)
{
}


/**************************************************************************
*
***************************************************************************/
void clSDS_ParkingList::onRequestDetailsForSelectedLocationResponse(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< RequestDetailsForSelectedLocationResponse >& /*response*/)
{
   if (_pGuiService)
   {
      _pGuiService->sendEventSignal(Event__SPEECH_DIALOG_GADGET_ACTIVATION);
   }
}


/**************************************************************************
*
***************************************************************************/
void clSDS_ParkingList::onStartGuidanceError(
   const ::boost::shared_ptr<NavigationServiceProxy>& /*proxy*/,
   const ::boost::shared_ptr< StartGuidanceError >& /*error*/)
{
}


/**************************************************************************
*
***************************************************************************/
void clSDS_ParkingList::onStartGuidanceResponse(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< StartGuidanceResponse >& /*response*/)
{
}


/**************************************************************************
*
***************************************************************************/
void clSDS_ParkingList::onPositionStatusInfoError(
   const ::boost::shared_ptr< NavigationSDSServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< PositionStatusInfoError >& /*error*/)
{
}


/**************************************************************************
*
***************************************************************************/
void clSDS_ParkingList::onPositionStatusInfoUpdate(
   const ::boost::shared_ptr< NavigationSDSServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< PositionStatusInfoUpdate >& update)
{
   if (update->hasPositionStatusInfo())
   {
      //Store position information.
      _naviGeoPosition.setLatitude(update->getPositionStatusInfo().getPositionWGS84().getLatitude());
      _naviGeoPosition.setLongitude(update->getPositionStatusInfo().getPositionWGS84().getLongitude());
      _startvehicleheading = static_cast<double>(update->getPositionStatusInfo().getVehicleHeading());
   }
}


/**************************************************************************
*
***************************************************************************/
tU32 clSDS_ParkingList::u32GetSize()
{
   return static_cast<unsigned long>(_parkingListItems.size());
}


/**************************************************************************
 *
 ***************************************************************************/
tVoid clSDS_ParkingList::vGetListInfo(sds2hmi_fi_tcl_e8_HMI_ListType::tenType listType)
{
   if (listType == sds2hmi_fi_tcl_e8_HMI_ListType::FI_EN_LIST_PARKING)
   {
      unsigned int  listID = static_cast<unsigned int >(T_e8_SxmPredefinedListIds__NEAR_BY);
      T_SxmListMode lSxmListMode;
      _sxmParkingProxy ->sendGetParkingInfoListStart(*this, listID);
   }
   else
   {
      _parkingListItems.clear();
      notifyListObserver();
   }
}


/**************************************************************************
*
***************************************************************************/
void clSDS_ParkingList::onGetParkingInfoListError(
   const ::boost::shared_ptr< sxm_parking_main_fi::Sxm_parking_main_fiProxy >& /*proxy*/,
   const ::boost::shared_ptr< sxm_parking_main_fi::GetParkingInfoListError >& /*error*/)
{
   _parkingListItems.clear();
   notifyListObserver();
}


/**************************************************************************
*
***************************************************************************/
void clSDS_ParkingList:: onGetParkingInfoListResult(
   const ::boost::shared_ptr< sxm_parking_main_fi::Sxm_parking_main_fiProxy >& /*proxy*/,
   const ::boost::shared_ptr< sxm_parking_main_fi::GetParkingInfoListResult >& result)
{
   unsigned int  listID = static_cast<unsigned int >(T_e8_SxmPredefinedListIds__NEAR_BY);
   _parkingListItems.clear();

   if ((result->hasParkingInfoList() && result->getListID() == listID) && (result->getParkingInfoList().size()))
   {
      setParkingListItems(result->getParkingInfoList());
   }
   notifyListObserver();
}


/**************************************************************************
*
***************************************************************************/
void clSDS_ParkingList::setParkingListItems(const std::vector< T_ParkingInfo >& lParkingList)
{
   std::vector< T_ParkingInfo >::const_iterator itrParkingList;

   for (size_t i = 0 ; i < lParkingList.size(); ++i)
   {
      const T_ParkingInfo parkingInfo = lParkingList[i];
      int sxmLatitude = DistanceDirectionUtility::vConvertWGS84ToLatLong(parkingInfo.getPrimaryLocation().getLatitude());
      int sxmLongtitude = DistanceDirectionUtility::vConvertWGS84ToLatLong(parkingInfo.getPrimaryLocation().getLongitude());

      DistanceDirectionUtility::stGeoCoordinates lParkingGeoCoordinates;
      lParkingGeoCoordinates.dLatitude = DistanceDirectionUtility::vConvertLatitudeLongitudeTodouble(sxmLatitude);
      lParkingGeoCoordinates.dLongitude = DistanceDirectionUtility::vConvertLatitudeLongitudeTodouble(sxmLongtitude);
      lParkingGeoCoordinates.dVehicleHeading = 0.0;

      DistanceDirectionUtility::stGeoCoordinates lNaviGeoCoordinates;
      lNaviGeoCoordinates.dLatitude =  _naviGeoPosition.getLatitude();
      lNaviGeoCoordinates.dLongitude = _naviGeoPosition.getLongitude();
      lNaviGeoCoordinates.dVehicleHeading =  _startvehicleheading;

      stParkingListItems lParkingListItems;
      lParkingListItems.longitude = lParkingGeoCoordinates.dLongitude ;
      lParkingListItems.latitude =  lParkingGeoCoordinates.dLatitude ;
      lParkingListItems.parkingStationName = parkingInfo.getName();
      lParkingListItems.cityName = parkingInfo.getCity();
      lParkingListItems.stateName = parkingInfo.getState();
      lParkingListItems.distanceWithUnits = DistanceDirectionUtility::getFormattedDistance(parkingInfo.getDistance(), _vehicleDistanceUnits);
      lParkingListItems.direction = static_cast<int>(DistanceDirectionUtility::oGetDirection(lParkingGeoCoordinates, lNaviGeoCoordinates)) ;
      lParkingListItems.parkingStationAddress = parkingInfo.getAddress() /* street Address */
            + ", " + parkingInfo.getCity() + ", " + parkingInfo.getState() + ", " + parkingInfo.getZip();
      lParkingListItems.parkingStationPhoneNumber = parkingInfo.getPhone();
      _parkingListItems.push_back(lParkingListItems);
   }
}


/**************************************************************************
*
***************************************************************************/
std::vector<clSDS_ListItems> clSDS_ParkingList::oGetItems(tU32 u32StartIndex, tU32 u32EndIndex)
{
   std::vector<clSDS_ListItems> oListItems;

   for (unsigned long u32Index = u32StartIndex; u32Index < ::std::min(u32EndIndex, u32GetSize()); u32Index++)
   {
      oListItems.push_back(getListItem(u32Index));
   }
   return oListItems;
}


/**************************************************************************
*
***************************************************************************/
clSDS_ListItems clSDS_ParkingList::getListItem(tU32 u32Index) const
{
   clSDS_ListItems listItems;
   listItems.oDescription.szString = getParkingStationName(u32Index);
   listItems.oDistance.szString = getDistance(u32Index);
   listItems.oDirectionSymbol.szString = StringUtils::toString(static_cast<unsigned int>(getDirection(u32Index)));
   return listItems;
}


/**************************************************************************
*
***************************************************************************/
std::string clSDS_ParkingList::getParkingStationName(tU32 u32Index)const
{
   if (u32Index < _parkingListItems.size())
   {
      return _parkingListItems[u32Index].parkingStationName;
   }
   return "";
}


/**************************************************************************
*
***************************************************************************/
std::string clSDS_ParkingList::getDistance(tU32 u32Index)const
{
   if (u32Index < _parkingListItems.size())
   {
      return _parkingListItems[u32Index].distanceWithUnits;
   }
   return "";
}


/**************************************************************************
*
***************************************************************************/
int clSDS_ParkingList::getDirection(tU32 u32Index) const
{
   if (u32Index < _parkingListItems.size())
   {
      return _parkingListItems[u32Index].direction;
   }
   return 0;
}


/**************************************************************************
*
***************************************************************************/
std::string clSDS_ParkingList::getParkingStationAddress(tU32 u32Index)const
{
   if (u32Index < _parkingListItems.size())
   {
      return _parkingListItems[u32Index].parkingStationAddress;
   }
   return "";
}


/**************************************************************************
*
***************************************************************************/
tBool clSDS_ParkingList::bSelectElement(tU32 u32SelectedIndex)
{
   if ((u32SelectedIndex > 0) && (u32SelectedIndex <= _parkingListItems.size()))
   {
      clSDS_StringVarList::vSetVariable("$(POI)", getParkingStationName(u32SelectedIndex - 1));
      clSDS_StringVarList::vSetVariable("$(POI_address)", getParkingStationAddress(u32SelectedIndex - 1));
      sendParkingLocationDetailsToNavi(u32SelectedIndex - 1);
      return TRUE;
   }
   return FALSE;
}


/***********************************************************************//**

***************************************************************************/
std::vector<sds2hmi_fi_tcl_HMIElementDescription> clSDS_ParkingList::getHmiElementDescription(unsigned int index)
{
   std::vector<sds2hmi_fi_tcl_HMIElementDescription> hmiElementDescriptionList;
   sds2hmi_fi_tcl_HMIElementDescription hmiElementDescription;

   if ((index  > 0) && ((index - 1) < _parkingListItems.size()))
   {
      hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_POINAME;
      hmiElementDescription.DescriptorValue = _parkingListItems[index - 1].parkingStationName.c_str();
      hmiElementDescriptionList.push_back(hmiElementDescription);

      hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_CITY;
      hmiElementDescription.DescriptorValue = _parkingListItems[index - 1].cityName.c_str();
      hmiElementDescriptionList.push_back(hmiElementDescription);

      hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_STATE;
      hmiElementDescription.DescriptorValue = _parkingListItems[index - 1].stateName.c_str();
      hmiElementDescriptionList.push_back(hmiElementDescription);
   }
   return hmiElementDescriptionList;
}


/***********************************************************************//**

***************************************************************************/
void clSDS_ParkingList::sendParkingLocationDetailsToNavi(unsigned long index)
{
   if ((index < _parkingListItems.size()))
   {
      POILocation poiLocation;
      org::bosch::cm::navigation::NavigationService::PositionWGS84 position;
      position.setLatitude(_parkingListItems[index].latitude);
      position.setLongitude(_parkingListItems[index].longitude);
      Location location;
      location.setPositionWGS84(position);
      location.setName(_parkingListItems[index].parkingStationName);
      location.setValid(true);
      poiLocation.setLocation(location);
      poiLocation.setAddress(_parkingListItems[index].parkingStationAddress);
      poiLocation.setPhoneNumber(_parkingListItems[index].parkingStationPhoneNumber);
      _naviProxy->sendSetLocationWithDetailsRequest(*this, poiLocation);
   }
}


/***********************************************************************//**

***************************************************************************/
void clSDS_ParkingList::vStartGuidance()
{
   _naviProxy->sendStartGuidanceRequest(*this);
}
