/**************************************************************************//**
 * \file       clSDS_LanguageMediator.cpp
 *
 * Takes care that the SDS language is consistent with the current menu language.
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "application/clSDS_LanguageMediator.h"
#include "application/clSDS_MyAppsList.h"
#include "application/clSDS_SDSStatus.h"
#include "application/clSDS_SdsControl.h"
#include "application/clSDS_KDSConfiguration.h"
#include "application/StringUtils.h"
#include "view_db/Sds_TextDB.h"
#include "SdsAdapter_Trace.h"
#include <string>


#ifdef DP_DATAPOOL_ID
#define DP_S_IMPORT_INTERFACE_FI
#include "dp_hmi_02_if.h"
#include "dp_tclfc_sds_adapter_SpeechSettings.h"
#endif


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/clSDS_LanguageMediator.cpp.trc.h"
#endif


#define ARRAY_SIZE(array)     (sizeof (array) / sizeof (array)[0])

using namespace sds_gui_fi::SettingsService;

struct LanguageMapping
{
   sds2hmi_fi_tcl_e16_ISO639_3_SDSLanguageCode::tenType sdsLanguageCode;
   sds2hmi_fi_tcl_e16_ISOCountryCode::tenType sdsCountryCode;
   vehicle_main_fi_types::T_e8_Language_Code vehicleLanguage;
};


/**
 * LANG_FULL_SDS() describes a language for which we provide display strings
 */
#define LANG_FULL_SDS(iso_lang, iso_country, vehicle_lang) \
   { \
      sds2hmi_fi_tcl_e16_ISO639_3_SDSLanguageCode::FI_EN_ISO_639_3_##iso_lang, \
      sds2hmi_fi_tcl_e16_ISOCountryCode::FI_EN_ISO_ALPHA_3_##iso_country, \
      vehicle_main_fi_types::T_e8_Language_Code__##vehicle_lang \
   }

/**
 * LANG_TTS_ONLY() describes a language which will not support SDS and help popups
 */
#define LANG_TTS_ONLY(iso_lang, iso_country, vehicle_lang) \
   { \
      sds2hmi_fi_tcl_e16_ISO639_3_SDSLanguageCode::FI_EN_ISO_639_3_##iso_lang, \
      sds2hmi_fi_tcl_e16_ISOCountryCode::FI_EN_ISO_ALPHA_3_##iso_country, \
      vehicle_main_fi_types::T_e8_Language_Code__##vehicle_lang \
   }

/**
 * The language map is used to translate language identifiers from SDS to Vehicle-FI
 * and vice versa. The following specification determines which languages
 * support full SDS and which support only TTS:
 * https://hi-dms.de.bosch.com/docushare/dsweb/View/Collection-237585
 */
static const LanguageMapping g_languageMap[] =
{
   LANG_FULL_SDS(ENG, USA, English_US),
   LANG_FULL_SDS(ENG, USA, English_US_for_PRC), // TODO jnd2hi: unclear how to support this special US English
   LANG_FULL_SDS(ENG, USA, English_US_for_JPN), // TODO jnd2hi: unclear how to support this special US English
   LANG_FULL_SDS(FRA, CAN, French_Canadian),
   LANG_FULL_SDS(SPA, MEX, Spanish_Mexican),
   LANG_FULL_SDS(SPA, MEX, Spanish_Latin_American),
   LANG_FULL_SDS(DEU, DEU, German),
   LANG_FULL_SDS(ENG, GBR, English_UK),
   LANG_FULL_SDS(FRA, FRA, French),
   LANG_FULL_SDS(SPA, ESP, Spanish),
   LANG_FULL_SDS(NLD, NLD, Dutch),
   LANG_FULL_SDS(ITA, ITA, Italian),
   LANG_FULL_SDS(POR, PRT, Portuguese),
   LANG_FULL_SDS(RUS, RUS, Russian),
   LANG_FULL_SDS(ARA, SAU, Arabic),
   LANG_FULL_SDS(BUL, BGR, Bulgarian),
   LANG_FULL_SDS(YUE, HKG, Chinese_Cantonese_Traditional_Chinese_character),
   LANG_FULL_SDS(CMN, CHN, Chinese_Mandarin_Simplified_Chinese_character),
   LANG_FULL_SDS(CMN, CHN, Chinese_Cantonese_Simplified_Chinese_character), // TODO jnd2hi: this system language id is not to be used -> NCG3D-19556
   LANG_FULL_SDS(CES, CZE, Czech),
   LANG_FULL_SDS(DAN, DNK, Danish),
   LANG_FULL_SDS(ENG, AUS, English_Australian),
   LANG_FULL_SDS(ENG, IND, English_India),
   LANG_FULL_SDS(FIN, FIN, Finnish),
   LANG_FULL_SDS(ELL, GRC, Greek),
   LANG_FULL_SDS(JPN, JPN, Japanese),
   LANG_FULL_SDS(KOR, KOR, Korean),
   LANG_FULL_SDS(CMN, TWN, Taiwanese),
   LANG_FULL_SDS(NOR, NOR, Norwegian),
   LANG_FULL_SDS(POL, POL, Polish),
   LANG_FULL_SDS(POR, BRA, Portuguese_Brazilian),
   LANG_FULL_SDS(SWE, SWE, Swedish),
   LANG_FULL_SDS(TUR, TUR, Turkish),
   LANG_FULL_SDS(RUS, RUS, Ukrainian),

   LANG_TTS_ONLY(HEB, ISR, Hebrew),
   LANG_TTS_ONLY(HIN, IND, Hindi),
   LANG_TTS_ONLY(HUN, HUN, Hungarian),
   LANG_TTS_ONLY(IND, IDN, Indonesian),
   LANG_TTS_ONLY(RON, ROU, Romanian),
   LANG_TTS_ONLY(SLK, SVK, Slovakian),
   LANG_TTS_ONLY(THA, THA, Thai),
};


/**
 * Fallback language table
 */
struct FallbackMapping
{
   clSDS_KDSConfiguration::Country destinationRegion;
   sds2hmi_fi_tcl_e16_ISO639_3_SDSLanguageCode::tenType reqLang;
   sds2hmi_fi_tcl_e16_ISOCountryCode::tenType reqCountry;
   sds2hmi_fi_tcl_e16_ISO639_3_SDSLanguageCode::tenType fallbackLang;
   sds2hmi_fi_tcl_e16_ISOCountryCode::tenType fallbackCountry;
};


#define LANG_FALLBACK(destRegion, reqLang, reqCountry, fallbackLang, fallbackCountry) \
   { \
      clSDS_KDSConfiguration::destRegion, \
      sds2hmi_fi_tcl_e16_ISO639_3_SDSLanguageCode::FI_EN_ISO_639_3_##reqLang, \
      sds2hmi_fi_tcl_e16_ISOCountryCode::FI_EN_ISO_ALPHA_3_##reqCountry, \
      sds2hmi_fi_tcl_e16_ISO639_3_SDSLanguageCode::FI_EN_ISO_639_3_##fallbackLang, \
      sds2hmi_fi_tcl_e16_ISOCountryCode::FI_EN_ISO_ALPHA_3_##fallbackCountry, \
   },


static const FallbackMapping g_fallbackLanguages[] =
{
   LANG_FALLBACK(ASR_NZR, ENG, GBR,  ENG, AUS)
};


/**************************************************************************//**
*
******************************************************************************/
static sds2hmi_fi_tcl_SDSLanguageID convertVehicleToSdsLanguage(::vehicle_main_fi_types::T_e8_Language_Code vehicleLanguage)
{
   sds2hmi_fi_tcl_SDSLanguageID sdsLanguage;
   for (size_t i = 0; i < ARRAY_SIZE(g_languageMap); i++)
   {
      if (g_languageMap[i].vehicleLanguage == vehicleLanguage)
      {
         sdsLanguage.ISO639_3_SDSLanguageCode.enType = g_languageMap[i].sdsLanguageCode;
         sdsLanguage.ISO3166_CountryCode.enType = g_languageMap[i].sdsCountryCode;
         break;
      }
   }
   return sdsLanguage;
}


/**************************************************************************//**
*
******************************************************************************/
static vehicle_main_fi_types::T_e8_Language_Code convertSdsToVehicleLanguage(const sds2hmi_fi_tcl_SDSLanguageID& sdsLanguage)
{
   for (size_t i = 0; i < ARRAY_SIZE(g_languageMap); i++)
   {
      if ((g_languageMap[i].sdsLanguageCode == sdsLanguage.ISO639_3_SDSLanguageCode.enType) &&
            (g_languageMap[i].sdsCountryCode == sdsLanguage.ISO3166_CountryCode.enType))
      {
         return g_languageMap[i].vehicleLanguage;
      }
   }
   return ::vehicle_main_fi_types::T_e8_Language_Code__UnSupported;
}


/**************************************************************************//**
*
******************************************************************************/
static sds2hmi_fi_tcl_SDSLanguageID getFallbackLanguage(const sds2hmi_fi_tcl_SDSLanguageID& requestedLanguage)
{
   sds2hmi_fi_tcl_SDSLanguageID sdsLanguage;
   clSDS_KDSConfiguration::Country destinationRegion = clSDS_KDSConfiguration::getMarketRegionType();
   for (size_t i = 0; i < ARRAY_SIZE(g_fallbackLanguages); i++)
   {
      if ((g_fallbackLanguages[i].destinationRegion == destinationRegion) &&
            (g_fallbackLanguages[i].reqLang == requestedLanguage.ISO639_3_SDSLanguageCode.enType) &&
            (g_fallbackLanguages[i].reqCountry == requestedLanguage.ISO3166_CountryCode.enType))
      {
         sdsLanguage.ISO639_3_SDSLanguageCode.enType = g_fallbackLanguages[i].fallbackLang;
         sdsLanguage.ISO3166_CountryCode.enType = g_fallbackLanguages[i].fallbackCountry;
         break;
      }
   }
   return sdsLanguage;
}


/**************************************************************************//**
 * Destructor
 ******************************************************************************/
clSDS_LanguageMediator::~clSDS_LanguageMediator()
{
   _pSDSStatus = NULL;
   _pSdsControl = NULL;
}


/**************************************************************************//**
 * Constructor
 ******************************************************************************/
clSDS_LanguageMediator::clSDS_LanguageMediator(
   clSDS_SDSStatus* pSDSStatus,
   SettingsService& settingsService,
   ::boost::shared_ptr< ::VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >vehicleProxy)

   : _actionRequestAvailable(false)
   , _requestPending(false)
   , _u32RetryCounter(0)
   , _systemLanguage(vehicle_main_fi_types::T_e8_Language_Code__Unknown)
   , _activeSpeaker(0)
   , _pSDSStatus(pSDSStatus)
   , _settingsService(settingsService)
   , _vehicleProxy(vehicleProxy)
   , _pSdsControl(NULL)
   , _isSDSSupportLanguage(false)
{
   _pSDSStatus->vRegisterObserver(this);
   _settingsService.addCommonSettingsObserver(this);

   _preferredGender = getSdsGender(_settingsService.getVoicePreference());
}


/**************************************************************************//**
* Request SDS to set a new language.
******************************************************************************/
tVoid clSDS_LanguageMediator::sendSpeakerRequest(tU16 speakerId)
{
   _requestPending = true;
   if (_pSdsControl)
   {
      _pSdsControl->vSetSpeaker(speakerId);
   }
   _u32RetryCounter++;
}


/**************************************************************************//**
* Returns true as long as SDS is operational. Returns false when SDS is in
* an IDLE/ERROR state. Note, that a speaker request must only be sent when
* SDS is in IDLE/ERROR.
******************************************************************************/
bool clSDS_LanguageMediator::sdsIsBusy() const
{
   return (!_pSDSStatus->bIsIdle() && !_pSDSStatus->bIsError() && !_pSDSStatus->bIsTTSOnlySupport());
}


/**************************************************************************//**
*
******************************************************************************/
tVoid clSDS_LanguageMediator::vSynchroniseSdsLanguage()
{
   if (_requestPending)
   {
      ETG_TRACE_COMP(("LanguageMediator: request pending"));
      return;
   }

   sds2hmi_fi_tcl_SDSLanguageID systemLanguage = convertVehicleToSdsLanguage(_systemLanguage);
   tU16 requiredSpeaker = getSpeakerIdForLanguage(systemLanguage, _preferredGender);
   if (requiredSpeaker == 0)
   {
      systemLanguage = getFallbackLanguage(systemLanguage);
      requiredSpeaker = getSpeakerIdForLanguage(systemLanguage, _preferredGender);
   }

   if (requiredSpeaker == 0)
   {
      ETG_TRACE_COMP(("LanguageMediator: speaker not available"));
      _isSDSSupportLanguage = false;
      return;
   }

   _isSDSSupportLanguage = true;
   ETG_TRACE_COMP(("LanguageMediator: requiredSpeaker:%d", requiredSpeaker));

   ETG_TRACE_COMP(("LanguageMediator: _activeSpeaker:%d", _activeSpeaker));

   if (requiredSpeaker == _activeSpeaker)
   {
      Sds_TextDB_vSetLanguage(StringUtils::toISOString(systemLanguage.ISO639_3_SDSLanguageCode.enType), StringUtils::toISOString(systemLanguage.ISO3166_CountryCode.enType));
      updateSdsLanguageInVehicleData();
      ETG_TRACE_COMP(("LanguageMediator: languages are synchronized"));
      return;
   }

   if (bTooManyRetries())
   {
      ETG_TRACE_COMP(("LanguageMediator: too many retries"));
      return;
   }

   if (sdsIsBusy())
   {
      ETG_TRACE_COMP(("LanguageMediator: SDS not ready to set language"));
      return;
   }

   if (!_actionRequestAvailable)
   {
      ETG_TRACE_COMP(("LanguageMediator: property CommonActionRequest not yet registered"));
      return;
   }

   sendSpeakerRequest(requiredSpeaker);
   ETG_TRACE_COMP(("LanguageMediator: SDS language change triggered"));
}


/**************************************************************************//**
*
******************************************************************************/
tVoid clSDS_LanguageMediator::updateSdsLanguageInVehicleData()
{
   if (_vehicleProxy->isAvailable() && _vehicleProxy->hasLanguage())
   {
      const ::std::vector< ::vehicle_main_fi_types::T_Language_SourceTable >& sourceTable = _vehicleProxy->getLanguage().getLangTable();
      ::std::vector< ::vehicle_main_fi_types::T_Language_SourceTable >::const_iterator iter = sourceTable.begin();
      ::vehicle_main_fi_types::T_e8_Language_Code sdsLanguage = convertSdsToVehicleLanguage(getLanguageOfActiveSpeaker());
      while (iter != sourceTable.end())
      {
         if ((iter->getEnLangSrcId() == ::vehicle_main_fi_types::T_e8_Language_SourceId__SDS) &&
               (iter->getEnLanguage() != sdsLanguage))
         {
            _vehicleProxy->sendSetLanguageStart(*this, ::vehicle_main_fi_types::T_e8_Language_SourceId__SDS, sdsLanguage);
            break;
         }
         ++iter;
      }
   }
}


/**************************************************************************//**
* Returns TRUE if retries are greater than 3.Otherwise FALSE.
******************************************************************************/
tBool clSDS_LanguageMediator::bTooManyRetries() const
{
   return (_u32RetryCounter >= 3);
}


/**************************************************************************//**
* Synchronising the SDS Language happens when SDS state changes
******************************************************************************/
tVoid clSDS_LanguageMediator::vSDSStatusChanged()
{
   if (sdsIsBusy())
   {
      // unblock subsequent requests as soon as SDS has left any of the IDLE/ERROR states
      _requestPending = false;
   }
   vSynchroniseSdsLanguage();
}


/**************************************************************************//**
* Sets the active speaker as indicated by property SDS_ActiveSpeaker.
* An id of value 0 indicates an invalid speaker id.
******************************************************************************/
tVoid clSDS_LanguageMediator::setActiveSpeaker(tU16 activeSpeaker)
{
   _requestPending = false;

   _activeSpeaker = activeSpeaker;
   if (activeSpeaker != 0)
   {
      _u32RetryCounter = 0;
   }
   vSynchroniseSdsLanguage();
}


/**************************************************************************//**
* Stores the available Languages supported by SDS
******************************************************************************/
tVoid clSDS_LanguageMediator::setAvailableSpeakers(const std::vector<sds2hmi_fi_tcl_LanguageAndSpeaker>& availableSpeakers)
{
   _availableSpeakers = availableSpeakers;
   _u32RetryCounter = 0;
   _settingsService.updateAvailableSpeakers(availableSpeakers);
   vSynchroniseSdsLanguage();
}


/**************************************************************************//**
*
******************************************************************************/
tVoid clSDS_LanguageMediator::setActionRequestAvailable(tBool isAvailable)
{
   _actionRequestAvailable = isAvailable;
   vSynchroniseSdsLanguage();
}


/**************************************************************************//**
* Gets Language ID for selected menu Language
******************************************************************************/
sds2hmi_fi_tcl_SDSLanguageID clSDS_LanguageMediator::getLanguageOfActiveSpeaker() const
{
   for (tU32 u32Counter = 0; u32Counter < _availableSpeakers.size(); u32Counter++)
   {
      if (_availableSpeakers[u32Counter].SpeakerId == _activeSpeaker)
      {
         return _availableSpeakers[u32Counter].LanguageID;
      }
   }
   sds2hmi_fi_tcl_SDSLanguageID oLanguage;
   return oLanguage;
}


/**************************************************************************//**
*
******************************************************************************/
tU16 clSDS_LanguageMediator::getSpeakerIdForLanguage(
   const sds2hmi_fi_tcl_SDSLanguageID& language,
   sds2hmi_fi_tcl_e8_Gender gender)
{
   // first try to find speaker with preferred gender
   for (size_t i = 0; i < _availableSpeakers.size(); i++)
   {
      if ((_availableSpeakers[i].LanguageID == language) &&
            (_availableSpeakers[i].Gender == gender))
      {
         return _availableSpeakers[i].SpeakerId;
      }
   }

   // find speaker disregarding preferred speaker
   for (size_t i = 0; i < _availableSpeakers.size(); i++)
   {
      if (_availableSpeakers[i].LanguageID == language)
      {
         return _availableSpeakers[i].SpeakerId;
      }
   }

   // TODO jnd2hi: determine fall-back language here and try to find speaker
   return 0;
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_LanguageMediator::onAvailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (_vehicleProxy == proxy)
   {
      _vehicleProxy ->sendLanguageUpReg(*this);
   }
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_LanguageMediator::onUnavailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (_vehicleProxy == proxy)
   {
      _vehicleProxy->sendLanguageRelUpRegAll();
   }
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_LanguageMediator::onLanguageError(
   const ::boost::shared_ptr< VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< VEHICLE_MAIN_FI::LanguageError >& /*error*/)
{
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_LanguageMediator::onLanguageStatus(
   const ::boost::shared_ptr< VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< VEHICLE_MAIN_FI::LanguageStatus >& status)
{
   _systemLanguage = status->getLanguage();
   _u32RetryCounter = 0;
   writeVehicleLanguageInDataPool(static_cast<char>(_systemLanguage));
   vSynchroniseSdsLanguage();
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_LanguageMediator::onSetLanguageError(
   const ::boost::shared_ptr< VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< VEHICLE_MAIN_FI::SetLanguageError >& /*error*/)
{
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_LanguageMediator::onSetLanguageResult(
   const ::boost::shared_ptr< VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< VEHICLE_MAIN_FI::SetLanguageResult >& /*result*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_LanguageMediator::vSetSDSControl(clSDS_SdsControl* pSdsControl)
{
   _pSdsControl = pSdsControl;
}


/**************************************************************************//**
 *
 ******************************************************************************/
sds2hmi_fi_tcl_e8_Gender clSDS_LanguageMediator::getSdsGender(sds_gui_fi::SettingsService::ActiveSpeakerGender guiGender) const
{
   sds2hmi_fi_tcl_e8_Gender gender;

   switch (guiGender)
   {
      case sds_gui_fi::SettingsService::ActiveSpeakerGender__FEMALE:
         gender.enType = sds2hmi_fi_tcl_e8_Gender::FI_EN_FEMALE;
         break;

      case sds_gui_fi::SettingsService::ActiveSpeakerGender__MALE:
      default:
         gender.enType = sds2hmi_fi_tcl_e8_Gender::FI_EN_MALE;
         break;
   }

   return gender;
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_LanguageMediator::voicePreferenceChanged()
{
   _preferredGender = getSdsGender(_settingsService.getVoicePreference());
   vSynchroniseSdsLanguage();
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool clSDS_LanguageMediator::bIsSDSLanguageSupport()
{
   return _isSDSSupportLanguage;
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_LanguageMediator::writeVehicleLanguageInDataPool(unsigned char systemLanguage)
{
   dp_tclfc_sds_adapter_SpeechSettingsSystemLanguage dpSytemLanguage;
   dpSytemLanguage.s32SetData(systemLanguage);
}
