/**************************************************************************//**
 * \file       clSDS_KDSConfiguration.cpp
 *
 * This file is part of the SdsAdapter component.
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/

#include "application/clSDS_KDSConfiguration.h"
#include "application/StringUtils.h"
#include "SdsAdapter_Trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/clSDS_KDSConfiguration.cpp.trc.h"
#endif

#define DP_S_IMPORT_INTERFACE_FI
#include "dp_generic_if.h"


clSDS_KDSConfiguration::clSDS_KDSConfiguration()
{
}


clSDS_KDSConfiguration::~clSDS_KDSConfiguration()
{
}


unsigned char clSDS_KDSConfiguration::getKDSConfigGeneric(const std::string& strConfigElement, const std::string& strConfigItem, unsigned char configValue)
{
   if (DP_s32GetConfigItem(strConfigElement.c_str(), strConfigItem.c_str(), &configValue, 1) != DP_S32_NO_ERR)
   {
      std::string trace = "KDS ReadError " + strConfigElement + "::" + strConfigItem;
      ETG_TRACE_ERR(("%s", trace.c_str()));
   }
   return configValue;
}


clSDS_KDSConfiguration::Country clSDS_KDSConfiguration::getMarketRegionType()
{
   return (Country)getKDSConfigGeneric("VehicleInformation", "DestinationRegion1", USA);
}


unsigned char clSDS_KDSConfiguration::getDABMounted()
{
   return getKDSConfigGeneric("CMVariantCoding", "DAB");
}


unsigned char clSDS_KDSConfiguration::getDABEnabled()
{
   // default to 'enabled' in case the config item is not available
   return getKDSConfigGeneric("SystemConfiguration1", "DAB", 1);
}


unsigned char clSDS_KDSConfiguration::getMaintenanceInfo()
{
   return getKDSConfigGeneric("SystemConfiguration1", "MaintenanceInfoFunction");
}


unsigned char clSDS_KDSConfiguration::getFuelConsumptionInfo()
{
   return (AllianceFuelConsumptionType)getKDSConfigGeneric("VehicleInformation", "FuelConsumptionInfo");
}


unsigned char clSDS_KDSConfiguration::getXMTunerMounted()
{
   return getKDSConfigGeneric("CMVariantCoding", "XMTuner");
}


unsigned char clSDS_KDSConfiguration::getDTVMounted()
{
   return getKDSConfigGeneric("CMVariantCoding", "DTV");
}


unsigned char clSDS_KDSConfiguration::getCDMounted()
{
   return getKDSConfigGeneric("CMVariantCoding", "CD");
}


unsigned char clSDS_KDSConfiguration::getDVDMounted()
{
   return getKDSConfigGeneric("CMVariantCoding", "DVD");
}


unsigned char clSDS_KDSConfiguration::getHEVInformationFunction()
{
   return getKDSConfigGeneric("SystemConfiguration1", "HEVInformationFunction");
}


unsigned char clSDS_KDSConfiguration::getBluetoothPhoneEnabled()
{
   return getKDSConfigGeneric("SystemConfiguration1", "BluetoothFunction");
}


clSDS_KDSConfiguration::AllianceVoiceRecognitionModeType clSDS_KDSConfiguration::getAllianceVRModeType()
{
   return (AllianceVoiceRecognitionModeType)getKDSConfigGeneric("SystemConfiguration1", "VoiceRecognition", EXPERT);
}


clSDS_KDSConfiguration::AllianceVehicleFleetType clSDS_KDSConfiguration::getAllianceVehicleFleetType()
{
   return (AllianceVehicleFleetType)getKDSConfigGeneric("SystemConfiguration1", "VehicleFleet", TAXI);
}


unsigned char clSDS_KDSConfiguration::getVRBeep()
{
   return getKDSConfigGeneric("SystemConfiguration1", "VRBeepOnly");
}


unsigned char clSDS_KDSConfiguration::getVRShortPrompts()
{
   return getKDSConfigGeneric("SystemConfiguration1", "VRShortPrompts");
}


clSDS_KDSConfiguration::FuelType clSDS_KDSConfiguration:: getVehicleEngineType()
{
   return (FuelType)getKDSConfigGeneric("VehicleInformation", "FuelType");
}


clSDS_KDSConfiguration::VehicleDistanceUnitType clSDS_KDSConfiguration::getVehicleDistanceUnitsType()
{
   return (VehicleDistanceUnitType)getKDSConfigGeneric("SystemConfiguration1", "SupportedDistanceUnits", EN_DIST_RESERVED);
}


clSDS_KDSConfiguration::VariantSelectionType clSDS_KDSConfiguration::getVariantSelectionType()
{
   return (VariantSelectionType)getKDSConfigGeneric("AIVIVariantCoding", "VariantSelection");
}


unsigned char clSDS_KDSConfiguration::getRDSAudioTuner1()
{
   return getKDSConfigGeneric("AMFMTunerParameter", "RDSAudioTuner1");
}


unsigned char clSDS_KDSConfiguration::getPIIgnore()
{
   return getKDSConfigGeneric("AMFMTunerParameter", "PIIgnore");
}


clSDS_KDSConfiguration::OEMType clSDS_KDSConfiguration::getVariantType()
{
   return (OEMType)getKDSConfigGeneric("CMVariantCoding", "OEMType");
}


clSDS_KDSConfiguration::DisplayOrientation clSDS_KDSConfiguration::getDisplayOrientation()
{
   return (DisplayOrientation)getKDSConfigGeneric("AIVIVariantCoding", "DisplayOrientation");
}


clSDS_KDSConfiguration::AutoBargeInState clSDS_KDSConfiguration::getAutoBargeIn()
{
   return (AutoBargeInState)getKDSConfigGeneric("SystemConfiguration1", "AutoBargeIn");
}


clSDS_KDSConfiguration::SystemTypeSelector clSDS_KDSConfiguration::getSystemTypeSelector()
{
   return (SystemTypeSelector)getKDSConfigGeneric("AIVIVariantCoding", "SystemTypeSelector");
}


clSDS_KDSConfiguration:: SystemVariant clSDS_KDSConfiguration::getSystemVariant()
{
   SystemTypeSelector enSystemType = getSystemTypeSelector();
   VariantSelectionType enVariantType = getVariantSelectionType();

   if (enVariantType == EN_VARIANT_SELECTION_TYPE_NAVIGATION && enSystemType == EN_SYSTEM_TYPE_SELECTOR_ENTRY)
   {
      return EN_ENTRY_NAVIGATION;
   }
   else if (enVariantType == EN_VARIANT_SELECTION_TYPE_NAVIGATION && enSystemType == EN_SYSTEM_TYPE_SELECTOR_MID)
   {
      return EN_MID_NAVIGATION;
   }
   else if (enVariantType == EN_VARIANT_SELECTION_TYPE_DISPLAY_AUDIO && enSystemType == EN_SYSTEM_TYPE_SELECTOR_ENTRY)
   {
      return EN_ENTRY_DISPLAY_AUDIO;
   }
   else if (enVariantType == EN_VARIANT_SELECTION_TYPE_DISPLAY_AUDIO && enSystemType == EN_SYSTEM_TYPE_SELECTOR_MID)
   {
      return EN_MID_DISPLAY_AUDIO;
   }
   else if (enVariantType == EN_VARIANT_SELECTION_TYPE_NAVIGATION && enSystemType == EN_SYSTEM_TYPE_SELECTOR_GENERIC)
   {
      return EN_GENERIC_NAVIGATION;
   }
   else if (enVariantType == EN_VARIANT_SELECTION_TYPE_NAVIGATION && enSystemType == EN_SYSTEM_TYPE_SELECTOR_EVOBUS)
   {
      return EN_EVOBUS_NAVIGATION;
   }
   return EN_NOT_CONFIGURED;
}


unsigned char clSDS_KDSConfiguration::getDefaultLanguage()
{
   return getKDSConfigGeneric("SystemConfiguration1", "DefaultLanguage");
}


unsigned char clSDS_KDSConfiguration::getSIRI()
{
   unsigned char siriEnable = 0;
   dp_tclKdsSystemConfiguration1 kdsSystemConfig1;
   if (kdsSystemConfig1.u8GetBTVR(siriEnable) != DP_U8_ELEM_STATUS_VALID)
   {
      ETG_TRACE_COMP(("clSDS_KDSConfiguration: KDS ReadError SIRI"));
   }
   return siriEnable;
}


unsigned char clSDS_KDSConfiguration::getVoiceAssistant()
{
   unsigned char voiceAssistant = 0;
   dp_tclKdsSystemConfiguration1 kdsSystemConfig1;
   if (kdsSystemConfig1.u8GetVA(voiceAssistant) != DP_U8_ELEM_STATUS_VALID)
   {
      ETG_TRACE_COMP(("clSDS_KDSConfiguration: KDS ReadError VoiceAssistant"));
   }
   return voiceAssistant;
}


unsigned char clSDS_KDSConfiguration::getCarplay()
{
   unsigned char carplay = 0;
   dp_tclKdsSystemConfiguration1 kdsSystemConfig1;
   if (kdsSystemConfig1.u8GetCarPlay(carplay) != DP_U8_ELEM_STATUS_VALID)
   {
      ETG_TRACE_COMP(("clSDS_KDSConfiguration: KDS ReadError Carplay"));
   }
   return carplay;
}


unsigned char clSDS_KDSConfiguration::getAndroidAuto()
{
   unsigned char androidauto = 0;
   dp_tclKdsSystemConfiguration1 kdsSystemConfig1;
   if (kdsSystemConfig1.u8GetAndroidAuto(androidauto) != DP_U8_ELEM_STATUS_VALID)
   {
      ETG_TRACE_COMP(("clSDS_KDSConfiguration: KDS ReadError AndroidAuto"));
   }
   return androidauto;
}
