/**************************************************************************//**
 * \file       clSDS_FmChannelList.cpp
 *
 * This file is part of the SdsAdapter component.
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "clSDS_FmChannelList.h"
#include "application/clSDS_TunerStateObserver.h"
#include "application/StringUtils.h"
#include "application/clSDS_KDSConfiguration.h"
#include <ctime>

#ifdef DP_DATAPOOL_ID
#define DP_S_IMPORT_INTERFACE_FI
#include "dp_generic_if.h"
#endif

#include "SdsAdapter_Trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/clSDS_FmChannelList.cpp.trc.h"
#endif


using namespace tuner_main_fi;
using namespace tuner_main_fi_types;


/**************************************************************************//**
 *
 ******************************************************************************/
clSDS_FmChannelList::~clSDS_FmChannelList()
{
   _pTunerStateObserver = NULL;
}


/**************************************************************************//**
 *
 ******************************************************************************/
clSDS_FmChannelList::clSDS_FmChannelList(
   ::boost::shared_ptr< sds_fm_fi::SdsFmService::SdsFmServiceProxy > sdsFmProxy,
   ::boost::shared_ptr<Tuner_main_fiProxy > tunerProxy)
   : _sdsFmProxy(sdsFmProxy)
   , _tunerProxy(tunerProxy)
   , _pTunerStateObserver(NULL)

{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_FmChannelList::onAvailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _tunerProxy)
   {
      ETG_TRACE_USR1(("clSDS_FmChannelList::onAvailable for port %s", proxy->getPortName().c_str()));
      _tunerProxy->sendFID_TUN_G_SURVIELLANCE_LISTUPDATEUpReg(*this);
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_FmChannelList::onUnavailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _tunerProxy)
   {
      _tunerProxy->sendFID_TUN_G_SURVIELLANCE_LISTUPDATERelUpRegAll();
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_FmChannelList::onStoreRDSChannelNamesError(
   const ::boost::shared_ptr< sds_fm_fi::SdsFmService::SdsFmServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< sds_fm_fi::SdsFmService::StoreRDSChannelNamesError >& /*error*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_FmChannelList::onStoreRDSChannelNamesResponse(
   const ::boost::shared_ptr< sds_fm_fi::SdsFmService::SdsFmServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< sds_fm_fi::SdsFmService::StoreRDSChannelNamesResponse >& /*response*/)
{
   // Notify SDS MW on tuner status as READY when DB updating is completed
   if (_pTunerStateObserver != NULL)
   {
      _pTunerStateObserver->vTunerStatusChanged(sds2hmi_fi_tcl_e8_TUN_Band::FI_EN_RDS_FM, sds2hmi_fi_tcl_e8_DeviceStatus::FI_EN_DEVICE_READY);
   }
   ETG_TRACE_USR1(("clSDS_FmChannelList::onStoreRDSChannelNamesResponse : tuner status ready"));
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_FmChannelList::onStoreHDChannelNamesError(
   const ::boost::shared_ptr< sds_fm_fi::SdsFmService::SdsFmServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< sds_fm_fi::SdsFmService::StoreHDChannelNamesError >& /*error*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_FmChannelList::onStoreHDChannelNamesResponse(
   const ::boost::shared_ptr< sds_fm_fi::SdsFmService::SdsFmServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< sds_fm_fi::SdsFmService::StoreHDChannelNamesResponse >& /*response*/)
{
   // Notify SDS MW on tuner status as READY when DB updating is completed
   if (_pTunerStateObserver != NULL)
   {
      _pTunerStateObserver->vTunerStatusChanged(sds2hmi_fi_tcl_e8_TUN_Band::FI_EN_HD_FM, sds2hmi_fi_tcl_e8_DeviceStatus::FI_EN_DEVICE_READY);
   }
   ETG_TRACE_USR1(("clSDS_FmChannelList::onStoreHDChannelNamesResponse : tuner status ready"));
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_FmChannelList::setFmChannelListObserver(clSDS_TunerStateObserver* pTunerStateObserver)
{
   _pTunerStateObserver = pTunerStateObserver;
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_FmChannelList::onFID_TUN_G_SURVIELLANCE_LISTUPDATEStatus(
   const ::boost::shared_ptr< tuner_main_fi::Tuner_main_fiProxy >& /*proxy*/,
   const ::boost::shared_ptr< tuner_main_fi::FID_TUN_G_SURVIELLANCE_LISTUPDATEStatus >& status)
{
   sds_fm_fi::SdsFmService::FMChannelItem storeRDSFmChannelElement;
   sds_fm_fi::SdsFmService::FMChannelItem storeFmHDChannelElement;
   const T_TunerConfigListElementList& resultConfigListElements = status->getTun_ConfigListElements();
   if (resultConfigListElements.size())
   {
      _fmChannelListHD.clear();
      _fmChannelListRDS.clear();
      std::string stationName = "";

      // **************************************************** for RDS stations  ****************************************************************//

      // if (clSDS_KDSConfiguration::getMarketRegionType() == clSDS_KDSConfiguration::OTHER_EUR)
      //{
      for (::std::vector< T_TunerConfigListElement >::const_iterator itr = resultConfigListElements.begin(); itr != resultConfigListElements.end(); ++itr)
      {
         if ((itr->getSPSName().size()) && (itr->getE8ConfigListID() == T_e8_Tun_ConfigListID__TUN_TUN_CONFIG_LIST_FM))
         {
            if (((itr->getE8ModulationType() == T_e8_Tun_ModulationType__TUN_MODULATION_ANALOG) || (itr->getE8ModulationType() == T_e8_Tun_ModulationType__TUN_MODULATION_DRM)))
            {
               storeRDSFmChannelElement.setObjectID(itr->getU32PI());
               storeRDSFmChannelElement.setStationName(itr->getSPSName());
               storeRDSFmChannelElement.setFrequency(itr->getU32Frequency());
               _fmChannelListRDS.push_back(storeRDSFmChannelElement);
            }
            else if ((itr->getE8ModulationType() == T_e8_Tun_ModulationType__TUN_MODULATION_HD))
            {
               stationName = itr->getSPSName();
               storeFmHDChannelElement.setFrequency(itr->getU32Frequency());
               if (itr->getB8HDNumOfAudioPrograms().getTUN_HD_AUDIOPROGRAM_MPS())
               {
                  stationName.append(" HD1");
                  storeFmHDChannelElement.setAudioProgram(T_e8_Tun_HDAudioProgram__TUN_AUDIO_PROGRAM_ONE);
               }
               else if (itr->getB8HDNumOfAudioPrograms().getTUN_HD_AUDIOPROGRAM_SPS1())
               {
                  stationName.append(" HD2");
                  storeFmHDChannelElement.setAudioProgram(T_e8_Tun_HDAudioProgram__TUN_AUDIO_PROGRAM_TWO);
               }
               else if (itr->getB8HDNumOfAudioPrograms().getTUN_HD_AUDIOPROGRAM_SPS2())
               {
                  stationName.append(" HD3");
                  storeFmHDChannelElement.setAudioProgram(T_e8_Tun_HDAudioProgram__TUN_AUDIO_PROGRAM_THREE);
               }
               else if (itr->getB8HDNumOfAudioPrograms().getTUN_HD_AUDIOPROGRAM_SPS3())
               {
                  stationName.append(" HD4");
                  storeFmHDChannelElement.setAudioProgram(T_e8_Tun_HDAudioProgram__TUN_AUDIO_PROGRAM_FOUR);
               }
               else if (itr->getB8HDNumOfAudioPrograms().getTUN_HD_AUDIOPROGRAM_SPS4())
               {
                  stationName.append(" HD5");
                  storeFmHDChannelElement.setAudioProgram(T_e8_Tun_HDAudioProgram__TUN_AUDIO_PROGRAM_FIVE);
               }
               else if (itr->getB8HDNumOfAudioPrograms().getTUN_HD_AUDIOPROGRAM_SPS5())
               {
                  stationName.append(" HD6");
                  storeFmHDChannelElement.setAudioProgram(T_e8_Tun_HDAudioProgram__TUN_AUDIO_PROGRAM_SIX);
               }
               else if (itr->getB8HDNumOfAudioPrograms().getTUN_HD_AUDIOPROGRAM_SPS6())
               {
                  stationName.append(" HD7");
                  storeFmHDChannelElement.setAudioProgram(T_e8_Tun_HDAudioProgram__TUN_AUDIO_PROGRAM_SEVEN);
               }
               else if (itr->getB8HDNumOfAudioPrograms().getTUN_HD_AUDIOPROGRAM_SPS7())
               {
                  stationName.append(" HD8");
                  storeFmHDChannelElement.setAudioProgram(T_e8_Tun_HDAudioProgram__TUN_AUDIO_PROGRAM_EIGHT);
               }
               storeFmHDChannelElement.setStationName(stationName);
               _fmChannelListHD.push_back(storeFmHDChannelElement);
            }
         }
      }
      setObjectIdsforHDStationList();
      // check if channel list is updated
      if (isFMListUpdated())
      {
         updateFmObjectIDToChannelNameMap();
         // send tuner status to SDS MW as UPDATING
         _pTunerStateObserver->vTunerStatusChanged(sds2hmi_fi_tcl_e8_TUN_Band::FI_EN_RDS_FM, sds2hmi_fi_tcl_e8_DeviceStatus::FI_EN_DEVICE_UPDATING);
         ETG_TRACE_USR4(("clSDS_FmChannelList::onFID_TUN_G_GET_CONFIG_LISTStatus : tuner status updating"));
         // send trigger to sds_fm to store channel list in FM DB
         _sdsFmProxy->sendStoreRDSChannelNamesRequest(*this, _fmChannelListRDS);
      }
      if (isFMHDListUpdated())
      {
         updateFmObjectIDToChannelNameMap();
         // send tuner status to SDS MW as UPDATING
         _pTunerStateObserver->vTunerStatusChanged(sds2hmi_fi_tcl_e8_TUN_Band::FI_EN_HD_FM, sds2hmi_fi_tcl_e8_DeviceStatus::FI_EN_DEVICE_UPDATING);
         ETG_TRACE_USR4(("clSDS_FmChannelList::onFID_TUN_G_GET_CONFIG_LISTStatus : Tuner Status Updating"));
         // send trigger to sds_fm to store channel list in FM DB
         _sdsFmProxy->sendStoreHDChannelNamesRequest(*this, _fmChannelListHD);
      }
   }
}


//}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_FmChannelList::onFID_TUN_G_SURVIELLANCE_LISTUPDATEError(
   const ::boost::shared_ptr< tuner_main_fi::Tuner_main_fiProxy >& /*proxy*/,
   const ::boost::shared_ptr< tuner_main_fi::FID_TUN_G_SURVIELLANCE_LISTUPDATEError >& /*error*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool clSDS_FmChannelList::isFMHDListUpdated()
{
   bool stationFound = false;
   for (std::vector<sds_fm_fi::SdsFmService::FMChannelItem>::const_iterator it = _fmChannelListHD.begin(); it != _fmChannelListHD.end(); ++it)
   {
      stationFound = false;
      for (std::map<uint32 , std::string>::const_iterator itr_map = _fmHDObjectIDToChannelNameMap.begin(); itr_map != _fmHDObjectIDToChannelNameMap.end(); ++itr_map)
      {
         if (it->getStationName() == itr_map->second)
         {
            stationFound = true;
            break;
         }
      }
      if (!stationFound)
      {
         return true;
      }
   }
   return false;
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_FmChannelList::updateFmObjectIDToChannelNameMap()
{
   _fmRDSObjectIDToChannelNameMap.clear();
   for (std::vector<sds_fm_fi::SdsFmService::FMChannelItem>::const_iterator it = _fmChannelListRDS.begin(); it != _fmChannelListRDS.end(); ++it)
   {
      _fmRDSObjectIDToChannelNameMap.insert(std::pair<uint32, std::string>(it->getObjectID(), it->getStationName()));
   }
   _fmRDSChannelNameHistory.insert(_fmRDSObjectIDToChannelNameMap.begin(), _fmRDSObjectIDToChannelNameMap.end());

   _fmHDObjectIDToChannelNameMap.clear();
   for (std::vector<sds_fm_fi::SdsFmService::FMChannelItem>::const_iterator it = _fmChannelListHD.begin(); it != _fmChannelListHD.end(); ++it)
   {
      _fmHDObjectIDToChannelNameMap.insert(std::pair<uint32, std::string>(it->getObjectID(), it->getStationName()));
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
uint32 clSDS_FmChannelList::getFrequencyForObjectIdRDS(uint32 objectID)
{
   clSDS_FMTunerObject fmObj = getPICodeAndFrequencyFromObjectID(objectID);
   if (fmObj.picode != 0)
   {
      for (size_t i = 0; i < _fmChannelListRDS.size(); ++i)
      {
         if (_fmChannelListRDS[i].getObjectID() == fmObj.picode)
         {
            return _fmChannelListRDS[i].getFrequency();
         }
      }
   }
   return fmObj.frequency;
}


/**************************************************************************//**
 *
 ******************************************************************************/
uint32 clSDS_FmChannelList::getFrequencyForObjectIdHD(uint32 objectID)
{
   for (size_t i = 0; i < _fmChannelListHD.size(); ++i)
   {
      if (_fmChannelListHD[i].getObjectID() == objectID)
      {
         return _fmChannelListHD[i].getFrequency();
      }
   }
   return 0;
}


/**************************************************************************//**
 *
 ******************************************************************************/
std::string clSDS_FmChannelList::getStationNameForObjectIdRDS(uint32 objectID) const
{
   clSDS_FMTunerObject fmObj = getPICodeAndFrequencyFromObjectID(objectID);
   if (fmObj.picode != 0)
   {
      std::map<uint32 , std::string>::const_iterator itmap = _fmRDSChannelNameHistory.find(fmObj.picode);
      if (itmap != _fmRDSChannelNameHistory.end())
      {
         return itmap->second;
      }
   }
   return "";
}


/**************************************************************************//**
 *
 ******************************************************************************/
clSDS_FMTunerObject clSDS_FmChannelList::getPICodeAndFrequencyFromObjectID(uint32 objectID) const
{
   clSDS_FMTunerObject fmObj;
   fmObj.frequency = (objectID >> 16) * 10;
   fmObj.picode = objectID & 0xFFFF;
   return fmObj;
}


/**************************************************************************//**
 *
 ******************************************************************************/
uint32 clSDS_FmChannelList::getAudioProgramForObjectID(uint32 objectID)
{
   for (size_t i = 0; i < _fmChannelListHD.size(); ++i)
   {
      if (_fmChannelListHD[i].getObjectID() == objectID)
      {
         return _fmChannelListHD[i].getAudioProgram();
      }
   }
   return 0;
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool clSDS_FmChannelList::isFMListUpdated()
{
   std::map<uint32, std::string>::const_iterator itr_map;

   for (std::vector<sds_fm_fi::SdsFmService::FMChannelItem>::const_iterator it = _fmChannelListRDS.begin(); it != _fmChannelListRDS.end(); ++it)
   {
      itr_map = _fmRDSObjectIDToChannelNameMap.find(it->getObjectID());
      if (_fmRDSObjectIDToChannelNameMap.end() == itr_map)
      {
         return true;
      }
      else
      {
         std::string newStationName = it->getStationName();
         std::string storedStationName = itr_map->second;
         if (StringUtils::trim(newStationName) != StringUtils::trim(storedStationName))
         {
            return true;
         }
      }
   }
   return false;
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_FmChannelList::setObjectIdsforHDStationList()
{
   // generate random numbers and set them as object IDs for HD stations
   time_t seconds;
   //Get value from system clock and place in seconds variable.
   time(&seconds);
   srand((unsigned int) seconds);
   for (std::vector<sds_fm_fi::SdsFmService::FMChannelItem>::iterator it = _fmChannelListHD.begin(); it != _fmChannelListHD.end(); ++it)
   {
      it->setObjectID(rand());
   }
}
