/**************************************************************************//**
 * \file       clSDS_Display.cpp
 *
 * clSDS_Display class implementation
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "application/clSDS_Display.h"
#include "application/clSDS_ListItems.h"
#include "application/clSDS_StringVarList.h"
#include "application/clSDS_SDSStatus.h"
#include "application/clSDS_ScreenData.h"
#include "application/clSDS_View.h"
#include "view_db/Sds_ViewDB.h"
#include "application/clSDS_ConfigurationFlags.h"
#include "SdsAdapter_Trace.h"
#define ETG_ENABLED
#include "trace_interface.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_I_TTFIS_CMD_PREFIX      "SdsAdapter_"
#define ETG_I_TRACE_CHANNEL         TR_TTFIS_SAAL
#define ETG_DEFAULT_TRACE_CLASS     TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/clSDS_Display.cpp.trc.h"
#endif

/**************************************************************************//**
* Destructor
******************************************************************************/
clSDS_Display::~clSDS_Display()
{
   _pSdsStatus = NULL;
}


/**************************************************************************//**
* Constructor
******************************************************************************/
clSDS_Display::clSDS_Display(
   clSDS_SDSStatus* pSdsStatus,
   GuiService& guiService,
   PopUpService& popupService)

   : _guiService(guiService)
   , _popupService(popupService)
   , _pSdsStatus(pSdsStatus)
   , _speechInputState(sds_gui_fi::PopUpService::SpeechInputState__UNKNOWN)
   , _isCommandListAvailable(true)
   , _cursorIndex(0)
   , _state(clSDS_Display::INACTIVE)
   , _pageNumber(0)
   , _numberOfPages(0)
   , _nextPreviousOptions(sds_gui_fi::PopUpService::NextPreviousOption__SINGLE_PAGE)
   , _speakableList(false)
   , _maxItemsPerPage(0)
{
   _pSdsStatus->vRegisterObserver(this);
}


/***********************************************************************//**
*
***************************************************************************/
tVoid clSDS_Display::vShowView(const sds_gui_fi::PopUpService::PopupRequestSignal& popupRequestSignal, const std::string& viewId)
{
   ETG_TRACE_USR4(("clSDS_Display::vShowView popupRequestSignal.getLayout() %s ", popupRequestSignal.getLayout().c_str()));
   ETG_TRACE_USR4(("clSDS_Display::vShowView popupRequestSignal.getHeader() %s ", popupRequestSignal.getHeader().c_str()));
   ETG_TRACE_USR4(("clSDS_Display::vShowView popupRequestSignal.getSpeechInputState() %d ", popupRequestSignal.getSpeechInputState()));
   ::std::vector< sds_gui_fi::PopUpService::TextField > textFieldVector;
   textFieldVector = popupRequestSignal.getTextFields();
   ETG_TRACE_USR4(("textFieldVector.size() = %d", textFieldVector.size()));
   for (::std::vector< sds_gui_fi::PopUpService::TextField >::iterator iter = textFieldVector.begin(); iter != textFieldVector.end(); ++iter)
   {
      ETG_TRACE_USR4(("iter.getTagName() = %s", iter->getTagName().c_str()));
      ETG_TRACE_USR4(("iter.getString() = %s", iter->getString().c_str()));
   }
   ETG_TRACE_USR4(("clSDS_Display::vShowView viewId %s ", viewId.c_str()));
   ETG_TRACE_USR4(("clSDS_Display::vShowView viewId %s ", viewId.c_str()));
   _viewId = viewId;
   _popupService.sendPopupRequestSignal(popupRequestSignal.getLayout(),
                                        popupRequestSignal.getHeader(),
                                        popupRequestSignal.getSpeechInputState(),
                                        popupRequestSignal.getTextFields(),
                                        _pageNumber,
                                        _numberOfPages,
                                        _cursorIndex,
                                        _nextPreviousOptions,
                                        popupRequestSignal.getSpeakableList());

   _popupService.setScreenID(viewId);
}


/***********************************************************************//**
*
***************************************************************************/
tVoid clSDS_Display::vCloseView(tBool /*bError*/)
{
   if (_state == clSDS_Display::ACTIVE)
   {
      _state = clSDS_Display::INACTIVE;
      _popupService.sendPopupRequestCloseSignal();
   }
}


/***********************************************************************//**
*
***************************************************************************/
tVoid clSDS_Display::vSDSStatusChanged()
{
   SetSpeechInputState();
   createAndShowView();
}


/***********************************************************************//**
*
***************************************************************************/
void clSDS_Display::cursorMoved(uint32 cursorIndex)
{
   _cursorIndex = cursorIndex;
   createAndShowView();
}


/***********************************************************************//**
*
***************************************************************************/
std::string clSDS_Display::getScreenId() const
{
   return _viewId;
}


/***********************************************************************//**
*
***************************************************************************/
tVoid clSDS_Display::vShowSDSView(clSDS_ScreenData& oScreenData, uint32 pageNumber, uint32 numberOfPages, tU8 nextPreviousOptions, unsigned int maxItemsperPage, sds2hmi_fi_tcl_e8_PopUp_UpdateType screenUpdateType)
{
   _state = clSDS_Display::ACTIVE;
   _items = oScreenData.oGetListItemsofView();
   _viewId = oScreenData.readScreenId();
   _headerTemplateId = oScreenData.readTemplateId();
   _headerValueMap = oScreenData.oGetHeaderVariableData();
   _cursorIndex = getSelectableCommandIndex();
   _pageNumber = pageNumber;
   _numberOfPages = numberOfPages;
   _nextPreviousOptions = getNextPreviousOption(nextPreviousOptions);
   _maxItemsPerPage = maxItemsperPage;
   _screenUpdateType =  screenUpdateType;

   std::map<std::string, clSDS_Template*>::const_iterator ittmp;
   ittmp = _templateHandlerMap.find(_headerTemplateId);

   if (_templateHandlerMap.end() != ittmp)
   {
      _headerValueMap = ittmp->second->translateHeaderValues(_headerValueMap);
   }

   createAndShowView();
}


/***********************************************************************//**
*
***************************************************************************/
tVoid clSDS_Display::vAddTemplateHandler(std::string templateName, clSDS_Template* templateHandler)
{
   _templateHandlerMap.insert(std::pair<std::string, clSDS_Template*>(templateName, templateHandler));
}


/***********************************************************************//**
*
***************************************************************************/
void clSDS_Display::setSpeakableList(bool isSpeakableList)
{
   _speakableList = isSpeakableList;
}


/***********************************************************************//**
*
***************************************************************************/
void clSDS_Display::createAndShowView()
{
   if ((_state == clSDS_Display::ACTIVE) && (_pSdsStatus && !_pSdsStatus->bIsPause()))
   {
      clSDS_View oView;
      sds_gui_fi::PopUpService::PopupRequestSignal popupRequestSignal;
      if (_pSdsStatus)
      {
         popupRequestSignal.setSpeechInputState(_speechInputState);
         if (_pSdsStatus->bIsListening() && Sds_ViewDB_oGetHeadlineBehaviorId(_viewId) == Default)
         {
            // Nissan-specific: clear the headline content upon first transition
            // to LISTENING mode and keep it clear until the next user input
            _headerTemplateId = "SDS_TEMPLATE_NULL";  // TODO jnd2hi: should be replaced with empty string; however then a fixed header might be shown in listening mode -> clSDS_View::vWriteHeadline()
         }
         else if (Sds_ViewDB_oGetHeadlineBehaviorId(_viewId) == StickyTemplate)
         {
            if (_screenUpdateType.enType == sds2hmi_fi_tcl_e8_PopUp_UpdateType::FI_EN_UPDATE_ALL)
            {
               _stickyHeaderTemplateId = _headerTemplateId;
               _stickyHeaderValueMap = _headerValueMap;
            }
            else
            {
               _headerTemplateId = _stickyHeaderTemplateId;
               _headerValueMap = _stickyHeaderValueMap;
            }
         }
      }
      oView.vCreateView(popupRequestSignal, _viewId, _items, getCombinedScreenVariableData(), _headerTemplateId, _headerValueMap, _cursorIndex, _pageNumber, _maxItemsPerPage);

      //oView.vCreateViewHeader(popupRequestSignal, _headerTemplateId, _headerValueMap, _viewId);
      getCommandListAvailability(oView);
      popupRequestSignal.setSpeakableList(_speakableList);
      vShowView(popupRequestSignal, _viewId);
   }
}


/***********************************************************************//**
*
***************************************************************************/
void clSDS_Display::SetSpeechInputState()
{
   if (_pSdsStatus)
   {
      switch (_pSdsStatus->getStatus())
      {
         case clSDS_SDSStatus::EN_IDLE:
            _speechInputState = sds_gui_fi::PopUpService::SpeechInputState__IDLE;
            _guiService.setSdsStatus(sds_gui_fi::SdsGuiService::SpeechInputState__IDLE);
            break;
         case clSDS_SDSStatus::EN_DIALOGOPEN:
            _speechInputState = sds_gui_fi::PopUpService::SpeechInputState__DIALOGOPEN;
            _guiService.setSdsStatus(sds_gui_fi::SdsGuiService::SpeechInputState__DIALOGOPEN);
            break;
         case clSDS_SDSStatus::EN_LOADING:
            _speechInputState = sds_gui_fi::PopUpService::SpeechInputState__LOADING;
            _guiService.setSdsStatus(sds_gui_fi::SdsGuiService::SpeechInputState__LOADING);
            break;
         case clSDS_SDSStatus::EN_ERROR:
            _speechInputState = sds_gui_fi::PopUpService::SpeechInputState__ERROR;
            _guiService.setSdsStatus(sds_gui_fi::SdsGuiService::SpeechInputState__ERROR);
            break;
         case clSDS_SDSStatus::EN_LISTENING:
            _speechInputState = sds_gui_fi::PopUpService::SpeechInputState__LISTENING;
            _guiService.setSdsStatus(sds_gui_fi::SdsGuiService::SpeechInputState__LISTENING);
            break;
         case clSDS_SDSStatus::EN_ACTIVE:
            _speechInputState = sds_gui_fi::PopUpService::SpeechInputState__ACTIVE;
            _guiService.setSdsStatus(sds_gui_fi::SdsGuiService::SpeechInputState__ACTIVE);
            break;
         case clSDS_SDSStatus::EN_PAUSE:
            _speechInputState = sds_gui_fi::PopUpService::SpeechInputState__PAUSE;
            _guiService.setSdsStatus(sds_gui_fi::SdsGuiService::SpeechInputState__PAUSE);
            break;
         case clSDS_SDSStatus::EN_IDLE_TTS_ONLY:
            _speechInputState = sds_gui_fi::PopUpService::SpeechInputState__IDLE_TTS_ONLY;
            _guiService.setSdsStatus(sds_gui_fi::SdsGuiService::SpeechInputState__IDLE);
            break;
         case clSDS_SDSStatus::EN_ACTIVE_PROMPTONLY:
            _speechInputState = sds_gui_fi::PopUpService::SpeechInputState__ACTIVE_PROMPTONLY;
            _guiService.setSdsStatus(sds_gui_fi::SdsGuiService::SpeechInputState__ACTIVE_PROMPTONLY);
            break;
         default:
            _speechInputState = sds_gui_fi::PopUpService::SpeechInputState__UNKNOWN;
            _guiService.setSdsStatus(sds_gui_fi::SdsGuiService::SpeechInputState__UNKNOWN);
            break;
      }

      _popupService.setSdsStatus(_speechInputState);
   }
}


/***********************************************************************//**
*
***************************************************************************/
bool clSDS_Display::hasCommandList() const
{
   return _isCommandListAvailable;
}


/***********************************************************************//**
*
***************************************************************************/
void clSDS_Display::getCommandListAvailability(const clSDS_View& sdsView)
{
   if (sdsView.getCommandListCount())
   {
      _isCommandListAvailable = true;
   }
   else
   {
      _isCommandListAvailable = false;
   }
}


/***********************************************************************//**
*
***************************************************************************/
std::map<std::string, std::string> clSDS_Display::getCombinedScreenVariableData() const
{
   std::map<std::string, std::string> combined = _screenVariableData;
   const std::map<std::string, std::string>& configurationData = clSDS_ConfigurationFlags::getConfigurationFlags();
   combined.insert(configurationData.begin(), configurationData.end());
   return combined;
}


/***********************************************************************//**
*
***************************************************************************/
unsigned int clSDS_Display::getPageNumber() const
{
   return _pageNumber;
}


/***********************************************************************//**
*
***************************************************************************/
void clSDS_Display::setScreenVariableData(clSDS_ScreenData& oScreenData)
{
   _screenVariableData = oScreenData.oGetScreenVariableData();
}


/***********************************************************************//**
*
***************************************************************************/
uint32 clSDS_Display::getSelectableCommandIndex() const
{
   std::vector<SelectableText> commandList = Sds_ViewDB_oGetCommandList(getCombinedScreenVariableData(), _viewId);
   ETG_TRACE_USR4(("clSDS_Display::getSelectableCommandIndex() commandList.size() = %d", commandList.size()));
   for (std::size_t index = 0; index != commandList.size(); ++index)
   {
      if (commandList[index].isSelectable == true)
      {
         return index;
      }
   }
   return 0;
}


/***********************************************************************//**
*
***************************************************************************/
sds_gui_fi::PopUpService::NextPreviousOption clSDS_Display::getNextPreviousOption(tU8 nextPreviousOtion)
{
   switch (nextPreviousOtion)
   {
      case SINGLE_PAGE:
         return sds_gui_fi::PopUpService::NextPreviousOption__SINGLE_PAGE;
         break;
      case PREVIOUS_PAGE:
         return sds_gui_fi::PopUpService::NextPreviousOption__PREVIOUS_PAGE;
         break;
      case NEXT_PAGE:
         return sds_gui_fi::PopUpService::NextPreviousOption__NEXT_PAGE;
         break;
      case NEXT_PREVIOUS_PAGE:
         return sds_gui_fi::PopUpService::NextPreviousOption__NEXT_PREVIOUS_PAGE;
         break;
   }
   return sds_gui_fi::PopUpService::NextPreviousOption__NEXT_PREVIOUS_PAGE;
}


/***********************************************************************//**
*
***************************************************************************/
void clSDS_Display::cursorIndexChanged(uint32 index)
{
   _cursorIndex = index;
}


/***********************************************************************//**
*
***************************************************************************/
unsigned int clSDS_Display::getMaxItemsPerPage() const
{
   return _maxItemsPerPage;
}
