/**************************************************************************//**
 * \file       clSDS_DabChannelList.cpp
 *
 * This file is part of the SdsAdapter component.
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "clSDS_DabChannelList.h"
#include "application/clSDS_TunerStateObserver.h"
#include "application/StringUtils.h"
#include "application/clSDS_KDSConfiguration.h"
#include <ctime>

#ifdef DP_DATAPOOL_ID
#define DP_S_IMPORT_INTERFACE_FI
#include "dp_generic_if.h"
#endif

#include "SdsAdapter_Trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/clSDS_DabChannelList.cpp.trc.h"
#endif


using namespace dabtuner_main_fi;
using namespace dabtuner_main_fi_types;

/**************************************************************************//**
 *
 ******************************************************************************/
clSDS_DabChannelList::~clSDS_DabChannelList()
{
   _pTunerStateObserver = NULL;
}


/**************************************************************************//**
 *
 ******************************************************************************/
clSDS_DabChannelList::clSDS_DabChannelList(
   ::boost::shared_ptr< sds_dab_fi::SdsDabService::SdsDabServiceProxy > sdsDabProxy,
   ::boost::shared_ptr<dabtuner_main_fi::Dabtuner_main_fiProxy> tunerDabProxy)
   : _sdsDabProxy(sdsDabProxy)
   , _tunerDabProxy(tunerDabProxy)
   , _pTunerStateObserver(NULL)

{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_DabChannelList::onAvailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _tunerDabProxy)
   {
      _tunerDabProxy->sendFID_DAB_G_LIST_UPDATEUpReg(*this);
      _tunerDabProxy->sendFID_DAB_G_SRVLIST_NOTIFICATIONUpReg(*this);
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_DabChannelList::onUnavailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _tunerDabProxy)
   {
      _tunerDabProxy->sendFID_DAB_G_LIST_UPDATERelUpRegAll();
      _tunerDabProxy->sendFID_DAB_G_SRVLIST_NOTIFICATIONRelUpRegAll();
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_DabChannelList::onFID_DAB_G_LIST_UPDATEError(const ::boost::shared_ptr< ::dabtuner_main_fi::Dabtuner_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::dabtuner_main_fi::FID_DAB_G_LIST_UPDATEError >& /*error*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_DabChannelList::onFID_DAB_G_LIST_UPDATEStatus(const ::boost::shared_ptr< ::dabtuner_main_fi::Dabtuner_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::dabtuner_main_fi::FID_DAB_G_LIST_UPDATEStatus >& status)
{
   sds_dab_fi::SdsDabService::DABChannelItem storeDabChannelElement;
   std::vector<dabtuner_main_fi_types::T_DAB_ChnListElement> _channelListElems;
   _channelListElems = status->getAvailSrvList();
   for (::std::vector< T_DAB_ChnListElement >::const_iterator itr = _channelListElems.begin(); itr != _channelListElems.end(); ++itr)
   {
      if (itr->getLabel().getSLabel().size())
      {
         storeDabChannelElement.setDabID(itr->getU32DABId());
         storeDabChannelElement.setListElementID(itr->getU16ListElementID());
         storeDabChannelElement.setServiceLabel(itr->getLabel().getSLabel().c_str());
         storeDabChannelElement.setSecComponentID(0);
         storeDabChannelElement.setEnsembleLabel(itr->getEnsembleLabel().getSLabel().c_str());
         /*Store the list info in sds_dab_fi::_dabChannel list*/
         _dabChannelList.push_back(storeDabChannelElement);
      }
   }
   if (isDABListUpdated())
   {
      updateDabObjectIDToChannelNameMap();
      // send tuner status to SDS MW as UPDATING
      _pTunerStateObserver->vTunerStatusChanged(sds2hmi_fi_tcl_e8_TUN_Band::FI_EN_DAB, sds2hmi_fi_tcl_e8_DeviceStatus::FI_EN_DEVICE_UPDATING);
      ETG_TRACE_USR4(("clSDS_DabChannelList::onFID_DAB_S_SET_SRV_LISTResult : tuner status updating"));
      // send trigger to sds_dab to store channel list in DAB DB
      _sdsDabProxy->sendStoreDabChannelNamesRequest(*this, _dabChannelList);
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_DabChannelList::setDabChannelListObserver(clSDS_TunerStateObserver* pTunerStateObserver)
{
   _pTunerStateObserver = pTunerStateObserver;
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_DabChannelList::onStoreDabChannelNamesError(
   const ::boost::shared_ptr< sds_dab_fi::SdsDabService::SdsDabServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< sds_dab_fi::SdsDabService::StoreDabChannelNamesError >& /*error*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_DabChannelList::onStoreDabChannelNamesResponse(
   const ::boost::shared_ptr< sds_dab_fi::SdsDabService::SdsDabServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< sds_dab_fi::SdsDabService::StoreDabChannelNamesResponse >& /*response*/)
{
   if (_pTunerStateObserver != NULL)
   {
      _pTunerStateObserver->vTunerStatusChanged(sds2hmi_fi_tcl_e8_TUN_Band::FI_EN_DAB, sds2hmi_fi_tcl_e8_DeviceStatus::FI_EN_DEVICE_READY);
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool clSDS_DabChannelList::isDABListUpdated()
{
   std::map<uint32, std::string>::const_iterator itr_map;

   for (std::vector<sds_dab_fi::SdsDabService::DABChannelItem>::const_iterator it = _dabChannelList.begin(); it != _dabChannelList.end(); ++it)
   {
      itr_map = _dabObjectIDToServiceLabelMap.find(it->getDabID());
      if (_dabObjectIDToServiceLabelMap.end() == itr_map)
      {
         return true;
      }
      else
      {
         std::string newServiceLabel = it->getServiceLabel(); //get service label from _dabChannelList
         std::string storedServiceLabel = itr_map->second;
         if (StringUtils::trim(newServiceLabel) != StringUtils::trim(storedServiceLabel))
         {
            return true;
         }
      }
   }
   return false;
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_DabChannelList::updateDabObjectIDToChannelNameMap()
{
   _dabObjectIDToServiceLabelMap.clear();
   for (std::vector<sds_dab_fi::SdsDabService::DABChannelItem>::const_iterator it = _dabChannelList.begin(); it != _dabChannelList.end(); ++it)
   {
      _dabObjectIDToServiceLabelMap.insert(std::pair<uint32, std::string>(it->getDabID(), it->getServiceLabel()));
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_DabChannelList::onFID_DAB_S_SRV_SELECTError(const ::boost::shared_ptr< ::dabtuner_main_fi::Dabtuner_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::dabtuner_main_fi::FID_DAB_S_SRV_SELECTError >& /*error*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_DabChannelList::onFID_DAB_S_SRV_SELECTResult(const ::boost::shared_ptr< ::dabtuner_main_fi::Dabtuner_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< ::dabtuner_main_fi::FID_DAB_S_SRV_SELECTResult >& /*result*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
uint32 clSDS_DabChannelList::getDABIDForObjectID(uint32 objectID)
{
   for (std::vector<sds_dab_fi::SdsDabService::DABChannelItem>::const_iterator it = _dabChannelList.begin(); it != _dabChannelList.end(); ++it)
   {
      if (it->getDabID() == objectID)
      {
         return it->getDabID();
      }
   }
   return 0;
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_DabChannelList::onFID_DAB_G_SRVLIST_NOTIFICATIONError(const ::boost::shared_ptr< ::dabtuner_main_fi::Dabtuner_main_fiProxy >& /*proxy*/, const ::boost::shared_ptr< ::dabtuner_main_fi::FID_DAB_G_SRVLIST_NOTIFICATIONError >& /*error*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_DabChannelList::onFID_DAB_G_SRVLIST_NOTIFICATIONStatus(const ::boost::shared_ptr< ::dabtuner_main_fi::Dabtuner_main_fiProxy >& /*proxy*/, const ::boost::shared_ptr< ::dabtuner_main_fi::FID_DAB_G_SRVLIST_NOTIFICATIONStatus >& /*status*/)
{
}
