/**************************************************************************//**
 * \file       StateNameRequestor.cpp
 *
 * This file is part of the SdsAdapter component.
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "application/StateNameRequestor.h"
#include "application/StateNameObserver.h"
#include "application/StringUtils.h"


using namespace org::bosch::cm::navigation::NavigationService;

/**************************************************************************//**
 * Destructor
 ******************************************************************************/
StateNameRequestor::~StateNameRequestor()
{
}


/**************************************************************************//**
 * Constructor
 ******************************************************************************/
StateNameRequestor::StateNameRequestor(
   ::boost::shared_ptr<NavigationServiceProxy> naviProxy)
   :  _navigationProxy(naviProxy)
{
   _availableStateList.clear();
}


/**************************************************************************//**
*
******************************************************************************/
void StateNameRequestor::addStateObserver(StateNameObserver* pObserver)
{
   if (pObserver != NULL)
   {
      _stateObservers.push_back(pObserver);
   }
}


/**************************************************************************//**
*
******************************************************************************/
void StateNameRequestor::notifyStateObservers()
{
   std::vector<StateNameObserver*>::iterator it = _stateObservers.begin();

   while (it != _stateObservers.end())
   {
      if (*it != NULL)
      {
         (*it)->statesUpdated();
      }
      ++it;
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void StateNameRequestor::onUnavailable(const boost::shared_ptr<asf::core::Proxy>& proxy,
                                       const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _navigationProxy)
   {
      _navigationProxy->sendSdsAddressWithOptionsDeregisterAll();
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void StateNameRequestor::onAvailable(const boost::shared_ptr<asf::core::Proxy>& proxy,
                                     const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _navigationProxy)
   {
      _navigationProxy->sendSdsAddressWithOptionsRegister(*this);
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void StateNameRequestor::getAllStatesforCountry(std::string countryName)
{
   std::vector< SDSAddressElement > requestedState;

   _currentCountryName = countryName;

   SDSAddressElement sdsAddressElement;

   sdsAddressElement.setAddressElementType(SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_COUNTRY);

   sdsAddressElement.setData(countryName);

   requestedState.push_back(sdsAddressElement);

   _navigationProxy->sendSdsCheckAddressRequest(*this, requestedState);
}


/***********************************************************************//**
 *
 ***************************************************************************/
void StateNameRequestor::onSdsCheckAddressError(const ::boost::shared_ptr<NavigationServiceProxy>& /*proxy*/,
      const ::boost::shared_ptr< SdsCheckAddressError >& /*Error*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void StateNameRequestor::onSdsCheckAddressResponse(const ::boost::shared_ptr<NavigationServiceProxy >& /*proxy*/,
      const ::boost::shared_ptr< SdsCheckAddressResponse >& /*osdsCheckAddressResponse*/)
{
   //Navi hall will send the response of SdsCheckAddress in form of property update.The below function will handle this.
   handleSdsCheckAddressWithState();
}


/***********************************************************************//**
 *
 ***************************************************************************/
void StateNameRequestor::onSdsAddressWithOptionsUpdate(const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
      const ::boost::shared_ptr< SdsAddressWithOptionsUpdate >& /*oSdsAddressWithOptionsUpdate*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void StateNameRequestor::onSdsAddressWithOptionsError(const ::boost::shared_ptr<NavigationServiceProxy>& /*proxy*/,
      const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationService::SdsAddressWithOptionsError >& /*error*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void StateNameRequestor::handleSdsCheckAddressWithState()
{
   std::vector<SDSAddressElementType> elementType;
   std::vector< SDSAddressElementType>:: iterator it;

   if (_navigationProxy->hasSdsAddressWithOptions())
   {
      elementType = _navigationProxy->getSdsAddressWithOptions().getAddressElementType();

      for (it = elementType.begin(); it != elementType.end(); ++it)
      {
         if ((*it) == SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_STATE)
         {
            requestStateNames();
            break;
         }
      }
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void StateNameRequestor::requestStateNames()
{
   _navigationProxy->sendSdsGetRefinementListRequest(*this, SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_COUNTRY);
}


/**************************************************************************//**
 *
 ******************************************************************************/
void StateNameRequestor::onSdsGetRefinementListError(const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
      const ::boost::shared_ptr< SdsGetRefinementListError >& /*error*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void StateNameRequestor::onSdsGetRefinementListResponse(const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
      const ::boost::shared_ptr< SdsGetRefinementListResponse >& oSdsRefinementListresponse)
{
   ::std::vector< RefinementListElement > stateList;
   ::std::vector< RefinementListElement >:: iterator it;

   stateList = oSdsRefinementListresponse->getRefinements();

   for (it = stateList.begin(); it != stateList.end(); ++it)
   {
      StateDetails stateCountryPair;

      stateCountryPair.countryName = _currentCountryName;
      stateCountryPair.stateName = it->getData();

      _availableStateList.push_back(stateCountryPair);
   }

   notifyStateObservers();
}


/***********************************************************************//**
 *
 ***************************************************************************/
std::string StateNameRequestor::getCountryForGivenStateName(std::string stateName)
{
   std::string countryName;
   std::vector< StateDetails >::iterator it;

   for (it = _availableStateList.begin(); it != _availableStateList.end(); it++)
   {
      if (stateName == it->stateName)
      {
         countryName = it->countryName;
         break;
      }
   }

   return countryName;
}


/***********************************************************************//**
 *
 ***************************************************************************/
void StateNameRequestor::clearStateList()
{
   _availableStateList.clear();
   _currentCountryName.clear();
}
