/**************************************************************************//**
 * \file       SettingsService.cpp
 *
 * This file is part of the SdsAdapter component.
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "application/SettingsService.h"
#include "application/CommonSettingsObserver.h"
#include "sds_gui_fi/SettingsService.h"

using namespace sds_gui_fi::SettingsService;

SettingsService::SettingsService()
   : SettingsServiceStub("SettingsServicePort")
{
}


SettingsService::~SettingsService()
{
}


void SettingsService::onBeepOnlyModeSet(const ::boost::shared_ptr< BeepOnlyModeSet >& payload)
{
   bool beepOnlyMode = payload->getBeepOnlyMode();
   setBeepOnlyMode(beepOnlyMode);
   notifyBeepOnlyModeChanged(beepOnlyMode);
}


void SettingsService::onShortPromptSet(const ::boost::shared_ptr< ShortPromptSet >& payload)
{
   bool promptMode = payload->getShortPrompt();
   setShortPrompt(promptMode);
   notifyShortPromptChanged(promptMode);
}


void SettingsService::onBestMatchAudioSet(const ::boost::shared_ptr< BestMatchAudioSet >& payload)
{
   bool bestMatchAudio = payload->getBestMatchAudio();
   setBestMatchAudio(bestMatchAudio);
   notifyBestMatchAudioChanged(bestMatchAudio);
}


void SettingsService::onBestMatchPhoneBookSet(const ::boost::shared_ptr< BestMatchPhoneBookSet >& payload)
{
   bool bestMatchPhoneBook = payload->getBestMatchPhoneBook();
   setBestMatchPhoneBook(bestMatchPhoneBook);
   notifyBestMatchPhoneBookChanged(bestMatchPhoneBook);
}


void SettingsService::onVoicePreferenceSet(const ::boost::shared_ptr< VoicePreferenceSet >& payload)
{
   setVoicePreference(payload->getVoicePreference());
   notifyVoicePreferenceChanged();
}


void SettingsService::onSpeechRateSet(const ::boost::shared_ptr< SpeechRateSet >& payload)
{
   tU32 speechRate = payload ->getSpeechRate();
   setSpeechRate(payload->getSpeechRate());
   notifySpeechRateChanged(speechRate);
}


void SettingsService::onVoiceBargeInSet(const ::boost::shared_ptr< VoiceBargeInSet >& payload)
{
   bool voiceBargeIn = payload->getVoiceBargeIn();
   setVoiceBargeIn(voiceBargeIn);
   notifyVoiceBargeInChanged(voiceBargeIn);
}


void SettingsService::addCommonSettingsObserver(CommonSettingsObserver* pObserver)
{
   _observers.push_back(pObserver);
}


void SettingsService::notifyVoicePreferenceChanged()
{
   for (size_t i = 0; i < _observers.size(); ++i)
   {
      if (_observers[i])
      {
         _observers[i]->voicePreferenceChanged();
      }
   }
}


void SettingsService::notifyBeepOnlyModeChanged(bool beepOnlyMode)
{
   for (size_t i = 0; i < _observers.size(); ++i)
   {
      if (_observers[i])
      {
         _observers[i]->updateBeepOnlyMode(beepOnlyMode);
      }
   }
}


void SettingsService::notifyShortPromptChanged(bool promptMode)
{
   for (size_t i = 0; i < _observers.size(); ++i)
   {
      if (_observers[i])
      {
         _observers[i]->promptModeChanged(promptMode);
      }
   }
}


void SettingsService::notifyBestMatchPhoneBookChanged(bool bestMatchPhoneBook)
{
   for (size_t i = 0; i < _observers.size(); ++i)
   {
      if (_observers[i])
      {
         _observers[i]->updateNBestMatchPhoneBook(bestMatchPhoneBook);
      }
   }
}


void SettingsService::notifyBestMatchAudioChanged(bool bestMatchAudio)
{
   for (size_t i = 0; i < _observers.size(); ++i)
   {
      if (_observers[i])
      {
         _observers[i]->updateNBestMatchAudio(bestMatchAudio);
      }
   }
}


void SettingsService::notifySpeechRateChanged(unsigned int speechrate)
{
   for (size_t i = 0; i < _observers.size(); ++i)
   {
      if (_observers[i])
      {
         _observers[i]->speechRateChanged(speechrate);
      }
   }
}


void SettingsService::notifyVoiceBargeInChanged(bool voiceBargeIn)
{
   for (size_t i = 0; i < _observers.size(); ++i)
   {
      if (_observers[i])
      {
         _observers[i]->updateVoiceBargeIn(voiceBargeIn);
      }
   }
}


void SettingsService::updateAvailableSpeakers(const std::vector<sds2hmi_fi_tcl_LanguageAndSpeaker>& availableSpeakers)
{
   std::vector<sds2hmi_fi_tcl_LanguageAndSpeaker>::const_iterator itr;
   std::vector<sds_gui_fi::SettingsService::SpeechRecognitionLanguage > sdsLangaugeList;

   for (itr = availableSpeakers.begin(); itr != availableSpeakers.end(); itr++)
   {
      sds_gui_fi::SettingsService::SpeechRecognitionLanguage sdsLanguage;

      sdsLanguage.setSpeakerId(itr->SpeakerId);
      sdsLanguage.setSpeakerName(itr->Speaker);
      sdsLanguage.setIsoLanguageCode(itr->LanguageID.ISO639_3_SDSLanguageCode.enType);
      sdsLanguage.setIsoCountryCode(itr->LanguageID.ISO3166_CountryCode.enType);
      sdsLanguage.setAsrSupported(itr->ASRsupported);
      sdsLanguage.setGender(itr->Gender.enType);
      sdsLangaugeList.push_back(sdsLanguage);
   }
   setSpeechRecognitionLanguageList(sdsLangaugeList);
}
