/**************************************************************************//**
 * \file       QuickChargingStation.cpp
 *
 * QuickChargingStation (AIF)
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "application/QuickChargingStation.h"
#include "application/ChargingStationListObserver.h"
#include "application/clSDS_StringVarList.h"
#include "SdsAdapter_Trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/QuickChargingStation.cpp.trc.h"
#endif


using namespace org::bosch::cm::navigation::NavigationService;


#define NUM_OF_DIGITS_IN_DIRECTION_VALUE 5


/**************************************************************************//**
*Constructor
******************************************************************************/
QuickChargingStation::QuickChargingStation(::boost::shared_ptr< NavigationServiceProxy > naviProxy): _naviProxy(naviProxy)
{
   _destinationIndex = 0;
   _isChargingStationListRequested = false;
   _restoreHMIListResponse = true;
}


/**************************************************************************//**
*Destructor
******************************************************************************/
QuickChargingStation::~QuickChargingStation()
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void QuickChargingStation::onAvailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _naviProxy)
   {
      _naviProxy->sendDestinationInformationRegister(*this);
      _naviProxy->sendDestinationInformationGet(*this);
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void QuickChargingStation::onUnavailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _naviProxy)
   {
      _naviProxy->sendDestinationInformationDeregisterAll();
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/


void QuickChargingStation::onDestinationInformationError(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< DestinationInformationError >& /*error*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/

void QuickChargingStation::onDestinationInformationUpdate(
   const ::boost::shared_ptr< NavigationServiceProxy >&  /*proxy*/,
   const ::boost::shared_ptr< DestinationInformationUpdate >& update)
{
   if (update->hasDestinationInformation())
   {
      _destinationIndex = (update->getDestinationInformation().size() - 1);
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/

void QuickChargingStation::onRequestFixedPOICategorySearchResultsError(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< RequestFixedPOICategorySearchResultsError >& /*error*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void QuickChargingStation::onRequestFixedPOICategorySearchResultsResponse(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< RequestFixedPOICategorySearchResultsResponse >& response)
{
   if (response->hasSearchResults())
   {
      _addressList = response->getSearchResults();
      notifyListObserver();
   }
}


/***********************************************************************//**

 ***************************************************************************/
void QuickChargingStation::onUndoSetLocationWithFixedPOICategoryInputError(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< UndoSetLocationWithFixedPOICategoryInputError >& /*error*/)
{
   _restoreHMIListResponse =  false;
   notifyRestoreHMIListObserver();
}


/***********************************************************************//**
 *
 ***************************************************************************/
void QuickChargingStation::onUndoSetLocationWithFixedPOICategoryInputResponse(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< UndoSetLocationWithFixedPOICategoryInputResponse >& /*response*/)
{
   _restoreHMIListResponse =  true;
   notifyRestoreHMIListObserver();
}


/***********************************************************************//**
 *
 ***************************************************************************/
void QuickChargingStation::onSetLocationWithFixedPOICategoryInputError(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< SetLocationWithFreeTextSearchInputError >& /*error*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void QuickChargingStation::onSetLocationWithFixedPOICategoryInputResponse(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< SetLocationWithFreeTextSearchInputResponse >& /*response*/)
{
   ETG_TRACE_USR1(("onSetLocationWithFixedPOICategoryInputResponse"));
   _isChargingStationListRequested = false;
   notifyChargingStationlistObserver();
}


/***********************************************************************//**
 *
 ***************************************************************************/
tU32 QuickChargingStation::u32GetSize()
{
   return _addressList.size();
}


/***********************************************************************//**
 *
 ***************************************************************************/
tVoid QuickChargingStation::vGetListInfo(sds2hmi_fi_tcl_e8_HMI_ListType::tenType listType)
{
   if (listType == sds2hmi_fi_tcl_e8_HMI_ListType::FI_EN_LIST_EVE_ALL_CHARGE_STATIONS || listType == sds2hmi_fi_tcl_e8_HMI_ListType::FI_EN_LIST_EVE_QUICK_CHARGE_STATIONS)
   {
      tU32 startIndex = 0;
      tU32 totalElements = 20;
      _naviProxy->sendRequestFixedPOICategorySearchResultsRequest(*this, FixedPOICategory__EV_CHARGING_STATION,
            FreeTextSearchScope__FREETEXT_SEARCH_SCOPE_POI_AT_CURRENT_POSITION, _destinationIndex, startIndex, totalElements);
      _isChargingStationListRequested =  true;
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
std::vector<clSDS_ListItems> QuickChargingStation::oGetItems(tU32 u32StartIndex, tU32 u32EndIndex)
{
   std::vector<clSDS_ListItems> oListItems;
   for (tU32 u32Index = u32StartIndex; u32Index < std::min(u32EndIndex, u32GetSize()); u32Index++)
   {
      oListItems.push_back(oGetListItem(u32Index));
   }
   return oListItems;
}


/**************************************************************************//**
*
******************************************************************************/
clSDS_ListItems QuickChargingStation::oGetListItem(tU32 u32Index)
{
   clSDS_ListItems oListItem;
   oListItem.oDescription.szString = oGetItem(u32Index);
   oListItem.oDistance.szString = oGetDisatance(u32Index);

   tChar szTempString[NUM_OF_DIGITS_IN_DIRECTION_VALUE] = "";
   OSALUTIL_s32SaveNPrintFormat(szTempString, sizeof(szTempString), "%d", oGetDirectionFromMapping(u32Index));
   oListItem.oDirectionSymbol.szString = szTempString;

   return oListItem;
}


/***********************************************************************//**
 *
 ***************************************************************************/
tBool QuickChargingStation::bSelectElement(tU32 u32SelectedIndex)
{
   if (u32SelectedIndex > 0)
   {
      clSDS_StringVarList::vSetVariable("$(POI_address)", oGetItem(u32SelectedIndex - 1));
      return TRUE;
   }
   return FALSE;
}


/***********************************************************************//**
 *
 ***************************************************************************/
std::string QuickChargingStation::oGetItem(tU32 u32Index)
{
   if (u32Index < _addressList.size())
   {
      return _addressList[u32Index].getData();
   }
   else
   {
      return "";
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
bool QuickChargingStation::getChargingStationListRequestStatus()
{
   return _isChargingStationListRequested;
}


/***********************************************************************//**
 *
 ***************************************************************************/
void QuickChargingStation::setChargingStationListRequestStatus(bool chargingStationSelected)
{
   _isChargingStationListRequested = chargingStationSelected;
}


/***********************************************************************//**
 *
 ***************************************************************************/
void QuickChargingStation::sendChargingStationListRequest(tU8 ambiguousValue)
{
   _naviProxy->sendSetLocationWithFixedPOICategoryInputRequest(*this, getRefinementID(ambiguousValue));
}


/***********************************************************************//**
 *
 ***************************************************************************/
void QuickChargingStation::vRestoreHMIListData()
{
   _naviProxy->sendUndoSetLocationWithFixedPOICategoryInputRequest(*this);
}


/***********************************************************************//**
 *
 ***************************************************************************/
void QuickChargingStation::registerChargingStationListObserver(ChargingStationListObserver* chargingStationListObserver)
{
   _chargingStationObserverlist.push_back(chargingStationListObserver);
}


/**************************************************************************//**
 *
 ******************************************************************************/
void QuickChargingStation::notifyChargingStationlistObserver()
{
   ETG_TRACE_USR1(("NotifyChargingStationlistObserver"));
   std::vector<ChargingStationListObserver*>::iterator iter = _chargingStationObserverlist.begin();
   while (iter != _chargingStationObserverlist.end())
   {
      if (*iter != NULL)
      {
         (*iter)->setLocationWithFixedPOICategoryCompleted();
      }
      ++iter;
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
bool QuickChargingStation::getResultType()
{
   return _restoreHMIListResponse;
}


/**************************************************************************//**
*
******************************************************************************/

std::vector<sds2hmi_fi_tcl_HMIElementDescrptionList> QuickChargingStation::getHmiListDescription()
{
   std::vector<sds2hmi_fi_tcl_HMIElementDescrptionList> chargingStationList;

   for (size_t i = 0; i < _addressList.size(); ++i)
   {
      sds2hmi_fi_tcl_HMIElementDescription hmiElementDescription;
      sds2hmi_fi_tcl_HMIElementDescrptionList hmiElementDescriptionList;

      if (!_addressList[i].getAddress().empty())
      {
         hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_POICATEGORY;
         hmiElementDescription.DescriptorValue.bSet(_addressList[i].getAddress().c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);

         hmiElementDescriptionList.DescriptionList.push_back(hmiElementDescription);
      }
      chargingStationList.push_back(hmiElementDescriptionList);
   }

   return chargingStationList;
}


/**************************************************************************//**
 *
 ******************************************************************************/

std::vector<sds2hmi_fi_tcl_HMIElementDescription>  QuickChargingStation::getHmiElementDescription(unsigned int index)
{
   sds2hmi_fi_tcl_HMIElementDescription hmiElementDescription;
   std::vector<sds2hmi_fi_tcl_HMIElementDescription> _hmiElementDescriptionList;
   _hmiElementDescriptionList.clear();
   if ((index <= _addressList.size()) && (index > 0))
   {
      if (!_addressList.at(index - 1).hasAddress())
      {
         hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_POICATEGORY;
         hmiElementDescription.DescriptorValue.bSet(_addressList[index - 1].getAddress().c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);

         _hmiElementDescriptionList.push_back(hmiElementDescription);
      }
   }
   return _hmiElementDescriptionList;
}


/***********************************************************************//**
 *
 ***************************************************************************/
tU32 QuickChargingStation::getRefinementID(tU32 absoluteValue)
{
   if (absoluteValue > 0 && absoluteValue <= _addressList.size())
   {
      return (_addressList.at(absoluteValue - 1).getId());
   }
   return 0;
}


/**************************************************************************//**
*
******************************************************************************/
std::string QuickChargingStation::oGetDisatance(tU32 u32Index)
{
   if (u32Index < _addressList.size())
   {
      return _addressList[u32Index].getDistance();
   }
   else
   {
      return "";
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
signed char QuickChargingStation::oGetDirectionFromMapping(tU32 u32Index)const
{
   if (_addressList[u32Index].getDirection() == DirectionDescription__NORTH ||
         _addressList[u32Index].getDirection()  == DirectionDescription__NORTH_NORTH_WEST)
   {
      return (EN_HEADING_INDEX_NORTH);
   }
   else if (_addressList[u32Index].getDirection()  == DirectionDescription__NORTH_NORTH_EAST ||
            _addressList[u32Index].getDirection()  == DirectionDescription__NORTH_EAST)
   {
      return (EN_HEADING_INDEX_NORTH_EAST);
   }
   else if (_addressList[u32Index].getDirection()  == DirectionDescription__EAST_NORTH_EAST ||
            _addressList[u32Index].getDirection()  == DirectionDescription__EAST)
   {
      return (EN_HEADING_INDEX_EAST);
   }
   else if (_addressList[u32Index].getDirection()  == DirectionDescription__EAST_SOUTH_EAST ||
            _addressList[u32Index].getDirection()  == DirectionDescription__SOUTH_EAST)
   {
      return (EN_HEADING_INDEX_SOUTH_EAST);
   }
   else if (_addressList[u32Index].getDirection() == DirectionDescription__SOUTH_SOUTH_EAST ||
            _addressList[u32Index].getDirection()  == DirectionDescription__SOUTH)
   {
      return (EN_HEADING_INDEX_SOUTH);
   }
   else if (_addressList[u32Index].getDirection()  == DirectionDescription__SOUTH_SOUTH_WEST ||
            _addressList[u32Index].getDirection() == DirectionDescription__SOUTH_WEST)
   {
      return (EN_HEADING_INDEX_SOUTH_WEST);
   }
   else if (_addressList[u32Index].getDirection()  == DirectionDescription__WEST_SOUTH_WEST ||
            _addressList[u32Index].getDirection()  == DirectionDescription__WEST)
   {
      return (EN_HEADING_INDEX_WEST);
   }
   else if (_addressList[u32Index].getDirection()  == DirectionDescription__WEST_NORTH_WEST ||
            _addressList[u32Index].getDirection()  == DirectionDescription__NORTH_WEST)
   {
      return (EN_HEADING_INDEX_NORTH_WEST);
   }
   else if (_addressList[u32Index].getDirection()  == DirectionDescription__INVALID)
   {
      return (EN_HEADING_INDEX_UNKNOWN);
   }
   else
   {
      return (EN_HEADING_INDEX_UNKNOWN);
   }
}
