/**************************************************************************//**
 * \file       NdsStringRequestor.cpp
 *
 * This file is part of the SdsAdapter component.
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "application/NdsStringRequestor.h"
#include "application/NdsStringObserver.h"
#include "application/StringUtils.h"


using namespace org::bosch::cm::navigation::NavigationSDSService;


/**************************************************************************//**
 * Destructor
 ******************************************************************************/
NdsStringRequestor::~NdsStringRequestor()
{
}


/**************************************************************************//**
 * Constructor
 ******************************************************************************/
NdsStringRequestor::NdsStringRequestor(
   ::boost::shared_ptr<org::bosch::cm::navigation::NavigationSDSService::NavigationSDSServiceProxy> naviProxy)
   : _navigationSDSProxy(naviProxy)
{
   _featureTypeRequestCounter = 0;
}


/**************************************************************************//**
*
******************************************************************************/
void NdsStringRequestor::addNdsStringObserver(NdsStringObserver* pObserver)
{
   if (pObserver != NULL)
   {
      _ndsStringObservers.push_back(pObserver);
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void NdsStringRequestor::getNdsFeautreType(
   std::vector<sds2hmi_fi_tcl_NDSID> ndsIdString,
   std::vector<sds2hmi_fi_tcl_NDSLanguageID> ndsLanguageId)
{
   std::vector<sds2hmi_fi_tcl_NDSID>::iterator it = ndsIdString.begin();

   while (it != ndsIdString.end())
   {
      switch (it->eNDSFeatureType.enType)
      {
         case sds2hmi_fi_tcl_e8_NDSFeatureType::FI_EN_NAMEOBJECT:
            getNdsIdBasedOnFeatureType(*it, _sdsNamedObjectList);
            break;
         case sds2hmi_fi_tcl_e8_NDSFeatureType::FI_EN_POINAME:
            getNdsIdBasedOnFeatureType(*it, _sdsPOINameList);
            break;
         case sds2hmi_fi_tcl_e8_NDSFeatureType::FI_EN_POI_CATEGORY:
            getNdsIdBasedOnFeatureType(*it, _sdsPOICategoryList);
            break;
         case sds2hmi_fi_tcl_e8_NDSFeatureType::FI_EN_POI_ATTRIBUTE:
            getNdsIdBasedOnFeatureType(*it, _sdsPOIBrandNameList);
            break;
         default:
            break;
      }

      ++it;
   }

   _sdsNdsLanguageId = ndsLanguageId;

   updateSDSRequestedFeatureType();

   getStringForNdsId();
}


/***********************************************************************//**
 *
 ***************************************************************************/
void NdsStringRequestor::updateSDSRequestedFeatureType()
{
   if (_sdsNamedObjectList.size())
   {
      _featureTypeRequestCounter++;
   }
   if (_sdsPOINameList.size())
   {
      _featureTypeRequestCounter++;
   }
   if (_sdsPOICategoryList.size())
   {
      _featureTypeRequestCounter++;
   }
   if (_sdsPOIBrandNameList.size())
   {
      _featureTypeRequestCounter++;
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void NdsStringRequestor::getNdsIdBasedOnFeatureType(
   sds2hmi_fi_tcl_NDSID ndsIdObject,
   std::vector<sds2hmi_fi_tcl_NDSID>& ndsObjectIdList)
{
   sds2hmi_fi_tcl_NDSID ndsNamedObject;

   ndsNamedObject.u16UpdateRegionId = ndsIdObject.u16UpdateRegionId;
   ndsNamedObject.u32ProductId = ndsIdObject.u32ProductId;
   ndsNamedObject.u8SupplierId = ndsIdObject.u8SupplierId;
   ndsNamedObject.u32NDSId = ndsIdObject.u32NDSId;

   ndsObjectIdList.push_back(ndsNamedObject);
}


/***********************************************************************//**
 *
 ***************************************************************************/
void NdsStringRequestor::getStringForNdsId()
{
   if (_sdsNamedObjectList.size())
   {
      _currentRequestedFeatureType.enType = sds2hmi_fi_tcl_e8_NDSFeatureType::FI_EN_NAMEOBJECT;
      sendNdsIdStringRequest(_sdsNamedObjectList, NDSFeatureType__NAMEDOBJECT);
      _sdsNamedObjectList.clear();
      _featureTypeRequestCounter--;
   }
   else if (_sdsPOINameList.size())
   {
      _currentRequestedFeatureType.enType = sds2hmi_fi_tcl_e8_NDSFeatureType::FI_EN_POINAME;
      sendNdsIdStringRequest(_sdsPOINameList, NDSFeatureType__POI_NAME);
      _sdsPOINameList.clear();
      _featureTypeRequestCounter--;
   }
   else if (_sdsPOICategoryList.size())
   {
      _currentRequestedFeatureType.enType = sds2hmi_fi_tcl_e8_NDSFeatureType::FI_EN_POI_CATEGORY;
      sendNdsIdStringRequest(_sdsPOICategoryList, NDSFeatureType__POI_CATEGORY);
      _sdsPOICategoryList.clear();
      _featureTypeRequestCounter--;
   }
   else if (_sdsPOIBrandNameList.size())
   {
      _currentRequestedFeatureType.enType = sds2hmi_fi_tcl_e8_NDSFeatureType::FI_EN_POI_ATTRIBUTE;
      sendNdsIdStringRequest(_sdsPOIBrandNameList, NDSFeatureType__POI_ATTRIBUTE);
      _sdsPOIBrandNameList.clear();
      _featureTypeRequestCounter--;
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void NdsStringRequestor::sendNdsIdStringRequest(
   std::vector<sds2hmi_fi_tcl_NDSID>& objectIdList,
   NDSFeatureType ndsFeatureType)
{
   std::vector< GlobalNamedObjectId > namedObjectIds;
   std::vector< LanguageEntry > languageEntryList;
   std::vector<sds2hmi_fi_tcl_NDSID>::iterator it = objectIdList.begin();

   while (it != objectIdList.end())
   {
      GlobalNamedObjectId namedObjectIdList;
      GlobalUpdateRegionId globalUpdateRegionId;

      globalUpdateRegionId.setUpdateRegionId(it->u16UpdateRegionId);
      namedObjectIdList.setGlobalUpdateRegionId(globalUpdateRegionId);

      globalUpdateRegionId.setProductId(it->u32ProductId);
      namedObjectIdList.setGlobalUpdateRegionId(globalUpdateRegionId);

      globalUpdateRegionId.setSupplierId(it->u8SupplierId);
      namedObjectIdList.setGlobalUpdateRegionId(globalUpdateRegionId);

      namedObjectIdList.setNamedObjectId(it->u32NDSId);

      namedObjectIds.push_back(namedObjectIdList);

      ++it;
   }

   getLanguageListEntry(languageEntryList);

   _navigationSDSProxy->sendGetNamesForNDSIdsRequest(*this, namedObjectIds, languageEntryList, ndsFeatureType, true);
}


/***********************************************************************//**
 *
 ***************************************************************************/
void NdsStringRequestor::getLanguageListEntry(std::vector< LanguageEntry >& languageEntryList)
{
   LanguageEntry languageEntry;

   for (size_t i = 0; i < _sdsNdsLanguageId.size(); ++i)
   {
      languageEntry.setISOLanguageCode(_sdsNdsLanguageId[i].ISO_LanguageCode);
      languageEntry.setISOCountryCode(_sdsNdsLanguageId[i].ISO_CountryCode);
      languageEntry.setTranscriptCode(_sdsNdsLanguageId[i].ScriptCode);
      languageEntryList.push_back(languageEntry);
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void NdsStringRequestor::onGetNamesForNDSIdsError(const ::boost::shared_ptr<NavigationSDSServiceProxy>& /*proxy*/,
      const ::boost::shared_ptr< GetNamesForNDSIdsError >& /*Error*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void NdsStringRequestor::onGetNamesForNDSIdsResponse(const ::boost::shared_ptr<NavigationSDSServiceProxy >& /*proxy*/,
      const ::boost::shared_ptr< GetNamesForNDSIdsResponse >& getNamesForNDSIdsResponse)
{
   if (sds2hmi_fi_tcl_e8_NDSFeatureType::FI_EN_NAMEOBJECT == _currentRequestedFeatureType.enType)
   {
      _ndsStringNamedObject = getNamesForNDSIdsResponse->getNamedObjectIdStringsList();
   }
   else if (sds2hmi_fi_tcl_e8_NDSFeatureType::FI_EN_POINAME == _currentRequestedFeatureType.enType)
   {
      _ndsStringPOIName = getNamesForNDSIdsResponse->getNamedObjectIdStringsList();
   }
   else if (sds2hmi_fi_tcl_e8_NDSFeatureType::FI_EN_POI_CATEGORY == _currentRequestedFeatureType.enType)
   {
      _ndsStringPOICategory = getNamesForNDSIdsResponse->getNamedObjectIdStringsList();
   }
   else if (sds2hmi_fi_tcl_e8_NDSFeatureType::FI_EN_POI_ATTRIBUTE == _currentRequestedFeatureType.enType)
   {
      _ndsStringPOIAttribute = getNamesForNDSIdsResponse->getNamedObjectIdStringsList();
   }

   if (_featureTypeRequestCounter)
   {
      getStringForNdsId();
   }
   else
   {
      notifyObjectIdStringListObservers();
      _featureTypeRequestCounter = 0;
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void NdsStringRequestor::notifyObjectIdStringListObservers()
{
   std::vector<NdsStringObserver*>::iterator it = _ndsStringObservers.begin();

   while (it != _ndsStringObservers.end())
   {
      if (*it != NULL)
      {
         (*it)->ndsNamesUpdated();
      }
      ++it;
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void NdsStringRequestor::getNdsNamedObjectList(std::vector< NamedObjectIdStrings >& namedObjectList)
{
   namedObjectList = _ndsStringNamedObject;
}


/***********************************************************************//**
 *
 ***************************************************************************/
void NdsStringRequestor::getNdsPOINameList(std::vector< NamedObjectIdStrings >& poiNameList)
{
   poiNameList = _ndsStringPOIName;
}


/***********************************************************************//**
 *
 ***************************************************************************/
void NdsStringRequestor::getNdsPOICategoryList(std::vector< NamedObjectIdStrings >& poiCategoryList)
{
   poiCategoryList =  _ndsStringPOICategory;
}


/***********************************************************************//**
 *
 ***************************************************************************/
void NdsStringRequestor::getNdsPOIAttributeList(std::vector< NamedObjectIdStrings >& poiAttributeList)
{
   poiAttributeList = _ndsStringPOIAttribute;
}


/***********************************************************************//**
 *
 ***************************************************************************/
void NdsStringRequestor::clearAllListData()
{
   _sdsNamedObjectList.clear();
   _sdsPOICategoryList.clear();
   _sdsPOIBrandNameList.clear();
   _sdsPOINameList.clear();

   _ndsStringNamedObject.clear();
   _ndsStringPOIName.clear();
   _ndsStringPOICategory.clear();
   _ndsStringPOIAttribute.clear();

   _featureTypeRequestCounter = 0;
}
