/**************************************************************************//**
 * \file       NdsPhonemeRequestor.cpp
 *
 * This file is part of the SdsAdapter component.
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "application/NdsPhonemeRequestor.h"
#include "application/NdsPhonemeObserver.h"
#include "application/StringUtils.h"


using namespace org::bosch::cm::navigation::NavigationSDSService;


/**************************************************************************//**
 * Destructor
 ******************************************************************************/
NdsPhonemeRequestor::~NdsPhonemeRequestor()
{
}


/**************************************************************************//**
 * Constructor
 ******************************************************************************/
NdsPhonemeRequestor::NdsPhonemeRequestor(
   ::boost::shared_ptr<org::bosch::cm::navigation::NavigationSDSService::NavigationSDSServiceProxy> naviProxy)
   : _navigationSDSProxy(naviProxy)
{
   _featureTypeRequestCounter = 0;
}


/**************************************************************************//**
*
******************************************************************************/
void NdsPhonemeRequestor::addNdsPhonemeObserver(NdsPhonemeObserver* pObserver)
{
   if (pObserver != NULL)
   {
      _ndsPhonemeObservers.push_back(pObserver);
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void NdsPhonemeRequestor::getNdsFeautreTypeForPhoneme(
   std::vector<sds2hmi_fi_tcl_NDSID> ndsIdPhoneme,
   std::vector<sds2hmi_fi_tcl_NDSLanguageID> ndsLanguageId)
{
   std::vector<sds2hmi_fi_tcl_NDSID>::iterator it = ndsIdPhoneme.begin();

   while (it != ndsIdPhoneme.end())
   {
      switch (it->eNDSFeatureType.enType)
      {
         case sds2hmi_fi_tcl_e8_NDSFeatureType::FI_EN_NAMEOBJECT:
            getNdsIdBasedOnFeatureType(*it, _sdsPhonemeObjectList);
            break;
         case sds2hmi_fi_tcl_e8_NDSFeatureType::FI_EN_POINAME:
            getNdsIdBasedOnFeatureType(*it, _sdsPhonemePOINameList);
            break;
         case sds2hmi_fi_tcl_e8_NDSFeatureType::FI_EN_POI_CATEGORY:
            getNdsIdBasedOnFeatureType(*it, _sdsPhonemePOICategoryList);
            break;
         case sds2hmi_fi_tcl_e8_NDSFeatureType::FI_EN_POI_ATTRIBUTE:
            getNdsIdBasedOnFeatureType(*it, _sdsPhonemePOIBrandNameList);
            break;
         default:
            break;
      }

      ++it;
   }

   _sdsNdsLanguageId = ndsLanguageId;

   updateSDSRequestedFeatureType();

   getPhonemeForNdsId();
}


/***********************************************************************//**
 *
 ***************************************************************************/
void NdsPhonemeRequestor::updateSDSRequestedFeatureType()
{
   if (_sdsPhonemeObjectList.size())
   {
      _featureTypeRequestCounter++;
   }
   if (_sdsPhonemePOINameList.size())
   {
      _featureTypeRequestCounter++;
   }
   if (_sdsPhonemePOICategoryList.size())
   {
      _featureTypeRequestCounter++;
   }
   if (_sdsPhonemePOIBrandNameList.size())
   {
      _featureTypeRequestCounter++;
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void NdsPhonemeRequestor::getNdsIdBasedOnFeatureType(
   sds2hmi_fi_tcl_NDSID ndsIdObject,
   std::vector<sds2hmi_fi_tcl_NDSID>& ndsObjectIdList)
{
   sds2hmi_fi_tcl_NDSID ndsNamedObject;

   ndsNamedObject.u16UpdateRegionId = ndsIdObject.u16UpdateRegionId;
   ndsNamedObject.u32ProductId = ndsIdObject.u32ProductId;
   ndsNamedObject.u8SupplierId = ndsIdObject.u8SupplierId;
   ndsNamedObject.u32NDSId = ndsIdObject.u32NDSId;

   ndsObjectIdList.push_back(ndsNamedObject);
}


/***********************************************************************//**
 *
 ***************************************************************************/
void NdsPhonemeRequestor::getPhonemeForNdsId()
{
   if (_sdsPhonemeObjectList.size())
   {
      _currentRequestedFeatureType.enType = sds2hmi_fi_tcl_e8_NDSFeatureType::FI_EN_NAMEOBJECT;
      sendNdsIdPhonemeRequest(_sdsPhonemeObjectList, NDSFeatureType__NAMEDOBJECT);
      _sdsPhonemeObjectList.clear();
      _featureTypeRequestCounter--;
   }
   else if (_sdsPhonemePOINameList.size())
   {
      _currentRequestedFeatureType.enType = sds2hmi_fi_tcl_e8_NDSFeatureType::FI_EN_POINAME;
      sendNdsIdPhonemeRequest(_sdsPhonemePOINameList, NDSFeatureType__POI_NAME);
      _sdsPhonemePOINameList.clear();
      _featureTypeRequestCounter--;
   }
   else if (_sdsPhonemePOICategoryList.size())
   {
      _currentRequestedFeatureType.enType = sds2hmi_fi_tcl_e8_NDSFeatureType::FI_EN_POI_CATEGORY;
      sendNdsIdPhonemeRequest(_sdsPhonemePOICategoryList, NDSFeatureType__POI_CATEGORY);
      _sdsPhonemePOICategoryList.clear();
      _featureTypeRequestCounter--;
   }
   else if (_sdsPhonemePOIBrandNameList.size())
   {
      _currentRequestedFeatureType.enType = sds2hmi_fi_tcl_e8_NDSFeatureType::FI_EN_POI_ATTRIBUTE;
      sendNdsIdPhonemeRequest(_sdsPhonemePOIBrandNameList, NDSFeatureType__POI_ATTRIBUTE);
      _sdsPhonemePOIBrandNameList.clear();
      _featureTypeRequestCounter--;
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void NdsPhonemeRequestor::getLanguageListEntry(std::vector< LanguageEntry >& languageEntryList)
{
   LanguageEntry languageEntry;

   if (_sdsNdsLanguageId.size())
   {
      languageEntry.setISOLanguageCode(_sdsNdsLanguageId[0].ISO_LanguageCode);
      languageEntry.setISOCountryCode(_sdsNdsLanguageId[0].ISO_CountryCode);
      languageEntry.setTranscriptCode(_sdsNdsLanguageId[0].ScriptCode);
      languageEntryList.push_back(languageEntry);
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void NdsPhonemeRequestor::sendNdsIdPhonemeRequest(
   std::vector<sds2hmi_fi_tcl_NDSID>& objectIdList,
   NDSFeatureType ndsFeatureType)
{
   std::vector< GlobalNamedObjectId > namedObjectIds;
   std::vector< LanguageEntry > languageEntryList;
   std::vector<sds2hmi_fi_tcl_NDSID>::iterator it = objectIdList.begin();

   while (it != objectIdList.end())
   {
      GlobalNamedObjectId namedObjectIdList;
      GlobalUpdateRegionId globalUpdateRegionId;

      globalUpdateRegionId.setUpdateRegionId(it->u16UpdateRegionId);
      namedObjectIdList.setGlobalUpdateRegionId(globalUpdateRegionId);

      globalUpdateRegionId.setProductId(it->u32ProductId);
      namedObjectIdList.setGlobalUpdateRegionId(globalUpdateRegionId);

      globalUpdateRegionId.setSupplierId(it->u8SupplierId);
      namedObjectIdList.setGlobalUpdateRegionId(globalUpdateRegionId);

      namedObjectIdList.setNamedObjectId(it->u32NDSId);

      namedObjectIds.push_back(namedObjectIdList);

      ++it;
   }

   getLanguageListEntry(languageEntryList);

   _navigationSDSProxy->sendGetPhonemesForNDSIdsRequest(*this, namedObjectIds, languageEntryList, ndsFeatureType, true);
}


/***********************************************************************//**
 *
 ***************************************************************************/
void NdsPhonemeRequestor::onGetPhonemesForNDSIdsError(const ::boost::shared_ptr<NavigationSDSServiceProxy>& /*proxy*/,
      const ::boost::shared_ptr< GetPhonemesForNDSIdsError >& /*Error*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void NdsPhonemeRequestor::onGetPhonemesForNDSIdsResponse(const ::boost::shared_ptr<NavigationSDSServiceProxy >& /*proxy*/,
      const ::boost::shared_ptr< GetPhonemesForNDSIdsResponse >& getNamesForNDSIdsResponse)
{
   if (sds2hmi_fi_tcl_e8_NDSFeatureType::FI_EN_NAMEOBJECT == _currentRequestedFeatureType.enType)
   {
      _ndsPhonemeObject = getNamesForNDSIdsResponse->getNamedObjectIdStringsList();
   }
   else if (sds2hmi_fi_tcl_e8_NDSFeatureType::FI_EN_POINAME == _currentRequestedFeatureType.enType)
   {
      _ndsPhonemePOIName = getNamesForNDSIdsResponse->getNamedObjectIdStringsList();
   }
   else if (sds2hmi_fi_tcl_e8_NDSFeatureType::FI_EN_POI_CATEGORY == _currentRequestedFeatureType.enType)
   {
      _ndsPhonemePOICategory = getNamesForNDSIdsResponse->getNamedObjectIdStringsList();
   }
   else if (sds2hmi_fi_tcl_e8_NDSFeatureType::FI_EN_POI_ATTRIBUTE == _currentRequestedFeatureType.enType)
   {
      _ndsPhonemePOIAttribute = getNamesForNDSIdsResponse->getNamedObjectIdStringsList();
   }

   if (_featureTypeRequestCounter)
   {
      getPhonemeForNdsId();
   }
   else
   {
      notifyObjectIdPhonemeListObservers();
      _featureTypeRequestCounter = 0;
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void NdsPhonemeRequestor::notifyObjectIdPhonemeListObservers()
{
   std::vector<NdsPhonemeObserver*>::iterator it = _ndsPhonemeObservers.begin();

   while (it != _ndsPhonemeObservers.end())
   {
      if (*it != NULL)
      {
         (*it)->ndsPhonemesUpdated();
      }
      ++it;
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void NdsPhonemeRequestor::getNdsPhonemeObjectList(std::vector< NamedObjectIdStrings >& phonemeObjectList)
{
   phonemeObjectList = _ndsPhonemeObject;
}


/***********************************************************************//**
 *
 ***************************************************************************/
void NdsPhonemeRequestor::getNdsPhonemePOINameList(std::vector< NamedObjectIdStrings >& phonemePOINameList)
{
   phonemePOINameList = _ndsPhonemePOIName;
}


/***********************************************************************//**
 *
 ***************************************************************************/
void NdsPhonemeRequestor::getNdsPhonemePOICategoryList(std::vector< NamedObjectIdStrings >& phonemePOICategoryList)
{
   phonemePOICategoryList =  _ndsPhonemePOICategory;
}


/***********************************************************************//**
 *
 ***************************************************************************/
void NdsPhonemeRequestor::getNdsPhonemePOIAttributeList(std::vector< NamedObjectIdStrings >& phonemePOIAttributeList)
{
   phonemePOIAttributeList = _ndsPhonemePOIAttribute;
}


/***********************************************************************//**
 *
 ***************************************************************************/
void NdsPhonemeRequestor::clearAllPhonemeListData()
{
   _sdsPhonemeObjectList.clear();
   _sdsPhonemePOICategoryList.clear();
   _sdsPhonemePOIBrandNameList.clear();
   _sdsPhonemePOINameList.clear();

   _ndsPhonemeObject.clear();
   _ndsPhonemePOIName.clear();
   _ndsPhonemePOICategory.clear();
   _ndsPhonemePOIAttribute.clear();

   _featureTypeRequestCounter = 0;
}
