/**************************************************************************//**
 * \file       NaviVoiceTagHandler.cpp
 *
 * This file is part of the SdsAdapter component.
 * This file is part of the SDS HMI application.
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/

#include "NaviVoiceTagHandler.h"
#include "SdsAdapter_Trace.h"
#include "application/clSDS_SdsControl.h"
#include "application/clSDS_SDSStatus.h"
#include "application/GuiService.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SDSADP_DETAILS
#define NAVIUSERWORDPROFILE 1
#include "trcGenProj/Header/NaviVoiceTagHandler.cpp.trc.h"
#endif

#define UTFUTIL_S_IMPORT_INTERFACE_GENERIC
#include "utf_if.h"

using namespace org::bosch::cm::navigation::NavigationService;
using namespace sds_gui_fi::SdsGuiService;


/**************************************************************************//**
 * Constructor
 ******************************************************************************/
NaviVoiceTagHandler::NaviVoiceTagHandler(
   ::boost::shared_ptr<NavigationServiceProxy> pNaviProxy,
   clSDS_SdsControl* pSdsControl,
   clSDS_SDSStatus* pSDSStatus,
   GuiService& guiService)
   : _navigationProxy(pNaviProxy)
   , _pSdsControl(pSdsControl)
   , _pSDSStatus(pSDSStatus)
   , _pGuiService(guiService)
{
   _pSDSStatus->vRegisterObserver(this);
}


/**************************************************************************//**
 * Destructor
 ******************************************************************************/
NaviVoiceTagHandler::~NaviVoiceTagHandler()
{
   _pSdsControl = NULL;
   _pSDSStatus = NULL;
}


/***********************************************************************//**
*
***************************************************************************/
void NaviVoiceTagHandler::onAvailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _navigationProxy)
   {
      _navigationProxy->sendSdsVoiceTagWithOptionsRegister(*this);
   }
}


/***********************************************************************//**
*
***************************************************************************/
void NaviVoiceTagHandler::onUnavailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _navigationProxy)
   {
      _navigationProxy->sendSdsVoiceTagWithOptionsDeregisterAll();
   }
}


tVoid NaviVoiceTagHandler::onSdsVoiceTagWithOptionsError(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< SdsVoiceTagWithOptionsError>& /*error*/)
{
}


tVoid NaviVoiceTagHandler::onSdsVoiceTagWithOptionsUpdate(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< SdsVoiceTagWithOptionsUpdate >& update)
{
   _voiceTagID = update->getSdsVoiceTagWithOptions();
   ETG_TRACE_USR1(("Received Value = %d" , _voiceTagID.getOptions()));
   handleVoiceTagUpdate();
}


tVoid NaviVoiceTagHandler::onSdsUpdateVoiceTagError(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< SdsVoiceTagWithOptionsError>& /*error*/)
{
}


tVoid NaviVoiceTagHandler::onSdsUpdateVoiceTagResponse(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< SdsUpdateVoiceTagResponse >& /*response*/)
{
}


void NaviVoiceTagHandler::onSetLocationWithDestinationMemoryEntryError(
   const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationService::NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationService::SetLocationWithDestinationMemoryEntryError >& /*error*/)
{
}


void NaviVoiceTagHandler::onSetLocationWithDestinationMemoryEntryResponse(
   const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationService::NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationService::SetLocationWithDestinationMemoryEntryResponse >& /*response*/)
{
   _navigationProxy->sendRequestDetailsForSelectedLocationRequest(*this);
}


void NaviVoiceTagHandler::onRequestDetailsForSelectedLocationError(
   const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationService::NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationService::RequestDetailsForSelectedLocationError >& /*error*/)
{
}


void NaviVoiceTagHandler::onRequestDetailsForSelectedLocationResponse(
   const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationService::NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationService::RequestDetailsForSelectedLocationResponse >& /*response*/)
{
   _pGuiService.sendEventSignal(Event__SPEECH_DIALOG_GADGET_ACTIVATION);
}


/**
 * _pSdsControl->vSetUWProfile is meant to be called only once when SDS_Status IDLE is received from SDS_MW
 * to ensure this, setNaviProfiletoSDS flag was declared as static
 */

tVoid NaviVoiceTagHandler::vSDSStatusChanged()
{
   static tBool setNaviProfiletoSDS = 0;
   if (_pSDSStatus != NULL && _pSDSStatus->getStatus() == clSDS_SDSStatus::EN_IDLE && setNaviProfiletoSDS == 0)
   {
      setNaviProfiletoSDS = NAVIUSERWORDPROFILE;
      if (_pSdsControl != NULL)
      {
         _pSdsControl->vSetUWProfile(setNaviProfiletoSDS, sds2hmi_fi_tcl_e8_Domain ::FI_EN_NAVIGATION);
      }
   }
}


tVoid NaviVoiceTagHandler::updateNaviUserWordList(std::vector<tU32> naviUWs)
{
   _NaviUserWords = naviUWs;
   std::vector<tU32>::iterator itr;
   uint32 userid = _voiceTagID.getSdsVoiceTagId();
   if (_voiceTagID.getOptions() == SDSVoiceTagOptions__DELETE_VOICETAG)
   {
      for (itr = _NaviUserWords.begin(); itr != _NaviUserWords.end(); ++itr)
      {
         if (*itr == userid)
         {
            ETG_TRACE_USR1(("Userword Not Deleted %d" , userid));
         }
         else
         {
            _voiceTagID.setOptions(SDSVoiceTagOptions__VOICETAG_DELETED);
            _navigationProxy->sendSdsUpdateVoiceTagRequest(*this, _voiceTagID);
            deleteListofIDs(_voiceTagID.getSdsVoiceTagId());
         }
      }
   }
   else
   {
      if (_voiceTagID.getOptions() == SDSVoiceTagOptions__CREATE_NEW_VOICETAG)
      {
         for (itr = _NaviUserWords.begin(); itr != _NaviUserWords.end(); ++itr)
         {
            if (*itr == userid)
            {
               ETG_TRACE_USR1(("Userwords present in CommonSetAvailableUserwords %d" , userid));
               _voiceTagID.setOptions(SDSVoiceTagOptions__NEW_VOICETAG_CREATED);
               _navigationProxy->sendSdsUpdateVoiceTagRequest(*this, _voiceTagID);
               listofIDs.push_back(_voiceTagID);
            }
            else
            {
               ETG_TRACE_USR1(("Userword is not present in CommonSetAvailableUserwords %d" , userid));
               _voiceTagID.setOptions(SDSVoiceTagOptions__NO_VOICETAG_CREATED);
            }
         }
      }
   }
}


//TODO raa8hi - review this implementation
tVoid NaviVoiceTagHandler::handleVoiceTagUpdate()
{
   if (_pSDSStatus != NULL && _pSDSStatus->getStatus() == clSDS_SDSStatus::EN_IDLE_TTS_ONLY)
   {
      if (_voiceTagID.getOptions() == SDSVoiceTagOptions__CREATE_NEW_VOICETAG || _voiceTagID.getOptions() == SDSVoiceTagOptions__PLAY_VOICETAG)
      {
         _pGuiService.sendEventSignal(Event__SPEECH_DIALOG_SHOW_POPUP_LANGUAGE_NOT_SUPPORTED);
      }
      handleVoiceTagBasedOnSDSStatus();
   }

   else if (_pSDSStatus != NULL && _pSDSStatus->getStatus() == clSDS_SDSStatus::EN_LOADING)
   {
      _pGuiService.sendEventSignal(Event__SPEECH_DIALOG_SHOW_POPUP_LOADING_OPEN);
      handleVoiceTagBasedOnSDSStatus();
   }

   else
   {
      handleVoiceTagOptions();
   }
}


void NaviVoiceTagHandler::handleVoiceTagBasedOnSDSStatus()
{
   switch (_voiceTagID.getOptions())
   {
      case SDSVoiceTagOptions__CREATE_NEW_VOICETAG:
      {
         _voiceTagID.setSdsVoiceTagId(0);
         _voiceTagID.setOptions(SDSVoiceTagOptions__NO_VOICETAG_CREATED);
         _navigationProxy->sendSdsUpdateVoiceTagRequest(*this, _voiceTagID);
      }
      break;
      case SDSVoiceTagOptions__DELETE_VOICETAG:
      {
         if (_pSdsControl != NULL)
         {
            _pSdsControl->vDeleteUserWord(_voiceTagID.getSdsVoiceTagId(), sds2hmi_fi_tcl_e8_Domain ::FI_EN_NAVIGATION);
         }
      }
      break;
      case SDSVoiceTagOptions__DELETE_ALL_VOICETAGS:
      {
         if (_pSdsControl != NULL)
         {
            _pSdsControl->vDeleteAllUserWords(NAVIUSERWORDPROFILE, sds2hmi_fi_tcl_e8_Domain ::FI_EN_NAVIGATION);
         }
      }
      break;
      default:
         break;
   }
}


void NaviVoiceTagHandler::handleVoiceTagOptions()
{
   switch (_voiceTagID.getOptions())
   {
      case SDSVoiceTagOptions__CREATE_NEW_VOICETAG:
      {
         createNewVoiceTag();
      }
      break;

      case SDSVoiceTagOptions__PLAY_VOICETAG:
      {
         if (_pSdsControl != NULL)
         {
            _pSdsControl->playUserWord(_voiceTagID.getSdsVoiceTagId(), sds2hmi_fi_tcl_e8_Domain ::FI_EN_NAVIGATION);
         }
      }
      break;

      case SDSVoiceTagOptions__DELETE_VOICETAG:
      {
         if (_pSdsControl != NULL)
         {
            _pSdsControl->vDeleteUserWord(_voiceTagID.getSdsVoiceTagId(), sds2hmi_fi_tcl_e8_Domain ::FI_EN_NAVIGATION);
         }
      }
      break;

      case SDSVoiceTagOptions__DELETE_ALL_VOICETAGS:
      {
         if (_pSdsControl != NULL)
         {
            _pSdsControl->vDeleteAllUserWords(NAVIUSERWORDPROFILE, sds2hmi_fi_tcl_e8_Domain ::FI_EN_NAVIGATION);
         }
      }
      break;
      default:
         break;
   }
}


tVoid NaviVoiceTagHandler::createNewVoiceTag()
{
   uint32 sdsVoiceTagId = _voiceTagID.getSdsVoiceTagId();
   if (sdsVoiceTagId != 0)
   {
      if (_pSdsControl != NULL)

      {
         _pSdsControl->replaceUserWord(sdsVoiceTagId, sds2hmi_fi_tcl_e8_Domain::FI_EN_NAVIGATION);
      }
   }
   else
   {
      if (_pSdsControl != NULL)
      {
         sdsVoiceTagId = generateUniqueVoiceTagID();
         _pSdsControl->recordUserWord(sdsVoiceTagId, sds2hmi_fi_tcl_e8_Domain::FI_EN_NAVIGATION);
      }
   }
   _voiceTagID.setSdsVoiceTagId(sdsVoiceTagId);
}


uint32 NaviVoiceTagHandler::generateUniqueVoiceTagID()
{
   tU32 voiceTagID = NAVIUSERWORDSTART;
   if (_NaviUserWords.size() != 0)
   {
      std::vector<tU32>::iterator result;
      result = std::max_element(_NaviUserWords.begin(), _NaviUserWords.end());
      voiceTagID = *(result);
      ++voiceTagID;
   }
   return voiceTagID;
}


std::string NaviVoiceTagHandler::getDestinationMemoryName(tU32 userwordID)
{
   if (_voiceTagID.getSdsVoiceTagId() == userwordID)
   {
      return _voiceTagID.getDestinationMemoryEntryName();
   }
   return "";
}


void NaviVoiceTagHandler::setVoiceTagMemoryID(tU32 voicetagID)
{
   std::vector<org::bosch::cm::navigation::NavigationService::SDSVoiceTagId>::iterator it;
   for (it = listofIDs.begin(); it != listofIDs.end(); ++it)
   {
      if (it->getSdsVoiceTagId() == voicetagID)
      {
         _navigationProxy->sendSetLocationWithDestinationMemoryEntryRequest(*this, (uint32)it->getDestinationMemoryEntryId());
         break;
      }
   }
}


void NaviVoiceTagHandler::deleteListofIDs(tU32 voicetagID)
{
   std::vector<org::bosch::cm::navigation::NavigationService::SDSVoiceTagId>::iterator it;
   for (it = listofIDs.begin(); it != listofIDs.end(); it++)
   {
      if (it->getSdsVoiceTagId() == voicetagID)
      {
         listofIDs.erase(it);
         break;
      }
   }
}
