/**************************************************************************//**
 * \file       NaviPOICategoryList.cpp
 *
 * This file is part of the SdsAdapter component.
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "application/NaviPOICategoryList.h"
#include "application/clSDS_StringVarList.h"
#include "view_db/Sds_ViewDB.h"
#include "SdsAdapter_Trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/NaviPOICategoryList.cpp.trc.h"
#endif


using namespace org::bosch::cm::navigation::NavigationService;


/**************************************************************************//**
*Destructor
******************************************************************************/
NaviPOICategoryList::~NaviPOICategoryList()
{
}


/**************************************************************************//**
*Constructor
******************************************************************************/
NaviPOICategoryList::NaviPOICategoryList(::boost::shared_ptr< NavigationServiceProxy > naviProxy): _naviProxy(naviProxy)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void NaviPOICategoryList::onAvailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _naviProxy)
   {
      _naviProxy->sendMenuPoiCategoryShortcutsRegister(*this);
      _naviProxy->sendMenuPoiCategoryShortcutsGet(*this);
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void NaviPOICategoryList::onUnavailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _naviProxy)
   {
      _naviProxy->sendMenuPoiCategoryShortcutsDeregisterAll();
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void NaviPOICategoryList::onMenuPoiCategoryShortcutsError(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationService::MenuPoiCategoryShortcutsError >& /*error*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void NaviPOICategoryList::onMenuPoiCategoryShortcutsUpdate(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationService::MenuPoiCategoryShortcutsUpdate >& update)
{
   if (update->hasMenuPoiCategoryShortcuts())
   {
      _sdsPOICategory.clear();
      _sdsPOICategory = update->getMenuPoiCategoryShortcuts();
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
std::vector<sds2hmi_fi_tcl_HMIElementDescription> NaviPOICategoryList::getHmiElementDescription(unsigned int index)
{
   std::vector<sds2hmi_fi_tcl_HMIElementDescription> hmiElementDescriptionList;
   sds2hmi_fi_tcl_HMIElementDescription hmiElementDescription;
   if (index > 0 && getCommandString(index - 1) != "")
   {
      hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_POICATEGORY;
      hmiElementDescription.DescriptorValue.bSet(getCommandString(index - 1).c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
      hmiElementDescription.DestinationIdList.push_back(getDestinationIdList(_sdsPOICategory[index - 1].getSdsNameObjectId()));

      hmiElementDescriptionList.push_back(hmiElementDescription);
   }
   return hmiElementDescriptionList;
}


/***********************************************************************//**
 *
 ***************************************************************************/
std::string NaviPOICategoryList::getCommandString(tU32 index)
{
   if (index < _sdsPOICategory.size())
   {
      return _sdsPOICategory[index].getCategoryName();
   }
   return "";
}


/***********************************************************************//**
 *
 ***************************************************************************/

tU32 NaviPOICategoryList::u32GetSize()
{
   return _sdsPOICategory.size();
}


/***********************************************************************//**
 *
 ***************************************************************************/

std::vector<clSDS_ListItems> NaviPOICategoryList::oGetItems(tU32 u32StartIndex, tU32 u32EndIndex)
{
   std::vector<clSDS_ListItems> oListItems;
   unsigned int minVal = std::min(u32EndIndex, u32GetSize());
   for (tU32 u32Index = u32StartIndex; u32Index < minVal; u32Index++)
   {
      clSDS_ListItems oListItem;
      oListItem.oCommand.szString = oGetItem(u32Index);
      oListItem.bIsListScreenWithoutIndex = TRUE;
      oListItems.push_back(oListItem);
   }
   return oListItems;
}


/**************************************************************************//**
*
******************************************************************************/
std::string NaviPOICategoryList::oGetItem(tU32 u32Index)
{
   if (u32Index < (_sdsPOICategory.size()))
   {
      return _sdsPOICategory[u32Index].getCategoryName();
   }
   else
   {
      return "";
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/

tBool NaviPOICategoryList::bSelectElement(tU32 u32SelectedIndex)
{
   return ((u32SelectedIndex > 0) && (u32SelectedIndex <= _sdsPOICategory.size()));
}


/***********************************************************************//**
 *
 ***************************************************************************/

std::vector<sds2hmi_fi_tcl_HMIElementDescrptionList> NaviPOICategoryList::getHmiListDescription()
{
   std::vector<sds2hmi_fi_tcl_HMIElementDescrptionList> poiCategoryList;

   for (size_t itr = 0; itr < _sdsPOICategory.size(); ++itr)
   {
      sds2hmi_fi_tcl_HMIElementDescription hmiElementDescription;
      sds2hmi_fi_tcl_HMIElementDescrptionList hmiElementDescriptionList;
      if (!_sdsPOICategory[itr].getCategoryName().empty())
      {
         hmiElementDescription.DescriptorTag.enType = sds2hmi_fi_tcl_e8_WordType::FI_EN_POICATEGORY;
         hmiElementDescription.DescriptorValue.bSet(_sdsPOICategory[itr].getCategoryName().c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
         hmiElementDescription.DestinationIdList.push_back(getDestinationIdList(_sdsPOICategory[itr].getSdsNameObjectId()));

         hmiElementDescriptionList.DescriptionList.push_back(hmiElementDescription);
      }
      poiCategoryList.push_back(hmiElementDescriptionList);
   }
   return poiCategoryList;
}


/***********************************************************************//**
 *
 **************************************************************************/

uint32 NaviPOICategoryList::getPOICategoryId(std::string categoryName)
{
   std::vector< PoiCategoryShortcut >::iterator it1;
   for (it1 = _sdsPOICategory.begin(); it1 != _sdsPOICategory.end(); ++it1)
   {
      if (it1->getCategoryName() == categoryName)
      {
         return it1->getPoiCategoryId();
      }
   }
   return 0;
}


sds2hmi_fi_tcl_NDSID NaviPOICategoryList::getDestinationIdList(GlobalNamedObjectId nameObjectId)
{
   sds2hmi_fi_tcl_NDSID dataNDSId;
   if (nameObjectId.getIsValid())
   {
      dataNDSId.u16UpdateRegionId = nameObjectId.getGlobalUpdateRegionId().getUpdateRegionId();
      dataNDSId.u32ProductId = nameObjectId.getGlobalUpdateRegionId().getProductId();
      dataNDSId.u8SupplierId = nameObjectId.getGlobalUpdateRegionId().getSupplierId();
      dataNDSId.u32NDSId = nameObjectId.getNamedObjectId();
      dataNDSId.eNDSFeatureType.enType = sds2hmi_fi_tcl_e8_NDSFeatureType::FI_EN_POI_CATEGORY;
   }
   return dataNDSId;
}
