/**************************************************************************//**
 * \file       NaviHapticHandOverInterface.cpp
 *
 * This file is part of the SdsAdapter component.
 *
 * \copyright  (C) 2017 Robert Bosch GmbH
 *             (C) 2017 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/

#include "application/NaviHapticHandOverInterface.h"
#include "Sds2HmiServer/functions/clSDS_Property_CommonActionRequest.h"
#include "application/clSDS_POIList.h"
#include "application/clSDS_MultipleDestinationsList.h"
#include "application/NaviHapticHandOverList.h"
#include "application/clSDS_NaviListItems.h"


using namespace org::bosch::cm::navigation::NavigationSDSService;


#include "SdsAdapter_Trace.h"
#define FI_S_IMPORT_INTERFACE_FI_TYPES
#include "fi_if.h"
#include "view_db/Sds_ViewDB.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/NaviHapticHandOverInterface.cpp.trc.h"
#endif

int HAND_OVER_DATA_SINGLE = 1;
int HAND_OVER_DATA_LIST_TWO = 2;
int HAND_OVER_DATA_LIST_SIX = 6;
int HAND_OVER_DATA_LIST_FIVE = 5;
int ADDITIONAL_INFO_LIST_COUNT = 1;
std::string INVALID_SCREEN = "NAVI_INACTIVE_SCREEN";
std::string HANDOVER_DATA_NONE = "NONE";
std::string HANDOVER_DATA_SINGLE = "SINGLE";
std::string HANDOVER_DATA_LIST = "LIST";
std::string HANDOVER_DATA_MULTIPLE = "MULTIPLE";

typedef std::vector<org::bosch::cm::navigation::NavigationSDSService::GlobalNamedObjectId> DataNameObjectList;

/**************************************************************************//**
 * Constructor
 ******************************************************************************/
NaviHapticHandOverInterface::NaviHapticHandOverInterface(
   ::boost::shared_ptr<NavigationSDSServiceProxy > naviProxy, clSDS_Property_CommonActionRequest* pCommonActionRequest,
   clSDS_MultipleDestinationsList* pMultipleDestinationList,
   clSDS_POIList* pPOIList,
   NaviPOICategoryList* pNaviPOICategoryList,
   NaviHapticHandOverList* pNaviHapticHandOverList,
   clSDS_NaviListItems* pNaviListItems):
   _naviSDSProxy(naviProxy)
   , _pPropertyCommonActionRequest(pCommonActionRequest)
   , _pMultipleDestinationList(pMultipleDestinationList)
   , _pPOIList(pPOIList)
   , _pNaviListItems(pNaviListItems)
   , _pNaviPOICategoryList(pNaviPOICategoryList)
   , _pNaviHapticHandOverList(pNaviHapticHandOverList)
   , _isNaviHandOverScreenActive(false)
   , _locationType(LocationType__DEFAULT)
   , _isPOISearch(false)
   , _POIListViaHaptic(false)
{
   mapHmiToSdsEntryPoint();
}


/**************************************************************************//**
 *Destructor
 ******************************************************************************/
NaviHapticHandOverInterface::~NaviHapticHandOverInterface()
{
   _pMultipleDestinationList = NULL;
   _pPOIList = NULL;
   _pNaviPOICategoryList = NULL;
   _pNaviHapticHandOverList =  NULL;
   _pNaviListItems = NULL;
}


/**************************************************************************//**
 *
 ******************************************************************************/
void NaviHapticHandOverInterface::onAvailable(const boost::shared_ptr<asf::core::Proxy>& proxy,
      const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _naviSDSProxy)
   {
      _naviSDSProxy->sendHandOverSessionStatusRegister(*this);
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void NaviHapticHandOverInterface::onUnavailable(const boost::shared_ptr<asf::core::Proxy>& proxy,
      const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _naviSDSProxy)
   {
      _naviSDSProxy->sendHandOverSessionStatusDeregisterAll();
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void NaviHapticHandOverInterface::onHandOverSessionStatusUpdate(const ::boost::shared_ptr< NavigationSDSServiceProxy >& proxy, const ::boost::shared_ptr< HandOverSessionStatusUpdate >& update)
{
   if (update->getHandOverSessionStatus().getIsHandOverScreenActive())
   {
      _isNaviHandOverScreenActive = true;
   }
   else
   {
      _isNaviHandOverScreenActive = false;
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void NaviHapticHandOverInterface::onHandOverSessionStatusError(const ::boost::shared_ptr< NavigationSDSServiceProxy >& proxy, const ::boost::shared_ptr< HandOverSessionStatusError >& error)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void NaviHapticHandOverInterface::onGetHapticHandOverDataError(const ::boost::shared_ptr< NavigationSDSServiceProxy >& proxy, const ::boost::shared_ptr< GetHapticHandOverDataError >& error)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void NaviHapticHandOverInterface::onGetHapticHandOverDataResponse(const ::boost::shared_ptr< NavigationSDSServiceProxy >& proxy, const ::boost::shared_ptr< GetHapticHandOverDataResponse >& response)
{
   ETG_TRACE_USR4(("NaviHapticHandOverInterface::onGetHapticHandOverDataResponse entry"));
   SDSHandOverScreenId hmiScreenId = response->getSdsHandOverScreenId();
   resetHandOverData();

   std::vector<HapticData> hapticDataList;
   hapticDataList = response->getDataList();
   _addressWithIdInfo = response->getSearchResults();
   _selectedDataList = response->getSelectedDataListItems();
   _locationType = response->getLocationType();
   _handOverOption = response->getSdsHandOverOptionsList();

   // remove ambigous list addresses provided by Navi
   std::vector<HapticData>::iterator it;
   for (it = hapticDataList.begin(); it != hapticDataList.end(); ++it)
   {
      _dataList.push_back(it->getData());
      setDestinationIdList(it->getSdsNameObjectId());
   }
   _handOverType = calculateHandOverType(_dataList.size());

   _uniqueList = removeDuplicateDataListItems(_dataList);

   if (hmiScreenId == org::bosch::cm::navigation::NavigationSDSService::SDSHandOverScreenId__NAVI_FORM_WITHOUT_STATE)
   {
      findEntryPointforEURformScreen(_selectedDataList, hmiScreenId);
   }

   getHandOverDataFromDB(hmiScreenId, _handOverType, _handOverSdsScreen);

   //Navi hall will send the response of GetHapticHandOverData method.The below function will read required SDS screen from HandOverDB.dat file based on this response.

   if (_handOverSdsScreen.compare(INVALID_SCREEN) != 0)
   {
      if (_pNaviListItems)
      {
         _pNaviListItems->vSetHapticHandOverListType();
      }
      handleNaviHandOverPTTRequest(_handOverSdsScreen, _selectedDataList, _handOverType, _handOverOption);
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void NaviHapticHandOverInterface::onSelectHapticHandOverListError(const ::boost::shared_ptr< NavigationSDSServiceProxy >& proxy, const ::boost::shared_ptr< SelectHapticHandOverListError >& error)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void NaviHapticHandOverInterface::onSelectHapticHandOverListResponse(const ::boost::shared_ptr< NavigationSDSServiceProxy >& proxy, const ::boost::shared_ptr< SelectHapticHandOverListResponse >& response)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void NaviHapticHandOverInterface::mapHmiToSdsEntryPoint()
{
   _handOverScreenInfoMap["SDS_ENTRY_NAVI_ENTER_ADDRESS"] = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_ENTER_ADDRESS;
   _handOverScreenInfoMap["SDS_ENTRY_NAVI_ENTER_POI"] = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_ENTER_POI;

   _handOverScreenInfoMap["SDS_ENTRY_NAVI_ADDRS_STEP_ENTER_COUNTRY"] = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_ADRS_STEP_ENTER_COUNTRY;
   _handOverScreenInfoMap["SDS_ENTRY_NAVI_ADDRS_STEP_ENTER_STATE"] = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_ADDRS_STEP_ENTER_STATE;
   _handOverScreenInfoMap["SDS_ENTRY_NAVI_ADDRS_STEP_HNOFIRST_HNO"] = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_ADDRS_STEP_HNOFIRST_HNO;
   _handOverScreenInfoMap["SDS_ENTRY_NAVI_ADDRS_STEP_HNOFIRST_STREET"] = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_ADDRS_STEP_HNOFIRST_STREET;
   _handOverScreenInfoMap["SDS_ENTRY_NAVI_ADDRS_STEP_HNOFIRST_STREET_CITY"] = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_ADDRS_STEP_HNOFIRST_STREET_CITY;
   _handOverScreenInfoMap["SDS_ENTRY_NAVI_STEP_CITYFIRST_CITY"] = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_STEP_CITYFIRST_CITY;
   _handOverScreenInfoMap["SDS_ENTRY_NAVI_STEP_CITYFIRST_HNO"] = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_STEP_CITYFIRST_HNO;
   _handOverScreenInfoMap["SDS_ENTRY_NAVI_STEP_CITYFIRST_HNO_STREET"] = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_STEP_CITYFIRST_HNO_STREET;

   if (clSDS_KDSConfiguration::JPN == clSDS_KDSConfiguration::getMarketRegionType())
   {
      _handOverScreenInfoMap["SDS_ENTRY_NAVI_ADDRS_STEP_CITYFIRST_CITY_STREET"] = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_ADDRS_STEP_OAZA;
   }
   else
   {
      _handOverScreenInfoMap["SDS_ENTRY_NAVI_ADDRS_STEP_CITYFIRST_CITY_STREET"] = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_ADDRS_STEP_CITYFIRST_STREET;
   }

   _handOverScreenInfoMap["SDS_ENTRY_NAVI_ADDRS_STEP_CITYFIRST_STREET_HNO"] = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_ADDRS_STEP_CITYFIRST_STREET_HNO;
   _handOverScreenInfoMap["SDS_ENTRY_NAVI_AMBIG_ADDRESS"] = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_AMBIG_ADDRESS;
   _handOverScreenInfoMap["SDS_ENTRY_NAVI_SET_DEST_ADDRERSS"] = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_SET_DEST_ADDRERSS;

   _handOverScreenInfoMap["SDS_ENTRY_NAVI_CITYCENTER_CITY"] = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_CITYCENTER_CITY;
   _handOverScreenInfoMap["SDS_ENTRY_NAVI_CITYCENTER_STATE"] = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_CITYCENTER_STEP_ENTER_STATE;
   _handOverScreenInfoMap["SDS_ENTRY_NAVI_AMBIG_CITY_CENTER"] = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_AMBIG_CITY_CENTER;
   _handOverScreenInfoMap["SDS_ENTRY_NAVI_SET_DEST_CITY_CENTER"] = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_SET_DEST_CITY_CENTER;
   _handOverScreenInfoMap["SDS_ENTRY_NAVI_INTERSECTION_STRT1"] = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_INTERSECTION_STRT1;
   _handOverScreenInfoMap["SDS_ENTRY_NAVI_INTERSECTION_STRT2"] = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_INTERSECTION_STRT2;
   _handOverScreenInfoMap["SDS_ENTRY_NAVI_INTERSECT_STEP_ENTER_CITY"] = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_INTERSECT_STEP_ENTER_CITY;
   _handOverScreenInfoMap["SDS_ENTRY_NAVI_INTERSECT_STEP_ENTER_STATE"] = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_INTERSECT_STEP_ENTER_STATE;
   _handOverScreenInfoMap["SDS_ENTRY_NAVI_INTERSECT_STEP_ENTER_COUNTRY"] = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_INTERSECT_STEP_ENTER_COUNTRY;
   _handOverScreenInfoMap["SDS_ENTRY_NAVI_ENTER_INTERSECTION"] = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_ENTER_INTERSECTION;
   _handOverScreenInfoMap["SDS_ENTRY_NAVI_SET_DEST_INTERSECTION"] = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_SET_DEST_INTERSECTION;

   _handOverScreenInfoMap["SDS_ENTRY_NAVI_ENTER_POI"] = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_ENTER_POI;
   _handOverScreenInfoMap["SDS_ENTRY_NAVI_ENTER_POICAT"] = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_ENTER_POICAT;
   _handOverScreenInfoMap["POI_LIST"] = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_ENTER_POI_IN_CITY_TAB;
   _handOverScreenInfoMap["SDS_ENTRY_NAVI_ENTER_POI_ALONG_ROUTE_TAB"] = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_ENTER_POI_ALONG_ROUTE_TAB;
   _handOverScreenInfoMap["SDS_ENTRY_NAVI_ENTER_POI_NEAR_DEST_TAB"] = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_ENTER_POI_NEAR_DEST_TAB;
   _handOverScreenInfoMap["SDS_ENTRY_NAVI_ENTER_POI_NEARBY_TAB"] = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_ENTER_POI_NEARBY_TAB;
   _handOverScreenInfoMap["SDS_ENTRY_NAVI_POI_IN_CITY"] = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_POI_IN_CITY;
   _handOverScreenInfoMap["SDS_ENTRY_NAVI_POI_STEP_ENTER_STATE"] = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_POI_STEP_ENTER_STATE;
   _handOverScreenInfoMap["SDS_ENTRY_NAVI_POI_STEP_ENTER_COUNTRY"] = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_POI_STEP_ENTER_COUNTRY;
   _handOverScreenInfoMap["SDS_ENTRY_NAVI_SET_DEST_POI"] = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_SET_DEST_POI;

   _handOverScreenInfoMap["SDS_ENTRY_NAVI"] = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI;
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool NaviHapticHandOverInterface::isHandOverSession() const
{
   return _isNaviHandOverScreenActive;
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool NaviHapticHandOverInterface::isPOIListViaHaptic() const
{
   return _POIListViaHaptic;
}


/**************************************************************************//**
 *
 ******************************************************************************/
void NaviHapticHandOverInterface::resetPOIListViaHaptic()
{
   _POIListViaHaptic = false;
}


/**************************************************************************//**
 *
 ******************************************************************************/
void NaviHapticHandOverInterface::sendNaviHandOverPTTRequest()
{
   ETG_TRACE_USR4(("NaviHapticHandOverInterface::sendNaviHandOverPTTrequest entered"));
   if (_naviSDSProxy)
   {
      _naviSDSProxy->sendGetHapticHandOverDataRequest(*this);
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool NaviHapticHandOverInterface::findSdsEntryPointFromNaviInput(const std::string& ScreenID, sds2hmi_fi_tcl_e8_SDS_EntryPoint::tenType& sdsScreen)
{
   ETG_TRACE_USR4(("NaviHapticHandOverInterface::findSdsScreenFromNaviInput entry %s", ScreenID.c_str()));
   bool sdsScreenFoundFromMap = false;
   if (0 < _handOverScreenInfoMap.size())
   {
      std::map<std::string, sds2hmi_fi_tcl_e8_SDS_EntryPoint::tenType >::iterator mapIterator =
         _handOverScreenInfoMap.find(ScreenID);
      if (_handOverScreenInfoMap.end() != mapIterator)
      {
         sdsScreen = mapIterator->second;
         sdsScreenFoundFromMap = true;
      }
      else
      {
         ETG_TRACE_USR4(("NaviHapticHandOverInterface::findSdsScreenFromNaviInput handOverScreenInfo empty"));
      }
   }
   return sdsScreenFoundFromMap;
}


/**************************************************************************//**
 *
 ******************************************************************************/
void NaviHapticHandOverInterface::handleNaviHandOverPTTRequest(const std::string& handOverSdsScreen , std::vector<SelectedDataItem> selectedDataList, const std::string& handOverType, std::vector<SDSHandOverOptions> handOverOption)
{
   sds2hmi_fi_tcl_e8_SDS_EntryPoint::tenType sdsScreen;

   if (!findSdsEntryPointFromNaviInput(handOverSdsScreen, sdsScreen))
   {
      return;
   }

   if (sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_ENTER_POI_IN_CITY_TAB == sdsScreen)
   {
      _isPOISearch = true;
      findPOIScreen(sdsScreen);
   }

   if (sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_POI_IN_CITY == sdsScreen && sds2hmi_fi_tcl_e8_Data_Exchange_Type::FI_EN_DATA_LIST == getHandOverType(handOverType))
   {
      _isPOISearch = true;
      sdsScreen = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_POI_IN_CITY;
   }

   size_t selectedDatalistCount = selectedDataList.size();
   if (sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_ADDRS_STEP_ENTER_STATE == sdsScreen && sds2hmi_fi_tcl_e8_Data_Exchange_Type::FI_EN_DATA_NONE == getHandOverType(handOverType) && selectedDatalistCount == 2)
   {
      sdsScreen = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_STEP_CITYFIRST_CITY;
   }

   sds2hmi_sdsfi_tclMsgCommonActionRequestStatus oMessage;
   oMessage.Action.enType = sds2hmi_fi_tcl_e8_Action::FI_EN_PTT_SHORT_PRESS;
   oMessage.Value = sdsScreen;

   sds2hmi_fi_tcl_Action_AdditionalInfo additionalInfo;

   additionalInfo.ActionInfoCategory.enType = sds2hmi_fi_tcl_e8_ActionInfoCategory::FI_EN_CATEGORY_DATA_EXCHANGE;
   additionalInfo.InfoValue = getHandOverType(handOverType);
   oMessage.AdditionalInfoList.push_back(additionalInfo);

   std::vector<SelectedDataItem>::const_iterator selectedDataIte;
   for (selectedDataIte = selectedDataList.begin(); selectedDataIte != selectedDataList.end(); ++selectedDataIte)
   {
      additionalInfo = sds2hmi_fi_tcl_Action_AdditionalInfo();

      if (!selectedDataIte->getData().empty())
      {
         additionalInfo.ActionInfoCategory.enType = sds2hmi_fi_tcl_e8_ActionInfoCategory::FI_EN_CATEGORY_DATA_DESCRPTION;
         if (HMISelectedAddressElementType__COUNTRY == selectedDataIte->getHmiSelectedAddressElementType())
         {
            additionalInfo.InfoValue = static_cast<uint32>(sds2hmi_fi_tcl_e8_WordType::FI_EN_COUNTRY);
         }
         else if (HMISelectedAddressElementType__STATE == selectedDataIte->getHmiSelectedAddressElementType())
         {
            if (true == clSDS_ConfigurationFlags::getNavStateAvailable())
            {
               additionalInfo.InfoValue = static_cast<uint32>(sds2hmi_fi_tcl_e8_WordType::FI_EN_STATE);
            }
         }
         else if (HMISelectedAddressElementType__CITY == selectedDataIte->getHmiSelectedAddressElementType())
         {
            additionalInfo.InfoValue = static_cast<uint32>(sds2hmi_fi_tcl_e8_WordType::FI_EN_CITY);
         }
         else if (HMISelectedAddressElementType__STREET == selectedDataIte->getHmiSelectedAddressElementType())
         {
            if (clSDS_KDSConfiguration::JPN == clSDS_KDSConfiguration::getMarketRegionType())
            {
               additionalInfo.InfoValue = static_cast<uint32>(sds2hmi_fi_tcl_e8_WordType::FI_EN_OAZA);
            }
            else
            {
               additionalInfo.InfoValue = static_cast<uint32>(sds2hmi_fi_tcl_e8_WordType::FI_EN_STREET);
            }
         }
         else if (HMISelectedAddressElementType__HOUSENUMBER == selectedDataIte->getHmiSelectedAddressElementType())
         {
            additionalInfo.InfoValue = static_cast<uint32>(sds2hmi_fi_tcl_e8_WordType::FI_EN_HOUSENUM);
         }
         else if (HMISelectedAddressElementType__JUNCTION == selectedDataIte->getHmiSelectedAddressElementType())
         {
            additionalInfo.InfoValue = static_cast<uint32>(sds2hmi_fi_tcl_e8_WordType::FI_EN_STREET2);
         }
         else if (HMISelectedAddressElementType__POI_CATEGORY == selectedDataIte->getHmiSelectedAddressElementType())
         {
            additionalInfo.InfoValue = static_cast<uint32>(sds2hmi_fi_tcl_e8_WordType::FI_EN_POICATEGORY);
         }
         else if (HMISelectedAddressElementType__POI_BRAND == selectedDataIte->getHmiSelectedAddressElementType())
         {
            additionalInfo.InfoValue = static_cast<uint32>(sds2hmi_fi_tcl_e8_WordType::FI_EN_POIBRAND);
         }
         else if (HMISelectedAddressElementType__POI_NAME == selectedDataIte->getHmiSelectedAddressElementType())
         {
            additionalInfo.InfoValue = static_cast<uint32>(sds2hmi_fi_tcl_e8_WordType::FI_EN_POINAME);
         }

         additionalInfo.InfoString.bSet(selectedDataIte->getData().c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);

         GlobalNamedObjectId nameObjectId = selectedDataIte->getSdsGlobalNameObjectId();
         sds2hmi_fi_tcl_NDSID selectedDataNDSId = sds2hmi_fi_tcl_NDSID();

         if (nameObjectId.getIsValid())
         {
            selectedDataNDSId.u16UpdateRegionId = nameObjectId.getGlobalUpdateRegionId().getUpdateRegionId();
            selectedDataNDSId.u32ProductId = nameObjectId.getGlobalUpdateRegionId().getProductId();
            selectedDataNDSId.u8SupplierId = nameObjectId.getGlobalUpdateRegionId().getSupplierId();
            selectedDataNDSId.u32NDSId = nameObjectId.getNamedObjectId();
            additionalInfo.DestinationIdList.push_back(selectedDataNDSId);
         }

         oMessage.AdditionalInfoList.push_back(additionalInfo);
      }

      else if (HMISelectedAddressElementType__HOUSENUMBER == selectedDataIte->getHmiSelectedAddressElementType())
      {
         additionalInfo = sds2hmi_fi_tcl_Action_AdditionalInfo();
         additionalInfo.ActionInfoCategory.enType = sds2hmi_fi_tcl_e8_ActionInfoCategory::FI_EN_CATEGORY_DATA_DESCRPTION;
         additionalInfo.InfoValue = static_cast<uint32>(sds2hmi_fi_tcl_e8_WordType::FI_EN_HOUSENUM);
         oMessage.AdditionalInfoList.push_back(additionalInfo);
      }
   }

   if (HANDOVER_DATA_SINGLE == handOverType)
   {
      additionalInfo = sds2hmi_fi_tcl_Action_AdditionalInfo();
      additionalInfo.ActionInfoCategory.enType = sds2hmi_fi_tcl_e8_ActionInfoCategory::FI_EN_CATEGORY_DATA_DESCRPTION;

      switch (sdsScreen)
      {
         case sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_ADDRS_STEP_ENTER_STATE:
            additionalInfo.InfoValue = static_cast<uint32>(sds2hmi_fi_tcl_e8_WordType::FI_EN_COUNTRY);
            break;

         case sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_ADDRS_STEP_HNOFIRST_STREET_CITY:
            additionalInfo.InfoValue = static_cast<uint32>(sds2hmi_fi_tcl_e8_WordType::FI_EN_CITY);
            break;

         case sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_CITYCENTER_CITY:
            additionalInfo.InfoValue = static_cast<uint32>(sds2hmi_fi_tcl_e8_WordType::FI_EN_CITY);
            break;

         case sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_INTERSECTION_STRT2:
            additionalInfo.InfoValue = static_cast<uint32>(sds2hmi_fi_tcl_e8_WordType::FI_EN_STREET2);
            break;

         case sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_INTERSECT_STEP_ENTER_STATE:
            additionalInfo.InfoValue = static_cast<uint32>(sds2hmi_fi_tcl_e8_WordType::FI_EN_STATE);
            break;

         case sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_INTERSECT_STEP_ENTER_CITY:
            additionalInfo.InfoValue = static_cast<uint32>(sds2hmi_fi_tcl_e8_WordType::FI_EN_CITY);
            break;

         case sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_INTERSECTION_STRT1:
            additionalInfo.InfoValue = static_cast<uint32>(sds2hmi_fi_tcl_e8_WordType::FI_EN_STREET);
            break;

         case sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_CITYCENTER_STEP_ENTER_STATE:
            additionalInfo.InfoValue = static_cast<uint32>(sds2hmi_fi_tcl_e8_WordType::FI_EN_STATE);
            break;

         case sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_ADDRS_STEP_CITYFIRST_STREET:
            additionalInfo.InfoValue = static_cast<uint32>(sds2hmi_fi_tcl_e8_WordType::FI_EN_STREET);
            break;

         case sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_STEP_CITYFIRST_HNO_STREET:
            additionalInfo.InfoValue = static_cast<uint32>(sds2hmi_fi_tcl_e8_WordType::FI_EN_STREET);
            break;

         case sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_STEP_CITYFIRST_CITY:
            additionalInfo.InfoValue = static_cast<uint32>(sds2hmi_fi_tcl_e8_WordType::FI_EN_CITY);
            break;

         default:
            break;
      }

      if (additionalInfo.InfoValue)
      {
         additionalInfo.InfoString.bSet(_uniqueList[0].c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
         additionalInfo.DestinationIdList = _dataNDSIdList;
         oMessage.AdditionalInfoList.push_back(additionalInfo);
      }
   }

   if (sds2hmi_fi_tcl_e8_Data_Exchange_Type::FI_EN_DATA_LIST == getHandOverType(handOverType) || _isPOISearch)
   {
      setHandOverHMIListContents(sdsScreen);
      setHandOverDataListItems(sdsScreen);
      _isPOISearch = false;
      _POIListViaHaptic = true;
   }
   if (sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_ENTER_ADDRESS == sdsScreen)
   {
      oMessage.AdditionalInfoList.clear();
   }

   _pPropertyCommonActionRequest->sendNaviHandoverPTTRequest(oMessage);
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool NaviHapticHandOverInterface::getHandOverDataFromDB(SDSHandOverScreenId screenID, const std::string& handOverType, std::string& sdsScreenID)
{
   ETG_TRACE_USR4(("NaviHapticHandOverInterface::getHandOverdataFromDB entry %s", handOverType.c_str()));
   if (true == clSDS_ConfigurationFlags::getNavStateAvailable())
   {
      sdsScreenID = Sds_HandOverDB_getSdsInput((int)screenID, handOverType);
   }
   else
   {
      sdsScreenID = Sds_HandOverDB_getSdsInput((int)screenID, handOverType, false);
   }
   return (sdsScreenID.empty()) ? false : true;
}


/**************************************************************************//**
 *
 ******************************************************************************/
void NaviHapticHandOverInterface::resetHandOverData()
{
   _dataList.clear();
   _uniqueList.clear();
   _dataNDSIdList.clear();
   _handOverOption.clear();
   _selectedDataList.clear();
   _addressWithIdInfo.clear();
}


/**************************************************************************//**
 *
 ******************************************************************************/
std::string NaviHapticHandOverInterface::calculateHandOverType(const int handOverDataListSize)
{
   if (handOverDataListSize == 0)
   {
      return HANDOVER_DATA_NONE;
   }
   else if (handOverDataListSize == HAND_OVER_DATA_SINGLE)
   {
      return HANDOVER_DATA_SINGLE;
   }
   else if (handOverDataListSize >= HAND_OVER_DATA_LIST_TWO && handOverDataListSize <= HAND_OVER_DATA_LIST_FIVE)
   {
      return HANDOVER_DATA_LIST;
   }
   else if (handOverDataListSize >= HAND_OVER_DATA_LIST_SIX)
   {
      return HANDOVER_DATA_MULTIPLE;
   }
   return "";
}


/**************************************************************************//**
 *
 ******************************************************************************/
sds2hmi_fi_tcl_e8_Data_Exchange_Type::tenType NaviHapticHandOverInterface::getHandOverType(const std::string& handOverType)
{
   if (0 == handOverType.compare(HANDOVER_DATA_NONE))
   {
      return sds2hmi_fi_tcl_e8_Data_Exchange_Type::FI_EN_DATA_NONE;
   }
   else if (0 == handOverType.compare(HANDOVER_DATA_SINGLE))
   {
      return sds2hmi_fi_tcl_e8_Data_Exchange_Type::FI_EN_DATA_SINGLE;
   }
   else if (0 == handOverType.compare(HANDOVER_DATA_LIST))
   {
      return sds2hmi_fi_tcl_e8_Data_Exchange_Type::FI_EN_DATA_LIST;
   }
   else if (0 == handOverType.compare(HANDOVER_DATA_MULTIPLE))
   {
      return sds2hmi_fi_tcl_e8_Data_Exchange_Type::FI_EN_DATA_MULTIPLE;
   }
   else
   {
      return sds2hmi_fi_tcl_e8_Data_Exchange_Type::FI_EN_DATA_NONE;
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
std::vector<std::string> NaviHapticHandOverInterface::removeDuplicateDataListItems(std::vector<std::string> ambigousDataList)
{
   std::vector<std::string> uniqueList;
   for (size_t i = 0; i < ambigousDataList.size(); ++i)
   {
      if (std::find(uniqueList.begin(), uniqueList.end(), ambigousDataList[i]) == uniqueList.end())
      {
         uniqueList.push_back(ambigousDataList[i]);
      }
   }
   return uniqueList;
}


/**************************************************************************//**
 *
 ******************************************************************************/
void NaviHapticHandOverInterface::setHandOverHMIListContents(sds2hmi_fi_tcl_e8_SDS_EntryPoint::tenType sdsScreen)
{
   switch (sdsScreen)
   {
      case sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_POI_IN_CITY:
      case sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_ENTER_POI:
      case sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_ENTER_POI_ALONG_ROUTE_TAB:
      case sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_ENTER_POI_IN_CITY_TAB:
      case sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_ENTER_POI_NEAR_DEST_TAB:
      case sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_ENTER_POI_NEARBY_TAB:
      case sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_SET_DEST_POI:
      {
         std::vector<org::bosch::cm::navigation::NavigationService::AddressListElement > poiList;
         org::bosch::cm::navigation::NavigationService::AddressListElement poiItem;

         for (size_t i = 0; i < _addressWithIdInfo.size(); ++i)
         {
            poiItem.setAddress((_addressWithIdInfo[i].getAddress().c_str()));
            poiItem.setData(_addressWithIdInfo[i].getData().c_str());
            poiItem.setId(_addressWithIdInfo[i].getId());
            poiItem.setDirection(getPOIDirection(_addressWithIdInfo[i].getDirection()));
            poiItem.setDistance(_addressWithIdInfo[i].getDistance().c_str());
            poiList.push_back(poiItem);
         }
         _pPOIList->setHandOverPOIList(poiList);
         break;
      }
      case sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_AMBIG_ADDRESS:
      case sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_AMBIG_CITY_CENTER:
      case sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_ENTER_INTERSECTION:
      {
         std::vector<org::bosch::cm::navigation::NavigationService::RefinementListElement > ambigusList;
         org::bosch::cm::navigation::NavigationService::RefinementListElement  ambiguosListItem;

         for (size_t i = 0; i < _addressWithIdInfo.size(); ++i)
         {
            ambiguosListItem.setData(_addressWithIdInfo[i].getData().c_str());
            ambiguosListItem.setId(_addressWithIdInfo[i].getId());
            ambiguosListItem.setDirection(getPOIDirection(_addressWithIdInfo[i].getDirection()));
            ambiguosListItem.setDistance(_addressWithIdInfo[i].getDistance().c_str());
            ambigusList.push_back(ambiguosListItem);
         }

         _pMultipleDestinationList->setHandOverList(ambigusList);
         break;
      }
      default :
         break;
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
org::bosch::cm::navigation::NavigationService::DirectionDescription NaviHapticHandOverInterface::getPOIDirection(org::bosch::cm::navigation::NavigationSDSService::DirectionDescription direction)
{
   org::bosch::cm::navigation::NavigationService::DirectionDescription poiDirection = org::bosch::cm::navigation::NavigationService::DirectionDescription__NORTH;
   switch (direction)
   {
      case org::bosch::cm::navigation::NavigationService::DirectionDescription__NORTH:
      {
         poiDirection =  org::bosch::cm::navigation::NavigationService::DirectionDescription__NORTH;
         break;
      }
      case org::bosch::cm::navigation::NavigationService::DirectionDescription__NORTH_NORTH_EAST:
      {
         poiDirection = org::bosch::cm::navigation::NavigationService::DirectionDescription__NORTH_NORTH_EAST;
         break;
      }
      case org::bosch::cm::navigation::NavigationService::DirectionDescription__NORTH_EAST:
      {
         poiDirection = org::bosch::cm::navigation::NavigationService::DirectionDescription__NORTH_EAST;
         break;
      }
      case org::bosch::cm::navigation::NavigationService::DirectionDescription__EAST_NORTH_EAST:
      {
         poiDirection = org::bosch::cm::navigation::NavigationService::DirectionDescription__EAST_NORTH_EAST;
         break;
      }
      case org::bosch::cm::navigation::NavigationService::DirectionDescription__EAST:
      {
         poiDirection =  org::bosch::cm::navigation::NavigationService::DirectionDescription__EAST;
         break;
      }
      case org::bosch::cm::navigation::NavigationService::DirectionDescription__EAST_SOUTH_EAST:
      {
         poiDirection = org::bosch::cm::navigation::NavigationService::DirectionDescription__EAST_SOUTH_EAST;
         break;
      }
      case org::bosch::cm::navigation::NavigationService::DirectionDescription__SOUTH_EAST:
      {
         poiDirection =  org::bosch::cm::navigation::NavigationService::DirectionDescription__SOUTH_EAST;
         break;
      }
      case org::bosch::cm::navigation::NavigationService::DirectionDescription__SOUTH_SOUTH_EAST:
      {
         poiDirection = org::bosch::cm::navigation::NavigationService::DirectionDescription__SOUTH_SOUTH_EAST;
         break;
      }
      case org::bosch::cm::navigation::NavigationService::DirectionDescription__SOUTH:
      {
         poiDirection = org::bosch::cm::navigation::NavigationService::DirectionDescription__SOUTH;
         break;
      }
      case org::bosch::cm::navigation::NavigationService::DirectionDescription__SOUTH_WEST:
      {
         poiDirection = org::bosch::cm::navigation::NavigationService::DirectionDescription__SOUTH_WEST;
         break;
      }
      case org::bosch::cm::navigation::NavigationService::DirectionDescription__WEST_SOUTH_WEST:
      {
         poiDirection = org::bosch::cm::navigation::NavigationService::DirectionDescription__WEST_SOUTH_WEST;
         break;
      }
      case org::bosch::cm::navigation::NavigationService::DirectionDescription__WEST:
      {
         poiDirection = org::bosch::cm::navigation::NavigationService::DirectionDescription__WEST;
         break;
      }
      case org::bosch::cm::navigation::NavigationService::DirectionDescription__WEST_NORTH_WEST:
      {
         poiDirection = org::bosch::cm::navigation::NavigationService::DirectionDescription__WEST_NORTH_WEST;
         break;
      }
      case org::bosch::cm::navigation::NavigationService::DirectionDescription__NORTH_WEST:
      {
         poiDirection = org::bosch::cm::navigation::NavigationService::DirectionDescription__WEST_NORTH_WEST;
         break;
      }
      case org::bosch::cm::navigation::NavigationService::DirectionDescription__NORTH_NORTH_WEST:
      {
         poiDirection = org::bosch::cm::navigation::NavigationService::DirectionDescription__NORTH_NORTH_WEST;
         break;
      }
      case org::bosch::cm::navigation::NavigationService::DirectionDescription__INVALID:
      {
         poiDirection = org::bosch::cm::navigation::NavigationService::DirectionDescription__INVALID;
         break;
      }
      default :
         break;
   }
   return poiDirection;
}


/**************************************************************************//**
 *
 ******************************************************************************/
void NaviHapticHandOverInterface::findEntryPointforEURformScreen(std::vector<SelectedDataItem> selectedDataList, SDSHandOverScreenId& sdsEURformScreen)
{
   std::vector<SelectedDataItem>::const_iterator selectedDataIte;
   for (selectedDataIte = selectedDataList.begin(); selectedDataIte != selectedDataList.end(); ++selectedDataIte)
   {
      if (!selectedDataIte->getData().empty())
      {
         if (HMISelectedAddressElementType__COUNTRY == selectedDataIte->getHmiSelectedAddressElementType())
         {
            sdsEURformScreen = org::bosch::cm::navigation::NavigationSDSService::SDSHandOverScreenId__NAVI_CITYFIRST_CITY;
         }
         else if (HMISelectedAddressElementType__CITY == selectedDataIte->getHmiSelectedAddressElementType())
         {
            sdsEURformScreen = org::bosch::cm::navigation::NavigationSDSService::SDSHandOverScreenId__NAVI_CITYFIRST_CITY_STREET;
         }
         else if (HMISelectedAddressElementType__STREET == selectedDataIte->getHmiSelectedAddressElementType())
         {
            sdsEURformScreen = org::bosch::cm::navigation::NavigationSDSService::SDSHandOverScreenId__NAVI_CITYFIRST_STREET_HNO;
         }
         else if (HMISelectedAddressElementType__HOUSENUMBER == selectedDataIte->getHmiSelectedAddressElementType())
         {
            sdsEURformScreen = org::bosch::cm::navigation::NavigationSDSService::SDSHandOverScreenId__NAVI_CITYFIRST_HNO_STREET;
         }
         else if (HMISelectedAddressElementType__JUNCTION == selectedDataIte->getHmiSelectedAddressElementType())
         {
            sdsEURformScreen = org::bosch::cm::navigation::NavigationSDSService::SDSHandOverScreenId__NAVI_INTERSECTION_STREET1;
         }
         else
         {
            sdsEURformScreen = org::bosch::cm::navigation::NavigationSDSService::SDSHandOverScreenId__NAVI_INACTIVE_SCREEN;
         }
      }
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void NaviHapticHandOverInterface::findPOIScreen(sds2hmi_fi_tcl_e8_SDS_EntryPoint::tenType& sdsPOIScreen)
{
   switch (_locationType)
   {
      case LocationType__NEAR_CURRENT_LOCATION:
      {
         sdsPOIScreen = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_ENTER_POI_NEARBY_TAB;
         break;
      }
      case LocationType__ALONG_ROUTE:
      {
         sdsPOIScreen = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_ENTER_POI_ALONG_ROUTE_TAB;
         break;
      }
      case LocationType__NEARBY_DESTINATION:
      {
         sdsPOIScreen = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_ENTER_POI_NEAR_DEST_TAB;
         break;
      }
      case LocationType__DEFAULT:
      {
         sdsPOIScreen = sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_ENTER_POI_IN_CITY_TAB;
         break;
      }
      default :
         break;
   }
}


void NaviHapticHandOverInterface::setHandOverDataListItems(sds2hmi_fi_tcl_e8_SDS_EntryPoint::tenType sdsScreen)
{
   ETG_TRACE_USR4(("NaviHapticHandOverInterface::setHandOverDataListItems entry"));
   switch (sdsScreen)
   {
      case sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_ADDRS_STEP_ENTER_STATE:
      case sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_ADDRS_STEP_HNOFIRST_STREET_CITY:
      case sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_ADDRS_STEP_CITYFIRST_STREET:
      case sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_STEP_CITYFIRST_CITY:
      case sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_STEP_CITYFIRST_HNO_STREET:
      case sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_CITYCENTER_CITY:
      case sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_CITYCENTER_STEP_ENTER_STATE:
      case sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_INTERSECTION_STRT1:
      case sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_INTERSECTION_STRT2:
      case sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_INTERSECT_STEP_ENTER_CITY:
      case sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_INTERSECT_STEP_ENTER_STATE:
      case sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_POI_IN_CITY:
      case sds2hmi_fi_tcl_e8_SDS_EntryPoint::FI_EN_SDS_ENTRY_NAVI_POI_STEP_ENTER_STATE:
      {
         if (NULL != _pNaviHapticHandOverList)
         {
            _pNaviHapticHandOverList->setHandOverHMIList(_uniqueList, _dataList, _dataNDSIdList, sdsScreen);
         }
         break;
      }
      default:
         break;
   }
}


void NaviHapticHandOverInterface::setDestinationIdList(GlobalNamedObjectId nameObjectId)
{
   sds2hmi_fi_tcl_NDSID dataNDSId;
   if (nameObjectId.getIsValid())
   {
      dataNDSId.u16UpdateRegionId = nameObjectId.getGlobalUpdateRegionId().getUpdateRegionId();
      dataNDSId.u32ProductId = nameObjectId.getGlobalUpdateRegionId().getProductId();
      dataNDSId.u8SupplierId = nameObjectId.getGlobalUpdateRegionId().getSupplierId();
      dataNDSId.u32NDSId = nameObjectId.getNamedObjectId();
   }
   _dataNDSIdList.push_back(dataNDSId);
}
