/**************************************************************************//**
 * \file       GuiService.cpp
 *
 * This file is part of the SdsAdapter component.
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/

#include "application/GuiService.h"
#include "application/DataHandoverHandler.h"
#include "application/NaviHapticHandOverInterface.h"
#include "application/clSDS_SessionControl.h"
#include "application/clSDS_XMLStringCreation.h"
#include "application/clSDS_MenuControl.h"
#include "application/clSDS_Userwords.h"
#include "application/clSDS_LanguageMediator.h"
#include "application/clSDS_KDSConfiguration.h"
#include "application/EarlyStartupPlayer.h"
#include "application/AudioSourceHandler.h"
#include "SdsAdapter_Trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/GuiService.cpp.trc.h"
#endif


using namespace sds_gui_fi::SdsGuiService;


GuiService::GuiService(clSDS_SessionControl* pSessionControl, EarlyStartupPlayer* pEarlyStartupPlayer, AudioSourceHandler* pAudioSourceHandler
                       , DataHandoverHandler* pDataHandoverHandler
                       , NaviHapticHandOverInterface* pNaviHapticHandOverInterface)
   : SdsGuiServiceStub("GuiServicePort")
   , _pSessionControl(pSessionControl)
   , _pMenuControl(NULL)
   , _pEarlyStartupPlayer(pEarlyStartupPlayer)
   , _pDataHandoverHandler(pDataHandoverHandler)
   , _audioSourceHandler(pAudioSourceHandler)
   , _pNaviHapticHandOverInterface(pNaviHapticHandOverInterface)
{
}


void GuiService::onPttPressRequest(const ::boost::shared_ptr< PttPressRequest >& request)
{
   if (_pSessionControl)
   {
      if (request->getPttPressType() == KeyState__KEY_HK_MFL_PTT_SHORT)
      {
         vPttShortPressed();
      }
   }
}


void GuiService::vPttShortPressed()
{
   if (_pSessionControl)
   {
      if (bIsPttPressAllowed())
      {
         if (_pSessionControl->bDownloadFailed())
         {
            if (_pSessionControl->bIsSDSLanguageSupport())
            {
               sendEventSignal(Event__SPEECH_DIALOG_SHOW_POPUP_NODATACARRIER_OPEN);
            }
            else
            {
               sendEventSignal(Event__SPEECH_DIALOG_SHOW_POPUP_LANGUAGE_NOT_SUPPORTED);
            }
         }
         else if (_pSessionControl->bIsInitialising())
         {
            sendEventSignal(Event__SPEECH_DIALOG_SHOW_POPUP_LOADING_OPEN);
         }
         else if (!_pSessionControl->bIsSDSLanguageSupport())
         {
            sendEventSignal(Event__SPEECH_DIALOG_SHOW_POPUP_LANGUAGE_NOT_SUPPORTED);
         }
         else if (_pSessionControl->bIsTTSOnlySupport())
         {
            sendEventSignal(Event__SPEECH_DIALOG_SHOW_POPUP_LANGUAGE_NOT_SUPPORTED);
         }
         else if (_pNaviHapticHandOverInterface != NULL && _pNaviHapticHandOverInterface->isHandOverSession())
         {
            ETG_TRACE_USR4(("GuiService::vPttShortPressed() sendNaviHandOverPTTrequest"));
            _pNaviHapticHandOverInterface->sendNaviHandOverPTTRequest();
         }
         else if (_pDataHandoverHandler != NULL && _pDataHandoverHandler->pttPressedOnDataHandoverScreen())
         {
            _pDataHandoverHandler->handleDataHandoverPTTrequest();
         }
         else
         {
            _pSessionControl->vSendPttEvent();
         }
      }
   }
}


void GuiService::onManualOperationRequest(const ::boost::shared_ptr< ManualOperationRequest >& request)
{
   if (_pMenuControl)
   {
      switch (request->getOperationType())
      {
         case OperationType__MANUAL_INTERVENTION_HAPTICAL_SELECTION:
            _pMenuControl->onElementSelected(request->getValue());
            break;

         case OperationType__MANUAL_INTERVENTION_ENCODER_FOCUS_MOVED:
            _pMenuControl->onCursorMoved(request->getValue());
            break;

         case OperationType__MANUAL_INTERVENTION_NEXT_PAGE:
            _pMenuControl->onRequestNextPage();
            break;

         case OperationType__MANUAL_INTERVENTION_PREV_PAGE:
            _pMenuControl->onRequestPreviousPage();
            break;

         case OperationType__MANUAL_INTERVENTION_HAPTICAL_SELECTION_HL:
            _pMenuControl->onElementSelectedHL(request->getValue());
            break;

         default:
            break;
      }
   }
}


GuiService::~GuiService()
{
   _pSessionControl = NULL;
   _pMenuControl = NULL;
   _pEarlyStartupPlayer = NULL;
   _pDataHandoverHandler = NULL;
   _audioSourceHandler = NULL;
   _pNaviHapticHandOverInterface = NULL;
}


void GuiService::onStartSessionContextRequest(const ::boost::shared_ptr<  sds_gui_fi::SdsGuiService::StartSessionContextRequest >& request)
{
   if (_pSessionControl)
   {
      if (_pSessionControl->bDownloadFailed() || _pSessionControl->bIsInitialising())
      {
         sendEventSignal(Event__SPEECH_DIALOG_SHOW_POPUP_LOADING_OPEN);
      }
      else if ((!_pSessionControl->bIsSDSLanguageSupport()) || (_pSessionControl->bIsTTSOnlySupport()))
      {
         sendEventSignal(Event__SPEECH_DIALOG_SHOW_POPUP_LANGUAGE_NOT_SUPPORTED);
      }
      else if (_pSessionControl->bIsSDSLanguageSupport())
      {
         _pSessionControl->sendStartSessionContext(request->getStartupContextType());
      }
   }
}


void GuiService::onTestModeUpdateRequest(const ::boost::shared_ptr< TestModeUpdateRequest >& /*request*/)
{
}


void GuiService::setMenuControl(clSDS_MenuControl* pMenuControl)
{
   _pMenuControl = pMenuControl;
}


bool GuiService::bIsPttPressAllowed()const
{
   if ((clSDS_KDSConfiguration::getAllianceVRModeType()) == clSDS_KDSConfiguration::OFF)
   {
      ETG_TRACE_USR1(("VR configuration set as '%d' ", clSDS_KDSConfiguration::getAllianceVRModeType()));
      return false;
   }
   else
   {
      return true;
   }
}


void GuiService::onSettingsCommandRequest(const ::boost::shared_ptr< SettingsCommandRequest >& /*request*/)
{
   if (_pMenuControl)
   {
      _pMenuControl->onSettingsRequest();
   }
}


void GuiService::onHelpCommandRequest(const ::boost::shared_ptr< HelpCommandRequest >& /*request*/)
{
   if (_pMenuControl)
   {
      _pMenuControl->onHelpRequest();
   }
}


void GuiService::onAbortSessionRequest(const ::boost::shared_ptr< AbortSessionRequest >& /*request*/)
{
   if (_pSessionControl)
   {
      sendEventSignal(Event__SPEECH_DIALOG_SDS_SESSION_ABORT);
      _pSessionControl->vAbortSession();
      ETG_TRACE_USR4(("Aborting SDS session"));
   }
}


void GuiService::onStopSessionRequest(const ::boost::shared_ptr< sds_gui_fi::SdsGuiService::StopSessionRequest >& /* request */)
{
   if (_pSessionControl)
   {
      _pSessionControl->vStopSession();
      ETG_TRACE_USR4(("Stoping SDS session"));
   }
}


void GuiService::onBackPressRequest(const ::boost::shared_ptr< sds_gui_fi::SdsGuiService::BackPressRequest >& /*request*/)
{
   if (_pSessionControl)
   {
      _pSessionControl->vBackButtonPressed();
      ETG_TRACE_USR4(("BackButtonPressed"));
   }
}


void GuiService::onPauseSessionRequest(const ::boost::shared_ptr< sds_gui_fi::SdsGuiService::PauseSessionRequest >& /* request */)
{
   if (_pSessionControl)
   {
      _pSessionControl->sendEnterPauseMode();
   }
}


void GuiService::onResumeSessionRequest(const ::boost::shared_ptr< sds_gui_fi::SdsGuiService::ResumeSessionRequest >& /* request */)
{
   if (_pSessionControl)
   {
      _pSessionControl->sendExitPauseMode();
   }
}


void GuiService::onStartEarlyHandlingRequest(const ::boost::shared_ptr< sds_gui_fi::SdsGuiService::StartEarlyHandlingRequest >& /*request*/)
{
   if (_pEarlyStartupPlayer)
   {
      _pEarlyStartupPlayer->requestStartPlayback();
   }
}


void GuiService::onStopEarlyHandlingRequest(const ::boost::shared_ptr< sds_gui_fi::SdsGuiService::StopEarlyHandlingRequest >& /*request*/)
{
   if (_pEarlyStartupPlayer)
   {
      _pEarlyStartupPlayer->requestStopPlayback();
   }
}


void GuiService::onEarlyPromptFinished()
{
   sendEarlyPromptFinishedSignal();
}


void GuiService::onEarlyPromptTerminated()
{
   sendEarlyPromptTerminatedSignal();
}


void GuiService::onCursorIndexForInitialFocusRequest(const ::boost::shared_ptr< sds_gui_fi::SdsGuiService::CursorIndexForInitialFocusRequest >& request)
{
   if (_pMenuControl)
   {
      _pMenuControl->onCursorIndexChanged(request->getIndex());
   }
}


void GuiService::onHighPrioAppStatusSet(const ::boost::shared_ptr< HighPrioAppStatusSet >& payload)
{
   if (payload->hasHighPrioAppStatus())
   {
      bool highPrioAppState = payload->getHighPrioAppStatus();
      setHighPrioAppStatus(highPrioAppState);

      if (_audioSourceHandler)
      {
         _audioSourceHandler->setHighPriorityAppStatus(highPrioAppState);
      }
   }
}
