/**************************************************************************//**
 * \file       DistanceDirectionUtility.cpp
 *
 * This file is part of the SdsAdapter component.
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/

#include <sstream>
#include <iomanip>
#include <math.h>
#include "application/DistanceDirectionUtility.h"


#define DIST_CONVERT_KM_100000  100000
#define MILE_TO_METER  1609.34
#define DIST_CONVERT_KM_1000  1000
#define DIST_1_MILE  1.0
#define DIST_CONVERT_MILES_TO_FEET  5280.0
#define DIST_THREE_FOURTH_MILE  0.75
#define DIST_HALF_MILE  0.5
#define DIST_ONE_FOURTH_MILE  0.25
#define DIST_1000_UNITS  1000.0
#define DIST_300_UNITS  300.0
#define DIST_100_UNITS  100.0
#define DIRECTION_MOD_VALUE 8


DistanceDirectionUtility::DistanceDirectionUtility()
{
}


DistanceDirectionUtility::~DistanceDirectionUtility()
{
}


/***********************************************************************//**

***************************************************************************/
std::string DistanceDirectionUtility::vConvertDistanceToString(float flValue)
{
   std::ostringstream buffer;
   flValue = (flValue < 0.0f) ? 0.0 : flValue;
   double doValue = static_cast <double>(flValue);
   doValue = static_cast <double>(round(doValue * 10) / 10);
   int decP = static_cast <int>((doValue - floor(doValue)) * 10);
   unsigned int precision = static_cast <unsigned int>((decP == 0) ? 0 : 1);
   buffer << std::fixed << std::setprecision(precision) << doValue ;
   return buffer.str();
}


/***********************************************************************//**

 ***************************************************************************/
double DistanceDirectionUtility::vCalculateDistanceWithDirection(
   const stGeoCoordinates& lSxmGeoCoordinates, const stGeoCoordinates& lNaviGeoCoodrdinates)
{
   double deltaLatitude  = lSxmGeoCoordinates.dLatitude - lNaviGeoCoodrdinates.dLatitude;
   double deltaLongitude = lSxmGeoCoordinates.dLongitude - lNaviGeoCoodrdinates.dLongitude;
   deltaLongitude *= ::cos(vConvertDegreesToRadians((lSxmGeoCoordinates.dLatitude + lNaviGeoCoodrdinates.dLatitude) / 2.0));

   // range of direction in degrees: -180 to + 180
   double carDirection = vConvertRadiansToDegrees(::atan2(deltaLongitude, deltaLatitude));

   // convert direction into North-relative angle:
   carDirection = ::fmod((carDirection + 360.0), 360.0);

   return carDirection;
}


/***********************************************************************//**

 ***************************************************************************/
double DistanceDirectionUtility::vConvertDegreesToRadians(double angle_degrees)
{
   return (M_PI / 180) * angle_degrees;
}


/***********************************************************************//**

 ***************************************************************************/
double DistanceDirectionUtility::vConvertRadiansToDegrees(double angle_degrees)
{
   return (180 / M_PI) * angle_degrees;
}


/***********************************************************************//**

 ***************************************************************************/
double DistanceDirectionUtility::vMakeCircular(double degree)
{
   double fmod_degree = ::fmod(degree, 360.0);
   return ::fabs(fmod_degree >= 0.0 ? fmod_degree : fmod_degree + 360.0);
}


/***********************************************************************//**

 ***************************************************************************/
std::string DistanceDirectionUtility::getFormattedDistance(float distanceInMiles, enDistanceUnitType enDistanceUnit)
{
   distanceInMiles = (distanceInMiles < 0.0) ? 0.0 : distanceInMiles; //handle negative distance
   ::std::string dDistanceInUnits = "km";
   switch (enDistanceUnit)
   {
      case EN_DISTANCE_MILES:
      {
         dDistanceInUnits = vAmericanDistanceUnitHandler(distanceInMiles);
      }
      break;

      case EN_DISTANCE_KM:
      {
         std::ostringstream dDistance;
         const char* kilomtr = " km";
         float distanceinmtrs = static_cast<float>(distanceInMiles * MILE_TO_METER);
         unsigned int u32DistToPOIMts = static_cast <unsigned int>(distanceinmtrs);
         if (u32DistToPOIMts >= DIST_CONVERT_KM_100000)  //Distance More than 100K == 100000 meters
         {
            dDistance << (u32DistToPOIMts / DIST_CONVERT_KM_1000) << kilomtr;
            dDistanceInUnits = dDistance.str();
         }
         else //between 1 Km and 100 km
         {
            dDistance << vConvertDistanceToString(distanceinmtrs / DIST_CONVERT_KM_1000) << kilomtr;
            dDistanceInUnits = dDistance.str();
         }
      }
      break;

      default://unknown unit
         break;
   }
   return dDistanceInUnits;
}


/***********************************************************************//**
 Distance in USA and CAN regions
 ***************************************************************************/
::std::string DistanceDirectionUtility::vAmericanDistanceUnitHandler(float dDistanceInMiles)
{
   unsigned int u32distanceInMiles = static_cast <unsigned int>(dDistanceInMiles);
   float distInFT = static_cast <float>(dDistanceInMiles * DIST_CONVERT_MILES_TO_FEET);
   unsigned int u32DistInFT = static_cast <unsigned int>(distInFT);
   std::ostringstream  dDistanceInUnits;
   const char* feet = " ft";
   const char* mile  = " mi";

   if (dDistanceInMiles >= DIST_100_UNITS)
   {
      dDistanceInUnits << u32distanceInMiles << mile;
   }
   else if ((dDistanceInMiles >= DIST_1_MILE) && (dDistanceInMiles < DIST_100_UNITS))
   {
      dDistanceInUnits << vConvertDistanceToString(dDistanceInMiles) << mile;
   }
   else if ((dDistanceInMiles < DIST_1_MILE) && (dDistanceInMiles >= DIST_THREE_FOURTH_MILE))
   {
      dDistanceInUnits << "0.75" << mile ; // 3/4 miles.
   }
   else if ((dDistanceInMiles < DIST_THREE_FOURTH_MILE) && (dDistanceInMiles >= DIST_HALF_MILE))
   {
      dDistanceInUnits << "0.5" << mile ;//  1/2 miles.
   }
   else if ((dDistanceInMiles < DIST_HALF_MILE) && (dDistanceInMiles >= DIST_ONE_FOURTH_MILE))
   {
      dDistanceInUnits << "0.25" << mile; // 1/4th miles.
   }
   else if ((dDistanceInMiles <  DIST_ONE_FOURTH_MILE) && (distInFT >= DIST_1000_UNITS))
   {
      dDistanceInUnits << "1000" << feet;
   }
   else if ((distInFT < DIST_1000_UNITS) && (distInFT >= DIST_300_UNITS))
   {
      dDistanceInUnits << vRoundUpToInteger(u32DistInFT, 100) << feet;//increment in 100
   }
   else if (distInFT <  DIST_300_UNITS)
   {
      dDistanceInUnits << vRoundUpToInteger(u32DistInFT, 50) << feet;//increment in 50
   }

   return dDistanceInUnits.str();
}


/***********************************************************************//**

 ***************************************************************************/
double DistanceDirectionUtility::vConvertLatitudeLongitudeTodouble(signed int dLatitudeLongitude)
{
   const double DOUBLETOINT = 1000000.0;
   double doLatLong = static_cast <double>(dLatitudeLongitude);
   return (doLatLong / DOUBLETOINT);
}


/***********************************************************************//**

 ***************************************************************************/
unsigned int DistanceDirectionUtility::vRoundUpToInteger(unsigned int numToRound, unsigned int multiple)
{
   if (multiple == 0)
   {
      return numToRound;
   }
   int rem = static_cast<int>(numToRound % multiple);
   numToRound = (rem == 0) ? numToRound : (((numToRound + (multiple / 2)) / multiple) * multiple) ;
   return numToRound;
}


/***********************************************************************//**

 ***************************************************************************/
bool DistanceDirectionUtility::bCheckLocationBoundary(double dLatitude, double dLongitude)
{
   // NAR Geo-coordinates i.e region Limits are taken from sms library.
   //"di_middleware_server\components\fc_sxm\smslib\sms\source\sms\source\_location_obj.h"

   bool bIsWithinLimit = true;

   const double MIN_LATITUDE = 0.0;
   const double MAX_LATITUDE = 90.0;
   const double MIN_LONGITUDE = -178.0;
   const double MAX_LONGITUDE = -30.0;

   if (dLatitude < MIN_LATITUDE || dLatitude > MAX_LATITUDE || dLongitude < MIN_LONGITUDE || dLongitude > MAX_LONGITUDE)
   {
      bIsWithinLimit = false;
   }

   return bIsWithinLimit;
}


/***********************************************************************//**
adjusts direction to current car heading
 ***************************************************************************/
double DistanceDirectionUtility::vCarRelativeDirection(double currentDirection, double carDirection)
{
   return vMakeCircular(carDirection - currentDirection);
}


/***********************************************************************//**
 *  8 Directions calculations like  enHeadingType.
 *  8 directions(modValue): 360 / 8 = 45° for each direction sector
 *  N = -22.5 to 22.5, NE = 22.5 to 67.5, .....
 *  East (90°), West(270°)    North  (0°)  S (180°)
 ***************************************************************************/
int DistanceDirectionUtility::getFormatDirectionInfo(double dActualAngle)
{
   double dirAngle = (DIRECTION_MOD_VALUE == 8) ? 45 : 22.5;
   int heading = -1;
   unsigned int uHeadingArr[8] = {0, 1, 2, 3, 4, 5, 6, 7}; //enHeadingType

   if (dActualAngle >= 0.0 && dActualAngle <= 360.0)
   {
      dActualAngle = round(dActualAngle);

      unsigned int val = static_cast <unsigned int>(round(dActualAngle / dirAngle));
      heading = static_cast <int>(uHeadingArr[(val % DIRECTION_MOD_VALUE)]);
   }
   return heading;
}


/***********************************************************************//**

 ***************************************************************************/
DistanceDirectionUtility::enHeadingType DistanceDirectionUtility::oGetDirection(
   const stGeoCoordinates& lSxmGeoCoordinates, const stGeoCoordinates& lNaviGeoCoodrdinates)
{
   if ((false == bCheckLocationBoundary(lSxmGeoCoordinates.dLatitude, lSxmGeoCoordinates.dLongitude))
         || (false == bCheckLocationBoundary(lNaviGeoCoodrdinates.dLatitude, lNaviGeoCoodrdinates.dLongitude)))
   {
      return EN_HEADING_INDEX_UNKNOWN; // Lat (or) Lon value is out of range or Invalid
   }

   //Get car direction from two set of coordinates.
   double cardirection = vCalculateDistanceWithDirection(lSxmGeoCoordinates, lNaviGeoCoodrdinates);
   double dHeading = vCarRelativeDirection(lNaviGeoCoodrdinates.dVehicleHeading, cardirection);
   enHeadingType eType = static_cast<enHeadingType>(getFormatDirectionInfo(dHeading));
   return eType;
}


/*********************************************************************************//**
Convert SXM Latitude longitude to  Navigation (WGS84 :World Geodetic System 1984 standard)
 Refer the "di_fi\components\fi\pos_fi\pos_fi.pdf" for conversion factor
 ***************************************************************************************/
int DistanceDirectionUtility::vConvertLatitudeLongitudeToWGS84(int dLatitudeLongitude)
{
   const double fFactorNavPos = (4294967295.0 / (360.0 * 1000000.0));
   int latlon = static_cast <int>((dLatitudeLongitude * fFactorNavPos));
   return latlon;
}


/*********************************************************************************//**
Convert SXM Latitude longitude to  Navigation (WGS84 :World Geodetic System 1984 standard)
 Refer the "di_fi\components\fi\pos_fi\pos_fi.pdf" for conversion factor
 ***************************************************************************************/
int DistanceDirectionUtility::vConvertWGS84ToLatLong(int latlongwsg)
{
   const double fFactorNavPos = ((360.0 * 1000000.0) / 4294967295.0);
   int latlon = static_cast <int>((latlongwsg * fFactorNavPos));
   return latlon;
}
