/**************************************************************************//**
 * \file       AudioSourceHandler.cpp
 *
 * AudioSourceHandler class implementation
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/

#include "AudioSourceHandler.h"
#include "clSDS_SDSStatus.h"
#include "SdsAudioSource.h"
#include "clSDS_SessionControl.h"

#include "SdsAdapter_Trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/AudioSourceHandler.cpp.trc.h"
#endif

/**************************************************************************//**
 * Constructor
 ******************************************************************************/
AudioSourceHandler::AudioSourceHandler(clSDS_SDSStatus* pSDSStatus, SdsAudioSource& audioSource, clSDS_SessionControl* pSessionControl)
   : _pSDSStatus(pSDSStatus)
   , _audioSource(audioSource)
   , _pSessionControl(pSessionControl)
   , _audioState(ARL_EN_ISRC_ACT_OFF)
   , _ignorePause(false)
   , _pauseSessionRequested(false)
   , _highPrioAppStatus(false)
   , _waitingForExitPause(false)
{
   _audioSource.addObserver(this);
   if (_pSDSStatus)
   {
      _pSDSStatus->vRegisterObserver(this);
   }
}


/**************************************************************************//**
 * Destructor
 ******************************************************************************/
AudioSourceHandler::~AudioSourceHandler()
{
   _pSDSStatus = NULL;
   _pSessionControl = NULL;
}


/**************************************************************************//**
*
******************************************************************************/
void AudioSourceHandler::onOutgoingCall()
{
   ETG_TRACE_USR4(("AudioSourceHandler::Entered onOutgoingCall"));
   _ignorePause = true;
}


/**************************************************************************//**
*
******************************************************************************/
void AudioSourceHandler::onAudioSourceStateChanged(arl_tenActivity state)
{
   _audioState = state;
   if (_pSDSStatus && _pSessionControl)
   {
      switch (state)
      {
         case ARL_EN_ISRC_ACT_PAUSE:
         {
            if (_pSDSStatus->bIsSessionInteractable() && !(_pSDSStatus->bIsPause()))
            {
               if (_ignorePause)
               {
                  ETG_TRACE_USR4(("AudioSourceHandler::Entered ignorePause"));
                  resetFlags();
               }
               else
               {
                  ETG_TRACE_USR4(("AudioSourceHandler::Entered Pause"));
                  _pSessionControl->sendEnterPauseMode();
                  _pauseSessionRequested = true;
               }
            }
            break;
         }
         case ARL_EN_ISRC_ACT_ON:
         {
            tryResumeSessionIfPaused();
            break;
         }
         case ARL_EN_ISRC_ACT_OFF:
         {
            if (_pSDSStatus->bIsSessionInteractable() || _pSDSStatus->bIsPause())
            {
               _pSessionControl->vAbortSession();
               resetFlags();
            }
            break;
         }
         default:
         {
            //do nothing
            break;
         }
      }
   }
}


void AudioSourceHandler::vSDSStatusChanged()
{
   if (_pSDSStatus)
   {
      if (_pSDSStatus->bIsPause() && _waitingForExitPause)
      {
         _pSessionControl->sendExitPauseMode();
         resetFlags();
      }
   }
}


void AudioSourceHandler::setHighPriorityAppStatus(bool isHighPrioApp)
{
   _highPrioAppStatus = isHighPrioApp;
   tryResumeSessionIfPaused();
}


void AudioSourceHandler::resetFlags()
{
   _ignorePause = false;
   _pauseSessionRequested = false;
   _waitingForExitPause = false;
}


void AudioSourceHandler::tryResumeSessionIfPaused()
{
   if (!(_pSessionControl && _pSDSStatus))
   {
      return;
   }

   if ((_audioState == ARL_EN_ISRC_ACT_ON) &&
         _pauseSessionRequested && !_highPrioAppStatus)
   {
      resetFlags();
      if (_pSDSStatus->bIsPause())
      {
         _pSessionControl->sendExitPauseMode();
      }
      else
      {
         _waitingForExitPause = true;
      }
   }
}
