/**************************************************************************//**
 * \file       clSDS_Property_NaviStatus.cpp
 *
 * Common Action Request property implementation
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "Sds2HmiServer/functions/clSDS_Property_NaviStatus.h"
#include "application/clSDS_ConfigurationFlags.h"
#include "SdsAdapter_Trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/clSDS_Property_NaviStatus.cpp.trc.h"
#endif


using namespace org::bosch::cm::navigation::NavigationService;


/**************************************************************************//**
* Destructor
******************************************************************************/
clSDS_Property_NaviStatus::~clSDS_Property_NaviStatus()
{
}


/**************************************************************************//**
* Constructor
******************************************************************************/
clSDS_Property_NaviStatus::clSDS_Property_NaviStatus(
   ahl_tclBaseOneThreadService* pService,
   ::boost::shared_ptr<org::bosch::cm::navigation::NavigationService::NavigationServiceProxy > naviProxy)
   : clServerProperty(SDS2HMI_SDSFI_C_U16_NAVISTATUS, pService)
   , _naviProxy(naviProxy)
{
   _sdsNaviStatus = sds2hmi_fi_tcl_e8_NAV_Status::FI_EN_IDLE;
   enVDEAvailInfo = sds2hmi_fi_tcl_e8_NAV_VDEAvailInfo::FI_EN_UNKNOWN;
   _detourOn = false;
}


/**************************************************************************//**
*
******************************************************************************/
tVoid clSDS_Property_NaviStatus::vSet(amt_tclServiceData* pInMsg)
{
   sds2hmi_sdsfi_tclMsgNaviStatusSet oMessage;
   vGetDataFromAmt(pInMsg, oMessage);
   enVDEAvailInfo = oMessage.VDEAvailInfo.enType;
   vSendNavStatus();
}


/********************************************************************************
 *
 ********************************************************************************/
void clSDS_Property_NaviStatus::addVDEAvailabilityObserver(VDEAvailabilityObserver* pVDEAvailabilityObserver)
{
   if (pVDEAvailabilityObserver != NULL)
   {
      _vDEAvailabilityObservers.push_back(pVDEAvailabilityObserver);
   }
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_Property_NaviStatus::notifyVDEAvailabilityObserver()
{
   for (size_t i = 0; i < _vDEAvailabilityObservers.size(); ++i)
   {
      _vDEAvailabilityObservers.at(i)->onVDEAvalabilityStatus();
   }
}


/*****************************************************************************
 *
 *****************************************************************************/
tVoid clSDS_Property_NaviStatus::vGet(amt_tclServiceData* /*pInMsg*/)
{
   vSendNavStatus();
}


/**************************************************************************//**
*
******************************************************************************/
tVoid clSDS_Property_NaviStatus::vUpreg(amt_tclServiceData* /*pInMsg*/)
{
   vSendNavStatus();
}


/***********************************************************************//**
*
***************************************************************************/
tVoid clSDS_Property_NaviStatus::vSendNavStatus()
{
   sds2hmi_sdsfi_tclMsgNaviStatusStatus oMessage;
   oMessage.MenuType.enType = sds2hmi_fi_tcl_e8_NAV_MenuType::FI_EN_MAP;
   oMessage.NavUnit.enType = sds2hmi_fi_tcl_e8_NAV_NavUnit::FI_EN_METRIC;
   oMessage.Status.enType = _sdsNaviStatus;
   oMessage.VDEAvailInfo.enType = enVDEAvailInfo;
   oMessage.DetourOn  = _detourOn;
   vStatus(oMessage);

   if (clSDS_ConfigurationFlags::isDisableNavigationForVDEUnavailable()) //3.x feature
   {
      clSDS_ConfigurationFlags::setDynamicVariable("VDEAvailable", oMessage.VDEAvailInfo.enType);
      if (enVDEAvailInfo == sds2hmi_fi_tcl_e8_NAV_VDEAvailInfo::FI_EN_NOTAVAILABLE ||
            enVDEAvailInfo == sds2hmi_fi_tcl_e8_NAV_VDEAvailInfo::FI_EN_UNKNOWN)
      {
         notifyVDEAvailabilityObserver();
      }
      if (enVDEAvailInfo == sds2hmi_fi_tcl_e8_NAV_VDEAvailInfo::FI_EN_AVAILABLE)
      {
         notifyVDEAvailabilityObserver();
      }
   }
}


/***********************************************************************//**
*
***************************************************************************/
void clSDS_Property_NaviStatus::onAvailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _naviProxy)
   {
      _naviProxy->sendNavStatusRegister(*this);
      _naviProxy->sendNavStatusGet(*this);
   }
}


/***********************************************************************//**
*
***************************************************************************/
void clSDS_Property_NaviStatus::onUnavailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _naviProxy)
   {
      _naviProxy->sendNavStatusDeregisterAll();
   }
}


/***********************************************************************//**
*
***************************************************************************/
void clSDS_Property_NaviStatus::onNavStatusUpdate(
   const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationService::NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationService::NavStatusUpdate >& update)
{
   bool routeGuidanceOn = false;
   switch (update->getNavStatus())
   {
      case NavStatus__NAVSTATUS_UNKNOWN:
      case NavStatus__NAVSTATUS_IDLE:
         _sdsNaviStatus = sds2hmi_fi_tcl_e8_NAV_Status::FI_EN_IDLE;
         break;
      case NavStatus__NAVSTATUS_BUSY:
         _sdsNaviStatus = sds2hmi_fi_tcl_e8_NAV_Status::FI_EN_BUSY;
         break;
      case NavStatus__NAVSTATUS_GUIDANCE_ACTIVE:
         _sdsNaviStatus = sds2hmi_fi_tcl_e8_NAV_Status::FI_EN_GUIDANCE_ACTIVE;
         routeGuidanceOn = true;
         break;
      case  NavStatus__NAVSTATUS_CALCULATING_ROUTE:
         _sdsNaviStatus = sds2hmi_fi_tcl_e8_NAV_Status::FI_EN_CALCULATING_ROUTE;
         break;
      case NavStatus__NAVSTATUS_GUIDANCE_ACTIVE_OFFBOARD:
         _sdsNaviStatus = sds2hmi_fi_tcl_e8_NAV_Status::FI_EN_GUIDANCE_ACTIVE_OFFBOARD;
         break;
      default:
         _sdsNaviStatus = sds2hmi_fi_tcl_e8_NAV_Status::FI_EN_BUSY;
         break;
   }
   clSDS_ConfigurationFlags::setDynamicVariable("RouteGuidanceOn", routeGuidanceOn);
   vSendNavStatus();
}


/***********************************************************************//**
*
***************************************************************************/
void clSDS_Property_NaviStatus::onNavStatusError(
   const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationService::NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationService::NavStatusError >& /*error*/)
{
}


/***********************************************************************//**
*
***************************************************************************/
sds2hmi_fi_tcl_e8_NAV_Status::tenType clSDS_Property_NaviStatus::getNaviStatus()
{
   return _sdsNaviStatus;
}


/***********************************************************************//**
*
***************************************************************************/
void clSDS_Property_NaviStatus::setNaviDetourStatus(bool detourStatus)
{
   _detourOn = detourStatus;
   vSendNavStatus();
}
