/**************************************************************************//**
 * \file       clSDS_Property_NaviCurrentCountryState.cpp
 *
 * Common Action Request property implementation
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "Sds2HmiServer/functions/clSDS_Property_NaviCurrentCountryState.h"
#include "SdsAdapter_Trace.h"
#include "application/clSDS_KDSConfiguration.h"
#include <string.h>

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/clSDS_Property_NaviCurrentCountryState.cpp.trc.h"
#endif


using namespace org::bosch::cm::navigation::NavigationSDSService;


/**************************************************************************//**

******************************************************************************/
clSDS_Property_NaviCurrentCountryState::~clSDS_Property_NaviCurrentCountryState()
{
   _pMethodNavDataRegisterDirectNDSUse = NULL;
}


/**************************************************************************//**
* Constructor
******************************************************************************/
clSDS_Property_NaviCurrentCountryState::clSDS_Property_NaviCurrentCountryState(
   ahl_tclBaseOneThreadService* pService,
   ::boost::shared_ptr< NavigationSDSServiceProxy> naviSDSProxy,
   clSDS_Method_NavDataRegisterDirectNDSUse* pMethodNavDataRegisterDirectNDSUse)

   : clServerProperty(SDS2HMI_SDSFI_C_U16_NAVICURRENTCOUNTRYSTATE, pService)
   , _naviSDSProxy(naviSDSProxy)
   , _pMethodNavDataRegisterDirectNDSUse(pMethodNavDataRegisterDirectNDSUse)
{
   if (pMethodNavDataRegisterDirectNDSUse)
   {
      pMethodNavDataRegisterDirectNDSUse->addGlobalDataForSdsObserver(this);
   }
}


/**************************************************************************//**
*
******************************************************************************/
tVoid clSDS_Property_NaviCurrentCountryState::vSet(amt_tclServiceData* /*pInMsg*/)
{
   sendResponse();
}


/**************************************************************************//**
*  Not noticed of get call for this property yet.
******************************************************************************/
tVoid clSDS_Property_NaviCurrentCountryState::vGet(amt_tclServiceData* /*pInMsg*/)
{
   sendResponse();
}


/**************************************************************************//**
*
******************************************************************************/
tVoid clSDS_Property_NaviCurrentCountryState::vUpreg(amt_tclServiceData* /*pInMsg*/)
{
   sendResponse();
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_Property_NaviCurrentCountryState::sendResponse()
{
   if (_naviSDSProxy->hasPositionStatusInfo() && _naviSDSProxy->hasPositionInfo())
   {
      if ((clSDS_KDSConfiguration::getMarketRegionType() == clSDS_KDSConfiguration::JPN) ||
            (clSDS_KDSConfiguration::getMarketRegionType() == clSDS_KDSConfiguration::KOR))
      {
         sendCountryStateStatus(_naviSDSProxy->getPositionStatusInfo(), _naviSDSProxy->getPositionInfo());
      }
      else
      {
         validateAndSendPosition(_naviSDSProxy->getPositionStatusInfo(), _naviSDSProxy->getPositionInfo());
      }
   }
   else
   {
      sendCountryStateStatus(PositionStatusInfo(), PositionInfo());
   }
}


/***********************************************************************//**

 ***************************************************************************/
void clSDS_Property_NaviCurrentCountryState::onUnavailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (_naviSDSProxy == proxy)
   {
      _naviSDSProxy->sendPositionInfoDeregisterAll();
      _naviSDSProxy->sendPositionStatusInfoDeregisterAll();
   }
}


/***********************************************************************//**

 ***************************************************************************/
void clSDS_Property_NaviCurrentCountryState::onAvailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (_naviSDSProxy == proxy)
   {
      ETG_TRACE_USR4(("clSDS_Property_NaviCurrentCountryState. Registering for PositionInformation"));
      _naviSDSProxy->sendPositionStatusInfoRegister(*this);
      _naviSDSProxy->sendPositionStatusInfoGet(*this);
      _naviSDSProxy->sendPositionInfoRegister(*this);
      _naviSDSProxy->sendPositionInfoGet(*this);
   }
}


/***********************************************************************//**

 ***************************************************************************/
void clSDS_Property_NaviCurrentCountryState::onPositionInfoError(
   const ::boost::shared_ptr< NavigationSDSServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< PositionInfoError >& /*error*/)
{
}


/***********************************************************************//**

 ***************************************************************************/

void clSDS_Property_NaviCurrentCountryState::onPositionInfoUpdate(
   const ::boost::shared_ptr< NavigationSDSServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< PositionInfoUpdate >& update)
{
   if (_naviSDSProxy->hasPositionStatusInfo())
   {
      if ((clSDS_KDSConfiguration::getMarketRegionType() == clSDS_KDSConfiguration::JPN) ||
            (clSDS_KDSConfiguration::getMarketRegionType() == clSDS_KDSConfiguration::KOR))
      {
         sendCountryStateStatus(_naviSDSProxy->getPositionStatusInfo(), update->getPositionInfo());
      }
      else
      {
         validateAndSendPosition(_naviSDSProxy->getPositionStatusInfo(), update->getPositionInfo());
      }
   }
}


/***********************************************************************//**

 ***************************************************************************/

void clSDS_Property_NaviCurrentCountryState::onPositionStatusInfoError(
   const ::boost::shared_ptr< NavigationSDSServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< PositionStatusInfoError >& /*error*/)
{
}


/***********************************************************************//**

 ***************************************************************************/

void clSDS_Property_NaviCurrentCountryState::onPositionStatusInfoUpdate(
   const ::boost::shared_ptr< NavigationSDSServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< PositionStatusInfoUpdate >& update)
{
   if (_naviSDSProxy->hasPositionInfo())
   {
      validateAndSendPosition(update->getPositionStatusInfo(), _naviSDSProxy->getPositionInfo());
   }
}


/***********************************************************************//**

 ***************************************************************************/
void clSDS_Property_NaviCurrentCountryState::sendCountryStateStatus(const PositionStatusInfo& positionStatus, const PositionInfo& position)
{
   sds2hmi_sdsfi_tclMsgNaviCurrentCountryStateStatus oMessage;
   sds2hmi_fi_tcl_NDSID countryNDSId;
   sds2hmi_fi_tcl_NDSID stateNDSId;
   sds2hmi_fi_tcl_NDSID cityNDSId;

   oMessage.Country.enType = convertNaviStringtoISOCountryCode(positionStatus.getCountryCode());
   oMessage.CountryName.bSet(position.getCountry().c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
   oMessage.State.bSet(positionStatus.getStateCode().c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
   oMessage.StateName.bSet(position.getProvince().c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
   oMessage.CityName.bSet(position.getCity().c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);

   UpdateRegionSpecifierRelations currentCountryDetails;
   if (position.getCurrentCountry().getNdsNamedObjectId() && _pMethodNavDataRegisterDirectNDSUse->getUpdateRegionSpecifierValues(position.getCurrentCountry().getNdsUpdateRegionSpecifier(), currentCountryDetails))
   {
      countryNDSId.u8SupplierId = currentCountryDetails.getSupplierId();
      countryNDSId.u16UpdateRegionId = currentCountryDetails.getUpdateRegionId();
      countryNDSId.u32ProductId = currentCountryDetails.getProductId();
      countryNDSId.u32NDSId = position.getCurrentCountry().getNdsNamedObjectId();
      ETG_TRACE_USR4((" clSDS_Property_NaviCurrentCountryState country: u8SupplierId= %d, u16UpdateRegionId= %d, u32ProductId=%d, u32NDSId=%d",
                      countryNDSId.u8SupplierId, countryNDSId.u16UpdateRegionId, countryNDSId.u32ProductId, countryNDSId.u32NDSId));
   }

   UpdateRegionSpecifierRelations currentStateDetails;
   if (position.getCurrentState().getNdsNamedObjectId() && _pMethodNavDataRegisterDirectNDSUse->getUpdateRegionSpecifierValues(position.getCurrentState().getNdsUpdateRegionSpecifier(), currentStateDetails))
   {
      stateNDSId.u8SupplierId = currentStateDetails.getSupplierId();
      stateNDSId.u16UpdateRegionId = currentStateDetails.getUpdateRegionId();
      stateNDSId.u32ProductId = currentStateDetails.getProductId();
      stateNDSId.u32NDSId = position.getCurrentState().getNdsNamedObjectId();
      ETG_TRACE_USR4((" clSDS_Property_NaviCurrentCountryState state: u8SupplierId= %d, u16UpdateRegionId= %d, u32ProductId=%d, u32NDSId=%d",
                      stateNDSId.u8SupplierId, stateNDSId.u16UpdateRegionId, stateNDSId.u32ProductId, stateNDSId.u32NDSId));
   }

   UpdateRegionSpecifierRelations currentCityDetails;
   if (position.getCurrentCity().getNdsNamedObjectId() && _pMethodNavDataRegisterDirectNDSUse->getUpdateRegionSpecifierValues(position.getCurrentCity().getNdsUpdateRegionSpecifier(), currentCityDetails))
   {
      cityNDSId.u8SupplierId = currentCityDetails.getSupplierId();
      cityNDSId.u16UpdateRegionId = currentCityDetails.getUpdateRegionId();
      cityNDSId.u32ProductId = currentCityDetails.getProductId();
      cityNDSId.u32NDSId = position.getCurrentCity().getNdsNamedObjectId();
      ETG_TRACE_USR4((" clSDS_Property_NaviCurrentCountryState city: u8SupplierId= %d, u16UpdateRegionId= %d, u32ProductId=%d, u32NDSId=%d",
                      cityNDSId.u8SupplierId, cityNDSId.u16UpdateRegionId, cityNDSId.u32ProductId, cityNDSId.u32NDSId));
   }

   oMessage.NDSLocationDetails.CountryNDSIdList.push_back(countryNDSId);
   oMessage.NDSLocationDetails.StateNDSIdList.push_back(stateNDSId);
   oMessage.NDSLocationDetails.CityNDSId = cityNDSId;
   if ((countryNDSId.u32NDSId != 0) || (clSDS_KDSConfiguration::getMarketRegionType() == clSDS_KDSConfiguration::JPN) || (clSDS_KDSConfiguration::getMarketRegionType() == clSDS_KDSConfiguration::KOR))
   {
      vStatus(oMessage);
   }
}


/***********************************************************************//**

 ***************************************************************************/
sds2hmi_fi_tcl_e16_ISOCountryCode::tenType clSDS_Property_NaviCurrentCountryState::convertNaviStringtoISOCountryCode(const std::string& countryCode)
{
   uint32 value = 0;
   for (size_t i = 0; i < countryCode.size(); i++)
   {
      value <<= 5;
      value += std::toupper(countryCode[i]) - '@';
   }
   ETG_TRACE_USR4(("country code %u = '%s'", value, countryCode.c_str()));
   return (sds2hmi_fi_tcl_e16_ISOCountryCode::tenType) value;
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_NaviCurrentCountryState::onNaviGlobalDataForSdsResponse()
{
   sendResponse();
}


void clSDS_Property_NaviCurrentCountryState::sendCountryStateStatusUsingTTFis(std::map<std::string, std::string> positionInfo)
{
   sds2hmi_sdsfi_tclMsgNaviCurrentCountryStateStatus oMessage;

   oMessage.Country.enType =  convertNaviStringtoISOCountryCode(positionInfo["countryCode"]);
   oMessage.CountryName.bSet(positionInfo["country"].c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
   oMessage.State.bSet(positionInfo["stateCode"].c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
   oMessage.StateName.bSet(positionInfo["state"].c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
   oMessage.CityName.bSet(positionInfo["city"].c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
   vStatus(oMessage);
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_NaviCurrentCountryState::validateAndSendPosition(const PositionStatusInfo& positionStatus, const PositionInfo& positionInfo)
{
   if (positionStatus.getIsUniqueIdValid() && positionInfo.getIsUniqueIdValid())
   {
      if (positionStatus.getUniqueId() == positionInfo.getUniqueId())
      {
         sendCountryStateStatus(positionStatus, positionInfo);
      }
   }
}
