/**************************************************************************//**
 * \file       clSDS_Property_CommonSDSConfiguration_Dyna.cpp

 *
 * Common SDS Configuration property implementation
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "Sds2HmiServer/functions/clSDS_Property_CommonSDSConfiguration_Dyna.h"
#include "Sds2HmiServer/functions/clSDS_Property_NavDataDataSetInfo.h"
#include "Sds2HmiServer/functions/clSDS_Property_NaviStatus.h"
#include "application/clSDS_QuickDialListCreator.h"
#include "application/clSDS_ConfigurationFlags.h"
#include "application/clSDS_KDSConfiguration.h"
#include "application/SettingsService.h"
#include "external/sds2hmi_fi.h"
#include "SdsAdapter_Trace.h"


#ifdef DP_DATAPOOL_ID
#define DP_S_IMPORT_INTERFACE_FI
#include "dp_hmi_02_if.h"
#include "dp_tclfc_sds_adapter_SpeechSettings.h"
#endif

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/clSDS_Property_CommonSDSConfiguration_Dyna.cpp.trc.h"
#endif


using namespace org::bosch::cm::navigation::NavigationService;
using namespace tcu_main_fi;
using namespace vehicle_main_fi_types;


/**************************************************************************//**
*
******************************************************************************/
clSDS_Property_CommonSDSConfiguration_Dyna::~clSDS_Property_CommonSDSConfiguration_Dyna()
{
   _pconfigFlags = NULL;
}


/**************************************************************************//**
*
******************************************************************************/
clSDS_Property_CommonSDSConfiguration_Dyna::clSDS_Property_CommonSDSConfiguration_Dyna(ahl_tclBaseOneThreadService* pService,
      ::boost::shared_ptr< Tcu_main_fiProxy > tcuFiProxy,
      ::boost::shared_ptr< NavigationServiceProxy > naviProxy,
      SettingsService* settingsService,
      clSDS_ConfigurationFlags* configFlags,
      clSDS_Property_NavDataDataSetInfo* pNavDataDataSetInfo,
      clSDS_Property_NaviStatus* pNaviStatus,
      boost::shared_ptr< VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy > vehicleProxy)
   : clServerProperty(SDS2HMI_SDSFI_C_U16_COMMONSDSCONFIGURATION_DYNA, pService)
   , _tcuFiProxy(tcuFiProxy)
   , _naviProxy(naviProxy)
   , _pconfigFlags(configFlags)
   , _vehicleProxy(vehicleProxy)
   , _tcuConnected(false)
{
   dp_tclfc_sds_adapter_SpeechSettingsBeepOnlyMode oBeepOnlyMode;
   dp_tclfc_sds_adapter_SpeechSettingsNBestMatchAudio oNBestMatchAudio;
   dp_tclfc_sds_adapter_SpeechSettingsNBestMatchPhoneBook oNBestMatchPhoneBook;
   dp_tclfc_sds_adapter_SpeechSettingsVoiceBargeIn oNVoiceBargeIn;
   _sdsConfigData.phoneBookNBestMatch = oNBestMatchPhoneBook.tGetData();
   _sdsConfigData.audioNBestMatch = oNBestMatchAudio.tGetData();
   _sdsConfigData.beepOnlyMode = oBeepOnlyMode.tGetData();
   _sdsConfigData.voiceBargeIn = (clSDS_ConfigurationFlags::isVoiceBargeInFeatureEnable()) ? oNVoiceBargeIn.tGetData() : false;
   _sdsConfigData.infoAvailable = false;
   _sdsConfigData.trafficServiceAvailable = false;
   _sdsConfigData.unitInKm = false;
   _sdsConfigData.naviAvailable = clSDS_ConfigurationFlags::getNavKey();
   _sdsConfigData.unitInCelsius = false;
   memset((&_sdsConfigData.phoneRelationshipFlags), 0 , sizeof(_sdsConfigData.phoneRelationshipFlags));

   if (settingsService)
   {
      settingsService->addCommonSettingsObserver(this);
   }
   if (pNavDataDataSetInfo)
   {
      pNavDataDataSetInfo->addNavDataObserver(this);
   }
   if (pNaviStatus)
   {
      pNaviStatus->addVDEAvailabilityObserver(this);
   }
   ETG_TRACE_USR4(("clSDS_Property_CommonSDSConfiguration_Dyna::clSDS_Property_CommonSDSConfiguration_Dyna _sdsConfigData.voiceBargeIn = %d", _sdsConfigData.voiceBargeIn));
}


/**************************************************************************//**
*
******************************************************************************/
tVoid clSDS_Property_CommonSDSConfiguration_Dyna::vGet(amt_tclServiceData* /*pInMsg*/)
{
   vSendStatus();
}


/**************************************************************************//**
*
******************************************************************************/
tVoid clSDS_Property_CommonSDSConfiguration_Dyna::vSet(amt_tclServiceData* /*pInMsg*/)
{
   vSendStatus();
}


/**************************************************************************//**
*
******************************************************************************/
tVoid clSDS_Property_CommonSDSConfiguration_Dyna::vUpreg(amt_tclServiceData* /*pInMsg*/)
{
   vSendStatus();
}


/**************************************************************************//**
*
******************************************************************************/
static void traceXmlString(const std::string& str)
{
   const size_t chunkSize = 100;
   for (size_t pos = 0; pos < str.size(); pos += chunkSize)
   {
      std::string chunk = str.substr(pos, chunkSize);
      ETG_TRACE_USR1(("SDS Config '%s'", chunk.c_str()));
   }
}


/**************************************************************************//**
*
******************************************************************************/
tVoid clSDS_Property_CommonSDSConfiguration_Dyna::vSendStatus()
{
   std::string dynamicString = clSDS_XMLStringCreation::getDynamicXmlString(_sdsConfigData);

   ETG_TRACE_USR1(("SDS Config === Dynamic ==="));
   traceXmlString(dynamicString);

   sds2hmi_sdsfi_tclMsgCommonSDSConfiguration_DynaStatus oMessage;
   oMessage.Value.bSet(dynamicString.c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
   vStatus(oMessage);
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_CommonSDSConfiguration_Dyna::onAvailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _tcuFiProxy)
   {
      _tcuFiProxy->sendConnectionUpReg(*this);
   }

   if (proxy == _naviProxy)
   {
      _naviProxy->sendTrafficServiceReceptionStatusRegister(*this);
      _naviProxy->sendTrafficServiceReceptionStatusGet(*this);
   }
   if (proxy == _vehicleProxy)
   {
      _vehicleProxy->sendDistanceUnitUpReg(*this);
      _vehicleProxy->sendDistanceUnitGet(*this);
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_CommonSDSConfiguration_Dyna::onUnavailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _tcuFiProxy)
   {
      _tcuFiProxy->sendConnectionRelUpRegAll();
   }

   if (proxy == _naviProxy)
   {
      _naviProxy->sendTrafficServiceReceptionStatusDeregisterAll();
   }
   if (proxy == _vehicleProxy)
   {
      _vehicleProxy->sendDistanceUnitRelUpRegAll();
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_CommonSDSConfiguration_Dyna::updateTcuAvailability()
{
   _sdsConfigData.infoAvailable = isInfAvailable(_tcuConnected);
   // B_INF_TCU is used in SDSHMI to check the Connect voice menu or Connect In touch on varaint type.
   clSDS_ConfigurationFlags::setDynamicVariable("B_INF_TCU", _tcuConnected);
   clSDS_ConfigurationFlags::setDynamicVariable("B_INF", _sdsConfigData.infoAvailable);
   vSendStatus();
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_CommonSDSConfiguration_Dyna::updateBeepOnlyMode(bool beepOnlyModeStatus)
{
   _sdsConfigData.beepOnlyMode = beepOnlyModeStatus;
   vSendStatus();
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_CommonSDSConfiguration_Dyna::updateVoiceBargeIn(bool voiceBargeInStatus)
{
   ETG_TRACE_USR4(("clSDS_Property_CommonSDSConfiguration_Dyna::updateVoiceBargeIn clSDS_ConfigurationFlags::isVoiceBargeInFeatureEnable() = %d", clSDS_ConfigurationFlags::isVoiceBargeInFeatureEnable()));
   if (clSDS_ConfigurationFlags::isVoiceBargeInFeatureEnable() == TRUE)
   {
      _sdsConfigData.voiceBargeIn = voiceBargeInStatus;
   }
   else
   {
      _sdsConfigData.voiceBargeIn = false;
   }
   ETG_TRACE_USR4(("clSDS_Property_CommonSDSConfiguration_Dyna::updateVoiceBargeIn _sdsConfigData.voiceBargeIn = %d", _sdsConfigData.voiceBargeIn));
   vSendStatus();
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_CommonSDSConfiguration_Dyna::updateNBestMatchAudio(bool audioNBest)
{
   _sdsConfigData.audioNBestMatch = audioNBest;
   vSendStatus();
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_CommonSDSConfiguration_Dyna::updateNBestMatchPhoneBook(bool phoneNBest)
{
   _sdsConfigData.phoneBookNBestMatch = phoneNBest;
   vSendStatus();
}


/**************************************************************************//**
 *
 ******************************************************************************/

void clSDS_Property_CommonSDSConfiguration_Dyna::onConnectionError(
   const ::boost::shared_ptr< Tcu_main_fiProxy >& /*proxy*/,
   const ::boost::shared_ptr< ConnectionError >& /*error*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_CommonSDSConfiguration_Dyna::onConnectionStatus(
   const ::boost::shared_ptr< Tcu_main_fiProxy >& /*proxy*/,
   const ::boost::shared_ptr< ConnectionStatus >& status)
{
   _tcuConnected = false;

   if (status->hasBConnected())
   {
      _tcuConnected = status->getBConnected();
   }

   if (clSDS_ConfigurationFlags::disableTCUforCurrentRegion())
   {
      _tcuConnected = false;
   }
   updateTcuAvailability();
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool clSDS_Property_CommonSDSConfiguration_Dyna::isInfAvailable(bool tcuAvailable)
{
   if ((clSDS_ConfigurationFlags::getInfFeeds()) &&
         (tcuAvailable ||
          clSDS_ConfigurationFlags::getInfTrafficKey() ||
          clSDS_ConfigurationFlags::getInfSXMKey() ||
          clSDS_ConfigurationFlags::getNavKey()))
   {
      return true;
   }
   return false;
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool clSDS_Property_CommonSDSConfiguration_Dyna::isRelationshipUsed(const std::set<std::string>& inUseRelationships, std::string cateroty)
{
   return (inUseRelationships.find(cateroty) != inUseRelationships.end());
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_CommonSDSConfiguration_Dyna::onInUseRelationshipsChanged(const std::set<std::string>& inUseRelationships)
{
   _sdsConfigData.phoneRelationshipFlags.momUsed       = isRelationshipUsed(inUseRelationships, QUICK_DIAL_RELATIONSHIP_CATEGORY_MOM);
   _sdsConfigData.phoneRelationshipFlags.dadUsed       = isRelationshipUsed(inUseRelationships, QUICK_DIAL_RELATIONSHIP_CATEGORY_DAD);
   _sdsConfigData.phoneRelationshipFlags.parentsUsed   = isRelationshipUsed(inUseRelationships, QUICK_DIAL_RELATIONSHIP_CATEGORY_PARENTS);
   _sdsConfigData.phoneRelationshipFlags.brotherUsed   = isRelationshipUsed(inUseRelationships, QUICK_DIAL_RELATIONSHIP_CATEGORY_BROTHER);
   _sdsConfigData.phoneRelationshipFlags.sisterUsed    = isRelationshipUsed(inUseRelationships, QUICK_DIAL_RELATIONSHIP_CATEGORY_SISTER);
   _sdsConfigData.phoneRelationshipFlags.childUsed     = isRelationshipUsed(inUseRelationships, QUICK_DIAL_RELATIONSHIP_CATEGORY_CHILD);
   _sdsConfigData.phoneRelationshipFlags.sonUsed       = isRelationshipUsed(inUseRelationships, QUICK_DIAL_RELATIONSHIP_CATEGORY_SON);
   _sdsConfigData.phoneRelationshipFlags.daughterUsed  = isRelationshipUsed(inUseRelationships, QUICK_DIAL_RELATIONSHIP_CATEGORY_DAUGHTER);
   _sdsConfigData.phoneRelationshipFlags.friendUsed    = isRelationshipUsed(inUseRelationships, QUICK_DIAL_RELATIONSHIP_CATEGORY_FRIEND);
   _sdsConfigData.phoneRelationshipFlags.partnerUsed   = isRelationshipUsed(inUseRelationships, QUICK_DIAL_RELATIONSHIP_CATEGORY_PARTNER);
   _sdsConfigData.phoneRelationshipFlags.wifeUsed      = isRelationshipUsed(inUseRelationships, QUICK_DIAL_RELATIONSHIP_CATEGORY_WIFE);
   _sdsConfigData.phoneRelationshipFlags.husbandUsed   = isRelationshipUsed(inUseRelationships, QUICK_DIAL_RELATIONSHIP_CATEGORY_HUSBAND);
   _sdsConfigData.phoneRelationshipFlags.homeUsed      = isRelationshipUsed(inUseRelationships, QUICK_DIAL_RELATIONSHIP_CATEGORY_HOME);
   _sdsConfigData.phoneRelationshipFlags.officeUsed    = isRelationshipUsed(inUseRelationships, QUICK_DIAL_RELATIONSHIP_CATEGORY_OFFICE);
   _sdsConfigData.phoneRelationshipFlags.assistantUsed = isRelationshipUsed(inUseRelationships, QUICK_DIAL_RELATIONSHIP_CATEGORY_ASSISTANT);
   _sdsConfigData.phoneRelationshipFlags.managerUsed   = isRelationshipUsed(inUseRelationships, QUICK_DIAL_RELATIONSHIP_CATEGORY_MANAGER);
   _sdsConfigData.phoneRelationshipFlags.otherUsed     = isRelationshipUsed(inUseRelationships, QUICK_DIAL_RELATIONSHIP_CATEGORY_OTHER);

   vSendStatus();
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_CommonSDSConfiguration_Dyna::onTrafficServiceReceptionStatusError(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< TrafficServiceReceptionStatusError >& /*error*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_CommonSDSConfiguration_Dyna::onTrafficServiceReceptionStatusUpdate(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< TrafficServiceReceptionStatusUpdate >& update)
{
   std::string TPEGservice = "TPEG";//Premium traffic Source
   std::string TMCservice = "TMC"; //Non-Premium traffic Source(sxm)
   _sdsConfigData.trafficServiceAvailable = false;

   if (update->hasTrafficServiceReceptionStatus())
   {
      TrafficServiceReceptionStatus receptionStatus  =  update->getTrafficServiceReceptionStatus();
      std::string serviceName =  receptionStatus.getServiceName().c_str() ;
      TrafficReceptionState enReceptionStatus = receptionStatus.getTrafficReceptionState();
      unsigned int trafficMessageCount = receptionStatus.getTrafficMessageCount();

      if ((serviceName == TPEGservice)  &&
            ((enReceptionStatus == TrafficReceptionState__OK) ||
             (enReceptionStatus != TrafficReceptionState__OK && trafficMessageCount > 0)))
      {
         _sdsConfigData.trafficServiceAvailable = true;
      }
      else if ((serviceName == TMCservice) &&
               ((enReceptionStatus == TrafficReceptionState__NOT_SUBSCRIBED) ||
                (enReceptionStatus == TrafficReceptionState__NO_ANTENNA) ||
                (enReceptionStatus == TrafficReceptionState__OK)))
      {
         _sdsConfigData.trafficServiceAvailable = true;
      }
      else
      {
         _sdsConfigData.trafficServiceAvailable = false;
      }
   }

   clSDS_ConfigurationFlags::setDynamicVariable("B_INF_TRAFFIC_AVAILABLE", _sdsConfigData.trafficServiceAvailable);
   vSendStatus();
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_CommonSDSConfiguration_Dyna::onNaviAvailabilityStatus()
{
   _sdsConfigData.naviAvailable = (clSDS_ConfigurationFlags::getNavKey());
   updateTcuAvailability();
   vSendStatus();
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Property_CommonSDSConfiguration_Dyna::onVDEAvalabilityStatus()
{
   _sdsConfigData.naviAvailable = (clSDS_ConfigurationFlags::getNavKey());
   updateTcuAvailability();
   vSendStatus();
}


/***************************************************************************
 *
 **************************************************************************/

void clSDS_Property_CommonSDSConfiguration_Dyna::onDistanceUnitError(
   const ::boost::shared_ptr< VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< VEHICLE_MAIN_FI::DistanceUnitError >& /*error*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/

void clSDS_Property_CommonSDSConfiguration_Dyna::onDistanceUnitStatus(
   const ::boost::shared_ptr< VEHICLE_MAIN_FI::VEHICLE_MAIN_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< VEHICLE_MAIN_FI::DistanceUnitStatus >& status)
{
   if (status->hasE8DistanceUnit())
   {
      if (status->getE8DistanceUnit() == ::vehicle_main_fi_types::T_e8_Vehicle_DistanceUnit__Km)
      {
         _sdsConfigData.unitInKm = true;
      }
      else
      {
         _sdsConfigData.unitInKm = false;
      }
      vSendStatus();
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Property_CommonSDSConfiguration_Dyna::hvacTemperatureUnitUpdated(bool isCelsiusON)
{
   if (_sdsConfigData.unitInCelsius != isCelsiusON)
   {
      _sdsConfigData.unitInCelsius = isCelsiusON;
      vSendStatus();
   }
}
