/**************************************************************************//**
 * \file       clSDS_Method_TunerSelectStation.cpp
 *
 * clSDS_Method_TunerSelectStation method class implementation
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "Sds2HmiServer/functions/clSDS_Method_TunerSelectStation.h"
#include "application/GuiService.h"
#include "application/clSDS_FmChannelList.h"
#include "application/clSDS_AmChannelList.h"
#include "application/clSDS_DabChannelList.h"
#include "SdsAdapter_Trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/clSDS_Method_TunerSelectStation.cpp.trc.h"
#endif


#define DP_S_IMPORT_INTERFACE_FI
#include "dp_generic_if.h"

#define MAX_CHANNEL_NUMBER 1000
#define MAX_PRESET_SXM 19


using namespace sxm_audio_main_fi;
using namespace sxm_audio_main_fi_types;
using namespace tuner_main_fi;
using namespace tuner_main_fi_types;
using namespace tunermaster_main_fi;
using namespace tunermaster_main_fi_types;
using namespace sds_gui_fi::SdsGuiService;
using namespace dabtuner_main_fi;
using namespace dabtuner_main_fi_types;

/**************************************************************************//**
 * Destructor
 ******************************************************************************/
clSDS_Method_TunerSelectStation::~clSDS_Method_TunerSelectStation()
{
   _pFmChannelList = NULL;
   _pDabChannelList = NULL;
}


/**************************************************************************//**
 * Constructor
 ******************************************************************************/
clSDS_Method_TunerSelectStation::clSDS_Method_TunerSelectStation(
   ahl_tclBaseOneThreadService* pService,
   ::boost::shared_ptr<Tuner_main_fiProxy > tunerProxy,
   GuiService& guiService,
   ::boost::shared_ptr<Sxm_audio_main_fiProxy > sxmAudioProxy,
   ::boost::shared_ptr< Tunermaster_main_fiProxy > tunerMasterProxy,
   clSDS_FmChannelList* pFmChannelList, clSDS_AmChannelList* pAmChannelList,
   ::boost::shared_ptr<dabtuner_main_fi::Dabtuner_main_fiProxy> tunerDabProxy
   , clSDS_DabChannelList* pDabChannelList)

   : clServerMethod(SDS2HMI_SDSFI_C_U16_TUNERSELECTSTATION, pService)
   , _guiService(guiService)
   , _sxmAudioProxy(sxmAudioProxy)
   , _tunerMasterProxy(tunerMasterProxy)
   , _tunerProxy(tunerProxy)
   , _tunerDabProxy(tunerDabProxy)
   , _pFmChannelList(pFmChannelList)
   , _pAmChannelList(pAmChannelList)
   , _pDabChannelList(pDabChannelList)

{
}


/**************************************************************************//**
 *
 ******************************************************************************/
tVoid clSDS_Method_TunerSelectStation::vMethodStart(amt_tclServiceData* pInMessage)
{
   sds2hmi_sdsfi_tclMsgTunerSelectStationMethodStart oMessage;
   vGetDataFromAmt(pInMessage, oMessage);
   vProcessMessage(oMessage);
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_TunerSelectStation::vProcessMessage(const sds2hmi_sdsfi_tclMsgTunerSelectStationMethodStart& oMessage)
{
   ETG_TRACE_USR1(("oMessage.SelectionType.enType%d", oMessage.SelectionType.enType));
   switch (oMessage.SelectionType.enType)
   {
      case sds2hmi_fi_tcl_e8_TUN_SelectionType::FI_EN_BYFREQUENCY:
      case sds2hmi_fi_tcl_e8_TUN_SelectionType::FI_EN_BYFREQUENCYANDSUBCHANNEL:
      case sds2hmi_fi_tcl_e8_TUN_SelectionType::FI_EN_BYCHANNEL:
         if (bTunerStation(oMessage))
         {
            vSendMethodResult();
            break;
         }
         else
         {
            if (oMessage.Band.enType == sds2hmi_fi_tcl_e8_TUN_Band::FI_EN_XM)
            {
               if (_sxmAudioProxy->isAvailable() == false)
               {
                  vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_CANNOTCOMPLETEACTION);
               }
               else
               {
                  vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_VALUEOUTOFRANGE);
               }
            }
            else
            {
               vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_CANNOTCOMPLETEACTION);
            }

            break;
         }

      case   sds2hmi_fi_tcl_e8_TUN_SelectionType::FI_EN_BYPRESET:
         if (bTuneToPresetChannel(oMessage))
         {
            ETG_TRACE_USR4(("bTuneToPresetChannel : true"));
            vSendMethodResult();
            break;
         }
         else
         {
            vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_PRESETNOTAVAILABLE);
            break;
         }
      case sds2hmi_fi_tcl_e8_TUN_SelectionType::FI_EN_BYID:
         if (bTuneToChannelName(oMessage))
         {
            vSendMethodResult();
         }
         else
         {
            vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_CANNOTCOMPLETEACTION);
         }
         break;
      default:
         vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_CANNOTCOMPLETEACTION);
         break;
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
tBool clSDS_Method_TunerSelectStation::bTunerStation(const sds2hmi_sdsfi_tclMsgTunerSelectStationMethodStart& oMessage)
{
   switch (oMessage.Band.enType)
   {
      case sds2hmi_fi_tcl_e8_TUN_Band::FI_EN_FM:
         _guiService.sendEventSignal(Event__SPEECH_DIALOG_SDS_PLAY_FM);
         ETG_TRACE_USR4(("FM frequency = %s", oMessage.Frequency.szValue));
         _tunerMasterProxy->sendFID_TUNMSTR_S_STATION_SELECTIONStart(*this, T_e8_StationSelection_Option__Frequency_Input,
               T_e8_Band__TUN_MSTR_BAND_FM, (tU32)(OSAL_dAsciiToDouble(oMessage.Frequency.szValue) * 1000), 0, (T_e8_List)0 , 0, 0, T_e8_ModulationType__TUN_MSTR_MODULATION_ANALOG , 0);
         _guiService.sendActivateAudioSourceSignal(AudioSource__FM);
         return TRUE;

      case sds2hmi_fi_tcl_e8_TUN_Band::FI_EN_AM_MW:
         _guiService.sendEventSignal(Event__SPEECH_DIALOG_SDS_PLAY_AM);
         ETG_TRACE_USR4(("AM frequency = %s", oMessage.Frequency.szValue));
         _tunerMasterProxy->sendFID_TUNMSTR_S_STATION_SELECTIONStart(*this, T_e8_StationSelection_Option__Frequency_Input,
               T_e8_Band__TUN_MSTR_BAND_MW, (tU32)(OSAL_dAsciiToDouble(oMessage.Frequency.szValue)), 0, (T_e8_List)0, 0, 0, T_e8_ModulationType__TUN_MSTR_MODULATION_ANALOG , 0);
         _guiService.sendActivateAudioSourceSignal(AudioSource__AM);
         return TRUE;

      case sds2hmi_fi_tcl_e8_TUN_Band::FI_EN_XM:

         if ((oMessage.Value < MAX_CHANNEL_NUMBER) && (_sxmAudioProxy->isAvailable() == true))
         {
            _guiService.sendEventSignal(Event__SPEECH_DIALOG_SDS_PLAY_XM);
            const uint16 cid = (uint16)oMessage.Value;
            _sxmAudioProxy->sendSelectChannelStart(*this, T_e8_SelectChannelMode__XMTUN_SELECT_CID, cid, 0, 0);
            _guiService.sendActivateAudioSourceSignal(AudioSource__XM);
            return TRUE;
         }
         return FALSE;

      case sds2hmi_fi_tcl_e8_TUN_Band::FI_EN_HD_FM:
         _guiService.sendEventSignal(Event__SPEECH_DIALOG_SDS_PLAY_FM);
         ETG_TRACE_USR4(("FM HD frequency = %s", oMessage.Frequency.szValue));
         _tunerMasterProxy->sendFID_TUNMSTR_S_STATION_SELECTIONStart(*this, T_e8_StationSelection_Option__StationName_Input,
               T_e8_Band__TUN_MSTR_BAND_FM, (tU32)(OSAL_dAsciiToDouble(oMessage.Frequency.szValue) * 1000), 0, (T_e8_List)0, 0, 0,
               T_e8_ModulationType__TUN_MSTR_MODULATION_HD, ((oMessage.Value) - 1));
         _guiService.sendActivateAudioSourceSignal(AudioSource__FM);
         return TRUE;

      case sds2hmi_fi_tcl_e8_TUN_Band::FI_EN_HD_AM:
         _guiService.sendEventSignal(Event__SPEECH_DIALOG_SDS_PLAY_AM);
         ETG_TRACE_USR4(("AM HD frequency = %s", oMessage.Frequency.szValue));
         _tunerMasterProxy->sendFID_TUNMSTR_S_STATION_SELECTIONStart(*this, T_e8_StationSelection_Option__StationName_Input,
               T_e8_Band__TUN_MSTR_BAND_MW, (tU32)(OSAL_dAsciiToDouble(oMessage.Frequency.szValue)), 0, (T_e8_List)0, 0, 0,
               T_e8_ModulationType__TUN_MSTR_MODULATION_HD, ((oMessage.Value) - 1));
         _guiService.sendActivateAudioSourceSignal(AudioSource__AM);
         return TRUE;

      default:
         return FALSE;
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
tBool clSDS_Method_TunerSelectStation::bTuneToPresetChannel(const sds2hmi_sdsfi_tclMsgTunerSelectStationMethodStart& oMessage)
{
   unsigned char numberofpresetsFM = 0xFF;
   unsigned char numberofpresetsAM = 0xFF;
   unsigned char numberofpresetsDAB = 0xFF;
   if (DP_S32_NO_ERR == DP_s32GetConfigItem("GenericTunerParameter", "NumberOfPresetsPerBankFM", &numberofpresetsFM, 1))
   {
      ETG_TRACE_USR4(("Number of presets for FM read from KDS"));
   }
   if (DP_S32_NO_ERR == DP_s32GetConfigItem("GenericTunerParameter", "NumberOfPresetsPerBankMW", &numberofpresetsAM, 1))
   {
      ETG_TRACE_USR4(("Number of presets for AM read from KDS"));
   }
   //Read DAB preset informations from KDS
   if (DP_S32_NO_ERR == DP_s32GetConfigItem("GenericTunerParameter", "NumberOfPresetsPerBankDAB", &numberofpresetsDAB, 1))
   {
      ETG_TRACE_USR4(("Number of presets for DAB read from KDS"));
   }

   const uint8 preset = (uint8)oMessage.Value;
   ETG_TRACE_USR4(("Prest value : %d", preset));

   switch (oMessage.Band.enType)
   {
      case sds2hmi_fi_tcl_e8_TUN_Band::FI_EN_DAB:
         if (oMessage.Value > 0 && oMessage.Value <= numberofpresetsDAB)
         {
            _guiService.sendEventSignal(Event__SPEECH_DIALOG_SDS_PLAY_DAB);
            _tunerMasterProxy->sendFID_TUNMSTR_S_STATION_SELECTIONStart(*this, T_e8_StationSelection_Option__Preset_Input,
                  T_e8_Band__TUN_MSTR_BAND_DAB, 0, preset, T_e8_List__BANK_DAB1, 0, 0, T_e8_ModulationType__TUN_MSTR_INVALID_TYPE, 0);
            _guiService.sendActivateAudioSourceSignal(AudioSource__DAB);
            return TRUE;
         }
         return FALSE;
      case sds2hmi_fi_tcl_e8_TUN_Band::FI_EN_FM:

         if (oMessage.Value > 0 && oMessage.Value <= numberofpresetsFM)
         {
            _guiService.sendEventSignal(Event__SPEECH_DIALOG_SDS_PLAY_FM);
            _tunerMasterProxy->sendFID_TUNMSTR_S_STATION_SELECTIONStart(*this, T_e8_StationSelection_Option__Preset_Input,
                  T_e8_Band__TUN_MSTR_BAND_FM, 0, preset, T_e8_List__BANK_FM1, 0, 0, T_e8_ModulationType__TUN_MSTR_INVALID_TYPE, 0);

            /* _tunerMasterProxy->sendFID_TUNMSTR_S_PRESET_HANDLINGStart(*this, T_e8_Preset_Action__TUNMSTR_PresetRecall, preset, T_e8_List__BANK_FM1, T_e8_Band__TUN_MSTR_BAND_FM);*/
            _guiService.sendActivateAudioSourceSignal(AudioSource__FM);
            return TRUE;
         }
         return FALSE;

      case sds2hmi_fi_tcl_e8_TUN_Band::FI_EN_AM_MW:
         if (oMessage.Value > 0 && oMessage.Value <= numberofpresetsAM)
         {
            _guiService.sendEventSignal(Event__SPEECH_DIALOG_SDS_PLAY_AM);
            _tunerMasterProxy->sendFID_TUNMSTR_S_STATION_SELECTIONStart(*this, T_e8_StationSelection_Option__Preset_Input,
                  T_e8_Band__TUN_MSTR_BAND_MW, 0, preset, T_e8_List__BANK_AM_MW1, 0, 0, T_e8_ModulationType__TUN_MSTR_INVALID_TYPE, 0);
            _guiService.sendActivateAudioSourceSignal(AudioSource__AM);
            return TRUE;
         }
         return FALSE;

      case sds2hmi_fi_tcl_e8_TUN_Band::FI_EN_XM:
         if (_sxmAudioProxy->isAvailable() == true)
         {
            if (oMessage.Value > 0 && oMessage.Value <= MAX_PRESET_SXM)

            {
               _guiService.sendEventSignal(Event__SPEECH_DIALOG_SDS_PLAY_XM);
               _sxmAudioProxy->sendRecallPresetStart(*this, preset);
               _guiService.sendActivateAudioSourceSignal(AudioSource__XM);
               return TRUE;
            }
            return FALSE;
         }
         return FALSE;

      default:
         return FALSE;
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
bool clSDS_Method_TunerSelectStation::bTuneToChannelName(const sds2hmi_sdsfi_tclMsgTunerSelectStationMethodStart& oMessage)
{
   ETG_TRACE_USR4(("bTuneToChannelName type : %d", oMessage.Band.enType));
   switch (oMessage.Band.enType)
   {
      case sds2hmi_fi_tcl_e8_TUN_Band::FI_EN_DAB:  // for DAB
      {
         _guiService.sendEventSignal(Event__SPEECH_DIALOG_SDS_PLAY_DAB);
         T_e8_DAB_ChnSelectMode selectMode = T_e8_DAB_ChnSelectMode__DAB_CHNSELECTMODE_PSDI;
         //TBD what MW will update
         uint32 u32Value = _pDabChannelList->getDABIDForObjectID(oMessage.Value);
         _tunerDabProxy->sendFID_DAB_S_SRV_SELECTStart(*this, selectMode, u32Value, 0 , false);
         _guiService.sendActivateAudioSourceSignal(AudioSource__DAB);
      }
      return true;

      case sds2hmi_fi_tcl_e8_TUN_Band::FI_EN_FM:
      case sds2hmi_fi_tcl_e8_TUN_Band::FI_EN_RDS_FM:// for RDS
      {
         _guiService.sendEventSignal(Event__SPEECH_DIALOG_SDS_PLAY_FM);
         clSDS_FMTunerObject fmObj = _pFmChannelList->getPICodeAndFrequencyFromObjectID(oMessage.Value);
         _tunerMasterProxy->sendFID_TUNMSTR_S_STATION_SELECTIONStart(*this, T_e8_StationSelection_Option__StationName_Input,
               T_e8_Band__TUN_MSTR_BAND_FM, fmObj.frequency, 0, (T_e8_List)0, fmObj.picode, 0, (T_e8_ModulationType)0, 0);
         _guiService.sendActivateAudioSourceSignal(AudioSource__FM);
      }
      return true;

      case sds2hmi_fi_tcl_e8_TUN_Band::FI_EN_HD_FM:
      {
         _guiService.sendEventSignal(Event__SPEECH_DIALOG_SDS_PLAY_FM);
         uint32 freq = _pFmChannelList->getFrequencyForObjectIdHD(oMessage.Value);
         uint8 audioProg = _pFmChannelList->getAudioProgramForObjectID(oMessage.Value);
         _tunerMasterProxy->sendFID_TUNMSTR_S_STATION_SELECTIONStart(*this, T_e8_StationSelection_Option__StationName_Input,
               T_e8_Band__TUN_MSTR_BAND_FM, freq , 0, (T_e8_List)0, 0, 0,
               T_e8_ModulationType__TUN_MSTR_MODULATION_HD, audioProg);
         _guiService.sendActivateAudioSourceSignal(AudioSource__FM);
      }
      return true;

      case sds2hmi_fi_tcl_e8_TUN_Band::FI_EN_HD_AM:
      {
         _guiService.sendEventSignal(Event__SPEECH_DIALOG_SDS_PLAY_AM);
         uint32 freq = _pAmChannelList->getFrequencyForObjectID(oMessage.Value);
         tU8 audioProg = _pAmChannelList->getAudioProgramForObjectID(oMessage.Value);
         _tunerMasterProxy->sendFID_TUNMSTR_S_STATION_SELECTIONStart(*this, T_e8_StationSelection_Option__StationName_Input,
               T_e8_Band__TUN_MSTR_BAND_MW, freq , 0, (T_e8_List)0, 0, 0,
               T_e8_ModulationType__TUN_MSTR_MODULATION_HD, audioProg);
         _guiService.sendActivateAudioSourceSignal(AudioSource__AM);
      }
      return true;

      case sds2hmi_fi_tcl_e8_TUN_Band::FI_EN_XM:
         return (bTunerStation(oMessage) == TRUE);

      default:
         return false;
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_TunerSelectStation::onSelectChannelError(const ::boost::shared_ptr<Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< SelectChannelError >& /*result*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_TunerSelectStation::onSelectChannelResult(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< SelectChannelResult >& /*result*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_TunerSelectStation::onFID_TUNMSTR_S_PRESET_HANDLINGError(const ::boost::shared_ptr< Tunermaster_main_fiProxy >& /*proxy*/, const ::boost::shared_ptr< FID_TUNMSTR_S_PRESET_HANDLINGError >& /*error*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_TunerSelectStation::onFID_TUNMSTR_S_PRESET_HANDLINGResult(const ::boost::shared_ptr< Tunermaster_main_fiProxy >& /*proxy*/, const ::boost::shared_ptr< FID_TUNMSTR_S_PRESET_HANDLINGResult >& /*result*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_TunerSelectStation::onRecallPresetError(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/, const ::boost::shared_ptr< RecallPresetError >& /*error*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_TunerSelectStation::onRecallPresetResult(const ::boost::shared_ptr< Sxm_audio_main_fiProxy >& /*proxy*/, const ::boost::shared_ptr< RecallPresetResult >& /*result*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_TunerSelectStation::onFID_TUN_S_SET_PIError(const ::boost::shared_ptr< Tuner_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< FID_TUN_S_SET_PIError >& /*error*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_TunerSelectStation::onFID_TUN_S_SET_PIResult(const ::boost::shared_ptr< Tuner_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< FID_TUN_S_SET_PIResult >& /*result*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_TunerSelectStation::onFID_TUN_S_SEL_HD_AUDIOPRGMDIRECTError(const ::boost::shared_ptr< Tuner_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< FID_TUN_S_SEL_HD_AUDIOPRGMDIRECTError >& /*error*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_TunerSelectStation::onFID_TUN_S_SEL_HD_AUDIOPRGMDIRECTResult(const ::boost::shared_ptr< Tuner_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< FID_TUN_S_SEL_HD_AUDIOPRGMDIRECTResult >& /*result*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_TunerSelectStation::onFID_DAB_S_SRV_SELECTError(const ::boost::shared_ptr< dabtuner_main_fi::Dabtuner_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< dabtuner_main_fi::FID_DAB_S_SRV_SELECTError >& /*error*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_TunerSelectStation::onFID_DAB_S_SRV_SELECTResult(const ::boost::shared_ptr< dabtuner_main_fi::Dabtuner_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< dabtuner_main_fi::FID_DAB_S_SRV_SELECTResult >& /*result*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_TunerSelectStation::onFID_TUNMSTR_S_STATION_SELECTIONError(const ::boost::shared_ptr< tunermaster_main_fi::Tunermaster_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< tunermaster_main_fi::FID_TUNMSTR_S_STATION_SELECTIONError >& /*error*/)
{
   ETG_TRACE_USR4(("onFID_TUNMSTR_S_STATION_SELECTIONError"));
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_TunerSelectStation::onFID_TUNMSTR_S_STATION_SELECTIONResult(const ::boost::shared_ptr< tunermaster_main_fi::Tunermaster_main_fiProxy >& /*proxy*/,
      const ::boost::shared_ptr< tunermaster_main_fi::FID_TUNMSTR_S_STATION_SELECTIONResult >& result)
{
   //int result = result.getOFID_TUNMSTR_S_STATION_SELECTION().getTunerMaster_StatusResponse().getU8TunermasterStatus();
   //int source = result.getOFID_TUNMSTR_S_STATION_SELECTION().getSource();
   //uint8 action = result->getAction();
   //ETG_TRACE_USR4(("onFID_TUNMSTR_S_STATION_SELECTIONResult : result : %d",result ));
   //ETG_TRACE_USR4(("onFID_TUNMSTR_S_STATION_SELECTIONResult : source : %d",source ));
//	ETG_TRACE_USR4(("onFID_TUNMSTR_S_STATION_SELECTIONResult : result : %d",action ));
}
