/**************************************************************************//**
 * \file       clSDS_Method_PhoneDialContact.cpp
 *
 * This file is part of the SdsAdapter component.
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/

#include "Sds2HmiServer/functions/clSDS_Method_PhoneDialContact.h"
#include "application/clSDS_StringVarList.h"
#include "application/AudioSourceHandler.h"
#include "application/GuiService.h"
#include "SdsAdapter_Trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/clSDS_Method_PhoneDialContact.cpp.trc.h"
#endif


using namespace MOST_Tel_FI;
using namespace MOST_PhonBk_FI;
using namespace most_PhonBk_fi_types;
using namespace sds_gui_fi::SdsGuiService;


clSDS_Method_PhoneDialContact::clSDS_Method_PhoneDialContact(ahl_tclBaseOneThreadService* pService
      , ::boost::shared_ptr< MOST_Tel_FIProxy > pTelProxy
      , ::boost::shared_ptr< MOST_PhonBk_FIProxy > pPhoneBkProxy
      , AudioSourceHandler* pAudioSourceHandler
      , GuiService& guiService)
   : clServerMethod(SDS2HMI_SDSFI_C_U16_PHONEDIALCONTACT, pService)
   , _telephoneProxy(pTelProxy)
   , _phoneBookProxy(pPhoneBkProxy)
   , _pAudioSourceHandler(pAudioSourceHandler)
   , _guiService(guiService)
{
   _numberType.enType = sds2hmi_fi_tcl_e8_PHN_NumberType::FI_EN_UNKNOWN;
}


clSDS_Method_PhoneDialContact::~clSDS_Method_PhoneDialContact()
{
   _pAudioSourceHandler = NULL;
}


void clSDS_Method_PhoneDialContact::vMethodStart(amt_tclServiceData* pInMsg)
{
   sds2hmi_sdsfi_tclMsgPhoneDialContactMethodStart oMessage;
   vGetDataFromAmt(pInMsg, oMessage);

   _numberType = oMessage.LocationType;

   ETG_TRACE_USR4(("clSDS_Method_PhoneDialContact::vMethodStart - numberType = %d", oMessage.LocationType.enType));
   ETG_TRACE_USR4(("clSDS_Method_PhoneDialContact::vMethodStart - contactID = %d", oMessage.ContactID));

   if (_phoneBookProxy->isAvailable())
   {
      _phoneBookProxy->sendGetContactDetailsStart(*this, oMessage.ContactID, T_e8_PhonBkContactDetailFilter__e8CDF_TELEPHONE);
   }
   else
   {
      vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_CANNOTCOMPLETEACTION);
   }
}


void clSDS_Method_PhoneDialContact::onDialError(const ::boost::shared_ptr< MOST_Tel_FIProxy >& /*proxy*/, const ::boost::shared_ptr< DialError >& /*error*/)
{
   vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_CANNOTCOMPLETEACTION);
}


void clSDS_Method_PhoneDialContact::onDialResult(const ::boost::shared_ptr< MOST_Tel_FIProxy >& /*proxy*/, const ::boost::shared_ptr< DialResult >& /*result*/)
{
   vSendMethodResult();
}


void clSDS_Method_PhoneDialContact::onGetContactDetailsError(const ::boost::shared_ptr< MOST_PhonBk_FIProxy >& /*proxy*/, const ::boost::shared_ptr< GetContactDetailsError >& /*error*/)
{
   vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_CANNOTCOMPLETEACTION);
}


void clSDS_Method_PhoneDialContact::onGetContactDetailsResult(const ::boost::shared_ptr< MOST_PhonBk_FIProxy >& /*proxy*/, const ::boost::shared_ptr< GetContactDetailsResult >& result)
{
   if (_telephoneProxy->isAvailable() && _pAudioSourceHandler)
   {
      std::string phoneNumber = getPhoneNumber(result->getOContactDetails());

      clSDS_StringVarList::vSetVariable("$(Number)", phoneNumber);

      _guiService.sendEventSignal(Event__SPEECH_DIALOG_SDS_PHONE_CALL_SIRI_VA);
      _telephoneProxy->sendDialStart(*this, phoneNumber, ::most_Tel_fi_types::T_e8_TelEchoCancellationNoiseReductionSetting__e8ECNR_NOCHANGE);
      _pAudioSourceHandler->onOutgoingCall();
   }
   else
   {
      vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_CANNOTCOMPLETEACTION);
   }
}


std::string clSDS_Method_PhoneDialContact::getPhoneNumber(const T_PhonBkContactDetails& contactDetails) const
{
   typedef std::map<sds2hmi_fi_tcl_e8_PHN_NumberType::tenType, std::string> NumberMap;
   NumberMap numbers;
   numbers[sds2hmi_fi_tcl_e8_PHN_NumberType::FI_EN_HOME1] = contactDetails.getSHomeNumber1();
   numbers[sds2hmi_fi_tcl_e8_PHN_NumberType::FI_EN_HOME2] = contactDetails.getSHomeNumber2();
   numbers[sds2hmi_fi_tcl_e8_PHN_NumberType::FI_EN_OFFICE1] = contactDetails.getSWorkNumber1();
   numbers[sds2hmi_fi_tcl_e8_PHN_NumberType::FI_EN_OFFICE2] = contactDetails.getSWorkNumber2();
   numbers[sds2hmi_fi_tcl_e8_PHN_NumberType::FI_EN_MOBILE1] = contactDetails.getSCellNumber1();
   numbers[sds2hmi_fi_tcl_e8_PHN_NumberType::FI_EN_MOBILE2] = contactDetails.getSCellNumber2();
   numbers[sds2hmi_fi_tcl_e8_PHN_NumberType::FI_EN_OTHER] = contactDetails.getSOtherNumber();
   numbers[sds2hmi_fi_tcl_e8_PHN_NumberType::FI_EN_UNKNOWN] = contactDetails.getSPreferredNumber();

   std::string phoneNumber = numbers[_numberType.enType];

   for (NumberMap::const_iterator it = numbers.begin(); (phoneNumber == "") && (it != numbers.end()); ++it)
   {
      phoneNumber = it->second;
   }

   ETG_TRACE_USR4(("clSDS_Method_PhoneDialContact::getPhoneNumber - phoneNumber = %s", phoneNumber.c_str()));

   return phoneNumber;
}
