/**************************************************************************//**
 * \file       clSDS_Method_NaviSetDestinationItem.cpp
 *
 * clSDS_Method_NaviSetDestinationItem method class implementation
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "Sds2HmiServer/functions/clSDS_Method_NaviSetDestinationItem.h"
#include "application/clSDS_ListScreen.h"
#include "application/clSDS_NaviListItems.h"
#include "application/clSDS_MultipleDestinationsList.h"
#include "application/clSDS_POIList.h"
#include "application/QuickChargingStation.h"
#include "application/StringUtils.h"
#include "Sds2HmiServer/functions/clSDS_Method_CommonShowDialog.h"
#include "application/CountryNameRequestor.h"
#include "SdsAdapter_Trace.h"
#include "application/GuiService.h"
#include "application/NaviHapticHandOverInterface.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/clSDS_Method_NaviSetDestinationItem.cpp.trc.h"
#endif


/**
 * Processing timeout for NaviSetDestinationItem in milliseconds.
 * The timeout is chosen 2 seconds shorter than the timeout in
 * SDS MW.
 */
#define NAVI_SET_DESTINATION_TIMEOUT   38000


using namespace org::bosch::cm::navigation::NavigationService;
using namespace sds_gui_fi::SdsGuiService;


/**************************************************************************//**
 * Destructor
 ******************************************************************************/
clSDS_Method_NaviSetDestinationItem::~clSDS_Method_NaviSetDestinationItem()
{
   _pMultipleDestiantionList = NULL;
   _pPOIList = NULL;
   _pNaviListItems = NULL;
   _pCommonShowDialog = NULL;
   _pGuiService = NULL;
   _pNdsCountryStateRequestor = NULL;
   _pNaviHapticHandOverInterface = NULL;
}


/**************************************************************************//**
 * Constructor
 ******************************************************************************/
clSDS_Method_NaviSetDestinationItem::clSDS_Method_NaviSetDestinationItem(
   ahl_tclBaseOneThreadService* pService,
   ::boost::shared_ptr<NavigationServiceProxy> naviProxy,
   clSDS_MultipleDestinationsList* pMultipleDestinationList,
   clSDS_POIList* pPOIList,
   clSDS_NaviListItems* pNaviListItems,
   clSDS_Method_CommonShowDialog* pCommonShowDialog,
   CountryNameRequestor* pCountryNameRequestor,
   QuickChargingStation* pQuickChargingStation,
   NaviHapticHandOverInterface* pNaviHapticHandOverInterface,
   clSDS_LanguageMediator* pLanguageMediator,
   NdsCountryStateRequestor* pNdsCountryStateRequestor,
   GuiService* pGuiService)

   : clServerMethod(SDS2HMI_SDSFI_C_U16_NAVISETDESTINATIONITEM, pService, NAVI_SET_DESTINATION_TIMEOUT)
   , _navigationProxy(naviProxy)
   , _pMultipleDestiantionList(pMultipleDestinationList)
   , _pPOIList(pPOIList)
   , _pNaviListItems(pNaviListItems)
   , _pCommonShowDialog(pCommonShowDialog)
   , _pCountryNameRequestor(pCountryNameRequestor)
   , _pQuickChargingStation(pQuickChargingStation)
   , _houseNumberSetForFTS(false)
   , _refinementValid(false)
   , _countryStateInfoRequested(false)
   , _pLanguageMediator(pLanguageMediator)
   , _pNdsCountryStateRequestor(pNdsCountryStateRequestor)
   , _pGuiService(pGuiService)
{
   _pMultipleDestiantionList->vRegisterAmbiguityListObserver(this);
   _refinmentType = NO_REFINEMENT_SELECTION;
   _pCountryNameRequestor->addCountryObserver(this);
   _pQuickChargingStation->registerChargingStationListObserver(this);
   _pNdsCountryStateRequestor->addCountryStateObserver(this);
   _pNaviHapticHandOverInterface = pNaviHapticHandOverInterface;
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_Method_NaviSetDestinationItem::onUnavailable(const boost::shared_ptr<asf::core::Proxy>& proxy,
      const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _navigationProxy)
   {
      _navigationProxy->sendSdsAddressWithOptionsDeregisterAll();
      _navigationProxy->sendDestinationInformationDeregisterAll();
      _navigationProxy->sendMenuPoiCategoryShortcutsDeregisterAll();
   }
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_Method_NaviSetDestinationItem::onAvailable(const boost::shared_ptr<asf::core::Proxy>& proxy,
      const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _navigationProxy)
   {
      _navigationProxy->sendSdsAddressWithOptionsRegister(*this);
      _navigationProxy->sendDestinationInformationRegister(*this);
      _navigationProxy->sendMenuPoiCategoryShortcutsRegister(*this);
      _navigationProxy->sendMenuPoiCategoryShortcutsGet(*this);
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviSetDestinationItem::onMenuPoiCategoryShortcutsError(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< MenuPoiCategoryShortcutsError >& /*error*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviSetDestinationItem::onMenuPoiCategoryShortcutsUpdate(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< MenuPoiCategoryShortcutsUpdate >& update)
{
   if (update->hasMenuPoiCategoryShortcuts())
   {
      _sdsPOICategory.clear();
      _sdsPOICategory = update->getMenuPoiCategoryShortcuts();
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviSetDestinationItem::onSetLocationWithFixedPOICategoryInputError(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< SetLocationWithFixedPOICategoryInputError >& /*error*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviSetDestinationItem::onSetLocationWithFixedPOICategoryInputResponse(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< SetLocationWithFixedPOICategoryInputResponse >& /*response*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviSetDestinationItem::onSdsCheckAddressError(const ::boost::shared_ptr<NavigationServiceProxy>& /*proxy*/,
      const ::boost::shared_ptr< SdsCheckAddressError >& /*Error*/)
{
   vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_DESTINATIONVALUENOTRESOLVED);
   _oRequestedsdsAddress.clear();
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviSetDestinationItem::onSdsCheckAddressResponse(const ::boost::shared_ptr<NavigationServiceProxy >& /*proxy*/,
      const ::boost::shared_ptr< SdsCheckAddressResponse >& /*osdsCheckAddressResponse*/)
{
   ETG_TRACE_USR1(("onSdsCheckAddressResponse()- Function entered"));

   //Navi hall will send the response of SdsCheckAddress in form of property update.The below function will handle this.
   getAddressOptionResultFromNavi();
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviSetDestinationItem::onSdsAddressWithOptionsUpdate(const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
      const ::boost::shared_ptr< SdsAddressWithOptionsUpdate >& /*oSdsAddressWithOptionsUpdate*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviSetDestinationItem::onSdsAddressWithOptionsError(const ::boost::shared_ptr<NavigationServiceProxy>& /*proxy*/,
      const ::boost::shared_ptr< SdsAddressWithOptionsError >& /*error*/)
{
   vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_DESTINATIONVALUENOTRESOLVED);
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviSetDestinationItem::onSetLocationWithSdsInputError(const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
      const ::boost::shared_ptr< SetLocationWithSdsInputError >& /*error*/)
{
   vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_CANNOTCOMPLETEACTION);
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviSetDestinationItem::onSetLocationWithSdsInputResponse(const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
      const ::boost::shared_ptr< SetLocationWithSdsInputResponse >& /*response*/)
{
   _navigationProxy->sendRequestDetailsForSelectedLocationRequest(*this);
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviSetDestinationItem::onSelectSdsRefinementError(const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
      const ::boost::shared_ptr< SelectSdsRefinementError >& /*error*/)
{
   vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_CANNOTCOMPLETEACTION);
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviSetDestinationItem::onSelectSdsRefinementResponse(const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
      const ::boost::shared_ptr< SelectSdsRefinementResponse >& /*response*/)
{
   ETG_TRACE_USR1(("onSdsSelectRefinementResponse()- Function entered"));

   if (_refinmentType == POI_SEARCH_REFINEMENT_SELECTION)
   {
      ETG_TRACE_USR1(("onSdsSelectRefinementResponse()- GettingPOIList"));
      requestPOIListToNavi();
   }
   else if (_refinmentType == ADDRESS_SEARCH_REFINEMENT_SELECTION)
   {
      ETG_TRACE_USR1(("onSdsSelectRefinementResponse()- Handling Addressoptions"));
      getAddressOptionResultFromNavi();
   }
   else
   {
      ETG_TRACE_USR1(("onSdsSelectRefinementResponse()-SelectionCriteria is wrong"));
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviSetDestinationItem::onDestinationInformationError(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< DestinationInformationError >& /*error*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviSetDestinationItem::onDestinationInformationUpdate(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< DestinationInformationUpdate >& update)
{
   ETG_TRACE_USR1(("clSDS_Method_NaviSetDestinationItem::onDestinationInformationUpdate() - destinationInformationSize = %d", update->getDestinationInformation().size()));
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviSetDestinationItem::onRequestFreeTextSearchResultsError(const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
      const ::boost::shared_ptr< RequestFreeTextSearchResultsError >& /*error*/)
{
   vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_DESTINATIONVALUENOTRESOLVED);
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviSetDestinationItem::onRequestFreeTextSearchResultsResponse(const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
      const ::boost::shared_ptr< RequestFreeTextSearchResultsResponse >& response)
{
   ETG_TRACE_USR1(("onRequestFreeTextSearchResultsResponse()- Function entered"));

   _pPOIList->setPOIList(response->getSearchResults());

   tU16 u16NumberofElements = (tU16) response->getSearchResults().size();

   ETG_TRACE_USR1(("onRequestFreeTextSearchResultsResponse()- u16NumberofElements = %d", u16NumberofElements));

   sendPOIResult(u16NumberofElements);

   _pPOIList->resetPOIListViaHaptic();
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviSetDestinationItem::onSetLocationWithFreeTextSearchInputError(const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
      const ::boost::shared_ptr< SetLocationWithFreeTextSearchInputError >& /*error*/)
{
   vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_CANNOTCOMPLETEACTION);
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviSetDestinationItem::onSetLocationWithFreeTextSearchInputResponse(const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
      const ::boost::shared_ptr< SetLocationWithFreeTextSearchInputResponse >& /*response*/)
{
   ETG_TRACE_USR1(("SetLocationWithFreeTextSearchInputRequestResponse is received"));
}


/***********************************************************************//**
 *
 ***************************************************************************/

void clSDS_Method_NaviSetDestinationItem::onRequestDetailsForSelectedLocationError(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< RequestDetailsForSelectedLocationError >& /*error*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviSetDestinationItem::onRequestDetailsForSelectedLocationResponse(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< RequestDetailsForSelectedLocationResponse >& /*response*/)
{
   if (_pGuiService)
   {
      _pGuiService->sendEventSignal(Event__SPEECH_DIALOG_GADGET_ACTIVATION);
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/


/**************************************************************************//**
 * nNumberOfElements represents tLocationDescription list size
 * nNumberOfElements can be either 0 or 1
 * nNumberOfElements == 1 and nAmbiguityAbsVal == 0xFF means that the ambiguity list has to be displayed
 * nNumberOfElements == 0 and nAmbiguityAbsVal <> 0xFF means that one element was selected from the ambiguity list
 ******************************************************************************/
tVoid clSDS_Method_NaviSetDestinationItem::vMethodStart(amt_tclServiceData* pInMessage)
{
   sds2hmi_sdsfi_tclMsgNaviSetDestinationItemMethodStart oMethodStart;
   vGetDataFromAmt(pInMessage, oMethodStart);

   _locationType = oMethodStart.tLocationType;
   _locationDescriptors = oMethodStart.tLocationDescription.Descriptors;

   if (_locationDescriptors.size() > 0)
   {
      listSelectionCriterionChanged(_locationDescriptors[0].SelectionCriterion.enType);
   }

   ETG_TRACE_USR1(("nAmbiguityAbsVal from SDS = %d", oMethodStart.nAmbiguityAbsVal));
   if (oMethodStart.nAmbiguityAbsVal == 0xFF)
   {
      // first step is to search the given Address or POI
      // afterwards only Navi service will inform whether the entry it's ambiguous or not
      if (!_pNdsCountryStateRequestor->isCountryStateInfoAvailable() && (clSDS_KDSConfiguration::getMarketRegionType() == clSDS_KDSConfiguration::USA))
      {
         // only for addresses that contain "state:" information, the country has to be determined before calling sendAddressRequestToNavi();
         // the retrieval operation is finished when countriesUpdated() is received
         _pNdsCountryStateRequestor->getNdsCountryStateIds();
         _countryStateInfoRequested = true;
      }
      else
      {
         sendAddressRequestToNavi();
      }
   }
   else
   {
      ETG_TRACE_USR1(("oMethodStart.nAmbiguityAbsVal != 0xFF"));
      // an entry was selected from the Ambiguty List
      sendListEntrySelectionToNavi(oMethodStart.nAmbiguityAbsVal);
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_NaviSetDestinationItem::listSelectionCriterionChanged(sds2hmi_fi_tcl_e16_SelectionCriterionType::tenType listType)
{
   if (_pNaviListItems)
   {
      _pNaviListItems->vSetNaviSelectionCriterionListType(listType);
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_NaviSetDestinationItem::sendAddressRequestToNavi()
{
   ETG_TRACE_USR1(("sendAddressRequestToNavi"));
   if (isHouseNumberRequest())
   {
      ETG_TRACE_USR1(("isHouseNumberRequest true"));
      handleHouseNrRequest();
   }
   else if (isOsdeAddressRequest())
   {
      ETG_TRACE_USR1(("isOsdeAddressRequest true"));
      handleOSDERequest();
   }
   else if (isPOISearchRequest())
   {
      ETG_TRACE_USR1(("handlePOIRequest true"));
      handlePOIRequest();
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
bool clSDS_Method_NaviSetDestinationItem::isHouseNumberRequest() const
{
   ETG_TRACE_USR1(("isHouseNumberRequest"));
   if (_locationDescriptors.size() > 0)
   {
      ETG_TRACE_USR1(("isHouseNumberRequest return true :%d", _locationDescriptors[0].SelectionCriterion.enType));
      return (_locationDescriptors[0].SelectionCriterion.enType == sds2hmi_fi_tcl_e16_SelectionCriterionType::FI_EN_HOUSENUMBER);
   }
   ETG_TRACE_USR1(("isHouseNumberRequest return false"));
   return FALSE;
}


/***********************************************************************//**
 *
 ***************************************************************************/
bool clSDS_Method_NaviSetDestinationItem::isPOISearchRequest() const
{
   if (_locationDescriptors.size() > 0)
   {
      return (_locationDescriptors[0].SelectionCriterion.enType == sds2hmi_fi_tcl_e16_SelectionCriterionType::FI_EN_OSDE_POI);
   }
   return FALSE;
}


/***********************************************************************//**
 *
 ***************************************************************************/
bool clSDS_Method_NaviSetDestinationItem::isOsdeAddressRequest() const
{
   ETG_TRACE_USR1(("isOsdeAddressRequest"));
   if (_locationDescriptors.size() > 0)
   {
      ETG_TRACE_USR1(("isOsdeAddressRequest return true :%d", _locationDescriptors[0].SelectionCriterion.enType));
      return (_locationDescriptors[0].SelectionCriterion.enType == sds2hmi_fi_tcl_e16_SelectionCriterionType::FI_EN_OSDE_ADDRESSES);
   }
   ETG_TRACE_USR1(("isOsdeAddressRequest return false"));
   return FALSE;
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviSetDestinationItem::handleOSDERequest()
{
   ETG_TRACE_USR1(("handleOSDERequest"));
   getContentFromDescriptor();
   sendSdsCheckAddressRequest();
}


/***********************************************************************//**
 *
 ***************************************************************************/
tVoid clSDS_Method_NaviSetDestinationItem::getAddressOptionResultFromNavi()
{
   ETG_TRACE_USR1(("getAddressOptionResultFromNavi"));
   resetResultValues();

   std::vector< SDSAddressElement> verifiedSdsAddress;
   SDSAddressOptions sdsAddressOptions;
   std::vector< SDSAddressElementType > sdsAddressElementType;

   if (_navigationProxy->hasSdsAddressWithOptions())
   {
      ETG_TRACE_USR1(("getAddressOptionResultFromNavi hasSdsAddressWithOptions"));
      verifiedSdsAddress = _navigationProxy->getSdsAddressWithOptions().getAddress();
      sdsAddressOptions = _navigationProxy->getSdsAddressWithOptions().getAddressOptions();
      sdsAddressElementType = _navigationProxy->getSdsAddressWithOptions().getAddressElementType();
   }

   handleVerifiedSdsAddress(verifiedSdsAddress);
   validateSdsAddressOptions(sdsAddressOptions, verifiedSdsAddress, sdsAddressElementType);
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviSetDestinationItem::getContentFromDescriptor()
{
   ETG_TRACE_USR1(("clSDS_Method_NaviSetDestinationItem::getContentFromDescriptor()"));

   _oPOIName.BrandName = getTaggedContent("attr:");
   _oPOIName.CategoryName = getTaggedContent("cat:");
   _oPOIName.BuildingName = getTaggedContent("poi:");

   const std::string stateName = getTaggedContent("state:");
   bool replacedStateName = false;

   if (stateName != "" && (clSDS_KDSConfiguration::USA == clSDS_KDSConfiguration::getMarketRegionType()))
   {
      //For regions which have state name the below function has to be called to identify the country to which it belongs.
      std::string correctStateName = _pNdsCountryStateRequestor->getNdsStateName(stateName.c_str(), convertISOLanguageCodetoString((_pLanguageMediator->getLanguageOfActiveSpeaker()).ISO639_3_SDSLanguageCode));
      setSearchInformation(_pNdsCountryStateRequestor->getNdsCountryName(), SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_COUNTRY);
      setSearchInformation(correctStateName, SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_STATE);
      replacedStateName = true;
   }
   else
   {
      //For regions where only city is available we can directly use the country name from SDS.
      setSearchInformation(getTaggedContent("country:"), SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_COUNTRY);
   }

   if (!replacedStateName)
   {
      setSearchInformation(stateName, SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_STATE);
   }

   setSearchInformation(getTaggedContent("city:"), SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_PLACE);
   // JPN Navigation changes
   setSearchInformation(getTaggedContent("oaza:"), SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_CITYDESTRICT);
   setSearchInformation(getTaggedContent("aza:"), SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_ROAD);

   setSearchInformation(getTaggedContent("street:"), SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_ROAD);
   setSearchInformation(getTaggedContent("street2:"), SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_CROSSROAD);
   setSearchInformation(getTaggedContent("houseno:"), SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_HOUSENUMBER);
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviSetDestinationItem::handleHouseNrRequest()
{
   ETG_TRACE_USR1(("handleHouseNrRequest"));
   std::string houseNrString = getTaggedContent("houseno:");
   _houseNumberSetForFTS = false;
   if (!houseNrString.empty())
   {
      ETG_TRACE_USR1(("handleHouseNrRequest()- House number is available - %s", houseNrString.c_str()));
      for (size_t i = 0; i < _firstRequestedsdsAddress.size(); ++i)
      {
         if (SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_HOUSENUMBER == _firstRequestedsdsAddress[i].getAddressElementType())
         {
            ETG_TRACE_USR1(("handleHouseNrRequest _houseNumberSetForFTS=true"));
            _houseNumberSetForFTS = true;
            SDSAddressElement sdsAddressElement;
            _firstRequestedsdsAddress.erase(_firstRequestedsdsAddress.begin() + i);
            sdsAddressElement.setAddressElementType(_firstRequestedsdsAddress[i].getAddressElementType());
            SDSElementOptions sdsElementOptions;
            sdsElementOptions.setIsAmbigious(false);
            sdsAddressElement.setElementOptions(sdsElementOptions);
            sdsAddressElement.setData(houseNrString.c_str());
            _firstRequestedsdsAddress.push_back(sdsAddressElement);
            break;
         }
      }
      _oRequestedsdsAddress.clear();
      _oRequestedsdsAddress = _firstRequestedsdsAddress;
      if (!_houseNumberSetForFTS)
      {
         setSearchInformation(houseNrString, SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_HOUSENUMBER);
      }
      _firstRequestedsdsAddress.clear();
      // JPN Navigation changes
      setSearchInformation(_oResultDescriptor.Oaza, SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_CITYDESTRICT);
      setSearchInformation(_oResultDescriptor.Aza, SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_ROAD);
      sendSdsCheckAddressRequest();
   }
   else
   {
      ETG_TRACE_USR1(("sendAddressRequestToNavi()- House number is not available"));
      setDestinationForNoHouseNr();
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviSetDestinationItem::NdsCountryStateUpdated()
{
   if (_pNdsCountryStateRequestor->isCountryStateInfoAvailable())
   {
      if (_countryStateInfoRequested)
      {
         sendAddressRequestToNavi();
         _countryStateInfoRequested = false;
      }
   }
   else
   {
      vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_CANNOTCOMPLETEACTION);
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviSetDestinationItem::handlePOIRequest()
{
   getContentFromDescriptor();

   if (_locationType.enType == sds2hmi_fi_tcl_e8_NAV_LocationType::FI_EN_DEFAULT)
   {
      sendSdsCheckAddressRequest();
   }
   else
   {
      requestPOIListToNavi();
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
std::string clSDS_Method_NaviSetDestinationItem::getTaggedContent(const char* tag) const
{
   if (_locationDescriptors.size() > 0)
   {
      std::string addressString = ((const sds2hmi_fi_tcl_StringPosition&) * _locationDescriptors[0].poDataData).String.szGet(sds2hmi_fi_tclString::FI_EN_UTF8);

      // JPN Navigation changes
      if (!strcmp("aza:", tag))
      {
         addressString = removeOazafromReceivedAddress(addressString);
      }
      size_t pos = addressString.find(tag);
      if (pos != std::string::npos)
      {
         std::string houseNumberString = addressString.substr(pos + strlen(tag));

         houseNumberString = removeDescriptorInformation(houseNumberString, "houseno:");
         houseNumberString = removeDescriptorInformation(houseNumberString, "oaza:");
         houseNumberString = removeDescriptorInformation(houseNumberString, "aza:");

         houseNumberString = removeDescriptorInformation(houseNumberString, "street:");
         houseNumberString = removeDescriptorInformation(houseNumberString, "street2:");
         houseNumberString = removeDescriptorInformation(houseNumberString, "poi:");
         houseNumberString = removeDescriptorInformation(houseNumberString, "cat:");
         houseNumberString = removeDescriptorInformation(houseNumberString, "attr:");
         houseNumberString = removeDescriptorInformation(houseNumberString, "city:");
         houseNumberString = removeDescriptorInformation(houseNumberString, "state:");
         houseNumberString = removeDescriptorInformation(houseNumberString, "country:");
         return StringUtils::trim(houseNumberString);
      }
   }
   return "";
}


/***********************************************************************//**
 *
 ***************************************************************************/
std::string clSDS_Method_NaviSetDestinationItem::removeDescriptorInformation(std::string addressString, const std::string& descriptorType) const
{
   if (addressString.find(descriptorType) != std::string::npos)
   {
      return addressString.erase(addressString.find(descriptorType));
   }
   return addressString;
}


/***********************************************************************//**
 *
 ***************************************************************************/
std::string clSDS_Method_NaviSetDestinationItem::removeOazafromReceivedAddress(std::string addressString) const
{
   size_t position =  addressString.find("oaza:");
   if (position != std::string::npos)
   {
      std::string azaString = addressString.substr(position);
      std::string oaza = azaString.substr(0, azaString.find(' '));
      addressString.erase(position, oaza.length());
   }

   return addressString;
}


/***********************************************************************//**
 *
 ***************************************************************************/
tVoid clSDS_Method_NaviSetDestinationItem::setSearchInformation(const std::string& searchContent, SDSAddressElementType searchElement)
{
   if (!searchContent.empty())
   {
      ETG_TRACE_USR1(("setSearchInformation()- SearchContent= %s", searchContent.c_str()));
      SDSAddressElement sdsAddressElement;
      sdsAddressElement.setAddressElementType(searchElement);
      SDSElementOptions sdsElementOptions;
      sdsElementOptions.setIsAmbigious(false);
      sdsAddressElement.setElementOptions(sdsElementOptions);
      sdsAddressElement.setData(searchContent);
      _oRequestedsdsAddress.push_back(sdsAddressElement);
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviSetDestinationItem::sendSdsCheckAddressRequest()
{
   ETG_TRACE_USR1(("Send address information to Navi Hall"));
   _firstRequestedsdsAddress.clear();
   _firstRequestedsdsAddress = _oRequestedsdsAddress;
   if (_navigationProxy->isAvailable())
   {
      _navigationProxy->sendSdsCheckAddressRequest(*this, _oRequestedsdsAddress);
   }
}


FreeTextSearchScope clSDS_Method_NaviSetDestinationItem::getFreeTextSearchScope()
{
   switch (_locationType.enType)
   {
      case sds2hmi_fi_tcl_e8_NAV_LocationType::FI_EN_DEFAULT:
         return FreeTextSearchScope__FREETEXT_SEARCH_SCOPE_POI_AT_LOCATION;

      case sds2hmi_fi_tcl_e8_NAV_LocationType::FI_EN_NEARBY:
         return FreeTextSearchScope__FREETEXT_SEARCH_SCOPE_POI_AT_CURRENT_POSITION;

      case sds2hmi_fi_tcl_e8_NAV_LocationType::FI_EN_ALONG_ROUTE:
         return FreeTextSearchScope__FREETEXT_SEARCH_SCOPE_POI_ALONG_ROUTE;

      case sds2hmi_fi_tcl_e8_NAV_LocationType::FI_EN_NEAR_DESTINATION:
         return FreeTextSearchScope__FREETEXT_SEARCH_SCOPE_POI_AT_DESTINATION_POSITION;

      default:
         return FreeTextSearchScope__FREETEXT_SEARCH_SCOPE_POI_AT_LOCATION;
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
tVoid clSDS_Method_NaviSetDestinationItem::requestPOIListToNavi()
{
   ETG_TRACE_USR1(("RequestPOIList to Navi"));
   const FreeTextSearchScope textSearchType = getFreeTextSearchScope();

   //const FixedPOICategory fixedPOICategory;

   if (textSearchType == FreeTextSearchScope__FREETEXT_SEARCH_SCOPE_POI_AT_LOCATION)
   {
      _navigationProxy->sendSetLocationWithSdsInputRequest(*this); //For AT_LOCATION case SDS-A has to call the SetLocation interface.
   }

   std::string searchString = getPOISearchString();
   ETG_TRACE_USR1(("searchStr =  %s", searchString.c_str()));

   if (searchString.empty())
   {
      vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_CANNOTCOMPLETEACTION);
   }
   else
   {
      tU32 startIndex = 0;
      uint8 destinationIndex = 0;
      if (_navigationProxy->hasDestinationInformation() &&
            _navigationProxy->getDestinationInformation().size() > 0)
      {
         destinationIndex = (uint8)(_navigationProxy->getDestinationInformation().size() - 1);
      }
      tU32 totalElements = 20;
      _navigationProxy->sendRequestFreeTextSearchResultsRequest(*this, searchString, textSearchType, destinationIndex, startIndex, totalElements);
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
std::string clSDS_Method_NaviSetDestinationItem::getPOISearchString()
{
   if (!_oPOIName.BrandName.empty())
   {
      return (setPOISearchStringForNavi("<?xml><BRAND>", "</BRAND>", _oPOIName.BrandName.c_str()));
   }
   else if (!_oPOIName.CategoryName.empty())
   {
      uint32 categoryId = getSdsPOICategoryId(_oPOIName.CategoryName.c_str());
      if (categoryId > 0)
      {
         return (setPOISearchStringForNavi("<?xml><CATEGORYID>", "</CATEGORYID>", (StringUtils::toString(categoryId))));
      }
      else if (!_oPOIName.CategoryName.compare("Freeways"))
      {
         return (setPOISearchStringForNavi("<?xml><CATEGORYID>", "</CATEGORYID>", (StringUtils::toString(33554633)))); //TODO:ATL5COB Remove the Hard-coded CategoryID for Freeways
      }
      else
      {
         return (setPOISearchStringForNavi("<?xml><CATEGORY>", "</CATEGORY>", _oPOIName.CategoryName.c_str()));
      }
   }
   else if (!_oPOIName.BuildingName.empty())
   {
      return (setPOISearchStringForNavi("<?xml><NAME>", "</NAME>", _oPOIName.BuildingName.c_str()));
   }
   return "";
}


/***********************************************************************//**
 *
 ***************************************************************************/
std::string clSDS_Method_NaviSetDestinationItem::setPOISearchStringForNavi(std::string startTag, std::string endTag, std::string poiName) const
{
   return (startTag.append(poiName).append(endTag));
}


/***********************************************************************//**
 *
 ***************************************************************************/


tVoid clSDS_Method_NaviSetDestinationItem::handleVerifiedSdsAddress(::std::vector< SDSAddressElement >& verifiedAddress)
{
   ETG_TRACE_USR1(("handleVerifiedSdsAddress()- Function entered"));

   ::std::vector<SDSAddressElement>::iterator it;

   for (it = verifiedAddress.begin(); it != verifiedAddress.end(); ++it)
   {
      if (it->getAddressElementType() == SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_HOUSENUMBER)
      {
         ETG_TRACE_USR1(("handleVerifiedSdsAddress()- Function entered HouseNr"));
         _oResultDescriptor.HouseNr = it->getData();
         _oResultDescriptor.bHouseNrValid = TRUE;
      }
      else if (it->getAddressElementType() == SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_ROAD)
      {
         ETG_TRACE_USR1(("handleVerifiedSdsAddress()- Function entered StreetName"));
         if (clSDS_KDSConfiguration::JPN == clSDS_KDSConfiguration::getMarketRegionType())
         {
            _oResultDescriptor.Aza = it->getData();
         }
         else
         {
            _oResultDescriptor.StreetName = it->getData();
         }
      }
      else if (it->getAddressElementType() == SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_CITYDESTRICT)
      {
         ETG_TRACE_USR1(("handleVerifiedSdsAddress()- Function entered citydesctrict"));
         if (clSDS_KDSConfiguration::JPN == clSDS_KDSConfiguration::getMarketRegionType())
         {
            _oResultDescriptor.Oaza = it->getData();
         }
      }
      else if (it->getAddressElementType() == SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_CROSSROAD)
      {
         ETG_TRACE_USR1(("handleVerifiedSdsAddress()- Function entered bCrossRoadAvailable"));
         _oResultDescriptor.CrossStreetName = it->getData();
         _oResultDescriptor.bCrossRoadAvailable = TRUE;
      }
      else if (it->getAddressElementType() == SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_PLACE)
      {
         ETG_TRACE_USR1(("handleVerifiedSdsAddress()- Function entered CityName"));
         _oResultDescriptor.CityName = it->getData();
      }
      else if (it->getAddressElementType() == SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_STATE)
      {
         ETG_TRACE_USR1(("handleVerifiedSdsAddress()- Function entered StateName"));
         _oResultDescriptor.StateName = it->getData();
      }
      else if (it->getAddressElementType() == SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_COUNTRY)
      {
         ETG_TRACE_USR1(("handleVerifiedSdsAddress()- Function entered CountryName"));
         _oResultDescriptor.CountryName = it->getData();
      }
      else if (it->getAddressElementType() == SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_PLACEREFINEMENT ||
               it->getAddressElementType() == SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_ROADREFINEMENT)
      {
         ETG_TRACE_USR1(("handleVerifiedSdsAddress()- Function entered refinement"));
         if (!(it->getData().empty()))
         {
            _refinementValid = true;
         }
      }
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
tVoid clSDS_Method_NaviSetDestinationItem::resetResultValues()
{
   _oResultDescriptor.HouseNr.clear();
   _oResultDescriptor.StreetName.clear();
   _oResultDescriptor.CrossStreetName.clear();
   _oResultDescriptor.CityName.clear();
   _oResultDescriptor.StateName.clear();
   _oResultDescriptor.CountryName.clear();
   _oResultDescriptor.Aza.clear();
   _oResultDescriptor.Oaza.clear();
   _oResultDescriptor.bAmbigous = FALSE;
   _oResultDescriptor.bNavigable = FALSE;
   _oResultDescriptor.bHouseNrAvailable = FALSE;
   _oResultDescriptor.bHouseNrValid = FALSE;
   _oResultDescriptor.bCrossRoadAvailable = FALSE;
   _oResultDescriptor.bRefinement = FALSE;
   _oResultDescriptor.removedInvalidEntry = FALSE;
   _oResultDescriptor.moreEntriesPossible = FALSE;
   _oResultDescriptor.azaAvailable = FALSE;
   _oRequestedsdsAddress.clear();
   _refinementValid = false;
}


/***********************************************************************//**l
 *
 ***************************************************************************/
tVoid clSDS_Method_NaviSetDestinationItem::validateSdsAddressOptions(const SDSAddressOptions& sdsAddressOptions, const ::std::vector< SDSAddressElement >& verifiedAddress,
      const std::vector< SDSAddressElementType >& sdsAddressElementType)
{
   ETG_TRACE_USR1(("validateSdsAddressOptions()- Function entered house number : %d : %d", sdsAddressOptions.getHouseNumberAvailable(), sdsAddressOptions.getHouseNumberValid()));

   _oResultDescriptor.bNavigable = sdsAddressOptions.getNavigable();
   _oResultDescriptor.bAmbigous = sdsAddressOptions.getAmbigious();
   _oResultDescriptor.bHouseNrAvailable = sdsAddressOptions.getHouseNumberAvailable();
   //_oResultDescriptor.bHouseNrValid = sdsAddressOptions.getHouseNumberValid();
   _oResultDescriptor.removedInvalidEntry = sdsAddressOptions.getRemovedInvalidEntry();
   _oResultDescriptor.moreEntriesPossible = sdsAddressOptions.getMoreEntriesPossible();

   findNextAddressElementType(sdsAddressElementType);

   validateAddressOptions(verifiedAddress);
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviSetDestinationItem::validateAddressOptions(const ::std::vector< SDSAddressElement >& verifiedAddress)
{
   if (isPOISearchRequest())
   {
      if (_oResultDescriptor.bAmbigous)
      {
         ETG_TRACE_USR1(("validateAddressOptions isPOISearchRequest vRequestAmbigListToNavi"));
         _pMultipleDestiantionList->vRequestAmbigListToNavi(_oResultDescriptor.ambigousElement);
      }
      else
      {
         requestPOIListToNavi();
      }
   }
   else if (isHouseNumberRequest())
   {
      if (_oResultDescriptor.bHouseNrAvailable && (!_oResultDescriptor.bHouseNrValid))
      {
         _pMultipleDestiantionList->vRequestAmbigListToNavi(SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_HOUSENUMBER);
      }
      else
      {
         checkForMoreEntries();
      }
   }
   else
   {
      if (_oResultDescriptor.removedInvalidEntry)
      {
         checkNextPossibleEntry(verifiedAddress);
      }
      else
      {
         checkForMoreEntries();
      }
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
tVoid clSDS_Method_NaviSetDestinationItem::setDestinationResultforAmbiguity()
{
   //Intersecting street with ambiguity
   if ((!_oResultDescriptor.StreetName.empty()) &&
         (!_oResultDescriptor.CrossStreetName.empty()) &&
         (_oResultDescriptor.bCrossRoadAvailable == TRUE))
   {
      ETG_TRACE_USR1(("setDestinationResultforAmbiguity()- sendDestinationResult for STREET_INTERSECTION_RESULT"));
      sendDestinationResult(STREET_INTERSECTION_RESULT);
   }
   else if ((_oResultDescriptor.bCrossRoadAvailable == TRUE) &&
            (_oResultDescriptor.bHouseNrValid == FALSE))
   {
      ETG_TRACE_USR1(("setDestinationResultforAmbiguity()- sendDestinationResult for STREET_INTERSECTION_RESULT"));
      sendDestinationResult(JUNCTION_RESULT);
   }
   //City center with Ambiguity
   else if (_oResultDescriptor.StreetName.empty() &&
            _oResultDescriptor.HouseNr.empty() &&
            isOsdeAddressRequest())
   {
      ETG_TRACE_USR1(("setDestinationResultforAmbiguity()- AddressMultipleResult for City center"));
      sendDestinationResult(ADDRESS_MULTIPLE_RESULT);
   }

   else if (_oResultDescriptor.StreetName.empty() &&
            _oResultDescriptor.HouseNr.empty() &&
            isPOISearchRequest())
   {
      ETG_TRACE_USR1(("setDestinationResultforAmbiguity()- sendDestinationResult for POI_MULTIPLE_RESULT"));
      sendDestinationResult(POI_MULTIPLE_RESULT);
   }
   else if ((_oResultDescriptor.bHouseNrValid == FALSE) &&
            (_oResultDescriptor.bHouseNrAvailable == TRUE))
   {
      ETG_TRACE_USR1(("setDestinationResultforAmbiguity()- sendDestinationResult for ADDRESS_MULTIPLE_RESULT_FOR_HOUSENR_OUTRANGE"));
      sendDestinationResult(ADDRESS_MULTIPLE_RESULT_FOR_HOUSENR_OUTRANGE);
   }
   else if ((_oResultDescriptor.bHouseNrAvailable == TRUE) &&
            (_oResultDescriptor.bHouseNrValid == TRUE))
   {
      ETG_TRACE_USR1(("setDestinationResultforAmbiguity()- ADDRESS_MULTIPLE_RESULT"));
      sendDestinationResult(ADDRESS_MULTIPLE_RESULT);
   }
   else if ((_oResultDescriptor.bHouseNrAvailable == FALSE) &&
            (_oResultDescriptor.bHouseNrValid == FALSE))
   {
      ETG_TRACE_USR1(("setDestinationResultforAmbiguity()- sendDestinationResult for STREET_INTERSECTION_MULTIPLE_RESULT"));
      sendDestinationResult(STREET_INTERSECTION_MULTIPLE_RESULT);
   }
   else
   {
      ETG_TRACE_USR1(("setDestinationResultforAmbiguity()- sendDestinationResult for SDS2HMI_SDSFI_C_U16_ERROR_CANNOTCOMPLETEACTION"));
      vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_CANNOTCOMPLETEACTION);
   }
}


bool clSDS_Method_NaviSetDestinationItem::setDestinationResultforNonAmbiguityRefinement()
{
   if ((_oResultDescriptor.bNavigable == TRUE) &&
         (! _oResultDescriptor.CrossStreetName.empty()) &&
         (! _oResultDescriptor.StreetName.empty()))
   {
      ETG_TRACE_USR1(("setDestinationResultforNonAmbiguity()- STREET_INTERSECTION_RESULT"));
      sendDestinationResult(STREET_INTERSECTION_RESULT);
      return true;
   }
   //Final Address Result after refinement
   else if ((_refinementValid) &&
            (_oResultDescriptor.bNavigable == TRUE) &&
            (_refinmentType == ADDRESS_SEARCH_REFINEMENT_SELECTION))
   {
      ETG_TRACE_USR1(("setDestinationResultforNonAmbiguity()- ADDRESS_RESULT"));
      sendDestinationResult(ADDRESS_RESULT);
      return true;
   }
   else if ((_oResultDescriptor.bCrossRoadAvailable == TRUE) &&
            (_oResultDescriptor.bHouseNrValid == FALSE))
   {
      ETG_TRACE_USR1(("setDestinationResultforNonAmbiguity()- JUNCTION_RESULT"));
      sendDestinationResult(JUNCTION_RESULT);
      return true;
   }
   //City center without Ambiguity
   else if (_oResultDescriptor.StreetName.empty() &&
            _oResultDescriptor.Oaza.empty() &&
            _oResultDescriptor.HouseNr.empty() &&
            (_oResultDescriptor.bNavigable == TRUE) &&
            isOsdeAddressRequest())
   {
      //   ETG_TRACE_USR1(("setDestinationResultforNonAmbiguity()- Refinement result"));
      ETG_TRACE_USR1(("setDestinationResultforNonAmbiguity()- AddressResult for City center"));
      sendDestinationResult(ADDRESS_RESULT);
      return true;
   }

   else if (_oResultDescriptor.StreetName.empty() &&
            _oResultDescriptor.HouseNr.empty() &&
            (_oResultDescriptor.bNavigable == TRUE) &&
            isPOISearchRequest())
   {
      ETG_TRACE_USR1(("setDestinationResultforNonAmbiguity()- POI"));
      requestPOIListToNavi();
      return true;
   }

   else if (_oResultDescriptor.CityName.empty() &&
            (_oResultDescriptor.bNavigable == FALSE) &&
            isPOISearchRequest())
   {
      ETG_TRACE_USR1(("setDestinationResultforNonAmbiguity()-requestPOIListToNavi"));
      requestPOIListToNavi();
      return true;
   }

   else if ((_oResultDescriptor.bHouseNrAvailable == TRUE) &&
            (_oResultDescriptor.azaAvailable == TRUE))
   {
      ETG_TRACE_USR1(("setDestinationResultforNonAmbiguity()- AZA_HNO_RESULT"));
      sendDestinationResult(AZA_HNO_RESULT);
      return true;
   }
   else
   {
      return false;
   }
}


tVoid clSDS_Method_NaviSetDestinationItem::setDestinationResultforNonAmbiguityAddressresult()
{
   if ((_oResultDescriptor.bHouseNrAvailable == FALSE) &&
         (_oResultDescriptor.azaAvailable == TRUE))
   {
      ETG_TRACE_USR1(("setDestinationResultforNonAmbiguity()- AZA_ONLY_RESULT"));
      sendDestinationResult(AZA_ONLY_RESULT);
   }

   else if ((_oResultDescriptor.bHouseNrAvailable == TRUE) &&
            (_oResultDescriptor.bHouseNrValid == FALSE))
   {
      ETG_TRACE_USR1(("setDestinationResultforNonAmbiguity()- ADDRESS_HOUSENO_OUTRANGE_RESULT"));
      sendDestinationResult(ADDRESS_HOUSENO_OUTRANGE_RESULT);
   }

   else if ((_oResultDescriptor.bHouseNrValid == TRUE) &&
            (_oResultDescriptor.bNavigable == TRUE))
   {
      ETG_TRACE_USR1(("setDestinationResultforNonAmbiguity()- Address Result"));
      sendDestinationResult(ADDRESS_RESULT);
   }
   else if (((!_oResultDescriptor.StreetName.empty()) || (!_oResultDescriptor.Aza.empty())) &&
            (_oResultDescriptor.HouseNr.empty()) &&
            (_oResultDescriptor.bNavigable == TRUE) &&
            (_oResultDescriptor.CrossStreetName.empty()) &&
            (_oResultDescriptor.moreEntriesPossible == FALSE) &&
            isOsdeAddressRequest())

   {
      ETG_TRACE_USR1(("setDestinationResultforNonAmbiguity()- Address Result without house no and intersection"));
      sendDestinationResult(ADDRESS_RESULT);
   }
   else if ((!_oResultDescriptor.Oaza.empty()) &&
            (_oResultDescriptor.HouseNr.empty()) &&
            (_oResultDescriptor.Aza.empty()) &&
            (_oResultDescriptor.bNavigable == TRUE) &&
            (_oResultDescriptor.moreEntriesPossible == FALSE) &&
            isOsdeAddressRequest())

   {
      ETG_TRACE_USR1(("setDestinationResultforNonAmbiguity()- Address Result without House No. and Aza"));
      sendDestinationResult(ADDRESS_RESULT);
   }
   else
   {
      ETG_TRACE_USR1(("setDestinationResultforNonAmbiguity()- SDS2HMI_SDSFI_C_U16_ERROR_CANNOTCOMPLETEACTION"));
      vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_CANNOTCOMPLETEACTION);
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
tVoid clSDS_Method_NaviSetDestinationItem::setDestinationResultforNonAmbiguity()
{
   ETG_TRACE_USR1(("setDestinationResultforNonAmbiguity()- entry"));

   if (!setDestinationResultforNonAmbiguityRefinement())
   {
      setDestinationResultforNonAmbiguityAddressresult();
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviSetDestinationItem::sendListEntrySelectionToNavi(tU8 ambiguousValue)
{
   // TODO jnd2hi: get rid of the hard-coded view ids below"
   ETG_TRACE_USR1(("sendListEntrySelectionToNavi()- ambiguousValue = %d", ambiguousValue));
   if ((_pCommonShowDialog->getViewId() == "SR_NAV_SelectAddress") ||
         (_pCommonShowDialog->getViewId() == "SR_NAV_SelectHouseNumber") ||
         (_pCommonShowDialog->getViewId() == "SR_NAV_Ambig_List_CityCenter") ||
         (_pCommonShowDialog->getViewId() == "SR_NAV_Ambig_List_CityCenterEur") ||
         (_pCommonShowDialog->getViewId() == "SR_NAV_Intersection_List"))
   {
      _refinmentType = ADDRESS_SEARCH_REFINEMENT_SELECTION;
      _navigationProxy->sendSelectSdsRefinementRequest(*this, _pMultipleDestiantionList->getRefinementID(ambiguousValue));
      ETG_TRACE_USR1(("SelectSdsRefinementRequest is sent to Navi hall for Address search"));
   }
   //SR_NAV_Ambig_List_City will be triggered during POI usecases.
   else if (_pCommonShowDialog->getViewId() == "SR_NAV_Ambig_List_City")
   {
      _refinmentType = POI_SEARCH_REFINEMENT_SELECTION;
      _navigationProxy->sendSelectSdsRefinementRequest(*this, _pMultipleDestiantionList->getRefinementID(ambiguousValue));
      ETG_TRACE_USR1(("SelectSdsRefinementRequest is sent to Navi hall for POI search"));
   }
   else if ((_pCommonShowDialog->getViewId() == "SR_NAV_SelectPoi") ||
            (_pCommonShowDialog->getViewId() == "SR_NAV_POI_AlongRoute_Tab") ||
            (_pCommonShowDialog->getViewId() == "SR_NAV_POI_InCity_Tab") ||
            (_pCommonShowDialog->getViewId() == "SR_NAV_POI_NearDestination_Tab") ||
            (_pCommonShowDialog->getViewId() == "SR_NAV_POI_Nearby_Tab"))
   {
      _refinmentType = NO_REFINEMENT_SELECTION;

      if (_pQuickChargingStation != NULL && _pQuickChargingStation->getChargingStationListRequestStatus())
      {
         _pQuickChargingStation->sendChargingStationListRequest(ambiguousValue);
      }
      else if (_pNaviHapticHandOverInterface != NULL && _pPOIList != NULL)
      {
         if (_pNaviHapticHandOverInterface->isPOIListViaHaptic() || _pPOIList->isPOIListViaHaptic())

         {
            _navigationProxy->sendSetLocationWithFixedPOICategoryInputRequest(*this, _pPOIList->getRefinementID(ambiguousValue));
            ETG_TRACE_USR1(("SetLocationWithFixedPOICategoryInputRequest is sent to Navi hall"));
            _pNaviHapticHandOverInterface->resetPOIListViaHaptic();
            _pPOIList->setPOIListViaHaptic();
            tU16 u16NumberofElements = 0;
            sendPOIResult(u16NumberofElements);
         }
         else
         {
            _navigationProxy->sendSetLocationWithFreeTextSearchInputRequest(*this, _pPOIList->getRefinementID(ambiguousValue));
            ETG_TRACE_USR1(("SetLocationWithFreeTextSearchInputRequest is sent to Navi hall"));
            tU16 u16NumberofElements = 0;
            sendPOIResult(u16NumberofElements);
         }
      }
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviSetDestinationItem::sendPOIResult(tU16 numberOfElements)
{
   ETG_TRACE_USR1(("clSDS_Method_NaviSetDestinationItem::sendPOIResult() - Function entered"));

   sds2hmi_sdsfi_tclMsgNaviSetDestinationItemMethodResult result;
   result.nNumberOfElements = numberOfElements;
   result.nAnswerOptions.vSetAmbiguous(FALSE);
   result.nAnswerOptions.vSetHouseNumberInRange(FALSE);
   result.nAnswerOptions.vSetInvalidAddress(FALSE);
   result.nAnswerOptions.vSetGuidancePossible(TRUE);
   vSendMethodResult(result);
}


/***********************************************************************//**
 *
 ***************************************************************************/
tVoid clSDS_Method_NaviSetDestinationItem::sendDestinationResult(enResultType resultType)
{
   ETG_TRACE_USR4(("setDestinationResult : ResultType = %d", resultType));

   if (_oResultDescriptor.bAmbigous == FALSE &&
         (resultType == ADDRESS_RESULT ||
          resultType == ADDRESS_NOHOUSENO_RESULT ||
          resultType == STREET_INTERSECTION_RESULT))
   {
      _navigationProxy->sendSetLocationWithSdsInputRequest(*this);
      _refinmentType = NO_REFINEMENT_SELECTION;
      ETG_TRACE_USR1(("SetLocationWithSdsInputRequest is sent to Navi hall"));
   }

   sds2hmi_sdsfi_tclMsgNaviSetDestinationItemMethodResult result;
   result.nNumberOfElements = 1;
   result.nAnswerOptions.vSetAmbiguous(_oResultDescriptor.bAmbigous);
   result.nAnswerOptions.vSetGuidancePossible(_oResultDescriptor.bNavigable);
   result.nAnswerOptions.vSetHouseNumberInRange(_oResultDescriptor.bHouseNrValid);
   result.nAnswerOptions.vSetInvalidAddress(FALSE);
   result.tSelectionCriterionType = getSelectionCriterionType(resultType);
   vSendMethodResult(result);
}


std::vector< sds2hmi_fi_tcl_e16_SelectionCriterionType > clSDS_Method_NaviSetDestinationItem::getSelectionCriterionType(enResultType resultType)
{
   std::vector< sds2hmi_fi_tcl_e16_SelectionCriterionType > selectionTypeList;
   sds2hmi_fi_tcl_e16_SelectionCriterionType selectionType;

   switch (resultType)
   {
      case ADDRESS_NOHOUSENO_RESULT:
      case ADDRESS_RESULT:
         selectionType.enType = sds2hmi_fi_tcl_e16_SelectionCriterionType::FI_EN_RESERVED;
         selectionTypeList.push_back(selectionType);
         break;

      case ADDRESS_HOUSENO_OUTRANGE_RESULT:
      case ADDRESS_MULTIPLE_RESULT_FOR_HOUSENR_OUTRANGE:
         selectionType.enType = sds2hmi_fi_tcl_e16_SelectionCriterionType::FI_EN_HOUSENUMBER;
         selectionTypeList.push_back(selectionType);
         break;

      case ADDRESS_MULTIPLE_RESULT:
         selectionType.enType = sds2hmi_fi_tcl_e16_SelectionCriterionType::FI_EN_OSDE_ADDRESSES;
         selectionTypeList.push_back(selectionType);
         break;

      case STREET_INTERSECTION_RESULT:
      case STREET_INTERSECTION_MULTIPLE_RESULT:
         selectionType.enType = sds2hmi_fi_tcl_e16_SelectionCriterionType::FI_EN_FINALDESTINATION;
         selectionTypeList.push_back(selectionType);
         break;

      case POI_MULTIPLE_RESULT:
         selectionType.enType = sds2hmi_fi_tcl_e16_SelectionCriterionType::FI_EN_OSDE_POI;
         selectionTypeList.push_back(selectionType);
         break;

      case JUNCTION_RESULT:
      {
         if (_oResultDescriptor.bHouseNrAvailable == TRUE)
         {
            selectionType.enType = sds2hmi_fi_tcl_e16_SelectionCriterionType::FI_EN_HOUSENUMBER;
            selectionTypeList.push_back(selectionType);
         }
         selectionType.enType = sds2hmi_fi_tcl_e16_SelectionCriterionType::FI_EN_JUNCTION;
         selectionTypeList.push_back(selectionType);
      }
      break;

      case AZA_ONLY_RESULT:
      {
         selectionType.enType = sds2hmi_fi_tcl_e16_SelectionCriterionType::FI_EN_AZA;
         selectionTypeList.push_back(selectionType);
      }
      break;
      case AZA_HNO_RESULT:
      {
         if (_oResultDescriptor.bHouseNrAvailable == TRUE)
         {
            selectionType.enType = sds2hmi_fi_tcl_e16_SelectionCriterionType::FI_EN_HOUSENUMBER;
            selectionTypeList.push_back(selectionType);
         }
         selectionType.enType = sds2hmi_fi_tcl_e16_SelectionCriterionType::FI_EN_AZA;
         selectionTypeList.push_back(selectionType);
      }
      break;

      default:
         break;
   }

   return selectionTypeList;
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviSetDestinationItem::setDestinationForNoHouseNr()
{
   ETG_TRACE_USR1(("setDestinationForNoHouseNr"));

   if (_oResultDescriptor.bAmbigous)
   {
      ETG_TRACE_USR1(("setDestinationForNoHouseNr vRequestAmbigListToNavi"));
      _oResultDescriptor.bHouseNrValid = TRUE;
      _oResultDescriptor.bHouseNrAvailable = TRUE;
      _pMultipleDestiantionList->vRequestAmbigListToNavi(_oResultDescriptor.ambigousElement);
   }
   else
   {
      sendDestinationResult(ADDRESS_NOHOUSENO_RESULT);
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviSetDestinationItem::vAmbiguityListResolved()
{
   ETG_TRACE_USR1(("vAmbiguityListResolved"));
   setDestinationResultforAmbiguity();
}


void clSDS_Method_NaviSetDestinationItem::setLocationWithFixedPOICategoryCompleted()
{
   ETG_TRACE_USR1(("setLocationWithFixedPOICategoryCompleted for charging station"));
   tU16 u16NumberofElements = 0;
   sendPOIResult(u16NumberofElements);
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NaviSetDestinationItem::countriesUpdated()
{
   sendAddressRequestToNavi();
}


/***********************************************************************//**
 * Available countries need to be retrieved from SDB
 * if the SDS request contains "state:" information in the location description string
 * and if the countries are not yet available in CountryNameRequestor (meaning that they had been retrieved before)
 ***************************************************************************/
bool clSDS_Method_NaviSetDestinationItem::isCountryRetrievalNecessary()
{
   return ((getTaggedContent("state:") != "") && (_pCountryNameRequestor->getCountryListSize() == 0));
}


/***********************************************************************//**
 *
 ***************************************************************************/

uint32 clSDS_Method_NaviSetDestinationItem::getSdsPOICategoryId(std::string categoryName)
{
   std::vector< PoiCategoryShortcut >::iterator it1;

   for (it1 = _sdsPOICategory.begin(); it1 != _sdsPOICategory.end(); ++it1)
   {
      if (it1->getCategoryName() == categoryName)
      {
         return it1->getPoiCategoryId();
      }
   }
   return 0;
}


void clSDS_Method_NaviSetDestinationItem::findNextAddressElementType(const std::vector< SDSAddressElementType >& sdsAddressElementType)
{
   ETG_TRACE_USR1(("findNextAddressElementType"));
   for (size_t i = 0; i < sdsAddressElementType.size(); ++i)
   {
      if (SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_HOUSENUMBER == sdsAddressElementType[i])
      {
         ETG_TRACE_USR1(("bHouseNrAvailable"));
         _oResultDescriptor.bHouseNrAvailable = TRUE;
      }
      else if (SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_ROAD == sdsAddressElementType[i] && clSDS_KDSConfiguration::JPN == clSDS_KDSConfiguration::getMarketRegionType())
      {
         ETG_TRACE_USR1(("azaAvailable"));
         _oResultDescriptor.azaAvailable = TRUE;
      }
      else if (SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_CROSSROAD == sdsAddressElementType[i])
      {
         ETG_TRACE_USR1(("bCrossRoadAvailable"));
         _oResultDescriptor.bCrossRoadAvailable = TRUE;
      }
      else if (SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_PLACEREFINEMENT == sdsAddressElementType[i] ||
               SDSAddressElementType__SDS_ADDRESS_ELEMENT_TYPE_ROADREFINEMENT == sdsAddressElementType[i])
      {
         ETG_TRACE_USR1(("bRefinement"));
         _oResultDescriptor.bRefinement = TRUE;
         _oResultDescriptor.ambigousElement = sdsAddressElementType[i];
      }
   }
}


void clSDS_Method_NaviSetDestinationItem::checkNextPossibleEntry(const ::std::vector< SDSAddressElement >& verifiedAddress)
{
   ETG_TRACE_USR1(("checkNextPossibleEntry"));
   // Checking next selection criteria
   if (_oResultDescriptor.bHouseNrAvailable || _oResultDescriptor.bCrossRoadAvailable || _oResultDescriptor.bNavigable || _oResultDescriptor.azaAvailable)
   {
      setDestinationResultforNonAmbiguity();
   }
   else
   {
      ETG_TRACE_USR1(("checkNextPossibleEntry handleDestinationItemResult"));
      handleDestinationItemResult();
   }
}


void clSDS_Method_NaviSetDestinationItem::checkForMoreEntries()
{
   ETG_TRACE_USR1(("checkForMoreEntries"));
   if (_oResultDescriptor.moreEntriesPossible)
   {
      ETG_TRACE_USR1(("checkForMoreEntries moreEntriesPossible"));
      // Checking next selection criteria
      if ((_oResultDescriptor.bHouseNrAvailable && (!_oResultDescriptor.bHouseNrValid)) || _oResultDescriptor.bCrossRoadAvailable || _oResultDescriptor.bNavigable)
      {
         ETG_TRACE_USR1(("checkForMoreEntries moreEntriesPossible setDestinationResultforNonAmbiguity"));
         setDestinationResultforNonAmbiguity();
      }
      else
      {
         ETG_TRACE_USR1(("checkForMoreEntries moreEntriesPossible !setDestinationResultforNonAmbiguity"));
         handleDestinationItemResult();
      }
   }
   else
   {
      ETG_TRACE_USR1(("checkForMoreEntries !!!moreEntriesPossible"));
      handleDestinationItemResult();
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/

std::string clSDS_Method_NaviSetDestinationItem::convertISOLanguageCodetoString(sds2hmi_fi_tcl_e16_ISO639_3_SDSLanguageCode languageCode)
{
   if (languageCode.enType == sds2hmi_fi_tcl_e16_ISO639_3_SDSLanguageCode::FI_EN_ISO_639_3_ENG)
   {
      return "eng";
   }
   else if (languageCode.enType == sds2hmi_fi_tcl_e16_ISO639_3_SDSLanguageCode::FI_EN_ISO_639_3_FRA)
   {
      return "fra";
   }
   else if (languageCode.enType == sds2hmi_fi_tcl_e16_ISO639_3_SDSLanguageCode::FI_EN_ISO_639_3_SPA)
   {
      return "spa";
   }
   else
   {
      return "";
   }
}


void clSDS_Method_NaviSetDestinationItem::handleDestinationItemResult()
{
   ETG_TRACE_USR1(("handleDestinationItemResult "));
   // Checking ambiguity in ROAD, PLACE or Complete Address
   if (_oResultDescriptor.bRefinement || _oResultDescriptor.bAmbigous)
   {
      _pMultipleDestiantionList->vRequestAmbigListToNavi(_oResultDescriptor.ambigousElement);
   }
   else
   {
      if (_oResultDescriptor.bNavigable)
      {
         setDestinationResultforNonAmbiguity();
      }
      else
      {
         _oRequestedsdsAddress.clear();
         vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_CANNOTCOMPLETEACTION);
      }
   }
}
