/**************************************************************************//**
 * \file       clSDS_Method_NaviGetWaypointListInfo.cpp
 *
 * clSDS_Method_NaviGetWaypointListInfo method class implementation
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "Sds2HmiServer/functions/clSDS_Method_NaviGetWaypointListInfo.h"


using namespace org::bosch::cm::navigation::NavigationService;


/**************************************************************************//**
* Destructor
******************************************************************************/
clSDS_Method_NaviGetWaypointListInfo::~clSDS_Method_NaviGetWaypointListInfo()
{
}


/**************************************************************************//**
* Constructor
******************************************************************************/
clSDS_Method_NaviGetWaypointListInfo::clSDS_Method_NaviGetWaypointListInfo(
   ahl_tclBaseOneThreadService* pService,
   ::boost::shared_ptr<NavigationServiceProxy> naviProxy,
   ::boost::shared_ptr<org::bosch::cm::navigation::NavigationSDSService::NavigationSDSServiceProxy> naviSDSProxy)

   : clServerMethod(SDS2HMI_SDSFI_C_U16_NAVIGETWAYPOINTLISTINFO, pService)
   , _naviProxy(naviProxy)
   , _naviSDSProxy(naviSDSProxy)
   , _waypointListSize(0)
   , _bWayListRequested(FALSE)
   , _bAddAsWayPoint(FALSE)
   , _maxWayPointCount(9)
{
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_Method_NaviGetWaypointListInfo::onUnavailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _naviProxy)
   {
      _naviProxy->sendWaypointListDeregisterAll();
   }
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_Method_NaviGetWaypointListInfo::onAvailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _naviProxy)
   {
      _naviProxy->sendWaypointListRegister(*this);
   }

   if (proxy == _naviSDSProxy)
   {
      _naviSDSProxy->sendGetMaxWaypointCountRequest(*this);
   }
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_Method_NaviGetWaypointListInfo::onWaypointListError(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< WaypointListError >& /*error*/)
{
   vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_CANNOTCOMPLETEACTION);
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_Method_NaviGetWaypointListInfo::onWaypointListUpdate(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< WaypointListUpdate >& update)
{
   _waypointListSize = update->getWaypointList().size();
   if (_bWayListRequested == TRUE)
   {
      vSendResult();
      _bWayListRequested = FALSE;
   }
   if (_bAddAsWayPoint == TRUE)
   {
      _naviProxy->sendApplyWaypointListChangeRequest(*this);
      _bAddAsWayPoint = FALSE;
   }
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_Method_NaviGetWaypointListInfo::onRequestWaypointListError(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< RequestWaypointListError >& /*error*/)
{
   vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_CANNOTCOMPLETEACTION);
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_Method_NaviGetWaypointListInfo::onRequestWaypointListResponse(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< RequestWaypointListResponse >& /*response*/)
{
}


/**************************************************************************//**
*
******************************************************************************/
tBool clSDS_Method_NaviGetWaypointListInfo::waypointListIsFull() const
{
   return ((_waypointListSize - 1) == _maxWayPointCount);
}


/**************************************************************************//**
*
******************************************************************************/
tVoid clSDS_Method_NaviGetWaypointListInfo::vMethodStart(amt_tclServiceData* /*pInMessage*/)
{
   _bWayListRequested = TRUE;
   vSendGetWaypointListRequest();
}


/**************************************************************************//**
*
******************************************************************************/
tVoid clSDS_Method_NaviGetWaypointListInfo::vSendGetWaypointListRequest()
{
   _naviProxy->sendRequestWaypointListRequest(*this);
}


/**************************************************************************//**
*
******************************************************************************/
tVoid clSDS_Method_NaviGetWaypointListInfo::vSendResult()
{
   sds2hmi_sdsfi_tclMsgNaviGetWaypointListInfoMethodResult oMessage;
   oMessage.bWaypointMustBeDeleted = waypointListIsFull();
   vSendMethodResult(oMessage);
}


void clSDS_Method_NaviGetWaypointListInfo::onSetAddAsWayPoint(tBool bAddAsWayPoint)
{
   _bAddAsWayPoint = bAddAsWayPoint;
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_Method_NaviGetWaypointListInfo::onApplyWaypointListChangeResponse(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< ApplyWaypointListChangeResponse >& /*response*/)
{
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_Method_NaviGetWaypointListInfo::onApplyWaypointListChangeError(
   const ::boost::shared_ptr< NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< ApplyWaypointListChangeError >& /*error*/)
{
}


/***********************************************************************//**
*
***************************************************************************/

void clSDS_Method_NaviGetWaypointListInfo::onGetMaxWaypointCountError(
   const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationSDSService::NavigationSDSServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationSDSService::GetMaxWaypointCountError >& /*error*/)
{
}


/***********************************************************************//**
*
***************************************************************************/

void clSDS_Method_NaviGetWaypointListInfo::onGetMaxWaypointCountResponse(
   const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationSDSService::NavigationSDSServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationSDSService::GetMaxWaypointCountResponse >& response)
{
   if (response->hasMaxWayPointCount())
   {
      _maxWayPointCount = response->getMaxWayPointCount();
   }
}
