/**************************************************************************//**
 * \file       clSDS_Method_NavDataRegisterDirectNDSUse.cpp
 *
 * clSDS_Method_NavDataRegisterDirectNDSUse method implementation
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "Sds2HmiServer/functions/clSDS_Method_NavDataRegisterDirectNDSUse.h"
#include "SdsAdapter_Trace.h"
#include "application/NaviGlobalDataForSdsObserver.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/clSDS_Method_NavDataRegisterDirectNDSUse.cpp.trc.h"
#endif


using namespace org::bosch::cm::navigation::NavigationSDSService;


enum BuildingBlockType
{
   SDS_BUILDING_BLOCK_UNDEFINED = 0,
   SDS_BUILDING_BLOCK_SPEECH = 9,
   SDS_BUILDING_BLOCK_EXTENSION = 255
};


clSDS_Method_NavDataRegisterDirectNDSUse::clSDS_Method_NavDataRegisterDirectNDSUse(ahl_tclBaseOneThreadService* pService,
      ::boost::shared_ptr< org::bosch::cm::navigation::NavigationService::NavigationServiceProxy> naviProxy ,
      ::boost::shared_ptr< NavigationSDSServiceProxy> naviSDSProxy,
      clSDS_Method_NavDataUnRegisterDirectNDSUse* pMethodNavDataUnRegisterDirectNDSUse)
   : clServerMethod(SDS2HMI_SDSFI_C_U16_NAVDATAREGISTERDIRECTNDSUSE, pService)
   , _naviSDSProxy(naviSDSProxy)
   , _naviProxy(naviProxy)
   , _pMethodNavDataUnRegisterDirectNDSUse(pMethodNavDataUnRegisterDirectNDSUse)
   , _globalDataResponseReceived(false)
   , _datasetRegistrationSuccess(false)
{
   if (_pMethodNavDataUnRegisterDirectNDSUse)
   {
      _pMethodNavDataUnRegisterDirectNDSUse->addDataSetDeRegistrationObserver(this);
   }

   clearsdsProducts();
}


clSDS_Method_NavDataRegisterDirectNDSUse::~clSDS_Method_NavDataRegisterDirectNDSUse()
{
}


tVoid clSDS_Method_NavDataRegisterDirectNDSUse::vMethodStart(amt_tclServiceData* pInMessage)
{
   _datasetRegistrationSuccess = false;
   sds2hmi_sdsfi_tclMsgNavDataRegisterDirectNDSUseMethodStart oMessage;
   vGetDataFromAmt(pInMessage, oMessage);
   NdsDatasetUser datasetUser = NdsDatasetUser__NDS_DATASET_USER__SDS;
   _naviSDSProxy->sendSdsDatasetUserRegistrationRequest(*this, datasetUser, oMessage.DataSetId);
}


/***********************************************************************//**
*
***************************************************************************/
void clSDS_Method_NavDataRegisterDirectNDSUse::onAvailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _naviProxy)
   {
      _naviProxy->sendNavStatusRegister(*this);
      _naviProxy->sendNavStatusGet(*this);
   }
}


/***********************************************************************//**
*
***************************************************************************/
void clSDS_Method_NavDataRegisterDirectNDSUse::onUnavailable(
   const boost::shared_ptr<asf::core::Proxy>& proxy,
   const asf::core::ServiceStateChange& /*stateChange*/)
{
   if (proxy == _naviProxy)
   {
      _naviProxy->sendNavStatusDeregisterAll();
   }
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_Method_NavDataRegisterDirectNDSUse::onGetGlobalDataForSdsError(
   const ::boost::shared_ptr< NavigationSDSServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< GetGlobalDataForSdsError >& /*error*/)
{
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_Method_NavDataRegisterDirectNDSUse::onGetGlobalDataForSdsResponse(
   const ::boost::shared_ptr< NavigationSDSServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< GetGlobalDataForSdsResponse >& response)
{
   setSdsProducts(response->getDeviceName(), response->getPersistentDatasetId(), response->getSdsProductsList());
   setUpdateRegionSpecifierRelationList(response->getUpdateRegionSpecifierRelationsList());
   notifyGlobalDataForSdsObservers();
   _globalDataResponseReceived = true;
   verifyAndSendResult();
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_Method_NavDataRegisterDirectNDSUse::onSdsDatasetUserRegistrationError(
   const ::boost::shared_ptr< NavigationSDSServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< SdsDatasetUserRegistrationError >& /*error*/)
{
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_Method_NavDataRegisterDirectNDSUse::onSdsDatasetUserRegistrationResponse(
   const ::boost::shared_ptr< NavigationSDSServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< SdsDatasetUserRegistrationResponse >& response)
{
   if (RegistrationAndDeRegistrationStatus__STATUS_OK == response->getRegistrationStatus())
   {
      _datasetRegistrationSuccess = true;
      verifyAndSendResult();
   }
   else
   {
      vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_CANNOTCOMPLETEACTION);
   }
}


/***********************************************************************//**
*
***************************************************************************/
void clSDS_Method_NavDataRegisterDirectNDSUse::onNavStatusUpdate(
   const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationService::NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationService::NavStatusUpdate >& update)
{
   switch (update->getNavStatus())
   {
      case ::org::bosch::cm::navigation::NavigationService::NavStatus__NAVSTATUS_IDLE:
      case ::org::bosch::cm::navigation::NavigationService::NavStatus__NAVSTATUS_GUIDANCE_ACTIVE:
      case ::org::bosch::cm::navigation::NavigationService::NavStatus__NAVSTATUS_CALCULATING_ROUTE:
      {
         if (!_globalDataResponseReceived)
         {
            _naviSDSProxy->sendGetGlobalDataForSdsRequest(*this);
         }
      }
      break;

      default:
         break;
   }
}


/***********************************************************************//**
*
***************************************************************************/
void clSDS_Method_NavDataRegisterDirectNDSUse::onNavStatusError(
   const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationService::NavigationServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< org::bosch::cm::navigation::NavigationService::NavStatusError >& /*error*/)
{
}


/**************************************************************************//**
*
******************************************************************************/

static sds2hmi_fi_tcl_e8NDSPhoneticTranscriptionFormat::tenType naviToSdsPhonemeFormat(PhoneticTranscriptionFormat naviPhonemeFormat)
{
   switch (naviPhonemeFormat)
   {
      case (PhoneticTranscriptionFormat__I_FLYTEC):
         return sds2hmi_fi_tcl_e8NDSPhoneticTranscriptionFormat::FI_EN_I_FLYTEC;

      case (PhoneticTranscriptionFormat__TA_IPA):
         return sds2hmi_fi_tcl_e8NDSPhoneticTranscriptionFormat::FI_EN_TA_IPA;

      case (PhoneticTranscriptionFormat__STARREC_SAMPA):
         return sds2hmi_fi_tcl_e8NDSPhoneticTranscriptionFormat::FI_EN_STARREC_SAMPA;

      case (PhoneticTranscriptionFormat__X_SAMPA):
         return sds2hmi_fi_tcl_e8NDSPhoneticTranscriptionFormat::FI_EN_X_SAMPA;

      case (PhoneticTranscriptionFormat__LH_PLUS):
         return sds2hmi_fi_tcl_e8NDSPhoneticTranscriptionFormat::FI_EN_LH_PLUS;

      case (PhoneticTranscriptionFormat__NT_SAMPA):
         return sds2hmi_fi_tcl_e8NDSPhoneticTranscriptionFormat::FI_EN_NT_SAMPA;

      case (PhoneticTranscriptionFormat__SVOX_PA):
         return sds2hmi_fi_tcl_e8NDSPhoneticTranscriptionFormat::FI_EN_SVOX_PA;

      case (PhoneticTranscriptionFormat__VFA):
         return sds2hmi_fi_tcl_e8NDSPhoneticTranscriptionFormat::FI_EN_VFA;

      default:
         return sds2hmi_fi_tcl_e8NDSPhoneticTranscriptionFormat::FI_EN_TA_IPA;
   }
}


/**************************************************************************//**
*
******************************************************************************/


static tU16 naviToSdsBlockType(NDSBuildingBlockType naviBlockType)
{
   switch (naviBlockType)
   {
      case NDSBuildingBlockType__SPEECH:
         return SDS_BUILDING_BLOCK_SPEECH;

      case NDSBuildingBlockType__EXTENSION:
         return SDS_BUILDING_BLOCK_EXTENSION;

      default:
         return (tU16)naviBlockType;
   }
}


/**************************************************************************//**
*
******************************************************************************/


void clSDS_Method_NavDataRegisterDirectNDSUse::setSdsProducts(
   const ::std::string& device,
   const ::std::string& persistentDataSetId,
   const ::std::vector<SdsProducts>& naviSdsProductsList)
{
   clearsdsProducts();
   _device.bSet(device.c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
   _persistentDataSetId.bSet(persistentDataSetId.c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
   for (size_t productIndex = 0; productIndex < naviSdsProductsList.size(); ++productIndex)
   {
      const SdsProducts& naviProduct = naviSdsProductsList[productIndex];
      sds2hmi_fi_tcl_SDSProduct sdsProduct;
      sdsProduct.SupplierId = naviProduct.getSupplierId();
      sdsProduct.ProductId = naviProduct.getProductId();
      for (size_t regionIndex = 0; regionIndex < naviProduct.getSdsUpdateRegionsList().size(); ++regionIndex)
      {
         const SdsUpdateRegions& updateRegion = naviProduct.getSdsUpdateRegionsList()[regionIndex];
         sds2hmi_fi_tcl_UpdateRegion sdsUpdateRegionItem;
         sdsUpdateRegionItem.UpdateRegionId = updateRegion.getUpdateRegionId();
         for (size_t blockIndex = 0; blockIndex < updateRegion.getSdsBuildingBlocksList().size(); ++blockIndex)
         {
            const SdsBuildingBlocks& buildingBlock = updateRegion.getSdsBuildingBlocksList()[blockIndex];
            sds2hmi_fi_tcl_BuildingBlock sdsBuildingBlockItem;
            sdsBuildingBlockItem.BuildingBlockId = buildingBlock.getBuildingBlockId();
            sdsBuildingBlockItem.BuildingBlockType = naviToSdsBlockType(buildingBlock.getNdsBuildingBlockType());
            sdsBuildingBlockItem.Path.bSet(buildingBlock.getUri().c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
            sdsBuildingBlockItem.EncryptionKeyId = buildingBlock.getEncryptionKeyId();
            sdsBuildingBlockItem.TypeNDSSupplierId = buildingBlock.getTypeNdsDbSupplierId();
            sdsBuildingBlockItem.TypeExtId = buildingBlock.getTypeExtId();
            sdsBuildingBlockItem.MetaData.HasPhonetic = buildingBlock.getSpeechMetadata().getHasPhonetic();
            sdsBuildingBlockItem.MetaData.PhoneticTranscriptionFormat.enType = naviToSdsPhonemeFormat(buildingBlock.getSpeechMetadata().getPhoneticTraMetadata().getPhoneticTranscriptionFormat());
            sdsUpdateRegionItem.BuilidingBlockList.push_back(sdsBuildingBlockItem);
            ETG_TRACE_USR4((
                              "NAVI-BuildingBlockType = %d, BuildingBlockType = %d, TypeNDSSupplierId = %d, Navi-TypeExtId = %d, TypeExtId = %d",
                              buildingBlock.getNdsBuildingBlockType(),
                              sdsBuildingBlockItem.BuildingBlockType,
                              sdsBuildingBlockItem.TypeNDSSupplierId,
                              buildingBlock.getTypeExtId(),
                              sdsBuildingBlockItem.TypeExtId));
         }

         for (size_t metadataIndex = 0; metadataIndex < updateRegion.getSdsRegionMetadataList().size(); ++metadataIndex)
         {
            const SdsRegionMetadata& regionMetadata = updateRegion.getSdsRegionMetadataList()[metadataIndex];
            sds2hmi_fi_tcl_RegionMetaData sdsRegionMetaDataItem;
            sdsRegionMetaDataItem.RegionMetaDataId = regionMetadata.getUpdateRegionId();
            sdsRegionMetaDataItem.ISO_CountryCodeTxt.bSet(regionMetadata.getIsoCountryCodeText().c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
            sdsRegionMetaDataItem.ISO_SubCountryCodeTxt.bSet(regionMetadata.getIsoSubCountryCodeText().c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
            sdsUpdateRegionItem.RegionMetaDataList.push_back(sdsRegionMetaDataItem);
         }
         sdsProduct.UpdateregionList.push_back(sdsUpdateRegionItem);
      }

      for (size_t languageIndex = 0; languageIndex < naviProduct.getSdsLanguagesList().size(); ++languageIndex)
      {
         const SdsLanguages& language = naviProduct.getSdsLanguagesList()[languageIndex];
         sds2hmi_fi_tcl_NDSLanguageID sdsNDSLanguageIdItem;
         sdsNDSLanguageIdItem.LanguageCode = language.getLanguageCode();
         sdsNDSLanguageIdItem.ISO_LanguageCode.bSet(language.getIsoLanguageCode().c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
         sdsNDSLanguageIdItem.ISO_CountryCode.bSet(language.getIsoCountryCode().c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
         sdsNDSLanguageIdItem.ScriptCode.bSet(language.getIsoScriptCode().c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
         sdsProduct.NDSLanguageList.push_back(sdsNDSLanguageIdItem);
      }
      _sdsProductsList.push_back(sdsProduct);
   }
}


void clSDS_Method_NavDataRegisterDirectNDSUse::clearsdsProducts()
{
   _device = "";
   _persistentDataSetId = "";
   _sdsProductsList.clear();
}


void clSDS_Method_NavDataRegisterDirectNDSUse::setUpdateRegionSpecifierRelationList(const ::std::vector <UpdateRegionSpecifierRelations>& updateRegionSpecifierRelationsList)
{
   _regionSpecifierRelationList = updateRegionSpecifierRelationsList;
}


bool clSDS_Method_NavDataRegisterDirectNDSUse::getUpdateRegionSpecifierValues(uint16 regionSpecifier, UpdateRegionSpecifierRelations& currentValue)
{
   for (size_t i = 0; i < _regionSpecifierRelationList.size(); ++i)
   {
      const UpdateRegionSpecifierRelations& relation = _regionSpecifierRelationList[i];
      if (relation.getUpdateRegionSpecifier() == regionSpecifier)
      {
         currentValue.setProductId(relation.getProductId());
         currentValue.setSupplierId(relation.getSupplierId());
         currentValue.setUpdateRegionId(relation.getUpdateRegionId());
         return true;
      }
   }
   return false;
}


/**************************************************************************//**
*
******************************************************************************/

void clSDS_Method_NavDataRegisterDirectNDSUse::verifyAndSendResult()
{
   if (_globalDataResponseReceived && _datasetRegistrationSuccess)
   {
      sds2hmi_sdsfi_tclMsgNavDataRegisterDirectNDSUseMethodResult result;
      result.Device = _device;
      result.PersistentDataSetId = _persistentDataSetId;
      result.SDSProductList = _sdsProductsList;
      vSendMethodResult(result);
   }
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_Method_NavDataRegisterDirectNDSUse::addGlobalDataForSdsObserver(NaviGlobalDataForSdsObserver* pObserver)
{
   if (pObserver != NULL)
   {
      _globalDataForSdsObservers.push_back(pObserver);
   }
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_Method_NavDataRegisterDirectNDSUse::notifyGlobalDataForSdsObservers()
{
   for (size_t i = 0; i < _globalDataForSdsObservers.size(); ++i)
   {
      _globalDataForSdsObservers.at(i)->onNaviGlobalDataForSdsResponse();
   }
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_Method_NavDataRegisterDirectNDSUse::onNaviDataSetDeRegistration()
{
   ETG_TRACE_USR1(("on NaviDataSetDeRegistration"));
   _globalDataResponseReceived = false;
}
