/**************************************************************************//**
 * \file       clSDS_Method_NavDataGetCountryStateList.cpp
 *
 * clSDS_Method_NavDataGetCountryStateList method class implementation
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "Sds2HmiServer/functions/clSDS_Method_NavDataGetCountryStateList.h"
#include "SdsAdapter_Trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS           TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/clSDS_Method_NavDataGetCountryStateList.cpp.trc.h"
#endif


using namespace org::bosch::cm::navigation::NavigationSDSService;


/**************************************************************************//**
 * Destructor
 ******************************************************************************/
clSDS_Method_NavDataGetCountryStateList::~clSDS_Method_NavDataGetCountryStateList()
{
}


/**************************************************************************//**
 * Constructor
 ******************************************************************************/
clSDS_Method_NavDataGetCountryStateList::clSDS_Method_NavDataGetCountryStateList(ahl_tclBaseOneThreadService* pService,
      ::boost::shared_ptr< NavigationSDSServiceProxy > pSds2NaviProxy)
   : clServerMethod(SDS2HMI_SDSFI_C_U16_NAVDATAGETCOUNTRYSTATELIST, pService)
   , _sds2NaviProxy(pSds2NaviProxy)
{
   _ndsRequestor = NDSID_UNKNWON;
   _ndsNameRequestor = NDSID_UNKNWON;
   _requestedsubCountryNdsIdSize = 0;
   _requestedsubCountryNdsIdforNameSize = 0;
   _countrySubCountryPhonemeList.clear();
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_NavDataGetCountryStateList::vMethodStart(amt_tclServiceData* pInMessage)
{
   sds2hmi_sdsfi_tclMsgNavDataGetCountryStateListMethodStart oMessage;
   vGetDataFromAmt(pInMessage, oMessage);
   _ndsLanguageId = oMessage.NDSLanguageID;
   _ndsRequestor = NDSID_FOR_COUNTRY;
   _ndsNameRequestor = NDSID_FOR_COUNTRY;
   _sds2NaviProxy->sendGetIdListOfCountryAndSubCountryRequest(*this);
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NavDataGetCountryStateList::onGetIdListOfCountryAndSubCountryResponse(
   const ::boost::shared_ptr< NavigationSDSServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< GetIdListOfCountryAndSubCountryResponse >& response)
{
   _countryIdList = response->getCountryIdList();

   _stateIdList = response->getSubCountryIdList();

   getPhonemesForIds(_countryIdList);
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NavDataGetCountryStateList::onGetIdListOfCountryAndSubCountryError(
   const ::boost::shared_ptr< NavigationSDSServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< GetIdListOfCountryAndSubCountryError >& /*error*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NavDataGetCountryStateList::onGetPhonemesForNDSIdsResponse(
   const ::boost::shared_ptr< NavigationSDSServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< GetPhonemesForNDSIdsResponse >& response)
{
   if (NDSID_FOR_COUNTRY == _ndsRequestor)
   {
      _countryPhonemeList = response->getNamedObjectIdStringsList();
      _ndsRequestor = NDSID_FOR_STATE;

      _requestedsubCountryNdsIdSize = 0;

      if (!_stateIdList.empty())
      {
         requestStatePhonemes(_stateIdList.at(_requestedsubCountryNdsIdSize));
      }
      else
      {
         getNamesForIds(_countryIdList);
      }
   }
   else if (NDSID_FOR_STATE == _ndsRequestor)
   {
      std::vector< NamedObjectIdStrings > subCountryPhonemeList;

      subCountryPhonemeList = response->getNamedObjectIdStringsList();

      updateCountrySubCountryPhonemes(subCountryPhonemeList);

      if (_requestedsubCountryNdsIdSize < _stateIdList.size())
      {
         requestStatePhonemes(_stateIdList.at(_requestedsubCountryNdsIdSize));
      }

      else
      {
         getNamesForIds(_countryIdList);
      }
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NavDataGetCountryStateList::onGetNamesForNDSIdsResponse(
   const ::boost::shared_ptr< NavigationSDSServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< GetNamesForNDSIdsResponse >& response)
{
   if (NDSID_FOR_COUNTRY == _ndsNameRequestor)
   {
      _countryNameList = response->getNamedObjectIdStringsList();
      _ndsNameRequestor = NDSID_FOR_STATE;

      _requestedsubCountryNdsIdforNameSize = 0;

      if (!_stateIdList.empty())
      {
         requestStateNames(_stateIdList.at(_requestedsubCountryNdsIdforNameSize));
      }
      else
      {
         sendCountryStatePhonemesAndNames();
      }
   }
   else if (NDSID_FOR_STATE == _ndsNameRequestor)
   {
      std::vector< NamedObjectIdStrings > subCountryNameList;

      subCountryNameList = response->getNamedObjectIdStringsList();

      updateCountrySubCountryNames(subCountryNameList);

      if (_requestedsubCountryNdsIdforNameSize < _stateIdList.size())
      {
         requestStateNames(_stateIdList.at(_requestedsubCountryNdsIdforNameSize));
      }
      else
      {
         sendCountryStatePhonemesAndNames();
      }
   }
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NavDataGetCountryStateList::onGetPhonemesForNDSIdsError(
   const ::boost::shared_ptr< NavigationSDSServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< GetPhonemesForNDSIdsError >& /*error*/)
{
}


/***********************************************************************//**
 *
 ***************************************************************************/
void clSDS_Method_NavDataGetCountryStateList::onGetNamesForNDSIdsError(
   const ::boost::shared_ptr< NavigationSDSServiceProxy >& /*proxy*/,
   const ::boost::shared_ptr< GetPhonemesForNDSIdsError >& /*error*/)
{
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_NavDataGetCountryStateList::getPhonemesForIds(std::vector< GlobalNamedObjectAndRegionId > ndsIds)
{
   std::vector< LanguageEntry > languageEntryList;
   NDSFeatureType ndsFeatureType;

   LanguageEntry languageEntry;
   languageEntry.setISOLanguageCode(getString(_ndsLanguageId.ISO_LanguageCode));
   languageEntry.setISOCountryCode(getString(_ndsLanguageId.ISO_CountryCode));
   languageEntry.setTranscriptCode(getString(_ndsLanguageId.ScriptCode));
   languageEntryList.push_back(languageEntry);

   ndsFeatureType = NDSFeatureType__NAMEDOBJECT;

   std::vector< GlobalNamedObjectId > namedObjectIdList;

   std::vector< GlobalNamedObjectAndRegionId >::iterator ndsIdsIt;

   for (ndsIdsIt = ndsIds.begin(); ndsIdsIt != ndsIds.end(); ++ndsIdsIt)
   {
      GlobalNamedObjectId namedObjectId;
      namedObjectId.setGlobalUpdateRegionId(ndsIdsIt->getGlobalUpdateRegionId());
      namedObjectId.setNamedObjectId(ndsIdsIt->getNamedObjectId());
      namedObjectIdList.push_back(namedObjectId);
   }

   _sds2NaviProxy->sendGetPhonemesForNDSIdsRequest(*this, namedObjectIdList, languageEntryList, ndsFeatureType, false);
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_NavDataGetCountryStateList::getNamesForIds(std::vector< GlobalNamedObjectAndRegionId > ndsIds)
{
   std::vector< LanguageEntry > languageEntryList;
   NDSFeatureType ndsFeatureType;

   LanguageEntry languageEntry;

   languageEntry.setISOLanguageCode(_ndsLanguageId.ISO_LanguageCode);
   languageEntry.setISOCountryCode(_ndsLanguageId.ISO_CountryCode);
   languageEntry.setTranscriptCode(_ndsLanguageId.ScriptCode);
   languageEntryList.push_back(languageEntry);

   languageEntry.setISOLanguageCode("");
   languageEntry.setISOCountryCode("");
   languageEntry.setTranscriptCode(_ndsLanguageId.ScriptCode);
   languageEntryList.push_back(languageEntry);

   ndsFeatureType = NDSFeatureType__NAMEDOBJECT;

   std::vector< GlobalNamedObjectId > namedObjectIdList;
   std::vector< GlobalNamedObjectAndRegionId >::iterator ndsIdsIt;

   for (ndsIdsIt = ndsIds.begin(); ndsIdsIt != ndsIds.end(); ++ndsIdsIt)
   {
      GlobalNamedObjectId namedObjectId;
      namedObjectId.setGlobalUpdateRegionId(ndsIdsIt->getGlobalUpdateRegionId());
      namedObjectId.setNamedObjectId(ndsIdsIt->getNamedObjectId());
      namedObjectIdList.push_back(namedObjectId);
   }

   _sds2NaviProxy->sendGetNamesForNDSIdsRequest(*this, namedObjectIdList, languageEntryList, ndsFeatureType, true);
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_NavDataGetCountryStateList::sendCountryStatePhonemesAndNames()
{
   sds2hmi_sdsfi_tclMsgNavDataGetCountryStateListMethodResult result;

   std::vector< NamedObjectIdStrings >::iterator countryPhonemeIt = _countryPhonemeList.begin();
   std::vector< NamedObjectIdStrings >::iterator countryNameIt = _countryNameList.begin();

   std::vector< GlobalNamedObjectAndRegionId >::iterator countryIt;

   while (countryPhonemeIt != _countryPhonemeList.end())
   {
      sds2hmi_fi_tcl_CountryPhoneme countryStatePhonemes;

      countryStatePhonemes.CountryId.u16UpdateRegionId = countryPhonemeIt->getGlobalNamedObjectId().getGlobalUpdateRegionId().getUpdateRegionId();
      countryStatePhonemes.CountryId.u32ProductId = countryPhonemeIt->getGlobalNamedObjectId().getGlobalUpdateRegionId().getProductId();
      countryStatePhonemes.CountryId.u8SupplierId = countryPhonemeIt->getGlobalNamedObjectId().getGlobalUpdateRegionId().getSupplierId();
      countryStatePhonemes.CountryId.u32NDSId = countryPhonemeIt->getGlobalNamedObjectId().getNamedObjectId();
      countryStatePhonemes.CountryId.eNDSFeatureType.enType = sds2hmi_fi_tcl_e8_NDSFeatureType::FI_EN_NAMEOBJECT;
      for (countryIt = _countryIdList.begin(); countryIt != _countryIdList.end(); ++countryIt)
      {
         if (countryStatePhonemes.CountryId.u32NDSId == countryIt->getNamedObjectId()  && countryStatePhonemes.CountryId.u16UpdateRegionId == countryIt->getGlobalUpdateRegionId().getUpdateRegionId())
         {
            countryStatePhonemes.CountryId.POIRootRegionID = countryIt->getPoiRegionId();
            countryStatePhonemes.CountryId.GlobalXRefId = countryIt->getGlobalXrefID();
            break;
         }
      }

      std::vector< ::std::string > phonemeList;
      phonemeList = countryPhonemeIt->getSdsNamedStringsList();

      setCountryPhonemes(phonemeList , countryStatePhonemes);

      result.CountryStateList.push_back(countryStatePhonemes);

      ++countryPhonemeIt;
   }

   while (countryNameIt != _countryNameList.end())
   {
      bool countryPhonemeAvailable = false;

      for (size_t i = 0; i < result.CountryStateList.size(); i++)
      {
         if ((countryNameIt->getGlobalNamedObjectId().getGlobalUpdateRegionId().getUpdateRegionId() == result.CountryStateList[i].CountryId.u16UpdateRegionId) && (countryNameIt->getGlobalNamedObjectId().getNamedObjectId() == result.CountryStateList[i].CountryId.u32NDSId))
         {
            std::vector< ::std::string > NameList;
            NameList = countryNameIt->getSdsNamedStringsList();

            setCountryNames(NameList, result.CountryStateList[i]);

            countryPhonemeAvailable = true;
            break;
         }
      }

      if (!countryPhonemeAvailable)
      {
         sds2hmi_fi_tcl_CountryPhoneme countryStatePhonemes;

         countryStatePhonemes.CountryId.u16UpdateRegionId = countryNameIt->getGlobalNamedObjectId().getGlobalUpdateRegionId().getUpdateRegionId();
         countryStatePhonemes.CountryId.u32ProductId = countryNameIt->getGlobalNamedObjectId().getGlobalUpdateRegionId().getProductId();
         countryStatePhonemes.CountryId.u8SupplierId = countryNameIt->getGlobalNamedObjectId().getGlobalUpdateRegionId().getSupplierId();
         countryStatePhonemes.CountryId.u32NDSId = countryNameIt->getGlobalNamedObjectId().getNamedObjectId();
         countryStatePhonemes.CountryId.eNDSFeatureType.enType = sds2hmi_fi_tcl_e8_NDSFeatureType::FI_EN_NAMEOBJECT;

         for (countryIt = _countryIdList.begin(); countryIt != _countryIdList.end(); ++countryIt)
         {
            if (countryStatePhonemes.CountryId.u32NDSId == countryIt->getNamedObjectId()  && countryStatePhonemes.CountryId.u16UpdateRegionId == countryIt->getGlobalUpdateRegionId().getUpdateRegionId())
            {
               countryStatePhonemes.CountryId.POIRootRegionID = countryIt->getPoiRegionId();
               countryStatePhonemes.CountryId.GlobalXRefId = countryIt->getGlobalXrefID();
               break;
            }
         }

         std::vector< ::std::string > NameList;
         NameList = countryNameIt->getSdsNamedStringsList();

         setCountryNames(NameList, countryStatePhonemes);

         result.CountryStateList.push_back(countryStatePhonemes);
      }

      ++countryNameIt;
   }

   for (size_t i = 0; i < result.CountryStateList.size(); i++)
   {
      setStatePhonemesAndNames(result.CountryStateList[i] , result.CountryStateList[i].CountryId.u32NDSId ,
                               result.CountryStateList[i].CountryId.u16UpdateRegionId);
   }

   vSendMethodResult(result);

   resetDataMembers();
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_NavDataGetCountryStateList::setCountryPhonemes(std::vector< ::std::string > countryPhonemeList,
      sds2hmi_fi_tcl_CountryPhoneme& countryPhonemeAndName)
{
   std::vector< ::std::string >::iterator phonemeIt;

   for (phonemeIt = countryPhonemeList.begin(); phonemeIt != countryPhonemeList.end(); ++phonemeIt)
   {
      sds2hmi_fi_tclString sdsString;
      sdsString.bSet(phonemeIt->c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
      countryPhonemeAndName.PhonemeList.push_back(sdsString);
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_NavDataGetCountryStateList::setCountryNames(std::vector< ::std::string > countryNameList,
      sds2hmi_fi_tcl_CountryPhoneme& countryPhonemeAndName)
{
   std::vector< ::std::string >::iterator nameIt;

   for (nameIt = countryNameList.begin(); nameIt != countryNameList.end(); ++nameIt)
   {
      sds2hmi_fi_tclString sdsString;
      sdsString.bSet(nameIt->c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
      countryPhonemeAndName.OrthographyList.push_back(sdsString);
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_NavDataGetCountryStateList::setStatePhonemesAndNames(
   sds2hmi_fi_tcl_CountryPhoneme& CountryPhoneme,
   tU32 countryNdsId,
   tU16 updateRegionId)
{
   std::vector < CountrySubCountryPhonemes >::iterator subCountryPhonemeListIt = _countrySubCountryPhonemeList.begin();
   std::vector < CountrySubCountryPhonemes >::iterator subCountryNameListIt = _countrySubCountryNameList.begin();

   while ((subCountryPhonemeListIt != _countrySubCountryPhonemeList.end()) && (subCountryNameListIt != _countrySubCountryNameList.end()))
   {
      if ((countryNdsId == subCountryPhonemeListIt->countryNdsId) && (updateRegionId == subCountryPhonemeListIt->updateRegionId) && (countryNdsId == subCountryNameListIt->countryNdsId) && (updateRegionId == subCountryNameListIt->updateRegionId))
      {
         std::vector< NamedObjectIdStrings >::iterator statePhonemeIt;
         std::vector< NamedObjectIdStrings >::iterator stateNameIt;

         std::vector< NamedObjectIdStrings> statePhonemeList;
         statePhonemeList = subCountryPhonemeListIt->namedObjectIdStrings;

         std::vector< NamedObjectIdStrings> stateNameList;
         stateNameList = subCountryNameListIt->namedObjectIdStrings;

         statePhonemeIt = statePhonemeList.begin();
         stateNameIt = stateNameList.begin();

         std::vector< SubCountryIds >::iterator countryIt;
         std::vector< GlobalNamedObjectAndRegionId > subCountryList;
         std::vector< GlobalNamedObjectAndRegionId >::iterator subCountryIt;

         for (countryIt = _stateIdList.begin(); countryIt != _stateIdList.end(); ++countryIt)

         {
            if (countryNdsId == countryIt->getCountryId().getNamedObjectId() && updateRegionId == countryIt->getCountryId().getGlobalUpdateRegionId().getUpdateRegionId())
            {
               subCountryList = countryIt->getSubCountryIdList();
               break;
            }
         }

         while (statePhonemeIt != statePhonemeList.end())
         {
            sds2hmi_fi_tcl_StatePhoneme statePhonemes;

            statePhonemes.StateId.u16UpdateRegionId = statePhonemeIt->getGlobalNamedObjectId().getGlobalUpdateRegionId().getUpdateRegionId();
            statePhonemes.StateId.u32ProductId = statePhonemeIt->getGlobalNamedObjectId().getGlobalUpdateRegionId().getProductId();
            statePhonemes.StateId.u8SupplierId = statePhonemeIt->getGlobalNamedObjectId().getGlobalUpdateRegionId().getSupplierId();
            statePhonemes.StateId.u32NDSId = statePhonemeIt->getGlobalNamedObjectId().getNamedObjectId();
            statePhonemes.StateId.eNDSFeatureType.enType = sds2hmi_fi_tcl_e8_NDSFeatureType::FI_EN_NAMEOBJECT;

            for (subCountryIt = subCountryList.begin(); subCountryIt != subCountryList.end(); subCountryIt++)
            {
               if (statePhonemes.StateId.u32NDSId  == subCountryIt->getNamedObjectId()  && statePhonemes.StateId.u16UpdateRegionId == subCountryIt->getGlobalUpdateRegionId().getUpdateRegionId())
               {
                  statePhonemes.StateId.POIRootRegionID = subCountryIt->getPoiRegionId();
                  statePhonemes.StateId.GlobalXRefId = subCountryIt->getGlobalXrefID();
                  break;
               }
            }

            std::vector< ::std::string > phonemeList;
            phonemeList = statePhonemeIt->getSdsNamedStringsList();

            std::vector< ::std::string >::iterator phonemeIt ;

            for (phonemeIt = phonemeList.begin(); phonemeIt != phonemeList.end(); ++phonemeIt)
            {
               sds2hmi_fi_tclString sdsString;
               sdsString.bSet(phonemeIt->c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
               statePhonemes.PhonemeList.push_back(sdsString);
            }

            CountryPhoneme.StateList.push_back(statePhonemes);
            ++statePhonemeIt;
         }

         while (stateNameIt != stateNameList.end())
         {
            bool statePhonemeAvailable = false;

            for (size_t i = 0; i < CountryPhoneme.StateList.size(); i++)
            {
               if ((stateNameIt->getGlobalNamedObjectId().getGlobalUpdateRegionId().getUpdateRegionId() == CountryPhoneme.StateList[i].StateId.u16UpdateRegionId) && (stateNameIt->getGlobalNamedObjectId().getNamedObjectId() == CountryPhoneme.StateList[i].StateId.u32NDSId))
               {
                  std::vector< ::std::string > nameList;
                  nameList = stateNameIt->getSdsNamedStringsList();
                  std::vector< ::std::string >::iterator nameIt;

                  for (nameIt = nameList.begin(); nameIt != nameList.end(); ++nameIt)
                  {
                     sds2hmi_fi_tclString sdsString;
                     sdsString.bSet(nameIt->c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
                     CountryPhoneme.StateList[i].OrthographyList.push_back(sdsString);
                  }

                  statePhonemeAvailable = true;
                  break;
               }
            }

            if (! statePhonemeAvailable)
            {
               sds2hmi_fi_tcl_StatePhoneme statePhonemes;

               statePhonemes.StateId.u16UpdateRegionId = stateNameIt->getGlobalNamedObjectId().getGlobalUpdateRegionId().getUpdateRegionId();
               statePhonemes.StateId.u32ProductId = stateNameIt->getGlobalNamedObjectId().getGlobalUpdateRegionId().getProductId();
               statePhonemes.StateId.u8SupplierId = stateNameIt->getGlobalNamedObjectId().getGlobalUpdateRegionId().getSupplierId();
               statePhonemes.StateId.u32NDSId = stateNameIt->getGlobalNamedObjectId().getNamedObjectId();
               statePhonemes.StateId.eNDSFeatureType.enType = sds2hmi_fi_tcl_e8_NDSFeatureType::FI_EN_NAMEOBJECT;

               for (subCountryIt = subCountryList.begin(); subCountryIt != subCountryList.end(); ++subCountryIt)
               {
                  if (statePhonemes.StateId.u32NDSId  == subCountryIt->getNamedObjectId()  && statePhonemes.StateId.u16UpdateRegionId == subCountryIt->getGlobalUpdateRegionId().getUpdateRegionId())
                  {
                     statePhonemes.StateId.POIRootRegionID = subCountryIt->getPoiRegionId();
                     statePhonemes.StateId.GlobalXRefId = subCountryIt->getGlobalXrefID();
                     break;
                  }
               }

               std::vector< ::std::string > nameList;
               nameList = stateNameIt->getSdsNamedStringsList();
               std::vector< ::std::string >::iterator nameIt;

               for (nameIt = nameList.begin(); nameIt != nameList.end(); ++nameIt)
               {
                  sds2hmi_fi_tclString sdsString;
                  sdsString.bSet(nameIt->c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
                  statePhonemes.OrthographyList.push_back(sdsString);
               }

               CountryPhoneme.StateList.push_back(statePhonemes);
            }
            ++stateNameIt;
         }
      }

      ++subCountryPhonemeListIt;
      ++subCountryNameListIt;
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_NavDataGetCountryStateList::requestStatePhonemes(SubCountryIds subCountryIds)
{
   std::vector< GlobalNamedObjectAndRegionId > subCountryIdList;

   subCountryIdList = subCountryIds.getSubCountryIdList();

   getPhonemesForIds(subCountryIdList);
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_NavDataGetCountryStateList::requestStateNames(SubCountryIds subCountryIds)
{
   std::vector< GlobalNamedObjectAndRegionId > subCountryIdList;

   subCountryIdList = subCountryIds.getSubCountryIdList();

   getNamesForIds(subCountryIdList);
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_NavDataGetCountryStateList::updateCountrySubCountryPhonemes(std::vector< NamedObjectIdStrings > stateList)
{
   if (_requestedsubCountryNdsIdSize < _stateIdList.size())
   {
      CountrySubCountryPhonemes countrySubCountryPhonemeObject;

      countrySubCountryPhonemeObject.countryNdsId = _stateIdList.at(_requestedsubCountryNdsIdSize).getCountryId().getNamedObjectId();
      countrySubCountryPhonemeObject.updateRegionId = _stateIdList.at(_requestedsubCountryNdsIdSize).getCountryId().getGlobalUpdateRegionId().getUpdateRegionId();
      countrySubCountryPhonemeObject.namedObjectIdStrings = stateList;

      _countrySubCountryPhonemeList.push_back(countrySubCountryPhonemeObject);

      _requestedsubCountryNdsIdSize++;
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_NavDataGetCountryStateList::updateCountrySubCountryNames(std::vector< NamedObjectIdStrings > stateList)
{
   if (_requestedsubCountryNdsIdforNameSize < _stateIdList.size())
   {
      CountrySubCountryPhonemes countrySubCountryNameObject;

      countrySubCountryNameObject.countryNdsId = _stateIdList.at(_requestedsubCountryNdsIdforNameSize).getCountryId().getNamedObjectId();
      countrySubCountryNameObject.updateRegionId = _stateIdList.at(_requestedsubCountryNdsIdforNameSize).getCountryId().getGlobalUpdateRegionId().getUpdateRegionId();
      countrySubCountryNameObject.namedObjectIdStrings = stateList;

      _countrySubCountryNameList.push_back(countrySubCountryNameObject);

      _requestedsubCountryNdsIdforNameSize++;
   }
}


/**************************************************************************//**
 *
 ******************************************************************************/
void clSDS_Method_NavDataGetCountryStateList::resetDataMembers()
{
   _ndsRequestor = NDSID_UNKNWON;
   _ndsNameRequestor = NDSID_UNKNWON;
   _countryIdList.clear();
   _stateIdList.clear();
   _countryPhonemeList.clear();
   _countryNameList.clear();
   _countrySubCountryPhonemeList.clear();
   _countrySubCountryNameList.clear();
   _requestedsubCountryNdsIdSize = 0;
   _requestedsubCountryNdsIdforNameSize = 0;
}
