/**************************************************************************//**
 * \file       clSDS_Method_MediaGetAmbiguityList.cpp
 *
 * clSDS_Method_MediaGetAmbiguityList method class implementation
 *
 * \copyright  (C) 2016 Robert Bosch GmbH
 *             (C) 2016 Robert Bosch Engineering and Business Solutions Limited
 *             The reproduction, distribution and utilization of this file
 *             as well as the communication of its contents to others without
 *             express authorization is prohibited. Offenders will be held
 *             liable for the payment of damages. All rights reserved in the
 *             event of the grant of a patent, utility model or design.
 *****************************************************************************/
#include "Sds2HmiServer/functions/clSDS_Method_MediaGetAmbiguityList.h"
#include "application/clSDS_NBestList.h"
#include "SdsAdapter_Trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SDSADP_DETAILS
#include "trcGenProj/Header/clSDS_Method_MediaGetAmbiguityList.cpp.trc.h"
#endif


using namespace mplay_MediaPlayer_FI;
using namespace MPlay_fi_types;


/**************************************************************************//**
* Destructor
******************************************************************************/
clSDS_Method_MediaGetAmbiguityList::~clSDS_Method_MediaGetAmbiguityList()
{
   _pNBestList = NULL;
}


/**************************************************************************//**
* Constructor
******************************************************************************/
clSDS_Method_MediaGetAmbiguityList::clSDS_Method_MediaGetAmbiguityList(
   ahl_tclBaseOneThreadService* pService,
   ::boost::shared_ptr< ::mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy > mediaPlayerProxy,
   clSDS_NBestList* pNBestList)

   : clServerMethod(SDS2HMI_SDSFI_C_U16_MEDIAGETAMBIGUITYLIST, pService)
   , _mediaPlayerProxy(mediaPlayerProxy)
   , _pNBestList(pNBestList)
   , _pendingResponses(0)
{
}


/**************************************************************************//**
* Request media objects for all given ambiguous media ids.
* Multiple request messages may be sent to the media player.
* We need to wait for all response messages before responding to SDS.
******************************************************************************/
tVoid clSDS_Method_MediaGetAmbiguityList::vMethodStart(amt_tclServiceData* pInMessage)
{
   sds2hmi_sdsfi_tclMsgMediaGetAmbiguityListMethodStart oMessage;
   vGetDataFromAmt(pInMessage, oMessage);
   _mediaObjects.clear();
   if (oMessage.ValueIDs.size() && _mediaPlayerProxy->isAvailable())
   {
      _pendingResponses = oMessage.ValueIDs.size();
      for (size_t i = 0; i < oMessage.ValueIDs.size(); i++)
      {
         _mediaPlayerProxy->sendGetMediaObjectStart(
            *this,
            oMessage.ValueIDs[i].ID,
            ::MPlay_fi_types::T_e8_MPlayCategoryType__e8CTY_NONE);
      }
   }
   else
   {
      vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_CANNOTCOMPLETEACTION);
   }
}


/**************************************************************************//**
* Error result received - yet count this is a result message.
******************************************************************************/
void clSDS_Method_MediaGetAmbiguityList::onGetMediaObjectError(
   const ::boost::shared_ptr< mplay_MediaPlayer_FI::Mplay_MediaPlayer_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< mplay_MediaPlayer_FI::GetMediaObjectError >& /*error*/)
{
   responseReceived();
}


/**************************************************************************//**
* Result received - just remeber and count a result message.
******************************************************************************/
void clSDS_Method_MediaGetAmbiguityList::onGetMediaObjectResult(
   const ::boost::shared_ptr<mplay_MediaPlayer_FI:: Mplay_MediaPlayer_FIProxy >& /*proxy*/,
   const ::boost::shared_ptr< mplay_MediaPlayer_FI::GetMediaObjectResult >& result)
{
   _mediaObjects.push_back(result->getOMediaObject());
   responseReceived();
}


/**************************************************************************//**
* Determine if all responses messages to
* Mplay_MediaPlayer_FIProxy::sendGetMediaObjectStart()
* have been received and trigger sending of the result.
******************************************************************************/
void clSDS_Method_MediaGetAmbiguityList::responseReceived()
{
   if (_pendingResponses)
   {
      _pendingResponses--;
      if (_pendingResponses == 0)
      {
         sendResult();
      }
   }
}


/**************************************************************************//**
* Evaluate media objects and send appropriate response to SDS_MW.
******************************************************************************/
void clSDS_Method_MediaGetAmbiguityList::sendResult()
{
   sds2hmi_sdsfi_tclMsgMediaGetAmbiguityListMethodResult result;
   result.AmbiguousEntries = convertMediaObjects();
   _mediaObjects.clear();

   if (result.AmbiguousEntries.size())
   {
      populateViewList(result.AmbiguousEntries);
      vSendMethodResult(result);
      traceResult(result.AmbiguousEntries);
   }
   else
   {
      vSendErrorMessage(SDS2HMI_SDSFI_C_U16_ERROR_MEDIAOBJNOTAVAILABLE);
   }
}


/**************************************************************************//**
* Convert media player objects to SDS media objects.
* May return less SDS media objects than media player objects in case a media
* player object evaluates to an empty string.
******************************************************************************/
std::vector<sds2hmi_fi_tcl_MediaIDString> clSDS_Method_MediaGetAmbiguityList::convertMediaObjects() const
{
   std::vector<sds2hmi_fi_tcl_MediaIDString> ambigList;

   for (size_t i = 0; i < _mediaObjects.size(); ++i)
   {
      const T_MPlayMediaObject& mediaObject = _mediaObjects[i];
      std::string listItem = makeListItemString(mediaObject);
      if (!listItem.empty())
      {
         sds2hmi_fi_tcl_MediaIDString ambigEntry;
         ambigEntry.ID = mediaObject.getU32Tag();
         ambigEntry.MediaType.enType = sds2hmi_fi_tcl_e8_MediaType::FI_EN_MEDIAOBJECTID;
         ambigEntry.String.bSet(listItem.c_str(), sds2hmi_fi_tclString::FI_EN_UTF8);
         ambigList.push_back(ambigEntry);
      }
   }
   return ambigList;
}


/**************************************************************************//**
*
******************************************************************************/
std::string clSDS_Method_MediaGetAmbiguityList::makeListItemString(const T_MPlayMediaObject& mediaObject) const
{
   const std::string& playlist = mediaObject.getSMetaDataField1();
   const std::string& artist = mediaObject.getSMetaDataField2();
   const std::string& album = mediaObject.getSMetaDataField3();
   const std::string& song = mediaObject.getSMetaDataField4();

   switch (mediaObject.getE8CategoryType())
   {
      case T_e8_MPlayCategoryType__e8CTY_SONG:
         return song + ", " + artist + ", " + album;

      case T_e8_MPlayCategoryType__e8CTY_ALBUM:
         return album + ", " + artist;

      case T_e8_MPlayCategoryType__e8CTY_ARTIST:
         return artist;

      case T_e8_MPlayCategoryType__e8CTY_PLAYLIST:
         //return trimAmbiguousPlaylistPath(mediaObject.getSFilename());
         return playlist;

      default:
         return "";
   }
}


/**************************************************************************//**
* Remove the leading "/AmbiguousPlaylists" from the playlist path.
* The path known to the user only starts after this prefix.
******************************************************************************/
std::string clSDS_Method_MediaGetAmbiguityList::trimAmbiguousPlaylistPath(const std::string& path) const
{
   const std::string header = "/AmbiguousPlaylists/";
   std::string trimmed = path;
   if (trimmed.compare(0, header.size(), header) == 0)
   {
      trimmed.erase(0, header.size() - 1);   // -1 to keep the second slash
   }
   return trimmed;
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_Method_MediaGetAmbiguityList::populateViewList(const std::vector<sds2hmi_fi_tcl_MediaIDString>& ambigList) const
{
   if (_pNBestList)
   {
      std::vector<clSDS_ListItems> list;
      for (size_t i = 0; i < ambigList.size(); ++i)
      {
         clSDS_ListItems item;
         item.oDescription.szString = ambigList[i].String.szValue;
         list.push_back(item);
      }
      _pNBestList->setItems(list);
   }
}


/**************************************************************************//**
*
******************************************************************************/
void clSDS_Method_MediaGetAmbiguityList::traceResult(const std::vector<sds2hmi_fi_tcl_MediaIDString>& ambigList) const
{
   for (size_t i = 0; i < ambigList.size(); ++i)
   {
      ETG_TRACE_USR1(("Media Id 0x%x MediaString %s", ambigList[i].ID, ambigList[i].String.szValue));
   }
}
