/**************************************************************************//**
* \file     clTunerClientServiceState.cpp
*
*           clTunerClientServiceState method class implementation
*
* \remark   Copyright: 2010 Robert Bosch GmbH, Hildesheim
******************************************************************************/

#include "sessionManagement/audio/clTunerClientServiceState.h"
#include "common/tuner_trace.h"

using namespace tuner::sessionManagement::audio;
using namespace tuner::common;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TUNER_TRACE_CLASS_SESSIONMANAGEMENT
#include "trcGenProj/Header/clTunerClientServiceState.cpp.trc.h"
#endif


/**************************************************************************//**
* Constructor
******************************************************************************/
clTunerClientServiceState::clTunerClientServiceState():
storeSourceId(SOURCEID_LIMIT)
, storeSourceActivity(SRCACTIVITY_UNKNOWN)
, allocateStoredSourceId(SOURCEID_LIMIT)
{
	for(tuner::common::enSourceId Src=tuner::common::FM1; Src < SOURCEID_LIMIT;
			Src = static_cast<tuner::common::enSourceId>(Src + 1))
	{
		SourceAvailableMap[Src] = tuner::common::TUNERCLIENTAVAILABITY_LIMIT;
	}
	/*Initialize tuner state map - NCG3D-39183*/
	currentTunerStateMap[tuner::common::AMFMTUNER] = tuner::common::TUNERTOBG ;
	currentTunerStateMap[tuner::common::DABTUNER] = tuner::common::TUNERTOBG ;
	previousTunerStateMap[tuner::common::AMFMTUNER] = tuner::common::TUNERSTATE_UNKNOWN ;
	previousTunerStateMap[tuner::common::DABTUNER] = tuner::common::TUNERSTATE_UNKNOWN ;
}


/**************************************************************************//**
* Destructor
******************************************************************************/
clTunerClientServiceState::~clTunerClientServiceState()
{
	vReset();
}

/**************************************************************************//**
*
******************************************************************************/
void clTunerClientServiceState::vReset()
{
	storeSourceId = SOURCEID_LIMIT;
	storeSourceActivity = SRCACTIVITY_UNKNOWN;
}

/***************************************************************************
 *
 ***************************************************************************/
bool clTunerClientServiceState::bIsServiceUnKnown(tuner::common::enSourceId sourceId)
{
	if (SourceAvailableMap.find(sourceId) != SourceAvailableMap.end())
	{
		return (SourceAvailableMap[sourceId] == tuner::common::TUNERCLIENTAVAILABITY_LIMIT)? true : false;
	}
	return false;
}

/***************************************************************************
 *
 ***************************************************************************/
bool clTunerClientServiceState::bIsServiceAvailable(tuner::common::enSourceId sourceId)
{
	if (SourceAvailableMap.find(sourceId) != SourceAvailableMap.end())
	{
		return (SourceAvailableMap[sourceId] == tuner::common::AVAILABLE)? true : false;
	}
	return false;
}
/***************************************************************************
 *
 ***************************************************************************/
bool clTunerClientServiceState::bIsServiceUnAvailable(tuner::common::enSourceId sourceId)
{
	if (SourceAvailableMap.find(sourceId) != SourceAvailableMap.end())
	{
		return (SourceAvailableMap[sourceId] == tuner::common::UNAVAILABLE)? true : false;
	}
	return false;
}

/***********
 *
 */

void clTunerClientServiceState::vOnServiceState(tuner::common::enTunerClient tunerClient, tuner::common::enTunerClientAvailability tunerClientAvailability)
{
	ETG_TRACE_USR4(("SESSION_MGT: vOnServiceState 	      Client         = %d\n", ETG_ENUM(TUNER_CLIENT, (tU8)tunerClient)));
	ETG_TRACE_USR4(("SESSION_MGT: vOnServiceState 	      Availability   = %d\n", ETG_ENUM(TUNER_CLIENT_AVAILABLITY, (tU8)tunerClientAvailability)));
	for(tuner::common::enSourceId Src=tuner::common::FM1; Src < SOURCEID_LIMIT;
			Src = static_cast<tuner::common::enSourceId>(Src + 1))
	{
		if(oGetTunerClient(Src) == tunerClient)
		{
			SourceAvailableMap[Src] = tunerClientAvailability;
		}
	}
}
/*******
 *
 */
void clTunerClientServiceState::vStoreRequest(tuner::common::enSourceId tunerSourceId, tuner::common::enSrcActivity tunerSourceActivity)
{
	if(bIsTemporarySource(tunerSourceId) == false)
	{
		storeSourceId = tunerSourceId;
		storeSourceActivity = tunerSourceActivity;
	}
}
/*******
 *
 */
void clTunerClientServiceState::vStoreRequest(tuner::common::enSourceId tunerSourceId, tunerString inputDevice)
{
	allocateStoredSourceId = tunerSourceId;
	allocateStoredInputDevice = inputDevice;
}
/************
 *
 *********/
tuner::common::enSourceId clTunerClientServiceState::getStoredSource()
{
	return storeSourceId;
}
/************
 *
 *********/
tuner::common::enSrcActivity clTunerClientServiceState::getStoredSourceActivity()
{
	return storeSourceActivity;
}
/************
 *
 *********/
void clTunerClientServiceState::vClearStoredRequest()
{
	storeSourceId = tuner::common::SOURCEID_LIMIT;
	storeSourceActivity = tuner::common::SRCACTIVITY_UNKNOWN;
}
/************
 *
 *********/
void clTunerClientServiceState::vOnNewTunerState(tuner::common::enTunerClient tunerClient, tuner::common::enTunerState tunerState)
{
	ETG_TRACE_USR4(("SESSION_MGT: vOnNewTunerState Client = %d State = %d\n", ETG_ENUM(TUNER_CLIENT, (tU8)tunerClient), ETG_ENUM(TUNER_STATE, (tU8)tunerState) ));
	currentTunerStateMap[tunerClient] = tunerState;
}
/************
 *
 *********/
tuner::common::enTunerState clTunerClientServiceState::oGetCurrentTunerState(tuner::common::enTunerClient tunerClient)
{
	if (currentTunerStateMap.find(tunerClient) != currentTunerStateMap.end())
	{
		ETG_TRACE_USR4(("SESSION_MGT: oGetCurrentTunerState 	   Client         = %d State = %d\n", ETG_ENUM(TUNER_CLIENT, (tU8)tunerClient), ETG_ENUM(TUNER_STATE, (tU8)currentTunerStateMap[tunerClient]) ));
		return currentTunerStateMap[tunerClient];
	}
	return tuner::common::TUNERSTATE_UNKNOWN;
}
/************
 *
 *********/
void clTunerClientServiceState::vClearTunerState(tuner::common::enTunerClient tunerClient)
{
	ETG_TRACE_USR4(("SESSION_MGT: vClearTunerState 	   Client         = %d\n", ETG_ENUM(TUNER_CLIENT, (tU8)tunerClient) ));
	currentTunerStateMap[tunerClient] = tuner::common::TUNERTOBG; // initialised to BG
}
/************
 *
 *********/
void clTunerClientServiceState::vStorePreviousTunerState()
{
	previousTunerStateMap[tuner::common::AMFMTUNER] = currentTunerStateMap[tuner::common::AMFMTUNER];
	previousTunerStateMap[tuner::common::DABTUNER] = currentTunerStateMap[tuner::common::DABTUNER];
	ETG_TRACE_USR4(("SESSION_MGT: vStorePreviousTunerState AMFMTUNER = %d, DABTUNER = %d\n", 
	ETG_ENUM(TUNER_STATE, (tU8)currentTunerStateMap[tuner::common::AMFMTUNER]), ETG_ENUM(TUNER_STATE, (tU8)currentTunerStateMap[tuner::common::DABTUNER])));
}
/************
 *
 *********/
tuner::common::enTunerState clTunerClientServiceState::oGetPreviousTunerState(tuner::common::enTunerClient tunerClient)
{
	if (previousTunerStateMap.find(tunerClient) != previousTunerStateMap.end())
	{
		ETG_TRACE_USR4(("SESSION_MGT: oGetPreviousTunerState 	   Client         = %d State = %d\n", ETG_ENUM(TUNER_CLIENT, (tU8)tunerClient), ETG_ENUM(TUNER_STATE, (tU8)previousTunerStateMap[tunerClient]) ));
		return previousTunerStateMap[tunerClient];
	}
	return tuner::common::TUNERSTATE_UNKNOWN;
}
/****************
 *
 */
void clTunerClientServiceState::vClearPreviousTunerState()
{
	previousTunerStateMap[tuner::common::AMFMTUNER] = tuner::common::TUNERSTATE_UNKNOWN;
	previousTunerStateMap[tuner::common::DABTUNER] = tuner::common::TUNERSTATE_UNKNOWN;
	ETG_TRACE_USR4(("SESSION_MGT: vClearPreviousTunerState"));
}
/****************
 *
 */
tuner::common::enSourceId clTunerClientServiceState::getAllocateStoredSourceId()
{
	return allocateStoredSourceId;
}
/****************
 *
 */
tunerString clTunerClientServiceState::getAllocateStoredInputDevice()
{
	return allocateStoredInputDevice;
}
/****************
 *
 */
void clTunerClientServiceState::vClearStoredAllocateRequest()
{
	allocateStoredSourceId = tuner::common::SOURCEID_LIMIT;
	allocateStoredInputDevice.clear();
}
