/************************************************************************
* FILE:           tunmstr_antdiag_i2cHandler.cpp
* PROJECT:        G3g
* SW-COMPONENT:   FC_TunerMaster.
*----------------------------------------------------------------------
*
* DESCRIPTION:   Implementation of the Class  tunmstr_antdiag_i2cHandler
*
*----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
* HISTORY:
* Date      | Author                            | Modification
* 20/01/2014| ECV4 ( RBEI ) Priya Prabhuswamy   | Initial version.
* 03.03.14  | ECV3 ( RBEI ) Priya Prabhuswamy   | adaptation for Different Type of External Power supply
* 17.03.14  | ECV3 ( RBEI ) Priya Prabhuswamy   | Bugfix for VWMIBB-1774
*************************************************************************/
#ifndef tunmstr_antdiag_i2cHandler_H
#include "tunmstr_antdiag_i2cHandler.h"
#endif

#ifndef tunmstr_antdiag_Measure_H
#include  "tunmstr_antdiag_Measure.h"
#endif

#ifndef tunmstr_antdiag_Manager_H
#include "tunmstr_antdiag_Manager.h"
#endif
#include "tunmstr_antdiag_Debounce.h"
#include "tunmstr_DiagLogClientHandler.h"
#include "tunmstr_MsgFromHMI.h"
#include <errno.h>
//Include for ETG trace
#include  "tunermaster_trace.h"
#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TUNMSTR_TRACE_CLASS_ANTDIAG
#include "trcGenProj/Header/tunmstr_antdiag_i2cHandler.cpp.trc.h"
#endif

#include "../convTables/antDiagMap.h"

#define TUNMSTR_ELMOSCUR_A1OVERCURERROR(u8Error)((u8Error==enA1_OverCurrentError))
#define TUNMSTR_ELMOSCUR_A2OVERCURERROR(u8Error)((u8Error==enA2_OverCurrentError))
#define TUNMSTR_MAX20084_A1_DIAG1ERROR(u8Error)(u8Error & 0x89)
#define TUNMSTR_MAX20084_A1_DIAG2ERROR(u8Error)(u8Error & 0x0a)
#define TUNMSTR_MAX20084_A2_DIAG1ERROR(u8Error)(u8Error & 0x45)
#define TUNMSTR_MAX20084_A2_DIAG2ERROR(u8Error)(u8Error & 0x06)
/*************************************************************************
 * FUNCTION:     tunmstr_antdiag_i2cHandler
 *
 * DESCRIPTION:  Constructor
 *
 * PARAMETER:    void
 *
 * RETURNVALUE:  void
 *************************************************************************/
tunmstr_antdiag_i2cHandler::tunmstr_antdiag_i2cHandler()
{
	ETG_TRACE_USR1(("tunmstr_antdiag_i2cHandler()"));
	m_potunmstr_antdiag_Measure=NULL;
	m_potunmstr_antdiag_Manager=NULL;
	m_potunmstr_antdiag_Debounce=NULL;
	m_potunmstr_DiagLogClientHandler=NULL;
	m_potunmstr_MsgFromHMI = NULL;
	for(tU8 u8Index=0;u8Index<enELMOSIcIndex;u8Index++)
	{
		m_blOutputA1Enabled[u8Index] = FALSE;
		m_blOutputA2Enabled[u8Index] = FALSE;
	}
	for(tU8 u8Index=0;u8Index<enMAX20084IcIndex;u8Index++)
	{
		m_blMax20084OutputA1En[u8Index] = FALSE;
		m_blMax20084OutputA2En[u8Index] = FALSE;
	}
	m_blDummyI2CHalfReadCompleted = false;
}
/*************************************************************************
 * FUNCTION:     tunmstr_antdiag_i2cHandler
 *
 * DESCRIPTION:  Destructor
 *
 * PARAMETER:    void
 *
 * RETURNVALUE:  void
 *************************************************************************/
tunmstr_antdiag_i2cHandler::~tunmstr_antdiag_i2cHandler()
{
	ETG_TRACE_USR1(("~tunmstr_antdiag_i2cHandler()"));
	m_potunmstr_antdiag_Measure=NULL;
	m_potunmstr_antdiag_Manager=NULL;
	m_potunmstr_antdiag_Debounce=NULL;
	m_potunmstr_DiagLogClientHandler=NULL;
	m_potunmstr_MsgFromHMI = NULL;
	for(tU8 u8Index=enELMOS1_IcIndex;u8Index<enELMOSIcIndex;u8Index++)
	{
		m_blOutputA1Enabled[u8Index] = FALSE;
		m_blOutputA2Enabled[u8Index] = FALSE;
	}
	for(tU8 u8Index=0;u8Index<enMAX20084IcIndex;u8Index++)
	{
		m_blMax20084OutputA1En[u8Index] = FALSE;
		m_blMax20084OutputA2En[u8Index] = FALSE;
	}
	m_blDummyI2CHalfReadCompleted = false;
}
/*************************************************************************
 * FUNCTION:    vSet_tunmstr_antdiag_Measure_Ptr
 *
 * DESCRIPTION: Set pointer to pointer tunmstr_antdiag_Measure
 *
 * PARAMETER:   tunmstr_antdiag_Measure* potunmstr_antdiag_Measure
 *
 * RETURNVALUE:  void
 *************************************************************************/
tVoid tunmstr_antdiag_i2cHandler::vSet_tunmstr_antdiag_Measure_Ptr
(tunmstr_antdiag_Measure* potunmstr_antdiag_Measure)
{
	if( potunmstr_antdiag_Measure != NULL )
	{
		m_potunmstr_antdiag_Measure = ( tunmstr_antdiag_Measure* )potunmstr_antdiag_Measure;
	}
}
/*************************************************************************
 * FUNCTION:    vSet_tunmstr_antdiag_Manager_Ptr
 *
 * DESCRIPTION: Set pointer to pointer tunmstr_antdiag_Manager
 *
 * PARAMETER:   tunmstr_antdiag_Manager* potunmstr_antdiag_Manager
 *
 * RETURNVALUE:  void
 *************************************************************************/
tVoid tunmstr_antdiag_i2cHandler::vSet_tunmstr_antdiag_Manager_Ptr
(tunmstr_antdiag_Manager* potunmstr_antdiag_Manager)
{
	if( potunmstr_antdiag_Manager != NULL )
	{
		m_potunmstr_antdiag_Manager = ( tunmstr_antdiag_Manager* )potunmstr_antdiag_Manager;
	}
}
/*************************************************************************
 * FUNCTION:    vSet_tunmstr_antdiag_Debounce_Ptr
 *
 * DESCRIPTION: Set pointer to pointer tunmstr_antdiag_Debounce
 *
 * PARAMETER:   tunmstr_antdiag_Debounce* potunmstr_antdiag_Debounce
 *
 * RETURNVALUE:  void
 *************************************************************************/
tVoid tunmstr_antdiag_i2cHandler::vSet_tunmstr_antdiag_Debounce_Ptr(tunmstr_antdiag_Debounce* potunmstr_antdiag_Debounce)
{
	if( potunmstr_antdiag_Debounce != NULL )
	{
		m_potunmstr_antdiag_Debounce = ( tunmstr_antdiag_Debounce* )potunmstr_antdiag_Debounce;
	}
}
/************************************************************************
 * FUNCTION:    vSet_tunmstr_DiagLogClientHandler_Ptr()
 *
 * DESCRIPTION: function to update m_potunmstr_DiagLogClientHandler
 *
 * PARAMETER:   tunmstr_DiagLogClientHandler* potunmstr_DiagLogClientHandler
 *
 * RETURNVALUE:  void
 *************************************************************************/
tVoid tunmstr_antdiag_i2cHandler::vSet_tunmstr_DiagLogClientHandler_Ptr(tunmstr_DiagLogClientHandler* potunmstr_DiagLogClientHandler)
{
	if( potunmstr_DiagLogClientHandler != NULL )
	{
		m_potunmstr_DiagLogClientHandler = ( tunmstr_DiagLogClientHandler* )potunmstr_DiagLogClientHandler;
	}
}

tVoid tunmstr_antdiag_i2cHandler::vSet_tunmstr_MsgFromHMI_Ptr(tunmstr_MsgFromHMI* potunmstr_MsgFromHMI )
{
	if( potunmstr_MsgFromHMI != NULL )
	{
		m_potunmstr_MsgFromHMI = ( tunmstr_MsgFromHMI* )potunmstr_MsgFromHMI;
	}

}
/*************************************************************************
 * FUNCTION:     vHandleTTFiS_CMD
 *
 * DESCRIPTION:  Fucntion to Handle the TTFI's Command
 *
 * PARAMETER:    const tU8* u8DataSpecial
 *
 * RETURNVALUE:  void
 *************************************************************************/
tVoid tunmstr_antdiag_i2cHandler::vHandleTTFiS_CMD(const tU8* u8DataSpecial)
{
	switch(u8DataSpecial[0])
	{
	case enReadI2CRegister:
	{
		trReadI2CReg((tenElmosRegAddress)u8DataSpecial[2],(tenCMExtPowerSupplyType)u8DataSpecial[1]);
		//ETG_TRACE_USR1(("vHandleTTFiS_CMD()u8DataSpecial[0]=%d u8DataSpecial[1]=%d u8Valu=%d",u8DataSpecial[0],u8DataSpecial[1]));
		break;
	}
	case enWriteI2CRegister:
	{
		//vWriteReg(u8DataSpecial[1],u8DataSpecial[2]);
		blWriteReg((tenElmosRegAddress)u8DataSpecial[2],u8DataSpecial[3],(tenCMExtPowerSupplyType)u8DataSpecial[1]);
		break;
	}
	default:
	{
	break;
	}
	}
}
tU8 tunmstr_antdiag_i2cHandler::u8GetICAddress(tenCMExtPowerSupplyType enCMExtPowerSupplyType)
{
	tU8 u8address =0x0;
	switch(enCMExtPowerSupplyType)
	{
		case enELMOS1_1:
		case enELMOS1_2: 	u8address= TUNMSTR_ANTDIAG_ELMOS1ADDRESS;
							break;
		case enELMOS2_1:
		case enELMOS2_2:	u8address= TUNMSTR_ANTDIAG_ELMOS2ADDRESS;
							break;
		case enELMOS3_1:
		case enELMOS3_2:
							u8address=	TUNMSTR_ANTDIAG_ELMOS3ADDRESS;
							break;
		case enMAX20084_1_1:
		case enMAX20084_1_2: 	u8address= TUNMSTR_ANTDIAG_MAX20084_1_ADDRESS;
							break;
		case enMAX20084_2_1:
		case enMAX20084_2_2:	u8address= TUNMSTR_ANTDIAG_MAX20084_2_ADDRESS;
							break;
		case enMAX20084_3_1:
		case enMAX20084_3_2:
							u8address=	TUNMSTR_ANTDIAG_MAX20084_3_ADDRESS;
							break;
		default:	break;
	}
	return u8address;
}
tVoid tunmstr_antdiag_i2cHandler::vInitialiseElmosRegisters(tenCMExtPowerSupplyType enCMExtPowerSupplyType)
{
	vDummyI2CHalfRead();
	blWriteReg((tU8)enA1_OverCurrentReg,TUNMSTR_ANTDIAG_U8MAX,enCMExtPowerSupplyType);
	blWriteReg((tU8)enA2_OverCurrentReg,TUNMSTR_ANTDIAG_U8MAX,enCMExtPowerSupplyType);
	blWriteReg((tU8)enErrorReg,0,enCMExtPowerSupplyType);
	blWriteELMOSStatusReg(enCMExtPowerSupplyType);
}
/* 1. Set output current limit, default value 270ma
 * 2. Set Over and open current register according to KDS
 * 3. Set warning current according to KDS
 * 4. Set output voltage according to KDS
 * 5. Set setup2 register with warning temp as 95 and blank time as 50ms
 * 6. set Mask register to assert FLT low when for reverse current, over current, short to battery i.e x001 0111
 */
tVoid tunmstr_antdiag_i2cHandler::vInitialiseMax20084Rg(tenCMExtPowerSupplyType enCMExtPowerSupplyType)
{
	tBool blDecrement = true;
	if(enCMExtPowerSupplyType == enMAX20084_1_1 || enCMExtPowerSupplyType == enMAX20084_2_1 || (enCMExtPowerSupplyType == enMAX20084_3_1))
	{
		blDecrement = false;
	}
	if(blDecrement)
	{
		blWriteReg((tU8)enMax20084ILim2Rg,u8GetMax20084ILimRegBitVal(TUNMSTR_ANTDIAG_MAX20084_LIMI_DEFAULT),enCMExtPowerSupplyType);
		blWriteReg((tU8)enMax20084ILim1Rg,u8GetMax20084ILimRegBitVal(TUNMSTR_ANTDIAG_MAX20084_LIMI_DEFAULT),(tenCMExtPowerSupplyType)(enCMExtPowerSupplyType-1));
		blWriteReg((tU8)enMax20084OIOL2Rg,u8GetMax20084OverCurNdOpenLoadCur(enCMExtPowerSupplyType),enCMExtPowerSupplyType);
		blWriteReg((tU8)enMax20084OIOL1Rg,u8GetMax20084OverCurNdOpenLoadCur((tenCMExtPowerSupplyType)(enCMExtPowerSupplyType-1)),(tenCMExtPowerSupplyType)(enCMExtPowerSupplyType-1));
		blWriteReg((tU8)enMax20084WarIRg,u8GetMax20084WarCur(enCMExtPowerSupplyType),enCMExtPowerSupplyType);
		blWriteReg((tU8)enMax20084WarIRg,u8GetMax20084WarCur(enCMExtPowerSupplyType),(tenCMExtPowerSupplyType)(enCMExtPowerSupplyType-1));
		blWriteReg((tU8)enMax20084OutV2Rg,u8GetMax20084OutVgValue(enCMExtPowerSupplyType),enCMExtPowerSupplyType);
		blWriteReg((tU8)enMax20084OutV1Rg,u8GetMax20084OutVgValue((tenCMExtPowerSupplyType)(enCMExtPowerSupplyType-1)),(tenCMExtPowerSupplyType)(enCMExtPowerSupplyType-1));
		blWriteReg((tU8)enMax20084SetUp2Rg,TUNMSTR_ANTDIAG_MAX20084_SETUPREG2_DEFAULT,enCMExtPowerSupplyType);
		blWriteReg((tU8)enMax20084MaskRg,TUNMSTR_ANTDIAG_MAX20084_MASKREG_DEFAULT,enCMExtPowerSupplyType);
	}
	else
	{
		blWriteReg((tU8)enMax20084ILim1Rg,u8GetMax20084ILimRegBitVal(TUNMSTR_ANTDIAG_MAX20084_LIMI_DEFAULT),enCMExtPowerSupplyType);
		blWriteReg((tU8)enMax20084ILim2Rg,u8GetMax20084ILimRegBitVal(TUNMSTR_ANTDIAG_MAX20084_LIMI_DEFAULT),(tenCMExtPowerSupplyType)(enCMExtPowerSupplyType+1));
		blWriteReg((tU8)enMax20084OIOL1Rg,u8GetMax20084OverCurNdOpenLoadCur(enCMExtPowerSupplyType),enCMExtPowerSupplyType);
		blWriteReg((tU8)enMax20084OIOL2Rg,u8GetMax20084OverCurNdOpenLoadCur((tenCMExtPowerSupplyType)(enCMExtPowerSupplyType+1)),(tenCMExtPowerSupplyType)(enCMExtPowerSupplyType+1));
		blWriteReg((tU8)enMax20084WarIRg,u8GetMax20084WarCur(enCMExtPowerSupplyType),enCMExtPowerSupplyType);
		blWriteReg((tU8)enMax20084WarIRg,u8GetMax20084WarCur(enCMExtPowerSupplyType),(tenCMExtPowerSupplyType)(enCMExtPowerSupplyType+1));
		blWriteReg((tU8)enMax20084OutV1Rg,u8GetMax20084OutVgValue(enCMExtPowerSupplyType),enCMExtPowerSupplyType);
		blWriteReg((tU8)enMax20084OutV2Rg,u8GetMax20084OutVgValue((tenCMExtPowerSupplyType)(enCMExtPowerSupplyType+1)),(tenCMExtPowerSupplyType)(enCMExtPowerSupplyType+1));
		blWriteReg((tU8)enMax20084SetUp2Rg,TUNMSTR_ANTDIAG_MAX20084_SETUPREG2_DEFAULT,enCMExtPowerSupplyType);
		blWriteReg((tU8)enMax20084MaskRg,TUNMSTR_ANTDIAG_MAX20084_MASKREG_DEFAULT,enCMExtPowerSupplyType);
	}
}
tU8 tunmstr_antdiag_i2cHandler::u8GetMax20084OverCurNdOpenLoadCur(tenCMExtPowerSupplyType enCMExtPowerSupplyType)
{
	if((m_potunmstr_antdiag_Debounce == NULL)|| (m_potunmstr_antdiag_Manager == NULL))
	{
		return 0;
	}
	tU8 u8OverNdOpenCur = u8GetMax20084OverIRegBitVal(m_potunmstr_antdiag_Debounce->u8GetShortCurrent(m_potunmstr_antdiag_Manager->enGetAntDiagClient(enCMExtPowerSupplyType)));
	u8OverNdOpenCur = (tU8)(u8OverNdOpenCur << 3);
	u8OverNdOpenCur = u8OverNdOpenCur | (u8GetMax20084OpenIRegBitVal(m_potunmstr_antdiag_Debounce->u8GetOpenCurrent(m_potunmstr_antdiag_Manager->enGetAntDiagClient(enCMExtPowerSupplyType))));
	return u8OverNdOpenCur;

}
tU8 tunmstr_antdiag_i2cHandler::u8GetMax20084WarCur(tenCMExtPowerSupplyType enCMExtPowerSupplyType)
{
	if((m_potunmstr_antdiag_Debounce == NULL)|| (m_potunmstr_antdiag_Manager == NULL))
	{
		return 0;
	}
	tU8 u8Cur = 0;
	if((enCMExtPowerSupplyType == enMAX20084_1_1) || (enCMExtPowerSupplyType == enMAX20084_2_1) || (enCMExtPowerSupplyType == enMAX20084_3_1))
	{
		u8Cur = u8GetMax20084WarIRegBitVal(m_potunmstr_antdiag_Debounce->u8GetWarningCurrent(m_potunmstr_antdiag_Manager->enGetAntDiagClient(enCMExtPowerSupplyType)));
		u8Cur = (tU8)(u8Cur << TUNMSTR_ANTDIAG_SHIFT_4BIT);
		u8Cur = u8Cur | u8GetMax20084WarIRegBitVal(m_potunmstr_antdiag_Debounce->u8GetWarningCurrent(m_potunmstr_antdiag_Manager->enGetAntDiagClient((tenCMExtPowerSupplyType)(enCMExtPowerSupplyType +1))));


	}
	else if((enCMExtPowerSupplyType == enMAX20084_1_2) || (enCMExtPowerSupplyType == enMAX20084_2_2) || (enCMExtPowerSupplyType == enMAX20084_3_2))
	{
		u8Cur = u8GetMax20084WarIRegBitVal(m_potunmstr_antdiag_Debounce->u8GetWarningCurrent(m_potunmstr_antdiag_Manager->enGetAntDiagClient((tenCMExtPowerSupplyType)(enCMExtPowerSupplyType -1))));
		u8Cur = (tU8)(u8Cur << TUNMSTR_ANTDIAG_SHIFT_4BIT);
		u8Cur = u8Cur |(u8GetMax20084WarIRegBitVal(m_potunmstr_antdiag_Debounce->u8GetWarningCurrent(m_potunmstr_antdiag_Manager->enGetAntDiagClient(enCMExtPowerSupplyType))));
	}	
	return u8Cur;
}
tU8 tunmstr_antdiag_i2cHandler::u8GetMax20084OutVg(tenCMExtPowerSupplyType enCMExtPowerSupplyType)
{
	if((ISPOWERSUPPLY_MAX20084(enCMExtPowerSupplyType)) && (m_potunmstr_antdiag_Debounce != NULL) && (m_potunmstr_antdiag_Manager !=NULL))
	{
		return u8GetMax20084OutVRegBitVal(
		((tFloat)(m_potunmstr_antdiag_Debounce->u8GetOutputVg(m_potunmstr_antdiag_Manager->enGetAntDiagClient(enCMExtPowerSupplyType))))/(tFloat)10.0);
	}
	return 0;
}
tU8 tunmstr_antdiag_i2cHandler::u8GetMax20084LimCur(tenCMExtPowerSupplyType enCMExtPowerSupplyType,tU16 u16Val)
{
	if(ISPOWERSUPPLY_MAX20084(enCMExtPowerSupplyType))
	{
		return u8GetMax20084ILimRegBitVal(u16Val);
	}
	return 0;
}
tBool tunmstr_antdiag_i2cHandler::blWriteReg(tU8 u8RegAdd,tU16 u16data,tenCMExtPowerSupplyType enCMExtPowerSupplyType)
{
	const char *dev = TUNMSTR_ANTDIAG_I2C_FILE_NAME;
	tU8 u8ICAdd = u8GetICAddress(enCMExtPowerSupplyType);
	//Open the i2c file to read the register
    tS8 s8Writefh = (tS8) open(dev, O_RDWR );
	if(s8Writefh < 0)
    {
		tInt errsv = errno;
		ETG_TRACE_ERR(("blWriteReg::Error in Opening I2C File with error:%d", errsv));
		ETG_TRACE_ERR(("blWriteReg: Error in Opening I2C File and error msg : %s\n", strerror(errsv)));
	    return false;
    }
	//the device address to communicate
    tS8 s8ioctlRet = (tS8)ioctl(s8Writefh, I2C_SLAVE, u8ICAdd);
    if(ISPOWERSUPPLY_ELMOS(enCMExtPowerSupplyType))
	{
    	ETG_TRACE_USR1(("blWriteReg-->s8ioctlRet=%d %02x Register=%02x u16data=%d",s8ioctlRet,
    		ETG_CENUM(tenCMExtPowerSupplyType,enCMExtPowerSupplyType),ETG_CENUM(tenElmosRegAddress,u8RegAdd),u16data));
	}
    else
    {
    	ETG_TRACE_USR1(("blWriteReg-->s8ioctlRet=%d %02x Register=%02x u16data=%d",s8ioctlRet,
    		ETG_CENUM(tenCMExtPowerSupplyType,enCMExtPowerSupplyType),ETG_CENUM(tenMax20084ICReg,u8RegAdd),u16data));
    }
	unsigned char writet[2];
	writet[0] = u8RegAdd;//Address of Register
	writet[1] = (unsigned char)u16data;//Data
	tS8 s8wrRet= (tS8) write(s8Writefh, &writet, TUNMSTR_ANTDIAG_2BYTE);
	if(s8wrRet!=TUNMSTR_ANTDIAG_2BYTE){
	int errsv = errno;
	ETG_TRACE_ERR(("blWriteReg: write() failed with error:%d and return value:%d", errsv, s8wrRet));
	ETG_TRACE_ERR(("blWriteReg: write() failed with error msg : %s\n", strerror(errsv)));
	ETG_TRACE_ERR(("blWriteReg-->Error in Writing Register s8wrRet=%d Register=%d, u16data=%d",s8wrRet,u8RegAdd,u16data));
		(void)close(s8Writefh);
		return false;
	}
	else
	{
		(void)close(s8Writefh);
		return true;
	}
	
}

/*************************************************************************
 * FUNCTION:     vWriteOverCurrentReg
 *
 * DESCRIPTION:  Function to update the ELMOS current register
 *
 * PARAMETER:    tenExtAntSupType enExtAntSupType
 *
 * RETURNVALUE:  void
 *************************************************************************/
tVoid tunmstr_antdiag_i2cHandler::vWriteOverCurrentReg(tenCMExtPowerSupplyType enCMExtPowerSupplyType,tU16 u16data)
{
	tFloat fData=u16data;
	tenElmosRegAddress enElmosRegAddress=NoneReg;
	if((enELMOS2_1==enCMExtPowerSupplyType)||(enELMOS1_1==enCMExtPowerSupplyType)|| (enELMOS3_1==enCMExtPowerSupplyType)||(enELMOS4_1==enCMExtPowerSupplyType))
	{
		enElmosRegAddress=enA1_OverCurrentReg;
	}
	else if((enELMOS2_2==enCMExtPowerSupplyType)||(enELMOS1_2==enCMExtPowerSupplyType)|| (enELMOS3_2==enCMExtPowerSupplyType)||(enELMOS4_2==enCMExtPowerSupplyType))
	{
		enElmosRegAddress=enA2_OverCurrentReg;
	}
	if(TUNMSTR_ANTDIAG_U8MAX!=u16data)
	{
		fData= (tFloat)fData/(tFloat)TUNMSTR_ANTDIAG_ELMOS_HIGHCURRENT_REG_MUL_FAC;
		u16data=(tU16)fData;
	}
	if(NoneReg!=enElmosRegAddress)
	{
		blWriteReg((tU8)enElmosRegAddress,u16data,enCMExtPowerSupplyType);
	}
}
/*Bugfix for NCG3D-28640*/
#ifdef VARIANT_S_FTR_ENABLE_FEATURE_RNAIVI
tVoid tunmstr_antdiag_i2cHandler::vWriteSupplyOverVoltageReg(tenCMExtPowerSupplyType enCMExtPowerSupplyType)
{
	blWriteReg((tU8)enSUP_OverVoltageReg,0xff,enCMExtPowerSupplyType);

}
#endif
/*************************************************************************
 * FUNCTION:     u8ReadELMOSReg
 *
 * DESCRIPTION:  Function to read the elmos register content
 *
 * PARAMETER:    elmosRegister enRegister,tenExtAntSupType enExtAntSupType
 *
 * RETURNVALUE:  void
 *************************************************************************/
trI2CRead tunmstr_antdiag_i2cHandler::trReadI2CReg(tU8 u8Register,tenCMExtPowerSupplyType enCMExtPowerSupplyType)
{
	trI2CRead otrI2CRead;
	const char *dev = TUNMSTR_ANTDIAG_I2C_FILE_NAME;
	tU8 u8ICAdd = u8GetICAddress(enCMExtPowerSupplyType);
	tS8 s8i2cFileHandler = (tS8) open(dev, O_RDWR );
	if(s8i2cFileHandler < 0)
	{
		tInt errsv = errno;
		ETG_TRACE_ERR(("trReadI2CReg::Error in Opening I2C File with error:%d", errsv));
		ETG_TRACE_ERR(("trReadI2CReg: Error in Opening I2C File and error msg : %s\n", strerror(errsv)));
		otrI2CRead.blReadSucess = false;
		return otrI2CRead;
	}
	//the device address to communicate
	tS8 s8ioctlRet = (tS8) ioctl(s8i2cFileHandler, I2C_SLAVE, u8ICAdd);
	tU8 readbuf[1]={0};
	tU8 writebuf[1];
	struct i2c_msg msgs[4];

	msgs[0].addr=u8ICAdd;
	msgs[0].flags=0x0;
	msgs[0].len=1;
	msgs[0].buf=writebuf;
	msgs[0].buf[0]=u8Register;

	msgs[1].addr=u8ICAdd;
	msgs[1].flags=0x0 | I2C_M_RD;
	msgs[1].len=1;
	msgs[1].buf=readbuf;

	struct i2c_rdwr_ioctl_data data;
	data.msgs=msgs;
	data.nmsgs=2;
	if (ioctl(s8i2cFileHandler, I2C_RDWR, &data) < 0) 
	{
		ETG_TRACE_ERR(("trReadI2CReg::Error in ioctl with"));
		int errsv = errno;
		ETG_TRACE_ERR(("trReadI2CReg: ioctl() failed with error:%d and return value", errsv));
		ETG_TRACE_ERR(("trReadI2CReg: ioctl() failed with error msg : %s\n", strerror(errsv)));
		otrI2CRead.blReadSucess = false;
		(void)close (s8i2cFileHandler);
		return otrI2CRead;
	}
    if(ISPOWERSUPPLY_ELMOS(enCMExtPowerSupplyType))
	{
    	ETG_TRACE_USR4(("trReadI2CReg::ELMOS enExtAntSupType=%02x Register=%02x s8ioctlRet=%d readbuf[0]=%d ",
    	    		ETG_CENUM(tenCMExtPowerSupplyType,enCMExtPowerSupplyType),ETG_CENUM(tenElmosRegAddress,u8Register),s8ioctlRet,readbuf[0]));
	}
    else
    {
    	ETG_TRACE_USR4(("trReadI2CReg::MAX20084 enExtAntSupType=%02x Register=%02x s8ioctlRet=%d readbuf[0]=%d ",
    		ETG_CENUM(tenCMExtPowerSupplyType,enCMExtPowerSupplyType),ETG_CENUM(tenMax20084ICReg,u8Register),s8ioctlRet,readbuf[0]));
    }

		otrI2CRead.blReadSucess =true;
		otrI2CRead.u8Value = readbuf[0];
		(void)close (s8i2cFileHandler);
		return otrI2CRead;
}
/*************************************************************************
 * FUNCTION:     vReadELMOSCurrent
 *************************************************************************/
tVoid tunmstr_antdiag_i2cHandler::vReadELMOSCurrent(
    tenCMExtPowerSupplyType enCMExtPowerSupplyType,tenAntDiagClients enAntDiagClients)
{
  if((m_potunmstr_antdiag_Debounce==NULL)||(m_potunmstr_DiagLogClientHandler==NULL))
  {
	  return;
  }
  if(!IsOutputPinIsEnabled(enCMExtPowerSupplyType))
   {
	   ETG_TRACE_USR4(("vReadELMOSCurrent()-> enCMExtPowerSupplyType=%02x is off",
			ETG_CENUM(tenCMExtPowerSupplyType,enCMExtPowerSupplyType)));
		if((enAntennState_Short==m_potunmstr_DiagLogClientHandler->u8GetAntenanstate(enAntDiagClients)) &&
			(m_potunmstr_antdiag_Debounce->IsShortCktTimeOver(enAntDiagClients)))
		{
			ETG_TRACE_USR4(("vReadELMOSCurrent()-> enCMExtPowerSupplyType=%02x Short circuit wait over time finished",
				ETG_CENUM(tenCMExtPowerSupplyType,enCMExtPowerSupplyType)));
			//Enable the corresponding Output pin of ELMOS
			vSetOutputEnabled(enCMExtPowerSupplyType,TRUE);
			//Update the status register of ELMOS
			if(blEnableOutput(enCMExtPowerSupplyType))
			{
				m_potunmstr_antdiag_Debounce->vClearShortWaitOverCounter(enCMExtPowerSupplyType);
				m_potunmstr_DiagLogClientHandler->blUpdateAntennaStatus(enAntDiagClients,TUNMSTR_ANTDIAG_ANTENNA_STATE_NUMBER_OF_ITEM);
			}
			else
			{
				//Update the status register of ELMOS
				vSetOutputEnabled(enCMExtPowerSupplyType,false);
			}
		}
	}
  else
  {
	  ETG_TRACE_USR4(("vReadELMOSCurrent()-> enCMExtPowerSupplyType=%02x is on",
			ETG_CENUM(tenCMExtPowerSupplyType,enCMExtPowerSupplyType)));
    if(!IsShortCktError(enCMExtPowerSupplyType))
    {
      vReadELMOSCurrentReg(enCMExtPowerSupplyType);
    }
  }
}
/*************************************************************************
 * FUNCTION:     vReadELMOSCurrentReg
 *************************************************************************/
tVoid tunmstr_antdiag_i2cHandler::vReadELMOSCurrentReg( tenCMExtPowerSupplyType enCMExtPowerSupplyType)
{
	switch (enCMExtPowerSupplyType)
		{
		case enELMOS1_1:
        case enELMOS2_1:
		case enELMOS3_1:
		case enELMOS4_1:
		{
			ETG_TRACE_USR1(("vReadELMOSCurrentReg()--> %02x Read A1 current",ETG_CENUM(tenCMExtPowerSupplyType,enCMExtPowerSupplyType)));
			vReadELMOSA1CurrentReg(enCMExtPowerSupplyType);
			break;
		}
		case enELMOS1_2:
        case enELMOS2_2:
		case enELMOS3_2:
		case enELMOS4_2:
		{
			ETG_TRACE_USR1(("vReadELMOSCurrentReg()-->%02x Read A2 current",ETG_CENUM(tenCMExtPowerSupplyType,enCMExtPowerSupplyType)));
			vReadELMOSA2CurrentReg(enCMExtPowerSupplyType);
			break;
		}
		default:
		{
			break;
		}
		}
}
/*************************************************************************
 * FUNCTION:     fReadCurrent
 *
 * DESCRIPTION:  Function to calculate ELMOS current
 *
 * PARAMETER:    tenExtAntSupType enExtAntSupType
 *
 * RETURNVALUE:  float
 *************************************************************************/
trI2CReadCurrent tunmstr_antdiag_i2cHandler::trReadELMOSCurrent(tenCMExtPowerSupplyType enCMExtPowerSupplyType)
{
	tU8 enRegToRead =NoneReg;
	trI2CReadCurrent otrI2CReadCurrent;
	if((enELMOS2_1==enCMExtPowerSupplyType)||(enELMOS1_1==enCMExtPowerSupplyType)|| (enELMOS3_1==enCMExtPowerSupplyType) || (enELMOS4_1==enCMExtPowerSupplyType) )
	{
		enRegToRead=enA1_LowCurrentReg;
	}
	else if((enELMOS2_2==enCMExtPowerSupplyType)||(enELMOS1_2==enCMExtPowerSupplyType) || (enELMOS3_2==enCMExtPowerSupplyType)|| (enELMOS4_2==enCMExtPowerSupplyType))
	{
		enRegToRead=enA2_LowCurrentReg;
	}
	trI2CRead otrI2CRead = trReadI2CReg((tenElmosRegAddress)enRegToRead,enCMExtPowerSupplyType);
	if(otrI2CRead.blReadSucess)
	{
		otrI2CReadCurrent.fValue = otrI2CRead.u8Value;
		if(otrI2CReadCurrent.fValue == TUNMSTR_ANTDIAG_U8MAX_FLOAT)
		{
			trI2CRead otrI2CRead = trReadI2CReg((tenElmosRegAddress)(enRegToRead+1),enCMExtPowerSupplyType);
			if(otrI2CRead.blReadSucess)
			{
				if(otrI2CRead.u8Value==TUNMSTR_ANTDIAG_U8MAX)
				otrI2CReadCurrent.fValue=TUNMSTR_ANTDIAG_U8MAX_FLOAT+1;
				else
				{
					otrI2CReadCurrent.fValue = otrI2CRead.u8Value;
					otrI2CReadCurrent.fValue= otrI2CReadCurrent.fValue*TUNMSTR_ANTDIAG_ELMOS_HIGHCURRENT_REG_MUL_FAC;
				}
			}
			else
			{
				otrI2CReadCurrent.blReadSucess =false;
			}
		}
		else
		{
			otrI2CReadCurrent.fValue=otrI2CReadCurrent.fValue*TUNMSTR_ANTDIAG_ELMOS_LOWCURRENT_REG_MUL_FAC;
		}
	}
	else
	{
		otrI2CReadCurrent.blReadSucess =false;
	}
	return otrI2CReadCurrent;
}
/*************************************************************************
 * FUNCTION:     vReadELMOSA1CurrentReg
 *
 * DESCRIPTION:  Function to read ELMOS A1 Antenna current
 * PARAMETER:    tenCMExtPowerSupplyType enCMExtPowerSupplyType
 *
 * RETURNVALUE:  tVoid
 *************************************************************************/
tVoid tunmstr_antdiag_i2cHandler::vReadELMOSA1CurrentReg(tenCMExtPowerSupplyType enCMExtPowerSupplyType)
{
	if(m_potunmstr_antdiag_Measure==NULL || m_potunmstr_antdiag_Manager == NULL)
	{
		return;
	}
	ETG_TRACE_USR1(("vReadELMOSA1CurrentReg()-->%02x Reading A1 current",ETG_CENUM(tenCMExtPowerSupplyType,enCMExtPowerSupplyType)));
	tenAntDiagClients enAntDiagClient = m_potunmstr_antdiag_Manager->enGetAntDiagClient(enCMExtPowerSupplyType);
	trI2CRead otrI2CRead = trReadI2CReg(enA1_VoltageReg,enCMExtPowerSupplyType);
	if(!otrI2CRead.blReadSucess)
	{
		m_potunmstr_DiagLogClientHandler->vHandleErrorLog(enAntDiagClient,enAntennState_Unknown,0.0);
		if(m_potunmstr_antdiag_Manager->u8GetPendingAntennaStateReqClient() == enAntDiagClient)
		{
			m_potunmstr_MsgFromHMI->vSendRfAntenna_TestMethodResult(enAntDiagClient,enAntennState_Unknown,0.0);
			m_potunmstr_antdiag_Manager->vSetPendingAntennaStateReqClient(enClientNone);
		}
		return;
	}
	tFloat fA1voltage= otrI2CRead.u8Value;
	fA1voltage=fA1voltage*TUNMSTR_ANTDIAG_ELMOS_OUTPUT_VG_MUL_FAC;
	//Check voltage error
	ETG_TRACE_USR1(("vReadELMOSA1CurrentReg()-->%02x fA1voltage=%f",ETG_CENUM(tenCMExtPowerSupplyType,enCMExtPowerSupplyType),fA1voltage));
    tFloat fCur=0.0;
	if(fA1voltage>=TUNMSTR_ANTDIAG_ELMOS_SHORTCKTOUTVG)
	{
		trI2CReadCurrent otrI2CReadCurrent = trReadELMOSCurrent(enCMExtPowerSupplyType);
		if(!otrI2CReadCurrent.blReadSucess)
		{
			m_potunmstr_DiagLogClientHandler->vHandleErrorLog(enAntDiagClient,enAntennState_Unknown,0.0);
			if(m_potunmstr_antdiag_Manager->u8GetPendingAntennaStateReqClient() == enAntDiagClient)
			{
				m_potunmstr_MsgFromHMI->vSendRfAntenna_TestMethodResult(enAntDiagClient,enAntennState_Unknown,0.0);
				m_potunmstr_antdiag_Manager->vSetPendingAntennaStateReqClient(enClientNone);
			}
			return;
		}
		fCur = otrI2CReadCurrent.fValue;
	}
	else if(fA1voltage!=0.0)
	{
		fCur=TUNMSTR_ANTDIAG_U8MAX_FLOAT+1.0;
		
	}
    m_potunmstr_antdiag_Measure->vUpdateAntennaCur(enCMExtPowerSupplyType,fCur);
}
/*************************************************************************
 * FUNCTION:     vReadELMOSA2CurrentReg
 *
 * DESCRIPTION:  Function to read ELMOS A2 Antenna current
 * PARAMETER:    tenCMExtPowerSupplyType enCMExtPowerSupplyType
 *
 * RETURNVALUE:  tVoid
 *************************************************************************/
tVoid tunmstr_antdiag_i2cHandler::vReadELMOSA2CurrentReg(tenCMExtPowerSupplyType enCMExtPowerSupplyType)
{
	if(m_potunmstr_antdiag_Measure==NULL || m_potunmstr_antdiag_Manager == NULL)
	{
		return;
	}
	ETG_TRACE_USR1(("vReadELMOSA2CurrentReg()-->%02x Reading A2 current",ETG_CENUM(tenCMExtPowerSupplyType,enCMExtPowerSupplyType)));
	trI2CRead otrI2CRead = trReadI2CReg(enA2_VoltageReg,enCMExtPowerSupplyType);
	tenAntDiagClients enAntDiagClient = m_potunmstr_antdiag_Manager->enGetAntDiagClient(enCMExtPowerSupplyType);
	if(!otrI2CRead.blReadSucess)
	{
		m_potunmstr_DiagLogClientHandler->vHandleErrorLog(enAntDiagClient,enAntennState_Unknown,0.0);
		if(m_potunmstr_antdiag_Manager->u8GetPendingAntennaStateReqClient() == enAntDiagClient)
		{
			m_potunmstr_MsgFromHMI->vSendRfAntenna_TestMethodResult(enAntDiagClient,enAntennState_Unknown,0.0);
			m_potunmstr_antdiag_Manager->vSetPendingAntennaStateReqClient(enClientNone);
		}
		return;
	}
	tFloat fA2voltage= otrI2CRead.u8Value;
	fA2voltage=fA2voltage*TUNMSTR_ANTDIAG_ELMOS_OUTPUT_VG_MUL_FAC;
	//Check voltage error
	ETG_TRACE_USR1(("vReadELMOSA2CurrentReg()-->%02x vReadELMOSA2CurrentReg()fA2voltage=%f",ETG_CENUM(tenCMExtPowerSupplyType,enCMExtPowerSupplyType),fA2voltage));
    tFloat fCur=0.0;
	if(fA2voltage>=TUNMSTR_ANTDIAG_ELMOS_SHORTCKTOUTVG)
	{
		trI2CReadCurrent otrI2CReadCurrent = trReadELMOSCurrent(enCMExtPowerSupplyType);
		if(!otrI2CReadCurrent.blReadSucess)
		{
			m_potunmstr_DiagLogClientHandler->vHandleErrorLog(enAntDiagClient,enAntennState_Unknown,0.0);
			if(m_potunmstr_antdiag_Manager->u8GetPendingAntennaStateReqClient() == enAntDiagClient)
			{
				m_potunmstr_MsgFromHMI->vSendRfAntenna_TestMethodResult(enAntDiagClient,enAntennState_Unknown,0.0);
				m_potunmstr_antdiag_Manager->vSetPendingAntennaStateReqClient(enClientNone);
			}
			return;
		}
		fCur = otrI2CReadCurrent.fValue;
	}
	else if(fA2voltage!=0.0)
	{
        fCur=TUNMSTR_ANTDIAG_U8MAX_FLOAT+1.0;
	}
    m_potunmstr_antdiag_Measure->vUpdateAntennaCur(enCMExtPowerSupplyType,fCur);
}
tVoid tunmstr_antdiag_i2cHandler::vReadMax20084Current(
    tenCMExtPowerSupplyType enCMExtPowerSupplyType,tenAntDiagClients enAntDiagClients)
{
  if((m_potunmstr_antdiag_Debounce==NULL)||(m_potunmstr_DiagLogClientHandler==NULL))
  {
	  return;
  }
  if(!IsOutputPinIsEnabled(enCMExtPowerSupplyType))
   {
	   ETG_TRACE_USR4(("vReadMax20084Current()-> enCMExtPowerSupplyType=%02x is off",
			ETG_CENUM(tenCMExtPowerSupplyType,enCMExtPowerSupplyType)));
		if((enAntennState_Short==m_potunmstr_DiagLogClientHandler->u8GetAntenanstate(enAntDiagClients)) &&
			(m_potunmstr_antdiag_Debounce->IsShortCktTimeOver(enAntDiagClients)))
		{
			ETG_TRACE_USR4(("vReadMax20084Current()-> enCMExtPowerSupplyType=%02x Short circuit wait over time finished",
				ETG_CENUM(tenCMExtPowerSupplyType,enCMExtPowerSupplyType)));
			//Enable the corresponding Output pin of max ic
			vSetOutputEnabled(enCMExtPowerSupplyType,TRUE);
			//Update the setup1 register of max ic
			if(blEnableOutput(enCMExtPowerSupplyType))
			{
				m_potunmstr_antdiag_Debounce->vClearShortWaitOverCounter(enCMExtPowerSupplyType);
				m_potunmstr_DiagLogClientHandler->blUpdateAntennaStatus(enAntDiagClients,TUNMSTR_ANTDIAG_ANTENNA_STATE_NUMBER_OF_ITEM);
			}
			else
			{
				//disable the corresponding Output pin of max ic
				vSetOutputEnabled(enCMExtPowerSupplyType,false);
			}
		}
	}
  else
  {
	  ETG_TRACE_USR4(("vReadMax20084Current()-> enCMExtPowerSupplyType=%02x is on",
			ETG_CENUM(tenCMExtPowerSupplyType,enCMExtPowerSupplyType)));
    if(!IsShortCktError(enCMExtPowerSupplyType))
    {
      vReadMax20084OutPutIReg(enCMExtPowerSupplyType);
    }
  }
}
tVoid tunmstr_antdiag_i2cHandler::vReadMax20084OutPutIReg(tenCMExtPowerSupplyType enCMExtPowerSupplyType)
{
	if((NULL == m_potunmstr_DiagLogClientHandler) || (NULL == m_potunmstr_antdiag_Manager) || (NULL == m_potunmstr_MsgFromHMI) ||
			(NULL ==m_potunmstr_antdiag_Measure) || (!ISPOWERSUPPLY_MAX20084(enCMExtPowerSupplyType)))
	{
		return;
	}
	tFloat fCur = 0;
	tU8 u8RegToRead = enMax20084OutI1Rg;
	if((enMAX20084_1_2==enCMExtPowerSupplyType)||(enMAX20084_2_2==enCMExtPowerSupplyType)|| (enMAX20084_3_2==enCMExtPowerSupplyType))
	{
		u8RegToRead = enMax20084OutI2Rg;
	}
	trI2CRead otrI2CReadCurrent = trReadI2CReg(u8RegToRead,enCMExtPowerSupplyType);
	if(!otrI2CReadCurrent.blReadSucess)
	{
		tenAntDiagClients enAntDiagClient = m_potunmstr_antdiag_Manager->enGetAntDiagClient(enCMExtPowerSupplyType);
		m_potunmstr_DiagLogClientHandler->vHandleErrorLog(enAntDiagClient,enAntennState_Unknown,0.0);
		if(m_potunmstr_antdiag_Manager->u8GetPendingAntennaStateReqClient() == enAntDiagClient)
		{
			m_potunmstr_MsgFromHMI->vSendRfAntenna_TestMethodResult(enAntDiagClient,enAntennState_Unknown,0.0);
			m_potunmstr_antdiag_Manager->vSetPendingAntennaStateReqClient(enClientNone);
		}
		return;
	}
	fCur = (tFloat)otrI2CReadCurrent.u8Value;
	m_potunmstr_antdiag_Measure->vUpdateAntennaCur(enCMExtPowerSupplyType,fCur);
}
tBool tunmstr_antdiag_i2cHandler::IsOutputPinIsEnabled(tenCMExtPowerSupplyType enCMExtPowerSupplyType)
{
	switch(enCMExtPowerSupplyType)
    {
        case enELMOS1_1:
        case enELMOS2_1:
		case enELMOS3_1:
		case enELMOS4_1:
         {
            return m_blOutputA1Enabled[enGetELMOSIC_Index(enCMExtPowerSupplyType)];
            break;
         }
        case enELMOS1_2:
        case enELMOS2_2:
		case enELMOS3_2:
		case enELMOS4_2:
         {
            return m_blOutputA2Enabled[enGetELMOSIC_Index(enCMExtPowerSupplyType)];
            break;
         }
        case enMAX20084_1_1:
        case enMAX20084_2_1:
		case enMAX20084_3_1:
         {
            return m_blMax20084OutputA1En[enGetMax20084IC_Index(enCMExtPowerSupplyType)];
            break;
         }
        case enMAX20084_1_2:
        case enMAX20084_2_2:
		case enMAX20084_3_2:
         {
            return m_blMax20084OutputA2En[enGetMax20084IC_Index(enCMExtPowerSupplyType)];
            break;
         }
        default:
            {
            break;
            }
    }
	return FALSE;
}
tVoid tunmstr_antdiag_i2cHandler::vSetOutputEnabled(tenCMExtPowerSupplyType enCMExtPowerSupplyType,tBool blValue)
{
	switch(enCMExtPowerSupplyType)
    {
        case enELMOS1_1:
        case enELMOS2_1:
        case enELMOS3_1:
        case enELMOS4_1:
         {
            m_blOutputA1Enabled[enGetELMOSIC_Index(enCMExtPowerSupplyType)]=blValue;
            break;
         }
        case enELMOS1_2:
        case enELMOS2_2:
        case enELMOS3_2:
        case enELMOS4_2:
         {
            m_blOutputA2Enabled[enGetELMOSIC_Index(enCMExtPowerSupplyType)]=blValue;
            break;
         }
		case enMAX20084_1_1:
		case enMAX20084_2_1:
		case enMAX20084_3_1:
         {
        	 m_blMax20084OutputA1En[enGetMax20084IC_Index(enCMExtPowerSupplyType)]=blValue;
            break;
         }
        case enMAX20084_1_2:
        case enMAX20084_2_2:
		case enMAX20084_3_2:
         {
        	 m_blMax20084OutputA2En[enGetMax20084IC_Index(enCMExtPowerSupplyType)]=blValue;
            break;
         }
        default:
            {
            break;
            }
    }
	ETG_TRACE_USR1(("vSetOutputEnabled(): %02x ELMOSA1Pin=%d, ELMOSA2Pin=%d MAXA1Pin=%d, MAXA2Pin=%d Value=%d",
	ETG_CENUM(tenCMExtPowerSupplyType,enCMExtPowerSupplyType),
	m_blOutputA1Enabled[enGetELMOSIC_Index(enCMExtPowerSupplyType)], m_blOutputA2Enabled[enGetELMOSIC_Index(enCMExtPowerSupplyType)],
	m_blMax20084OutputA1En[enGetMax20084IC_Index(enCMExtPowerSupplyType)], m_blMax20084OutputA2En[enGetMax20084IC_Index(enCMExtPowerSupplyType)],blValue));
}
/*************************************************************************
 * FUNCTION:     enGetELMOSIC_Index
 *
 * DESCRIPTION:  Function to get ELMOS1 or ELMOS2 index for A1 and A2 pins

 * PARAMETER:    tenExtAntSupType enExtAntSupType
 *
 * RETURNVALUE:  tenELMOSPowerICIndex
 ************************************************************************/
tenELMOSPowerICIndex tunmstr_antdiag_i2cHandler::enGetELMOSIC_Index(tenCMExtPowerSupplyType enCMExtPowerSupplyType)
{
	tenELMOSPowerICIndex enIndex=enELMOS1_IcIndex;
	switch(enCMExtPowerSupplyType)
	{
		case enELMOS1_1:
		case enELMOS1_2:
			enIndex=enELMOS1_IcIndex;
			break;
		case enELMOS2_1:
		case enELMOS2_2:
			enIndex=enELMOS2_IcIndex;
			break;
		case enELMOS3_1:
		case enELMOS3_2:
			enIndex=enELMOS3_IcIndex;
			break;
		case enELMOS4_1:
		case enELMOS4_2:
			enIndex=enELMOS4_IcIndex;
			break;
		default:
			break;
	}
	ETG_TRACE_USR1(("enGetELMOSIC_Index(): enIndex=%d",enIndex));
	return enIndex;
}
tenMAX20084IcIndex tunmstr_antdiag_i2cHandler::enGetMax20084IC_Index(tenCMExtPowerSupplyType enCMExtPowerSupplyType)
{
	tenMAX20084IcIndex enIndex=enMAX20084_1_IcIndex;
	switch(enCMExtPowerSupplyType)
	{
		case enMAX20084_1_1:
		case enMAX20084_1_2:
			enIndex=enMAX20084_1_IcIndex;
			break;
		case enMAX20084_2_1:
		case enMAX20084_2_2:
			enIndex=enMAX20084_2_IcIndex;
			break;
		case enMAX20084_3_1:
		case enMAX20084_3_2:
			enIndex=enMAX20084_3_IcIndex;
			break;
		default:
			break;
	}
	return enIndex;
}
tBool tunmstr_antdiag_i2cHandler::blEnableOutput(tenCMExtPowerSupplyType enCMExtPowerSupplyType)
{
	if(ISPOWERSUPPLY_ELMOS(enCMExtPowerSupplyType))
	{
		return blWriteELMOSStatusReg(enCMExtPowerSupplyType);
	}
	else if(ISPOWERSUPPLY_MAX20084(enCMExtPowerSupplyType))
	{
		return blEnableMAX20084Output(enCMExtPowerSupplyType);
	}
	return false;
}
tenMax20084ICReg tunmstr_antdiag_i2cHandler::enGetMax20084IC_OutVgRegAddr(tenCMExtPowerSupplyType enCMExtPowerSupplyType)
{
	tenMax20084ICReg enOutVgRegAddr = enMax2008RgNone;
	switch(enCMExtPowerSupplyType)
	{
		case enMAX20084_1_1:
		case enMAX20084_2_1:
		case enMAX20084_3_1:
			enOutVgRegAddr = enMax20084OutV1Rg;
			break;
		case enMAX20084_1_2:
		case enMAX20084_2_2:
		case enMAX20084_3_2:
			enOutVgRegAddr = enMax20084OutV2Rg;
			break;
		default:
			break;
	}
	return enOutVgRegAddr;
}
tU8 tunmstr_antdiag_i2cHandler::u8GetMax20084OutVgValue(tenCMExtPowerSupplyType enCMExtPowerSupplyType, tBool blTurnOn)const
{
	if((m_potunmstr_antdiag_Manager == NULL) || (m_potunmstr_antdiag_Debounce == NULL) || (!blTurnOn))
	{
		return 0;
	}
	ETG_TRACE_USR4(("u8GetMax20084OutVgValue() =%d",u8GetMax20084OutVRegBitVal((tFloat)(m_potunmstr_antdiag_Debounce->u8GetOutputVg(m_potunmstr_antdiag_Manager->enGetAntDiagClient(enCMExtPowerSupplyType)))/(tFloat)10.0)));
	return (u8GetMax20084OutVRegBitVal((tFloat)(m_potunmstr_antdiag_Debounce->u8GetOutputVg(m_potunmstr_antdiag_Manager->enGetAntDiagClient(enCMExtPowerSupplyType)))/(tFloat)10.0));
}
/*************************************************************************
 * FUNCTION:     vWriteELMOSStatusReg
 *
 * DESCRIPTION:  Write into ELMOS status reg

 * PARAMETER:    tenExtAntSupType enExtAntSupType
 *
 * RETURNVALUE:  void
 *************************************************************************/
tBool tunmstr_antdiag_i2cHandler::blWriteELMOSStatusReg(tenCMExtPowerSupplyType enCMExtPowerSupplyType)
{
	tU8 u8Index=enGetELMOSIC_Index(enCMExtPowerSupplyType);
	ETG_TRACE_USR4(("blWriteELMOSStatusReg() m_blOutputA1Enabled=%d m_blOutputA2Enabled=%d",m_blOutputA1Enabled[u8Index],m_blOutputA2Enabled[u8Index]));
	tBool blRet = false;
	tBool blSendAntennaStatus = false;
	if(ISPOWERSUPPLY_ELMOS(enCMExtPowerSupplyType))
	{
		if((!m_blOutputA1Enabled[u8Index]) && (!m_blOutputA2Enabled[u8Index]))
		{
			ETG_TRACE_USR1(("blWriteELMOSStatusReg()-->Both ELMOS clients at A1 and A2 are disabled"));	
			blRet = blWriteReg((tU8)enStatusReg, enDisableBothA1AndA2,enCMExtPowerSupplyType); //Both disabled
			blSendAntennaStatus = true;
		}
		else if((!m_blOutputA1Enabled[u8Index]) && (m_blOutputA2Enabled[u8Index]) )
		{
			ETG_TRACE_USR1(("blWriteELMOSStatusReg()-->ELMOS client at A1 is disabled"));
			blRet =  blWriteReg((tU8)enStatusReg, enEnableOnlyA2,enCMExtPowerSupplyType);  //Only A1 disabled
			blSendAntennaStatus = true;
		}

		if ((m_blOutputA1Enabled[u8Index]) && (m_blOutputA2Enabled[u8Index]))
		{
			ETG_TRACE_USR1(("blWriteELMOSStatusReg()-->Both ELMOS clients at A1 and A2 are enabled"));
			blRet =  blWriteReg((tU8)enStatusReg, enEnableBothA1AndA2,enCMExtPowerSupplyType); //Both A1 and A2 enabled
			blSendAntennaStatus = true;
		}

		else if ((m_blOutputA1Enabled[u8Index]) && (!m_blOutputA2Enabled[u8Index]))
		{
			ETG_TRACE_USR1(("blWriteELMOSStatusReg()-->ELMOS client at A2 is disabled"));
			blRet =  blWriteReg((tU8)enStatusReg, enEnableOnlyA1,enCMExtPowerSupplyType); //Only A2 disabled
			blSendAntennaStatus = true;
		}
		ETG_TRACE_USR1(("blWriteELMOSStatusReg()--> for %02x: m_blOutputA1Enabled=%d, m_blOutputA2Enabled=%d",
			ETG_CENUM(tenCMExtPowerSupplyType,enCMExtPowerSupplyType),m_blOutputA1Enabled[u8Index],m_blOutputA2Enabled[u8Index]));
	}
	if((!blRet) && (blSendAntennaStatus) && (NULL != m_potunmstr_antdiag_Manager))
	{
		m_potunmstr_antdiag_Manager->vUpdateAntenaaStatetoHMI(m_potunmstr_antdiag_Manager->enGetAntDiagClient(enCMExtPowerSupplyType),enAntennState_Unknown, 0.0);
	}
	return blRet;
}
tBool tunmstr_antdiag_i2cHandler::blEnableMAX20084Output(tenCMExtPowerSupplyType enCMExtPowerSupplyType)
{

	tBool blRet = false;
	tBool blSendAntennaStatus = false;
	if(ISPOWERSUPPLY_MAX20084(enCMExtPowerSupplyType))
	{
		tU8 u8Index=enGetMax20084IC_Index(enCMExtPowerSupplyType);
		ETG_TRACE_USR4(("blEnableMAX20084Output() m_blMax20084OutputA1En=%d m_blMax20084OutputA2En=%d",m_blMax20084OutputA1En[u8Index],m_blMax20084OutputA2En[u8Index]));
		if((!m_blMax20084OutputA1En[u8Index]) && (!m_blMax20084OutputA2En[u8Index]))
		{
			ETG_TRACE_USR1(("blEnableMAX20084Output()-->Both MAX20084 clients at A1 and A2 are disabled"));
			blRet = blWriteReg((tU8)enMax20084SetUp1Rg, enMax20084DisableBothA1AndA2,enCMExtPowerSupplyType); //Both disabled
			blSendAntennaStatus = true;
		}
		else if((!m_blMax20084OutputA1En[u8Index]) && (m_blMax20084OutputA2En[u8Index]) )
		{
			ETG_TRACE_USR1(("blEnableMAX20084Output()-->MAX20084 client at A1 is disabled"));
			blRet =  blWriteReg((tU8)enMax20084SetUp1Rg, enMax20084EnableOnlyA2,enCMExtPowerSupplyType);  //Only A1 disabled
			blSendAntennaStatus = true;
		}

		if ((m_blMax20084OutputA1En[u8Index]) && (m_blMax20084OutputA2En[u8Index]))
		{
			ETG_TRACE_USR1(("blEnableMAX20084Output()-->Both MAX20084 clients at A1 and A2 are enabled"));
			blRet =  blWriteReg((tU8)enMax20084SetUp1Rg, enMax20084EnableBothA1AndA2,enCMExtPowerSupplyType); //Both A1 and A2 enabled
			blSendAntennaStatus = true;
		}

		else if ((m_blMax20084OutputA1En[u8Index]) && (!m_blMax20084OutputA2En[u8Index]))
		{
			ETG_TRACE_USR1(("blEnableMAX20084Output()-->MAX20084 client at A2 is disabled"));
			blRet =  blWriteReg((tU8)enMax20084SetUp1Rg, enMax20084EnableOnlyA1,enCMExtPowerSupplyType); //Only A2 disabled
			blSendAntennaStatus = true;
		}
		ETG_TRACE_USR1(("blEnableMAX20084Output()--> for %02x: m_blMax20084OutputA1En=%d, m_blMax20084OutputA2En=%d",
			ETG_CENUM(tenCMExtPowerSupplyType,enCMExtPowerSupplyType),m_blMax20084OutputA1En[u8Index],m_blMax20084OutputA2En[u8Index]));
	}
	if((!blRet) && (blSendAntennaStatus) && (NULL != m_potunmstr_antdiag_Manager))
	{
		m_potunmstr_antdiag_Manager->vUpdateAntenaaStatetoHMI(m_potunmstr_antdiag_Manager->enGetAntDiagClient(enCMExtPowerSupplyType),enAntennState_Unknown, 0.0);
	}
	return blRet;
}
tBool  tunmstr_antdiag_i2cHandler::blMAX20084ClearErrorLatch(tenCMExtPowerSupplyType enCMExtPowerSupplyType)
{
	tBool blRet = false;
	if(ISPOWERSUPPLY_MAX20084(enCMExtPowerSupplyType))
	{
		tU8 u8Index = enGetMax20084IC_Index(enCMExtPowerSupplyType);
		tU8 u8EnableVal = 0x0;
		tU8 u8DisableVal = 0x0;
		if((!m_blMax20084OutputA1En[u8Index]) && (!m_blMax20084OutputA2En[u8Index]))
		{
			u8EnableVal= enMax20084DisableBothA1AndA2;
		}
		if((!m_blMax20084OutputA1En[u8Index]) && (m_blMax20084OutputA2En[u8Index]))
		{
			u8EnableVal= enMax20084EnableOnlyA2;
		}
		if ((m_blMax20084OutputA1En[u8Index]) && (m_blMax20084OutputA2En[u8Index]))
		{
			u8EnableVal= enMax20084EnableBothA1AndA2;
		}
		if ((m_blMax20084OutputA1En[u8Index]) && (!m_blMax20084OutputA2En[u8Index]))
		{
			u8EnableVal= enMax20084EnableOnlyA1;
		}
		u8DisableVal = u8GetMAX20084ErrorLatchMaskVal(enCMExtPowerSupplyType) & u8EnableVal;
		blRet = blWriteReg((tU8)enMax20084SetUp1Rg,u8DisableVal,enCMExtPowerSupplyType);
		blRet = blWriteReg((tU8)enMax20084SetUp1Rg,u8EnableVal,enCMExtPowerSupplyType);
	}
	return blRet;
}
tU8 tunmstr_antdiag_i2cHandler::u8GetMAX20084ErrorLatchMaskVal(tenCMExtPowerSupplyType enCMExtPowerSupplyType)
{
	tU8 u8MaskVal = enMax20084EnableBothA1AndA2;
	switch(enCMExtPowerSupplyType)
	{
		case enMAX20084_1_1:
		case enMAX20084_2_1:
		case enMAX20084_3_1:
			u8MaskVal = enMax20084EnableOnlyA2;
			break;
		case enMAX20084_1_2:
		case enMAX20084_2_2:
		case enMAX20084_3_2:
			u8MaskVal = enMax20084EnableOnlyA1;
			break;
		default:
			break;

	}
	return u8MaskVal;
}
/*************************************************************************
 * FUNCTION:     blIsShortCktError
 *
 * DESCRIPTION: Returns true if error register contains value due to short circuit error

 * PARAMETER:    tenCMExtPowerSupplyType enCMExtPowerSupplyType
 *
 * RETURNVALUE:  bool
 *************************************************************************/
tBool tunmstr_antdiag_i2cHandler::IsShortCktError(tenCMExtPowerSupplyType enCMExtPowerSupplyType)
{
    tBool blRet= FALSE;
    trI2CRead  otrI2CRead;
    if(ISPOWERSUPPLY_ELMOS(enCMExtPowerSupplyType))
    	otrI2CRead = trReadI2CReg(enErrorReg,enCMExtPowerSupplyType);
    else
    	otrI2CRead = trReadI2CReg(enMax20084Diag1Rg,enCMExtPowerSupplyType);
	ETG_TRACE_USR1(("IsShortCktError() =%d",otrI2CRead.u8Value));
	if(!otrI2CRead.blReadSucess)
	{
		return blRet;
	}
	if(m_potunmstr_antdiag_Measure==NULL)
    {
    	return blRet;
    }
    switch(enCMExtPowerSupplyType)
    {
        case enELMOS1_1:
		case enELMOS2_1:
		case enELMOS3_1:
		case enELMOS4_1:
        {
            if(TUNMSTR_ELMOSCUR_A1OVERCURERROR((otrI2CRead.u8Value & enA1_ShortDetectionMask)))
            {
                blRet=TRUE;
            }
            break;
        }
        case enELMOS1_2:
		case enELMOS2_2:
		case enELMOS3_2:
		case enELMOS4_2:
        {
            if(TUNMSTR_ELMOSCUR_A2OVERCURERROR((otrI2CRead.u8Value & enA2_ShortDetectionMask)))
            {
                blRet=TRUE;
            }
            break;
        
        }
		case enMAX20084_1_1:
		case enMAX20084_2_1:
		case enMAX20084_3_1:
		{
			if(TUNMSTR_MAX20084_A1_DIAG1ERROR(otrI2CRead.u8Value))
			{
				blRet=TRUE;
			}
			else
			{
				otrI2CRead = trReadI2CReg(enMax20084Diag2Rg,enCMExtPowerSupplyType);
				if(otrI2CRead.blReadSucess)
				{
					if(TUNMSTR_MAX20084_A1_DIAG2ERROR(otrI2CRead.u8Value))
						blRet=TRUE;
				}

			}
			break;
		}
		case enMAX20084_1_2:
		case enMAX20084_2_2:
		case enMAX20084_3_2:
		{
			if(TUNMSTR_MAX20084_A2_DIAG1ERROR(otrI2CRead.u8Value))
			{
				blRet=TRUE;
			}
			else
			{
				otrI2CRead = trReadI2CReg(enMax20084Diag2Rg,enCMExtPowerSupplyType);
				if(otrI2CRead.blReadSucess)
				{
					if(TUNMSTR_MAX20084_A2_DIAG2ERROR(otrI2CRead.u8Value))
						blRet=TRUE;
				}

			}
			break;
		}
        default:
        {
            break;
        }
    }
    if(blRet)
    {
   	if(ISPOWERSUPPLY_ELMOS(enCMExtPowerSupplyType))
    		blRet = blWriteReg((tU8)enErrorReg,0,enCMExtPowerSupplyType);
    	if(ISPOWERSUPPLY_MAX20084(enCMExtPowerSupplyType))
    		blRet = blMAX20084ClearErrorLatch(enCMExtPowerSupplyType);
    	m_potunmstr_antdiag_Measure->vUpdateAntennaCur(enCMExtPowerSupplyType,(tFloat)256.0);
    }
    return blRet;
}
tVoid tunmstr_antdiag_i2cHandler::vClearAllInternalState()
{
	for(tU8 u8Index=0;u8Index<enELMOSIcIndex;u8Index++)
	{
		m_blOutputA1Enabled[u8Index] = FALSE;
		m_blOutputA2Enabled[u8Index] = FALSE;
	}
}
tBool tunmstr_antdiag_i2cHandler::blTurnOnUsingGPIORegister(tU8 u8RegisterAddress,tU8 u8SlaveAddress, tU8 u8MaskVal)
{
	const char *dev = TUNMSTR_ANTDIAG_I2C_FILE_NAME;
	signed char fd = (signed char)open(dev,O_RDWR);
	if(fd <0)
	{
		tInt errsv = errno;
		ETG_TRACE_ERR(("blTurnOnUsingGPIORegister::Error in Opening I2C File with error:%d", errsv));
		ETG_TRACE_ERR(("blTurnOnUsingGPIORegister: Error in Opening I2C File and error msg : %s\n", strerror(errsv)));
		return 0;
	}
	unsigned char readbuf[1]={0};
	unsigned char writebuf[1] ={0};
	struct i2c_msg msgs[2];
	msgs[0].addr=u8SlaveAddress;
	msgs[0].flags=0x0;
	msgs[0].len=1;
	msgs[0].buf=writebuf;
	msgs[0].buf[0]=u8RegisterAddress;

	msgs[1].addr=u8SlaveAddress;
	msgs[1].flags=0x0 | I2C_M_RD;
	msgs[1].len=1;
	msgs[1].buf=readbuf;
	struct i2c_rdwr_ioctl_data data;
	data.msgs=msgs;
	data.nmsgs=2;
	if (ioctl(fd, I2C_RDWR, &data) < 0)
	{
		int errsv = errno;
		ETG_TRACE_ERR(("blTurnOnUsingGPIORegister: ioctl() failed with error:%d and return value", errsv));
		ETG_TRACE_ERR(("blTurnOnUsingGPIORegister: ioctl() failed with error msg : %s\n", strerror(errsv)));
		(void)close (fd);
		return false;
	}
	if( ioctl(fd, I2C_SLAVE, u8SlaveAddress)<0)
	{
		tInt errsv = errno;
		ETG_TRACE_ERR(("blTurnOnUsingGPIORegister::Error in Opening I2C File with error:%d", errsv));
		ETG_TRACE_ERR(("blTurnOnUsingGPIORegister: Error in Opening I2C File and error msg : %s\n", strerror(errsv)));
		(void)close (fd);
		return 0;
	}
	unsigned char writedata[TUNMSTR_ANTDIAG_2BYTE];
	writedata[0]= u8RegisterAddress;// write address of GPIO register
	writedata[1]= readbuf[0] | u8MaskVal;// set the bit 4 to turn on the phantom pin
	signed char writeret= (tU8) write(fd,&writedata,TUNMSTR_ANTDIAG_2BYTE);//3 byte of data
	if(writeret!=TUNMSTR_ANTDIAG_2BYTE)//if the 3 data is not written successfully
	{
		tInt errsv = errno;
		ETG_TRACE_ERR(("blTurnOnUsingGPIORegister::Error in Opening I2C File with error:%d", errsv));
		ETG_TRACE_ERR(("blTurnOnUsingGPIORegister: Error in Opening I2C File and error msg : %s\n", strerror(errsv)));
	}
	(void)close (fd);
	ETG_TRACE_ERR(("blTurnOnUsingGPIORegister Success"));
	return true;
}
tBool tunmstr_antdiag_i2cHandler::blTurnOffUsingGPIORegister(tU8 u8RegisterAddress,tU8 u8SlaveAddress,tU8 u8MaskVal)
{
	const char *dev = TUNMSTR_ANTDIAG_I2C_FILE_NAME;
	signed char fd = (signed char) open(dev,O_RDWR);
	if(fd <0)
	{
		tInt errsv = errno;
		ETG_TRACE_ERR(("blTurnOffUsingGPIORegister::Error in Opening I2C File with error:%d", errsv));
		ETG_TRACE_ERR(("blTurnOffUsingGPIORegister: Error in Opening I2C File and error msg : %s\n", strerror(errsv)));
		return 0;
	}
	unsigned char readbuf[1]={0};
	unsigned char writebuf[1] ={0};
	struct i2c_msg msgs[2];
	msgs[0].addr=u8SlaveAddress;
	msgs[0].flags=0x0;
	msgs[0].len=1;
	msgs[0].buf=writebuf;
	msgs[0].buf[0]=u8RegisterAddress;

	msgs[1].addr=u8SlaveAddress;
	msgs[1].flags=0x0 | I2C_M_RD;
	msgs[1].len=1;
	msgs[1].buf=readbuf;
	struct i2c_rdwr_ioctl_data data;
	data.msgs=msgs;
	data.nmsgs=2;
	if (ioctl(fd, I2C_RDWR, &data) < 0)
	{
		int errsv = errno;
		ETG_TRACE_ERR(("blTurnOffUsingGPIORegister: ioctl() failed with error:%d and return value", errsv));
		ETG_TRACE_ERR(("blTurnOffUsingGPIORegister: ioctl() failed with error msg : %s\n", strerror(errsv)));
		(void)close (fd);
		return false;
	}
	if( ioctl(fd, I2C_SLAVE, u8SlaveAddress)<0)
	{
		tInt errsv = errno;
		ETG_TRACE_ERR(("blTurnOffUsingGPIORegister::Error in Opening I2C File with error:%d", errsv));
		ETG_TRACE_ERR(("blTurnOffUsingGPIORegister: Error in Opening I2C File and error msg : %s\n", strerror(errsv)));
		(void)close (fd);
		return 0;
	}
	unsigned char writedata[2];
	writedata[0]= u8RegisterAddress;
	writedata[1]= (readbuf[0] & u8MaskVal);
	signed char writeret= (tU8)write(fd,&writedata,TUNMSTR_ANTDIAG_2BYTE);
	if(writeret!=TUNMSTR_ANTDIAG_2BYTE)//if the 3 data is not written successfully
	{
		tInt errsv = errno;
		ETG_TRACE_ERR(("blTurnOffUsingGPIORegister::Error in Opening I2C File with error:%d", errsv));
		ETG_TRACE_ERR(("blTurnOffUsingGPIORegister: Error in Opening I2C File and error msg : %s\n", strerror(errsv)));
	}
	(void)close (fd);
	ETG_TRACE_ERR(("blTurnOffUsingGPIORegister Success"));
	return true;
}
tVoid tunmstr_antdiag_i2cHandler::vDummyI2CHalfRead()
{
   if(NULL == m_potunmstr_antdiag_Manager)
   {
	   return;
   }
   if(!m_blDummyI2CHalfReadCompleted)
   {
	   for(tU8 enCMExtPowerSupplyType=enELMOS1_1; enCMExtPowerSupplyType<=enELMOS4_2; enCMExtPowerSupplyType++)
	   	{
		   if(m_potunmstr_antdiag_Manager->blIsExtSup_Supports((tenCMExtPowerSupplyType)enCMExtPowerSupplyType))
		   {
			   ETG_TRACE_USR1(("vDummyI2CHalfRead"));
			   const char *dev = TUNMSTR_ANTDIAG_I2C_FILE_NAME;
			   tU8 u8ICAdd = u8GetICAddress((tenCMExtPowerSupplyType)enCMExtPowerSupplyType);
			   tS8 s8i2cFileHandler = (tS8)open(dev, O_RDWR);
			   if(s8i2cFileHandler < 0)
			   {
				   tInt errsv = errno;
				   ETG_TRACE_ERR(("vDummyI2CHalfRead::Error in Opening I2C File with error:%d", errsv));
				   ETG_TRACE_ERR(("vDummyI2CHalfRead: Error in Opening I2C File and error msg : %s\n", strerror(errsv)));
			   }
			   else
			   {
				   //the device address to communicate
				   tS8 s8ioctlRet = (tS8)ioctl(s8i2cFileHandler, I2C_SLAVE, u8ICAdd);
				   tU8 readbuf[1] = {0};
				   struct i2c_msg msgs[4];
				   msgs[0].addr = u8ICAdd;
				   msgs[0].flags = 0x0 | I2C_M_RD;
				   msgs[0].len = 1;
				   msgs[0].buf = readbuf;
				   struct i2c_rdwr_ioctl_data data;
				   data.msgs = msgs;
				   data.nmsgs = 1;
				   if(ioctl(s8i2cFileHandler, I2C_RDWR, &data) < 0)
				   {
					   ETG_TRACE_ERR(("vDummyI2CHalfRead::Error in ioctl with"));
					   int errsv = errno;
					   ETG_TRACE_ERR(("vDummyI2CHalfRead: ioctl() failed with error:%d and return value", errsv));
					   ETG_TRACE_ERR(("vDummyI2CHalfRead: ioctl() failed with error msg : %s\n", strerror(errsv)));
				   }
				   (void)close(s8i2cFileHandler);
			   }
		   }
		   if(enELMOS2_2 == enCMExtPowerSupplyType)
		   {
			  enCMExtPowerSupplyType = enInfineonTLF4277_2;
		   }
	   	}
   }
   m_blDummyI2CHalfReadCompleted = true;
}
