/************************************************************************
* FILE:           tunmstr_antdiag_Measure.cpp
* PROJECT:        G3g
* SW-COMPONENT:   FC_TunerMaster.
*----------------------------------------------------------------------
* DESCRIPTION:    Implementation of the Class  tunmstr_antdiag_Measure
*
*----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
* HISTORY:
* Date      | Author                            | Modification
* 25/05/2012| ECV4 ( RBEI ) Priya Prabhuswamy   | Initial version.
* 25.04.13  | NGP1KOR							| First version of the G3g after porting 
* 03.03.14  | ECV3 ( RBEI ) Priya Prabhuswamy   | adaptation for Different Type of External Power supply
*************************************************************************/


// System includes.
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"
#include "tunmstr_antdiag_Measure.h"
#include "tunmstr_antdiag_Manager.h"
#include "tunmstr_antdiag_Debounce.h"
#include "tunmstr_DiagLogClientHandler.h"
#include "tunmstr_antdiag_i2cHandler.h"
#include <errno.h>
// ETG Trace Defines
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TUNMSTR_TRACE_CLASS_ANTDIAG
#include "trcGenProj/Header/tunmstr_antdiag_Measure.cpp.trc.h"
#endif

/*************************************************************************
 * FUNCTION:     tunmstr_antdiag_Measure::tunmstr_antdiag_Measure( )
 *
 * DESCRIPTION:  Constructor
 *
 * PARAMETER:    void
 *
 * RETURNVALUE:  void
 *************************************************************************/
tunmstr_antdiag_Measure::tunmstr_antdiag_Measure( )
{
	m_potunmstr_antdiag_Debounce=NULL;
	m_potunmstr_antdiag_Manager=NULL;
	m_potunmstr_DiagLogClientHandler=NULL;
	m_hInfineonIOHandler=OSAL_ERROR;
	m_hInfineon2IOHandler=OSAL_ERROR;
	m_hMAXIOHandler=OSAL_ERROR;
	m_blnfineonADCChannel=FALSE;
	m_blnfineon2ADCChannel=FALSE;
	m_bMAXADCChannel=FALSE;
	m_flnfineon_Cur=0;
	m_flnfineon2_Cur=0;
	m_fMax_Cur=0;
	m_fMaxIC_CalculatedCur=0;
	m_fELMOS1_1Cur=0;
	m_fELMOS1_2Cur=0;
	m_fELMOS2_1Cur=0;
	m_fELMOS2_2Cur=0;
	m_fELMOS3_1Cur=0;
	m_fELMOS3_2Cur=0;
	m_fELMOS4_1Cur=0;
	m_fELMOS4_2Cur=0;
	m_fTPS7B7701_Cur=0;
	m_fTPS7B7702_1_1Cur=0;
	m_fTPS7B7702_1_2Cur=0;
	m_fMAX20084_1_1_Cur = 0;
	m_fMAX20084_1_2_Cur = 0;
	m_fMAX20084_2_1_Cur = 0;
	m_fMAX20084_2_2_Cur = 0;
	m_fMAX20084_3_1_Cur = 0;
	m_fMAX20084_3_2_Cur = 0;
	m_hTPS7B7701_IOHandler=OSAL_ERROR;
	m_hTPS7B7702_1_1IOHandler=OSAL_ERROR;
	m_hTPS7B7702_1_2IOHandler=OSAL_ERROR;
	m_blTPS7B7701ADCChannelOPen=FALSE;
	m_blTPS7B7702_1_1ADCChannelOpen=FALSE;
	m_blTPS7B7702_1_2ADCChannelOpen=FALSE;
}
/*************************************************************************
 * FUNCTION:     tunmstr_antdiag_Measuer::~tunmstr_antdiag_Measuer( )
 *
 * DESCRIPTION:  Destructor
 *
 * PARAMETER:    void
 *
 * RETURNVALUE:  void
 *************************************************************************/
tunmstr_antdiag_Measure::~tunmstr_antdiag_Measure( )
{
	//vCloselnfineonADCChannel ();
	m_potunmstr_antdiag_Debounce=NULL;
	m_potunmstr_antdiag_Manager=NULL;
	m_potunmstr_DiagLogClientHandler=NULL;
	m_hInfineonIOHandler=OSAL_ERROR;
	m_hInfineon2IOHandler=OSAL_ERROR;
	m_hMAXIOHandler=OSAL_ERROR;
	m_blnfineonADCChannel=FALSE;
	m_blnfineon2ADCChannel=FALSE;
	m_bMAXADCChannel=FALSE;
	m_flnfineon_Cur=0;
	m_flnfineon2_Cur=0;
	m_fMax_Cur=0;
	m_fELMOS1_1Cur=0;
	m_fELMOS1_2Cur=0;
	m_fELMOS2_1Cur=0;
	m_fELMOS2_2Cur=0;
	m_fMaxIC_CalculatedCur=0;
	m_fELMOS3_1Cur=0;
	m_fELMOS3_2Cur=0;
	m_fELMOS4_1Cur=0;
	m_fELMOS4_2Cur=0;
	m_fTPS7B7701_Cur=0;
	m_fTPS7B7702_1_1Cur=0;
	m_fTPS7B7702_1_2Cur=0;
	m_fMAX20084_1_1_Cur = 0;
	m_fMAX20084_1_2_Cur = 0;
	m_fMAX20084_2_1_Cur = 0;
	m_fMAX20084_2_2_Cur = 0;
	m_fMAX20084_3_1_Cur = 0;
	m_fMAX20084_3_2_Cur = 0;
	m_hTPS7B7701_IOHandler=OSAL_ERROR;
	m_hTPS7B7702_1_1IOHandler=OSAL_ERROR;
	m_hTPS7B7702_1_2IOHandler=OSAL_ERROR;
	m_blTPS7B7701ADCChannelOPen=FALSE;
	m_blTPS7B7702_1_1ADCChannelOpen=FALSE;
	m_blTPS7B7702_1_2ADCChannelOpen=FALSE;

}
/*************************************************************************
 * FUNCTION:    tunmstr_antdiag_Measure::vSet_tunmstr_antdiag_Debounce_Ptr( tunmstr_antdiag_Debounce* potunmstr_antdiag_Debounce )
 *
 * DESCRIPTION: Set pointer to pointer tunmstr_antdiag_Debounce
 *
 * PARAMETER:   tunmstr_antdiag_Debounce* potunmstr_antdiag_Debounce
 *
 * RETURNVALUE:  void
 *************************************************************************/
tVoid tunmstr_antdiag_Measure::vSet_tunmstr_antdiag_Debounce_Ptr(tunmstr_antdiag_Debounce* potunmstr_antdiag_Debounce)
{
	if( potunmstr_antdiag_Debounce != NULL )
	{
		m_potunmstr_antdiag_Debounce = ( tunmstr_antdiag_Debounce* )potunmstr_antdiag_Debounce;
	}
}
/*************************************************************************
 * FUNCTION:    tunmstr_antdiag_Measure::vSet_tunmstr_antdiag_Debounce_Ptr( tunmstr_antdiag_Debounce* potunmstr_antdiag_Debounce )
 *
 * DESCRIPTION: Set pointer to pointer tunmstr_antdiag_Debounce
 *
 * PARAMETER:   tunmstr_antdiag_Debounce* potunmstr_antdiag_Debounce
 *
 * RETURNVALUE:  void
 *************************************************************************/
tVoid tunmstr_antdiag_Measure::vSet_tunmstr_antdiag_Manager(tunmstr_antdiag_Manager* potunmstr_antdiag_Manager)
{
	if( potunmstr_antdiag_Manager != NULL )
	{
		m_potunmstr_antdiag_Manager = ( tunmstr_antdiag_Manager* )potunmstr_antdiag_Manager;
	}
}
/************************************************************************
 * FUNCTION:    vSet_tunmstr_DiagLogClientHandler_Ptr()
 *
 * DESCRIPTION: function to update m_potunmstr_DiagLogClientHandler
 *
 * PARAMETER:   tunmstr_DiagLogClientHandler* potunmstr_DiagLogClientHandler
 *
 * RETURNVALUE:  void
 *************************************************************************/
tVoid tunmstr_antdiag_Measure::vSet_tunmstr_DiagLogClientHandler_Ptr(tunmstr_DiagLogClientHandler* potunmstr_DiagLogClientHandler)
{
	if( potunmstr_DiagLogClientHandler != NULL )
	{
		m_potunmstr_DiagLogClientHandler = ( tunmstr_DiagLogClientHandler* )potunmstr_DiagLogClientHandler;
	}
}
/************************************************************************
 * FUNCTION:    vOpenlnfineonADCChannel()
 *
 * DESCRIPTION: function to open ADC channel of Infineon IC
 *
 * PARAMETER:   
 *
 * RETURNVALUE:  void
 *************************************************************************/
tVoid tunmstr_antdiag_Measure::	vOpenlnfineonADCChannel()
{
	if(!m_blnfineonADCChannel)
	{
		tCString stADCChannel;
		stADCChannel=OSAL_C_STRING_DEVICE_ADC_TLF4277_CSO;
		m_hInfineonIOHandler= OSAL_IOOpen((tCString)stADCChannel,OSAL_EN_READWRITE);
		if(m_hInfineonIOHandler==OSAL_ERROR)
		{
			ETG_TRACE_ERR(("vOpenlnfineonADCChannel()->ERRRO in Opening lnfineon IC ADC channel"));
			m_blnfineonADCChannel=FALSE;
			return;
		}
		else
		{
			m_blnfineonADCChannel=TRUE;
			ETG_TRACE_USR1(("vOpenlnfineonADCChannel() success"));
		}
	}
}
/*************************************************************************
 * FUNCTION:     vOpenlnfineon2ADCChannel
 *
 * DESCRIPTION:	Open ADC channel infineon2 IC
 *************************************************************************/
tVoid tunmstr_antdiag_Measure::	vOpenlnfineon2ADCChannel()
{
	if(!m_blnfineon2ADCChannel)
	{
		tCString stADCChannel;
		stADCChannel=OSAL_C_STRING_DEVICE_ADC_TLF4277_CSO_2;
		m_hInfineon2IOHandler= OSAL_IOOpen((tCString)stADCChannel,OSAL_EN_READWRITE);
		if(m_hInfineon2IOHandler==OSAL_ERROR)
		{
			ETG_TRACE_ERR(("vOpenlnfineon2ADCChannel()->ERRRO in Opening lnfineon2 IC ADC channel"));
			m_blnfineon2ADCChannel=FALSE;
			return;
		}
		else
		{
			m_blnfineon2ADCChannel=TRUE;
			ETG_TRACE_USR1(("vOpenlnfineon2ADCChannel()-> success"));
		}
	}
}
/*************************************************************************
 * FUNCTION:     vOpenMAXADCChannel
 *
 * DESCRIPTION:	Open ADC channel of MAX IC
 *************************************************************************/
tVoid tunmstr_antdiag_Measure::	vOpenMAXADCChannel()
{
	if(!m_bMAXADCChannel)
	{
		tCString stADCChannel;
		stADCChannel=OSAL_C_STRING_DEVICE_ADC_MAX16946_SENS;
		m_hMAXIOHandler= OSAL_IOOpen((tCString)stADCChannel,OSAL_EN_READWRITE);
		if(m_hMAXIOHandler==OSAL_ERROR)
		{
			ETG_TRACE_ERR(("vOpenMAXADCChannel()->ERRRO in Opening Max IC ADC channel"));
			m_bMAXADCChannel=FALSE;
			return;
		}
		else
		{
			m_bMAXADCChannel=TRUE;
			ETG_TRACE_USR1(("vOpenMAXADCChannel()-> success"));
		}
	}
}
/*Not tested, due to unavailabilty of the hardware*/
tVoid tunmstr_antdiag_Measure::vOpenTPS7B7701ADCChannel()
{
	/*if(!m_blTPS7B7701ADCChannelOPen)
	{
		tCString stADCChannel;
		stADCChannel=OSAL_C_STRING_DEVICE_ADC_TPS7B7701;
		m_hTPS7B7701_IOHandler= OSAL_IOOpen((tCString)stADCChannel,OSAL_EN_READWRITE);
		if(m_hTPS7B7701_IOHandler==OSAL_ERROR)
		{
			ETG_TRACE_ERR(("vOpenTPS7B7701ADCChannel()->ERRRO in Opening TPS7B7701 ADC channel"));
			m_blTPS7B7701ADCChannelOPen=FALSE;
			return;
		}
		else
		{
			m_blTPS7B7701ADCChannelOPen=TRUE;
			ETG_TRACE_USR1(("vOpenTPS7B7701ADCChannel()-> success"));
		}
	}*/
}
tVoid tunmstr_antdiag_Measure::vOpenTPS7B7702_1_1ADCChannel()
{
	if(!m_blTPS7B7702_1_1ADCChannelOpen)
	{
		tCString stADCChannel = OSAL_C_STRING_DEVICE_ADC_TPS7B7702_1_SENS1;
		m_hTPS7B7702_1_1IOHandler = OSAL_IOOpen((tCString)stADCChannel,OSAL_EN_READWRITE);
		if(m_hTPS7B7702_1_1IOHandler == OSAL_ERROR)
		{
			ETG_TRACE_ERR(("vOpenTPS7B7702_1_1ADCChannel()->ERRRO in Opening TPS7B7702_1_1 ADC channel"));
			tInt errsv = errno;
			ETG_TRACE_ERR(("vOpenTPS7B7702_1_1ADCChannel::ERRRO in Opening TPS7B7702_1_1 ADC channel with error:%d", errsv));
			ETG_TRACE_ERR(("vOpenTPS7B7702_1_1ADCChannel: ERRRO in Opening TPS7B7702_1_1 ADC Channel and error msg : %s\n", strerror(errsv)));
			m_blTPS7B7702_1_1ADCChannelOpen = FALSE;
			return;
		}
		else
		{
			m_blTPS7B7702_1_1ADCChannelOpen = TRUE;
			ETG_TRACE_USR1(("vOpenTPS7B7702_1_1ADCChannel()-> success"));
		}
	}
}
tVoid tunmstr_antdiag_Measure::vOpenTPS7B7702_1_2ADCChannel()
{
	if(!m_blTPS7B7702_1_2ADCChannelOpen)
	{
		tCString stADCChannel = OSAL_C_STRING_DEVICE_ADC_TPS7B7702_1_SENS2;
		m_hTPS7B7702_1_2IOHandler= OSAL_IOOpen((tCString)stADCChannel,OSAL_EN_READWRITE);
		if(m_hTPS7B7702_1_2IOHandler == OSAL_ERROR)
		{
			ETG_TRACE_ERR(("vOpenTPS7B7702_1_2ADCChannel()->ERRRO in Opening TPS7B7702_1_2 ADC channel"));
			tInt errsv = errno;
			ETG_TRACE_ERR(("vOpenTPS7B7702_1_2ADCChannel::ERRRO in Opening TPS7B7702_1_2 ADC channel with error:%d", errsv));
			ETG_TRACE_ERR(("vOpenTPS7B7702_1_2ADCChannel: ERRRO in Opening TPS7B7702_1_2 ADC Channel and error msg : %s\n", strerror(errsv)));
			m_blTPS7B7702_1_2ADCChannelOpen = FALSE;
			return;
		}
		else
		{
			m_blTPS7B7702_1_2ADCChannelOpen = TRUE;
			ETG_TRACE_USR1(("vOpenTPS7B7702_1_2ADCChannel()-> success"));
		}
	}
}
tVoid tunmstr_antdiag_Measure::vCloselnfineonADCChannel()
{
	OSAL_s32IOClose ( m_hInfineonIOHandler );
	m_blnfineonADCChannel=FALSE;
}
/*************************************************************************
 * FUNCTION:     vCloselnfineon2ADCChannel
 *
 * DESCRIPTION:Function to close Infineon2 IC ADC channel
 *************************************************************************/
tVoid tunmstr_antdiag_Measure::vCloselnfineon2ADCChannel()
{
	OSAL_s32IOClose ( m_hInfineon2IOHandler );
	m_blnfineon2ADCChannel=FALSE;
}
/*************************************************************************
 * FUNCTION:     tunmstr_antdiag_Measuer::vClosechannel
 *
 * DESCRIPTION:	This function will Close ADC channel for the requested antenna
 *				
 * PARAMETER:	tU8 u8Client,tU8 u8Whichantenna
 *
 * RETURNVALUE:	tVoid
 *************************************************************************/
tVoid tunmstr_antdiag_Measure::vCloseMAXADCChannel()
{
	OSAL_s32IOClose ( m_hMAXIOHandler );
	m_bMAXADCChannel=FALSE;
}
/*Not tested, due to unavailabilty of the hardware*/
tVoid tunmstr_antdiag_Measure::vCloseTPS7B7701ADCChannel()
{
	/*OSAL_s32IOClose ( m_hTPS7B7701_IOHandler );
	m_blTPS7B7701ADCChannelOPen=FALSE;*/
}
tVoid tunmstr_antdiag_Measure::vCloseTPS7B7702_1_1ADCChannel()
{
	OSAL_s32IOClose ( m_hTPS7B7702_1_1IOHandler );
	m_blTPS7B7702_1_1ADCChannelOpen=FALSE;
}
tVoid tunmstr_antdiag_Measure::vCloseTPS7B7702_1_2ADCChannel()
{
	OSAL_s32IOClose ( m_hTPS7B7702_1_2IOHandler );
	m_blTPS7B7702_1_2ADCChannelOpen=FALSE;
}
/*************************************************************************
 * FUNCTION:     tunmstr_antdiag_Measuer::fCalculateInfineonCurrent
 *
 * DESCRIPTION:	This function will calculate the current as per the ADC value measured for Infineon IC
 *				
 * PARAMETER:	tU16 u16adcSample
 *
 * RETURNVALUE:	float
 *************************************************************************/
tFloat tunmstr_antdiag_Measure::fCalculateInfineonCurrent(tU16 u16adcSample)const
{
	tFloat fCurrent=0;
	/*Infineon IC
		I = U(DAB_ANT1_SENSE) * 58,1 mA/V
		U(ADC) = 3,36V * ADC_Readout / (2^n -1)*/
#ifdef VARIANT_S_FTR_ENABLE_FEATURE_SUZUKI
	fCurrent=(((u16adcSample*3.3)/(TUNMSTR_ANTDIAG_DAB_CURRENT_CAL_REDUCTION_FACT))* TUNMSTR_ANTDIAG_INFINEON_SUZ_VREF);
#else
#ifdef VARIANT_S_FTR_ENABLE_FEATURE_PSA_RCC
	//Below formula is valid only for PSA DAB antenna
	fCurrent=(((u16adcSample*3.3)/(TUNMSTR_ANTDIAG_TENBIT_VALUE))* TUNMSTR_ANTDIAG_INFINEON_PSA_VREF);
#else
	fCurrent= (tFloat) (((u16adcSample*3.3)/(TUNMSTR_ANTDIAG_DAB_CURRENT_CAL_REDUCTION_FACT))* TUNMSTR_ANTDIAG_INFINEON_OTHER_VREF);
#endif
#endif

	ETG_TRACE_USR4(("fCalculateInfineonCurrent()->fCurrent=%f",fCurrent));
	return fCurrent;
}
/*************************************************************************
 * FUNCTION:     fCalculateInfineon2Current
 *
 * DESCRIPTION:	This function will calculate Infineon2 current 
 				as per the ADC value measured for Infineon2 IC
 *
 * PARAMETER:	tU16 u16adcSample
 *
 * RETURNVALUE:	float
 *************************************************************************/
tFloat tunmstr_antdiag_Measure::fCalculateInfineon2Current(tU16 u16adcSample)const
{
	tFloat fCurrent=0;
//Only PSA HW has 2Infineon IC. For other project update the formula.
// Other than PSA project,update the current as 0
#ifdef VARIANT_S_FTR_ENABLE_FEATURE_PSA_RCC
	fCurrent=((u16adcSample*3.3)/TUNMSTR_ANTDIAG_TENBIT_VALUE) * TUNMSTR_ANTDIAG_INFINEON2_GPS_VREF;
#endif
	//etg trace is added to remove the Prio2 Lint
	ETG_TRACE_USR4(("fCalculateInfineon2Current()->u16adcSample=%d",u16adcSample));
	return fCurrent;
}
/*Not tested, due to unavailabilty of the hardware*/
tFloat tunmstr_antdiag_Measure::fCalculateTPS7B7701Current(tU16 /*u16adcSample*/)const
{
	tFloat fCurrent=0;
	return fCurrent;
}
tFloat tunmstr_antdiag_Measure::fCalculateTPS7B7702_1_1Current(tU16 u16adcSample,tenAntDiagClients enAntDiagClients)
{
	tFloat fCurrent=0;
	fCurrent = (tFloat)((((TUNMSTR_ANTDIAG_TPS7B7702_VREF * u16adcSample) * TUNMSTR_ANTDIAG_TPS7B7702_OUTI_TO_ISENSE_RATIO)/fGetTPS7B7702RLIM(enAntDiagClients)) / TUNMSTR_ANTDIAG_12BIT_VALUE);
	ETG_TRACE_USR4(("fCalculateTPS7B7702_1_1Current()->fCurrent=%f",fCurrent));
	return fCurrent;
}
tFloat tunmstr_antdiag_Measure::fGetTPS7B7702RLIM(tenAntDiagClients enAntDiagClients)
{
	if(enAntDiagClients == enFM1)
	{
		return TUNMSTR_ANTDIAG_TPS7B7702_FM1_RLIM;
	}
	else if (enAntDiagClients == enGPS)
	{
		return TUNMSTR_ANTDIAG_TPS7B7702_GPS_RLIM;
	}
	return (tFloat)0.0;
}
tFloat tunmstr_antdiag_Measure::fCalculateTPS7B7702_1_2Current(tU16 u16adcSample, tenAntDiagClients enAntDiagClients)
{
	tFloat fCurrent=0;
	fCurrent = (tFloat)((((TUNMSTR_ANTDIAG_TPS7B7702_VREF * u16adcSample) * TUNMSTR_ANTDIAG_TPS7B7702_OUTI_TO_ISENSE_RATIO)/fGetTPS7B7702RLIM(enAntDiagClients)) / TUNMSTR_ANTDIAG_12BIT_VALUE);
	ETG_TRACE_USR4(("fCalculateTPS7B7702_1_2Current()->fCurrent=%f",fCurrent));
	return fCurrent;
}
/*************************************************************************
 * FUNCTION:     vReadInfineonIC_ADCVal
 *
 * DESCRIPTION:	This function will read the ADC channel 
 *
 * PARAMETER:	tU8 u8WhichClient
 *
 * RETURNVALUE:	tVoid
 *************************************************************************/
tVoid tunmstr_antdiag_Measure::vReadInfineonIC_ADCVal()
{
	if((NULL!=m_potunmstr_antdiag_Manager)&& (NULL!=m_potunmstr_antdiag_Debounce)&& (NULL!=m_potunmstr_DiagLogClientHandler))
	{
		tBool blGPIOpinState=m_potunmstr_antdiag_Manager->blGetExtPwrSupEN_PinState(enInfineonTLF4277_1);
		if((blGPIOpinState==TUNMSTR_ANTDIAG_GPIO_STATE_OUTPUT)&&(IsShortCkt_TimeOver(enInfineonTLF4277_1)))
		{
			blGPIOpinState=m_potunmstr_antdiag_Manager->blTurnOnGPIOPin(OSAL_EN_TLF4277_EN);
			m_potunmstr_antdiag_Manager->vSetExtPwrSupEN_PinState(enInfineonTLF4277_1,blGPIOpinState);
			m_potunmstr_antdiag_Debounce->vClearShortWaitOverCounter(enInfineonTLF4277_1);
			m_potunmstr_DiagLogClientHandler->blUpdateAntennaStatus(m_potunmstr_antdiag_Manager->enGetAntDiagClient(enInfineonTLF4277_1),enAntennState_Unknown);
		}
		else if((blGPIOpinState==(tBool)TUNMSTR_ANTDIAG_GPIO_STATE_INPUT)&& (m_blnfineonADCChannel==TRUE))
		{
			tPU16 pu16ADCdata  = NULL;
			tU16 u16ADCdata=0;
			pu16ADCdata = &u16ADCdata;
			tU32 u32NoOfBytesRead = (tU32)OSAL_s32IORead(m_hInfineonIOHandler,(tPS8)pu16ADCdata ,TUNMSTR_ANTDIAG_ADC_READ_2BYTE);
			ETG_TRACE_USR1((" vReadInfineonIC_ADCVal()->Infineon IC ADC sample =%d u32NoOfBytesRead=%d",*pu16ADCdata,u32NoOfBytesRead));
			//// only 10 Bits are valid!
			u16ADCdata = *pu16ADCdata & 0x03FF;
			tFloat fCurrent=fCalculateInfineonCurrent(u16ADCdata);
			vUpdateAntennaCur(enInfineonTLF4277_1,fCurrent);
		}
	}
}
/*************************************************************************
 * FUNCTION:     vReadInfineon2IC_ADCVal

* DESCRIPTION:	Function to calculate the Infineon2 IC current. 
				Formula will be changed in the next release.
 *************************************************************************/
tVoid tunmstr_antdiag_Measure::vReadInfineon2IC_ADCVal()
{
	if((m_potunmstr_antdiag_Manager!=NULL)&& (m_potunmstr_antdiag_Debounce!=NULL)&& (m_potunmstr_DiagLogClientHandler!=NULL))
	{
	tBool blGPIOpinState=m_potunmstr_antdiag_Manager->blGetExtPwrSupEN_PinState(enInfineonTLF4277_2);
	if((blGPIOpinState==TUNMSTR_ANTDIAG_GPIO_STATE_OUTPUT)&&(IsShortCkt_TimeOver(enInfineonTLF4277_2)))
	{
		blGPIOpinState=m_potunmstr_antdiag_Manager->blTurnOnGPIOPin(OSAL_EN_TLF4277_EN_2);
		m_potunmstr_antdiag_Manager->vSetExtPwrSupEN_PinState(enInfineonTLF4277_2,blGPIOpinState);
		m_potunmstr_antdiag_Debounce->vClearShortWaitOverCounter(enInfineonTLF4277_2);
		m_potunmstr_DiagLogClientHandler->blUpdateAntennaStatus(m_potunmstr_antdiag_Manager->enGetAntDiagClient(enInfineonTLF4277_2)
				,enAntennState_Unknown);
	}
	else if((blGPIOpinState==(tBool)TUNMSTR_ANTDIAG_GPIO_STATE_INPUT)&& (m_blnfineon2ADCChannel==TRUE))
	{
		tPU16 pu16ADCdata  = NULL;
		tU16 u16ADCdata=0;
		pu16ADCdata = &u16ADCdata;
	    tU32 u32NoOfBytesRead = (tU32)OSAL_s32IORead(m_hInfineon2IOHandler,(tPS8)pu16ADCdata ,TUNMSTR_ANTDIAG_ADC_READ_2BYTE);
		ETG_TRACE_USR1((" vReadInfineon2IC_ADCVal()->Infineon2 ADC sample =%d u32NoOfBytesRead=%d",*pu16ADCdata,u32NoOfBytesRead));
		//// only 10 Bits are valid!
		u16ADCdata = *pu16ADCdata & 0x03FF;
		tFloat fCurrent=fCalculateInfineon2Current(u16ADCdata);
		vUpdateAntennaCur(enInfineonTLF4277_2,fCurrent);
	}
	}
}
/*************************************************************************
 * FUNCTION:     vReadMAXIC_ADCVal

* DESCRIPTION:	Function to calculate the MAX IC current. 	
 *************************************************************************/
tVoid tunmstr_antdiag_Measure::vReadMAXIC_ADCVal()
{
	if((NULL!=m_potunmstr_antdiag_Manager)&& (NULL!=m_potunmstr_antdiag_Debounce)&& (NULL!=m_potunmstr_DiagLogClientHandler))
	{
	tBool blGPIOpinState=m_potunmstr_antdiag_Manager->blGetExtPwrSupEN_PinState(enMAXIM_16946);
	if((blGPIOpinState==TUNMSTR_ANTDIAG_GPIO_STATE_OUTPUT) && (IsShortCkt_TimeOver(enMAXIM_16946)))
	{
		blGPIOpinState=m_potunmstr_antdiag_Manager->blTurnOnGPIOPin(OSAL_EN_MAX16946_SHDN);
		m_potunmstr_antdiag_Manager->vSetExtPwrSupEN_PinState(enMAXIM_16946,blGPIOpinState);
		m_potunmstr_antdiag_Debounce->vClearShortWaitOverCounter(enMAXIM_16946);
		m_potunmstr_DiagLogClientHandler->blUpdateAntennaStatus(m_potunmstr_antdiag_Manager->enGetAntDiagClient(enMAXIM_16946),
				enAntennState_Unknown);
	}
	else if((blGPIOpinState==(tBool)TUNMSTR_ANTDIAG_GPIO_STATE_INPUT)&& (m_bMAXADCChannel==TRUE))
	{
		tPU16 pu16ADCdata  = NULL;
		tU16 u16ADCdata=0;
		pu16ADCdata = &u16ADCdata;
		tU32 u32NoOfBytesRead = (tU32)OSAL_s32IORead(m_hMAXIOHandler,(tPS8)pu16ADCdata ,TUNMSTR_ANTDIAG_ADC_READ_2BYTE);
		ETG_TRACE_USR1((" vReadMAXIC_ADCVal()->MAXIC ADC Valu =%d u32NoOfBytesRead=%d",*pu16ADCdata,u32NoOfBytesRead));
	    // only 10 Bits are valid!
		u16ADCdata = *pu16ADCdata & 0x03FF;
		tFloat fCurrent=fCalculateMaxICCurrent(u16ADCdata);
		vUpdateAntennaCur(enMAXIM_16946,fCurrent);
	}
	}
}
/*Not tested, due to unavailabilty of the hardware*/
tVoid tunmstr_antdiag_Measure::vReadTPS7B7701_ADCVal()
{
	/*if((NULL!=m_potunmstr_antdiag_Manager)&& (NULL!=m_potunmstr_antdiag_Debounce)&& (NULL!=m_potunmstr_DiagLogClientHandler))
	{
		tBool blGPIOpinState=m_potunmstr_antdiag_Manager->blGetExtPwrSupEN_PinState(enTPS7B7701_1);
		if((blGPIOpinState==TUNMSTR_ANTDIAG_GPIO_STATE_OUTPUT)&&(IsShortCkt_TimeOver(enTPS7B7701_1)))
		{
			blGPIOpinState=m_potunmstr_antdiag_Manager->blTurnOnGPIOPin(OSAL_EN_TPS7B7701_EN_1);
			m_potunmstr_antdiag_Manager->vSetExtPwrSupEN_PinState(enTPS7B7701_1,blGPIOpinState);
			m_potunmstr_antdiag_Debounce->vClearShortWaitOverCounter(enTPS7B7701_1);
			m_potunmstr_DiagLogClientHandler->blUpdateAntennaStatus(m_potunmstr_antdiag_Manager->enGetAntDiagClient(enTPS7B7701_1),enAntennState_Unknown);
		}
		else if((blGPIOpinState==(tBool)TUNMSTR_ANTDIAG_GPIO_STATE_INPUT)&& (m_blTPS7B7701ADCChannelOPen==TRUE))
		{
			tPU16 pu16ADCdata  = NULL;
			pu16ADCdata = &u16ADCdata;
			tU32 u32NoOfBytesRead = (tU32)OSAL_s32IORead(m_hTPS7B7701_IOHandler,(tPS8)pu16ADCdata ,TUNMSTR_ANTDIAG_ADC_READ_2BYTE);
			ETG_TRACE_USR1((" vReadTPS7B7701_ADCVal()->TPS7B7701 IC ADC sample =%d u32NoOfBytesRead=%d",*pu16ADCdata,u32NoOfBytesRead));
			tU16 u16ADCData = *pu16ADCdata & 0x0FFF;
			tFloat fCurrent=fCalculateTPS7B7701Current(u16ADCData);
			vUpdateAntennaCur(enTPS7B7701_1,fCurrent);
		}
	}*/
}
tVoid tunmstr_antdiag_Measure::vReadTPS7B7702_1_1ADCVal(tenAntDiagClients enAntDiagClients)
{
	if((NULL!= m_potunmstr_antdiag_Manager)&& (NULL!= m_potunmstr_antdiag_Debounce)&& (NULL!= m_potunmstr_DiagLogClientHandler))
	{
		tBool blGPIOpinState=m_potunmstr_antdiag_Manager->blGetExtPwrSupEN_PinState(enTPS7B7702_1_1);
		if((blGPIOpinState==TUNMSTR_ANTDIAG_GPIO_STATE_OUTPUT)&&(IsShortCkt_TimeOver(enTPS7B7702_1_1)))
		{
			blGPIOpinState=m_potunmstr_antdiag_Manager->blTurnOnGPIOPin(OSAL_EN_TPS7B7702_EN_1);
			m_potunmstr_antdiag_Manager->vSetExtPwrSupEN_PinState(enTPS7B7702_1_1,blGPIOpinState);
			m_potunmstr_antdiag_Debounce->vClearShortWaitOverCounter(enTPS7B7702_1_1);
			m_potunmstr_DiagLogClientHandler->blUpdateAntennaStatus(m_potunmstr_antdiag_Manager->enGetAntDiagClient(enTPS7B7702_1_1),enAntennState_Unknown);
		}
		else if((blGPIOpinState==(tBool)TUNMSTR_ANTDIAG_GPIO_STATE_INPUT)&& (m_blTPS7B7702_1_1ADCChannelOpen==TRUE))
		{
			tPU16 pu16ADCdata  = NULL;
			tU16 u16ADCdata =0;
			pu16ADCdata = &u16ADCdata;
			tU32 u32NoOfBytesRead =(tU32)OSAL_s32IORead(m_hTPS7B7702_1_1IOHandler,(tPS8)pu16ADCdata ,TUNMSTR_ANTDIAG_ADC_READ_2BYTE);
			ETG_TRACE_USR1((" vReadTPS7B7702_1_1ADCVal()->TPS7B7702_1_1 IC ADC sample =%d u32NoOfBytesRead=%d",*pu16ADCdata,u32NoOfBytesRead));
			u16ADCdata = *pu16ADCdata& TUNMSTR_ANTDIAG_12BIT_VALUE;
			tFloat fCurrent=fCalculateTPS7B7702_1_1Current(u16ADCdata,enAntDiagClients);
			vUpdateAntennaCur(enTPS7B7702_1_1,fCurrent);
		}
	}
}
tVoid tunmstr_antdiag_Measure::vReadTPS7B7702_1_2ADCVal(tenAntDiagClients enAntDiagClients)
{
	if((NULL!=m_potunmstr_antdiag_Manager)&& (NULL!=m_potunmstr_antdiag_Debounce)&& (NULL!=m_potunmstr_DiagLogClientHandler))
	{
		tBool blGPIOpinState=m_potunmstr_antdiag_Manager->blGetExtPwrSupEN_PinState(enTPS7B7702_1_2);
		if((blGPIOpinState==TUNMSTR_ANTDIAG_GPIO_STATE_OUTPUT)&&(IsShortCkt_TimeOver(enTPS7B7702_1_2)))
		{
			if(m_potunmstr_antdiag_Manager->blIsB1A2S1Sample())
			{
			blGPIOpinState=m_potunmstr_antdiag_Manager->blTurnOnGPIOPin(OSAL_EN_TPS7B7702_EN_1);
			}
			else
			{
				blGPIOpinState=m_potunmstr_antdiag_Manager->blTurnOnGPIOPin(OSAL_EN_TPS7B7702_EN_2);
			}
			m_potunmstr_antdiag_Manager->vSetExtPwrSupEN_PinState(enTPS7B7702_1_2,blGPIOpinState);
			m_potunmstr_antdiag_Debounce->vClearShortWaitOverCounter(enTPS7B7702_1_2);
			m_potunmstr_DiagLogClientHandler->blUpdateAntennaStatus(m_potunmstr_antdiag_Manager->enGetAntDiagClient(enTPS7B7702_1_2),enAntennState_Unknown);
		}
		else if((blGPIOpinState==(tBool)TUNMSTR_ANTDIAG_GPIO_STATE_INPUT)&& (m_blTPS7B7702_1_2ADCChannelOpen==TRUE))
		{
			tPU16 pu16ADCdata  = NULL;
			tU16 u16ADCdata =0;
			pu16ADCdata = &u16ADCdata;
			tU32 u32NoOfBytesRead =(tU32)OSAL_s32IORead(m_hTPS7B7702_1_2IOHandler,(tPS8)pu16ADCdata ,TUNMSTR_ANTDIAG_ADC_READ_2BYTE);
			ETG_TRACE_USR1((" vReadTPS7B7702_1_2ADCVal()->TPS7B7702_1_2 IC ADC sample =%d u32NoOfBytesRead=%d",*pu16ADCdata,u32NoOfBytesRead));
			u16ADCdata = *pu16ADCdata & TUNMSTR_ANTDIAG_12BIT_VALUE;
			tFloat fCurrent=fCalculateTPS7B7702_1_2Current(u16ADCdata,enAntDiagClients);
			vUpdateAntennaCur(enTPS7B7702_1_2,fCurrent);
		}
	}
}
	/*NFLT pin state is not required to identify the antenna status,
	may be during issue debug, it might be required*/
/*tBool tunmstr_antdiag_Measure::blReadTPS7B7702_1_1_NFLTPinState()
{
	tBool blRet = false;
	OSAL_tIODescriptor hFd=OSAL_ERROR;

	hFd = OSAL_IOOpen(OSAL_C_STRING_DEVICE_GPIO, OSAL_EN_READWRITE);
	if ( OSAL_ERROR != hFd )
	{
		OSAL_tGPIODevID DevID = (OSAL_tGPIODevID) OSAL_EN_TPS7B7702_NFLT_1;
		OSAL_trGPIOData Data = {DevID, TRUE};
		Data.tId = (OSAL_tGPIODevID) OSAL_EN_TPS7B7702_NFLT_1;
		if( OSAL_ERROR != OSAL_s32IOControl ( hFd,OSAL_C_32_IOCTRL_GPIO_IS_STATE_ACTIVE,(intptr_t)&Data) )
		{
			blRet = Data.unData.bState;
		}
	}
	OSAL_s32IOClose(hFd);
	ETG_TRACE_USR4(("blReadTPS7B7702_1_1_NFLTPinState()-> NFLTPinState=%d",blRet));
	return blRet;
}*/
/*************************************************************************
 * FUNCTION:    vUpdateAntennaCur
 *
 * DESCRIPTION:	function to update the antenna current
 *				
 * PARAMETER:	tenExtAntSupType enExtAntSupType,tFloat fCurrent
 *
 * RETURNVALUE:	tVoid
 *************************************************************************/
tVoid tunmstr_antdiag_Measure::vUpdateAntennaCur
(tenCMExtPowerSupplyType enCMExtPowerSupplyType,tFloat fCurrent)
{
	if((m_potunmstr_antdiag_Debounce==NULL)|| (m_potunmstr_antdiag_Manager==NULL))
	{
		return;
	}
	tenAntDiagClients enAntDiagClient=m_potunmstr_antdiag_Manager->enGetAntDiagClient(enCMExtPowerSupplyType);
    switch(enCMExtPowerSupplyType)
	{
		case enInfineonTLF4277_1:
		{
			m_flnfineon_Cur=fCurrent;
			break;
		}
		case enInfineonTLF4277_2:
		{
			m_flnfineon2_Cur=fCurrent;
			break;
		}
		case enMAXIM_16946:
		{
			m_fMax_Cur=fCurrent;
			break;
		}
		case enELMOS1_1:
		{
			/*No offset adjustment is required for ELMOS current calculation  Fix for JIRA issue suzuki-17229*/
			/*tFloat fBatteryVg=m_potunmstr_antdiag_Manager->fGet_BatteryVg();
			if((fBatteryVg>8.0)&&(fBatteryVg<13.5))
			{
				fCurrent=fCurrent+TUNMSTR_ANTDIAG_ELMOS_ADD_AROUND_CUR;
			}*/
			m_fELMOS1_1Cur=fCurrent;
            break;
        }
        case enELMOS1_2:
		{
			m_fELMOS1_2Cur=fCurrent;
            break;
		}
		case enELMOS2_1:
		{
			m_fELMOS2_1Cur=fCurrent;
			break;
		}
		case enELMOS2_2:
		{
			m_fELMOS2_2Cur=fCurrent;
			break;
		}
		case enELMOS3_1:
		{
			m_fELMOS3_1Cur=fCurrent;
			break;
		}
		case enELMOS3_2:
		{
			m_fELMOS3_2Cur=fCurrent;
			break;
		}
		case enELMOS4_1:
		{
			m_fELMOS4_1Cur=fCurrent;
			break;
		}
		case enELMOS4_2:
		{
			m_fELMOS4_2Cur=fCurrent;
			break;
		}
		case enTPS7B7701_1:
		{
			m_fTPS7B7701_Cur=fCurrent;
			break;
		}
		case enTPS7B7702_1_1:
		{
			m_fTPS7B7702_1_1Cur=fCurrent;
			break;
		}
		case enTPS7B7702_1_2:
		{
			m_fTPS7B7702_1_2Cur=fCurrent;
			break;
		}
		case enMAX20084_1_1:
		{
			m_fMAX20084_1_1_Cur = fCurrent;
			break;
		}
		case enMAX20084_1_2:
		{
			m_fMAX20084_1_2_Cur = fCurrent;
			break;
		}
		case enMAX20084_2_1:
		{
			m_fMAX20084_2_1_Cur = fCurrent;
			break;
		}
		case enMAX20084_2_2:
		{
			m_fMAX20084_2_2_Cur = fCurrent;
			break;
		}
		case enMAX20084_3_1:
		{
			m_fMAX20084_3_1_Cur = fCurrent;
			break;
		}
		case enMAX20084_3_2:
		{
			m_fMAX20084_3_2_Cur = fCurrent;
			break;
		}
		default:
		{
			break;
		}
	}
m_potunmstr_antdiag_Debounce->vHandleAntennaCurrent(enAntDiagClient,fCurrent);
}
/*************************************************************************
 * FUNCTION:     IsShortCkt_TimeOver
 *
 * DESCRIPTION:	This function returns true if the short circuit wait over time is done for request client 
 *
 * PARAMETER:	tVoid
 *
 * RETURNVALUE:	tVoid
 *************************************************************************/
tBool tunmstr_antdiag_Measure::IsShortCkt_TimeOver(tenCMExtPowerSupplyType enCMExtPowerSupplyType)const
{
	tBool blRet =FALSE;
	if((NULL==m_potunmstr_antdiag_Manager)||(NULL==m_potunmstr_antdiag_Debounce) ||(NULL==m_potunmstr_DiagLogClientHandler))
	{
		return blRet;
	}
	if(enAntennState_Short==m_potunmstr_DiagLogClientHandler->u8GetAntenanstate(m_potunmstr_antdiag_Manager->enGetAntDiagClient(enCMExtPowerSupplyType)))
	{
		blRet=m_potunmstr_antdiag_Debounce->IsShortCktTimeOver(m_potunmstr_antdiag_Manager->enGetAntDiagClient(enCMExtPowerSupplyType));
	}
	return blRet;
}
/*************************************************************************
 * FUNCTION:     fCalculateMaxICCurrent
 *
 * DESCRIPTION:	This function will read the MAX IC ADC channel 
 *
 * PARAMETER:	tVoid
 *
 * RETURNVALUE:	tVoid
 *************************************************************************/
tFloat tunmstr_antdiag_Measure::fCalculateMaxICCurrent(tU16 u16adcSample)
{
	if((NULL==m_potunmstr_antdiag_Debounce)||(NULL==m_potunmstr_antdiag_Manager))
	{
		return 0;
	}
	tFloat fCurrent=0;
	#ifdef VARIANT_S_FTR_ENABLE_FEATURE_GMMY16
		fCurrent=(((((TUNMSTR_ANTDIAG_MAX_MY16_VREF * u16adcSample)/TUNMSTR_ANTDIAG_TENBIT_VALUE)-TUNMSTR_ANTDIAG_MAX_MY16_SUB_FACTOR)/TUNMSTR_ANTDIAG_MAX_MY16_RSENSE)/TUNMSTR_ANTDIAG_MAX_MY16_AOUT_GAIN);
		m_fMaxIC_CalculatedCur=fCurrent;
		tenMaxICGPIOPinState enMaxICGPIOPinStateen=enFindMaxICGPIOPinState(fCurrent);
		if(enMaxICGPIOPinStateen ==enMaxICOpenGPIOPinActive)
		{
			fCurrent=0.0;
		}
		else if(enMaxICGPIOPinStateen==enMaxICShortGPIOPinActive)
		{
			fCurrent=255.0;
		}
		else
		{
			fCurrent=(m_potunmstr_antdiag_Debounce->u8GetOpen2AmpCurrent(m_potunmstr_antdiag_Manager->enGetAntDiagClient(enMAXIM_16946)))+1.0;
		}
	#else
		fCurrent= (tFloat) (((u16adcSample*3.3)/(TUNMSTR_ANTDIAG_DAB_CURRENT_CAL_REDUCTION_FACT))* 58.1);
	#endif
	ETG_TRACE_USR1(("***********fCalculateMaxICCurrent()::fCurrent=%f m_fMaxIC_CalculatedCur=%f",fCurrent,m_fMaxIC_CalculatedCur));
	return fCurrent;
}
/*************************************************************************
* FUNCTION:     enReadMaxICShortGPIOPinState
*
* DESCRIPTION:  Function to read the Max IC short GPIO pin state
*************************************************************************/
tenMaxICGPIOPinState tunmstr_antdiag_Measure::enReadMaxICShortGPIOPinState()const
{
	tenMaxICGPIOPinState enMaxICGPIOPinState =enMaxICShortndOpenGPIOPinInActive;
	OSAL_tIODescriptor hFd=OSAL_ERROR;
	//Read short Pin state
	hFd = OSAL_IOOpen(OSAL_C_STRING_DEVICE_GPIO, OSAL_EN_READWRITE);
	if ( OSAL_ERROR != hFd )
	{
		OSAL_tGPIODevID DevID = (OSAL_tGPIODevID) OSAL_EN_MAX16946_SC;
		OSAL_trGPIOData Data = {DevID, TRUE};
		Data.tId = (OSAL_tGPIODevID) OSAL_EN_MAX16946_SC;
		#ifndef VARIANT_S_FTR_ENABLE_FEATURE_PSA_RCC
		if ( OSAL_ERROR != OSAL_s32IOControl ( hFd,OSAL_C_32_IOCTRL_GPIO_SET_INPUT,(intptr_t) DevID) )
		#endif
		{
			if( OSAL_ERROR != OSAL_s32IOControl ( hFd,OSAL_C_32_IOCTRL_GPIO_IS_STATE_ACTIVE,(intptr_t)&Data) )
			{
				if(!Data.unData.bState)
				{
					enMaxICGPIOPinState=enMaxICShortGPIOPinActive;
				}
			}
		}
	}
	OSAL_s32IOClose(hFd);
	ETG_TRACE_USR4(("enReadMaxICShortGPIOPinState()-> MaxICShortGPIOPinState=%02x",
	ETG_CENUM(tenMaxICGPIOPinState,enMaxICGPIOPinState)));
	return enMaxICGPIOPinState;
}
/*************************************************************************
* FUNCTION:     enReadMaxICShortGPIOPinState
*
* DESCRIPTION:  Function to read the Max IC Open GPIO pin state
*************************************************************************/
tenMaxICGPIOPinState tunmstr_antdiag_Measure::enReadMaxICOpenGPIOPinState()const
{
	tenMaxICGPIOPinState enMaxICGPIOPinState =enMaxICShortndOpenGPIOPinInActive;
	OSAL_tIODescriptor hFd=OSAL_ERROR;
	//Read short Pin state
	hFd = OSAL_IOOpen(OSAL_C_STRING_DEVICE_GPIO, OSAL_EN_READWRITE);
	if ( OSAL_ERROR != hFd )
	{
		OSAL_tGPIODevID DevID = (OSAL_tGPIODevID) OSAL_EN_MAX16946_OL;
		OSAL_trGPIOData Data = {DevID, TRUE};
		Data.tId = (OSAL_tGPIODevID) OSAL_EN_MAX16946_OL;
		#ifndef VARIANT_S_FTR_ENABLE_FEATURE_PSA_RCC
		if ( OSAL_ERROR != OSAL_s32IOControl ( hFd,OSAL_C_32_IOCTRL_GPIO_SET_INPUT,(intptr_t) DevID) )
#endif
		{
			if( OSAL_ERROR != OSAL_s32IOControl ( hFd,OSAL_C_32_IOCTRL_GPIO_IS_STATE_ACTIVE,(intptr_t)&Data) )
			{
				if(!Data.unData.bState)
				{
					enMaxICGPIOPinState=enMaxICOpenGPIOPinActive;
				}
			}
		}
	}
	OSAL_s32IOClose(hFd);
	ETG_TRACE_USR4(("enReadMaxICOpenGPIOPinState()-> MaxICOpenGPIOPinState=%02x",
	ETG_CENUM(tenMaxICGPIOPinState,enMaxICGPIOPinState)));
	return enMaxICGPIOPinState;
}
/*************************************************************************
* FUNCTION:     enReadMaxICShortGPIOPinState
*
* DESCRIPTION:  Function to find out the the Max IC Open GPIO and Short pin state

	extend the concept (confirmed by Frank Pape and Michael Frerichs):
-	For open and short detection, only the status of the MAX open pin and MAX short pin is evaluated.
-	If open is detected, the pin will not switch back to ok automatically. Therefore, the antenna power supply
	has to be deactivated as soon as the calculated current is above 10 mA for and has to be activated after a wait time of at least 150s.
-	If short is detected, the antenna power supply must not be switched off. Short state is completely handled by the IC 
	and the short pin state will change to ok, it the system recovers from the short situation.
-	The currently configured thresholds for open (5mA) and short (40 mA) must not be evaluated. Open and short detection will only be done based on the pin states.
-	Added 01.04.2015: The monitor cycle is 1 s.
-	Added 01.04.2015: For the qualified status, the conditions has to be stable for 3 consecutive Monitoring Cycles

*************************************************************************/
tenMaxICGPIOPinState tunmstr_antdiag_Measure::enFindMaxICGPIOPinState(tFloat fCurrent)
{
	tenMaxICGPIOPinState enMaxICGPIOPinState =enMaxICShortndOpenGPIOPinInActive;
	if(m_potunmstr_antdiag_Manager!=NULL)
	{
		if(enMaxICShortGPIOPinActive==enReadMaxICShortGPIOPinState())
		{
			enMaxICGPIOPinState=enMaxICShortGPIOPinActive;
		}
		else
		{
			if((fCurrent>10.0) && (enMaxICOpenGPIOPinActive==enReadMaxICOpenGPIOPinState()))
			{
				//Turn Off and Turn on the Max IC and read the Open pin state
				OSAL_tGPIODevID DevIDShdn = (OSAL_tGPIODevID) OSAL_EN_MAX16946_SHDN;
				OSAL_trGPIOData Data = {DevIDShdn, FALSE};
				OSAL_tIODescriptor hFd=OSAL_ERROR;
				hFd = OSAL_IOOpen(OSAL_C_STRING_DEVICE_GPIO, OSAL_EN_READWRITE);
				tS32 _s32RetVal=OSAL_s32IOControl(hFd, OSAL_C_32_IOCTRL_GPIO_SET_INACTIVE_STATE, (intptr_t) &Data);
				if(OSAL_OK	==_s32RetVal)
				{
					m_potunmstr_antdiag_Manager->vSetExtPwrSupEN_PinState(enMAXIM_16946,false);
					OSAL_trGPIOData oData = {DevIDShdn, TRUE};
					tS32 s32RetVal=OSAL_s32IOControl(hFd, OSAL_C_32_IOCTRL_GPIO_SET_ACTIVE_STATE, (intptr_t) &oData);
					if(OSAL_OK==s32RetVal)
					{
						m_potunmstr_antdiag_Manager->vSetExtPwrSupEN_PinState(enMAXIM_16946,true);
					}
				}
			}
			//Read Open Pin State
			enMaxICGPIOPinState=enReadMaxICOpenGPIOPinState();
		}
		ETG_TRACE_USR4(("enFindMaxICGPIOPinState()-> MaxICGPIOPinState=%02x",
				ETG_CENUM(tenMaxICGPIOPinState,enMaxICGPIOPinState)));
	}
	return enMaxICGPIOPinState;
}
/*************************************************************************
* FUNCTION:     fGetMaxIC_CalculatedCur
*
* DESCRIPTION:  Function to get the Max IC calcualted current
*************************************************************************/
tFloat tunmstr_antdiag_Measure::fGetMaxIC_CalculatedCur()
{
	ETG_TRACE_USR4(("fGetMaxIC_CalculatedCur()-> m_fMaxIC_CalculatedCur=%f",m_fMaxIC_CalculatedCur));
	return m_fMaxIC_CalculatedCur;
}
tVoid tunmstr_antdiag_Measure::vClearAllInternalState()
{
	m_hInfineonIOHandler=OSAL_ERROR;
	m_hInfineon2IOHandler=OSAL_ERROR;
	m_hMAXIOHandler=OSAL_ERROR;
	m_blnfineonADCChannel=FALSE;
	m_blnfineon2ADCChannel=FALSE;
	m_bMAXADCChannel=FALSE;
	m_flnfineon_Cur=0;
	m_flnfineon2_Cur=0;
	m_fMax_Cur=0;
	m_fELMOS1_1Cur=0;
	m_fELMOS1_2Cur=0;
	m_fELMOS2_1Cur=0;
	m_fELMOS2_2Cur=0;
	m_fELMOS3_1Cur=0;
	m_fELMOS3_2Cur=0;
	m_fELMOS4_1Cur=0;
	m_fELMOS4_2Cur=0;
	m_fMaxIC_CalculatedCur=0;
	m_fMaxIC_CalculatedCur=0;
	m_hTPS7B7701_IOHandler=OSAL_ERROR;
	m_hTPS7B7702_1_1IOHandler=OSAL_ERROR;
	m_hTPS7B7702_1_2IOHandler=OSAL_ERROR;
	m_blTPS7B7701ADCChannelOPen=FALSE;
	m_blTPS7B7702_1_1ADCChannelOpen=FALSE;
	m_blTPS7B7702_1_2ADCChannelOpen=FALSE;
	m_fTPS7B7701_Cur=0;
	m_fTPS7B7702_1_1Cur=0;
	m_fTPS7B7702_1_2Cur=0;
}
