/************************************************************************
* FILE:           tunmstr_antdiag_Debounce.cpp
* PROJECT:        G3g
* SW-COMPONENT:   FC_TunerMaster.
*----------------------------------------------------------------------
*
* DESCRIPTION:  Implementation of the Class tunmstr_antdiag_Debounce
*
*----------------------------------------------------------------------
* COPYRIGHT: (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
* HISTORY:
* Date      | Author                            | Modification
* 25/05/2012| ECV4 ( RBEI ) Priya Prabhuswamy   | Initial version.
* 25.04.13  | NGP1KOR							| First version of the G3g after porting 
* 03.03.14  | ECV3 ( RBEI ) Priya Prabhuswamy   | adaptation for Different Type of External Power supply
* 17.03.14  | ECV3 ( RBEI ) Priya Prabhuswamy   | Default configuration for MIB
*************************************************************************/
// System includes.
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#include "tunmstr_antdiag_Debounce.h"

#ifndef tunmstr_DiagLogClientHandler_H
#include "tunmstr_DiagLogClientHandler.h"
#endif

#ifndef tunmstr_antdiag_Manager_H
#include "tunmstr_antdiag_Manager.h"
#endif

#include "tunmstr_antdiag_i2cHandler.h"

#ifndef TUNERMASTER_TRACE_H
#include "tunermaster_trace.h"
#endif

#include "tunmstr_MsgFromHMI.h"

#ifndef tunmaster_Config_H
#include "tunermaster_Config.h"
#endif
#ifndef tunmstr_antdiag_Measure_H
#include "tunmstr_antdiag_Measure.h"
#endif

#include "kdsconfig/clConfigInterface.h"

// ETG Trace Defines
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TUNMSTR_TRACE_CLASS_ANTDIAG
#include "trcGenProj/Header/tunmstr_antdiag_Debounce.cpp.trc.h"
#endif

/*************************************************************************
*
* FUNCTION:     tunmstr_antdiag_Debounce::tunmstr_antdiag_Debounce( )
*
* DESCRIPTION:  Constructor
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*
*************************************************************************/
tunmstr_antdiag_Debounce::tunmstr_antdiag_Debounce( )
{
	m_potunmstr_antdiag_Manager=NULL;
	m_potunmstr_DiagLogClientHandler=NULL;
	m_potunmstr_antdiag_i2cHandler=NULL;
	m_potunmstr_MsgFromHMI=NULL;
	m_potunmstr_antdiag_Measure=NULL;
	m_poclConfigInterface = NULL;
	for (tU8 i=0;i<TUNMSTR_ANTDIAG_MAX_NUMBEROF_ANTDIAGCLIENT;i++)
	{
		m_u8ShortCounter[i]=0;
		m_u8OpenCounter[i]=0;
		m_u8CloseCounter[i]=0;
		m_u8OpenToAmpCounter[i]=0;
		//Antenna Current
		m_fAntennacurrent[i]=0.0;
		m_u8ShortWaitOverCounter[i]=0;
		m_u8AntennaStatus[i]=enAntennState_Unknown;
			//Current Range
		m_u8OpenCur[i]=TUNMSTR_ANTDIAG_OpenCur;
		m_u8OpenClearCur[i]=TUNMSTR_ANTDIAG_OpenClearCur;
		m_u8Open2AmpCur[i]=0;
		m_u8Open2AmpClr_LwCur[i]=0;
		m_u8Open2AmpClr_HgCur[i]=0;
		m_u8ShortCur[i]=TUNMSTR_ANTDIAG_ShortCur;
		m_u8ShortClearCur[i]=TUNMSTR_ANTDIAG_ShortClearCur;
		//Monitor cycel
		m_u8NoOfMon_Cycle[i]=1;
		//Short Ckt Wait for period
		m_u8ShortCkt_Wait[i]=1;
		m_u8AntennaOutPutVg[i]= 0;
		m_u8AntennaWarningCur[i]=0;
	}
}
/*************************************************************************
*
* FUNCTION:     tunmstr_antdiag_Debounce::~tunmstr_antdiag_Debounce( )
*
* DESCRIPTION:  Destructor
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*
*************************************************************************/
tunmstr_antdiag_Debounce::~tunmstr_antdiag_Debounce( )
{
	m_potunmstr_antdiag_Manager=NULL;
	m_potunmstr_DiagLogClientHandler=NULL;
	m_potunmstr_antdiag_i2cHandler=NULL;
	m_potunmstr_MsgFromHMI=NULL;
	m_potunmstr_antdiag_Measure=NULL;
	m_poclConfigInterface = NULL;
	for (tU8 i=0;i<TUNMSTR_ANTDIAG_MAX_NUMBEROF_ANTDIAGCLIENT;i++)
	{
		m_u8ShortCounter[i]=0;
		m_u8OpenCounter[i]=0;
		m_u8CloseCounter[i]=0;
		m_u8OpenToAmpCounter[i]=0;
		//Antenna Current
		m_fAntennacurrent[i]=0.0;
		m_u8ShortWaitOverCounter[i]=1;
		m_u8AntennaStatus[i]=(tU8)enAntennState_Unknown;
			//Current Range
		m_u8OpenCur[i]=0;
		m_u8OpenClearCur[i]=0;
		m_u8Open2AmpCur[i]=0;
		m_u8Open2AmpClr_LwCur[i]=0;
		m_u8Open2AmpClr_HgCur[i]=0;
		m_u8ShortCur[i]=0;
		m_u8ShortClearCur[i]=0;
		//Monitor cycel
		m_u8NoOfMon_Cycle[i]=0;
		//Short Ckt Wait for period
		m_u8ShortCkt_Wait[i]=0;
		m_u8AntennaOutPutVg[i]= 0;
		m_u8AntennaWarningCur[i]= 0;
	}
}
/*************************************************************************
*
* FUNCTION:    vSet_tunmstr_antdiag_Manager( tunmstr_antdiag_Manager* potunmstr_antdiag_Manager )
*
* DESCRIPTION: update tunmstr_antdiag_Manager pointer variable
*
* PARAMETER:   tunmstr_antdiag_Manager* potunmstr_antdiag_Manager
*
* RETURNVALUE:  void
*************************************************************************/
tVoid tunmstr_antdiag_Debounce::vSet_tunmstr_antdiag_Manager(tunmstr_antdiag_Manager* potunmstr_antdiag_Manager)
{
	if( potunmstr_antdiag_Manager != NULL )
	{
		m_potunmstr_antdiag_Manager = ( tunmstr_antdiag_Manager* )potunmstr_antdiag_Manager;
	}
}
/*************************************************************************
* FUNCTION:    vSet_tunmstr_DiagLogClientHandler( tunmstr_DiagLogClientHandler* potunmstr_DiagLogClientHandler )
*
* DESCRIPTION: update tunmstr_DiagLogClientHandler pointer variable
*
* PARAMETER:   tunmstr_DiagLogClientHandler* potunmstr_DiagLogClientHandler
*
* RETURNVALUE:  void
*************************************************************************/
tVoid tunmstr_antdiag_Debounce::vSet_tunmstr_DiagLogClientHandler_Ptr(tunmstr_DiagLogClientHandler* potunmstr_DiagLogClientHandler)
{
	if( potunmstr_DiagLogClientHandler != NULL )
	{
		m_potunmstr_DiagLogClientHandler = ( tunmstr_DiagLogClientHandler* )potunmstr_DiagLogClientHandler;
	}
}
/******************************************************************************
*	FUNCTION:      vSet_tunmstr_antdiag_i2cHandler

*	DESCRIPTION:   function to update m_potunmstr_antdiag_i2cHandler

*	PARAMETERS:    tunmstr_antdiag_i2cHandler* Ptunmstr_antdiag_i2cHandler

*	RETURN TYPE:   tVoid
*********************************************************************************/
tVoid tunmstr_antdiag_Debounce::vSet_tunmstr_antdiag_i2cHandler
(tunmstr_antdiag_i2cHandler* Ptunmstr_antdiag_i2cHandler)
{
	m_potunmstr_antdiag_i2cHandler=Ptunmstr_antdiag_i2cHandler;
}
/******************************************************************************
*	FUNCTION:      tunmstr_antdiag_Debounce

*	DESCRIPTION:   function to update m_potunmstr_MsgFromHMI

*	PARAMETERS:    tunmstr_MsgFromHMI* potunmstr_MsgFromHMI 

*	RETURN TYPE:   tVoid
*********************************************************************************/
tVoid tunmstr_antdiag_Debounce::vSet_tunmstr_MsgFromHMI_Ptr(tunmstr_MsgFromHMI* potunmstr_MsgFromHMI )
{
	if( potunmstr_MsgFromHMI != NULL )
	{
		m_potunmstr_MsgFromHMI = ( tunmstr_MsgFromHMI* )potunmstr_MsgFromHMI;
	}

}
/*************************************************************************
* FUNCTION:    vSet_tunmstr_antdiag_Measure_Ptr( tunmstr_antdiag_Measure* potunmstr_antdiag_Measure )
*
* DESCRIPTION: update tunmstr_antdiag_Measure pointer variable
*
* PARAMETER:   tunmstr_antdiag_Measure* potunmstr_antdiag_Measure
*
* RETURNVALUE:  void
*************************************************************************/
tVoid tunmstr_antdiag_Debounce::vSet_tunmstr_antdiag_Measure_Ptr(tunmstr_antdiag_Measure* potunmstr_antdiag_Measure)
{
	if( potunmstr_antdiag_Measure != NULL )
	{
		m_potunmstr_antdiag_Measure = ( tunmstr_antdiag_Measure* )potunmstr_antdiag_Measure;
	}
}


/*************************************************************************
* FUNCTION:    vSet_tunmstr_ConfigInterface_Ptr
*
* DESCRIPTION: 
*
* PARAMETER:   clConfigInterface* poclConfigInterface
*
* RETURNVALUE:  void
*************************************************************************/
tVoid tunmstr_antdiag_Debounce::vSet_tunmstr_ConfigInterface_Ptr
(clConfigInterface* poclConfigInterface)
{
	if( poclConfigInterface != NULL )
	{
		m_poclConfigInterface = poclConfigInterface;
	}

}


/*************************************************************************
 * FUNCTION:    vUpdateAntennaParam_FromKDS()
 *
 * DESCRIPTION: function to update the member variables by reading the Antenna Parameter set of KDS
 *
 * PARAMETER:   void
 *
 * RETURNVALUE:  void
 *************************************************************************/
tVoid tunmstr_antdiag_Debounce::vUpdateAntennaParam_FromKDS(tunerVector<tU8> &AntennaPar)
{
	if((NULL==m_poclConfigInterface) || (NULL ==m_potunmstr_antdiag_Manager))
	{
		return;
	}
	if(AntennaPar.size()> enAntennaParameter_idx_DTV4WarningCur)
	{
		tU16 u16KDSindex= enAntennaParameter_idx_FM1OpenCur;
		for(tU8 u8index=(tU8)enkdsFM1;u8index<enkdsNone;)
		{
			tU8 u8AntDiagClientInx= (tU8)m_potunmstr_antdiag_Manager->enGetAntDaigClientInx((tenAntDiagKDS)u8index);
			m_u8OpenCur[u8AntDiagClientInx]=AntennaPar.at(u16KDSindex);
			u16KDSindex++;
			m_u8OpenClearCur[u8AntDiagClientInx]=AntennaPar.at(u16KDSindex);
			u16KDSindex++;u16KDSindex++;u16KDSindex++;
			m_u8Open2AmpCur[u8AntDiagClientInx]=AntennaPar.at(u16KDSindex);
			u16KDSindex++;
			m_u8Open2AmpClr_LwCur[u8AntDiagClientInx]=AntennaPar.at(u16KDSindex);
			u16KDSindex++;
			m_u8Open2AmpClr_HgCur[u8AntDiagClientInx]=AntennaPar.at(u16KDSindex);
			u16KDSindex++;u16KDSindex++;u16KDSindex++;
			m_u8ShortCur[u8AntDiagClientInx]=AntennaPar.at(u16KDSindex);
			u16KDSindex++;
			m_u8ShortClearCur[u8AntDiagClientInx]=AntennaPar.at(u16KDSindex);
			u16KDSindex++;u16KDSindex++;u16KDSindex++;u16KDSindex++;
			if(0xff==AntennaPar.at(u16KDSindex))
			{
				m_u8NoOfMon_Cycle[u8AntDiagClientInx]=1;
			}
			else
			{
				m_u8NoOfMon_Cycle[u8AntDiagClientInx]=AntennaPar.at(u16KDSindex);
			}
			u16KDSindex++;
			m_u8ShortCkt_Wait[u8AntDiagClientInx]=AntennaPar.at(u16KDSindex);
			u16KDSindex++;u16KDSindex++;
			m_u8AntennaOutPutVg[u8AntDiagClientInx]= AntennaPar.at(u16KDSindex);
			u16KDSindex++;
			m_u8AntennaWarningCur[u8AntDiagClientInx]= AntennaPar.at(u16KDSindex);
			u8index++;
			u16KDSindex= (tU16)(enAntennaParameter_idx_FM1OpenCur +(u8index*TUNMSTR_ANTDIAG_DIFFERENCE_BTW_ANT_CLIENT_INDEX));
			std::cout<<"u16KDSindex="<<u16KDSindex<<std::endl;
		}
		// DAB antenna open load has to be set to 8 mA. Bugfix for PSARCCB-11288
#ifdef VARIANT_S_FTR_ENABLE_FEATURE_PSA_RCC
	m_u8OpenCur[enDAB1]		=TUNMSTR_ANTDIAG_PSA_DAB_OPENCUR_RANGE;
	m_u8OpenClearCur[enDAB1] =TUNMSTR_ANTDIAG_PSA_DAB_OPENCUR_RANGE;
	m_u8Open2AmpCur[enDAB1]	=TUNMSTR_ANTDIAG_PSA_DAB_OPENCUR_RANGE;
	m_u8Open2AmpClr_LwCur[enDAB1] =TUNMSTR_ANTDIAG_PSA_DAB_OPENCUR_RANGE;
	m_u8Open2AmpClr_HgCur[enDAB1] =TUNMSTR_ANTDIAG_PSA_DAB_OPENCUR_RANGE;
#endif

		for(tU8 u8index=(tU8)enFM1;u8index<(tU8)enClientNone;)
		{
			ETG_TRACE_USR4(("****************************Client [%02x ] **********************************",ETG_CENUM(tenAntDiagClients,u8index)));
			ETG_TRACE_USR4(("OpenCur = %d		OpenClear         = %d",
					m_u8OpenCur[u8index],m_u8OpenClearCur[u8index]));
			ETG_TRACE_USR4(("Open2AmpCur = %d		Open2AmpClr_LwCur = %d		Open2AmpClr_HgCur =%d",
					m_u8Open2AmpCur[u8index],m_u8Open2AmpClr_LwCur[u8index],m_u8Open2AmpClr_HgCur[u8index]));
			ETG_TRACE_USR4(("ShortCur    = %d		ShortClearCur     = %d",
					m_u8ShortCur[u8index],m_u8ShortClearCur[u8index]));
			ETG_TRACE_USR4(("WarningCur    = %d		OutputVg     = %d",
					m_u8AntennaWarningCur[u8index],m_u8AntennaOutPutVg[u8index]));
			ETG_TRACE_USR4(("NoOfMon_Cycle=%d        ShortCkt_Wait=%d",
					m_u8NoOfMon_Cycle[u8index],m_u8ShortCkt_Wait[u8index]));
			u8index=(tU8)(u8index+1);
			ETG_TRACE_USR4(("****************************Done **********************************"));
		}
	}
}
/*************************************************************************
* FUNCTION:     IsCounterReachMaxLmt
*
* DESCRIPTION:	 This function will check any of the FM Counters has reached the Maximum limit
*				
* PARAMETER:	 tVoid
*
* RETURNVALUE:	 tBool blCounterReachMaxLmt
*************************************************************************/
tBool tunmstr_antdiag_Debounce::IsCounterReachMaxLmt(tenAntDiagClients enAntDiagClient)
{
	tBool blCounterReachMaxLmt =TRUE;
	if (m_u8OpenCounter[enAntDiagClient] ==m_u8NoOfMon_Cycle[enAntDiagClient])
	{
		ETG_TRACE_USR4(("IsCounterReachMaxLmt()-> %02x Open counter reached the max Lmt",
			ETG_CENUM(tenAntDiagClients,enAntDiagClient)));
		m_u8AntennaStatus[enAntDiagClient]=(tU8)enAntennState_Open;
	}
	else if (m_u8CloseCounter[enAntDiagClient] ==m_u8NoOfMon_Cycle[enAntDiagClient])
	{
		ETG_TRACE_USR4(("IsCounterReachMaxLmt()-> %02x close counter reached the max Lmt",
			ETG_CENUM(tenAntDiagClients,enAntDiagClient)));
		m_u8AntennaStatus[enAntDiagClient]=(tU8)enAntennState_Ok;
	}
	else if (m_u8ShortCounter[enAntDiagClient] ==m_u8NoOfMon_Cycle[enAntDiagClient])
	{
		m_u8AntennaStatus[enAntDiagClient]=(tU8)enAntennState_Short;
		ETG_TRACE_USR4(("IsCounterReachMaxLmt()-> %02x short counter reached the max Lmt" ,
			ETG_CENUM(tenAntDiagClients,enAntDiagClient)));
	}
	else if (m_u8OpenToAmpCounter[enAntDiagClient] ==m_u8NoOfMon_Cycle[enAntDiagClient])
	{
		m_u8AntennaStatus[enAntDiagClient]=(tU8)enAntennState_OpentoAmplifier;
		ETG_TRACE_USR4(("IsCounterReachMaxLmt()-> %02x OpentoAmp counter reached the max Lmt" ,
			ETG_CENUM(tenAntDiagClients,enAntDiagClient)));
	}
	else
	{
		ETG_TRACE_USR4(("IsCounterReachMaxLmt()->None %02x of the counters reached max limit",
			ETG_CENUM(tenAntDiagClients,enAntDiagClient)));
		blCounterReachMaxLmt=FALSE;
	}
    return   blCounterReachMaxLmt;
}
/*************************************************************************
* FUNCTION:    vClearCounter
*
* DESCRIPTION:	This function will clear all the antenna diagnosis counter
*				
* PARAMETER:	tU8 u8WhichClient,tU8 u8Whichantenna,tU8 u8AntennaStatus
* RETURNVALUE:	tVoid
*************************************************************************/
tVoid tunmstr_antdiag_Debounce::vClearCounter(tenAntDiagClients enAntDiagClient)
{
	if(enAntDiagClient>=enClientNone)
	{
		return;
	}
	else
	{
		m_u8ShortCounter[enAntDiagClient]=0;
		m_u8OpenCounter[enAntDiagClient]=0;
		m_u8CloseCounter[enAntDiagClient]=0;
		m_u8OpenToAmpCounter[enAntDiagClient]=0;
	}
}
tVoid tunmstr_antdiag_Debounce::vHandleNewAntennaState(tenAntDiagClients enAntDiagClient,tFloat fCurrent)
{
    if((enAntDiagClient>=enClientNone) ||(NULL==m_potunmstr_DiagLogClientHandler))
	{
		return;
	}
    m_fAntennacurrent[enAntDiagClient]=fCurrent;
    m_potunmstr_DiagLogClientHandler->vHandleErrorLog(enAntDiagClient,m_u8AntennaStatus[enAntDiagClient],fCurrent);
}
tVoid tunmstr_antdiag_Debounce::vHandleAntennaStateShort(tenAntDiagClients enAntDiagClient)
{
    if((NULL!=m_potunmstr_antdiag_Manager) && (NULL!=m_potunmstr_antdiag_i2cHandler))
	{
	tenCMExtPowerSupplyType enCMExtPowerSupplyType=m_potunmstr_antdiag_Manager->enGetExtPowerSupplyType(enAntDiagClient);
	if((ISPOWERSUPPLY_ELMOS(enCMExtPowerSupplyType) )|| (ISPOWERSUPPLY_MAX20084(enCMExtPowerSupplyType)))
    {
		m_potunmstr_antdiag_i2cHandler->vSetOutputEnabled(enCMExtPowerSupplyType,false);
		if(!m_potunmstr_antdiag_i2cHandler->blEnableOutput(enCMExtPowerSupplyType))
		{
			m_potunmstr_antdiag_i2cHandler->vSetOutputEnabled(enCMExtPowerSupplyType,true);
		}
    }
	else
	{
		//Turn off the GPIO pin
		tBool blRes=m_potunmstr_antdiag_Manager->blTurnOffGPIOPin(m_potunmstr_antdiag_Manager->enGetOSALGPIOPinName(enCMExtPowerSupplyType));
		m_potunmstr_antdiag_Manager->vSetExtPwrSupEN_PinState(enCMExtPowerSupplyType,!(blRes));
	}
	}
}
/*************************************************************************
* FUNCTION:     vHandleAntennaCurrent
*
* DESCRIPTION:	 function to handle Current 
*				
* PARAMETER:	tU8 u8index tFloat fCurrent
*
* RETURNVALUE:	 tVoid
*************************************************************************/
tVoid tunmstr_antdiag_Debounce::vHandleAntennaCurrent(tenAntDiagClients enAntDiagClient,tFloat fCurrent)
{
	ETG_TRACE_USR4(("vHandleAntennaCurrent()-> =%02x current  is =%f",
	ETG_CENUM(tenAntDiagClients,enAntDiagClient),fCurrent));
	if(enAntDiagClient>=enClientNone)
	{
		return;
	}
	switch(m_u8AntennaStatus[enAntDiagClient])
	{
	case enAntennState_Open:
		{
			vHandlePrevAntennaStateOpen(enAntDiagClient,fCurrent);
			break;
		}
	case enAntennState_Deactivated_HighVg:
	case enAntennState_Deactivated:
	case enAntennState_Ok:
	case enAntennState_Unknown:
#ifdef VARIANT_S_FTR_ENABLE_FEATURE_PSA_RCC
	case enAntennState_ELMOSOpen_Noresult:
#endif
	case TUNMSTR_ANTDIAG_ANTENNA_STATE_NUMBER_OF_ITEM:
		{
			vHandlePrevAntennaStateOkorUnk(enAntDiagClient,fCurrent);
			break;
		}
	case enAntennState_Short:
		{
			vHandlePrevAntennaStateShort(enAntDiagClient,fCurrent);
			break;
		}
	case enAntennState_OpentoAmplifier:
		{
			vHandlePrevAntStateOpen2Amp(enAntDiagClient,fCurrent);
			break;
		}
	default:
		{
			break;
		}
	}
	vSendPendingAntennaStateRequest(enAntDiagClient,fCurrent);
    if(IsCounterReachMaxLmt(enAntDiagClient))
    {
      vClearCounter(enAntDiagClient);
      vHandleNewAntennaState(enAntDiagClient,fCurrent);
    }

}
/*************************************************************************
* FUNCTION:     vSendPendingAntennaStateRequest
*
* DESCRIPTION:	 function to handle send the antenna status and current for the requested client
				 for FID FID_TUNMSTR_S_RF_ANTENNA_TEST
*				
* PARAMETER:	 tU8 u8index,tFloat fCurrent
*
* RETURNVALUE:	 tVoid
*************************************************************************/
tVoid tunmstr_antdiag_Debounce::vSendPendingAntennaStateRequest(tenAntDiagClients enAntDiagClient,tFloat fCurrent)
{
	if((NULL!=m_potunmstr_antdiag_Manager)&&(NULL!=m_potunmstr_antdiag_Measure)&& (NULL!=m_potunmstr_MsgFromHMI) && (NULL!=m_potunmstr_antdiag_i2cHandler))
	{
		ETG_TRACE_USR4(("vSendPendingAntennaStateRequest-> %02x: m_u8OpenCounter=%d, m_u8OpenToAmpCounter=%d, m_u8CloseCounter=%d, m_u8ShortCounter=%d",
				ETG_CENUM(tenAntDiagClients, enAntDiagClient),m_u8OpenCounter[enAntDiagClient], m_u8OpenToAmpCounter[enAntDiagClient],
				m_u8CloseCounter[enAntDiagClient],m_u8ShortCounter[enAntDiagClient]));
	tU8 u8AntennaState=enAntennState_Unknown;
    if(enAntDiagClient>=enClientNone)
    {
        return;
    }
    if(enAntDiagClient == (tenAntDiagClients)m_potunmstr_antdiag_Manager->u8GetPendingAntennaStateReqClient())
    {
		if(m_u8OpenCounter[enAntDiagClient]>0)
		{
			u8AntennaState=enAntennState_Open;
		}
		else if(m_u8OpenToAmpCounter[enAntDiagClient]>0)
		{
			u8AntennaState=enAntennState_OpentoAmplifier;
		}
		else if(m_u8CloseCounter[enAntDiagClient]>0)
		{
			u8AntennaState=enAntennState_Ok;
		}
		else if(m_u8ShortCounter[enAntDiagClient]>0)
		{
			u8AntennaState=enAntennState_Short;
		}
		else
		{
			
		}
		if(m_potunmstr_antdiag_Manager->enGetExtPowerSupplyType(enAntDiagClient)==enMAXIM_16946)
		{
			fCurrent=m_potunmstr_antdiag_Measure->fGetMaxIC_CalculatedCur();
		}
        m_potunmstr_antdiag_Manager->vSetPendingAntennaStateReqClient(enClientNone);
#ifdef VARIANT_S_FTR_ENABLE_FEATURE_PSA_RCC
	if(u8AntennaState ==enAntennState_Open)
	{
		tenCMExtPowerSupplyType enCMExtPowerSupplyType=m_potunmstr_antdiag_Manager->enGetExtPowerSupplyType(enAntDiagClient);
		if(ISPOWERSUPPLY_ELMOS(enCMExtPowerSupplyType))
		{
			trI2CRead otrI2CRead = m_potunmstr_antdiag_i2cHandler->trReadI2CReg(enSUP_VoltageReg,enCMExtPowerSupplyType);
			if(!otrI2CRead.blReadSucess)
			{
				m_potunmstr_MsgFromHMI->vSendRfAntenna_TestMethodResult(enAntDiagClient,enAntennState_Unknown,0.0);
				return;
			}
			tFloat fSupvoltage = otrI2CRead.u8Value;
			fSupvoltage=TUNMSTR_ANTDIAG_ELMOS_SUPPLY_VG_MUL_FAC*fSupvoltage;
			if(fSupvoltage<=TUNMSTR_ANTDIAG_ELMOS_SUPPLY_VG)
			{
				u8AntennaState=enAntennState_Unknown;
				ETG_TRACE_USR4(("Supply Voltage of ELMOS is %f <= %f V,hence antenna state is update as unknow",TUNMSTR_ANTDIAG_ELMOS_SUPPLY_VG,fSupvoltage));
			}
		}
	}
#endif
        m_potunmstr_MsgFromHMI->vSendRfAntenna_TestMethodResult(enAntDiagClient,u8AntennaState,fCurrent);
    }
	}
}
/*************************************************************************
* FUNCTION:     vHandlePrevAntennaStateOpen
*
* DESCRIPTION:	 function to handle FM1 Current 
*				
* PARAMETER:	 tFloat fCurrent
*
* RETURNVALUE:	 tVoid
*************************************************************************/
tVoid tunmstr_antdiag_Debounce::vHandlePrevAntennaStateOpen(tU8 u8index,tFloat fCurrent)
{
	ETG_TRACE_USR4(("vHandlePrevAntennaStateOpen()-> u8index=%02x fCurrent=%f",
	ETG_CENUM(tenAntDiagClients,u8index),fCurrent));
	if(u8index>=TUNMSTR_ANTDIAG_MAX_NUMBEROF_ANTDIAGCLIENT)
	{
		//Outof Range
		return;
	}
	//<15ma
	if(fCurrent<=m_u8OpenClearCur[u8index])
	{
		m_u8OpenCounter[u8index]++;
		m_u8OpenToAmpCounter[u8index]=0;
		m_u8CloseCounter[u8index]=0;
		m_u8ShortCounter[u8index] =0;
	}
	//15<i<30
	else if((fCurrent>m_u8OpenClearCur[u8index])&&(fCurrent<=m_u8Open2AmpCur[u8index]))
	{
		m_u8OpenToAmpCounter[u8index]++;
		m_u8OpenCounter[u8index]=0;
		m_u8CloseCounter[u8index]=0;
		m_u8ShortCounter[u8index] =0;
	}
	//30<i<110
	else if((fCurrent>m_u8Open2AmpCur[u8index])&&(fCurrent<=m_u8ShortCur[u8index]))
	{
		m_u8CloseCounter[u8index]++;
		m_u8OpenCounter[u8index]=0;
		m_u8OpenToAmpCounter[u8index]=0;
		m_u8ShortCounter[u8index] =0;
	}
	//>110
	else
	{
		m_u8ShortCounter[u8index]++;
		m_u8OpenCounter[u8index]=0;
		m_u8OpenToAmpCounter[u8index]=0;
		m_u8CloseCounter[u8index] =0;
	}
	ETG_TRACE_USR4(("vHandlePrevAntennaStateOpen()m_u8OpenCounter=%d m_u8ShortCounter=%d m_u8CloseCounter=%d",
		m_u8OpenCounter[u8index],m_u8ShortCounter[u8index],m_u8CloseCounter[u8index]));
}
/*************************************************************************
* FUNCTION:     vHandlePrevFM1AntennaStateOkorUnk
*
* DESCRIPTION:	 function to handle  FM1 Current 
*				
* PARAMETER:	 tFloat fCurrent
*
* RETURNVALUE:	 tVoid
*************************************************************************/
tVoid tunmstr_antdiag_Debounce::vHandlePrevAntennaStateOkorUnk(tU8 u8index,tFloat fCurrent)
{
	ETG_TRACE_USR4(("vHandlePrevAntennaStateOkorUnk()-> u8index=%02x fCurrent=%f",
	ETG_CENUM(tenAntDiagClients,u8index),fCurrent));
	if(u8index>=enClientNone)
	{
		//Outof Range
		return;
	}
	//<10ma
	if(fCurrent<m_u8OpenCur[u8index])
	{
		m_u8OpenCounter[u8index]++;
		m_u8OpenToAmpCounter[u8index]=0;
		m_u8CloseCounter[u8index]=0;
		m_u8ShortCounter[u8index] =0;
	}
	//10<i<30
	else if((fCurrent>=m_u8OpenCur[u8index])&&(fCurrent<=m_u8Open2AmpCur[u8index]))
	{
		m_u8OpenToAmpCounter[u8index]++;
		m_u8OpenCounter[u8index]=0;
		m_u8CloseCounter[u8index]=0;
		m_u8ShortCounter[u8index] =0;
	}
	//30<i<110
	else if((fCurrent>m_u8Open2AmpCur[u8index])&&(fCurrent<=m_u8ShortCur[u8index]))
	{
		m_u8CloseCounter[u8index]++;
		m_u8OpenCounter[u8index]=0;
		m_u8OpenToAmpCounter[u8index]=0;
		m_u8ShortCounter[u8index] =0;
	}
	//>110
	else
	{
		m_u8ShortCounter[u8index]++;
		m_u8OpenCounter[u8index]=0;
		m_u8OpenToAmpCounter[u8index]=0;
		m_u8CloseCounter[u8index] =0;
	}
}
tU8 tunmstr_antdiag_Debounce::u8GetShortCktWaitOverCounterValue(tenAntDiagClients enAntDiagClient)
{
	if(enAntDiagClient>=enClientNone)
	{
		//Outof Range
		return 0;
	}
	else
	{
		ETG_TRACE_USR4(("m_u8ShortCounter[enAntDiagClient]=%d m_u8ShortWaitOverCounter=%d",m_u8ShortCounter[enAntDiagClient],m_u8ShortWaitOverCounter[enAntDiagClient]));
		return m_u8ShortWaitOverCounter[enAntDiagClient];
	}
}
tU8 tunmstr_antdiag_Debounce::u8GetWaringCurrent(tenAntDiagClients enAntDiagClient)
{
	if(enAntDiagClient>=enClientNone)
	{
		//Outof Range
		return 0;
	}
	else
	{

		ETG_TRACE_USR4(("u8GetWaringCurrent()-> warning current for %02x antenna = %d",
			ETG_CENUM(tenAntDiagClients,enAntDiagClient),m_u8AntennaWarningCur[enAntDiagClient]));
		return m_u8AntennaWarningCur[enAntDiagClient];
	}
}
/*************************************************************************
* FUNCTION:     vHandlePrevFM1AntennaStateShort
*
* DESCRIPTION:	 function to handle FM1 Current 
*				
* PARAMETER:	 tFloat fCurrent
*
* RETURNVALUE:	 tVoid
*************************************************************************/
tVoid tunmstr_antdiag_Debounce::vHandlePrevAntennaStateShort(tU8 u8index,tFloat fCurrent)
{
	ETG_TRACE_USR4(("vHandlePrevAntennaStateOkorUnk()-> u8index=%02x fCurrent=%f",
	ETG_CENUM(tenAntDiagClients,u8index),fCurrent));
	if(u8index>=enClientNone)
	{
		//Outof Range
		return;
	}
	//<10ma
	if(fCurrent<m_u8OpenCur[u8index])
	{
		m_u8OpenCounter[u8index]++;
		m_u8OpenToAmpCounter[u8index]=0;
		m_u8CloseCounter[u8index]=0;
		m_u8ShortCounter[u8index] =0;
	}
	//10<i<30
	else if((fCurrent>=m_u8OpenCur[u8index])&&(fCurrent<=m_u8Open2AmpCur[u8index]))
	{
		m_u8OpenToAmpCounter[u8index]++;
		m_u8OpenCounter[u8index]=0;
		m_u8CloseCounter[u8index]=0;
		m_u8ShortCounter[u8index] =0;
	}
	//30<i<110
	else if((fCurrent>m_u8Open2AmpCur[u8index])&&(fCurrent<=m_u8ShortClearCur[u8index]))
	{
		m_u8CloseCounter[u8index]++;
		m_u8OpenCounter[u8index]=0;
		m_u8OpenToAmpCounter[u8index]=0;
		m_u8ShortCounter[u8index] =0;
	}
	//>100
	else
	{
		m_u8ShortCounter[u8index]++;
		m_u8OpenCounter[u8index]=0;
		m_u8OpenToAmpCounter[u8index]=0;
		m_u8CloseCounter[u8index] =0;
	}
}
/*************************************************************************
* FUNCTION:     vHandlePrevFM1AntStateOpen2Amp
*
* DESCRIPTION:	 function to handle FM1 Current 
*				
* PARAMETER:	 tFloat fCurrent
*
* RETURNVALUE:	 tVoid
*************************************************************************/
tVoid tunmstr_antdiag_Debounce::vHandlePrevAntStateOpen2Amp(tU8 u8index,tFloat fCurrent)
{
	ETG_TRACE_USR4(("vHandlePrevAntStateOpen2Amp()-> u8index=%02x fCurrent=%f",
	ETG_CENUM(tenAntDiagClients,u8index),fCurrent));
	if(u8index>=TUNMSTR_ANTDIAG_MAX_NUMBEROF_ANTDIAGCLIENT)
	{
		//Outof Range
		return;
	}
	//<5ma
	if(fCurrent<m_u8Open2AmpClr_LwCur[u8index])
	{
		m_u8OpenCounter[u8index]++;
		m_u8OpenToAmpCounter[u8index]=0;
		m_u8CloseCounter[u8index]=0;
		m_u8ShortCounter[u8index] =0;
	}
	//5<i<35
	else if((fCurrent>=m_u8Open2AmpClr_LwCur[u8index])&&(fCurrent<=m_u8Open2AmpClr_HgCur[u8index]))
	{
		m_u8OpenToAmpCounter[u8index]++;
		m_u8OpenCounter[u8index]=0;
		m_u8CloseCounter[u8index]=0;
		m_u8ShortCounter[u8index] =0;
	}
	//35<i<110
	else if((fCurrent>m_u8Open2AmpClr_HgCur[u8index])&&(fCurrent<=m_u8ShortCur[u8index]))
	{
		m_u8CloseCounter[u8index]++;
		m_u8OpenCounter[u8index]=0;
		m_u8OpenToAmpCounter[u8index]=0;
		m_u8ShortCounter[u8index] =0;
	}
	//>110
	else
	{
		m_u8ShortCounter[u8index]++;
		m_u8OpenCounter[u8index]=0;
		m_u8OpenToAmpCounter[u8index]=0;
		m_u8CloseCounter[u8index] =0;
	}
}
/*************************************************************************
* FUNCTION:     u8GetAntennaStatus
*
* DESCRIPTION:	 function to read the antenna status of client
*				
* PARAMETER:	 tenAntDiagClients enAntDiagClient
*
* RETURNVALUE:	 tU8
*************************************************************************/
tU8 tunmstr_antdiag_Debounce::u8GetAntennaStatus(tenAntDiagClients enAntDiagClient)const
{
	if(enAntDiagClient>=enClientNone)
	{
		return ((tU8)enAntennState_Unknown);
	}
	else
	{
		return (m_u8AntennaStatus[(tU8)enAntDiagClient]);
	}
}
tVoid tunmstr_antdiag_Debounce::vUpdateAntennaStatus(tenAntDiagClients enAntDiagClient,tU8 u8AntennState )
{
	if(enAntDiagClient>=enClientNone)
	{
	}
	else
	{
		m_u8AntennaStatus[(tU8)enAntDiagClient]=u8AntennState;
	}
}
/*************************************************************************
* FUNCTION:     vClearAllAntennaStatus
*
* DESCRIPTION:  Function to clear the all antenna state

* PARAMETER:    tenAntDiagClients enAntDiagClient
*
* RETURNVALUE:  tVoid
*************************************************************************/
tVoid tunmstr_antdiag_Debounce::vClearAllAntennaStatus()
{
	for(tU8 u8index=0; u8index<TUNMSTR_ANTDIAG_MAX_NUMBEROF_ANTDIAGCLIENT; u8index++)	
	{
		m_u8ShortCounter[u8index]=0;
		m_u8OpenCounter[u8index]=0;
		m_u8CloseCounter[u8index]=0;
		m_u8OpenToAmpCounter[u8index]=0;
		//Antenna Current
		m_fAntennacurrent[u8index]=0.0;
		m_u8AntennaStatus[u8index]=(tU8)enAntennState_Unknown;
	}
}

/*************************************************************************
* FUNCTION:     IsShortCktTimeOver
*
* DESCRIPTION:  Function to check the  ShortWaitOverCounter reaches the max limit, if not increment the counter by 1

* PARAMETER:    tenAntDiagClients enAntDiagClient
*
* RETURNVALUE:  tVoid
*************************************************************************/
tBool tunmstr_antdiag_Debounce::IsShortCktTimeOver(tenAntDiagClients enAntDiagClient)
{
	tBool blRet=FALSE;
	if(enAntDiagClient>=enClientNone)
	{
		return blRet;
	}
	//If AntennaShortCircuitWaitTime =255, Don't turn on the antenna in the current power cycle
	if((m_u8ShortCkt_Wait[enAntDiagClient]!=TUNMSTR_ANTDIAG_U8MAX) && (m_u8ShortCkt_Wait[enAntDiagClient]!=TUNMSTR_ANTDIAG_U8ZERO))
	{
		if((m_u8ShortWaitOverCounter[enAntDiagClient])>=(m_u8ShortCkt_Wait[enAntDiagClient]))
		{
			blRet =TRUE;
		}
		else
		{
			vIncrementShortCktWaitOverCounter(enAntDiagClient);
		}
	}
	ETG_TRACE_USR4(("IsShortCktTimeOver()blRet=%d m_u8ShortWaitOverCounter=%d",blRet,m_u8ShortWaitOverCounter[enAntDiagClient]));
	return blRet;
}
/*************************************************************************
* FUNCTION:     vIncrementShortCktWaitOverCounter
*
* DESCRIPTION:  Function to increment the ShortWaitOverCounter

* PARAMETER:    tenAntDiagClients enAntDiagClient
*
* RETURNVALUE:  tVoid
*************************************************************************/
tVoid tunmstr_antdiag_Debounce::vIncrementShortCktWaitOverCounter(tenAntDiagClients enAntDiagClient)
{
	if(enAntDiagClient>=enClientNone)
	{
		return;
	}
	else
	{
		m_u8ShortWaitOverCounter[(tU8)enAntDiagClient]++;
		ETG_TRACE_USR4(("vIncrementShortCktWaitOverCounter()m_u8ShortWaitOverCounter=%d ",m_u8ShortWaitOverCounter[(tU8)enAntDiagClient]));
	}
}
/*************************************************************************
* FUNCTION:     vClearShortWaitOverCounter
*
* DESCRIPTION:  Function to clear the ShortWaitOverCounter

* PARAMETER:    tenCMExtPowerSupplyType enCMExtPowerSupplyType
*
* RETURNVALUE:  tVoid
*************************************************************************/
tVoid tunmstr_antdiag_Debounce::vClearShortWaitOverCounter(tenCMExtPowerSupplyType enCMExtPowerSupplyType)
{
	if(NULL==m_potunmstr_antdiag_Manager)
	{
		return;
	}
    m_u8ShortWaitOverCounter[m_potunmstr_antdiag_Manager->enGetAntDiagClient(enCMExtPowerSupplyType)]=0;
}
/*************************************************************************
* FUNCTION:     fGetAntennaCurrent
*
* DESCRIPTION:  Function to get the request antenna current

* PARAMETER:    tU8 u8Client
*
* RETURNVALUE:  tU8
*************************************************************************/
tFloat tunmstr_antdiag_Debounce::fGetAntennaCurrent(tU8 u8Client)const
{
	if(u8Client>=enClientNone)
	{
		return 0;
	}
	else
	{
		return m_fAntennacurrent[u8Client];
	}
}
/*************************************************************************
* FUNCTION:     u8GetShortCurrent
*
* DESCRIPTION:  Function to get the request antenna short current
*************************************************************************/
tU8 tunmstr_antdiag_Debounce::u8GetShortCurrent(tU8 u8Client)
{
	if(u8Client>=enClientNone)
	{
		return 0;
	}
	return m_u8ShortCur[u8Client];
}
tU8 tunmstr_antdiag_Debounce::u8GetOpenCurrent(tU8 u8Client)
{
	if(u8Client>=enClientNone)
	{
		return 0;
	}
	return m_u8OpenCur[u8Client];
}
tU8 tunmstr_antdiag_Debounce::u8GetOutputVg(tU8 u8Client)
{
	if(u8Client>=enClientNone)
	{
		return 0;
	}
	return m_u8AntennaOutPutVg[u8Client];
}
tU8 tunmstr_antdiag_Debounce::u8GetWarningCurrent(tU8 u8Client)
{
	if(u8Client>=enClientNone)
	{
		return 0;
	}
	ETG_TRACE_USR4(("u8GetWarningCurrent()-> =%d client is =%02x",
			m_u8AntennaWarningCur[u8Client],
	ETG_CENUM(tenAntDiagClients,u8Client)));
	return m_u8AntennaWarningCur[u8Client];
}
/*************************************************************************
* FUNCTION:     u8GetShortCkt_WaitOverTime
*************************************************************************/
tU8 tunmstr_antdiag_Debounce::u8GetShortCkt_WaitOverTime(tenAntDiagClients enAntDiagClient)
{
	if(enAntDiagClient>=enClientNone)
	{
		return 0;
	}
	return m_u8ShortCkt_Wait[enAntDiagClient];
}
/*************************************************************************
* FUNCTION:     u8GetShortCurrent
*
* DESCRIPTION:  Function to get the request antenna open2Amp current
*************************************************************************/
tU8 tunmstr_antdiag_Debounce::u8GetOpen2AmpCurrent(tenAntDiagClients enAntDiagClient)
{
	if((enAntDiagClient>=enClientNone) ||(NULL==m_potunmstr_DiagLogClientHandler))
	{
		return 0;
	}
	return m_u8Open2AmpCur[enAntDiagClient];
}
/*************************************************************************
* FUNCTION:     vClearAntennaState
*
* DESCRIPTION:  Clears the antenna status of requested Client
*************************************************************************/
tVoid tunmstr_antdiag_Debounce::vClearAntennaState(tenAntDiagClients enAntDiagClient)
{
	if((enAntDiagClient>=enClientNone)||(NULL==m_potunmstr_DiagLogClientHandler))
	{
		return ;
	}
	m_u8AntennaStatus[enAntDiagClient]=(tU8)enAntennState_Unknown;
	tBool blRes=m_potunmstr_DiagLogClientHandler->blUpdateAntennaStatus(enAntDiagClient,enAntennState_Unknown);
	ETG_TRACE_USR4(("vClearAntennaState() blRes=%d ",blRes));
}
tVoid tunmstr_antdiag_Debounce::vClearAllInternalState()
{
	for (tU8 i=0;i<TUNMSTR_ANTDIAG_MAX_NUMBEROF_ANTDIAGCLIENT;i++)
	{
		m_u8ShortCounter[i]=0;
		m_u8OpenCounter[i]=0;
		m_u8CloseCounter[i]=0;
		m_u8OpenToAmpCounter[i]=0;
		m_fAntennacurrent[i]=0.0;
		m_u8ShortWaitOverCounter[i]=0;
		m_u8AntennaStatus[i]=enAntennState_Unknown;
		m_u8OpenCur[i]=TUNMSTR_ANTDIAG_OpenCur;
		m_u8OpenClearCur[i]=TUNMSTR_ANTDIAG_OpenClearCur;
		m_u8Open2AmpCur[i]=0;
		m_u8Open2AmpClr_LwCur[i]=0;
		m_u8Open2AmpClr_HgCur[i]=0;
		m_u8ShortCur[i]=TUNMSTR_ANTDIAG_ShortCur;
		m_u8ShortClearCur[i]=TUNMSTR_ANTDIAG_ShortClearCur;
		m_u8NoOfMon_Cycle[i]=1;
		m_u8ShortCkt_Wait[i]=1;
	}
}
