/************************************************************************
* FILE:           tunmstr_DiaglibListner.cpp
* PROJECT:        G3g
* SW-COMPONENT:   FC_TunerMaster.
*----------------------------------------------------------------------
*
* DESCRIPTION:    Implementation of the Class  tunmstr_DiaglibListner
*
*----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
* HISTORY:
* Date      | Author                       | Modification
* 25/05/2012| ECV4 ( RBEI ) Priya Prabhuswamy  | Initial version.
* 25.04.13  | NGP1KOR							| First version of the G3g after porting from NISSAN LCN2Kai
* 17.03.14  | ECV3 ( RBEI ) Priya Prabhuswamy  | adaptation for Different Type of External Power supply
 *************************************************************************/
#ifndef tunmstr_DiaglibListner_H
#include "tunmstr_DiaglibListner.h"
#endif
#include "tunermaster_trace.h"
#include "tunmstr_DiagLogClientHandler.h"
#include "tunmstr_antdiag_Manager.h"
#include "tunmstr_antdiag_i2cHandler.h"
#include "tunermaster_main.h"
#include "tunmstr_MsgToHMI.h"
#include "tunmstr_antdiag_Defines.h"
#include "tunmstr_antdiag_Debounce.h"
#include "tunmstr_antdiag_Measure.h"
#include "kdsconfig/clConfigInterface.h"
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TUNMSTR_TRACE_CLASS_DIAGLIBHANDLER
#include "trcGenProj/Header/tunmstr_DiaglibListner.cpp.trc.h"
#endif

#ifndef VARIANT_S_FTR_ENABLE_TUNERMASTER_MOCK
#define DP_S_IMPORT_INTERFACE_FI
#include "dp_generic_if.h"
#include "dp_tuner_if.h"
#endif
#define TUN_CONFIG_DEFAULT_AVAILABLEBAND_AM_MASKVAL  0x06
#define ISTUNMSTR_ROUTINEID(u32RoutineId)((ROUTINEID_DAB_ANTENNA_MODE==u32RoutineId)||(ROUTINEID_ANTENNA_FEED_SELE==u32RoutineId) || (ROUTINEID_ANTENNA_2_FEED_SELE==u32RoutineId))
/************************************************************************
 * FUNCTION:     tunmstr_DiaglibListner::tunmstr_DiaglibListner( )
 *
 * DESCRIPTION:  Default Constructor
 *
 * PARAMETER:    void
 *
 * RETURNVALUE:  void
 *************************************************************************/
tunmstr_DiaglibListner::tunmstr_DiaglibListner(tunmaster_tclApp* poMainApp)
{
	ETG_TRACE_USR4(("tunmstr_DiaglibListner"));
	m_potunermaster_main = poMainApp;
	m_potunmstr_antdiag_Manager=NULL;
	m_potunmstr_antdiag_i2cHandler=NULL;
	m_potunmstr_MsgToHMI=NULL;
	m_potunmstr__DiagLog=NULL;
	m_potunmstr_antdiag_Debounce=NULL;
	m_potunmstr_antdiag_Measure = NULL;
#ifndef VARIANT_S_FTR_ENABLE_FEATURE_GMMY16
	m_potunmstr_Preset_Handler=NULL;
	m_poclConfigInterface=NULL;
#endif
#ifdef VARIANT_S_FTR_ENABLE_SRCCHANGE_LIB
	m_poArlSourceImpl=NULL;
#endif
	m_poFactory=NULL;
	m_poDiagIF=NULL;
	m_poCcaIF=NULL;
	m_poSystemSetIF=NULL;
	m_poRoutineCtrlIF=NULL;
	tU8 u8MajorVersion=DIAGLIB_U8_MAJOR_VERSION;
	tU8 u8MinorVersion=DIAGLIB_U8_MINOR_VERSION;
	m_poFactory=diaglib::tclDiagnosisInterfaceFactory::poGetInstance();
	m_MsgContext=0;
	if(m_poFactory != OSAL_NULL)
	{
		if(m_poFactory->bGetVersion(u8MajorVersion, u8MinorVersion) == TRUE)
		{
			if(u8MajorVersion !=DIAGLIB_U8_MAJOR_VERSION)
			{
				ETG_TRACE_ERR(("Major version is not equal to 1"));
			}
		}

		if(m_potunermaster_main!=NULL)
		{
			m_poDiagIF=m_poFactory->poCreateDiagnosisInterface(diaglib::EN_PROTOCOL_CCA,static_cast<tVoid*>(m_potunermaster_main));
			ETG_TRACE_USR4(("m_poDiagIF=%p",m_poDiagIF));
			if(OSAL_NULL != m_poDiagIF)
			{
				m_poDiagIF->bGetInterface<diaglib::tclCCAServiceIF>(&m_poCcaIF);
				m_poDiagIF->bGetInterface<diaglib::tclRoutineControlIF>(&m_poRoutineCtrlIF);
				m_poDiagIF->bGetInterface<diaglib::tclSystemSetIF>(&m_poSystemSetIF);
				m_poRoutineCtrlIF->vRegisterListener(ROUTINEID_ANTENNA_FEED_SELE,(diaglib::tclRoutineControlListenerIF *)this);
				m_poRoutineCtrlIF->vRegisterListener(ROUTINEID_DAB_ANTENNA_MODE,(diaglib::tclRoutineControlListenerIF *)this);
				m_poRoutineCtrlIF->vRegisterListener(ROUTINEID_ANTENNA_2_FEED_SELE,(diaglib::tclRoutineControlListenerIF *)this);
				if (m_poSystemSetIF != NULL)
				{
					m_poSystemSetIF->vRegisterListener(SYSSETID_TUNE_MASTER, diaglib::EN_TYPE_CODING, (diaglib::tclSystemSetListenerIF *)this);
					m_poSystemSetIF->vRegisterListener(SYSSETID_TUNE_MASTER, diaglib::EN_TYPE_DEFAULT_HMI, (diaglib::tclSystemSetListenerIF *)this);
					m_poSystemSetIF->vRegisterListener(SYSSETID_TUNE_MASTER, diaglib::EN_TYPE_DEFAULT_TEF, (diaglib::tclSystemSetListenerIF *)this);
#ifdef VARIANT_S_FTR_ENABLE_FEATURE_PSA_RCC
					m_poSystemSetIF->vRegisterListener(SYSSETID_TUNE_MASTER, diaglib::EN_TYPE_UM_RESET_ALL, (diaglib::tclSystemSetListenerIF *)this);
					m_poSystemSetIF->vRegisterListener(SYSSETID_TUNE_MASTER, diaglib::EN_TYPE_UM_RESET_USER, (diaglib::tclSystemSetListenerIF *)this);
					m_poSystemSetIF->vRegisterListener(SYSSETID_TUNE_MASTER, diaglib::EN_TYPE_UM_SWITCH_USER, (diaglib::tclSystemSetListenerIF *)this);
#endif
				}
			}
		}

	}
	ETG_TRACE_USR4(("SystemSetIF=%p DiagIF=%p RoutineCtrlIF=%p Factory=%p",
			m_poSystemSetIF,m_poDiagIF,m_poRoutineCtrlIF,m_poFactory));
}
/*************************************************************************
 * FUNCTION:     tunmstr_DiaglibListner::~tunmstr_DiaglibListner( )
 *
 * DESCRIPTION:  Destructor
 *
 * PARAMETER:    void
 *
 * RETURNVALUE:  void
 *************************************************************************/
tunmstr_DiaglibListner::~tunmstr_DiaglibListner( )
{
	m_potunermaster_main = NULL;
	m_poRoutineCtrlIF=NULL;
	m_poSystemSetIF=NULL;
	m_potunmstr_antdiag_Manager=NULL;
	m_potunmstr__DiagLog=NULL;
	m_potunmstr_antdiag_i2cHandler=NULL;
	m_potunmstr_MsgToHMI=NULL;
	m_potunmstr_antdiag_Debounce=NULL;
	m_potunmstr_antdiag_Measure = NULL;
#ifndef VARIANT_S_FTR_ENABLE_FEATURE_GMMY16
	m_potunmstr_Preset_Handler=NULL;
	m_poclConfigInterface=NULL;
#endif
#ifdef VARIANT_S_FTR_ENABLE_SRCCHANGE_LIB
	m_poArlSourceImpl=NULL;
#endif
	m_poFactory=NULL;
	m_poDiagIF=NULL;
	m_poCcaIF=NULL;
	m_poSystemSetIF=NULL;
	m_poRoutineCtrlIF=NULL;
	m_MsgContext=0;
	while(!m_stTunmstrRoutineControlMsg.empty())
	{
	   m_stTunmstrRoutineControlMsg.pop();
	}
}
/*************************************************************************
 * FUNCTION:    tunmstr_DiaglibListner::vSet_tunmstr_antdiag_Manager(tunmstr_antdiag_Manager* potunmstr_antdiag_Manager)
 *
 * DESCRIPTION: Set pointer to tunmstr_antdiag_Manager
 *
 * PARAMETER:   tunmstr_antdiag_Manager* potunmstr_antdiag_Manager
 *
 * RETURNVALUE:  void
 *************************************************************************/
tVoid tunmstr_DiaglibListner::vSet_tunmstr_antdiag_Manager(tunmstr_antdiag_Manager* potunmstr_antdiag_Manager)
{
	if( potunmstr_antdiag_Manager != NULL )
	{
		m_potunmstr_antdiag_Manager = ( tunmstr_antdiag_Manager* )potunmstr_antdiag_Manager;
	}
}
/*************************************************************************
 * FUNCTION:    tunmstr_DiaglibListner::vSet_tunmstr_MsgToHMI_Ptr
 *
 * DESCRIPTION: Set pointer to pointer tunmstr_MsgToHMI
 *
 * PARAMETER:   tunmstr_MsgToHMI* potunmstr_MsgToHMI
 *
 * RETURNVALUE:  void
 *************************************************************************/
tVoid tunmstr_DiaglibListner::vSet_tunmstr_MsgToHMI_Ptr(tunmstr_MsgToHMI* potunmstr_MsgToHMI)
{
	if(potunmstr_MsgToHMI!=NULL)
	{
		m_potunmstr_MsgToHMI =(tunmstr_MsgToHMI*) potunmstr_MsgToHMI;
	}
}
/******************************************************************************
*	FUNCTION:      vSet_tunmstr_antdiag_i2cHandler

*	DESCRIPTION:   function to update m_potunmstr_antdiag_i2cHandler

*	PARAMETERS:    tunmstr_antdiag_i2cHandler* Ptunmstr_antdiag_i2cHandler

*	RETURN TYPE:   tVoid
*********************************************************************************/
tVoid tunmstr_DiaglibListner::vSet_tunmstr_antdiag_i2cHandler
(tunmstr_antdiag_i2cHandler* Ptunmstr_antdiag_i2cHandler)
{
	m_potunmstr_antdiag_i2cHandler=Ptunmstr_antdiag_i2cHandler;
}

/*************************************************************************
* FUNCTION:    vSet_tunmstr_antdiag_Measure_Ptr( tunmstr_DiagLogClientHandler* potunmstr_DiagLogClientHandler )
*
* DESCRIPTION: update tunmstr_DiagLogClientHandler pointer variable
*
* PARAMETER:   tunmstr_DiagLogClientHandler* potunmstr_DiagLogClientHandler
*
* RETURNVALUE:  void
*************************************************************************/
tVoid tunmstr_DiaglibListner::vSet_tunmstr_DiagLogClientHandler_Ptr(tunmstr_DiagLogClientHandler* potunmstr_DiagLogClientHandler)
{
	if( potunmstr_DiagLogClientHandler != NULL )
	{
		m_potunmstr__DiagLog = ( tunmstr_DiagLogClientHandler* )potunmstr_DiagLogClientHandler;
	}
}
/*************************************************************************
* FUNCTION:    vSet_tunmstr_antdiag_Debounce_Ptr
*
* DESCRIPTION: update m_potunmstr_antdiag_Debounce pointer variable
*
* PARAMETER:   tunmstr_antdiag_Debounce* potunmstr_antdiag_Debounce
*
* RETURNVALUE:  void
*************************************************************************/
tVoid tunmstr_DiaglibListner::vSet_tunmstr_antdiag_Debounce_Ptr
(tunmstr_antdiag_Debounce* potunmstr_antdiag_Debounce)
{
	if( potunmstr_antdiag_Debounce != NULL )
	{
		m_potunmstr_antdiag_Debounce = ( tunmstr_antdiag_Debounce* )potunmstr_antdiag_Debounce;
	}
}
#ifndef VARIANT_S_FTR_ENABLE_FEATURE_GMMY16
tVoid tunmstr_DiaglibListner::vSet_tunmstr_Preset_Handler_Ptr( tunmstr_Preset_Handler* potunmstr_Preset_Handler)
{
	if( potunmstr_Preset_Handler != NULL )
	{
		m_potunmstr_Preset_Handler = ( tunmstr_Preset_Handler* )potunmstr_Preset_Handler;
	}
}
tVoid tunmstr_DiaglibListner::vSet_tunmstr_ConfigInterface_Ptr
(clConfigInterface* poclConfigInterface)
{
	if( poclConfigInterface != NULL )
	{
		m_poclConfigInterface = poclConfigInterface;
	}

}
#endif
tVoid tunmstr_DiaglibListner::vSet_tunmstr_antdiag_Measure_Ptr
(tunmstr_antdiag_Measure* potunmstr_antdiag_Measure)
{
	if( potunmstr_antdiag_Measure != NULL )
	{
		m_potunmstr_antdiag_Measure = ( tunmstr_antdiag_Measure* )potunmstr_antdiag_Measure;
	}
}

#ifdef VARIANT_S_FTR_ENABLE_SRCCHANGE_LIB
tVoid tunmstr_DiaglibListner::vSet_tunmstr_ArlSourceImpl_Ptr(tuner::sessionManagement::audio::clArlSourceImpl* poArlSourceImpl)
{
	if( poArlSourceImpl != NULL )
	{
		m_poArlSourceImpl = poArlSourceImpl;
	}
}
#endif


/*************************************************************************
 * FUNCTION: 		vOnSystemSetPrepare
 *************************************************************************/
tU32 tunmstr_DiaglibListner::vOnSystemSetPrepare( tU32 u32SystemSetID,
		diaglib::tenSystemSetType u32SystemSetType,
		diaglib::tContext MsgContext
)
{
	ETG_TRACE_USR4(("vOnSystemSetPrepare()->SystemSetID=%d SystemSetType=%d ",u32SystemSetType,u32SystemSetID));
	tU32 u32RetValue	=	diaglib::U32_DIAGLIB_RETURN_NOT_OK;
	if (m_poSystemSetIF == NULL)
	{
		return u32RetValue;
	}
	diaglib::tenSystemSetResult _enSysSetPrepareResult = diaglib::EN_SYSTEMSET_NOT_OK;
	diaglib::tclParameterVector _oErrorList;
	diaglib::trParameter rParam;
	rParam.enType = diaglib::EN_PARAMETER_TYPE_U8;
	if(TUNERMASTER_SYSTEMSETID(u32SystemSetID))
	{
		if((tBool)TUNERMASTER_REGISTERED_SYSTEMSETTYEP((tU32)u32SystemSetType))
		{
			_enSysSetPrepareResult=diaglib::EN_SYSTEMSET_OK;
			u32RetValue =	diaglib::U32_DIAGLIB_RETURN_OK;
		}
		else
		{
			rParam.u8Value = (tU8)diaglib::EN_ERROR_INCOMPATIBLE_PARAMETER_SIGNATURE;
			_oErrorList.push_back(rParam);
		}
	}
	else
	{
		rParam.u8Value = (tU8)diaglib::EN_ERROR_ID_NOT_SUPPORTED;
		_oErrorList.push_back(rParam);
	}
	if (TRUE == m_poSystemSetIF->bSendSystemSetPrepareResult( _enSysSetPrepareResult, _oErrorList, MsgContext ) )
	{
		ETG_TRACE_USR4(("vOnSystemSetPrepare()->bSendSystemSetPrepareResult has sent"));
	}
	else
	{
		ETG_TRACE_USR4(("vOnSystemSetPrepare()->bSendSystemSetPrepareResult has not sent"));
	}
	return u32RetValue;
}
/*************************************************************************
 * FUNCTION: 		vOnSystemSet
 *************************************************************************/

tU32 tunmstr_DiaglibListner::vOnSystemSet( tU32 u32SystemSetID,
		diaglib::tenSystemSetType u32SystemSetType,
		diaglib::tContext MsgContext
)
{
	tU32 u32RetValue=diaglib::U32_DIAGLIB_RETURN_NOT_OK;
	if (m_poSystemSetIF == NULL)
	{
		return u32RetValue;
	}
	// temp vars
	diaglib::tenSystemSetResult _enSysSetResult = diaglib::EN_SYSTEMSET_NOT_OK;
	diaglib::tclParameterVector _oErrorList;
	diaglib::trParameter rParam;
	rParam.enType = diaglib::EN_PARAMETER_TYPE_U8;
	m_MsgContext=MsgContext;
	if(TUNERMASTER_SYSTEMSETID(u32SystemSetID))
	{
		if((tBool)TUNERMASTER_REGISTERED_SYSTEMSETTYEP((tU32)u32SystemSetType))
		{
			vHandleSystemSetRequest(u32SystemSetType);
		}
		else
		{

			rParam.u8Value = (tU8)diaglib::EN_ERROR_INCOMPATIBLE_PARAMETER_SIGNATURE;
			_oErrorList.push_back(rParam);
			m_poSystemSetIF->bSendSystemSetResult( _enSysSetResult, _oErrorList, MsgContext);
		}
	}
	else
	{
		rParam.u8Value = (tU8)diaglib::EN_ERROR_ID_NOT_SUPPORTED;
		_oErrorList.push_back(rParam);
		m_poSystemSetIF->bSendSystemSetResult( _enSysSetResult, _oErrorList, MsgContext);
		ETG_TRACE_USR4(("vOnSystemSet()->bSendSystemSetResult has sent"));
	}
	return u32RetValue;
}
/*************************************************************************
 * FUNCTION: 		vOnSystemSetFinished
 *************************************************************************/
tU32 tunmstr_DiaglibListner::vOnSystemSetFinished( tU32 u32SystemSetID, 
		diaglib::tenSystemSetType u32SystemSetType,
		diaglib::tContext MsgContext
)
{
	ETG_TRACE_USR4(("vOnSystemSetFinished()->SystemSetID=%d SystemSetType=%d ",u32SystemSetType,u32SystemSetID));
	tU32 u32RetValue= diaglib::U32_DIAGLIB_RETURN_NOT_OK;
	if (m_poSystemSetIF == NULL)
	{
		return u32RetValue;
	}
	if(TUNERMASTER_SYSTEMSETID(u32SystemSetID))
	{
		if((tBool)TUNERMASTER_REGISTERED_SYSTEMSETTYEP((tU32)u32SystemSetType))
		{
			u32RetValue =	diaglib::U32_DIAGLIB_RETURN_OK;
			vHandleSystemSetFinishRequest(u32SystemSetType);
		}
	}
	if (TRUE == m_poSystemSetIF->bAcknowledgeSystemSetFinished( MsgContext ) )
	{
		ETG_TRACE_USR4(("vOnSystemSetFinished()->vOnSystemSetFinishedresult has sent"));
	}
	else
	{
		ETG_TRACE_USR4(("vOnSystemSetFinished()->vOnSystemSetFinishedresult has not sent"));
	}
	return u32RetValue;
}
/*************************************************************************
 * FUNCTION: 		vOnIoControl
 *
 * DESCRIPTION:
 *
 * PARAMETER:  		tU32 u32IoControlId, diaglib::tenIoControlAction enActionId,
 *			  		diaglib::tContext MsgContext
 *
 * RETURNVALUE:
 *************************************************************************/
/*tU32 tunmstr_DiaglibListner::vOnIoControl(tU32 u32IoControlId,
		diaglib::tenIoControlAction enActionId,
		const diaglib::tclParameterVector& oParameterVector,
		diaglib::tContext MsgContext)
{
	(tVoid)oParameterVector;
	(tVoid)MsgContext;
	tU32 _u32RetValue = 0;
	// todo: wait for dianosis-declaration for Antenna
	ETG_TRACE_USR4(("vOnIoControl: u32IoControlID=%u enActionId=%u",
			u32IoControlId, enActionId));
	return _u32RetValue;
}*/
/*************************************************************************
 * FUNCTION: vOnIoControlFreeze
 *
 * DESCRIPTION:
 *
 * PARAMETER:  tU32 u32IoControlId,diaglib::tContext MsgContext
 *
 * RETURNVALUE:
 *************************************************************************/
//tU32 tunmstr_DiaglibListner::vOnIoControlFreeze (tU32 u32IoControlId,diaglib::tContext MsgContext)
//{
//	(tVoid)u32IoControlId;
//	ETG_TRACE_USR4(("vOnIoControlFreeze()-> u32IoControlId=%u MsgContext=%u",
//		u32IoControlId,MsgContext));
//	tU32 u32Ret=0;
//	if (m_blIoControlRunning)
//	{
//		//vSendResult(MsgContext, diaglib::EN_ERROR_ROUTINE_STILL_RUNNING);
//		return u32Ret;
//	}
//	m_blIoControlRunning=TRUE;
//	return u32Ret;
//}
/*************************************************************************
 * FUNCTION: vOnIoControlUnfreeze
 *
 * DESCRIPTION:
 *
 * PARAMETER:  tU32 u32IoControlId,diaglib::tContext MsgContext
 *
 * RETURNVALUE:
 *************************************************************************/
//tU32 tunmstr_DiaglibListner::vOnIoControlUnfreeze (tU32 u32IoControlId,diaglib::tContext MsgContext)
//{
//	(tVoid)u32IoControlId;
//	ETG_TRACE_USR4(("vOnIoControlFreeze()-> u32IoControlId=%u MsgContext=%u",
//	u32IoControlId,MsgContext));
//	tU32 u32Ret=0;
//	if (m_blIoControlRunning)
//	{
//		//vSendResult(MsgContext, diaglib::EN_ERROR_ROUTINE_STILL_RUNNING);
//		return u32Ret;
//	}
//	m_blIoControlRunning=TRUE;
//	return u32Ret;
//}
/*************************************************************************
 * FUNCTION:     enRoutineCntrlAntDiagClient()
 * DESCRIPTION:  Returns the client for which routine control is requested
 * PARAMETER:    tU32 u32RoutineId
 * RETURNVALUE:  tenAntDiagClients
 *************************************************************************/
tenAntDiagClients tunmstr_DiaglibListner::enRoutineCntrlAntDiagClient(tU32 u32RoutineId)
{
	if(ROUTINEID_ANTENNA_FEED_SELE==u32RoutineId)
	{
		return enFM1;
	}
	else if(ROUTINEID_DAB_ANTENNA_MODE==u32RoutineId)
	{
		return enDAB1;
	}
	else if(ROUTINEID_ANTENNA_2_FEED_SELE==u32RoutineId)
	{
		return enFM2;
	}
	else
	{
		return enClientNone;
	}
}
/*************************************************************************
 * FUNCTION:     enGetRoutineCntrlAntennaFeedReq()
 * DESCRIPTION:  Returns the routine control request for the client
  * PARAMETER:   diaglib::tclParameterVector& oParamVector,tU32 u32RoutineId
 * RETURNVALUE:  tenAntennaFeed
 *************************************************************************/
tenAntennaFeed tunmstr_DiaglibListner::enGetRoutineCntrlAntennaFeedReq(const diaglib::tclParameterVector& oParamVector,tU32 u32RoutineId)
{
	switch(u32RoutineId)
	{
	case (tU32)ROUTINEID_ANTENNA_FEED_SELE:
		{
		return (tenAntennaFeed)oParamVector[ROUTINEID_ANTENNA_FEED_SELE_START_u8RequestedAntennaFeed].u8Value;
		break;
		}
	case (tU32)ROUTINEID_DAB_ANTENNA_MODE:
		{
		return (tenAntennaFeed)oParamVector[ROUTINEID_DAB_ANTENNA_MODE_START_u8SetDABAntennaPowerMode].u8Value;
		break;
		}
	case ROUTINEID_ANTENNA_2_FEED_SELE:
	{
		return (tenAntennaFeed)oParamVector[ROUTINEID_ANTENNA_2_FEED_SELE_START_u8ActualAntennaFeed].u8Value;
		break;
		}
	default:
	{
		break;
	}
	}
	return enNonePower;
}

/*************************************************************************
 * FUNCTION:     blHandleGPIOstateOff()
 * DESCRIPTION:  Function takes care of the request if the Client is already turned off
 * PARAMETER:    tenAntDiagClients enAntDiagClient
 * RETURNVALUE:  tenAntennaFeed
 *************************************************************************/
tBool tunmstr_DiaglibListner::blHandleGPIOstateOff(tenAntDiagClients enAntDiagClient)
{
	ETG_TRACE_USR4(("blHandleGPIOstateOff"));
	if((m_potunmstr__DiagLog!=NULL)&& (m_potunmstr_antdiag_Debounce!=NULL) && (m_potunmstr_antdiag_Manager!=NULL))
	{
		tU8 u8AntennaState=m_potunmstr__DiagLog->u8GetAntenanstate(enAntDiagClient);
		if((enAntennState_Deactivated_LowVg==u8AntennaState) || (enAntennState_Deactivated_HighVg==u8AntennaState))
		{
			return false;
		}
		else if(u8AntennaState ==enAntennState_Short)
		{
			m_potunmstr_antdiag_Debounce->vClearShortWaitOverCounter(m_potunmstr_antdiag_Manager->enGetExtPowerSupplyType(enAntDiagClient));
		}
		m_potunmstr_antdiag_Debounce->vClearCounter(enAntDiagClient);
		m_potunmstr__DiagLog->blUpdateAntennaStatus(enAntDiagClient,enAntennState_Unknown);
		return true;
	}
	return false;
}

/*************************************************************************
 * FUNCTION:     vOnRoutineControlStart()
 * DESCRIPTION:  Start the Selftest routine
 * PARAMETER:    u32RoutineId :
 *               u32Bitmask   :
 *               oParamVector :
 *               MsgContext   :
 * RETURNVALUE:  tVoid
 *************************************************************************/
tU32 tunmstr_DiaglibListner::vOnRoutineControlStart ( 
		tU32 u32RoutineId,
		tU32 u32Bitmask,
		const diaglib::tclParameterVector& oParamVector,
		diaglib::tContext MsgContext
)
{
	if(NULL!=m_potunmstr_antdiag_Manager)
	{
	ETG_TRACE_USR4(("ANTDIAG::ROUTINECONTROL::vOnRoutineControlStart()"));
	OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(u32RoutineId);
	OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(u32Bitmask);
	OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(oParamVector);
	if(diaglib::bCheckParameterSignature(oParamVector,u32GetStartLength(u32RoutineId),
			u32GetParam1(u32RoutineId))==FALSE)
	{
		vSendError(diaglib::EN_ERROR_INCOMPATIBLE_PARAMETER_SIGNATURE, MsgContext);
		return diaglib::U32_DIAGLIB_RETURN_NOT_OK;
	}
	}
	tenAntennaFeed enAntennaFeed =enGetRoutineCntrlAntennaFeedReq(oParamVector,u32RoutineId);
	return u32HandleRoutineControlStart(u32RoutineId,enAntennaFeed,MsgContext);
}
/*************************************************************************
 * FUNCTION:     u32HandleRoutineControlStart()
 * DESCRIPTION:  Handles the routine control request
 * PARAMETER:    u32RoutineId :
 *               oParamVector :
 *               MsgContext   :
 * RETURNVALUE:  tU32
 *************************************************************************/
tU32 tunmstr_DiaglibListner::u32HandleRoutineControlStart
(tU32 u32RoutineId,tenAntennaFeed enAntennaFeed,diaglib::tContext MsgContext)
{
	if(m_potunmstr_antdiag_Manager==NULL)
	{
		return diaglib::U32_DIAGLIB_RETURN_NOT_OK;
	}
	tU32 u32Ret=diaglib::U32_DIAGLIB_RETURN_OK;
	if(ISTUNMSTR_ROUTINEID(u32RoutineId))
	{
		tenAntDiagClients enAntDiagClient =enRoutineCntrlAntDiagClient(u32RoutineId);
		switch(enAntennaFeed)
		{
		   case enDisablePower:
		   case enEnablePower:
		   case enPhantomPower:
		   {
		      stTunmstrRoutineControlMsg ostTunmstrRoutineControlMsg;
		      ostTunmstrRoutineControlMsg.MsgContext = MsgContext;
		      ostTunmstrRoutineControlMsg.enAntDiagClient = enAntDiagClient;
		      ostTunmstrRoutineControlMsg.enAntennaFeed = enAntennaFeed;
		      vInsertMsgToRoutineControlMsgQ(ostTunmstrRoutineControlMsg);
		      m_potunmstr_antdiag_Manager->vPostEvent(TUNMSTR_ANTDIAG_U32_EVENT_MASK_ROUTINE_REQ);
		      break;
		   }

		   default:
		   {
		      u32Ret=diaglib::U32_DIAGLIB_RETURN_NOT_OK;
		      break;
		   }
		}
	}
	else
	{
		vSendError(diaglib::EN_ERROR_INCOMPATIBLE_PARAMETER_SIGNATURE, MsgContext);
		u32Ret=diaglib::U32_DIAGLIB_RETURN_NOT_OK;
	}
	return u32Ret;
}
/*************************************************************************
 * FUNCTION:     vHandleEnablePowerRequest()
 * DESCRIPTION:  Handles the request to enable antenna
 * PARAMETER:    tenAntDiagClients enAntDiagClient,diaglib::tContext MsgContext
 * RETURNVALUE:  void
 *************************************************************************/
tVoid tunmstr_DiaglibListner::vHandleEnablePowerRequest
(tenAntDiagClients enAntDiagClient,diaglib::tContext MsgContext)
{
	ETG_TRACE_USR4(("ANTDIAG::ROUTINECONTROL::vHandleEnablePowerRequest() for client %06x",
			ETG_CENUM(tenAntDiagClients, enAntDiagClient)));
	if(m_potunmstr_antdiag_Manager==NULL)
	{
		ETG_TRACE_USR4(("ANTDIAG::ROUTINECONTROL::vHandleEnablePowerRequest() m_potunmstr_antdiag_Manager pointer is null"));
		return;
	}
	diaglib::tenRoutineCtrlResult enRoutineStatus=diaglib::EN_ROUTINE_CONTROL_NOT_OK;
	diaglib::tclParameterVector oResultVector;
	vBuildResultVector(oResultVector,enEnablePower);
	tenCMExtPowerSupplyType enCMExtPowerSupplyType=m_potunmstr_antdiag_Manager->enGetExtPowerSupplyType(enAntDiagClient);
	if((enunknown!=enCMExtPowerSupplyType) && (m_potunmstr_antdiag_Manager->blIsExtSup_Supports(enCMExtPowerSupplyType)))
	{
		m_potunmstr_antdiag_Manager->vSetExtPowerSupply(enCMExtPowerSupplyType, true);
		//If antenna power supply is turned off
		if(!(m_potunmstr_antdiag_Manager->blGetExtPwrSupEN_PinState(enCMExtPowerSupplyType))&& (blHandleGPIOstateOff(enAntDiagClient)))
		{
			//Turn on
			if((ISPOWERSUPPLY_ELMOS(enCMExtPowerSupplyType))|| (ISPOWERSUPPLY_MAX20084(enCMExtPowerSupplyType)))
			{
				m_potunmstr_antdiag_i2cHandler->vSetOutputEnabled(enCMExtPowerSupplyType,true);
				if(m_potunmstr_antdiag_i2cHandler->blEnableOutput(enCMExtPowerSupplyType))
				{
					tBool blRet = false;
					if(ISPOWERSUPPLY_ELMOS(enCMExtPowerSupplyType))
					blRet= m_potunmstr_antdiag_Manager->blTurnOnELMOSIC(enCMExtPowerSupplyType);
					if(ISPOWERSUPPLY_MAX20084(enCMExtPowerSupplyType))
					blRet = m_potunmstr_antdiag_Manager->blTurnOnMAX20084Antenna(enCMExtPowerSupplyType);
					m_potunmstr_antdiag_Manager->vSetExtPwrSupEN_PinState(enCMExtPowerSupplyType, blRet);
				}
				else
				{
					if(OSAL_NULL !=m_poRoutineCtrlIF)
					{
						m_poRoutineCtrlIF->bSendRoutineCtrlMethodResult(enRoutineStatus,oResultVector,MsgContext);
					}
					return;
				}
			}
			else
			{
				m_potunmstr_antdiag_Manager->vTurnOnExtPowerSupplyType(enCMExtPowerSupplyType);
			}
			ETG_TRACE_USR4(("ANTDIAG::ROUTINECONTROL::%06x client enabled successfully",
								ETG_CENUM(tenAntDiagClients,enAntDiagClient)));
		}
		 enRoutineStatus=diaglib::EN_ROUTINE_CONTROL_OK;
	}
	if(OSAL_NULL !=m_poRoutineCtrlIF)
	{
		m_poRoutineCtrlIF->bSendRoutineCtrlMethodResult(enRoutineStatus,oResultVector,MsgContext);
	}
}
/*************************************************************************
 * FUNCTION:     vHandleEnablePhantomPowerRequest()
 * DESCRIPTION:  Handles the request to enable Phantom antenna (other than RN IVI)
 * PARAMETER:    tenAntDiagClients enAntDiagClient,diaglib::tContext MsgContext
 * RETURNVALUE:  void
 *************************************************************************/
tVoid tunmstr_DiaglibListner::vHandleEnablePhantomPowerRequest
(tenAntDiagClients enAntDiagClient,diaglib::tContext MsgContext)
{
	ETG_TRACE_USR4(("ANTDIAG::ROUTINECONTROL::vHandleEnablePhantomPowerRequest() for client %06x",
			ETG_CENUM(tenAntDiagClients,enAntDiagClient)));
	if(m_potunmstr_antdiag_Manager==NULL)
	{
		ETG_TRACE_ERR(("enAntDiagClient=%d",enAntDiagClient));
		return;
	}
	diaglib::tenRoutineCtrlResult enRoutineStatus=diaglib::EN_ROUTINE_CONTROL_OK;
	diaglib::tclParameterVector oResultVector;
	vBuildResultVector(oResultVector,enPhantomPower);
	tenCMExtPowerSupplyType enCMExtPowerSupplyType=m_potunmstr_antdiag_Manager->enGetExtPowerSupplyType(enAntDiagClient);
	if((enunknown!=enCMExtPowerSupplyType) && (m_potunmstr_antdiag_Manager->blIsExtSup_Supports(enCMExtPowerSupplyType)))
	{
		m_potunmstr_antdiag_Manager->vSetExtPowerSupply(enCMExtPowerSupplyType, true);
		//If antenna power supply is turned off
		if(!(m_potunmstr_antdiag_Manager->blGetExtPwrSupEN_PinState(enCMExtPowerSupplyType))&& (blHandleGPIOstateOff(enAntDiagClient)))
		{
			//Turn on
			if((ISPOWERSUPPLY_ELMOS(enCMExtPowerSupplyType))|| (ISPOWERSUPPLY_MAX20084(enCMExtPowerSupplyType)))
			{
				m_potunmstr_antdiag_i2cHandler->vSetOutputEnabled(enCMExtPowerSupplyType,true);
				if(m_potunmstr_antdiag_i2cHandler->blEnableOutput(enCMExtPowerSupplyType))
				{
					tBool blRet = false;
					if(ISPOWERSUPPLY_ELMOS(enCMExtPowerSupplyType))
					blRet = m_potunmstr_antdiag_Manager->blTurnOnELMOSIC(enCMExtPowerSupplyType);
					if(ISPOWERSUPPLY_MAX20084(enCMExtPowerSupplyType))
					blRet = m_potunmstr_antdiag_Manager->blTurnOnMAX20084Antenna(enCMExtPowerSupplyType);
					m_potunmstr_antdiag_Manager->vSetExtPwrSupEN_PinState(enCMExtPowerSupplyType, blRet);
				}
				else
				{
					enRoutineStatus=diaglib::EN_ROUTINE_CONTROL_NOT_OK;
					if(OSAL_NULL !=m_poRoutineCtrlIF)
					m_poRoutineCtrlIF->bSendRoutineCtrlMethodResult(enRoutineStatus,oResultVector,MsgContext);
					return;
				}
			}
			else
			{
			m_potunmstr_antdiag_Manager->vTurnOnExtPowerSupplyType(enCMExtPowerSupplyType);
			}
			ETG_TRACE_USR4(("ANTDIAG::ROUTINECONTROL::%06x client phantom power enabled successfully",
								ETG_CENUM(tenAntDiagClients,enAntDiagClient)));
		}
	}
	if(OSAL_NULL !=m_poRoutineCtrlIF)
	{
		m_poRoutineCtrlIF->bSendRoutineCtrlMethodResult(enRoutineStatus,oResultVector,MsgContext);
	}
}
/*************************************************************************
 * FUNCTION:     vHandleDisablePowerRequest()
 * DESCRIPTION:  Handles the request to disable antenna
 * PARAMETER:    tenAntDiagClients enAntDiagClient,diaglib::tContext MsgContext
 * RETURNVALUE:  void
 *************************************************************************/
tVoid tunmstr_DiaglibListner::vHandleDisablePowerRequest
(tenAntDiagClients enAntDiagClient,diaglib::tContext MsgContext)
{
	ETG_TRACE_USR4(("ANTDIAG::ROUTINECONTROL::vHandleDisablePowerRequest() for client %06x", ETG_CENUM(tenAntDiagClients, enAntDiagClient)));
	if((m_potunmstr_antdiag_Manager!=NULL) && (m_potunmstr_antdiag_Debounce!=NULL) 
	&& (m_potunmstr__DiagLog!=NULL)&& (m_potunmstr_MsgToHMI!=NULL) && (m_potunmstr_antdiag_i2cHandler!=NULL))
	{
		diaglib::tenRoutineCtrlResult enRoutineStatus=diaglib::EN_ROUTINE_CONTROL_NOT_OK;
		diaglib::tclParameterVector oResultVector;
		vBuildResultVector(oResultVector,enDisablePower);
		tenCMExtPowerSupplyType enCMExtPowerSupplyType=m_potunmstr_antdiag_Manager->enGetExtPowerSupplyType(enAntDiagClient);
		if((enunknown!=enCMExtPowerSupplyType) && (m_potunmstr_antdiag_Manager->blIsExtSup_Supports(enCMExtPowerSupplyType)))
		{
			m_potunmstr_antdiag_Manager->vSetExtPowerSupply(enCMExtPowerSupplyType, true);
			//If antenna power supply is turned on
			if(m_potunmstr_antdiag_Manager->blGetExtPwrSupEN_PinState(enCMExtPowerSupplyType))
			{
				//Turn off
				if((ISPOWERSUPPLY_ELMOS(enCMExtPowerSupplyType))|| (ISPOWERSUPPLY_MAX20084(enCMExtPowerSupplyType)))
				{
					//Disable the corresponding Output pin of ELMOS
					m_potunmstr_antdiag_i2cHandler->vSetOutputEnabled(enCMExtPowerSupplyType,false);
					//Update the status register of ELMOS
					 if(!m_potunmstr_antdiag_i2cHandler->blEnableOutput(enCMExtPowerSupplyType))
					 {
						if(OSAL_NULL !=m_poRoutineCtrlIF)
						m_poRoutineCtrlIF->bSendRoutineCtrlMethodResult(enRoutineStatus,oResultVector,MsgContext);
						return;
					 }
				}
				else
				{
					m_potunmstr_antdiag_Manager->vTurnOffExtPowerSupply(enCMExtPowerSupplyType);
				}
				m_potunmstr_antdiag_Debounce->vClearCounter(enAntDiagClient);
				m_potunmstr__DiagLog->blUpdateAntennaStatus(enAntDiagClient,enAntennState_Deactivated);
				m_potunmstr_MsgToHMI->vSend_g_RFAntennaTest(enAntDiagClient,enAntennState_Deactivated,TUNMSTR_ANTDIAG_ZERO_FLOAT);
				enRoutineStatus=diaglib::EN_ROUTINE_CONTROL_OK;
				ETG_TRACE_USR4(("ANTDIAG::ROUTINECONTROL::%06x client disabled successfully",
						ETG_CENUM(tenAntDiagClients,enAntDiagClient)));
			}
			else if (blHandleGPIOstateOff(enAntDiagClient))
			{
				//Update Antenna status as de-activated
				m_potunmstr__DiagLog->blUpdateAntennaStatus(enAntDiagClient,enAntennState_Deactivated);
				m_potunmstr_MsgToHMI->vSend_g_RFAntennaTest(enAntDiagClient,enAntennState_Deactivated,TUNMSTR_ANTDIAG_ZERO_FLOAT);
				 enRoutineStatus=diaglib::EN_ROUTINE_CONTROL_OK;
			}
		}
		if(OSAL_NULL !=m_poRoutineCtrlIF)
		{
			m_poRoutineCtrlIF->bSendRoutineCtrlMethodResult(enRoutineStatus,oResultVector,MsgContext);
		}
	}
}
/*************************************************************************
 * FUNCTION:     vOnRoutineControlAbort()
 * DESCRIPTION:  Abort the Selftest routine
 * PARAMETER:    u32RoutineId :
 *               MsgContext   :
 * RETURNVALUE:  tVoid
 *************************************************************************/

tU32 tunmstr_DiaglibListner::vOnRoutineControlAbort ( 
		tU32 u32RoutineId,
		diaglib::tContext MsgContext
)
{
	ETG_TRACE_USR4(("vOnRoutineControlAbort()"));
	OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(u32RoutineId);

	// +++ Abort the running test specified by the given ID. +++
	//Abort not supported

	vSendError(diaglib::EN_ERROR_ID_NOT_SUPPORTED,MsgContext);
	return diaglib::U32_DIAGLIB_RETURN_OK;
}

/*************************************************************************
 * FUNCTION:     vOnGetRoutineResult()
 * DESCRIPTION:  Currently it is not supported
 * PARAMETER:    u32RoutineId :
              MsgContext   :
 * RETURNVALUE:  tU32 : status
 *************************************************************************/

tU32 tunmstr_DiaglibListner::vOnGetRoutineResult ( 
		tU32 u32RoutineId,
		diaglib::tContext MsgContext
)
{
	ETG_TRACE_USR4(("vOnGetRoutineResult()"));
	OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(u32RoutineId);
	//Not Supported
	vSendError(diaglib::EN_ERROR_ID_NOT_SUPPORTED,MsgContext);
	return diaglib::U32_DIAGLIB_RETURN_OK;
}
/*************************************************************************
 * FUNCTION:     vSendError()
 * DESCRIPTION:  Send Error to diaglib
 * PARAMETER:    enErrorCode : ErrorCode
 *               MsgContext  : Context of Message
 * RETURNVALUE:  tVoid
 *************************************************************************/
tVoid tunmstr_DiaglibListner::vSendError (diaglib::tenInternalError enErrorCode, diaglib::tContext MsgContext) const
{
	ETG_TRACE_USR4(("vSendError()"));
	diaglib::trParameter rParam;
	diaglib::tclParameterVector oResultVector;

	rParam.enType = diaglib::EN_PARAMETER_TYPE_U8ERROR;
	rParam.u8ErrorValue = enErrorCode;

	oResultVector.push_back(rParam);

	if(OSAL_NULL != m_poRoutineCtrlIF)
	{

		m_poRoutineCtrlIF->bSendRoutineCtrlMethodResult( diaglib::EN_ROUTINE_CONTROL_NOT_OK,
				oResultVector,
				MsgContext
		);
	}
}
/******************************************************************************
 *FUNCTION:    	tunmstr_DiaglibListner::vBuildResultVector
 *
 * DESCRIPTION: prepare message data for result message
 *
 * PARAMETER:   diaglib::tclParameterVector &oResultVector, tU8 u8Param1
 *
 * RETURNVALUE:  void
 *******************************************************************************/
tVoid tunmstr_DiaglibListner::vBuildResultVector (diaglib::tclParameterVector &oResultVector, tU8 u8Param1) const
{
	diaglib::trParameter rParam;

	rParam.enType = diaglib::EN_PARAMETER_TYPE_U8;
	rParam.u8Value = u8Param1;
	oResultVector.push_back(rParam);
}
/******************************************************************************
 *FUNCTION:    	tunmstr_DiaglibListner::vOnNewAppState
 *******************************************************************************/
tVoid tunmstr_DiaglibListner::vOnNewAppState(tU32 u32OldAppState,tU32 u32NewAppState)
{
	ETG_TRACE_USR1(("vOnNewAppState()->u32NewAppState= %d,u32OldAppState=%d",u32NewAppState,u32OldAppState));
	if(u32OldAppState==u32NewAppState)
	{
		return;
	}
	if( m_poCcaIF == OSAL_NULL )
	{
		ETG_TRACE_USR1(("vOnNewAppState()->Null Pointer"));
		return;
	}
	m_poCcaIF->vOnNewAppState(u32OldAppState, u32NewAppState);
}
/*************************************************************************
* FUNCTION:     tunmstr_antdiag_Manager::vDiagLibDispatchMessage
*
* DESCRIPTION:	 Dispatch fucntion for Diaglib messages
*
* PARAMETER:	amt_tclServiceData* poMessage
*
* RETURNVALUE:	tVoid
*************************************************************************/

tVoid tunmstr_DiaglibListner::vDiagLibDispatchMessage(amt_tclServiceData* poMessage)
{
	if( m_poCcaIF == OSAL_NULL )
	{
		ETG_TRACE_USR1(("vDiagLibDispatchMessage()->null pointer"));
		return;
	}
	m_poCcaIF->vDispatchMessage(poMessage);
	ETG_TRACE_USR1(("vDiagLibDispatchMessage()poMessage=%p",poMessage));
}

/*************************************************************************
* FUNCTION:     tunmstr_antdiag_Manager::bGetServiceVersion
*
* DESCRIPTION:	 Will read the Major, minor and Patch version of Diaglib service
*
* PARAMETER:	tU16& rfu16MajorVersion, tU16& rfu16MinorVersion, tU16& rfu16PatchVersion
*
* RETURNVALUE:	tVoid
*************************************************************************/
tBool tunmstr_DiaglibListner::bGetServiceVersion ( tU16& rfu16MajorVersion, 
												   tU16& rfu16MinorVersion, 
												   tU16& rfu16PatchVersion)
{
	ETG_TRACE_USR4(("bGetServiceVersion()")); 
	tBool _bRetValue = FALSE;
	if ( m_poCcaIF == OSAL_NULL )
	{
		// +++ Trace Error +++
		ETG_TRACE_COMP(("bGetServiceVersion()-> null-pointer m_poCcaIF")); 
	}
	else
	{
		_bRetValue = m_poCcaIF->bGetServiceVersion(rfu16MajorVersion, 
			rfu16MinorVersion, 
			rfu16PatchVersion );
	}
	ETG_TRACE_USR4(("bGetServiceVersion()->major=%u, minor=%u, path=%u, retVal=%u",
		rfu16MajorVersion,
		rfu16MinorVersion,
		rfu16PatchVersion,
		_bRetValue)); 

	return _bRetValue;
}
/*************************************************************************
* FUNCTION:     tunmstr_antdiag_Manager::u16GetServiceId
*
* DESCRIPTION:	 Function to  read Service Id
*
* PARAMETER:	tVoid
*
* RETURNVALUE:	tU16
*************************************************************************/
tU16 tunmstr_DiaglibListner::u16GetServiceId()
{
	if( m_poCcaIF == OSAL_NULL )
	{
	 ETG_TRACE_USR1(("u16GetServiceId()->null pointer"));
	 return 0;
	}  
	return(m_poCcaIF->u16GetServiceId());
}
/*************************************************************************
* FUNCTION:     u32GetStartLength
*
* DESCRIPTION:	 Function to  read Start length of Request Routine ID
*
* PARAMETER:	u32RoutineId
*
* RETURNVALUE:	tU32
*************************************************************************/
tU32 tunmstr_DiaglibListner::u32GetStartLength(tU32 u32RoutineId)const
{
	ETG_TRACE_USR1(("u32RoutineId=%d()",u32RoutineId));
	tU32 u32StartLength=ROUTINEID_ANTENNA_FEED_SELE_START_LEN;
	switch(u32RoutineId)
	{
	case ROUTINEID_ANTENNA_FEED_SELE:
		{
			u32StartLength=ROUTINEID_ANTENNA_FEED_SELE_START_LEN;
			break;
		}
	case ROUTINEID_DAB_ANTENNA_MODE:
		{
			u32StartLength=ROUTINEID_DAB_ANTENNA_MODE_START_LEN;
			break;
		}
	case ROUTINEID_ANTENNA_2_FEED_SELE:
		{
			u32StartLength=ROUTINEID_ANTENNA_2_FEED_SELE_START_LEN;
			break;
		}
	default:
		{
			break;
		}
	}
	return u32StartLength;
}
/*************************************************************************
* FUNCTION:     u32GetParam1
*
* DESCRIPTION:	 Function to  read Param1 of Request Routine ID
*
* PARAMETER:	u32RoutineId
*
* RETURNVALUE:	tU32
*************************************************************************/
tU32 tunmstr_DiaglibListner::u32GetParam1(tU32 u32RoutineId)const
{
	ETG_TRACE_USR1(("u32GetParam1=%d()",u32RoutineId));
	tU32 u32Param1=ROUTINEID_ANTENNA_FEED_SELE_START_PARAM1;
	switch(u32RoutineId)
	{
	case ROUTINEID_ANTENNA_FEED_SELE:
		{
			u32Param1=ROUTINEID_ANTENNA_FEED_SELE_START_PARAM1;
			break;
		}
	case ROUTINEID_DAB_ANTENNA_MODE:
		{
			u32Param1=ROUTINEID_DAB_ANTENNA_MODE_START_PARAM1;
			break;
		}
	case ROUTINEID_ANTENNA_2_FEED_SELE:
		{
			u32Param1=ROUTINEID_ANTENNA_2_FEED_SELE_START_PARAM1;
			break;
		}
	default:
		{
			break;
		}
	}
	return u32Param1;
}
tVoid tunmstr_DiaglibListner::vHandleSystemSetRequest(tU32 u32SystemSetType)
{
	ETG_TRACE_USR1(("vHandleSystemSetRequest() u32SystemSetType=%d",u32SystemSetType));
#ifndef VARIANT_S_FTR_ENABLE_FEATURE_GMMY16
	if((m_potunmstr_Preset_Handler==NULL)|| (NULL==m_poSystemSetIF) || (m_potunmstr_MsgToHMI == NULL) || (m_poclConfigInterface == NULL))
	{
		return;
	}
	if(TUNERMASTER_HANDLE_PRESETLIST(u32SystemSetType))
	{
		m_potunmstr_Preset_Handler->vHandleSystemSetForPresetList();
	}

	if(u32SystemSetType==diaglib::EN_TYPE_DEFAULT_HMI)
	{

#ifdef VARIANT_S_FTR_ENABLE_SRCCHANGE_LIB
	/*If AM is configured in amfm parameter set and AM is enabled in EOL paramter */
		arl_tenSource enSource = ARL_SRC_AM;
		arl_tenSrcAvailability enSrcAvail=ARL_EN_SRC_NOT_AVAILABLE;
		arl_tenAvailabilityReason enAvailabilityReason=ARL_EN_REASON_SAMEMEDIA;

	if((m_poclConfigInterface->u32GetConfigData("AvailableBands") & TUN_CONFIG_DEFAULT_AVAILABLEBAND_AM_MASKVAL) && (m_poclConfigInterface->u32GetConfigData("CFGAM")))
	{		//Send status of AM source availability to HMI
		enSrcAvail=ARL_EN_SRC_PLAYABLE;
		m_potunmstr_MsgToHMI->vSend_g_Tuner_Source_Availability_status();
	}

	m_poArlSourceImpl->vUpdateAudioSourceAvailabelStatus(enSource,enSrcAvail,enAvailabilityReason);
#endif

	}
//For system set coding, application state is change from Off to Normal, hence below code is commented
	/*if(u32SystemSetType==diaglib::EN_TYPE_CODING)
	{
		vHandleSystemSet_CodingRequest();
	}
	else*/
	{
		vSendSystemSetResult();
	}


#endif
}
tVoid::tunmstr_DiaglibListner::vSendSystemSetResult()
{
	if(m_poSystemSetIF!=NULL)
	{
		diaglib::tenSystemSetResult _enSysSetResult = diaglib::EN_SYSTEMSET_OK;
		diaglib::tclParameterVector _oErrorList;
		diaglib::tContext MsgContext =m_MsgContext;
		m_poSystemSetIF->bSendSystemSetResult( _enSysSetResult, _oErrorList, MsgContext);
		ETG_TRACE_USR1(("bSendSystemSetResult()"));
	}
}
tVoid tunmstr_DiaglibListner::vHandleSystemSetFinishRequest(tU32 u32SystemSetType)
{
	ETG_TRACE_USR1(("vHandleSystemSetFinishRequest() SystemSetType=%d", u32SystemSetType));
#ifndef VARIANT_S_FTR_ENABLE_FEATURE_GMMY16
	if (TUNERMASTER_HANDLE_PRESETLIST(u32SystemSetType))
	{
		ETG_TRACE_USR1(("vHandleSystemSetFinishRequest()"));
		if (NULL != m_potunmstr_Preset_Handler)
		{
			m_potunmstr_Preset_Handler->vHandleSystemSetFinishForPresetList();
		}
	}
#endif
}

tVoid tunmstr_DiaglibListner::vHandleTTFiS_CMD()
{
	vHandleSystemSet_CodingRequest();
}
tVoid tunmstr_DiaglibListner::vHandleSystemSet_CodingRequest()
{
	/*
	 * 1. Stop Antenna Diag timer
	 * 2. Turn off all the external power supply
	 * 3. Clear all internal variables with default values
	 * 4. Update Antenna state as De-activated
	 * 5. Read the KDS and start the antenna Diagnosis
	 * */
	if ((NULL == m_potunmstr_antdiag_Manager) || (NULL == m_potunmstr__DiagLog) ||
			(NULL == m_potunmstr_antdiag_i2cHandler) || (NULL == m_potunmstr_antdiag_Debounce) ||
#ifndef VARIANT_S_FTR_ENABLE_FEATURE_GMMY16
			(NULL==m_poclConfigInterface)||
#endif
	(NULL == m_potunmstr_antdiag_Measure))
	{
		return;
	}
	m_potunmstr_antdiag_Manager->vResetAntennaDiagTimers();
	m_potunmstr_antdiag_Manager->vTurnOffExtPowerSupply();
	m_potunmstr_antdiag_Manager->vClearAllInternalState();
	m_potunmstr__DiagLog->vClearAllInternalState();
	m_potunmstr_antdiag_i2cHandler->vClearAllInternalState();
	m_potunmstr_antdiag_Debounce->vClearAllInternalState();
	m_potunmstr_antdiag_Measure->vClearAllInternalState();
	for(tU8 u8index=(tU8)enFM1;u8index<(tU8)enClientNone;u8index++)
	{
		m_potunmstr_antdiag_Manager->vUpdateAntenaaStatetoHMI(u8index,enAntennState_Deactivated,TUNMSTR_ANTDIAG_ZERO_FLOAT);
	}
#ifndef VARIANT_S_FTR_ENABLE_FEATURE_GMMY16
	m_poclConfigInterface->vStartKDSRead();
#endif
}
tVoid tunmstr_DiaglibListner::vInsertMsgToRoutineControlMsgQ(stTunmstrRoutineControlMsg ostTunmstrRoutineControlMsg)
{
   std::lock_guard<std::mutex> lock(m_TunmstrRoutineControlMsgLock);
   m_stTunmstrRoutineControlMsg.push(ostTunmstrRoutineControlMsg);
}
tBool tunmstr_DiaglibListner::IsQEmpty()
{
   tBool blRet = true;
   std::lock_guard<std::mutex> lock(m_TunmstrRoutineControlMsgLock);
   if(!m_stTunmstrRoutineControlMsg.empty())
   {
      blRet = false;
   }
   return blRet;
}
tVoid tunmstr_DiaglibListner::vProcessRoutineControlReq()
{
   do
   {
      m_TunmstrRoutineControlMsgLock.lock();
      stTunmstrRoutineControlMsg ostTunmstrRoutineControlMsg;
      ostTunmstrRoutineControlMsg.enAntDiagClient = enClientNone;
      if(!m_stTunmstrRoutineControlMsg.empty())
      {
         ostTunmstrRoutineControlMsg = m_stTunmstrRoutineControlMsg.front();
         m_stTunmstrRoutineControlMsg.pop();
      }
      m_TunmstrRoutineControlMsgLock.unlock();
      if(ostTunmstrRoutineControlMsg.enAntDiagClient != enClientNone)
         vHandleRoutineControlReq(ostTunmstrRoutineControlMsg);
   }while(!IsQEmpty());
}
tVoid tunmstr_DiaglibListner::vHandleRoutineControlReq(stTunmstrRoutineControlMsg ostTunmstrRoutineControlMsg)
{
   switch(ostTunmstrRoutineControlMsg.enAntennaFeed)
   {
      case enDisablePower:
      {
         vHandleDisablePowerRequest(ostTunmstrRoutineControlMsg.enAntDiagClient,ostTunmstrRoutineControlMsg.MsgContext);
         break;
      }
      case enEnablePower:
      {
         vHandleEnablePowerRequest(ostTunmstrRoutineControlMsg.enAntDiagClient,ostTunmstrRoutineControlMsg.MsgContext);
         break;
      }
      case enPhantomPower:
      {
         vHandleEnablePhantomPowerRequest(ostTunmstrRoutineControlMsg.enAntDiagClient,ostTunmstrRoutineControlMsg.MsgContext);
         break;
      }
      default:
         break;
   }
}
