/************************************************************************
* FILE:           tun_Utility.cpp
* PROJECT:        Ford H/L RNS
* SW-COMPONENT: 
*----------------------------------------------------------------------
*
* DESCRIPTION:    Tuner Utility tasks are done .
*              
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
* HISTORY:      
* Date      | Author                       | Modification
* 12.05.05  | CM-DI/ESA2 ( RBIN ) Dinesh   | Initial version.
* 25.04.13  | NGP1KOR    | First version of the G3g after porting 
				from NISSAN LCN2Kai
*************************************************************************/


#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"




#include "tun_Utility.h"

/*------------------------------*/
/* Utility functions            */
/*------------------------------*/


/*************************************************************************
* FUNCTION:     tun_Utility::tun_Utility( )
* 
* DESCRIPTION:  Constructor
*
* PARAMETER:    void
*
* RETURNVALUE:  NAP
*
* vnd4kor: 13.1.2014
*************************************************************************/
tun_Utility::tun_Utility()
{
	/** Filling the map */  
	m_mapRdsEBU2Unicode[0x24]= 0xa4;
	m_mapRdsEBU2Unicode[0x5e]= 0x2014;
	m_mapRdsEBU2Unicode[0x60]= 0x2016;
	m_mapRdsEBU2Unicode[0x7f]= TUN_CHAR_SPACE;
	m_mapRdsEBU2Unicode[0x7e]= 0xaf;
	m_mapRdsEBU2Unicode[0x80]= 0xe1;
	m_mapRdsEBU2Unicode[0x81]= 0xe0;
	m_mapRdsEBU2Unicode[0x82]= 0xe9;
	m_mapRdsEBU2Unicode[0x83]= 0xe8;
	m_mapRdsEBU2Unicode[0x84]= 0xed;
	m_mapRdsEBU2Unicode[0x85]= 0xec;
	m_mapRdsEBU2Unicode[0x86]= 0xf3;
	m_mapRdsEBU2Unicode[0x87]= 0xf2;
	m_mapRdsEBU2Unicode[0x88]= 0xfa;
	m_mapRdsEBU2Unicode[0x89]= 0xf9;
	m_mapRdsEBU2Unicode[0x8a]= 0xd1;
	m_mapRdsEBU2Unicode[0x8b]= 0xc7;
	m_mapRdsEBU2Unicode[0x8c]= 0x15e;
	m_mapRdsEBU2Unicode[0x8d]= 0xdf;
	m_mapRdsEBU2Unicode[0x8e]= 0xa1;
	m_mapRdsEBU2Unicode[0x8f]= 0x132;

	m_mapRdsEBU2Unicode[0x90]= 0xe2;
	m_mapRdsEBU2Unicode[0x91]= 0xe4;
	m_mapRdsEBU2Unicode[0x92]= 0xea;
	m_mapRdsEBU2Unicode[0x93]= 0xeb;
	m_mapRdsEBU2Unicode[0x94]= 0xee;
	m_mapRdsEBU2Unicode[0x95]= 0xef;
	m_mapRdsEBU2Unicode[0x96]= 0xf4;
	m_mapRdsEBU2Unicode[0x97]= 0xf6;
	m_mapRdsEBU2Unicode[0x98]= 0xfb;
	m_mapRdsEBU2Unicode[0x99]= 0xfc;
	m_mapRdsEBU2Unicode[0x9a]= 0xf1;
	m_mapRdsEBU2Unicode[0x9b]= 0xe7;
	m_mapRdsEBU2Unicode[0x9c]= 0x15f;
	m_mapRdsEBU2Unicode[0x9d]= 0x11f;
	m_mapRdsEBU2Unicode[0x9e]= 0x131;
	m_mapRdsEBU2Unicode[0x9f]= 0x133;

	
	m_mapRdsEBU2Unicode[0xa0]= 0xaa;
	m_mapRdsEBU2Unicode[0xa1]= 0x03b1;
	m_mapRdsEBU2Unicode[0xa2]= 0xa9;
	m_mapRdsEBU2Unicode[0xa3]= 0x2030;
	m_mapRdsEBU2Unicode[0xa4]= 0x11e;
	m_mapRdsEBU2Unicode[0xa5]= 0x11b;
	m_mapRdsEBU2Unicode[0xa6]= 0x148;
	m_mapRdsEBU2Unicode[0xa7]= 0x151;
	m_mapRdsEBU2Unicode[0xa8]= 0x03c0;
	m_mapRdsEBU2Unicode[0xa9]= 0x20ac;
	m_mapRdsEBU2Unicode[0xaa]= 0xa3;
	m_mapRdsEBU2Unicode[0xab]= 0x24;
	m_mapRdsEBU2Unicode[0xac]= 0x2190;
	m_mapRdsEBU2Unicode[0xad]= 0x2191;
	m_mapRdsEBU2Unicode[0xae]= 0x2192;
	m_mapRdsEBU2Unicode[0xaf]= 0x2193;

	m_mapRdsEBU2Unicode[0xb0]= 0xba;
	m_mapRdsEBU2Unicode[0xb1]= 0xb9;
	m_mapRdsEBU2Unicode[0xb2]= 0xb2;
	m_mapRdsEBU2Unicode[0xb3]= 0xb3;
	m_mapRdsEBU2Unicode[0xb4]= 0xb1;
	m_mapRdsEBU2Unicode[0xb5]= 0x130;
	m_mapRdsEBU2Unicode[0xb6]= 0x144;
	m_mapRdsEBU2Unicode[0xb7]= 0x171;
	m_mapRdsEBU2Unicode[0xb8]= 0xb5;
	m_mapRdsEBU2Unicode[0xb9]= 0xbf;
	m_mapRdsEBU2Unicode[0xba]= 0xf7;
	m_mapRdsEBU2Unicode[0xbb]= 0xb0;
	m_mapRdsEBU2Unicode[0xbc]= 0xbc;
	m_mapRdsEBU2Unicode[0xbd]= 0xbd;
	m_mapRdsEBU2Unicode[0xbe]= 0xbe;
	m_mapRdsEBU2Unicode[0xbf]= 0xa7;
			
	m_mapRdsEBU2Unicode[0xc0]= 0xc1;
	m_mapRdsEBU2Unicode[0xc1]= 0xc0;
	m_mapRdsEBU2Unicode[0xc2]= 0xc9;
	m_mapRdsEBU2Unicode[0xc3]= 0xc8;
	m_mapRdsEBU2Unicode[0xc4]= 0xcd;
	m_mapRdsEBU2Unicode[0xc5]= 0xcc;
	m_mapRdsEBU2Unicode[0xc6]= 0xd3;
	m_mapRdsEBU2Unicode[0xc7]= 0xd2;
	m_mapRdsEBU2Unicode[0xc8]= 0xda;
	m_mapRdsEBU2Unicode[0xc9]= 0xd9;
	m_mapRdsEBU2Unicode[0xca]= 0x158;
	m_mapRdsEBU2Unicode[0xcb]= 0x10c;
	m_mapRdsEBU2Unicode[0xcc]= 0x160;
	m_mapRdsEBU2Unicode[0xcd]= 0x17d;
	m_mapRdsEBU2Unicode[0xce]= 0x110;
	m_mapRdsEBU2Unicode[0xcf]= 0x013f;
			
	m_mapRdsEBU2Unicode[0xd0]= 0xc2;
	m_mapRdsEBU2Unicode[0xd1]= 0xc4;
	m_mapRdsEBU2Unicode[0xd2]= 0xca;
	m_mapRdsEBU2Unicode[0xd3]= 0xcb;
	m_mapRdsEBU2Unicode[0xd4]= 0xce;
	m_mapRdsEBU2Unicode[0xd5]= 0xcf;
	m_mapRdsEBU2Unicode[0xd6]= 0xd4;
	m_mapRdsEBU2Unicode[0xd7]= 0xd6;
	m_mapRdsEBU2Unicode[0xd8]= 0xdb;
	m_mapRdsEBU2Unicode[0xd9]= 0xdc;
	m_mapRdsEBU2Unicode[0xda]= 0x159;
	m_mapRdsEBU2Unicode[0xdb]= 0x10d;
	m_mapRdsEBU2Unicode[0xdc]= 0x161;
	m_mapRdsEBU2Unicode[0xdd]= 0x17e;
	m_mapRdsEBU2Unicode[0xde]= 0x111;
	m_mapRdsEBU2Unicode[0xdf]= 0x0140;
	
	m_mapRdsEBU2Unicode[0xe0]= 0xc3;
	m_mapRdsEBU2Unicode[0xe1]= 0xc5;
	m_mapRdsEBU2Unicode[0xe2]= 0xc6;
	m_mapRdsEBU2Unicode[0xe3]= 0x152;
	m_mapRdsEBU2Unicode[0xe4]= 0x0177;
	m_mapRdsEBU2Unicode[0xe5]= 0xdd;
	m_mapRdsEBU2Unicode[0xe6]= 0xd5;
	m_mapRdsEBU2Unicode[0xe7]= 0xd8;
	m_mapRdsEBU2Unicode[0xe8]= 0xde;
	m_mapRdsEBU2Unicode[0xe9]= 0x014a;
	m_mapRdsEBU2Unicode[0xea]= 0x154;
	m_mapRdsEBU2Unicode[0xeb]= 0x106;
	m_mapRdsEBU2Unicode[0xec]= 0x15a;
	m_mapRdsEBU2Unicode[0xed]= 0x179;
	m_mapRdsEBU2Unicode[0xee]= 0x0166;
	m_mapRdsEBU2Unicode[0xef]= 0xf0;

	
	m_mapRdsEBU2Unicode[0xf0]= 0xe3;
	m_mapRdsEBU2Unicode[0xf1]= 0xe5;
	m_mapRdsEBU2Unicode[0xf2]= 0xe6;
	m_mapRdsEBU2Unicode[0xf3]= 0x153;
	m_mapRdsEBU2Unicode[0xf4]= 0x0175;
	m_mapRdsEBU2Unicode[0xf5]= 0xfd;
	m_mapRdsEBU2Unicode[0xf6]= 0xf5;
	m_mapRdsEBU2Unicode[0xf7]= 0xf8;
	m_mapRdsEBU2Unicode[0xf8]= 0xfe;
	m_mapRdsEBU2Unicode[0xf9]= 0x014b;
	m_mapRdsEBU2Unicode[0xfa]= 0x155;
	m_mapRdsEBU2Unicode[0xfb]= 0x107;
	m_mapRdsEBU2Unicode[0xfc]= 0x15b;
	m_mapRdsEBU2Unicode[0xfd]= 0x17a;
	m_mapRdsEBU2Unicode[0xfe]= 0x167;



}


/*************************************************************************
*
* FUNCTION:     tun_Utility::vConvert16BitTo2SeparateBytes( )
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  
*
*************************************************************************/
tVoid tun_Utility::vConvert16BitTo2SeparateBytes( tU8* pu8Message, tU16 u16Frequency )const
{
   pu8Message[0] = (tU8) (( u16Frequency & 0xff00) >> 8  );
   pu8Message[1] = (tU8) (  u16Frequency & 0x00ff);

}




/*************************************************************************
*
* FUNCTION:     tun_Utility::u16Convert2SeparateBytesTo16Bit( )
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  
*
*************************************************************************/
tU16  tun_Utility::u16Convert2SeparateBytesTo16Bit( const tU8* pu8NewData )const
{ 
    tU16 u16Frequency;

    u16Frequency  = ( tU16 ) ( ( (tU16)  pu8NewData[0]  ) << 8  ) ;
    u16Frequency  = ( tU16 ) (( u16Frequency)  | (  pu8NewData[1]  ));
    
    
    return u16Frequency;
}


/*************************************************************************
*
* FUNCTION:     tun_Utility::vConvert32BitTo4SeparateBytes( )
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  
*
*************************************************************************/
tVoid tun_Utility::vConvert32BitTo4SeparateBytes( tU8* pu8Message, tU32 u32Frequency )const
{ 
    
    pu8Message[0] = (tU8) (( u32Frequency & 0xff000000) >> 24 );
    pu8Message[1] = (tU8) (( u32Frequency & 0x00ff0000) >> 16 );
    pu8Message[2] = (tU8) (( u32Frequency & 0x0000ff00) >> 8  );
    pu8Message[3] = (tU8) (  u32Frequency & 0x000000ff);

}



/*************************************************************************
*
* FUNCTION:     tun_Utility::u32Convert4SeparateBytesTo32Bit( )
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  
*
*************************************************************************/
tU32 tun_Utility::u32Convert4SeparateBytesTo32Bit( const tU8* pu8NewData )const
{ 
    tU32 u32Frequency;

    u32Frequency  = ( tU32 ) ( ( (tU32)  pu8NewData[0]  ) << 24 ) ;
    u32Frequency |= ( tU32 ) ( ( (tU32)  pu8NewData[1]  ) << 16 ) ;
    u32Frequency |= ( tU32 ) ( ( (tU32)  pu8NewData[2]  ) << 8  ) ;
    u32Frequency |= ( tU32 ) (   (tU32)  pu8NewData[3]  );
    
    return u32Frequency;
}
/*************************************************************************
*
* FUNCTION:     tun_Utility::u32Convert3SeparateBytesTo32Bit( )
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE:
*
*************************************************************************/
tU32 tun_Utility::u32Convert3SeparateBytesTo32Bit( const tU8* pu8NewData) const
{
   tU32 u32data = 0;

   u32data |= ( tU32 ) ( ( (tU32)  pu8NewData[0]  ) << 16 ) ;
   u32data |= ( tU32 ) ( ( (tU32)  pu8NewData[1]  ) << 8  ) ;
   u32data |= ( tU32 ) (   (tU32)  pu8NewData[2]  );
   return u32data;
}

//#if 0

/*************************************************************************
*
* FUNCTION:     tun_Utility::u32ConvertFromUnicode2UTF8( )
* 
* DESCRIPTION:  Converts the Unicode character to UTF8 character
*
* PARAMETER:    tU16 - Unicode character
*
* RETURNVALUE:  tU32 - UTF8 character.
*
*************************************************************************/
tU32 tun_Utility::u32ConvertFromUnicode2UTF8( tU16 u16UnicodeChar )const
{ 
    tU32 u32UTF8Char = (tU32) TUN_INVALID_UTF8_CHAR;

    // If the Unicode character is less than 128(0x7f), equivalent UTF8 is one byte long
    // No conversion is required.
    if( u16UnicodeChar < (tU16) 128 )
    {
      u32UTF8Char = (tU32) u16UnicodeChar;

    }
    // If the Unicode data value is between 128( 0x7f) and 2047 (0x7ff), 
    // equivalent UTF8 is two byte long.
    else if( (u16UnicodeChar >= (tU16) 128) && (u16UnicodeChar <= (tU16) 2047) )
    {
      // Get the UTF8 MSB:
      u32UTF8Char = 192 + (u16UnicodeChar / 64 ); 

      u32UTF8Char = (tU32)(u32UTF8Char << 8);

      // OR with UTF8 LSB:
      u32UTF8Char |= ( 128 + (u16UnicodeChar % 64 ) );

    }
    /* 3 byte UTF 8 character.*/
    else if( (u16UnicodeChar >= (tU16) 2048) && (u16UnicodeChar < (tU16) 65535) )
    {
      tU16 u16TempUTF8Char = 0;
      // Get the UTF8 MSB. Byte1 = 224 + (ud div 4096)
      u32UTF8Char = (tU32) 224 + (u16UnicodeChar / 4096 ); 

      u32UTF8Char = (u32UTF8Char << 16);

      // Get the 2nd UTF8 MSB. Byte2 = 128 + ((ud div 64) mod 64)
      u16TempUTF8Char = (tU16)((tU16) 128 + ((u16UnicodeChar / 64 ) % 64));

      u16TempUTF8Char = ( tU16 )(u16TempUTF8Char << 8);

      u32UTF8Char |= (tU32) u16TempUTF8Char;

      // Get the LSB. Byte3 = 128 + (ud mod 64)
      u32UTF8Char |= ( 128 + (u16UnicodeChar % 64 ) );
    }
    
    return u32UTF8Char;
}


/*************************************************************************
*
* FUNCTION:     tun_Utility::u32ConvertFromRdsEBU2UTF8( )
* 
* DESCRIPTION:  Converts the RDS data in Air to UTF8 equivalent
*
* PARAMETER:    tU8 - RDS character received in air
*
* RETURNVALUE:  tU32 - UTF8 character.
*
*************************************************************************/
tU32 tun_Utility::u32ConvertFromRdsEBU2UTF8( tU8 u8RdsChar ) const
{ 
    tU32 u32UTF8Char = (tU32) TUN_INVALID_UTF8_CHAR;
     
    // Convert from RDS character to equivalent Unicode format.
    tU16 u16UniCodeChar = u16ConvertFromRdsEBU2Unicode( u8RdsChar );

    // Convert Unicode to UTF8 format.
    u32UTF8Char = u32ConvertFromUnicode2UTF8( u16UniCodeChar );     
    
    return u32UTF8Char;

}

/*************************************************************************
*
* FUNCTION:     tun_Utility::u16ConvertFromRdsEBU2Unicode( )
* 
* DESCRIPTION:  Convert the Rds data character received in air (EBU ISO 646 )
*               to Unicode data.
*
* PARAMETER:    RDS character
*
* RETURNVALUE:  Unicode equivalent value.
*
*************************************************************************/
tU16 tun_Utility::u16ConvertFromRdsEBU2Unicode( tU8 u8RdsChar )const
{ 
  
    mapRdsEBU2Unicode::const_iterator pos;
	if(!m_mapRdsEBU2Unicode.empty())
	{
		/** Find the RdsChar in map */
		pos = m_mapRdsEBU2Unicode.find(u8RdsChar);
	}

	/** Check if iterator reached end */
		if(m_mapRdsEBU2Unicode.end() != pos )
		{
			/** Return corresponding unicode value */
			return pos->second ;
		}
		else if((u8RdsChar >= 0x20) && (u8RdsChar <= 0x7d))
		{
			/** Return corresponding unicode value */
			return (tU16)u8RdsChar;
		}
		else
		{
			/** Invalid RdsChar */
			return TUN_INVALID_UNICODE_CHARACTER;
		}
	
}
//#endif
/*************************************************************************
*
* FUNCTION:     tun_Utility::u32GetCRC32( )
* 
* DESCRIPTION:  This Routine calculates the CRC32-Value for a given value,
*               passed as parameter
*               
*
* PARAMETER:    ubData     : New Value, which should be CRC-Checked
*               ulwPrevCRC : actual (previous) value of Checksum
*
* RETURNVALUE:  new value for Checksum.
*
*************************************************************************/
tU32 tun_Utility::u32GetCRC32( tU8 u8Data, tU32 u32PrevCRC) const
{
  tUInt unCnt;
  tU32 u32Temp = (u32PrevCRC ^ u8Data) & 0x000000FFUL;

  for (unCnt = 8U; unCnt > 0U; unCnt--)        /* Loop for each bit in char */
  {
      if ((u32Temp & 1UL) != 0UL)
      {
         u32Temp = (u32Temp >> 1) ^ TUN_CRC32_POLYNOM;
      }
      else {
        u32Temp >>= 1;
      }
  }
  u32PrevCRC = (u32PrevCRC >> 8) ^ u32Temp;

  return u32PrevCRC;                           /* return new Checksum-value */
}


/*************************************************************************
*
* FUNCTION:     tun_Utility::u32CalculateCRC32( )
* 
* DESCRIPTION:  This Routine calculates a CRC32 check sum for the
*               given buffer
*
* PARAMETER:    pu8Data    : data buffer
*               u32Length  : u32Length number of bytes
*
* RETURNVALUE:  Checksum.
*
*************************************************************************/
tU32 tun_Utility::u32CalculateCRC32( tPU8 pu8Data, tU32 u32Length) const
{
  tUInt unI;
  
  tU32 u32Crc32 = TUN_CRC32_DEFAULT;

  for(unI = 0; unI < u32Length; unI++)
  {
    if(unI == 440)
    {
    	// do nothing
    }
    else
    {
	  
	  u32Crc32 = u32GetCRC32( pu8Data[ unI], u32Crc32);
    }
  }
  return TUN_CRC32_RESULT( u32Crc32);
}


/*** END OF FILE *************************************************************/
