/************************************************************************
* FILE:           tun_RUIF_WaitingThread.cpp
* PROJECT:        Ford H/L RNS
* SW-COMPONENT: 
*----------------------------------------------------------------------
*
* DESCRIPTION:    Wating thread, waits for events.
*              
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
* HISTORY:      
* Date      | Author                       | Modification
* 03.02.05  | CM-DI/ESA2 ( RBIN ) Dinesh   | Initial version.
*
*************************************************************************/


#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"


#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#define SCD_S_IMPORT_INTERFACE_GENERIC
#include "scd_if.h"

#ifndef VARIANT_S_FTR_ENABLE_AIL_MOCK
#define AIL_S_IMPORT_INTERFACE_GENERIC
#include "ail_if.h"         // use AIL template with MessageMaps
#endif


#include "tun_Utility.h"
#include "tun_Manager.h"

#include "../Header/tun_MsgToHMI.h"

//#include "tun_MsgToHMI.h"
#include "tun_MsgDispatcher.h"

#include "tun_DrvAdrIf.hpp"

#include "tun_MsgToADR.h"

#include "tun_RUIF_WaitingThread.h"

#include "tun_defines.h"
#include "tun_trace.h"
#include "datatypes/tVector.h"

/*****************************************************************************
 * ETG trace
 * 0x2603:    TUN_TRACE_CLASS_WTHREAD
 ****************************************************************************/
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TUN_TRACE_CLASS_WTHREAD
#include "trcGenProj/Header/tun_RUIF_WaitingThread.cpp.trc.h"
#endif

/*-------------------*/
/* Static functions  */
/*-------------------*/
OSAL_tTimerHandle tun_RUIF_WaitingThread::_hTxTimer = 0;
OSAL_tSemHandle tun_RUIF_WaitingThread::m_hMsgRUIFWaitingThreadSem  = OSAL_C_INVALID_HANDLE;


/*************************************************************************
*
* FUNCTION:     tun_RUIF_WaitingThread::tun_RUIF_WaitingThread( )
* 
* DESCRIPTION:  constructor.
*
* PARAMETER:    -
*
* RETURNVALUE:  void
*
*************************************************************************/
tun_RUIF_WaitingThread::tun_RUIF_WaitingThread(  ):
		_hEvTuner(OSAL_C_INVALID_HANDLE),
		_fdTrace(OSAL_ERROR),
		m_potu_Manager( NULL ),
		m_potun_MsgDispatcher( NULL )
{
  ETG_TRACE_USR1(( " tun_RUIF_WaitingThread::tun_RUIF_WaitingThread() -> Constructor" ));

  /*-------------------*/
   /* Add timeout event */
   /*-------------------*/
   
   tS32 ret = OSAL_s32TimerCreate( ( OSAL_tpfCallback )vGetTxTimeout,
                                            ( tPVoid )this, &_hTxTimer ); 

	m_hMsgRUIFWaitingThreadSem = tuner_tclApp::_hADRDataUpdateSem;
   

   /* Initialize thread while loop */
   _bTerminate = FALSE;
	tBool bThreadSetUpRet = bThreadSetup(  );
	ETG_TRACE_USR1(( " tun_RUIF_WaitingThread::tun_RUIF_WaitingThread() -> Constructor Ret =%d bThreadSetUpRet =%d ", ret,bThreadSetUpRet)); 
   
}



/*************************************************************************
*
* FUNCTION:     tun_RUIF_WaitingThread::~tun_RUIF_WaitingThread( )
* 
* DESCRIPTION:  Destructor.
*
* PARAMETER:    -
*
* RETURNVALUE:  void
*
*************************************************************************/
tun_RUIF_WaitingThread::~tun_RUIF_WaitingThread( )
{
  ETG_TRACE_USR1(( " tun_RUIF_WaitingThread::~tun_RUIF_WaitingThread() -> Destructor" ));
  
  /* Delete the created thread */
  OSAL_s32ThreadDelete(_hInputThreadId);
  
  if( _hEvTuner != OSAL_NULL)
  {
      OSAL_s32EventClose( _hEvTuner );
      OSAL_s32EventDelete( EVENT_TUNER_SYSTEM_EVENT_HANDLE );
	  
  }

  m_potu_Manager = NULL;

  m_potun_MsgDispatcher = NULL;

}



/*************************************************************************
*
* FUNCTION:     tun_RUIF_WaitingThread::vSet_tu_Manager_Ptr( )
* 
* DESCRIPTION:  Destructor
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_RUIF_WaitingThread::vSet_tu_Manager_Ptr( tu_Manager* potu_Manager )
{
  if( potu_Manager != NULL )
  {
    m_potu_Manager = potu_Manager;
  }
}


/*************************************************************************
*
* FUNCTION:     tun_RUIF_WaitingThread::vSet_tun_MsgDispatcher_Ptr( )
* 
* DESCRIPTION:  Destructor
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_RUIF_WaitingThread::vSet_tun_MsgDispatcher_Ptr( tun_MsgDispatcher* potun_MsgDispatcher )
{
  if( potun_MsgDispatcher != NULL )
  {
    m_potun_MsgDispatcher = potun_MsgDispatcher;
  }
}


/*************************************************************************
*
* FUNCTION:     tun_RUIF_WaitingThread::bThreadSetup( )
* 
* DESCRIPTION:  Creates the tuner waiting thread.
*
* PARAMETER:    -
*
* RETURNVALUE:  void
*
*************************************************************************/
tBool tun_RUIF_WaitingThread::bThreadSetup( )
{
    tBool bRetVal = true;
    tS32 s32OsalError = OSAL_ERROR;    


    OSAL_trThreadAttribute  rAttr;
    tC8                     szThreadName[OSAL_C_U32_MAX_NAMELENGTH] = "\0";

    // Note:Maximum size of thread name limited to 8
    OSAL_szStringCopy( szThreadName, "TUN_RU");
	


    //------------------------------------------------------
    // Read data from registry, if fails use default values.
    //------------------------------------------------------

    // Local variables to store data read from registry
    tU32 u32ThreadPrio = 0;
    tU32 u32StackSize  = 0;     

    // Read thread priority from registry
    if ( FALSE == scd_bGetAppConfigurationValue 
                                      ( 
                                         // CCA App ID
                                         CCA_C_U16_APP_TUNER,

                                         // Registry path
                                         TUN_REGPATH_THREAD,

                                         // Registry entry for thread priority
                                         TUN_REGVALUE_THREAD_PRIO_NAME,

                                         // Read priority value
                                         &u32ThreadPrio 
                                       ) 
          )
    {
      // Assign default value.
      u32ThreadPrio = TUN_WAITINGTHREAD_DEFAULT_PRIO;
    
      // Indicate that the thread is running on default priority
      ETG_TRACE_ERR(( " tun_RUIF_WaitingThread::bThreadSetup() ->  the thread is running on default priority (=%d)."
                      ,u32ThreadPrio
                    ));
    }
    else
    {
      ETG_TRACE_USR1(( " tun_RUIF_WaitingThread::bThreadSetup() ->  the thread is running on Registry priority=%d."
                      ,u32ThreadPrio
                    ));
    }


    // Read thread stack size from registry
  if ( FALSE == scd_bGetAppConfigurationValue 
                                    ( 
                                       // CCA App ID
                                       CCA_C_U16_APP_TUNER,

                                       // Registry path
                                       TUN_REGPATH_THREAD,

                                       // Registry entry for stack size
                                       TUN_REGVALUE_STACK_SIZE_NAME,

                                       // Read stack size value
                                       &u32StackSize 
                                     )
     )
  {
    u32StackSize  = TUN_WAITINGTHREAD_DEFAULT_STACKSIZE;

    // Indicate that the thread is running on default priority
    ETG_TRACE_ERR(( " tun_RUIF_WaitingThread::bThreadSetup() ->  the thread is running with default stack size (=%d)."
                    ,u32StackSize
                  ));
  }
  else
  {
    ETG_TRACE_USR1(( " tun_RUIF_WaitingThread::bThreadSetup() ->  the thread is running with Registry stack size =%d."
                    ,u32StackSize
                  ));
  }




    s32OsalError = OSAL_s32EventCreate( EVENT_TUNER_SYSTEM_EVENT_HANDLE, &_hEvTuner);

    s32OsalError = (tS32)OSAL_u32ErrorCode( );
    

    if( s32OsalError == OSAL_ERROR)
    {
       bRetVal = false;   
         
      ETG_TRACE_ERR(( " tun_RUIF_WaitingThread::bThreadSetup() -> Event creation failed errorcode=%x"
                      ,s32OsalError
                    ));
    }
    else
    {
       ETG_TRACE_USR1(( " tun_RUIF_WaitingThread::bThreadSetup() ->  Event creation successful."
                    ));
    }

    // Initialize thread parameters
    rAttr.szName       = szThreadName;
    rAttr.s32StackSize = (tS32)u32StackSize; 
    rAttr.u32Priority  = u32ThreadPrio; 
    rAttr.pfEntry      = (OSAL_tpfThreadEntry)vInputThread;
    rAttr.pvArg        = (tPVoid)this;
    
    _hInputThreadId = OSAL_ThreadSpawn(&rAttr);

    if ( _hInputThreadId == OSAL_ERROR )
      {
        s32OsalError = (tS32)OSAL_u32ErrorCode(  );
        ETG_TRACE_ERR(( " tun_RUIF_WaitingThread::bThreadSetup() -> Thread spawn failed errorcode=%x"
                        ,s32OsalError
                      ));
        bRetVal = false;
      }
    else
    {
        ETG_TRACE_USR1(( " tun_RUIF_WaitingThread::bThreadSetup() -> Thread creation successful."
                      ));
    }
    return bRetVal;
    
}



/*************************************************************************
*
* FUNCTION:     tun_RUIF_WaitingThread::vInputThread( )
* 
* DESCRIPTION:  Thread waits for events here.
*
* PARAMETER:    -
*
* RETURNVALUE:  void
*
*************************************************************************/

tVoid tun_RUIF_WaitingThread::vInputThread(tVoid *pvArg)
{
   amt_tclBaseMessage     oMsgObject;


   OSAL_tEventMask                hEvRequest = 0; 
   tun_RUIF_WaitingThread*        potun_RUIF_WaitingThread;

   potun_RUIF_WaitingThread = (tun_RUIF_WaitingThread*) pvArg;



   ET_TRACE_INFO_BIN( TUN_TRACE_CLASS_WTHREAD, ET_EN_T16 _ TUN_TRACE_WTHREAD_THREAD_STARTED _ ET_EN_DONE);
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
   tU32 u32mask =    EVENT_INC_RX_READY | EVENT_TERMINATE | EVENT_FROM_MAINTHREAD_UPDATE_MANTUN | EVENT_INC_TX_READY | EVENT_PING_TIMER | EVENT_NEW_ADR_STATE | EVENT_ADVISORY_MSG | EVENT_HDAUDIODECODING;
#else
   tU32 u32mask =    EVENT_INC_RX_READY | EVENT_TERMINATE | EVENT_FROM_MAINTHREAD_UPDATE_MANTUN | EVENT_INC_TX_READY | EVENT_PING_TIMER | EVENT_NEW_ADR_STATE ;
#endif
   /*---------------------*/
   /* - main input loop - */
   /*---------------------*/

   


   while ( (potun_RUIF_WaitingThread->_bTerminate) == FALSE )
   {
     
     ET_TRACE_INFO_BIN( TUN_TRACE_CLASS_WTHREAD, ET_EN_T16 _ TUN_TRACE_WTHREAD_LOOP_RUNNING _ ET_EN_DONE);
      
      
      if (OSAL_OK == OSAL_s32EventWait (  potun_RUIF_WaitingThread -> _hEvTuner,
                                          u32mask,
                                          OSAL_EN_EVENTMASK_OR,
                                          OSAL_C_TIMEOUT_FOREVER, 
                                          &hEvRequest)
         ) 
      {
         ET_TRACE_INFO_BIN( TUN_TRACE_CLASS_WTHREAD, ET_EN_T16 _ TUN_TRACE_WTHREAD_EVENT_MASK _
                            ET_EN_T32 _ hEvRequest _ ET_EN_DONE);
        
         /*--------------*/
         /* clear event  */
         /*--------------*/
        OSAL_s32EventPost (potun_RUIF_WaitingThread -> _hEvTuner,~hEvRequest,OSAL_EN_EVENTMASK_AND);
        
         /*====================================================================
          * EVENT_INC_RX_READY
          *------------------------------------------------------------------*/
         if ( (EVENT_INC_RX_READY & hEvRequest) && ((potun_RUIF_WaitingThread->_bTerminate) == FALSE) )
         {
           /*---------------------------*/ 
           /* message from SPI received */
           /*---------------------------*/
           ETG_TRACE_USR1(( " tun_RUIF_WaitingThread::vInputThread() ->  New Rx Event received.\n"
                            "================ BEGIN of message dispatching ===================="
                         ));

           potun_RUIF_WaitingThread -> vHandleReceivedMessage(  );

           ETG_TRACE_USR1(( " tun_RUIF_WaitingThread::vInputThread()\n"
                            "================ END   of message dispatching ===================="
                         ));

         }
         /*==================================================================*/

         /*====================================================================
          * EVENT_INC_TX_READY
          *------------------------------------------------------------------*/
         if( (EVENT_INC_TX_READY & hEvRequest) && ((potun_RUIF_WaitingThread->_bTerminate) == FALSE) )
         {
           /*---------------------------*/ 
           /* message to SPI compleeted */
           /*---------------------------*/
           ET_TRACE_INFO_BIN( TUN_TRACE_CLASS_WTHREAD, ET_EN_T16 _ TUN_TRACE_TX_EVENT_RECEIVED _ ET_EN_DONE);
           
           // now try to start the next transfer
           potun_RUIF_WaitingThread->vHandleConfirmation();
         }
         /*==================================================================*/

         /*====================================================================
          * EVENT_PING_TIMER
          *------------------------------------------------------------------*/
         if( (EVENT_PING_TIMER & hEvRequest) && ((potun_RUIF_WaitingThread->_bTerminate) == FALSE) )
         {
           ETG_TRACE_USR1(( " tun_RUIF_WaitingThread::vInputThread() -> EVENT_PING_TIMER\n" ));
           tun_DrvAdrIf::instance()->vPingTimerExpired();
         }
         /*==================================================================*/

         /*====================================================================
          * EVENT_NEW_ADR_STATE
          *------------------------------------------------------------------*/
         if( (EVENT_NEW_ADR_STATE & hEvRequest) && ((potun_RUIF_WaitingThread->_bTerminate) == FALSE) )
         {
           ETG_TRACE_USR1(( " tun_RUIF_WaitingThread::vInputThread() -> EVENT_NEW_ADR_STATE\n" ));
           tun_DrvAdrIf::instance()->vNewADRState();
         }
         /*==================================================================*/

#ifdef CODE_NOT_USED
         //Handle Event received from main thread (For manual tuning frequence updation )
         if ( (EVENT_FROM_MAINTHREAD_UPDATE_MANTUN & hEvRequest) && ( (potun_RUIF_WaitingThread->_bTerminate) == FALSE) )
         {
           TUN_TRACE(99);
           
           potun_RUIF_WaitingThread -> vHandleReceivedEvents( EVENT_FROM_MAINTHREAD_UPDATE_MANTUN );
           
           TUN_TRACE(98);
         }
#endif

#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
 		 /*====================================================================
          * EVENT_ADVISORY_MSG
          *------------------------------------------------------------------*/
         if( (EVENT_ADVISORY_MSG & hEvRequest) && ((potun_RUIF_WaitingThread->_bTerminate) == FALSE) )
         {
           ETG_TRACE_USR1(( " tun_RUIF_WaitingThread::vInputThread() -> EVENT_ADVISORY_MSG\n" ));
           tun_MsgToHMI::instance()->vAdvisoryTimerExpired();
         }
		 /*====================================================================
          * EVENT_HDAUDIODECODING
          *------------------------------------------------------------------*/
         if( (EVENT_HDAUDIODECODING & hEvRequest) && ((potun_RUIF_WaitingThread->_bTerminate) == FALSE) )
         {
           ETG_TRACE_USR1(( " tun_RUIF_WaitingThread::vInputThread() -> EVENT_HDAUDIODECODING\n" ));
           tun_MsgToHMI::instance()->vHDStationInfoBitTimerExpired();
         }
#endif

         /*====================================================================
          * EVENT_TERMINATE
          *------------------------------------------------------------------*/
         if( EVENT_TERMINATE & hEvRequest)
         {
           // Do nothing ...
         }
         /*==================================================================*/
         
      }
      else
      {       
        ET_TRACE_INFO_BIN( TUN_TRACE_CLASS_WTHREAD, ET_EN_T16 _ TUN_TRACE_SYSLAYER_LOOP_NOEVENT _ ET_EN_DONE);
      }
      
   } /* while ( (poAudioVd->_bTerminate) == FALSE ) */

   ET_TRACE_INFO_BIN ( TUN_TRACE_CLASS_WTHREAD, ET_EN_T16 _ TUN_TRACE_SYSLAYER_LOOP_END _ ET_EN_DONE);

   return ;
}
#ifdef CODE_NOT_USED
/*************************************************************************
*
* FUNCTION:     tun_RUIF_WaitingThread::vHandleReceivedEvents( )
* 
* DESCRIPTION:  Thread waits for events here.
*
* PARAMETER:    -
*
* RETURNVALUE:  void
*
*************************************************************************/

tVoid tun_RUIF_WaitingThread::vHandleReceivedEvents(tU16 u16Event)
{
  m_potun_MsgDispatcher -> bDispatchRUMessage ( u16Event );
  TUN_TRACE(73);
}
#endif



/*************************************************************************
*
* FUNCTION:     tun_RUIF_WaitingThread::vHandleReceivedMessage( )
* 
* DESCRIPTION:  Thread waits for events here.
*
* PARAMETER:    -
*
* RETURNVALUE:  void
*
*************************************************************************/

tVoid tun_RUIF_WaitingThread::vHandleReceivedMessage(  )
{
	(tVoid)OSAL_s32SemaphoreWait( m_hMsgRUIFWaitingThreadSem, OSAL_C_U32_INFINITE );

	/** Loop until the last message in the queue is processed*/
	while(! tun_DrvAdrIf::instance()->bCheckIfQueueEmpty())
	{
		tPU8  pu8RxBuffer;
		tU32  u32MessageLen =0;

		ET_TRACE_INFO_BIN ( TUN_TRACE_CLASS_WTHREAD, ET_EN_T16 _ TUN_TRACE_WTHREAD_HANDLE_RX_MSG _ ET_EN_DONE);

		u32MessageLen = tun_DrvAdrIf::instance()->u32GetRxDataBuffer( &pu8RxBuffer);

		if( u32MessageLen != 0)
		{
			/*-----------------------------------*/
			/* New message in received, from RU  */
			/*-----------------------------------*/
			ET_TRACE_INFO_BIN ( TUN_TRACE_CLASS_WTHREAD, ET_EN_T16 _ TUN_TRACE_WTHREAD_RX_MSG_INDICATION _ ET_EN_DONE);


			/*------------------------------------------------*/
			/* Dispatch the RU message to the concerned class */
			/*------------------------------------------------*/       
			if( m_potun_MsgDispatcher != NULL )
			{
				m_potun_MsgDispatcher->bDispatchRUMessage( pu8RxBuffer, u32MessageLen);

				if(TUN_DRVADRIF_GET_U16( &pu8RxBuffer[enAdrMsgOffset_FBLOCK_ID]) != (tU16)enFBlockId_TMCTuner)
				{
		            tunerVector<unsigned char> msg(pu8RxBuffer, pu8RxBuffer+u32MessageLen);
		            tun_MsgToADR::instance()->vHandleMessage(msg);
				}

			}
			tun_DrvAdrIf::instance()->vFreeRxDataBuffer();

		}
	}
	(tVoid)OSAL_s32SemaphorePost( m_hMsgRUIFWaitingThreadSem );
}
/*************************************************************************
*
* FUNCTION:     tun_RUIF_WaitingThread::vSetTxTimeOut( )
* 
* DESCRIPTION:  Thread waits for events here.
*
* PARAMETER:    -
*
* RETURNVALUE:  void
*
*************************************************************************/

tVoid tun_RUIF_WaitingThread::vSetTxTimeOut( tU8 u8MsTimeout )
{
  /*--------------*/
  /* Timer active */
  /*--------------*/
  (tVoid)OSAL_s32TimerSetTime( _hTxTimer, (tU32) u8MsTimeout, 0 );

  ET_TRACE_INFO_BIN ( TUN_TRACE_CLASS_WTHREAD, ET_EN_T16 _ TUN_TRACE_WTHREAD_TIMER_EVENT _ ET_EN_DONE);
}





/*************************************************************************
*
* FUNCTION:     tun_RUIF_WaitingThread::vGetTxTimeout( )
* 
* DESCRIPTION:  Thread waits for events here.
*
* PARAMETER:    -
*
* RETURNVALUE:  void
*
*************************************************************************/

OSAL_tpfCallback tun_RUIF_WaitingThread::vGetTxTimeout( const tVoid* pArg )//pArg declared as poiting to const to solve prio3 lint warning
{
  /* validation of the pointer and data length */

  if(pArg == NULL )
  {
      NORMAL_M_ASSERT( pArg != NULL ); 
  }


  return 0;
}

/*************************************************************************
*
* FUNCTION:     tun_RUIF_WaitingThread::vPostEventToSelf( )
* 
* DESCRIPTION:  Post an self event to Waiting thread.
*
* PARAMETER:    -
*
* RETURNVALUE:  void
*
*************************************************************************/

tVoid tun_RUIF_WaitingThread::vPostEventToSelf(  ) const//function made const to solve prio3 lint warning
{
   /*------------------*/
  /* set event handle */
  /*------------------*/
   OSAL_tEventHandle hRxSPI = 0;
          
   ET_TRACE_INFO_BIN( TUN_TRACE_CLASS_WTHREAD, ET_EN_T16 _ TUN_TRACE_WTHREAD_SELF_POSE_EVENT _ ET_EN_DONE);

   if(OSAL_s32EventOpen(EVENT_TUNER_SYSTEM_EVENT_HANDLE, &hRxSPI) == OSAL_OK)
   {
      
      OSAL_s32EventPost ( hRxSPI, EVENT_TERMINATE, OSAL_EN_EVENTMASK_OR);
      OSAL_s32EventClose( hRxSPI);
	  
      ET_TRACE_INFO_BIN( TUN_TRACE_CLASS_WTHREAD, ET_EN_T16 _ TUN_TRACE_WTHREAD_SELF_POSE_EVENT_SUCC _ ET_EN_DONE );
   }
   else
   {
     ET_TRACE_ERROR_BIN( TUN_TRACE_CLASS_WTHREAD, ET_EN_T16 _ TUN_TRACE_WTHREAD_SELF_POSE_EVENT_FAIL _ ET_EN_DONE);
   }
}


/*****************************************************************************
 * 
 ****************************************************************************/
tVoid tun_RUIF_WaitingThread::vHandleConfirmation( tVoid) const
{

  tun_MsgToADR::instance()->vMessageSentConfirmation();

}
