/************************************************************************
* FILE:           tun_MsgToADR.cpp
* PROJECT:        g3g
* SW-COMPONENT: 
*----------------------------------------------------------------------
*
* DESCRIPTION:    All the message sent to RU is sent by this class.
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------

* HISTORY:      
* Date      | Author                       | Modification
* 21.01.05  | CM-DI/ESA2 ( RBIN ) Dinesh   | Initial version.
* 25.04.13  | NGP1KOR    | First version of the G3g after porting 
				from NISSAN LCN2Kai
*************************************************************************/



#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#include "tun_main.h"

#include "tun_ActualData.h"

#include "tun_Utility.h"

#include "tun_MsgToHMI.h"
#include "tun_Config.h"
#include "tun_Manager.h"

#include "tun_DrvAdrIf.hpp"

#include "tun_MsgToADR.h"
#include "tun_diaghandler.h"
#include "tun_MsgDefines.h"
#include "conversionTable/midwTunerToAdr3Map.h"
#include "tun_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
	#include "Aars/clAars_HDTestMode.h"
	#include "AMFMTuner/clTmcTuner_Method_S_SET_ACTIVE_PRESETLIST.h"
	#include "Aars/clAars_ATPresetList_XX.h"
	#include "Aars/clAars_CM_SetUserOptions.h"
#endif
#ifndef VARIANT_P_MIDDLEWARE_FC_TMCTUNER_GTEST
	#include "TunerApp/clTmcTuner_AdditionalVersionInfo_Update.h"
#endif
/*****************************************************************************
 * ETG trace
 * 0x2619:    TUN_TRACE_CLASS_MSGTOADR
 ****************************************************************************/
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TUN_TRACE_CLASS_MSGTOADR
#include "trcGenProj/Header/tun_MsgToADR.cpp.trc.h"
#endif


#include "../Header/tun_defines.h"
#include "../Diagnostics/tun_Diag_defines.h"
#ifndef VARIANT_S_FTR_ENABLE_DP_CLIENT_MOCK
   #define DP_S_IMPORT_INTERFACE_FI
   #include "dp_tuner_if.h"
   #include "dp_generic_if.h"
#else
   #include"diaglib_if.h"
#endif

#include "TunerApp/clTmcTuner_SetConfigSections.h"
#include "Aars/clAars_CM_AdditionalVersionInfo.h"
#include "Aars/clAars_GetConfigurationInfo.h"
#include "kdsconfig/clConfigInterface.h"

#ifdef VARIANT_S_FTR_ENABLE_AARSPROXY
#include "AarsDBusProxy/tuner_gio_dbus_handler.h"
#include "AarsDBusProxy/tuner_gio_dbus_aarsproc_proxy.h"
#endif
#include "tun_MessageQ.h"
// Initialize the semaphore variable:
OSAL_tSemHandle tun_MsgToADR::m_hSendToRU_Sync = OSAL_C_INVALID_HANDLE;

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR::tun_MsgToADR( )
* 
* DESCRIPTION:  Constructor
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*
*************************************************************************/
tun_MsgToADR::tun_MsgToADR(  ):u32MessageLength(0),
                             m_bReadCheckSumAfterConfig( FALSE),
                             m_bTestModeActivity( FALSE),
							 m_bResetADRFlag(TRUE),
							 m_bLandscapeRequested(FALSE),
							 m_u32ADRVersion(0),
							 m_NumofAtBandReqAtStartUp(0),
                             u8HWCrc(0),
							 m_bAtBandSent(FALSE),
							 m_u8AtBandRequested(0xff),
							 bSetDefaultSettingMsgSent(FALSE)

{ 
	m_bNotificationSent = FALSE;
	m_bPersonalisationReq = FALSE;
    memset(m_sADRHWVersion,0,enCM_VersionInf_len_HardwareVersion);
    memset(m_sADRSWVersion,0,enCM_VersionInf_len_SoftwareVersion);
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
    m_poclTmcTuner_Method_S_SET_ACTIVE_PRESETLIST=NULL;
    m_poclAars_ATPresetList_XX=NULL;
#endif
    /*-----------------------*/
    /* Create a message Q    */
    /*-----------------------*/
    m_ptun_MessageQ = new tun_MessageQ( );
	m_potun_Utility = NULL;
	m_potun_ActualData = NULL;
	m_potun_Config = NULL;
	m_potu_Manager = NULL;
	m_potun_MsgToHMI = NULL;
	m_potun_main = NULL;
	m_poclConfigInterface = NULL;
	m_potun_tclDiagHandler=NULL;
    m_poclAars_CM_SetUserOptions = NULL;
	m_u8CurrentUserId=0;
	// Create semaphore to synchronise the data sent to RU.
	tS32 s32RetCode = OSAL_s32SemaphoreCreate( TUNER_SEND_DATA_TO_RU_SYNC_SEM_NAME, &m_hSendToRU_Sync, 1);
	if (OSAL_OK != s32RetCode)
	{
		NORMAL_M_ASSERT (OSAL_OK == s32RetCode);
	}
}





/*************************************************************************
*
* FUNCTION:     tu_Manager::~tu_Manager( )
* 
* DESCRIPTION:  Destructor
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*
*************************************************************************/
tun_MsgToADR::~tun_MsgToADR( )
{
  if( m_ptun_MessageQ != NULL )
  {
    vDeleteNoThrow( m_ptun_MessageQ );
  }
  m_potun_Utility = NULL;

  m_potun_ActualData = NULL;

  m_potun_Config = NULL;
  
    m_potun_MsgToHMI = NULL;
	
	m_potu_Manager = NULL;

    m_potun_main = NULL;

    m_poclConfigInterface = NULL;
	m_bAtBandSent = false;
  OSAL_s32SemaphoreClose( m_hSendToRU_Sync );
  OSAL_s32SemaphoreDelete( TUNER_SEND_DATA_TO_RU_SYNC_SEM_NAME );
  m_bTestModeActivity = FALSE;
  m_bResetADRFlag = TRUE;
  u8HWCrc = 0;
  m_bLandscapeRequested =  FALSE;
  m_bNotificationSent = FALSE;
  m_bPersonalisationReq = FALSE;
  m_u8CurrentUserId=0;
  bSetDefaultSettingMsgSent=FALSE;
  m_potun_tclDiagHandler=NULL;
  m_NumofAtBandReqAtStartUp = 0;
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
  m_poclTmcTuner_Method_S_SET_ACTIVE_PRESETLIST=NULL;
  m_poclAars_ATPresetList_XX=NULL;
  m_poclAars_CM_SetUserOptions = NULL;
#endif
}


/*************************************************************************
*
* FUNCTION:     tun_MsgToADR::vSet_tun_Utility_Ptr( )
* 
* DESCRIPTION:  
*
* PARAMETER:    tVoid
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_MsgToADR::vSet_tun_Utility_Ptr( tun_Utility* potun_Utility )
{
  if( potun_Utility != NULL )
  {  
    m_potun_Utility = potun_Utility;
  }
  else
  {
    /* Assert to do */
  }
}


/*************************************************************************
*
* FUNCTION:     tun_MsgToADR::vSet_tun_MsgToHMI_Ptr( )
* 
* DESCRIPTION:   With this config data can be accessed
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_MsgToADR::vSet_tun_MsgToHMI_Ptr( tun_MsgToHMI* potun_MsgToHMI )
{
  if( potun_MsgToHMI != NULL )
  {
    m_potun_MsgToHMI = ( tun_MsgToHMI* )potun_MsgToHMI;
  }
  
}


tVoid tun_MsgToADR::vSet_tun_main_Ptr( tuner_tclApp* potun_main )
{
  if( potun_main != NULL )
  {
    m_potun_main = ( tuner_tclApp* )potun_main;
  }
  
}

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR::vSet_tun_ActualData_Ptr( )
* 
* DESCRIPTION:  
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_MsgToADR::vSet_tun_ActualData_Ptr( tun_ActualData* potun_ActualData )
{
  if( potun_ActualData != NULL )
  {
    m_potun_ActualData = ( tun_ActualData* )potun_ActualData;
  }
}



/*************************************************************************
*
* FUNCTION:     tun_MsgToADR::vSet_tun_Config_Ptr( )
* 
* DESCRIPTION:   With this config data can be accessed
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_MsgToADR::vSet_tun_Config_Ptr( tun_Config* potun_Config )
{
  if( potun_Config != NULL )
  {
    m_potun_Config = ( tun_Config* )potun_Config;
  }
  
}

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR::vSet_tun_ConfigInterface_Ptr( )
*
* DESCRIPTION:   With this config data can be accessed
*
* PARAMETER:    clConfigInterface* poclConfigInterface
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_MsgToADR::vSet_tun_ConfigInterface_Ptr( clConfigInterface* poclConfigInterface )
{
  if( poclConfigInterface != NULL )
  {
	  m_poclConfigInterface = poclConfigInterface;
  }

}


/*************************************************************************
*
* FUNCTION:     tu_Manager::vSet_tu_Manager_Ptr( )
* 
* DESCRIPTION:   With this config data can be accessed
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_MsgToADR::vSet_tu_Manager_Ptr( tu_Manager* potu_Manager )
{
  if( potu_Manager != NULL )
  {
    m_potu_Manager = ( tu_Manager* )potu_Manager;
  }
  
}
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
tVoid tun_MsgToADR::vSet_TmcTuner_Method_S_SET_ACTIVE_PRESETLISTPtr(clTmcTuner_Method_S_SET_ACTIVE_PRESETLIST* poclTmcTuner_Method_S_SET_ACTIVE_PRESETLIST)
{
	if(NULL != poclTmcTuner_Method_S_SET_ACTIVE_PRESETLIST)
	{
		m_poclTmcTuner_Method_S_SET_ACTIVE_PRESETLIST =(clTmcTuner_Method_S_SET_ACTIVE_PRESETLIST*)poclTmcTuner_Method_S_SET_ACTIVE_PRESETLIST;
	}

}
#endif


/*************************************************************************
*
* FUNCTION:     tu_Manager::vSet_tun_tclDiagHandler( )
*
* DESCRIPTION:
*
* PARAMETER:    tun_tclDiagHandler* potun_tclDiagHandler
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_MsgToADR :: vSet_tun_tclDiagHandler(tun_tclDiagHandler* potun_tclDiagHandler )
{
   if( potun_tclDiagHandler != NULL )
   {
      m_potun_tclDiagHandler = ( tun_tclDiagHandler* )potun_tclDiagHandler;
   }
   else
   {
      NORMAL_M_ASSERT( potun_tclDiagHandler != NULL );
   }
}


/*************************************************************************
*
* FUNCTION:     tun_MsgToADR::vSet_SetUserOptions_Ptr( )
* 
* DESCRIPTION:  
*
* PARAMETER:    tVoid
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_MsgToADR::vSet_SetUserOptions_Ptr( clAars_CM_SetUserOptions* poclAars_CM_SetUserOptions )
{
  if( poclAars_CM_SetUserOptions != NULL )
  {  
    m_poclAars_CM_SetUserOptions = poclAars_CM_SetUserOptions;
  }
  else
  {
    /* Assert to do */
  }
}


/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSend( )
* 
* DESCRIPTION:  Serialize the message into a buffer and trace it. Afterwarts
*               put the filles buffer into the message queue.
*
* PARAMETER:    tun_TxMsg* pu8Message
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vSend( tun_TxMsg* pMsg)
{
  tun_SsiTxBuffer oTxBuffer;
  
  pMsg->vSerialize( &oTxBuffer);
  pMsg->vTrace();

  vWaitOnMessageQue( oTxBuffer.au8TxBuffer, oTxBuffer.u32MessageLen );
}


/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSetAdrCommunication(  tBool bState)
* 
* DESCRIPTION:  Communication state of ADR.
*               
*
* PARAMETER:     tBool bState  TRUE  : communication started
*                              FALSE : communication stopped
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vSetAdrCommunication( tBool bState)
{
  if((m_potun_main==NULL) || (m_poclConfigInterface == NULL))
  {
	  return;
  }
  /*======================================================================
   * sends the first message after startup message from ADR
   *--------------------------------------------------------------------*/
  if( bState == TRUE)
  {

    // Synchronise the function call made by many clients with semaphore
    (tVoid)OSAL_s32SemaphoreWait( m_hSendToRU_Sync, OSAL_C_U32_INFINITE );
	if(!m_poclConfigInterface->u32GetConfigData("SBR"))
	{
		if(m_potun_main->u32GetRestartTimeValue()==enDgramSendError)
		{	
			(tVoid)OSAL_s32TimerSetTime(m_potun_main->_hTimerHandler,0, 0);
			ETG_TRACE_ERRMEM(("tun_MsgToADR::vSetAdrCommunication(): reset dgramSendError Timer"));
			m_potun_main->vSetRestartTimeValue(enNoError);
		}
	}
    vSendNextMessageFromQueue();

    // On exit, release the semaphore
    (tVoid)OSAL_s32SemaphorePost( m_hSendToRU_Sync );
  }
  /*====================================================================*/
}

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vClearMessageQueue( )
*
* DESCRIPTION:  Clearing all message in queue
*
* PARAMETER:    tVoid
*
* RETURNVALUE:  tVoid
*
*************************************************************************/
tVoid tun_MsgToADR::vClearMessageQueue( tVoid)
{
    ETG_TRACE_USR1(( " tun_MsgToADR::vClearMessageQueue()"));
    if (m_ptun_MessageQ != NULL)
    {
        // Synchronise the function call made by many clients with semaphore
        (tVoid)OSAL_s32SemaphoreWait( m_hSendToRU_Sync, OSAL_C_U32_INFINITE );

        m_ptun_MessageQ->vClearQueue();

        // On exit, release the semaphore
        (tVoid)OSAL_s32SemaphorePost( m_hSendToRU_Sync );
    }
}

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendNextMessageFromQueue( )
* 
* DESCRIPTION:  Reads a message from queue and send it via SSI to ADR
*
* PARAMETER:    tVoid
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vSendNextMessageFromQueue( tVoid)
{
    if ((m_ptun_MessageQ != NULL) && (m_potun_main != NULL) && (m_poclConfigInterface != NULL))
    {
        if((FALSE == bSetDefaultSettingMsgSent) && (m_potun_main->m_enRestartTimeValue==enNoError))
        {
            tU8  u8aMessageBuffer[ TUN_DRVADRIF_CONF_SSI_AMFM_TUNER_TX_BUFFER_SIZE ];
            tU32 u32NewMessageLen;

            /*---------------------------------------*/
            /* Get next message from the message Que */
            /*---------------------------------------*/
            if( m_ptun_MessageQ -> fGetNextMessage( u8aMessageBuffer, &u32NewMessageLen ) )
            {
                /*----------------------------------------------*/
                /* Waiting for confirmation, message put in SPI */
                /*----------------------------------------------*/
                m_ptun_MessageQ -> vWaitingForConfirmation( );

                /*----------------------------*/
                /* Message sent to Radio unit */
                /*----------------------------*/
                tun_DrvAdrIf::instance()->vSendMessage( u8aMessageBuffer, u32NewMessageLen );

                ET_TRACE_BIN( TUN_TRACE_CLASS_MSGTOADR, TR_LEVEL_USER_4, ET_EN_T16 _ TUN_TRACE_MSGTOADR_SEND_MSGTO_RU_2 _ ET_EN_DONE );

                ET_TRACE_BIN( TUN_TRACE_CLASS_MSGTOADR, TR_LEVEL_USER_4, ET_EN_T16 _ TUN_TRACE_MSGTOADR_WAITING_FOR_CONF_2 _ ET_EN_DONE );
            }
        }
    }
}

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vWaitOnMessageQue( )
* 
* DESCRIPTION:  Message to be sent to RU is put in Que.
*
* PARAMETER:    tU8* pu8Message, tU32 u32MessageLen
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vWaitOnMessageQue( unsigned char* pu8Message,  unsigned long  u32MessageLen, unsigned char bSendAlways)
{
	ETG_TRACE_USR4(( " tun_MsgToADR::vWaitOnMessageQue() -> Write message into the Queue, bSendAlways = %d", bSendAlways));
	// Synchronise the function call made by many clients with semaphore
	tS32 ret = OSAL_s32SemaphoreWait( m_hSendToRU_Sync, OSAL_C_U32_INFINITE );


	if( ( pu8Message != NULL ) && ( u32MessageLen != 0 ) )
	{
		/*---------------------------*/
		/* Add the message in the Q  */
		/*---------------------------*/
		if( m_ptun_MessageQ != NULL )
		{
			if( (tun_DrvAdrIf::instance()->bIsAdrInDiagnoseSession() == FALSE) || (bSendAlways == TRUE) )
			{
				m_ptun_MessageQ ->vAddNewMessageInQ( pu8Message, (tU32)u32MessageLen, 0);

				ETG_TRACE_USR4(( " tun_MsgToADR::vWaitOnMessageQue() -> New message in queue added ret=%d",ret ));
			}
			else
			{
				ETG_TRACE_COMP(( " tun_MsgToADR::vWaitOnMessageQue() -> ADR interface currently disabled, new message is not posted." ));
			}
		}

		/*-----------------------------------------------------------------------------------*/
		/* Send new message to RU, only if not waiting for confirmation for previous message */
		/*-----------------------------------------------------------------------------------*/
		if( m_ptun_MessageQ != NULL )
		{
			if( tun_DrvAdrIf::instance()->bIsAdrCommunicationRunning() == TRUE)
			{
				/*-------------------------------------------------------------------------*/
				/*  Check if not waiting for confirmation for the previous message sent    */
				/*-------------------------------------------------------------------------*/
				if( !( m_ptun_MessageQ -> fCheckWaitingForConfirmation( ) ) )
				{

					vSendNextMessageFromQueue();

				}

			} /* if( tun_DrvAdrIf::instance()->bIsAdrCommunicationRunning() == TRUE) */

		} /* if( m_ptun_MessageQ != NULL ) */

	} /* if( ( pu8Message != NULL ) && ( u32MessageLen != 0 ) ) */
	else
	{
		ETG_TRACE_ERR(( " tun_MsgToADR::vWaitOnMessageQue() -> Invalid parameter." ));
	}

	// On exit, release the semaphore
	OSAL_s32SemaphorePost( m_hSendToRU_Sync );

}


/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vMessageSentConfirmation( )
* 
* DESCRIPTION:  Confirmation on message sent to RU. 
*
* PARAMETER:    tVoid
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vMessageSentConfirmation( )
{
   if(m_poclConfigInterface == NULL)
   {
	   return;
   }
   ET_TRACE_BIN( TUN_TRACE_CLASS_MSGTOADR, TR_LEVEL_USER_4, ET_EN_T16 _ TUN_TRACE_MSGTOADR_MSG_SENT_CONF _ ET_EN_DONE ); 

   // Synchronise the function call made by many clients with semaphore
   tS32 ret = OSAL_s32SemaphoreWait( m_hSendToRU_Sync, OSAL_C_U32_INFINITE );
   
  /*----------------------------------------------------------------*/
  /* External message sent confirmation come, prepare next message  */
  /*----------------------------------------------------------------*/  
  if( m_ptun_MessageQ != NULL )
  {
    m_ptun_MessageQ ->vMessageSuccessfullySent( );

	if(TRUE == bSetDefaultSettingMsgSent)
	{
		m_ptun_MessageQ -> vWaitingForConfirmation( );
		ETG_TRACE_USR4(( " tun_MsgToADR::vWaitOnMessageQue() -> Invalid parameter ret=%d",ret ));
	}
   
  }

  /*-------------------------*/
  /* Send new message to RU  */
  /*-------------------------*/
  if(( tun_DrvAdrIf::instance()->bIsAdrCommunicationRunning() == TRUE) )
  {
    vSendNextMessageFromQueue();
  }
  
  // On exit, release the semaphore
  OSAL_s32SemaphorePost( m_hSendToRU_Sync ); 
}


/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendSnapshot ( u8Action )
* 
* DESCRIPTION:  tells HMI if there is an error in any of the supervised supplies 
*
* PARAMETER:    tU8 u8Action
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vSendSnapshot (tU8 u8Action )
{
  TUN_ASSERT_RETURN( m_potun_Utility != NULL);

  tU8 u8aMessage[ TUN_CM_SNAPSHOT_SET_LEN ];

  // send message CM_SnapshotMode

  vFormat_Msg_Header(u8aMessage, (tU16)enClientId_AmFmTuner, (tU16)enFBlockId_AmFmTuner,(tU8)enInstanceId_All,(tU16)enMsg_CM_SnapshotMode, (tU8)enOpType_SET);

  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA_LEN], 0x0001);

  u8aMessage[ enAdrMsgOffset_DATA] = u8Action;
  

  ETG_TRACE_USR2(( " tun_MsgToADR::vSendSnapshot -> , Snapshot mode=%u "
                    ,u8aMessage[ enAdrMsgOffset_DATA]
                  ));
  
  /* Message to be sent is put in Que   */
  vWaitOnMessageQue( u8aMessage, TUN_CM_SNAPSHOT_SET_LEN );

}




/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSend_TUNFgBg( )
* 
* DESCRIPTION:  Send FG / BG  status to RU.
*
* PARAMETER:    tU8 u8SubCommand, tU8 u8AudioSink
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vSend_TUNFgBg (tU8 u8TunerHeader, tU8 u8SubCommand, tU8 u8AudioSink, tBool bTunerState, tU8 u8PhysicalTuner)
{
  (tVoid)u8TunerHeader;  /* unused parameter */

  /*trMsg_CM_AudioSource_Set oMsg_CM_AudioSourceTuner;

  if( u8TunerHeader == TU_FOREGROUND)
  {
    oMsg_CM_AudioSourceTuner.enInstanceId = enInstanceId_AudioSource1;
  }
  else  // TU_BACKGROUND
  {
    oMsg_CM_AudioSourceTuner.enInstanceId = enInstanceId_AudioSource2;
  }

  if( u8SubCommand == TU_FOREGROUND)
  {
    if( u8AudioSink == 0)      // FI_EN_TUN_USR1_CHANGE_TO_FG_BG
    {
      oMsg_CM_AudioSourceTuner.enAudioSource = enAudioSource_User_audio;
    }
    else
    {
      oMsg_CM_AudioSourceTuner.enAudioSource = enAudioSource_TA_audio;
    }
  }
  else // TU_BACKGROUND
  {
    oMsg_CM_AudioSourceTuner.enAudioSource = enAudioSource_No_audio;
  }

  vSend( &oMsg_CM_AudioSourceTuner);*/
  
    ETG_TRACE_USR2(( " tun_MsgToADR::vSend_TUNFgBg -> , TestmodeActivity=0x%02x"
                    ,m_bTestModeActivity
                  ));
    

		TUN_ASSERT_RETURN( m_potun_Utility != NULL);
	
      if(u8PhysicalTuner != 0xFF) //Testmode active
      {
        ETG_TRACE_USR2(( " tun_MsgToADR::vSend_TUNFgBg -> , Send SpecialDiagnosiMode -> Mode = 0x%02x, Physical Tuner= 0x%02x"
                    ,enCM_SpecialDiagnosisModeMode_MEASURE_MODE
          ,u8PhysicalTuner
                  ));
          
        vSendSpecialDiagnosisMode(enCM_SpecialDiagnosisModeMode_MEASURE_MODE, (tenPhysicalTunerId)u8PhysicalTuner);
          
      }
      else //Testmode is not active
      {
          tU8   u8aMessage[ TUN_CM_AUDIOSOURCE_SET_LEN];
          m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_CLIENT_ID], (tU16)enClientId_AmFmTuner);
        m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FBLOCK_ID], (tU16)enFBlockId_AmFmTuner);
        u8aMessage[ enAdrMsgOffset_INSTANCE_ID] = (tU8)enInstanceId_AudioSource1;
        
        m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FKT_ID], (tU16)enMsg_CM_AudioSource);
        u8aMessage[ enAdrMsgOffset_OP_TYPE] = (tU8)enOpType_SETGET;
        m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA_LEN], 0x0001);
        
        ETG_TRACE_USR2(( " tun_MsgToADR::vSend_TUNFgBg -> , Send CM_AudioSource -> u8SubCommand = 0x%02x, u8AudioSink= 0x%02x"
                    ,u8SubCommand
          ,u8AudioSink
                  ));
          
        if( u8SubCommand == TU_FOREGROUND)
        { 
          if( u8AudioSink == 0) //User
          {
            u8aMessage[ enAdrMsgOffset_DATA] = (tU8)enAudioSource_User_audio;
            ETG_TRACE_USR2(( " tun_MsgToADR::vSend_TUNFgBg -> , Send CM_AudioSource -> set to User Audio"));
          }
          else if(u8AudioSink == 1)// TA
          {
            u8aMessage[ enAdrMsgOffset_DATA] = (tU8)enAudioSource_TA_audio;
            ETG_TRACE_USR2(( " tun_MsgToADR::vSend_TUNFgBg -> , Send CM_AudioSource -> set to TA Audio"));
          }
          else if(u8AudioSink == 3)// PTY
          {
        	  u8aMessage[ enAdrMsgOffset_DATA] = enAudioSource_PTY_audio;
        	  ETG_TRACE_USR2(( " tun_MsgToADR::vSend_TUNFgBg -> , Send CM_AudioSource -> set to PTY Audio"));
          }

          else
          {
          }
          
        }
        else
        {
          if(u8AudioSink == 1 || u8AudioSink == 3)// TA or PTY
          {
            ETG_TRACE_USR2(( " tun_MsgToADR::vSend_TUNFgBg -> , Send CM_AudioSource -> TA sent to background"));
            if(bTunerState) //Tuner1 was in Background
            {
              ETG_TRACE_USR2(( " tun_MsgToADR::vSend_TUNFgBg -> Tuner in BG earlier, set to No Audio"));
              u8aMessage[ enAdrMsgOffset_DATA] = (tU8)enAudioSource_No_audio;

            }
            else //Tuner1 was in Foreground
            {
              ETG_TRACE_USR2(( " tun_MsgToADR::vSend_TUNFgBg -> Tuner in FG earlier, set to User Audio"));
              u8aMessage[ enAdrMsgOffset_DATA] = (tU8)enAudioSource_User_audio;
            }
          }
          else          
          {
            ETG_TRACE_USR2(( " tun_MsgToADR::vSend_TUNFgBg -> set to No Audio"));
            u8aMessage[ enAdrMsgOffset_DATA] = (tU8)enAudioSource_No_audio;


          }
        }
        /* CM_AudioSource Message to be sent is put in Que   */
        vWaitOnMessageQue( u8aMessage, TUN_CM_AUDIOSOURCE_SET_LEN );
      }
    
}

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendTunerFGBG( )
* 
* DESCRIPTION:  Send specified tuner to FG/BG
*
* PARAMETER:    tun_tenInstanceId e8InstanceId, tenAudioSource e8AudioSource
*
* RETURNVALUE:  tVoid 
*
* vnd4kor 3/2/2014
*************************************************************************/
tVoid tun_MsgToADR::vSendTunerFGBG(tun_tenInstanceId e8InstanceId, \
								   tenAudioSource e8AudioSource )
{
	TUN_ASSERT_RETURN( m_potun_Utility != NULL);
	
	tU8 u8aMessage[ TUN_CM_AUDIOSOURCE_SET_LEN];
	m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_CLIENT_ID], (tU16)enClientId_AmFmTuner);
	m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FBLOCK_ID], (tU16)enFBlockId_AmFmTuner);
	u8aMessage[ enAdrMsgOffset_INSTANCE_ID] = (tU8)e8InstanceId;

	m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FKT_ID], (tU16)enMsg_CM_AudioSource);
	u8aMessage[ enAdrMsgOffset_OP_TYPE] = (tU8)enOpType_SETGET;
	m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA_LEN], 0x0001);

	ETG_TRACE_USR2(( " tun_MsgToADR::vSendTunerFGBG -> , Send CM_AudioSource -> e8InstanceId = 0x%02x, e8AudioSource= 0x%02x"
		,e8InstanceId
		,e8AudioSource
		));

	u8aMessage[ enAdrMsgOffset_DATA] = (tU8)e8AudioSource;

	 /* CM_AudioSource Message to be sent is put in Que   */
    vWaitOnMessageQue( u8aMessage, TUN_CM_AUDIOSOURCE_SET_LEN );


}

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendSelListElem( )
* 
* DESCRIPTION:  Select a element in the stations list.
*
* PARAMETER:    tU8 u8ListElementID ,tU8 u8ListType 
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vSendSelListElem ( tU16 u16ListID, tU8 u8ListElementID, tU8 u8ListType)
{
   ETG_TRACE_COMP(( " tun_MsgToADR::vSendSelListElem() -> u16ListID=%04x, u8ListElementID=%d,u8ListType=%02x"
                  ,u16ListID
                  ,u8ListElementID
                  ,u8ListType
                ));

  TUN_ASSERT_RETURN( m_potun_Utility != NULL);
  


  if( u8ListType == TUN_LIST_TYPE_CONFIG)
  {
	tU8 au8Message[ TUN_CM_ONLY_HEADER + 4];
	vFormat_Msg_Header(au8Message, (tU16)enClientId_AmFmTuner, (tU16)enFBlockId_AmFmTuner,(tU8)enInstanceId_AudioSource1,(tU16)enMsg_CM_StationsList, (tU8)enOpType_SET);
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_DATA_LEN], sizeof( au8Message) - (tU8)enAdrMsgOffset_HEADER_LEN);

    au8Message[ enCM_StationsList_idx_PosX]    = tU8(u8ListElementID + 1);
    au8Message[ enCM_StationsList_idx_PosY]    = (tU8)enCM_StationsListPosY_Selected;
    au8Message[ enCM_StationsList_idx_Data]    = 0x01;   /* set to 'active' */
	au8Message[ (tU8)enCM_StationsList_idx_Data + 1]    = (tU8)u16ListID;   

  
    /* Message to be sent is put in Que   */
    vWaitOnMessageQue( au8Message, sizeof( au8Message));
  }
   else //selection of preset list element
  {
	tU8 au8Message[ TUN_CM_ONLY_HEADER + 3];
	vFormat_Msg_Header(au8Message, (tU16)enClientId_AmFmTuner, (tU16)enFBlockId_AmFmTuner,(tU8)enInstanceId_AudioSource1,u16ListID, (tU8)enOpType_SETGET);
        m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_DATA_LEN], sizeof( au8Message) - (tU8)enAdrMsgOffset_HEADER_LEN);

    au8Message[ enCM_PresetList_idx_PosX]    = u8ListElementID;
    au8Message[ enCM_PresetList_idx_PosY]    = (tU8)enCM_PresetListPosY_Selected;
    au8Message[ enCM_PresetList_idx_Data]    = 0x01;   // set to 'active'
		
	ETG_TRACE_USR2(( " tun_MsgToADR::vSendSelListElem -> ListElementID = %d", u8ListElementID));			
    vWaitOnMessageQue( au8Message, sizeof( au8Message));	
  }

}

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vGetPresetList( )
* 
* DESCRIPTION:  Get Complete Preset list.
*
* PARAMETER:    tU16 u16ListID 
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vGetPresetList ( tU16 u16ListID )
{
	TUN_ASSERT_RETURN( m_potun_Utility != NULL);


	tU8 au8Message[ TUN_CM_ONLY_HEADER + 2];
	vFormat_Msg_Header(au8Message, (tU16)enClientId_AmFmTuner, (tU16)enFBlockId_AmFmTuner,(tU8)enInstanceId_AudioSource1,u16ListID, (tU8)enOpType_GET);
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_DATA_LEN], sizeof( au8Message) - (tU8)enAdrMsgOffset_HEADER_LEN);

    au8Message[ enCM_PresetList_idx_PosX]    = 0;
    au8Message[ enCM_PresetList_idx_PosY]    = 0;

		
	ETG_TRACE_USR2(( " tun_MsgToADR::vGetPresetList -> u16ListID = %d", u16ListID));	

	/** Work around to maintain correctness of MethodResult handling logic
	for Preset Handling */
	 PresetHandlingData data;
	 data.u8Action = GET_ALL_ELEMENTS;
	 data.u8ListElement = 0;
	 data.u16CmdCounter = 0;

	m_listPresetAction.push_back(data);

    vWaitOnMessageQue( au8Message, sizeof( au8Message));
}


/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendSaveListElem( )
* 
* DESCRIPTION:  Select a element in the Preset list.
*
* PARAMETER:    tU8 u8ListElementID ,tU8 u8ListType 
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vSendSaveListElem ( tU16 u16ListID, tU8 u8ListElementID )
{
  ETG_TRACE_COMP(( " tun_MsgToADR::vSendSaveListElem() -> u16ListID=%02x, u8ListElementID=%d"
                  ,u16ListID
                  ,u8ListElementID
                ));

  TUN_ASSERT_RETURN( m_potun_Utility != NULL);
  tU8 au8Message[ TUN_CM_ONLY_HEADER + 3];
  
	/*CM_AtPresetSave should have a SetGet optype introduced*/
  vFormat_Msg_Header(au8Message, (tU16)enClientId_AmFmTuner, (tU16)enFBlockId_AmFmTuner,(tU8)enInstanceId_AudioSource1,(tU16)enMsg_CM_ATPresetSave, (tU8)enOpType_SET);
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_DATA_LEN], sizeof( au8Message) - (tU8)enAdrMsgOffset_HEADER_LEN);
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_DATA], u16ListID);
      
	au8Message[ (tU8)enAdrMsgOffset_DATA+2] = u8ListElementID;
    
   /* Message to be sent is put in Que   */
    vWaitOnMessageQue( au8Message, sizeof( au8Message));
  
 } 


/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendSpecialSaveListElem( )
*
* DESCRIPTION:  Store non tuned stations.
*
* PARAMETER:    tU8 u8ListElementID ,tU8 u8ListType , tU32 u32Frequency , tU16 u16PI
*
* RETURNVALUE:  tVoid
*
*************************************************************************/
tVoid tun_MsgToADR::vSendSpecialSaveListElem (tU16 u16ListID, tU8 u8ListElementID , tU32 u32Frequency , tU16 u16PI)
{

	ETG_TRACE_COMP(( " tun_MsgToADR::vSendSpecialSaveListElem() -> u16ListID=%04x, u8ListElementID=%d , u32Frequency = %d , u16PI = 0x%x"
		   	   	   	   	   	   	   , u16ListID, u8ListElementID , u32Frequency , u16PI));

   	TUN_ASSERT_RETURN( m_potun_Utility != NULL);

	tU8 au8Message[ TUN_CM_ONLY_HEADER + TUN_CM_PRESETLIST_SPECIALSAVE_DATA_LEN];
	vFormat_Msg_Header(au8Message, (tU16)enClientId_AmFmTuner, (tU16)enFBlockId_AmFmTuner,(tU8)enInstanceId_AudioSource1,u16ListID, (tU8)enOpType_SETGET);
		m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_DATA_LEN], sizeof( au8Message) - (tU8)enAdrMsgOffset_HEADER_LEN);

	au8Message[ enCM_PresetList_idx_PosX]    = u8ListElementID;
	au8Message[ enCM_PresetList_idx_PosY]    = (tU8)enCM_PresetListPosY_selectAll;
	au8Message[ enCM_PresetList_idx_Data + enCM_PresetList_idx_Selected] = 0;
	m_potun_Utility->vConvert32BitTo4SeparateBytes( &au8Message[ enCM_PresetList_idx_Data + enCM_PresetList_idx_Frequency], u32Frequency );
	for (tU8 u8index = 0; u8index < enCM_PresetListt_len_Sendername ; u8index++)
	{
		au8Message[ enCM_PresetList_idx_Data + enCM_PresetList_idx_Sendername + u8index] = 0;
	}
	m_potun_Utility->vConvert16BitTo2SeparateBytes( &au8Message[ enCM_PresetList_idx_Data + enCM_PresetList_idx_PI], u16PI );
	au8Message[ enCM_PresetList_idx_Data + enCM_PresetList_idx_StationNameInfo] = 0;
	au8Message[ enCM_PresetList_idx_Data + enCM_PresetList_idx_TpTmcInfo] = 0;
	au8Message[ enCM_PresetList_idx_Data + enCM_PresetList_idx_PTY] = 0;
	au8Message[ enCM_PresetList_idx_Data + enCM_PresetList_idx_Level_Comp] = 0;

	vWaitOnMessageQue( au8Message, sizeof( au8Message));

}


/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendSetAtBand( )
* 
* DESCRIPTION:  Send at band request to set band.
*
* PARAMETER:    tVoid
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vSendSetAtBand ( tU8 u8Band )
{
  
	TUN_ASSERT_RETURN( m_potun_Utility != NULL);
	
	
	m_bAtBandSent = true;
   
 	  tU8   au8Message[ TUN_CM_ATBAND_SET_LEN]= {0};

	  vFormat_Msg_Header(au8Message, (tU16)enClientId_AmFmTuner, (tU16)enFBlockId_AmFmTuner,(tU8)enInstanceId_AudioSource1,(tU16)enMsg_CM_ATBand, (tU8)enOpType_SET);
      m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_DATA_LEN], 0x0001);

#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
      u8Band = u8ConvertTunerbandToCombiband(u8Band);
#endif
	  m_u8AtBandRequested = u8Band;
      /*Increment by 1 to map to ADR catalogue*/
      au8Message[ enAdrMsgOffset_DATA] = u8Band;
  
	  ETG_TRACE_USR2(( " TestMode Active: tun_MsgToADR::vSendSetAtBand -> Band = %02x"
                    ,ETG_CENUM(tenBand, u8Band)
                  ));
    
    /* Message to be sent is put in Que   */
    vWaitOnMessageQue( au8Message, TUN_CM_ATBAND_SET_LEN );
 
}
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: u8ConvertTunerbandToCombiband( )
*
* DESCRIPTION:  Check if combiband is enabled, then send band as CB1
*
* PARAMETER:    tU8 u8Band
*
* RETURNVALUE:  tU8 u8band
*
*************************************************************************/
tU8 tun_MsgToADR::u8ConvertTunerbandToCombiband( tU8 u8Band )
{
	if( m_poclConfigInterface == NULL)
	{
		NORMAL_M_ASSERT( m_poclConfigInterface != NULL );
		return (tU8)-1;
	}
   if(u8Band == midw_fi_tcl_e8_Tun_TunerBand::FI_EN_TUN_BAND_MW || u8Band == midw_fi_tcl_e8_Tun_TunerBand::FI_EN_TUN_BAND_LW)
    {
       if(PSA_COMBIBAND_VALUE == m_poclConfigInterface->u32GetConfigData("CombiBands") )
       {
          u8Band = (tU8)midw_fi_tcl_e8_Tun_TunerBand::FI_EN_TUN_BAND_CB1; //send combiband for psa
          ETG_TRACE_USR2(( " tun_MsgToADR::u8ConvertTunerbandToCombiband -> band is changed to CB1"));
       }
    }
   return u8Band;
}

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: u8ConvertCombibandToTunerband( )
*
* DESCRIPTION:  Check if combiband is enabled, then convert
*                             it to corresponding Tuner band
*
* PARAMETER:    tU8 u8Band, tU32 u32Frequency
*
* RETURNVALUE:  tU8 u8band
*
*************************************************************************/
tU8 tun_MsgToADR::u8ConvertCombibandToTunerband( tU8 u8Band, tU32 u32Frequency )
{
	if( m_poclConfigInterface == NULL)
	{
		NORMAL_M_ASSERT( m_poclConfigInterface != NULL );
		return (tU8)-1;
	}
    if((u8Band == midw_fi_tcl_e8_Tun_TunerBand::FI_EN_TUN_BAND_CB1) && (PSA_COMBIBAND_VALUE == m_poclConfigInterface->u32GetConfigData("CombiBands")))
    {
       //for PSA
       if((u32Frequency >= (unsigned int) m_poclConfigInterface->u32GetConfigData("LowLW")) && (u32Frequency<=(unsigned int)m_poclConfigInterface->u32GetConfigData("HighLW")))
       {
          u8Band = midw_fi_tcl_e8_Tun_TunerBand::FI_EN_TUN_BAND_LW;
       }
       else if((u32Frequency>=(unsigned int)m_poclConfigInterface->u32GetConfigData("LowMW")) && (u32Frequency<=(unsigned int)m_poclConfigInterface->u32GetConfigData("HighMW")))
       {
          u8Band = midw_fi_tcl_e8_Tun_TunerBand::FI_EN_TUN_BAND_MW;
       }
       else
       {

       }

    }
   return u8Band;
}
#endif
/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendSetBand( )
* 
* DESCRIPTION:  Save the current station in selected static list.
*
* PARAMETER:    tVoid
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vSendSetBand ( tU8 u8Band )
{
  
	if((m_potun_MsgToHMI == NULL)||( m_potun_Utility == NULL)||(m_potun_Config == NULL)||(m_poclConfigInterface == NULL) || (m_potun_MsgToHMI == NULL)
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
		|| (m_poclTmcTuner_Method_S_SET_ACTIVE_PRESETLIST == NULL)
#endif
		)
	{
		TUN_ASSERT_RETURN( m_potun_MsgToHMI != NULL);
		TUN_ASSERT_RETURN( m_potun_Utility != NULL);
		TUN_ASSERT_RETURN( m_potun_Config != NULL);
		TUN_ASSERT_RETURN( m_poclConfigInterface != NULL);
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
		TUN_ASSERT_RETURN( m_poclTmcTuner_Method_S_SET_ACTIVE_PRESETLIST != NULL);
#endif

		return;
	}
  
    ETG_TRACE_USR2(( " tun_MsgToADR::vSendSetBand -> Testmode not active, Band = %02x"
                    ,ETG_CENUM(tenBand, u8Band)));
    
    tU8   u8aMessage[ TUN_CM_SYSTEMPRESET_SET_LEN] = {0};
    



      vFormat_Msg_Header(u8aMessage, (tU16)enClientId_AmFmTuner, (tU16)enFBlockId_AmFmTuner,(tU8)enInstanceId_AudioSource1,(tU16)enMsg_CM_SystemPresets, (tU8)enOpType_SETGET);
      m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA_LEN], 0x0003);

	  
		/**Added for Suzuki project.*/
#ifdef VARIANT_S_FTR_ENABLE_ACTIVEPRESETLIST_BANDCHG

		ETG_TRACE_USR2(("Entering into the Suzuki switch for Band Change."));
		/**Get the active preset sent by HMI and request for the System presets for that particular PresetList.*/
		tU16 u16ActivePresetList = m_potun_MsgToHMI->u16GetListIDForSetActivePresetList();
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
		u16ActivePresetList=m_poclTmcTuner_Method_S_SET_ACTIVE_PRESETLIST->U16GetADR3PresetListIDForSetActivePresetList();
#endif
		/**Below changes are wrt SUZUKI-19723*/
		/**When the command comes from Diagnosis or any other source, who doesn't care for BAND LEVEL ,then
		we are tunning to the first BAND LEVEL in the corresponding BAND*/
		if((u8Band == enBand_FM) && (u16ActivePresetList > enMsg_CM_ATPresetList_01))
		{
			u16ActivePresetList = enMsg_CM_ATPresetList_00;
		}
		if((u8Band == enBand_MW) && ((u16ActivePresetList < enMsg_CM_ATPresetList_02) || (u16ActivePresetList > enMsg_CM_ATPresetList_03)))
		{
			u16ActivePresetList = enMsg_CM_ATPresetList_02;
		}
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
		if((u8Band == enBand_CB1) &&  (m_poclConfigInterface->u32GetConfigData("DRMSupported"))&& (u16ActivePresetList <enMsg_CM_ATPresetList_04))
		{
		   u16ActivePresetList = enMsg_CM_ATPresetList_04;
		}
#endif
		m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA], u16ActivePresetList);
#else
		unsigned short adr3presetlist = 0;
		/** Get Default HMI List Id for the band */
		/** Get Default ADR List Id for the band and assign it to u8aMessage */
		adr3presetlist = U16GetADR3PresetListID((midw_fi_tcl_e8_Tun_TunerBand::tenType)u8Band);
		ETG_TRACE_USR2(("No. of lists not greater than 1, list ID : %d", adr3presetlist));

		if(u8Band == midw_fi_tcl_e8_Tun_TunerBand::FI_EN_TUN_BAND_FM)
		{
			if(m_poclConfigInterface->u32GetConfigData("NumberOfPresetListFM") > 1)
			{
				if(m_potun_MsgToHMI->u16GetFMActivePresetListId()!= 0xff)
				{
					adr3presetlist = m_potun_MsgToHMI->u16GetFMActivePresetListId();
				}
				ETG_TRACE_USR2(("FM list ID : %d", adr3presetlist));
			}
		}
		else if(u8Band == midw_fi_tcl_e8_Tun_TunerBand::FI_EN_TUN_BAND_MW)
		{
			if(m_poclConfigInterface->u32GetConfigData("NumberOfPresetListMW") > 1)
			{
				if(m_potun_MsgToHMI->u16GetAMActivePresetListId()!= 0xff)
				{
					adr3presetlist = m_potun_MsgToHMI->u16GetAMActivePresetListId();
				}
				ETG_TRACE_USR2(("MW list ID : %d", adr3presetlist));
			}
		}
		
		else if((u8Band == midw_fi_tcl_e8_Tun_TunerBand::FI_EN_TUN_BAND_CB1) && (m_poclConfigInterface->u32GetConfigData("DRMSupported")))
		{
			if(m_poclConfigInterface->u32GetConfigData("NumberOfPresetListDRM") > 1)
			{
				if(m_potun_MsgToHMI->u16GetDRMActivePresetListId() != 0xff)
				{
					adr3presetlist = m_potun_MsgToHMI->u16GetDRMActivePresetListId();
				}
				ETG_TRACE_USR2(("DRM list ID : %d", adr3presetlist));
			}
		}


		if(adr3presetlist == 0)
		{
			ETG_TRACE_USR2(("ADR3 presetlist id 0, list ID : %d", adr3presetlist));
			
			if(m_poclConfigInterface->u32GetConfigData("PresetListType"))
			{
				adr3presetlist = 0x0f20;
			}
			else
			{
				adr3presetlist = U16GetADR3PresetListID((midw_fi_tcl_e8_Tun_TunerBand::tenType)u8Band);
			}			
			
		}
	
			m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA], \
				   adr3presetlist );
#endif
	   

      u8aMessage[ (tU8)enAdrMsgOffset_DATA + 2] =  0; //Last tuning preset
    
    ETG_TRACE_USR2(( " tun_MsgToADR::vSendSetBand -> Destinaltion List = 0x%02x 0x%02x"
                    ,u8aMessage[ enAdrMsgOffset_DATA]
					,u8aMessage[ (tU8)enAdrMsgOffset_DATA + 1]
                  ));
	/* Message to be sent is put in Que   */
    vWaitOnMessageQue( u8aMessage, TUN_CM_SYSTEMPRESET_SET_LEN );

}


/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSetvPresetListSetActive( )
* 
* DESCRIPTION:  Select a element in the Preset list.
*
* PARAMETER:    tU16 u16ListID 
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vSetPresetListSetActive ( tU16 u16ListID )
{
	if(u16ListID == INVALID_ENTRY)
	{
		ETG_TRACE_USR2(( " tun_MsgToADR::vSetPresetListSetActive -> Invalid List Id =%x",u16ListID ));
		return;
	}
	ETG_TRACE_USR2(( " tun_MsgToADR::vSetPresetListSetActive -> u16ListID=%x",u16ListID ));
  /** Fill the adr message */
  TUN_ASSERT_RETURN( m_potun_Utility != NULL);
  tU8 au8Message[ TUN_CM_ONLY_HEADER + 2];
  
  vFormat_Msg_Header(au8Message, (tU16)enClientId_AmFmTuner, (tU16)enFBlockId_AmFmTuner,(tU8)enInstanceId_AudioSource1,(tU16)enMsg_CM_ATPresetListSetActive, (tU8)enOpType_STARTRESULT);
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_DATA_LEN], sizeof( au8Message) - (tU16)enAdrMsgOffset_HEADER_LEN);
  
  /** Data : FktID Received */
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_DATA],u16ListID);
  

   /* Message to be sent is put in Que   */
    vWaitOnMessageQue( au8Message, sizeof( au8Message));
  
 } 

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendSaveListElem( )
* 
* DESCRIPTION:  Save the current station in selected static list.
*
* PARAMETER:    tVoid
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vGetBandDetails( tU8 u8Band )
{
	tU8   u8aMessage[ TUN_CM_ATBAND_SET_LEN];
    
	if(!bGetNotificationSentStatus())
	{
		vInc_m_NumofAtBandReqAtStartUp();
		ETG_TRACE_USR2((" tun_MsgToADR::bGetNotificationSentStatus not sent m_NumofAtBandReqAtStartUp = %d ", m_NumofAtBandReqAtStartUp));
	}

    if (m_potun_Utility!=NULL)
    {
      m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_CLIENT_ID], (tU16)enClientId_AmFmTuner);
      m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FBLOCK_ID], (tU16)enFBlockId_AmFmTuner);
      u8aMessage[ (tU8)enAdrMsgOffset_INSTANCE_ID] = (tU8)enInstanceId_AudioSource1;
      m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FKT_ID], (tU16)enMsg_CM_ATBand);
      u8aMessage[ (tU8)enAdrMsgOffset_OP_TYPE] = (tU8)enOpType_GET;
      m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA_LEN], 0x0001);
      
      /*Increment by 1 to map to ADR catalogue*/
      u8aMessage[ (tU8)enAdrMsgOffset_DATA] = u8Band;
    }
    ETG_TRACE_USR2(( " tun_MsgToADR::vGetBandDetails -> Band = %02x"
                    ,ETG_CENUM(tenBand, u8Band)
                  ));
    
    /* Message to be sent is put in Que   */
    vWaitOnMessageQue( u8aMessage, TUN_CM_ATBAND_SET_LEN );
}


/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: Inc_m_NumofAtBandReqAtStartUp( )
*
* DESCRIPTION:  increment the value of m_NumofAtBandReqAtStartUp by 1
*
* PARAMETER:    tVoid
*
* RETURNVALUE:  tVoid
*
*************************************************************************/
tVoid tun_MsgToADR::vInc_m_NumofAtBandReqAtStartUp()
{
	m_NumofAtBandReqAtStartUp++;
}

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: Get_m_NumofAtBandReqAtStartUp( )
*
* DESCRIPTION:  return the value of m_NumofAtBandReqAtStartUp
*
* PARAMETER:    tVoid
*
* RETURNVALUE:  tVoid
*
*************************************************************************/
tU8 tun_MsgToADR::u8Get_m_NumofAtBandReqAtStartUp()
{
	return m_NumofAtBandReqAtStartUp;
}

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: Dec_m_NumofAtBandReqAtStartUp( )
*
* DESCRIPTION:  decrement the value of m_NumofAtBandReqAtStartUp by 1
*
* PARAMETER:    tVoid
*
* RETURNVALUE:  tVoid
*
*************************************************************************/
tVoid tun_MsgToADR::vDec_m_NumofAtBandReqAtStartUp()
{
	m_NumofAtBandReqAtStartUp--;
	ETG_TRACE_USR2((" tun_MsgToADR::Dec_m_NumofAtBandReqAtStartUp:bGetNotificationSentStatus not sent m_NumofAtBandReqAtStartUp = %d ", m_NumofAtBandReqAtStartUp));
}

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendAcceptAF( )
* 
* DESCRIPTION:  Sends a request to ADR to accept the current frequency as 
				the base frequency
*
* PARAMETER:    tVoid
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vSendAcceptAF()
{
	tU8   u8aMessage[ TUN_CM_ONLY_HEADER];
    
    if (m_potun_Utility!=NULL)
    {
    	vFormat_Msg_Header(u8aMessage, (tU16)enClientId_AmFmTuner,(tU16)enFBlockId_AmFmTuner, (tU8)enInstanceId_AudioSource1, (tU16)enMsg_CM_AcceptAf, (tU8)enOpType_SET );

      m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA_LEN], 0x0000);

    }
    ETG_TRACE_USR2(( " tun_MsgToADR::vSendAcceptAF "));
    
    /* Message to be sent is put in Que   */
    vWaitOnMessageQue( u8aMessage, TUN_CM_ONLY_HEADER );
}

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendATFrequency( )
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vSendATFrequency( tenATFrequencyAbsOrRel enAbsOrRel, tU8 u8NSteps, tU32 u32Frequency, tenATFrequencyDirection enDirection, tU8 u8TunerHeader)
{

  TUN_ASSERT_RETURN( m_potun_Utility != NULL);

  tU16 u16MessageLen = TUN_ATFREQUENCY_SET_LEN;
  tU8 au8Message[ TUN_ATFREQUENCY_SET_LEN];

  tU8 u8OpType = (tU8)enOpType_SET;

  if(enAbsOrRel == enATFrequencyAbsOrRel_AbsoluteValue)
  {
    m_potun_Utility->vConvert32BitTo4SeparateBytes( &au8Message[ enATFrequency_idx_Frequency], u32Frequency );

    ETG_TRACE_COMP(( " tun_MsgToADR::vSendATFrequency() -> AbsoluteValue, u8TunerHeader = 0x%02X, u32Frequency = %d kHz"
                    ,u8TunerHeader
                    ,u32Frequency
                  ));
  }
  else
  {
    u16MessageLen = TUN_ATFREQUENCY_INCREMENT_LEN;
    au8Message[ enATFrequency_idx_NSteps] = u8NSteps;

    ETG_TRACE_COMP(( " tun_MsgToADR::vSendATFrequency() -> RelativeValue u8TunerHeader = 0x%02X, enDirection = %02x, u8NSteps = %d"
                    ,u8TunerHeader
                    ,ETG_CENUM( tenATFrequencyDirection, enDirection)
                    ,u8NSteps
                  ));

    if( enDirection == enATFrequencyDirection_Up)
    {
    	u8OpType = (tU8)enOpType_Increment;
    }
    else
    {
    	u8OpType = (tU8)enOpType_Decrement;
    }
  }

  vFormat_Msg_Header(au8Message, (tU16)enClientId_AmFmTuner, (tU16)enFBlockId_AmFmTuner,(tU8)enInstanceId_AudioSource1,(tU16)enMsg_ATFrequency, (tU8)u8OpType);
 

  m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_DATA_LEN], (tU16)(u16MessageLen - (tU16)enAdrMsgOffset_HEADER_LEN));
  
  /* Message to be sent is put in Que   */
  vWaitOnMessageQue( au8Message, u16MessageLen );
}


/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendSeekStart( )
* 
* DESCRIPTION:  Starts a new seek..
*
* PARAMETER:    tU8 u8SeekType, tU8 u8Direction, tU32 u32Frequency
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vSendSeekStart (tU8 u8SeekType, tU32 u32Frequency)
{
   tU8   u8aMessage[ TUN_CM_ATSEEK_SET_LEN];
  
  /*Send CM_ATSeek[Set]SeekMode, Freq*/
  if (m_potun_Utility!=NULL)
  {
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_CLIENT_ID], (tU16)enClientId_AmFmTuner);
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FBLOCK_ID], (tU16)enFBlockId_AmFmTuner);
    u8aMessage[ enAdrMsgOffset_INSTANCE_ID] = (tU8)enInstanceId_AudioSource1;
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FKT_ID], (tU16)enMsg_CM_ATSeek);
    u8aMessage[ enAdrMsgOffset_OP_TYPE] = (tU8)enOpType_SET;
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA_LEN], 5);
    u8aMessage[ enCM_ATSeek_idx_TSeekMode] =  u8SeekType;
    m_potun_Utility->vConvert32BitTo4SeparateBytes(&u8aMessage[ enCM_ATSeek_idx_Frequency], u32Frequency );
  }
  
  ETG_TRACE_USR2(( " tun_MsgToADR::vSendSeekStart -> SeekMode=%02x, Freq = %u kHz"
                    , ETG_CENUM( tenATSeekMode, u8SeekType)
                    , u32Frequency
                  ));
          
  /* Message to be sent is put in Que   */
  vWaitOnMessageQue( u8aMessage, TUN_CM_ATSEEK_SET_LEN );

}



/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendGetSeekStatus( )
*
* DESCRIPTION:  gets seek status.
*
* PARAMETER:
*
* RETURNVALUE:  tVoid
*
*************************************************************************/
tVoid tun_MsgToADR::vSendGetSeekStatus()
{

 tU8   u8aMessage[ TUN_CM_ONLY_HEADER];

 /* Send AnnouncementSwitch[Get] */
 if (m_potun_Utility!=NULL)
 {
    vFormat_Msg_Header(u8aMessage, (tU16)enClientId_AmFmTuner, (tU16)enFBlockId_AmFmTuner,(tU8)enInstanceId_AudioSource1,(tU16)enMsg_CM_ATSeek, (tU8)enOpType_GET);
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA_LEN], 0x0000);
 }
 ETG_TRACE_USR2(( " tun_MsgToADR::vSendGetSeekStatus "));

 /* Message to be sent is put in Que   */
 vWaitOnMessageQue( u8aMessage, TUN_CM_ONLY_HEADER );

}


/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vUpdateLandscape( )
* 
* DESCRIPTION:  Sens Landscape update request to ADR for FM and MW bands
*
* PARAMETER:    tU32 u32Bands
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vUpdateLandscape(tU32 u32Bands)
{
   tU8   u8aMessage[ TUN_C_LANDSCAPEUPDATE_STARTRESULT_LEN];
  
  /*Send CM_LandscapeUpdate[StartResult]Bands*/
  if (m_potun_Utility!=NULL)
  {
    vFormat_Msg_Header(u8aMessage, (tU16)enClientId_AmFmTuner, (tU16)enFBlockId_AmFmTuner,(tU8)enInstanceId_All,(tU16)enMsg_CM_UpdateLandscape, (tU8)enOpType_STARTRESULT);

    m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA_LEN], 0x0004);
    m_potun_Utility->vConvert32BitTo4SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA], u32Bands );
  }
  
  ETG_TRACE_USR2(( " tun_MsgToADR::vUpdateLandscape -> Bands = %u"
                    , u32Bands
                  ));

  vSetLandscapeRequested(TRUE);
  /* Message to be sent is put in Que   */
  vWaitOnMessageQue( u8aMessage, TUN_C_LANDSCAPEUPDATE_STARTRESULT_LEN );
}


/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vUpdateLandscape( )
*
* DESCRIPTION:  Sets if Landscape update was requested to ADR
*
* PARAMETER:    tBool
*
* RETURNVALUE:  tVoid
*
*************************************************************************/
tVoid tun_MsgToADR::vSetLandscapeRequested(tBool bLandscapeReq)
{
	m_bLandscapeRequested = bLandscapeReq;
}

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: bGetLandscapeRequested( )
*
* DESCRIPTION:  returns if Landscape update was requested to ADR
*
* PARAMETER:    tVoid
*
* RETURNVALUE:  tBool
*
*************************************************************************/
tBool tun_MsgToADR::bGetLandscapeRequested() const
{
	return m_bLandscapeRequested;
}


/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendCommit( )
* 
* DESCRIPTION:  Commit message requested to ADR3
*
* PARAMETER:    tVoid
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vSendCommit()
{
   tU8   u8aMessage[ TUN_CM_ONLY_HEADER];
  
  /*Send vSendCommit[StartResult]*/
  if (m_potun_Utility!=NULL)
  {
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_CLIENT_ID], (tU16)enClientId_AmFmTuner);
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FBLOCK_ID], (tU16)enFBlockId_AmFmTuner);
    u8aMessage[ enAdrMsgOffset_INSTANCE_ID] = (tU8)enInstanceId_All;
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FKT_ID],  (tU16)enMsg_CM_Commit);
    u8aMessage[ enAdrMsgOffset_OP_TYPE] = (tU8)enOpType_STARTRESULT;
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA_LEN], 0x0000);
  }
  
  ETG_TRACE_USR2(( " tun_MsgToADR::vSendCommit" ));
          
  /* Message to be sent is put in Que   */
  vWaitOnMessageQue( u8aMessage, TUN_CM_ONLY_HEADER );
}

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendAFMode( )
* 
* DESCRIPTION:  Switches AF Mode ON or OFF
*
* PARAMETER:    None
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vSendAFMode(tU32 u32AFModeValue)
{
  TUN_ASSERT_RETURN( m_potun_Utility != NULL);
  TUN_ASSERT_RETURN( m_potun_Config != NULL);

    tU8   u8aMessage[ TUN_CM_AFMODE_SETGET_LEN];
  
  /*Send CM_AfBestSwitch[SetGet]AFBestOnOff*/
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_CLIENT_ID], (tU16)enClientId_AmFmTuner);
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FBLOCK_ID], (tU16)enFBlockId_AmFmTuner);
  u8aMessage[ enAdrMsgOffset_INSTANCE_ID] = (tU8)enInstanceId_AudioSource1;
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FKT_ID], (tU16)enMsg_AFBestSwitch);
  u8aMessage[ enAdrMsgOffset_OP_TYPE] = (tU8)enOpType_SETGET;
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA_LEN], 0x0001);

  u8aMessage[ enAdrMsgOffset_DATA] =  ( ( u32AFModeValue & 0x00000001)) ? 0x01 : 0x00;

/** For VW-MIBE , 
Whenever HMI sends AF_Best_Switch OFF, 
if 
	AF_Temporary EOL Parameter is set to Temporary, then send AFBest_Off_Temporary(2) to ADR 
else
	send Normal_OFF (0) */
 #ifdef VARIANT_S_FTR_ENABLE_FEATURE_VW_MIBE

  ETG_TRACE_USR2(( " tun_MsgToADR::vSendAFMode -> , EOL AFTemporarySwitch=0x%02x"
                    ,m_potun_Config->m_rKdsEOLParameter.bGetAFTemporarySwitch() 
                  ));

  if(u8aMessage[ enAdrMsgOffset_DATA] == 0x00)
  {
	  u8aMessage[ enAdrMsgOffset_DATA] = (m_potun_Config->m_rKdsEOLParameter.bGetAFTemporarySwitch() == false)? 0x00: 0x02;
  }
#endif
  
  ETG_TRACE_USR2(( " tun_MsgToADR::vSendAFMode -> , AFMode=0x%02x"
                    ,u8aMessage[ enAdrMsgOffset_DATA]
                  ));
          
  /* Message to be sent is put in Que   */
  vWaitOnMessageQue( u8aMessage, TUN_CM_AFMODE_SETGET_LEN );
}

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vGetAFMode( )
* 
* DESCRIPTION:  Get the last set AF Switch status
*
* PARAMETER:    None
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vGetAFMode()
{
  tU8   u8aMessage[ TUN_CM_ONLY_HEADER];
  
  /*Send CM_AfBestSwitch[Get]*/
  if (m_potun_Utility!=NULL)
  {
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_CLIENT_ID], (tU16)enClientId_AmFmTuner);
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FBLOCK_ID], (tU16)enFBlockId_AmFmTuner);
    u8aMessage[ enAdrMsgOffset_INSTANCE_ID] = (tU8)enInstanceId_AudioSource1; //check if instance ID 0xff can be sent??
#ifdef VARIANT_S_FTR_ENABLE_FEATURE_SUZUKI
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FKT_ID], (tU16)enMsg_AFBestSwitchFeatures);
#else
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FKT_ID], (tU16)enMsg_AFBestSwitch);
#endif
    u8aMessage[ enAdrMsgOffset_OP_TYPE] = (tU8)enOpType_GET;
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA_LEN], 0x0000);  
  }
  
  ETG_TRACE_USR2(( " tun_MsgToADR::vGetAFMode"));
          
  /* Message to be sent is put in Que   */
  vWaitOnMessageQue( u8aMessage, TUN_CM_ONLY_HEADER );
}


/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendSetDDASwitch( )
* 
* DESCRIPTION:  Switches DDA Mode ON or OFF
*
* PARAMETER:    None
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vSendSetDDASwitch(tU32 u32DDAMode)
{
  TUN_ASSERT_RETURN( m_potun_Utility != NULL);


  tU8   u8aMessage[ TUN_CM_DDAMODE_SETGET_LEN];
  
  /*Send CM_DDASwitch[SetGet]Phase diversity OnOff*/
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_CLIENT_ID], (tU16)enClientId_AmFmTuner);
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FBLOCK_ID], (tU16)enFBlockId_AmFmTuner);
  u8aMessage[ enAdrMsgOffset_INSTANCE_ID] = (tU8)enInstanceId_All;//not evaluated by ADR
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FKT_ID], (tU16)enMsg_CM_DDASwitch);
  
  u8aMessage[ enAdrMsgOffset_OP_TYPE] = (tU8)enOpType_SETGET;
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA_LEN], 0x0001);
  
  u8aMessage[ enAdrMsgOffset_DATA] = ( ( u32DDAMode & 0x00080000)) ? 0x01 : 0x00;
  
  ETG_TRACE_USR2(( " tun_MsgToADR::vSendSetDDASwitch -> , DDAMode=0x%02x"
                    ,u8aMessage[10]
                  ));
  
  /* Message to be sent is put in Que   */
  vWaitOnMessageQue( u8aMessage, TUN_CM_DDAMODE_SETGET_LEN );
}

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vGetCalibrationData( )
* 
* DESCRIPTION:  Gets the data that reflects the current tuner calibration
        state of ADR3
*
* PARAMETER:    None
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vGetCalibrationData()
{
  tU8   u8aMessage[ TUN_CM_ONLY_HEADER];
  
  /*Send CM_CalibrationData[Get]*/
  if (m_potun_Utility!=NULL)
  {
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_CLIENT_ID], (tU16)enClientId_AmFmTuner);
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FBLOCK_ID], (tU16)enFBlockId_AmFmTuner);
    u8aMessage[ enAdrMsgOffset_INSTANCE_ID] = (tU8)enInstanceId_All;
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FKT_ID], (tU16)enMsg_CM_CalibrationData);
    u8aMessage[ enAdrMsgOffset_OP_TYPE] = (tU8)enOpType_GET;
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA_LEN], 0x0000);  
  }
  
  ETG_TRACE_USR2(( " tun_MsgToADR::vGetCalibrationData"));
          
  /* Message to be sent is put in Que   */
  vWaitOnMessageQue( u8aMessage, TUN_CM_ONLY_HEADER );
}


/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendStationListExit(tU8 u8ConfigListID)
* 
* DESCRIPTION:  
*
* PARAMETER:    
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vSendStationListExit(tU8 u8ConfigListID)
{

  TUN_ASSERT_RETURN( m_potun_Utility != NULL);
  ETG_TRACE_USR2(( " tun_MsgToADR::vSendStationListExit "));
  tU8   u8aMessage[ TUN_CM_ONLY_HEADER + 2];

  vFormat_Msg_Header(u8aMessage, (tU16)enClientId_AmFmTuner, (tU16)enFBlockId_AmFmTuner,(tU8)enInstanceId_All,(tU16)enMsg_CM_StationListRelease, (tU8)enOpType_SET);
  
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA_LEN], 0x0002);
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA], (tU16)u8ConfigListID);  
  
  /* Message to be sent is put in Que   */
  vWaitOnMessageQue( u8aMessage, sizeof(u8aMessage) );
}


/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendGetDDASwitch( )
* 
* DESCRIPTION:  Gets the DDA mode
*
* PARAMETER:    None
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vSendGetDDASwitch()
{
    tU8   u8aMessage[ TUN_CM_ONLY_HEADER];
  
  /*Send CM_DDASwitch[Get]*/
  if (m_potun_Utility!=NULL)
  {
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_CLIENT_ID], (tU16)enClientId_AmFmTuner);
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FBLOCK_ID], (tU16)enFBlockId_AmFmTuner);
    u8aMessage[ enAdrMsgOffset_INSTANCE_ID] = (tU8)enInstanceId_All; //not evaluated by ADR
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FKT_ID], (tU16)enMsg_CM_DDASwitch);
    
    u8aMessage[ enAdrMsgOffset_OP_TYPE] = (tU8)enOpType_GET;  
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA_LEN], 0x0000);  
  }
  
  ETG_TRACE_USR2(( " tun_MsgToADR::vSendGetDDASwitch "));
  
  /* Message to be sent is put in Que   */
  vWaitOnMessageQue( u8aMessage, TUN_CM_ONLY_HEADER );
}

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendTAMode( )
* 
* DESCRIPTION:  Switches TA functionality ON or OFF
*
* PARAMETER:    None
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vSendTAMode(tU8 u8TAMode)
{
  TUN_ASSERT_RETURN( m_potun_Utility != NULL);

  tU8   u8aMessage[ TUN_CM_TAMODE_SETGET_LEN];
  
  /*Send TPSwitch[SetGet]TP Switch OnOff*/
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_CLIENT_ID], (tU16)enClientId_AmFmTuner);
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FBLOCK_ID], (tU16)enFBlockId_AmFmTuner);
  u8aMessage[ enAdrMsgOffset_INSTANCE_ID] = (tU8)enInstanceId_AudioSource1;
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FKT_ID], (tU16)enMsg_TPSwitch);
  u8aMessage[ enAdrMsgOffset_OP_TYPE] = (tU8)enOpType_SETGET;
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA_LEN], 0x0001);
  
  u8aMessage[ enAdrMsgOffset_DATA] = u8TAMode;
  
  ETG_TRACE_USR2(( " tun_MsgToADR::vSendTAMode -> TAMode = 0x%02x "
  ,u8aMessage[ enAdrMsgOffset_DATA]));
  
  /* Message to be sent is put in Que   */
  vWaitOnMessageQue( u8aMessage, TUN_CM_TAMODE_SETGET_LEN );
}

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendLSMSetup( )
*
* DESCRIPTION:  sets the last tuned band and/or the last tuned frequency to
* 				the given values for the audio 512 tuner
*
* PARAMETER:    None
*
* RETURNVALUE:  tVoid
*
*************************************************************************/
tVoid tun_MsgToADR::vSendLSMSetup(tU8 u8Band, tU32 u32Frequency,tU16 u16PICode,tU32 u32HDStnID_DRMSrvID,tU8 u8ModulationType)
{
	TUN_ASSERT_RETURN( m_potun_Utility != NULL);

	  tU8   u8aMessage[ TUN_CM_LSMSETUP_SET_LEN];

	  /*Send TPSwitch[SetGet]TP Switch OnOff*/
	  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_CLIENT_ID], (tU16)enClientId_AmFmTuner);
	  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FBLOCK_ID], (tU16)enFBlockId_AmFmTuner);
	  u8aMessage[ enAdrMsgOffset_INSTANCE_ID] = (tU8)enInstanceId_AudioSource1;
	  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FKT_ID], (tU16)enMsg_CM_ATLsmSetup);
	  u8aMessage[ enAdrMsgOffset_OP_TYPE] = (tU8)enOpType_SETGET;
	  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA_LEN], TUN_CM_LSMSETUP_SET_DATA_LEN);

#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
	  u8Band = u8ConvertTunerbandToCombiband(u8Band);
#endif
	  u8aMessage[ (tU8)enCM_ATLsmSetup_idx_Band] = u8Band;
	  m_potun_Utility->vConvert32BitTo4SeparateBytes( &u8aMessage[ (tU8)enCM_ATLsmSetup_idx_Frequency], u32Frequency );
	  m_potun_Utility->vConvert32BitTo4SeparateBytes( &u8aMessage[ (tU8)enCM_ATLsmSetup_idx_PICode], u16PICode );

	  ETG_TRACE_USR2(( " tun_MsgToADR::vSendLSMSetup -> Band = 0x%02x, Frequency = %d, PICode = 0x%x "
	  ,u8aMessage[ (tU8)enCM_ATLsmSetup_idx_Band ]
	  ,u8aMessage[ (tU8)enCM_ATLsmSetup_idx_Frequency]
	  ,u8aMessage[ (tU8)enCM_ATLsmSetup_idx_PICode]));
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
	  m_potun_Utility->vConvert32BitTo4SeparateBytes( &u8aMessage[ (tU8)enCM_ATLsmSetup_idx_HDStationId_DRMServiceId], u32HDStnID_DRMSrvID);	  
	  u8aMessage[ (tU8)enCM_ATLsmSetup_idx_ModulationType] = 0x0;
	  m_potun_Utility->vConvert32BitTo4SeparateBytes( &u8aMessage[ (tU8)enCM_ATLsmSetup_idx_RFUBytes], u8ModulationType);	  
#else
	  /** Solving lint issue : Symbol 'u32HDStnID_DRMSrvID' & 'u8ModulationType' not referenced */
	  u32HDStnID_DRMSrvID = u8ModulationType;
#endif
	  /* Message to be sent is put in Que   */
	  vWaitOnMessageQue( u8aMessage, TUN_CM_LSMSETUP_SET_LEN );
}



/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendREGMode( )
* 
* DESCRIPTION:  Sets REG mode to FIX or Auto
*
* PARAMETER:    None
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vSendREGMode(tU8 u8REGMode)
{
  TUN_ASSERT_RETURN( m_potun_Utility != NULL);

  tU8   u8aMessage[ TUN_CM_REGMODE_SETGET_LEN];
  
  /* Send REG[SetGet]REGMode AUTO/ON */
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_CLIENT_ID], (tU16)enClientId_AmFmTuner);
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FBLOCK_ID], (tU16)enFBlockId_AmFmTuner);
  u8aMessage[ enAdrMsgOffset_INSTANCE_ID] = (tU8)enInstanceId_AudioSource1;
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FKT_ID], (tU16)enMsg_REG);
  u8aMessage[ enAdrMsgOffset_OP_TYPE] = (tU8)enOpType_SETGET;
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA_LEN], 0x0001);
  
  u8aMessage[ enAdrMsgOffset_DATA] = u8REGMode;
  
  ETG_TRACE_USR2(( " tun_MsgToADR::vSendREGMode -> , u32REGMode=0x%02x"
                    ,u8aMessage[10]
                  ));
          
  /* Message to be sent is put in Que   */
  vWaitOnMessageQue( u8aMessage, TUN_CM_REGMODE_SETGET_LEN );
}

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vGetREGMode( )
* 
* DESCRIPTION:  Gets the current REG switch status
*
* PARAMETER:    None
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vGetREGMode()
{
  tU8   u8aMessage[ TUN_CM_ONLY_HEADER];
  
  /* Send REG[Get] */
  if (m_potun_Utility!=NULL)
  {
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_CLIENT_ID], (tU16)enClientId_AmFmTuner);
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FBLOCK_ID], (tU16)enFBlockId_AmFmTuner);
    u8aMessage[ enAdrMsgOffset_INSTANCE_ID] = (tU8)enInstanceId_AudioSource1;
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FKT_ID], (tU16)enMsg_REG);
    u8aMessage[ enAdrMsgOffset_OP_TYPE] = (tU8)enOpType_GET;
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA_LEN], 0x0000);  
  }
  ETG_TRACE_USR2(( " tun_MsgToADR::vGetREGMode "));
  
  /* Message to be sent is put in Que   */
  vWaitOnMessageQue( u8aMessage, TUN_CM_ONLY_HEADER );
}

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendTestTunerSwitches( )
* 
* DESCRIPTION:  Changes the setting for DDS and DMQ
*
* PARAMETER:    None
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vSendTestTunerSwitches(tU8 u8DDS, tU8 u8DMQ)
{
   tU8   u8aMessage[ TUN_CM_TESTTUNERSWITCHES_SET_LEN];
  
  /*Send CM_DDASwitch[SetGet]Phase diversity OnOff*/
  if (m_potun_Utility!=NULL)
  {
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_CLIENT_ID], (tU16)enClientId_AmFmTuner);
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FBLOCK_ID], (tU16)enFBlockId_AmFmTuner);
    u8aMessage[ enAdrMsgOffset_INSTANCE_ID] = (tU8)enInstanceId_All;
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FKT_ID], (tU16)enMsg_CM_TestTunerSwitches);
    u8aMessage[ enAdrMsgOffset_OP_TYPE] = (tU8)enOpType_SET;
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA_LEN], 0x0002);
    
    u8aMessage[ enAdrMsgOffset_DATA] = u8DDS;
    u8aMessage[ (tU8)enAdrMsgOffset_DATA + 1] = u8DMQ;
  }
  
  /* Message to be sent is put in Que   */
  vWaitOnMessageQue( u8aMessage, TUN_CM_TESTTUNERSWITCHES_SET_LEN );
}


/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendSpecialDiagnosisMode
* 
* DESCRIPTION:  Sets the special diagnosis mode
*
* PARAMETER:    None
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vSendSpecialDiagnosisMode( tenCM_SpecialDiagnosisModeMode enMode, tenPhysicalTunerId enPhysicalTunerID)
{
  TUN_ASSERT_RETURN( m_potun_Utility  != NULL);
  TUN_ASSERT_RETURN( m_potun_MsgToHMI != NULL);

  tU8   u8aMessage[ TUN_CM_SPECIAL_DIAGNOSIS_MODE_SET_LEN];
  
  /*Send CM_SpecialDiagnosisMode[Set]Mode, PhysicalTuner*/
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_CLIENT_ID], (tU16)enClientId_AmFmTuner);
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FBLOCK_ID], (tU16)enFBlockId_AmFmTuner);
  u8aMessage[ enAdrMsgOffset_INSTANCE_ID] = (tU8)enInstanceId_All;
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FKT_ID], (tU16)enMsg_CM_SpecialDiagnosisMode);
  
  u8aMessage[ enAdrMsgOffset_OP_TYPE] = (tU8)enOpType_SETGET;
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA_LEN], sizeof( u8aMessage) - (tU8)enAdrMsgOffset_HEADER_LEN);
  
  u8aMessage[ enCM_SpecialDiagnosisMode_idx_Mode] = (tU8)enMode;
  u8aMessage[ enCM_SpecialDiagnosisMode_idx_PhysicalTunerId] = (tU8)enPhysicalTunerID;

  
  ETG_TRACE_USR2(( " tun_MsgToADR::vSendSpecialDiagnosisMode ->PhysicalTuner=%d"
                    ,u8aMessage[ enCM_SpecialDiagnosisMode_idx_PhysicalTunerId]
                  ));
  
  /* Message to be sent is put in Que   */
  vWaitOnMessageQue( u8aMessage, sizeof( u8aMessage) );

}


/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vGetActivePhyTunerAndMeasureMode
* 
* DESCRIPTION:  Gets the current active physical tuner
*
* PARAMETER:    None
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vGetActivePhyTunerAndMeasureMode()
{
	TUN_ASSERT_RETURN( m_potun_Utility  != NULL);
	ETG_TRACE_USR2(( "  tun_MsgToADR::vGetActivePhyTunerAndMeasureMode"));
	
	tU8   u8aMessage[ TUN_CM_ONLY_HEADER];
  
  /*Send CM_SpecialDiagnosisMode[Set]Mode, PhysicalTuner*/
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_CLIENT_ID], (tU16)enClientId_AmFmTuner);
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FBLOCK_ID], (tU16)enFBlockId_AmFmTuner);
  u8aMessage[ enAdrMsgOffset_INSTANCE_ID] = (tU8)enInstanceId_All;
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FKT_ID], (tU16)enMsg_CM_SpecialDiagnosisMode);
  
  u8aMessage[ enAdrMsgOffset_OP_TYPE] = (tU8)enOpType_GET;
   m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA_LEN], 0x0000);  

          
  /* Message to be sent is put in Que   */
  vWaitOnMessageQue( u8aMessage, TUN_CM_ONLY_HEADER );
	
}

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSetResetADR( )
* 
* DESCRIPTION:  Sets the flag to enable/disable resetting of ADR
*
* PARAMETER:    tBool bResetADR
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vSetResetADR(tBool bResetADR)
{
	 ETG_TRACE_USR2(( "  tun_MsgToADR::vSetResetADR ->ResetADR=%d"
                    ,bResetADR
                  ));
	m_bResetADRFlag = bResetADR;
}


/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: bGetResetADR( )
* 
* DESCRIPTION:  Returns the flag whaich indicates if resetting of ADR
				should be enabled/disabled
*
* PARAMETER:    None
*
* RETURNVALUE:  tBool
*
*************************************************************************/
tBool tun_MsgToADR::bGetResetADR()
{
	ETG_TRACE_USR2(( "  tun_MsgToADR::GetResetADR ->ResetADR=%d"
                    ,m_bResetADRFlag
                  ));
	return m_bResetADRFlag;
}

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendTestmode( )
* 
* DESCRIPTION:  Switches ON or OFF the testmode auto notifications
*
* PARAMETER:    None
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vSendTestmode(tU32 u32TestmodeValue, tU16 u16MsgId)
{
  TUN_ASSERT_RETURN( m_potun_Utility != NULL);
  
  if(u32TestmodeValue == 0x00001000) /*Testmode ON*/
  {
    m_bTestModeActivity = TRUE;
  }
  else
  {
    m_bTestModeActivity = FALSE;
  }
  
  
    tU8   u8aMessage[ TUN_CM_TESTMODE_SET_LEN];
  
  /*Send CM_Testmode[Set]TestmodeActivity */
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_CLIENT_ID], (tU16)enClientId_AmFmTuner);
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FBLOCK_ID], (tU16)enFBlockId_AmFmTuner);
  u8aMessage[ enAdrMsgOffset_INSTANCE_ID] = (tU8)enInstanceId_All;
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FKT_ID], u16MsgId);
  u8aMessage[ enAdrMsgOffset_OP_TYPE] = (tU8)enOpType_SET;
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA_LEN], 0x0001);
  u8aMessage[ enAdrMsgOffset_DATA] = ( ( u32TestmodeValue & enTUN_FEATURE_TM_DATA_MODE)) ? enFeatureSwitchON : enFeatureSwitchOFF;
  
  ETG_TRACE_USR2(( " tun_MsgToADR::vSendTestmode -> , TestmodeValue=0x%02x"
                    ,u8aMessage[ enAdrMsgOffset_DATA]
                  ));
          
  /* Message to be sent is put in Que   */
  vWaitOnMessageQue( u8aMessage, TUN_CM_TESTMODE_SET_LEN );
  	  		    
  


#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
  TUN_ASSERT_RETURN( m_poclConfigInterface != NULL);
  /** Send HD TestMode ON/OFF if HD is suppoerted by variant ****/
  if(m_poclConfigInterface->u32GetConfigData("HDSupported"))
  {	  
	TUN_ASSERT_RETURN( m_potun_main != NULL);
	TUN_ASSERT_RETURN( m_potun_main->m_poclAars_HDTestMode != NULL);

	m_potun_main->m_poclAars_HDTestMode->vSetHDTestMode(u8aMessage[ enAdrMsgOffset_DATA]);
	/*vHDSendTestmode(u8aMessage[ enAdrMsgOffset_DATA]);*/
  }
#endif

  
}

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: bGetTestModeActivity
* 
* DESCRIPTION:  Returns the testmode activity ON/OFF
*
* PARAMETER:    tVoid
*
* RETURNVALUE:   tBool
*
*************************************************************************/
tBool tun_MsgToADR::bGetTestModeActivity()
{

	ETG_TRACE_USR2(( " tun_MsgToADR::bGetTestModeActivity -> , TestmodeValue=0x%02x"
	                    ,m_bTestModeActivity
	                  ));
	return m_bTestModeActivity;
}


/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendPersistentStorage
* 
* DESCRIPTION:  Switches ON the persistent storage
*
* PARAMETER:    None
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vSendPersistentStorage(tU32 u32PersistentStorageMode)
{
  tU8   u8aMessage[ TUN_CM_PERSISTENT_STORAGE_SET_LEN];
  
  /*turning off persistent storage setting should be taken care before switching testmode activity ON*/
  
  /*Send CM_PersistentStorage[Set]PersitentStorageOn = ON*/
  if (m_potun_Utility!=NULL)
  {
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_CLIENT_ID], (tU16)enClientId_AmFmTuner);
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FBLOCK_ID], (tU16)enFBlockId_AmFmTuner);
    u8aMessage[ enAdrMsgOffset_INSTANCE_ID] = (tU8)enInstanceId_All;
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FKT_ID], (tU16)enMsg_CM_PersistentStorage);
    u8aMessage[ enAdrMsgOffset_OP_TYPE] = (tU8)enOpType_SET;
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA_LEN], 0x0001);

    u8aMessage[ enAdrMsgOffset_DATA] = ( ( u32PersistentStorageMode & 0x01000000)) ? 0x01 : 0x00; 
  }
  
  /* Message to be sent is put in Que   */
  vWaitOnMessageQue( u8aMessage, TUN_CM_PERSISTENT_STORAGE_SET_LEN );
}



/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vHandleAMFMSettings
* 
* DESCRIPTION:  Requests all the data required on entering AM FM Settings screen
*
* PARAMETER:    None
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vHandleAMFMSettings(tU32 u32AMFMSetup)
{
  (tVoid)u32AMFMSetup; /* unused parameter */

  /*Get the phase diversity value*/
  vSendGetDDASwitch(); 
  
  /*Get the HiCut Value*/
  vGetReceptionParameter(TUN_CM_RECEPTIONPAREMETER_POS_HICUT);
  
  /*Get the Sharx Value*/
  vGetReceptionParameter(TUN_CM_RECEPTIONPAREMETER_POS_SHARX);
  
  /*Get the current TP switch status*/
  //check if any handling is required if the instance ID has to be specified??
  vSendGetAnnouncementSwitch();
  
  /*Get the last set AF switch status*/
  //check if any handling is required if the instance ID has to be specified??
  vGetAFMode();
  
}



/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendSetHiCut( )
* 
* DESCRIPTION:  Sets the Hicut value
*
* PARAMETER:    None.
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vSendSetHiCut (tU8 u8AbsVal)
{
  TUN_ASSERT_RETURN( m_potun_Utility != NULL);

  tU8 u8aMessage[ TUN_C_RECEPTION_PARAM_SET_LEN ];

  /*Send CM_ReceptionParameter[SetGet]HiCut*/
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_CLIENT_ID], (tU16)enClientId_AmFmTuner);
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FBLOCK_ID], (tU16)enFBlockId_AmFmTuner);
  u8aMessage[ enAdrMsgOffset_INSTANCE_ID] = (tU8)enInstanceId_All;
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FKT_ID], (tU16)enMsg_CM_ReceptionParameter);
  u8aMessage[ enAdrMsgOffset_OP_TYPE] = (tU8)enOpType_SETGET;
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA_LEN], 0x0004);
  u8aMessage[ enAdrMsgOffset_DATA] = 2 ; //pos = HiCut
  u8aMessage[ (tU8)enAdrMsgOffset_DATA + 1] =  0;
  u8aMessage[ (tU8)enAdrMsgOffset_DATA + 2] =  u8AbsVal;// Data
  u8aMessage[ (tU8)enAdrMsgOffset_DATA + 3] =  0; // LevelType = Fielstrength

  ETG_TRACE_USR2(( " tun_MsgToADR::vSendSetHiCut -> , posX=%u, posY=%u, HiCut=%u, LevelType=%u"
                    ,u8aMessage[10]
                    ,u8aMessage[11]
          ,u8aMessage[12]
          ,u8aMessage[13]
                  ));
  
  /* Message to be sent is put in Que   */
  vWaitOnMessageQue( u8aMessage, TUN_C_RECEPTION_PARAM_SET_LEN );
}

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendDefaultSetting( )
*
* DESCRIPTION:  Default setting operation is performed
*
* PARAMETER:    tU8 u8DefaultSettingOption.
*
* RETURNVALUE:  tVoid
*
*************************************************************************/
tVoid tun_MsgToADR::vSendDefaultSetting(tU8 u8DefaultSettingOption)
{
	  TUN_ASSERT_RETURN( m_potun_Utility != NULL);
	  TUN_ASSERT_RETURN( m_potun_main != NULL);
	  TUN_ASSERT_RETURN( m_potun_MsgToHMI != NULL);

	  tU8 u8aMessage[ TUN_CM_DEFAULT_SETTING_START_LEN ];

	  /*Send CM_DefaultSetting[StartResult]*/
	  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_CLIENT_ID], (tU16)enClientId_AmFmTuner);
	  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FBLOCK_ID], (tU16)enFBlockId_AmFmTuner);
	  u8aMessage[ enAdrMsgOffset_INSTANCE_ID] = (tU8)enInstanceId_All;
	  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FKT_ID], (tU16)enMsg_CM_DefaultSetting);
	  u8aMessage[ enAdrMsgOffset_OP_TYPE] = (tU8)enOpType_STARTRESULT;
	  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA_LEN], 0x0001);
	  u8aMessage[ enAdrMsgOffset_DATA] = u8DefaultSettingOption; //Only 0 supported

	  ETG_TRACE_USR2(( " tun_MsgToADR::vSendDefaultSetting" ));

	  /* Message to be sent is put in Que   */
	    vWaitOnMessageQue( u8aMessage, TUN_CM_DEFAULT_SETTING_START_LEN );
	    bSetDefaultSettingMsgSent=TRUE;
		m_potun_MsgToHMI->vUpdatePresetListID_DP((tU16)enMsg_CM_ATPresetList_00);
}

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vGetReceptionParameter
* 
* DESCRIPTION:  Requests the specified reception parameter
*
* PARAMETER:    None.
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vGetReceptionParameter (tU8 u8Parameter)
{
  TUN_ASSERT_RETURN( m_potun_Utility != NULL);

  tU8 u8aMessage[ TUN_C_RECEPTION_PARAM_GET_LEN ];

  /*Send CM_ReceptionParameter[SetGet]HiCut*/
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_CLIENT_ID], (tU16)enClientId_AmFmTuner);
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FBLOCK_ID], (tU16)enFBlockId_AmFmTuner);
  u8aMessage[ enAdrMsgOffset_INSTANCE_ID] = (tU8)enInstanceId_All;
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FKT_ID], (tU16)enMsg_CM_ReceptionParameter);
  u8aMessage[ enAdrMsgOffset_OP_TYPE] = (tU8)enOpType_GET;
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA_LEN], 0x0002);
  u8aMessage[ enAdrMsgOffset_DATA] = u8Parameter ; /* HiCut/Sharx */ /** X position */
  u8aMessage[ (tU8)enAdrMsgOffset_DATA + 1] =  0; /** Y position */
             
  ETG_TRACE_USR2(( " tun_MsgToADR::vGetReceptionParameter -> , u8Parameter=%u "
                    ,u8aMessage[ enAdrMsgOffset_DATA]
                  ));
  
  /* Message to be sent is put in Que   */
  vWaitOnMessageQue( u8aMessage, TUN_C_RECEPTION_PARAM_GET_LEN );
}

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendGetAFList( )
* 
* DESCRIPTION:  With this message HMI can request the contents of the AF-List. 
*               
*               The command contains the possibility of notifying to a continous 
*               update if the testmode is activated.
*               
*               The response delivers Only the length of the list, 
*               the list itsself is delivered by a set of CM_ATAfLis
*
*
*
* PARAMETER:    None.
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vSendGetAFList ( tU8 u8Notification )
{

  TUN_ASSERT_RETURN( m_potun_Utility != NULL);

  tenAtAfListActivity enAtAfListActivity = enAtAfListActivity_auto_notifications_off;
  tU16 u16MessageLen = TUN_CM_ATAFLIST_SET_LEN;
  tU8 au8Message[ TUN_CM_ATAFLIST_SET_LEN];
  tun_tenOpType enOpType = enOpType_SET;

  switch( u8Notification)
  {

    /*----------------------------------------------------------------------*/

    case 2:  // TUN_SINGLE_REQUEST

      ETG_TRACE_COMP(( " tun_MsgToADR::vSendGetAFList() -> Get"
                    ));

      u16MessageLen = TUN_CM_ATAFLIST_GET_LEN;
      enOpType = enOpType_GET;

      break;
  
    /*----------------------------------------------------------------------*/

	case 1:  // TUN_NOTIFICATION_ON_TEST_MODE_ON

		enAtAfListActivity = enAtAfListActivity_auto_notifications_on;

		au8Message[enCM_ATAfList_idx_AtAfListActivity] = (tU8)enAtAfListActivity;

		ETG_TRACE_COMP((" tun_MsgToADR::vSendGetAFList() -> Set AtAfListActivity = %02x"
			, ETG_CENUM(tenAtAfListActivity, enAtAfListActivity)
			));
		break;
    /*----------------------------------------------------------------------*/

    default:  // + TUN_NOTIFICATION_OFF

      au8Message[ enCM_ATAfList_idx_AtAfListActivity] = (tU8)enAtAfListActivity;
 
      ETG_TRACE_COMP(( " tun_MsgToADR::vSendGetAFList() -> Set AtAfListActivity = %02x"
                      ,ETG_CENUM( tenAtAfListActivity, enAtAfListActivity)
                    ));

      break;

    /*----------------------------------------------------------------------*/
  }
  
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_CLIENT_ID], (tU16)enClientId_AmFmTuner);
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_FBLOCK_ID], (tU16)enFBlockId_AmFmTuner);
  au8Message[ enAdrMsgOffset_INSTANCE_ID] = (tU8)enInstanceId_All;
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_FKT_ID], (tU16)enMsg_CM_ATEnhancedAfList);
  au8Message[ enAdrMsgOffset_OP_TYPE] = (tU8)enOpType;
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_DATA_LEN], tU16(u16MessageLen - (tU16)enAdrMsgOffset_HEADER_LEN));

  /* Message to be sent is put in Que   */
  vWaitOnMessageQue( au8Message, u16MessageLen );
}



/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendMakeConfigList ( )
* 
* DESCRIPTION:  Command to RU to activate the making of Config list
*
* PARAMETER:    tU8 u8ConfigListID 
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vSendMakeConfigList ( tU32 u32PtyFilter, tU8 u8ConfigListID, tU8 u8SortCriteria )
{
  if(( m_potun_Utility == NULL) ||( m_potun_Config == NULL))
  {
	  TUN_ASSERT_RETURN( m_potun_Utility != NULL);
	  TUN_ASSERT_RETURN( m_potun_Config != NULL);
	  return;
  }
  
  tU8 au8Message[ TUN_CM_STATIONLISTCONFIG_STARTRESULT_LEN];
  
  /** band selection based on config list id */
  tU32 u32Band = u32GetBandFromConfigListID(u8ConfigListID);
  
  tenCM_StationListConfigRdsFilter enRdsFilter = enCM_StationListConfigRdsFilter_RDS_NONRDS;
  
  ETG_TRACE_COMP(( " tun_MsgToADR::vSendMakeConfigList() -> PTY=0x%08X, u8ListID=%d, u32Band=0x%08X, enCM_StationListConfigRdsFilter=%x, enSorting=%d"
                  ,u32PtyFilter
                  ,u8ConfigListID
                  ,u32Band
                  ,ETG_CENUM( tenCM_StationListConfigRdsFilter, enRdsFilter)
                  ,u8SortCriteria
                ));

  vFormat_Msg_Header(au8Message, (tU16)enClientId_AmFmTuner, (tU16)enFBlockId_AmFmTuner,(tU8)enInstanceId_AudioSource1,(tU16)enMsg_CM_StationListConfig, (tU8)enOpType_STARTRESULT);
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_DATA_LEN], TUN_CM_STATIONLISTCONFIG_STARTRESULT_LEN - (tU8)enAdrMsgOffset_HEADER_LEN);

  au8Message[ enCM_StationListConfig_idx_ListId] = u8ConfigListID;
  m_potun_Utility->vConvert32BitTo4SeparateBytes( &au8Message[ enCM_StationListConfig_idx_Band], u32Band);

#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
    au8Message[ enCM_StationListConfig_idx_AnalogFilter] = (tU8)enRdsFilter;
    au8Message[ enCM_StationListConfig_idx_DigitalFilter] = (tU8)0;
	m_potun_Utility->vConvert32BitTo4SeparateBytes( &au8Message[ enCM_StationListConfig_idx_RFUBytes], 0);
#else
  au8Message[ enCM_StationListConfig_idx_RdsFilter] = (tU8)enRdsFilter;
#endif

#ifdef VARIANT_S_FTR_ENABLE_FEATURE_VW_MIBE
 if(m_potun_Config->m_rKdsEOLParameter.bGetMultipleEntrySwitch())
  {
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
	au8Message[ enCM_StationListConfig_idx_AnalogFilter] = (tU8)enCM_StationListConfigRdsFilter_RDS_NONRDS_RDSunknown;
#else
	  au8Message[ enCM_StationListConfig_idx_RdsFilter] = (tU8)enCM_StationListConfigRdsFilter_RDS_NONRDS_RDSunknown;
#endif
  }
#endif

  m_potun_Utility->vConvert32BitTo4SeparateBytes( &au8Message[ enCM_StationListConfig_idx_PtyFilter], u32PtyFilter);
  au8Message[ enCM_StationListConfig_idx_Sorting] = u8SortCriteria;
  
  /* Message to be sent is put in Que   */
  vWaitOnMessageQue( au8Message, TUN_CM_STATIONLISTCONFIG_STARTRESULT_LEN );
}



/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: u32GetBandFromConfigListID (tU8 u8ConfigListID)
* 
* DESCRIPTION:  Get correspoding Band for ConfigListID
*
* PARAMETER:   tU8 u8ConfigListID
*
* RETURNVALUE:  tU32 
*
* AUTHOR: vnd4kor ECV2 19/02/2014
*************************************************************************/
tU32 tun_MsgToADR::u32GetBandFromConfigListID ( tU8 u8ConfigListID) const
{
	tU32 u32Band = 0x00;
  /** band selection based on config list id */
  switch (u8ConfigListID)
  {
  case enCM_StationListConfigListID_FM : 
	  {
		  u32Band = TUN_CM_STATIONLISTCONFIG_BAND_BIT_MSK_FM;
	  }
	  break;
  case enCM_StationListConfigListID_MW : 
	  {
		  u32Band = TUN_CM_STATIONLISTCONFIG_BAND_BIT_MSK_MW;
	  }
	  break;
  case enCM_StationListConfigListID_LW :
	  {
		  u32Band = TUN_CM_STATIONLISTCONFIG_BAND_BIT_MSK_LW;
	  }
	  break;
  case enCM_StationListConfigListID_AM : 
	  {
		  u32Band |= TUN_CM_STATIONLISTCONFIG_BAND_BIT_MSK_MW;
		  u32Band |= TUN_CM_STATIONLISTCONFIG_BAND_BIT_MSK_LW;
	  }
	  break;
  default:
	  break;
  }

  return u32Band;
}

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendGetConfigList ( )
* 
* DESCRIPTION:  Command to ADR3 to Get the current active 
*               Config list using CM_StationsList
*
* PARAMETER:    
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vSendGetConfigList ( tU8 u8ConfigListID, tU8 u8PosX, tU8 u8PosY )
{ 
  TUN_ASSERT_RETURN( m_potun_Utility != NULL);

  tU8 au8Message[ TUN_CM_ONLY_HEADER + 3];

  ETG_TRACE_COMP(( " tun_MsgToADR::vSendGetConfigList() -> u8ListID=%d"
                  ,u8ConfigListID
                ));

  m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_CLIENT_ID], (tU16)enClientId_AmFmTuner);
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_FBLOCK_ID], (tU16)enFBlockId_AmFmTuner);
  au8Message[ enAdrMsgOffset_INSTANCE_ID] = (tU8)enInstanceId_All;
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_FKT_ID], (tU16)enMsg_CM_StationsList);
  au8Message[ enAdrMsgOffset_OP_TYPE] = (tU8)enOpType_GET;
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_DATA_LEN], sizeof( au8Message) - (tU8)enAdrMsgOffset_HEADER_LEN);

  au8Message[ enCM_StationsList_idx_PosX] =  u8PosX;  
  au8Message[ enCM_StationsList_idx_PosY] =  u8PosY;

  au8Message[ enCM_StationsList_get_idx_ListId]  =  u8ConfigListID;
  
  /* Message to be sent is put in Que   */
  vWaitOnMessageQue( au8Message, sizeof( au8Message) );

}

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendGetTMCList ( )
* 
* DESCRIPTION:  Command to ADR3 to Get the current active 
*               TMC list
*
* PARAMETER:    
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vSendGetTMCList ()
{

  TUN_ASSERT_RETURN( m_potun_Utility != NULL);

  tU8 au8Message[ TUN_CM_ONLY_HEADER + 2];

  ETG_TRACE_COMP(( " tun_MsgToADR::vSendGetTMCList() "));
  
  vFormat_Msg_Header(au8Message, (tU16)enClientId_TMCTuner, (tU16)enFBlockId_TMCTuner,(tU8)enInstanceId_All,(tU16)enMsg_CM_TMCStations, (tU8)enOpType_GET);


  m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_DATA_LEN], sizeof( au8Message) - (tU16)enAdrMsgOffset_HEADER_LEN);

  au8Message[ enAdrMsgOffset_DATA] = 0;
  au8Message[ (tU8)enAdrMsgOffset_DATA+1] = 0;
  /* Message to be sent is put in Que   */
  vWaitOnMessageQue( au8Message, sizeof( au8Message) );

}


/*************************************************************************
*
* FUNCTION:     tun_MsgToADR :: vSendUpdateConfigList ( );
* 
* DESCRIPTION:  Command to RU to activate the Updating of Config list
*
* PARAMETER:    tU8 u8ConfigListID, tU8 u8UpdateMode 
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR :: vSendUpdateConfigList (tU8 u8ConfigListID, tU8 u8UpdateMode)
{
  TUN_ASSERT_RETURN( m_potun_Utility != NULL);

  tU8 au8Message[ TUN_CM_ONLY_HEADER + 1];

  ETG_TRACE_COMP(( " tun_MsgToADR::vSendUpdateConfigList() -> u8ListID=%d, u8UpdateMode=%d"
                  ,u8ConfigListID
                  ,u8UpdateMode
                ));

  if( u8UpdateMode == 0x01)
  {
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_CLIENT_ID], (tU16)enClientId_AmFmTuner);
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_FBLOCK_ID], (tU16)enFBlockId_AmFmTuner);
    au8Message[ enAdrMsgOffset_INSTANCE_ID] = (tU8)enInstanceId_AudioSource1;
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_FKT_ID], (tU16)enMsg_CM_StationListRelease);
    au8Message[ enAdrMsgOffset_OP_TYPE] = (tU8)enOpType_SET;
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_DATA_LEN], sizeof( au8Message) - (tU8)enAdrMsgOffset_HEADER_LEN);

    au8Message[ enCM_StationListRelease_idx_ListId]  =  u8ConfigListID;
  
    /* Message to be sent is put in Que   */
    vWaitOnMessageQue( au8Message, sizeof( au8Message));
  }

}


/*************************************************************************
*
* FUNCTION:     tun_MsgToADR :: vSend_SetPI ( );
* 
* DESCRIPTION:  Command to RU to get the responses from RU without giving 
*               respective command
*
* PARAMETER:    tU8 u8DataMode, tU8 u8RequestedMsg
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR :: vSend_SetPI (tU8 u8TunerHeader ,tU8 u8AbsOrRel, tU8 u8RelSteps)
{
  (tVoid)u8TunerHeader;  /* unused parameter */
  (tVoid)u8AbsOrRel;     /* unused parameter */

  TUN_ASSERT_RETURN( m_potun_Utility != NULL);

  /*Currently there is no usecase where we get Set_PI request with Absolute PI stepping,
  hence u8AbsOrRel is not evaluated*/ 
  
  tU8 u8aMessage[ TUN_C_ATPI_LEN ];

  /*Send CM_ReceptionParameter[SetGet]HiCut*/
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_CLIENT_ID], (tU16)enClientId_AmFmTuner);
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FBLOCK_ID], (tU16)enFBlockId_AmFmTuner);
  u8aMessage[ enAdrMsgOffset_INSTANCE_ID] = (tU8)enInstanceId_AudioSource1;
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FKT_ID], (tU16)enMsg_ATPI);
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA_LEN], 0x0001);
  
  if(u8RelSteps == TUN_SEEK_DIRECTION_UP)
  {
    u8aMessage[ enAdrMsgOffset_OP_TYPE] = (tU8)enOpType_Increment;
  }
  else
  {
    u8aMessage[ enAdrMsgOffset_OP_TYPE] = (tU8)enOpType_Decrement;
  }
  /*Increment or decrement by 1*/
  u8aMessage[ enAdrMsgOffset_DATA] = 0x01 ;     
  
  ETG_TRACE_USR2(( " tun_MsgToADR::vSend_SetPI -> , enOpType=%d, Steps=%d"
                    ,ETG_CENUM( tun_tenOpType, u8aMessage[ enAdrMsgOffset_OP_TYPE])
                    ,u8aMessage[ enAdrMsgOffset_DATA] 
                  ));
          
  /* Message to be sent is put in Que   */
  vWaitOnMessageQue( u8aMessage, TUN_C_ATPI_LEN );
  
  
  
  
 /* if( u8AbsOrRel == TU_ABS_VALUE)
  {
    NORMAL_M_ASSERT_ALWAYS(); // tbd
  }
  else // u8AbsOrRel == TU_REL_VALUE
  {
    if( u8RelSteps == TUN_SEEK_DIRECTION_UP)
    {
      trMsg_ATPI_Increment oMsg_ATPI_Increment( 0x01);  // 1 increment
    
      if( u8TunerHeader == TU_FOREGROUND)
      {
        oMsg_ATPI_Increment.enInstanceId = enInstanceId_AudioSource1;
      }
      else  // u8TunerHeader == TU_BACKGROUND
      {
        oMsg_ATPI_Increment.enInstanceId = enInstanceId_AudioSource2;
      }
      vSend( &oMsg_ATPI_Increment);
    }
    else   // u8RelSteps == TUN_SEEK_DIRECTION_DOWN
    {
      trMsg_ATPI_Decrement oMsg_ATPI_Decrement( 0x01);  // 1 decrement

      if( u8TunerHeader == TU_FOREGROUND)
      {
        oMsg_ATPI_Decrement.enInstanceId = enInstanceId_AudioSource1;
      }
      else  // u8TunerHeader == TU_BACKGROUND
      {
        oMsg_ATPI_Decrement.enInstanceId = enInstanceId_AudioSource2;
      }
      vSend( &oMsg_ATPI_Decrement);
    }
  }*/
}

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSend_SetSharxLevel( )
* 
* DESCRIPTION:  Which this command the Sharx level can be modified.
*
* PARAMETER:    tU8,tU8,tU8.
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vSend_SetSharxLevel (tU8 u8SharxStep)
{ 
  TUN_ASSERT_RETURN( m_potun_Utility != NULL);

  tU8 u8aMessage[ TUN_C_RECEPTION_PARAM_SET_LEN ];

  /*Send CM_ReceptionParameter[SetGet]HiCut*/
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_CLIENT_ID], (tU16)enClientId_AmFmTuner);
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FBLOCK_ID], (tU16)enFBlockId_AmFmTuner);
  u8aMessage[ enAdrMsgOffset_INSTANCE_ID] = (tU8)enInstanceId_All;
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FKT_ID], (tU16)enMsg_CM_ReceptionParameter);
  u8aMessage[ enAdrMsgOffset_OP_TYPE] = (tU8)enOpType_SETGET;
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA_LEN], 0x0004);
  u8aMessage[ enAdrMsgOffset_DATA] = 1 ; //pos = Sharx
  u8aMessage[ (tU8)enAdrMsgOffset_DATA + 1] =  0;
  u8aMessage[ (tU8)enAdrMsgOffset_DATA + 2] =  u8SharxStep;// Data
  u8aMessage[ (tU8)enAdrMsgOffset_DATA + 3] =  0; // LevelType = Fielstrength
  
  ETG_TRACE_USR2(( " tun_MsgToADR::vSend_SetSharxLevel -> , posX=%u, posY=%u, SharxStep=%u, LevelType=%u"
                    ,u8aMessage[10]
                    ,u8aMessage[11]
          ,u8aMessage[12]
          ,u8aMessage[13]
                  ));
          
  /* Message to be sent is put in Que   */
  vWaitOnMessageQue( u8aMessage, TUN_C_RECEPTION_PARAM_SET_LEN );

}

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendGetAtStationInfo( )
* 
* DESCRIPTION:  Request ADR to send the status for AtStationInfo
*
* PARAMETER:    tVoid

*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vSendGetAtStationInfo()
{
	tU8   u8aMessage[ TUN_CM_ONLY_HEADER];
  
  /*Send CM_AtStationInfo[Get]*/
  if (m_potun_Utility!=NULL)
  {
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_CLIENT_ID], (tU16)enClientId_AmFmTuner);
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FBLOCK_ID], (tU16)enFBlockId_AmFmTuner);
    u8aMessage[ enAdrMsgOffset_INSTANCE_ID] = (tU8)enInstanceId_AudioSource1; 
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FKT_ID], (tU16)enMsg_CM_ATStationInfo);
    u8aMessage[ enAdrMsgOffset_OP_TYPE] = (tU8)enOpType_GET;
    m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA_LEN], 0x0000);  
  }  
  ETG_TRACE_USR2(( " tun_MsgToADR::vSendGetAtStationInfo"));
          
  /* Message to be sent is put in Que   */
  vWaitOnMessageQue( u8aMessage, TUN_CM_ONLY_HEADER );
}


/*************************************************************************
*
* FUNCTION:     tun_MsgToADR::vSendSetATMContainer()
* 
* DESCRIPTION:   Send a data transparent to ADR manager
*
* PARAMETER:    tU8 - No.of data bytes sent, tu8* - data bytes

*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vSendSetATMContainer(tU8 u8DataLength,const tU8* u8DataSpecial) 
{
  TUN_ASSERT_RETURN( m_potun_Utility != NULL);
  TUN_ASSERT_RETURN( (TUN_DRVADRIF_CONF_SSI_AMFM_TUNER_TX_BUFFER_SIZE - (u8DataLength + (tU8)enAdrMsgOffset_HEADER_LEN)) >= 0);

  tU8   u8aMessage[ TUN_DRVADRIF_CONF_SSI_AMFM_TUNER_TX_BUFFER_SIZE];
  
  ETG_TRACE_COMP(( " tun_MsgToADR::vSendSetATMContainer() -> u8DataLength = 0x%02x"
    , u8DataLength
    ));
    
  /*Send CM_Container[StartResult]*/
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_CLIENT_ID], (tU16)enClientId_AmFmTuner);
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FBLOCK_ID], (tU16)enFBlockId_AmFmTuner);
  u8aMessage[ enAdrMsgOffset_INSTANCE_ID] = (tU8)enInstanceId_All;
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FKT_ID], (tU16)enMsg_CM_Container);
  u8aMessage[ enAdrMsgOffset_OP_TYPE] = (tU8)enOpType_STARTRESULT;
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA_LEN], u8DataLength);
  
  for(tU8 u8ContainerData = 0; u8ContainerData <  u8DataLength; u8ContainerData++ ) 
  {
    u8aMessage[(tU8)enAdrMsgOffset_DATA + u8ContainerData] = u8DataSpecial[u8ContainerData];
    ETG_TRACE_USR2(( " tun_MsgToADR::vSendSetATMContainer() -> u8aMessage[%d] = 0x%02x"
      , (tU8)enAdrMsgOffset_DATA + u8ContainerData
      , u8aMessage[(tU8)enAdrMsgOffset_DATA + u8ContainerData]
    ));
  }

  /* Message to be sent is put in Que   */
  vWaitOnMessageQue( u8aMessage, (tU8)enAdrMsgOffset_HEADER_LEN + u8DataLength, TRUE);
}


/*************************************************************************
*
* FUNCTION:     tun_MsgToADR::vSendSetRDBID()
*
* DESCRIPTION:  Sends RDBID Wish list
*
* PARAMETER:  const tU8* pu8RDBIDList ,tU8 u8ListIndex

*
* RETURNVALUE:  tVoid
*
*************************************************************************/
tVoid tun_MsgToADR::vSendSetRDBID(const tU8* pu8RDBIDList ,tU8 u8ListIndex)
{
  TUN_ASSERT_RETURN( m_potun_Utility != NULL);
  

  tU8   u8aMessage[TUN_DRVADRIF_CONF_SSI_AMFM_TUNER_TX_BUFFER_SIZE];

  ETG_TRACE_COMP(( " tun_MsgToADR::vSendSetRDBID() " ));


  /*Send CM_TMCSelectionFilter[Set]*/
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_CLIENT_ID], (tU16)enClientId_TMCTuner);
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FBLOCK_ID], (tU16)enFBlockId_TMCTuner);
  u8aMessage[ enAdrMsgOffset_INSTANCE_ID] = (tU8)enInstanceId_All;
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FKT_ID], (tU16)enMsg_CM_TMCSelectionFilter);
  u8aMessage[ enAdrMsgOffset_OP_TYPE] = (tU8)enOpType_SETGET;
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA_LEN], tU16(u8ListIndex+2));

  u8aMessage[ enAdrMsgOffset_DATA] = 0;
  u8aMessage[ (tU8)enAdrMsgOffset_DATA+1] = 0;

  if(pu8RDBIDList != NULL)
  {
	for(tU8 u8Index = 0; u8Index < u8ListIndex; u8Index++)
     {
	  u8aMessage[ (tU8)enAdrMsgOffset_DATA+u8Index + 2] = pu8RDBIDList[u8Index];
     }
  }
  
  /* Message to be sent is put in Que   */
  vWaitOnMessageQue( u8aMessage, TUN_CM_SETRDBID_SET_LEN + u8ListIndex);
}

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vMsgToADRFromTrace( )
* 
* DESCRIPTION:  Send data to Radio unit from trace component.
*
* PARAMETER:    
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vMsgToADRFromTrace( tPU8 pu8Msg, tU32 u32Len )
{
  tU8  au8Buffer[ TUN_DRVADRIF_CONF_SSI_AMFM_TUNER_TX_BUFFER_SIZE];
  tU16 u16DataLen = 0;
  
  /* copy parameters: u16ClientID, u16FBlockId, u8InstanceId, u16FunctionId and u8OpType */
  OSAL_pvMemoryCopy( au8Buffer, &pu8Msg[ 0], (tU8)enAdrMsgOffset_DATA_LEN);

  /* set data length */
  if( u32Len >= (tU32)enAdrMsgOffset_DATA_LEN)
  {
    u16DataLen = (tU16)(u32Len - (tU32)enAdrMsgOffset_DATA_LEN);
  }
  TUN_DRVADRIF_SET_U16( &au8Buffer[ enAdrMsgOffset_DATA_LEN], u16DataLen);
 
  /* copy the data */
  OSAL_pvMemoryCopy( &au8Buffer[ enAdrMsgOffset_DATA], &pu8Msg[ enAdrMsgOffset_DATA_LEN], u16DataLen);

  vWaitOnMessageQue( au8Buffer, u32Len + 2, TRUE);
}


tVoid tun_MsgToADR::vMsgToADRFromTrace( tU8 u8Command, tU8 u8Para1, tU8 u8Para2, tU8 u8Para3)
{
    if ((m_potun_Utility!=NULL)&& (m_potun_ActualData !=NULL))
    {
     switch( u8Command )
     {
     case TUN_C_SEEK:
       {
          tU32 u32Frequency = m_potun_ActualData->m_otun_Data_Station.u32GetFrequency();

          vSendSeekStart (u8Para2 /* u8SeekType */, u32Frequency/* u32Frequency */); 
       }
       break;
     case TUN_C_SET_SCANTIME:
       {
       }
       break;
     case TUN_C_SET_HICUT:
       {
         vSendSetHiCut (u8Para1 /* u8AbsVal */); 
       }
       break;

     case TUN_C_MAKE_CONFIG_LIST:
       {
         vSendMakeConfigList( u8Para1 /* WhichTuner */, u8Para2/* u8ConfigListID */,u8Para3);
       }
      break;

     case TUN_C_UPDATE_CONFIG_LIST:
       {
         vSendUpdateConfigList(u8Para1 /* u8ConfigListID */, u8Para2 /*u8UpdateMode */);
       }
      break;

     case TUN_C_SET_PI:
       {

        vSend_SetPI (u8Para1 /* u8TunerHeader */, u8Para2 /* u8AbsOrRel */,u8Para3 /* u8RelSteps */);

       }
      break;

     case TUN_C_SET_SHARX_LEVEL:
       {

        vSend_SetSharxLevel (u8Para3 /*u8SharxStep */ );

       }
      break;
     case TUN_C_SNAPSHOT:
      {
       vSendSnapshot ( u8Para1 /*u8Action */);
        
      }
      break;
         

     default:
       {
           // do nothing
       }
       break;
     }
}
}


/*****************************************************************************
 * Incomming message from TTFis
 ****************************************************************************/
tVoid tun_MsgToADR::vTraceCommand( tPU8 pu8Msg, tU32 u32Len )
{
  (tVoid)u32Len;  /* unused parameter */

  switch( pu8Msg[ 0])
  {
    /*----------------------------------------------------------------------*/

    case TUN_TRACE_TUN_MSGTOADR_SETCONFIGURATION:
      {
        tenTraceSetConfiguration enTraceSetConfiguration = (tenTraceSetConfiguration)pu8Msg[ 1];
        tU32 u32CRC;

        ETG_TRACE_FATAL(( " tun_MsgToADR::vTraceCommand() -> TTFis command: TUN_TRACE_TUN_MSGTOADR_SETCONFIGURATION.\n"
                          "=> bTraceOnly        : %02x\n"
                         ,ETG_CENUM( tenTraceSetConfiguration,    enTraceSetConfiguration)
                       ));

        (tVoid)bSendCM_SetConfiguration( enTraceSetConfiguration, &u32CRC);

        ETG_TRACE_FATAL(( " tun_MsgToADR::vTraceCommand() -> TTFis command: TUN_TRACE_TUN_MSGTOADR_SETCONFIGURATION u32CRC=0x%08X"
                         ,u32CRC
                       ));
      }
      break;

    /*----------------------------------------------------------------------*/

    default:
      ETG_TRACE_FATAL(( " tun_MsgToADR::vTraceCommand() -> Error: unknown command."
                     ));
      break;

    /*----------------------------------------------------------------------*/
  }
}


/* *****************************************************************************

   FUNCTION:      vDeleteNoThrow

   DESCRIPTION:   Deletes memory without throwing an exception.

   PARAMETERS:    rfpArg: reference of pointer to memory, to be deleted

   RETURN TYPE:   void

   HINT:          Why this function? If delete is called inside the destructor,
                  Lint complains with warning 1551 that delete might throw an
                  execption, which is not caught.

                  But try / catch must not be used, and delete does not throw
                  an exception, anyway. To prevent a Lint warning deactivation,
                  this function guarantees that no exception is thrown.
                  Furthermore, it sets the given pointer to 0, and Lint is
                  happy.

                  It is a template, because two different memory types have to
                  be deleted inside this class.

----------------------------------------------------------------------------- */

template< class T > void tun_MsgToADR::vDeleteNoThrow( T*& rfpArg ) throw()
{
   OSAL_DELETE rfpArg;
   rfpArg = 0;
}


/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vConfigureAdr( )
* 
* DESCRIPTION:  Is called by CM_Startup message from ADR
*
* PARAMETER:    None
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vConfigureAdr( tVoid)
{
   ETG_TRACE_COMP(( " tun_MsgToADR::vConfigureAdr() ->" ));

  vSendCM_VersionInfo();
  TUN_ASSERT_RETURN( m_potun_main != NULL);
  /*if(m_potun_main->bIsSBRVariant())
  {
	  vNotifyFunctionIDs();
  }
  else*/
  {
    //vGetUser();
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
  	TUN_ASSERT_RETURN( m_potun_main->m_poclAars_GetConfigurationInfo  != NULL);
  	m_potun_main->m_poclAars_GetConfigurationInfo->vGetConfigurationinfo();
#endif
#ifndef VARIANT_S_FTR_ENABLE_DP_CLIENT_MOCK
    m_potun_main->m_poclAars_AdditionalVersionInfo->vGetAdditionalVersionInfo();
#endif
  }
}



tVoid tun_MsgToADR::vGetUser()
{
#ifdef VARIANT_S_FTR_ENABLE_FEATURE_PSA_RCC
ETG_TRACE_COMP(( " tun_MsgToADR::vGetUser() ->" ));

tU8   u8aMessage[ TUN_CM_ONLY_HEADER];

if (m_potun_Utility!=NULL)
{
   vFormat_Msg_Header(u8aMessage, (tU16)enClientId_AmFmTuner, (tU16)enFBlockId_AmFmTuner,(tU8)enInstanceId_All,(tU16)enMsg_CM_GetUser, (tU8)enOpType_GET);
   m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA_LEN], 0x0000);
}
vWaitOnMessageQue( u8aMessage, TUN_CM_ONLY_HEADER );
#endif
#ifdef VARIANT_S_FTR_ENABLE_FEATURE_RNAIVI
ETG_TRACE_COMP(( " tun_MsgToADR::vGetUser() ->" ));
tU8   u8aMessage[ TUN_CM_ONLY_HEADER];
if (m_potun_Utility!=NULL)
{
   vFormat_Msg_Header(u8aMessage, (tU16)enClientId_AmFmTuner, (tU16)enFBlockId_AmFmTuner,(tU8)enInstanceId_AudioSource1,(tU16)0x0fec, (tU8)enOpType_GET);
   m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA_LEN], 0x0000);
}
vWaitOnMessageQue( u8aMessage, TUN_CM_ONLY_HEADER );
#endif
}





/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vNotifyFunctionIDs( )
* 
* DESCRIPTION:  Set notifications for all used ADR messages
*
* PARAMETER:    None
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vNotifyFunctionIDs( tVoid)
{
	ETG_TRACE_USR1(( " tun_MsgToADR::vNotifyFunctionIDs() -> %p			Manager: %p", (void *)m_potun_Config, (void *)m_potu_Manager));
  TUN_ASSERT_RETURN( m_potun_Config  != NULL);
  TUN_ASSERT_RETURN( m_poclConfigInterface  != NULL);

  /*==========================================================================
   * set all ADR notifications
   *------------------------------------------------------------------------*/
 /* trMsg_Notification_Set oMsg_Notification_Set_AMFMTuner( enNotificationControl_SetAll, enClientId_AmFmTuner);
  vSend( &oMsg_Notification_Set_AMFMTuner);

  trMsg_Notification_Set oMsg_Notification_Set_TMCTuner( enNotificationControl_SetAll, enClientId_TMCTuner);
    vSend( &oMsg_Notification_Set_TMCTuner);*/

    vSendNotificationRequest((tU8)enNotificationControl_SetAll, (tU16)enClientId_AmFmTuner, (tU16)enFBlockId_AmFmTuner);
	
	ETG_TRACE_USR1(( " tun_MsgToADR::vNotifyFunctionIDs --------------------------------------    1 " ));

    TUN_ASSERT_RETURN( m_poclConfigInterface != NULL);
	/*Only if FM TMC Server is enabled, notify for TMC messages*/
	if( m_potun_Config->u8GetTMCServer())
	{
		vSendNotificationRequest((tU8)enNotificationControl_SetAll, (tU16)enClientId_TMCTuner, (tU16)enFBlockId_TMCTuner);
	}

	if(NULL != m_potu_Manager)
	{
		ETG_TRACE_USR1(( " tun_MsgToADR::Entered into the componentsstatus" ));
		tun_DrvAdrIf::instance()->vSetCommunicationState(enComState_Run);
		m_potu_Manager -> vUpdateComponentStatus();
	}
	else
	{
		ETG_TRACE_USR1(( " tun_MsgToADR::POINTER IS NULL--------------------------------------" ));
	}
		
  /*========================================================================*/

  /*==========================================================================
   * clear notification for ATStationInfo
   *------------------------------------------------------------------------*/
 /*   oMsg_Notification_Set_AMFMTuner.enControl = enNotificationControl_ClearFunction;
    oMsg_Notification_Set_AMFMTuner.AddFktID( enMsg_ATStationInfo);
  vSend( &oMsg_Notification_Set_AMFMTuner);*/
  /*========================================================================*/



  /*==========================================================================
   * set tuner 1 to  'audio tuner'
   *------------------------------------------------------------------------*/
  /*Msg_CM_AudioSource_Set oMsg_CM_AudioSource_Set;

  oMsg_CM_AudioSource_Set.enAudioSource = enAudioSource_User_audio;
  oMsg_CM_AudioSource_Set.enOpType      = enOpType_SETGET;
  vSend( &oMsg_CM_AudioSource_Set);*/
  /*========================================================================*/
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
	/** Send Set PSD data at startup to get autonotifications ****/
	   if(m_poclConfigInterface->u32GetConfigData("HDSupported"))
	   {
		   vSendSetPSD_Data();
	   }

#endif
}


/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: bSendCM_SetConfiguration( )
* 
* DESCRIPTION:  Send the configuration to the ADR by command CM_SetConfiguration
*
* PARAMETER: [in]   enTraceType - trace | send
*
* PARAMETER: [out]  pu32Crc - CRC checksum
*
* RETURNVALUE:  tBool  TRUE  : message is send
*                      FALSE : Error, message is not send
*
*************************************************************************/
tBool tun_MsgToADR::bSendCM_SetConfiguration(  tenTraceSetConfiguration enTraceType, tPU32 pu32Crc)
{
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
	TUN_ASSERT_RETURN_VAL( m_potun_main != NULL, FALSE);
    TUN_ASSERT_RETURN_VAL( m_potun_main->m_poclTmcTuner_SetConfigSections  != NULL, FALSE);
    TUN_ASSERT_RETURN_VAL( m_potun_MsgToHMI  != NULL, FALSE);
    TUN_ASSERT_RETURN_VAL( m_potun_Config  != NULL, FALSE);
	TUN_ASSERT_RETURN_VAL( m_poclConfigInterface != NULL, FALSE);
	m_potun_main->m_poclTmcTuner_SetConfigSections->vSendSetConfigSections((tU8)enTraceType,(unsigned int*)pu32Crc,&u8HWCrc);
	
	/** initialise the RDS switch member variable with KDS data */
	m_potun_MsgToHMI->vSetRDSSwitchStatus((tU8)(m_poclConfigInterface->u32GetConfigData("RDSSwitchAudioTuner1")));

//	if( ( (tU8)enTraceType & (tU8)enTraceSetConfiguration_Send) != 0)
//	{
//		m_bSetConfigurationMessageSent = TRUE;
//	}
	return TRUE;
#else
  TUN_ASSERT_RETURN_VAL( m_potun_Utility != NULL, FALSE);
  TUN_ASSERT_RETURN_VAL( m_potun_Config  != NULL, FALSE);
  TUN_ASSERT_RETURN_VAL( m_potun_MsgToHMI != NULL, FALSE);

  tChar sKDSVersionId[ TUN_CONFIG_VERSIONID_LEN];

  m_potun_Config->vGetVersionId( sKDSVersionId, sizeof( sKDSVersionId));

  /*==========================================================================
   * Calculate the message length depending on the KDS entrys for Preset.
   * Check the message length and do nothing (return FALSE), if the buffer
   * is to small for the current konfig data.
   *------------------------------------------------------------------------*/
  tU32 u32MessageLen =  (tU32)enCM_SetConfiguration_idx_PresetList                    /* lebgth of constat part of message          */
    + ( m_potun_Config->u8GetNoFMpresetBanks() * 2)                             /* size for 'FM Band' and 'number of Aresets' */
    + ( m_potun_Config->u8GetNoFMpresetBanks() * m_potun_Config->u8GetNoOfFMpresetPerBank() * 4)   /* size for the FM Presets */
    + ( m_potun_Config->u8GetNoAMpresetBanks1() * 2)                            /* size for 'AM Band' and 'number of Presets' */
    + ( m_potun_Config->u8GetNoAMpresetBanks1() * m_potun_Config->u8GetNoOfAMpresetPerBank1() * 4) /* size for the AM Presets */
    + ( m_potun_Config->u8GetNoMIXEDpresetBanks() * 2)
    + ( m_potun_Config->u8GetNoMIXEDpresetBanks() * m_potun_Config->u8GetNoOfMIXEDpresetPerBank() * 4)
    + 2;                                                                        /* two bytes for 'end of list'*/
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
	u32MessageLen=u32MessageLen+(m_potun_Config->u8GetNoDRMpresetBanks()*2)+(m_potun_Config->u8GetNoDRMpresetBanks()* m_potun_Config->u8GetNoOfDRMpresetPerBank()*4);
#endif
  tU16 u16NumberOfPayloadBytes = u32MessageLen - (tU32)enCM_SetConfiguration_idx_NumOfSeekPeriods;

  ETG_TRACE_COMP(( " tun_MsgToADR::bSendCM_SetConfiguration() -> enTraceType=%x, sKDSVersionId=%11s\n"
                   " u32MessageLen           : %8d bytes\n"
                   " u16NumberOfPayloadBytes : %8d bytes\n"
                  ,ETG_CENUM(tenTraceSetConfiguration,         enTraceType)
                  ,sKDSVersionId
                  ,u32MessageLen
                  ,u16NumberOfPayloadBytes
                 ));

  TUN_ASSERT_RETURN_VAL( u32MessageLen < TUN_CM_SETCONFIGURATION_STARTRESULT_LEN, FALSE);

  tU8 au8Message[ TUN_CM_SETCONFIGURATION_STARTRESULT_LEN];
  /*========================================================================*/


  m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_CLIENT_ID], (tU16)enClientId_AmFmTuner);
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_FBLOCK_ID], (tU16)enFBlockId_AmFmTuner);
  au8Message[ enAdrMsgOffset_INSTANCE_ID] = (tU8)enInstanceId_All;
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_FKT_ID], (tU16)enMsg_CM_SetConfiguration);
  au8Message[ enAdrMsgOffset_OP_TYPE] = (tU8)enOpType_STARTRESULT;
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_DATA_LEN], u32MessageLen - (tU32)enAdrMsgOffset_HEADER_LEN);
  
  OSAL_pvMemoryCopy( &au8Message[ enCM_SetConfiguration_idx_VersionId], sKDSVersionId, sizeof( sKDSVersionId));
  OSAL_pvMemorySet(  &au8Message[ (tU16)enCM_SetConfiguration_idx_VersionId + sizeof( sKDSVersionId)], TUN_INIT_UNUSED_BYTE, (tU16)enCM_SetConfiguration_len_VersionId - sizeof( sKDSVersionId)); // set the reserved bytes to 0xAA
  m_potun_Utility->vConvert16BitTo2SeparateBytes( &au8Message[ enCM_SetConfiguration_idx_NumberOfBytes ],  u16NumberOfPayloadBytes);

  /***************************************************************************
   * the config data
   **************************************************************************/

  /*==========================================================================
   * Stage 1:
   * Tuning behaviour (not audio tuner dependent):
   *------------------------------------------------------------------------*/
  tU8  u8NumOfSeekPeriods          = m_potun_Config->u8GetNumOfSeekPeriods();
  tU8  u8ScanTime                  = m_potun_Config->u8GetScanTime();
  //tU8  u8AfBestDelayTimeAfterSeek  = m_potun_Config->u8GetAfBestDelayTimeAfterSeek();
  tU8  u8TpSeekStrategy            = m_potun_Config->u8GetTpSeekStrategy();
  tU32 u32PtyFilterForSeek         = m_potun_Config->u32GetPtyFilterForSeek();
  tU16 u16AutoSeekStepDelay        = m_potun_Config->u16GetAutoSeekStepDelay();
  tU16 u16PemanentManualStepDelay  = m_potun_Config->u16GetPemanentManualStepDelay();

  tU8  u8TunerBehaviourSwitches    = m_potun_Config->u8GetTuningBehaviourSwitches();
  //tU8  u8WrongPiAcceptDelay        = m_potun_Config->u8GetWrongPiAcceptDelay();
  //tU8  u8TuningSwitches            = m_potun_Config->u8GetTuningSwitches();
  tU16 u16MuteRamp1                = m_potun_Config->u16GetMUTE_RAMP_1();
  tU16 u16DemuteRamp1              = m_potun_Config->u16GetDEMUTE_RAMP_1();
  tU16 u16MuteRamp2                = m_potun_Config->u16GetMUTE_RAMP_2();
  tU16 u16DemuteRamp2              = m_potun_Config->u16GetDEMUTE_RAMP_2();
  /*==========================================================================
   * Stage 2:
   * Behaviour of  ATstation info auto notification (not audio tuner dependent):
   *------------------------------------------------------------------------*/
  /*tU8  u8RdsLostTimeout            = m_potun_Config->u8GetRdsLostTimeout();
  tU8  u8BehaviouralSwitches       = m_potun_Config->u8GetBehaviouralSwitches();*/
  tU8  u8ATStationInfoSwitches       = m_potun_Config->u8GetATStationInfoSwitches();

  /*==========================================================================
   * Stage 3:
   * Timer Configuration (not audio tuner dependent):
   *------------------------------------------------------------------------*/

  tU8 u8TimerScanRadio					= m_potun_Config->u8GetTimerScanRadio();
  tU8 u8TimerInhibitLstSortAfterSkip	= m_potun_Config->u8GetTimerInhibitLstSortAfterSkip();
  tU8 u8TimerDelayTPstnSelAfterVol0		= m_potun_Config->u8GetTimerDelayTPstnSelAfterVol0();
  tU8 u8TimerDelayNoTPstnFound			= m_potun_Config->u8GetTimerDelayNoTPstnFound();
  tU8 u8TimerNonRDS_Detect				= m_potun_Config->u8GetTimerNonRDS_Detect();
  tU8 u8TimerPS_DelAfterRDSLost			= m_potun_Config->u8GetTimerPS_DelAfterRDSLost();
  tU8 u8TimerWrongPIAccept				= m_potun_Config->u8GetTimerWrongPIAccept();
  tU8 u8TimerScrollPSDetect				= m_potun_Config->u8GetTimerScrollPSDetect();
  tU8 u8TimerDelayTPstnSelAfterRDSLost	= m_potun_Config->u8GetTimerDelayTPstnSelAfterRDSLost();
  tU8 u8TimerTAendAfterRDSLost			= m_potun_Config->u8GetTimerTAendAfterRDSLost();
  tU8 u8TimerAM_MinTunTime				= m_potun_Config->u8GetTimerAM_MinTunTime();
  tU8 u8TimerCursorDelayAfterStnSel		= m_potun_Config->u8GetTimerCursorDelayAfterStnSel();
  tU8 u8TimerAF_InhibitAfterManTune		= m_potun_Config->u8GetTimerAF_InhibitAfterManTune();
  tU8 u8TimerAcceptPIAfterRecall		= m_potun_Config->u8GetTimerAcceptPIAfterRecall();
  tU8 u8TimerPTY_Hold					= m_potun_Config->u8GetTimerPTY_Hold();



  /*==========================================================================
   * Stage 4:
   * Parameters that influence the reception behaviour (not audio tuner dependent):
   *------------------------------------------------------------------------*/
  
  tU8  u8FieldStrengthSeekLevel_FM_RDS          = m_potun_Config->u8GetFieldStrengthSeekLevel_FM_RDS();
  tU8  u8QualitySeekLevel_FM_RDS				= m_potun_Config->u8GetQualitySeekLevel_FM_RDS();
  tU8  u8FieldStrengthSeekLevel_FM_NONRDS      = m_potun_Config->u8GetFieldStrengthSeekLevel_FM_NONRDS();
  tU8  u8QualitySeekLevel_FM_NONRDS			= m_potun_Config->u8GetQualitySeekLevel_FM_NONRDS();
  tU8  u8FieldStrengthSeekLevel_LW              = m_potun_Config->u8GetFieldStrengthSeekLevel_LW();
  tU8  u8QualitySeekLevel_LW				    = m_potun_Config->u8GetQualitySeekLevel_LW();
  tU8  u8FieldStrengthSeekLevel_MW              = m_potun_Config->u8GetFieldStrengthSeekLevel_MW();
  tU8  u8QualitySeekLevel_MW				    = m_potun_Config->u8GetQualitySeekLevel_MW();
  tU8  u8FieldStrengthSeekLevel_SW              = m_potun_Config->u8GetFieldStrengthSeekLevel_SW(); 
  tU8  u8QualitySeekLevel_SW				    = m_potun_Config->u8GetQualitySeekLevel_SW();
  tU8  u8FieldStrengthSeekLevel_TRF             = m_potun_Config->u8GetFieldStrengthSeekLevel_TRF(); 
  tU8  u8QualitySeekLevel_TRF				    = m_potun_Config->u8GetQualitySeekLevel_TRF();
  tU8  u8FieldStrengthSeekLevel_WB              = m_potun_Config->u8GetFieldStrengthSeekLevel_WB(); 
  tU8  u8QualitySeekLevel_WB				    = m_potun_Config->u8GetQualitySeekLevel_WB();
  
  tU8  u8FieldStrengthListLevel_FM_RDS          = m_potun_Config->u8GetFieldStrengthListLevel_FM_RDS();
  tU8  u8QualityListLevel_FM_RDS				= m_potun_Config->u8GetQualityListLevel_FM_RDS();
  tU8  u8FieldStrengthListLevel_FM_NONRDS      = m_potun_Config->u8GetFieldStrengthListLevel_FM_NONRDS();
  tU8  u8QualityListLevel_FM_NONRDS			= m_potun_Config->u8GetQualityListLevel_FM_NONRDS();
  tU8  u8FieldStrengthListLevel_LW              = m_potun_Config->u8GetFieldStrengthListLevel_LW();
  tU8  u8QualityListLevel_LW				    = m_potun_Config->u8GetQualityListLevel_LW();
  tU8  u8FieldStrengthListLevel_MW              = m_potun_Config->u8GetFieldStrengthListLevel_MW();
  tU8  u8QualityListLevel_MW				    = m_potun_Config->u8GetQualityListLevel_MW();
  tU8  u8FieldStrengthListLevel_SW              = m_potun_Config->u8GetFieldStrengthListLevel_SW(); 
  tU8  u8QualityListLevel_SW				    = m_potun_Config->u8GetQualityListLevel_SW();
  tU8  u8FieldStrengthListLevel_TRF             = m_potun_Config->u8GetFieldStrengthListLevel_TRF(); 
  tU8  u8QualityListLevel_TRF				    = m_potun_Config->u8GetQualityListLevel_TRF();
  tU8  u8FieldStrengthListLevel_WB              = m_potun_Config->u8GetFieldStrengthListLevel_WB(); 
  tU8  u8QualityListLevel_WB				    = m_potun_Config->u8GetQualityListLevel_WB();

  /*
  tU8  u8FieldStrengthSeekLevel_FM_RDS          = m_potun_Config->u8GetSeekLevel_FM_RDS();
  tU8  u8QualitySeekLevel_SW              = 0x0F;
  tU8  u8SeekLevel_MW              = m_potun_Config->u8GetSeekLevel_MW();
  tU8  u8SeekLevel_LW              = 0x0F;
  tU8  u8SeekLevel_TRF             = 0x0F;
  tU8  u8SeekLevel_WB              = 0x0F;
  tU8  u8SeekLevel_FM_NONRDS       = m_potun_Config->u8GetSeekLevel_FM_NONRDS();
  tU8  u8ListLevel_FM_RDS          = m_potun_Config->u8GetListLevel_FM_RDS();
  tU8  u8ListLevel_SW              = 0x0F;
  tU8  u8ListLevel_MW              = m_potun_Config->u8GetListLevel_MW();
  tU8  u8ListLevel_LW              = 0x0F;
  tU8  u8ListLevel_TRF             = 0x0F;
  tU8  u8ListLevel_WB              = 0x0F;
  tU8  u8ListLevel_FM_NONRDS       = m_potun_Config->u8GetListLevel_FM_NONRDS();
  tenLevelType enLevelType         = m_potun_Config->enGetLevelType();*/

  tU8  u8FM_SHARX                  = m_potun_Config->u8GetFM_SHARX();
  tU8  u8AM_SHARX                  = m_potun_Config->u8GetAM_SHARX();
  tU8  u8FM_HICUT                  = m_potun_Config->u8GetFM_HICUT();
  tU8  u8AM_HICUT                  = m_potun_Config->u8GetAM_HICUT();
  /*==========================================================================
   * Stage 5:
   * Area setup for CONFIG area (no CONFIG arear needed) (not audio tuner dependent):
   *------------------------------------------------------------------------*/
  tU32 u32AvailableBands           = m_potun_Config->u32GetAvailableBands();
  tU32 u32CombiBands			   = m_potun_Config->u32GetCombiBands();
  tU32 u32FM_LowFrequency          = m_potun_Config->u32GetMinFrqFM();
  tU32 u32FM_HighFrequency         = m_potun_Config->u32GetMaxFrqFM();
  tU16 u16FM_StepSizeAutoSeek      = m_potun_Config->u16GetStepSizeFM();
  tU16 u16FM_StepSizeManualStep    = m_potun_Config->u16GetStepSizeManualFM();
  tU32 u32FM_LSM                   = m_potun_Config->u32GetFM_LSM();
  tU32 u32LW_LowFrequency          = m_potun_Config->u32GetMinFrqLW();
  tU32 u32LW_HighFrequency         = m_potun_Config->u32GetMaxFrqLW();
  tU16 u16LW_StepSizeAutoSeek      = m_potun_Config->u16GetStepSizeLW();
  tU16 u16LW_StepSizeManualStep    = m_potun_Config->u16GetStepSizeManualLW();
  tU32 u32LW_LSM                   = m_potun_Config->u32GetLW_LSM();
  tU32 u32MW_LowFrequency          = m_potun_Config->u32GetMinFrqMW();
  tU32 u32MW_HighFrequency         = m_potun_Config->u32GetMaxFrqMW();
  tU16 u16MW_StepSizeAutoSeek      = m_potun_Config->u16GetStepSizeMW();
  tU16 u16MW_StepSizeManualStep    = m_potun_Config->u16GetStepSizeManualMW();
  tU32 u32MW_LSM                   = m_potun_Config->u32GetMW_LSM();
  tU8  u8special_behaviour         = TUN_INIT_UNUSED_BYTE;
  /*==========================================================================
   * Stage 6:
   * User settings (audio tuner dependent):
   *------------------------------------------------------------------------*/
  tU8  u8TP_Switch_AudioTuner1     = m_potun_Config->u8TP_Switch_AudioTuner1();
  tU8  u8TP_Switch_AudioTuner2     = m_potun_Config->u8TP_Switch_AudioTuner2();
  tU8  u8AF_Switch_AudioTuner1     = m_potun_Config->u8AF_Switch_AudioTuner1();
  tU8  u8AF_Switch_AudioTuner2     = m_potun_Config->u8AF_Switch_AudioTuner2();
  tenDDA_Switch enDDA_Switch_AudioTuner1    = m_potun_Config->enDDA_Switch_AudioTuner1();
  tenDDA_Switch enDDA_Switch_AudioTuner2    = m_potun_Config->enDDA_Switch_AudioTuner2();
  tU8  u8RDS_Switch_AudioTuner1    = m_potun_Config->u8RDS_Switch_AudioTuner1();
  /** initialise the RDS switch member variable with KDS data */
  m_potun_MsgToHMI->vSetRDSSwitchStatus(u8RDS_Switch_AudioTuner1);
  tU8  u8RDS_Switch_AudioTuner2    = m_potun_Config->u8RDS_Switch_AudioTuner1(); 
  tU8  u8PS_FIX_Switch_AudioTuner1 = m_potun_Config->u8PS_FIX_Switch_AudioTuner1();
  tU8  u8PS_FIX_Switch_AudioTuner2 = m_potun_Config->u8PS_FIX_Switch_AudioTuner2();
  tU8  u8Stereo_Switch_AudioTuner1 = m_potun_Config->u8Stereo_Switch_AudioTuner1();
  tU8  u8Stereo_Switch_AudioTuner2 = m_potun_Config->u8Stereo_Switch_AudioTuner1(); 
  tenREG_Switch enREG_Switch_AudioTuner1    = m_potun_Config->enREG_Switch_AudioTuner1();
  tenREG_Switch enREG_Switch_AudioTuner2    = m_potun_Config->enREG_Switch_AudioTuner2();
  tU8  u8PTY31_Switch_AudioTuner1 = m_potun_Config->u8PTY31_Switch_AudioTuner1();
  tU8  u8PTY31_Switch_AudioTuner2 = m_potun_Config->u8PTY31_Switch_AudioTuner2();
  tU8  u8DDS_Switch_AudioTuner1 = m_potun_Config->u8DDS_Switch_AudioTuner1();
  tU8  u8DDS_Switch_AudioTuner2 = m_potun_Config->u8DDS_Switch_AudioTuner2();
  tU8  u8DMQ_Switch_AudioTuner1 = m_potun_Config->u8DMQ_Switch_AudioTuner1();
  tU8  u8DMQ_Switch_AudioTuner2 = m_potun_Config->u8DMQ_Switch_AudioTuner2();

   /*==========================================================================
   * Stage 7:
   * HD parameter configuration:
   *------------------------------------------------------------------------*/

  tU8 u8BlendThresholdFMHybrid = m_potun_Config->u8GetBlendThresholdFMHybrid() ;
  tU16 u16DigiAudioScalingFMHybrid = m_potun_Config->u16GetDigiAudioScalingFMHybrid() ;
  tU16 u16DigiAudioDelayFMHybridPrimaryBBRate = m_potun_Config->u16GetDigiAudioDelayFMHybridPrimaryBBRate() ;
  tU8 u8BlendRateFMHybrid = m_potun_Config->u8GetBlendRateFMHybrid() ;			
  tU8 u8BlendThresholdAMHybrid = m_potun_Config->u8GetBlendThresholdAMHybrid() ;			
  tU16 u16AM_MPSAudioScaling = m_potun_Config->u16GetAM_MPSAudioScaling() ;					
  tU8 u8AM_MPSBlendRate = m_potun_Config->u8GetAM_MPSBlendRate() ;						
  tU16 u16DigiAudioDelayAM_MPSPrimaryBBRate = m_potun_Config->u16GetDigiAudioDelayAM_MPSPrimaryBBRate() ;    
  tU8 u8DigiAudioPhaseInversion = m_potun_Config->u8GetDigiAudioPhaseInversion() ;				
  tU8 u8BlendDecision = m_potun_Config->u8GetBlendDecision() ;						
  tU8 u8FM_CDNO_BlendDecision = m_potun_Config->u8GetFM_CDNO_BlendDecision() ;				
  tU8 u8AM_CDNO_BlendDecision = m_potun_Config->u8GetAM_CDNO_BlendDecision() ;	

  /*==========================================================================
   * Stage 8:
   * HF Parameter (not physical tuner dependent):
   *------------------------------------------------------------------------*/
  tU8  u8FM_Channel_Separation     = m_potun_Config->u8GetFmChannelSeparation();
  tU8  u8FM_Limiting               = m_potun_Config->u8GetFmLimiting();
  tU8  u8FM_Multipath              = m_potun_Config->u8GetFmMultipath();
  tU8  u8FM_Softmute               = m_potun_Config->u8GetFmSoftMute();
  tU8  u8AM_Softmute               = m_potun_Config->u8GetAmSoftMute();
  tU8  u8FM_Hicut_Configuration    = m_potun_Config->u8GetFmHighCut();
  tU8  u8AM_Hicut_Configuration    = m_potun_Config->u8GetAM_Hicut_Configuration();
  tU8  u8FM_Deemphasis             = m_potun_Config->u8GetFmDeemphasis();
  tU8  u8AM_Deemphasis             = m_potun_Config->u8GetAM_Deemphasis();
  tU8  u8FM_Frequency_Response     = m_potun_Config->u8GetFmFrequencyResponse();
  tU8  u8AM_Frequency_Response     = m_potun_Config->u8GetAmFrequencyResponse();
  tU8  u8FM_Miscellaneous          = m_potun_Config->u8GetFmMisc();
  tU8  u8AM_Miscellaneous          = m_potun_Config->u8GetAmMisc();
  tU8  u8AM_Antenna_Attenuation    = m_potun_Config->u8GetFmAmplificationLevel();
  tU8  u8FM_Antenna_Attenuation    = m_potun_Config->u8GetAmAmplificationLevel();
  tenFmDabHdTunerMode enPhysicalTunerConfigId = m_potun_Config->enGetPhysicalTunerConfigId();
  tU8  u8AntennaConfigId           = m_potun_Config->u8GetAntennaConfigId();
#ifdef  VARIANT_S_FTR_ENABLE_ADR_API_3_20
  tU8  u8AMFMHicutExtender           = m_potun_Config->u8GetAMFMHicutExtender();
  tU8  u8FMSoftmutehandling           = m_potun_Config->u8GetFMSoftmutehandling();
  tU8  u8AMFMMisc2           = m_potun_Config->u8GetAMFMMisc2();
#endif
  /*==========================================================================
   * Stage 9:
   * RDS Parameter
   *------------------------------------------------------------------------*/
  tU8  u8ExchangeAntennaLevel      = m_potun_Config->u8GetExchangeAntennaLevel();
  tU8  u8PitestLevel               = m_potun_Config->u8GetPitestLevel();
  tU8  u8NrdsLevel                 = m_potun_Config->u8GetNrdsLevel();
  tU8  u8EonLevel                  = m_potun_Config->u8GetEonLevel();
  tU8  u8FmMemoLevel               = m_potun_Config->u8GetFMMemoLevel();
  tU8  u8PanicFieldstLevel         = m_potun_Config->u8GetPanicFieldstLevel();
  tU8  u8ParamRdsLevel             = m_potun_Config->u8GetParamRDSLevel();
  tU8  u8RdsDataAcceptLevel        = m_potun_Config->u8GetRDSDataAcceptLevel();
  tU8  u8PiStepLevel               = m_potun_Config->u8GetPIStepLevel();
  tU8  u8DdaAntennaDistLevel       = m_potun_Config->u8GetDDAAntennaDistLevel();
  tU8  u8RdsSynDelayCnt            = m_potun_Config->u8GetRDSSynDelayCnt();
  tU8  u8MpMeasureLevel            = m_potun_Config->u8GetMpMeasureLevel();
  tU8  u8MpLevel2Level             = m_potun_Config->u8GetMpLevel2Level();
  tU8  u8MpVerybadLevel            = m_potun_Config->u8GetMpVeryBadLevel();
  tU8  u8MpMediumLevel             = m_potun_Config->u8GetMpMediumLevel();
  tU8  u8RdsBadLevel               = m_potun_Config->u8GetRDSBadLevel();
  tU8  u8IllegalAf                 = m_potun_Config->u8GetIllegalAf();
  tU8  u8AfSaveLevel               = m_potun_Config->u8GetAfSaveLevel();
  tU8  u8FirstRundum               = m_potun_Config->u8GetFirstRundum();
  tU8  u8SecondRundum              = m_potun_Config->u8GetSecondRundum();
  tU8  u8ThirdRundum               = m_potun_Config->u8GetThirdRundum();
  tU8  u8PIStepMode                = m_potun_Config->u8PIStepMode();
  tU8  u8AudioOffScanMode          = m_potun_Config->u8GetAudioOffScanMode();
  tU8  u8AMLearnCycle              = m_potun_Config->u8GetAMLearnCycle();
  tU8  u8PIIgnore                  = m_potun_Config->u8GetPIIgnore();
  /** Free1 Byte */
  tU8  u8PISearchTime              = m_potun_Config->u8GetPISearchTime();
  /** Free2 Byte */
  tU8  u8DisableSeamlessBlending   = m_potun_Config->u8GetDisableSeamlessBlending();
  tU8  u8SlAdjThreshold            = m_potun_Config->u8GetSlAdjThreshold();
  tU8  u8SlMpThreshold             = m_potun_Config->u8GetSlMpThreshold();
  tU8  u8SpecialFmStartFrequency   = m_potun_Config->u8GetSpecialFMStartFrequency();
  tU8  u8HDScanEveryFreq		   = m_potun_Config->u8GetHDScanEveryFrequency();
  tU8  u8HWSubConfig			   = m_potun_Config->u8GetHWSubConfig();
  /*==========================================================================
   * Stage 10:
   * Testmode and AF-List configuration:
   *------------------------------------------------------------------------*/
  tU8  u8TestModeDataUpdateInterval             = m_potun_Config->u8GetTestModeDataUpdateInterval();
  tU8  u8AFListUpdateInterval   = m_potun_Config->u8GetAFListUpdateInterval();
  tU8  u8AFListReleaseBehaviour   = m_potun_Config->u8GetAFListReleaseBehaviour();

  /*========================================================================*/
  /*==========================================================================
   * Stage 11:
   * Station list configuration:
   *------------------------------------------------------------------------*/
  tU8  u8MaxNumberofStationsInStnList             = m_potun_Config->u8GetMaxNumberofStationsInStnList();
 
  /*========================================================================*/
  /*==========================================================================
   * Stage 12:
   * Preset list configuration (not audio tuner dependent):
   *------------------------------------------------------------------------*/
   /* see below */
  /*========================================================================*/


  /***************************************************************************
   * fill the message buffer with the config data
   **************************************************************************/

  /*==========================================================================
   * Stage 1:
   * Tuning behaviour (not audio tuner dependent):
   *------------------------------------------------------------------------*/
                                                   au8Message[ enCM_SetConfiguration_idx_NumOfSeekPeriods         ] = u8NumOfSeekPeriods;
                                                   au8Message[ enCM_SetConfiguration_idx_ScanTime                 ] = u8ScanTime;
								OSAL_pvMemorySet( &au8Message[ enCM_SetConfiguration_idx_reserved0                ], TUN_INIT_UNUSED_BYTE, (tU8)enCM_SetConfiguration_len_reserved0);
												   au8Message[ enCM_SetConfiguration_idx_TpSeekStrategy           ] = u8TpSeekStrategy;
  m_potun_Utility->vConvert32BitTo4SeparateBytes( &au8Message[ enCM_SetConfiguration_idx_PtyFilterForSeek         ],  u32PtyFilterForSeek);
  m_potun_Utility->vConvert16BitTo2SeparateBytes( &au8Message[ enCM_SetConfiguration_idx_AutoSeekStepDelay        ],  u16AutoSeekStepDelay);
  m_potun_Utility->vConvert16BitTo2SeparateBytes( &au8Message[ enCM_SetConfiguration_idx_PemanentManualStepDelay  ],  u16PemanentManualStepDelay);
								OSAL_pvMemorySet( &au8Message[ enCM_SetConfiguration_idx_reserved1                ], TUN_INIT_UNUSED_BYTE, (tU8)enCM_SetConfiguration_len_reserved1);
												   au8Message[ enCM_SetConfiguration_idx_TuningBehaviourSwitches  ] = u8TunerBehaviourSwitches;
  m_potun_Utility->vConvert16BitTo2SeparateBytes( &au8Message[ enCM_SetConfiguration_idx_MUTE_RAMP_1              ],  u16MuteRamp1);
  m_potun_Utility->vConvert16BitTo2SeparateBytes( &au8Message[ enCM_SetConfiguration_idx_DEMUTE_RAMP_1            ],  u16DemuteRamp1);
  m_potun_Utility->vConvert16BitTo2SeparateBytes( &au8Message[ enCM_SetConfiguration_idx_MUTE_RAMP_2              ],  u16MuteRamp2);
  m_potun_Utility->vConvert16BitTo2SeparateBytes( &au8Message[ enCM_SetConfiguration_idx_DEMUTE_RAMP_2            ],  u16DemuteRamp2);
								OSAL_pvMemorySet( &au8Message[ enCM_SetConfiguration_idx_reserved2                ], TUN_INIT_UNUSED_BYTE, (tU8)enCM_SetConfiguration_len_reserved2);
  
  /*==========================================================================
   * Stage 2:
   * ATStation info Configuration (not audio tuner dependent):
   *------------------------------------------------------------------------*/
                                OSAL_pvMemorySet( &au8Message[ enCM_SetConfiguration_idx_reserved3                ], TUN_INIT_UNUSED_BYTE, (tU8)enCM_SetConfiguration_len_reserved3);
                                                   au8Message[ enCM_SetConfiguration_idx_ATStationInfoBits        ] = u8ATStationInfoSwitches;
                                OSAL_pvMemorySet( &au8Message[ enCM_SetConfiguration_idx_reserved4                ], TUN_INIT_UNUSED_BYTE, (tU8)enCM_SetConfiguration_len_reserved4);


  /*==========================================================================
   * Stage 3:
   * Timer Configuration:
   *------------------------------------------------------------------------*/
							au8Message[ enCM_SetConfiguration_idx_t_SCAN_radio									] = u8TimerScanRadio;
                            au8Message[ enCM_SetConfiguration_idx_t_inhibit_list_sort_after_skip				] = u8TimerInhibitLstSortAfterSkip;
                            au8Message[ enCM_SetConfiguration_idx_t_delay_TP_station_select_after_Vol_zero      ] = u8TimerDelayTPstnSelAfterVol0;
                            au8Message[ enCM_SetConfiguration_idx_t_delay_no_TP_station_found					] = u8TimerDelayNoTPstnFound;
                            au8Message[ enCM_SetConfiguration_idx_t_non_RDS_detect								] = u8TimerNonRDS_Detect;
                            au8Message[ enCM_SetConfiguration_idx_t_PS_delete_after_RDS_lost					] = u8TimerPS_DelAfterRDSLost;
                            au8Message[ enCM_SetConfiguration_idx_t_wrongPI_accept								] = u8TimerWrongPIAccept;
                            au8Message[ enCM_SetConfiguration_idx_t_scroll_PS_detect							] = u8TimerScrollPSDetect;
                            au8Message[ enCM_SetConfiguration_idx_t_delay_TP_station_select_after_RDS_lost      ] = u8TimerDelayTPstnSelAfterRDSLost;
                            au8Message[ enCM_SetConfiguration_idx_t_TA_end_after_RDS_lost						] = u8TimerTAendAfterRDSLost;
                            au8Message[ enCM_SetConfiguration_idx_t_AM_min_tun_time								] = u8TimerAM_MinTunTime;
                            au8Message[ enCM_SetConfiguration_idx_t_cursor_delay_after_station_select           ] = u8TimerCursorDelayAfterStnSel;
                            au8Message[ enCM_SetConfiguration_idx_t_AF_inhibit_after_man_tune					] = u8TimerAF_InhibitAfterManTune;
                            au8Message[ enCM_SetConfiguration_idx_t_accept_PI_after_recall						] = u8TimerAcceptPIAfterRecall;			
							au8Message[ enCM_SetConfiguration_idx_t_PTY_hold									] = u8TimerPTY_Hold;
								OSAL_pvMemorySet( &au8Message[ enCM_SetConfiguration_idx_reserved5              ], TUN_INIT_UNUSED_BYTE, (tU8)enCM_SetConfiguration_idx_reserved5);

  /*==========================================================================
   * Stage 4:
   * Parameters that influence the reception behaviour (not audio tuner dependent):
   *------------------------------------------------------------------------*/
							au8Message[ enCM_SetConfiguration_idx_FieldstrengthSeekLevel_FM_RDS					] = u8FieldStrengthSeekLevel_FM_RDS;
							au8Message[ enCM_SetConfiguration_idx_QualitySeekLevel_FM_RDS						] = u8QualitySeekLevel_FM_RDS;
                            au8Message[ enCM_SetConfiguration_idx_FieldstrengthSeekLevel_FM_NONRDS				] = u8FieldStrengthSeekLevel_FM_NONRDS;
                            au8Message[ enCM_SetConfiguration_idx_QualitySeekLevel_FM_NONRDS					] = u8QualitySeekLevel_FM_NONRDS;
                            au8Message[ enCM_SetConfiguration_idx_FieldstrengthSeekLevel_LW						] = u8FieldStrengthSeekLevel_LW;
                            au8Message[ enCM_SetConfiguration_idx_QualitySeekLevel_LW							] = u8QualitySeekLevel_LW;
                            au8Message[ enCM_SetConfiguration_idx_FieldstrengthSeekLevel_MW						] = u8FieldStrengthSeekLevel_MW;
                            au8Message[ enCM_SetConfiguration_idx_QualitySeekLevel_MW							] = u8QualitySeekLevel_MW;
                            au8Message[ enCM_SetConfiguration_idx_FieldstrengthSeekLevel_SW						] = u8FieldStrengthSeekLevel_SW;
                            au8Message[ enCM_SetConfiguration_idx_QualitySeekLevel_SW							] = u8QualitySeekLevel_SW;
                            au8Message[ enCM_SetConfiguration_idx_FieldstrengthSeekLevel_TRF					] = u8FieldStrengthSeekLevel_TRF;
                            au8Message[ enCM_SetConfiguration_idx_QualitySeekLevel_TRF							] = u8QualitySeekLevel_TRF;
                            au8Message[ enCM_SetConfiguration_idx_FieldstrengthSeekLevel_WB						] = u8FieldStrengthSeekLevel_WB;
                            au8Message[ enCM_SetConfiguration_idx_QualitySeekLevel_WB							] = u8QualitySeekLevel_WB;
												   
							au8Message[ enCM_SetConfiguration_idx_FieldstrengthListLevel_FM_RDS					] = u8FieldStrengthListLevel_FM_RDS;
                            au8Message[ enCM_SetConfiguration_idx_QualityListLevel_FM_RDS						] = u8QualityListLevel_FM_RDS;
                            au8Message[ enCM_SetConfiguration_idx_FieldstrengthListLevel_FM_NONRDS		        ] = u8FieldStrengthListLevel_FM_NONRDS;
                            au8Message[ enCM_SetConfiguration_idx_QualityListLevel_FM_NONRDS					] = u8QualityListLevel_FM_NONRDS;
                            au8Message[ enCM_SetConfiguration_idx_FieldstrengthListLevel_LW						] = u8FieldStrengthListLevel_LW;
                            au8Message[ enCM_SetConfiguration_idx_QualityListLevel_LW							] = u8QualityListLevel_LW;
                            au8Message[ enCM_SetConfiguration_idx_FieldstrengthListLevel_MW						] = u8FieldStrengthListLevel_MW;
                            au8Message[ enCM_SetConfiguration_idx_QualityListLevel_MW							] = u8QualityListLevel_MW;
                            au8Message[ enCM_SetConfiguration_idx_FieldstrengthListLevel_SW						] = u8FieldStrengthListLevel_SW;
                            au8Message[ enCM_SetConfiguration_idx_QualityListLevel_SW							] = u8QualityListLevel_SW;
                            au8Message[ enCM_SetConfiguration_idx_FieldstrengthListLevel_TRF					] = u8FieldStrengthListLevel_TRF;
                            au8Message[ enCM_SetConfiguration_idx_QualityListLevel_TRF							] = u8QualityListLevel_TRF;
                            au8Message[ enCM_SetConfiguration_idx_FieldstrengthListLevel_WB						] = u8FieldStrengthListLevel_WB;
                            au8Message[ enCM_SetConfiguration_idx_QualityListLevel_WB							] = u8QualityListLevel_WB;
                            
												 au8Message[ enCM_SetConfiguration_idx_FM_SHARX                 ] = u8FM_SHARX;
												 au8Message[ enCM_SetConfiguration_idx_AM_SHARX                 ] = u8AM_SHARX;
												 au8Message[ enCM_SetConfiguration_idx_FM_HICUT                 ] = u8FM_HICUT;
												 au8Message[ enCM_SetConfiguration_idx_AM_HICUT                 ] = u8AM_HICUT;
							  OSAL_pvMemorySet( &au8Message[ enCM_SetConfiguration_idx_reserved6                ], TUN_INIT_UNUSED_BYTE, (tU8)enCM_SetConfiguration_len_reserved6);
  /*==========================================================================
   * Stage 5:
   * Area setup for CONFIG area (no CONFIG arear needed) (not audio tuner dependent):
   *------------------------------------------------------------------------*/
  m_potun_Utility->vConvert32BitTo4SeparateBytes( &au8Message[ enCM_SetConfiguration_idx_AvailableBands           ], u32AvailableBands);
  m_potun_Utility->vConvert32BitTo4SeparateBytes( &au8Message[ enCM_SetConfiguration_idx_CombiBandConfig          ], u32CombiBands);
  /** FM Band Config */
  m_potun_Utility->vConvert32BitTo4SeparateBytes( &au8Message[ enCM_SetConfiguration_idx_FM_LowFrequency          ], u32FM_LowFrequency);
  m_potun_Utility->vConvert32BitTo4SeparateBytes( &au8Message[ enCM_SetConfiguration_idx_FM_HighFrequency         ], u32FM_HighFrequency);
  m_potun_Utility->vConvert16BitTo2SeparateBytes( &au8Message[ enCM_SetConfiguration_idx_FM_StepSizeAutoSeek      ], u16FM_StepSizeAutoSeek);
  m_potun_Utility->vConvert16BitTo2SeparateBytes( &au8Message[ enCM_SetConfiguration_idx_FM_StepSizeManualStep    ], u16FM_StepSizeManualStep);
  m_potun_Utility->vConvert32BitTo4SeparateBytes( &au8Message[ enCM_SetConfiguration_idx_FM_LSM                   ], u32FM_LSM);
  /** LW Band Config */
  m_potun_Utility->vConvert32BitTo4SeparateBytes( &au8Message[ enCM_SetConfiguration_idx_LW_LowFrequency          ], u32LW_LowFrequency);
  m_potun_Utility->vConvert32BitTo4SeparateBytes( &au8Message[ enCM_SetConfiguration_idx_LW_HighFrequency         ], u32LW_HighFrequency);
  m_potun_Utility->vConvert16BitTo2SeparateBytes( &au8Message[ enCM_SetConfiguration_idx_LW_StepSizeAutoSeek      ], u16LW_StepSizeAutoSeek);
  m_potun_Utility->vConvert16BitTo2SeparateBytes( &au8Message[ enCM_SetConfiguration_idx_LW_StepSizeManualStep    ], u16LW_StepSizeManualStep);
  m_potun_Utility->vConvert32BitTo4SeparateBytes( &au8Message[ enCM_SetConfiguration_idx_LW_LSM                   ], u32LW_LSM);
  /** MW Band Config */                                
  m_potun_Utility->vConvert32BitTo4SeparateBytes( &au8Message[ enCM_SetConfiguration_idx_MW_LowFrequency          ], u32MW_LowFrequency);
  m_potun_Utility->vConvert32BitTo4SeparateBytes( &au8Message[ enCM_SetConfiguration_idx_MW_HighFrequency         ], u32MW_HighFrequency);
  m_potun_Utility->vConvert16BitTo2SeparateBytes( &au8Message[ enCM_SetConfiguration_idx_MW_StepSizeAutoSeek      ], u16MW_StepSizeAutoSeek);
  m_potun_Utility->vConvert16BitTo2SeparateBytes( &au8Message[ enCM_SetConfiguration_idx_MW_StepSizeManualStep    ], u16MW_StepSizeManualStep);
  m_potun_Utility->vConvert32BitTo4SeparateBytes( &au8Message[ enCM_SetConfiguration_idx_MW_LSM                   ], u32MW_LSM);

                                OSAL_pvMemorySet( &au8Message[ enCM_SetConfiguration_idx_other_band_configs       ], TUN_INIT_UNUSED_BYTE, (tU16)enCM_SetConfiguration_len_other_band_configs);
                                OSAL_pvMemorySet( &au8Message[ enCM_SetConfiguration_idx_reserved7				  ], TUN_INIT_UNUSED_BYTE, (tU8)enCM_SetConfiguration_len_reserved7);
                                
  /*==========================================================================
   * Stage 6:
   * User settings (audio tuner dependent):
   *------------------------------------------------------------------------*/
                                                   au8Message[ enCM_SetConfiguration_idx_TP_Switch_AudioTuner1		] = u8TP_Switch_AudioTuner1;
                                                   au8Message[ enCM_SetConfiguration_idx_TP_Switch_AudioTuner2		] = u8TP_Switch_AudioTuner2;
                                                   au8Message[ enCM_SetConfiguration_idx_AF_Switch_AudioTuner1		] = u8AF_Switch_AudioTuner1;
                                                   au8Message[ enCM_SetConfiguration_idx_AF_Switch_AudioTuner2		] = u8AF_Switch_AudioTuner2;
                                                   au8Message[ enCM_SetConfiguration_idx_DDA_Switch_AudioTuner1		] = (tU8)enDDA_Switch_AudioTuner1;
                                                   au8Message[ enCM_SetConfiguration_idx_DDA_Switch_AudioTuner2		] = (tU8)enDDA_Switch_AudioTuner2;
                                                   au8Message[ enCM_SetConfiguration_idx_RDS_Switch_AudioTuner1		] = u8RDS_Switch_AudioTuner1;
                                                   au8Message[ enCM_SetConfiguration_idx_RDS_Switch_AudioTuner2		] = u8RDS_Switch_AudioTuner2;
                                                   au8Message[ enCM_SetConfiguration_idx_PS_FIX_Switch_AudioTuner1	] = u8PS_FIX_Switch_AudioTuner1;
                                                   au8Message[ enCM_SetConfiguration_idx_PS_FIX_Switch_AudioTuner2	] = u8PS_FIX_Switch_AudioTuner2;
                                                   au8Message[ enCM_SetConfiguration_idx_Stereo_Switch_AudioTuner1	] = u8Stereo_Switch_AudioTuner1;
                                                   au8Message[ enCM_SetConfiguration_idx_Stereo_Switch_AudioTuner2	] = u8Stereo_Switch_AudioTuner2;
                                                   au8Message[ enCM_SetConfiguration_idx_REG_Switch_AudioTuner1		] = (tU8)enREG_Switch_AudioTuner1;
                                                   au8Message[ enCM_SetConfiguration_idx_REG_Switch_AudioTuner2		] = (tU8)enREG_Switch_AudioTuner2;
                                                   au8Message[ enCM_SetConfiguration_idx_PTY31_Switch_AudioTuner1	] = u8PTY31_Switch_AudioTuner1;
                                                   au8Message[ enCM_SetConfiguration_idx_PTY31_Switch_AudioTuner2   ] = u8PTY31_Switch_AudioTuner2;
                                                   au8Message[ enCM_SetConfiguration_idx_DDS_Switch_AudioTuner1		] = u8DDS_Switch_AudioTuner1;
                                                   au8Message[ enCM_SetConfiguration_idx_DDS_Switch_AudioTuner2		] = u8DDS_Switch_AudioTuner2;
                                                   au8Message[ enCM_SetConfiguration_idx_DMQ_Switch_AudioTuner1		] = u8DMQ_Switch_AudioTuner1;
                                                   au8Message[ enCM_SetConfiguration_idx_DMQ_Switch_AudioTuner2		] = u8DMQ_Switch_AudioTuner2;

												   if(m_potun_Config -> bIsHDSupported())
												   {
									OSAL_pvMemorySet( &au8Message[ enCM_SetConfiguration_idx_reserved8              ], TUN_INIT_UNUSED_BYTE, (tU16)enCM_SetConfiguration_len_reserved8);

   /*==========================================================================
   * Stage 7:
   * HD parameter configuration:
   *------------------------------------------------------------------------*/
												   au8Message[ enCM_SetConfiguration_idx_BlendThresholdFMHybrid				] = u8BlendThresholdFMHybrid;
  m_potun_Utility->vConvert16BitTo2SeparateBytes( &au8Message[ enCM_SetConfiguration_idx_DigiAudioScalingFMHybrid			], u16DigiAudioScalingFMHybrid);
  m_potun_Utility->vConvert16BitTo2SeparateBytes( &au8Message[ enCM_SetConfiguration_idx_DigiAudioDelayFMHybridPrimaryBBRate], u16DigiAudioDelayFMHybridPrimaryBBRate);
                                                   au8Message[ enCM_SetConfiguration_idx_BlendRateFMHybrid					] = u8BlendRateFMHybrid;
                                                   au8Message[ enCM_SetConfiguration_idx_u8BlendThresholdAMHybrid			] = u8BlendThresholdAMHybrid;
  m_potun_Utility->vConvert16BitTo2SeparateBytes( &au8Message[ enCM_SetConfiguration_idx_u16AM_MPSAudioScaling				], u16AM_MPSAudioScaling);
                                                   au8Message[ enCM_SetConfiguration_idx_AM_MPSBlendRate					] = u8AM_MPSBlendRate;
  m_potun_Utility->vConvert16BitTo2SeparateBytes( &au8Message[ enCM_SetConfiguration_idx_DigiAudioDelayAM_MPSPrimaryBBRate	], u16DigiAudioDelayAM_MPSPrimaryBBRate);
                                                   au8Message[ enCM_SetConfiguration_idx_DigiAudioPhaseInversion			] = u8DigiAudioPhaseInversion;
                                                   au8Message[ enCM_SetConfiguration_idx_BlendDecision						] = u8BlendDecision;
                                                   au8Message[ enCM_SetConfiguration_idx_FM_CDNO_BlendDecision				] = u8FM_CDNO_BlendDecision;
                                                   au8Message[ enCM_SetConfiguration_idx_AM_CDNO_BlendDecision				] = u8AM_CDNO_BlendDecision;

									OSAL_pvMemorySet( &au8Message[ enCM_SetConfiguration_idx_reserved15						], TUN_INIT_UNUSED_BYTE, (tU16)enCM_SetConfiguration_len_reserved15);

												   }
												   else
												   {
													   /** adding 20 bytes of uncoonfigured HD parameters to length "enCM_SetConfiguration_len_reserved8" */
									OSAL_pvMemorySet( &au8Message[ enCM_SetConfiguration_idx_reserved8              ], TUN_INIT_UNUSED_BYTE, (tU16)enCM_SetConfiguration_len_reserved8 +20);
												   }
  /*==========================================================================
   * Stage 8:
   * HF parameter configuration:
   *------------------------------------------------------------------------*/
								
                                                   au8Message[ enCM_SetConfiguration_idx_FM_Channel_Separation    ] = u8FM_Channel_Separation;
                                                   au8Message[ enCM_SetConfiguration_idx_FM_Limiting              ] = u8FM_Limiting;
                                                   au8Message[ enCM_SetConfiguration_idx_FM_Multipath             ] = u8FM_Multipath;
                                                   au8Message[ enCM_SetConfiguration_idx_FM_Softmute              ] = u8FM_Softmute;
                                                   au8Message[ enCM_SetConfiguration_idx_AM_Softmute              ] = u8AM_Softmute;
                                                   au8Message[ enCM_SetConfiguration_idx_FM_Hicut_Configuration   ] = u8FM_Hicut_Configuration;
                                                   au8Message[ enCM_SetConfiguration_idx_AM_Hicut_Configuration   ] = u8AM_Hicut_Configuration;
                                                   au8Message[ enCM_SetConfiguration_idx_FM_Deemphasis            ] = u8FM_Deemphasis;
                                                   au8Message[ enCM_SetConfiguration_idx_AM_Deemphasis            ] = u8AM_Deemphasis;
                                                   au8Message[ enCM_SetConfiguration_idx_FM_Frequency_Response    ] = u8FM_Frequency_Response;
                                                   au8Message[ enCM_SetConfiguration_idx_AM_Frequency_Response    ] = u8AM_Frequency_Response;
                                                   au8Message[ enCM_SetConfiguration_idx_FM_Miscellaneous         ] = u8FM_Miscellaneous;
                                                   au8Message[ enCM_SetConfiguration_idx_AM_Miscellaneous         ] = u8AM_Miscellaneous;
                                                   au8Message[ enCM_SetConfiguration_idx_AM_Antenna_Attenuation   ] = u8AM_Antenna_Attenuation;
                                                   au8Message[ enCM_SetConfiguration_idx_FM_Antenna_Attenuation   ] = u8FM_Antenna_Attenuation;
                                                   au8Message[ enCM_SetConfiguration_idx_PhysicalTunerConfigId    ] = (tU8)enPhysicalTunerConfigId;
                                                   au8Message[ enCM_SetConfiguration_idx_AntennaConfigId          ] = u8AntennaConfigId;
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
                                                   au8Message[ enCM_SetConfiguration_idx_AMFM_Hicut_Extender          ] = u8AMFMHicutExtender;
                                                   au8Message[ enCM_SetConfiguration_idx_FM_Softmute_Timing          ] = u8FMSoftmutehandling;
                                                   au8Message[ enCM_SetConfiguration_idx_AMFM_Misc2          ] = u8AMFMMisc2;
#else
                                OSAL_pvMemorySet( &au8Message[ enCM_SetConfiguration_idx_reserved9                ], TUN_INIT_UNUSED_BYTE, (tU8)enCM_SetConfiguration_len_reserved9);
#endif
  /*==========================================================================
   * Stage 9:
   * RDS Parameter
   *------------------------------------------------------------------------*/
                                                   au8Message[ enCM_SetConfiguration_idx_EXCHANGE_ANTENNA_LEVEL		] = u8ExchangeAntennaLevel;
                                                   au8Message[ enCM_SetConfiguration_idx_PITEST_LEVEL				] = u8PitestLevel;
                                                   au8Message[ enCM_SetConfiguration_idx_NRDS_LEVEL					] = u8NrdsLevel;
                                                   au8Message[ enCM_SetConfiguration_idx_EON_LEVEL					] = u8EonLevel;
                                                   au8Message[ enCM_SetConfiguration_idx_FM_MEMO_LEVEL				] = u8FmMemoLevel;
                                                   au8Message[ enCM_SetConfiguration_idx_PANIC_FIELDST_LEVEL		] = u8PanicFieldstLevel;
                                                   au8Message[ enCM_SetConfiguration_idx_PARAM_RDS_LEVEL			] = u8ParamRdsLevel;
                                                   au8Message[ enCM_SetConfiguration_idx_RDS_DATA_ACCEPT_LEVEL		] = u8RdsDataAcceptLevel;
                                                   au8Message[ enCM_SetConfiguration_idx_PI_STEP_LEVEL				] = u8PiStepLevel;
                                                   au8Message[ enCM_SetConfiguration_idx_DDA_ANTENNA_DIST_LEVEL		] = u8DdaAntennaDistLevel;
                                                   au8Message[ enCM_SetConfiguration_idx_RDS_SYN_DELAY_CNT			] = u8RdsSynDelayCnt;
                                                   au8Message[ enCM_SetConfiguration_idx_MP_MEASURE_LEVEL			] = u8MpMeasureLevel;
                                                   au8Message[ enCM_SetConfiguration_idx_MP_LEVEL2_LEVEL			] = u8MpLevel2Level;
                                                   au8Message[ enCM_SetConfiguration_idx_MP_VERYBAD_LEVEL			] = u8MpVerybadLevel;
                                                   au8Message[ enCM_SetConfiguration_idx_MP_MEDIUM_LEVEL			] = u8MpMediumLevel;
                                                   au8Message[ enCM_SetConfiguration_idx_RDS_BAD_LEVEL				] = u8RdsBadLevel;
                                                   au8Message[ enCM_SetConfiguration_idx_ILLEGAL_AF					] = u8IllegalAf;
                                                   au8Message[ enCM_SetConfiguration_idx_AF_SAVE_LEVEL				] = u8AfSaveLevel;
                                                   au8Message[ enCM_SetConfiguration_idx_FIRST_RUNDUM				] = u8FirstRundum;
                                                   au8Message[ enCM_SetConfiguration_idx_SECOND_RUNDUM				] = u8SecondRundum;
                                                   au8Message[ enCM_SetConfiguration_idx_THIRD_RUNDUM				] = u8ThirdRundum;
                                                   au8Message[ enCM_SetConfiguration_idx_PI_STEP_MODE				] = u8PIStepMode;
                                                   au8Message[ enCM_SetConfiguration_idx_AUDIO_OFF_SCANMODE			] = u8AudioOffScanMode;
                                                   au8Message[ enCM_SetConfiguration_idx_AM_LEARN_CYCLE				] = u8AMLearnCycle;
                                                   au8Message[ enCM_SetConfiguration_idx_PI_IGNORE					] = u8PIIgnore;
                                                   au8Message[ enCM_SetConfiguration_idx_PI_SEARCH_TIME             ] = u8PISearchTime;
                                                   au8Message[ enCM_SetConfiguration_idx_DISABLE_SEAMLESS_BLENDING  ] = u8DisableSeamlessBlending;
                                                   au8Message[ enCM_SetConfiguration_idx_SL_ADJ_THRESHOLD			] = u8SlAdjThreshold;
                                                   au8Message[ enCM_SetConfiguration_idx_SL_MP_THRESHOLD			] = u8SlMpThreshold;
                                                   au8Message[ enCM_SetConfiguration_idx_SPECIAL_FM_START_FREQUENCY	] = u8SpecialFmStartFrequency;
												   if(m_potun_Config -> bIsHDSupported())
												   {
   								OSAL_pvMemorySet( &au8Message[ enCM_SetConfiguration_idx_reserved14					], TUN_INIT_UNUSED_BYTE, 1);
												   au8Message[ enCM_SetConfiguration_idx_HD_Scan_Every_Freq			] = u8HDScanEveryFreq;
												   }
												   else
												   {
   								OSAL_pvMemorySet( &au8Message[ enCM_SetConfiguration_idx_reserved14					], TUN_INIT_UNUSED_BYTE, (tU8)enCM_SetConfiguration_len_reserved14);
												   }
												   au8Message[ enCM_SetConfiguration_idx_HW_SubConfig				] = u8HWSubConfig;
                                OSAL_pvMemorySet( &au8Message[ enCM_SetConfiguration_idx_reserved10					], TUN_INIT_UNUSED_BYTE, (tU8)enCM_SetConfiguration_len_reserved10);

							
  /*==========================================================================
   * Stage 10:
   * TestMode and AF-List configuration 
   *------------------------------------------------------------------------*/
								OSAL_pvMemorySet( &au8Message[ enCM_SetConfiguration_idx_reserved11							], TUN_INIT_UNUSED_BYTE, (tU8)enCM_SetConfiguration_len_reserved11);
												   au8Message[ enCM_SetConfiguration_idx_TestModeDataUpdateInterval			] = u8TestModeDataUpdateInterval;
                                                   au8Message[ enCM_SetConfiguration_idx_AFListUpdateInterval				] = u8AFListUpdateInterval;
                                                   au8Message[ enCM_SetConfiguration_idx_AFListReleaseBehaviour				] = u8AFListReleaseBehaviour;
                                OSAL_pvMemorySet( &au8Message[ enCM_SetConfiguration_idx_reserved12							], TUN_INIT_UNUSED_BYTE, (tU8)enCM_SetConfiguration_len_reserved12);
								                  
  /*==========================================================================
   * Stage 11:
   * Station list configuration:
   *------------------------------------------------------------------------*/
  
												   au8Message[ enCM_SetConfiguration_idx_MaxNumberofStationsInStnList		] = u8MaxNumberofStationsInStnList;
								OSAL_pvMemorySet( &au8Message[ enCM_SetConfiguration_idx_reserved13							], TUN_INIT_UNUSED_BYTE, (tU16)enCM_SetConfiguration_idx_reserved13);
								
                                                 
   /*==========================================================================
   * Stage 12:
   * Preset list configuration:
   *------------------------------------------------------------------------*/
	
  {
    tUInt unBankNo;
    tUInt unPresetNo = 0;
    tUInt unPresetLoop;
    tUInt unIndex = (tU16)enCM_SetConfiguration_idx_PresetList;
    /* FM lists */

    /*==========================================================================
     * FM preset lists
     *------------------------------------------------------------------------*/
    for( unBankNo = 0; unBankNo < m_potun_Config->u8GetNoFMpresetBanks(); unBankNo++)
    {

      au8Message[ unIndex++] = (tU8)enBand_FM;
	  
      au8Message[ unIndex++] = m_potun_Config->u8GetNoOfFMpresetPerBank();
      

      for( unPresetLoop = 0; unPresetLoop < m_potun_Config->u8GetNoOfFMpresetPerBank(); unPresetLoop++, unPresetNo++)
      {
        m_potun_Utility->vConvert32BitTo4SeparateBytes( &au8Message[ unIndex], m_potun_Config->u32GetPreset( (tU32)unPresetNo));
        unIndex += 4;
      }
    }
    /*==========================================================================
     * AM preset lists
     *------------------------------------------------------------------------*/
    for( unBankNo = 0; unBankNo < m_potun_Config->u8GetNoAMpresetBanks1(); unBankNo++)
    {
      au8Message[ unIndex++] = (tU8)enBand_MW;
	  
      au8Message[ unIndex++] = m_potun_Config->u8GetNoOfAMpresetPerBank1();
      

      for( unPresetLoop = 0; unPresetLoop < m_potun_Config->u8GetNoOfAMpresetPerBank1(); unPresetLoop++, unPresetNo++)
      {
        m_potun_Utility->vConvert32BitTo4SeparateBytes( &au8Message[ unIndex], m_potun_Config->u32GetPreset( (tU32)unPresetNo));
        unIndex += 4;
      }
    }
	for( unBankNo = 0; unBankNo < m_potun_Config->u8GetNoMIXEDpresetBanks(); unBankNo++)
    {
      au8Message[ unIndex++] = (tU8)enAvailBand_GetCurrent;

      au8Message[ unIndex++] = m_potun_Config->u8GetNoOfMIXEDpresetPerBank();


      for( unPresetLoop = 0; unPresetLoop < m_potun_Config->u8GetNoOfMIXEDpresetPerBank(); unPresetLoop++, unPresetNo++)
      {
        m_potun_Utility->vConvert32BitTo4SeparateBytes( &au8Message[ unIndex], m_potun_Config->u32GetPreset( (tU32)unPresetNo));
        unIndex += 4;
      }
    }
	/*DRM Preset List*/
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
	for( unBankNo = 0; unBankNo < m_potun_Config->u8GetNoDRMpresetBanks(); unBankNo++)
    {
      au8Message[ unIndex++] = (tU8)enBand_MW;

      au8Message[ unIndex++] = m_potun_Config->u8GetNoOfDRMpresetPerBank();
      /*Default DRM Preset frequency is not yet defined, currently value 0 is packed for all the DRM Presets*/
      for( unPresetLoop = 0; unPresetLoop < m_potun_Config->u8GetNoOfDRMpresetPerBank(); unPresetLoop++, unPresetNo++)
      {
        //DRM Preset related parameters are not yet defined in GenericTunerParameterset of KDS

    	m_potun_Utility->vConvert32BitTo4SeparateBytes( &au8Message[ unIndex], m_potun_Config->u32GetPreset( (tU32)unPresetNo));
        unIndex += 4;
      }
    }
#endif
    /*==========================================================================
     * End of List
     *------------------------------------------------------------------------*/
      au8Message[ unIndex++] = 0x00;
      
      au8Message[ unIndex] = 0x00;
    /*========================================================================*/
  }
  /*========================================================================*/



  /***************************************************************************
   * trace out the data
   **************************************************************************/

  if( etg_bIsTraceActiveShort(((0xFFFFu & (tU16)TUN_TRACE_CLASS_MSGTOADR) << 16) | (tU16)TR_LEVEL_USER_2) == TRUE)
  {
    /***************************************************************************
     * print plain message
     **************************************************************************/
    if( ( (tU8)enTraceType & (tU8)enTraceSetConfiguration_TraceOnlyPlain) != 0)
    {
      ETG_TRACE_USR2(( " BEGIN ------------------------------------------------------------------------------------"));
      /*==========================================================================
       * Stage 1:
       * Tuning behaviour (not audio tuner dependent):
       *------------------------------------------------------------------------*/
      ETG_TRACE_USR2(( " > Stage 1: Tuning behaviour (not audio tuner dependent):"                                  ));
      ETG_TRACE_USR2(( " u8NumOfSeekPeriods          : %d s"                      ,u8NumOfSeekPeriods               ));
      ETG_TRACE_USR2(( " u8ScanTime                  : %d s"                      ,u8ScanTime                       ));
      ETG_TRACE_USR2(( " u8TpSeekStrategy            : %d"                        ,u8TpSeekStrategy                 ));
      ETG_TRACE_USR2(( " u32PtyFilterForSeek         : 0x%08X"                    ,u32PtyFilterForSeek              ));
      ETG_TRACE_USR2(( " u16AutoSeekStepDelay        : %d ms"                     ,u16AutoSeekStepDelay             ));
      ETG_TRACE_USR2(( " u16PemanentManualStepDelay  : %d ms"                     ,u16PemanentManualStepDelay       ));
      ETG_TRACE_USR2(( " u8TunerBehaviourSwitches    : %d ms"                     ,u8TunerBehaviourSwitches       	));
      ETG_TRACE_USR2(( " u16MuteRamp1                : 0x%04X"                    ,u16MuteRamp1                     ));
      ETG_TRACE_USR2(( " u16DemuteRamp1              : 0x%04X"                    ,u16DemuteRamp1                   ));
      ETG_TRACE_USR2(( " u16MuteRamp2                : 0x%04X"                    ,u16MuteRamp2                     ));
      ETG_TRACE_USR2(( " u16DemuteRamp2              : 0x%04X"                    ,u16DemuteRamp2                   ));

      /*==========================================================================
       * Stage 2:
       * Behaviour of  station info auto notification (not audio tuner dependent):
       *------------------------------------------------------------------------*/
      ETG_TRACE_USR2(( " > Stage 2: Behaviour of  station info auto notification (not audio tuner dependent):"      ));
      ETG_TRACE_USR2(( " u8ATStationInfoSwitches            : %02X "                      ,u8ATStationInfoSwitches	));

      /*==========================================================================
       * Stage 3:
       * Timer Configuration (not audio tuner dependent):
       *------------------------------------------------------------------------*/
      ETG_TRACE_USR2(( " > Stage 3: Timer Configuration (not audio tuner dependent)" ));
	  ETG_TRACE_USR2(( " u8TimerScanRadio					: %d  s"                      ,u8TimerScanRadio));
      ETG_TRACE_USR2(( " u8TimerInhibitLstSortAfterSkip     : %d  s"                      ,u8TimerInhibitLstSortAfterSkip));
      ETG_TRACE_USR2(( " u8TimerDelayTPstnSelAfterVol0      : %d  s"                      ,u8TimerDelayTPstnSelAfterVol0));
      ETG_TRACE_USR2(( " u8TimerDelayNoTPstnFound           : %d  s"                      ,u8TimerDelayNoTPstnFound));
      ETG_TRACE_USR2(( " u8TimerNonRDS_Detect				: %d  s"                      ,u8TimerNonRDS_Detect));
      ETG_TRACE_USR2(( " u8TimerPS_DelAfterRDSLost          : %d  s"                      ,u8TimerPS_DelAfterRDSLost));
      ETG_TRACE_USR2(( " u8TimerWrongPIAccept				: %d  s"                      ,u8TimerWrongPIAccept));
      ETG_TRACE_USR2(( " u8TimerScrollPSDetect				: %d  s"                      ,u8TimerScrollPSDetect));
      ETG_TRACE_USR2(( " u8TimerDelayTPstnSelAfterRDSLost   : %d  s"                      ,u8TimerDelayTPstnSelAfterRDSLost));
      ETG_TRACE_USR2(( " u8TimerTAendAfterRDSLost           : %d  s"                      ,u8TimerTAendAfterRDSLost));
      ETG_TRACE_USR2(( " u8TimerAM_MinTunTime				: %d  s"                      ,u8TimerAM_MinTunTime));
      ETG_TRACE_USR2(( " u8TimerCursorDelayAfterStnSel      : %d  s"                      ,u8TimerCursorDelayAfterStnSel));
      ETG_TRACE_USR2(( " u8TimerAF_InhibitAfterManTune      : %d  s"                      ,u8TimerAF_InhibitAfterManTune));
      ETG_TRACE_USR2(( " u8TimerAcceptPIAfterRecall         : %d  s"                      ,u8TimerAcceptPIAfterRecall));
      ETG_TRACE_USR2(( " u8TimerPTY_Hold					: %d  s"                      ,u8TimerPTY_Hold));

	  /*==========================================================================
       * Stage 4:
       * Parameters that influence the reception behaviour (not audio tuner dependent):
       *------------------------------------------------------------------------*/
      ETG_TRACE_USR2(( " > Stage 4: Parameters that influence the reception behaviour (not audio tuner dependent):" ));
	  ETG_TRACE_USR2(( " u8FieldStrengthSeekLevel_FM_RDS        : 0x%02X"                    ,u8FieldStrengthSeekLevel_FM_RDS ));
      ETG_TRACE_USR2(( " u8QualitySeekLevel_FM_RDS              : 0x%02X"                    ,u8QualitySeekLevel_FM_RDS ));
      ETG_TRACE_USR2(( " u8FieldStrengthSeekLevel_FM_NONRDS     : 0x%02X"                    ,u8FieldStrengthSeekLevel_FM_NONRDS ));
      ETG_TRACE_USR2(( " u8QualitySeekLevel_FM_NONRDS           : 0x%02X"                    ,u8QualitySeekLevel_FM_NONRDS ));
      ETG_TRACE_USR2(( " u8FieldStrengthSeekLevel_LW            : 0x%02X"                    ,u8FieldStrengthSeekLevel_LW ));
      ETG_TRACE_USR2(( " u8QualitySeekLevel_LW					: 0x%02X"                    ,u8QualitySeekLevel_LW ));
      ETG_TRACE_USR2(( " u8FieldStrengthSeekLevel_MW			: 0x%02X"                    ,u8FieldStrengthSeekLevel_MW ));
      ETG_TRACE_USR2(( " u8QualitySeekLevel_MW					: 0x%02X"                    ,u8QualitySeekLevel_MW ));
      ETG_TRACE_USR2(( " u8FieldStrengthSeekLevel_SW            : 0x%02X"                    ,u8FieldStrengthSeekLevel_SW ));
      ETG_TRACE_USR2(( " u8QualitySeekLevel_SW					: 0x%02X"                    ,u8QualitySeekLevel_SW ));
      ETG_TRACE_USR2(( " u8FieldStrengthSeekLevel_TRF           : 0x%02X"                    ,u8FieldStrengthSeekLevel_TRF ));
      ETG_TRACE_USR2(( " u8QualitySeekLevel_TRF					: 0x%02X"                    ,u8QualitySeekLevel_TRF ));
      ETG_TRACE_USR2(( " u8FieldStrengthSeekLevel_WB            : 0x%02X"                    ,u8FieldStrengthSeekLevel_WB ));
      ETG_TRACE_USR2(( " u8QualitySeekLevel_WB					: 0x%02X"                    ,u8QualitySeekLevel_WB ));

      ETG_TRACE_USR2(( " u8FieldStrengthListLevel_FM_RDS		: 0x%02X"                    ,u8FieldStrengthListLevel_FM_RDS ));
      ETG_TRACE_USR2(( " u8QualityListLevel_FM_RDS				: 0x%02X"                    ,u8QualityListLevel_FM_RDS ));
      ETG_TRACE_USR2(( " u8FieldStrengthListLevel_FM_NONRDS     : 0x%02X"                    ,u8FieldStrengthListLevel_FM_NONRDS ));
      ETG_TRACE_USR2(( " u8QualityListLevel_FM_NONRDS			: 0x%02X"                    ,u8QualityListLevel_FM_NONRDS ));
      ETG_TRACE_USR2(( " u8FieldStrengthListLevel_LW			: 0x%02X"                    ,u8FieldStrengthListLevel_LW ));
      ETG_TRACE_USR2(( " u8QualityListLevel_LW					: 0x%02X"                    ,u8QualityListLevel_LW ));
      ETG_TRACE_USR2(( " u8FieldStrengthListLevel_MW			: 0x%02X"                    ,u8FieldStrengthListLevel_MW ));
      ETG_TRACE_USR2(( " u8QualityListLevel_MW					: 0x%02X"                    ,u8QualityListLevel_MW ));
      ETG_TRACE_USR2(( " u8FieldStrengthListLevel_SW			: 0x%02X"                    ,u8FieldStrengthListLevel_SW ));
      ETG_TRACE_USR2(( " u8QualityListLevel_SW					: 0x%02X"                    ,u8QualityListLevel_SW ));
      ETG_TRACE_USR2(( " u8FieldStrengthListLevel_TRF			: 0x%02X"                    ,u8FieldStrengthListLevel_TRF ));
      ETG_TRACE_USR2(( " u8QualityListLevel_TRF					: 0x%02X"                    ,u8QualityListLevel_TRF ));
      ETG_TRACE_USR2(( " u8FieldStrengthListLevel_WB			: 0x%02X"                    ,u8FieldStrengthListLevel_WB ));
      ETG_TRACE_USR2(( " u8QualityListLevel_WB					: 0x%02X"                    ,u8QualityListLevel_WB ));

      ETG_TRACE_USR2(( " u8FM_SHARX       : %d"                 ,u8FM_SHARX ));
      ETG_TRACE_USR2(( " u8AM_SHARX       : %d"                 ,u8AM_SHARX ));
      ETG_TRACE_USR2(( " u8FM_HICUT       : %d"                 ,u8FM_HICUT ));
	  ETG_TRACE_USR2(( " u8AM_HICUT       : %d"                 ,u8AM_HICUT ));

      /*==========================================================================
       * Stage 5:
       * Area setup for CONFIG area (no CONFIG arear needed) (not audio tuner dependent):
       *------------------------------------------------------------------------*/
      ETG_TRACE_USR2(( " > Stage 5: Area setup for CONFIG area (no CONFIG arear needed) (not audio tuner dependent):"));
      ETG_TRACE_USR2(( " u32AvailableBands           : 0x%08X"                     , u32AvailableBands              ));
      ETG_TRACE_USR2(( " u32CombiBands				 : 0x%08X"                     , u32CombiBands                  ));
      ETG_TRACE_USR2(( " u32FM_LowFrequency          : %8d kHz"                    , u32FM_LowFrequency             ));
      ETG_TRACE_USR2(( " u32FM_HighFrequency         : %8d kHz"                    , u32FM_HighFrequency            ));
      ETG_TRACE_USR2(( " u16FM_StepSizeAutoSeek      : %8d kHz"                    , u16FM_StepSizeAutoSeek         ));
      ETG_TRACE_USR2(( " u16FM_StepSizeManualStep    : %8d kHz"                    , u16FM_StepSizeManualStep       ));
      ETG_TRACE_USR2(( " u32FM_LSM                   : %8d kHz"                    , u32FM_LSM                      ));
      ETG_TRACE_USR2(( " u32LW_LowFrequency          : %8d kHz"                    , u32LW_LowFrequency             ));
      ETG_TRACE_USR2(( " u32LW_HighFrequency         : %8d kHz"                    , u32LW_HighFrequency            ));
      ETG_TRACE_USR2(( " u16LW_StepSizeAutoSeek      : %8d kHz"                    , u16LW_StepSizeAutoSeek         ));
      ETG_TRACE_USR2(( " u16LW_StepSizeManualStep    : %8d kHz"                    , u16LW_StepSizeManualStep       ));
      ETG_TRACE_USR2(( " u32LW_LSM                   : %8d kHz"                    , u32LW_LSM                      ));
      ETG_TRACE_USR2(( " u32MW_LowFrequency          : %8d kHz"                    , u32MW_LowFrequency             ));
      ETG_TRACE_USR2(( " u32MW_HighFrequency         : %8d kHz"                    , u32MW_HighFrequency            ));
      ETG_TRACE_USR2(( " u16MW_StepSizeAutoSeek      : %8d kHz"                    , u16MW_StepSizeAutoSeek         ));
      ETG_TRACE_USR2(( " u16MW_StepSizeManualStep    : %8d kHz"                    , u16MW_StepSizeManualStep       ));
      ETG_TRACE_USR2(( " u32MW_LSM                   : %8d kHz"                    , u32MW_LSM                      ));
      ETG_TRACE_USR2(( " u8special_behaviour         : 0x%02X"                     , u8special_behaviour            ));

      /*==========================================================================
       * Stage 6:
       * User settings (audio tuner dependent):
       *------------------------------------------------------------------------*/
      ETG_TRACE_USR2(( " > Stage 6: User settings (audio tuner dependent):"                                         ));
      ETG_TRACE_USR2(( " u8TP_Switch_AudioTuner1     : 0x%02X"                    ,u8TP_Switch_AudioTuner1          ));
      ETG_TRACE_USR2(( " u8TP_Switch_AudioTuner2     : 0x%02X"                    ,u8TP_Switch_AudioTuner2          ));
      ETG_TRACE_USR2(( " u8AF_Switch_AudioTuner1     : 0x%02X"                    ,u8AF_Switch_AudioTuner1          ));
      ETG_TRACE_USR2(( " u8AF_Switch_AudioTuner2     : 0x%02X"                    ,u8AF_Switch_AudioTuner2          ));
      ETG_TRACE_USR2(( " enDDA_Switch_AudioTuner1    : %02x"                      ,ETG_CENUM( tenDDA_Switch, enDDA_Switch_AudioTuner1) ));
      ETG_TRACE_USR2(( " enDDA_Switch_AudioTuner2    : %02x"                      ,ETG_CENUM( tenDDA_Switch, enDDA_Switch_AudioTuner2) ));
      ETG_TRACE_USR2(( " u8RDS_Switch_AudioTuner1    : 0x%02X"                    ,u8RDS_Switch_AudioTuner1         ));
      ETG_TRACE_USR2(( " u8RDS_Switch_AudioTuner2    : 0x%02X"                    ,u8RDS_Switch_AudioTuner2         ));
      ETG_TRACE_USR2(( " u8PS_FIX_Switch_AudioTuner1 : 0x%02X"                    ,u8PS_FIX_Switch_AudioTuner1      ));
      ETG_TRACE_USR2(( " u8PS_FIX_Switch_AudioTuner2 : 0x%02X"                    ,u8PS_FIX_Switch_AudioTuner2      ));
      ETG_TRACE_USR2(( " u8Stereo_Switch_AudioTuner1 : 0x%02X"                    ,u8Stereo_Switch_AudioTuner1      ));
      ETG_TRACE_USR2(( " u8Stereo_Switch_AudioTuner2 : 0x%02X"                    ,u8Stereo_Switch_AudioTuner2      ));
      ETG_TRACE_USR2(( " enREG_Switch_AudioTuner1    : %02x"                      ,ETG_CENUM( tenREG_Switch, enREG_Switch_AudioTuner1) ));
      ETG_TRACE_USR2(( " enREG_Switch_AudioTuner2    : %02x"                      ,ETG_CENUM( tenREG_Switch, enREG_Switch_AudioTuner2) ));
      ETG_TRACE_USR2(( " u8PTY31_Switch_AudioTuner1	 : 0x%02X"                    ,u8PTY31_Switch_AudioTuner1      ));
      ETG_TRACE_USR2(( " u8PTY31_Switch_AudioTuner2  : 0x%02X"                    ,u8PTY31_Switch_AudioTuner2      ));
      ETG_TRACE_USR2(( " u8DDS_Switch_AudioTuner1    : 0x%02X"                    ,u8DDS_Switch_AudioTuner1      ));
      ETG_TRACE_USR2(( " u8DDS_Switch_AudioTuner2    : 0x%02X"                    ,u8DDS_Switch_AudioTuner2      ));
      ETG_TRACE_USR2(( " u8DMQ_Switch_AudioTuner1    : 0x%02X"                    ,u8DMQ_Switch_AudioTuner1      ));
      ETG_TRACE_USR2(( " u8DMQ_Switch_AudioTuner2    : 0x%02X"                    ,u8DMQ_Switch_AudioTuner2      ));

 if(m_potun_Config -> bIsHDSupported())
 {
      /*==========================================================================
       * Stage 7:
       * HD Parameter :
       *------------------------------------------------------------------------*/
      ETG_TRACE_USR2(( " > Stage 7: HD Parameter					: "																	));
      ETG_TRACE_USR2(( " u8BlendThresholdFMHybrid					: 0x%02X"                    ,u8BlendThresholdFMHybrid				));
      ETG_TRACE_USR2(( " u16DigiAudioScalingFMHybrid				: 0x%04X"                    ,u16DigiAudioScalingFMHybrid           ));
      ETG_TRACE_USR2(( " u16DigiAudioDelayFMHybridPrimaryBBRate     : 0x%04X"                    ,u16DigiAudioDelayFMHybridPrimaryBBRate));
      ETG_TRACE_USR2(( " u8BlendRateFMHybrid						: 0x%02X"                    ,u8BlendRateFMHybrid                   ));
      ETG_TRACE_USR2(( " u8BlendThresholdAMHybrid					: 0x%02X"                    ,u8BlendThresholdAMHybrid              ));
      ETG_TRACE_USR2(( " u16AM_MPSAudioScaling						: 0x%04X"                    ,u16AM_MPSAudioScaling					));
      ETG_TRACE_USR2(( " u8AM_MPSBlendRate							: 0x%02X"                    ,u8AM_MPSBlendRate						));
      ETG_TRACE_USR2(( " u16DigiAudioDelayAM_MPSPrimaryBBRate       : 0x%04X"                    ,u16DigiAudioDelayAM_MPSPrimaryBBRate  ));
      ETG_TRACE_USR2(( " u8DigiAudioPhaseInversion					: 0x%02X"                    ,u8DigiAudioPhaseInversion             ));
      ETG_TRACE_USR2(( " u8BlendDecision							: 0x%02X"                    ,u8BlendDecision						));
      ETG_TRACE_USR2(( " u8FM_CDNO_BlendDecision					: 0x%02X"                    ,u8FM_CDNO_BlendDecision				));
      ETG_TRACE_USR2(( " u8AM_CDNO_BlendDecision					: 0x%02X"                    ,u8AM_CDNO_BlendDecision               ));
 }
	 
	  /*==========================================================================
       * Stage 8:
       * HF Parameter (not physical tuner dependent):
       *------------------------------------------------------------------------*/
      ETG_TRACE_USR2(( " > Stage 7: HF Parameter (not physical tuner dependent):"                                   ));
      ETG_TRACE_USR2(( " u8FM_Channel_Separation     : 0x%02X"                    ,u8FM_Channel_Separation          ));
      ETG_TRACE_USR2(( " u8FM_Limiting               : 0x%02X"                    ,u8FM_Limiting                    ));
      ETG_TRACE_USR2(( " u8FM_Multipath              : 0x%02X"                    ,u8FM_Multipath                   ));
      ETG_TRACE_USR2(( " u8FM_Softmute               : 0x%02X"                    ,u8FM_Softmute                    ));
      ETG_TRACE_USR2(( " u8AM_Softmute               : 0x%02X"                    ,u8AM_Softmute                    ));
      ETG_TRACE_USR2(( " u8FM_Hicut_Configuration    : 0x%02X"                    ,u8FM_Hicut_Configuration         ));
      ETG_TRACE_USR2(( " u8AM_Hicut_Configuration    : 0x%02X"                    ,u8AM_Hicut_Configuration         ));
      ETG_TRACE_USR2(( " u8FM_Deemphasis             : 0x%02X"                    ,u8FM_Deemphasis                  ));
      ETG_TRACE_USR2(( " u8AM_Deemphasis             : 0x%02X"                    ,u8AM_Deemphasis                  ));
      ETG_TRACE_USR2(( " u8FM_Frequency_Response     : 0x%02X"                    ,u8FM_Frequency_Response          ));
      ETG_TRACE_USR2(( " u8AM_Frequency_Response     : 0x%02X"                    ,u8AM_Frequency_Response          ));
      ETG_TRACE_USR2(( " u8FM_Miscellaneous          : 0x%02X"                    ,u8FM_Miscellaneous               ));
      ETG_TRACE_USR2(( " u8AM_Miscellaneous          : 0x%02X"                    ,u8AM_Miscellaneous               ));
      ETG_TRACE_USR2(( " u8AM_Antenna_Attenuation    : 0x%02X"                    ,u8AM_Antenna_Attenuation         ));
      ETG_TRACE_USR2(( " u8FM_Antenna_Attenuation    : 0x%02X"                    ,u8FM_Antenna_Attenuation         ));
      ETG_TRACE_USR2(( " enPhysicalTunerConfigId     : %02x"                      ,ETG_CENUM( tenFmDabHdTunerMode, enPhysicalTunerConfigId) ));
      ETG_TRACE_USR2(( " u8AntennaConfigId           : 0x%02X"                    ,u8AntennaConfigId                ));

      /*==========================================================================
       * Stage 9:
       * RDS Parameter
       *------------------------------------------------------------------------*/
      ETG_TRACE_USR2(( " > Stage 8: RDS Parameter:"                                                                 ));
      ETG_TRACE_USR2(( " u8ExchangeAntennaLevel      : 0x%02X"                    ,u8ExchangeAntennaLevel           ));
      ETG_TRACE_USR2(( " u8PitestLevel               : %2d dBuV"                  ,u8PitestLevel                    ));
      ETG_TRACE_USR2(( " u8NrdsLevel                 : %2d dBuV"                  ,u8NrdsLevel                      ));
      ETG_TRACE_USR2(( " u8EonLevel                  : %2d dBuV"                  ,u8EonLevel                       ));
      ETG_TRACE_USR2(( " u8FmMemoLevel               : %2d dBuV"                  ,u8FmMemoLevel                    ));
      ETG_TRACE_USR2(( " u8PanicFieldstLevel         : %2d dBuV"                  ,u8PanicFieldstLevel              ));
      ETG_TRACE_USR2(( " u8ParamRdsLevel             : %2d dBuV"                  ,u8ParamRdsLevel                  ));
      ETG_TRACE_USR2(( " u8RdsDataAcceptLevel        : %2d dBuV"                  ,u8RdsDataAcceptLevel             ));
      ETG_TRACE_USR2(( " u8PiStepLevel               : %2d dBuV"                  ,u8PiStepLevel                    ));
      ETG_TRACE_USR2(( " u8DdaAntennaDistLevel       : %3d"                       ,u8DdaAntennaDistLevel            ));
      ETG_TRACE_USR2(( " u8RdsSynDelayCnt            : %3d"                       ,u8RdsSynDelayCnt                 ));
      ETG_TRACE_USR2(( " u8MpMeasureLevel            : %3d"                       ,u8MpMeasureLevel                 ));
      ETG_TRACE_USR2(( " u8MpLevel2Level             : %3d"                       ,u8MpLevel2Level                  ));
      ETG_TRACE_USR2(( " u8MpVerybadLevel            : %3d"                       ,u8MpVerybadLevel                 ));
      ETG_TRACE_USR2(( " u8MpMediumLevel             : %3d"                       ,u8MpMediumLevel                  ));
      ETG_TRACE_USR2(( " u8RdsBadLevel               : %3d"                       ,u8RdsBadLevel                    ));
      ETG_TRACE_USR2(( " u8IllegalAf                 : %3d"                       ,u8IllegalAf                      ));
      ETG_TRACE_USR2(( " u8AfSaveLevel               : %3d"                       ,u8AfSaveLevel                    ));
      ETG_TRACE_USR2(( " u8FirstRundum               : %3d seconds"               ,u8FirstRundum                    ));
      ETG_TRACE_USR2(( " u8SecondRundum              : %3d seconds"               ,u8SecondRundum                   ));
      ETG_TRACE_USR2(( " u8ThirdRundum               : %3d seconds"               ,u8ThirdRundum                    ));
      ETG_TRACE_USR2(( " u8PIStepMode                : %3d"                       ,u8PIStepMode                     ));
      ETG_TRACE_USR2(( " u8AudioOffScanMode          : %3d"                       ,u8AudioOffScanMode               ));
      ETG_TRACE_USR2(( " u8AMLearnCycle              : %3d seconds"               ,u8AMLearnCycle                   ));
      ETG_TRACE_USR2(( " u8PIIgnore                  : %3d seconds"               ,u8PIIgnore                       ));
	  /** Free1 Byte */
      ETG_TRACE_USR2(( " u8PISearchTime              : %3d seconds"               ,u8PISearchTime                   )); 
      /** Free2 Byte */
	  ETG_TRACE_USR2(( " u8DisableSeamlessBlending   : %3d seconds"               ,u8DisableSeamlessBlending        ));
      ETG_TRACE_USR2(( " u8SlAdjThreshold            : %3d seconds"               ,u8SlAdjThreshold                 ));
      ETG_TRACE_USR2(( " u8SlMpThreshold             : %3d seconds"               ,u8SlMpThreshold                  ));
      ETG_TRACE_USR2(( " u8SpecialFmStartFrequency   : %3d seconds"               ,u8SpecialFmStartFrequency        ));
	  ETG_TRACE_USR2(( " u8HDScanEveryFreq           : %3d "					  ,u8HDScanEveryFreq				));
	  ETG_TRACE_USR2(( " u8HWSubConfig               : %3d seconds"               ,u8HWSubConfig					));

	   /*==========================================================================
       * Stage 10:
       * Testmode and AF-List configuration:
       *------------------------------------------------------------------------*/
		ETG_TRACE_USR2(( " > Stage 8: Testmode and AF-List configuration:" ));
		ETG_TRACE_USR2(( " u8TestModeDataUpdateInterval      : %d  s"        ,u8TestModeDataUpdateInterval     ));
		ETG_TRACE_USR2(( " u8AFListUpdateInterval	         : %d  s"        ,u8AFListUpdateInterval           ));
		ETG_TRACE_USR2(( " u8AFListReleaseBehaviour	         : 0x%02X"        ,u8AFListReleaseBehaviour         ));

	   /*==========================================================================
		* Stage 11:
		* Station list configuration:
		*------------------------------------------------------------------------*/
		ETG_TRACE_USR2(( " > Stage 9: Station list configuration:" ));
		ETG_TRACE_USR2(( " u8MaxNumberofStationsInStnList      : %d"        ,u8MaxNumberofStationsInStnList  ));
		
 
    
      /*==========================================================================
       * Stage 12:
       * Preset list configuration (not audio tuner dependent):
       *------------------------------------------------------------------------*/
      {
        tUInt unBankNo   = 0;
        tUInt unBankLoop;
        tUInt unPresetNo = 0;
        tUInt unPresetLoop;
        /* FM lists */

        ETG_TRACE_USR2(( " > Preset list configuration (not audio tuner dependent):"                                ));
        /*==========================================================================
         * FM preset lists
         *------------------------------------------------------------------------*/
        for( unBankLoop = 0; unBankLoop < m_potun_Config->u8GetNoFMpresetBanks(); unBankLoop++, unBankNo++)
        {

          ETG_TRACE_USR2(( " enPresetList%02d_Band         : %02x (read from KDS)", unBankNo, ETG_CENUM(tenBand, enBand_FM) ));
          ETG_TRACE_USR2(( " u8PresetList%02d_NumOfPresets : %2d (read from KDS)", unBankNo,  m_potun_Config->u8GetNoOfFMpresetPerBank() ));

          for( unPresetLoop = 0; unPresetLoop < m_potun_Config->u8GetNoOfFMpresetPerBank(); unPresetLoop++, unPresetNo++)
          {
            ETG_TRACE_USR2(( " u8PresetList%-02dPreset%-02d      : %8d kHz (read from KDS)", unBankNo, unPresetLoop, m_potun_Config->u32GetPreset( (tU32)unPresetNo) ));
          }
        }
        /*==========================================================================
         * AM preset lists
         *------------------------------------------------------------------------*/
        for( unBankLoop = 0; unBankLoop < m_potun_Config->u8GetNoAMpresetBanks1(); unBankLoop++, unBankNo++)
        {
		ETG_TRACE_USR2(( " enPresetList%02d_Band         : %x (read from KDS)", unBankNo, ETG_CENUM(tenBand, enBand_MW) ));


            ETG_TRACE_USR2(( " u8PresetList%02d_NumOfPresets : %2d (read from KDS)", unBankNo,  m_potun_Config->u8GetNoOfAMpresetPerBank1() ));

          for( unPresetLoop = 0; unPresetLoop < m_potun_Config->u8GetNoOfAMpresetPerBank1(); unPresetLoop++, unPresetNo++)
          {
            ETG_TRACE_USR2(( " u8PresetList%02dPreset%02d      : %8d kHz (read from KDS)", unBankNo, unPresetLoop, m_potun_Config->u32GetPreset( (tU32)unPresetNo) ));
          }
        }

#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
        /*==========================================================================
         * drm preset lists
         *------------------------------------------------------------------------*/
        for( unBankLoop = 0; unBankLoop < m_potun_Config->u8GetNoDRMpresetBanks(); unBankLoop++, unBankNo++)
        {

        	ETG_TRACE_USR2(( " enPresetList%02d_DRMBank         : %02x (read from KDS)", unBankNo, ETG_CENUM(tenBand, enBand_MW) ));
        	ETG_TRACE_USR2(( " u8PresetList%02d_NumOfPresets : %2d (read from KDS)", unBankNo,  m_potun_Config->u8GetNoOfDRMpresetPerBank() ));

        	for( unPresetLoop = 0; unPresetLoop < m_potun_Config->u8GetNoOfDRMpresetPerBank(); unPresetLoop++, unPresetNo++)
        	{
        		ETG_TRACE_USR2(( " u8PresetList%-02dPreset%-02d      : %8d kHz (read from KDS)", unBankNo, unPresetLoop, m_potun_Config->u32GetPreset( (tU32)unPresetNo) ));
        	}
        }
#endif
		for( unBankLoop = 0; unBankLoop < m_potun_Config->u8GetNoMIXEDpresetBanks(); unBankLoop++, unBankNo++)
        {
          ETG_TRACE_USR2(( " enPresetList%02d_Band         : %02x (read from KDS)", unBankNo, ETG_CENUM(tenBand, enAvailBand_GetCurrent) ));

          ETG_TRACE_USR2(( " u8PresetList%02d_NumOfPresets : %2d (read from KDS)", unBankNo,  m_potun_Config->u8GetNoOfMIXEDpresetPerBank() ));

          for( unPresetLoop = 0; unPresetLoop < m_potun_Config->u8GetNoOfMIXEDpresetPerBank(); unPresetLoop++, unPresetNo++)
          {
            ETG_TRACE_USR2(( " u8PresetList%-02dPreset%-02d      : %8d kHz (read from KDS)", unBankNo, unPresetLoop, m_potun_Config->u32GetPreset( (tU32)unPresetNo) ));
          }
        }
        /*==========================================================================
         * End of List
         *------------------------------------------------------------------------*/
        ETG_TRACE_USR2(( " u16PresetConfig_End         :   0x0000"  ));
        /*========================================================================*/
      }
      ETG_TRACE_USR2(( " END   ------------------------------------------------------------------------------------"));
    }
    
    /***************************************************************************
     * print raw message
     **************************************************************************/
    if( ( (tU8)enTraceType & (tU8)enTraceSetConfiguration_TraceOnlyRaw) != 0)
    {
      tUInt unIndex;
      tUInt unLen = 32; /* values per trace line */

      ETG_TRACE_USR2(( " BEGIN ------------------------------------------------------------------------------------"));
#   ifndef LSIM
      ETG_TRACE_USR2(( "       %02x"
                      ,ETG_LIST_LEN( enAdrMsgOffset_HEADER_LEN)
                      ,ETG_LIST_PTR_T8( au8Message)
                    ));
      ETG_TRACE_USR2(( "       %02x"
                      ,ETG_LIST_LEN( (tU8)enCM_SetConfiguration_len_VersionId + (tU8)enCM_SetConfiguration_len_NumberOfBytes)
                      ,ETG_LIST_PTR_T8( &au8Message[ enCM_SetConfiguration_idx_VersionId])
                    ));
#endif
      for( unIndex = 0; (unIndex < u16NumberOfPayloadBytes) && (unLen > 0); unIndex += unLen)
      {
        if( (u16NumberOfPayloadBytes - unIndex) < unLen)
        {
          unLen = u16NumberOfPayloadBytes - unIndex;
        }
#   ifndef LSIM
        ETG_TRACE_USR2(( " [%3d] %02x"
                        ,unIndex
                        ,ETG_LIST_LEN( unLen)
                        ,ETG_LIST_PTR_T8( &au8Message[ (tU8)enCM_SetConfiguration_idx_NumOfSeekPeriods + unIndex])
                      ));
#endif
      }
      ETG_TRACE_USR2(( " END   ------------------------------------------------------------------------------------"));
    }
  }
  
  /***************************************************************************
   * calculate CRC check sum
   **************************************************************************/
  if( pu32Crc != OSAL_NULL)
  {
    *pu32Crc = m_potun_Utility->u32CalculateCRC32( &au8Message[ enAdrMsgOffset_DATA],  u32MessageLen - (tU8)enAdrMsgOffset_HEADER_LEN);

    ETG_TRACE_USR2(( " tun_MsgToADR::bSendCM_SetConfiguration() -> u32CRC=0x%08X"
                     ,*pu32Crc
                   ));
  }
  
  u8HWCrc = (tU8)enPhysicalTunerConfigId;

  /***************************************************************************
   * send message
   **************************************************************************/
  if( ( (tU8)enTraceType & (tU8)enTraceSetConfiguration_Send) != 0)
  {
    /* Message to be sent is put in Que   */
    vWaitOnMessageQue( au8Message, u32MessageLen );
//    m_bSetConfigurationMessageSent = TRUE;
  }
  /*========================================================================*/
  return TRUE;
#endif
}


/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vHandle_r_SetConfiguration( )
* 
* DESCRIPTION:  This function is called by incoming ADR message
*               CM_SetConfiguration.
*
* PARAMETER:    None
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vHandle_r_SetConfiguration( tPU8 pu8Message, tU32 /*u32MessageLen*/)
{
  TUN_ASSERT_RETURN( m_ptun_MessageQ != NULL);

  if(    ( pu8Message[ enAdrMsgOffset_OP_TYPE] == (tU8)enOpType_RESULT)
      && ( pu8Message[ enAdrMsgOffset_DATA]    == 0x00)
    )
  {
    ETG_TRACE_COMP(( " tun_MsgToADR::vHandle_r_SetConfiguration() -> Configruation finished, read new check sum from ADR"
                  ));

    /*==========================================================================
     * read the checksum after the configuration
     *------------------------------------------------------------------------*/
    m_bReadCheckSumAfterConfig = TRUE;
	/**Reset the variable and set the m_fWaitingForConfirmation to FALSE to send the other messages.*/
//    m_bSetConfigurationMessageSent = FALSE;
//	m_ptun_MessageQ -> vResetWaitingForConfirmation();
	/***/

#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
    TUN_ASSERT_RETURN( m_potun_main != NULL);
    TUN_ASSERT_RETURN( m_potun_main->m_poclAars_GetConfigurationInfo  != NULL);
    m_potun_main->m_poclAars_GetConfigurationInfo->vGetConfigurationinfo();
#endif


    /*========================================================================*/
#if defined(VARIANT_S_FTR_ENABLE_FEATURE_RNAIVI) || defined(VARIANT_S_FTR_ENABLE_FEATURE_PSA_RCC)

    vGetUser();
    //vStartupSetUserReq();
#else
    vDoNotification();
#endif

    if(m_potun_tclDiagHandler->blGetSystemSetFinished() == FALSE)
    {
        m_potun_tclDiagHandler->vSendSystemSetResponse();
    }

  }
  else
  {
    ETG_TRACE_ERR(( " tun_MsgToADR::vHandle_r_SetConfiguration() -> Configruation failed"
                  ));
  }

}

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendCM_VersionInfo( )
* 
* DESCRIPTION:  Requests the hard- and software version ID
*
* PARAMETER:    None
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vSendCM_VersionInfo( tVoid)
{
  ETG_TRACE_COMP(( " tun_MsgToADR::vSendCM_VersionInfo() ->" ));

  TUN_ASSERT_RETURN( m_potun_Utility != NULL);

  /*==========================================================================
   * Read the version ID from ADR by CM_VersionInfo
   *------------------------------------------------------------------------*/
  tU8 au8Message[ TUN_CM_VERSIONINFO_SET_LEN];
  
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_CLIENT_ID], (tU16)enClientId_AmFmTuner);
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_FBLOCK_ID], (tU16)enFBlockId_AmFmTuner);
  au8Message[ enAdrMsgOffset_INSTANCE_ID] = (tU8)enInstanceId_All;
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_FKT_ID], (tU16)enMsg_CM_VersionInfo);
  au8Message[ enAdrMsgOffset_OP_TYPE] = (tU8)enOpType_GET;
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_DATA_LEN], 0);
  
  /* Message to be sent is put in Que   */
  vWaitOnMessageQue( au8Message, TUN_CM_VERSIONINFO_SET_LEN );
  /*========================================================================*/
}


/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vHandle_r_CM_VersionInfo( )
* 
* DESCRIPTION:  This function is called by incoming ADR message
*               CM_VersionInfo.
*
* PARAMETER:    None
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vHandle_r_CM_VersionInfo( tPU8 pu8Message, tU32 u32MessageLen) 
{

  (tVoid) u32MessageLen;                /* unused paremeter (makes Lint happy) */

  if( pu8Message[ enAdrMsgOffset_OP_TYPE] == (tU8)enOpType_STATUS)
  {
    ETG_TRACE_COMP(( " tun_MsgToADR::vHandle_r_CM_VersionInfo() ->\n"
                     " SoftwareVersion                  : %64s\n"
                     " HardwareVersion                  : %64s\n"
                    ,(tChar*)&pu8Message[ enCM_VersionInf_idx_SoftwareVersion]
                    ,(tChar*)&pu8Message[ enCM_VersionInf_idx_HardwareVersion]
                  ));

    /*==========================================================================
     * Write SoftwareVersion into Registry
     *------------------------------------------------------------------------*/
    dp_tclregVersionsAdrHwVersion oAdrHwVersion;
    tU8 u8DPRead = oAdrHwVersion.u8GetData(m_sADRHWVersion , (tU8)enCM_VersionInf_len_HardwareVersion);
    ETG_TRACE_USR2(( " tun_MsgToADR::vHandle_r_CM_VersionInfo , HW version DP read status : %d ", u8DPRead));

    if(strcmp((tCString)m_sADRHWVersion , (tCString)&pu8Message[ enCM_VersionInf_idx_HardwareVersion]))
    {
    	ETG_TRACE_USR2(( " tun_MsgToADR::vHandle_r_CM_VersionInfo , ADR Hardware version changed "));
    oAdrHwVersion.vSetData( (tString)&pu8Message[ enCM_VersionInf_idx_HardwareVersion], (tU8)enCM_VersionInf_len_HardwareVersion);
	oAdrHwVersion.u8GetData(m_sADRHWVersion , (tU8)enCM_VersionInf_len_HardwareVersion);
    }

    /*========================================================================*/
    
    /*==========================================================================
     * Write HardwareVersion into Registry
     *------------------------------------------------------------------------*/
    dp_tclregVersionsAdrSwVersion oAdrSwVersion;
    u8DPRead = oAdrSwVersion.u8GetData(m_sADRSWVersion , (tU8)enCM_VersionInf_len_SoftwareVersion);
    ETG_TRACE_USR2(( " tun_MsgToADR::vHandle_r_CM_VersionInfo , SW version DP read status : %d ", u8DPRead));

    if(strcmp((tCString)m_sADRSWVersion , (tCString)&pu8Message[ enCM_VersionInf_idx_SoftwareVersion]))
    {
    	ETG_TRACE_USR2(( " tun_MsgToADR::vHandle_r_CM_VersionInfo , ADR Software version changed "));
    oAdrSwVersion.vSetData( (tString)&pu8Message[ enCM_VersionInf_idx_SoftwareVersion], (tU8)enCM_VersionInf_len_SoftwareVersion);
	oAdrSwVersion.u8GetData(m_sADRSWVersion , (tU8)enCM_VersionInf_len_SoftwareVersion);
    }

    /*========================================================================*/
  }
}
/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vHandle_r_CM_AdditionalVersionInfo( )
*
* DESCRIPTION:  This function is called by incoming ADR message
*               CM_AdditionalVersionInfo.
*
* PARAMETER:    None
*
* RETURNVALUE:  tVoid
*
*************************************************************************/
tVoid tun_MsgToADR::vHandle_r_CM_AdditionalVersionInfo(tPU8 pu8Message) 
{
    ETG_TRACE_COMP(( " tun_MsgToADR::vHandle_r_CM_AdditionalVersionInfo()" ));
    if( pu8Message[ enAdrMsgOffset_OP_TYPE] == (tU8)enOpType_STATUS)
    {
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20

		m_u32ADRVersion = 0;
		m_u32ADRVersion = m_u32ADRVersion + pu8Message[ enCM_AdditionalVersionInf_idx_VersionDigit1] * 100;
		m_u32ADRVersion = m_u32ADRVersion + pu8Message[ enCM_AdditionalVersionInf_idx_VersionDigit2] * 10;
		m_u32ADRVersion += pu8Message[ enCM_AdditionalVersionInf_idx_VersionDigit3];

		if(m_u32ADRVersion < 330)
		{
			TUN_ASSERT_RETURN( m_potun_main != NULL);
			tFloat fADRVersion =(tFloat)( (tFloat)m_u32ADRVersion/(tFloat)100.0);
			 ETG_TRACE_COMP(( " tun_MsgToADR::vHandle_r_CM_AdditionalVersionInfo() ->\n"
				 "***********************************************\n"
				 "=>		ADR VERSION NOT COMPATIBLE			*\n"
				 "=>			 REQUIRED API 3.30				*\n"
				 "=>			 RECEIVED API %1.2f				*\n"
				 "=>			TUNER WILL SHUTDOWN				*\n"
				 "***********************************************\n"
                 ,fADRVersion
				 ));
			m_potun_main->vTunerShutDownProcess();
		}
#endif
        TUN_ASSERT_RETURN( m_potun_Utility != NULL);
        tU16 u16VersionNumber = m_potun_Utility->u16Convert2SeparateBytesTo16Bit( &pu8Message[ enCM_AdditionalVersionInf_idx_VersionNumber] );
        ETG_TRACE_COMP(( " tun_MsgToADR::vHandle_r_CM_AdditionalVersionInfo() ->\n"
                       " u16VersionNumber                  : %x\n"
                      , u16VersionNumber
                    ));
      tChar sVersionNo[ADR_SW_CUSTOMER_ADDITIONALVERSION_LENGTH];
      OSAL_s32PrintFormat(sVersionNo, "%x", u16VersionNumber );
      ETG_TRACE_COMP(( " tun_MsgToADR::vHandle_r_CM_AdditionalVersionInfo() ->\n"
                       " sVersionNo                  : %16s\n"
                      , sVersionNo
                    ));
      dp_tclregVersionsAdrSwVersionCustomer oAdrSwVersionCustomer;
      tString sSwCustomerVersion = sVersionNo;
      ETG_TRACE_COMP(( " tun_MsgToADR::vHandle_r_CM_AdditionalVersionInfo() ->\n"
                       " sSwCustomerVersion                  : %s\n"
                      , sSwCustomerVersion
                    ));
                    
      //oAdrSwVersionCustomer.vSetData( (tString)sSwCustomerVersion, ADR_SW_CUSTOMER_ADDITIONALVERSION_LENGTH );
      char sCustomerVersion[ADR_SW_CUSTOMER_ADDITIONALVERSION_LENGTH] = {0};
      tU8 u8DPRead = oAdrSwVersionCustomer.u8GetData(sCustomerVersion , ADR_SW_CUSTOMER_ADDITIONALVERSION_LENGTH);
      ETG_TRACE_USR2(( " tun_MsgToADR::vHandle_r_CM_AdditionalVersionInfo , DP read status : %d ", u8DPRead));

      if(strcmp(sCustomerVersion , sVersionNo))
      {
    	  ETG_TRACE_USR2(( " tun_MsgToADR::vHandle_r_CM_AdditionalVersionInfo , CustomerVersion changed "));
        oAdrSwVersionCustomer.vSetData( (tString)sSwCustomerVersion, ADR_SW_CUSTOMER_ADDITIONALVERSION_LENGTH );
      }
    }
}
/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendCM_AdditionalVersionInfo( )
*
* DESCRIPTION:  Requests the version number
*
* PARAMETER:    None
*
* RETURNVALUE:  tVoid
*
*************************************************************************/
tVoid tun_MsgToADR::vSendCM_AdditionalVersionInfo( tVoid)
{
  ETG_TRACE_COMP(( " tun_MsgToADR::vSendCM_AdditionalVersionInfo() ->" ));
  TUN_ASSERT_RETURN( m_potun_Utility != NULL);
  /*==========================================================================
   * Read the version ID from ADR by CM_AdditionalVersionInfo
   *------------------------------------------------------------------------*/
  tU8 au8Message[ TUN_CM_ADDITIONALVERSIONINFO_GET_LEN];
  vFormat_Msg_Header(au8Message, (tU16)enClientId_AmFmTuner, (tU16)enFBlockId_AmFmTuner,(tU8)enInstanceId_All,(tU16)enMsg_CM_AdditionalVersionInfo, (tU8)enOpType_GET);
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_DATA_LEN], 0x0000);
	/* Message to be sent is put in Que   */
  vWaitOnMessageQue( au8Message, TUN_CM_ADDITIONALVERSIONINFO_GET_LEN );
}

tVoid tun_MsgToADR::vFormat_Msg_Header(tU8* pu8Message, tU16 u16ClientID, tU16 u16FBlockID, tU8 u8InstanceID, tU16 u16FktID, tU8 u8Optype) const
{
	ETG_TRACE_USR2(( " tun_MsgToADR::vFormat_Msg_Header" ));
	if(m_potun_Utility != NULL)
	{
	   m_potun_Utility->vConvert16BitTo2SeparateBytes(&pu8Message[ enAdrMsgOffset_CLIENT_ID], u16ClientID);
	   m_potun_Utility->vConvert16BitTo2SeparateBytes(&pu8Message[ enAdrMsgOffset_FBLOCK_ID], u16FBlockID);
	   pu8Message[ enAdrMsgOffset_INSTANCE_ID] = (tU8)u8InstanceID;
	   m_potun_Utility->vConvert16BitTo2SeparateBytes(&pu8Message[ enAdrMsgOffset_FKT_ID], u16FktID);
	   pu8Message[ enAdrMsgOffset_OP_TYPE] = (tU8)u8Optype;
	}
}


/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vPrintADRVersionInformation( )
*
* DESCRIPTION:  Prints ADR VERSIONS 
*
* PARAMETER:    None
*
* RETURNVALUE:  tVoid
*
* AUTHOR : vnd4kor 06 Oct 2015
*************************************************************************/
tVoid tun_MsgToADR::vPrintADRVersionInformation()
{
#ifndef VARIANT_S_FTR_ENABLE_DP_CLIENT_MOCK
	TUN_ASSERT_RETURN( m_potun_main != NULL);
	TUN_ASSERT_RETURN( m_potun_main->m_poclTmcTuner_AdditionalVersionInfo_Update != NULL);

	ETG_TRACE_USR2(( " tun_MsgToADR::ADR VERSIIONS\n"
		"*****************************************************************************************\n"
		"=>	ADR HW VERSION		= %64s\n"
		"=>	ADR SW VERSION		= %64s\n"
		"=>	ADR API NUMBER		= %1.2f\n"
		"*****************************************************************************************\n"
		,m_sADRHWVersion
		,m_sADRSWVersion
		,m_potun_main->m_poclTmcTuner_AdditionalVersionInfo_Update->fGetAdrVersionInfo()));
#endif
}

tVoid tun_MsgToADR::vSendNotificationRequest(tU8 u8NotificationControl, tU16 u16ClientID, tU16 u16FBlockID)
{
	ETG_TRACE_USR2(( " tun_MsgToADR::vSendNotificationRequest "));
	//tU8   u8aMessage[ 20];

	    if (m_potun_Utility!=NULL)
	    {


	      if(u16ClientID == (tU16)enClientId_AmFmTuner )
	      {
	    	  tU8   u8aMessage[ TUN_NOTIFICATION_AMFM_SET_LEN];
	    	  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_CLIENT_ID], u16ClientID);
	    	  	      m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FBLOCK_ID], u16FBlockID);
	    	  	      u8aMessage[ enAdrMsgOffset_INSTANCE_ID] = (tU8)enInstanceId_All;
	    	  	      m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_FKT_ID], (tU16)enMsg_Notification);
	    	  	      u8aMessage[ enAdrMsgOffset_OP_TYPE] = (tU8)enOpType_SET;
	    	  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA_LEN], 0x0003);
	    	  u8aMessage[ enAdrMsgOffset_DATA] = u8NotificationControl;
	    	  	      m_potun_Utility->vConvert16BitTo2SeparateBytes( &u8aMessage[(tU8)enAdrMsgOffset_DATA + 1], u16ClientID);

	    	  	    vWaitOnMessageQue( u8aMessage, TUN_NOTIFICATION_AMFM_SET_LEN );
	    	  	  vSetNotificationSentStatus(true);
	      }
	      else
	      {
	    	  tU8   u8aMessage1[ TUN_NOTIFICATION_TMC_SET_LEN];
	    	 	    	  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage1[ enAdrMsgOffset_CLIENT_ID], u16ClientID);
	    	 	    	  	      m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage1[ enAdrMsgOffset_FBLOCK_ID], u16FBlockID);
	    	 	    	  	      u8aMessage1[ enAdrMsgOffset_INSTANCE_ID] = (tU8)enInstanceId_All;
	    	 	    	  	      m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage1[ enAdrMsgOffset_FKT_ID], (tU16)enMsg_Notification);
	    	 	    	  	      u8aMessage1[ enAdrMsgOffset_OP_TYPE] = (tU8)enOpType_SET;
	    	  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage1[ enAdrMsgOffset_DATA_LEN], 0x0007);
	    	  u8aMessage1[ enAdrMsgOffset_DATA] = (tU8)enNotificationControl_SetFunction;
	    	  	    	  	      m_potun_Utility->vConvert16BitTo2SeparateBytes( &u8aMessage1[(tU8)enAdrMsgOffset_DATA + 1], u16ClientID);
	    	  m_potun_Utility->vConvert16BitTo2SeparateBytes( &u8aMessage1[(tU8)enAdrMsgOffset_DATA + 3], 0x0F01);
	    	  m_potun_Utility->vConvert16BitTo2SeparateBytes( &u8aMessage1[(tU8)enAdrMsgOffset_DATA + 5], 0x0F02);
	    	 
	    	  vWaitOnMessageQue( u8aMessage1, TUN_NOTIFICATION_TMC_SET_LEN );

	      }

	    }

	    /* Message to be sent is put in Que   */
	    //vWaitOnMessageQue( u8aMessage, 20 );
}

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendStationNameFix( )
* 
* DESCRIPTION:  Send request to ADR to freeze or unfreeze the PSName
*
* PARAMETER:    None
*
* RETURNVALUE:  tVoid 
*
*************************************************************************/
tVoid tun_MsgToADR::vSendStationNameFix(tBool bPSFreeze)
{
	 TUN_ASSERT_RETURN( m_potun_Utility != NULL);

	  tU8 u8aMessage[ TUN_CM_STATIONNAMEFIX_SETGET_LEN ];

	  // send message CM_StationNameFix

	  vFormat_Msg_Header(u8aMessage, (tU16)enClientId_AmFmTuner, (tU16)enFBlockId_AmFmTuner,(tU8)enInstanceId_AudioSource1,(tU16)enMsg_CM_StationNameFix, (tU8)enOpType_SETGET);

	  m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA_LEN], 0x0001);

	  u8aMessage[ enAdrMsgOffset_DATA] = bPSFreeze;
	  

	  ETG_TRACE_USR2(( " tun_MsgToADR::vSendStationNameFix -> ,PSFixOnorOff=%u "
			    ,u8aMessage[ enAdrMsgOffset_DATA]
			  ));
	  
	  /* Message to be sent is put in Que   */
	  vWaitOnMessageQue( u8aMessage, TUN_CM_STATIONNAMEFIX_SETGET_LEN );
}

 /*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendSetRDSSwitch( )
* 
* DESCRIPTION:  Send request to ADR to set RDS switch ON/OFF
*
* PARAMETER:    u32RDSMode
*
* RETURNVALUE:  tVoid 
*
* AUTHOR vnd4kor
*************************************************************************/
 tVoid tun_MsgToADR::vSendSetRDSSwitch(tU32 u32RDSMode)
 {
	 TUN_ASSERT_RETURN( m_potun_Utility != NULL);

	 tU8   u8aMessage[ TUN_CM_RDSMODE_SETGET_LEN];

	 /*Send RDSSwitch[SetGet] OnOff*/
	 vFormat_Msg_Header(u8aMessage, (tU16)enClientId_AmFmTuner, (tU16)enFBlockId_AmFmTuner,(tU8)enInstanceId_AudioSource1,(tU16)enMsg_RDSSwitch, (tU8)enOpType_SETGET);

	 m_potun_Utility->vConvert16BitTo2SeparateBytes(\
					&u8aMessage[ enAdrMsgOffset_DATA_LEN], 0x0001);

	 u8aMessage[ enAdrMsgOffset_DATA] = ( ( u32RDSMode & (tU32)enTUN_FEATURE_RDS_AUTO_MODE)) ?\
												(tU8)enFeatureSwitchON : (tU8)enFeatureSwitchOFF;

	 ETG_TRACE_USR2(( " tun_MsgToADR::vSendSetRDSSwitch -> , RDSAutoMode=0x%02x"
		 ,u8aMessage[10]
	 ));

	 /* Message to be sent is put in Que   */
	 vWaitOnMessageQue( u8aMessage, TUN_CM_RDSMODE_SETGET_LEN );
 }
 /*************************************************************************
 *
 * FUNCTION:     tun_MsgToADR:: vSendAutostore( )
 *
 * DESCRIPTION:  Send request to ADR to do autostore
 *
 * PARAMETER:    u32ListID, u8MaxNoOfStations, enUnUsedPresets,
 *               enSortingCriteria, u8BehaviouralSwitch
 *
 * RETURNVALUE:  tVoid
 *
 * History
 * 9/3/2013  | initial version  | ngp1kor, pmn3kor
 *************************************************************************/
 tVoid tun_MsgToADR::vSendAutostore(tU16 u16ListID, tU8 u8MaxNoOfStations, \
                                    tenAMFMAutostore_UnUsedPreset enUnUsedPresets,\
                                    tenAMFMAutostore_Sort enSortingCriteria, \
                                    tU8 u8BehaviouralSwitch)
 {
     ETG_TRACE_USR1(("tun_MsgToADR::vSendAutostore -> start autostore"));
     TUN_ASSERT_RETURN( m_potun_Utility != NULL);

     /* TUN_CM_AUTOSTORE_STARTRESULT_LEN got by adding header size and payload for autostore*/
     tU8 u8aMessage[TUN_CM_AUTOSTORE_STARTRESULT_LEN];
     /*send Autostore[StartResult] gives info about stations*/
	 vFormat_Msg_Header(u8aMessage, (tU16)enClientId_AmFmTuner, (tU16)enFBlockId_AmFmTuner,(tU8)enInstanceId_AudioSource1,(tU16)enMsg_CM_Autostore, (tU8)enOpType_STARTRESULT);
     
     /**data length is payload size of autostore*/
     m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA_LEN], TUN_CM_AUTOSTORE_DATA_LEN);

     /**List ID updating*/
     m_potun_Utility->vConvert16BitTo2SeparateBytes( &u8aMessage[enAdrMsgOffset_DATA], u16ListID);

     u8aMessage[(tU8)enAdrMsgOffset_DATA+(tU8)enCM_Autostore_idx_MaxNumberOfStations] = \
															u8MaxNoOfStations;
     u8aMessage[(tU8)enAdrMsgOffset_DATA+(tU8)enCM_Autostore_idx_UnusedPresetTreatment] = \
															(tU8)enUnUsedPresets;
     u8aMessage[(tU8)enAdrMsgOffset_DATA+(tU8)enCM_Autostore_idx_SortingCriteria] = \
															(tU8)enSortingCriteria;
     u8aMessage[(tU8)enAdrMsgOffset_DATA+(tU8)enCM_Autostore_idx_BehaviouralSwitches] = \
															u8BehaviouralSwitch;

     ETG_TRACE_USR1(( " tun_MsgToADR::vSendAutostore -> send the message to queue"));

     vWaitOnMessageQue( u8aMessage, TUN_CM_AUTOSTORE_STARTRESULT_LEN );








 }

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vGetRDSMode( )
* 
* DESCRIPTION:  Gets the current RDS switch status
*
* PARAMETER:    None
*
* RETURNVALUE:  tVoid 
*
* AUTHOR: vnd4kor
*************************************************************************/
 tVoid tun_MsgToADR::vGetRDSMode()
 {
	   tU8   u8aMessage[ TUN_CM_ONLY_HEADER];
  
  /* Send RDS[Get] */
  if (m_potun_Utility!=NULL)
  {
     vFormat_Msg_Header(u8aMessage, (tU16)enClientId_AmFmTuner, (tU16)enFBlockId_AmFmTuner,(tU8)enInstanceId_AudioSource1,(tU16)enMsg_RDSSwitch, (tU8)enOpType_GET);
     m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA_LEN], 0x0000);  
  }
  ETG_TRACE_USR2(( " tun_MsgToADR::vGetRDSMode "));
  
  /* Message to be sent is put in Que   */
  vWaitOnMessageQue( u8aMessage, TUN_CM_ONLY_HEADER );
 }

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendDeletePreset( )
* 
* DESCRIPTION:  Sends command to adr to delete single element in preset list
*
* PARAMETER:    tU16 u16ListID, tU8 u8ListElementID
*
* RETURNVALUE:  tVoid 
*
* AUTHOR: vnd4kor : 6/9/2013
*************************************************************************/
 tVoid tun_MsgToADR::vSendDeleteSinglePreset ( tU16 u16ListID, tU8 u8ListElementID )
 {
	 TUN_ASSERT_RETURN( m_potun_Utility != NULL);

	 tU8   u8aMessage[ TUN_CM_DEL_SINGLE_PRESET_SETGET_LEN];

	 /*Fill Send Delete Preset [SetGet] msg */
	 /** Client ID */
	  vFormat_Msg_Header(u8aMessage, (tU16)enClientId_AmFmTuner, (tU16)enFBlockId_AmFmTuner,(tU8)enInstanceId_All,u16ListID, (tU8)enOpType_SETGET);

	 /** Payload Length */
	 m_potun_Utility->vConvert16BitTo2SeparateBytes(\
					&u8aMessage[ enAdrMsgOffset_DATA_LEN],\
					TUN_CM_DEL_SINGLE_PRESET_DATA_LEN);


	 /** List element to be deleted */
	 u8aMessage[ enCM_PresetList_idx_PosX]    = u8ListElementID;

	 /** Set Frequency of element to '0' to delete it */
	 u8aMessage[ enCM_PresetList_idx_PosY]    = (tU8)enCM_PresetListPosY__Frequency;
     m_potun_Utility->vConvert32BitTo4SeparateBytes(&\
					u8aMessage[ enCM_PresetList_idx_Data], 0x00000000);  
	

	 ETG_TRACE_USR2(( " tun_MsgToADR::vSendDeletePreset -> , u16ListID=0x%04X,  u8ListElementID = 0x%02X",u16ListID,u8ListElementID ));

	 /* Message to be sent is put in Que   */
	 vWaitOnMessageQue( u8aMessage, TUN_CM_DEL_SINGLE_PRESET_SETGET_LEN );
 }

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendDeleteAllPreset( )
* 
* DESCRIPTION:  Sends command to adr to delete all elements in preset list
*
* PARAMETER:    tU16 u16ListID
*
* RETURNVALUE:  tVoid 
*
* AUTHOR: vnd4kor : 10/9/2013
*************************************************************************/
 tVoid tun_MsgToADR::vSendDeleteAllPreset ( tU16 u16ListID)
 {
	 TUN_ASSERT_RETURN( m_potun_Utility != NULL);
	 TUN_ASSERT_RETURN( m_potun_Config != NULL);
	 TUN_ASSERT_RETURN( m_poclConfigInterface != NULL);

	 tU8 u8NumOfPresets = 0;

	 tU8 u8Band = oGetADR3ToMidwTunerBand(u16ListID);
	 switch(u8Band)
	 {
	 case enAvailBand_FM:
		 {
			 	 u8NumOfPresets = (tU8)m_poclConfigInterface->u32GetConfigData("NumberOfPresetsPerBankFM");
		 }
		 break;
	 case enAvailBand_MW:
	 case enAvailBand_LW:
		 {
			 	 u8NumOfPresets = (tU8)m_poclConfigInterface->u32GetConfigData("NumberOfPresetsPerBankMW");
		 }
		 break;
	 case enAvailBand_GetCurrent:
		 {
			 	 u8NumOfPresets = (tU8)m_poclConfigInterface->u32GetConfigData("NumberOfPresetsPerBankForMixedAllBands");
		 }
			 break;
	 default:
		 {
			 ETG_TRACE_USR2(( " tun_MsgToADR::vSendDeleteAllPreset -> Error NumberofPresets = %d"\
				 ,u8NumOfPresets));
			 return;
		 }
	 }
	 /** Delete all Preset elements Msg Length */

tU16 u16PresetSetGetLength =  (tU16)(u8NumOfPresets * enCM_PresetList_len_Frequency + enCM_PresetList_idx_Data);
tU16 u16PresetDataLen = (tU16)(u8NumOfPresets * enCM_PresetList_len_Frequency  + enCM_PresetList_len_Pos);

	 tU8  *u8aMessage = new tU8[u16PresetSetGetLength];

	 if(u8aMessage != NULL)
	 {
	 /*Fill Send Delete Preset [SetGet] msg */
	  vFormat_Msg_Header(u8aMessage, (tU16)enClientId_AmFmTuner, (tU16)enFBlockId_AmFmTuner,(tU8)enInstanceId_All,u16ListID, (tU8)enOpType_SETGET);

	 /** Payload Length */
	 m_potun_Utility->vConvert16BitTo2SeparateBytes(\
					&u8aMessage[ enAdrMsgOffset_DATA_LEN],\
					u16PresetDataLen);


	 /** List element to be deleted */
	 u8aMessage[ enCM_PresetList_idx_PosX]    = CM_PresetListPosX_selectALL;

	 /** Set Frequency of all element to '0' to delete it */
	 u8aMessage[ enCM_PresetList_idx_PosY]    = (tU8)enCM_PresetListPosY__Frequency;

	 


	 for (tU8 u8Index = 0; u8Index != u8NumOfPresets; u8Index++)
	 {
		 m_potun_Utility->vConvert32BitTo4SeparateBytes(\
						/** Adding Header length */
						&u8aMessage[ (tU8)enCM_PresetList_idx_Data +\
						/** Adding 4 bytes of Frequency of each element */
						(u8Index * (tU8)enCM_PresetList_len_Frequency)],	
						/** Data : Frequency set to '0' to delete element */
						0x00000000);								

	 }
		

	 ETG_TRACE_USR2(( " tun_MsgToADR::vSendDeleteAllPreset -> , u16ListID=0x%04X "\
						,u16ListID));

	 /* Message to be sent is put in Que   */
	 try
	 {
		vWaitOnMessageQue( u8aMessage, u16PresetSetGetLength );
	 }
	 catch(std::bad_alloc)
	 {

	 }

	 delete[] u8aMessage;
	 }
 }
 
 /*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendSurveilTP( )
* 
* DESCRIPTION:  Start/Stop Surveillance
*
* PARAMETER:    tU16 u16PI, tBool bSurveilTp, tU8 u8Strategy
*
* RETURNVALUE:  tVoid 
*************************************************************************/
 tVoid tun_MsgToADR::vSendSurveilTP(tU16 u16PI, tBool bSurveilTp, tU8 u8Strategy)
 {
	 ETG_TRACE_COMP(( " tun_MsgToADR::vSendSurveilTP() -> u16PI=0x%x, SurveilTP=%d, Strategy = %d"
                  ,u16PI
                  ,bSurveilTp
				  ,u8Strategy
                ));

  TUN_ASSERT_RETURN( m_potun_Utility != NULL);
  tU8 au8Message[ TUN_CM_SURVEILTP_SET_LEN];
  
	/*CM_AtPresetSave should have a SetGet optype introduced*/
  vFormat_Msg_Header(au8Message, (tU16)enClientId_AmFmTuner, (tU16)enFBlockId_AmFmTuner,(tU8)enInstanceId_AudioSource1,(tU16)enMsg_CM_SurveilTp, (tU8)enOpType_SET);
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_DATA_LEN], 0x0004);
  m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_DATA], u16PI);
      
	au8Message[ (tU8)enAdrMsgOffset_DATA+2] = bSurveilTp;
	au8Message[ (tU8)enAdrMsgOffset_DATA+3] = u8Strategy;
    
   /* Message to be sent is put in Que   */
    vWaitOnMessageQue( au8Message, sizeof( au8Message));
 }

 /*************************************************************************
 *
 * FUNCTION:     tun_MsgToADR:: vSendSetEnggParam( )
 * 
 * DESCRIPTION:  Sends command to adr to set engg param value
 *
 * PARAMETER:    tU8 u8ParameterId,tS16 s16ParameterValue
 *
 * RETURNVALUE:  tVoid 
 *
 * AUTHOR: vnd4kor : 4.10.13
 *************************************************************************/
 tVoid tun_MsgToADR::vSendSetEnggParam(tU8 u8ParameterId,tU32 u32ParameterValue)
 {
	 ETG_TRACE_USR4(( " tun_MsgToADR::vSendSetEnggParam()"));

	 TUN_ASSERT_RETURN( m_potun_Utility != NULL);
	 tU8 au8Message[ TUN_CM_SETENGGPARAM_SETGET_LEN ];

	 /** Fill Send CM_SetEngineeringParameter [SETGET] Msg */
	 vFormat_Msg_Header(au8Message,\
		 (tU16)enClientId_AmFmTuner,\
		 (tU16)enFBlockId_AmFmTuner,\
		 (tU8)enInstanceId_All,\
		 (tU16)enMsg_CM_SetEngineeringParameter,\
		 (tU8)enOpType_SETGET);

	 m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_DATA_LEN],\
		 TUN_CM_SETENGGPARAM_SETGET_DATA_LEN);

	 /** Pack Data to be sent to ADR*/
	 au8Message[ enCM_SetEnggParam_idx_ParameterId ] = u8ParameterId;
	 m_potun_Utility->vConvert32BitTo4SeparateBytes(\
		 &au8Message[ enCM_SetEnggParam_idx_ParameterValue], \
		 u32ParameterValue);

	 /* Message to be sent is put in Que   */
	 vWaitOnMessageQue( au8Message, sizeof( au8Message));

 }
/*************************************************************************
 *
 * FUNCTION:     tun_MsgToADR:: vGetEngineeringParameter( )
 * 
 * DESCRIPTION:  Sends command to adr to get engg param value
 *
 * PARAMETER:    tU8 u8ParameterId
 *
 * RETURNVALUE:  tVoid 
 *
 * AUTHOR: vnd4kor : 20.11.13
 *************************************************************************/
 tVoid tun_MsgToADR::vGetEngineeringParameter(tU8 u8ParameterId)
{
	 ETG_TRACE_USR4(( " tun_MsgToADR::vGetEngineeringParameter()"));

	 TUN_ASSERT_RETURN( m_potun_Utility != NULL);
	 tU8 au8Message[ TUN_CM_SETENGGPARAM_GET_LEN ];

	 /** Fill Send CM_SetEngineeringParameter [GET] Msg */
	 vFormat_Msg_Header(au8Message,\
		 (tU16)enClientId_AmFmTuner,\
		 (tU16)enFBlockId_AmFmTuner,\
		 (tU8)enInstanceId_All,\
		 (tU16)enMsg_CM_SetEngineeringParameter,\
		 (tU8)enOpType_GET);

	 m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_DATA_LEN],\
		 TUN_CM_SETENGGPARAM_GET_DATA_LEN );

	 /** Pack Data to be sent to ADR*/
	 au8Message[ enCM_SetEnggParam_idx_ParameterId ] = u8ParameterId;
	 
	 /* Message to be sent is put in Que   */
	 vWaitOnMessageQue( au8Message, sizeof( au8Message));

}
 /*************************************************************************
 *
 * FUNCTION:     tun_MsgToADR:: vSendSetTuningActionConfig( )
 * 
 * DESCRIPTION:  Sends command to adr to set tuning Action config parameters
 *
 * PARAMETER:    tU8 u8ConfigParamType,tU32 u32ConfigParamValue
 *
 * RETURNVALUE:  tVoid 
 *
 * AUTHOR: vnd4kor : 8.10.13
 *************************************************************************/
 tVoid tun_MsgToADR::vSendSetTuningActionConfig(tU8 u8ConfigParamType,tU32 u32ConfigParamValue)
 {
	 ETG_TRACE_USR4(( " tun_MsgToADR::vSendSetTuningActionConfig()"));
	 TUN_ASSERT_RETURN( m_potun_Utility != NULL);

	 tU8 au8Message[ TUN_CM_TUN_ACTION_CONFIG_FOUR_BYTE_SET_LEN ];

	  /** Fill Send CM_TuningActionConfiguration [SET] Msg */
	 vFormat_Msg_Header(au8Message,\
		 (tU16)enClientId_AmFmTuner,\
		 (tU16)enFBlockId_AmFmTuner,\
		 (tU8)enInstanceId_All,\
		 (tU16)enMsg_CM_TuningActionConfiguration,\
		 (tU8)enOpType_SET);

	au8Message[enCM_TuningActionConfiguration_idx_PosX] = u8ConfigParamType;
	au8Message[enCM_TuningActionConfiguration_idx_PosY] = 0;
	
	 switch(u8ConfigParamType)
	 {
	 case TAC_PTYFilter:
		 {
			 /** Payload length = 6 bytes , Pos = 2bytes , Data = 4 bytes */
			m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_DATA_LEN],\
			TUN_CM_TUN_ACTION_CONFIG_FOUR_BYTE_DATA_LEN );

			/** Assign 4 byte PTYFilter value */
			m_potun_Utility->vConvert32BitTo4SeparateBytes( \
				&au8Message[ enCM_TuningActionConfiguration_idx_Data ], u32ConfigParamValue );

		 }
		 break;
	 case TAC_AutoSeekStepDelay:
	 case TAC_ManualStepDelay:
		 {
			 /** Payload length = 4 bytes , Pos = 2bytes , Data = 2 bytes */
			m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_DATA_LEN],\
			TUN_CM_TUN_ACTION_CONFIG_TWO_BYTE_DATA_LEN );

			/** Assign 2 byte AutoSeekStepDelay/ManualStepDelay value */
			au8Message[ enCM_TuningActionConfiguration_idx_Data ] = (tU8)u32ConfigParamValue;

		 }
		 break;
	 default:
		 {
			 /** Payload length = 3 bytes , Pos = 2bytes , Data = 1 byte */
			m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_DATA_LEN],\
			TUN_CM_TUN_ACTION_CONFIG_DEFAULT_DATA_LEN );

			/** Assign 1 byte data value */
			au8Message[ enCM_TuningActionConfiguration_idx_Data ] = (tU8)u32ConfigParamValue;

		 }
		 break;
	 }
	

	 /* Message to be sent is put in Que   */
	 vWaitOnMessageQue( au8Message, sizeof( au8Message));

 }

  /*************************************************************************
 *
 * FUNCTION:     tun_MsgToADR:: vGetTuningActionConfigValues( )
 * 
 * DESCRIPTION:  Sends command to adr to get tuning Action config parameter values
 *
 * PARAMETER:    tU8 u8ConfigParamType
 *
 * RETURNVALUE:  tVoid 
 *
 * AUTHOR: vnd4kor : 8.10.13
 *************************************************************************/
 tVoid tun_MsgToADR::vGetTuningActionConfigValues(tU8 u8ConfigParamType)
 {
	 ETG_TRACE_USR4(( " tun_MsgToADR::vGetTuningActionConfigValues()"));
	 TUN_ASSERT_RETURN( m_potun_Utility != NULL);

	 tU8 au8Message[ TUN_CM_TUN_ACTION_CONFIG_GET_LEN ];

	  /** Fill Send CM_TuningActionConfiguration [GET] Msg */
	 vFormat_Msg_Header(au8Message,\
		 (tU16)enClientId_AmFmTuner,\
		 (tU16)enFBlockId_AmFmTuner,\
		 (tU8)enInstanceId_All,\
		 (tU16)enMsg_CM_TuningActionConfiguration,\
		 (tU8)enOpType_GET);

	 /** Payload */
	 m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_DATA_LEN],\
			TUN_CM_TUN_ACTION_CONFIG_GET_DATA_LEN );

    /** Fill Msg */
	au8Message[enCM_TuningActionConfiguration_idx_PosX] = u8ConfigParamType;
	au8Message[enCM_TuningActionConfiguration_idx_PosY] = 0;

    /* Message to be sent is put in Que   */
	vWaitOnMessageQue( au8Message, sizeof( au8Message));


 }

 /*************************************************************************
 *
 * FUNCTION:     tun_MsgToADR:: vSendTestTuneFrequency( )
 * 
 * DESCRIPTION:  Sends command to adr to select AF List Element
 *
 * PARAMETER:    tU8 u8PhysicalTunerId,tU8 u8Mode,tU32 u32Frequency
 *
 * RETURNVALUE:  tVoid 
 *
 * AUTHOR: vnd4kor : 21.10.13
 *************************************************************************/
 tVoid tun_MsgToADR::vSendTestTuneFrequency(tU8 u8PhysicalTunerId,\
											tU8 u8Mode,tU32 u32Frequency)
 {
	 ETG_TRACE_USR4(( " tun_MsgToADR::vSendTestTuneFrequency()"));
	 TUN_ASSERT_RETURN( m_potun_Utility != NULL);

	 tU8 au8Message[ TUN_CM_TEST_TUNE_FREQUENCY_SET_LEN ];

	  /** Fill Send enMsg_CM_ATTestTuneFrequency [SET] Msg */
	 vFormat_Msg_Header(au8Message,\
		 (tU16)enClientId_AmFmTuner,\
		 (tU16)enFBlockId_AmFmTuner,\
		 (tU8)enInstanceId_All,\
		 (tU16)enMsg_CM_ATTestTuneFrequency,\
		 (tU8)enOpType_SET);

	 /** Payload */
	 m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_DATA_LEN],\
			TUN_CM_TEST_TUNE_FREQUENCY_SET_DATA_LEN );

    /** Fill Msg */
	 /** Physical tuner Id to which the frequency must be tuned */
	au8Message[enCM_ATTestTuneFrequency_idx_PhysicalTunerID] = u8PhysicalTunerId;
	/** Mode to be used (AF_JUMP ON/OFF) */
	au8Message[enCM_ATTestTuneFrequency_idx_Mode] = u8Mode;
	/** Frequency to be tuned */
	m_potun_Utility->vConvert32BitTo4SeparateBytes(&au8Message[ \
					enCM_ATTestTuneFrequency_idx_Frequency],\
					u32Frequency );


    /* Message to be sent is put in Que   */
	vWaitOnMessageQue( au8Message, sizeof( au8Message));

 }

 /*************************************************************************
 *
 * FUNCTION:     tun_MsgToADR:: vSendSetAnnouncementSwitch( )
 * 
 * DESCRIPTION:  Sends command to adr to Set TA/PTY switch to ON/OFF
 *
 * PARAMETER:    tU8 u8AnnouncementType
 *
 * RETURNVALUE:  tVoid 
 *
 * AUTHOR: vnd4kor : 28.10.13
 *************************************************************************/
 tVoid tun_MsgToADR::vSendSetAnnouncementSwitch(tU8 u8AnnouncementType)
 {
	 ETG_TRACE_USR4(( " tun_MsgToADR::vSendSetAnnouncementSwitch()"));
	 TUN_ASSERT_RETURN( m_potun_Utility != NULL);

	 tU8 au8Message[ TUN_CM_ANNOUNCEMENT_SWITCH_SETGET_LEN ];

	  /** Fill Send enMsg_CM_AnnouncementSwitch [SETGET] Msg */
	 vFormat_Msg_Header(au8Message,\
		 (tU16)enClientId_AmFmTuner,\
		 (tU16)enFBlockId_AmFmTuner,\
		 (tU8)enInstanceId_AudioSource1,\
		 (tU16)enMsg_CM_AnnouncementSwitch,\
		 (tU8)enOpType_SETGET);

	 /** Payload */
	 m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_DATA_LEN],\
			TUN_CM_ANNOUNCEMENT_SWITCH_SETGET_DATA_LEN );

    /** Fill Msg */
	/** Announcement Type to be switched on */
	au8Message[enAdrMsgOffset_DATA ] = u8AnnouncementType;
	

    /* Message to be sent is put in Que   */
	vWaitOnMessageQue( au8Message, sizeof( au8Message));


 }

 /*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendGetAnnouncementSwitch( )
* 
* DESCRIPTION:  Gets the current Announcement switch status
*
* PARAMETER:    None
*
* RETURNVALUE:  tVoid 
*
* AUTHOR: vnd4kor 29/10/2013
*************************************************************************/
 tVoid tun_MsgToADR::vSendGetAnnouncementSwitch()
 {
   tU8   u8aMessage[ TUN_CM_ONLY_HEADER];
  
  /* Send AnnouncementSwitch[Get] */
  if (m_potun_Utility!=NULL)
  {
     vFormat_Msg_Header(u8aMessage, (tU16)enClientId_AmFmTuner, (tU16)enFBlockId_AmFmTuner,(tU8)enInstanceId_AudioSource1,(tU16)enMsg_CM_AnnouncementSwitch, (tU8)enOpType_GET);
     m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA_LEN], 0x0000);  
  }
  ETG_TRACE_USR2(( " tun_MsgToADR::vSendGetAnnouncementSwitch "));
  
  /* Message to be sent is put in Que   */
  vWaitOnMessageQue( u8aMessage, TUN_CM_ONLY_HEADER );
 }

 
 /*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendAnnouncementEscape( )
* 
* DESCRIPTION:  Send Cancel announcement request to ADR
*
* PARAMETER:    None
*
* RETURNVALUE:  tVoid 
*
* AUTHOR: vnd4kor 29/10/2013
*************************************************************************/
 tVoid tun_MsgToADR::vSendAnnouncementEscape(tU8 u8AnnouncementType)
 {
	 ETG_TRACE_USR4(( " tun_MsgToADR::vSendAnnouncementEscape()"));
	 TUN_ASSERT_RETURN( m_potun_Utility != NULL);

	 tU8 au8Message[ TUN_CM_ANNOUNCEMENT_ESCAPE_STARTRESULT_LEN ];

	  /** Fill Send enMsg_CM_AnnouncementEscape [STARTRESULT] Msg */
	 vFormat_Msg_Header(au8Message,\
		 (tU16)enClientId_AmFmTuner,\
		 (tU16)enFBlockId_AmFmTuner,\
		 (tU8)enInstanceId_AudioSource1,\
		 (tU16)enMsg_CM_AnnouncementEscape,\
		 (tU8)enOpType_STARTRESULT);

	 /** Payload */
	 m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_DATA_LEN],\
			TUN_CM_ANNOUNCEMENT_ESCAPE_STARTRESULT_DATA_LEN );

    /** Fill Msg */
	/** Announcement Type to be Cancelled */
	au8Message[enAdrMsgOffset_DATA ] = u8AnnouncementType;
	

    /* Message to be sent is put in Que   */
	vWaitOnMessageQue( au8Message, sizeof( au8Message));

 }

 /*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendSetPiPsRelation( )
* 
* DESCRIPTION:  Sets PI PS Relation 
*
* PARAMETER:    tU16 u16PI,tU8* sPSName
*
* RETURNVALUE:  tVoid 
*
* AUTHOR: vnd4kor 11/11/2013
*************************************************************************/
 tVoid tun_MsgToADR::vSendSetPiPsRelation(tU16 u16PI,tU8* sPSName)
 {
	 ETG_TRACE_USR4(( " tun_MsgToADR::vSendSetPiPsRelation()"));
	 TUN_ASSERT_RETURN( m_potun_Utility != NULL);

	 tU8 au8Message[ TUN_CM_PI_PS_RELATION_SET_LEN ];

	  /** Fill Send enMsg_CM_PiPsRelation [SETGET] Msg */
	 vFormat_Msg_Header(au8Message,\
		 (tU16)enClientId_AmFmTuner,\
		 (tU16)enFBlockId_AmFmTuner,\
		 (tU8)enInstanceId_AudioSource1,\
		 (tU16)enMsg_CM_PiPsRelation,\
		 (tU8)enOpType_SETGET);

	 /** Payload */
	 m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_DATA_LEN],\
			TUN_CM_PI_PS_RELATION_SET_DATA_LEN );

    /** Fill Msg */
	/** PI*/
	m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enCM_PiPsRelation_idx_PI],\
		u16PI);

	/** PS name */
	for(tU8 i=0;i<8;i++)
	{
		au8Message[ (tU8)enCM_PiPsRelation_idx_PSName + i] = sPSName[i];
	}

    /* Message to be sent is put in Que   */
	vWaitOnMessageQue( au8Message, sizeof( au8Message));

 }

 
 /*************************************************************************
*
* FUNCTION:     tun_MsgToADR::vSendSetPreparePresetStorage( )
* 
* DESCRIPTION:  Sets PS Freeze ON/OFF 
*
* PARAMETER:    tU8 u8Mode
*
* RETURNVALUE:  tVoid 
*
* AUTHOR: vnd4kor 11/11/2013
*************************************************************************/
 tVoid tun_MsgToADR::vSendSetPreparePresetStorage(tU8 u8Mode)
 {
	 ETG_TRACE_USR4(( " tun_MsgToADR::vSendSetPreparePresetStorage()"));
	 TUN_ASSERT_RETURN( m_potun_Utility != NULL);

	 tU8 au8Message[ TUN_CM_PREPARE_PRESET_STORAGE_SET_LEN  ];

	  /** Fill Send enMsg_CM_ATPreparePresetStorage [SET] Msg */
	 vFormat_Msg_Header(au8Message,\
		 (tU16)enClientId_AmFmTuner,\
		 (tU16)enFBlockId_AmFmTuner,\
		 (tU8)enInstanceId_AudioSource1,\
		 (tU16)enMsg_CM_ATPreparePresetStorage,\
		 (tU8)enOpType_SET);

	 /** Payload */
	 m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_DATA_LEN],\
			TUN_CM_PREPARE_PRESET_STORAGE_SET_DATA_LEN );

    /** Fill Msg */
	au8Message[ enAdrMsgOffset_DATA] = u8Mode;

    /* Message to be sent is put in Que   */
	vWaitOnMessageQue( au8Message, sizeof( au8Message));
	
 }
 
/*************************************************************************
*
* FUNCTION:     tun_MsgToADR::vSendSpeedInfo( )
* 
* DESCRIPTION:  Sets Speed information obtained from CSM_Speed from vehicle data 
*
* PARAMETER:    tU8 u8Speed
*
* RETURNVALUE:  tVoid 
*
* AUTHOR: vnd4kor 26/11/2013
*************************************************************************/
 tVoid tun_MsgToADR::vSendSpeedInfo(tU8 u8Speed)
 {
	 ETG_TRACE_USR4(( " tun_MsgToADR::vSendSpeedInfo()"));
	 TUN_ASSERT_RETURN( m_potun_Utility != NULL);

	 tU8 au8Message[ TUN_CM_SPEED_INFORMATION_SETGET_LEN  ];

	  /** Fill Send enMsg_CM_ATPreparePresetStorage [SET] Msg */
	 vFormat_Msg_Header(au8Message,\
		 (tU16)enClientId_AmFmTuner,\
		 (tU16)enFBlockId_AmFmTuner,\
		 (tU8)enInstanceId_All,\
		 (tU16)enMsg_CM_SpeedInformation,\
		 (tU8)enOpType_SETGET);

	 /** Payload */
	 m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_DATA_LEN],\
			TUN_CM_SPEED_INFORMATION_SETGET_DATA_LEN );

    /** Fill Msg */
	au8Message[ enAdrMsgOffset_DATA] = u8Speed;

    /* Message to be sent is put in Que   */
	vWaitOnMessageQue( au8Message, sizeof( au8Message));

 }

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR::vSend_GetRadiotext( )
* 
* DESCRIPTION:  Gets Current Radio text 
*
* PARAMETER:    None
*
* RETURNVALUE:  tVoid 
*
* AUTHOR: vnd4kor 9/12/2013
*************************************************************************/
 tVoid tun_MsgToADR:: vSend_GetRadiotext()
 {
	 ETG_TRACE_USR4(( " tun_MsgToADR::vSend_GetRadiotext()"));
	 TUN_ASSERT_RETURN( m_potun_Utility != NULL);

	 tU8 au8Message[ TUN_CM_CURRENTRADIOTEST_GET_LEN ];

	  /** Fill Send enMsg_CurrentRadiotext [GET] Msg */
	 vFormat_Msg_Header(au8Message,\
		 (tU16)enClientId_AmFmTuner,\
		 (tU16)enFBlockId_AmFmTuner,\
		 (tU8)enInstanceId_AudioSource1,\
		 (tU16)enMsg_CM_CurrentRadiotext,\
		 (tU8)enOpType_GET);

	 /** Payload = 0*/
	 m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_DATA_LEN],0);

    /* Message to be sent is put in Que   */
	vWaitOnMessageQue( au8Message, sizeof( au8Message));



 }

 /*************************************************************************
*
* FUNCTION:     tun_MsgToADR::vSendSetEvaluationParameter( )
* 
* DESCRIPTION:  Sets Evaluation Parameter value 
*
* PARAMETER:    tU8* puchData
*
* RETURNVALUE:  tVoid 
*
* AUTHOR: vnd4kor 25/3/2014
*************************************************************************/
 tVoid tun_MsgToADR::vSendSetEvaluationParameter(tU8* puchData)
 {

	 ETG_TRACE_USR4(( " tun_MsgToADR::vSendSetEvaluationParameter()"));
	 TUN_ASSERT_RETURN( m_potun_Utility != NULL);

	 tU8 au8Message[ TUN_CM_EVALUATION_PARAMETER_SETGET_LEN ];

	  /** Fill Send enMsg_CM_EvaluationParameter [GET] Msg */
	 vFormat_Msg_Header(au8Message,\
		 (tU16)enClientId_AmFmTuner,\
		 (tU16)enFBlockId_AmFmTuner,\
		 (tU8)enInstanceId_AudioSource1,\
		 (tU16)enMsg_CM_EvaluationParameter,\
		 (tU8)enOpType_SETGET);
	 	 /** Payload */
	 m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_DATA_LEN],\
			TUN_CM_EVALUATION_PARAMETER_SETGET_DATA_LEN );

	  ETG_TRACE_USR4(( " tun_MsgToADR::vSendSetEvaluationParameter() Received values : %d %d %d %d",
		  puchData[1],
		  puchData[2],
		  puchData[3],
		  puchData[4]
	  ));
	

	 au8Message[enCM_EvaluationParameter_Section] = puchData[1];
	 au8Message[enCM_EvaluationParameter_idx_ParameterId] = puchData[2];
	 au8Message[(tU8)enCM_EvaluationParameter_idx_ParameterId + 1] = puchData[3];
	 au8Message[enCM_EvaluationParameter_idx_ParameterValue] = puchData[4];

	 
    /* Message to be sent is put in Que   */
	vWaitOnMessageQue( au8Message, sizeof( au8Message));


 }


 /*************************************************************************
*
* FUNCTION:     tun_MsgToADR::vSendCancelAutostore( )
* 
* DESCRIPTION:  Sends cancel autostore request to ADR
*
* PARAMETER:    tVoid
*
* RETURNVALUE:  tVoid 
*
* AUTHOR: vnd4kor 13/5/2014
*************************************************************************/
 tVoid tun_MsgToADR::vSendCancelAutostore()
 {
	 ETG_TRACE_USR4(( " tun_MsgToADR::vSendCancelAutostore()"));
	 TUN_ASSERT_RETURN( m_potun_Utility != NULL);

	 tU8 au8Message[ TUN_CM_CANCEL_AUTOSTORE_SET_LEN ];

	  /** Fill Send enMsg_CancelAutostore [SET] Msg */
	 vFormat_Msg_Header(au8Message,\
		 (tU16)enClientId_AmFmTuner,\
		 (tU16)enFBlockId_AmFmTuner,\
		 (tU8)enInstanceId_AudioSource1,\
		 (tU16)enMsg_CM_CancelAutostore,\
		 (tU8)enOpType_SET);
	 	 /** Payload */
	 m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_DATA_LEN],\
			TUN_CM_CANCEL_AUTOSTORE_SET_DATA_LEN );
 
    /* Message to be sent is put in Que   */
	vWaitOnMessageQue( au8Message, sizeof( au8Message));


 }

 /*************************************************************************
*
* FUNCTION:     tun_MsgToADR::vSendSetUser( )
*
* DESCRIPTION:  selects a certain user
*
* PARAMETER:
*
* RETURNVALUE:  tVoid
*
* AUTHOR:
*************************************************************************/
 tVoid tun_MsgToADR::vSendSetUser(tU8 u8UserID)
 {

	 TUN_ASSERT_RETURN( m_potun_Utility != NULL);
	 TUN_ASSERT_RETURN( m_potun_main != NULL);

	 tU8 au8Message[ TUN_CM_SETUSER_LEN ];

	  /** Fill Send enMsg_CM_SetUser [GET] Msg */
	 vFormat_Msg_Header(au8Message,\
		 (tU16)enClientId_AmFmTuner,\
		 (tU16)enFBlockId_AmFmTuner,\
		 (tU8)enInstanceId_All,\
		 (tU16)enMsg_CM_SetUser,\
		 (tU8)enOpType_STARTRESULT);
	 	 /** Payload */

	 m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_DATA_LEN], TUN_CM_SETUSER_DATA_LEN);

	 ETG_TRACE_USR4(( " tun_MsgToADR::vSendSetUser() userID : %d" , u8UserID));

	 au8Message[enAdrMsgOffset_DATA] = u8UserID;

	 for(tU8 u8index = 1 ; u8index <= TUN_CM_SETUSER_DATA_LEN-1 ; u8index++)
	 {
		 au8Message[enAdrMsgOffset_DATA + u8index] = 0;
	 }

    /* Message to be sent is put in Que   */
	vWaitOnMessageQue( au8Message, sizeof( au8Message));

	if(bGetPersonalisationReq() == TRUE)
	{
		//m_potun_main->vTunerShutDownProcess();
	}
 }

 
  /*************************************************************************
*
* FUNCTION:     tun_MsgToADR::vSetAtBandSent( )
* 
* DESCRIPTION:  Setter for m_bAtBandSent
*
* PARAMETER:    tBool
*
* RETURNVALUE:  tVoid 
*
* AUTHOR: vnd4kor 21/5/2014
*************************************************************************/
  tVoid tun_MsgToADR::vSetAtBandSent(tBool bValue)
  {
	  m_bAtBandSent = bValue;
  }
/*************************************************************************
*
* FUNCTION:     tun_MsgToADR::bGetAtBandSent( )
* 
* DESCRIPTION:  Getter for m_bAtBandSent
*
* PARAMETER:    tVoid
*
* RETURNVALUE:  tBool 
*
* AUTHOR: vnd4kor 21/5/2014
*************************************************************************/
  tBool tun_MsgToADR::bGetAtBandSent()const
  {
	  return m_bAtBandSent;
  }

/*************************************************************************
* FUNCTION:     tun_MsgToADR::bGetAtBandSent( )
* DESCRIPTION:  Getter for m_u8AtBandRequested
* PARAMETER:    None
* RETURNVALUE:  tU8
* AUTHOR: gds5kor 31/3/2016
*************************************************************************/
  tU8 tun_MsgToADR::oGetAtBandRequested() const
  {
	  return m_u8AtBandRequested;
  }

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR::bGetNotificationSentStatus( )
*
* DESCRIPTION:  Getter for m_bNotificationSent
*
* PARAMETER:    tVoid
*
* RETURNVALUE:  tBool
*
* AUTHOR:
*************************************************************************/
tBool tun_MsgToADR::bGetNotificationSentStatus()const
{
	  return m_bNotificationSent;
}

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR::vSetNotificationSentStatus( )
*
* DESCRIPTION:  Setter for m_bNotificationSent
*
* PARAMETER:    tBool
*
* RETURNVALUE:  tVoid
*
* AUTHOR:
*************************************************************************/
 tVoid tun_MsgToADR::vSetNotificationSentStatus(tBool bValue)
 {
	 m_bNotificationSent = bValue;
 }
 

 /*************************************************************************
 *
 * FUNCTION:     tun_MsgToADR::bGetPersonalisationReq( )
 *
 * DESCRIPTION:  Getter for m_bPersonalisationReq
 *
 * PARAMETER:    tVoid
 *
 * RETURNVALUE:  tBool
 *
 * AUTHOR:
 *************************************************************************/
 tBool tun_MsgToADR::bGetPersonalisationReq()const
 {
 	  return m_bPersonalisationReq;
 }

 /*************************************************************************
 *
 * FUNCTION:     tun_MsgToADR::vSetPersonalisationReq( )
 *
 * DESCRIPTION:  Setter for m_bPersonalisationReq
 *
 * PARAMETER:    tBool
 *
 * RETURNVALUE:  tVoid
 *
 * AUTHOR:
 *************************************************************************/
  tVoid tun_MsgToADR::vSetPersonalisationReq(tBool bValue)
  {
	  m_bPersonalisationReq = bValue;
  }


/*************************************************************************
*
* FUNCTION:     tun_MsgToADR::vHandle_CM_SetUser( )
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE:  tVoid
*
* AUTHOR:
*************************************************************************/
tVoid tun_MsgToADR::vHandle_CM_SetUser()
{
	TUN_ASSERT_RETURN( m_ptun_MessageQ != NULL);
	TUN_ASSERT_RETURN( m_potun_main != NULL);

	if(bGetPersonalisationReq() == TRUE)
	{
		vSetPersonalisationReq(FALSE);
	}
	else
	{
		vDoNotification();
	}
}
tVoid tun_MsgToADR::vHandle_CM_GetUser(const tU8* pu8Message)
{
	if(pu8Message == NULL)
	{
			NORMAL_M_ASSERT( pu8Message != NULL );
		return;
	}
	m_u8CurrentUserId = pu8Message[enAdrMsgOffset_DATA];
	vStartupSetUserReq();
	ETG_TRACE_USR2(( "vHandle_CM_GetUser CurrentUserId=%d",m_u8CurrentUserId));
}
/*************************************************************************
*
* FUNCTION:     tun_MsgToADR::vStartupSetUserReq( )
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE:  tVoid
*
* AUTHOR:
*************************************************************************/
tVoid tun_MsgToADR::vStartupSetUserReq()
{
	TUN_ASSERT_RETURN( m_ptun_MessageQ != NULL);
	 dp_tclSrvIf oSrvEndUser;
	 tU8 u8UserID = 0xFF;
	 oSrvEndUser.s32GetEndUser(u8UserID);
	 ETG_TRACE_USR4(( " tun_MsgToADR::vStartupSetUserReq() userID : %d" , u8UserID));
#ifdef VARIANT_S_FTR_ENABLE_FEATURE_PSA_RCC
	 if((u8UserID <= 0x03) && (m_u8CurrentUserId!=u8UserID))
	 {
		 vSendSetUser(u8UserID);
	 }
	 else
	 {
		 vDoNotification();
	 }
#endif
#ifdef VARIANT_S_FTR_ENABLE_FEATURE_RNAIVI
	if((u8UserID != 0xFF) && (m_u8CurrentUserId != u8UserID))
	{
		ETG_TRACE_COMP(( " tun_MsgToADR::vStartupSetUserReq() ->" ));
		tU8   u8aMessage[ TUN_CM_ONLY_HEADER + 5];
		vFormat_Msg_Header(u8aMessage, (tU16)enClientId_AmFmTuner, (tU16)enFBlockId_AmFmTuner,(tU8)enInstanceId_AudioSource1,(tU16)0x0fec, (tU8)enOpType_SETGET);
		m_potun_Utility->vConvert16BitTo2SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA_LEN], 5);
		u8aMessage[enAdrMsgOffset_DATA] = u8UserID;
		if(u8UserID == 0)
		{
			m_potun_Utility->vConvert32BitTo4SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA+1], 2);
		}
		else
		{
			m_potun_Utility->vConvert32BitTo4SeparateBytes(&u8aMessage[ enAdrMsgOffset_DATA+1], 0);
		}
		vWaitOnMessageQue( u8aMessage, sizeof( u8aMessage) );
	}
	else
	{
		vDoNotification();
	}
#endif
}


/*************************************************************************
*
* FUNCTION:     tun_MsgToADR::vSet_bSetConfigurationMessageSent( )
*
* DESCRIPTION:  Setter for m_bSetConfigurationMessageSent
*
* PARAMETER:    tBool
*
* RETURNVALUE:  tVoid
*
* AUTHOR:
*************************************************************************/
 tVoid tun_MsgToADR::vSetDefaultSettingMsg(tBool bValue)
 {
	 if (m_ptun_MessageQ == NULL)
	 {
		 return;
	 }
	 bSetDefaultSettingMsgSent = bValue;
	 m_ptun_MessageQ -> vResetWaitingForConfirmation();
 }
 tBool tun_MsgToADR::blGetDefaultSettingMsgSent()
 {
	return bSetDefaultSettingMsgSent;
 }
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vSendSetPSD_Data( )
* 
* DESCRIPTION:  Set required PSD Data in CM_ ATHD_PSDDataFrame.
*
* PARAMETER:    tVoid
*
* RETURNVALUE:  tVoid 
*
* Author: Vinay ECV2 10/10/2014
*************************************************************************/
tVoid tun_MsgToADR::vSendSetPSD_Data()
{
	TUN_ASSERT_RETURN( m_potun_Utility != NULL);
	TUN_ASSERT_RETURN( m_poclConfigInterface != NULL);

	tU8 au8Message[ TUN_CM_ATHD_PSD_DATA_SET_LEN];

	/** fill header info */
	vFormat_Msg_Header(au8Message, (tU16)enClientId_AmFmTuner, (tU16)enFBlockId_AmFmTuner,\
		(tU8)enInstanceId_AudioSource1,enMsg_CM_ATHD_PSDDataFrame, (tU8)enOpType_SET);

	/** number of bytes in msg */
    m_potun_Utility->vConvert16BitTo2SeparateBytes(\
		&au8Message[ enAdrMsgOffset_DATA_LEN], sizeof( au8Message) - (tU8)enAdrMsgOffset_HEADER_LEN);

	/** Fill data */
    au8Message[ enCM_ATHD_PSDDataFrame_idx_ProgramBitMask] \
		= CM_ATHD_PSDDataFrame_PROG_BITMASK;

	m_potun_Utility->vConvert16BitTo2SeparateBytes(\
		&au8Message[ enCM_ATHD_PSDDataFrame_idx_FieldBitMask], (tU16)((tU16)m_poclConfigInterface->u32GetConfigData("PSDFieldBitMask") << 8));

	au8Message[ enCM_ATHD_PSDDataFrame_idx_CommentsBitMask] \
		= (tU8)m_poclConfigInterface->u32GetConfigData("HDPSDCommentsBitMask");

    au8Message[ enCM_ATHD_PSDDataFrame_idx_UFIDBitMask] \
		= (tU8)m_poclConfigInterface->u32GetConfigData("HDPSDUFIDBitMask");

    au8Message[ enCM_ATHD_PSDDataFrame_idx_CommersialBitMask] \
		= (tU8)m_poclConfigInterface->u32GetConfigData("HDPSDCommercialBitMask");

		
	ETG_TRACE_USR2(( " tun_MsgToADR::vSendSetPSD_Data Sent "));	


    vWaitOnMessageQue( au8Message, sizeof( au8Message));
}
#endif

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: bGetValue_m_bReadCheckSumAfterConfig();
*
* DESCRIPTION:  To get the value of the member variable m_bReadCheckSumAfterConfig().
*
* PARAMETER:    tVoid
*
* RETURNVALUE:  tBool
*
* Author:
*************************************************************************/
tBool tun_MsgToADR::bGetValue_m_bReadCheckSumAfterConfig()
{
	return m_bReadCheckSumAfterConfig;
}

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: bSetValue_m_bReadCheckSumAfterConfig(tBool bValue);
*
* DESCRIPTION:  To get the value of the member variable m_bReadCheckSumAfterConfig().
*
* PARAMETER:    tBool
*
* RETURNVALUE:  tVoid
*
* Author:
*************************************************************************/
tVoid tun_MsgToADR::vSetValue_m_bReadCheckSumAfterConfig(tBool bValue)
{
	m_bReadCheckSumAfterConfig = bValue;
}

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR:: vDoNotification();
*
* DESCRIPTION:  To notify the clients
*
* PARAMETER:    tVoid
*
* RETURNVALUE:  tVoid
*
* Author:
*************************************************************************/
tVoid tun_MsgToADR::vDoNotification()
{
		vNotifyFunctionIDs();
		/*request the TPSwitch Status*/
		vSendGetAnnouncementSwitch();
		// Get Seek Status
		vSendGetSeekStatus();
#ifdef VARIANT_S_FTR_ENABLE_FEATURE_RNAIVI
		// send DARC register output for japan region
		if((m_poclConfigInterface != NULL ) && (m_poclConfigInterface->u32GetConfigData("Destination_Region") == JAPAN_REGION))
		{
			vSendDARC_RegisterOutput();
		}
#endif
}
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
tVoid tun_MsgToADR::vSet_clAars_ATPresetList_XXPtr(clAars_ATPresetList_XX* poclAars_ATPresetList_XX)
  {
		if(NULL != poclAars_ATPresetList_XX)
	   {
		  m_poclAars_ATPresetList_XX = ( clAars_ATPresetList_XX* )poclAars_ATPresetList_XX;
	   }
  }
#endif

tVoid tun_MsgToADR::vGetDefaultPresetListAfterSetUser()
{
	if((NULL!=m_poclConfigInterface)&& (NULL!=m_potun_MsgToHMI))
	{
		tU16 U16adrListId;
		#ifdef VARIANT_S_FTR_ENABLE_FEATURE_PSA_RCC
			U16adrListId= 0x0f20;
		#else
			if ((tBool)m_poclConfigInterface->u32GetConfigData("PresetListType"))
				U16adrListId= 0x0f20;
			else 
				U16adrListId=U16GetADR3PresetListID((midw_fi_tcl_e8_Tun_TunerBand::tenType)m_potun_MsgToHMI->m_u8PrevBand);
		#endif
		#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
			if(NULL!=m_poclAars_ATPresetList_XX)
			{
				m_poclAars_ATPresetList_XX->vGetPresetList(U16adrListId);
			}
		#else
		vGetPresetList(U16adrListId);
		#endif
	}
}
tVoid tun_MsgToADR::vGetDefaultPresetListAfterDefaultSetting()
{
	if((NULL==m_poclConfigInterface) || (NULL==m_potun_MsgToHMI) ||(NULL==m_potun_Config))
	{
		return;
	}
	#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
		if((NULL==m_poclAars_ATPresetList_XX)|| (NULL==m_poclTmcTuner_Method_S_SET_ACTIVE_PRESETLIST))
		{
			return;
		}
	#endif
	#ifdef VARIANT_S_FTR_ENABLE_ACTIVEPRESETLIST_BANDCHG
		#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
			m_poclAars_ATPresetList_XX->vGetPresetList(m_poclTmcTuner_Method_S_SET_ACTIVE_PRESETLIST->U16GetADR3PresetListIDForSetActivePresetList());
		#else
			vGetPresetList(m_potun_MsgToHMI->u16GetListIDForSetActivePresetList());
		#endif
	#else
		if ((tBool)m_poclConfigInterface->u32GetConfigData("PresetListType") == FALSE)
		{
			/** if more than one preset lists are present in the same band, and if set active preset list is not used*/
			tU8 u8HMIListID = (tU8)oGetMidwTunerBandToPresetListID((midw_fi_tcl_e8_Tun_TunerBand::tenType)m_potun_MsgToHMI->m_u8PrevBand);
			#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
				m_poclAars_ATPresetList_XX->vGetPresetList(U16GetADR3PresetListID((midw_fi_tcl_e8_Tun_ListID::tenType)u8HMIListID));
			#else
				vGetPresetList(U16GetADR3PresetListID((midw_fi_tcl_e8_Tun_ListID::tenType)u8HMIListID));
			#endif
		}
		else
		{
			//#ifdef VARIANT_S_FTR_ENABLE_FEATURE_PSA_RCC
				#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
					m_poclAars_ATPresetList_XX->vGetPresetList(0x0f20);
				#else
					vGetPresetList(0x0f20);
				#endif
			//#endif
		}
	#endif
}

/*************************************************************************
*
* FUNCTION:     tun_MsgToADR::vSendDARC_RegisterOutput( )
*
* DESCRIPTION:  send DARC register output to receive DARC data
*
* PARAMETER:
*
* RETURNVALUE:  tVoid
*
* AUTHOR:
*************************************************************************/
tVoid tun_MsgToADR::vSendDARC_RegisterOutput()
{

	 TUN_ASSERT_RETURN( m_potun_Utility != NULL);
	 TUN_ASSERT_RETURN( m_potun_main != NULL);

	 tU8 au8Message[ TUN_CM_DARC_REGISTEROUTPUT_LEN ];

	  /** Fill Send enMsg_CM_DARC_RegisterOutput [SET] Msg */
	 vFormat_Msg_Header(au8Message,\
		 (tU16)enClientId_AmFmTuner,\
		 (tU16)enFBlockId_AmFmTuner,\
		 (tU8)enInstanceId_All,\
		 (tU16)enMsg_CM_DARC_RegisterOutput,\
		 (tU8)enOpType_SET);
	 	 /** Payload */

	 m_potun_Utility->vConvert16BitTo2SeparateBytes(&au8Message[ enAdrMsgOffset_DATA_LEN], TUN_CM_DARC_REGISTEROUTPUT_DATA_LEN);

	 ETG_TRACE_USR4(( " tun_MsgToADR::vSendDARC_RegisterOutput() " ));

	 au8Message[enAdrMsgOffset_DATA] = 0X01; // register data
	 au8Message[enAdrMsgOffset_DATA+1] = 0X00;
	 au8Message[enAdrMsgOffset_DATA+2] = 0X00;

   /* Message to be sent is put in Que   */
	vWaitOnMessageQue( au8Message, sizeof( au8Message));

}


