/************************************************************************
* FILE:           tun_Config.cpp
* PROJECT:        Ford H/L RNS
* SW-COMPONENT: 
*----------------------------------------------------------------------
*
* DESCRIPTION:    Tuner component specific configuration.
*              
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
* HISTORY:      
* Date      | Author                       | Modification
* 20.01.05  | CM-DI/ESA2 ( RBIN ) Dinesh   | Initial version.
*
*************************************************************************/



#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#include "tun_trace.h"

#include "tun_PrjConfig.h"
#include "tun_Config.h"

#include "tun_Utility.h"
#include "tun_Msg_types.h"
#include "tun_defines.h"
#include "tun_MsgToADR.h"

#include "kdsconfig/clConfigInterface.h"
#include "tun_defines.h"

/**Ngp1kor: Currently commented for migration to G3 Platform.*/
#ifndef VARIANT_S_FTR_ENABLE_DP_CLIENT_MOCK
   #define DP_S_IMPORT_INTERFACE_FI
   #include "dp_generic_if.h"
#else
   #include"diaglib_if.h"
#endif
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TUN_TRACE_CLASS_KDSCONFIG
#include "trcGenProj/Header/tun_Config.cpp.trc.h"
#endif

#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_TUNERFI_TYPES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_TUNERFI_ERRORCODES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_TUNERFI_FUNCTIONIDS
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_TUNERFI_SERVICEINFO
#include "midw_fi_if.h"
/*************************************************************************
*
* FUNCTION:     tun_Config::tun_Config( )
* 
* DESCRIPTION:  Constructor
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*
*************************************************************************/
tun_Config::tun_Config( )
{
#ifdef VARIANT_S_FTR_ENABLE_FEATURE_SUZUKI	
	tun_ConfigSuzukiParameters();
#else
#ifdef VARIANT_S_FTR_ENABLE_FEATURE_JACCII
	tun_ConfigJACCIIParameters();
#else
#ifdef VARIANT_S_FTR_ENABLE_FEATURE_VW_MIBE
	tun_ConfigVMMIBEParameters();
#else
#ifdef VARIANT_S_FTR_ENABLE_FEATURE_PSA_RCC
	tun_ConfigPSARCCParameters();
	/** Default for G3g */
#else
#ifdef VARIANT_S_FTR_ENABLE_FEATURE_RNAIVI
	tun_ConfigRNAIVIParameters();
#else
#ifdef VARIANT_S_FTR_ENABLE_FEATURE_CAF
	tun_ConfigCAFParameters();
#else
	/** FM1 List */
	m_mapHMIListIDtoADRListID[0]	= 0x0f20;
	m_mapHMIListIDtoBand[0] = (tU8)enAvailBand_FM;
	/** MW1 List */
	m_mapHMIListIDtoADRListID[4]	= 0x0f21;
	m_mapHMIListIDtoBand[4] = (tU8)enAvailBand_MW;
#endif /** END CAF */
#endif /** END AIVI */
#endif /** END PSA */
#endif /** END VW_MIBE */
#endif /** END JAC */ 
#endif /** END SUZUKI */



//#ifdef VARIANT_S_FTR_ENABLE_FEATURE_VW_MIBE
//	/** initially production mode assigned to OFF state */
//	m_bProductionModeState = false;
//#endif
m_poclConfigInterface = NULL;	
}

tVoid tun_Config::tun_ConfigSuzukiParameters()
{
	/** FM1 List */
	m_mapHMIListIDtoADRListID[0]	= 0x0f20;
	m_mapHMIListIDtoBand[0] = (tU8)enAvailBand_FM;
	/** FM2 List */
	m_mapHMIListIDtoADRListID[1]	= 0x0f21;
	m_mapHMIListIDtoBand[1] = (tU8)enAvailBand_FM;
	/** FM-AST List */
	m_mapHMIListIDtoADRListID[3]	= 0x0f22;
	m_mapHMIListIDtoBand[3] = (tU8)enAvailBand_FM;
	/** MW1 List */
	m_mapHMIListIDtoADRListID[4]	= 0x0f23;
	m_mapHMIListIDtoBand[4] = (tU8)enAvailBand_MW;
	/** AM-AST List */
	/** For Suzuki there is no LW band hence AM = MW */
	m_mapHMIListIDtoADRListID[12]	= 0x0f24;
	m_mapHMIListIDtoBand[12] = (tU8)enAvailBand_MW;
	/*DRM Preset List*/
	   #ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
	      m_mapHMIListIDtoADRListID[midw_fi_tcl_e8_Tun_ListID::FI_EN_TUN_BANK_DRM1]  = enMsg_CM_ATPresetList_05;
	      m_mapHMIListIDtoBand[midw_fi_tcl_e8_Tun_ListID::FI_EN_TUN_BANK_DRM1]=(tU8)enAvailBand_CB1;
	      m_mapHMIListIDtoADRListID[midw_fi_tcl_e8_Tun_ListID::FI_EN_TUN_BANK_DRM_AST]  = enMsg_CM_ATPresetList_06;
	      m_mapHMIListIDtoBand[midw_fi_tcl_e8_Tun_ListID::FI_EN_TUN_BANK_DRM_AST]=(tU8)enAvailBand_CB1;
	   #endif
}

tVoid tun_Config::tun_ConfigJACCIIParameters()
{
	/** FM1 List */
	m_mapHMIListIDtoADRListID[0] = 0x0f20;
	m_mapHMIListIDtoBand[0] = (tU8)enAvailBand_FM;
	/** MW1 List */
	m_mapHMIListIDtoADRListID[4] = 0x0f21;
	m_mapHMIListIDtoBand[4] = (tU8)enAvailBand_MW;
}

tVoid tun_Config::tun_ConfigVMMIBEParameters()
{
	/** FM1 List */
	m_mapHMIListIDtoADRListID[0]	= 0x0f20;
	m_mapHMIListIDtoBand[0] = (tU8)enAvailBand_FM;
	/** MW1 List */
	m_mapHMIListIDtoADRListID[4]	= 0x0f21;
	m_mapHMIListIDtoBand[4] = (tU8)enAvailBand_MW;
}

tVoid tun_Config::tun_ConfigPSARCCParameters()
{
	/** FM1 List */
	m_mapHMIListIDtoADRListID[17] = 0x0f20;
	m_mapHMIListIDtoBand[17] = (tU8)enAvailBand_GetCurrent;//255
}

tVoid tun_Config::tun_ConfigRNAIVIParameters()
{
	m_mapHMIListIDtoADRListID[0]	= 0x0f20;
	m_mapHMIListIDtoBand[0] = (tU8)enAvailBand_FM;
	m_mapHMIListIDtoADRListID[4]	= 0x0f21;
	m_mapHMIListIDtoBand[4] = (tU8)enAvailBand_MW;
}

tVoid tun_Config::tun_ConfigCAFParameters()
{
	/** FM1 List */
	m_mapHMIListIDtoADRListID[0]	= 0x0f20;
	m_mapHMIListIDtoBand[0] = (tU8)enAvailBand_FM;
	/** FM2 List */
	m_mapHMIListIDtoADRListID[1]	= 0x0f21;
	m_mapHMIListIDtoBand[1] = (tU8)enAvailBand_FM;
	/** FM-AST List */
	m_mapHMIListIDtoADRListID[3]	= 0x0f22;
	m_mapHMIListIDtoBand[3] = (tU8)enAvailBand_FM;
	/** MW1 List */
	m_mapHMIListIDtoADRListID[4]	= 0x0f23;
	m_mapHMIListIDtoBand[4] = (tU8)enAvailBand_MW;
	/*DRM Preset List*/
	#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
		m_mapHMIListIDtoADRListID[midw_fi_tcl_e8_Tun_ListID::FI_EN_TUN_BANK_DRM1]	= enMsg_CM_ATPresetList_05;
		m_mapHMIListIDtoBand[midw_fi_tcl_e8_Tun_ListID::FI_EN_TUN_BANK_DRM1]=(tU8)enAvailBand_CB1;
		m_mapHMIListIDtoADRListID[midw_fi_tcl_e8_Tun_ListID::FI_EN_TUN_BANK_DRM_AST]	= enMsg_CM_ATPresetList_06;
		m_mapHMIListIDtoBand[midw_fi_tcl_e8_Tun_ListID::FI_EN_TUN_BANK_DRM_AST]=(tU8)enAvailBand_CB1;
	#endif
	/** AM-AST List */
	/** For Suzuki there is no LW band hence AM = MW */
	m_mapHMIListIDtoADRListID[12]	= 0x0f24;
	m_mapHMIListIDtoBand[12] = (tU8)enAvailBand_MW;
}


/*************************************************************************
*
* FUNCTION:     tun_Config::~tun_Config( )
* 
* DESCRIPTION:  Destructor
*
* PARAMETER:    void
*
* RETURNVALUE:  void
*
*************************************************************************/
tun_Config::~tun_Config( )
{
//#ifdef VARIANT_S_FTR_ENABLE_FEATURE_VW_MIBE
//	/** initially production mode assigned to OFF state */
//	m_bProductionModeState = false;
//#endif
	m_poclConfigInterface = NULL;
}


/*************************************************************************
*
* FUNCTION:     tu_Manager:: bGetFrequencyValid( )
* 
* DESCRIPTION:  Returns true if the Frequency is within the Band given..
*
* PARAMETER:    tU8 u8Band,tU32 u32Frequncy
*
* RETURNVALUE:  tBool 
*
*************************************************************************/
tBool tun_Config:: bGetFrequencyValid( tU8 u8Band,tU32 u32Frequency)const
{
  tBool flag=false;

	if(m_poclConfigInterface  == NULL)
	{
	NORMAL_M_ASSERT (m_poclConfigInterface != NULL);
	return false;
	}

	if ((bIsFrequencyWithinTheRange(u8Band,u32Frequency)) && (bIsFrequencyStepValid(u8Band,u32Frequency)))
    {
		flag=true;
		if(midw_fi_tcl_e8_Tun_TunerBand::FI_EN_TUN_BAND_WB == u8Band)
		{
			flag = WB_FREQUENCY_CHECK(u32Frequency);
		}
    }

	ETG_TRACE_ERR(( " tun_Config::bGetFrequencyValid() Flag : %d\n",	flag	));
	return(flag);

}


tVoid tun_Config:: vGetVersionId( tChar* pBuffer, tU32 u32BufferLen) const
{
  TUN_ASSERT_RETURN( u32BufferLen >= TUN_CONFIG_VERSIONID_LEN);
  OSAL_pvMemoryCopy( pBuffer, m_rNewKdsFmAmTunerParameter.sTag, sizeof( m_rNewKdsFmAmTunerParameter.sTag));
  OSAL_pvMemoryCopy( &pBuffer[ sizeof( m_rNewKdsFmAmTunerParameter.sTag)], m_rNewKdsFmAmTunerParameter.sVersion, sizeof( m_rNewKdsFmAmTunerParameter.sVersion));
}

/*************************************************************************
*
* FUNCTION:   vReadKDSConfig()
*
* DESCRIPTION: Read various KDS configuration parameters 
*
* PARAMETER:  Enumeration type
*
* RETURNVALUE: tBool
*
*************************************************************************/
tVoid tun_Config::vReadKDSConfig( tenKDSParameter enKDSParameter)
{

	ETG_TRACE_USR1(( " tun_Config::vReadKDSConfig() -> enKDSParameter=%x"
		,ETG_CENUM( tenKDSParameter, enKDSParameter)
		));


	//Open the KDS device before reading it
	OSAL_tIODescriptor IOKdsHandle = OSAL_IOOpen(OSAL_C_STRING_DEVICE_KDS,OSAL_EN_READWRITE);

	/** Check if IOOpen was successful */
	if( OSAL_ERROR == IOKdsHandle )
	{
		ETG_TRACE_ERR(( " tun_Config::vReadKDSConfig() -> KDS open operation failed, OSAL_u32ErrorCode=0x%08X"
			,OSAL_u32ErrorCode()
			));
		return;
	}

	tS32 s32KdsReadReturnValue = OSAL_ERROR;

	switch(enKDSParameter)
	{
		/*----------------------------------------------------------------------*/


	case TUN_NEW_KDS_FM_AM_Tuner_Parameter_Set:
		{

			tU8 pu8Data[TUN_CONFIG_NEW_KDS_FM_AM_TUNER_PARAMETER_LENGTH];

			/** First read 230 bytes  */
			s32KdsReadReturnValue = s32ReadNewKDS(IOKdsHandle,pu8Data,TUN_CONFIG_NEW_KDS_FM_AM_TUNER_PARAMETER_START_POINT_1,\
				TUN_CONFIG_NEW_KDS_FM_AM_TUNER_PARAMETER_READ_LEN_1);

			/** If success read remaining bytes  */
			if(s32KdsReadReturnValue != OSAL_ERROR)
			{
				s32KdsReadReturnValue = s32ReadNewKDS(IOKdsHandle,pu8Data + TUN_CONFIG_NEW_KDS_FM_AM_TUNER_PARAMETER_READ_LEN_1,\
					TUN_CONFIG_NEW_KDS_FM_AM_TUNER_PARAMETER_START_POINT_2,\
					TUN_CONFIG_NEW_KDS_FM_AM_TUNER_PARAMETER_READ_LEN_2);
			}


			if(s32KdsReadReturnValue != OSAL_ERROR)
			{
				/** Parse and Print  */
				m_rNewKdsFmAmTunerParameter.vParse(pu8Data,TUN_CONFIG_NEW_KDS_FM_AM_TUNER_PARAMETER_LENGTH);
				m_rNewKdsFmAmTunerParameter.vTrace();

			}
			else
			{
				ETG_TRACE_ERR(( " tun_Config::vReadKDSConfig() -> Reading FMAM PARAMETERS ERROR"		));
			}

			break;
		}
	case TUN_NEW_KDS_GENERIC_Tuner_Parameter_Set:
		{
			tU8 pu8Data[TUN_CONFIG_NEW_KDS_GENERIC_TUNER_PARAMETER_LENGTH];

			/** First read 230 bytes  */
			s32KdsReadReturnValue = s32ReadNewKDS(IOKdsHandle,pu8Data,TUN_CONFIG_NEW_KDS_GENERIC_TUNER_PARAMETER_START_POINT_1,\
				TUN_CONFIG_NEW_KDS_GENERIC_TUNER_PARAMETER_READ_LEN_1);

			ETG_TRACE_USR1(( " TUN_NEW_KDS_GENERIC_Tuner_Parameter_Set***************  -> %d", 	s32KdsReadReturnValue	));

			/** If success read remaining bytes  */
			if(s32KdsReadReturnValue != OSAL_ERROR)
			{
				s32KdsReadReturnValue = s32ReadNewKDS(IOKdsHandle,pu8Data + TUN_CONFIG_NEW_KDS_GENERIC_TUNER_PARAMETER_READ_LEN_1,\
					TUN_CONFIG_NEW_KDS_GENERIC_TUNER_PARAMETER_START_POINT_2,\
					TUN_CONFIG_NEW_KDS_GENERIC_TUNER_PARAMETER_READ_LEN_2);
			}


			if(s32KdsReadReturnValue != OSAL_ERROR)
			{
				/** Parse and Print  */
				m_rNewKdsGenericTunerParameter.vParse(pu8Data,TUN_CONFIG_NEW_KDS_GENERIC_TUNER_PARAMETER_LENGTH);
				m_rNewKdsGenericTunerParameter.vTrace();

			}
			else
			{
				ETG_TRACE_ERR(( " tun_Config::vReadKDSConfig() -> Reading GENERIC PARAMETERS ERROR"		));
			}
			break;
		}

		/*----------------------------------------------------------------------*/

	case TUN_KDS_CM_VARIANT_CODING:
		{
			tU8 pu8Data[TUN_CONFIG_KDS_CM_VARIANT_CODING_LENGTH];


			s32KdsReadReturnValue = s32ReadNewKDS(IOKdsHandle,pu8Data,TUN_CONFIG_KDS_CM_VARIANT_CODING,\
				TUN_CONFIG_KDS_CM_VARIANT_CODING_LENGTH);

			ETG_TRACE_USR1(( " TUN_KDS_CM_VARIANT_CODING ***************  -> %d", 	s32KdsReadReturnValue	));


			if(s32KdsReadReturnValue != OSAL_ERROR)
			{
				/** Parse and Print  */
				m_rKdsCMVariantCoding.vParse(pu8Data,TUN_CONFIG_KDS_CM_VARIANT_CODING_LENGTH);
				m_rKdsCMVariantCoding.vTrace();

			}
			else
			{
				ETG_TRACE_ERR(( " tun_Config::vReadKDSConfig() -> Reading CM VARIANT CODING ERROR"		));
			}
			break;
		}
#if FEATURE_VW_MIBE
		/** Reading End Of Line Parameters */
	case TUN_KDS_CM_EOL_PARAMETER:
		{
			tU8 pu8Data[TUN_CONFIG_KDS_CM_EOL_PARAMETER_LENGTH];

			s32KdsReadReturnValue = s32ReadNewKDS(IOKdsHandle,pu8Data,TUN_CONFIG_KDS_CM_EOL_PARAMETER,\
				TUN_CONFIG_KDS_CM_EOL_PARAMETER_LENGTH);

			ETG_TRACE_USR1(( " TUN_KDS_CM_EOL_PARAMETER ***************  -> %d", 	s32KdsReadReturnValue	));


			if(s32KdsReadReturnValue != OSAL_ERROR)
			{
				/** Parse and Print  */
				m_rKdsEOLParameter.vParse(pu8Data,TUN_CONFIG_KDS_CM_EOL_PARAMETER_LENGTH);
				m_rKdsEOLParameter.vTrace();

				vUpdateFreqAccording2EOLFMTunerBand();
				vUpdateFreqAccording2EOLAMTunerBand();
			}
			else
			{
				ETG_TRACE_ERR(( " tun_Config::vReadKDSConfig() -> Reading CM EOL PARAMETER ERROR"		));
			}

			break;
		}
		/** Reading Production Mode Parameters */
	case TUN_KDS_CM_PROD_MODE_SETTINGS:
		{

			tU8 pu8Data[TUN_CONFIG_KDS_PRODUCTION_MODE_SETTINGS_LENGTH];


			s32KdsReadReturnValue = s32ReadNewKDS(IOKdsHandle,pu8Data,TUN_CONFIG_KDS_PRODUCTION_MODE_SETTINGS,\
				TUN_CONFIG_KDS_PRODUCTION_MODE_SETTINGS_LENGTH);

			ETG_TRACE_USR1(( " TUN_KDS_CM_PRODUCTION_MODE_SETTINGS ***************  -> %d", 	s32KdsReadReturnValue	));


			if(s32KdsReadReturnValue != OSAL_ERROR)
			{
				/** Parse and Print  */
				m_rKdsProdModeSettings.vParse(pu8Data,TUN_CONFIG_KDS_PRODUCTION_MODE_SETTINGS_LENGTH);
				m_rKdsProdModeSettings.vTrace();

				if((m_rKdsProdModeSettings.m_u8ProdModeByte1 == 0x00) &&
					(m_rKdsProdModeSettings.m_u8ProdModeByte2 == 0x04) &&
					(m_rKdsProdModeSettings.m_u8ProdModeByte3 == 0x10) 
					)
				{
					ETG_TRACE_USR1(( " tun_Config::vReadKDSConfig() -> KDS_PROD_MODE --- > ON " ));

					//m_bProductionModeState = true;

					///** Set tuner 1 to measure mode */
					//m_ptun_MsgToADR->vSendSpecialDiagnosisMode(
					//	enCM_SpecialDiagnosisModeMode_MEASURE_MODE,enPhysicalTunerId_Tuner1);
					///** Set tuner 1 to User Audio */
					//m_ptun_MsgToADR->vSendTunerFGBG(enInstanceId_AudioSource1,enAudioSource_User_audio);

					///** Set tuner 2 to measure mode */
					//m_ptun_MsgToADR->vSendSpecialDiagnosisMode(
					//	enCM_SpecialDiagnosisModeMode_MEASURE_MODE,enPhysicalTunerId_Tuner2);
					///** Tune frequency from tuner 1 to tuner 2*/
					//
					//m_ptun_MsgToADR->vSendATFrequency(enATFrequencyAbsOrRel_AbsoluteValue,
					//	0,m_u32LastFrequency,enATFrequencyDirection_unused,0);

					///** Set tuner 1 to measure mode */
					//m_ptun_MsgToADR->vSendSpecialDiagnosisMode(
					//	enCM_SpecialDiagnosisModeMode_MEASURE_MODE,enPhysicalTunerId_Tuner1);


				}
				else
				{
					ETG_TRACE_USR1(( " tun_Config::vReadKDSConfig() -> KDS_PROD_MODE --- > OFF" ));

					//m_bProductionModeState = false;

					///** Put tuner one to Normal mode */
					//m_ptun_MsgToADR->vSendSpecialDiagnosisMode(
					//	enCM_SpecialDiagnosisModeMode_NORMAL_MODE,enPhysicalTunerId_Tuner1);
				}


			}
			else
			{
				ETG_TRACE_ERR(( " tun_Config::vReadKDSConfig() -> Reading CM PRODUCTION MODE SETTINGS ERROR"		));
			}

		}
		break;
		/** Reading Quality Level Offset Parameters */
	case TUN_KDS_CM_QUALITY_LEVEL_OFFSET:
		{
			tU8 pu8Data[TUN_CONFIG_KDS_QUALITYLEVEL_OFFSET_LENGTH];


			s32KdsReadReturnValue = s32ReadNewKDS(IOKdsHandle,pu8Data,TUN_CONFIG_KDS_QUALITYLEVEL_OFFSET,\
				TUN_CONFIG_KDS_QUALITYLEVEL_OFFSET_LENGTH);

			ETG_TRACE_USR1(( " TUN_KDS_QUALITYLEVEL_OFFSET ***************  -> %d", 	s32KdsReadReturnValue	));


			if(s32KdsReadReturnValue != OSAL_ERROR)
			{
				/** Parse and Print  */
				m_rKdsQualityLevelOffset.vParse(pu8Data,TUN_CONFIG_KDS_QUALITYLEVEL_OFFSET_LENGTH);
				m_rKdsQualityLevelOffset.vTrace();

			}
			else
			{
				ETG_TRACE_ERR(( " tun_Config::vReadKDSConfig() -> Reading CM QUALITY LEVEL OFFSET ERROR"		));
			}


			break;
		}
#endif

#if FEATURE_PSA_RCC
	case TUN_KDS_CM_EOL_CONFIG_RADIO_PARAMETER:
	{
#ifdef DP_U16_KDSADR_TEL_FCT_RADIO
		tU8 pu8Data[TUN_CONFIG_KDS_CM_EOL_PARAMETER_LENGTH];
		dp_tclKdsTEL_Fct_RADIO oDp_TEL_Fct_RADIO;

		/** Read from Data pool into local array */
		tU8 u8KdsReadReturnValue = oDp_TEL_Fct_RADIO.u8GetTEL_Fct_RADIO(pu8Data,DP_U8_KDSLEN_TEL_FCT_RADIO_COMPLETE);
		ETG_TRACE_USR2(( " tun_Config::vReadKDSConfig() :->u8KdsReadReturnValue : %d", u8KdsReadReturnValue));

		if(u8KdsReadReturnValue == DP_U8_ELEM_STATUS_VALID)
		{
			m_rKdsEOLConfigRadioParameter.vParse();
			m_rKdsEOLConfigRadioParameter.vTrace();
			vUpdateAvailableBandsAccording2EOL();
			vUpdateFrequency_StepSizeFromEOL();
			vUpdateDeemphasisAccording2Country();

			vReadKDSEOLConfigAudio();
		}
#endif

		break;
	}
#endif
		/*----------------------------------------------------------------------*/
	
		/*----------------------------------------------------------------------*/

	case TUN_KDS_CM_HD_RADIO_PARAM_SET:
		{
			tU8 pu8Data[TUN_CONFIG_KDS_HD_RADIO_PARAM_SET_LENGTH];


			s32KdsReadReturnValue = s32ReadNewKDS(IOKdsHandle,pu8Data,TUN_KDS_CM_HD_RADIO_PARAM_SET_KEY,\
				TUN_CONFIG_KDS_HD_RADIO_PARAM_SET_LENGTH);

			ETG_TRACE_USR1(( " TUN_KDS_CM_HD_RADIO_PARAM_SET ***************  -> %d", 	s32KdsReadReturnValue	));


			if(s32KdsReadReturnValue != OSAL_ERROR)
			{
				/** Parse and Print  */
				m_rKdsHDRadioParameterSet.vParse(pu8Data,TUN_CONFIG_KDS_HD_RADIO_PARAM_SET_LENGTH);
				m_rKdsHDRadioParameterSet.vTrace();

			}
			else
			{
				ETG_TRACE_ERR(( " tun_Config::vReadKDSConfig() -> Reading CM HD RADIO PARAMETER ERROR"		));
			}
			break;
		}	

		/*----------------------------------------------------------------------*/

	case TUN_KDS_CM_DRM_TUNER_PARAM_SET:
		{
			tU8 pu8Data[TUN_CONFIG_KDS_DRM_TUNER_PARAM_SET_LENGTH];


			s32KdsReadReturnValue = s32ReadNewKDS(IOKdsHandle,pu8Data,TUN_KDS_CM_DRM_TUNER_PARAM_SET_KEY,\
				TUN_CONFIG_KDS_DRM_TUNER_PARAM_SET_LENGTH);

			ETG_TRACE_USR1(( " TUN_KDS_CM_DRM_TUNER_PARAM_SET ***************  -> %d", 	s32KdsReadReturnValue	));


			if(s32KdsReadReturnValue != OSAL_ERROR)
			{
				/** Parse and Print  */
				m_rKdsDRMTunerParameterSet.vParse(pu8Data,TUN_CONFIG_KDS_DRM_TUNER_PARAM_SET_LENGTH);
				m_rKdsDRMTunerParameterSet.vTrace();

			}
			else
			{
				ETG_TRACE_ERR(( " tun_Config::vReadKDSConfig() -> Reading CM DRM TUNER PARAMETER SET ERROR"		));
			}
			break;
		}
	default:
		break;
		/*----------------------------------------------------------------------*/
	}

	ETG_TRACE_USR1(( " tun_Config::vReadKDSConfig() -> %02x : Read status [0 = Success, -1 = ERROR] = %d ",
		ETG_CENUM( tenKDSParameter, enKDSParameter),s32KdsReadReturnValue));
			

	/** Close IO handle */
	tS32 s32ReturnValue = OSAL_s32IOClose(IOKdsHandle);

		ETG_TRACE_USR1(( " tun_Config::vReadKDSConfig() -> KDS close operation result [0 = Success, -1 = ERROR] : s32ReturnValue = %d",
			s32ReturnValue));
}



/*************************************************************************
*
* FUNCTION:   tun_Config::s32ReadNewKDS()
*
* DESCRIPTION: Reads KDS data as specified by the length. Max value to be passed is 230 (one chunk)
*
* PARAMETER:  OSAL_tIODescriptor IOKdsHandle, tU8* u8Buf, tU16 u16KdsKey, tU16 u16Len       
*
* RETURNVALUE: tS32
*
* vnd4kor 12/4/2014 
*************************************************************************/
tS32 tun_Config::s32ReadNewKDS(OSAL_tIODescriptor IOKdsHandle, tU8* u8Buf, tU16 u16KdsKey, tU16 u16Len) const
{
	/**Reading Production mode settings */
	tsKDSEntry sKDS_EntryData;

	sKDS_EntryData.u16Entry = u16KdsKey;
	sKDS_EntryData.u16EntryLength = u16Len;
	sKDS_EntryData.u16EntryFlags = M_KDS_ENTRY_FLAG_NONE;
	sKDS_EntryData.au8EntryData[0] = 0;

	tS32 s32RetVal = OSAL_s32IORead(IOKdsHandle,(tPS8)&sKDS_EntryData,\
				sizeof(sKDS_EntryData));

    if (OSAL_ERROR != s32RetVal) {
          (tVoid)memcpy((tVoid*)u8Buf, sKDS_EntryData.au8EntryData, sKDS_EntryData.u16EntryLength);
       }

	return s32RetVal;
}



 

/*****************************************************************************
******************* New KDS AM FM Parameter Set ******************************
*****************************************************************************/

/*****************************************************************************
 * Constructor
 ****************************************************************************/
tun_trNewKdsFmAmTunerParameter::tun_trNewKdsFmAmTunerParameter( tVoid )
{
  ETG_TRACE_USR1_CLS(( TUN_TRACE_CLASS_TUNAPP, \
	  " tun_trNewKdsFmAmTunerParameter::tun_trNewKdsFmAmTunerParameter() -> Constructor"
      ));

  tun_trNewKdsFmAmTunerParameter::vSetToDefaultValues();

}

/*****************************************************************************
 * Set all KDS values to default
 ****************************************************************************/
tVoid tun_trNewKdsFmAmTunerParameter::vSetToDefaultValues( tVoid)
{
  ETG_TRACE_USR1_CLS(( TUN_TRACE_CLASS_TUNAPP, " tun_trNewKdsFmAmTunerParameter::vSetToDefaultValues() -> "
                ));

  bKdsDataRead                    = FALSE;
  
  sTag[ 0]     = 'D';
  sTag[ 1]     = 'e';
  sTag[ 2]     = 'f';
  sTag[ 3]     = 'a';
  sTag[ 4]     = 'u';
  sTag[ 5]     = 'l';
  sTag[ 6]     = 't';
  sTag[ 7]     = '-';

  sVersion[ 0] = '0';
  sVersion[ 1] = '0';
  sVersion[ 2] = '2';

    /** FM  */
  u32LowFM                        = TUN_CONFIG_DEFAULT_LOW_FM;
  u32HighFM                       = TUN_CONFIG_DEFAULT_HIGH_FM;
  u8StepSizeFM                    = TUN_CONFIG_DEFAULT_STEP_SIZE_FM;
  u8StepSizeManualFM              = TUN_CONFIG_DEFAULT_STEP_SIZE_MANUAL_FM;
  /** MW */
  u32LowMW                        = TUN_CONFIG_DEFAULT_LOW_MW;
  u32HighMW                       = TUN_CONFIG_DEFAULT_HIGH_MW;
  u8StepSizeMW                    = TUN_CONFIG_DEFAULT_STEP_SIZE_MW;
  u8StepSizeManualMW              = TUN_CONFIG_DEFAULT_STEP_SIZE_MANUAL_MW;
  /** LW */
  u32LowLW                        = TUN_CONFIG_DEFAULT_LOW_LW;
  u32HighLW                       = TUN_CONFIG_DEFAULT_HIGH_LW;
  u8StepSizeLW                    = TUN_CONFIG_DEFAULT_STEP_SIZE_LW;
  u8StepSizeManualLW              = TUN_CONFIG_DEFAULT_STEP_SIZE_MANUAL_LW;

  u8FeatureControlI				  = TUN_CONFIG_DEFAULT_FEATURE_CONTROL_I;
  u8FeatureControlII			  = TUN_CONFIG_DEFAULT_FEATURE_CONTROL_II;
  
  u32FM_LSM                       = TUN_CONFIG_DEFAULT_FM_LSM;
  u32MW_LSM                       = TUN_CONFIG_DEFAULT_MW_LSM;
  u32LW_LSM                       = TUN_CONFIG_DEFAULT_LW_LSM;
  
  u8ShowRadioTextStatusMessage				= TUN_CONFIG_DEFAULT_SHOWRADIOTEXT_STATUSMSG; 
  u8T_RadioTextNotAvailable					= TUN_CONFIG_DEFAULT_T_RADIOTEXT_NOTAVAILABLE;
  u8T_Timeout_Status_RT						= TUN_CONFIG_DEFAULT_T_TIMEOUT_STATUS_RT;
  u8T_ShowTextMin							= TUN_CONFIG_DEFAULT_T_SHOWTEXTMIN;
  u16T_MinDisplayDurationPerRadioTextPage	= TUN_CONFIG_DEFAULT_T_MINDISPLAYDURATIONPER_RTPAGE;
  u8RadioTextMinLength						= TUN_CONFIG_DEFAULT_RADIOTEXT_MINLENGTH;  
  u8RadioTextSpeedThreshold					= TUN_CONFIG_DEFAULT_RADIOTEXT_SPEEDTHRESHOLD;
  u8RadioTextSpeedThresholdHysteresis		= TUN_CONFIG_DEFAULT_RADIOTEXT_SPEEDTHRESHOLDHYSTERISIS;
 
  u8EmergencySeekScanNoOfStations = TUN_CONFIG_DEFAULT_EMERGENCYSEEKSCAN_NOOFSTNS; 
  u8ShowFreqDuringFreqScan		  = TUN_CONFIG_DEFAULT_SHOW_FREQDURING_FREQSCAN;
  u8SeekMode					  = TUN_CONFIG_DEFAULT_SEEKMODE;

  u8NumOfSeekPeriods              = TUN_CONFIG_DEFAULT_NUM_OF_SEEK_PERIODS;
  u8ScanTime                      = TUN_CONFIG_DEFAULT_SCAN_TIME;

  u8AFBestDelayTimeAftManualTuning		= TUN_CONFIG_DEFAULT_AFBESTDELAY_TIMEAFTMANUALTUNING;
  u8AFBestDisplayFreezeAftManualTuning	= TUN_CONFIG_DEFAULT_AFBEST_DISPLAYFREEZEAFTMANUALTUNING;

  u16AutoSeekStepDelay            = TUN_CONFIG_DEFAULT_AUTO_SEEK_STEP_DELAY;
  u16PemanentManualStepDelay      = TUN_CONFIG_DEFAULT_PEMANENT_MANUAL_STEP_DELAY;

  u8WrongPIAcceptDelayAftStnSelection = TUN_CONFIG_DEFAULT_WRONGPIACCEPTDELAY_AFTSTNSEL; //3sec (Skoda)
  u8WrongPIAcceptAftStnlost			  = TUN_CONFIG_DEFAULT_WRONGPIACCEPT_AFTSTNLOST;//3 sec (Skoda)

  u8MultiManualStepsWithIntermediateFreq = TUN_CONFIG_DEFAULT_MULTIMANUALSTEPSWITH_INTERMEDIATEFREQ;//1 VW, rest all 0
  u8StationLists						 = TUN_CONFIG_DEFAULT_STATIONLIST;//3 for all, 9 for PSA
  u8FMListSortingDefault				 = TUN_CONFIG_DEFAULT_FMLISTSORTINGDEFAULT;//03
  u8AMListSortingDefault				 = TUN_CONFIG_DEFAULT_AMLISTSORTINGDEFAULT;//03

  u8TimerInhibitLstSortAfterSkip		= TUN_CONFIG_DEFAULT_TIMERINHIBITLSTSORTAFTERSKIP;
  u8TimerCursorDelayAfterStnSel			= TUN_CONFIG_DEFAULT_TIMERCURSORDELAYAFTERSTNSEL;

  u8TimeAMMinTuneListRemove				= TUN_CONFIG_DEFAULT_T_AM_MINTUNELISTREMOVE;//30 sec
  u8TimeAMStnRemoveAftDeslect			= TUN_CONFIG_DEFAULT_T_AM_STNREMOVEAFTDESELECT;//30 sec
  u8AMStnListUpdateStrategy				= TUN_CONFIG_DEFAULT_AMSTNLIST_UPDATESTRATEGY;//1 min

  u8TestModeDataUpdateInterval	= TUN_CONFIG_DEFAULT_TESTMODE_DATA_UPDATE_INT;
  u8AFListUpdateInterval		= TUN_CONFIG_DEFAULT_AFLIST_UPDATE_INT;
  u8AFListResort				= TUN_CONFIG_DEFAULT_AFLISTRESORT;//0
  u8AFListReleaseBehaviour		= TUN_CONFIG_DEFAULT_AFLIST_RELEASE_BEHAVIOUR;

  u8LandscapeUpdateOnShutdown	= TUN_CONFIG_DEFAULT_LANDSCAPEUPDATE_ONSHUTDOWN; //3 for all, 9 for PSA
  u8TunerSwitches				= TUN_CONFIG_DEFAULT_TUNERSWITCHES;//3 for PSA , 0 for Suz

  u8ADRFmChannelSeparation           = TUN_CONFIG_DEFAULT_FM_CHANNEL_SEPARATION;
  u8ADRFmLimiting                    = TUN_CONFIG_DEFAULT_FM_LIMITING;
  u8ADRFmMultipath                   = TUN_CONFIG_DEFAULT_FM_MULTIPATH;
  u8ADRFmHighCut                     = TUN_CONFIG_DEFAULT_FM_HIGHCUT;
  u8ADRFmSoftMute                    = TUN_CONFIG_DEFAULT_FM_SOFTMUTE;
  u8ADRAmSoftMute                    = TUN_CONFIG_DEFAULT_AM_SOFTMUTE;
  u8ADRFmDeemphasis                  = TUN_CONFIG_DEFAULT_FM_DEEMPHASIS;
  u8ADRFmFrequencyResponse           = TUN_CONFIG_DEFAULT_FM_FREQUENCY_RESPONSE;
  u8ADRAmFrequencyResponse           = TUN_CONFIG_DEFAULT_AM_FREQUENCY_RESPONSE;
  u8ADRFmMisc                        = TUN_CONFIG_DEFAULT_FM_MISC;
  u8ADRAmMisc                        = TUN_CONFIG_DEFAULT_AM_MISC;
  u8ADRFmAmplificationLevel          = TUN_CONFIG_DEFAULT_FM_AMPLIFICATION_LEVEL;
  u8ADRAmAmplificationLevel          = TUN_CONFIG_DEFAULT_AM_AMPLIFICATION_LEVEL;
  u8ADRAM_Hicut_Configuration        = TUN_CONFIG_DEFAULT_AM_HICUT_CONFIGURATION;
  u8ADRAM_Deemphasis                 = TUN_CONFIG_DEFAULT_AM_DEEMPHASIS;

  u16MUTE_RAMP_1                  = TUN_CONFIG_DEFAULT_MUTE_RAMP_1;
  u16DEMUTE_RAMP_1                = TUN_CONFIG_DEFAULT_DEMUTE_RAMP_1;
  u16MUTE_RAMP_2                  = TUN_CONFIG_DEFAULT_MUTE_RAMP_2;
  u16DEMUTE_RAMP_2                = TUN_CONFIG_DEFAULT_DEMUTE_RAMP_2;

  u8BehaviourSwitches			  = TUN_CONFIG_DEFAULT_BEHAVIOURAL_SWITCHES;

  u8FieldStrengthSeekLevel_FM_RDS       = TUN_CONFIG_DEFAULT_FS_SEEKLEVEL_FM_RDS; 
  u8QualitySeekLevel_FM_RDS				= TUN_CONFIG_DEFAULT_Q_SEEKLEVEL_FM_RDS;
  u8FieldStrengthSeekLevel_FM_NONRDS    = TUN_CONFIG_DEFAULT_FS_SEEKLEVEL_FM_NONRDS;
  u8QualitySeekLevel_FM_NONRDS			= TUN_CONFIG_DEFAULT_Q_SEEKLEVEL_FM_NONRDS;
  u8FieldStrengthSeekLevel_LW           = TUN_CONFIG_DEFAULT_FS_SEEKLEVEL_LW;
  u8QualitySeekLevel_LW				    = TUN_CONFIG_DEFAULT_Q_SEEKLEVEL_LW;
  u8FieldStrengthSeekLevel_MW           = TUN_CONFIG_DEFAULT_FS_SEEKLEVEL_MW;
  u8QualitySeekLevel_MW				    = TUN_CONFIG_DEFAULT_Q_SEEKLEVEL_MW;
  u8FieldStrengthSeekLevel_SW           = TUN_CONFIG_DEFAULT_FS_SEEKLEVEL_SW; 
  u8QualitySeekLevel_SW				    = TUN_CONFIG_DEFAULT_Q_SEEKLEVEL_SW;
  u8FieldStrengthSeekLevel_TRF          = TUN_CONFIG_DEFAULT_FS_SEEKLEVEL_TRF; 
  u8QualitySeekLevel_TRF				= TUN_CONFIG_DEFAULT_Q_SEEKLEVEL_TRF;
  u8FieldStrengthSeekLevel_WB           = TUN_CONFIG_DEFAULT_FS_SEEKLEVEL_WB; 
  u8QualitySeekLevel_WB				    = TUN_CONFIG_DEFAULT_Q_SEEKLEVEL_WB;
  
  u8FieldStrengthListLevel_FM_RDS       = TUN_CONFIG_DEFAULT_FS_LISTLEVEL_FM_RDS;
  u8QualityListLevel_FM_RDS				= TUN_CONFIG_DEFAULT_Q_LISTLEVEL_FM_RDS;
  u8FieldStrengthListLevel_FM_NONRDS    = TUN_CONFIG_DEFAULT_FS_LISTLEVEL_FM_NONRDS;
  u8QualityListLevel_FM_NONRDS			= TUN_CONFIG_DEFAULT_Q_LISTLEVEL_FM_NONRDS;
  u8FieldStrengthListLevel_LW           = TUN_CONFIG_DEFAULT_FS_LISTLEVEL_LW;
  u8QualityListLevel_LW				    = TUN_CONFIG_DEFAULT_Q_LISTLEVEL_LW;
  u8FieldStrengthListLevel_MW           = TUN_CONFIG_DEFAULT_FS_LISTLEVEL_MW;
  u8QualityListLevel_MW				    = TUN_CONFIG_DEFAULT_Q_LISTLEVEL_MW;
  u8FieldStrengthListLevel_SW           = TUN_CONFIG_DEFAULT_FS_LISTLEVEL_SW; 
  u8QualityListLevel_SW				    = TUN_CONFIG_DEFAULT_Q_LISTLEVEL_SW;
  u8FieldStrengthListLevel_TRF          = TUN_CONFIG_DEFAULT_FS_LISTLEVEL_TRF; 
  u8QualityListLevel_TRF				= TUN_CONFIG_DEFAULT_Q_LISTLEVEL_TRF;
  u8FieldStrengthListLevel_WB           = TUN_CONFIG_DEFAULT_FS_LISTLEVEL_WB; 
  u8QualityListLevel_WB				    = TUN_CONFIG_DEFAULT_Q_LISTLEVEL_WB;

  u8SHARX                         = TUN_CONFIG_DEFAULT_SHARX;
  u8HICUT                         = TUN_CONFIG_DEFAULT_HICUT;

  u32AvailableBands               = TUN_CONFIG_DEFAULT_AVAILABLE_BANDS;
  u32CombiBands					  = TUN_CONFIG_DEFAULT_COMBI_BANDS;

  u8ADRSwitches1                  = TUN_CONFIG_DEFAULT_ADR_SWITCHES1;//TUN_CONFIG_DEFAULT_ADR_SWITCHES1; f1
  u8ADRSwitches2                  = TUN_CONFIG_DEFAULT_ADR_SWITCHES2;//TUN_CONFIG_DEFAULT_ADR_SWITCHES2;31
  u8ADRSwitches3                  = TUN_CONFIG_DEFAULT_ADR_SWITCHES3;//TUN_CONFIG_DEFAULT_ADR_SWITCHES3;0
  u8ADRSwitches4                  = TUN_CONFIG_DEFAULT_ADR_SWITCHES4;//0c

  enPhysicalTunerConfigId         = TUN_CONFIG_DEFAULT_PHYSICAL_TUNER_CONFIG_ID;
  u8AntennaConfigId               = TUN_CONFIG_DEFAULT_ANTENNA_CONFIG_ID;
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
  u8AMFMMisc2                       = TUN_CONFIG_DEFAULT_AMFMMISC2;
#endif

  u8ExchangeAntennaLevel          = TUN_CONFIG_DEFAULT_EXCHANGE_ANTENNA_LEVEL;
  u8PitestLevel                   = TUN_CONFIG_DEFAULT_PITEST_LEVEL;
  u8NrdsLevel                     = TUN_CONFIG_DEFAULT_NRDS_LEVEL;
  u8EonLevel                      = TUN_CONFIG_DEFAULT_EON_LEVEL;
  u8FMMemoLevel                   = TUN_CONFIG_DEFAULT_FM_MEMO_LEVEL;
  u8PanicFieldstLevel             = TUN_CONFIG_DEFAULT_PANIC_FIELDST_LEVEL;
  u8ParamRDSLevel                 = TUN_CONFIG_DEFAULT_PARAM_RDS_LEVEL;
  u8RDSDataAcceptLevel            = TUN_CONFIG_DEFAULT_RDS_DATA_ACCEPT_LEVEL;
  u8PIStepLevel                   = TUN_CONFIG_DEFAULT_PI_STEP_LEVEL;
  u8DDAAntennaDistLevel           = TUN_CONFIG_DEFAULT_DDA_ANTENNA_DIST_LEVEL;
  u8RDSSynDelayCnt                = TUN_CONFIG_DEFAULT_RDS_SYN_DELAY_CNT;
  u8MpMeasureLevel                = TUN_CONFIG_DEFAULT_MP_MEASURE_LEVEL;
  u8MpLevel2Level                 = TUN_CONFIG_DEFAULT_MP_LEVEL2_LEVEL;
  u8MpVeryBadLevel                = TUN_CONFIG_DEFAULT_MP_VERY_BAD_LEVEL;
  u8MpMediumLevel                 = TUN_CONFIG_DEFAULT_MP_MEDIUM_LEVEL;
  u8RDSBadLevel                   = TUN_CONFIG_DEFAULT_RDS_BAD_LEVEL;
  u8IllegalAf                     = TUN_CONFIG_DEFAULT_ILLEGAL_AF;
  u8AfSaveLevel                   = TUN_CONFIG_DEFAULT_AF_SAVE_LEVEL;
  u8FirstRundum                   = TUN_CONFIG_DEFAULT_FIRST_RUNDUM;
  u8SecondRundum                  = TUN_CONFIG_DEFAULT_SECOND_RUNDUM;
  u8ThirdRundum                   = TUN_CONFIG_DEFAULT_THIRD_RUNDUM;
  u8PIStepMode					  = TUN_CONFIG_DEFAULT_PISTEP_MODE;
  u8AudioOffScanMode              = TUN_CONFIG_DEFAULT_AUDIO_OFF_SCAN_MODE;
  u8AMLearnCycle                  = TUN_CONFIG_DEFAULT_AM_LEARN_CYCLE;
  u8PIIgnore                      = TUN_CONFIG_DEFAULT_PI_IGNORE;
  u8PISearchTime                  = TUN_CONFIG_DEFAULT_FREE1;
  u8DisableSeamlessblending       = TUN_CONFIG_DEFAULT_FREE2;
  u8SlAdjThreshold                = TUN_CONFIG_DEFAULT_SL_ADJ_THRESHOLD;
  u8SlMpThreshold                 = TUN_CONFIG_DEFAULT_SL_MP_THRESHOLD;
  u8SpecialFMStartFrequency       = TUN_CONFIG_DEFAULT_SPECIAL_FM_START_FREQUENCY;
  u8FMSharxLimit				  = TUN_CONFIG_DEFAULT_FMSHARXLIMIT;//0
  u8HWSubConfig					  = TUN_CONFIG_DEFAULT_HWSUBCONFIG;
  u8HDScanEveryFrequency		  = TUN_CONFIG_DEFAULT_HD_SCAN_EVERY_FREQ;

  u8FM2DABDelayTime								= TUN_CONFIG_DEFAULT_FM2DAB_DELAY_TIME; 

  u8AMFMHicutExtender							= TUN_CONFIG_DEFAULT_AMFM_HICUT_EXTENDER;
  u8FMSoftwareMuteTiming						= TUN_CONFIG_DEFAULT_FM_SOFTMUTE_TIMING;
  u8AMFMMisscellaneous2							= TUN_CONFIG_DEFAULT_AMFM_MISSCELLANEOUS_2;
  u8FMHicutExtenderFieldstrengthThreshold		= TUN_CONFIG_DEFAULT_FM_HICUT_EXTENDER_FIELDSTRENGTH_THRESHOLD;
  u8FMHicutExtenderFieldstrengthFilterOffset	= TUN_CONFIG_DEFAULT_FM_HICUT_EXTENDER_FIELDSTRENGTH_FILTER_OFFSET;
  u8FMChannelSeparationMaximum					= TUN_CONFIG_DEFAULT_FM_CHANNEL_SEPERATION_MAXIMUM;
  u8FMChannelSeparationTiming					= TUN_CONFIG_DEFAULT_FM_CHANNEL_SEPERATION_TIMING;
  u8FMHicutFieldstrengthThreshold				= TUN_CONFIG_DEFAULT_FM_HICUT_FIELDSTRENGTH_THRESHOLD;
  u8FMHicutTiming								= TUN_CONFIG_DEFAULT_FM_HICUT_TIMING;
  u8FMDMQSensitivity							= TUN_CONFIG_DEFAULT_FM_DMQ_SENSITIVITY;
  u8FMDeviationAdaption							= TUN_CONFIG_DEFAULT_FM_DEVIATION_ADAPTION;
  u8AMNoiseBlanker228kHzSensitivity				= TUN_CONFIG_DEFAULT_AM_NOISE_BLANKER_228KHZ_SENSITIVITY;
  u8AMNoiseBlanker45_6kHzSensitivity			= TUN_CONFIG_DEFAULT_AM_NOISE_BLANKER_45_6KHZ_SENSITIVITY;
  u8AMHicutExtenderFieldstrengthThreshold		= TUN_CONFIG_DEFAULT_AM_HICUT_EXTENDER_FIELDSTRENGTH_THRESHOLD;
  u8AMHicutExtenderFieldstrengthFilterOffset	= TUN_CONFIG_DEFAULT_AM_HICUT_EXTENDER_FIELDSTRENGTH_FILTER_OFFSET;

  u8DemodulationSwitchAudioTuner1				= TUN_CONFIG_DEFAULT_DEMODULATION_SWITCH_AUDIO_TUNER_1;
  u8DemodulationSwitchAudioTuner2				= TUN_CONFIG_DEFAULT_DEMODULATION_SWITCH_AUDIO_TUNER_2;
 

  u8TimerNonRDS_Detect					= TUN_CONFIG_DEFAULT_TIMERNONRDS_DETECT;
  u8TimerPS_DelAfterRDSLost				= TUN_CONFIG_DEFAULT_TIMERPS_DELAFTERRDSLOST;
  u8TimerScrollPSDetect					= TUN_CONFIG_DEFAULT_TIMERSCROLLPSDETECT;

  u8TimerSpeedTuneTrigger			= TUN_CONFIG_DEFAULT_T_SPEEDTUNETRIGGER;//250 msec
  u8N_SpeedTuneTrigger				= TUN_CONFIG_DEFAULT_N_SPEEDTUNETRIGGER;//3 rotations
  u8N_SpeedTuneMultiplier_FM		= TUN_CONFIG_DEFAULT_N_SPEEDTUNEMULTIPLIER_FM;//5
  u8N_SpeedTuneMultiplier_AM		= TUN_CONFIG_DEFAULT_N_SPEEDTUNEMULTIPLIER_AM;//10
  u8NoOfPSChangesForScrollDetect	= TUN_CONFIG_DEFAULT_NOOFPSCHANGES_FORSCROLLDETECT;//3

  u8TimerPTY_Hold				= TUN_CONFIG_DEFAULT_TIMERPTY_HOLD;
  u16CRC 						= 0x00;//0 for all projects

  /** No KDS entry for these config params present.
  Assigning default values */
  u8TunerMaster								= TUN_CONFIG_DEFAULT_TUNER_MASTER;
  u32PtyFilterForSeek						= TUN_CONFIG_DEFAULT_PTY_FILTER_FOR_SEEK;
  u8ATStationInfoSwitches					= TUN_CONFIG_DEFAULT_ATSTATIONINFO_SWITCHES;
  u8TimerScanRadio							= TUN_CONFIG_DEFAULT_TIMERSCANRADIO;
  u8TimerAM_MinTunTime						= TUN_CONFIG_DEFAULT_TIMERAM_MINTUNTIME;
  u8TimerAF_InhibitAfterManTune				= TUN_CONFIG_DEFAULT_TIMERAF_INHIBITAFTERMANTUNE;
  u8MaxNumberofStationsInStnList			= TUN_CONFIG_DEFAULT_MAX_NO_STNS_IN_STATIONSLIST;


}

/*****************************************************************************
 * 
 ****************************************************************************/
tVoid tun_trNewKdsFmAmTunerParameter::vParse( tPU8 pu8KdsBuffer, tU32 u32Length)
{


  ETG_TRACE_USR1(( " tun_trNewKdsFmAmTunerParameter::vParse() ->\n"
                   " u32Length                   : %d bytes\n"
                   " KDS config                  : %d bytes\n"
                   " expected                    : %d bytes\n"
                   ,u32Length
                   ,288//DIA_C_U16_KDS_NISSAN_FMAMTUNER_PARAM_SET_LEN
                   ,TUN_CONFIG_NEW_KDS_FM_AM_TUNER_PARAMETER_LENGTH
                ));

  if( u32Length < TUN_CONFIG_NEW_KDS_FM_AM_TUNER_PARAMETER_LENGTH)
  {
    ETG_TRACE_ERR(( " tun_trNewKdsFmAmTunerParameter::vParse() -> Error: wrong length of KDS entry, default values are used now!\n"
                     " u32Length                   : %d bytes\n"
                     " KDS config                  : %d bytes\n"
                     " expected                    : %d bytes\n"
                     ,u32Length
                     ,288//DIA_C_U16_KDS_NISSAN_FMAMTUNER_PARAM_SET_LEN
                     ,TUN_CONFIG_NEW_KDS_FM_AM_TUNER_PARAMETER_LENGTH
                  ));
  }
  else

  {
    OSAL_pvMemoryCopy( sTag, &pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_Tag], (tU8)enNewKdsFmAmTunerParameter_len_Tag);
    OSAL_pvMemoryCopy( sVersion, &pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_MajorVersion], \
		(tU8)enNewKdsFmAmTunerParameter_len_MajorVersion + (tU8)enNewKdsFmAmTunerParameter_len_MinorVersion);

	u32LowFM						= TUN_CONFIG_GET_U32( &pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_LowFM]);
    u32HighFM						= TUN_CONFIG_GET_U32( &pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_HighFM]);
    u8StepSizeFM					= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_StepSizeFM];
    u8StepSizeManualFM				= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_StepSizeManualFM];
    u32LowMW						= TUN_CONFIG_GET_U32( &pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_LowMW]);
    u32HighMW						= TUN_CONFIG_GET_U32( &pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_HighMW]);
    u8StepSizeMW					= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_StepSizeMW];
    u8StepSizeManualMW				= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_StepSizeManualMW];
    u32LowLW						= TUN_CONFIG_GET_U32( &pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_LowLW]);
    u32HighLW						= TUN_CONFIG_GET_U32( &pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_HighLW]);
    u8StepSizeLW					= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_StepSizeLW];
    u8StepSizeManualLW				= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_StepSizeManualLW];
    
	u8FeatureControlI				= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_FeatureControlI];
	u8FeatureControlII				= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_FeatureControlII];

    u32FM_LSM						= TUN_CONFIG_GET_U32( &pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_FM_LSM]);
    u32MW_LSM						= TUN_CONFIG_GET_U32( &pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_MW_LSM]);
	u32LW_LSM						= TUN_CONFIG_GET_U32( &pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_LW_LSM]);

	u8ShowRadioTextStatusMessage	= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_ShowRadioTextStatusMessage];
	u8T_RadioTextNotAvailable		= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_T_RadioTextNotAvailable];
	u8T_Timeout_Status_RT			= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_T_Timeout_Status_RT];
	u8T_ShowTextMin					= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_T_ShowTextMin];
	u16T_MinDisplayDurationPerRadioTextPage = TUN_CONFIG_GET_U16( &pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_T_MinDisplayDurationPerRadioTextPage]);
	u8RadioTextMinLength			= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_RadioTextMinLength];
	u8RadioTextSpeedThreshold		= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_RadioTextSpeedThreshold];
	u8RadioTextSpeedThresholdHysteresis = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_RadioTextSpeedThresholdHysteresis];

	u8EmergencySeekScanNoOfStations = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_EmergencySeekScanNoOfStations];
    u8ShowFreqDuringFreqScan		= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_ShowFreqDuringFreqScan];
    u8SeekMode						= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_SeekMode];

    u8NumOfSeekPeriods              = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_NumOfSeekPeriods];
    u8ScanTime                      = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_ScanTime];
	/** assigned same as u8ScanTime i.e. KDS AMFM Parameter set byte 71. 
	since no seperate KDS parameter available */
    u8TimerAF_InhibitAfterManTune	= pu8KdsBuffer [ enNewKdsFmAmTunerParameter_idx_AFBestDelayTimeAftManualTuning ];
	u8TimerScanRadio				= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_ScanTime];

	u8AFBestDelayTimeAftManualTuning = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_AFBestDelayTimeAftManualTuning];
    u8AFBestDisplayFreezeAftManualTuning = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_AFBestDisplayFreezeAftManualTuning];

    u16AutoSeekStepDelay            = TUN_CONFIG_GET_U16( &pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_AutoSeekStepDelay]);
    u16PemanentManualStepDelay      = TUN_CONFIG_GET_U16( &pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_PemanentManualStepDelay]);

    u8WrongPIAcceptDelayAftStnSelection = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_WrongPIAcceptDelayAftStnSelection]; //78
    u8WrongPIAcceptAftStnlost		= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_WrongPIAcceptAftStnlost]; //79

	u8MultiManualStepsWithIntermediateFreq = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_MultiManualStepsWithIntermediateFreq];
    u8StationLists					= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_StationLists];
    u8FMListSortingDefault			= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_FMListSortingDefault];
    u8AMListSortingDefault			= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_AMListSortingDefault];

    u8TimerInhibitLstSortAfterSkip	= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_TimerInhibitLstSortAfterSkip];
	u8TimerCursorDelayAfterStnSel	= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_TimerCursorDelayAfterStnSel];

    u8TimeAMMinTuneListRemove		= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_TimeAMMinTuneListRemove];
    u8TimeAMStnRemoveAftDeslect		= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_TimeAMStnRemoveAftDeslect];
    u8AMStnListUpdateStrategy		= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_AMStnListUpdateStrategy];
 
	u8TestModeDataUpdateInterval	= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_TestModeDataUpdateInterval];
    u8AFListUpdateInterval			= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_AFListUpdateInterval];
    u8AFListResort					= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_AFListResort];
    u8AFListReleaseBehaviour		= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_AFListReleaseBehaviour];

	u8LandscapeUpdateOnShutdown		= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_LandscapeUpdateOnShutdown];
    u8TunerSwitches					= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_TunerSwitches];
	if(u8TunerSwitches & TUN_LASTPASSED_FREQ_BIT)
	{
	  u8TunerSwitches &= TUN_LASTPASSED_FREQ_DISABLE;
	}
	else
	{
	  u8TunerSwitches |= TUN_LASTPASSED_FREQ_ENABLE;
	}

	u8ADRFmChannelSeparation		= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_ADRFmChannelSeparation];

    u8ADRFmLimiting                 = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_ADRFmLimiting];
    u8ADRFmMultipath                = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_ADRFmMultipath];
    u8ADRFmHighCut                  = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_ADRFmHighCut];
    u8ADRFmSoftMute                 = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_ADRFmSoftMute];
    u8ADRAmSoftMute                 = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_ADRAmSoftMute];
    u8ADRFmDeemphasis               = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_ADRFmDeemphasis];
    u8ADRFmFrequencyResponse        = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_ADRFmFrequencyResponse];
    u8ADRAmFrequencyResponse        = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_ADRAmFrequencyResponse];
    u8ADRFmMisc                     = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_ADRFmMisc];
    u8ADRAmMisc                     = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_ADRAmMisc];
    u8ADRFmAmplificationLevel       = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_ADRFmAmplificationLevel];
    u8ADRAmAmplificationLevel       = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_ADRAmAmplificationLevel];
    u8ADRAM_Hicut_Configuration     = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_ADRAM_Hicut_Configuration];
    u8ADRAM_Deemphasis              = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_ADRAM_Deemphasis];
    u16MUTE_RAMP_1                  = TUN_CONFIG_GET_U16( &pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_MUTE_RAMP_1]);
    u16DEMUTE_RAMP_1                = TUN_CONFIG_GET_U16( &pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_DEMUTE_RAMP_1]);
    u16MUTE_RAMP_2                  = TUN_CONFIG_GET_U16( &pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_MUTE_RAMP_2]);
    u16DEMUTE_RAMP_2                = TUN_CONFIG_GET_U16( &pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_DEMUTE_RAMP_2]);
    u8BehaviourSwitches             = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_BehaviourSwitches];

    u8FieldStrengthSeekLevel_FM_RDS	= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_SeekLevel_FM_RDS_FldStrength];
	u8QualitySeekLevel_FM_RDS		= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_SeekLevel_FM_RDS_Quality];
	u8FieldStrengthSeekLevel_FM_NONRDS	= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_SeekLevel_FM_NONRDS_FldStrength];
	u8QualitySeekLevel_FM_NONRDS	= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_SeekLevel_FM_NONRDS_Quality];
	u8FieldStrengthSeekLevel_LW		= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_SeekLevel_LW_FldStrength];
	u8QualitySeekLevel_LW			= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_SeekLevel_LW_Quality];
	u8FieldStrengthSeekLevel_MW		= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_SeekLevel_MW_FldStrength];
	u8QualitySeekLevel_MW			= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_SeekLevel_MW_Quality];
	u8FieldStrengthSeekLevel_SW		= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_SeekLevel_SW_FldStrength];
	u8QualitySeekLevel_SW			= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_SeekLevel_SW_Quality];
	u8FieldStrengthSeekLevel_TRF	= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_SeekLevel_TRF_FldStrength];
	u8QualitySeekLevel_TRF			= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_SeekLevel_TRF_Quality];
	u8FieldStrengthSeekLevel_WB		= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_SeekLevel_WB_FldStrength];
	u8QualitySeekLevel_WB			= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_SeekLevel_WB_Quality];
  
	u8FieldStrengthListLevel_FM_RDS	= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_ListLevel_FM_RDS_FldStrength];
	u8QualityListLevel_FM_RDS		= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_ListLevel_FM_RDS_Quality];
	u8FieldStrengthListLevel_FM_NONRDS	= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_ListLevel_FM_NONRDS_FldStrength];
	u8QualityListLevel_FM_NONRDS	= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_ListLevel_FM_NONRDS_Quality];
	u8FieldStrengthListLevel_LW		= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_ListLevel_LW_FldStrength];
	u8QualityListLevel_LW			= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_ListLevel_LW_Quality];
	u8FieldStrengthListLevel_MW		= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_ListLevel_MW_FldStrength];
	u8QualityListLevel_MW			= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_ListLevel_MW_Quality];
	u8FieldStrengthListLevel_SW		= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_ListLevel_SW_FldStrength];
	u8QualityListLevel_SW			= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_ListLevel_SW_Quality];
	u8FieldStrengthListLevel_TRF	= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_ListLevel_TRF_FldStrength];
	u8QualityListLevel_TRF			= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_ListLevel_TRF_Quality];
	u8FieldStrengthListLevel_WB		= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_ListLevel_WB_FldStrength];
	u8QualityListLevel_WB			= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_ListLevel_WB_Quality];

	u8SHARX							= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_SHARX];
	u8HICUT							= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_HICUT];

    u32AvailableBands				= TUN_CONFIG_GET_U32( &pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_AvailableBands]);
    u32CombiBands					= TUN_CONFIG_GET_U32( &pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_CombiBands]);

    u8ADRSwitches1                  = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_ADRSwitches1];
    u8ADRSwitches2                  = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_ADRSwitches2];
    u8ADRSwitches3                  = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_ADRSwitches3];
    u8ADRSwitches4                  = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_ADRSwitches4];

    enPhysicalTunerConfigId         = (tenFmDabHdTunerMode)pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_PhysicalTunerConfigId];
    u8AntennaConfigId               = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_AntennaConfigId];
    u8ExchangeAntennaLevel          = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_EXCHANGE_ANTENNA_LEVEL];
    u8PitestLevel                   = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_PITEST_LEVEL];
    u8NrdsLevel                     = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_NRDS_LEVEL];
    u8EonLevel                      = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_EON_LEVEL];
    u8FMMemoLevel                   = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_FM_MEMO_LEVEL];
    u8PanicFieldstLevel             = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_PANIC_FIELDST_LEVEL];
    u8ParamRDSLevel                 = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_PARAM_RDS_LEVEL];
    u8RDSDataAcceptLevel            = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_RDS_DATA_ACCEPT_LEVEL];
    u8PIStepLevel                   = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_PI_STEP_LEVEL];
    u8DDAAntennaDistLevel           = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_DDA_ANTENNA_DIST_LEVEL];
    u8RDSSynDelayCnt                = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_RDS_SYN_DELAY_CNT];
    u8MpMeasureLevel                = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_MP_MEASURE_LEVEL];
    u8MpLevel2Level                 = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_MP_LEVEL2_LEVEL];
    u8MpVeryBadLevel                = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_MP_VERYBAD_LEVEL];
    u8MpMediumLevel                 = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_MP_MEDIUM_LEVEL];
    u8RDSBadLevel                   = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_RDS_BAD_LEVEL];
    u8IllegalAf                     = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_ILLEGAL_AF];
    u8AfSaveLevel                   = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_AF_SAVE_LEVEL];
    u8FirstRundum                   = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_FIRST_RUNDUM];
    u8SecondRundum                  = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_SECOND_RUNDUM];
    u8ThirdRundum                   = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_THIRD_RUNDUM];
    u8PIStepMode                    = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_PISTEP_MODE];
    u8AudioOffScanMode              = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_AUDIO_OFF_SCAN_MODE];
    u8AMLearnCycle                  = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_AM_LEARN_CYCLE];
    u8PIIgnore                      = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_PI_IGNORE];
    u8PISearchTime                  = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_PI_SEARCH_TIME];
    u8DisableSeamlessblending       = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_DisableSeamlessBlending];
    u8SlAdjThreshold                = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_SL_ADJ_THRESHOLD];
    u8SlMpThreshold                 = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_SL_MP_THRESHOLD];
    u8SpecialFMStartFrequency       = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_SPECIAL_FM_START_FREQUENCY];
	u8FMSharxLimit					= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_FM_Sharx_Limit];
	u8HDScanEveryFrequency			= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_HDScanEveryFrequency];
	u8HWSubConfig					= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_HW_SubConfig];

	u8FM2DABDelayTime								= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_FM2DABDelayTime]; 

	u8AMFMHicutExtender								= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_AMFMHicutExtender];
	u8FMSoftwareMuteTiming							= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_FMSoftMuteTiming];
	u8AMFMMisscellaneous2							= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_AMFMMisscellaneous2];
	u8FMHicutExtenderFieldstrengthThreshold			= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_FMHicutExtenderFieldstrengthThreshold];
	u8FMHicutExtenderFieldstrengthFilterOffset		= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_FMHicutExtenderFieldstrengthFilterOffset];
	u8FMChannelSeparationMaximum					= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_FMChannelSeperationMaximum];
	u8FMChannelSeparationTiming						= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_FMChannelSeperationTiming];
	u8FMHicutFieldstrengthThreshold					= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_FMHicutFieldstrengthThreshold];
	u8FMHicutTiming									= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_FMHicutTiming];
	u8FMDMQSensitivity								= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_FMDMQSensitivity];
	u8FMDeviationAdaption							= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_FMDeviationAdaption];
	u8AMNoiseBlanker228kHzSensitivity				= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_AMNoiseBlanker228kHzSensitivity];
	u8AMNoiseBlanker45_6kHzSensitivity				= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_AMNoiseBlanker45_6kHzSensitivity];
	u8AMHicutExtenderFieldstrengthThreshold			= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_AMHicutExtenderFieldstrengthThreshold];
	u8AMHicutExtenderFieldstrengthFilterOffset		= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_AMHicutExtenderFieldstrengthFilterOffset];

	u8DemodulationSwitchAudioTuner1					= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_DemodulationSwitchAudioTuner1];
	u8DemodulationSwitchAudioTuner2					= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_DemodulationSwitchAudioTuner2];

	u8TimerNonRDS_Detect			= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_TimerNonRDS_Detect];
    u8TimerPS_DelAfterRDSLost		= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_TimerPS_DelAfterRDSLost];
    u8TimerScrollPSDetect			= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_TimerScrollPSDetect];
    u8TimerSpeedTuneTrigger			= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_TimerSpeedTuneTrigger];
    u8N_SpeedTuneTrigger			= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_N_SpeedTuneTrigger];
    u8N_SpeedTuneMultiplier_FM		= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_N_SpeedTuneMultiplier_FM];
    u8N_SpeedTuneMultiplier_AM		= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_N_SpeedTuneMultiplier_AM];
    u8NoOfPSChangesForScrollDetect  = pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_NoOfPSChangesForScrollDetect];
    u8TimerPTY_Hold					= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_TimePTYHold];

    u16CRC							= pu8KdsBuffer[ enNewKdsFmAmTunerParameter_idx_CRC];

    bKdsDataRead = TRUE;
  }
}


/*****************************************************************************
 * 
 ****************************************************************************/
tVoid tun_trNewKdsFmAmTunerParameter::vTrace( tVoid)
{
  if( etg_bIsTraceActiveShort(((0xFFFFu & (tU16)TUN_TRACE_CLASS_KDSCONFIG) << 16) | (tU16)TR_LEVEL_USER_2) == TRUE)
  {
    
    ETG_TRACE_USR2(( " BEGIN ------------------------------------------------------------------------------------"            ));
    ETG_TRACE_USR2(( " bKdsDataRead                : 0x%02X  (0x00: Default values, 0x01: data from KDS read)"  ,bKdsDataRead ));
    ETG_TRACE_USR2(( " sTag                        : %8s"        ,sTag                                 ));
    ETG_TRACE_USR2(( " sVersion                    : %3s"        ,sVersion                             ));
//    ETG_TRACE_USR2(( " u8TunerMaster               : 0x%02X"     ,u8TunerMaster                        ));
//    ETG_TRACE_USR2(( " enArea                      : %02x"       ,ETG_CENUM( tenTunerArea,    enArea)  ));
    ETG_TRACE_USR2(( " u32LowFM                    : %8d kHz"    ,u32LowFM                             ));
    ETG_TRACE_USR2(( " u32HighFM                   : %8d kHz"    ,u32HighFM                            ));
    ETG_TRACE_USR2(( " u8tepSizeFM                 : %8d kHz"    ,u8StepSizeFM                         ));
    ETG_TRACE_USR2(( " u8StepSizeManualFM          : %8d kHz"    ,u8StepSizeManualFM                   ));
    ETG_TRACE_USR2(( " u32LowMW                    : %8d kHz"    ,u32LowMW                             ));
    ETG_TRACE_USR2(( " u32HighMW                   : %8d kHz"    ,u32HighMW                            ));
    ETG_TRACE_USR2(( " u8StepSizeMW                : %8d kHz"    ,u8StepSizeMW                         ));
    ETG_TRACE_USR2(( " u8StepSizeManualMW          : %8d kHz"    ,u8StepSizeManualMW                   ));
    ETG_TRACE_USR2(( " u32LowLW                    : %8d kHz"    ,u32LowLW                             ));
    ETG_TRACE_USR2(( " u32HighLW                   : %8d kHz"    ,u32HighLW                            ));
    ETG_TRACE_USR2(( " u8StepSizeLW                : %8d kHz"    ,u8StepSizeLW                         ));
    ETG_TRACE_USR2(( " u8StepSizeManualLW          : %8d kHz"    ,u8StepSizeManualLW                   ));

    ETG_TRACE_USR2(( " u8FeatureControlI           : 	  0x%02X",u8FeatureControlI                    ));
    ETG_TRACE_USR2(( " u8FeatureControlII          : 	  0x%02X",u8FeatureControlII                   ));
 
	ETG_TRACE_USR2(( " u32FM_LSM                   : %8d kHz"    ,u32FM_LSM                            ));
    ETG_TRACE_USR2(( " u32MW_LSM                   : %8d kHz"    ,u32MW_LSM							   ));
    ETG_TRACE_USR2(( " u32LW_LSM                   : %8d kHz"    ,u32LW_LSM                            ));

    ETG_TRACE_USR2(( " u8ShowRadioTextStatusMessage:	  0x%02X",u8ShowRadioTextStatusMessage         ));
    ETG_TRACE_USR2(( " u8T_RadioTextNotAvailable   : %8d"        ,u8T_RadioTextNotAvailable            ));
    ETG_TRACE_USR2(( " u8T_Timeout_Status_RT       : %8ds"       ,u8T_Timeout_Status_RT				   ));
    ETG_TRACE_USR2(( " u8T_ShowTextMin			   : %8ds"       ,u8T_ShowTextMin					   ));
    ETG_TRACE_USR2(( " u16T_MinDisplayDurationPerRadioTextPage	: %8ds"        ,u16T_MinDisplayDurationPerRadioTextPage    ));
    ETG_TRACE_USR2(( " u8RadioTextMinLength		   : %8d"        ,u8RadioTextMinLength				   ));
    ETG_TRACE_USR2(( " u8RadioTextSpeedThreshold   : %8d"        ,u8RadioTextSpeedThreshold            ));
    ETG_TRACE_USR2(( " u8RadioTextSpeedThresholdHysteresis		: %8d"        ,u8RadioTextSpeedThresholdHysteresis         ));


    ETG_TRACE_USR2(( " u8EmergencySeekScanNoOfStations			: %8d"        ,u8EmergencySeekScanNoOfStations			   ));
    ETG_TRACE_USR2(( " u8ShowFreqDuringFreqScan    : %8d"        ,u8ShowFreqDuringFreqScan            ));
    ETG_TRACE_USR2(( " u8SeekMode				   : %8d"        ,u8SeekMode						  ));

    ETG_TRACE_USR2(( " u8NumOfSeekPeriods          : %8d  "        ,u8NumOfSeekPeriods                ));
    ETG_TRACE_USR2(( " u8ScanTime                  : %8d s"        ,u8ScanTime                        ));
    ETG_TRACE_USR2(( " u8AFBestDelayTimeAftManualTuning         : %8d s"       ,u8AFBestDelayTimeAftManualTuning           ));
    ETG_TRACE_USR2(( " u8AFBestDisplayFreezeAftManualTuning     : %8d s"       ,u8AFBestDisplayFreezeAftManualTuning	   ));
    ETG_TRACE_USR2(( " u16AutoSeekStepDelay        : %8d ms"       ,u16AutoSeekStepDelay              ));
    ETG_TRACE_USR2(( " u16PemanentManualStepDelay  : %8d ms"       ,u16PemanentManualStepDelay        ));
    ETG_TRACE_USR2(( " u8WrongPIAcceptDelayAftStnSelection      : %8d ms"       ,u8WrongPIAcceptDelayAftStnSelection       ));
    ETG_TRACE_USR2(( " u8WrongPIAcceptAftStnlost   : %8d ms"       ,u8WrongPIAcceptAftStnlost         ));

    ETG_TRACE_USR2(( " u8MultiManualStepsWithIntermediateFreq   :	  0x%02X"   ,u8MultiManualStepsWithIntermediateFreq    ));
    ETG_TRACE_USR2(( " u8StationLists			   :	  0x%02X"  ,u8StationLists			          ));
    ETG_TRACE_USR2(( " u8FMListSortingDefault      : %8d   "       ,u8FMListSortingDefault            ));
    ETG_TRACE_USR2(( " u8AMListSortingDefault      : %8d   "       ,u8AMListSortingDefault            ));

    ETG_TRACE_USR2(( " u8TimerInhibitLstSortAfterSkip		    : %8ds  "       ,u8TimerInhibitLstSortAfterSkip            ));
    ETG_TRACE_USR2(( " u8TimerCursorDelayAfterStnSel		    : %8ds  "       ,u8TimerCursorDelayAfterStnSel             ));


    ETG_TRACE_USR2(( " u8TimeAMMinTuneListRemove	: %8ds  "       ,u8TimeAMMinTuneListRemove         ));
    ETG_TRACE_USR2(( " u8TimeAMStnRemoveAftDeslect	: %8ds  "       ,u8TimeAMStnRemoveAftDeslect       ));
    ETG_TRACE_USR2(( " u8AMStnListUpdateStrategy	: %8d   "       ,u8AMStnListUpdateStrategy         ));

    ETG_TRACE_USR2(( " u8TestModeDataUpdateInterval	: %8ds  "       ,u8TestModeDataUpdateInterval      ));
    ETG_TRACE_USR2(( " u8AFListUpdateInterval		: %8ds  "       ,u8AFListUpdateInterval			   ));
    ETG_TRACE_USR2(( " u8AFListResort				: %8d   "       ,u8AFListResort					   ));
    ETG_TRACE_USR2(( " u8AFListReleaseBehaviour		: %8d   "       ,u8AFListReleaseBehaviour          ));

    ETG_TRACE_USR2(( " u8LandscapeUpdateOnShutdown	:	  0x%02X"   ,u8LandscapeUpdateOnShutdown	   ));
    ETG_TRACE_USR2(( " u8TunerSwitches				:	  0x%02X"   ,u8TunerSwitches		           ));

    ETG_TRACE_USR2(( " u8ADRFmChannelSeparation     :     0x%02X"   ,u8ADRFmChannelSeparation          ));
    ETG_TRACE_USR2(( " u8ADRFmLimiting              :     0x%02X"   ,u8ADRFmLimiting                   ));
    ETG_TRACE_USR2(( " u8ADRFmMultipath             :     0x%02X"   ,u8ADRFmMultipath                  ));
    ETG_TRACE_USR2(( " u8ADRFmHighCut               :     0x%02X"   ,u8ADRFmHighCut                    ));
    ETG_TRACE_USR2(( " u8ADRFmSoftMute              :     0x%02X"   ,u8ADRFmSoftMute                   ));
    ETG_TRACE_USR2(( " u8ADRAmSoftMute              :     0x%02X"   ,u8ADRAmSoftMute                   ));
    ETG_TRACE_USR2(( " u8ADRFmDeemphasis            :     0x%02X"   ,u8ADRFmDeemphasis                 ));
    ETG_TRACE_USR2(( " u8ADRFmFrequencyResponse     :     0x%02X"   ,u8ADRFmFrequencyResponse          ));
    ETG_TRACE_USR2(( " u8ADRAmFrequencyResponse     :     0x%02X"   ,u8ADRAmFrequencyResponse          ));
    ETG_TRACE_USR2(( " u8ADRFmMisc                  :     0x%02X"   ,u8ADRFmMisc                       ));
    ETG_TRACE_USR2(( " u8ADRAmMisc                  :     0x%02X"   ,u8ADRAmMisc                       ));
    ETG_TRACE_USR2(( " u8ADRFmAmplificationLevel    :     0x%02X"   ,u8ADRFmAmplificationLevel         ));
    ETG_TRACE_USR2(( " u8ADRAmAmplificationLevel    :     0x%02X"   ,u8ADRAmAmplificationLevel         ));
    ETG_TRACE_USR2(( " u8ADRAM_Hicut_Configuration  :     0x%02X"   ,u8ADRAM_Hicut_Configuration       ));
    ETG_TRACE_USR2(( " u8ADRAM_Deemphasis           :     0x%02X"   ,u8ADRAM_Deemphasis                ));
    ETG_TRACE_USR2(( " u16MUTE_RAMP_1               :     0x%04X"   ,u16MUTE_RAMP_1                    ));
    ETG_TRACE_USR2(( " u16DEMUTE_RAMP_1             :     0x%04X"   ,u16DEMUTE_RAMP_1                  ));
    ETG_TRACE_USR2(( " u16MUTE_RAMP_2               :     0x%04X"   ,u16MUTE_RAMP_2                    ));
    ETG_TRACE_USR2(( " u16DEMUTE_RAMP_2             :     0x%04X"   ,u16DEMUTE_RAMP_2                  ));

    ETG_TRACE_USR2(( " u8BehaviourSwitches          :     0x%02X"   ,u8BehaviourSwitches               ));

    ETG_TRACE_USR2(( " u8FieldStrengthSeekLevel_FM_RDS		: %8d   "       ,u8FieldStrengthSeekLevel_FM_RDS    ));
    ETG_TRACE_USR2(( " u8QualitySeekLevel_FM_RDS			: %8d   "       ,u8QualitySeekLevel_FM_RDS          ));
    ETG_TRACE_USR2(( " u8FieldStrengthSeekLevel_FM_NONRDS	: %8d   "       ,u8FieldStrengthSeekLevel_FM_NONRDS ));
    ETG_TRACE_USR2(( " u8QualitySeekLevel_FM_NONRDS			: %8d   "       ,u8QualitySeekLevel_FM_NONRDS       ));
    ETG_TRACE_USR2(( " u8FieldStrengthSeekLevel_LW			: %8d   "       ,u8FieldStrengthSeekLevel_LW        ));
    ETG_TRACE_USR2(( " u8QualitySeekLevel_LW				: %8d   "       ,u8QualitySeekLevel_LW				));
    ETG_TRACE_USR2(( " u8FieldStrengthSeekLevel_MW			: %8d   "       ,u8FieldStrengthSeekLevel_MW        ));
    ETG_TRACE_USR2(( " u8QualitySeekLevel_MW				: %8d   "       ,u8QualitySeekLevel_MW				));
    ETG_TRACE_USR2(( " u8FieldStrengthSeekLevel_SW			: %8d   "       ,u8FieldStrengthSeekLevel_SW        ));
    ETG_TRACE_USR2(( " u8QualitySeekLevel_SW				: %8d   "       ,u8QualitySeekLevel_SW				));
    ETG_TRACE_USR2(( " u8FieldStrengthSeekLevel_TRF			: %8d   "       ,u8FieldStrengthSeekLevel_TRF       ));
    ETG_TRACE_USR2(( " u8QualitySeekLevel_TRF				: %8d   "       ,u8QualitySeekLevel_TRF				));
    ETG_TRACE_USR2(( " u8FieldStrengthSeekLevel_WB			: %8d   "       ,u8FieldStrengthSeekLevel_WB        ));
    ETG_TRACE_USR2(( " u8QualitySeekLevel_WB				: %8d   "       ,u8QualitySeekLevel_WB				));



    ETG_TRACE_USR2(( " u8FieldStrengthListLevel_FM_RDS		: %8d   "       ,u8FieldStrengthListLevel_FM_RDS    ));
    ETG_TRACE_USR2(( " u8QualityListLevel_FM_RDS			: %8d   "       ,u8QualityListLevel_FM_RDS          ));
    ETG_TRACE_USR2(( " u8FieldStrengthListLevel_FM_NONRDS	: %8d   "       ,u8FieldStrengthListLevel_FM_NONRDS ));
    ETG_TRACE_USR2(( " u8QualityListLevel_FM_NONRDS			: %8d   "       ,u8QualityListLevel_FM_NONRDS       ));
    ETG_TRACE_USR2(( " u8FieldStrengthListLevel_LW			: %8d   "       ,u8FieldStrengthListLevel_LW        ));
    ETG_TRACE_USR2(( " u8QualityListLevel_LW				: %8d   "       ,u8QualityListLevel_LW				));
    ETG_TRACE_USR2(( " u8FieldStrengthListLevel_MW			: %8d   "       ,u8FieldStrengthListLevel_MW        ));
    ETG_TRACE_USR2(( " u8QualityListLevel_MW				: %8d   "       ,u8QualityListLevel_MW				));
    ETG_TRACE_USR2(( " u8FieldStrengthListLevel_SW			: %8d   "       ,u8FieldStrengthListLevel_SW        ));
    ETG_TRACE_USR2(( " u8QualityListLevel_SW				: %8d   "       ,u8QualityListLevel_SW				));
    ETG_TRACE_USR2(( " u8FieldStrengthListLevel_TRF			: %8d   "       ,u8FieldStrengthListLevel_TRF       ));
    ETG_TRACE_USR2(( " u8QualityListLevel_TRF				: %8d   "       ,u8QualityListLevel_TRF				));
    ETG_TRACE_USR2(( " u8FieldStrengthListLevel_WB			: %8d   "       ,u8FieldStrengthListLevel_WB        ));
    ETG_TRACE_USR2(( " u8QualityListLevel_WB				: %8d   "       ,u8QualityListLevel_WB				));
	
	ETG_TRACE_USR2(( " u8SHARX                     :     0x%02X"   ,u8SHARX                           ));
    ETG_TRACE_USR2(( " u8HICUT                     :     0x%02X"   ,u8HICUT                           ));

    ETG_TRACE_USR2(( " u32AvailableBands           : 0x%08X"       ,u32AvailableBands                 ));
    ETG_TRACE_USR2(( " u32CombiBands			   : 0x%08X"       ,u32CombiBands                     ));

    ETG_TRACE_USR2(( " u8ADRSwitches1              :     0x%02X"   ,u8ADRSwitches1                    ));
    ETG_TRACE_USR2(( " u8ADRSwitches2              :     0x%02X"   ,u8ADRSwitches2                    ));
    ETG_TRACE_USR2(( " u8ADRSwitches3              :     0x%02X"   ,u8ADRSwitches3                    ));
    ETG_TRACE_USR2(( " u8ADRSwitches4              :     0x%02X"   ,u8ADRSwitches4                    ));

    ETG_TRACE_USR2(( " enPhysicalTunerConfigId     :     0x%02X"   ,ETG_CENUM( tenFmDabHdTunerMode, enPhysicalTunerConfigId) ));
    ETG_TRACE_USR2(( " u8AntennaConfigId           :     0x%02X"   ,u8AntennaConfigId                 ));
    ETG_TRACE_USR2(( " u8ExchangeAntennaLevel      : %8d dBuV"     ,u8ExchangeAntennaLevel            ));
    ETG_TRACE_USR2(( " u8PitestLevel               : %8d dBuV"     ,u8PitestLevel                     ));
    ETG_TRACE_USR2(( " u8NrdsLevel                 : %8d dBuV"     ,u8NrdsLevel                       ));
    ETG_TRACE_USR2(( " u8EonLevel                  : %8d dBuV"     ,u8EonLevel                        ));
    ETG_TRACE_USR2(( " u8FMMemoLevel               : %8d dBuV"     ,u8FMMemoLevel                     ));
    ETG_TRACE_USR2(( " u8PanicFieldstLevel         : %8d dBuV"     ,u8PanicFieldstLevel               ));
    ETG_TRACE_USR2(( " u8ParamRDSLevel             : %8d dBuV"     ,u8ParamRDSLevel                   ));
    ETG_TRACE_USR2(( " u8RDSDataAcceptLevel        : %8d dBuV"     ,u8RDSDataAcceptLevel              ));
    ETG_TRACE_USR2(( " u8PIStepLevel               : %8d dBuV"     ,u8PIStepLevel                     ));
    ETG_TRACE_USR2(( " u8DDAAntennaDistLevel       : %8d"          ,u8DDAAntennaDistLevel             ));
    ETG_TRACE_USR2(( " u8RDSSynDelayCnt            : %8d"          ,u8RDSSynDelayCnt                  ));
    ETG_TRACE_USR2(( " u8MpMeasureLevel            : %8d"          ,u8MpMeasureLevel                  ));
    ETG_TRACE_USR2(( " u8MpLevel2Level             : %8d"          ,u8MpLevel2Level                   ));
    ETG_TRACE_USR2(( " u8MpVeryBadLevel            : %8d"          ,u8MpVeryBadLevel                  ));
    ETG_TRACE_USR2(( " u8MpMediumLevel             : %8d"          ,u8MpMediumLevel                   ));
    ETG_TRACE_USR2(( " u8RDSBadLevel               : %8d"          ,u8RDSBadLevel                     ));
    ETG_TRACE_USR2(( " u8IllegalAf                 : %8d"          ,u8IllegalAf                       ));
    ETG_TRACE_USR2(( " u8AfSaveLevel               : %8d"          ,u8AfSaveLevel                     ));
    ETG_TRACE_USR2(( " u8FirstRundum               : %8d"          ,u8FirstRundum                     ));
    ETG_TRACE_USR2(( " u8SecondRundum              : %8d s"        ,u8SecondRundum                    ));
    ETG_TRACE_USR2(( " u8ThirdRundum               : %8d s"        ,u8ThirdRundum                     ));
    ETG_TRACE_USR2(( " u8PIStepMode                : %8d s"        ,u8PIStepMode                      ));
    ETG_TRACE_USR2(( " u8AudioOffScanMode          : %8d"          ,u8AudioOffScanMode                ));
    ETG_TRACE_USR2(( " u8AMLearnCycle              : %8d"          ,u8AMLearnCycle                    ));
    ETG_TRACE_USR2(( " u8PIIgnore                  : %8d"          ,u8PIIgnore                        ));
    ETG_TRACE_USR2(( " u8PISearchTime              : %8d"          ,u8PISearchTime					  ));
    ETG_TRACE_USR2(( " u8DisableSeamlessblending   : %8d"          ,u8DisableSeamlessblending         ));
    ETG_TRACE_USR2(( " u8SlAdjThreshold            : %8d"          ,u8SlAdjThreshold                  ));
    ETG_TRACE_USR2(( " u8SlMpThreshold             : %8d"          ,u8SlMpThreshold                   ));
    ETG_TRACE_USR2(( " u8SpecialFMStartFrequency   : %8d"          ,u8SpecialFMStartFrequency         ));
	ETG_TRACE_USR2(( " u8FMSharxLimit			   : %8d"          ,u8FMSharxLimit					  ));
	ETG_TRACE_USR2(( " u8HDScanEveryFrequency	   : %8d"          ,u8HDScanEveryFrequency			  ));
	ETG_TRACE_USR2(( " u8HWSubConfig			   : %8d"          ,u8HWSubConfig					  ));
	ETG_TRACE_USR2(( " u8FM2DABDelayTime		   : %8d"          ,u8FM2DABDelayTime				  ));

	
	ETG_TRACE_USR2(( " u8AMFMHicutExtender							: %8d"          ,u8AMFMHicutExtender							));
	ETG_TRACE_USR2(( " u8FMSoftwareMuteTiming						: %8d"          ,u8FMSoftwareMuteTiming							));
	ETG_TRACE_USR2(( " u8AMFMMisscellaneous2						: %8d"          ,u8AMFMMisscellaneous2							));
	ETG_TRACE_USR2(( " u8FMHicutExtenderFieldstrengthThreshold		: %8d"          ,u8FMHicutExtenderFieldstrengthThreshold		));
	ETG_TRACE_USR2(( " u8FMHicutExtenderFieldstrengthFilterOffset	: %8d"          ,u8FMHicutExtenderFieldstrengthFilterOffset		));
	ETG_TRACE_USR2(( " u8FMChannelSeparationMaximum					: %8d"          ,u8FMChannelSeparationMaximum					));
	ETG_TRACE_USR2(( " u8FMChannelSeparationTiming					: %8d"          ,u8FMChannelSeparationTiming					));
	ETG_TRACE_USR2(( " u8FMHicutFieldstrengthThreshold				: %8d"          ,u8FMHicutFieldstrengthThreshold				));
	ETG_TRACE_USR2(( " u8FMHicutTiming								: %8d"          ,u8FMHicutTiming								));
	ETG_TRACE_USR2(( " u8FMDMQSensitivity							: %8d"          ,u8FMDMQSensitivity								));
	ETG_TRACE_USR2(( " u8FMDeviationAdaption						: %8d"          ,u8FMDeviationAdaption							));
	ETG_TRACE_USR2(( " u8AMNoiseBlanker228kHzSensitivity			: %8d"          ,u8AMNoiseBlanker228kHzSensitivity				));
	ETG_TRACE_USR2(( " u8AMNoiseBlanker45_6kHzSensitivity			: %8d"          ,u8AMNoiseBlanker45_6kHzSensitivity				));
	ETG_TRACE_USR2(( " u8AMHicutExtenderFieldstrengthThreshold		: %8d"          ,u8AMHicutExtenderFieldstrengthThreshold		));
	ETG_TRACE_USR2(( " u8AMHicutExtenderFieldstrengthFilterOffset	: %8d"          ,u8AMHicutExtenderFieldstrengthFilterOffset		));

	ETG_TRACE_USR2(( " u8DemodulationSwitchAudioTuner1				: %8d"          ,u8DemodulationSwitchAudioTuner1				));
	ETG_TRACE_USR2(( " u8DemodulationSwitchAudioTuner2				: %8d"          ,u8DemodulationSwitchAudioTuner2				));
	
    ETG_TRACE_USR2(( " u8TimerNonRDS_Detect		   : %8ds"         ,u8TimerNonRDS_Detect              ));
    ETG_TRACE_USR2(( " u8TimerPS_DelAfterRDSLost   : %8ds"         ,u8TimerPS_DelAfterRDSLost		  ));
    ETG_TRACE_USR2(( " u8TimerScrollPSDetect	   : %8ds"         ,u8TimerScrollPSDetect			  ));

    ETG_TRACE_USR2(( " u8TimerSpeedTuneTrigger	   : %8ds"         ,u8TimerSpeedTuneTrigger			  ));
    ETG_TRACE_USR2(( " u8N_SpeedTuneTrigger		   : %8d"          ,u8N_SpeedTuneTrigger			  ));
    ETG_TRACE_USR2(( " u8N_SpeedTuneMultiplier_FM  : %8d"          ,u8N_SpeedTuneMultiplier_FM		  ));
    ETG_TRACE_USR2(( " u8N_SpeedTuneMultiplier_AM  : %8d"          ,u8N_SpeedTuneMultiplier_AM		  ));
    ETG_TRACE_USR2(( " u8NoOfPSChangesForScrollDetect  : %8d"      ,u8NoOfPSChangesForScrollDetect	  ));

    ETG_TRACE_USR2(( " u16CRC						: 0x%04X"      ,u16CRC							  ));

    ETG_TRACE_USR2(( " END   ------------------------------------------------------------"            ));
  }
}

/******************* New KDS AM FM Parameter Set ENDS ***********************/



/*****************************************************************************
******************* New KDS GENERIC Parameter Set ******************************
*****************************************************************************/

/*****************************************************************************
 * Constructor
 ****************************************************************************/
tun_trNewKdsGenericTunerParameter::tun_trNewKdsGenericTunerParameter( tVoid)
{
  ETG_TRACE_USR1_CLS(( TUN_TRACE_CLASS_TUNAPP, \
	  " tun_trNewKdsGenericTunerParameter::tun_trNewKdsGenericTunerParameter() -> Constructor"
      ));

  tun_trNewKdsGenericTunerParameter::vSetToDefaultValues();

}

/*****************************************************************************
 * Set all KDS values to default
 ****************************************************************************/
tVoid tun_trNewKdsGenericTunerParameter::vSetToDefaultValues( tVoid)
{
  ETG_TRACE_USR1_CLS(( TUN_TRACE_CLASS_TUNAPP, " tun_trNewKdsGenericTunerParameter::vSetToDefaultValues() -> "
                ));

  bKdsDataRead                    = FALSE;
  
  sTag[ 0]     = 'D';
  sTag[ 1]     = 'e';
  sTag[ 2]     = 'f';
  sTag[ 3]     = 'a';
  sTag[ 4]     = 'u';
  sTag[ 5]     = 'l';
  sTag[ 6]     = 't';
  sTag[ 7]     = '-';

  sVersion[ 0] = '0';//GENERIC_TO_BE_DECIDED
  sVersion[ 1] = '0';//GENERIC_TO_BE_DECIDED
  sVersion[ 2] = '2';//GENERIC_TO_BE_DECIDED

  u32InitialAutoStoreListId				= TUN_CONFIG_DEFAULT_INITIAL_AUTOSTORE_LISTID; 
  u8AutostoreUnusedPresetParameter		= TUN_CONFIG_DEFAULT_AUTOSTORE_UNUSED_PRESET_PARAM; 
  u8InitialAutostore_NoOfStations		= TUN_CONFIG_DEFAULT_INITIAL_AUTOSTORE_NO_OF_STATIONS; 
  u8UserAutostore_NoOfStations			= TUN_CONFIG_DEFAULT_USER_AUTOSTORE_NO_OF_STATIONS;
  u8AutostoreSortingCriteriaFM			= TUN_CONFIG_DEFAULT_AUTOSTORE_SORTING_CRITERIA_FM;
  u8AutostoreSortingCriteriaAM			= TUN_CONFIG_DEFAULT_AUTOSTORE_SORTING_CRITERIA_AM;

  u8AutostoreSortingCriteriaDAB			= TUN_CONFIG_DEFAULT_AUTOSTORE_SORTING_CRITERIA_DAB;
  u8AutostoreBehavioralSwitches			= TUN_CONFIG_DEFAULT_AUTOSTORE_BEHAVIOURAL_SWITCHES; 

  u8BandToTuneAfterInitialAutostore		= TUN_CONFIG_DEFAULT_BANDTOTUNE_AFTR_INITIAL_AUTOSTORE; 
  u8BandToTuneAfterUserAutostore		= TUN_CONFIG_DEFAULT_BANDTOTUNE_AFTR_USER_AUTOSTORE; 

  //u8ReservedByte1[16];/**---------------------------  24 - 39 byte */  

  u8MuteOrAttenuationForAbsStnChngAMFM	= TUN_CONFIG_DEFAULT_MUTEORATTENUATION_FORABSSTNCHNG_AMFM;
  u8MuteOrAttenuationForSeekAMFM		= TUN_CONFIG_DEFAULT_MUTEORATTENUATION_FORSEEK_AMFM;
  u8MuteOrAttenuationForManualTuningAMFM = TUN_CONFIG_DEFAULT_MUTEORATTENUATION_FORMANUALTUNING_AMFM;

  u8TimetoReleaseAttenuationAMFM		= TUN_CONFIG_DEFAULT_TIME_TO_RELEASE_ATTENUATION_AMFM;
  u8TimeToReleaseDAB_FMLink				= TUN_CONFIG_DEFAULT_TIME_TO_RELEASE_DAB_FMLINK;

  u8TimeToReleaseDABPlus_FMLink			= TUN_CONFIG_DEFAULT_TIME_TO_RELEASE_DABPLUS_FMLINK;

  u8TimeToReleaseDMBA_FMLink			= TUN_CONFIG_DEFAULT_TIME_TO_RELEASE_DMBA_FMLINK;

  u8LevelCompensationForStationLoad		= TUN_CONFIG_DEFAULT_LEVEL_COMPENSATION_FOR_STNLOAD;
  //u8ReservedByte2[6];/**----------------------------  48 - 53 byte */

  u8MonitorFM_TA						= TUN_CONFIG_DEFAULT_MONITOR_FMTA;
  u8AudioTunerFM_TPSeek					= TUN_CONFIG_DEFAULT_AUDIOTUNER_FMTPSEEK;
  u8AnnouncementRequestHandling			= TUN_CONFIG_DEFAULT_ANNOUNCEMENT_REQHANDLING;

  u8FM_TPseekStrategy					= TUN_CONFIG_DEFAULT_TP_SEEK_STRATEGY;
  u8TimeDelayFM_TPStnAftVolZero			= TUN_CONFIG_DEFAULT_TIMERDELAYTPSTNSELAFTERVOL0;
  u8TimeDelayNoFM_TPStnFound			= TUN_CONFIG_DEFAULT_TIMERDELAYNOTPSTNFOUND;

  u8TimeDelayFM_TPStnSelAftRDSLost		= TUN_CONFIG_DEFAULT_TIMERDELAYTPSTNSELAFTERRDSLOST; 
  u8TimeFM_TAEndAftRDSLost				= TUN_CONFIG_DEFAULT_TIMERTAENDAFTERRDSLOST;

  u8BackgroundFM_TA						= TUN_CONFIG_DEFAULT_BACKGROUNDFM_TA;

  u8TraficAnnoInOtherSrc				= TUN_CONFIG_DEFAULT_TRAFFIC_ANNO_INOTHSRC;
  u8AnnouncementPrio_FM_TA				= TUN_CONFIG_DEFAULT_ANNOUNCEMENT_PRIO_FM_TA;
  u8AnnouncementPrio_FM_PTY31			= TUN_CONFIG_DEFAULT_ANNOUNCEMENT_PRIO_FM_PTY31;
  u8AnnouncementPrio_FM_NEWS			= TUN_CONFIG_DEFAULT_ANNOUNCEMENT_PRIO_FM_NEWS;
  u8AnnouncementPrio_DAB_Alarm			= TUN_CONFIG_DEFAULT_ANNOUNCEMENT_PRIO_DAB_ALARM;


  
  u8AnnouncementPrio_DAB_RoadTrafficFlash = TUN_CONFIG_DEFAULT_ANNOUNCEMENT_PRIO_DAB_ROADTRAFFICFLASH;
  u8AnnouncementPrio_DAB_TransportFlash	= TUN_CONFIG_DEFAULT_ANNOUNCEMENT_PRIO_DAB_TRANSPORTFLASH;
  u8AnnouncementPrio_DAB_WarningService = TUN_CONFIG_DEFAULT_ANNOUNCEMENT_PRIO_DAB_WARNIINGSERVICE;


  u8AnnouncementPrio_DAB_NewsFlash			= TUN_CONFIG_DEFAULT_ANNOUNCEMENT_PRIO_DAB_NEWSFLASH;
  u8AnnouncementPrio_DAB_AreaWeatherFlash	= TUN_CONFIG_DEFAULT_ANNOUNCEMENT_PRIO_DAB_AREAWEATHERFLASH;
  u8AnnouncementPrio_DAB_EventAnnouncement	= TUN_CONFIG_DEFAULT_ANNOUNCEMENT_PRIO_DAB_EVENTANNO;
  u8AnnouncementPrio_DAB_SpecialEvent		= TUN_CONFIG_DEFAULT_ANNOUNCEMENT_PRIO_DAB_SPECIALEVENT;
  u8AnnouncementPrio_DAB_ProgramInfo		= TUN_CONFIG_DEFAULT_ANNOUNCEMENT_PRIO_DAB_PROGRAMINFO;
  u8AnnouncementPrio_DAB_SportReport		= TUN_CONFIG_DEFAULT_ANNOUNCEMENT_PRIO_DAB_SPORTREPORT;
  u8AnnouncementPrio_DAB_FinancialReport	= TUN_CONFIG_DEFAULT_ANNOUNCEMENT_PRIO_DAB_FINANCIALREPORT;
  u8AnnouncementPrio_DAB_reserved1			= TUN_CONFIG_DEFAULT_ANNOUNCEMENT_PRIO_DAB_RESERVED1;
  u8AnnouncementPrio_DAB_reserved2			= TUN_CONFIG_DEFAULT_ANNOUNCEMENT_PRIO_DAB_RESERVED2;
  u8AnnouncementPrio_DAB_reserved3			= TUN_CONFIG_DEFAULT_ANNOUNCEMENT_PRIO_DAB_RESERVED3;
  u8AnnouncementPrio_DAB_reserved4			= TUN_CONFIG_DEFAULT_ANNOUNCEMENT_PRIO_DAB_RESERVED4;
  u8AnnouncementPrio_DAB_reserved5			= TUN_CONFIG_DEFAULT_ANNOUNCEMENT_PRIO_DAB_RESERVED5;

  //u8ReservedByte3[10];/**----------------------------  83 - 92 byte */

  u8TMCServer1								= DIA_TUNER_FM_TMCSERVER;

  //u8ReservedByte4[40];/**----------------------------  94 - 133 byte */

  u8PresetListType					= TUN_CONFIG_DEFAULT_PRESETLISTTYPE;
  u8NumberOfPresetListFM			= TUN_CONFIG_DEFAULT_NO_OFPRESETLISTFM;
  u8NumberOfBanksHmiFM				= TUN_CONFIG_DEFAULT_NO_FM_PRESET_BANKS;
  u8NumberOfPresetsPerBankFM		= TUN_CONFIG_DEFAULT_NO_OF_FM_PRESET_PER_BANK;
//  u8ReservedByte5;/**-------------------------------  138th byte */
  u8NumberOfPresetListMW			= TUN_CONFIG_DEFAULT_NO_OFPRESETLISTMW;
  u8NumberOfBanksHmiMW				= TUN_CONFIG_DEFAULT_NO_AM_PRESET_BANKS1;
  u8NumberOfPresetsPerBankMW		= TUN_CONFIG_DEFAULT_NO_OF_AM_PRESET_PER_BANK1;
//  u8ReservedByte6;/**-------------------------------  142th byte */
  u8NumberOfPresetListLW			= TUN_CONFIG_DEFAULT_NO_OFPRESETLISTLW;
  u8NumberOfBanksHmiLW				= TUN_CONFIG_DEFAULT_NO_OFBANKSHMILW;
  u8NumberOfPresetsPerBankLW		= TUN_CONFIG_DEFAULT_NO_OFPRESETPERBANKLW;
//  u8ReservedByte7;/**-------------------------------  146th byte */
  u8NumberOfPresetListDAB			= TUN_CONFIG_DEFAULT_NO_OFPRESETLISTDAB; 
  u8NumberOfBanksHmiDAB				= TUN_CONFIG_DEFAULT_NO_OFBANKSHMIDAB;
  u8NumberOfPresetsPerBankDAB		= TUN_CONFIG_DEFAULT_NO_OFPRESETPERBANKDAB;
//  u8ReservedByte8;/**-------------------------------  150th byte */
  u8NumberOfPresetListForMixedAllBands		= TUN_CONFIG_DEFAULT_NO_OFPRESETLISTFOR_MIXEDALLBANDS;
  u8NumberOfBanksHmiForMixedAllBands		= TUN_CONFIG_DEFAULT_NO_OFBANKSHMIFOR_MIXEDALLBANDS;
  u8NumberOfPresetsPerBankForMixedAllBands	= TUN_CONFIG_DEFAULT_NO_OFPRESETPERBANKFOR_MIXEDALLBANDS;

#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
  u8NumberOfPresetListDRM			= TUN_CONFIG_DEFAULT_NO_OFPRESETLISTDRM;
  u8NumberOfBanksHmiDRM				= TUN_CONFIG_DEFAULT_NO_DRM_PRESET_BANKS;
  u8NumberOfPresetsPerBankDRM		= TUN_CONFIG_DEFAULT_NO_OF_DRM_PRESET_PER_BANK;
#endif 
//  u8ReservedByte9;/**-------------------------------  154th byte */
//  u8ReservedByte10[16];/**--------------------------  155 - 170th byte */

  /** need to assign preset list values */
//  u32Preset[61];// = GENERIC_TO_BE_DECIDED;
 /* for(tU8 u8Index = 0; u8Index < TUN_CONFIG_NEW_KDS_GENERIC_NO_OF_PRESETS ; u8Index++)
	{
		u32Preset[u8Index]  = 0x00000000;// GENERIC_TO_BE_DECIDED;
	}
	*/
	au32Preset[0] = TUN_CONFIG_DEFAULT_PRESET0;
	au32Preset[1] = TUN_CONFIG_DEFAULT_PRESET1;
	au32Preset[2] = TUN_CONFIG_DEFAULT_PRESET2;
	au32Preset[3] = TUN_CONFIG_DEFAULT_PRESET3;
	au32Preset[4] = TUN_CONFIG_DEFAULT_PRESET4;
	au32Preset[5] = TUN_CONFIG_DEFAULT_PRESET5;
	au32Preset[6] = TUN_CONFIG_DEFAULT_PRESET6;
	au32Preset[7] = TUN_CONFIG_DEFAULT_PRESET7;
	au32Preset[8] = TUN_CONFIG_DEFAULT_PRESET8;
	au32Preset[9] = TUN_CONFIG_DEFAULT_PRESET9;
	au32Preset[10] = TUN_CONFIG_DEFAULT_PRESET10;
	au32Preset[11] = TUN_CONFIG_DEFAULT_PRESET11;
	au32Preset[12] = TUN_CONFIG_DEFAULT_PRESET12;
	au32Preset[13] = TUN_CONFIG_DEFAULT_PRESET13;
	au32Preset[14] = TUN_CONFIG_DEFAULT_PRESET14;
	au32Preset[15] = TUN_CONFIG_DEFAULT_PRESET15;
	au32Preset[16] = TUN_CONFIG_DEFAULT_PRESET16;
	au32Preset[17] = TUN_CONFIG_DEFAULT_PRESET17;
	au32Preset[18] = TUN_CONFIG_DEFAULT_PRESET18;
	au32Preset[19] = TUN_CONFIG_DEFAULT_PRESET19;
	au32Preset[20] = TUN_CONFIG_DEFAULT_PRESET20;
	au32Preset[21] = TUN_CONFIG_DEFAULT_PRESET21;
	au32Preset[22] = TUN_CONFIG_DEFAULT_PRESET22;
	au32Preset[23] = TUN_CONFIG_DEFAULT_PRESET23;
	au32Preset[24] = TUN_CONFIG_DEFAULT_PRESET24;
	au32Preset[25] = TUN_CONFIG_DEFAULT_PRESET25;
	au32Preset[26] = TUN_CONFIG_DEFAULT_PRESET26;
	au32Preset[27] = TUN_CONFIG_DEFAULT_PRESET27;
	au32Preset[28] = TUN_CONFIG_DEFAULT_PRESET28;
	au32Preset[29] = TUN_CONFIG_DEFAULT_PRESET29;
	au32Preset[30] = TUN_CONFIG_DEFAULT_PRESET30;
	au32Preset[31] = TUN_CONFIG_DEFAULT_PRESET31;
	au32Preset[32] = TUN_CONFIG_DEFAULT_PRESET32;
	au32Preset[33] = TUN_CONFIG_DEFAULT_PRESET33;
	au32Preset[34] = TUN_CONFIG_DEFAULT_PRESET34;
	au32Preset[35] = TUN_CONFIG_DEFAULT_PRESET35;
	au32Preset[36] = TUN_CONFIG_DEFAULT_PRESET36;
	au32Preset[37] = TUN_CONFIG_DEFAULT_PRESET37;
	au32Preset[38] = TUN_CONFIG_DEFAULT_PRESET38;
	au32Preset[39] = TUN_CONFIG_DEFAULT_PRESET39;
	au32Preset[40] = TUN_CONFIG_DEFAULT_PRESET40;
	au32Preset[41] = TUN_CONFIG_DEFAULT_PRESET41;
	au32Preset[42] = TUN_CONFIG_DEFAULT_PRESET42;
	au32Preset[43] = TUN_CONFIG_DEFAULT_PRESET43;
	au32Preset[44] = TUN_CONFIG_DEFAULT_PRESET44;
	au32Preset[45] = TUN_CONFIG_DEFAULT_PRESET45;
	au32Preset[46] = TUN_CONFIG_DEFAULT_PRESET46;
	au32Preset[47] = TUN_CONFIG_DEFAULT_PRESET47;
	au32Preset[48] = TUN_CONFIG_DEFAULT_PRESET48;
	au32Preset[49] = TUN_CONFIG_DEFAULT_PRESET49;
	au32Preset[50] = TUN_CONFIG_DEFAULT_PRESET50;
	au32Preset[51] = TUN_CONFIG_DEFAULT_PRESET51;
	au32Preset[52] = TUN_CONFIG_DEFAULT_PRESET52;
	au32Preset[53] = TUN_CONFIG_DEFAULT_PRESET53;
	au32Preset[54] = TUN_CONFIG_DEFAULT_PRESET54;
	au32Preset[55] = TUN_CONFIG_DEFAULT_PRESET55;
	au32Preset[56] = TUN_CONFIG_DEFAULT_PRESET56;
	au32Preset[57] = TUN_CONFIG_DEFAULT_PRESET57;
	au32Preset[58] = TUN_CONFIG_DEFAULT_PRESET58;
	au32Preset[59] = TUN_CONFIG_DEFAULT_PRESET59;
	u16CRC = 0x00;//0 for all projects

}

/*****************************************************************************
 * 
 ****************************************************************************/
tVoid tun_trNewKdsGenericTunerParameter::vParse( tPU8 pu8KdsBuffer, tU32 u32Length)
{


  ETG_TRACE_USR1(( " tun_trNewKdsGenericTunerParameter::vParse() ->\n"
                   " u32Length                   : %d bytes\n"
                   " KDS config                  : %d bytes\n"
                   " expected                    : %d bytes\n"
                   ,u32Length
                   ,417//diagnosis define needed
                   ,TUN_CONFIG_NEW_KDS_GENERIC_TUNER_PARAMETER_LENGTH
                ));

  if( u32Length < TUN_CONFIG_NEW_KDS_GENERIC_TUNER_PARAMETER_LENGTH)
  {
    ETG_TRACE_ERR(( " tun_trNewKdsGenericTunerParameter::vParse() -> Error: wrong length of KDS entry, default values are used now!\n"
                     " u32Length                   : %d bytes\n"
                     " KDS config                  : %d bytes\n"
                     " expected                    : %d bytes\n"
                     ,u32Length
                     ,417//DIA_C_U16_KDS_NISSAN_FMAMTUNER_PARAM_SET_LEN
                     ,TUN_CONFIG_NEW_KDS_GENERIC_TUNER_PARAMETER_LENGTH
                  ));
  }
  else

  {
	  OSAL_pvMemoryCopy( sTag, &pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_Tag], (tU8)enNewKdsGenericTunerParameter_len_Tag);
	  OSAL_pvMemoryCopy( sVersion, &pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_MajorVersion], \
		  (tU8)enNewKdsGenericTunerParameter_len_MajorVersion + (tU8)enNewKdsGenericTunerParameter_len_MinorVersion);

	  u32InitialAutoStoreListId				= TUN_CONFIG_GET_U32( &pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_InitialAutoStoreListId]);
	  u8AutostoreUnusedPresetParameter		= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_AutostoreUnusedPresetParameter];
	  u8InitialAutostore_NoOfStations		= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_InitialAutostore_NoOfStations];
	  u8UserAutostore_NoOfStations			= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_UserAutostore_NoOfStations]; 
	  u8AutostoreSortingCriteriaFM			= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_AutostoreSortingCriteriaFM]; 
	  u8AutostoreSortingCriteriaAM			= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_AutostoreSortingCriteriaAM]; 
	  u8AutostoreSortingCriteriaDAB			= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_AutostoreSortingCriteriaDAB]; 
	  u8AutostoreBehavioralSwitches			= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_AutostoreBehavioralSwitches]; 
	  u8BandToTuneAfterInitialAutostore		= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_BandToTuneAfterInitialAutostore]; 
	  u8BandToTuneAfterUserAutostore		= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_BandToTuneAfterUserAutostore]; 

	  //u8ReservedByte1[16];/**---------------------------  24 - 39 byte */  

	  u8MuteOrAttenuationForAbsStnChngAMFM	= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_MuteOrAttenuationForAbsStnChngAMFM];
	  u8MuteOrAttenuationForSeekAMFM		= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_MuteOrAttenuationForSeekAMFM];
	  u8MuteOrAttenuationForManualTuningAMFM = pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_MuteOrAttenuationForManualTuningAMFM];

	  u8TimetoReleaseAttenuationAMFM		= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_TimetoReleaseAttenuationAMFM];
	  u8TimeToReleaseDAB_FMLink				= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_TimeToReleaseDAB_FMLink];
	  u8TimeToReleaseDABPlus_FMLink			= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_TimeToReleaseDABPlus_FMLink];
	  u8TimeToReleaseDMBA_FMLink			= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_TimeToReleaseDMBA_FMLink];

	  u8LevelCompensationForStationLoad		= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_LevelCompensationForStationLoad];
	  //u8ReservedByte2[6];/**----------------------------  48 - 53 byte */

	  u8MonitorFM_TA						= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_MonitorFM_TA];
	  u8AudioTunerFM_TPSeek					= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_AudioTunerFM_TPSeek];
	  u8AnnouncementRequestHandling			= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_AnnouncementRequestHandling];

	  u8FM_TPseekStrategy					= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_FM_TPseekStrategy];
	  u8TimeDelayFM_TPStnAftVolZero			= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_TimeDelayFM_TPStnAftVolZero];
	  u8TimeDelayNoFM_TPStnFound			= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_TimeDelayNoFM_TPStnFound];
	  u8TimeDelayFM_TPStnSelAftRDSLost		= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_TimeDelayFM_TPStnSelAftRDSLost];
	  u8TimeFM_TAEndAftRDSLost				= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_TimeFM_TAEndAftRDSLost];

	  u8BackgroundFM_TA						= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_BackgroundFM_TA];

	  u8TraficAnnoInOtherSrc				= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_TraficAnnoInOtherSrc];
	  u8AnnouncementPrio_FM_TA				= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_AnnouncementPrio_FM_TA];
	  u8AnnouncementPrio_FM_PTY31			= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_AnnouncementPrio_FM_PTY31];
	  u8AnnouncementPrio_FM_NEWS			= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_AnnouncementPrio_FM_NEWS];
	  u8AnnouncementPrio_DAB_Alarm			= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_AnnouncementPrio_DAB_Alarm];
	  u8AnnouncementPrio_DAB_RoadTrafficFlash = pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_AnnouncementPrio_DAB_RoadTrafficFlash];
	  u8AnnouncementPrio_DAB_TransportFlash = pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_AnnouncementPrio_DAB_TransportFlash];
	  u8AnnouncementPrio_DAB_WarningService = pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_AnnouncementPrio_DAB_WarningService];
	  u8AnnouncementPrio_DAB_NewsFlash		= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_AnnouncementPrio_DAB_NewsFlash];
	  u8AnnouncementPrio_DAB_AreaWeatherFlash = pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_AnnouncementPrio_DAB_AreaWeatherFlash];
	  u8AnnouncementPrio_DAB_EventAnnouncement = pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_AnnouncementPrio_DAB_EventAnnouncement];
	  u8AnnouncementPrio_DAB_SpecialEvent	= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_AnnouncementPrio_DAB_SpecialEvent];
	  u8AnnouncementPrio_DAB_ProgramInfo	= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_AnnouncementPrio_DAB_ProgramInfo];
	  u8AnnouncementPrio_DAB_SportReport	= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_AnnouncementPrio_DAB_SportReport];
	  u8AnnouncementPrio_DAB_FinancialReport = pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_AnnouncementPrio_DAB_FinancialReport];
	  u8AnnouncementPrio_DAB_reserved1		= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_AnnouncementPrio_DAB_reserved1];
	  u8AnnouncementPrio_DAB_reserved2		= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_AnnouncementPrio_DAB_reserved2];
	  u8AnnouncementPrio_DAB_reserved3		= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_AnnouncementPrio_DAB_reserved3];
	  u8AnnouncementPrio_DAB_reserved4		= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_AnnouncementPrio_DAB_reserved4];
	  u8AnnouncementPrio_DAB_reserved5		= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_AnnouncementPrio_DAB_reserved5];

	  //u8ReservedByte3[10];/**----------------------------  83 - 92 byte */

	  u8TMCServer1							= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_TMCServer1];

	  //u8ReservedByte4[40];/**----------------------------  94 - 133 byte */
	  u8PresetListType						= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_PresetListType];
	  u8NumberOfPresetListFM				= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_NumberOfPresetListFM];
	  u8NumberOfBanksHmiFM					= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_NumberOfBanksHmiFM];
	  u8NumberOfPresetsPerBankFM			= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_NumberOfPresetsPerBankFM];
	  //  u8ReservedByte5;/**-------------------------------  138th byte */
	  u8NumberOfPresetListMW				= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_NumberOfPresetListMW];
	  u8NumberOfBanksHmiMW					= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_NumberOfBanksHmiMW];
	  u8NumberOfPresetsPerBankMW			= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_NumberOfPresetsPerBankMW];
	  //  u8ReservedByte6;/**-------------------------------  142th byte */
	  u8NumberOfPresetListLW				= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_NumberOfPresetListLW];
	  u8NumberOfBanksHmiLW					= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_NumberOfBanksHmiLW];
	  u8NumberOfPresetsPerBankLW			= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_NumberOfPresetsPerBankLW];
	  //  u8ReservedByte7;/**-------------------------------  146th byte */
	  u8NumberOfPresetListDAB				= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_NumberOfPresetListDAB];
	  u8NumberOfBanksHmiDAB					= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_NumberOfBanksHmiDAB];
	  u8NumberOfPresetsPerBankDAB			= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_NumberOfPresetsPerBankDAB];
	  //  u8ReservedByte8;/**-------------------------------  150th byte */
	  u8NumberOfPresetListForMixedAllBands	= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_NumberOfPresetListForMixedAllBands];
	  u8NumberOfBanksHmiForMixedAllBands	= pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_NumberOfBanksHmiForMixedAllBands];
	  u8NumberOfPresetsPerBankForMixedAllBands = pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_NumberOfPresetsPerBankForMixedAllBands];
	  //  u8ReservedByte9;/**-------------------------------  154th byte */
	  //  u8ReservedByte10[16];/**--------------------------  155 - 170th byte */
	  for(tU8 u8Index = 0; u8Index < TUN_CONFIG_NEW_KDS_GENERIC_NO_OF_PRESETS ; u8Index++)
	  {
		 au32Preset[u8Index]  = TUN_CONFIG_GET_U32( &pu8KdsBuffer[ (tU8)enNewKdsGenericTunerParameter_idx_PresetStartPoint 
			  +
			  (tU8)enNewKdsGenericTunerParameter_len_Preset * u8Index
		  ]);
	  }

	  u16CRC = pu8KdsBuffer[ enNewKdsGenericTunerParameter_idx_CRC];

	  bKdsDataRead = TRUE;
  }
}

/*****************************************************************************
 * 
 ****************************************************************************/
tVoid tun_trNewKdsGenericTunerParameter::vTrace( tVoid)
{
  if( etg_bIsTraceActiveShort(((0xFFFFu & (tU16)TUN_TRACE_CLASS_KDSCONFIG) << 16) | (tU16)TR_LEVEL_USER_2) == TRUE)
  {
    
    ETG_TRACE_USR2(( " BEGIN ------------------------------------------------------------------------------------"            ));
    ETG_TRACE_USR2(( " bKdsDataRead                : 0x%02X  (0x00: Default values, 0x01: data from KDS read)"  ,bKdsDataRead ));
    ETG_TRACE_USR2(( " sTag                        : %8s"        ,sTag                                 ));
    ETG_TRACE_USR2(( " sVersion                    : %3s"        ,sVersion                             ));

	ETG_TRACE_USR2(( " u32InitialAutoStoreListId                    : 0x%08d  " ,u32InitialAutoStoreListId						));
    ETG_TRACE_USR2(( " u8AutostoreUnusedPresetParameter             : %8d  "    ,u8AutostoreUnusedPresetParameter               ));
    ETG_TRACE_USR2(( " u8InitialAutostore_NoOfStations              : %8d  "    ,u8InitialAutostore_NoOfStations                ));
    ETG_TRACE_USR2(( " u8UserAutostore_NoOfStations					: %8d  "    ,u8UserAutostore_NoOfStations					));
    ETG_TRACE_USR2(( " u8AutostoreSortingCriteriaFM                 : %8d  "    ,u8AutostoreSortingCriteriaFM                   ));
    ETG_TRACE_USR2(( " u8AutostoreSortingCriteriaAM                 : %8d  "    ,u8AutostoreSortingCriteriaAM                   ));
    ETG_TRACE_USR2(( " u8AutostoreSortingCriteriaDAB				: %8d  "    ,u8AutostoreSortingCriteriaDAB                  ));
    ETG_TRACE_USR2(( " u8AutostoreBehavioralSwitches				: %8d  "    ,u8AutostoreBehavioralSwitches                  ));
    ETG_TRACE_USR2(( " u8BandToTuneAfterInitialAutostore            : %8d  "    ,u8BandToTuneAfterInitialAutostore				));
    ETG_TRACE_USR2(( " u8BandToTuneAfterUserAutostore               : %8d  "    ,u8BandToTuneAfterUserAutostore                 ));
    ETG_TRACE_USR2(( " u8MuteOrAttenuationForAbsStnChngAMFM         : %8d  "    ,u8MuteOrAttenuationForAbsStnChngAMFM           ));
    ETG_TRACE_USR2(( " u8MuteOrAttenuationForSeekAMFM				: %8d  "    ,u8MuteOrAttenuationForSeekAMFM                 ));

    ETG_TRACE_USR2(( " u8MuteOrAttenuationForManualTuningAMFM       : %8d  "	,u8MuteOrAttenuationForManualTuningAMFM         ));
    ETG_TRACE_USR2(( " u8TimetoReleaseAttenuationAMFM				: %8d  "	,u8TimetoReleaseAttenuationAMFM                 ));
 
	ETG_TRACE_USR2(( " u8TimeToReleaseDAB_FMLink					: %8d  "    ,u8TimeToReleaseDAB_FMLink                      ));
    ETG_TRACE_USR2(( " u8TimeToReleaseDABPlus_FMLink                : %8d  "    ,u8TimeToReleaseDABPlus_FMLink					));
    ETG_TRACE_USR2(( " u8TimeToReleaseDMBA_FMLink                   : %8d  "    ,u8TimeToReleaseDMBA_FMLink                     ));

    ETG_TRACE_USR2(( " u8LevelCompensationForStationLoad			: %8d  "	,u8LevelCompensationForStationLoad				));
    ETG_TRACE_USR2(( " u8MonitorFM_TA								: %8d  "    ,u8MonitorFM_TA									));
    ETG_TRACE_USR2(( " u8AnnouncementRequestHandling				: %8d  "    ,u8AnnouncementRequestHandling					));
    ETG_TRACE_USR2(( " u8FM_TPseekStrategy							: %8d  "    ,u8FM_TPseekStrategy							));
    ETG_TRACE_USR2(( " u8TimeDelayFM_TPStnAftVolZero				: %8ds "    ,u8TimeDelayFM_TPStnAftVolZero					));
    ETG_TRACE_USR2(( " u8TimeDelayNoFM_TPStnFound					: %8ds "    ,u8TimeDelayNoFM_TPStnFound						));

    ETG_TRACE_USR2(( " u8TimeDelayFM_TPStnSelAftRDSLost				: %8ds "    ,u8TimeDelayFM_TPStnSelAftRDSLost				));
    ETG_TRACE_USR2(( " u8TimeFM_TAEndAftRDSLost						: %8ds "    ,u8TimeFM_TAEndAftRDSLost						));


    ETG_TRACE_USR2(( " u8BackgroundFM_TA							: %8d  "    ,u8BackgroundFM_TA								));
    ETG_TRACE_USR2(( " u8TraficAnnoInOtherSrc						: %8d  "    ,u8TraficAnnoInOtherSrc							));
    ETG_TRACE_USR2(( " u8AnnouncementPrio_FM_TA						: %8d  "    ,u8AnnouncementPrio_FM_TA						));

    ETG_TRACE_USR2(( " u8AnnouncementPrio_FM_PTY31					: %8d  "    ,u8AnnouncementPrio_FM_PTY31					));
    ETG_TRACE_USR2(( " u8AnnouncementPrio_FM_NEWS					: %8d  "    ,u8AnnouncementPrio_FM_NEWS                     ));
    ETG_TRACE_USR2(( " u8AnnouncementPrio_DAB_Alarm					: %8d  "    ,u8AnnouncementPrio_DAB_Alarm					));
    ETG_TRACE_USR2(( " u8AnnouncementPrio_DAB_RoadTrafficFlash		: %8d  "    ,u8AnnouncementPrio_DAB_RoadTrafficFlash		));
    ETG_TRACE_USR2(( " u8AnnouncementPrio_DAB_TransportFlash        : %8d  "    ,u8AnnouncementPrio_DAB_TransportFlash          ));

    ETG_TRACE_USR2(( " u8AnnouncementPrio_DAB_WarningService		: %8d  "    ,u8AnnouncementPrio_DAB_WarningService			));
    ETG_TRACE_USR2(( " u8AnnouncementPrio_DAB_NewsFlash				: %8d  "    ,u8AnnouncementPrio_DAB_NewsFlash				)); 

    ETG_TRACE_USR2(( " u8AnnouncementPrio_DAB_AreaWeatherFlash		: %8d  "	,u8AnnouncementPrio_DAB_AreaWeatherFlash		));
    ETG_TRACE_USR2(( " u8AnnouncementPrio_DAB_EventAnnouncement		: %8d  "	,u8AnnouncementPrio_DAB_EventAnnouncement		));
    ETG_TRACE_USR2(( " u8AnnouncementPrio_DAB_SpecialEvent			: %8d  "    ,u8AnnouncementPrio_DAB_SpecialEvent            ));
    ETG_TRACE_USR2(( " u8AnnouncementPrio_DAB_ProgramInfo			: %8d  "    ,u8AnnouncementPrio_DAB_ProgramInfo				));

    ETG_TRACE_USR2(( " u8AnnouncementPrio_DAB_SportReport		    : %8d  "    ,u8AnnouncementPrio_DAB_SportReport				));
    ETG_TRACE_USR2(( " u8AnnouncementPrio_DAB_FinancialReport		: %8d  "    ,u8AnnouncementPrio_DAB_FinancialReport         ));


    ETG_TRACE_USR2(( " u8AnnouncementPrio_DAB_reserved1				: %8d  "    ,u8AnnouncementPrio_DAB_reserved1				));
    ETG_TRACE_USR2(( " u8AnnouncementPrio_DAB_reserved2				: %8d  "    ,u8AnnouncementPrio_DAB_reserved2				));
    ETG_TRACE_USR2(( " u8AnnouncementPrio_DAB_reserved3				: %8d  "    ,u8AnnouncementPrio_DAB_reserved3				));
    ETG_TRACE_USR2(( " u8AnnouncementPrio_DAB_reserved4				: %8d  "    ,u8AnnouncementPrio_DAB_reserved4				));
    ETG_TRACE_USR2(( " u8AnnouncementPrio_DAB_reserved5				: %8d  "    ,u8AnnouncementPrio_DAB_reserved5				));

    ETG_TRACE_USR2(( " u8TMCServer1									: %8d  "    ,u8TMCServer1									));
    ETG_TRACE_USR2(( " u8PresetListType								: %8d  "    ,u8PresetListType								));

    ETG_TRACE_USR2(( " u8NumberOfPresetListFM						: %8d  "	,u8NumberOfPresetListFM							));
    ETG_TRACE_USR2(( " u8NumberOfBanksHmiFM							: %8d  "	,u8NumberOfBanksHmiFM							));


    ETG_TRACE_USR2(( " u8NumberOfPresetsPerBankFM					: %8d  "    ,u8NumberOfPresetsPerBankFM						));
    ETG_TRACE_USR2(( " u8NumberOfPresetListMW						: %8d  "    ,u8NumberOfPresetListMW							));
    ETG_TRACE_USR2(( " u8NumberOfBanksHmiMW							: %8d  "    ,u8NumberOfBanksHmiMW							));
    ETG_TRACE_USR2(( " u8NumberOfPresetsPerBankMW					: %8d  "    ,u8NumberOfPresetsPerBankMW						));
    ETG_TRACE_USR2(( " u8NumberOfPresetListLW						: %8d  "    ,u8NumberOfPresetListLW							));
    ETG_TRACE_USR2(( " u8NumberOfBanksHmiLW							: %8d  "    ,u8NumberOfBanksHmiLW							));
    ETG_TRACE_USR2(( " u8NumberOfPresetsPerBankLW					: %8d  "    ,u8NumberOfPresetsPerBankLW						));
    ETG_TRACE_USR2(( " u8NumberOfPresetListDAB						: %8d  "    ,u8NumberOfPresetListDAB						));
    ETG_TRACE_USR2(( " u8NumberOfBanksHmiDAB						: %8d  "    ,u8NumberOfBanksHmiDAB							));
    ETG_TRACE_USR2(( " u8NumberOfPresetsPerBankDAB					: %8d  "    ,u8NumberOfPresetsPerBankDAB					));
    ETG_TRACE_USR2(( " u8NumberOfPresetListForMixedAllBands			: %8d  "    ,u8NumberOfPresetListForMixedAllBands			));
    ETG_TRACE_USR2(( " u8NumberOfBanksHmiForMixedAllBands			: %8d  "    ,u8NumberOfBanksHmiForMixedAllBands				));
    ETG_TRACE_USR2(( " u8NumberOfPresetsPerBankForMixedAllBands		: %8d  "    ,u8NumberOfPresetsPerBankForMixedAllBands       ));

  for(tU8 u8Index = 0; u8Index < TUN_CONFIG_NEW_KDS_GENERIC_NO_OF_PRESETS ; u8Index++)
	{
		ETG_TRACE_USR2(( "au32Preset[%d]							: %8d  "    ,u8Index,au32Preset[u8Index]						));
	}



    ETG_TRACE_USR2(( " u16CRC										: 0x%04X"   ,u16CRC											));

    ETG_TRACE_USR2(( " END   ------------------------------------------------------------"            ));
  }
}

/******************* New KDS GENERIC Parameter Set ENDS ***********************/




/*****************************************************************************
* CM VARIANT CODING
*****************************************************************************/

/*****************************************************************************
 * Constructor
 ****************************************************************************/
tun_trKdsCMVariantCoding::tun_trKdsCMVariantCoding( tVoid)
{
  ETG_TRACE_USR1_CLS(( TUN_TRACE_CLASS_TUNAPP, " tun_trKdsCMVariantCoding::tun_trKdsCMVariantCoding() -> Constructor"
                ));

  tun_trKdsCMVariantCoding::vSetToDefaultValues();

}


/*****************************************************************************
 * Set all KDS values to default
 ****************************************************************************/
tVoid tun_trKdsCMVariantCoding::vSetToDefaultValues( tVoid)
{
  ETG_TRACE_USR1_CLS(( TUN_TRACE_CLASS_TUNAPP, " tun_trKdsCMVariantCoding::vSetToDefaultValues() -> "
                ));

  bKdsDataRead                    = FALSE;
  u8HWAssemblyInfo				  = 0;
  u8HWAssemblyInfo3              = 0;
  enFmDabHdTunerMode              = enFmDabHdTunerMode_reserved_0x00;
}


/*****************************************************************************
 * 
 ****************************************************************************/
tVoid tun_trKdsCMVariantCoding::vParse( tPU8 pu8KdsBuffer, tU32 u32Length)
{
	  ETG_TRACE_USR1(( " tun_trKdsCMVariantCoding::vParse() ->u32Length=%u\n"
                   ,u32Length));
  ETG_TRACE_USR1(( " tun_trKdsCMVariantCoding::vParse() ->\n"
                   " u32Length                   : %d bytes\n"
                   " KDS config                  : %d bytes\n"
                   " expected                    : %d bytes\n"
                   ,u32Length
                   ,32
                   ,TUN_CONFIG_KDS_CM_VARIANT_CODING_LENGTH
                ));
  u8HWAssemblyInfo				= pu8KdsBuffer[ enKdsCMVariantCoding_idx_HWAssemblyInfo];
  u8HWAssemblyInfo3            = pu8KdsBuffer[ enKdsCMVariantCoding_idx_HWAssemblyInfo3];
  enFmDabHdTunerMode            = (tenFmDabHdTunerMode)pu8KdsBuffer[ enKdsCMVariantCoding_idx_FmDabHdTunerMode];

  bKdsDataRead = TRUE;
}


/*****************************************************************************
 * 
 ****************************************************************************/
tVoid tun_trKdsCMVariantCoding::vTrace( tVoid)
{
  if( etg_bIsTraceActiveShort(((0xFFFFu & (tU16)TUN_TRACE_CLASS_KDSCONFIG) << 16) | (tU16)TR_LEVEL_USER_2) == TRUE)
  {
    ETG_TRACE_USR2(( " BEGIN ------------------------------------------------------------------------------------"            ));
    ETG_TRACE_USR2(( " bKdsDataRead                : 0x%02X  (0x00: Default values, 0x01: date from KDS read)"  ,bKdsDataRead ));
    ETG_TRACE_USR2(( " u8HWAssemblyInfo            : %02x"         ,u8HWAssemblyInfo  ));
    ETG_TRACE_USR2(( " u8HWAssemblyInfo3            : %02x"         ,u8HWAssemblyInfo3  ));
    ETG_TRACE_USR2(( " enFmDabHdTunerMode          : %02x"         ,ETG_CENUM( tenFmDabHdTunerMode,    enFmDabHdTunerMode)  ));
    ETG_TRACE_USR2(( " END   ------------------------------------------------------------------------------------"            ));
  }
}



/************** CM VARIANT CODING ENDS **************************************/

/************** CM HD RADIO PARAMETERS *************************************/
/*****************************************************************************
 * Constructor HDRadioParameterSet
 ****************************************************************************/
tun_trKdsHDRadioParameterSet::tun_trKdsHDRadioParameterSet( tVoid)
{
  ETG_TRACE_USR1_CLS(( TUN_TRACE_CLASS_TUNAPP, " tun_trKdsHDRadioParameterSet::tun_trKdsHDRadioParameterSet() -> Constructor"
                ));

  tun_trKdsHDRadioParameterSet::vSetToDefaultValues();

}


/*****************************************************************************
 * Set all HDRadioParameterSet KDS values to default 
 ****************************************************************************/
tVoid tun_trKdsHDRadioParameterSet::vSetToDefaultValues( tVoid)
{
  ETG_TRACE_USR1_CLS(( TUN_TRACE_CLASS_TUNAPP, " tun_trKdsHDRadioParameterSet::vSetToDefaultValues() -> "
                ));
  
  bKdsDataRead                    = FALSE;
  
  sTag[ 0]     = 'D';
  sTag[ 1]     = 'e';
  sTag[ 2]     = 'f';
  sTag[ 3]     = 'a';
  sTag[ 4]     = 'u';
  sTag[ 5]     = 'l';
  sTag[ 6]     = 't';
  sTag[ 7]     = '-';
  sVersion[ 0] = '0';
  sVersion[ 1] = '0';
  sVersion[ 2] = '0';
  
  u8BlendThresholdFMHybrid					= TUN_CONFIG_DEFAULT_HD_BLEND_THRESHOLD_FMHYBRID;
  u16DigiAudioScalingFMHybrid				= TUN_CONFIG_DEFAULT_HD_DIGI_AUDIO_SCALING_FMHYBRID;
  u16DigiAudioDelayFMHybridPrimaryBBRate	= TUN_CONFIG_DEFAULT_HD_DIGI_AUDIO_DELAY_FMHYBRID_PRIMARY_BBRATE;
  u8BlendRateFMHybrid						= TUN_CONFIG_DEFAULT_HD_BLEND_RATE_FMHYBRID;			
  u8BlendThresholdAMHybrid					= TUN_CONFIG_DEFAULT_HD_BLEND_THRESHOLD_AMHYBRID;			
  u16AM_MPSAudioScaling						= TUN_CONFIG_DEFAULT_HD_AM_MPS_AUDIO_SCALING;					
  u8AM_MPSBlendRate							= TUN_CONFIG_DEFAULT_HD_AM_MPS_BLEND_RATE;						
  u16DigiAudioDelayAM_MPSPrimaryBBRate		= TUN_CONFIG_DEFAULT_HD_DIGI_AUDIO_DELAY_AM_MPS_PRIMARY_BBRATE;    
  u8DigiAudioPhaseInversion					= TUN_CONFIG_DEFAULT_HD_DIGI_AUDIO_PHASEINVERSION;				
  u8BlendDecision							= TUN_CONFIG_DEFAULT_HD_BLEND_DECISION;						
  u8FM_CDNO_BlendDecision					= TUN_CONFIG_DEFAULT_HD_FM_CDNO_BLEND_DECISION;				
  u8AM_CDNO_BlendDecision					= TUN_CONFIG_DEFAULT_HD_AM_CDNO_BLEND_DECISION;				
  u8FMSPSBlendThreshold						= TUN_CONFIG_DEFAULT_HD_FM_SPS_BLEND_THRESHOLD;
  u16FMSPSAudioScaling						= TUN_CONFIG_DEFAULT_HD_FM_SPS_AUDIO_SCALING;
  u8FMSPSBlendRate							= TUN_CONFIG_DEFAULT_HD_FM_SPS_BLEND_RATE;
  u8BallgameModeAudioOutput					= TUN_CONFIG_DEFAULT_HD_BALLGAME_MODE_AUDIO_OUTPUT;
  u8FilteredDSQMTimeConstant				= TUN_CONFIG_DEFAULT_HD_FILTERED_DSQM_TIME_CONSTANT;
  u8EZBlendEnable							= TUN_CONFIG_DEFAULT_HD_EZ_BLEND_ENABLE;
  u16DSQMSeekThreshold						= TUN_CONFIG_DEFAULT_HD_DSQM_SEEK_THRESHOLD;	
  u8HDImageType								= TUN_CONFIG_DEFAULT_HD_IMAGETYPE;
  u8HDPSDFieldBitMask						= TUN_CONFIG_DEFAULT_HD_PSD_FIELD_BITMASK;
  u8HDPSDCommentsBitMask					= TUN_CONFIG_DEFAULT_HD_PSD_COMMENTS_BITMASK;
  u8HDPSDCommercialBitMask					= TUN_CONFIG_DEFAULT_HD_PSD_COMMERCIAL_BITMASK;
  u8HDPSD_UFIDBitMask						= TUN_CONFIG_DEFAULT_HD_PSD_UFID_BITMASK;
  u8Timer1									= TUN_CONFIG_DEFAULT_HD_TIMER1;								
  u8Timer2									= TUN_CONFIG_DEFAULT_HD_TIMER2;								
  u8Timer3									= TUN_CONFIG_DEFAULT_HD_TIMER3;								
  u8Timer4									= TUN_CONFIG_DEFAULT_HD_TIMER4;								
  u8Timer5									= TUN_CONFIG_DEFAULT_HD_TIMER5;		
}


/*****************************************************************************
 * Parse HDRadioParameterSet KDS array read 
 ****************************************************************************/
tVoid tun_trKdsHDRadioParameterSet::vParse( tPU8 pu8KdsBuffer, tU32 u32Length)
{
  ETG_TRACE_USR1(( " tun_trKdsHDRadioParameterSet::vParse() ->\n"
                   " u32Length                   : %d bytes\n"
                   " KDS config                  : %d bytes\n"
                   " expected                    : %d bytes\n"
                   ,u32Length
                   ,TUN_CONFIG_KDS_HD_RADIO_PARAM_SET_LENGTH//DIA_C_U16_KDS_CM_CONFIG_STRING_LEN
                   ,TUN_CONFIG_KDS_HD_RADIO_PARAM_SET_LENGTH
                ));

  OSAL_pvMemoryCopy( sTag, &pu8KdsBuffer[ enKdsHDRadioParameterSet_idx_tag], enKdsHDRadioParameterSet_len_tag);
  OSAL_pvMemoryCopy( sVersion, &pu8KdsBuffer[ enKdsHDRadioParameterSet_idx_MajorVersion], \
	  enKdsHDRadioParameterSet_len_MajorVersion + enKdsHDRadioParameterSet_len_MinorVersion);
  u8BlendThresholdFMHybrid = pu8KdsBuffer[ enKdsHDRadioParameterSet_idx_BlendThresholdFMHybrid];
  u16DigiAudioScalingFMHybrid = \
	  TUN_CONFIG_GET_U16( &pu8KdsBuffer[ enKdsHDRadioParameterSet_idx_DigiAudioScalingFMHybrid]);
  u16DigiAudioDelayFMHybridPrimaryBBRate = \
	  TUN_CONFIG_GET_U16( &pu8KdsBuffer[ enKdsHDRadioParameterSet_idx_DigiAudioDelayFMHybridPrimaryBBRate]);
  u8BlendRateFMHybrid = pu8KdsBuffer[ enKdsHDRadioParameterSet_idx_BlendRateFMHybrid];			
  u8BlendThresholdAMHybrid = pu8KdsBuffer[ enKdsHDRadioParameterSet_idx_BlendThresholdAMHybrid];			
  u16AM_MPSAudioScaling = TUN_CONFIG_GET_U16( &pu8KdsBuffer[ enKdsHDRadioParameterSet_idx_AM_MPSAudioScaling]);					
  u8AM_MPSBlendRate = pu8KdsBuffer[ enKdsHDRadioParameterSet_idx_AM_MPSBlendRate];						
  u16DigiAudioDelayAM_MPSPrimaryBBRate = \
	  TUN_CONFIG_GET_U16( &pu8KdsBuffer[ enKdsHDRadioParameterSet_idx_DigiAudioDelayAM_MPSPrimaryBBRate]);    
  u8DigiAudioPhaseInversion = pu8KdsBuffer[ enKdsHDRadioParameterSet_idx_DigiAudioPhaseInversion];				
  u8BlendDecision = pu8KdsBuffer[ enKdsHDRadioParameterSet_idx_BlendDecision];						
  u8FM_CDNO_BlendDecision = pu8KdsBuffer[ enKdsHDRadioParameterSet_idx_FM_CDNO_BlendDecision];				
  u8AM_CDNO_BlendDecision = pu8KdsBuffer[ enKdsHDRadioParameterSet_idx_AM_CDNO_BlendDecision];	
  u8FMSPSBlendThreshold	= pu8KdsBuffer[enKdsHDRadioParameterSet_idx_FMSPSBlendThreshold];	
  u16FMSPSAudioScaling	= TUN_CONFIG_GET_U16( &pu8KdsBuffer[ enKdsHDRadioParameterSet_idx_FMSPSAudioScaling]); 
  u8FMSPSBlendRate	= pu8KdsBuffer[enKdsHDRadioParameterSet_idx_FMSPSBlendRate];
  u8BallgameModeAudioOutput	= pu8KdsBuffer[enKdsHDRadioParameterSet_idx_BallgameModeAudioOutput];
  u8FilteredDSQMTimeConstant	= pu8KdsBuffer[enKdsHDRadioParameterSet_idx_FilteredDSQMTimeConstant];
  u8EZBlendEnable	= pu8KdsBuffer[enKdsHDRadioParameterSet_idx_EZBlendEnable];
  u16DSQMSeekThreshold	= TUN_CONFIG_GET_U16( &pu8KdsBuffer[enKdsHDRadioParameterSet_idx_DSQMSeekThreshold]); 
  u8HDImageType		= pu8KdsBuffer[enKdsHDRadioParameterSet_idx_HDImageType];
  u8HDPSDFieldBitMask		= pu8KdsBuffer[enKdsHDRadioParameterSet_idx_PSDFieldBitMask];
  u8HDPSDCommentsBitMask	= pu8KdsBuffer[enKdsHDRadioParameterSet_idx_HDPSDCommentsBitMask];
  u8HDPSDCommercialBitMask	= pu8KdsBuffer[enKdsHDRadioParameterSet_idx_HDPSDCommercialBitMask];
  u8HDPSD_UFIDBitMask	= pu8KdsBuffer[enKdsHDRadioParameterSet_idx_HDPSD_UFIDBitMask];
  u8Timer1 = pu8KdsBuffer[ enKdsHDRadioParameterSet_idx_Timer1];								
  u8Timer2 = pu8KdsBuffer[ enKdsHDRadioParameterSet_idx_Timer2];								
  u8Timer3 = pu8KdsBuffer[ enKdsHDRadioParameterSet_idx_Timer3];								
  u8Timer4 = pu8KdsBuffer[ enKdsHDRadioParameterSet_idx_Timer4];								
  u8Timer5 = pu8KdsBuffer[ enKdsHDRadioParameterSet_idx_Timer5];					
  bKdsDataRead = TRUE;


}


/*****************************************************************************
 * Trace for HDRadioParameterSet
 ****************************************************************************/
tVoid tun_trKdsHDRadioParameterSet::vTrace( tVoid)
{
  if( etg_bIsTraceActiveShort(((0xFFFFu & (tU16)TUN_TRACE_CLASS_KDSCONFIG) << 16) | (tU16)TR_LEVEL_USER_2) == TRUE)
  {
    ETG_TRACE_USR2(( " BEGIN --- HDRadioParameterSet ----------------------------------------------------------"            ));    
	ETG_TRACE_USR2(( " bKdsDataRead                : 0x%02X  (0x00: Default values, 0x01: date from KDS read)"  ,bKdsDataRead ));
    ETG_TRACE_USR2(( " sTag                        : %8s"        ,sTag                                 ));
    ETG_TRACE_USR2(( " sVersion                    : %3s"        ,sVersion                             ));

    ETG_TRACE_USR2(( " u8BlendThresholdFMHybrid					: %02x"         , u8BlendThresholdFMHybrid));
    ETG_TRACE_USR2(( " u16DigiAudioScalingFMHybrid				: %04x"         ,u16DigiAudioScalingFMHybrid));
    ETG_TRACE_USR2(( " u16DigiAudioDelayFMHybridPrimaryBBRate   : %04x"         ,u16DigiAudioDelayFMHybridPrimaryBBRate));
    ETG_TRACE_USR2(( " u8BlendRateFMHybrid						: %02x"         ,u8BlendRateFMHybrid));
    ETG_TRACE_USR2(( " u8BlendThresholdAMHybrid					: %02x"         ,u8BlendThresholdAMHybrid));
    ETG_TRACE_USR2(( " u16AM_MPSAudioScaling					: %04x"         ,u16AM_MPSAudioScaling));
    ETG_TRACE_USR2(( " u8AM_MPSBlendRate						: %02x"         ,u8AM_MPSBlendRate));
    ETG_TRACE_USR2(( " u16DigiAudioDelayAM_MPSPrimaryBBRate     : %04x"         ,u16DigiAudioDelayAM_MPSPrimaryBBRate));
    ETG_TRACE_USR2(( " u8DigiAudioPhaseInversion				: %02x"         ,u8DigiAudioPhaseInversion));
    ETG_TRACE_USR2(( " u8BlendDecision							: %02x"         ,u8BlendDecision));
    ETG_TRACE_USR2(( " u8FM_CDNO_BlendDecision					: %02x"         ,u8FM_CDNO_BlendDecision));
    ETG_TRACE_USR2(( " u8AM_CDNO_BlendDecision					: %02x"         ,u8AM_CDNO_BlendDecision));
    ETG_TRACE_USR2(( " u8FMSPSBlendThreshold					: %02x"         ,u8FMSPSBlendThreshold));
    ETG_TRACE_USR2(( " u16FMSPSAudioScaling						: %04x"         ,u16FMSPSAudioScaling));
    ETG_TRACE_USR2(( " u8FMSPSBlendRate							: %02x"         ,u8FMSPSBlendRate));
    ETG_TRACE_USR2(( " u8BallgameModeAudioOutput				: %02x"         ,u8BallgameModeAudioOutput));
    ETG_TRACE_USR2(( " u8FilteredDSQMTimeConstant				: %02x"         ,u8FilteredDSQMTimeConstant));
    ETG_TRACE_USR2(( " u8EZBlendEnable							: %02x"         ,u8EZBlendEnable));
    ETG_TRACE_USR2(( " u16DSQMSeekThreshold						: %04x"         ,u16DSQMSeekThreshold));
    ETG_TRACE_USR2(( " u8HDImageType							: %02x"         ,u8HDImageType));
    ETG_TRACE_USR2(( " u8HDPSDFieldBitMask						: %02x"         ,u8HDPSDFieldBitMask));
    ETG_TRACE_USR2(( " u8HDPSDCommentsBitMask					: %02x"         ,u8HDPSDCommentsBitMask));
    ETG_TRACE_USR2(( " u8HDPSDCommercialBitMask					: %02x"         ,u8HDPSDCommercialBitMask));
    ETG_TRACE_USR2(( " u8HDPSD_UFIDBitMask						: %02x"         ,u8HDPSD_UFIDBitMask));
    ETG_TRACE_USR2(( " u8Timer1									: %02x"         ,u8Timer1));
    ETG_TRACE_USR2(( " u8Timer2									: %02x"         ,u8Timer2));
    ETG_TRACE_USR2(( " u8Timer3									: %02x"         ,u8Timer3));
    ETG_TRACE_USR2(( " u8Timer4									: %02x"         ,u8Timer4));
    ETG_TRACE_USR2(( " u8Timer5									: %02x"         ,u8Timer5));
    ETG_TRACE_USR2(( " END   ------------------------------------------------------------------------------------"));
  }
}

/************** CM HD RADIO PARAMETERS ENDS *********************************/

/************** CM DRM TUNER PARAMETER SET *************************************/
/*****************************************************************************
 * Constructor DRMTunerParameterSet
 ****************************************************************************/
tun_trKdsDRMTunerParameterSet::tun_trKdsDRMTunerParameterSet( tVoid)
{
  ETG_TRACE_USR1_CLS(( TUN_TRACE_CLASS_TUNAPP, " tun_trKdsDRMTunerParameterSet::tun_trKdsDRMTunerParameterSet() -> Constructor"
                ));

  tun_trKdsDRMTunerParameterSet::vSetToDefaultValues();

}


/*****************************************************************************
 * Set all DRMTunerParameterSet KDS values to default 
 ****************************************************************************/
tVoid tun_trKdsDRMTunerParameterSet::vSetToDefaultValues( tVoid)
{
  ETG_TRACE_USR1_CLS(( TUN_TRACE_CLASS_TUNAPP, " tun_trKdsDRMTunerParameterSet::vSetToDefaultValues() -> "
                ));
  
  bKdsDataRead                    = FALSE;

  sTag[ 0]     = 'D';
  sTag[ 1]     = 'e';
  sTag[ 2]     = 'f';
  sTag[ 3]     = 'a';
  sTag[ 4]     = 'u';
  sTag[ 5]     = 'l';
  sTag[ 6]     = 't';
  sTag[ 7]     = '-';
  sVersion[ 0] = '0';
  sVersion[ 1] = '0';
  sVersion[ 2] = '0';

  u8DRMBandEnable								=	TUN_CONFIG_DEFAULT_DRM_BAND_ENABLE;
  u8DRMMWTunningStepSizeForSearch				=	TUN_CONFIG_DEFAULT_DRM_MW_TUNNING_STEP_SIZE_FOR_SEARCH;
  u8DRMMWTunningStepSizeForManualStep			=	TUN_CONFIG_DEFAULT_DRM_MW_TUNNING_STEP_SIZE_FOR_MANUAL_STEP;
  u8DRMServiceFollowing							=	TUN_CONFIG_DEFAULT_DRM_SERVICE_FOLLOWING;			
  u8DRM_DRMAFSearchDelayTime					=	TUN_CONFIG_DEFAULT_DRM_DRM_AF_SEARCH_DELAY_TIME;			
  u8DRMServiceListUnlearnTimeThreshold			=	TUN_CONFIG_DEFAULT_DRM_SERVICE_LIST_UNLEARN_TIME_THRESHOLD; 
  u8DRMServiceListLearnAcceptanceTime			=	TUN_CONFIG_DEFAULT_DRM_SERVICE_LIST_LEARN_ACCEPTANCE_TIME;						
  u8DRMServiceListLearnAcceptanceThreshold		=	TUN_CONFIG_DEFAULT_DRM_SERVICE_LIST_LEARN_ACCEPTANCE_THRESHOLD;				
  u8DRMQualityThresholdForAudioMute				=	TUN_CONFIG_DEFAULT_DRM_QUALITY_THRESHOLD_FOR_AUDIO_MUTE;			
  u8DRMAudioErrorConcealmentLevel				=	TUN_CONFIG_DEFAULT_DRM_AUDIO_ERROR_CONCEALMENT_LEVEL;
  s8DRMMWSeekStopFieldStrengthThreshold         =   TUN_CONFIG_DEFAULT_DRM_SEEK_THRESHOLD;
  u8DRMSwitches                                                = TUN_CONFIG_DEFAULT_DRM_SWITCHES;
}


/*****************************************************************************
 * Parse DRMTunerParameterSet KDS array read 
 ****************************************************************************/
tVoid tun_trKdsDRMTunerParameterSet::vParse( tPU8 pu8KdsBuffer, tU32 u32Length)
{
  ETG_TRACE_USR1(( " tun_trKdsDRMTunerParameterSet::vParse() ->\n"
                   " u32Length                   : %d bytes\n"
                   " KDS config                  : %d bytes\n"
                   " expected                    : %d bytes\n"
                   ,u32Length
                   ,TUN_CONFIG_KDS_DRM_TUNER_PARAM_SET_LENGTH
                   ,TUN_CONFIG_KDS_DRM_TUNER_PARAM_SET_LENGTH
                ));

  OSAL_pvMemoryCopy( sTag, &pu8KdsBuffer[ enKdsDRMTunerParameterSet_idx_tag], enKdsDRMTunerParameterSet_len_tag);
  OSAL_pvMemoryCopy( sVersion, &pu8KdsBuffer[ enKdsDRMTunerParameterSet_idx_MajorVersion], \
	  enKdsDRMTunerParameterSet_len_MajorVersion + enKdsDRMTunerParameterSet_len_MinorVersion);
  u8DRMBandEnable = pu8KdsBuffer[ enKdsDRMTunerParameterSet_idx_DRMBandEnable];
  u8DRMMWTunningStepSizeForSearch = pu8KdsBuffer[ enKdsDRMTunerParameterSet_idx_DRMMWTunningStepSizeForSearch];
  u8DRMMWTunningStepSizeForManualStep = pu8KdsBuffer[ enKdsDRMTunerParameterSet_idx_DRMMWTunningStepSizeForManualStep];
  u8DRMServiceFollowing = pu8KdsBuffer[ enKdsDRMTunerParameterSet_idx_DRMServiceFollowing];			
  u8DRM_DRMAFSearchDelayTime = pu8KdsBuffer[ enKdsDRMTunerParameterSet_idx_DRM_DRMAFSearchDelayTime];			
  u8DRMServiceListUnlearnTimeThreshold = pu8KdsBuffer[ enKdsDRMTunerParameterSet_idx_DRMServiceListUnlearnTimeThreshold];					
  u8DRMServiceListLearnAcceptanceTime = pu8KdsBuffer[ enKdsDRMTunerParameterSet_idx_DRMServiceListLearnAcceptanceTime];						
  u8DRMServiceListLearnAcceptanceThreshold = pu8KdsBuffer[ enKdsDRMTunerParameterSet_idx_DRMServiceListLearnAcceptanceThreshold];    
  u8DRMQualityThresholdForAudioMute = pu8KdsBuffer[ enKdsDRMTunerParameterSet_idx_DRMQualityThresholdForAudioMute];				
  u8DRMAudioErrorConcealmentLevel = pu8KdsBuffer[ enKdsDRMTunerParameterSet_idx_DRMAudioErrorConcealmentLevel];
  s8DRMMWSeekStopFieldStrengthThreshold = pu8KdsBuffer[enKdsDRMTunerParameterSet_idx_DRMMWSeekStopFieldStrengthThreshold];
  //work around till KDS actual value is defined. Has to be removed once KDS value is given in Suzuki
  if(s8DRMMWSeekStopFieldStrengthThreshold == 0)
  {
     ETG_TRACE_USR2(("workaround : s8DRMMWSeekStopFieldStrengthThreshold"
           "is set to default 24 till KDS proper value is given"));
     s8DRMMWSeekStopFieldStrengthThreshold = 0x18; //int 24
  }
  u8DRMSwitches = pu8KdsBuffer[enKdsDRMTunerParameterSet_idx_DRMSwitches];
  bKdsDataRead = TRUE;
}


/*****************************************************************************
 * Trace for DRMTunerParameterSet
 ****************************************************************************/
tVoid tun_trKdsDRMTunerParameterSet::vTrace( tVoid)
{
  if( etg_bIsTraceActiveShort(((0xFFFFu & (tU16)TUN_TRACE_CLASS_KDSCONFIG) << 16) | (tU16)TR_LEVEL_USER_2) == TRUE)
  {
    ETG_TRACE_USR2(( " BEGIN --- DRMTunerParameterSet ----------------------------------------------------------"            ));    
	ETG_TRACE_USR2(( " bKdsDataRead									: 0x%02X  (0x00: Default values, 0x01: date from KDS read)"  ,bKdsDataRead ));
    ETG_TRACE_USR2(( " sTag											: %8s"			,sTag										));
    ETG_TRACE_USR2(( " sVersion										: %3s"			,sVersion									));
    ETG_TRACE_USR2(( " u8DRMBandEnable								: %02x"         ,u8DRMBandEnable							));
    ETG_TRACE_USR2(( " u8DRMMWTunningStepSizeForSearch				: %02xkHz"      ,u8DRMMWTunningStepSizeForSearch			));
    ETG_TRACE_USR2(( " u8DRMMWTunningStepSizeForManualStep			: %02xkHz"      ,u8DRMMWTunningStepSizeForManualStep		));
    ETG_TRACE_USR2(( " u8DRMServiceFollowing						: %02x"         ,u8DRMServiceFollowing						));
    ETG_TRACE_USR2(( " u8DRM_DRMAFSearchDelayTime					: %02xms"       ,u8DRM_DRMAFSearchDelayTime					));
    ETG_TRACE_USR2(( " u8DRMServiceListUnlearnTimeThreshold			: %02xs"        ,u8DRMServiceListUnlearnTimeThreshold		));
    ETG_TRACE_USR2(( " u8DRMServiceListLearnAcceptanceTime			: %02xms"       ,u8DRMServiceListLearnAcceptanceTime		));
    ETG_TRACE_USR2(( " u8DRMServiceListLearnAcceptanceThreshold     : %02x"         ,u8DRMServiceListLearnAcceptanceThreshold	));
    ETG_TRACE_USR2(( " u8DRMQualityThresholdForAudioMute			: %02x"         ,u8DRMQualityThresholdForAudioMute			));
    ETG_TRACE_USR2(( " u8DRMAudioErrorConcealmentLevel				: %02x"         ,u8DRMAudioErrorConcealmentLevel			));
    ETG_TRACE_USR2(( " s8DRMMWSeekStopFieldStrengthThreshold        : %02xdBuV"     ,s8DRMMWSeekStopFieldStrengthThreshold      ));
    ETG_TRACE_USR2(( " u8DRMSwitches                                : %02x"         ,u8DRMSwitches                              ));
    ETG_TRACE_USR2(( " END   -----------------------------------------------------------------------------------"				));
  }
}

/************** CM DRM TUNER PARAMETERS ENDS *********************************/

/*****************************************************************************
 * Incomming message from TTFis
 ****************************************************************************/
tVoid tun_Config::vTraceCommand( tPU8 pu8Msg, tU32 u32Len )
{

	if(m_poclConfigInterface  == NULL)
	{
	NORMAL_M_ASSERT (m_poclConfigInterface != NULL);
	return;
	}

  if( u32Len < 2)
  {
    ETG_TRACE_FATAL(( " tun_Config::vTraceCommand( -> Error: Message is to short Len=%d bytes."
                   ,u32Len
                   ));
    return;
  }


  tU8 u8TunParameterSet;
  tU8 u8CommandType;

  switch( pu8Msg[ 1])
  {
    /*----------------------------------------------------------------------*/


    case TUN_NEW_KDS_FM_AM_Tuner_Parameter_Set:
      {
        ETG_TRACE_FATAL(( " tun_Config::vTraceCommand() -> TTFis command: TUN_NEW_KDS_FM_AM_Tuner_Parameter_Set."
                       ));
        u8TunParameterSet = TUN_KDS_FM_AM_Tuner_Parameter_Set;
      }
      break;
    case TUN_NEW_KDS_GENERIC_Tuner_Parameter_Set:
      {
        ETG_TRACE_FATAL(( " tun_Config::vTraceCommand() -> TTFis command: TUN_NEW_KDS_GENERIC_Tuner_Parameter_Set."
                       ));
        u8TunParameterSet = TUN_KDS_GENERIC_Tuner_Parameter_Set;
      }
      break;

    /*----------------------------------------------------------------------*/

    case TUN_KDS_CM_VARIANT_CODING:
      {
        ETG_TRACE_FATAL(( " tun_Config::vTraceCommand() -> TTFis command: TUN_KDS_CM_VARIANT_CODING."
                       ));
        u8TunParameterSet = TUN_KDS_VARIANT_CODING;
      }
      break;    
	  
	 /*----------------------------------------------------------------------*/

    case TUN_KDS_CM_HD_RADIO_PARAM_SET:
      {
        ETG_TRACE_FATAL(( " tun_Config::vTraceCommand() -> TTFis command: TUN_KDS_CM_HD_RADIO_PARAM_SET."
                       ));
        u8TunParameterSet = TUN_KDS_HD_RADIO_PARAM_SET;
      }
      break;	

	 /*----------------------------------------------------------------------*/

    case TUN_KDS_CM_DRM_TUNER_PARAM_SET:
      {
        ETG_TRACE_FATAL(( " tun_Config::vTraceCommand() -> TTFis command: TUN_KDS_CM_DRM_TUNER_PARAM_SET."
                       ));
        u8TunParameterSet = TUN_KDS_DRM_TUNER_PARAM_SET;
      }
      break;

#if FEATURE_VW_MIBE
    /*----------------------------------------------------------------------*/
	case TUN_KDS_CM_EOL_PARAMETER:
		{
			ETG_TRACE_FATAL(( " tun_Config::vTraceCommand() -> TTFis command: TUN_KDS_CM_EOL_PARAMETER."
                       ));
        prKds = &m_rKdsEOLParameter;
		}
		break;
	case TUN_KDS_CM_PROD_MODE_SETTINGS:
		{
			ETG_TRACE_FATAL(( " tun_Config::vTraceCommand() -> TTFis command: TUN_KDS_CM_PROD_MODE_SETTINGS."));
        prKds = &m_rKdsEOLParameter;
		}
		break;
	case TUN_KDS_CM_QUALITY_LEVEL_OFFSET:
		{
			ETG_TRACE_FATAL(( " tun_Config::vTraceCommand() -> TTFis command: TUN_KDS_CM_QUALITY_LEVEL_OFFSET."));
        prKds = &m_rKdsQualityLevelOffset;
		}
		break;
#endif
    default:
      {
        ETG_TRACE_FATAL(( " tun_Config::vTraceCommand() -> Error: unknown command."
                       ));
      }
      return;

    /*----------------------------------------------------------------------*/
  }


  switch( pu8Msg[ 0])
  {
    /*----------------------------------------------------------------------*/

    case TUN_TRACE_TUN_CONFIG_PRINT:
      {
        ETG_TRACE_FATAL(( " tun_Config::vTraceCommand() -> TTFis command: TUN_TRACE_TUN_CONFIG_PRINT. Please enable trace class  TUN_TRACE_CLASS_KDSCONFIG!"
                       ));
        u8CommandType = TUN_TRACE_TUN_CONFIG_PRINT;
        m_poclConfigInterface->vHandleTTFisCommands(u8CommandType,u8TunParameterSet);
      }
      break;

    /*----------------------------------------------------------------------*/

    case TUN_TRACE_TUN_CONFIG_SET_TO_DEFAULT:
      {
        ETG_TRACE_FATAL(( " tun_Config::vTraceCommand() -> TTFis command: TUN_TRACE_TUN_CONFIG_SET_TO_DEFAULT."
                       ));

        u8CommandType = TUN_TRACE_TUN_CONFIG_SET_TO_DEFAULT;
        m_poclConfigInterface->vHandleTTFisCommands(u8CommandType,u8TunParameterSet);
      }
      break;

    /*----------------------------------------------------------------------*/

    case TUN_TRACE_TUN_CONFIG_READ_KDS:
      {
        ETG_TRACE_FATAL(( " tun_Config::vTraceCommand() -> TTFis command: TUN_TRACE_TUN_CONFIG_READ_KDS."
                       ));
        u8CommandType = TUN_TRACE_TUN_CONFIG_READ_KDS;
        m_poclConfigInterface->vHandleTTFisCommands(u8CommandType,u8TunParameterSet);
      }
      break;

    /*----------------------------------------------------------------------*/

    default:
      ETG_TRACE_FATAL(( " tun_Config::vTraceCommand() -> Error: unknown command."
                     ));
      break;

    /*----------------------------------------------------------------------*/
  }
}

/*************************************************************************
*
* FUNCTION:     tun_Config::vSet_tun_ConfigInterface_Ptr( )
*
* DESCRIPTION:   With this config data can be accessed
*
* PARAMETER:    clConfigInterface* poclConfigInterface
*
* RETURNVALUE:  void
*
*************************************************************************/
tVoid tun_Config::vSet_tun_ConfigInterface_Ptr( clConfigInterface* poclConfigInterface )
{
  if( poclConfigInterface != NULL )
  {
	  m_poclConfigInterface = poclConfigInterface;
  }

}

/*************************************************************************
* FUNCTION:    tun_Config::u8GetHMIListIDFromMap
*
* DESCRIPTION: returns HMI List ID from HMI - ADR List ID Mapping
*
* PARAMETER:   u16ADRListID
*
* RETURNVALUE: tU8
*
* History:
* InitialVersion
* 
* Author vnd4kor 03.09.2013
*************************************************************************/

tU8 tun_Config::u8GetHMIListIDFromMap(const tU16 u16ADRListID)
{
	/** Check for map empty */
	if(!m_mapHMIListIDtoADRListID.empty())
	{
		/** Iterate through the map */
		for(mapListIDs::const_iterator pos = m_mapHMIListIDtoADRListID.begin();
			pos != m_mapHMIListIDtoADRListID.end();
			++pos)
		{
			/** Check for presence of ADR List ID */
			if(u16ADRListID == pos->second)
			{
				/** Return Corresponding HMI LIST ID */
				return pos->first;
			}
		}
	}
	/** Match Not Found or Empty MAP : returning Invalid List ID */
	return (tU8)TUN_BANK_NONE;
}

/*************************************************************************
* FUNCTION:    tun_Config::u16GetADRListIDFromMap
*
* DESCRIPTION: returns ADR List ID from HMI - ADR List ID Mapping
*
* PARAMETER:   u16ADRListID
*
* RETURNVALUE: tU16
*
* History:
* InitialVersion
* 
* Author vnd4kor 03.09.2013
*************************************************************************/

tU16 tun_Config::u16GetADRListIDFromMap(const tU8 u8HMIListID)
{	
	/** Check for map empty */
	if(!m_mapHMIListIDtoADRListID.empty())
	{
		/** Find the HMI LIST ID in map */
		mapListIDs::const_iterator pos = m_mapHMIListIDtoADRListID.find(u8HMIListID);

		/** returning Corresponding ADR List ID if found else 
		returning invalid list id */
		return ((m_mapHMIListIDtoADRListID.end() != pos )? pos->second : (tU8)TUN_BANK_NONE);
	}
	return (tU8)TUN_BANK_NONE;
}


/*************************************************************************
* FUNCTION:    tun_Config::u8GetBandForHMIListID
*
* DESCRIPTION: Get Band associated with HMI list Id
*
* PARAMETER:   u8HMIListID
*
* RETURNVALUE: tU8
*
* History:
* InitialVersion
* 
* Author vnd4kor 06.01.2014
*************************************************************************/
tU8 tun_Config::u8GetBandForHMIListID(const tU8 u8HMIListID)
{
	/** Check for map empty */
	if(!m_mapHMIListIDtoBand.empty())
	{
		/** Find the HMI LIST ID in map */
		mapListIDBands::const_iterator pos = m_mapHMIListIDtoBand.find(u8HMIListID);

		/** returning Corresponding ADR List ID if found else 
		returning invalid list id */
		return ((m_mapHMIListIDtoBand.end() != pos )? pos->second : (tU8)enAvailBand_NONE);
	}
	return (tU8)enAvailBand_NONE;

}
tU8 tun_Config::u8GetNoOfPresetsPerBank(tU16 U16PresetListID)
{
	tU8 u8PresetsPerBank =0;

	if(m_poclConfigInterface != NULL)
	{
		switch(U16PresetListID)
		{
			case midw_fi_tcl_e8_Tun_ListID::FI_EN_TUN_BANK_FM1:
			case midw_fi_tcl_e8_Tun_ListID::FI_EN_TUN_BANK_FM2:

				{
					u8PresetsPerBank=(tU8)m_poclConfigInterface->u32GetConfigData("NumberOfPresetsPerBankFM");
					break;
				}
			case midw_fi_tcl_e8_Tun_ListID::FI_EN_TUN_BANK_FM_AST:
			{
				if(m_poclConfigInterface->u32GetConfigData("UserAutostoreNoOfStations"))
					u8PresetsPerBank=(tU8)m_poclConfigInterface->u32GetConfigData("UserAutostoreNoOfStations");
				else
					u8PresetsPerBank=(tU8)m_poclConfigInterface->u32GetConfigData("NumberOfPresetsPerBankFM");

				break;
			}
			case midw_fi_tcl_e8_Tun_ListID::FI_EN_TUN_BANK_AM_MW1:
			case midw_fi_tcl_e8_Tun_ListID::FI_EN_TUN_BANK_AM_AST:
			case midw_fi_tcl_e8_Tun_ListID::FI_EN_TUN_BANK_AM_MWAST:
				{
					u8PresetsPerBank=(tU8)m_poclConfigInterface->u32GetConfigData("NumberOfPresetsPerBankMW");
					break;
				}
		#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
			case midw_fi_tcl_e8_Tun_ListID::FI_EN_TUN_BANK_DRM1:
			case midw_fi_tcl_e8_Tun_ListID::FI_EN_TUN_BANK_DRM_AST:
				{
					u8PresetsPerBank=u8GetNoOfDRMpresetPerBank();
					break;
				}
		#endif
			case midw_fi_tcl_e8_Tun_ListID::FI_EN_TUN_BANK_MIX1:
				{
					u8PresetsPerBank=(tU8)m_poclConfigInterface->u32GetConfigData("NumberOfPresetsPerBankForMixedAllBands");
					break;
				}
			default:
				{
					break;
				}
		}
	}
	return u8PresetsPerBank;
}
/*************************************************************************
* FUNCTION:    tun_Config::u8GetDefaultHMIListIDForBand
*
* DESCRIPTION: Get Default HMIListID for Band 
*
* PARAMETER:   u8Band
*
* RETURNVALUE: tU8
*
* History:
* InitialVersion
* 
* Author vnd4kor 06.01.2014
*************************************************************************/
tU8 tun_Config::u8GetDefaultHMIListIDForBand(const tU8 u8Band)
{
	/** Check for map empty */
	if(!m_mapHMIListIDtoBand.empty())
	{
		/** Iterate through the map */
		for(mapListIDBands::const_iterator pos = m_mapHMIListIDtoBand.begin();
			pos != m_mapHMIListIDtoBand.end();
			++pos)
		{
			/** Check for presence of Band */
			if(u8Band == pos->second)
			{
				/** Return Corresponding HMI LIST ID */
				return pos->first;
			}
		}
	}
	/** Match Not Found or Empty MAP : returning Invalid List ID */
	return (tU8)TUN_BANK_NONE;

}

/*************************************************************************
* FUNCTION:    tun_Config::u8AF_Switch_AudioTuner1
*
* DESCRIPTION: returns AF switch based on EOL or AMFM parameter set
*
* PARAMETER:   tVoid
*
* RETURNVALUE: tU8
*
* History:
* InitialVersion
*************************************************************************/
tU8 tun_Config::u8AF_Switch_AudioTuner1( tVoid) const
{ 
	tU8 u8AFSwitchTuner1;
	u8AFSwitchTuner1 = ( (( m_rNewKdsFmAmTunerParameter.u8ADRSwitches1 & 0x04) != 0) ? 0x01 : 0x00 ); 


#if FEATURE_VW_MIBE
	/**According to the Req: MIB-2_RQ_RAD-TUN_148
			Other Parameter: AF Deactivation  (Default Value: 0=NO alt. 1=YES)
				Influence only given when RDS Functionality is switched ON: 
		NO:   AF function switch is set to ON  (Search AF and  follow station to new frequency)
		YES: AF function switch is set to OFF  (Don't change tuned frequency, no AF jumping)
	*/

	u8AFSwitchTuner1  = ( (m_rKdsEOLParameter.bGetAFDeactivation() == 0x01) ? 0x00 : 0x01 ); 

#endif
	return (u8AFSwitchTuner1);
}

/*************************************************************************
* FUNCTION:    tun_Config::u8RDS_Switch_AudioTuner1
*
* DESCRIPTION: returns RDS switch based on EOL or AMFM parameter set
*
* PARAMETER:   tVoid
*
* RETURNVALUE: tU8
*
* History:
* InitialVersion
*************************************************************************/
tU8 tun_Config::u8RDS_Switch_AudioTuner1( tVoid) const 
{ 
	tU8 u8RDSSwitch;

	u8RDSSwitch = ((( m_rNewKdsFmAmTunerParameter.u8ADRSwitches2 & 0x01) != 0) ? 0x01 : 0x00);



#if FEATURE_VW_MIBE
		/**
		MIB-2_RQ_RAD-TUN_147

		Other Parameter: RDS Deactivation  (Default Value: 0=NO alt. 1=YES)
		Influence: 
		NO:   RDS function switch is set to ON (RDS function ON)
		YES: RDS function switch is set to OFF (RDS function OFF)

		*/
	u8RDSSwitch  = ( (m_rKdsEOLParameter.bGetRDSDeactivation() == 0x01) ? 0x00 : 0x01 ); 

#endif
	return u8RDSSwitch;
}
/*************************************************************************
* FUNCTION:    tun_Config::u8RDS_Switch_AudioTuner1
*
* DESCRIPTION: returns RDS switch based on EOL or AMFM parameter set
*
* PARAMETER:   tVoid
*
* RETURNVALUE: tU8
*
* History:
* InitialVersion
*************************************************************************/
tenDDA_Switch tun_Config::enDDA_Switch_AudioTuner1( tVoid) const

{
	tenDDA_Switch enDDASwitch;

	enDDASwitch = (tenDDA_Switch)(((m_rNewKdsFmAmTunerParameter.u8ADRSwitches1 >> 4) & 0x03) ? 0x01 : 0x00);



#if FEATURE_VW_MIBE
	/**
	MIB-2_RQ_RAD-TUN_140

Other Parameter: Single ANT Mode (Default Value: 0=OFF alt. 1=ON)
Influence: 
OFF: operation on both antennas (AM+FM and FM only antenna).
ON:   operation with AM+FM antenna only, diagnostics  for second antenna ("FM only antenna") and monitoring of its phantom power supply is deactivated
	*/

	enDDASwitch  = ( (m_rKdsEOLParameter.bGetFMAntenna() == 0x01) ? enDDA_Switch_disable : enDDA_Switch_enable ); 

#endif
	return (enDDASwitch); 
}
/*************************************************************************
* FUNCTION:    tun_Config::u8GetSpecialFMStartFrequency
*
* DESCRIPTION: update the Special start frequency.
*
* PARAMETER:   tVoid
*
* RETURNVALUE: tU8
*
* History:
* InitialVersion
*************************************************************************/
tU8  tun_Config::u8GetSpecialFMStartFrequency( tVoid) const   
{ 

	tU8 u8SpecialStartFreq = m_rNewKdsFmAmTunerParameter.u8SpecialFMStartFrequency;

#if FEATURE_VW_MIBE
	if((tU8)FMTunerBand_NAR == m_rKdsEOLParameter.u8GetFMTunerBand())
	{
		u8SpecialStartFreq  = 50; 
	}

#endif

	return (u8SpecialStartFreq);
}
/*************************************************************************
* FUNCTION:    tun_Config::u8GetSpecialFMStartFrequency
*
* DESCRIPTION: update the Special start frequency.
*
* PARAMETER:   tVoid
*
* RETURNVALUE: tU8
*
* History:
* InitialVersion
*************************************************************************/
tU8  tun_Config::u8GetPIIgnore( tVoid) const
{
	tU8 u8PIIgnoreSwitch = m_rNewKdsFmAmTunerParameter.u8PIIgnore; 


#if FEATURE_VW_MIBE
		u8PIIgnoreSwitch  = m_rKdsEOLParameter.bGetPIIgnoreSwitch(); 
#endif
		return u8PIIgnoreSwitch;
}


/*************************************************************************
*
* FUNCTION:   u8GetQualitySeekLevel_FM_RDS()
*
* DESCRIPTION: Read QualitySeekLevel_FM_RDS from KDS  parameters 
*
* PARAMETER:  tVoid       
*
* RETURNVALUE: tU8
*
* vnd4kor 5/2/2014
*************************************************************************/

tU8 tun_Config::u8GetQualitySeekLevel_FM_RDS( tVoid) const          
{ 
	tU8 u8QualitySeekLevel_FM_RDS = m_rNewKdsFmAmTunerParameter.u8QualitySeekLevel_FM_RDS; 

#if FEATURE_VW_MIBE
		u8QualitySeekLevel_FM_RDS  = m_rKdsQualityLevelOffset.u8GetFMQualityLevelOffset(); 
#endif
		return u8QualitySeekLevel_FM_RDS;
}


/*************************************************************************
*
* FUNCTION:   u8GetQualitySeekLevel_FM_NONRDS()
*
* DESCRIPTION: Read QualitySeekLevel_FM_NONRDS from KDS  parameters 
*
* PARAMETER:  tVoid       
*
* RETURNVALUE: tU8
*
* vnd4kor 5/2/2014
*************************************************************************/

tU8 tun_Config::u8GetQualitySeekLevel_FM_NONRDS( tVoid) const          
{ 
	tU8 u8QualitySeekLevel_FM_NONRDS = m_rNewKdsFmAmTunerParameter.u8QualitySeekLevel_FM_NONRDS; 

#if FEATURE_VW_MIBE
		u8QualitySeekLevel_FM_NONRDS  = m_rKdsQualityLevelOffset.u8GetFMQualityLevelOffset(); 
#endif
		return u8QualitySeekLevel_FM_NONRDS;
}

/*************************************************************************
*
* FUNCTION:   u8GetQualitySeekLevel_MW()
*
* DESCRIPTION: Read QualitySeekLevel_MW from KDS  parameters 
*
* PARAMETER:  tVoid       
*
* RETURNVALUE: tU8
*
* vnd4kor 5/2/2014
*************************************************************************/

tU8 tun_Config::u8GetQualitySeekLevel_MW( tVoid) const          
{ 
	tU8 u8QualitySeekLevel_MW = m_rNewKdsFmAmTunerParameter.u8QualitySeekLevel_MW; 

#if FEATURE_VW_MIBE
		u8QualitySeekLevel_MW  = m_rKdsQualityLevelOffset.u8GetAMQualityLevelOffset(); 
#endif
		return u8QualitySeekLevel_MW;
}


/*************************************************************************
*
* FUNCTION:   u8GetQualityListLevel_FM_RDS()
*
* DESCRIPTION: Read QualityListLevel_FM_RDS from KDS  parameters 
*
* PARAMETER:  tVoid       
*
* RETURNVALUE: tU8
*
* vnd4kor 5/2/2014
*************************************************************************/

tU8 tun_Config::u8GetQualityListLevel_FM_RDS( tVoid) const          
{ 
	tU8 u8QualityListLevel_FM_RDS = m_rNewKdsFmAmTunerParameter.u8QualityListLevel_FM_RDS; 

#if FEATURE_VW_MIBE
		u8QualityListLevel_FM_RDS  = m_rKdsQualityLevelOffset.u8GetFMQualityLevelOffset(); 
#endif
		return u8QualityListLevel_FM_RDS;
}

/*************************************************************************
*
* FUNCTION:   u8GetQualityListLevel_FM_NONRDS()
*
* DESCRIPTION: Read QualityListLevel_FM_NONRDS from KDS  parameters 
*
* PARAMETER:  tVoid       
*
* RETURNVALUE: tU8
*
* vnd4kor 5/2/2014
*************************************************************************/

tU8 tun_Config::u8GetQualityListLevel_FM_NONRDS( tVoid) const          
{
	tU8 u8QualityListLevel_FM_NONRDS = m_rNewKdsFmAmTunerParameter.u8QualityListLevel_FM_NONRDS; 

#if FEATURE_VW_MIBE
		u8QualityListLevel_FM_NONRDS  = m_rKdsQualityLevelOffset.u8GetFMQualityLevelOffset(); 
#endif
		return u8QualityListLevel_FM_NONRDS;
}

/*************************************************************************
*
* FUNCTION:   u8GetQualityListLevel_MW()
*
* DESCRIPTION: Read QualityListLevel_MW from KDS  parameters 
*
* PARAMETER:  tVoid       
*
* RETURNVALUE: tU8
*
* vnd4kor 5/2/2014
*************************************************************************/

tU8 tun_Config::u8GetQualityListLevel_MW( tVoid) const          
{ 
	tU8 u8QualityListLevel_MW = m_rNewKdsFmAmTunerParameter.u8QualityListLevel_MW; 

#if FEATURE_VW_MIBE
		u8QualityListLevel_MW  = m_rKdsQualityLevelOffset.u8GetAMQualityLevelOffset(); 
#endif
		return u8QualityListLevel_MW;
}

/*************************************************************************
* FUNCTION:    tun_Config::u8GetATStationInfoSwitches
*
* DESCRIPTION: update the ATstationInfo Bits for configuration..
*
* PARAMETER:   tVoid
*
* RETURNVALUE: tU8
*
* History:
* InitialVersion
BIT0:
	Configure if a fieldstrength change triggers an CM_ATSTationInfo
		0: off
		1: on
Bit 1:
	Background TA
		0: off
		1: on
Bit 2:
	Audio tuner TP seek
		0: off
		1: on
Bit 3:
	Smoth frequencies for multiple manual steps
		0: off
		1: on
BIT 4-7 Unused
*************************************************************************/
tU8 tun_Config::u8GetATStationInfoSwitches( tVoid)
{ 
	tU8 u8ATStationInfoSwitches = 0;
	tU8 u8BehaviourSwitches = 0;
	tU32 u32BackgroundFMTA = 0;

	vGetATStationInfoSwitches(u8ATStationInfoSwitches);
	vGetBehaviourSwitches(u8BehaviourSwitches);
	vGetBackgroundFMTA(u32BackgroundFMTA);

	if(u8BehaviourSwitches)
	{
		u8ATStationInfoSwitches |= u8BehaviourSwitches;
	}
	else
	{
		u8ATStationInfoSwitches = (tU8)(u8ATStationInfoSwitches & (~u8BehaviourSwitches));
	}

	if(u32BackgroundFMTA)
	{
		u8ATStationInfoSwitches |= TUN_BG_TA_ENABLE;
	}
	else
	{
		u8ATStationInfoSwitches &= TUN_BG_TA_DISABLE;
	}
	return u8ATStationInfoSwitches;
}

/*************************************************************************
*
* FUNCTION:   u8PTY31_Switch_AudioTuner1()
*
* DESCRIPTION: Read u8PTY31_Switch_AudioTuner1 from KDS  parameters 
*
* PARAMETER:  tVoid       
*
* RETURNVALUE: tU8
*
* vnd4kor 5/2/2014
*************************************************************************/
tU8  tun_Config::u8PTY31_Switch_AudioTuner1( tVoid) const
{
	tU8 u8PTY31 = (( m_rNewKdsFmAmTunerParameter.u8ADRSwitches4 & 0x01) != 0) ? 0x01 : 0x00;

#if FEATURE_VW_MIBE
	u8PTY31 = m_rKdsEOLParameter.bGetPTY31Switch();
#endif
	return u8PTY31;
}

/*************************************************************************
*
* FUNCTION:   u8GetBehaviouralSwitchForAutostore()
*
* DESCRIPTION: Return Behavioral switch value read from KDS  parameters
*
* PARAMETER:  tVoid
*
* RETURNVALUE: tU8
*
* pmn3kor 26/5/2014
*************************************************************************/
tU8 tun_Config::u8GetBehaviouralSwitchForAutostore(tVoid)
{
    /* contains Project specifice KDS value*/
    return m_rNewKdsGenericTunerParameter.u8AutostoreBehavioralSwitches;
}
/*************************************************************************
*
* FUNCTION:   u8GetUnusedPresetForAutostore()
*
* DESCRIPTION: Return UnusedPreset value read from KDS  parameters
*
* PARAMETER:  tVoid
*
* RETURNVALUE: tU8
*
* pmn3kor 26/5/2014
*************************************************************************/
tU8 tun_Config::u8GetUnusedPresetForAutostore(tVoid)
{
    /* contains Project specifice KDS value*/
    return m_rNewKdsGenericTunerParameter.u8AutostoreUnusedPresetParameter;
}
/*************************************************************************
*
* FUNCTION:   u8GetMaxNoOfStationsForAutostore()
*
* DESCRIPTION: Return MaxNoOfStations read from KDS  parameters
*
* PARAMETER:  tVoid
*
* RETURNVALUE: tU8
*
* pmn3kor 26/5/2014
*************************************************************************/
tU8 tun_Config::u8GetMaxNoOfStationsForAutostore(tVoid)
{
    /* contains Project specifice KDS value*/
    return m_rNewKdsGenericTunerParameter.u8UserAutostore_NoOfStations;
}
/*************************************************************************
*
* FUNCTION:   u8GetMaxNoOfStationsForAutostore()
*
* DESCRIPTION: Return sorting criteria read from KDS  parameters
*
* PARAMETER:  tVoid
*
* RETURNVALUE: tU8
*
* pmn3kor 26/5/2014
*************************************************************************/
tU8 tun_Config::u8GetSortingCriteriaForAutostore(tU16 u16FunctionIdOfList)
{
	tU8 u8AutostoreSortingCriteriaFM = 0;
	tU8 u8AutostoreSortingCriteriaAM = 0;

	vGetAutostoreSortingCriteriaFM(u8AutostoreSortingCriteriaFM);
	vGetAutostoreSortingCriteriaAM(u8AutostoreSortingCriteriaAM);

	if(u16FunctionIdOfList <= (tU8)TUN_BANK_FM_AST)
		/* if active preset list is FM then send FM sorting criteria*/
		return u8AutostoreSortingCriteriaFM;
	else
		/* if active presetlist is AM (MW and LW) send AM sorting criteria*/
		return u8AutostoreSortingCriteriaAM;

}

#if FEATURE_VW_MIBE

/** KDS QUALITY LEVEL OFFSET PARAMETER Function Definitions **/
/*****************************************************************************
 * Constructor
 ****************************************************************************/
tun_trKdsQualityLevelOffset::tun_trKdsQualityLevelOffset(tVoid)
{
  ETG_TRACE_USR1_CLS(( TUN_TRACE_CLASS_TUNAPP, \
	  " tun_trKdsQualityLevelOffset::tun_trKdsQualityLevelOffset() -> Constructor"
                ));

  tun_trKdsQualityLevelOffset::vSetToDefaultValues();

}

/*****************************************************************************
 * Set all KDS values to default
 ****************************************************************************/
tVoid tun_trKdsQualityLevelOffset::vSetToDefaultValues( tVoid)
{
  ETG_TRACE_USR1_CLS(( TUN_TRACE_CLASS_TUNAPP, \
	  " tun_trKdsQualityLevelOffset::vSetToDefaultValues() -> "
                ));
  m_bReadKDSQualityLevelOffset = false;

  m_u8AMQualityLevel = TUN_CONFIG_DEFAULT_Q_SEEKLEVEL_MW;
  m_u8FMQualityLevel = TUN_CONFIG_DEFAULT_Q_SEEKLEVEL_FM_RDS;
}

tVoid tun_trKdsQualityLevelOffset::vParse( tPU8 pu8KdsBuffer, tU32 u32Length)
{
  ETG_TRACE_USR1(( " tun_trKdsQualityLevelOffset::vParse() ->"));

  m_u8AMQualityLevel = pu8KdsBuffer[TUN_KDS_AM_QUALITY_LEVEL_OFFSET];
  m_u8FMQualityLevel = pu8KdsBuffer[TUN_KDS_FM_QUALITY_LEVEL_OFFSET];

  m_bReadKDSQualityLevelOffset = true;
	
 (void)u32Length;
}

/*****************************************************************************
 * 
 ****************************************************************************/
tVoid tun_trKdsQualityLevelOffset::vTrace( tVoid)
{
  if( etg_bIsTraceActiveShort(((0xFFFFu & (tU16)TUN_TRACE_CLASS_KDSCONFIG) << 16) | (tU16)TR_LEVEL_USER_2) == TRUE)
  {
	  	ETG_TRACE_USR2(( " BEGIN ----------------------- KDS QUALITY LEVEL OFFSET -----------------------------------" ));
		ETG_TRACE_USR2(( " m_u8FMQualityLevel    : 0x%02X",m_u8FMQualityLevel));
		ETG_TRACE_USR2(( " m_u8AMQualityLevel    : 0x%02X",m_u8AMQualityLevel));
		ETG_TRACE_USR2(( " ENDS- ----------------------- KDS QUALITY LEVEL OFFSET -----------------------------------" ));
  }
}

/*****************************************************************************/
/**************** KDS QUALITY LEVEL OFFSET DEFINITIONS ENDS*******************/
/*****************************************************************************/

/** KDS PRODUCTION MODE SETTINGS Function Definitions **/
/*****************************************************************************
 * Constructor
 ****************************************************************************/
tun_trKdsProdModeSettings::tun_trKdsProdModeSettings(tVoid)
{
  ETG_TRACE_USR1_CLS(( TUN_TRACE_CLASS_TUNAPP, \
	  " tun_trKdsQualityLevelOffset::tun_trKdsProdModeSettings() -> Constructor"
                ));

  tun_trKdsProdModeSettings::vSetToDefaultValues();
}

/*****************************************************************************
 * Set all KDS values to default
 ****************************************************************************/
tVoid tun_trKdsProdModeSettings::vSetToDefaultValues( tVoid)
{
  ETG_TRACE_USR1_CLS(( TUN_TRACE_CLASS_TUNAPP, \
	  " tun_trKdsProdModeSettings::vSetToDefaultValues() -> "
                ));
  m_bReadKDSProdModeSettings = false;

  m_u8ProdModeByte1 = 0x0;
  m_u8ProdModeByte2 = 0x0;
  m_u8ProdModeByte3 = 0x0;
}

tVoid tun_trKdsProdModeSettings::vParse( tPU8 pu8KdsBuffer, tU32 u32Length)
{
  ETG_TRACE_USR1(( " tun_trKdsProdModeSettings::vParse() ->"));

  m_u8ProdModeByte1 = pu8KdsBuffer[TUN_KDS_PROD_MODE_BYTE_ONE];
  m_u8ProdModeByte2 = pu8KdsBuffer[TUN_KDS_PROD_MODE_BYTE_TWO];
  m_u8ProdModeByte3 = pu8KdsBuffer[TUN_KDS_PROD_MODE_BYTE_THREE];

  m_bReadKDSProdModeSettings = true;
	
 (void)u32Length;
}

/*****************************************************************************
 * 
 ****************************************************************************/
tVoid tun_trKdsProdModeSettings::vTrace( tVoid)
{
  if( etg_bIsTraceActiveShort(((0xFFFFu & (tU16)TUN_TRACE_CLASS_KDSCONFIG) << 16) | (tU16)TR_LEVEL_USER_2) == TRUE)
  {
	  	ETG_TRACE_USR2(( " BEGIN ----------------------- PRODUCTION MODE SETTINGS -----------------------------------" ));
		ETG_TRACE_USR2(( " m_u8ProdModeByte1    : 0x%02X",m_u8ProdModeByte1));
		ETG_TRACE_USR2(( " m_u8ProdModeByte2    : 0x%02X",m_u8ProdModeByte2));
		ETG_TRACE_USR2(( " m_u8ProdModeByte3    : 0x%02X",m_u8ProdModeByte3));
		ETG_TRACE_USR2(( " ENDS- ----------------------- PRODUCTION MODE SETTINGS -----------------------------------" ));
  }
}
/*****************************************************************************/
/************** KDS PRODUCTION MODE SETTINGS DEFINITIONS ENDS*****************/
/*****************************************************************************/


/** EOL PARAMETER Function Definitions **/
/*****************************************************************************
 * Constructor
 ****************************************************************************/
tun_trKdsEOLParameter::tun_trKdsEOLParameter( tVoid)
{
  ETG_TRACE_USR1_CLS(( TUN_TRACE_CLASS_TUNAPP, " tun_trKdsEOLParameter::tun_trKdsEOLParameter() -> Constructor"
                ));

  tun_trKdsEOLParameter::vSetToDefaultValues();

}



/*****************************************************************************
 * Set all KDS values to default
 ****************************************************************************/
tVoid tun_trKdsEOLParameter::vSetToDefaultValues( tVoid)
{
  ETG_TRACE_USR1_CLS(( TUN_TRACE_CLASS_TUNAPP, " tun_trKdsEOLParameter::vSetToDefaultValues() -> "
                ));
  m_u8FMTunerBand = (tU8)FMTunerBand_NOSET;
  m_u8AMTunerBand = (tU8)AMTunerBand_NOSET;
  m_bFMAntenna = FALSE;
  m_bPIIgnoreSwitch = FALSE;
  m_bAFTemporarySwitch = FALSE;
  m_bPTY31AlarmSwitch = FALSE;
  m_bMultipleEntrySwitch = FALSE;
  m_bRDSDeactivation = FALSE;
  m_bAFDeactivation = FALSE;

  m_bReadKDSData = FALSE;

}


/*****************************************************************************
 * 
 ****************************************************************************/
tVoid tun_trKdsEOLParameter::vParse( tPU8 pu8KdsBuffer, tU32 u32Length)
{
  ETG_TRACE_USR1(( " tun_trKdsEOLParameter::vParse() ->"));

  m_bReadKDSData = TRUE;

  /**Get only Lower Nibble*/
  m_u8FMTunerBand = 0x0f & pu8KdsBuffer[ enKdsEOLParameter_idx_FMAMTunerBand];
  /**Get only Higher Nibble*/
  m_u8AMTunerBand = 0x0f & (pu8KdsBuffer[ enKdsEOLParameter_idx_FMAMTunerBand] >> 4);

  	/**Byte 12 and BIT 0*/
  /**Returns true if Second Antenna is not available.*/
  m_bFMAntenna = ( ((pu8KdsBuffer[ enKdsEOLParameter_idx_FMAntenna] & EOL_FMANTENNA) != 0)? TRUE : FALSE);
  /**get BIT 1 from Byte 13 which give AF Temporary details.*/
  m_bAFTemporarySwitch = (((pu8KdsBuffer[ enKdsEOLParameter_idx_RDS_AF_RadioText_PI_DabAlarm	] & EOL_AF_TEMPORARYSWITCH) != 0)? TRUE : FALSE);
  /**get BIT 4 from Byte 13 which give PI Ignore switch details.*/
  m_bPIIgnoreSwitch = (((pu8KdsBuffer[ enKdsEOLParameter_idx_RDS_AF_RadioText_PI_DabAlarm	] & EOL_PI_IGNORESWITCH) != 0)? TRUE : FALSE);
  /**get BIT 0 from Byte 14 which give PTY31 switch details.*/
  m_bPTY31AlarmSwitch = (((pu8KdsBuffer[ enKdsEOLParameter_idx_PTY31_Mulitiple_RDS_AF_Switch] & EOL_PTY31_ALARM_SWITCH)!= 0)? TRUE : FALSE);
  /**get BIT 3 from Byte 14 which give Multiple entry switch details.*/
  m_bMultipleEntrySwitch = (((pu8KdsBuffer[ enKdsEOLParameter_idx_PTY31_Mulitiple_RDS_AF_Switch] & EOL_MULTIPLEENTRY_SWITCH)!= 0)? TRUE : FALSE);
  /**get BIT 4 from Byte 14 which give RDS Deactivation switch details.*/
  m_bRDSDeactivation = (((pu8KdsBuffer[ enKdsEOLParameter_idx_PTY31_Mulitiple_RDS_AF_Switch] & EOL_RDS_DEACTIVATION)!= 0)? TRUE : FALSE);
  /**get BIT 5 from Byte 14 which give AF Deactivation switch details.*/
  m_bAFDeactivation = (((pu8KdsBuffer[ enKdsEOLParameter_idx_PTY31_Mulitiple_RDS_AF_Switch] & EOL_AF_DEACTIVATION)!= 0)? TRUE : FALSE);


  (void)u32Length;

}
/*****************************************************************************
 * 
 ****************************************************************************/
tVoid tun_trKdsEOLParameter::vTrace( tVoid)
{
  if( etg_bIsTraceActiveShort(((0xFFFFu & (tU16)TUN_TRACE_CLASS_KDSCONFIG) << 16) | (tU16)TR_LEVEL_USER_2) == TRUE)
  {
	ETG_TRACE_USR2(( " BEGIN -------------------------------tun_trKdsEOLParameter-----------------------------------------------------"            ));
	ETG_TRACE_USR2(( " m_bReadKDSData                : 0x%02X  (0x00: Default values, 0x01: date from KDS read)"  ,m_bReadKDSData ));
	ETG_TRACE_USR2(( " m_u8FMTunerBand               : %02X  "		,ETG_CENUM( tenFMTunerBand,    m_u8FMTunerBand)  ));
    ETG_TRACE_USR2(( " m_u8AMTunerBand               : %02X "			,ETG_CENUM( tenAMTunerBand,    m_u8AMTunerBand)          ));
	ETG_TRACE_USR2(( " m_bFMAntenna					 : %02x"			,m_bFMAntenna ));
    ETG_TRACE_USR2(( " m_bAFTemporarySwitch		     : %02x"			,ETG_CENUM( tenAFEOLStatus, m_bAFTemporarySwitch) ));
    ETG_TRACE_USR2(( " m_bPIIgnoreSwitch			 : %02x"			,m_bPIIgnoreSwitch ));
    ETG_TRACE_USR2(( " m_bPTY31AlarmSwitch			 : %02x"			,ETG_CENUM( tenPTY31EOLStatus, m_bPTY31AlarmSwitch) ));
	ETG_TRACE_USR2(( " m_bMultipleEntrySwitch        : %02x"			,m_bMultipleEntrySwitch));
	ETG_TRACE_USR2(( " m_bRDSDeactivation            : %02x"			,m_bRDSDeactivation));
	ETG_TRACE_USR2(( " m_bAFDeactivation             : %02x"			,m_bAFDeactivation));


    ETG_TRACE_USR2(( " END   -------------------------------tun_trKdsEOLParameter-----------------------------------------------------"            ));
  }
}

/*************************************************************************
*
* FUNCTION:   vUpdateFreqAccording2EOLFMTunerBand()
*
* DESCRIPTION: update the member variable for FM freq based on the FM Tuner band from EOL
*
* PARAMETER:  tVoid       
*
* RETURNVALUE: tVoid
*
*************************************************************************/
tVoid tun_Config::vUpdateFreqAccording2EOLFMTunerBand()
{
	m_rNewKdsFmAmTunerParameter.u8SpecialFMStartFrequency = 0;

	switch (m_rKdsEOLParameter.u8GetFMTunerBand())
	{
	case FMTunerBand_NOSET:
	case FMTunerBand_EU_RDW:
		{
			m_rNewKdsFmAmTunerParameter.u32LowFM = TUN_CONFIG_EU_RDW_LOW_FM;
			m_rNewKdsFmAmTunerParameter.u32HighFM = TUN_CONFIG_EU_RDW_HIGH_FM;
			m_rNewKdsFmAmTunerParameter.u8StepSizeFM = TUN_CONFIG_EU_RDW_STEP_SIZE_FM;
			m_rNewKdsFmAmTunerParameter.u8StepSizeManualFM = TUN_CONFIG_EU_RDW_STEP_SIZE_FM;
			m_rNewKdsFmAmTunerParameter.u8ADRFmDeemphasis = TUN_CONFIG_EU_RDW_FM_DEEMPHASIS;
		}
		break;
	case FMTunerBand_NAR:
		{
			m_rNewKdsFmAmTunerParameter.u32LowFM = TUN_CONFIG_NAR_LOW_FM;
			m_rNewKdsFmAmTunerParameter.u32HighFM = TUN_CONFIG_NAR_HIGH_FM;
			m_rNewKdsFmAmTunerParameter.u8StepSizeFM = TUN_CONFIG_NAR_STEP_SIZE_FM;
			m_rNewKdsFmAmTunerParameter.u8StepSizeManualFM = TUN_CONFIG_NAR_STEP_SIZE_FM;
			m_rNewKdsFmAmTunerParameter.u8ADRFmDeemphasis = TUN_CONFIG_NAR_FM_DEEMPHASIS;
			m_rNewKdsFmAmTunerParameter.u8SpecialFMStartFrequency = 50;
		}
		break;
	case FMTunerBand_JP:
		{
			m_rNewKdsFmAmTunerParameter.u32LowFM = TUN_CONFIG_JP_LOW_FM;
			m_rNewKdsFmAmTunerParameter.u32HighFM = TUN_CONFIG_JP_HIGH_FM;
			m_rNewKdsFmAmTunerParameter.u8StepSizeFM = TUN_CONFIG_JP_STEP_SIZE_FM;			
			m_rNewKdsFmAmTunerParameter.u8StepSizeManualFM = TUN_CONFIG_JP_STEP_SIZE_FM;
			m_rNewKdsFmAmTunerParameter.u8ADRFmDeemphasis = TUN_CONFIG_JP_FM_DEEMPHASIS;
		}
		break;
	case FMTunerBand_KOR:
		{
			m_rNewKdsFmAmTunerParameter.u32LowFM = TUN_CONFIG_KOR_LOW_FM;
			m_rNewKdsFmAmTunerParameter.u32HighFM = TUN_CONFIG_KOR_HIGH_FM;
			m_rNewKdsFmAmTunerParameter.u8StepSizeFM = TUN_CONFIG_KOR_STEP_SIZE_FM;		
			m_rNewKdsFmAmTunerParameter.u8StepSizeManualFM = TUN_CONFIG_KOR_STEP_SIZE_FM;
			m_rNewKdsFmAmTunerParameter.u8ADRFmDeemphasis = TUN_CONFIG_KOR_FM_DEEMPHASIS;
		}
		break;
	case FMTunerBand_CN:
		{
			m_rNewKdsFmAmTunerParameter.u32LowFM = TUN_CONFIG_CN_LOW_FM;
			m_rNewKdsFmAmTunerParameter.u32HighFM = TUN_CONFIG_CN_HIGH_FM;
			m_rNewKdsFmAmTunerParameter.u8StepSizeFM = TUN_CONFIG_CN_STEP_SIZE_FM;
			m_rNewKdsFmAmTunerParameter.u8StepSizeManualFM = TUN_CONFIG_CN_STEP_SIZE_FM;
			m_rNewKdsFmAmTunerParameter.u8ADRFmDeemphasis = TUN_CONFIG_CN_FM_DEEMPHASIS;
		}
		break;
	default:
		break;
	}
}
/*************************************************************************
*
* FUNCTION:   vUpdateFreqAccording2EOLAMTunerBand()
*
* DESCRIPTION: update the member variable for AM freq based on the AM Tuner band from EOL
*
* PARAMETER:  tVoid       
*
* RETURNVALUE: tVoid
*
*************************************************************************/
tVoid tun_Config::vUpdateFreqAccording2EOLAMTunerBand()
{
	switch (m_rKdsEOLParameter.u8GetAMTunerBand())
	{
	case AMTunerBand_EU_RDW:
		{
			m_rNewKdsFmAmTunerParameter.u32LowMW = TUN_CONFIG_EU_RDW_LOW_MW;
			m_rNewKdsFmAmTunerParameter.u32HighMW = TUN_CONFIG_EU_RDW_HIGH_MW;
			m_rNewKdsFmAmTunerParameter.u8StepSizeMW = TUN_CONFIG_EU_RDW_STEP_SIZE_MW;
			m_rNewKdsFmAmTunerParameter.u8StepSizeManualMW = TUN_CONFIG_EU_RDW_STEP_SIZE_MW;
		}
		break;
	case AMTunerBand_NAR:
		{
			m_rNewKdsFmAmTunerParameter.u32LowMW = TUN_CONFIG_NAR_LOW_MW;
			m_rNewKdsFmAmTunerParameter.u32HighMW = TUN_CONFIG_NAR_HIGH_MW;
			m_rNewKdsFmAmTunerParameter.u8StepSizeMW = TUN_CONFIG_NAR_STEP_SIZE_MW;
			m_rNewKdsFmAmTunerParameter.u8StepSizeManualMW = TUN_CONFIG_NAR_STEP_SIZE_MW;
		}
		break;
	case AMTunerBand_JP:
		{
			m_rNewKdsFmAmTunerParameter.u32LowMW = TUN_CONFIG_JP_LOW_MW;
			m_rNewKdsFmAmTunerParameter.u32HighMW = TUN_CONFIG_JP_HIGH_MW;
			m_rNewKdsFmAmTunerParameter.u8StepSizeMW = TUN_CONFIG_JP_STEP_SIZE_MW;
			m_rNewKdsFmAmTunerParameter.u8StepSizeManualMW = TUN_CONFIG_JP_STEP_SIZE_MW;
		}
		break;
	case AMTunerBand_NOSET:
	case AMTunerBand_EU:
		{
			m_rNewKdsFmAmTunerParameter.u32LowMW = TUN_CONFIG_EU_LOW_MW;
			m_rNewKdsFmAmTunerParameter.u32HighMW = TUN_CONFIG_EU_HIGH_MW;
			m_rNewKdsFmAmTunerParameter.u8StepSizeMW = TUN_CONFIG_EU_STEP_SIZE_MW;
			m_rNewKdsFmAmTunerParameter.u8StepSizeManualMW = TUN_CONFIG_EU_STEP_SIZE_MW;
		}
		break;
	case AMTunerBand_AUS:
		{
			m_rNewKdsFmAmTunerParameter.u32LowMW = TUN_CONFIG_AUS_LOW_MW;
			m_rNewKdsFmAmTunerParameter.u32HighMW = TUN_CONFIG_AUS_HIGH_MW;
			m_rNewKdsFmAmTunerParameter.u8StepSizeMW = TUN_CONFIG_AUS_STEP_SIZE_MW;
			m_rNewKdsFmAmTunerParameter.u8StepSizeManualMW = TUN_CONFIG_AUS_STEP_SIZE_MW;
		}
		break;
	default:
		break;
	}
}


/*************************************************************************
*
* FUNCTION:   bGetFMAntenna()
*
* DESCRIPTION: Returns TRUE if Only one antenna is available else 2 antennas are available
*
* PARAMETER:  tVoid       
*
* RETURNVALUE: tBool
*
*************************************************************************/
tBool tun_trKdsEOLParameter::bGetFMAntenna() const
{
	return m_bFMAntenna;
}
/*************************************************************************
*
* FUNCTION:   u8GetFMTunerBand()
*
* DESCRIPTION: Read FM Tuner Band from KDS EOL parameters 
*
* PARAMETER:  tVoid       
*
* RETURNVALUE: tU8
*
*************************************************************************/
tU8 tun_trKdsEOLParameter::u8GetFMTunerBand() const
{
	return m_u8FMTunerBand;
}
  /*************************************************************************
*
* FUNCTION:   u8GetAMTunerBand()
*
* DESCRIPTION: Read AM Tuner Band from KDS EOL parameters 
*
* PARAMETER:  tVoid       
*
* RETURNVALUE: tU8
*
*************************************************************************/
tU8 tun_trKdsEOLParameter::u8GetAMTunerBand() const
{
	return m_u8AMTunerBand;
}
  /*************************************************************************
*
* FUNCTION:   bGetPIIgnoreSwitch()
*
* DESCRIPTION: Read PI Ignore Switch from KDS EOL parameters 
*
* PARAMETER:  tVoid       
*
* RETURNVALUE: tBool
*
*************************************************************************/
tBool tun_trKdsEOLParameter::bGetPIIgnoreSwitch() const
  {
	return m_bPIIgnoreSwitch;
}
/*************************************************************************
*
* FUNCTION:   bGetAFTemporarySwitch()
*
* DESCRIPTION: Read AF Temporary Switch from KDS EOL parameters 
*
* PARAMETER:  tVoid       
*
* RETURNVALUE: tBool
*
*************************************************************************/
tBool tun_trKdsEOLParameter::bGetAFTemporarySwitch() const
{
	return m_bAFTemporarySwitch;
}
/*************************************************************************
*
* FUNCTION:   bGetPTY31Switch()
*
* DESCRIPTION: Read PTY31 Switch from KDS EOL parameters 
*
* PARAMETER:  tVoid       
*
* RETURNVALUE: tBool
*
*************************************************************************/
tBool tun_trKdsEOLParameter::bGetPTY31Switch() const
{
	return m_bPTY31AlarmSwitch;
}
    /*************************************************************************
*
* FUNCTION:   bGetMultipleEntrySwitch()
*
* DESCRIPTION: Read MultipleEntry Switch from KDS EOL parameters 
*
* PARAMETER:  tVoid       
*
* RETURNVALUE: tBool
*
*************************************************************************/
  tBool tun_trKdsEOLParameter::bGetMultipleEntrySwitch() const
  {
	return m_bMultipleEntrySwitch;
}
  /*************************************************************************
*
* FUNCTION:   bGetRDSDeactivation()
*
* DESCRIPTION: Read RDSDeactivation Switch from KDS EOL parameters 
*
* PARAMETER:  tVoid       
*
* RETURNVALUE: tBool
*
*************************************************************************/
  tBool tun_trKdsEOLParameter::bGetRDSDeactivation() const
  {
	return m_bRDSDeactivation;
}
/*************************************************************************
*
* FUNCTION:   bGetAFDeactivation()
*
* DESCRIPTION: Read AFDeactivation Switch from KDS EOL parameters 
*
* PARAMETER:  tVoid       
*
* RETURNVALUE: tBool
*
*************************************************************************/
  tBool tun_trKdsEOLParameter::bGetAFDeactivation() const
  {
	return m_bAFDeactivation;
}

#endif

#if FEATURE_PSA_RCC

/*****************************************************************************
* Constructor
****************************************************************************/
tun_trKdsEOLConfigRadioParameter::tun_trKdsEOLConfigRadioParameter( tVoid)
{
	ETG_TRACE_USR1_CLS(( TUN_TRACE_CLASS_TUNAPP, " tun_trKdsEOLConfigRadioParameter::tun_trKdsEOLConfigRadioParameter() -> Constructor"
			  ));

	tun_trKdsEOLConfigRadioParameter::vSetToDefaultValues();
}


/*****************************************************************************
* Set all KDS values to default
****************************************************************************/
tVoid tun_trKdsEOLConfigRadioParameter::vSetToDefaultValues( tVoid)
{
	ETG_TRACE_USR1_CLS(( TUN_TRACE_CLASS_TUNAPP, " tun_trKdsEOLConfigRadioParameter::vSetToDefaultValues() -> "
			  ));

	m_u8Country_Code = TUN_CONFIG_DEFAULT_COUNTRY_CODE;
	m_u8AMAvail = TUN_CONFIG_DEFAULT_AM_AVAILABLE;
}


/*****************************************************************************
*
****************************************************************************/
tVoid tun_trKdsEOLConfigRadioParameter::vParse( tVoid )
{
	ETG_TRACE_USR1(( " tunmstr_trKdsEOLParameter::vParse() ->\n"));
#ifdef	DP_U16_KDSADR_TEL_FCT_RADIO
	tU8 u8DpReadStatus = 0xFF;
	dp_tclKdsTEL_Fct_RADIO oDp_TEL_Fct_RADIO;

	u8DpReadStatus = oDp_TEL_Fct_RADIO.u8GetRADIOAMBandFunction(m_u8AMAvail);
	ETG_TRACE_USR2(( " AM avail read status : %d" , u8DpReadStatus));

	u8DpReadStatus = oDp_TEL_Fct_RADIO.u8GetRADIOCountryZone(m_u8Country_Code);
	ETG_TRACE_USR2(( " CountryCode read status : %d" , u8DpReadStatus));

	u8DpReadStatus = oDp_TEL_Fct_RADIO.u8GetRADIORDSFollowUp(m_u8RDS_FollowUP);
	ETG_TRACE_USR2(( " RDSFollowUp read status : %d" , u8DpReadStatus));

#endif
}

/*****************************************************************************
*
****************************************************************************/
tVoid tun_trKdsEOLConfigRadioParameter::vTrace( tVoid)
{
if( etg_bIsTraceActiveShort(((0xFFFFu & (tU16)TUN_TRACE_CLASS_KDSCONFIG) << 16) | (tU16)TR_LEVEL_USER_2) == TRUE)
{
	ETG_TRACE_USR2(( " m_u8AMAvail               : %8d  "        	,m_u8AMAvail));
	ETG_TRACE_USR2(( " m_u8Country_Code          : %8d  "        	,m_u8Country_Code));
	ETG_TRACE_USR2(( " m_u8RDS_FollowUP          : %8d  "        	,m_u8RDS_FollowUP));
}
}


/*****************************************************************************
* Constructor
****************************************************************************/
tun_trKdsEOLConfigAudioParameter::tun_trKdsEOLConfigAudioParameter( tVoid)
{
	ETG_TRACE_USR1_CLS(( TUN_TRACE_CLASS_TUNAPP, " tun_trKdsEOLConfigAudioParameter::tun_trKdsEOLConfigAudioParameter() -> Constructor"
				  ));

	tun_trKdsEOLConfigAudioParameter::vSetToDefaultValues();
}


/*****************************************************************************
* Set all KDS values to default
****************************************************************************/
tVoid tun_trKdsEOLConfigAudioParameter::vSetToDefaultValues( tVoid)
{
	ETG_TRACE_USR1_CLS(( TUN_TRACE_CLASS_TUNAPP, " tun_trKdsEOLConfigAudioParameter::vSetToDefaultValues() -> "
				  ));
	m_u8SeekSensitivity = TUN_CONFIG_DEFAULT_SEEK_SENSITIVITY;

}


/*****************************************************************************
*
****************************************************************************/
tVoid tun_trKdsEOLConfigAudioParameter::vParse( tVoid)
{
	ETG_TRACE_USR1(( " tun_trKdsEOLConfigAudioParameter::vParse() ->"));
#ifdef	DP_U16_KDSADR_TEL_FCT_AUDIO
	tU8 u8DpReadStatus = 0xFF;
	dp_tclKdsTEL_Fct_AUDIO oDp_TEL_Fct_AUDIO;

	u8DpReadStatus = oDp_TEL_Fct_AUDIO.u8GetAUDIORadioReceptionThreshold(m_u8SeekSensitivity);
	ETG_TRACE_USR2(( " Seek Sensitivity read status : %d" , u8DpReadStatus));
#endif

}


/*****************************************************************************
*
****************************************************************************/
tVoid tun_trKdsEOLConfigAudioParameter::vTrace( tVoid)
{
	if( etg_bIsTraceActiveShort(((0xFFFFu & (tU16)TUN_TRACE_CLASS_KDSCONFIG) << 16) | (tU16)TR_LEVEL_USER_2) == TRUE)
	{
		ETG_TRACE_USR2(( " m_u8SeekSensitivity               : %8d  "        	,m_u8SeekSensitivity));
	}
}

/*************************************************************************
*
* FUNCTION:   u8GetCountryCode()
*
* DESCRIPTION: Read Country code from KDS EOL parameters
*
* PARAMETER:  tVoid
*
* RETURNVALUE: tU8
*
*************************************************************************/
tBool tun_trKdsEOLConfigRadioParameter::u8GetCountryCode() const
{
	return m_u8Country_Code;
}

/*************************************************************************
*
* FUNCTION:   u8GetAMAvail()
*
* DESCRIPTION: Read AM available from KDS EOL parameters
*
* PARAMETER:  tVoid
*
* RETURNVALUE: tU8
*
*************************************************************************/
tBool tun_trKdsEOLConfigRadioParameter::u8GetAMAvail() const
{
	return m_u8AMAvail;
}


/*************************************************************************
*
* FUNCTION:   u8GetRDSFollowUP()
*
* DESCRIPTION: Read RDS followup from KDS EOL parameters
*
* PARAMETER:  tVoid
*
* RETURNVALUE: tU8
*
*************************************************************************/
tBool tun_trKdsEOLConfigRadioParameter::u8GetRDSFollowUP() const
{
	return m_u8RDS_FollowUP;
}


/*************************************************************************
*
* FUNCTION:   vUpdateAvailableBandsAccording2EOL()
*
* DESCRIPTION: update the Available bands from EOL
*
* PARAMETER:  tVoid
*
* RETURNVALUE: tVoid
*
*************************************************************************/
tVoid tun_Config::vUpdateAvailableBandsAccording2EOL()
{
	tU32 u32Availablebands = u32GetAvailableBands();

	if(m_rKdsEOLConfigRadioParameter.u8GetAMAvail())
	{
		if(m_rKdsEOLConfigRadioParameter.u8GetCountryCode() == enPSA_TunerArea_Europe)
		{
			u32Availablebands |= TUN_CONFIG_AM_AVAIL_EU; /* Enable both MW and LW */
		}
		else
		{
			/* Enable only MW */
			u32Availablebands &= TUN_CONFIG_LW_NOTAVAIL;
			u32Availablebands |= TUN_CONFIG_AM_AVAIL_RESTAREA;
		}
	}
	else
	{
		u32Availablebands &= TUN_CONFIG_AM_NOTAVAIL; /*Disable MW and LW */
		m_rNewKdsFmAmTunerParameter.u32CombiBands  = 0;
	}

	m_rNewKdsFmAmTunerParameter.u32AvailableBands = u32Availablebands;
}

/*************************************************************************
*
* FUNCTION:   vReadKDSEOLConfigAudio()
*
* DESCRIPTION: Read KDS Config Audio EOL parameter
*
* PARAMETER:
*
* RETURNVALUE: tVoid
*
*************************************************************************/
tVoid tun_Config::vReadKDSEOLConfigAudio()
{
	ETG_TRACE_USR2(( " vReadKDSEOLConfigAudio :->"));


#ifdef	DP_U16_KDSADR_TEL_FCT_AUDIO
		tU8 pu8Data[TUN_CONFIG_KDS_CM_EOL_PARAMETER_LENGTH];
		dp_tclKdsTEL_Fct_AUDIO oDp_TEL_Fct_AUDIO;
		tU8 u8KdsReadReturnValue = oDp_TEL_Fct_AUDIO.u8GetTEL_Fct_AUDIO(pu8Data,DP_U8_KDSLEN_TEL_FCT_AUDIO_COMPLETE);
		ETG_TRACE_USR2(( " vReadKDSEOLConfigAudio :->u8KdsReadReturnValue : %d", u8KdsReadReturnValue));

		if(u8KdsReadReturnValue ==  DP_U8_ELEM_STATUS_VALID)
		{
			m_rKdsEOLConfigAudioParameter.vParse();
			m_rKdsEOLConfigAudioParameter.vTrace();

			vUpdateFieldStrengthSeekLevel();
		}
#endif
}

/*************************************************************************
*
* FUNCTION:   u8GetSeekSensitivity()
*
* DESCRIPTION: return seek sensitivity bits value
*
* PARAMETER:
*
* RETURNVALUE: tU8
*
*************************************************************************/
tU8 tun_trKdsEOLConfigAudioParameter::u8GetSeekSensitivity() const
{
	return m_u8SeekSensitivity;
}
/*************************************************************************
	tVoid tun_Config::vUpdateDeemphasisAccording2Country()
*************************************************************************/
tVoid tun_Config::vUpdateDeemphasisAccording2Country()
{
	tU8 u8CountryCode = m_rKdsEOLConfigRadioParameter.u8GetCountryCode();
	switch((enTunerArea)u8CountryCode)
	{
		case enPSA_TunerArea_Europe:
		{
			m_rNewKdsFmAmTunerParameter.u8ADRFmDeemphasis = TUN_CONFIG_EU_RDW_FM_DEEMPHASIS;
			break;
		}
		case enPSA_TunerArea_Japan:
		{
			m_rNewKdsFmAmTunerParameter.u8ADRFmDeemphasis = TUN_CONFIG_JP_FM_DEEMPHASIS;
			break;
		}
		case enPSA_TunerArea_South_America:
		{
			m_rNewKdsFmAmTunerParameter.u8ADRFmDeemphasis = TUN_CONFIG_SAR_FM_DEEMPHASIS;
			break;
		}
		case enPSA_TunerArea_Asia_China:
		{
			m_rNewKdsFmAmTunerParameter.u8ADRFmDeemphasis = TUN_CONFIG_ASIA_CN_FM_DEEMPHASIS;
			break;
		}
		case enPSA_TunerArea_Arabia:
		{
			m_rNewKdsFmAmTunerParameter.u8ADRFmDeemphasis = TUN_CONFIG_ARABIA_FM_DEEMPHASIS;
			break;
		}
		case enPSA_TunerArea_North_America:
		{
			m_rNewKdsFmAmTunerParameter.u8ADRFmDeemphasis = TUN_CONFIG_NAR_FM_DEEMPHASIS;
			break;
		}
		case enPSA_TunerArea_Korea:
		{
			m_rNewKdsFmAmTunerParameter.u8ADRFmDeemphasis = TUN_CONFIG_KOREA_FM_DEEMPHASIS;
			break;
		}
		case enPSA_TunerArea_Canada:
		{
			break;
		}
		default:
		{
			break;
		}
	}
}

/*************************************************************************
*
* FUNCTION:   vUpdateFMFrequency_StepSizeFromEOL()
*
* DESCRIPTION: update the Available Frequency from EOL
*
* PARAMETER:tU8 u8Band
*
* RETURNVALUE: tVoid
*
*************************************************************************/
tVoid tun_Config::vUpdateFrequency_StepSizeFromEOL()
{
	tU8 u8CountryCode = m_rKdsEOLConfigRadioParameter.u8GetCountryCode();
	ETG_TRACE_USR2(( " vUpdateFrequency_StepSizeFromEOL, country code : %d", u8CountryCode));

	if(m_rKdsEOLConfigRadioParameter.u8GetRDSFollowUP())
	{
		m_rNewKdsFmAmTunerParameter.u8ADRSwitches1 |= TUN_CONFIG_AFSWITCH_ON;
	}
	else
	{
		m_rNewKdsFmAmTunerParameter.u8ADRSwitches1 &= TUN_CONFIG_AFSWITCH_OFF;
	}

	switch((enTunerArea)u8CountryCode)
	{
	case enPSA_TunerArea_Europe:
		{
				m_rNewKdsFmAmTunerParameter.u32LowFM  			= TUN_CONFIG_EU_LOW_FM;
				m_rNewKdsFmAmTunerParameter.u32HighFM 			= TUN_CONFIG_EU_HIGH_FM;
				m_rNewKdsFmAmTunerParameter.u8StepSizeFM 		= TUN_CONFIG_EU_STEP_SIZE_FM;
				m_rNewKdsFmAmTunerParameter.u8StepSizeManualFM	= TUN_CONFIG_EU_MANUAL_STEP_SIZE_FM;

				m_rNewKdsFmAmTunerParameter.u32LowMW  			= TUN_CONFIG_EU_LOW_MW;
				m_rNewKdsFmAmTunerParameter.u32HighMW 			= TUN_CONFIG_EU_HIGH_MW;
				m_rNewKdsFmAmTunerParameter.u8StepSizeMW 		= TUN_CONFIG_EU_STEP_SIZE_MW;
				m_rNewKdsFmAmTunerParameter.u8StepSizeManualMW	= TUN_CONFIG_EU_MANUAL_STEP_SIZE_MW;

				m_rNewKdsFmAmTunerParameter.u32LowLW  			= TUN_CONFIG_EU_LOW_LW;
				m_rNewKdsFmAmTunerParameter.u32HighLW 			= TUN_CONFIG_EU_HIGH_LW;
				m_rNewKdsFmAmTunerParameter.u8StepSizeLW 		= TUN_CONFIG_EU_STEP_SIZE_LW;
				m_rNewKdsFmAmTunerParameter.u8StepSizeManualLW	= TUN_CONFIG_EU_MANUAL_STEP_SIZE_LW;

			break;
		}
	case enPSA_TunerArea_Japan:
		{
				m_rNewKdsFmAmTunerParameter.u32LowFM  			= TUN_CONFIG_JP_LOW_FM;
				m_rNewKdsFmAmTunerParameter.u32HighFM 			= TUN_CONFIG_JP_HIGH_FM;
				m_rNewKdsFmAmTunerParameter.u8StepSizeFM 		= TUN_CONFIG_JP_STEP_SIZE_FM;
				m_rNewKdsFmAmTunerParameter.u8StepSizeManualFM	= TUN_CONFIG_JP_MANUAL_STEP_SIZE_FM;

				m_rNewKdsFmAmTunerParameter.u32LowMW  			= TUN_CONFIG_JP_LOW_MW;
				m_rNewKdsFmAmTunerParameter.u32HighMW 			= TUN_CONFIG_JP_HIGH_MW;
				m_rNewKdsFmAmTunerParameter.u8StepSizeMW 		= TUN_CONFIG_JP_STEP_SIZE_MW;
				m_rNewKdsFmAmTunerParameter.u8StepSizeManualMW	= TUN_CONFIG_JP_MANUAL_STEP_SIZE_MW;

				m_rNewKdsFmAmTunerParameter.u32LowLW  			= TUN_CONFIG_LOW_LW;
				m_rNewKdsFmAmTunerParameter.u32HighLW 			= TUN_CONFIG_HIGH_LW;
				m_rNewKdsFmAmTunerParameter.u8StepSizeLW 		= TUN_CONFIG_STEP_SIZE_LW;
				m_rNewKdsFmAmTunerParameter.u8StepSizeManualLW	= TUN_CONFIG_MANUAL_STEP_SIZE_LW;

				m_rNewKdsFmAmTunerParameter.u32CombiBands = 0;
				m_rNewKdsFmAmTunerParameter.u32LW_LSM = 0;

				m_rNewKdsFmAmTunerParameter.u8ADRSwitches2 = m_rNewKdsFmAmTunerParameter.u8ADRSwitches2 & 0xfc; //Set the Bit0 and Bit1 to zero, to disable the RDS
			break;
		}
	case enPSA_TunerArea_South_America:
		{
				m_rNewKdsFmAmTunerParameter.u32LowFM  			= TUN_CONFIG_SAR_LOW_FM;
				m_rNewKdsFmAmTunerParameter.u32HighFM 			= TUN_CONFIG_SAR_HIGH_FM;
				m_rNewKdsFmAmTunerParameter.u8StepSizeFM 		= TUN_CONFIG_SAR_STEP_SIZE_FM;
				m_rNewKdsFmAmTunerParameter.u8StepSizeManualFM	= TUN_CONFIG_SAR_MANUAL_STEP_SIZE_FM;

				m_rNewKdsFmAmTunerParameter.u32LowMW  			= TUN_CONFIG_SAR_LOW_MW;
				m_rNewKdsFmAmTunerParameter.u32HighMW 			= TUN_CONFIG_SAR_HIGH_MW;
				m_rNewKdsFmAmTunerParameter.u8StepSizeMW 		= TUN_CONFIG_SAR_STEP_SIZE_MW;
				m_rNewKdsFmAmTunerParameter.u8StepSizeManualMW	= TUN_CONFIG_SAR_MANUAL_STEP_SIZE_MW;

				m_rNewKdsFmAmTunerParameter.u32LowLW  			= TUN_CONFIG_LOW_LW;
				m_rNewKdsFmAmTunerParameter.u32HighLW 			= TUN_CONFIG_HIGH_LW;
				m_rNewKdsFmAmTunerParameter.u8StepSizeLW 		= TUN_CONFIG_STEP_SIZE_LW;
				m_rNewKdsFmAmTunerParameter.u8StepSizeManualLW	= TUN_CONFIG_MANUAL_STEP_SIZE_LW;
				//for SouthAmerica, RDS should be On and AF switch should be off
				tU8 u8ADRSwitches = m_rNewKdsFmAmTunerParameter.u8ADRSwitches1;
				u8ADRSwitches &= TUN_CONFIG_AFSWITCH_OFF;
				m_rNewKdsFmAmTunerParameter.u8ADRSwitches1 = u8ADRSwitches;
				m_rNewKdsFmAmTunerParameter.u32CombiBands = 0;
				m_rNewKdsFmAmTunerParameter.u32LW_LSM = 0;

			break;
		}
	case enPSA_TunerArea_Asia_China:
		{
				m_rNewKdsFmAmTunerParameter.u32LowFM  			= TUN_CONFIG_ASIA_CN_LOW_FM;
				m_rNewKdsFmAmTunerParameter.u32HighFM 			= TUN_CONFIG_ASIA_CN_HIGH_FM;
				m_rNewKdsFmAmTunerParameter.u8StepSizeFM 		= TUN_CONFIG_ASIA_CN_STEP_SIZE_FM;
				m_rNewKdsFmAmTunerParameter.u8StepSizeManualFM	= TUN_CONFIG_ASIA_CN_MANUAL_STEP_SIZE_FM;

				m_rNewKdsFmAmTunerParameter.u32LowMW  			= TUN_CONFIG_ASIA_CN_LOW_MW;
				m_rNewKdsFmAmTunerParameter.u32HighMW 			= TUN_CONFIG_ASIA_CN_HIGH_MW;
				m_rNewKdsFmAmTunerParameter.u8StepSizeMW 		= TUN_CONFIG_ASIA_CN_STEP_SIZE_MW;
				m_rNewKdsFmAmTunerParameter.u8StepSizeManualMW	= TUN_CONFIG_ASIA_CN_MANUAL_STEP_SIZE_MW;

				m_rNewKdsFmAmTunerParameter.u32LowLW  			= TUN_CONFIG_LOW_LW;
				m_rNewKdsFmAmTunerParameter.u32HighLW 			= TUN_CONFIG_HIGH_LW;
				m_rNewKdsFmAmTunerParameter.u8StepSizeLW 		= TUN_CONFIG_STEP_SIZE_LW;
				m_rNewKdsFmAmTunerParameter.u8StepSizeManualLW	= TUN_CONFIG_MANUAL_STEP_SIZE_LW;

				m_rNewKdsFmAmTunerParameter.u32CombiBands = 0;
				m_rNewKdsFmAmTunerParameter.u32LW_LSM = 0;

				m_rNewKdsFmAmTunerParameter.u8ADRSwitches2 = m_rNewKdsFmAmTunerParameter.u8ADRSwitches2 & 0xfc; //Set the Bit0 and Bit1 to zero, to disable the RDS
			break;
		}
	case enPSA_TunerArea_Arabia:
		{
				m_rNewKdsFmAmTunerParameter.u32LowFM  			= TUN_CONFIG_ARABIA_LOW_FM;
				m_rNewKdsFmAmTunerParameter.u32HighFM 			= TUN_CONFIG_ARABIA_HIGH_FM;
				m_rNewKdsFmAmTunerParameter.u8StepSizeFM 		= TUN_CONFIG_ARABIA_STEP_SIZE_FM;
				m_rNewKdsFmAmTunerParameter.u8StepSizeManualFM	= TUN_CONFIG_ARABIA_MANUAL_STEP_SIZE_FM;

				m_rNewKdsFmAmTunerParameter.u32LowMW  			= TUN_CONFIG_ARABIA_LOW_MW;
				m_rNewKdsFmAmTunerParameter.u32HighMW 			= TUN_CONFIG_ARABIA_HIGH_MW;
				m_rNewKdsFmAmTunerParameter.u8StepSizeMW 		= TUN_CONFIG_ARABIA_STEP_SIZE_MW;
				m_rNewKdsFmAmTunerParameter.u8StepSizeManualMW	= TUN_CONFIG_ARABIA_MANUAL_STEP_SIZE_MW;

				m_rNewKdsFmAmTunerParameter.u32LowLW  			= TUN_CONFIG_LOW_LW;
				m_rNewKdsFmAmTunerParameter.u32HighLW 			= TUN_CONFIG_HIGH_LW;
				m_rNewKdsFmAmTunerParameter.u8StepSizeLW 		= TUN_CONFIG_STEP_SIZE_LW;
				m_rNewKdsFmAmTunerParameter.u8StepSizeManualLW	= TUN_CONFIG_MANUAL_STEP_SIZE_LW;

				m_rNewKdsFmAmTunerParameter.u32CombiBands = 0;
				m_rNewKdsFmAmTunerParameter.u32LW_LSM = 0;
				m_rNewKdsFmAmTunerParameter.u8ADRSwitches2 = m_rNewKdsFmAmTunerParameter.u8ADRSwitches2 & 0xfc;

			break;
		}
	case enPSA_TunerArea_North_America:
		{
				m_rNewKdsFmAmTunerParameter.u32LowFM  			= TUN_CONFIG_NAR_LOW_FM;
				m_rNewKdsFmAmTunerParameter.u32HighFM 			= TUN_CONFIG_NAR_HIGH_FM;
				m_rNewKdsFmAmTunerParameter.u8StepSizeFM 		= TUN_CONFIG_NAR_STEP_SIZE_FM;
				m_rNewKdsFmAmTunerParameter.u8StepSizeManualFM	= TUN_CONFIG_NAR_MANUAL_STEP_SIZE_FM;

				m_rNewKdsFmAmTunerParameter.u32LowMW  			= TUN_CONFIG_NAR_LOW_MW;
				m_rNewKdsFmAmTunerParameter.u32HighMW 			= TUN_CONFIG_NAR_HIGH_MW;
				m_rNewKdsFmAmTunerParameter.u8StepSizeMW 		= TUN_CONFIG_NAR_STEP_SIZE_MW;
				m_rNewKdsFmAmTunerParameter.u8StepSizeManualMW	= TUN_CONFIG_NAR_MANUAL_STEP_SIZE_MW;

				m_rNewKdsFmAmTunerParameter.u32LowLW  			= TUN_CONFIG_LOW_LW;
				m_rNewKdsFmAmTunerParameter.u32HighLW 			= TUN_CONFIG_HIGH_LW;
				m_rNewKdsFmAmTunerParameter.u8StepSizeLW 		= TUN_CONFIG_STEP_SIZE_LW;
				m_rNewKdsFmAmTunerParameter.u8StepSizeManualLW	= TUN_CONFIG_MANUAL_STEP_SIZE_LW;

				m_rNewKdsFmAmTunerParameter.u32CombiBands = 0;
				m_rNewKdsFmAmTunerParameter.u32LW_LSM = 0;
				m_rNewKdsFmAmTunerParameter.u8ADRSwitches2 = m_rNewKdsFmAmTunerParameter.u8ADRSwitches2 & 0xfc;

			break;
		}
	case enPSA_TunerArea_Korea:
	{
		m_rNewKdsFmAmTunerParameter.u32LowFM  			= TUN_CONFIG_KOREA_LOW_FM;
		m_rNewKdsFmAmTunerParameter.u32HighFM 			= TUN_CONFIG_KOREA_HIGH_FM;
		m_rNewKdsFmAmTunerParameter.u8StepSizeFM 		= TUN_CONFIG_KOREA_STEP_SIZE_FM;
		m_rNewKdsFmAmTunerParameter.u8StepSizeManualFM	= TUN_CONFIG_KOREA_MANUAL_STEP_SIZE_FM;
		m_rNewKdsFmAmTunerParameter.u32LowMW  			= TUN_CONFIG_KOREA_LOW_MW;
		m_rNewKdsFmAmTunerParameter.u32HighMW 			= TUN_CONFIG_KOREA_HIGH_MW;
		m_rNewKdsFmAmTunerParameter.u8StepSizeMW 		= TUN_CONFIG_KOREA_STEP_SIZE_MW;
		m_rNewKdsFmAmTunerParameter.u8StepSizeManualMW	= TUN_CONFIG_KOREA_MANUAL_STEP_SIZE_MW;
		m_rNewKdsFmAmTunerParameter.u32LowLW  			= TUN_CONFIG_LOW_LW;
		m_rNewKdsFmAmTunerParameter.u32HighLW 			= TUN_CONFIG_HIGH_LW;
		m_rNewKdsFmAmTunerParameter.u8StepSizeLW 		= TUN_CONFIG_STEP_SIZE_LW;
		m_rNewKdsFmAmTunerParameter.u8StepSizeManualLW	= TUN_CONFIG_MANUAL_STEP_SIZE_LW;
		m_rNewKdsFmAmTunerParameter.u32CombiBands = 0;
		m_rNewKdsFmAmTunerParameter.u32LW_LSM = 0;
		m_rNewKdsFmAmTunerParameter.u8ADRSwitches2 = m_rNewKdsFmAmTunerParameter.u8ADRSwitches2 & 0xfc; //Set the Bit0 and Bit1 to zero, to disable the RDS
		break;
	}
	case enPSA_TunerArea_Canada:
	{
		break;
	}
	default :
		{
			ETG_TRACE_USR2(( " vUpdateFrequency_StepSizeFromEOL , invalid area "));
			break;
		}
	};
	ETG_TRACE_USR2((" u32FM_LSM                   : %8d kHz u32LowFM=%d", m_rNewKdsFmAmTunerParameter.u32FM_LSM, m_rNewKdsFmAmTunerParameter.u32LowFM));
	if (m_rNewKdsFmAmTunerParameter.u32FM_LSM < m_rNewKdsFmAmTunerParameter.u32LowFM)
	{
		m_rNewKdsFmAmTunerParameter.u32FM_LSM = m_rNewKdsFmAmTunerParameter.u32LowFM;
	}
	if (m_rNewKdsFmAmTunerParameter.u32MW_LSM < m_rNewKdsFmAmTunerParameter.u32LowMW)
	{
		m_rNewKdsFmAmTunerParameter.u32MW_LSM = m_rNewKdsFmAmTunerParameter.u32LowMW;
	}
	if (m_rNewKdsFmAmTunerParameter.u32LW_LSM < m_rNewKdsFmAmTunerParameter.u32LowLW)
	{
		m_rNewKdsFmAmTunerParameter.u32LW_LSM = m_rNewKdsFmAmTunerParameter.u32LowLW;
	}

}

/*************************************************************************
*
* FUNCTION:   vUpdateFieldStrengthSeekLevel()
*
* DESCRIPTION: update the Fieldstrength seek level
*
* PARAMETER:
*
* RETURNVALUE: tVoid
*
*************************************************************************/
tVoid tun_Config::vUpdateFieldStrengthSeekLevel()
{
	tU8 u8SeekSensitivitybit = m_rKdsEOLConfigAudioParameter.u8GetSeekSensitivity();
	tU8 u8CountryCode = m_rKdsEOLConfigRadioParameter.u8GetCountryCode();
	ETG_TRACE_USR2(( " vUpdateFieldStrengthSeekLevel, Seek Sensitivity bit : %d , country code : %d",
					u8SeekSensitivitybit , u8CountryCode));

	if(u8SeekSensitivitybit == TUN_CONFIG_SEEK_SENSITIVITY_BITSVAL1)
	{
		m_rNewKdsFmAmTunerParameter.u8FieldStrengthSeekLevel_FM_RDS = TUN_CONFIG_SEEK_SENSITIVITY_FS1_FM;
		m_rNewKdsFmAmTunerParameter.u8FieldStrengthSeekLevel_FM_NONRDS = TUN_CONFIG_SEEK_SENSITIVITY_FS1_FM;
		m_rNewKdsFmAmTunerParameter.u8FieldStrengthSeekLevel_MW = TUN_CONFIG_SEEK_SENSITIVITY_FS1_AM;

		if(u8CountryCode == (tU8)enPSA_TunerArea_Europe)
		{
			m_rNewKdsFmAmTunerParameter.u8FieldStrengthSeekLevel_LW = TUN_CONFIG_SEEK_SENSITIVITY_FS1_AM;
		}
	}
	else if(u8SeekSensitivitybit == TUN_CONFIG_SEEK_SENSITIVITY_BITSVAL2)
	{
		m_rNewKdsFmAmTunerParameter.u8FieldStrengthSeekLevel_FM_RDS = TUN_CONFIG_SEEK_SENSITIVITY_FS2_FM;
		m_rNewKdsFmAmTunerParameter.u8FieldStrengthSeekLevel_FM_NONRDS = TUN_CONFIG_SEEK_SENSITIVITY_FS2_FM;
		m_rNewKdsFmAmTunerParameter.u8FieldStrengthSeekLevel_MW = TUN_CONFIG_SEEK_SENSITIVITY_FS2_AM;

		if(u8CountryCode == (tU8)enPSA_TunerArea_Europe)
		{
			m_rNewKdsFmAmTunerParameter.u8FieldStrengthSeekLevel_LW = TUN_CONFIG_SEEK_SENSITIVITY_FS2_AM;
		}
	}
	else if(u8SeekSensitivitybit == TUN_CONFIG_SEEK_SENSITIVITY_BITSVAL3)
	{
		m_rNewKdsFmAmTunerParameter.u8FieldStrengthSeekLevel_FM_RDS = TUN_CONFIG_SEEK_SENSITIVITY_FS1_FM;
		m_rNewKdsFmAmTunerParameter.u8FieldStrengthSeekLevel_FM_NONRDS = TUN_CONFIG_SEEK_SENSITIVITY_FS1_FM;
		m_rNewKdsFmAmTunerParameter.u8FieldStrengthSeekLevel_MW = TUN_CONFIG_SEEK_SENSITIVITY_FS2_AM;

		if(u8CountryCode == (tU8)enPSA_TunerArea_Europe)
		{
			m_rNewKdsFmAmTunerParameter.u8FieldStrengthSeekLevel_LW = TUN_CONFIG_SEEK_SENSITIVITY_FS2_AM;
		}
	}
	else
	{
		ETG_TRACE_USR2(( " vUpdateFrequency_StepSizeFromEOL, ignore updation of seek sensitivity value "));
	}
}


#endif


/*************************************************************************
*
* FUNCTION:   u8GetTuningBehaviourSwitches()
*
* DESCRIPTION:
*
* PARAMETER:  tVoid
*
* RETURNVALUE:
*
*************************************************************************/
tU8 tun_Config::u8GetTuningBehaviourSwitches()
{

	tU8 u8TunerSwitches = 0;

	vGetTunerSwitches(u8TunerSwitches);

	if(u8TunerSwitches & TUN_LASTPASSED_FREQ_BIT)
	{
	  u8TunerSwitches &= TUN_LASTPASSED_FREQ_DISABLE;
	}
	else
	{
	  u8TunerSwitches |= TUN_LASTPASSED_FREQ_ENABLE;
	}
	return u8TunerSwitches;
}


/*************************************************************************
*
* FUNCTION:   u8GetTuningBehaviourSwitches()
*
* DESCRIPTION:
*
* PARAMETER:  tVoid
*
* RETURNVALUE:
*
*************************************************************************/
tBool tun_Config::fGetLandscapeSeekNeeded( tVoid) const
{
	if(m_poclConfigInterface  == NULL)
	{
	NORMAL_M_ASSERT (m_poclConfigInterface != NULL);
	return false;
	}
	return (((tU8)m_poclConfigInterface->u32GetConfigData("TunerMaster") & DIA_TUNER_LANDSCAPESEEK_NEEDED) != 0) ? TRUE : FALSE;
}


/*************************************************************************
*
* FUNCTION:   bIsHDSupported()
*
* DESCRIPTION: Says weather HD is supported by the current KDS configuration 
*
* PARAMETER:  tVoid       
*
* RETURNVALUE: tBool : true - HD suppported , false - HD not supported
*
*************************************************************************/
tBool tun_Config::bIsHDSupported() const
{
	tBool bHDAvailable = (m_rKdsCMVariantCoding.u8HWAssemblyInfo & EXTRACT_HD_BIT_HW_CONFIGURATION) == EXTRACT_HD_BIT_HW_CONFIGURATION ? true: false;
	ETG_TRACE_USR2((" tun_Config::bIsHDSupported()-> %d (1 - HD Supported , 0 - HD Not Supported )",
	  bHDAvailable ));

	return bHDAvailable;
}
/*************************************************************************
*
* FUNCTION:   bIsDRMSupported()
*
* DESCRIPTION: Says whether DRM is supported by the current configuration.
* Bit 2 and 3 are used for DRM info in CMVariantCoding 15th byte.
*
* PARAMETER:  tVoid
*
* RETURNVALUE: tU8 : 0-info not available (0x00)
*                                      1- mounted (0x04)
*                                      2-not mounted (0x08)
*
*************************************************************************/

tU8 tun_Config::bIsDRMSupported() const
{
   ETG_TRACE_USR2((" tun_Config::bIsDRMSupported-> 0x%02x (Bit values: 0 - info not available, 1- mounted, 2-not mounted )",
   m_rKdsCMVariantCoding.u8HWAssemblyInfo3 & EXTRACT_DRM_BIT_HW_CONFIGURATION ));

   return (m_rKdsCMVariantCoding.u8HWAssemblyInfo3 & EXTRACT_DRM_BIT_HW_CONFIGURATION) == EXTRACT_DRM_BIT_HW_CONFIGURATION ? true: false;
}


/*************************************************************************
*
* FUNCTION:   vGetAutostoreSortingCriteriaFM()
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE: tVoid
*
*************************************************************************/
tVoid tun_Config::vGetAutostoreSortingCriteriaFM(tU8 &AutostoreSortingCriteriaFM)
{
	TUN_ASSERT_RETURN( m_poclConfigInterface  != NULL);
	AutostoreSortingCriteriaFM = (tU8)m_poclConfigInterface->u32GetConfigData("AutostoreSortingCriteriaFM");
}

/*************************************************************************
*
* FUNCTION:   vGetAutostoreSortingCriteriaAM()
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE: tVoid
*
*************************************************************************/
tVoid tun_Config::vGetAutostoreSortingCriteriaAM(tU8 &AutostoreSortingCriteriaAM)
{
	TUN_ASSERT_RETURN( m_poclConfigInterface  != NULL);
	AutostoreSortingCriteriaAM = (tU8)m_poclConfigInterface->u32GetConfigData("AutostoreSortingCriteriaAM");
}

/*************************************************************************
*
* FUNCTION:   vGetTunerSwitches()
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE: tVoid
*
*************************************************************************/
tVoid tun_Config::vGetTunerSwitches(tU8 &TunerSwitches)
{
	TUN_ASSERT_RETURN( m_poclConfigInterface  != NULL);
	TunerSwitches = (tU8)m_poclConfigInterface->u32GetConfigData("TunerSwitches");
}

/*************************************************************************
*
* FUNCTION:   vGetATStationInfoSwitches()
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE: tVoid
*
*************************************************************************/
tVoid tun_Config::vGetATStationInfoSwitches(tU8 &ATStationInfoSwitches)
{
	TUN_ASSERT_RETURN( m_poclConfigInterface  != NULL);
	ATStationInfoSwitches = (tU8)m_poclConfigInterface->u32GetConfigData("ATStationInfoSwitches");
}

/*************************************************************************
*
* FUNCTION:   vGetBehaviourSwitches()
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE: tVoid
*
*************************************************************************/
tVoid tun_Config::vGetBehaviourSwitches(tU8 &BehaviourSwitches)
{
	TUN_ASSERT_RETURN( m_poclConfigInterface  != NULL);
	BehaviourSwitches = (tU8)m_poclConfigInterface->u32GetConfigData("BehaviourSwitches");
}

/*************************************************************************
*
* FUNCTION:   vGetBackgroundFMTA()
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE: tVoid
*
*************************************************************************/
tVoid tun_Config::vGetBackgroundFMTA(tU32 &BackgroundFMTA)
{
	TUN_ASSERT_RETURN( m_poclConfigInterface  != NULL);
	BackgroundFMTA = m_poclConfigInterface->u32GetConfigData("BackgroundFMTA");
}


/*************************************************************************
*
* FUNCTION:   u8GetTMCServer()
*
* DESCRIPTION:
*
* PARAMETER:
*
* RETURNVALUE: tBool
*
*************************************************************************/
tBool tun_Config::u8GetTMCServer() const
{
	if(m_poclConfigInterface  == NULL)
	{
	NORMAL_M_ASSERT (m_poclConfigInterface != NULL);
	return false;
	}
	return((( (tU8)m_poclConfigInterface->u32GetConfigData("TMCServer1") & DIA_TUNER_FM_TMCSERVER) != 0)     ? TRUE : FALSE);

}



/*** END OF FILE *************************************************************/

tU8 tun_Config::u8GetPresetListType(tVoid)
{
	if(m_poclConfigInterface  == NULL)
	{
		NORMAL_M_ASSERT (m_poclConfigInterface != NULL);
		return 0xff;
	}
    return (tU8)m_poclConfigInterface->u32GetConfigData("PresetListType");
}

tVoid tun_Config::vGetFrequencyRangeToDeleteElement(tU32 &u32StartFrequency, tU32 &u32EndFrequency, midw_fi_tcl_e8_Tun_TunerBand::tenType enBand)
{
	if(NULL==m_poclConfigInterface)
	{
		return;
	}
	switch(enBand)
	{
		case midw_fi_tcl_e8_Tun_TunerBand::FI_EN_TUN_BAND_CB1:
		{
			 if(m_poclConfigInterface->u32GetConfigData("CombiBands") &0x0006)
			{
				u32StartFrequency = m_poclConfigInterface->u32GetConfigData("LowLW");
				u32EndFrequency = m_poclConfigInterface->u32GetConfigData("HighMW");
	}
			break;
		}
		case midw_fi_tcl_e8_Tun_TunerBand::FI_EN_TUN_BAND_LW:
	{
			u32StartFrequency = m_poclConfigInterface->u32GetConfigData("LowLW");
			u32EndFrequency = m_poclConfigInterface->u32GetConfigData("HighLW");
			break;
		}
		case midw_fi_tcl_e8_Tun_TunerBand::FI_EN_TUN_BAND_MW:
		{
			u32StartFrequency = m_poclConfigInterface->u32GetConfigData("LowMW");
			u32EndFrequency = m_poclConfigInterface->u32GetConfigData("HighMW");
			break;
		}
		default:
			break;
	}
}
tU32 tun_Config::u32GetLSMFrequency(midw_fi_tcl_e8_Tun_TunerBand::tenType midwTunerBand)
{
	if(NULL==m_poclConfigInterface)
	{
		return 0;
	}
	switch(midwTunerBand)
	{
		case midw_fi_tcl_e8_Tun_TunerBand::FI_EN_TUN_BAND_CB1:
		{
			return m_poclConfigInterface->u32GetConfigData("MW_LSM");
			break;
		}
		case midw_fi_tcl_e8_Tun_TunerBand::FI_EN_TUN_BAND_FM:
		{
			return m_poclConfigInterface->u32GetConfigData("FM_LSM");
			break;
		}
		case midw_fi_tcl_e8_Tun_TunerBand::FI_EN_TUN_BAND_LW:
		{
			return m_poclConfigInterface->u32GetConfigData("LW_LSM");
			break;
		}
		case midw_fi_tcl_e8_Tun_TunerBand::FI_EN_TUN_BAND_MW:
		{
			return m_poclConfigInterface->u32GetConfigData("MW_LSM");
			break;
		}
		default:
		{
			break;
		}
	}
	return 0;

}

/*************************************************************************
*
* FUNCTION:     tun_Config:: bIsFrequencyWithinTheRange( )
* 
* DESCRIPTION:  Returns true if the Frequency is within the Range
*
* PARAMETER:    tU32 u32Frequncy
*
* RETURNVALUE:  tBool 
*
*************************************************************************/
tBool tun_Config:: bIsFrequencyWithinTheRange(tU8 u8Band, tU32 u32Frequency)const
{
  tBool flag=false;

	if(m_poclConfigInterface  == NULL)
	{
	NORMAL_M_ASSERT (m_poclConfigInterface != NULL);
	return false;
	}

	switch(u8Band)
	{
	case midw_fi_tcl_e8_Tun_TunerBand::FI_EN_TUN_BAND_FM:
		if(u32Frequency>=(unsigned int)m_poclConfigInterface->u32GetConfigData("LowFM") && u32Frequency<=(unsigned int)m_poclConfigInterface->u32GetConfigData("HighFM"))
		{
			flag=true;
		}
    break;

	case midw_fi_tcl_e8_Tun_TunerBand::FI_EN_TUN_BAND_LW:
	case midw_fi_tcl_e8_Tun_TunerBand::FI_EN_TUN_BAND_MW:
		if ((u32Frequency>=(unsigned int)m_poclConfigInterface->u32GetConfigData("LowLW") && u32Frequency<=(unsigned int)m_poclConfigInterface->u32GetConfigData("HighLW")) ||
    		 (u32Frequency>=(unsigned int)m_poclConfigInterface->u32GetConfigData("LowMW") && u32Frequency<=(unsigned int)m_poclConfigInterface->u32GetConfigData("HighMW")))
		{
			flag=true;
        }
     break;
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
	case midw_fi_tcl_e8_Tun_TunerBand::FI_EN_TUN_BAND_CB1:
	{
		if( m_poclConfigInterface->u32GetConfigData("CombiBands") == PSA_COMBIBAND_VALUE)
		{
			if ((u32Frequency>=(unsigned int)m_poclConfigInterface->u32GetConfigData("LowLW") && u32Frequency<=(unsigned int)m_poclConfigInterface->u32GetConfigData("HighLW")) ||
    			(u32Frequency>=(unsigned int)m_poclConfigInterface->u32GetConfigData("LowMW") && u32Frequency<=(unsigned int)m_poclConfigInterface->u32GetConfigData("HighMW")))
			{
				flag=true;
			}
		}
		else
		{
			if(u32Frequency>=(unsigned int)m_poclConfigInterface->u32GetConfigData("LowMW") && u32Frequency<=(unsigned int)m_poclConfigInterface->u32GetConfigData("HighMW"))
			{
				flag=true;
			}
		}
	}
  break;
#endif
	case midw_fi_tcl_e8_Tun_TunerBand::FI_EN_TUN_BAND_TRF:
	{
		if ((u32Frequency>=(unsigned int)TUN_CONFIG_DEFAULT_TRF_LOW_FREQ )&& (u32Frequency<=(unsigned int)TUN_CONFIG_DEFAULT_TRF_HIGH_FREQ))
		{
			flag=true;
		}
		break;
	}
	case midw_fi_tcl_e8_Tun_TunerBand::FI_EN_TUN_BAND_WB:
	{
		if(u32Frequency>=(unsigned int)m_poclConfigInterface->u32GetConfigData("LowWB")&& u32Frequency<=(unsigned int)m_poclConfigInterface->u32GetConfigData("HighWB"))
		{
			flag=true;
		}
		break;
	}
	default:
    flag=false;
    break;
  }

	return(flag);

}

/*************************************************************************
*
* FUNCTION:     tun_Config:: bIsFrequencyStepValid( )
* 
* DESCRIPTION:  Returns true if the Frequency step is valid
*
* PARAMETER:    tU32 u32Frequncy
*
* RETURNVALUE:  tBool 
*
*************************************************************************/
tBool tun_Config:: bIsFrequencyStepValid(tU8 u8Band, tU32 u32Frequency)const
{
  tBool flag=false;

	if(m_poclConfigInterface  == NULL)
	{
	NORMAL_M_ASSERT (m_poclConfigInterface != NULL);
	return false;
	}

	switch(u8Band)
	{
	case midw_fi_tcl_e8_Tun_TunerBand::FI_EN_TUN_BAND_FM:
		if((unsigned int)m_poclConfigInterface->u32GetConfigData("ManualStepSizeFM"))
		{
		if(((u32Frequency - (unsigned int)m_poclConfigInterface->u32GetConfigData("LowFM")) % (unsigned int)m_poclConfigInterface->u32GetConfigData("ManualStepSizeFM")) == 0)
		{
			flag=true;
			}
		}
    break;

	case midw_fi_tcl_e8_Tun_TunerBand::FI_EN_TUN_BAND_LW:
	case midw_fi_tcl_e8_Tun_TunerBand::FI_EN_TUN_BAND_MW:
		if((unsigned int)m_poclConfigInterface->u32GetConfigData("ManualStepSizeLW"))
		{
			if(((u32Frequency - (unsigned int)m_poclConfigInterface->u32GetConfigData("LowLW")) % (unsigned int)m_poclConfigInterface->u32GetConfigData("ManualStepSizeLW")) == 0)
			{
				flag=true;
			}
		}
		if((unsigned int)m_poclConfigInterface->u32GetConfigData("ManualStepSizeMW"))
		{
			if(((u32Frequency - (unsigned int)m_poclConfigInterface->u32GetConfigData("LowMW")) % (unsigned int)m_poclConfigInterface->u32GetConfigData("ManualStepSizeMW")) == 0)
		{
			flag=true;
        }
		}		
     break;
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
	case midw_fi_tcl_e8_Tun_TunerBand::FI_EN_TUN_BAND_CB1:
	{
		if( m_poclConfigInterface->u32GetConfigData("CombiBands") == PSA_COMBIBAND_VALUE)
		{
			if((unsigned int)m_poclConfigInterface->u32GetConfigData("ManualStepSizeLW"))
			{
				if(((u32Frequency - (unsigned int)m_poclConfigInterface->u32GetConfigData("LowLW")) % (unsigned int)m_poclConfigInterface->u32GetConfigData("ManualStepSizeLW")) == 0)
				{
					flag=true;
				}
			}
			if((unsigned int)m_poclConfigInterface->u32GetConfigData("ManualStepSizeMW"))
			{
				if(((u32Frequency - (unsigned int)m_poclConfigInterface->u32GetConfigData("LowMW")) % (unsigned int)m_poclConfigInterface->u32GetConfigData("ManualStepSizeMW")) == 0)
			{
				flag=true;
				}
			}
		}
		else
		{
			if((unsigned int)m_poclConfigInterface->u32GetConfigData("ManualStepSizeMW"))
			{
			if(((u32Frequency - (unsigned int)m_poclConfigInterface->u32GetConfigData("LowMW")) % (unsigned int)m_poclConfigInterface->u32GetConfigData("ManualStepSizeMW")) == 0) 
			{
				flag=true;
				}
			}
		}
	}
  break;
#endif
	case midw_fi_tcl_e8_Tun_TunerBand::FI_EN_TUN_BAND_TRF:
	{
		flag=true;
		break;
	}
	case midw_fi_tcl_e8_Tun_TunerBand::FI_EN_TUN_BAND_WB:
	{
		if((unsigned int)m_poclConfigInterface->u32GetConfigData("ManualStepSizeWB"))
		{
			if(((u32Frequency - (unsigned int)m_poclConfigInterface->u32GetConfigData("LowWB")) % (unsigned int)m_poclConfigInterface->u32GetConfigData("ManualStepSizeWB")) == 0) 
			{
				flag=true;
			}
		}
		break;
	}
	default:
    flag=false;
    break;
  }
	
	return(flag);
}