#ifndef TUN_DRVADRIF_HPP
#define TUN_DRVADRIF_HPP
/************************************************************************
* FILE:         tun_DrvAdrIf.hpp
* PROJECT:      g3g
* SW-COMPONENT:
*----------------------------------------------------------------------
*
* DESCRIPTION:
*
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
* HISTORY:
* Date       | Author                 | Modification
* --.--.2012 | CM-AI/PJ-VW34 Loss     | initial version
*
*************************************************************************/

#include <list>
#include <algorithm>
#include <queue>

using namespace std;

#ifndef ETG_S_IMPORT_INTERFACE_MOCKUP
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"
#endif

#include "tun_Class_ForwardDecl.h"
#include "tun_defines.h"
#include "tun_trace.h"
#include "tun_Singleton.h"
#include "tun_MsgToADR.h"
#include "tun_main.h"
#include "tun_SPM_Client.h"

class clConfigInterface;
/**Ngp1kor: Currently commented for migration to G3 Platform.*/
#ifdef VARIANT_S_FTR_ENABLE_OPTADR
#include "ssi_if.h"
#endif


/*****************************************************************************
 * ETG trace
 * 0x2619:    TUN_TRACE_CLASS_DRVADRIF
 ****************************************************************************/
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TUN_TRACE_CLASS_DRVADRIF
#include "trcGenProj/Header/tun_DrvAdrIf.hpp.trc.h"
#endif

#define TUN_DRVADRIF_CONF_DEVICE_AMFM_TUNER "/dev/adr3/amfm_tuner"
//#define TUN_DRVADRIF_CONF_AMFM_TUNER_CLIENT_ID                 0x0010
//#define TUN_DRVADRIF_CONF_AMFM_TUNER_INSTANCE_ID               0x01


#define TUN_DRVADRIF_GET_U32(pu8) ((tU32)(((tU32)((pu8)[0])<<24) + ((tU32)((pu8)[1])<<16) + ((tU32)((pu8)[2])<<8) + (pu8)[3]))
#define TUN_DRVADRIF_GET_U24(pu8) ((tU32)(((tU32)((pu8)[0])<<16) + ((tU32)((pu8)[1])<<8) + (pu8)[2]))
#define TUN_DRVADRIF_GET_U16(pu8) ((tU16)(((tU16)((pu8)[0])<<8) + (pu8)[1]))

//lint -emacro(572, TUN_DRVADRIF_SET_U32) prio2: Excessive shift
//lint -emacro(778, TUN_DRVADRIF_SET_U32) prio2: Constant expression evaluates to 0 in operation '>>'
#define TUN_DRVADRIF_SET_U32(pu8, u32)   \
    (pu8)[0]=(tU8)(((tU32)(u32)>>24)& 0xFF);       \
    (pu8)[1]=(tU8)(((tU32)(u32)>>16)& 0xFF);  \
    (pu8)[2]=(tU8)(((tU32)(u32)>>8) & 0xFF);  \
    (pu8)[3]=(tU8)(((tU32)(u32))    & 0xFF);

//lint -emacro(572, TUN_DRVADRIF_SET_U16) prio2: Excessive shift
//lint -emacro(778, TUN_DRVADRIF_SET_U16) prio2: Constant expression evaluates to 0 in operation '>>'
#define TUN_DRVADRIF_SET_U16(pu8, u16)   \
    (pu8)[0]=(tU8)(((tU16)(u16)>>8) & 0xFF);         \
    (pu8)[1]=(tU8)(((tU16)(u16))    & 0xFF);



#define LUNID 0x14

/*============================================================================
 *
 *--------------------------------------------------------------------------*/
#define TUN_DRVADRIF_OFFSET_CLIENT_ID     0   /*   2 bytes        */
#define TUN_DRVADRIF_OFFSET_FBLOCK_ID     2   /*   2 bytes        */
#define TUN_DRVADRIF_OFFSET_INSTANCE_ID   4   /*   1 byte         */
#define TUN_DRVADRIF_OFFSET_FKT_ID        5   /*   2 bytes        */
#define TUN_DRVADRIF_OFFSET_OP_TYPE       7   /*   1 byte         */
#define TUN_DRVADRIF_OFFSET_DATA_LEN      8   /*   2 bytes        */
#define TUN_DRVADRIF_OFFSET_DATA         10
#define TUN_DRVADRIF_HEADER_LEN          TUN_DRVADRIF_OFFSET_DATA
/*==========================================================================*/

/****************************************************************************/
/*                                                                          */
/* CLASS DEFINITION                                                         */
/*                                                                          */
/****************************************************************************/


/*****************************************************************************
 * Base class for message call back
 ****************************************************************************/
class tun_CallBack
{
  public:
  tun_CallBack( tVoid) : _bActive( TRUE)
  {
  }

    virtual void vCall( tun_RxMsg* poMsg) = 0;

  tBool _bActive;
};

/*****************************************************************************
 * This template creates for each receiver and each message one call back
 * function.
 ****************************************************************************/
template < class DESTINATION, class MSG>
class tun_MsgCallBack : public tun_CallBack
{
   public:
     virtual void vCall( tun_RxMsg* poMsg)
     {
       DESTINATION::instance()->vMessage( (MSG*)poMsg);
     }
};

/*****************************************************************************
 * This template function creates a static object of tun_MsgCallBack and
 * returns its address.
 ****************************************************************************/
template < class DESTINATION, class MSG>
inline tun_CallBack* tun_poGetMsgCallBack( tVoid)
{
   static tun_MsgCallBack< DESTINATION, MSG> oCallBack;

   return &oCallBack;
};

/*****************************************************************************
 * This template function creates a static subscriber list for all messages.
 ****************************************************************************/
template< class MSG>
list< tun_CallBack *>* TUN_poGetSubscriberList( tVoid)
{
  static list< tun_CallBack*> oSubscriberList;

  return &oSubscriberList;
}

/*****************************************************************************
 * Provides the address of call back list for the message type given by
 * the parameter 'poTypeInentifierDummy'.
 ****************************************************************************/
template< class MSG>
list< tun_CallBack *>* TUN_poGetSubscriberList( MSG* poTypeInentifierDummy)
{
  /* unused parameter, it contains the 'this' pointer of the caller. */
  /* It is needed to identify the type of 'this' pointer.            */
  (tVoid)poTypeInentifierDummy;

  list< tun_CallBack*>* poLi = TUN_poGetSubscriberList< MSG>();

  /* returns the call back list for the corresponding type */
  return poLi;
}


/*****************************************************************************
 * This definition installs the call back list within a message
 ****************************************************************************/
#define TUN_INSTALL_MESSAGE_CALLBACKLIST \
  virtual list <tun_CallBack* >* poGetSubscriberList( tVoid) { return TUN_poGetSubscriberList( this); }



/*****************************************************************************
 * Base class for all classes which want to receive ADR messages. It creates
 * also a static ponter to the instance of receiver (singleton).
 ****************************************************************************/
template <class ClassName>
class tun_MsgHandler
{
  public:

    virtual ~tun_MsgHandler( tVoid) { }

    template< class MSG>
    void vSubscribe( void)
    {
      /* create a static call back object.*/
      tun_CallBack* poCB = tun_poGetMsgCallBack< ClassName, MSG>();

      /* take pointer to the subcriber list for the given message */
      list< tun_CallBack*>* poSubscriberList = TUN_poGetSubscriberList< MSG>();

      /* creates an iterator */
      list< tun_CallBack*>::iterator iter= find( poSubscriberList->begin(), poSubscriberList->end(), poCB);

      if( iter == poSubscriberList->end())
      {
        /* add the call back to the end */
        poSubscriberList->push_back( poCB);
      }
      else
      {
        /* the message is already subscribed, reactivete it */
        (*iter)->_bActive = TRUE;
      }
    }

    template< class MSG>
    void vUnSubscribe( void)
    {
      /* create a static call back object.*/
      tun_CallBack* poCB = tun_poGetMsgCallBack< ClassName, MSG>();

      /* take pointer to the subcriber list for the given message */
      list< tun_CallBack*>* poSubscriberList = TUN_poGetSubscriberList< MSG>();

      /* creates an iterator */
      list< tun_CallBack*>::iterator iter= find(poSubscriberList->begin(), poSubscriberList->end(), poCB);

      if( iter != poSubscriberList->end())
      {
        /* deactivate the call back */
        (*iter)->_bActive = FALSE;
      }
    }
};



/*****************************************************************************
 *
 ****************************************************************************/
class tun_SsiTxBuffer
{
  public:

    /*-------------*/
    /* constructor */
    /*-------------*/
    tun_SsiTxBuffer( tVoid)
    {
      u32MessageLen   = 0; // makes lint happy
      au8TxBuffer[ 0] = 0; // makes lint happy
    }

    tU32 u32MessageLen;
    tU8  au8TxBuffer[ TUN_DRVADRIF_CONF_SSI_AMFM_TUNER_TX_BUFFER_SIZE];
};

class tun_SsiRxBuffer
{
  public:

    /*-------------*/
    /* constructor */
    /*-------------*/
    tun_SsiRxBuffer( tPCU8 pcu8SetRxBuffer, tU32 u32SetRxMessageLen) :
      u32RxMessageLen( u32SetRxMessageLen),
      pcu8RxBuffer(pcu8SetRxBuffer )
    {
    }

    tun_tenAmFmTunerMsg enGetFunctionId( tVoid) const
    {
      return (tun_tenAmFmTunerMsg)TUN_DRVADRIF_GET_U16( &pcu8RxBuffer[ TUN_DRVADRIF_OFFSET_FKT_ID]);
    }

    tun_tenOpType enGetOpType( tVoid) const
    {
      return (tun_tenOpType)pcu8RxBuffer[ TUN_DRVADRIF_OFFSET_OP_TYPE];
    }

    tU32   u32RxMessageLen;
    tPCU8  pcu8RxBuffer;
};


/****************************************************************************/
// interface
class tun_iSerialize
{
  public:
    virtual tVoid vSerialize( tun_SsiTxBuffer& rRawOutput) = 0;

};


/****************************************************************************/
class tun_TxMsg
{
  public:
    tun_TxMsg( tVoid)
    {
      enClientId   = enClientId_AmFmTuner;
      enFBlockId   = enFBlockId_AmFmTuner;
      enInstanceId = enInstanceId_AudioSource1;
      enFunctionId = enMsg_Unknown_Message_0x0000;
      enOpType     = enOpType_INVALID;
      u16DataLen   = 0;
    }

    virtual tVoid vSerialize( tPU8 pu8Data) = 0;
    virtual tVoid vTrace( tVoid) = 0;

    tVoid vSerialize( tun_SsiTxBuffer* pRawOutput)
    {
      TUN_DRVADRIF_SET_U16( &pRawOutput->au8TxBuffer[ enAdrMsgOffset_CLIENT_ID], enClientId);
      TUN_DRVADRIF_SET_U16( &pRawOutput->au8TxBuffer[ enAdrMsgOffset_FBLOCK_ID], enFBlockId);
      pRawOutput->au8TxBuffer[ enAdrMsgOffset_INSTANCE_ID] = (tU8)enInstanceId;
      TUN_DRVADRIF_SET_U16( &pRawOutput->au8TxBuffer[ enAdrMsgOffset_FKT_ID], enFunctionId);
      pRawOutput->au8TxBuffer[ enAdrMsgOffset_OP_TYPE]     = (tU8)enOpType;

      vSerialize( &pRawOutput->au8TxBuffer[ enAdrMsgOffset_DATA]);

      TUN_DRVADRIF_SET_U16( &pRawOutput->au8TxBuffer[ enAdrMsgOffset_DATA_LEN], u16DataLen);
      pRawOutput->u32MessageLen =   enAdrMsgOffset_DATA     /* header length */
                                  + u16DataLen;                  /* + payload     */
    }

    tun_tenClientId      enClientId;
    tun_tenFBlockId      enFBlockId;
    tun_tenInstanceId    enInstanceId;
    tun_tenAmFmTunerMsg  enFunctionId;
    tun_tenOpType        enOpType;
    tU16                 u16DataLen;
};


/****************************************************************************/
class tun_RxMsg
{
  public:

    virtual ~tun_RxMsg( tVoid) {};
    tun_RxMsg( tVoid) : bIsValid( FALSE)
    {
      enClientId   = enClientId_undefined;
      enFBlockId   = enFBlockId_undefined;
      enFunctionId = enMsg_Unknown_Message_0x0000;
      enInstanceId = enInstanceId_undefined;
      enOpType     = enOpType_INVALID;
      u16DataLen   = 0;
    };

    virtual list <tun_CallBack* > *poGetSubscriberList() = 0;
    virtual tVoid vTrace( tVoid) = 0;
    virtual tBool bParse( tPCU8 pcu8Data) = 0;
    virtual tun_tenErrorCode enParseError( tPCU8 pcu8Data)
    {
      return (tun_tenErrorCode)pcu8Data[ 0];
    }

    tun_tenErrorCode enParse( tun_SsiRxBuffer* pRawInput)
    {
      tun_tenErrorCode enRetVal = enErrorCode_No_Error;

      enClientId   = (tun_tenClientId)TUN_DRVADRIF_GET_U16( &pRawInput->pcu8RxBuffer[ enAdrMsgOffset_CLIENT_ID]);
      enFBlockId   = (tun_tenFBlockId)TUN_DRVADRIF_GET_U16( &pRawInput->pcu8RxBuffer[ enAdrMsgOffset_FBLOCK_ID]);
      enInstanceId = (tun_tenInstanceId)pRawInput->pcu8RxBuffer[ enAdrMsgOffset_INSTANCE_ID];
      enFunctionId = (tun_tenAmFmTunerMsg)TUN_DRVADRIF_GET_U16( &pRawInput->pcu8RxBuffer[ enAdrMsgOffset_FKT_ID]);
      enOpType     = (tun_tenOpType)pRawInput->pcu8RxBuffer[ enAdrMsgOffset_OP_TYPE];
      u16DataLen   = TUN_DRVADRIF_GET_U16( &pRawInput->pcu8RxBuffer[ enAdrMsgOffset_DATA_LEN]);

      if( enOpType == enOpType_INVALID)
      {
        enRetVal = enErrorCode_OpType_INVALID;
      }
      else
      {
        if( enOpType == enOpType_ERROR)
        {
          enRetVal = enParseError( &pRawInput->pcu8RxBuffer[ enAdrMsgOffset_DATA]);
          bIsValid = FALSE;
        }
        else
        {
          bIsValid = (pRawInput->u32RxMessageLen >= enAdrMsgOffset_HEADER_LEN) ? TRUE : FALSE;
          bIsValid = (bIsValid & ((tBool) (bParse( &pRawInput->pcu8RxBuffer[ enAdrMsgOffset_DATA]))));
        }
      }

      return enRetVal;
    }


    tun_tenClientId      enClientId;
    tun_tenFBlockId      enFBlockId;
    tun_tenInstanceId    enInstanceId;
    tun_tenAmFmTunerMsg  enFunctionId;
    tun_tenOpType        enOpType;
    tU16                 u16DataLen;
    tBool                bIsValid;
};



/*****************************************************************************
 * Error
 *  #### ####  ####   ##  ####
 *  #    #   # #   # #  # #   #
 *  #    #   # #   # #  # #   #
 *  ###  ####  ####  #  # ####
 *  #    # #   # #   #  # # #
 *  #    #  #  #  #  #  # #  #
 *  #### #   # #   #  ##  #   #
 *
 *
 ****************************************************************************/

/*****************************************************************************
 * OPType: Error
 ****************************************************************************/
struct trMsg_Generic_Error  :
  public tun_RxMsg
{
  TUN_INSTALL_MESSAGE_CALLBACKLIST;

  trMsg_Generic_Error( tVoid) :
    enError( enErrorCode_Invalid_format),
    enErrOpType( enOpType_INVALID),
    enErrFunctionId( enMsg_Unknown_Message_0x0000),
    u8Parameter( 0)
  {

  }

  virtual tBool bParse( tPCU8 pcu8Data) { (tVoid)pcu8Data; return FALSE; }

  virtual tun_tenErrorCode enParseError( tPCU8 pcu8Data)
  {
    if( u16DataLen > 0)
    {
      enError = (tun_tenErrorCode)pcu8Data[ 0];

      switch( enError)
      {
        /*------------------------------------------------------------------*/

        case enErrorCode_OPType_not_available:
          enErrOpType = (tun_tenOpType)pcu8Data[ 1];
          break;

        /*------------------------------------------------------------------*/

        case enErrorCode_Parameter_wrong_or_out_of_range:
          u8Parameter = pcu8Data[ 1];
          break;

        /*------------------------------------------------------------------*/

        case enErrorCode_Parameter_not_available:
          u8Parameter = pcu8Data[ 1];
          break;

        /*------------------------------------------------------------------*/

        case enErrorCode_Given_FktId_cannot_be_registered:
          enErrFunctionId = (tun_tenAmFmTunerMsg)TUN_DRVADRIF_GET_U16( &pcu8Data[ 1]);
          break;

        /*------------------------------------------------------------------*/

        default:
          //u8ErrorInfo = pcu8Data[ 1];
          break;

        /*------------------------------------------------------------------*/
      }
    }

    return enError;
  }

  virtual tVoid vTrace( tVoid)
  {
    switch( enError)
    {
      /*------------------------------------------------------------------*/

      case enErrorCode_OPType_not_available:
        ETG_TRACE_ERR(( " trMsg_Generic_Error: Invalid OPType\n"
          "->01 ClientID   : %04x\n"
          "->02 FBlock     : %04x\n"
          "->03 InstanceId : %02x\n"
          "->04 FunctionId : %04x\n"
          "->05 OpType     : %02x\n"
          "->06 %d bytes payload\n"
          "=>   ErrorCode  : %02x\n"
          "=>   OPType     : %02x\n"
          ,ETG_CENUM(tun_tenClientId,         enClientId)
          ,ETG_CENUM(tun_tenFBlockId,         enFBlockId)
          ,ETG_CENUM(tun_tenInstanceId,       enInstanceId)
          ,ETG_CENUM(tun_tenAmFmTunerMsg,     enFunctionId)
          ,ETG_CENUM(tun_tenOpType,           enOpType)
          ,u16DataLen
          ,ETG_CENUM(tun_tenErrorCode,        enError)
          ,ETG_CENUM(tun_tenOpType,           enErrOpType)
          ));
        break;

      /*------------------------------------------------------------------*/

      case enErrorCode_Parameter_wrong_or_out_of_range:
      case enErrorCode_Parameter_not_available:
        ETG_TRACE_ERR(( " trMsg_Generic_Error: Index of first byte that contains wrong data\n"
          "->01 ClientID   : %04x\n"
          "->02 FBlock     : %04x\n"
          "->03 InstanceId : %02x\n"
          "->04 FunctionId : %04x\n"
          "->05 OpType     : %02x\n"
          "->06 %d bytes payload\n"
          "=>   ErrorCode             : %02x\n"
          "=>   Parameter Nr.         : %02x\n"
          ,ETG_CENUM(tun_tenClientId,         enClientId)
          ,ETG_CENUM(tun_tenFBlockId,         enFBlockId)
          ,ETG_CENUM(tun_tenInstanceId,       enInstanceId)
          ,ETG_CENUM(tun_tenAmFmTunerMsg,     enFunctionId)
          ,ETG_CENUM(tun_tenOpType,           enOpType)
          ,u16DataLen
          ,ETG_CENUM(tun_tenErrorCode,        enError)
          ,u8Parameter
          ));
        break;

      /*------------------------------------------------------------------*/

      case enErrorCode_Given_FktId_cannot_be_registered:
        ETG_TRACE_ERR(( " trMsg_Generic_Error: First Fktid that cannot be registered\n"
          "->01 ClientID   : %04x\n"
          "->02 FBlock     : %04x\n"
          "->03 InstanceId : %02x\n"
          "->04 FunctionId : %04x\n"
          "->05 OpType     : %02x\n"
          "->06 %d bytes payload\n"
          "=>   ErrorCode             : %02x\n"
          "=>   FktId                 : %04x\n"
          ,ETG_CENUM(tun_tenClientId,         enClientId)
          ,ETG_CENUM(tun_tenFBlockId,         enFBlockId)
          ,ETG_CENUM(tun_tenInstanceId,       enInstanceId)
          ,ETG_CENUM(tun_tenAmFmTunerMsg,     enFunctionId)
          ,ETG_CENUM(tun_tenOpType,           enOpType)
          ,u16DataLen
          ,ETG_CENUM(tun_tenErrorCode,        enError)
          ,ETG_CENUM(tun_tenAmFmTunerMsg,     enErrFunctionId)
          ));
        break;

      /*------------------------------------------------------------------*/

      default:
        ETG_TRACE_ERR(( " trMsg_Generic_Error\n"
          "->01 ClientID   : %04x\n"
          "->02 FBlock     : %04x\n"
          "->03 InstanceId : %02x\n"
          "->04 FunctionId : %04x\n"
          "->05 OpType     : %02x\n"
          "->06 %d bytes payload\n"
          "=>   ErrorCode             : %02x\n"
          ,ETG_CENUM(tun_tenClientId,         enClientId)
          ,ETG_CENUM(tun_tenFBlockId,         enFBlockId)
          ,ETG_CENUM(tun_tenInstanceId,       enInstanceId)
          ,ETG_CENUM(tun_tenAmFmTunerMsg,     enFunctionId)
          ,ETG_CENUM(tun_tenOpType,           enOpType)
          ,u16DataLen
          ,ETG_CENUM(tun_tenErrorCode,        enError)
          ));
        break;

      /*------------------------------------------------------------------*/
    }
  }

  tun_tenErrorCode         enError;
  //tU8                      u8ErrorInfo;
  tun_tenOpType            enErrOpType;
  tun_tenAmFmTunerMsg      enErrFunctionId;
  tU8                      u8Parameter;
};




/*****************************************************************************
 * CM_Startup
 *
 *   ##  #   #       ### #####  ##  ####  ##### #   # ###
 *  #  # #   #      #      #   #  # #   #   #   #   # #  #
 *  #    ## ##      #      #   #  # #   #   #   #   # #  #
 *  #    # # #       ##    #   #### ####    #   #   # ###
 *  #    #   #         #   #   #  # # #     #   #   # #
 *  #  # #   #         #   #   #  # #  #    #   #   # #
 *   ##  #   # #### ###    #   #  # #   #   #    ###  #
 *
 * OPType:
 ****************************************************************************/

/*****************************************************************************
 * OPType: Status
 ****************************************************************************/
struct trMsg_CM_Startup_Status  :
  public tun_RxMsg
{
  TUN_INSTALL_MESSAGE_CALLBACKLIST;

  trMsg_CM_Startup_Status( tVoid)
  {
  }

  virtual tBool bParse(tPCU8 pcu8Data)
  {
    (tVoid)pcu8Data;  /* unused parameter */

    return TRUE;
  }

  virtual tVoid vTrace( tVoid)
  {
    ETG_TRACE_USR1(( " trMsg_CM_Startup_Status\n"
      "->01 ClientID   : %04x\n"
      "->02 FBlock     : %04x\n"
      "->03 InstanceId : %02x\n"
      "->04 FunctionId : %04x\n"
      "->05 OpType     : %02x\n"
      "->06 %d bytes payload\n"
      ,ETG_CENUM(tun_tenClientId,         enClientId)
      ,ETG_CENUM(tun_tenFBlockId,         enFBlockId)
      ,ETG_CENUM(tun_tenInstanceId,       enInstanceId)
      ,ETG_CENUM(tun_tenAmFmTunerMsg,     enFunctionId)
      ,ETG_CENUM(tun_tenOpType,           enOpType)
      ,u16DataLen
      ));
  }

};


/*****************************************************************************
 * CM_Ping
 *
 *   ##  #   #      ###  ### #   #  ###
 *  #  # #   #      #  #  #  ##  # #   #
 *  #    ## ##      #  #  #  # # # #
 *  #    # # #      ###   #  # # # # ###
 *  #    #   #      #     #  #  ## #   #
 *  #  # #   #      #     #  #  ## #   #
 *   ##  #   # #### #    ### #   #  ###
 *
 * OPType: SetGet, Status, Error
 ****************************************************************************/

/*****************************************************************************
 * OPType: SetGet
 ****************************************************************************/
struct trMsg_CM_Ping_SetGet :
  public tun_TxMsg
{
  public:
    trMsg_CM_Ping_SetGet( tU8 u8SetValue = 0x00) :
      u8Value( u8SetValue)
    {
      enInstanceId    = enInstanceId_All;
      enOpType        = enOpType_SETGET;
      enFunctionId    = enMsg_CM_Ping;
      u16DataLen      = 1;
    }

    virtual tVoid vSerialize( tPU8 pu8Data)
    {
      pu8Data[ 0]    = u8Value;
    }

    virtual tVoid vTrace( tVoid)
    {
      ETG_TRACE_USR4(( " trMsg_CM_Ping_SetGet\n"
        "->01 ClientID   : %04x\n"
        "->02 FBlock     : %04x\n"
        "->03 InstanceId : %02x\n"
        "->04 FunctionId : %04x\n"
        "->05 OpType     : %02x\n"
        "->06 %d bytes payload\n"
        "=> Value        : %02x\n"
        ,ETG_CENUM(tun_tenClientId,         enClientId)
        ,ETG_CENUM(tun_tenFBlockId,         enFBlockId)
        ,ETG_CENUM(tun_tenInstanceId,       enInstanceId)
        ,ETG_CENUM(tun_tenAmFmTunerMsg,     enFunctionId)
        ,ETG_CENUM(tun_tenOpType,           enOpType)
        ,u16DataLen
        ,u8Value
        ));
    }

    tU8                     u8Value;
};


/*****************************************************************************
 * OPType: Status
 ****************************************************************************/
struct trMsg_CM_Ping_Status  :
  public tun_RxMsg
{
  TUN_INSTALL_MESSAGE_CALLBACKLIST;

  trMsg_CM_Ping_Status( tVoid) :
    u8Value( 0x00)
  {
  }

  virtual tBool bParse(tPCU8 pcu8Data)
  {
    u8Value = pcu8Data[ 0];

    return TRUE;
  }

  virtual tVoid vTrace( tVoid)
  {
    ETG_TRACE_USR4(( " trMsg_CM_Ping_Status\n"
      "->01 ClientID   : %04x\n"
      "->02 FBlock     : %04x\n"
      "->03 InstanceId : %02x\n"
      "->04 FunctionId : %04x\n"
      "->05 OpType     : %02x\n"
      "->06 %d bytes payload\n"
      "=>   Data                  : %d\n"
      ,ETG_CENUM(tun_tenClientId,         enClientId)
      ,ETG_CENUM(tun_tenFBlockId,         enFBlockId)
      ,ETG_CENUM(tun_tenInstanceId,       enInstanceId)
      ,ETG_CENUM(tun_tenAmFmTunerMsg,     enFunctionId)
      ,ETG_CENUM(tun_tenOpType,           enOpType)
      ,u16DataLen
      ,u8Value
      ));
  }

  tU8                      u8Value;
};



/*****************************************************************************
 * CM_PrepareShutdown
 *
 *   ##  #   #      ###  ####  #### ###   ##  ####  ####  ### #  # #   # ##### ###   ##  #   # #   #
 *  #  # #   #      #  # #   # #    #  # #  # #   # #    #    #  # #   #   #   #  # #  # #   # ##  #
 *  #    ## ##      #  # #   # #    #  # #  # #   # #    #    #  # #   #   #   #  # #  # #   # # # #
 *  #    # # #      ###  ####  ###  ###  #### ####  ###   ##  #### #   #   #   #  # #  # #   # # # #
 *  #    #   #      #    # #   #    #    #  # # #   #       # #  # #   #   #   #  # #  # # # # #  ##
 *  #  # #   #      #    #  #  #    #    #  # #  #  #       # #  # #   #   #   #  # #  # # # # #  ##
 *   ##  #   # #### #    #   # #### #    #  # #   # #### ###  #  #  ###    #   ###   ##   # #  #   #
 *
 * OPType:
 ****************************************************************************/


/*****************************************************************************
 * CM_Shutdown
 *
 *   ##  #   #       ### #  # #   # ##### ###   ##  #   # #   #
 *  #  # #   #      #    #  # #   #   #   #  # #  # #   # ##  #
 *  #    ## ##      #    #  # #   #   #   #  # #  # #   # # # #
 *  #    # # #       ##  #### #   #   #   #  # #  # #   # # # #
 *  #    #   #         # #  # #   #   #   #  # #  # # # # #  ##
 *  #  # #   #         # #  # #   #   #   #  # #  # # # # #  ##
 *   ##  #   # #### ###  #  #  ###    #   ###   ##   # #  #   #
 *
 * OPType:
 ****************************************************************************/



/*****************************************************************************
 * FktIds
 *
 *  #### #   # ##### ### ###   ###
 *  #    #  #    #    #  #  # #
 *  #    # #     #    #  #  # #
 *  ###  ##      #    #  #  #  ##
 *  #    # #     #    #  #  #    #
 *  #    #  #    #    #  #  #    #
 *  #    #   #   #   ### ###  ###
 *
 * OPType:
 ****************************************************************************/


/*****************************************************************************
 * Notification
 *
 *  #   #  ##  ##### ### #### ###  ##   ##  ##### ###  ##  #   #
 *  ##  # #  #   #    #  #     #  #  # #  #   #    #  #  # ##  #
 *  # # # #  #   #    #  #     #  #    #  #   #    #  #  # # # #
 *  # # # #  #   #    #  ###   #  #    ####   #    #  #  # # # #
 *  #  ## #  #   #    #  #     #  #    #  #   #    #  #  # #  ##
 *  #  ## #  #   #    #  #     #  #  # #  #   #    #  #  # #  ##
 *  #   #  ##    #   ### #    ###  ##  #  #   #   ###  ##  #   #
 *
 * Register/Unregister properties for auto notification
 * OPType: Set, Get, Status, Error
 ****************************************************************************/
#define TUN_DRVADRIF_CONF_NOTIFIKATION_MAX_FKTIDLIST  20

#if (( TUN_DRVADRIF_CONF_NOTIFIKATION_MAX_FKTIDLIST * 2 + TUN_DRVADRIF_HEADER_LEN) > TUN_DRVADRIF_CONF_SSI_AMFM_TUNER_TX_BUFFER_SIZE)

  #error  "SSI buffer is not big enough"

#endif


/*****************************************************************************
 * OPType: Set
 ****************************************************************************/
struct trMsg_Notification_Set :
  public tun_TxMsg
{
  public:
    trMsg_Notification_Set( tenNotificationControl enSetControl, tun_tenClientId enSetClientID)
    {
      enInstanceId     = enInstanceId_All;
      enControl        = enSetControl;
      enClientID       = enSetClientID;
      enOpType         = enOpType_SET;
      enFunctionId     = enMsg_Notification;
      u8NumberOfFktID  = 0;
      u16DataLen       = 3;
      aenFktIDList[ 0] = enMsg_Unknown_Message_0x0000;  // makes lint happy
    }

    virtual tVoid vSerialize( tPU8 pu8Data)
    {
      pu8Data[ 0]    = (tU8)enControl;
      TUN_DRVADRIF_SET_U16( &pu8Data[ 1], enClientID);

      if( ( enControl == enNotificationControl_SetFunction) || (enControl == enNotificationControl_ClearFunction))
      {
        for( tU8 u8Index = 0; u8Index < u8NumberOfFktID; u8Index++)
        {
          TUN_DRVADRIF_SET_U16( &pu8Data[ 3 + (u8Index * 2)], aenFktIDList[ u8Index]);
        }
        u16DataLen = (tU16)(u16DataLen + ( (tU16)(u8NumberOfFktID * 2)));
      }
    }

    virtual tVoid vTrace( tVoid)
    {
#   ifndef LSIM
	  ETG_TRACE_USR1(( " trMsg_Notification_Set\n"
        "->01 ClientID   : %04x\n"
        "->02 FBlock     : %04x\n"
        "->03 InstanceId : %02x\n"
        "->04 FunctionId : %04x\n"
        "->05 OpType     : %02x\n"
        "->06 %d bytes payload\n"
        "=> Control      : %02x\n"
        "=> ClientID     : %04x\n"
        "=> FktIDList    : %*p\n"
        ,ETG_CENUM(tun_tenClientId,         enClientId)
        ,ETG_CENUM(tun_tenFBlockId,         enFBlockId)
        ,ETG_CENUM(tun_tenInstanceId,       enInstanceId)
        ,ETG_CENUM(tun_tenAmFmTunerMsg,     enFunctionId)
        ,ETG_CENUM(tun_tenOpType,           enOpType)
        ,u16DataLen
        ,ETG_CENUM(tenNotificationControl,  enControl)
        ,ETG_CENUM(tun_tenClientId,        enClientID)
        ,ETG_LIST_LEN(u8NumberOfFktID), ETG_LIST_PTR_T16((tPU16)aenFktIDList)
        ));
#endif
    }

    void AddFktID( tun_tenAmFmTunerMsg enFktID)
    {
      if( u8NumberOfFktID < TUN_DRVADRIF_CONF_NOTIFIKATION_MAX_FKTIDLIST)
      {
        if( ( enControl == enNotificationControl_SetFunction) || (enControl == enNotificationControl_ClearFunction))
        {
          aenFktIDList[ u8NumberOfFktID] = enFktID;
          u8NumberOfFktID++;
        }
      }
      else
      {
        /* buffer to small : TUN_DRVADRIF_CONF_NOTIFIKATION_MAX_FKTIDLIST */
        NORMAL_M_ASSERT_ALWAYS();
      }
    }

    tenNotificationControl  enControl;
    tun_tenClientId         enClientID;
    tU8                     u8NumberOfFktID;
    tun_tenAmFmTunerMsg     aenFktIDList[ TUN_DRVADRIF_CONF_NOTIFIKATION_MAX_FKTIDLIST];
};


/*****************************************************************************
 * NotificationCheck
 *
 *  #   #  ##  ##### ### #### ###  ##   ##  ##### ###  ##  #   #  ##  #  # ####  ##  #   #
 *  ##  # #  #   #    #  #     #  #  # #  #   #    #  #  # ##  # #  # #  # #    #  # #  #
 *  # # # #  #   #    #  #     #  #    #  #   #    #  #  # # # # #    #  # #    #    # #
 *  # # # #  #   #    #  ###   #  #    ####   #    #  #  # # # # #    #### ###  #    ##
 *  #  ## #  #   #    #  #     #  #    #  #   #    #  #  # #  ## #    #  # #    #    # #
 *  #  ## #  #   #    #  #     #  #  # #  #   #    #  #  # #  ## #  # #  # #    #  # #  #
 *  #   #  ##    #   ### #    ###  ##  #  #   #   ###  ##  #   #  ##  #  # ####  ##  #   #
 *
 * OPType:
 ****************************************************************************/


/*****************************************************************************
 * CM_AudioSource
 *
 *   ##  #   #       ##  #   # ###  ###  ##   ###  ##  #   # ####   ##  ####
 *  #  # #   #      #  # #   # #  #  #  #  # #    #  # #   # #   # #  # #
 *  #    ## ##      #  # #   # #  #  #  #  # #    #  # #   # #   # #    #
 *  #    # # #      #### #   # #  #  #  #  #  ##  #  # #   # ####  #    ###
 *  #    #   #      #  # #   # #  #  #  #  #    # #  # #   # # #   #    #
 *  #  # #   #      #  # #   # #  #  #  #  #    # #  # #   # #  #  #  # #
 *   ##  #   # #### #  #  ###  ###  ###  ##  ###   ##   ###  #   #  ##  ####
 *
 * OPType: Status
 ****************************************************************************/
struct trMsg_CM_AudioSource_Status  :
  public tun_RxMsg
{
  TUN_INSTALL_MESSAGE_CALLBACKLIST;

  virtual tBool bParse(tPCU8 pcu8Data)
  {
    u8AudioSource = pcu8Data[ 0];
    return TRUE;
  }

  virtual tVoid vTrace( tVoid)
  {
    ETG_TRACE_USR4(( " trMsg_CM_AudioSource_Status\n"
      "->01 ClientID   : %04x\n"
      "->02 FBlock     : %04x\n"
      "->03 InstanceId : %02x\n"
      "->04 FunctionId : %04x\n"
      "->05 OpType     : %02x\n"
      "->06 %d bytes payload\n"
      "=> AudioSource : %02x\n"
      ,ETG_CENUM(tun_tenClientId,         enClientId)
      ,ETG_CENUM(tun_tenFBlockId,         enFBlockId)
      ,ETG_CENUM(tun_tenInstanceId,       enInstanceId)
      ,ETG_CENUM(tun_tenAmFmTunerMsg,     enFunctionId)
      ,ETG_CENUM(tun_tenOpType,           enOpType)
      ,u16DataLen
      ,u8AudioSource
      ));
  }
  tU8  u8AudioSource;
};


/*****************************************************************************
 * OPType: Set
 ****************************************************************************/
struct trMsg_CM_AudioSource_Set :
  public tun_TxMsg
{
  public:
    trMsg_CM_AudioSource_Set( tVoid) :
      enAudioSource( enAudioSource_No_audio)
    {
      enOpType        = enOpType_SET;
      enFunctionId    = enMsg_CM_AudioSource;
      u16DataLen      = 1;
    }

    virtual tVoid vSerialize( tPU8 pu8Data)
    {
      pu8Data[ 0]    = (tU8)enAudioSource;
    }

    virtual tVoid vTrace( tVoid)
    {
      ETG_TRACE_USR1(( " trMsg_CM_AudioSource_Set\n"
        "->01 ClientID   : %04x\n"
        "->02 FBlock     : %04x\n"
        "->03 InstanceId : %02x\n"
        "->04 FunctionId : %04x\n"
        "->05 OpType     : %02x\n"
        "->06 %d bytes payload\n"
        "=> AudioSource  : %02x\n"
        ,ETG_CENUM(tun_tenClientId,         enClientId)
        ,ETG_CENUM(tun_tenFBlockId,         enFBlockId)
        ,ETG_CENUM(tun_tenInstanceId,       enInstanceId)
        ,ETG_CENUM(tun_tenAmFmTunerMsg,     enFunctionId)
        ,ETG_CENUM(tun_tenOpType,           enOpType)
        ,u16DataLen
        ,ETG_CENUM(tenAudioSource,  enAudioSource)
        ));
    }

    tenAudioSource   enAudioSource;
};



 struct trMsg_CM_TMCTrafficInfos_Status  :
    public tun_RxMsg
  {
    TUN_INSTALL_MESSAGE_CALLBACKLIST;

    virtual tBool bParse(tPCU8 pcu8Data)
    {
      u8Data = 0;
      u8NoOfListElem = pcu8Data[enAdrMsgOffset_NoOf8AMessages_index];
      u8NoOfListElem = (tU8)(u8NoOfListElem * 5);
      u88A = enAdrMsgOffset_NoOf8AMessages_index+1;
      pu8TMCData = pcu8Data;

      u8CountryCode       = pcu8Data[ u8Data ++];
      u8TableNo              = pcu8Data[u8Data++];
      u8ServiceID            = pcu8Data[u8Data++];
       u8Encrypted            = pcu8Data[u8Data++];
       u16ECC                  =  pcu8Data[u8Data++];
       u16AppID        = TUN_DRVADRIF_GET_U16(&pcu8Data[u8Data]);
       u8Data = (tU8)(u8Data+2);
      u16ProgramID            = TUN_DRVADRIF_GET_U16(&pcu8Data[u8Data]);
      u8Data = (tU8)(u8Data+2);
      u8EncryptedStatus           = pcu8Data[u8Data++];
      u8Encid               = pcu8Data[u8Data++];
      u8TestBits            = pcu8Data[u8Data++];
      u8MediumTypeId         = pcu8Data[u8Data++];
      u8MsgSourceNo               = pcu8Data[u8Data++];
      OSAL_pvMemoryCopy( sPSName, &pcu8Data[ u8Data], 16);
      u8Data = (tU8)(u8Data+16);
      OSAL_pvMemoryCopy( sServiceProviderName, &pcu8Data[ u8Data], 16);
      u8Data = (tU8)(u8Data +16);

	  u8Data++; //NoOfBufferedTrafficInfos

	  //tU8 u8Index = 0;
	//  OSAL_pvMemoryCopy(  &pu8TMCData[u8Index], &pcu8Data[ u8Data],u8NoOfListElem);

	  ETG_TRACE_USR4(( " trMsg_CM_TMCTrafficInfos_Status -- u8NoOfListElem = %d", u8NoOfListElem ));
      /*for(tU8 u8Index = 0; u8Index < u8NoOfListElem; u8Index ++)
      {
    	  ETG_TRACE_USR4(( " trMsg_CM_TMCTrafficInfos_Status -- u8Index = %d", u8Index ));
    	  pu8TMCData[u8Index] = pcu8Data[u8Data];
    	  u8Data++;
      }*/

      return TRUE;
    }

    virtual tVoid vTrace( tVoid)
    {
      ETG_TRACE_USR4(( " trMsg_CM_TMCTrafficInfos_Status -- Part1\n"
        "-> CountryCode   : %02x\n"
        "-> TableNo     : %02x\n"
        "-> ServID   : %02x\n"
        "-> Encrypted : %02x\n"
        "-> ECC     : %04x\n"
        "-> AppID:   %02x\n"
        "-> ProgramID:   %04x\n"
        ,u8CountryCode
        ,u8TableNo
        ,u8ServiceID
        ,u8Encrypted
        ,u16ECC
        ,u16AppID
        ,u16ProgramID
        ));

      ETG_TRACE_USR4(( " trMsg_CM_TMCTrafficInfos_Status -- Part2\n"
              "-> EncryptedStatus:   %02x\n"
              "-> EncID:   %02x\n"
              "->TestBits : %02x\n"
          	 "->mediumTypeID : %02x\n"
          	 "->MsgSourceNo : %02x\n"
          	 "->sPSName : %16s\n"
          	 "->sServiceProviderName : %16s\n"
              ,u8EncryptedStatus
              ,u8Encid
              ,u8TestBits
              ,u8MediumTypeId
              ,u8MsgSourceNo
              ,sPSName
              ,sServiceProviderName
              ));

      for(tU8 u8Index = 0; u8Index < u8NoOfListElem; u8Index ++)
      {
    	  ETG_TRACE_USR4(( " ->8AData : %02x\n"
    	 , pu8TMCData[u88A]
    	              ));
    	  u88A++;
      }


    }
    tU8  u8Data;
    tU8 u88A;
    tU8  u8AudioSource;
    tU8 u8NoOfListElem;
    tPCU8 pu8TMCData;
    tU8 u8CountryCode;
    tU8 u8TableNo;
    tU8 u8ServiceID;
    tU8 u8Encrypted;
    tU16 u16ECC;
    tU16 u16AppID;
           tU16 u16ProgramID;
           tU8 u8EncryptedStatus;
           tU8 u8Encid;
           tU8 u8TestBits;
           tU8 u8MediumTypeId;

           tU8 u8MsgSourceNo;
           tChar                    sPSName[ 16];
           tChar                    sServiceProviderName[ 16];

  };

/*****************************************************************************
 * CM_ATBand
 *
 *   ##  #   #       ##  ##### ###   ##  #   # ###
 *  #  # #   #      #  #   #   #  # #  # ##  # #  #
 *  #    ## ##      #  #   #   #  # #  # # # # #  #
 *  #    # # #      ####   #   ###  #### # # # #  #
 *  #    #   #      #  #   #   #  # #  # #  ## #  #
 *  #  # #   #      #  #   #   #  # #  # #  ## #  #
 *   ##  #   # #### #  #   #   ###  #  # #   # ###
 *
 *
 * OPType: Set, Get, Status
 ****************************************************************************/


/*****************************************************************************
 * OPType: Set
 ****************************************************************************/
struct trMsg_CM_ATBand_Set :
  public tun_TxMsg
{
  public:
    trMsg_CM_ATBand_Set( tVoid) :
      enBand( enBand_FM)
    {
      enOpType        = enOpType_SET;
      enFunctionId    = enMsg_CM_ATBand;
      u16DataLen      = 1;
    }

    virtual tVoid vSerialize( tPU8 pu8Data)
    {
      pu8Data[ 0]    = (tU8)enBand;
    }

    virtual tVoid vTrace( tVoid)
    {
      ETG_TRACE_USR1(( " trMsg_CM_ATBand_Set\n"
        "->01 ClientID   : %04x\n"
        "->02 FBlock     : %04x\n"
        "->03 InstanceId : %02x\n"
        "->04 FunctionId : %04x\n"
        "->05 OpType     : %02x\n"
        "->06 %d bytes payload\n"
        "=> Band         : %02x\n"
        ,ETG_CENUM(tun_tenClientId,         enClientId)
        ,ETG_CENUM(tun_tenFBlockId,         enFBlockId)
        ,ETG_CENUM(tun_tenInstanceId,       enInstanceId)
        ,ETG_CENUM(tun_tenAmFmTunerMsg,     enFunctionId)
        ,ETG_CENUM(tun_tenOpType,           enOpType)
        ,u16DataLen
        ,ETG_CENUM(tenBand,  enBand)
        ));
    }

    tenBand   enBand;
};


/*****************************************************************************
 * ATFrequency
 *
 *   ##  ##### #### ####  ####  ##   #   # #### #   #  ##  #   #
 *  #  #   #   #    #   # #    #  #  #   # #    ##  # #  # #   #
 *  #  #   #   #    #   # #    #  #  #   # #    # # # #    #   #
 *  ####   #   ###  ####  ###  #  #  #   # ###  # # # #     # #
 *  #  #   #   #    # #   #    #  #  #   # #    #  ## #      #
 *  #  #   #   #    #  #  #    #  #  #   # #    #  ## #  #   #
 *  #  #   #   #    #   # ####  ## #  ###  #### #   #  ##    #
 *
 * OPType:
 ****************************************************************************/

/*****************************************************************************
 * OPType: Set
 ****************************************************************************/
struct trMsg_ATFrequency_Set :
  public tun_TxMsg
{
  public:
    trMsg_ATFrequency_Set( tU32 u32SetFrequency) :
      u32Frequency( u32SetFrequency)
    {
      enOpType        = enOpType_SET;
      enFunctionId    = enMsg_ATFrequency;
      u16DataLen      = 4;
    }

    virtual tVoid vSerialize( tPU8 pu8Data)
    {
      TUN_DRVADRIF_SET_U32( &pu8Data[ 0], u32Frequency);
    }

    virtual tVoid vTrace( tVoid)
    {
      ETG_TRACE_USR1(( " trMsg_ATFrequency_Set\n"
        "->01 ClientID   : %04x\n"
        "->02 FBlock     : %04x\n"
        "->03 InstanceId : %02x\n"
        "->04 FunctionId : %04x\n"
        "->05 OpType     : %02x\n"
        "->06 %d bytes payload\n"
        "=> Frequency    : %04x\n"
        ,ETG_CENUM(tun_tenClientId,         enClientId)
        ,ETG_CENUM(tun_tenFBlockId,         enFBlockId)
        ,ETG_CENUM(tun_tenInstanceId,       enInstanceId)
        ,ETG_CENUM(tun_tenAmFmTunerMsg,     enFunctionId)
        ,ETG_CENUM(tun_tenOpType,           enOpType)
        ,u16DataLen
        ,u32Frequency
        ));
    }

    tU32   u32Frequency;
};


/*****************************************************************************
 * CM_CalibrationData
 * OPType: Status
 ****************************************************************************/
struct trMsg_CM_CalibrationData_Status:
  public tun_RxMsg
{
  TUN_INSTALL_MESSAGE_CALLBACKLIST;

  virtual tBool bParse(tPCU8 pcu8Data)
  {
    u16CalibrationList[0] = TUN_DRVADRIF_GET_U16( &pcu8Data[0]);
	u16CalibrationList[1] = TUN_DRVADRIF_GET_U16( &pcu8Data[2]);
	u16CalibrationList[2] = TUN_DRVADRIF_GET_U16( &pcu8Data[4]);
	u16CalibrationList[3] = TUN_DRVADRIF_GET_U16( &pcu8Data[6]);
	u16CalibrationList[4] = TUN_DRVADRIF_GET_U16( &pcu8Data[10]);
    return TRUE;
  }

  virtual tVoid vTrace( tVoid)
  {
    ETG_TRACE_USR4(( " trMsg_CM_CalibrationData_Status\n"
      "->01 ClientID   : %04x\n"
      "->02 FBlock     : %04x\n"
      "->03 InstanceId : %02x\n"
      "->04 FunctionId : %04x\n"
      "->05 OpType     : %02x\n"
      "->06 %d bytes payload\n"
      ,ETG_CENUM(tun_tenClientId,         enClientId)
      ,ETG_CENUM(tun_tenFBlockId,         enFBlockId)
      ,ETG_CENUM(tun_tenInstanceId,       enInstanceId)
      ,ETG_CENUM(tun_tenAmFmTunerMsg,     enFunctionId)
      ,ETG_CENUM(tun_tenOpType,           enOpType)
      ,u16DataLen
      ));

	  ETG_TRACE_USR4(( " trMsg_CM_CalibrationData_Status_1\n"
      "=> Calibration FM1 Tuner : %02x\n"
	  "=> Calibration FM2 Tuner : %02x\n"
	  "=> Calibration FM3 Tuner : %02x\n"
	  "=> Calibration FM4 Tuner : %02x\n"
	  "=> Calibration MW2 Tuner : %02x\n"
      ,u16CalibrationList[0]
	  ,u16CalibrationList[1]
	  ,u16CalibrationList[2]
	  ,u16CalibrationList[3]
	  ,u16CalibrationList[4]
      ));
  }
  tU16  u16CalibrationList[5];
};


/*****************************************************************************
 * ATSeek
 *
 *   ##  #####  ### #### #### #   #
 *  #  #   #   #    #    #    #  #
 *  #  #   #   #    #    #    # #
 *  ####   #    ##  ###  ###  ##
 *  #  #   #      # #    #    # #
 *  #  #   #      # #    #    #  #
 *  #  #   #   ###  #### #### #   #
 *
 * Do seek of given type in given direction for given audio tuner
 * OPType: Set, Get, SetGet, Status, Error
 ****************************************************************************/


/*****************************************************************************
 * OPType: Set
 ****************************************************************************/
 struct trMsg_ATSeek_Set :
  public tun_TxMsg
{
  public:

    trMsg_ATSeek_Set( tVoid) :
      enATSeekMode( enATSeekMode_Off)
    {
      enOpType       = enOpType_SET;
      enFunctionId   = enMsg_ATSeek;
      u16DataLen     = 1;
    }

    virtual tVoid vSerialize( tPU8 pu8Data)
    {
      pu8Data[ 0]    = (tU8)enATSeekMode;
    }

    virtual tVoid vTrace( tVoid)
    {
      ETG_TRACE_USR1(( " trMsg_ATSeek_Set\n"
        "->01 ClientID   : %04x\n"
        "->02 FBlock     : %04x\n"
        "->03 InstanceId : %02x\n"
        "->04 FunctionId : %04x\n"
        "->05 OpType     : %02x\n"
        "->06 %d bytes payload\n"
        "=> ATSeekMode   : %02x\n"
        ,ETG_CENUM(tun_tenClientId,         enClientId)
        ,ETG_CENUM(tun_tenFBlockId,         enFBlockId)
        ,ETG_CENUM(tun_tenInstanceId,       enInstanceId)
        ,ETG_CENUM(tun_tenAmFmTunerMsg,     enFunctionId)
        ,ETG_CENUM(tun_tenOpType,           enOpType)
        ,u16DataLen
        ,ETG_CENUM(tenATSeekMode,           enATSeekMode)
        ));
    }

    tenATSeekMode enATSeekMode;
 };

 /*****************************************************************************
 * OPType: Status
 ****************************************************************************/
 struct trMsg_CM_ATSeek_Status:
  public tun_RxMsg
{
  TUN_INSTALL_MESSAGE_CALLBACKLIST;

  virtual tBool bParse(tPCU8 pcu8Data)
  {
    u8SeekMode = pcu8Data[0];
    return TRUE;
  }

  virtual tVoid vTrace( tVoid)
  {
    ETG_TRACE_USR4(( " trMsg_CM_ATSeek_Status\n"
      "->01 ClientID   : %04x\n"
      "->02 FBlock     : %04x\n"
      "->03 InstanceId : %02x\n"
      "->04 FunctionId : %04x\n"
      "->05 OpType     : %02x\n"
      "->06 %d bytes payload\n"
      "=> SeekMode : %02x\n"
      ,ETG_CENUM(tun_tenClientId,         enClientId)
      ,ETG_CENUM(tun_tenFBlockId,         enFBlockId)
      ,ETG_CENUM(tun_tenInstanceId,       enInstanceId)
      ,ETG_CENUM(tun_tenAmFmTunerMsg,     enFunctionId)
      ,ETG_CENUM(tun_tenOpType,           enOpType)
      ,u16DataLen
      ,u8SeekMode
      ));
  }
  tU8 u8SeekMode;
};


 /*****************************************************************************
 * OPType: Set
 ****************************************************************************/
 struct trMsg_CM_ATSeek_Set :
  public tun_TxMsg
{
  public:

    trMsg_CM_ATSeek_Set( tU32 u32SetFrequency = 0) :
      enATSeekMode( enATSeekMode_Off),
      u32Frequency( u32SetFrequency)
    {
      enOpType       = enOpType_SET;
      enFunctionId   = enMsg_CM_ATSeek;
      u16DataLen     = 5;
    }

    virtual tVoid vSerialize( tPU8 pu8Data)
    {
      pu8Data[ 0]    = (tU8)enATSeekMode;
      TUN_DRVADRIF_SET_U32( &pu8Data[ 1], u32Frequency);
    }

    virtual tVoid vTrace( tVoid)
    {
      ETG_TRACE_USR1(( " trMsg_CM_ATSeek_Set\n"
        "->01 ClientID   : %04x\n"
        "->02 FBlock     : %04x\n"
        "->03 InstanceId : %02x\n"
        "->04 FunctionId : %04x\n"
        "->05 OpType     : %02x\n"
        "->06 %d bytes payload\n"
        "=> Frequency    : %d kHz\n"
        "=> ATSeekMode   : %02x\n"
        ,ETG_CENUM(tun_tenClientId,         enClientId)
        ,ETG_CENUM(tun_tenFBlockId,         enFBlockId)
        ,ETG_CENUM(tun_tenInstanceId,       enInstanceId)
        ,ETG_CENUM(tun_tenAmFmTunerMsg,     enFunctionId)
        ,ETG_CENUM(tun_tenOpType,           enOpType)
        ,u16DataLen
        ,u32Frequency
        ,ETG_CENUM(tenATSeekMode,           enATSeekMode)
        ));
    }

    tenATSeekMode enATSeekMode;
    tU32          u32Frequency;
 };


/*****************************************************************************
 * ATPI
 *
 *   ##  ##### ###  ###
 *  #  #   #   #  #  #
 *  #  #   #   #  #  #
 *  ####   #   ###   #
 *  #  #   #   #     #
 *  #  #   #   #     #
 *  #  #   #   #    ###
 *
 * Load station with given PI to given audio tuner
 * OPType: Set, Get, SetGet, Increment, Decrement, Decrement, Error
 ****************************************************************************/

/*****************************************************************************
 * OPType: Increment
 ****************************************************************************/
struct trMsg_ATPI_Increment :
  public tun_TxMsg
{
  public:

    trMsg_ATPI_Increment( tU8 u8SetNsteps = 0x01)
    {
      enOpType       = enOpType_Increment;
      enFunctionId   = enMsg_ATPI;
      u8Nsteps       = u8SetNsteps;
      u16DataLen     = 1;
    }

    virtual tVoid vSerialize( tPU8 pu8Data)
    {
      pu8Data[ 0]    = u8Nsteps;
    }

    virtual tVoid vTrace( tVoid)
    {
      ETG_TRACE_USR1(( " trMsg_ATPI_Increment\n"
        "->01 ClientID   : %04x\n"
        "->02 FBlock     : %04x\n"
        "->03 InstanceId : %02x\n"
        "->04 FunctionId : %04x\n"
        "->05 OpType     : %02x\n"
        "->06 %d bytes payload\n"
        "=>   %02x\n"
        ,ETG_CENUM(tun_tenClientId,         enClientId)
        ,ETG_CENUM(tun_tenFBlockId,         enFBlockId)
        ,ETG_CENUM(tun_tenInstanceId,       enInstanceId)
        ,ETG_CENUM(tun_tenAmFmTunerMsg,     enFunctionId)
        ,ETG_CENUM(tun_tenOpType,           enOpType)
        ,u16DataLen
        ,u8Nsteps
        ));
    }

    tU8 u8Nsteps;
 };


/*****************************************************************************
 * OPType: Decrement
 ****************************************************************************/
struct trMsg_ATPI_Decrement :
  public tun_TxMsg
{
  public:

    trMsg_ATPI_Decrement( tU8 u8SetNsteps = 0x01)
    {
      enOpType       = enOpType_Decrement;
      enFunctionId   = enMsg_ATPI;
      u8Nsteps       = u8SetNsteps;
      u16DataLen     = 1;
    }

    virtual tVoid vSerialize( tPU8 pu8Data)
    {
      pu8Data[ 0]    = u8Nsteps;
    }

    virtual tVoid vTrace( tVoid)
    {
      ETG_TRACE_USR1(( " trMsg_ATPI_Decrement\n"
        "->01 ClientID   : %04x\n"
        "->02 FBlock     : %04x\n"
        "->03 InstanceId : %02x\n"
        "->04 FunctionId : %04x\n"
        "->05 OpType     : %02x\n"
        "->06 %d bytes payload\n"
        "=>   %02x\n"
        ,ETG_CENUM(tun_tenClientId,         enClientId)
        ,ETG_CENUM(tun_tenFBlockId,         enFBlockId)
        ,ETG_CENUM(tun_tenInstanceId,       enInstanceId)
        ,ETG_CENUM(tun_tenAmFmTunerMsg,     enFunctionId)
        ,ETG_CENUM(tun_tenOpType,           enOpType)
        ,u16DataLen
        ,u8Nsteps
        ));
    }


    tU8 u8Nsteps;
 };


/*****************************************************************************
 * CM_AudioTunerStation
 *
 *   ##  #   #       ##  #   # ###  ###  ##  ##### #   # #   # #### ####   ### #####  ##  ##### ###  ##  #   #
 *  #  # #   #      #  # #   # #  #  #  #  #   #   #   # ##  # #    #   # #      #   #  #   #    #  #  # ##  #
 *  #    ## ##      #  # #   # #  #  #  #  #   #   #   # # # # #    #   # #      #   #  #   #    #  #  # # # #
 *  #    # # #      #### #   # #  #  #  #  #   #   #   # # # # ###  ####   ##    #   ####   #    #  #  # # # #
 *  #    #   #      #  # #   # #  #  #  #  #   #   #   # #  ## #    # #      #   #   #  #   #    #  #  # #  ##
 *  #  # #   #      #  # #   # #  #  #  #  #   #   #   # #  ## #    #  #     #   #   #  #   #    #  #  # #  ##
 *   ##  #   # #### #  #  ###  ###  ###  ##    #    ###  #   # #### #   # ###    #   #  #   #   ###  ##  #   #
 *
 * OPType:
 ****************************************************************************/


/*****************************************************************************
 * CM_Autostore
 *
 *   ##  #   #       ##  #   # #####  ##   ### #####  ##  ####  ####
 *  #  # #   #      #  # #   #   #   #  # #      #   #  # #   # #
 *  #    ## ##      #  # #   #   #   #  # #      #   #  # #   # #
 *  #    # # #      #### #   #   #   #  #  ##    #   #  # ####  ###
 *  #    #   #      #  # #   #   #   #  #    #   #   #  # # #   #
 *  #  # #   #      #  # #   #   #   #  #    #   #   #  # #  #  #
 *   ##  #   # #### #  #  ###    #    ##  ###    #    ##  #   # ####
 *
 * OPType:
 ****************************************************************************/


/*****************************************************************************
 * CM_TuningActionConfiguration
 *
 *   ##  #   #      ##### #   # #   # ### #   #  ###   ##   ##  ##### ###  ##  #   #  ##   ##  #   # #### ###  ###  #   # ####   ##  ##### ###  ##  #   #
 *  #  # #   #        #   #   # ##  #  #  ##  # #   # #  # #  #   #    #  #  # ##  # #  # #  # ##  # #     #  #   # #   # #   # #  #   #    #  #  # ##  #
 *  #    ## ##        #   #   # # # #  #  # # # #     #  # #      #    #  #  # # # # #    #  # # # # #     #  #     #   # #   # #  #   #    #  #  # # # #
 *  #    # # #        #   #   # # # #  #  # # # # ### #### #      #    #  #  # # # # #    #  # # # # ###   #  # ### #   # ####  ####   #    #  #  # # # #
 *  #    #   #        #   #   # #  ##  #  #  ## #   # #  # #      #    #  #  # #  ## #    #  # #  ## #     #  #   # #   # # #   #  #   #    #  #  # #  ##
 *  #  # #   #        #   #   # #  ##  #  #  ## #   # #  # #  #   #    #  #  # #  ## #  # #  # #  ## #     #  #   # #   # #  #  #  #   #    #  #  # #  ##
 *   ##  #   # ####   #    ###  #   # ### #   #  ###  #  #  ##    #   ###  ##  #   #  ##   ##  #   # #    ###  ###   ###  #   # #  #   #   ###  ##  #   #
 *
 * OPType:
 ****************************************************************************/


/*****************************************************************************
 * CM_SetActiveArea
 *
 *   ##  #   #       ### #### #####  ##   ##  ##### ### #   # ####  ##  ####  ####  ##
 *  #  # #   #      #    #      #   #  # #  #   #    #  #   # #    #  # #   # #    #  #
 *  #    ## ##      #    #      #   #  # #      #    #  #   # #    #  # #   # #    #  #
 *  #    # # #       ##  ###    #   #### #      #    #  #   # ###  #### ####  ###  ####
 *  #    #   #         # #      #   #  # #      #    #  #   # #    #  # # #   #    #  #
 *  #  # #   #         # #      #   #  # #  #   #    #   # #  #    #  # #  #  #    #  #
 *   ##  #   # #### ###  ####   #   #  #  ##    #   ###   #   #### #  # #   # #### #  #
 *
 * OPType:
 ****************************************************************************/


/*****************************************************************************
 * CM_GetActiveArea
 *
 *   ##  #   #       ###  #### #####  ##   ##  ##### ### #   # ####  ##  ####  ####  ##
 *  #  # #   #      #   # #      #   #  # #  #   #    #  #   # #    #  # #   # #    #  #
 *  #    ## ##      #     #      #   #  # #      #    #  #   # #    #  # #   # #    #  #
 *  #    # # #      # ### ###    #   #### #      #    #  #   # ###  #### ####  ###  ####
 *  #    #   #      #   # #      #   #  # #      #    #  #   # #    #  # # #   #    #  #
 *  #  # #   #      #   # #      #   #  # #  #   #    #   # #  #    #  # #  #  #    #  #
 *   ##  #   # ####  ###  ####   #   #  #  ##    #   ###   #   #### #  # #   # #### #  #
 *
 * OPType:
 ****************************************************************************/


/*****************************************************************************
 * CM_AreaConfiguration
 *
 *   ##  #   #       ##  ####  ####  ##   ##   ##  #   # #### ###  ###  #   # ####   ##  ##### ###  ##  #   #
 *  #  # #   #      #  # #   # #    #  # #  # #  # ##  # #     #  #   # #   # #   # #  #   #    #  #  # ##  #
 *  #    ## ##      #  # #   # #    #  # #    #  # # # # #     #  #     #   # #   # #  #   #    #  #  # # # #
 *  #    # # #      #### ####  ###  #### #    #  # # # # ###   #  # ### #   # ####  ####   #    #  #  # # # #
 *  #    #   #      #  # # #   #    #  # #    #  # #  ## #     #  #   # #   # # #   #  #   #    #  #  # #  ##
 *  #  # #   #      #  # #  #  #    #  # #  # #  # #  ## #     #  #   # #   # #  #  #  #   #    #  #  # #  ##
 *   ##  #   # #### #  # #   # #### #  #  ##   ##  #   # #    ###  ###   ###  #   # #  #   #   ###  ##  #   #
 *
 * OPType:
 ****************************************************************************/


/*****************************************************************************
 * CM_BandConfiguration
 *
 *   ##  #   #      ###   ##  #   # ###   ##   ##  #   # #### ###  ###  #   # ####   ##  ##### ###  ##  #   #
 *  #  # #   #      #  # #  # ##  # #  # #  # #  # ##  # #     #  #   # #   # #   # #  #   #    #  #  # ##  #
 *  #    ## ##      #  # #  # # # # #  # #    #  # # # # #     #  #     #   # #   # #  #   #    #  #  # # # #
 *  #    # # #      ###  #### # # # #  # #    #  # # # # ###   #  # ### #   # ####  ####   #    #  #  # # # #
 *  #    #   #      #  # #  # #  ## #  # #    #  # #  ## #     #  #   # #   # # #   #  #   #    #  #  # #  ##
 *  #  # #   #      #  # #  # #  ## #  # #  # #  # #  ## #     #  #   # #   # #  #  #  #   #    #  #  # #  ##
 *   ##  #   # #### ###  #  # #   # ###   ##   ##  #   # #    ###  ###   ###  #   # #  #   #   ###  ##  #   #
 *
 * OPType:
 ****************************************************************************/


/*****************************************************************************
 * CM_DefaultSetting
 *
 *   ##  #   #      ###  #### ####  ##  #   # #    #####  ### #### ##### ##### ### #   #  ###
 *  #  # #   #      #  # #    #    #  # #   # #      #   #    #      #     #    #  ##  # #   #
 *  #    ## ##      #  # #    #    #  # #   # #      #   #    #      #     #    #  # # # #
 *  #    # # #      #  # ###  ###  #### #   # #      #    ##  ###    #     #    #  # # # # ###
 *  #    #   #      #  # #    #    #  # #   # #      #      # #      #     #    #  #  ## #   #
 *  #  # #   #      #  # #    #    #  # #   # #      #      # #      #     #    #  #  ## #   #
 *   ##  #   # #### ###  #### #    #  #  ###  ####   #   ###  ####   #     #   ### #   #  ###
 *
 * OPType:
 ****************************************************************************/


/*****************************************************************************
 * CM_AcceptAf
 *
 *   ##  #   #       ##   ##   ##  #### ###  #####  ##  ####
 *  #  # #   #      #  # #  # #  # #    #  #   #   #  # #
 *  #    ## ##      #  # #    #    #    #  #   #   #  # #
 *  #    # # #      #### #    #    ###  ###    #   #### ###
 *  #    #   #      #  # #    #    #    #      #   #  # #
 *  #  # #   #      #  # #  # #  # #    #      #   #  # #
 *   ##  #   # #### #  #  ##   ##  #### #      #   #  # #
 *
 * OPType:
 ****************************************************************************/


/*****************************************************************************
 * CM_AcceptAf
 *
 *   ##  #   #       ##   ##   ##  #### ###  #####  ##  ####
 *  #  # #   #      #  # #  # #  # #    #  #   #   #  # #
 *  #    ## ##      #  # #    #    #    #  #   #   #  # #
 *  #    # # #      #### #    #    ###  ###    #   #### ###
 *  #    #   #      #  # #    #    #    #      #   #  # #
 *  #  # #   #      #  # #  # #  # #    #      #   #  # #
 *   ##  #   # #### #  #  ##   ##  #### #      #   #  # #
 *
 * OPType:
 ****************************************************************************/


/*****************************************************************************
 * ATStationInfo
 *
 *   ##  #####  ### #####  ##  ##### ###  ##  #   # ### #   # ####  ##
 *  #  #   #   #      #   #  #   #    #  #  # ##  #  #  ##  # #    #  #
 *  #  #   #   #      #   #  #   #    #  #  # # # #  #  # # # #    #  #
 *  ####   #    ##    #   ####   #    #  #  # # # #  #  # # # ###  #  #
 *  #  #   #      #   #   #  #   #    #  #  # #  ##  #  #  ## #    #  #
 *  #  #   #      #   #   #  #   #    #  #  # #  ##  #  #  ## #    #  #
 *  #  #   #   ###    #   #  #   #   ###  ##  #   # ### #   # #     ##
 *
 * Provides station data for given audio tuner
 * OPType: Get, Status, Error
 ****************************************************************************/


/*============================================================================
 *
 *--------------------------------------------------------------------------*/
struct trPos
{
  tU8  u8X;
  tU8  u8Y;
};
/*==========================================================================*/

/*****************************************************************************
 * OPType: Status
 ****************************************************************************/
struct trMsg_ATStationInfo_Status  :
  public tun_RxMsg
{
  TUN_INSTALL_MESSAGE_CALLBACKLIST;

  virtual tBool bParse(tPCU8 pcu8Data)
  {
    tBool bRet = FALSE;

    rPos.u8X = pcu8Data[ 0];
    rPos.u8Y = pcu8Data[ 1];

    tenATStationInfo_Data enData = (tenATStationInfo_Data)rPos.u8X;

    switch( enData)
    {
      /* ------------------------------------------------------------------ */

      case enATStationInfo_Data_AllData:
        if( u16DataLen == 20)
        {
          OSAL_pvMemoryCopy( sStationname, &pcu8Data[ 2], 8);
          enStationnameInfo       = (tenStationnameInfo)pcu8Data[ 10];
          u32Frequency            = TUN_DRVADRIF_GET_U32( &pcu8Data[ 11]);
          u8PTY                   = pcu8Data[ 15];
          u16PI                   = TUN_DRVADRIF_GET_U16( &pcu8Data[ 16]);
          u8InfoSwitches          = pcu8Data[ 18];
          u8ReceptionQuality      = pcu8Data[ 19];
          bRet = TRUE;
        }
        break;

      /* ------------------------------------------------------------------ */

      case enATStationInfo_Data_Stationname:
        break;

      /* ------------------------------------------------------------------ */

      case enATStationInfo_Data_StationnameInfo:
        break;

      /* ------------------------------------------------------------------ */

      case enATStationInfo_Data_Frequency:
        break;

      /* ------------------------------------------------------------------ */

      case enATStationInfo_Data_PTY:
        break;

      /* ------------------------------------------------------------------ */

      case enATStationInfo_Data_PI:
        break;

      /* ------------------------------------------------------------------ */

      case enATStationInfo_Data_ATStationInfoSwitches:
        break;

      /* ------------------------------------------------------------------ */

      case enATStationInfo_Data_ReceptionQuality:
        break;

      /* ------------------------------------------------------------------ */

      default:
        break;

      /* ------------------------------------------------------------------ */

    }

    return bRet;
  }

  virtual tVoid vTrace( tVoid)
  {
    ETG_TRACE_USR1(( " trMsg_ATStationInfo_Status\n"
      //"->01 ClientID   : %04x\n"
      //"->02 FBlock     : %04x\n"
      //"->03 InstanceId : %02x\n"
      //"->04 FunctionId : %04x\n"
      //"->05 OpType     : %02x\n"
      "->06 %d bytes payload\n"
      "=>   Data                  : %d\n"
      "=>   Stationname           : %8s\n"
      "=>   StationnameInfo       : %d\n"
      "=>   Frequency             : %d kHz\n"
      "=>   PTY                   : %d\n"
      "=>   PI                    : %02x\n"
      "=>   InfoSwitches          : %02x\n"
      "=>   ReceptionQuality      : %d dB/uV\n"
//      ,ETG_CENUM(tun_tenClientId,         enClientId)
//      ,ETG_CENUM(tun_tenFBlockId,         enFBlockId)
//      ,ETG_CENUM(tun_tenInstanceId,       enInstanceId)
//      ,ETG_CENUM(tun_tenAmFmTunerMsg,     enFunctionId)
//      ,ETG_CENUM(tun_tenOpType,           enOpType)
      ,u16DataLen
      ,ETG_CENUM(tenATStationInfo_Data,   rPos.u8X)
      ,sStationname
      ,ETG_CENUM(tenStationnameInfo,      enStationnameInfo)
      ,u32Frequency
      ,u8PTY
      ,u16PI
      ,u8InfoSwitches
      ,u8ReceptionQuality
      ));
  }

  trPos                    rPos;
  tChar                    sStationname[ 8];
  tenStationnameInfo       enStationnameInfo;
  tU32                     u32Frequency;
  tU8                      u8PTY;
  tU16                     u16PI;
  tU8                      u8InfoSwitches;
  tU8                      u8ReceptionQuality;
};


/*****************************************************************************
 * CM_ATStationInfo
 *
 *   ##  #   #       ##  #####  ### #####  ##  ##### ###  ##  #   # ### #   # ####  ##
 *  #  # #   #      #  #   #   #      #   #  #   #    #  #  # ##  #  #  ##  # #    #  #
 *  #    ## ##      #  #   #   #      #   #  #   #    #  #  # # # #  #  # # # #    #  #
 *  #    # # #      ####   #    ##    #   ####   #    #  #  # # # #  #  # # # ###  #  #
 *  #    #   #      #  #   #      #   #   #  #   #    #  #  # #  ##  #  #  ## #    #  #
 *  #  # #   #      #  #   #      #   #   #  #   #    #  #  # #  ##  #  #  ## #    #  #
 *   ##  #   # #### #  #   #   ###    #   #  #   #   ###  ##  #   # ### #   # #     ##
 *
 * OPType: Get, Status, Error
 ****************************************************************************/


/*****************************************************************************
 * OPType: Status
 ****************************************************************************/
struct trMsg_CM_ATStationInfo_Status  :
  public tun_RxMsg
{
  TUN_INSTALL_MESSAGE_CALLBACKLIST;

  virtual tBool bParse(tPCU8 pcu8Data)
  {                                                                       // index | length
    OSAL_pvMemoryCopy( sStationname, &pcu8Data[ 0], 8);                   //     0 |      8
    enStationnameInfo       = (tenCM_ATStationnameInfo)pcu8Data[ 8];      //     8 |      1
    u32Frequency            = TUN_DRVADRIF_GET_U32( &pcu8Data[ 9]);       //     9 |      4
    u8PTY                   = pcu8Data[ 13];                              //    13 |      1
    u16PI                   = TUN_DRVADRIF_GET_U16( &pcu8Data[ 14]);      //    14 |      2
    enATStationTpInfo       = (tenATStationTpInfo)pcu8Data[ 16];          //    16 |      1
    enATStationTmcInfo	    = (tenATStationTmcInfo)pcu8Data[ 17];         //    17 |      1
    u8Fieldstrength   	    = pcu8Data[ 18];                              //    18 |      1
    u8Quality          	    = pcu8Data[ 19];                              //    19 |      1
    enBand          	      = (tenBand)pcu8Data[ 20];                     //    20 |      1
    u16ActivePresetList     = TUN_DRVADRIF_GET_U16( &pcu8Data[ 21]);      //    21 |      2
    u8ActiveStationList     = pcu8Data[ 23];                              //    23 |      1
    u16PresetAutocompare    = TUN_DRVADRIF_GET_U16( &pcu8Data[ 24]);      //    24 |      2
    u16StationListAutocompare = TUN_DRVADRIF_GET_U16( &pcu8Data[ 26]);    //    26 |      2
    u8ATStationInfo         = pcu8Data[ 28];                              //    28 |      1
                                                                          //----------------
                                                                          //    29 bytes
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
    b8HDStationInfoBits			= pcu8Data[ enCM_ATStationInfo_idx_HDStationInfoBits - enAdrMsgOffset_HEADER_LEN];
    u32StationInfoHDStationID	= TUN_DRVADRIF_GET_U32(&pcu8Data[ enCM_ATStationInfo_idx_HDStationId - enAdrMsgOffset_HEADER_LEN]);
	OSAL_pvMemoryCopy( sHDCallSign, &pcu8Data[ enCM_ATStationInfo_idx_HDCallSign - enAdrMsgOffset_HEADER_LEN], 16);
    e8HDCallSignTextEncoding	= pcu8Data[ enCM_ATStationInfo_idx_HDCallsignTextEncoding - enAdrMsgOffset_HEADER_LEN];
    u8HDStationGenre			= pcu8Data[ enCM_ATStationInfo_idx_HDStationGenre - enAdrMsgOffset_HEADER_LEN];
    b8HDAudioPrgmAvail			= pcu8Data[ enCM_ATStationInfo_idx_HDNumberOfAudioPrograms - enAdrMsgOffset_HEADER_LEN];
    u8HDQuality					= pcu8Data[ enCM_ATStationInfo_idx_HDQuality - enAdrMsgOffset_HEADER_LEN];
    e8ActiveHDAudPrgm			= pcu8Data[ enCM_ATStationInfo_idx_HDActiveAudioProgram - enAdrMsgOffset_HEADER_LEN];

    b8DRMStationInfoBits = pcu8Data[enCM_ATStationInfo_idx_DRMStationInfoBits - enAdrMsgOffset_HEADER_LEN];
    u32DRMServiceId = TUN_DRVADRIF_GET_U32(&pcu8Data[enCM_ATStationInfo_idx_DRMServiceId - enAdrMsgOffset_HEADER_LEN]);
    OSAL_pvMemoryCopy(sDRMServiceLabel, &pcu8Data[enCM_ATStationInfo_idx_DRMServiceLabel - enAdrMsgOffset_HEADER_LEN], 64);
    u8DRMPTY = pcu8Data[enCM_ATStationInfo_idx_DRMPTY - enAdrMsgOffset_HEADER_LEN];
    b16DRMAnnouBits = TUN_DRVADRIF_GET_U16(&pcu8Data[ enCM_ATStationInfo_idx_DRMAnnoSupport - enAdrMsgOffset_HEADER_LEN]);
    b8DRMNumberOfAudioServices = pcu8Data[enCM_ATStationInfo_idx_DRMNumberOfAudioServices - enAdrMsgOffset_HEADER_LEN];
    u8DRMQuality = pcu8Data[enCM_ATStationInfo_idx_DRMQuality - enAdrMsgOffset_HEADER_LEN];
    u8DRMServiceShortId = pcu8Data[enCM_ATStationInfo_idx_DRMServiceShortId - enAdrMsgOffset_HEADER_LEN];
    u8Drmlanguage = pcu8Data[enCM_ATStationInfo_idx_Drmlanguage - enAdrMsgOffset_HEADER_LEN];
    u32DrmlanguageCode = TUN_DRVADRIF_GET_U24(&pcu8Data[enCM_ATStationInfo_idx_DrmlanguageCode - enAdrMsgOffset_HEADER_LEN]);
    u16DrmCountryCode = TUN_DRVADRIF_GET_U16(&pcu8Data[enCM_ATStationInfo_idx_DrmCountryCode - enAdrMsgOffset_HEADER_LEN]);
    s8DrmFrequencyCorrection = pcu8Data[enCM_ATStationInfo_idx_DrmFrequencyCorrection - enAdrMsgOffset_HEADER_LEN];
#endif
    return TRUE;
  }

  virtual tVoid vTrace( tVoid)
  {
    ETG_TRACE_USR1(( " trMsg_CM_ATStationInfo_Status -> Part1\n"
      "->01 ClientID   : %04x\n"
      "->02 FBlock     : %04x\n"
      "->03 InstanceId : %02x\n"
//      "->04 FunctionId : %04x\n"
      "->05 OpType     : %02x\n"
      "->06 %d bytes payload\n"
      "=>   Stationname           : %8s\n"
      "=>   StationnameInfo       : %d\n"
      "=>   Frequency             : %u kHz\n"
      "=>   PTY                   : %d\n"
      ,ETG_CENUM(tun_tenClientId,         enClientId)
      ,ETG_CENUM(tun_tenFBlockId,         enFBlockId)
      ,ETG_CENUM(tun_tenInstanceId,       enInstanceId)
//      ,ETG_CENUM(tun_tenAmFmTunerMsg,     enFunctionId)
      ,ETG_CENUM(tun_tenOpType,           enOpType)
      ,u16DataLen
      ,sStationname
      ,ETG_CENUM(tenCM_ATStationnameInfo, enStationnameInfo)
      ,u32Frequency
      ,u8PTY
      ));
    ETG_TRACE_USR1(( " trMsg_CM_ATStationInfo_Status -> Part2\n"
      "=>   PI                    : %02x\n"
      "=>   ATStationTpInfo       : %02x\n"
      "=>   ATStationTmcInfo      : %02x\n"
      "=>   Fieldstrength         : %d dB/uV\n"
      "=>   u8Quality             : %02x\n"
      "=>   Band                  : %02x\n"
      "=>   ActivePresetList      : %04x\n"
      "=>   PresetAutocompare     : %04x\n"
      "=>   StationListAutocompare: %04x\n"
      "=>   ATStationInfo         : %02x\n"
      ,u16PI
      ,ETG_CENUM(tenATStationTpInfo,      enATStationTpInfo)
      ,ETG_CENUM(tenATStationTmcInfo,     enATStationTmcInfo)
      ,u8Fieldstrength
      ,u8Quality
      ,ETG_CENUM(tenBand,                 enBand)
      ,u16ActivePresetList
      ,u16PresetAutocompare
      ,u16StationListAutocompare
      ,u8ATStationInfo
      ));
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
	 ETG_TRACE_USR1(( " trMsg_CM_ATStationInfo_Status -> Part3 -> HD Fields \n"
      "=>   b8HDStationInfoBits         : %02x\n"
      "=>   u32StationInfoHDStationID   : %08x\n"
	  "=>   sHDCallSign                 : %16s\n"
      "=>   e8HDCallSignTextEncoding    : %02x\n"
      "=>   u8HDStationGenre            : %02x\n"
	  "=>   b8HDAudioPrgmAvail          : %02x\n"
      "=>   u8HDQuality                 : %02x\n"
      "=>   e8ActiveHDAudPrgm           : %02x\n"
      ,b8HDStationInfoBits
      ,u32StationInfoHDStationID
      ,sHDCallSign
      ,e8HDCallSignTextEncoding
      ,u8HDStationGenre
      ,b8HDAudioPrgmAvail
      ,u8HDQuality
      ,e8ActiveHDAudPrgm
      ));


	 ETG_TRACE_USR1(( " trMsg_CM_ATStationInfo_Status -> Part4 -> DRM Fields \n"
	  "=>   b8DRMStationInfoBits        : %02x\n"
	  "=>   u32DRMServiceId   			: %08x\n"
	  "=>   sDRMServiceLabel            : %64s\n"
	  "=>   u8DRMPTY    				: %02x\n"
	  "=>   b16DRMAnnouBits            	: %04x\n"
	  "=>   b8DRMNumberOfAudioServices  : %02x\n"
	  "=>   u8DRMQuality                : %02x\n"
	  "=>   u8DRMServiceShortId         : %02x\n"
	  ,b8DRMStationInfoBits
	  ,u32DRMServiceId
	  ,sDRMServiceLabel
	  ,u8DRMPTY
	  ,b16DRMAnnouBits
	  ,b8DRMNumberOfAudioServices
	  ,u8DRMQuality
	  ,u8DRMServiceShortId
	  ));

	 ETG_TRACE_USR1(( " trMsg_CM_ATStationInfo_Status -> Part4 -> DRM Fields \n"
	  "=>   u8Drmlanguage         		: %02d\n"
	  "=>   u32DrmlanguageCode   		: %08d\n"
	  "=>   u16DrmCountryCode           : %04d\n"
	  "=>   s8DrmFrequencyCorrection    : %02i\n"
	  ,u8Drmlanguage
	  ,u32DrmlanguageCode
	  ,u16DrmCountryCode
	  ,s8DrmFrequencyCorrection
	  ));
#endif
  }

  tChar                    sStationname[ 8];
  tenCM_ATStationnameInfo  enStationnameInfo;
  tU32                     u32Frequency;
  tU8                      u8PTY;
  tU16                     u16PI;
  tenATStationTpInfo       enATStationTpInfo;
  tenATStationTmcInfo      enATStationTmcInfo;
  tU8                      u8Fieldstrength;
  tU8                      u8Quality;
  tenBand                  enBand;
  tU16                     u16ActivePresetList;
  tU8                      u8ActiveStationList;
  tU16                     u16PresetAutocompare;
  tU16                     u16StationListAutocompare;
  tU8                      u8ATStationInfo;

#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
   tU8 b8HDStationInfoBits;
   tU32 u32StationInfoHDStationID;
   tChar sHDCallSign[16];
   tU8 e8HDCallSignTextEncoding;
   tU8 u8HDStationGenre;
   tU8 b8HDAudioPrgmAvail;
   tU8 u8HDQuality;
   tU8 e8ActiveHDAudPrgm;

   tU8 b8DRMStationInfoBits;
   tU32 u32DRMServiceId;
   tChar sDRMServiceLabel[64];
   tU8 u8DRMPTY;
   tU16 b16DRMAnnouBits;
   tU8 b8DRMNumberOfAudioServices;
   tU8 u8DRMQuality;
   tU8 u8DRMServiceShortId;
   tU8 u8Drmlanguage;
   tU32 u32DrmlanguageCode;
   tU16 u16DrmCountryCode;
   tS8 s8DrmFrequencyCorrection;
#endif
};



/*****************************************************************************
 * CM_ATStationInfoConfig
 *
 *   ##  #   #       ##  #####  ### #####  ##  ##### ###  ##  #   # ### #   # ####  ##   ##   ##  #   # #### ###  ###
 *  #  # #   #      #  #   #   #      #   #  #   #    #  #  # ##  #  #  ##  # #    #  # #  # #  # ##  # #     #  #   #
 *  #    ## ##      #  #   #   #      #   #  #   #    #  #  # # # #  #  # # # #    #  # #    #  # # # # #     #  #
 *  #    # # #      ####   #    ##    #   ####   #    #  #  # # # #  #  # # # ###  #  # #    #  # # # # ###   #  # ###
 *  #    #   #      #  #   #      #   #   #  #   #    #  #  # #  ##  #  #  ## #    #  # #    #  # #  ## #     #  #   #
 *  #  # #   #      #  #   #      #   #   #  #   #    #  #  # #  ##  #  #  ## #    #  # #  # #  # #  ## #     #  #   #
 *   ##  #   # #### #  #   #   ###    #   #  #   #   ###  ##  #   # ### #   # #     ##   ##   ##  #   # #    ###  ###
 *
 * OPType:
 ****************************************************************************/


/*****************************************************************************
 * CM_ATTunerState
 *
 *   ##  #   #       ##  ##### ##### #   # #   # #### ####   ### #####  ##  ##### ####
 *  #  # #   #      #  #   #     #   #   # ##  # #    #   # #      #   #  #   #   #
 *  #    ## ##      #  #   #     #   #   # # # # #    #   # #      #   #  #   #   #
 *  #    # # #      ####   #     #   #   # # # # ###  ####   ##    #   ####   #   ###
 *  #    #   #      #  #   #     #   #   # #  ## #    # #      #   #   #  #   #   #
 *  #  # #   #      #  #   #     #   #   # #  ## #    #  #     #   #   #  #   #   #
 *   ##  #   # #### #  #   #     #    ###  #   # #### #   # ###    #   #  #   #   ####
 *
 * OPType: Status
 ****************************************************************************/
struct trMsg_CM_ATTunerState_Status  :
  public tun_RxMsg
{
  TUN_INSTALL_MESSAGE_CALLBACKLIST;

  virtual tBool bParse(tPCU8 pcu8Data)
  {
    u8Band = pcu8Data[ 0];
	u8TunerState = pcu8Data[1];
    return TRUE;
  }

  virtual tVoid vTrace( tVoid)
  {
    ETG_TRACE_USR4(( " trMsg_CM_ATTunerState_Status\n"
      "->01 ClientID   : %04x\n"
      "->02 FBlock     : %04x\n"
      "->03 InstanceId : %02x\n"
      "->04 FunctionId : %04x\n"
      "->05 OpType     : %02x\n"
      "->06 %d bytes payload\n"
      "=> Band : %02x\n"
	  "=> TunerState : %02x\n"
      ,ETG_CENUM(tun_tenClientId,         enClientId)
      ,ETG_CENUM(tun_tenFBlockId,         enFBlockId)
      ,ETG_CENUM(tun_tenInstanceId,       enInstanceId)
      ,ETG_CENUM(tun_tenAmFmTunerMsg,     enFunctionId)
      ,ETG_CENUM(tun_tenOpType,           enOpType)
      ,u16DataLen
      ,u8Band
	  ,u8TunerState
      ));
  }
  tU8  u8Band;
  tU8 u8TunerState;
};


/*****************************************************************************
 * TPSwitch
 *
 *  ##### ###   ### #   # ### #####  ##  #  #
 *    #   #  # #    #   #  #    #   #  # #  #
 *    #   #  # #    #   #  #    #   #    #  #
 *    #   ###   ##  #   #  #    #   #    ####
 *    #   #       # # # #  #    #   #    #  #
 *    #   #       # # # #  #    #   #  # #  #
 *    #   #    ###   # #  ###   #    ##  #  #
 *
 * OPType: Status
 ****************************************************************************/
struct trMsg_CM_TPSwitch_Status  :
  public tun_RxMsg
{
  TUN_INSTALL_MESSAGE_CALLBACKLIST;

  virtual tBool bParse(tPCU8 pcu8Data)
  {
    bTPMode = pcu8Data[ 0];
    return TRUE;
  }

  virtual tVoid vTrace( tVoid)
  {
    ETG_TRACE_USR4(( " trMsg_CM_TPSwitch_Status\n"
      "->01 ClientID   : %04x\n"
      "->02 FBlock     : %04x\n"
      "->03 InstanceId : %02x\n"
      "->04 FunctionId : %04x\n"
      "->05 OpType     : %02x\n"
      "->06 %d bytes payload\n"
      "=> TPMode : %02x\n"
      ,ETG_CENUM(tun_tenClientId,         enClientId)
      ,ETG_CENUM(tun_tenFBlockId,         enFBlockId)
      ,ETG_CENUM(tun_tenInstanceId,       enInstanceId)
      ,ETG_CENUM(tun_tenAmFmTunerMsg,     enFunctionId)
      ,ETG_CENUM(tun_tenOpType,           enOpType)
      ,u16DataLen
      ,bTPMode
      ));
  }
  tBool  bTPMode;
};



/*****************************************************************************
 * AFBestSwitch
 *
 *   ##  #### ###  ####  ### #####  ### #   # ### #####  ##  #  #
 *  #  # #    #  # #    #      #   #    #   #  #    #   #  # #  #
 *  #  # #    #  # #    #      #   #    #   #  #    #   #    #  #
 *  #### ###  ###  ###   ##    #    ##  #   #  #    #   #    ####
 *  #  # #    #  # #       #   #      # # # #  #    #   #    #  #
 *  #  # #    #  # #       #   #      # # # #  #    #   #  # #  #
 *  #  # #    ###  #### ###    #   ###   # #  ###   #    ##  #  #
 *
 * OPType: Status
 ****************************************************************************/
struct trMsg_CM_AFBestSwitch_Status  :
  public tun_RxMsg
{
  TUN_INSTALL_MESSAGE_CALLBACKLIST;

  virtual tBool bParse(tPCU8 pcu8Data)
  {
    bAFBestSwitch = pcu8Data[ 0];
    return TRUE;
  }

  virtual tVoid vTrace( tVoid)
  {
    ETG_TRACE_USR4(( " trMsg_CM_AFBestSwitch_Status\n"
      "->01 ClientID   : %04x\n"
      "->02 FBlock     : %04x\n"
      "->03 InstanceId : %02x\n"
      "->04 FunctionId : %04x\n"
      "->05 OpType     : %02x\n"
      "->06 %d bytes payload\n"
      "=> AFBestSwitch : %02x\n"
      ,ETG_CENUM(tun_tenClientId,         enClientId)
      ,ETG_CENUM(tun_tenFBlockId,         enFBlockId)
      ,ETG_CENUM(tun_tenInstanceId,       enInstanceId)
      ,ETG_CENUM(tun_tenAmFmTunerMsg,     enFunctionId)
      ,ETG_CENUM(tun_tenOpType,           enOpType)
      ,u16DataLen
      ,bAFBestSwitch
      ));
  }
  tBool  bAFBestSwitch;
};

/*****************************************************************************
 * RDSSwitch
 *
 *  ####  ###   ###  ### #   # ### #####  ##  #  #
 *  #   # #  # #    #    #   #  #    #   #  # #  #
 *  #   # #  # #    #    #   #  #    #   #    #  #
 *  ####  #  #  ##   ##  #   #  #    #   #    ####
 *  # #   #  #    #    # # # #  #    #   #    #  #
 *  #  #  #  #    #    # # # #  #    #   #  # #  #
 *  #   # ###  ###  ###   # #  ###   #    ##  #  #
 *
 * OPType:
 ****************************************************************************/
struct trMsg_RDSSwitch_Status  :
  public tun_RxMsg
{
  TUN_INSTALL_MESSAGE_CALLBACKLIST;

  virtual tBool bParse(tPCU8 pcu8Data)
  {
    bRDSSwitch = pcu8Data[ 0];
    return TRUE;
  }

  virtual tVoid vTrace( tVoid)
  {
    ETG_TRACE_USR4(( " trMsg_RDSSwitch_Status\n"
      "->01 ClientID   : %04x\n"
      "->02 FBlock     : %04x\n"
      "->03 InstanceId : %02x\n"
      "->04 FunctionId : %04x\n"
      "->05 OpType     : %02x\n"
      "->06 %d bytes payload\n"
      "=> RDSSwitch : %02x\n"
      ,ETG_CENUM(tun_tenClientId,         enClientId)
      ,ETG_CENUM(tun_tenFBlockId,         enFBlockId)
      ,ETG_CENUM(tun_tenInstanceId,       enInstanceId)
      ,ETG_CENUM(tun_tenAmFmTunerMsg,     enFunctionId)
      ,ETG_CENUM(tun_tenOpType,           enOpType)
      ,u16DataLen
      ,bRDSSwitch
      ));
  }
  tBool  bRDSSwitch;
};


/*****************************************************************************
 * CM_StationNameFix
 *
 *   ##  #   #       ### #####  ##  ##### ###  ##  #   # #   #  ##  #   # #### #### ### #   #
 *  #  # #   #      #      #   #  #   #    #  #  # ##  # ##  # #  # #   # #    #     #  #   #
 *  #    ## ##      #      #   #  #   #    #  #  # # # # # # # #  # ## ## #    #     #   # #
 *  #    # # #       ##    #   ####   #    #  #  # # # # # # # #### # # # ###  ###   #    #
 *  #    #   #         #   #   #  #   #    #  #  # #  ## #  ## #  # #   # #    #     #   # #
 *  #  # #   #         #   #   #  #   #    #  #  # #  ## #  ## #  # #   # #    #     #  #   #
 *   ##  #   # #### ###    #   #  #   #   ###  ##  #   # #   # #  # #   # #### #    ### #   #
 *
 * OPType:
 ****************************************************************************/


/*****************************************************************************
 * CM_MonoStereo
 *
 *   ##  #   #      #   #  ##  #   #  ##   ### ##### #### ####  ####  ##
 *  #  # #   #      #   # #  # ##  # #  # #      #   #    #   # #    #  #
 *  #    ## ##      ## ## #  # # # # #  # #      #   #    #   # #    #  #
 *  #    # # #      # # # #  # # # # #  #  ##    #   ###  ####  ###  #  #
 *  #    #   #      #   # #  # #  ## #  #    #   #   #    # #   #    #  #
 *  #  # #   #      #   # #  # #  ## #  #    #   #   #    #  #  #    #  #
 *   ##  #   # #### #   #  ##  #   #  ##  ###    #   #### #   # ####  ##
 *
 * OPType:
 ****************************************************************************/


/*****************************************************************************
 * REG
 *
 *  ####  ####  ###
 *  #   # #    #   #
 *  #   # #    #
 *  ####  ###  # ###
 *  # #   #    #   #
 *  #  #  #    #   #
 *  #   # ####  ###
 *
 * OPType: Status
 ****************************************************************************/
struct trMsg_CM_REG_Status  :
  public tun_RxMsg
{
  TUN_INSTALL_MESSAGE_CALLBACKLIST;

  virtual tBool bParse(tPCU8 pcu8Data)
  {
    u8REGMode = pcu8Data[ 0];
    return TRUE;
  }

  virtual tVoid vTrace( tVoid)
  {
    ETG_TRACE_USR4(( " trMsg_CM_REG_Status\n"
      "->01 ClientID   : %04x\n"
      "->02 FBlock     : %04x\n"
      "->03 InstanceId : %02x\n"
      "->04 FunctionId : %04x\n"
      "->05 OpType     : %02x\n"
      "->06 %d bytes payload\n"
      "=> u8REGMode : %02x\n"
      ,ETG_CENUM(tun_tenClientId,         enClientId)
      ,ETG_CENUM(tun_tenFBlockId,         enFBlockId)
      ,ETG_CENUM(tun_tenInstanceId,       enInstanceId)
      ,ETG_CENUM(tun_tenAmFmTunerMsg,     enFunctionId)
      ,ETG_CENUM(tun_tenOpType,           enOpType)
      ,u16DataLen
      ,u8REGMode
      ));
  }
  tU8  u8REGMode;
};

/*****************************************************************************
 * Announcement Switch
 * OPType: Status
 ****************************************************************************/
struct trMsg_CM_AnnouncementSwitch_Status  :
  public tun_RxMsg
{
  TUN_INSTALL_MESSAGE_CALLBACKLIST;

  virtual tBool bParse(tPCU8 pcu8Data)
  {
    u8AnnouncementSwitch = pcu8Data[ 0];
    return TRUE;
  }

  virtual tVoid vTrace( tVoid)
  {
    ETG_TRACE_USR4(( " trMsg_CM_AnnouncementSwitch_Status\n"
      "->01 ClientID   : %04x\n"
      "->02 FBlock     : %04x\n"
      "->03 InstanceId : %02x\n"
      "->04 FunctionId : %04x\n"
      "->05 OpType     : %02x\n"
      "->06 %d bytes payload\n"
      "=> u8AnnouncementSwitch : %02x\n"
      ,ETG_CENUM(tun_tenClientId,         enClientId)
      ,ETG_CENUM(tun_tenFBlockId,         enFBlockId)
      ,ETG_CENUM(tun_tenInstanceId,       enInstanceId)
      ,ETG_CENUM(tun_tenAmFmTunerMsg,     enFunctionId)
      ,ETG_CENUM(tun_tenOpType,           enOpType)
      ,u16DataLen
      ,u8AnnouncementSwitch
      ));
  }
  tU8  u8AnnouncementSwitch;
};


/*****************************************************************************
 * CM_DDASwitch
 *
 *   ##  #   #      ###  ###   ##   ### #   # ### #####  ##  #  #
 *  #  # #   #      #  # #  # #  # #    #   #  #    #   #  # #  #
 *  #    ## ##      #  # #  # #  # #    #   #  #    #   #    #  #
 *  #    # # #      #  # #  # ####  ##  #   #  #    #   #    ####
 *  #    #   #      #  # #  # #  #    # # # #  #    #   #    #  #
 *  #  # #   #      #  # #  # #  #    # # # #  #    #   #  # #  #
 *   ##  #   # #### ###  ###  #  # ###   # #  ###   #    ##  #  #
 *
 * OPType: Status
 ****************************************************************************/
struct trMsg_CM_DDASwitch_Status  :
  public tun_RxMsg
{
  TUN_INSTALL_MESSAGE_CALLBACKLIST;

  virtual tBool bParse(tPCU8 pcu8Data)
  {
    u8DDAMode = pcu8Data[ 0];
    return TRUE;
  }

  virtual tVoid vTrace( tVoid)
  {
    ETG_TRACE_USR4(( " trMsg_CM_DDASwitch_Status\n"
      "->01 ClientID   : %04x\n"
      "->02 FBlock     : %04x\n"
      "->03 InstanceId : %02x\n"
      "->04 FunctionId : %04x\n"
      "->05 OpType     : %02x\n"
      "->06 %d bytes payload\n"
      "=> DDAMode : %02x\n"
      ,ETG_CENUM(tun_tenClientId,         enClientId)
      ,ETG_CENUM(tun_tenFBlockId,         enFBlockId)
      ,ETG_CENUM(tun_tenInstanceId,       enInstanceId)
      ,ETG_CENUM(tun_tenAmFmTunerMsg,     enFunctionId)
      ,ETG_CENUM(tun_tenOpType,           enOpType)
      ,u16DataLen
      ,u8DDAMode
      ));
  }
  tU8  u8DDAMode;
};


/*****************************************************************************
 * TAEscape
 *
 *  #####  ##  ####  ###  ##   ##  ###  ####
 *    #   #  # #    #    #  # #  # #  # #
 *    #   #  # #    #    #    #  # #  # #
 *    #   #### ###   ##  #    #### ###  ###
 *    #   #  # #       # #    #  # #    #
 *    #   #  # #       # #  # #  # #    #
 *    #   #  # #### ###   ##  #  # #    ####
 *
 * OPType:
 ****************************************************************************/


/*****************************************************************************
 * CM_PersistentStorage
 *
 *   ##  #   #      ###  #### ####   ### ###  ### ##### #### #   # #####  ### #####  ##  ####   ##   ###  ####
 *  #  # #   #      #  # #    #   # #     #  #      #   #    ##  #   #   #      #   #  # #   # #  # #   # #
 *  #    ## ##      #  # #    #   # #     #  #      #   #    # # #   #   #      #   #  # #   # #  # #     #
 *  #    # # #      ###  ###  ####   ##   #   ##    #   ###  # # #   #    ##    #   #  # ####  #### # ### ###
 *  #    #   #      #    #    # #      #  #     #   #   #    #  ##   #      #   #   #  # # #   #  # #   # #
 *  #  # #   #      #    #    #  #     #  #     #   #   #    #  ##   #      #   #   #  # #  #  #  # #   # #
 *   ##  #   # #### #    #### #   # ###  ### ###    #   #### #   #   #   ###    #    ##  #   # #  #  ###  ####
 *
 * OPType:  Set, SetGet, Status, Error
 ****************************************************************************/

/*****************************************************************************
 * OPType: SetGet
 ****************************************************************************/
struct trMsg_CM_PersistentStorage_Set :
  public tun_TxMsg
{
  public:
    trMsg_CM_PersistentStorage_Set( tBool bSetPersistentStorageOn = FALSE)
    {
      enInstanceId    = enInstanceId_All;
      enOpType        = enOpType_SET;
      enFunctionId    = enMsg_CM_PersistentStorage;
      u16DataLen      = 1;
      bPersistentStorageOn = bSetPersistentStorageOn;
    }

    virtual tVoid vSerialize( tPU8 pu8Data)
    {
      pu8Data[ 0]    = (tU8)bPersistentStorageOn;
    }

    virtual tVoid vTrace( tVoid)
    {
      ETG_TRACE_USR4(( " trMsg_CM_PersistentStorage_Set\n"
        "->01 ClientID   : %04x\n"
        "->02 FBlock     : %04x\n"
        "->03 InstanceId : %02x\n"
        "->04 FunctionId : %04x\n"
        "->05 OpType     : %02x\n"
        "->06 %d bytes payload\n"
        "=> PersistentStorageOn : %02x\n"
        ,ETG_CENUM(tun_tenClientId,         enClientId)
        ,ETG_CENUM(tun_tenFBlockId,         enFBlockId)
        ,ETG_CENUM(tun_tenInstanceId,       enInstanceId)
        ,ETG_CENUM(tun_tenAmFmTunerMsg,     enFunctionId)
        ,ETG_CENUM(tun_tenOpType,           enOpType)
        ,u16DataLen
        ,bPersistentStorageOn
        ));
    }

    tBool                   bPersistentStorageOn;
};

/*****************************************************************************
 * OPType: SetGet
 ****************************************************************************/
struct trMsg_CM_PersistentStorage_SetGet :
  public tun_TxMsg
{
  public:
    trMsg_CM_PersistentStorage_SetGet( tBool bSetPersistentStorageOn = FALSE)
    {
      enInstanceId    = enInstanceId_All;
      enOpType        = enOpType_SETGET;
      enFunctionId    = enMsg_CM_PersistentStorage;
      u16DataLen      = 1;
      bPersistentStorageOn = bSetPersistentStorageOn;
    }

    virtual tVoid vSerialize( tPU8 pu8Data)
    {
      pu8Data[ 0]    = (tU8)bPersistentStorageOn;
    }

    virtual tVoid vTrace( tVoid)
    {
      ETG_TRACE_USR4(( " trMsg_CM_PersistentStorage_SetGet\n"
        "->01 ClientID   : %04x\n"
        "->02 FBlock     : %04x\n"
        "->03 InstanceId : %02x\n"
        "->04 FunctionId : %04x\n"
        "->05 OpType     : %02x\n"
        "->06 %d bytes payload\n"
        "=> PersistentStorageOn : %02x\n"
        ,ETG_CENUM(tun_tenClientId,         enClientId)
        ,ETG_CENUM(tun_tenFBlockId,         enFBlockId)
        ,ETG_CENUM(tun_tenInstanceId,       enInstanceId)
        ,ETG_CENUM(tun_tenAmFmTunerMsg,     enFunctionId)
        ,ETG_CENUM(tun_tenOpType,           enOpType)
        ,u16DataLen
        ,bPersistentStorageOn
        ));
    }

    tBool                   bPersistentStorageOn;
};


/*****************************************************************************
 * OPType: Status
 ****************************************************************************/
struct trMsg_CM_PersistentStorage_Status  :
  public tun_RxMsg
{
  TUN_INSTALL_MESSAGE_CALLBACKLIST;

  trMsg_CM_PersistentStorage_Status( tVoid) :
    bPersistentStorageOn( FALSE)
  {
  }

  virtual tBool bParse(tPCU8 pcu8Data)
  {
    bPersistentStorageOn = (tBool)pcu8Data[ 0];

    return TRUE;
  }

  virtual tVoid vTrace( tVoid)
  {
    ETG_TRACE_USR4(( " trMsg_CM_PersistentStorage_Status\n"
      "->01 ClientID   : %04x\n"
      "->02 FBlock     : %04x\n"
      "->03 InstanceId : %02x\n"
      "->04 FunctionId : %04x\n"
      "->05 OpType     : %02x\n"
      "->06 %d bytes payload\n"
      "=> PersistentStorageOn : %02x\n"
      ,ETG_CENUM(tun_tenClientId,         enClientId)
      ,ETG_CENUM(tun_tenFBlockId,         enFBlockId)
      ,ETG_CENUM(tun_tenInstanceId,       enInstanceId)
      ,ETG_CENUM(tun_tenAmFmTunerMsg,     enFunctionId)
      ,ETG_CENUM(tun_tenOpType,           enOpType)
      ,u16DataLen
      ,bPersistentStorageOn
      ));
  }

  tBool                   bPersistentStorageOn;
};


/*****************************************************************************
 * TAMessage
 *
 *  #####  ##  #   # ####  ###  ###  ##   ###  ####
 *    #   #  # #   # #    #    #    #  # #   # #
 *    #   #  # ## ## #    #    #    #  # #     #
 *    #   #### # # # ###   ##   ##  #### # ### ###
 *    #   #  # #   # #       #    # #  # #   # #
 *    #   #  # #   # #       #    # #  # #   # #
 *    #   #  # #   # #### ###  ###  #  #  ###  ####
 *
 * OPType:
 ****************************************************************************/


/*****************************************************************************
 * CM_DateAndTime
 *
 *   ##  #   #      ###   ##  ##### ####  ##  #   # ###  ##### ### #   # ####
 *  #  # #   #      #  # #  #   #   #    #  # ##  # #  #   #    #  #   # #
 *  #    ## ##      #  # #  #   #   #    #  # # # # #  #   #    #  ## ## #
 *  #    # # #      #  # ####   #   ###  #### # # # #  #   #    #  # # # ###
 *  #    #   #      #  # #  #   #   #    #  # #  ## #  #   #    #  #   # #
 *  #  # #   #      #  # #  #   #   #    #  # #  ## #  #   #    #  #   # #
 *   ##  #   # #### ###  #  #   #   #### #  # #   # ###    #   ### #   # ####
 *
 * OPType:
 ****************************************************************************/


/*****************************************************************************
 * CurrentRadiotext
 *
 *   ##  #   # ####  ####  #### #   # ##### ####   ##  ###  ###  ##  ##### #### #   # #####
 *  #  # #   # #   # #   # #    ##  #   #   #   # #  # #  #  #  #  #   #   #    #   #   #
 *  #    #   # #   # #   # #    # # #   #   #   # #  # #  #  #  #  #   #   #     # #    #
 *  #    #   # ####  ####  ###  # # #   #   ####  #### #  #  #  #  #   #   ###    #     #
 *  #    #   # # #   # #   #    #  ##   #   # #   #  # #  #  #  #  #   #   #     # #    #
 *  #  # #   # #  #  #  #  #    #  ##   #   #  #  #  # #  #  #  #  #   #   #    #   #   #
 *   ##   ###  #   # #   # #### #   #   #   #   # #  # ###  ###  ##    #   #### #   #   #
 *
 * OPType:
 ****************************************************************************/


/*****************************************************************************
 * RadiotextMetaData
 *
 *  ####   ##  ###  ###  ##  ##### #### #   # ##### #   # #### #####  ##  ###   ##  #####  ##
 *  #   # #  # #  #  #  #  #   #   #    #   #   #   #   # #      #   #  # #  # #  #   #   #  #
 *  #   # #  # #  #  #  #  #   #   #     # #    #   ## ## #      #   #  # #  # #  #   #   #  #
 *  ####  #### #  #  #  #  #   #   ###    #     #   # # # ###    #   #### #  # ####   #   ####
 *  # #   #  # #  #  #  #  #   #   #     # #    #   #   # #      #   #  # #  # #  #   #   #  #
 *  #  #  #  # #  #  #  #  #   #   #    #   #   #   #   # #      #   #  # #  # #  #   #   #  #
 *  #   # #  # ###  ###  ##    #   #### #   #   #   #   # ####   #   #  # ###  #  #   #   #  #
 *
 * OPType:
 ****************************************************************************/


/*****************************************************************************
 * CM_RadiotextMetaData
 *
 *   ##  #   #      ####   ##  ###  ###  ##  ##### #### #   # ##### #   # #### #####  ##  ###   ##  #####  ##
 *  #  # #   #      #   # #  # #  #  #  #  #   #   #    #   #   #   #   # #      #   #  # #  # #  #   #   #  #
 *  #    ## ##      #   # #  # #  #  #  #  #   #   #     # #    #   ## ## #      #   #  # #  # #  #   #   #  #
 *  #    # # #      ####  #### #  #  #  #  #   #   ###    #     #   # # # ###    #   #### #  # ####   #   ####
 *  #    #   #      # #   #  # #  #  #  #  #   #   #     # #    #   #   # #      #   #  # #  # #  #   #   #  #
 *  #  # #   #      #  #  #  # #  #  #  #  #   #   #    #   #   #   #   # #      #   #  # #  # #  #   #   #  #
 *   ##  #   # #### #   # #  # ###  ###  ##    #   #### #   #   #   #   # ####   #   #  # ###  #  #   #   #  #
 *
 * OPType:
 ****************************************************************************/


/*****************************************************************************
 * CM_ReceptionParameter
 *
 *   ##  #   #      ####  ####  ##  #### ###  ##### ###  ##  #   # ###   ##  ####   ##  #   # #### ##### #### ####
 *  #  # #   #      #   # #    #  # #    #  #   #    #  #  # ##  # #  # #  # #   # #  # #   # #      #   #    #   #
 *  #    ## ##      #   # #    #    #    #  #   #    #  #  # # # # #  # #  # #   # #  # ## ## #      #   #    #   #
 *  #    # # #      ####  ###  #    ###  ###    #    #  #  # # # # ###  #### ####  #### # # # ###    #   ###  ####
 *  #    #   #      # #   #    #    #    #      #    #  #  # #  ## #    #  # # #   #  # #   # #      #   #    # #
 *  #  # #   #      #  #  #    #  # #    #      #    #  #  # #  ## #    #  # #  #  #  # #   # #      #   #    #  #
 *   ##  #   # #### #   # ####  ##  #### #      #   ###  ##  #   # #    #  # #   # #  # #   # ####   #   #### #   #
 *
 * OPType: Status
 ****************************************************************************/
struct trMsg_CM_ReceptionParameter_Status  :
  public tun_RxMsg
{
  TUN_INSTALL_MESSAGE_CALLBACKLIST;

  virtual tBool bParse(tPCU8 pcu8Data)
  {
    u8ReceptionParam = pcu8Data[ 0];
	u8ReceivedData = pcu8Data[2];

    return TRUE;
  }

  virtual tVoid vTrace( tVoid)
  {
    ETG_TRACE_USR4(( " trMsg_CM_ReceptionParameter_Status\n"
      "->01 ClientID   : %04x\n"
      "->02 FBlock     : %04x\n"
      "->03 InstanceId : %02x\n"
      "->04 FunctionId : %04x\n"
      "->05 OpType     : %02x\n"
      "->06 %d bytes payload\n"
      "=> Pos: ReceptionParameter : %02x\n"
	  "=> Data : %02x\n"
      ,ETG_CENUM(tun_tenClientId,         enClientId)
      ,ETG_CENUM(tun_tenFBlockId,         enFBlockId)
      ,ETG_CENUM(tun_tenInstanceId,       enInstanceId)
      ,ETG_CENUM(tun_tenAmFmTunerMsg,     enFunctionId)
      ,ETG_CENUM(tun_tenOpType,           enOpType)
      ,u16DataLen
      ,u8ReceptionParam
	  ,u8ReceivedData
      ));
  }

  tU8  u8ReceptionParam;
  tU8   u8ReceivedData;
};


/*****************************************************************************
 * CM_SurveilReception
 *
 *   ##  #   #       ### #   # ####  #   # #### ### #    ####  ####  ##  #### ###  ##### ###  ##  #   #
 *  #  # #   #      #    #   # #   # #   # #     #  #    #   # #    #  # #    #  #   #    #  #  # ##  #
 *  #    ## ##      #    #   # #   # #   # #     #  #    #   # #    #    #    #  #   #    #  #  # # # #
 *  #    # # #       ##  #   # ####  #   # ###   #  #    ####  ###  #    ###  ###    #    #  #  # # # #
 *  #    #   #         # #   # # #   #   # #     #  #    # #   #    #    #    #      #    #  #  # #  ##
 *  #  # #   #         # #   # #  #   # #  #     #  #    #  #  #    #  # #    #      #    #  #  # #  ##
 *   ##  #   # #### ###   ###  #   #   #   #### ### #### #   # ####  ##  #### #      #   ###  ##  #   #
 *
 * OPType:
 ****************************************************************************/


/*****************************************************************************
 * CM_SurveilTp
 *
 *   ##  #   #       ### #   # ####  #   # #### ### #    ##### ###
 *  #  # #   #      #    #   # #   # #   # #     #  #      #   #  #
 *  #    ## ##      #    #   # #   # #   # #     #  #      #   #  #
 *  #    # # #       ##  #   # ####  #   # ###   #  #      #   ###
 *  #    #   #         # #   # # #   #   # #     #  #      #   #
 *  #  # #   #         # #   # #  #   # #  #     #  #      #   #
 *   ##  #   # #### ###   ###  #   #   #   #### ### ####   #   #
 *
 * OPType:
 ****************************************************************************/


/*****************************************************************************
 * CM_SurveillanceConfiguration
 *
 *   ##  #   #       ### #   # ####  #   # #### ### #    #     ##  #   #  ##  ####  ##   ##  #   # #### ###  ###  #   # ####   ##  ##### ###  ##  #   #
 *  #  # #   #      #    #   # #   # #   # #     #  #    #    #  # ##  # #  # #    #  # #  # ##  # #     #  #   # #   # #   # #  #   #    #  #  # ##  #
 *  #    ## ##      #    #   # #   # #   # #     #  #    #    #  # # # # #    #    #    #  # # # # #     #  #     #   # #   # #  #   #    #  #  # # # #
 *  #    # # #       ##  #   # ####  #   # ###   #  #    #    #### # # # #    ###  #    #  # # # # ###   #  # ### #   # ####  ####   #    #  #  # # # #
 *  #    #   #         # #   # # #   #   # #     #  #    #    #  # #  ## #    #    #    #  # #  ## #     #  #   # #   # # #   #  #   #    #  #  # #  ##
 *  #  # #   #         # #   # #  #   # #  #     #  #    #    #  # #  ## #  # #    #  # #  # #  ## #     #  #   # #   # #  #  #  #   #    #  #  # #  ##
 *   ##  #   # #### ###   ###  #   #   #   #### ### #### #### #  # #   #  ##  ####  ##   ##  #   # #    ###  ###   ###  #   # #  #   #   ###  ##  #   #
 *
 * OPType:
 ****************************************************************************/


/*****************************************************************************
 * TaInfo
 *
 *  #####  ##  ### #   # ####  ##
 *    #   #  #  #  ##  # #    #  #
 *    #   #  #  #  # # # #    #  #
 *    #   ####  #  # # # ###  #  #
 *    #   #  #  #  #  ## #    #  #
 *    #   #  #  #  #  ## #    #  #
 *    #   #  # ### #   # #     ##
 *
 * OPType: Status
 ****************************************************************************/
struct trMsg_CM_TaInfo_Status  :
  public tun_RxMsg
{
  TUN_INSTALL_MESSAGE_CALLBACKLIST;

  virtual tBool bParse(tPCU8 pcu8Data)
  {
    u32Frequency = TUN_DRVADRIF_GET_U32( &pcu8Data[ 0]);
	u16PI = TUN_DRVADRIF_GET_U16( &pcu8Data[ 4]);
	OSAL_pvMemoryCopy( sTaStationname, &pcu8Data[6], 8);

    return TRUE;
  }

  virtual tVoid vTrace( tVoid)
  {
    ETG_TRACE_USR4(( " trMsg_CM_TaInfo_Status\n"
      "->01 ClientID   : %04x\n"
      "->02 FBlock     : %04x\n"
      "->03 InstanceId : %02x\n"
      "->04 FunctionId : %04x\n"
      "->05 OpType     : %02x\n"
      "->06 %d bytes payload\n"
      "=> Frequency    : %02x\n"
	  "=> PI           : %02x\n"
	  "=> TaStationname : %8s\n"
      ,ETG_CENUM(tun_tenClientId,         enClientId)
      ,ETG_CENUM(tun_tenFBlockId,         enFBlockId)
      ,ETG_CENUM(tun_tenInstanceId,       enInstanceId)
      ,ETG_CENUM(tun_tenAmFmTunerMsg,     enFunctionId)
      ,ETG_CENUM(tun_tenOpType,           enOpType)
      ,u16DataLen
      ,u32Frequency
	  ,u16PI
	  ,sTaStationname
      ));
  }

  tU32  u32Frequency;
  tU16 u16PI;
  tChar  sTaStationname[ 8];
};


/*****************************************************************************
 * CM_ATPresetList
 *
 *   ##  #   #       ##  ##### ###  ####  ####  ### #### ##### #    ###  ### #####
 *  #  # #   #      #  #   #   #  # #   # #    #    #      #   #     #  #      #
 *  #    ## ##      #  #   #   #  # #   # #    #    #      #   #     #  #      #
 *  #    # # #      ####   #   ###  ####  ###   ##  ###    #   #     #   ##    #
 *  #    #   #      #  #   #   #    # #   #       # #      #   #     #     #   #
 *  #  # #   #      #  #   #   #    #  #  #       # #      #   #     #     #   #
 *   ##  #   # #### #  #   #   #    #   # #### ###  ####   #   #### ### ###    #
 *
 * OPType:
 ****************************************************************************/


/*****************************************************************************
 * CM_ATPresetSave
 *
 *   ##  #   #       ##  ##### ###  ####  ####  ### #### #####  ###  ##  #   # ####
 *  #  # #   #      #  #   #   #  # #   # #    #    #      #   #    #  # #   # #
 *  #    ## ##      #  #   #   #  # #   # #    #    #      #   #    #  # #   # #
 *  #    # # #      ####   #   ###  ####  ###   ##  ###    #    ##  #### #   # ###
 *  #    #   #      #  #   #   #    # #   #       # #      #      # #  # #   # #
 *  #  # #   #      #  #   #   #    #  #  #       # #      #      # #  #  # #  #
 *   ##  #   # #### #  #   #   #    #   # #### ###  ####   #   ###  #  #   #   ####
 *
 * OPType:
 ****************************************************************************/


/*****************************************************************************
 * CM_SystemPresets
 *
 *   ##  #   #       ### #   #  ### ##### #### #   # ###  ####  ####  ### #### #####  ###
 *  #  # #   #      #    #   # #      #   #    #   # #  # #   # #    #    #      #   #
 *  #    ## ##      #    #   # #      #   #    ## ## #  # #   # #    #    #      #   #
 *  #    # # #       ##   # #   ##    #   ###  # # # ###  ####  ###   ##  ###    #    ##
 *  #    #   #         #   #      #   #   #    #   # #    # #   #       # #      #      #
 *  #  # #   #         #   #      #   #   #    #   # #    #  #  #       # #      #      #
 *   ##  #   # #### ###    #   ###    #   #### #   # #    #   # #### ###  ####   #   ###
 *
 * OPType: Set, Status, Error
 ****************************************************************************/





/*****************************************************************************
 * CM_ATPresetListSetActive
 *
 *   ##  #   #       ##  ##### ###  ####  ####  ### #### ##### #    ###  ### #####  ### #### #####  ##   ##  ##### ### #   # ####
 *  #  # #   #      #  #   #   #  # #   # #    #    #      #   #     #  #      #   #    #      #   #  # #  #   #    #  #   # #
 *  #    ## ##      #  #   #   #  # #   # #    #    #      #   #     #  #      #   #    #      #   #  # #      #    #  #   # #
 *  #    # # #      ####   #   ###  ####  ###   ##  ###    #   #     #   ##    #    ##  ###    #   #### #      #    #  #   # ###
 *  #    #   #      #  #   #   #    # #   #       # #      #   #     #     #   #      # #      #   #  # #      #    #  #   # #
 *  #  # #   #      #  #   #   #    #  #  #       # #      #   #     #     #   #      # #      #   #  # #  #   #    #   # #  #
 *   ##  #   # #### #  #   #   #    #   # #### ###  ####   #   #### ### ###    #   ###  ####   #   #  #  ##    #   ###   #   ####
 *
 * OPType:
 ****************************************************************************/


/*****************************************************************************
 * CM_ATPresetListConfig
 *
 *   ##  #   #       ##  ##### ###  ####  ####  ### #### ##### #    ###  ### #####  ##   ##  #   # #### ###  ###
 *  #  # #   #      #  #   #   #  # #   # #    #    #      #   #     #  #      #   #  # #  # ##  # #     #  #   #
 *  #    ## ##      #  #   #   #  # #   # #    #    #      #   #     #  #      #   #    #  # # # # #     #  #
 *  #    # # #      ####   #   ###  ####  ###   ##  ###    #   #     #   ##    #   #    #  # # # # ###   #  # ###
 *  #    #   #      #  #   #   #    # #   #       # #      #   #     #     #   #   #    #  # #  ## #     #  #   #
 *  #  # #   #      #  #   #   #    #  #  #       # #      #   #     #     #   #   #  # #  # #  ## #     #  #   #
 *   ##  #   # #### #  #   #   #    #   # #### ###  ####   #   #### ### ###    #    ##   ##  #   # #    ###  ###
 *
 * OPType:
 ****************************************************************************/


/*****************************************************************************
 * CM_ATPresetListResetAll
 *
 *   ##  #   #       ##  ##### ###  ####  ####  ### #### ##### #    ###  ### ##### ####  ####  ### #### #####  ##  #    #
 *  #  # #   #      #  #   #   #  # #   # #    #    #      #   #     #  #      #   #   # #    #    #      #   #  # #    #
 *  #    ## ##      #  #   #   #  # #   # #    #    #      #   #     #  #      #   #   # #    #    #      #   #  # #    #
 *  #    # # #      ####   #   ###  ####  ###   ##  ###    #   #     #   ##    #   ####  ###   ##  ###    #   #### #    #
 *  #    #   #      #  #   #   #    # #   #       # #      #   #     #     #   #   # #   #       # #      #   #  # #    #
 *  #  # #   #      #  #   #   #    #  #  #       # #      #   #     #     #   #   #  #  #       # #      #   #  # #    #
 *   ##  #   # #### #  #   #   #    #   # #### ###  ####   #   #### ### ###    #   #   # #### ###  ####   #   #  # #### ####
 *
 * OPType:
 ****************************************************************************/


/*****************************************************************************
 * CM_StationListUpdate
 *
 *   ##  #   #       ### #####  ##  ##### ###  ##  #   # #    ###  ### ##### #   # ###  ###   ##  ##### ####
 *  #  # #   #      #      #   #  #   #    #  #  # ##  # #     #  #      #   #   # #  # #  # #  #   #   #
 *  #    ## ##      #      #   #  #   #    #  #  # # # # #     #  #      #   #   # #  # #  # #  #   #   #
 *  #    # # #       ##    #   ####   #    #  #  # # # # #     #   ##    #   #   # ###  #  # ####   #   ###
 *  #    #   #         #   #   #  #   #    #  #  # #  ## #     #     #   #   #   # #    #  # #  #   #   #
 *  #  # #   #         #   #   #  #   #    #  #  # #  ## #     #     #   #   #   # #    #  # #  #   #   #
 *   ##  #   # #### ###    #   #  #   #   ###  ##  #   # #### ### ###    #    ###  #    ###  #  #   #   ####
 *
 * OPType:
 ****************************************************************************/


/*****************************************************************************
 * CM_StationListRelease
 *
 *   ##  #   #       ### #####  ##  ##### ###  ##  #   # #    ###  ### ##### ####  #### #    ####  ##   ### ####
 *  #  # #   #      #      #   #  #   #    #  #  # ##  # #     #  #      #   #   # #    #    #    #  # #    #
 *  #    ## ##      #      #   #  #   #    #  #  # # # # #     #  #      #   #   # #    #    #    #  # #    #
 *  #    # # #       ##    #   ####   #    #  #  # # # # #     #   ##    #   ####  ###  #    ###  ####  ##  ###
 *  #    #   #         #   #   #  #   #    #  #  # #  ## #     #     #   #   # #   #    #    #    #  #    # #
 *  #  # #   #         #   #   #  #   #    #  #  # #  ## #     #     #   #   #  #  #    #    #    #  #    # #
 *   ##  #   # #### ###    #   #  #   #   ###  ##  #   # #### ### ###    #   #   # #### #### #### #  # ###  ####
 *
 * OPType:
 ****************************************************************************/


/*****************************************************************************
 * CM_StationsList
 *
 *   ##  #   #       ### #####  ##  ##### ###  ##  #   #  ### #    ###  ### #####
 *  #  # #   #      #      #   #  #   #    #  #  # ##  # #    #     #  #      #
 *  #    ## ##      #      #   #  #   #    #  #  # # # # #    #     #  #      #
 *  #    # # #       ##    #   ####   #    #  #  # # # #  ##  #     #   ##    #
 *  #    #   #         #   #   #  #   #    #  #  # #  ##    # #     #     #   #
 *  #  # #   #         #   #   #  #   #    #  #  # #  ##    # #     #     #   #
 *   ##  #   # #### ###    #   #  #   #   ###  ##  #   # ###  #### ### ###    #
 *
 * OPType:
 ****************************************************************************/


/*****************************************************************************
 * CM_Container
 *
 *   ##  #   #       ##   ##  #   # #####  ##  ### #   # #### ####
 *  #  # #   #      #  # #  # ##  #   #   #  #  #  ##  # #    #   #
 *  #    ## ##      #    #  # # # #   #   #  #  #  # # # #    #   #
 *  #    # # #      #    #  # # # #   #   ####  #  # # # ###  ####
 *  #    #   #      #    #  # #  ##   #   #  #  #  #  ## #    # #
 *  #  # #   #      #  # #  # #  ##   #   #  #  #  #  ## #    #  #
 *   ##  #   # ####  ##   ##  #   #   #   #  # ### #   # #### #   #
 *
 * OPType:
 ****************************************************************************/


/*****************************************************************************
 * CM_SpecialDiagnosisMode
 *
 *   ##  #   #       ### ###  ####  ##  ###  ##  #    ###  ###  ##   ###  #   #  ##   ### ###  ### #   #  ##  ###  ####
 *  #  # #   #      #    #  # #    #  #  #  #  # #    #  #  #  #  # #   # ##  # #  # #     #  #    #   # #  # #  # #
 *  #    ## ##      #    #  # #    #     #  #  # #    #  #  #  #  # #     # # # #  # #     #  #    ## ## #  # #  # #
 *  #    # # #       ##  ###  ###  #     #  #### #    #  #  #  #### # ### # # # #  #  ##   #   ##  # # # #  # #  # ###
 *  #    #   #         # #    #    #     #  #  # #    #  #  #  #  # #   # #  ## #  #    #  #     # #   # #  # #  # #
 *  #  # #   #         # #    #    #  #  #  #  # #    #  #  #  #  # #   # #  ## #  #    #  #     # #   # #  # #  # #
 *   ##  #   # #### ###  #    ####  ##  ### #  # #### ###  ### #  #  ###  #   #  ##  ###  ### ###  #   #  ##  ###  ####
 *
 * OPType:
 ****************************************************************************/


/*****************************************************************************
 * CM_TestMode
 *
 *   ##  #   #      ##### ####  ### ##### #   #  ##  ###  ####
 *  #  # #   #        #   #    #      #   #   # #  # #  # #
 *  #    ## ##        #   #    #      #   ## ## #  # #  # #
 *  #    # # #        #   ###   ##    #   # # # #  # #  # ###
 *  #    #   #        #   #       #   #   #   # #  # #  # #
 *  #  # #   #        #   #       #   #   #   # #  # #  # #
 *   ##  #   # ####   #   #### ###    #   #   #  ##  ###  ####
 *
 * OPType:  Set, Status, Error
 ****************************************************************************/


/*****************************************************************************
 * OPType: Set
 ****************************************************************************/
struct trMsg_CM_TestMode_Set :
  public tun_TxMsg
{
  public:
    trMsg_CM_TestMode_Set( tVoid) :
      enTestmodeActivity( enTestmodeActivity_Testmode_auto_notifications_off)
    {
      enInstanceId    = enInstanceId_All;
      enOpType        = enOpType_SET;
      enFunctionId    = enMsg_CM_EnhancedTestMode;
      u16DataLen      = 1;
    }

    virtual tVoid vSerialize( tPU8 pu8Data)
    {
      pu8Data[ 0]    = (tU8)enTestmodeActivity;
    }

    virtual tVoid vTrace( tVoid)
    {
      ETG_TRACE_USR4(( " trMsg_CM_TestMode_Set\n"
        "->01 ClientID   : %04x\n"
        "->02 FBlock     : %04x\n"
        "->03 InstanceId : %02x\n"
        "->04 FunctionId : %04x\n"
        "->05 OpType     : %02x\n"
        "->06 %d bytes payload\n"
        "=> TestmodeActivity : %02x\n"
        ,ETG_CENUM(tun_tenClientId,         enClientId)
        ,ETG_CENUM(tun_tenFBlockId,         enFBlockId)
        ,ETG_CENUM(tun_tenInstanceId,       enInstanceId)
        ,ETG_CENUM(tun_tenAmFmTunerMsg,     enFunctionId)
        ,ETG_CENUM(tun_tenOpType,           enOpType)
        ,u16DataLen
        ,ETG_CENUM(tenTestmodeActivity,     enTestmodeActivity)
        ));
    }

    tenTestmodeActivity    enTestmodeActivity;
};


/*****************************************************************************
 * OPType: Status
 ****************************************************************************/
struct trMsg_CM_TestMode_Status :
  public tun_RxMsg
{
  TUN_INSTALL_MESSAGE_CALLBACKLIST;

  virtual tBool bParse(tPCU8 pcu8Data)
  {                                                                          // index | length
    enModulation                  = (tenModulation)pcu8Data[  0];            //     0 |      1
    u32Frequency                  = TUN_DRVADRIF_GET_U32( &pcu8Data[  1]);   //     1 |      4
    u16PI                         = TUN_DRVADRIF_GET_U16( &pcu8Data[  5]);   //     5 |      2
    u8Quality                     = pcu8Data[  7];                           //     7 |      1
    u8Fieldstrength               = pcu8Data[  8];                           //     8 |      1
    u8Multipath                   = pcu8Data[  9];                           //     9 |      1
    u8NeighbourChannelDistortion  = pcu8Data[ 10];                           //    10 |      1
    u8AFListLength                = pcu8Data[ 11];                           //    11 |      1
    u8Hicut                       = pcu8Data[ 12];                           //    12 |      1
    u8Sharx                       = pcu8Data[ 13];                           //    13 |      1
    OSAL_pvMemoryCopy( sStationname, &pcu8Data[ 14], 8);                     //    14 |      8
    u8ChannelSeparation           = pcu8Data[ 22];                           //    22 |      1
    u8RdsErrorRate                = pcu8Data[ 23];                           //    23 |      1
    u8Deviation                   = pcu8Data[ 24];                           //    24 |      1
    u8InfoBits                    = pcu8Data[ 25];                           //    25 |      1
                                                                             //----------------
                                                                             //    26 bytes
    return TRUE;
  }

  virtual tVoid vTrace( tVoid)
  {
    ETG_TRACE_USR1(( " trMsg_CM_TestMode_Status -> Part1\n"
      //"->01 ClientID   : %04x\n"
      //"->02 FBlock     : %04x\n"
      //"->03 InstanceId : %02x\n"
      //"->04 FunctionId : %04x\n"
      //"->05 OpType     : %02x\n"
      "->06 %d bytes payload\n"
      "=>   Modulation            : %d\n"
      "=>   Frequency             : %d kHz\n"
      "=>   PI                    : %02x\n"
      "=>   Quality               : %02x\n"
      "=>   Fieldstrength         : %d dB\n"
      "=>   Multipath             : %02x\n"
      "=>   NeighbourChannelDistortion : %02x\n"
      "=>   AFListLength          : %d\n"
      "=>   Hicut                 : %d\n"
//      ,ETG_CENUM(tun_tenClientId,         enClientId)
//      ,ETG_CENUM(tun_tenFBlockId,         enFBlockId)
//      ,ETG_CENUM(tun_tenInstanceId,       enInstanceId)
//      ,ETG_CENUM(tun_tenAmFmTunerMsg,     enFunctionId)
//      ,ETG_CENUM(tun_tenOpType,           enOpType)
      ,u16DataLen
      ,ETG_CENUM(tenModulation,           enModulation)
      ,u32Frequency
      ,u16PI
      ,u8Quality
      ,u8Fieldstrength
      ,u8Multipath
      ,u8NeighbourChannelDistortion
      ,u8AFListLength
      ,u8Hicut
      ));

    ETG_TRACE_USR1(( " trMsg_CM_TestMode_Status -> Part2\n"
      "=>   Sharx                 : %d\n"
      "=>   Stationname           : %8s\n"
      "=>   ChannelSeparation     : %02x\n"
      "=>   RdsErrorRate          : %d percent\n"
      "=>   Deviation             : %d kHz\n"
      "=>   InfoBits              : %02x\n"
      ,u8Sharx
      ,sStationname
      ,u8ChannelSeparation
      ,u8RdsErrorRate
      ,u8Deviation
      ,u8InfoBits
      ));

  }


  tenModulation     enModulation;
  tU32              u32Frequency;
  tU16              u16PI;
  tU8               u8Quality;
  tU8               u8Fieldstrength;
  tU8               u8Multipath;
  tU8               u8NeighbourChannelDistortion;
  tU8               u8AFListLength;
  tU8               u8Hicut;
  tU8               u8Sharx;
  tChar             sStationname[ 8];
  tU8               u8ChannelSeparation;
  tU8               u8RdsErrorRate;
  tU8               u8Deviation;
  tU8               u8InfoBits;
};


/*****************************************************************************
 * CM_ATAfLis
 *
 *   ##  #   #       ##  #####  ##  #### #    ###  ###
 *  #  # #   #      #  #   #   #  # #    #     #  #
 *  #    ## ##      #  #   #   #  # #    #     #  #
 *  #    # # #      ####   #   #### ###  #     #   ##
 *  #    #   #      #  #   #   #  # #    #     #     #
 *  #  # #   #      #  #   #   #  # #    #     #     #
 *   ##  #   # #### #  #   #   #  # #    #### ### ###
 *
 * OPType:
 ****************************************************************************/


/*****************************************************************************
 * CM_TestTunerSwitches
 *
 *   ##  #   #      ##### ####  ### ##### ##### #   # #   # #### ####   ### #   # ### #####  ##  #  # ####  ###
 *  #  # #   #        #   #    #      #     #   #   # ##  # #    #   # #    #   #  #    #   #  # #  # #    #
 *  #    ## ##        #   #    #      #     #   #   # # # # #    #   # #    #   #  #    #   #    #  # #    #
 *  #    # # #        #   ###   ##    #     #   #   # # # # ###  ####   ##  #   #  #    #   #    #### ###   ##
 *  #    #   #        #   #       #   #     #   #   # #  ## #    # #      # # # #  #    #   #    #  # #       #
 *  #  # #   #        #   #       #   #     #   #   # #  ## #    #  #     # # # #  #    #   #  # #  # #       #
 *   ##  #   # ####   #   #### ###    #     #    ###  #   # #### #   # ###   # #  ###   #    ##  #  # #### ###
 *
 * OPType:
 ****************************************************************************/


/*****************************************************************************
 * CM_SetConfiguration
 *
 *   ##  #   #       ### #### #####  ##   ##  #   # #### ###  ###  #   # ####   ##  ##### ###  ##  #   #
 *  #  # #   #      #    #      #   #  # #  # ##  # #     #  #   # #   # #   # #  #   #    #  #  # ##  #
 *  #    ## ##      #    #      #   #    #  # # # # #     #  #     #   # #   # #  #   #    #  #  # # # #
 *  #    # # #       ##  ###    #   #    #  # # # # ###   #  # ### #   # ####  ####   #    #  #  # # # #
 *  #    #   #         # #      #   #    #  # #  ## #     #  #   # #   # # #   #  #   #    #  #  # #  ##
 *  #  # #   #         # #      #   #  # #  # #  ## #     #  #   # #   # #  #  #  #   #    #  #  # #  ##
 *   ##  #   # #### ###  ####   #    ##   ##  #   # #    ###  ###   ###  #   # #  #   #   ###  ##  #   #
 *
 * OPType:
 ****************************************************************************/


/*****************************************************************************
 * CM_GetConfiguration
 *
 *   ##  #   #       ###  #### #####  ##   ##  #   # #### ###  ###  #   # ####   ##  ##### ###  ##  #   #
 *  #  # #   #      #   # #      #   #  # #  # ##  # #     #  #   # #   # #   # #  #   #    #  #  # ##  #
 *  #    ## ##      #     #      #   #    #  # # # # #     #  #     #   # #   # #  #   #    #  #  # # # #
 *  #    # # #      # ### ###    #   #    #  # # # # ###   #  # ### #   # ####  ####   #    #  #  # # # #
 *  #    #   #      #   # #      #   #    #  # #  ## #     #  #   # #   # # #   #  #   #    #  #  # #  ##
 *  #  # #   #      #   # #      #   #  # #  # #  ## #     #  #   # #   # #  #  #  #   #    #  #  # #  ##
 *   ##  #   # ####  ###  ####   #    ##   ##  #   # #    ###  ###   ###  #   # #  #   #   ###  ##  #   #
 *
 * OPType:
 ****************************************************************************/

/*****************************************************************************
 * TMCSelectionFilter
 *
 *  ##### #   #  ##   ### #### #    ####  ##  ##### ###  ##  #   # #### ### #    ##### #### ####
 *    #   #   # #  # #    #    #    #    #  #   #    #  #  # ##  # #     #  #      #   #    #   #
 *    #   ## ## #    #    #    #    #    #      #    #  #  # # # # #     #  #      #   #    #   #
 *    #   # # # #     ##  ###  #    ###  #      #    #  #  # # # # ###   #  #      #   ###  ####
 *    #   #   # #       # #    #    #    #      #    #  #  # #  ## #     #  #      #   #    # #
 *    #   #   # #  #    # #    #    #    #  #   #    #  #  # #  ## #     #  #      #   #    #  #
 *    #   #   #  ##  ###  #### #### ####  ##    #   ###  ##  #   # #    ### ####   #   #### #   #
 *
 * OPType:
 ****************************************************************************/


/*****************************************************************************
 * TMCData
 *
 *  ##### #   #  ##  ###   ##  #####  ##
 *    #   #   # #  # #  # #  #   #   #  #
 *    #   ## ## #    #  # #  #   #   #  #
 *    #   # # # #    #  # ####   #   ####
 *    #   #   # #    #  # #  #   #   #  #
 *    #   #   # #  # #  # #  #   #   #  #
 *    #   #   #  ##  ###  #  #   #   #  #
 *
 * OPType:
 ****************************************************************************/


/*****************************************************************************
 * TMCStations
 *
 *  ##### #   #  ##   ### #####  ##  ##### ###  ##  #   #  ###
 *    #   #   # #  # #      #   #  #   #    #  #  # ##  # #
 *    #   ## ## #    #      #   #  #   #    #  #  # # # # #
 *    #   # # # #     ##    #   ####   #    #  #  # # # #  ##
 *    #   #   # #       #   #   #  #   #    #  #  # #  ##    #
 *    #   #   # #  #    #   #   #  #   #    #  #  # #  ##    #
 *    #   #   #  ##  ###    #   #  #   #   ###  ##  #   # ###
 *
 * OPType:
 ****************************************************************************/


/*****************************************************************************
 * CM_TMCStationsRelease
 *
 *   ##  #   #      ##### #   #  ##   ### #####  ##  ##### ###  ##  #   #  ### ####  #### #    ####  ##   ### ####
 *  #  # #   #        #   #   # #  # #      #   #  #   #    #  #  # ##  # #    #   # #    #    #    #  # #    #
 *  #    ## ##        #   ## ## #    #      #   #  #   #    #  #  # # # # #    #   # #    #    #    #  # #    #
 *  #    # # #        #   # # # #     ##    #   ####   #    #  #  # # # #  ##  ####  ###  #    ###  ####  ##  ###
 *  #    #   #        #   #   # #       #   #   #  #   #    #  #  # #  ##    # # #   #    #    #    #  #    # #
 *  #  # #   #        #   #   # #  #    #   #   #  #   #    #  #  # #  ##    # #  #  #    #    #    #  #    # #
 *   ##  #   # ####   #   #   #  ##  ###    #   #  #   #   ###  ##  #   # ###  #   # #### #### #### #  # ###  ####
 *
 * OPType:
 ****************************************************************************/


/*****************************************************************************
 * TMCStatus
 *
 *  ##### #   #  ##   ### #####  ##  ##### #   #  ###
 *    #   #   # #  # #      #   #  #   #   #   # #
 *    #   ## ## #    #      #   #  #   #   #   # #
 *    #   # # # #     ##    #   ####   #   #   #  ##
 *    #   #   # #       #   #   #  #   #   #   #    #
 *    #   #   # #  #    #   #   #  #   #   #   #    #
 *    #   #   #  ##  ###    #   #  #   #    ###  ###
 *
 * OPType:
 ****************************************************************************/




/* ######################################################################## */
/* ######################################################################## */
/* ######################################################################## */


/*****************************************************************************
 * ADR Driver Interface
 ****************************************************************************/

/*============================================================================
 * Communication state
 *--------------------------------------------------------------------------*/
typedef enum
{
  enComState_Off              = 0x00,
  enComState_WaitForStartup   = 0x01,
  enComState_Run              = 0x02,
  enComState_DiagnoseSession  = 0x03

} tun_tenComState;
/*==========================================================================*/

/*============================================================================
 * ADR state
 *--------------------------------------------------------------------------*/
typedef enum
{
  enADRState_INIT             = 0x00,
  enADRState_DEAD             = 0x01,
  enADRState_ALIVE            = 0x02,
  enADRState_DNL              = 0x03,
  enADRState_Unknown          = 0xFF
} tun_tenADRState;
/*==========================================================================*/

#ifdef VARIANT_S_FTR_ENABLE_INC_ADAPTATION_TO_NISSANLCN2KAI
class tun_INCComm
{
public:
	tun_INCComm();

	~tun_INCComm();

	bool bOpenADRSocket(uint8_t u8Domain,\
									   uint16_t u16Type,\
									   int iChannel) const;
	static tVoid vWaitForADRMessage(tVoid* pvArg);

	static int m_nADRSocketFd;

	OSAL_tThreadID       _hRxThreadId;


	tVoid vGetThreadPrioandStacksize(tU32 &u32ThreadPrio,tU32 &u32StackSize) const;

	tVoid vRxThreadSetup();

	tVoid vOpenINCCommunication();
	tVoid vSet_tuner_tclApp(tuner_tclApp* m_potunMain);
	tVoid vCloseAdr3Communication();
	static tuner_tclApp* m_potunMain;
};
#endif
class tun_DrvAdrIf :
  public TUN_Singleton< tun_DrvAdrIf>,
  public tun_MsgHandler< tun_DrvAdrIf>
{
  friend class TUN_Singleton< tun_DrvAdrIf>;

  private:

    static tPU8     pu8RxDataBuffer;
    static tU32     u32RxDataLength;
    static tun_tenADRState m_enADRState;

	static tPU8 u8CopyRxDataBuffer;


    tVoid*          m_pvHandle;
    tun_tenComState m_enCommunicationState;
    static tU32            m_u32CommunicationLostCounter;
	#if defined  (VARIANT_S_FTR_ENABLE_FEATURE_PSA_RCC) || defined (VARIANT_S_FTR_ENABLE_FEATURE_SUZUKI)|| defined (VARIANT_S_FTR_ENABLE_FEATURE_INF4CV)
    static tU8             m_u8ADR3ResetCounter;
	static tBool             m_blClearSystemFailureErrorLog;
    #endif
    tU8             m_u8PingCounter;

	static OSAL_tSemHandle     m_hMsgDrvAdrIfSem;

	/** class to hold the data received from ADR */
	class rxDataFromAdr
	{
		tPU8 pu8DataPointer;
		tU32 u32DataLength;

	public:
		rxDataFromAdr()
	{
			pu8DataPointer = NULL;
			u32DataLength = 0;
	}
		tVoid vSetDataPointer(tPU8 up8Data)
		{
			pu8DataPointer = up8Data;
		}
		tVoid vSetDataLength(tU32 u32Length)
		{
			u32DataLength = u32Length;
		}

		tPU8 up8GetDataPointer() const
		{
			return pu8DataPointer;
		}
		tU32 u32GetDataLength() const
		{
			return u32DataLength;
		}

	};
	static queue<rxDataFromAdr> m_ReceiverQueue;

    OSAL_tIODescriptor m_hAdr3CtrlDevice;

    //tuner_tclApp* m_potunMain;

      /*-------------*/
    /* constructor */
    /*-------------*/
    tun_DrvAdrIf();
    clConfigInterface* m_poclConfigInterface;


  public:

	  OSAL_tTimerHandle  m_hTimerPingTimer;

	  rxDataFromAdr m_structPopMessage;
	 /*--------------*/
    /* Destructors  */
    /*--------------*/
    virtual ~tun_DrvAdrIf( tVoid);

    tVoid vStartCommunication( tVoid);
    tVoid vStopCommunication( tVoid);
    static tuner_tclApp* m_potunMain;
    static tun_tclSPMClient* m_potunSPMClient;
	static tun_MsgToHMI* m_potun_MsgToHMI;
	static tun_DiaglogHandler* m_potun_DiaglogHandler;



  private:
   /*--------------------------------------*/
   /* Call back function for SSI interface */
   /*--------------------------------------*/
/**Ngp1kor: Currently commented for migration to G3 Platform.*/
#ifdef VARIANT_S_FTR_ENABLE_OPTADR
   static tVoid SsiDataConfirmationCallbackFunction( tVoid *pvHandle, enSsiConfState enState);
   static tVoid SsiDataIndicationCallbackFunction( tVoid *pvHandle, tU8 *pu8Data, tU32 u32Length);
#endif
   /*-------------------------------------------*/
   /* Call back function for ADR control driver */
   /*-------------------------------------------*/
   

   /*--------------------------------------*/
   /* Call back function for timer         */
   /*--------------------------------------*/
   static tVoid vPingTimerCallback(tVoid* pArg);
      tU8 m_u8Dgram_SendFailurCount;
#if defined  (VARIANT_S_FTR_ENABLE_FEATURE_PSA_RCC) || defined (VARIANT_S_FTR_ENABLE_FEATURE_SUZUKI) || defined (VARIANT_S_FTR_ENABLE_FEATURE_INF4CV)
   tVoid vUpdateTunerDTC();
#endif

  public:
	static tVoid vCallBackADRState( tU32 u32State);
    tVoid vNewADRState( tVoid);
    tVoid vPingTimerExpired( tVoid);
    tVoid vResetAdr( tVoid);
    tU32  u32GetRxDataBuffer( tPU8* ppu8RxData);
	/**vnd4kor : to solve prio 2 warning*/
    tVoid vFreeRxDataBuffer( tVoid);
	tBool bCheckIfQueueEmpty()const;
    tVoid vSendMessage( tU8* pu8TxMessage, tU32 u32StreamMsgLen);
    tBool bIsAdrCommunicationRunning( tVoid) const{ return ( m_enCommunicationState == enComState_Run) || ( m_enCommunicationState == enComState_DiagnoseSession); }
    tBool bIsAdrInDiagnoseSession( tVoid) const { return m_enCommunicationState == enComState_DiagnoseSession; }
    tVoid vTraceCommand( tPU8 pu8Msg, tU32 u32Len );
	/**vnd4kor : to solve prio 2 warning*/
    tVoid vSet_tunMain_Ptr(tuner_tclApp* ptunMain)const;
	tVoid vSetCommunicationState(tun_tenComState m_enCommunicationState);
    /**vnd4kor : to solve prio 2 warning*/
	tVoid vSet_tun_tclSPMClient_Ptr(tun_tclSPMClient* ptunSPMClient)const;
	tVoid vSet_tun_MsgToHMI_Ptr(tun_MsgToHMI* ptun_MsgToHMI) const;
	tVoid vSet_tun_DiaglogHandler_Ptr(tun_DiaglogHandler* potun_DiaglogHandler)const;
	tVoid vSet_tun_ConfigInterface_Ptr( clConfigInterface* poclConfigInterface );
    tun_tenADRState enGetADRState()const;
    tVoid vSetADRCommnState(tun_tenComState enComState);
	tun_tenComState vGetADRCommnState()const;

	static tVoid vPrintTrace(tU32 u32StreamMsgLen, const tU8* pu8TxMessage, tU8 u8TraceType);
#if defined VARIANT_S_FTR_ENABLE_INC_ADAPTATION_TO_NISSANLCN2KAI || defined  VARIANT_S_FTR_ENABLE_AARSPROXY
	static tVoid INCDataIndication(tU8 *pu8Data, tU32 u32Length);

	tVoid INCDataConfirmation() const;
#endif
   /*--------------------------------------*/
   /* subsceibed ADR messages              */
   /*--------------------------------------*/
    tVoid vMessage( trMsg_CM_Startup_Status* prMsg);
	tVoid vPrintADRVersionAtDefinedPingCount();
	tVoid vPrintCurrentTunerStateUpdates();
    tVoid vHandle_r_Ping(  const tU8* pu8Message, tU32 u32MessageLen);

};



#endif /* #ifndef TUN_DRVADRIF_HPP */
/*** END OF FILE *************************************************************/
