/************************************************************************
* FILE:           tun_Config.h
* PROJECT:        g3g
* SW-COMPONENT: 
*----------------------------------------------------------------------
*
* DESCRIPTION:    Tuner component, configuration.
*              
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
* HISTORY:      
* Date      | Author                       | Modification
* 20.01.05  | CM-DI/ESA2 ( RBIN ) Dinesh   | Initial version.
* 03.09.2013 | ECG  vnd4kor | Added member variable m_mapHMIListIDtoADRListID
* to map HMI List ID to ADR List ID
*************************************************************************/

#ifndef TUN_CONFIG_H
#define TUN_CONFIG_H

#define DIAGNOSTICS_KDS_ENTRY_IMPORT_INTERFACE
#include "fc_diagnosis_if.h"
#include "tun_Class_ForwardDecl.h"
#include "tun_PrjConfig.h"
#include "tun_Singleton.h"
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_TUNERFI_TYPES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_TUNERFI_ERRORCODES
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_TUNERFI_FUNCTIONIDS
#define MIDW_FI_S_IMPORT_INTERFACE_MIDW_TUNERFI_SERVICEINFO
#include "midw_fi_if.h"
/** vnd4kor: importing std map required for list id map */
#include <map>
#include <list>

using namespace std;
/****************************************************************************/
/*                                                                          */
/* DEFINITIONS                                                              */
/*                                                                          */
/****************************************************************************/
#define TUN_CONFIG_GET_U32(pu8) ((tU32)(((tU32)((pu8)[0])<<24) + ((tU32)((pu8)[1])<<16) + ((tU32)((pu8)[2])<<8) + (pu8)[3]))
//#define TUN_CONFIG_GET_U24(pu8) ((tU32)(((tU32)((pu8)[1])<<16) + ((tU32)((pu8)[2])<<8) + (pu8)[3]))
#define TUN_CONFIG_GET_U16(pu8) ((tU16)(((tU16)((pu8)[0])<<8) + (pu8)[1]))


#ifdef VARIANT_S_FTR_ENABLE_FEATURE_VW_MIBE
#define FEATURE_VW_MIBE 1
#else
#define FEATURE_VW_MIBE 0
#endif
#ifdef VARIANT_S_FTR_ENABLE_FEATURE_PSA_RCC
#define FEATURE_PSA_RCC 1
#else
#define FEATURE_PSA_RCC 0
#endif

class clConfigInterface;

/****************************************************************************/
/*                                                                          */
/* CLASS DEFINITION                                                         */
/*                                                                          */
/****************************************************************************/

/*****************************************************************************
 * data structure definition for read KDS value  'FM/AM Tuner Parameter Set'
 * https://hi-dms.de.bosch.com/docushare/dsweb/Services/Document-429364/LCN2_DiagnosticServices_2.0.0.pdf
 ****************************************************************************/


/*============================================================================
 * Base class for KDS entries
 *--------------------------------------------------------------------------*/
struct tun_trKds
{
  virtual tVoid vSetToDefaultValues( tVoid) = 0;
  virtual tVoid vParse( tPU8 pu8KdsBuffer, tU32 u32Length) = 0;
  virtual tVoid vTrace( tVoid) = 0;
};
/*==========================================================================*/


#define TUN_CONFIG_VERSIONID_LEN              11 /* enKdsFmAmTunerParameter_len_Tag + enKdsFmAmTunerParameter_len_MajorVersion + enKdsFmAmTunerParameter_len_MinorVersion */


/*==========================================================================*/

/*============================================================================
 * KDS type Tuner area
 *--------------------------------------------------------------------------*/
typedef enum
{
  enTunerArea_Europe                 = 0x00,
  enTunerArea_Russia                 = 0x01,
  enTunerArea_North_America          = 0x02,
  enTunerArea_South_Africa           = 0x03,
  enTunerArea_Golf_Coast             = 0x04,
  enTunerArea_Australia_and_Oceania  = 0x05,
  enTunerArea_South_America          = 0x06,
  enTunerArea_Thailand               = 0x07
} tenTunerArea;
/*==========================================================================*/

/*============================================================================
 * KDS type FM Antenna Type
 *--------------------------------------------------------------------------*/
typedef enum
{
  enFmAntenna_Passive_1                       = 0x00,
  enFmAntenna_Active_AGC_1                    = 0x01,
  enFmAntenna_Active_ELCAA                    = 0x02,
  enFmAntenna_Passive_and_AGC_Diversity       = 0x03,
  enFmAntenna_Passive_2                       = 0x04,
  enFmAntenna_Active_AGC_2                    = 0x05,
  enFmAntenna_Active_AGC_with_phantom_supply  = 0x06,
  enFmAntenna_New_2                           = 0x07
} tenFmAntenna;
/*==========================================================================*/

/*============================================================================
 * KDS type SeekMode Type
 *--------------------------------------------------------------------------*/
typedef enum
{
  enSeekMode_PIstepping      = 0x00,
  enSeekMode_frequencySearch = 0x01
} tenSeekMode;
/*==========================================================================*/

/*============================================================================
 * KDS type DDA_Switch
 *--------------------------------------------------------------------------*/
typedef enum
{
  enDDA_Switch_disable            = 0x00,
  enDDA_Switch_enable             = 0x01,
  enDDA_Switch_forcedON           = 0x02
} tenDDA_Switch;
/*==========================================================================*/

/*============================================================================
 * KDS type REG_Switch
 *--------------------------------------------------------------------------*/
typedef enum
{
  enREG_Switch_OFF                = 0x00,
  enREG_Switch_AUTO               = 0x01,
  enREG_Switch_ON_FIX             = 0x04
} tenREG_Switch;
/*==========================================================================*/

/*============================================================================
 * KDS type Level Type 
 *--------------------------------------------------------------------------*/
typedef enum
{
  enLevelType_fieldstrenght      = 0x00,
  enLevelType_quality            = 0x01
} tenLevelType;
/*==========================================================================*/


/*============================================================================
 * Autostore data
 *History
 *9/3/2013  | ngp1kor, pmn3kor | added data types for autostore
 *10/9/2013 |  ngp1kor, pmn3kor| added project variants for defines.
 *--------------------------------------------------------------------------*/
typedef enum
{
    enAMFMAutostore_UnUsedPreset_KeepSame = 0,
    enAMFMAutostore_UnUsedPreset_DeleteUnUsedStations = 1,
    enAMFMAutostore_UnUsedPreset_SetToLowestFreq = 2
}tenAMFMAutostore_UnUsedPreset;

typedef enum
{
    enAMFMAutostore_Sort_Fieldstrength = 0,
    enAMFMAutostore_Sort_Quality = 1,
    enAMFMAutostore_Sort_Frequency = 2,
    enAMFMAutostore_Sort_PIChain = 3
}tenAMFMAutostore_Sort;



/*==========================================================================*/


/*============================================================================
 * KDS type FmDabHdTunerMode
 *--------------------------------------------------------------------------*/
typedef enum
{
/**Changed from 00 to FF to support ADR3 for handling the default configuration as KDS is not available.*/
  enFmDabHdTunerMode_reserved_0x00					= 0xff,
  enFmDabHdTunerMode_TunVar1_2FE_DDA				= 0x01,
  enFmDabHdTunerMode_TunVar2_1FE					= 0x02,
  enFmDabHdTunerMode_TunVar3_2FE					= 0x03,
  enFmDabHdTunerMode_reserved_0x04					= 0x04,
  enFmDabHdTunerMode_TunVar5_3FE_DDA				= 0x05,
  enFmDabHdTunerMode_TunVar6_3FE_DAB				= 0x06,
  enFmDabHdTunerMode_TunVar7_4FE_DDA_DAB			= 0x07,
  enFmDabHdTunerMode_TunVar8_2FE_HD					= 0x08,
  enFmDabHdTunerMode_TunVar9_2FE_DDA_HD				= 0x09,
  enFmDabHdTunerMode_TunVar10_3FE_DDA_HD			= 0x0A,
  enFmDabHdTunerMode_TunVar11_3FE_DAB_SBM			= 0x0B,
  enFmDabHdTunerMode_TunVar12_4FE_DDA_DAB_SBM		= 0x0C,
  enFmDabHdTunerMode_TunVar13_2FE_DDA_DDS			= 0x0D,
  enFmDabHdTunerMode_TunVar14_3FE_DDA_DDS_DAB		= 0x0E,
  enFmDabHdTunerMode_TunVar15_3FE_DDA_DDS_DAB		= 0x0F,
  enFmDabHdTunerMode_TunVar16_2FE					= 0x10,
  enFmDabHdTunerMode_TunVar17_3FE_DAB				= 0x11,
  enFmDabHdTunerMode_TunVar18_2FE_DDA_DDS			= 0x12,
  enFmDabHdTunerMode_TunVar19_2FE_DDA_DDS			= 0x13,
  enFmDabHdTunerMode_TunVar20_3FE_DDA_DDS_DAB		= 0x14,
  enFmDabHdTunerMode_TunVar21_3FE_DDA_DDS_DAB		= 0x15,
  enFmDabHdTunerMode_TunVar22_3FE_DDA_DDS_DAB		= 0x16,
  enFmDabHdTunerMode_TunVar23						= 0x17,
  enFmDabHdTunerMode_TunVar24						= 0x18,
  enFmDabHdTunerMode_TunVar25						= 0x19,
  enFmDabHdTunerMode_TunVar26						= 0x1A,
  enFmDabHdTunerMode_TunVar27						= 0x1B,
  enFmDabHdTunerMode_TunVar28						= 0x1C,
  enFmDabHdTunerMode_TunVar29						= 0x1D,
  enFmDabHdTunerMode_TunVar30						= 0x1E,
  enFmDabHdTunerMode_TunVar31						= 0x1F,
  enFmDabHdTunerMode_TunVar32_2FE_DDA				= 0x20,
  enFmDabHdTunerMode_TunVar33_3FE_DDA_DAB			= 0x21,
  enFmDabHdTunerMode_TunVar34_2FE_DAB				= 0x22,
  enFmDabHdTunerMode_TunVar50                       = 0x32,
  enFmDabHdTunerMode_TunVar46_2FE_AM_1FE_DDA_HD					= 0x2E
} tenFmDabHdTunerMode;
/*==========================================================================*/



typedef enum
{
	FMTunerBand_NOSET = 0,
	FMTunerBand_EU_RDW,
	FMTunerBand_NAR,
	FMTunerBand_JP,
	FMTunerBand_KOR,
	FMTunerBand_CN,
	FMTunerBand_RESERVE = 255
}tenFMTunerBand;

typedef enum
{
	AMTunerBand_NOSET = 0,
	AMTunerBand_EU_RDW,
	AMTunerBand_NAR,
	AMTunerBand_JP,
	AMTunerBand_EU,
	AMTunerBand_AUS,
	AMTunerBand_RESERVE = 255
}tenAMTunerBand;


#if FEATURE_VW_MIBE

#define TUN_CONFIG_KDS_CM_EOL_PARAMETER_LENGTH			25
#define TUN_CONFIG_KDS_CM_EOL_PARAMETER					0x0D10

typedef enum
{
  TUN_KDS_AM_QUALITY_LEVEL_OFFSET,
  TUN_KDS_FM_QUALITY_LEVEL_OFFSET
} tenQualityLevelOffsetParam;

typedef enum
{
  TUN_KDS_PROD_MODE_BYTE_ONE,
  TUN_KDS_PROD_MODE_BYTE_TWO,
  TUN_KDS_PROD_MODE_BYTE_THREE
} tenKDSProdModeParam;


struct tun_trKdsQualityLevelOffset : public tun_trKds
{
  tBool m_bReadKDSQualityLevelOffset;
	
  tU8 m_u8AMQualityLevel;
  tU8 m_u8FMQualityLevel;

  tun_trKdsQualityLevelOffset( tVoid);

  virtual tVoid vSetToDefaultValues( tVoid);
  virtual tVoid vParse( tPU8 pu8KdsBuffer, tU32 u32Length);
  virtual tVoid vTrace( tVoid);

  tU8 u8GetAMQualityLevelOffset()const {return m_u8AMQualityLevel;}
  tU8 u8GetFMQualityLevelOffset()const {return m_u8FMQualityLevel;}
};

#define TUN_CONFIG_KDS_QUALITYLEVEL_OFFSET_LENGTH					2
#define TUN_CONFIG_KDS_QUALITYLEVEL_OFFSET							0x0D92	


struct tun_trKdsProdModeSettings : public tun_trKds
{
  tBool m_bReadKDSProdModeSettings;
	
  tU8 m_u8ProdModeByte1,m_u8ProdModeByte2,m_u8ProdModeByte3;
  
  tun_trKdsProdModeSettings( tVoid);

  virtual tVoid vSetToDefaultValues( tVoid);
  virtual tVoid vParse( tPU8 pu8KdsBuffer, tU32 u32Length);
  virtual tVoid vTrace( tVoid);


};

#define TUN_CONFIG_KDS_PRODUCTION_MODE_SETTINGS_LENGTH				3
#define TUN_CONFIG_KDS_PRODUCTION_MODE_SETTINGS						0x0D32	

struct tun_trKdsEOLParameter : public tun_trKds
{
	tBool m_bReadKDSData;

	/**Byte 9 and BIT 0-3*/
	tU8 m_u8FMTunerBand;
	/**Byte 9 and BIT 4-7*/
	tU8 m_u8AMTunerBand;
	/**Byte 12 and BIT 0*/
	tBool m_bFMAntenna;
	/**Byte 13 and BIT 1*/
	tBool m_bAFTemporarySwitch;
	/**Byte 13 and BIT 4*/
	tBool m_bPIIgnoreSwitch;
	/**Byte 14 and BIT 0*/
	tBool m_bPTY31AlarmSwitch;
	/**Byte 14 and BIT 3*/
	tBool m_bMultipleEntrySwitch;
	/**Byte 14 and BIT 4*/
	tBool m_bRDSDeactivation;
	/**Byte 14 and BIT 5*/
	tBool m_bAFDeactivation;


	tun_trKdsEOLParameter( tVoid);


  virtual tVoid vSetToDefaultValues( tVoid);
  virtual tVoid vParse( tPU8 pu8KdsBuffer, tU32 u32Length);
  virtual tVoid vTrace( tVoid);

  /**EOL Parameter */
  /**Returns FM Tuner band*/
  tU8 u8GetFMTunerBand() const;
  /**Returns AM Tuner Band*/
  tU8 u8GetAMTunerBand() const;
  /**Returns AF Temporary switch status*/
  tBool bGetAFTemporarySwitch() const;
  /**Returns PI Ignore switch status*/
  tBool bGetPIIgnoreSwitch() const;
  /**Returns Multipleentry switch status*/
  tBool bGetMultipleEntrySwitch() const;
  /**Returns PTY31 switch status*/
  tBool bGetPTY31Switch() const;
  /**Returns RDS Deactivation status*/
  tBool bGetRDSDeactivation() const;
  /**Returns AF Deactivation status*/
  tBool bGetAFDeactivation() const;
  /**Returns TRUE if Only one antenna is available else 2 antennas are available*/
  tBool bGetFMAntenna() const;


};
#endif



#if FEATURE_PSA_RCC

#define TUN_CONFIG_KDS_CM_EOL_PARAMETER_LENGTH			4

struct tun_trKdsEOLConfigRadioParameter
{
	tU8  m_u8Country_Code; 		/*Byte 3 and bit 7*/
	tU8  m_u8AMAvail;     		/*Byte 3 and bit 4*/
	tU8  m_u8RDS_FollowUP;

	tun_trKdsEOLConfigRadioParameter( tVoid);

	tVoid vSetToDefaultValues( tVoid);
	tVoid vParse( tVoid);
	tVoid vTrace( tVoid);


	  tU8  u8GetCountryCode() const;
	  tU8  u8GetAMAvail() const;
	  tU8  u8GetRDSFollowUP() const;

};


struct tun_trKdsEOLConfigAudioParameter
{
	tU8 m_u8SeekSensitivity;     /* bit 10 and bit 11 */

	tun_trKdsEOLConfigAudioParameter( tVoid);

	tVoid vSetToDefaultValues( tVoid);
	tVoid vParse( tVoid);
	tVoid vTrace( tVoid);


	tU8  u8GetSeekSensitivity() const;

};
#endif


/*============================================================================
 * Indices for New KDS byte stream 'FM/AM Tuner Parameter Set'
 *--------------------------------------------------------------------------*/
typedef enum
{
  enNewKdsFmAmTunerParameter_idx_Tag                        =   0, enNewKdsFmAmTunerParameter_len_Tag                        =   8,
  enNewKdsFmAmTunerParameter_idx_MajorVersion               =   8, enNewKdsFmAmTunerParameter_len_MajorVersion               =   1,
  enNewKdsFmAmTunerParameter_idx_MinorVersion               =   9, enNewKdsFmAmTunerParameter_len_MinorVersion               =   2,
  enNewKdsFmAmTunerParameter_idx_ReservedByte1              =  11, enNewKdsFmAmTunerParameter_len_ReservedByte1              =   1,
  enNewKdsFmAmTunerParameter_idx_LowFM                      =  12, enNewKdsFmAmTunerParameter_len_LowFM                      =   4,
  enNewKdsFmAmTunerParameter_idx_HighFM                     =  16, enNewKdsFmAmTunerParameter_len_HighFM                     =   4,
  enNewKdsFmAmTunerParameter_idx_StepSizeFM                 =  20, enNewKdsFmAmTunerParameter_len_StepSizeFM                 =   1,
  enNewKdsFmAmTunerParameter_idx_StepSizeManualFM           =  21, enNewKdsFmAmTunerParameter_len_StepSizeManualFM           =   1,
  enNewKdsFmAmTunerParameter_idx_LowMW                      =  22, enNewKdsFmAmTunerParameter_len_LowMW                      =   4,
  enNewKdsFmAmTunerParameter_idx_HighMW                     =  26, enNewKdsFmAmTunerParameter_len_HighMW                     =   4,
  enNewKdsFmAmTunerParameter_idx_StepSizeMW                 =  30, enNewKdsFmAmTunerParameter_len_StepSizeMW                 =   1,
  enNewKdsFmAmTunerParameter_idx_StepSizeManualMW           =  31, enNewKdsFmAmTunerParameter_len_StepSizeManualMW           =   1,
  enNewKdsFmAmTunerParameter_idx_LowLW                      =  32, enNewKdsFmAmTunerParameter_len_LowLW                      =   4,
  enNewKdsFmAmTunerParameter_idx_HighLW                     =  36, enNewKdsFmAmTunerParameter_len_HighLW                     =   4,
  enNewKdsFmAmTunerParameter_idx_StepSizeLW                 =  40, enNewKdsFmAmTunerParameter_len_StepSizeLW                 =   1,
  enNewKdsFmAmTunerParameter_idx_StepSizeManualLW           =  41, enNewKdsFmAmTunerParameter_len_StepSizeManualLW           =   1,
  enNewKdsFmAmTunerParameter_idx_FeatureControlI            =  42, enNewKdsFmAmTunerParameter_len_FeatureControlI            =   1,
  enNewKdsFmAmTunerParameter_idx_FeatureControlII           =  43, enNewKdsFmAmTunerParameter_len_FeatureControlII           =   1,
  enNewKdsFmAmTunerParameter_idx_FM_LSM                     =  44, enNewKdsFmAmTunerParameter_len_FM_LSM                     =   4,
  enNewKdsFmAmTunerParameter_idx_MW_LSM                     =  48, enNewKdsFmAmTunerParameter_len_MW_LSM                     =   4,
  enNewKdsFmAmTunerParameter_idx_LW_LSM                     =  52, enNewKdsFmAmTunerParameter_len_LW_LSM                     =   4,
  enNewKdsFmAmTunerParameter_idx_ShowRadioTextStatusMessage =  56, enNewKdsFmAmTunerParameter_len_ShowRadioTextStatusMessage =   1,
  enNewKdsFmAmTunerParameter_idx_T_RadioTextNotAvailable    =  57, enNewKdsFmAmTunerParameter_len_T_RadioTextNotAvailable    =   1,
  enNewKdsFmAmTunerParameter_idx_T_Timeout_Status_RT        =  58, enNewKdsFmAmTunerParameter_len_T_Timeout_Status_RT        =   1,
  enNewKdsFmAmTunerParameter_idx_T_ShowTextMin              =  59, enNewKdsFmAmTunerParameter_len_T_ShowTextMin              =   1,
  enNewKdsFmAmTunerParameter_idx_T_MinDisplayDurationPerRadioTextPage =  60, enNewKdsFmAmTunerParameter_len_T_MinDisplayDurationPerRadioTextPage =   2,
  enNewKdsFmAmTunerParameter_idx_RadioTextMinLength         =  62, enNewKdsFmAmTunerParameter_len_RadioTextMinLength         =   1,
  enNewKdsFmAmTunerParameter_idx_RadioTextSpeedThreshold    =  63, enNewKdsFmAmTunerParameter_len_RadioTextSpeedThreshold    =   1,
  enNewKdsFmAmTunerParameter_idx_RadioTextSpeedThresholdHysteresis    =  64, enNewKdsFmAmTunerParameter_len_RadioTextSpeedThresholdHysteresis    =   1,
  enNewKdsFmAmTunerParameter_idx_ReservedByte2              =  65, enNewKdsFmAmTunerParameter_len_ReservedByte2              =   2,
  enNewKdsFmAmTunerParameter_idx_EmergencySeekScanNoOfStations        =  67, enNewKdsFmAmTunerParameter_len_EmergencySeekScanNoOfStations        =   1,
  enNewKdsFmAmTunerParameter_idx_ShowFreqDuringFreqScan     =  68, enNewKdsFmAmTunerParameter_len_ShowFreqDuringFreqScan     =   1,
  enNewKdsFmAmTunerParameter_idx_SeekMode                   =  69, enNewKdsFmAmTunerParameter_len_SeekMode                   =   1,
  enNewKdsFmAmTunerParameter_idx_NumOfSeekPeriods           =  70, enNewKdsFmAmTunerParameter_len_NumOfSeekPeriods           =   1,
  enNewKdsFmAmTunerParameter_idx_ScanTime                   =  71, enNewKdsFmAmTunerParameter_len_ScanTime                   =   1,
  enNewKdsFmAmTunerParameter_idx_AFBestDelayTimeAftManualTuning       =  72, enNewKdsFmAmTunerParameter_len_AFBestDelayTimeAftManualTuning       =   1,
  enNewKdsFmAmTunerParameter_idx_AFBestDisplayFreezeAftManualTuning   =  73, enNewKdsFmAmTunerParameter_len_AFBestDisplayFreezeAftManualTuning   =   1,
  enNewKdsFmAmTunerParameter_idx_AutoSeekStepDelay          =  74, enNewKdsFmAmTunerParameter_len_AutoSeekStepDelay          =   2,
  enNewKdsFmAmTunerParameter_idx_PemanentManualStepDelay    =  76, enNewKdsFmAmTunerParameter_len_PemanentManualStepDelay    =   2,
  enNewKdsFmAmTunerParameter_idx_WrongPIAcceptDelayAftStnSelection    =  78, enNewKdsFmAmTunerParameter_len_WrongPIAcceptDelayAftStnSelection    =   1,
  enNewKdsFmAmTunerParameter_idx_WrongPIAcceptAftStnlost    =  79, enNewKdsFmAmTunerParameter_len_WrongPIAcceptAftStnlost    =   1,
  enNewKdsFmAmTunerParameter_idx_MultiManualStepsWithIntermediateFreq =  80, enNewKdsFmAmTunerParameter_len_MultiManualStepsWithIntermediateFreq =   1,
  enNewKdsFmAmTunerParameter_idx_StationLists				=  81, enNewKdsFmAmTunerParameter_len_StationLists				 =   1,
  enNewKdsFmAmTunerParameter_idx_FMListSortingDefault		=  82, enNewKdsFmAmTunerParameter_len_FMListSortingDefault		 =   1,
  enNewKdsFmAmTunerParameter_idx_AMListSortingDefault		=  83, enNewKdsFmAmTunerParameter_len_AMListSortingDefault		 =   1,
  enNewKdsFmAmTunerParameter_idx_TimerInhibitLstSortAfterSkip		  =  84, enNewKdsFmAmTunerParameter_len_TimerInhibitLstSortAfterSkip 		 =   1,
  enNewKdsFmAmTunerParameter_idx_TimerCursorDelayAfterStnSel		  =  85, enNewKdsFmAmTunerParameter_len_TimerCursorDelayAfterStnSel			 =   1,
  enNewKdsFmAmTunerParameter_idx_TimeAMMinTuneListRemove	=  86, enNewKdsFmAmTunerParameter_len_TimeAMMinTuneListRemove	 =   1,
  enNewKdsFmAmTunerParameter_idx_TimeAMStnRemoveAftDeslect	=  87, enNewKdsFmAmTunerParameter_len_TimeAMStnRemoveAftDeslect	 =   1,
  enNewKdsFmAmTunerParameter_idx_AMStnListUpdateStrategy	=  88, enNewKdsFmAmTunerParameter_len_AMStnListUpdateStrategy	 =   1,
  enNewKdsFmAmTunerParameter_idx_ReservedByte3				=  89, enNewKdsFmAmTunerParameter_len_ReservedByte3				 =   4,
  enNewKdsFmAmTunerParameter_idx_TestModeDataUpdateInterval	=  93, enNewKdsFmAmTunerParameter_len_TestModeDataUpdateInterval =   1,
  enNewKdsFmAmTunerParameter_idx_AFListUpdateInterval		=  94, enNewKdsFmAmTunerParameter_len_AFListUpdateInterval		 =   1,
  enNewKdsFmAmTunerParameter_idx_AFListResort				=  95, enNewKdsFmAmTunerParameter_len_AFListResort				 =   1,
  enNewKdsFmAmTunerParameter_idx_AFListReleaseBehaviour		=  96, enNewKdsFmAmTunerParameter_len_AFListReleaseBehaviour	 =   1,
  enNewKdsFmAmTunerParameter_idx_LandscapeUpdateOnShutdown	=  97, enNewKdsFmAmTunerParameter_len_LandscapeUpdateOnShutdown	 =   1,
  enNewKdsFmAmTunerParameter_idx_TunerSwitches				=  98, enNewKdsFmAmTunerParameter_len_TunerSwitches				 =   1,
  enNewKdsFmAmTunerParameter_idx_ADRFmChannelSeparation     =  99, enNewKdsFmAmTunerParameter_len_ADRFmChannelSeparation	 =   1,
  enNewKdsFmAmTunerParameter_idx_ADRFmLimiting              = 100, enNewKdsFmAmTunerParameter_len_ADRFmLimiting				 =   1,
  enNewKdsFmAmTunerParameter_idx_ADRFmMultipath             = 101, enNewKdsFmAmTunerParameter_len_ADRFmMultipath             =   1,
  enNewKdsFmAmTunerParameter_idx_ADRFmHighCut               = 102, enNewKdsFmAmTunerParameter_len_ADRFmHighCut               =   1,
  enNewKdsFmAmTunerParameter_idx_ADRFmSoftMute              = 103, enNewKdsFmAmTunerParameter_idx_len_ADRFmSoftMute          =   1,
  enNewKdsFmAmTunerParameter_idx_ADRAmSoftMute              = 104, enNewKdsFmAmTunerParameter_len_ADRAmSoftMute              =   1,
  enNewKdsFmAmTunerParameter_idx_ADRFmDeemphasis            = 105, enNewKdsFmAmTunerParameter_len_ADRFmDeemphasis            =   1,
  enNewKdsFmAmTunerParameter_idx_ADRFmFrequencyResponse     = 106, enNewKdsFmAmTunerParameter_len_ADRFmFrequencyResponse     =   1,
  enNewKdsFmAmTunerParameter_idx_ADRAmFrequencyResponse     = 107, enNewKdsFmAmTunerParameter_len_ADRAmFrequencyResponse     =   1,
  enNewKdsFmAmTunerParameter_idx_ADRFmMisc                  = 108, enNewKdsFmAmTunerParameter_len_ADRFmMisc                  =   1,
  enNewKdsFmAmTunerParameter_idx_ADRAmMisc                  = 109, enNewKdsFmAmTunerParameter_len_ADRAmMisc                  =   1,
  enNewKdsFmAmTunerParameter_idx_ADRFmAmplificationLevel    = 110, enNewKdsFmAmTunerParameter_len_ADRFmAmplificationLevel    =   1,
  enNewKdsFmAmTunerParameter_idx_ADRAmAmplificationLevel    = 111, enNewKdsFmAmTunerParameter_len_ADRAmAmplificationLevel    =   1,
  enNewKdsFmAmTunerParameter_idx_ADRAM_Hicut_Configuration  = 112, enNewKdsFmAmTunerParameter_len_ADRAM_Hicut_Configuration  =   1,
  enNewKdsFmAmTunerParameter_idx_ADRAM_Deemphasis           = 113, enNewKdsFmAmTunerParameter_len_ADRAM_Deemphasis           =   1,
  enNewKdsFmAmTunerParameter_idx_MUTE_RAMP_1                = 114, enNewKdsFmAmTunerParameter_len_MUTE_RAMP_1                =   2,
  enNewKdsFmAmTunerParameter_idx_DEMUTE_RAMP_1              = 116, enNewKdsFmAmTunerParameter_len_DEMUTE_RAMP_1              =   2,
  enNewKdsFmAmTunerParameter_idx_MUTE_RAMP_2                = 118, enNewKdsFmAmTunerParameter_len_MUTE_RAMP_2                =   2,
  enNewKdsFmAmTunerParameter_idx_DEMUTE_RAMP_2              = 120, enNewKdsFmAmTunerParameter_len_DEMUTE_RAMP_2              =   2,
  enNewKdsFmAmTunerParameter_idx_BehaviourSwitches          = 122, enNewKdsFmAmTunerParameter_len_BehaviourSwitches          =   1,

  enNewKdsFmAmTunerParameter_idx_SeekLevel_FM_RDS_FldStrength      = 123, enNewKdsFmAmTunerParameter_len_SeekLevel_FM_RDS_FldStrength    =   1,
  enNewKdsFmAmTunerParameter_idx_SeekLevel_FM_RDS_Quality		   = 124, enNewKdsFmAmTunerParameter_len_SeekLevel_FM_RDS_Quality		 =   1,
  enNewKdsFmAmTunerParameter_idx_SeekLevel_FM_NONRDS_FldStrength   = 125, enNewKdsFmAmTunerParameter_len_SeekLevel_FM_NONRDS_FldStrength =   1,
  enNewKdsFmAmTunerParameter_idx_SeekLevel_FM_NONRDS_Quality       = 126, enNewKdsFmAmTunerParameter_len_SeekLevel_FM_NONRDS_Quality     =   1,
  enNewKdsFmAmTunerParameter_idx_SeekLevel_LW_FldStrength          = 127, enNewKdsFmAmTunerParameter_len_SeekLevel_LW_FldStrength        =   1,
  enNewKdsFmAmTunerParameter_idx_SeekLevel_LW_Quality              = 128, enNewKdsFmAmTunerParameter_len_SeekLevel_LW_Quality            =   1,
  enNewKdsFmAmTunerParameter_idx_SeekLevel_MW_FldStrength          = 129, enNewKdsFmAmTunerParameter_len_SeekLevel_MW_FldStrength        =   1,
  enNewKdsFmAmTunerParameter_idx_SeekLevel_MW_Quality              = 130, enNewKdsFmAmTunerParameter_len_SeekLevel_MW_Quality            =   1,
  enNewKdsFmAmTunerParameter_idx_SeekLevel_SW_FldStrength          = 131, enNewKdsFmAmTunerParameter_len_SeekLevel_SW_FldStrength        =   1,
  enNewKdsFmAmTunerParameter_idx_SeekLevel_SW_Quality              = 132, enNewKdsFmAmTunerParameter_len_SeekLevel_SW_Quality            =   1,
  enNewKdsFmAmTunerParameter_idx_SeekLevel_TRF_FldStrength         = 133, enNewKdsFmAmTunerParameter_len_SeekLevel_TRF_FldStrength       =   1,
  enNewKdsFmAmTunerParameter_idx_SeekLevel_TRF_Quality             = 134, enNewKdsFmAmTunerParameter_len_SeekLevel_TRF_Quality           =   1,
  enNewKdsFmAmTunerParameter_idx_SeekLevel_WB_FldStrength          = 135, enNewKdsFmAmTunerParameter_len_SeekLevel_WB_FldStrength        =   1,
  enNewKdsFmAmTunerParameter_idx_SeekLevel_WB_Quality              = 136, enNewKdsFmAmTunerParameter_len_SeekLevel_WB_Quality            =   1,

  enNewKdsFmAmTunerParameter_idx_ListLevel_FM_RDS_FldStrength      = 137, enNewKdsFmAmTunerParameter_len_ListLevel_FM_RDS_FldStrength    =   1,
  enNewKdsFmAmTunerParameter_idx_ListLevel_FM_RDS_Quality          = 138, enNewKdsFmAmTunerParameter_len_ListLevel_FM_RDS_Quality		 =   1,
  enNewKdsFmAmTunerParameter_idx_ListLevel_FM_NONRDS_FldStrength   = 139, enNewKdsFmAmTunerParameter_len_ListLevel_FM_NONRDS_FldStrength =   1,
  enNewKdsFmAmTunerParameter_idx_ListLevel_FM_NONRDS_Quality       = 140, enNewKdsFmAmTunerParameter_len_ListLevel_FM_NONRDS_Quality     =   1,
  enNewKdsFmAmTunerParameter_idx_ListLevel_LW_FldStrength          = 141, enNewKdsFmAmTunerParameter_len_ListLevel_LW_FldStrength        =   1,
  enNewKdsFmAmTunerParameter_idx_ListLevel_LW_Quality              = 142, enNewKdsFmAmTunerParameter_len_ListLevel_LW_Quality            =   1,
  enNewKdsFmAmTunerParameter_idx_ListLevel_MW_FldStrength          = 143, enNewKdsFmAmTunerParameter_len_ListLevel_MW_FldStrength        =   1,
  enNewKdsFmAmTunerParameter_idx_ListLevel_MW_Quality              = 144, enNewKdsFmAmTunerParameter_len_ListLevel_MW_Quality            =   1,
  enNewKdsFmAmTunerParameter_idx_ListLevel_SW_FldStrength          = 145, enNewKdsFmAmTunerParameter_len_ListLevel_SW_FldStrength        =   1,
  enNewKdsFmAmTunerParameter_idx_ListLevel_SW_Quality              = 146, enNewKdsFmAmTunerParameter_len_ListLevel_SW_Quality            =   1,
  enNewKdsFmAmTunerParameter_idx_ListLevel_TRF_FldStrength         = 147, enNewKdsFmAmTunerParameter_len_ListLevel_TRF_FldStrength       =   1,
  enNewKdsFmAmTunerParameter_idx_ListLevel_TRF_Quality             = 148, enNewKdsFmAmTunerParameter_len_ListLevel_TRF_Quality           =   1,
  enNewKdsFmAmTunerParameter_idx_ListLevel_WB_FldStrength          = 149, enNewKdsFmAmTunerParameter_len_ListLevel_WB_FldStrength        =   1,
  enNewKdsFmAmTunerParameter_idx_ListLevel_WB_Quality              = 150, enNewKdsFmAmTunerParameter_len_ListLevel_WB_Quality            =   1,
  
 
  enNewKdsFmAmTunerParameter_idx_SHARX                      = 151, enNewKdsFmAmTunerParameter_len_SHARX                      =   1,
  enNewKdsFmAmTunerParameter_idx_HICUT                      = 152, enNewKdsFmAmTunerParameter_len_HICUT                      =   1,
  enNewKdsFmAmTunerParameter_idx_AvailableBands             = 153, enNewKdsFmAmTunerParameter_len_AvailableBands             =   4,
  enNewKdsFmAmTunerParameter_idx_CombiBands					= 157, enNewKdsFmAmTunerParameter_len_CombiBands				 =   4,
 
  enNewKdsFmAmTunerParameter_idx_ADRSwitches1               = 161, enNewKdsFmAmTunerParameter_len_ADRSwitches1               =   1,
  enNewKdsFmAmTunerParameter_idx_ADRSwitches2               = 162, enNewKdsFmAmTunerParameter_len_ADRSwitches2               =   1,
  enNewKdsFmAmTunerParameter_idx_ADRSwitches3               = 163, enNewKdsFmAmTunerParameter_len_ADRSwitches3               =   1,
  enNewKdsFmAmTunerParameter_idx_ADRSwitches4               = 164, enNewKdsFmAmTunerParameter_len_ADRSwitches4               =   1,

  
  enNewKdsFmAmTunerParameter_idx_PhysicalTunerConfigId      = 165, enNewKdsFmAmTunerParameter_len_PhysicalTunerConfigId      =   1,
  enNewKdsFmAmTunerParameter_idx_AntennaConfigId            = 166, enNewKdsFmAmTunerParameter_len_AntennaConfigId            =   1,
  enNewKdsFmAmTunerParameter_idx_EXCHANGE_ANTENNA_LEVEL     = 167, enNewKdsFmAmTunerParameter_len_EXCHANGE_ANTENNA_LEVEL     =   1,
  enNewKdsFmAmTunerParameter_idx_PITEST_LEVEL               = 168, enNewKdsFmAmTunerParameter_len_PITEST_LEVEL               =   1,
  enNewKdsFmAmTunerParameter_idx_NRDS_LEVEL                 = 169, enNewKdsFmAmTunerParameter_len_NRDS_LEVEL                 =   1,
  enNewKdsFmAmTunerParameter_idx_EON_LEVEL                  = 170, enNewKdsFmAmTunerParameter_len_EON_LEVEL                  =   1,
  enNewKdsFmAmTunerParameter_idx_FM_MEMO_LEVEL              = 171, enNewKdsFmAmTunerParameter_len_FM_MEMO_LEVEL              =   1,
  enNewKdsFmAmTunerParameter_idx_PANIC_FIELDST_LEVEL        = 172, enNewKdsFmAmTunerParameter_len_PANIC_FIELDST_LEVEL        =   1,
  enNewKdsFmAmTunerParameter_idx_PARAM_RDS_LEVEL            = 173, enNewKdsFmAmTunerParameter_len_PARAM_RDS_LEVEL            =   1,
  enNewKdsFmAmTunerParameter_idx_RDS_DATA_ACCEPT_LEVEL      = 174, enNewKdsFmAmTunerParameter_len_RDS_DATA_ACCEPT_LEVEL      =   1,
  enNewKdsFmAmTunerParameter_idx_PI_STEP_LEVEL              = 175, enNewKdsFmAmTunerParameter_len_PI_STEP_LEVEL              =   1,
  enNewKdsFmAmTunerParameter_idx_DDA_ANTENNA_DIST_LEVEL     = 176, enNewKdsFmAmTunerParameter_len_DDA_ANTENNA_DIST_LEVEL     =   1,
  enNewKdsFmAmTunerParameter_idx_RDS_SYN_DELAY_CNT          = 177, enNewKdsFmAmTunerParameter_len_RDS_SYN_DELAY_CNT          =   1,
  enNewKdsFmAmTunerParameter_idx_MP_MEASURE_LEVEL           = 178, enNewKdsFmAmTunerParameter_len_MP_MEASURE_LEVEL           =   1,
  enNewKdsFmAmTunerParameter_idx_MP_LEVEL2_LEVEL            = 179, enNewKdsFmAmTunerParameter_len_MP_LEVEL2_LEVEL            =   1,
  enNewKdsFmAmTunerParameter_idx_MP_VERYBAD_LEVEL           = 180, enNewKdsFmAmTunerParameter_len_MP_VERYBAD_LEVEL           =   1,
  enNewKdsFmAmTunerParameter_idx_MP_MEDIUM_LEVEL            = 181, enNewKdsFmAmTunerParameter_len_MP_MEDIUM_LEVEL            =   1,
  enNewKdsFmAmTunerParameter_idx_RDS_BAD_LEVEL              = 182, enNewKdsFmAmTunerParameter_len_RDS_BAD_LEVEL              =   1,
  enNewKdsFmAmTunerParameter_idx_ILLEGAL_AF                 = 183, enNewKdsFmAmTunerParameter_len_ILLEGAL_AF                 =   1,
  enNewKdsFmAmTunerParameter_idx_AF_SAVE_LEVEL              = 184, enNewKdsFmAmTunerParameter_len_AF_SAVE_LEVEL              =   1,
  enNewKdsFmAmTunerParameter_idx_FIRST_RUNDUM               = 185, enNewKdsFmAmTunerParameter_len_FIRST_RUNDUM               =   1,
  enNewKdsFmAmTunerParameter_idx_SECOND_RUNDUM              = 186, enNewKdsFmAmTunerParameter_len_SECOND_RUNDUM              =   1,
  enNewKdsFmAmTunerParameter_idx_THIRD_RUNDUM               = 187, enNewKdsFmAmTunerParameter_len_THIRD_RUNDUM               =   1,
  enNewKdsFmAmTunerParameter_idx_PISTEP_MODE                = 188, enNewKdsFmAmTunerParameter_len_PISTEP_MODE                =   1,
  enNewKdsFmAmTunerParameter_idx_AUDIO_OFF_SCAN_MODE        = 189, enNewKdsFmAmTunerParameter_len_AUDIO_OFF_SCAN_MODE        =   1,
  enNewKdsFmAmTunerParameter_idx_AM_LEARN_CYCLE             = 190, enNewKdsFmAmTunerParameter_len_AM_LEARN_CYCLE             =   1,
  enNewKdsFmAmTunerParameter_idx_PI_IGNORE                  = 191, enNewKdsFmAmTunerParameter_len_PI_IGNORE                  =   1,
  enNewKdsFmAmTunerParameter_idx_PI_SEARCH_TIME             = 192, enNewKdsFmAmTunerParameter_len_PI_SEARCH_TIME             =   1,
  enNewKdsFmAmTunerParameter_idx_DisableSeamlessBlending    = 193, enNewKdsFmAmTunerParameter_len_DisableSeamlessBlending    =   1,
  enNewKdsFmAmTunerParameter_idx_SL_ADJ_THRESHOLD           = 194, enNewKdsFmAmTunerParameter_len_SL_ADJ_THRESHOLD           =   1,
  enNewKdsFmAmTunerParameter_idx_SL_MP_THRESHOLD            = 195, enNewKdsFmAmTunerParameter_len_SL_MP_THRESHOLD            =   1,
  enNewKdsFmAmTunerParameter_idx_SPECIAL_FM_START_FREQUENCY = 196, enNewKdsFmAmTunerParameter_len_SPECIAL_FM_START_FREQUENCY =   1,
  enNewKdsFmAmTunerParameter_idx_FM_Sharx_Limit				= 197, enNewKdsFmAmTunerParameter_len_FM_Sharx_Limit			 =   1,
  enNewKdsFmAmTunerParameter_idx_HDScanEveryFrequency		= 198, enNewKdsFmAmTunerParameter_len_HDScanEveryFrequency       =   1,
  enNewKdsFmAmTunerParameter_idx_HW_SubConfig				= 199, enNewKdsFmAmTunerParameter_len_HW_SubConfig			     =   1,
  enNewKdsFmAmTunerParameter_idx_FM2DABDelayTime			= 200, enNewKdsFmAmTunerParameter_len_FM2DABDelayTime		     =   1,

  enNewKdsFmAmTunerParameter_idx_ReservedByte4				= 201, enNewKdsFmAmTunerParameter_len_ReservedByte4				 =   8,

  enNewKdsFmAmTunerParameter_idx_AMFMHicutExtender							= 209, enNewKdsFmAmTunerParameter_len_AMFMHicutExtender							=   1,
  enNewKdsFmAmTunerParameter_idx_FMSoftMuteTiming							= 210, enNewKdsFmAmTunerParameter_len_FMSoftMuteTiming							=   1,
  enNewKdsFmAmTunerParameter_idx_AMFMMisscellaneous2						= 211, enNewKdsFmAmTunerParameter_len_AMFMMisscellaneous2						=   1,
  enNewKdsFmAmTunerParameter_idx_FMHicutExtenderFieldstrengthThreshold		= 212, enNewKdsFmAmTunerParameter_len_FMHicutExtenderFieldstrengthThreshold		=   1,
  enNewKdsFmAmTunerParameter_idx_FMHicutExtenderFieldstrengthFilterOffset	= 213, enNewKdsFmAmTunerParameter_len_FMHicutExtenderFieldstrengthFilterOffset	=   1,
  enNewKdsFmAmTunerParameter_idx_FMChannelSeperationMaximum					= 214, enNewKdsFmAmTunerParameter_len_FMChannelSeperationMaximum				=   1,
  enNewKdsFmAmTunerParameter_idx_FMChannelSeperationTiming					= 215, enNewKdsFmAmTunerParameter_len_FMChannelSeperationTiming					=   1,
  enNewKdsFmAmTunerParameter_idx_FMHicutFieldstrengthThreshold				= 216, enNewKdsFmAmTunerParameter_len_FMHicutFieldstrengthThreshold				=   1,
  enNewKdsFmAmTunerParameter_idx_FMHicutTiming								= 217, enNewKdsFmAmTunerParameter_len_FMHicutTiming								=   1,
  enNewKdsFmAmTunerParameter_idx_FMDMQSensitivity							= 218, enNewKdsFmAmTunerParameter_len_FMDMQSensitivity							=   1,
  enNewKdsFmAmTunerParameter_idx_FMDeviationAdaption						= 219, enNewKdsFmAmTunerParameter_len_FMDeviationAdaption						=   1,
  enNewKdsFmAmTunerParameter_idx_AMNoiseBlanker228kHzSensitivity			= 220, enNewKdsFmAmTunerParameter_len_AMNoiseBlanker228kHzSensitivity			=   1,
  enNewKdsFmAmTunerParameter_idx_AMNoiseBlanker45_6kHzSensitivity			= 221, enNewKdsFmAmTunerParameter_len_AMNoiseBlanker45_6kHzSensitivity			=   1,
  enNewKdsFmAmTunerParameter_idx_AMHicutExtenderFieldstrengthThreshold		= 222, enNewKdsFmAmTunerParameter_len_AMHicutExtenderFieldstrengthThreshold		=   1,
  enNewKdsFmAmTunerParameter_idx_AMHicutExtenderFieldstrengthFilterOffset	= 223, enNewKdsFmAmTunerParameter_len_AMHicutExtenderFieldstrengthFilterOffset	=   1,
  
  enNewKdsFmAmTunerParameter_idx_ReservedByte5	= 224, enNewKdsFmAmTunerParameter_len_ReservedByte5	=   8,
  
  enNewKdsFmAmTunerParameter_idx_DemodulationSwitchAudioTuner1	= 232, enNewKdsFmAmTunerParameter_len_DemodulationSwitchAudioTuner1	=   1,
  enNewKdsFmAmTunerParameter_idx_DemodulationSwitchAudioTuner2	= 233, enNewKdsFmAmTunerParameter_len_DemodulationSwitchAudioTuner2	=   1,
  
  enNewKdsFmAmTunerParameter_idx_ReservedByte6	= 234, enNewKdsFmAmTunerParameter_len_ReservedByte6	=   3,

  enNewKdsFmAmTunerParameter_idx_TimerNonRDS_Detect			= 237, enNewKdsFmAmTunerParameter_len_TimerNonRDS_Detect		 =   1,
  enNewKdsFmAmTunerParameter_idx_TimerPS_DelAfterRDSLost	= 238, enNewKdsFmAmTunerParameter_len_TimerPS_DelAfterRDSLost    =   1,
  enNewKdsFmAmTunerParameter_idx_TimerScrollPSDetect		= 239, enNewKdsFmAmTunerParameter_len_TimerScrollPSDetect        =   1,

  enNewKdsFmAmTunerParameter_idx_TimerSpeedTuneTrigger			= 240, enNewKdsFmAmTunerParameter_len_TimerSpeedTuneTrigger			=   1,
  enNewKdsFmAmTunerParameter_idx_N_SpeedTuneTrigger				= 241, enNewKdsFmAmTunerParameter_len_N_SpeedTuneTrigger			=   1,
  enNewKdsFmAmTunerParameter_idx_N_SpeedTuneMultiplier_FM		= 242, enNewKdsFmAmTunerParameter_len_N_SpeedTuneMultiplier_FM      =   1,
  enNewKdsFmAmTunerParameter_idx_N_SpeedTuneMultiplier_AM		= 243, enNewKdsFmAmTunerParameter_len_N_SpeedTuneMultiplier_AM      =   1,
  enNewKdsFmAmTunerParameter_idx_NoOfPSChangesForScrollDetect	= 244, enNewKdsFmAmTunerParameter_len_NoOfPSChangesForScrollDetect	=   1,
  enNewKdsFmAmTunerParameter_idx_TimePTYHold					= 245, enNewKdsFmAmTunerParameter_len_TimePTYHold					=   1,

  enNewKdsFmAmTunerParameter_idx_ReservedByte7					= 246, enNewKdsFmAmTunerParameter_len_ReservedByte7			=   40,
  enNewKdsFmAmTunerParameter_idx_CRC							= 286, enNewKdsFmAmTunerParameter_len_CRC                   =   2

 } tenNewKdsFmAmTunerParameter;                                                                                         /* ------- */
#define TUN_CONFIG_NEW_KDS_FM_AM_TUNER_PARAMETER_LENGTH                                                                      288
#define TUN_CONFIG_NEW_KDS_FM_AM_TUNER_PARAMETER_START_POINT_1																0x0410
#define TUN_CONFIG_NEW_KDS_FM_AM_TUNER_PARAMETER_READ_LEN_1																     230
#define TUN_CONFIG_NEW_KDS_FM_AM_TUNER_PARAMETER_START_POINT_2																0x0411
#define TUN_CONFIG_NEW_KDS_FM_AM_TUNER_PARAMETER_READ_LEN_2																      57
#define TUN_CONFIG_VERSIONID_LEN              11 /* enKdsFmAmTunerParameter_len_Tag + enKdsFmAmTunerParameter_len_MajorVersion + enKdsFmAmTunerParameter_len_MinorVersion */


/*==========================================================================*/


/** -------------- New AMFMParameter Set Begins --------------------------------- */

struct tun_trNewKdsFmAmTunerParameter : public tun_trKds
{
  /** Check whether KDS data is read or no*/
  tBool bKdsDataRead;

  /** Parameter set idetification tag ---------------------  0 - 7 byte */
  tChar sTag[ 8];
  /** Major(1 byte) and Minor(2 byte) Version -------------	 8 - 10 byte */
  tChar sVersion[ 3];
 
  //tU8 u8ReservedByte1;/**------------------------------------  11th byte */

  tU8   u8TunerMaster; /** local */

  /** 0- RDS,1- AF,2- REG,3- PS,4- RADIOTEXT,5- DDA,6- DDS,7- DATASCAN */
  tU8   u8FeatureControlI;/**------------------------------  42nd byte */
  /** 0- TA,1- PTY31,2- AF TEMPORARY ON/OFF, 3-7 reserved */
  tU8   u8FeatureControlII;/**-----------------------------  43rd byte */

  /** ----------- Radio text related parameters -----------------------*/
  tU8   u8ShowRadioTextStatusMessage; /**-------------------  56th byte */
  tU8   u8T_RadioTextNotAvailable;/**-----------------------  57th byte */
  tU8   u8T_Timeout_Status_RT;/**---------------------------  58th byte */
  tU8   u8T_ShowTextMin;/**---------------------------------  59th byte */
  tU16  u16T_MinDisplayDurationPerRadioTextPage;/**---------  60 - 61 byte */  
  tU8   u8RadioTextMinLength;/**----------------------------  62nd byte */  
  tU8   u8RadioTextSpeedThreshold;/**-----------------------  63rd byte */  
  tU8   u8RadioTextSpeedThresholdHysteresis;/**-------------  64th byte */  
  //tU16  u16ReservedByte2;/**--------------------------------  65 - 66 byte */
  /** ----------- Radio text related parameters Ends ------------------*/  

  tU8   u8EmergencySeekScanNoOfStations;/**-----------------  67th byte */
  tU8   u8ShowFreqDuringFreqScan;/**------------------------  68th byte */
  tU8   u8SeekMode;/**--------------------------------------  69th byte */ 

  tU8   u8AFBestDelayTimeAftManualTuning;/**----------------  72nd byte */
  tU8   u8AFBestDisplayFreezeAftManualTuning;/**------------  73rd byte */

  tU8   u8WrongPIAcceptDelayAftStnSelection;/**-------------  78th byte */
  tU8   u8WrongPIAcceptAftStnlost;/**-----------------------  79th byte */
  /** 0- ON/OFF, 1-7 Reserved */
  tU8   u8MultiManualStepsWithIntermediateFreq;/**----------  80th byte */
  /** 0- Fm,1- MW,2- LW,3- MW+LW,4-7 reserved */
  tU8   u8StationLists;/**----------------------------------  81st byte */
  tU8   u8FMListSortingDefault;/**--------------------------  82nd byte */
  tU8   u8AMListSortingDefault;/**--------------------------  83rd byte */

  tU8   u8TimeAMMinTuneListRemove;/**-----------------------  86th byte */
  tU8   u8TimeAMStnRemoveAftDeslect;/**---------------------  87th byte */
  tU8   u8AMStnListUpdateStrategy;/**-----------------------  88th byte */
  //tU32  u32ReservedByte3;/**--------------------------------  89 -92 byte */

  tU8   u8LandscapeUpdateOnShutdown;/**---------------------  97th byte */
  tU8   u8TunerSwitches;/**---------------------------------  98th byte */

  tU8   u8BehaviourSwitches;/**-----------------------------  122nd byte */


  //tU8   u8ReservedByte3[39];/**-----------------------------  198-236 byte */

  tU8   u8TimerSpeedTuneTrigger;/**-------------------------  240th byte */
  tU8   u8N_SpeedTuneTrigger;/**----------------------------  241st byte */
  tU8   u8N_SpeedTuneMultiplier_FM;/**----------------------  242nd byte */
  tU8   u8N_SpeedTuneMultiplier_AM;/**----------------------  243rd byte */
  tU8   u8NoOfPSChangesForScrollDetect;/**------------------  244th byte */

  //tU8   u8ReservedByte4[40];/**-----------------------------  246-285 byte */
  tU16  u16CRC;/**------------------------------------------  286-287 byte */
  
//  tU8   u8TMCServer;
////  tU16  u16AntennaLevelopen0;
////  tU16  u16AntennaLevelClosed0;
////  tU16  u16AntennaLevelShort0;
////  tU16  u16AntennaLevelopen1;
////  tU16  u16AntennaLevelClosed1;
////  tU16  u16AntennaLevelShort1;
////  tenFmAntenna enFmAntennaType;
//  tU8   u8NoFMpresetBanks;
//  tU8   u8NoOfFMpresetPerBank;
//  tU8   u8NoAMpresetBanks1;
//  tU8   u8NoOfAMpresetPerBank1;
//  tU32  au32Preset[ TUN_CONFIG_NUMBER_OF_PRESETS];
//  tU8   u8HMIbehaviour;

  /*==========================================================================
  * Stage 1:
  * Tuning behaviour (not audio tuner dependent):
  *------------------------------------------------------------------------*/
  tU8   u8NumOfSeekPeriods;/**-----------------------------  70th byte */
  tU8   u8ScanTime;/**-------------------------------------  71st byte */
 
  tU32  u32PtyFilterForSeek; /** local */
  // tU8   u8AfBestDelayTimeAfterSeek;
  //tU8   u8TpSeekStrategy;
  tU16  u16AutoSeekStepDelay;/**---------------------------  74 - 75 byte */
  tU16  u16PemanentManualStepDelay;/**---------------------  76 - 77 byte */
  //tU8   u8WrongPiAcceptDelay;
  tU16  u16MUTE_RAMP_1;/**---------------------------------  114 - 115 byte */
  tU16  u16DEMUTE_RAMP_1;/**-------------------------------  116 - 117 byte */
  tU16  u16MUTE_RAMP_2;/**---------------------------------  118 - 119 byte */
  tU16  u16DEMUTE_RAMP_2;/**-------------------------------  120 - 121 byte */
  
  /*==========================================================================
  * Stage 2:
  * Behaviour of  ATstation info auto notification (not audio tuner dependent):
  *------------------------------------------------------------------------*/
  tU8   u8ATStationInfoSwitches; /** local */

  /*==========================================================================
  * Stage 3:
  * Timer Configuration (not audio tuner dependent):
  *------------------------------------------------------------------------*/

  tU8 u8TimerScanRadio; /** local */
  tU8 u8TimerInhibitLstSortAfterSkip;/**-------------------  84th byte */
  //tU8 u8TimerDelayTPstnSelAfterVol0;
  //tU8 u8TimerDelayNoTPstnFound;
  tU8 u8TimerNonRDS_Detect;/**-----------------------------  237th byte */
  tU8 u8TimerPS_DelAfterRDSLost;/**------------------------  238th byte */
  //tU8 u8TimerWrongPIAccept;
  tU8 u8TimerScrollPSDetect;/**----------------------------  239th byte */
  //tU8 u8TimerDelayTPstnSelAfterRDSLost;
  //tU8 u8TimerTAendAfterRDSLost;
  tU8 u8TimerAM_MinTunTime; /** local */
  tU8 u8TimerCursorDelayAfterStnSel;/**--------------------  85th byte */
  tU8 u8TimerAF_InhibitAfterManTune; /** local */
  tU8 u8TimerPTY_Hold;/**----------------------------------  245th byte */

  /*==========================================================================
  * Stage 4:
  * Parameters that influence the reception behaviour (not audio tuner dependent):
  *------------------------------------------------------------------------*/

  tU8 u8FieldStrengthSeekLevel_FM_RDS;/**------------------  123rd byte */
  tU8 u8QualitySeekLevel_FM_RDS;/**------------------------  124th byte */
  tU8 u8FieldStrengthSeekLevel_FM_NONRDS;/**---------------  125th byte */
  tU8 u8QualitySeekLevel_FM_NONRDS;/**---------------------  126th byte */
  tU8 u8FieldStrengthSeekLevel_LW;/**----------------------  127th byte */
  tU8 u8QualitySeekLevel_LW;/**----------------------------  128th byte */
  tU8 u8FieldStrengthSeekLevel_MW;/**----------------------  129th byte */
  tU8 u8QualitySeekLevel_MW;/**----------------------------  130th byte */
  tU8 u8FieldStrengthSeekLevel_SW;/**----------------------  131st byte */
  tU8 u8QualitySeekLevel_SW;/**----------------------------  132nd byte */
  tU8 u8FieldStrengthSeekLevel_TRF;/**---------------------  133rd byte */
  tU8 u8QualitySeekLevel_TRF;/**---------------------------  134th byte */
  tU8 u8FieldStrengthSeekLevel_WB;/**----------------------  135th byte */
  tU8 u8QualitySeekLevel_WB;/**----------------------------  136th byte */
  
  tU8 u8FieldStrengthListLevel_FM_RDS;/**------------------  137th byte */
  tU8 u8QualityListLevel_FM_RDS;/**------------------------  138th byte */
  tU8 u8FieldStrengthListLevel_FM_NONRDS;/**---------------  139th byte */
  tU8 u8QualityListLevel_FM_NONRDS;/**---------------------  140th byte */
  tU8 u8FieldStrengthListLevel_LW;/**----------------------  141st byte */
  tU8 u8QualityListLevel_LW;/**----------------------------  142nd byte */
  tU8 u8FieldStrengthListLevel_MW;/**----------------------  143rd byte */
  tU8 u8QualityListLevel_MW;/**----------------------------  144th byte */
  tU8 u8FieldStrengthListLevel_SW;/**----------------------  145th byte */
  tU8 u8QualityListLevel_SW;/**----------------------------  146th byte */
  tU8 u8FieldStrengthListLevel_TRF;/**---------------------  147thrd byte */
  tU8 u8QualityListLevel_TRF;/**---------------------------  148th byte */
  tU8 u8FieldStrengthListLevel_WB;/**----------------------  149th byte */
  tU8 u8QualityListLevel_WB;/**----------------------------  150th byte */

  tU8 u8SHARX;/**------------------------------------------  151st byte */
  tU8 u8HICUT;/**------------------------------------------  152nd byte */

  /*==========================================================================
  * Stage 5:
  * Area setup for CONFIG area (no CONFIG arear needed) (not audio tuner dependent):
  *------------------------------------------------------------------------*/

  tU32  u32AvailableBands;/**------------------------------  153 - 156 byte */
  tU32  u32CombiBands;/**----------------------------------  157 - 160 byte */
  /** FM */
  tU32  u32LowFM; /**--------------------------------------  12 - 15 byte */
  tU32  u32HighFM;/**--------------------------------------  16 - 19 byte */
  tU8   u8StepSizeFM;/**---------------------------------  20th byte */
  tU8   u8StepSizeManualFM;/**---------------------------  21st byte */
  tU32  u32FM_LSM;/**--------------------------------------  44 - 47 byte */
  /** LW */
  tU32  u32LowLW;/**---------------------------------------  32 - 35 byte */
  tU32  u32HighLW;/**--------------------------------------  36 - 39 byte */
  tU8  u8StepSizeLW;/**----------------------------------  40th  byte */
  tU8  u8StepSizeManualLW;/**----------------------------  41st byte */
  tU32  u32LW_LSM;/**--------------------------------------  52 - 55 byte */
  /** MW */
  tU32  u32LowMW;/**---------------------------------------  22 - 25 byte */
  tU32  u32HighMW;/**--------------------------------------  26 - 29 byte */
  tU8  u8StepSizeMW;/**----------------------------------  30th byte */
  tU8  u8StepSizeManualMW;/**----------------------------  31st byte */
  tU32  u32MW_LSM;/**--------------------------------------  48 - 51 byte */

  /*==========================================================================
  * Stage 6:
  * User settings (audio tuner dependent):
  *------------------------------------------------------------------------*/

  /** 0- TP1,1- TP2,2- AF1,3- AF2,4-5 DDA1,6-7 DDA2*/
  tU8   u8ADRSwitches1;/**---------------------------------------  161st byte */
  /** 0- RDS1,1- RDS2,2- PSFix1,3- PSFix2,4- Stereo1,5- Stereo2,6-7 Reserved */
  tU8   u8ADRSwitches2;/**---------------------------------------  162nd byte */
  /** 0-2 REG1, 3-5 REG2, 6-7 Reserved */
  tU8   u8ADRSwitches3;/**---------------------------------------  163rd byte */
  /** 0- PTY31 1,1- PTY31 2,2- DDS1,3- DDS2,4- DMQ1,5- DMQ2,6-7 reserved */
  tU8   u8ADRSwitches4;/**---------------------------------------  164th byte */


 
  /*==========================================================================
  * Stage 7:
  * HF Parameter (not physical tuner dependent):
  *------------------------------------------------------------------------*/
  tU8   u8ADRFmChannelSeparation;/**---------------------------  99th byte */
  tU8   u8ADRFmLimiting;/**------------------------------------  100th byte */
  tU8   u8ADRFmMultipath;/**-----------------------------------  101st byte */
  tU8   u8ADRFmHighCut;/**-------------------------------------  102nd byte */
  tU8   u8ADRFmSoftMute;/**------------------------------------  103rd byte */
  tU8   u8ADRAmSoftMute;/**------------------------------------  104th byte */
  tU8   u8ADRFmDeemphasis;/**----------------------------------  105th byte */
  tU8   u8ADRFmFrequencyResponse;/**---------------------------  106th byte */
  tU8   u8ADRAmFrequencyResponse;/**---------------------------  107th byte */
  tU8   u8ADRFmMisc;/**----------------------------------------  108th byte */
  tU8   u8ADRAmMisc;/**----------------------------------------  109th byte */
  tU8   u8ADRFmAmplificationLevel;/**--------------------------  110th byte */
  tU8   u8ADRAmAmplificationLevel;/**--------------------------  111th byte */
  tU8   u8ADRAM_Hicut_Configuration;/**------------------------  112th byte */
  tU8   u8ADRAM_Deemphasis;/**---------------------------------  113th byte */

  tU8   u8AntennaConfigId;/**-------------------------------  166th byte */
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
  tU8   u8AMFMMisc2; /**-------------------------------------------569th byte*/
#endif
  /*==========================================================================
  * Stage 8:
  * RDS Parameter
  *------------------------------------------------------------------------*/
 
  //tU8   u8RdsLostTimeout;
  //tU8   u8BehaviouralSwitches;
  //tenLevelType   enLevelType;
  //tU8   u8SpecialFeatures;
  tU8   u8ExchangeAntennaLevel;/**-------------------------------  167th byte */
  tU8   u8PitestLevel;/**----------------------------------------  168th byte */
  tU8   u8NrdsLevel;/**------------------------------------------  169th byte */
  tU8   u8EonLevel;/**-------------------------------------------  170th byte */
  tU8   u8FMMemoLevel;/**----------------------------------------  171st byte */
  tU8   u8PanicFieldstLevel;/**----------------------------------  172nd byte */
  tU8   u8ParamRDSLevel;/**--------------------------------------  173rd byte */
  tU8   u8RDSDataAcceptLevel;/**---------------------------------  174th byte */
  tU8   u8PIStepLevel;/**----------------------------------------  175th byte */
  tU8   u8DDAAntennaDistLevel;/**--------------------------------  176th byte */
  tU8   u8RDSSynDelayCnt;/**-------------------------------------  177th byte */
  tU8   u8MpMeasureLevel;/**-------------------------------------  178th byte */
  tU8   u8MpLevel2Level;/**--------------------------------------  179th byte */
  tU8   u8MpVeryBadLevel;/**-------------------------------------  180th byte */
  tU8   u8MpMediumLevel;/**--------------------------------------  181st byte */
  tU8   u8RDSBadLevel;/**----------------------------------------  182nd byte */
  tU8   u8IllegalAf;/**------------------------------------------  183rd byte */
  tU8   u8AfSaveLevel;/**----------------------------------------  184th byte */
  tU8   u8FirstRundum;/**----------------------------------------  185th byte */
  tU8   u8SecondRundum;/**---------------------------------------  186th byte */
  tU8   u8ThirdRundum;/**----------------------------------------  187th byte */
  tU8   u8PIStepMode;/**-----------------------------------------  188th byte */
  tU8   u8AudioOffScanMode;/**-----------------------------------  189th byte */
  tU8   u8AMLearnCycle;/**---------------------------------------  190th byte */
  tU8   u8PIIgnore;/**-------------------------------------------  191st byte */
  tU8   u8PISearchTime;/**---------------------------------------  192nd byte */
  tU8   u8DisableSeamlessblending;/**----------------------------  193rd byte */
  tU8   u8SlAdjThreshold;/**-------------------------------------  194th byte */
  tU8   u8SlMpThreshold;/**--------------------------------------  195th byte */
  tU8   u8SpecialFMStartFrequency;/**----------------------------  196th byte */
  tU8   u8FMSharxLimit;/**---------------------------------------  197th byte */
  tU8   u8HDScanEveryFrequency;/**-------------------------------  198th byte */
  tU8   u8HWSubConfig; /**---------------------------------------  199th byte */
  tU8   u8FM2DABDelayTime; /**-----------------------------------  200th byte */

  tU8   u8AMFMHicutExtender;/**----------------------------------  209th byte */
  tU8   u8FMSoftwareMuteTiming;/**-------------------------------  210th byte */
  tU8   u8AMFMMisscellaneous2;/**--------------------------------  211th byte */
  tU8   u8FMHicutExtenderFieldstrengthThreshold;/**--------------  212th byte */
  tU8   u8FMHicutExtenderFieldstrengthFilterOffset;/**-----------  213th byte */
  tU8   u8FMChannelSeparationMaximum;/**-------------------------  214th byte */
  tU8   u8FMChannelSeparationTiming;/**--------------------------  215th byte */
  tU8   u8FMHicutFieldstrengthThreshold;/**----------------------  216th byte */
  tU8   u8FMHicutTiming;/**--------------------------------------  217th byte */
  tU8   u8FMDMQSensitivity;/**-----------------------------------  218th byte */
  tU8   u8FMDeviationAdaption;/**--------------------------------  219th byte */
  tU8   u8AMNoiseBlanker228kHzSensitivity;/**--------------------  220th byte */
  tU8   u8AMNoiseBlanker45_6kHzSensitivity;/**-------------------  221th byte */
  tU8   u8AMHicutExtenderFieldstrengthThreshold;/**--------------  222th byte */
  tU8   u8AMHicutExtenderFieldstrengthFilterOffset;/**-----------  223th byte */

  tU8   u8DemodulationSwitchAudioTuner1;/**----------------------  232th byte */
  tU8   u8DemodulationSwitchAudioTuner2;/**----------------------  233th byte */
 
  /*==========================================================================
  * Stage 9:
  * Testmode and AF-List configuration:
  *------------------------------------------------------------------------*/
 
  tU8 u8TestModeDataUpdateInterval;/**---------------------------  93rd byte */
  tU8 u8AFListUpdateInterval;/**---------------------------------  94th byte */
  tU8 u8AFListResort;/**-----------------------------------------  95th byte */
  tU8 u8AFListReleaseBehaviour;/**-------------------------------  96th byte */



  /*==========================================================================
  * Stage 10:
  * Station list configuration:
  *------------------------------------------------------------------------*/
 tU8 u8MaxNumberofStationsInStnList; /** local */
  

  tenFmDabHdTunerMode enPhysicalTunerConfigId;/**----------------  165th byte */


  //tU16  u16CRC;

  tun_trNewKdsFmAmTunerParameter( tVoid );

  virtual tVoid vSetToDefaultValues( tVoid);
  virtual tVoid vParse( tPU8 pu8KdsBuffer, tU32 u32Length);
  virtual tVoid vTrace( tVoid);
};


/** -------------- New AMFMParameter Set Ends   --------------------------------- */


/*============================================================================
 * Indices for New KDS byte stream 'Generic Tuner Parameter Set'
 *--------------------------------------------------------------------------*/
typedef enum
{
  enNewKdsGenericTunerParameter_idx_Tag									=   0, enNewKdsGenericTunerParameter_len_Tag								=   8,
  enNewKdsGenericTunerParameter_idx_MajorVersion						=   8, enNewKdsGenericTunerParameter_len_MajorVersion						=   1,
  enNewKdsGenericTunerParameter_idx_MinorVersion						=   9, enNewKdsGenericTunerParameter_len_MinorVersion						=   2,

  enNewKdsGenericTunerParameter_idx_InitialAutoStoreListId				=  11, enNewKdsGenericTunerParameter_len_InitialAutoStoreListId				=   4,
  enNewKdsGenericTunerParameter_idx_AutostoreUnusedPresetParameter		=  15, enNewKdsGenericTunerParameter_len_AutostoreUnusedPresetParameter     =   1,
  enNewKdsGenericTunerParameter_idx_InitialAutostore_NoOfStations		=  16, enNewKdsGenericTunerParameter_len_InitialAutostore_NoOfStations      =   1,
  enNewKdsGenericTunerParameter_idx_UserAutostore_NoOfStations			=  17, enNewKdsGenericTunerParameter_len_UserAutostore_NoOfStations         =   1,
  enNewKdsGenericTunerParameter_idx_AutostoreSortingCriteriaFM			=  18, enNewKdsGenericTunerParameter_len_AutostoreSortingCriteriaFM         =   1,
  enNewKdsGenericTunerParameter_idx_AutostoreSortingCriteriaAM			=  19, enNewKdsGenericTunerParameter_len_AutostoreSortingCriteriaAM         =   1,
  enNewKdsGenericTunerParameter_idx_AutostoreSortingCriteriaDAB			=  20, enNewKdsGenericTunerParameter_len_AutostoreSortingCriteriaDAB        =   1,
  enNewKdsGenericTunerParameter_idx_AutostoreBehavioralSwitches			=  21, enNewKdsGenericTunerParameter_len_AutostoreBehavioralSwitches        =   1,
  enNewKdsGenericTunerParameter_idx_BandToTuneAfterInitialAutostore		=  22, enNewKdsGenericTunerParameter_len_BandToTuneAfterInitialAutostore    =   1,
  enNewKdsGenericTunerParameter_idx_BandToTuneAfterUserAutostore		=  23, enNewKdsGenericTunerParameter_len_BandToTuneAfterUserAutostore       =   1,

  enNewKdsGenericTunerParameter_idx_ReservedByte1						=  24, enNewKdsGenericTunerParameter_len_ReservedByte1                      =   16,

  enNewKdsGenericTunerParameter_idx_MuteOrAttenuationForAbsStnChngAMFM  =  40, enNewKdsGenericTunerParameter_len_MuteOrAttenuationForAbsStnChngAMFM =   1,
  enNewKdsGenericTunerParameter_idx_MuteOrAttenuationForSeekAMFM        =  41, enNewKdsGenericTunerParameter_len_MuteOrAttenuationForSeekAMFM       =   1,
  enNewKdsGenericTunerParameter_idx_MuteOrAttenuationForManualTuningAMFM=  42, enNewKdsGenericTunerParameter_len_MuteOrAttenuationForManualTuningAMFM  =   1,
  enNewKdsGenericTunerParameter_idx_TimetoReleaseAttenuationAMFM        =  43, enNewKdsGenericTunerParameter_len_TimetoReleaseAttenuationAMFM       =   1,
  enNewKdsGenericTunerParameter_idx_TimeToReleaseDAB_FMLink             =  44, enNewKdsGenericTunerParameter_len_TimeToReleaseDAB_FMLink            =   1,
  enNewKdsGenericTunerParameter_idx_TimeToReleaseDABPlus_FMLink         =  45, enNewKdsGenericTunerParameter_len_TimeToReleaseDABPlus_FMLink        =   1,
  enNewKdsGenericTunerParameter_idx_TimeToReleaseDMBA_FMLink            =  46, enNewKdsGenericTunerParameter_len_TimeToReleaseDMBA_FMLink           =   1,
  enNewKdsGenericTunerParameter_idx_LevelCompensationForStationLoad		=  47, enNewKdsGenericTunerParameter_len_LevelCompensationForStationLoad	=   1,

  enNewKdsGenericTunerParameter_idx_ReservedByte2						=  48, enNewKdsGenericTunerParameter_len_ReservedByte2						=   6,

  enNewKdsGenericTunerParameter_idx_MonitorFM_TA						=  54, enNewKdsGenericTunerParameter_len_MonitorFM_TA						=   1,
  enNewKdsGenericTunerParameter_idx_AudioTunerFM_TPSeek					=  55, enNewKdsGenericTunerParameter_len_AudioTunerFM_TPSeek				=   1,
  enNewKdsGenericTunerParameter_idx_AnnouncementRequestHandling			=  56, enNewKdsGenericTunerParameter_len_AnnouncementRequestHandling		=   1,
  enNewKdsGenericTunerParameter_idx_FM_TPseekStrategy					=  57, enNewKdsGenericTunerParameter_len_FM_TPseekStrategy					=   1,
  enNewKdsGenericTunerParameter_idx_TimeDelayFM_TPStnAftVolZero			=  58, enNewKdsGenericTunerParameter_len_TimeDelayFM_TPStnAftVolZero		=   1,
  enNewKdsGenericTunerParameter_idx_TimeDelayNoFM_TPStnFound			=  59, enNewKdsGenericTunerParameter_len_TimeDelayNoFM_TPStnFound			=   1,
  enNewKdsGenericTunerParameter_idx_TimeDelayFM_TPStnSelAftRDSLost      =  60, enNewKdsGenericTunerParameter_len_TimeDelayFM_TPStnSelAftRDSLost     =   1,
  enNewKdsGenericTunerParameter_idx_TimeFM_TAEndAftRDSLost				=  61, enNewKdsGenericTunerParameter_len_TimeFM_TAEndAftRDSLost				=   1,
  enNewKdsGenericTunerParameter_idx_BackgroundFM_TA						=  62, enNewKdsGenericTunerParameter_len_BackgroundFM_TA					=   1,
  enNewKdsGenericTunerParameter_idx_TraficAnnoInOtherSrc                =  63, enNewKdsGenericTunerParameter_len_TraficAnnoInOtherSrc               =   1,
  enNewKdsGenericTunerParameter_idx_AnnouncementPrio_FM_TA				=  64, enNewKdsGenericTunerParameter_len_AnnouncementPrio_FM_TA				=   1,
  enNewKdsGenericTunerParameter_idx_AnnouncementPrio_FM_PTY31           =  65, enNewKdsGenericTunerParameter_len_AnnouncementPrio_FM_PTY31          =   1,
  enNewKdsGenericTunerParameter_idx_AnnouncementPrio_FM_NEWS			=  66, enNewKdsGenericTunerParameter_len_AnnouncementPrio_FM_NEWS			=   1,
  enNewKdsGenericTunerParameter_idx_AnnouncementPrio_DAB_Alarm			=  67, enNewKdsGenericTunerParameter_len_AnnouncementPrio_DAB_Alarm			=   1,
  enNewKdsGenericTunerParameter_idx_AnnouncementPrio_DAB_RoadTrafficFlash  =  68, enNewKdsGenericTunerParameter_len_AnnouncementPrio_DAB_RoadTrafficFlash =   1,
  enNewKdsGenericTunerParameter_idx_AnnouncementPrio_DAB_TransportFlash =  69, enNewKdsGenericTunerParameter_len_AnnouncementPrio_DAB_TransportFlash =   1,
  enNewKdsGenericTunerParameter_idx_AnnouncementPrio_DAB_WarningService =  70, enNewKdsGenericTunerParameter_len_AnnouncementPrio_DAB_WarningService =   1,


  enNewKdsGenericTunerParameter_idx_AnnouncementPrio_DAB_NewsFlash		=  71, enNewKdsGenericTunerParameter_len_AnnouncementPrio_DAB_NewsFlash		=   1,
  enNewKdsGenericTunerParameter_idx_AnnouncementPrio_DAB_AreaWeatherFlash  =  72, enNewKdsGenericTunerParameter_len_AnnouncementPrio_DAB_AreaWeatherFlash  =   1,
  enNewKdsGenericTunerParameter_idx_AnnouncementPrio_DAB_EventAnnouncement =  73, enNewKdsGenericTunerParameter_len_AnnouncementPrio_DAB_EventAnnouncement =   1,
  enNewKdsGenericTunerParameter_idx_AnnouncementPrio_DAB_SpecialEvent	=  74, enNewKdsGenericTunerParameter_len_AnnouncementPrio_DAB_SpecialEvent	=   1,
  enNewKdsGenericTunerParameter_idx_AnnouncementPrio_DAB_ProgramInfo	=  75, enNewKdsGenericTunerParameter_len_AnnouncementPrio_DAB_ProgramInfo	=   1,

  enNewKdsGenericTunerParameter_idx_AnnouncementPrio_DAB_SportReport	=  76, enNewKdsGenericTunerParameter_len_AnnouncementPrio_DAB_SportReport	=   1,
  enNewKdsGenericTunerParameter_idx_AnnouncementPrio_DAB_FinancialReport  =  77, enNewKdsGenericTunerParameter_len_AnnouncementPrio_DAB_FinancialReport  =   1,
  enNewKdsGenericTunerParameter_idx_AnnouncementPrio_DAB_reserved1		=  78, enNewKdsGenericTunerParameter_len_AnnouncementPrio_DAB_reserved1		=   1,
  enNewKdsGenericTunerParameter_idx_AnnouncementPrio_DAB_reserved2		=  79, enNewKdsGenericTunerParameter_len_AnnouncementPrio_DAB_reserved2		=   1,
  enNewKdsGenericTunerParameter_idx_AnnouncementPrio_DAB_reserved3		=  80, enNewKdsGenericTunerParameter_len_AnnouncementPrio_DAB_reserved3		=   1,
  enNewKdsGenericTunerParameter_idx_AnnouncementPrio_DAB_reserved4		=  81, enNewKdsGenericTunerParameter_len_AnnouncementPrio_DAB_reserved4		=   1,
  enNewKdsGenericTunerParameter_idx_AnnouncementPrio_DAB_reserved5		=  82, enNewKdsGenericTunerParameter_len_AnnouncementPrio_DAB_reserved5		=   1,

  enNewKdsGenericTunerParameter_idx_ReservedByte3						=  83, enNewKdsGenericTunerParameter_len_ReservedByte3						=   10,
  
  enNewKdsGenericTunerParameter_idx_TMCServer1							=  93, enNewKdsGenericTunerParameter_len_TMCServer1							=   1,

  enNewKdsGenericTunerParameter_idx_ReservedByte4						=  94, enNewKdsGenericTunerParameter_len_ReservedByte4						=   40,

  /************* 92 done *************/

  enNewKdsGenericTunerParameter_idx_PresetListType						=  134, enNewKdsGenericTunerParameter_len_PresetListType					=   1,
  enNewKdsGenericTunerParameter_idx_NumberOfPresetListFM				=  135, enNewKdsGenericTunerParameter_len_NumberOfPresetListFM				=   1,
  enNewKdsGenericTunerParameter_idx_NumberOfBanksHmiFM					=  136, enNewKdsGenericTunerParameter_len_NumberOfBanksHmiFM				=   1,
  enNewKdsGenericTunerParameter_idx_NumberOfPresetsPerBankFM			=  137, enNewKdsGenericTunerParameter_len_NumberOfPresetsPerBankFM			=   1,

  enNewKdsGenericTunerParameter_idx_ReservedByte5						=  138, enNewKdsGenericTunerParameter_len_ReservedByte5						=   1,

  enNewKdsGenericTunerParameter_idx_NumberOfPresetListMW				=  139, enNewKdsGenericTunerParameter_len_NumberOfPresetListMW				=   1,
  enNewKdsGenericTunerParameter_idx_NumberOfBanksHmiMW					=  140, enNewKdsGenericTunerParameter_len_NumberOfBanksHmiMW				=   1,
  enNewKdsGenericTunerParameter_idx_NumberOfPresetsPerBankMW			=  141, enNewKdsGenericTunerParameter_len_NumberOfPresetsPerBankMW			=   1,

  enNewKdsGenericTunerParameter_idx_ReservedByte6						=  142, enNewKdsGenericTunerParameter_len_ReservedByte6						=   1,

  enNewKdsGenericTunerParameter_idx_NumberOfPresetListLW				=  143, enNewKdsGenericTunerParameter_len_NumberOfPresetListLW				=   1,
  enNewKdsGenericTunerParameter_idx_NumberOfBanksHmiLW					=  144, enNewKdsGenericTunerParameter_len_NumberOfBanksHmiLW				=   1,
  enNewKdsGenericTunerParameter_idx_NumberOfPresetsPerBankLW			=  145, enNewKdsGenericTunerParameter_len_NumberOfPresetsPerBankLW			=   1,

  enNewKdsGenericTunerParameter_idx_ReservedByte7						=  146, enNewKdsGenericTunerParameter_len_ReservedByte7						=   1,
  
  enNewKdsGenericTunerParameter_idx_NumberOfPresetListDAB				=  147, enNewKdsGenericTunerParameter_len_NumberOfPresetListDAB				=   1,
  enNewKdsGenericTunerParameter_idx_NumberOfBanksHmiDAB					=  148, enNewKdsGenericTunerParameter_len_NumberOfBanksHmiDAB				=   1,
  enNewKdsGenericTunerParameter_idx_NumberOfPresetsPerBankDAB			=  149, enNewKdsGenericTunerParameter_len_NumberOfPresetsPerBankDAB			=   1,

  enNewKdsGenericTunerParameter_idx_ReservedByte8						=  150, enNewKdsGenericTunerParameter_len_ReservedByte8						=   1,

  enNewKdsGenericTunerParameter_idx_NumberOfPresetListForMixedAllBands	=  151, enNewKdsGenericTunerParameter_len_NumberOfPresetListForMixedAllBands=   1,
  enNewKdsGenericTunerParameter_idx_NumberOfBanksHmiForMixedAllBands	=  152, enNewKdsGenericTunerParameter_len_NumberOfBanksHmiForMixedAllBands	=   1,
  enNewKdsGenericTunerParameter_idx_NumberOfPresetsPerBankForMixedAllBands	=  153, enNewKdsGenericTunerParameter_len_NumberOfPresetsPerBankForMixedAllBands =   1,

  enNewKdsGenericTunerParameter_idx_ReservedByte9						=  154, enNewKdsGenericTunerParameter_len_ReservedByte9						=   1,

  enNewKdsGenericTunerParameter_idx_ReservedByte10						=  155, enNewKdsGenericTunerParameter_len_ReservedByte10					=   16,

  enNewKdsGenericTunerParameter_idx_PresetStartPoint					=  171, enNewKdsGenericTunerParameter_len_Preset							=   4,

  enNewKdsGenericTunerParameter_idx_CRC									= 415, enNewKdsGenericTunerParameter_len_CRC								=   2

 } tenNewKdsGenericTunerParameter;                                                                                         /* ------- */
#define TUN_CONFIG_NEW_KDS_GENERIC_TUNER_PARAMETER_LENGTH                                                                      417
#define TUN_CONFIG_NEW_KDS_GENERIC_NO_OF_PRESETS																				61
#define TUN_CONFIG_NEW_KDS_GENERIC_TUNER_PARAMETER_START_POINT_1																0x0420
#define TUN_CONFIG_NEW_KDS_GENERIC_TUNER_PARAMETER_READ_LEN_1																	230
#define TUN_CONFIG_NEW_KDS_GENERIC_TUNER_PARAMETER_START_POINT_2																0x0421
#define TUN_CONFIG_NEW_KDS_GENERIC_TUNER_PARAMETER_READ_LEN_2																	186

/** -------------- New Generic Tuner Parameter Set Begins --------------------------------- */

struct tun_trNewKdsGenericTunerParameter : public tun_trKds
{
  /** Check whether KDS data is read or no*/
  tBool bKdsDataRead;

  /** Parameter set idetification tag ---------------------  0 - 7 byte */
  tChar sTag[ 8];
  /** Major(1 byte) and Minor(2 byte) Version -------------	 8 - 10 byte */
  tChar sVersion[ 3];
 
  tU32  u32InitialAutoStoreListId;/**----------------------  11 - 14 byte */

  tU8   u8AutostoreUnusedPresetParameter;/**---------------  15th byte */
  
  tU8   u8InitialAutostore_NoOfStations;/**----------------  16th byte */

  tU8   u8UserAutostore_NoOfStations; /**------------------  17th byte */
  tU8   u8AutostoreSortingCriteriaFM;/**-------------------  18th byte */
  tU8   u8AutostoreSortingCriteriaAM;/**-------------------  19th byte */

  tU8   u8AutostoreSortingCriteriaDAB;/**------------------  20th byte */
  tU8   u8AutostoreBehavioralSwitches;/**------------------  21st byte */

  tU8   u8BandToTuneAfterInitialAutostore;/**--------------  22nd byte */  
  tU8   u8BandToTuneAfterUserAutostore;/**-----------------  23rd byte */  

  //tU8  u8ReservedByte1[16];/**---------------------------  24 - 39 byte */  

  tU8   u8MuteOrAttenuationForAbsStnChngAMFM;/**-----------  40th byte */
  tU8   u8MuteOrAttenuationForSeekAMFM;/**-----------------  41st byte */
  tU8   u8MuteOrAttenuationForManualTuningAMFM;/**---------  42nd byte */ 

  tU8   u8TimetoReleaseAttenuationAMFM;/**-----------------  43rd byte */
  tU8   u8TimeToReleaseDAB_FMLink;/**----------------------  44th byte */

  tU8   u8TimeToReleaseDABPlus_FMLink;/**------------------  45th byte */

  tU8   u8TimeToReleaseDMBA_FMLink;/**---------------------  46th byte */

  tU8   u8LevelCompensationForStationLoad;/**--------------  47th byte */
  //tU8  u8ReservedByte2[6];/**----------------------------  48 - 53 byte */

  tU8   u8MonitorFM_TA;/**---------------------------------  54th byte */
  tU8   u8AudioTunerFM_TPSeek;/**--------------------------  55th byte */
  tU8   u8AnnouncementRequestHandling;/**------------------  56th byte */

  tU8   u8FM_TPseekStrategy;/**----------------------------  57th byte */
  tU8   u8TimeDelayFM_TPStnAftVolZero;/**------------------  58th byte */
  tU8   u8TimeDelayNoFM_TPStnFound;/**---------------------  59th byte */

  tU8   u8TimeDelayFM_TPStnSelAftRDSLost;/**---------------  60th byte */
  tU8   u8TimeFM_TAEndAftRDSLost;/**-----------------------  61st byte */

  tU8   u8BackgroundFM_TA;/**------------------------------  62nd byte */

  tU8   u8TraficAnnoInOtherSrc;/**-------------------------  63rd byte */
  tU8   u8AnnouncementPrio_FM_TA;/**-----------------------  64th byte */
  tU8   u8AnnouncementPrio_FM_PTY31;/**--------------------  65th byte */
  tU8   u8AnnouncementPrio_FM_NEWS;/**---------------------  66th byte */
  tU8   u8AnnouncementPrio_DAB_Alarm;/**-------------------  67th byte */


  
  tU8   u8AnnouncementPrio_DAB_RoadTrafficFlash;/**--------  68th byte */
  tU8   u8AnnouncementPrio_DAB_TransportFlash;/**----------  69th byte */
  tU8   u8AnnouncementPrio_DAB_WarningService;/**----------  70th byte */


  tU8   u8AnnouncementPrio_DAB_NewsFlash;/**---------------  71st byte */
  tU8   u8AnnouncementPrio_DAB_AreaWeatherFlash;/**--------  72nd byte */
  tU8   u8AnnouncementPrio_DAB_EventAnnouncement;/**-------  73rd byte */
  tU8   u8AnnouncementPrio_DAB_SpecialEvent;/**------------  74th byte */
  tU8   u8AnnouncementPrio_DAB_ProgramInfo;/**-------------  75th byte */
  tU8   u8AnnouncementPrio_DAB_SportReport;/**-------------  76th byte */
  tU8   u8AnnouncementPrio_DAB_FinancialReport;/**---------  77th byte */
  tU8   u8AnnouncementPrio_DAB_reserved1;/**---------------  78th byte */
  tU8   u8AnnouncementPrio_DAB_reserved2;/**---------------  79th byte */
  tU8   u8AnnouncementPrio_DAB_reserved3;/**---------------  80th byte */
  tU8   u8AnnouncementPrio_DAB_reserved4;/**---------------  81st byte */
  tU8   u8AnnouncementPrio_DAB_reserved5;/**---------------  82nd byte */

  //tU8  u8ReservedByte3[10];/**----------------------------  83 - 92 byte */

  tU8   u8TMCServer1;/**-----------------------------------  93rd byte */

  //tU8  u8ReservedByte4[40];/**----------------------------  94 - 133 byte */

  tU8   u8PresetListType;/**-------------------------------  134th byte */
  tU8   u8NumberOfPresetListFM;/**-------------------------  135th byte */
  tU8   u8NumberOfBanksHmiFM;/**---------------------------  136th byte */
  tU8   u8NumberOfPresetsPerBankFM;/**---------------------  137th byte */
//  tU8   u8ReservedByte5;/**-------------------------------  138th byte */
  tU8   u8NumberOfPresetListMW;/**-------------------------  139th byte */
  tU8   u8NumberOfBanksHmiMW;/**---------------------------  140th byte */
  tU8   u8NumberOfPresetsPerBankMW;/**---------------------  141st byte */
//  tU8   u8ReservedByte6;/**-------------------------------  142th byte */
  tU8   u8NumberOfPresetListLW;/**-------------------------  143rd byte */
  tU8   u8NumberOfBanksHmiLW;/**---------------------------  144th byte */
  tU8   u8NumberOfPresetsPerBankLW;/**---------------------  145th byte */
//  tU8   u8ReservedByte7;/**-------------------------------  146th byte */
  tU8   u8NumberOfPresetListDAB;/**------------------------  147th byte */
  tU8   u8NumberOfBanksHmiDAB;/**--------------------------  148th byte */
  tU8   u8NumberOfPresetsPerBankDAB;/**--------------------  149th byte */

  /*DRM Preset List*/
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
  tU8   u8NumberOfPresetListDRM;
  tU8   u8NumberOfBanksHmiDRM;
  tU8   u8NumberOfPresetsPerBankDRM;
#endif 
//  tU8   u8ReservedByte8;/**-------------------------------  150th byte */
  tU8   u8NumberOfPresetListForMixedAllBands;/**-----------  151st byte */
  tU8   u8NumberOfBanksHmiForMixedAllBands;/**-------------  152nd byte */
  tU8   u8NumberOfPresetsPerBankForMixedAllBands;/**-------  153rd byte */
//  tU8   u8ReservedByte9;/**-------------------------------  154th byte */
//  tU8   u8ReservedByte10[16];/**--------------------------  155 - 170th byte */

  tU32  au32Preset[61];/**----------------------------------  171 - 414th byte*/

  tU16  u16CRC;/**-----------------------------------------  415 - 416th byte */

  //tU16  u16CRC;

  tun_trNewKdsGenericTunerParameter( tVoid);

  virtual tVoid vSetToDefaultValues( tVoid);
  virtual tVoid vParse( tPU8 pu8KdsBuffer, tU32 u32Length);
  virtual tVoid vTrace( tVoid);
};


/** -------------- New Generic Tuner Parameter Set Ends   --------------------------------- */



/* =========================================================================*
* Available Bands
* --------------------------------------------------------------------------*/
typedef enum
{
	enAvailBand_FM = 0,
	enAvailBand_LW = 1,
	enAvailBand_MW = 2,
	enAvailBand_AM = 3,
	enAvailBand_CB1 = 20,
	enAvailBand_NONE = 254,
	/** Used to set Mixed Preset list band */
	enAvailBand_GetCurrent = 255 
}tenAvailableBands;
/*==========================================================================*/
/*============================================================================
 * Indices for KDS byte stream 'CMVariantCoding'
 *--------------------------------------------------------------------------*/
typedef enum
{
 enKdsCMVariantCoding_idx_HWAssemblyInfo          =   4, enKdsCMVariantCoding_len_HWAssemblyInfo          =   1,
 enKdsCMVariantCoding_idx_HWAssemblyInfo3          =   15, enKdsCMVariantCoding_len_HWAssemblyInfo3          =   1,
 enKdsCMVariantCoding_idx_FmDabHdTunerMode        =   5, enKdsCMVariantCoding_len_FmDabHdTunerMode        =   1
} tenKdsCMVariantCoding;                                                                                         /* ------- */
#define TUN_CONFIG_KDS_CM_VARIANT_CODING_LENGTH																		32
#define TUN_CONFIG_KDS_CM_VARIANT_CODING																			0x0dF4

#define EXTRACT_HD_BIT_HW_ASSEMBLYINFO				0x40
/**************************************************/

struct tun_trKdsCMVariantCoding : public tun_trKds
{
  tBool bKdsDataRead;
  tU8   u8HWAssemblyInfo;
  tU8   u8HWAssemblyInfo3;
  tenFmDabHdTunerMode enFmDabHdTunerMode;
  
  tun_trKdsCMVariantCoding( tVoid);

  virtual tVoid vSetToDefaultValues( tVoid);
  virtual tVoid vParse( tPU8 pu8KdsBuffer, tU32 u32Length);
  virtual tVoid vTrace( tVoid);
};

/*============================================================================
* KDS structure for HD Parameter set
* ==========================================================================*/
#define EXTRACT_HD_BIT_HW_CONFIGURATION																					0x40

/*============================================================================
 * Indices for KDS byte stream 'HDRadioParameterSet'
 *==========================================================================*/
typedef enum
{
  enKdsHDRadioParameterSet_idx_tag									=   0, enKdsHDRadioParameterSet_len_tag										=   8,
  enKdsHDRadioParameterSet_idx_MajorVersion							=   8, enKdsHDRadioParameterSet_len_MajorVersion							=   1,
  enKdsHDRadioParameterSet_idx_MinorVersion							=   9, enKdsHDRadioParameterSet_len_MinorVersion							=   2,
  enKdsHDRadioParameterSet_idx_BlendThresholdFMHybrid				=   11, enKdsHDRadioParameterSet_len_BlendThresholdFMHybrid					=   1,
  enKdsHDRadioParameterSet_idx_DigiAudioScalingFMHybrid				=   12, enKdsHDRadioParameterSet_len_DigiAudioScalingFMHybrid				=   2,
  enKdsHDRadioParameterSet_idx_DigiAudioDelayFMHybridPrimaryBBRate  =   14, enKdsHDRadioParameterSet_len_DigiAudioDelayFMHybridPrimaryBBRate	=   2,
  enKdsHDRadioParameterSet_idx_BlendRateFMHybrid					=   16, enKdsHDRadioParameterSet_len_BlendRateFMHybrid						=   1,
  enKdsHDRadioParameterSet_idx_BlendThresholdAMHybrid				=   17, enKdsHDRadioParameterSet_len_BlendThresholdAMHybrid					=   1,
  enKdsHDRadioParameterSet_idx_AM_MPSAudioScaling					=   18, enKdsHDRadioParameterSet_len_AM_MPSAudioScaling						=   2,
  enKdsHDRadioParameterSet_idx_AM_MPSBlendRate						=   20, enKdsHDRadioParameterSet_len_AM_MPSBlendRate						=   1,
  enKdsHDRadioParameterSet_idx_DigiAudioDelayAM_MPSPrimaryBBRate    =	21, enKdsHDRadioParameterSet_len_DigiAudioDelayAM_MPSPrimaryBBRate		=   2,
  enKdsHDRadioParameterSet_idx_DigiAudioPhaseInversion				=	23, enKdsHDRadioParameterSet_len_DigiAudioPhaseInversion				=   1,
  enKdsHDRadioParameterSet_idx_BlendDecision						=	24, enKdsHDRadioParameterSet_len_BlendDecision							=   1,
  enKdsHDRadioParameterSet_idx_FM_CDNO_BlendDecision				=	25, enKdsHDRadioParameterSet_len_FM_CDNO_BlendDecision					=   1,
  enKdsHDRadioParameterSet_idx_AM_CDNO_BlendDecision				=	26, enKdsHDRadioParameterSet_len_AM_CDNO_BlendDecision					=   1,

  enKdsHDRadioParameterSet_idx_FMSPSBlendThreshold					=	27, enKdsHDRadioParameterSet_len_FMSPSBlendThreshold					=   1,
  enKdsHDRadioParameterSet_idx_FMSPSAudioScaling					=	28, enKdsHDRadioParameterSet_len_FMSPSAudioScaling						=   2,
  enKdsHDRadioParameterSet_idx_FMSPSBlendRate						=	30, enKdsHDRadioParameterSet_len_FMSPSBlendRate							=   1,
  enKdsHDRadioParameterSet_idx_BallgameModeAudioOutput				=	31, enKdsHDRadioParameterSet_len_BallgameModeAudioOutput				=   1,
  enKdsHDRadioParameterSet_idx_FilteredDSQMTimeConstant				=	32, enKdsHDRadioParameterSet_len_FilteredDSQMTimeConstant				=   1,
  enKdsHDRadioParameterSet_idx_EZBlendEnable						=	33, enKdsHDRadioParameterSet_len_EZBlendEnable							=   1,
  enKdsHDRadioParameterSet_idx_DSQMSeekThreshold					=	34, enKdsHDRadioParameterSet_len_DSQMSeekThreshold						=   2,

  enKdsHDRadioParameterSet_idx_Reserved0							=	36, enKdsHDRadioParameterSet_len_Reserved0								=   12,

  enKdsHDRadioParameterSet_idx_HDImageType							=	48, enKdsHDRadioParameterSet_len_HDImageType							=   1,
  enKdsHDRadioParameterSet_idx_PSDFieldBitMask						=	49, enKdsHDRadioParameterSet_len_PSDFieldBitMask						=   1,
  enKdsHDRadioParameterSet_idx_HDPSDCommentsBitMask					=	50, enKdsHDRadioParameterSet_len_HDPSDCommentsBitMask					=   1,
  enKdsHDRadioParameterSet_idx_HDPSDCommercialBitMask				=	51, enKdsHDRadioParameterSet_len_HDPSDCommercialBitMask					=   1,
  enKdsHDRadioParameterSet_idx_HDPSD_UFIDBitMask				    =	52, enKdsHDRadioParameterSet_len_HDPSD_UFIDBitMask						=   1,
  
  enKdsHDRadioParameterSet_idx_Reserved1							=	53, enKdsHDRadioParameterSet_len_Reserved1								=   4,

  enKdsHDRadioParameterSet_idx_Timer1								=	57, enKdsHDRadioParameterSet_len_Timer1									=   1,
  enKdsHDRadioParameterSet_idx_Timer2								=	58, enKdsHDRadioParameterSet_len_Timer2									=   1,
  enKdsHDRadioParameterSet_idx_Timer3								=	59, enKdsHDRadioParameterSet_len_Timer3									=   1,
  enKdsHDRadioParameterSet_idx_Timer4								=	60, enKdsHDRadioParameterSet_len_Timer4									=   1,
  enKdsHDRadioParameterSet_idx_Timer5								=	61, enKdsHDRadioParameterSet_len_Timer5									=   1,
  enKdsHDRadioParameterSet_idx_Reserved2							=	62, enKdsHDRadioParameterSet_len_Reserved2								=   2
} tenKdsHDRadioParameterSet;                                                                                         /* ------- */
#define TUN_CONFIG_KDS_HD_RADIO_PARAM_SET_LENGTH                    64
#define TUN_KDS_CM_HD_RADIO_PARAM_SET_KEY							0x0450
struct tun_trKdsHDRadioParameterSet : public tun_trKds
{
  tBool bKdsDataRead;
  tChar sTag[ 8];
  tChar sVersion[ 3];
  tU8 u8BlendThresholdFMHybrid;
  tU16 u16DigiAudioScalingFMHybrid;
  tU16 u16DigiAudioDelayFMHybridPrimaryBBRate;
  tU8 u8BlendRateFMHybrid;			
  tU8 u8BlendThresholdAMHybrid;			
  tU16 u16AM_MPSAudioScaling;					
  tU8 u8AM_MPSBlendRate;						
  tU16 u16DigiAudioDelayAM_MPSPrimaryBBRate;    
  tU8 u8DigiAudioPhaseInversion;				
  tU8 u8BlendDecision;						
  tU8 u8FM_CDNO_BlendDecision;				
  tU8 u8AM_CDNO_BlendDecision;				
  tU8 u8FMSPSBlendThreshold;
  tU16 u16FMSPSAudioScaling;
  tU8 u8FMSPSBlendRate;
  tU8 u8BallgameModeAudioOutput;
  tU8 u8FilteredDSQMTimeConstant;
  tU8 u8EZBlendEnable;
  tU16 u16DSQMSeekThreshold;
  tU8 u8HDImageType;
  tU8 u8HDPSDFieldBitMask;
  tU8 u8HDPSDCommentsBitMask;
  tU8 u8HDPSDCommercialBitMask;
  tU8 u8HDPSD_UFIDBitMask;
  tU8 u8Timer1;								
  tU8 u8Timer2;								
  tU8 u8Timer3;								
  tU8 u8Timer4;								
  tU8 u8Timer5;								
	

  tun_trKdsHDRadioParameterSet( tVoid);
  virtual tVoid vSetToDefaultValues( tVoid);
  virtual tVoid vParse( tPU8 pu8KdsBuffer, tU32 u32Length);
  virtual tVoid vTrace( tVoid);
};


/*============================================================================
* KDS structure for DRM Parameter set
* ==========================================================================*/
#define EXTRACT_DRM_BIT_HW_CONFIGURATION                                                               0x04

/*============================================================================
 * Indices for KDS byte stream 'DRMTunerParameterSet'
 *==========================================================================*/
typedef enum
{
  enKdsDRMTunerParameterSet_idx_tag										=   0,  enKdsDRMTunerParameterSet_len_tag										=   8,
  enKdsDRMTunerParameterSet_idx_MajorVersion							=   8,  enKdsDRMTunerParameterSet_len_MajorVersion								=   1,
  enKdsDRMTunerParameterSet_idx_MinorVersion							=   9,  enKdsDRMTunerParameterSet_len_MinorVersion								=   2,
  enKdsDRMTunerParameterSet_idx_DRMBandEnable							=   11, enKdsDRMTunerParameterSet_len_DRMBandEnable								=   1,
  enKdsDRMTunerParameterSet_idx_DRMMWTunningStepSizeForSearch			=   12, enKdsDRMTunerParameterSet_len_DRMMWTunningStepSizeForSearch				=   1,
  enKdsDRMTunerParameterSet_idx_DRMMWTunningStepSizeForManualStep		=   13, enKdsDRMTunerParameterSet_len_DRMMWTunningStepSizeForManualStep			=   1,
  enKdsDRMTunerParameterSet_idx_Reserved1								=   14, enKdsDRMTunerParameterSet_len_Reserved1									=   4,
  enKdsDRMTunerParameterSet_idx_DRMServiceFollowing						=   18, enKdsDRMTunerParameterSet_len_DRMServiceFollowing						=   1,
  enKdsDRMTunerParameterSet_idx_DRM_DRMAFSearchDelayTime				=   19, enKdsDRMTunerParameterSet_len_DRM_DRMAFSearchDelayTime					=   1,
  enKdsDRMTunerParameterSet_idx_Reserved2								=   20, enKdsDRMTunerParameterSet_len_Reserved2									=   3,
  enKdsDRMTunerParameterSet_idx_DRMServiceListUnlearnTimeThreshold		=	23, enKdsDRMTunerParameterSet_len_DRMServiceListUnlearnTimeThreshold		=   1,
  enKdsDRMTunerParameterSet_idx_Reserved3								=	24, enKdsDRMTunerParameterSet_len_Reserved3									=   1,
  enKdsDRMTunerParameterSet_idx_DRMServiceListLearnAcceptanceTime		=	25, enKdsDRMTunerParameterSet_len_DRMServiceListLearnAcceptanceTime			=   1,
  enKdsDRMTunerParameterSet_idx_DRMServiceListLearnAcceptanceThreshold	=	26, enKdsDRMTunerParameterSet_len_DRMServiceListLearnAcceptanceThreshold	=   1,
  enKdsDRMTunerParameterSet_idx_DRMQualityThresholdForAudioMute			=	27, enKdsDRMTunerParameterSet_len_DRMQualityThresholdForAudioMute			=   1,
  enKdsDRMTunerParameterSet_idx_Reserved4								=	28, enKdsDRMTunerParameterSet_len_Reserved4									=   1,
  enKdsDRMTunerParameterSet_idx_DRMAudioErrorConcealmentLevel			=	29, enKdsDRMTunerParameterSet_len_DRMAudioErrorConcealmentLevel				=   1,
  enKdsDRMTunerParameterSet_idx_Reserved5								=	30, enKdsDRMTunerParameterSet_len_Reserved5									=   2,
  enKdsDRMTunerParameterSet_idx_DRMMWSeekStopFieldStrengthThreshold     =   32, enKdsDRMTunerParameterSet_len_DRMMWSeekStopFieldStrengthThreshold 		= 	1,
  enKdsDRMTunerParameterSet_idx_DRMSwitches               				=   33, enKdsDRMTunerParameterSet_len_DRMSwitches 								= 	1,
  enKdsDRMTunerParameterSet_idx_Reserved6								=	34, enKdsDRMTunerParameterSet_len_Reserved6									=   13

} tenKdsDRMTunerParameterSet;                                                                                         /* ------- */
#define TUN_CONFIG_KDS_DRM_TUNER_PARAM_SET_LENGTH                   48
#define TUN_KDS_CM_DRM_TUNER_PARAM_SET_KEY							0x0460
struct tun_trKdsDRMTunerParameterSet : public tun_trKds
{
  tBool bKdsDataRead;
  tChar sTag[ 8];
  tChar sVersion[ 3];
  tU8 u8DRMBandEnable;
  tU8 u8DRMMWTunningStepSizeForSearch;
  tU8 u8DRMMWTunningStepSizeForManualStep;
  tU8 u8DRMServiceFollowing;			
  tU8 u8DRM_DRMAFSearchDelayTime;			
  tU8 u8DRMServiceListUnlearnTimeThreshold; 
  tU8 u8DRMServiceListLearnAcceptanceTime;						
  tU8 u8DRMServiceListLearnAcceptanceThreshold;				
  tU8 u8DRMQualityThresholdForAudioMute;			
  tU8 u8DRMAudioErrorConcealmentLevel;
  tS8 s8DRMMWSeekStopFieldStrengthThreshold;
  tU8 u8DRMSwitches;

  tun_trKdsDRMTunerParameterSet( tVoid);
  virtual tVoid vSetToDefaultValues( tVoid);
  virtual tVoid vParse( tPU8 pu8KdsBuffer, tU32 u32Length);
  virtual tVoid vTrace( tVoid);
};


/*****************************************************************************
 * class tun_Config definition
 ****************************************************************************/

//Enumeration for KDS configuration parameters
typedef enum
{
  TUN_KDS_INVALID_PARAMETER         = 255
// ,TUN_KDS_FM_AM_Tuner_Parameter_Set = 0
// ,TUN_KDS_CM_CONFIG_STRING = 1
 ,TUN_KDS_CM_EOL_PARAMETER = 2
 ,TUN_KDS_CM_PROD_MODE_SETTINGS
 ,TUN_KDS_CM_QUALITY_LEVEL_OFFSET
 ,TUN_KDS_CM_VARIANT_CODING
 ,TUN_NEW_KDS_FM_AM_Tuner_Parameter_Set 
 ,TUN_NEW_KDS_GENERIC_Tuner_Parameter_Set 
 ,TUN_KDS_CM_EOL_CONFIG_RADIO_PARAMETER
 ,TUN_KDS_CM_HD_RADIO_PARAM_SET
 ,TUN_KDS_CM_DRM_TUNER_PARAM_SET
} tenKDSParameter;

class tun_Config : public TUN_Singleton< tun_Config>
{
  friend class TUN_Singleton< tun_Config>;

private:
  /* private Constructor (Singleton pattern) */
  tun_Config( );
  /* ---- Fix for logiscope review ------------*/
  tVoid tun_ConfigSuzukiParameters();
  tVoid tun_ConfigVMMIBEParameters();
  tVoid tun_ConfigJACCIIParameters();
  tVoid tun_ConfigPSARCCParameters();
  tVoid tun_ConfigRNAIVIParameters();
  tVoid tun_ConfigCAFParameters();

public:
  /* Destructors */
  virtual ~tun_Config( );
  //To reslove complier error
  #if FEATURE_VW_MIBE
  /** productionmodestate */
  //tBool m_bProductionModeState;

  tun_trKdsEOLParameter m_rKdsEOLParameter;
  tun_trKdsQualityLevelOffset m_rKdsQualityLevelOffset;
  tun_trKdsProdModeSettings m_rKdsProdModeSettings;
#endif

  tBool bGetFrequencyValid( tU8 u8Band,tU32 u32Frequency) const;
  tBool bIsFrequencyWithinTheRange( tU8 u8Band,tU32 u32Frequency) const;
  tBool bIsFrequencyStepValid( tU8 u8Band,tU32 u32Frequency) const;

  tBool bGetKdsDataAvailable( tVoid) const          { return m_rNewKdsFmAmTunerParameter.bKdsDataRead; } // FALSE: default values, TRUE: KDS data read
  tVoid vGetVersionId( tChar* pBuffer, tU32 u32BufferLen) const;

  tU8  u8GetFeatureControlI( tVoid) const           { return m_rNewKdsFmAmTunerParameter.u8FeatureControlI; }
  tBool u8GetTMCServer( tVoid) const;
  tU8  u8GetNoFMpresetBanks( tVoid) const           { return m_rNewKdsGenericTunerParameter.u8NumberOfBanksHmiFM; }
  tU8  u8GetNoOfFMpresetPerBank( tVoid) const       { return m_rNewKdsGenericTunerParameter.u8NumberOfPresetsPerBankFM; }
  tU8  u8GetNoAMpresetBanks1( tVoid) const          { return m_rNewKdsGenericTunerParameter.u8NumberOfBanksHmiMW; }
  tU8  u8GetNoOfAMpresetPerBank1( tVoid) const      { return m_rNewKdsGenericTunerParameter.u8NumberOfPresetsPerBankMW; }
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
  tU8  u8GetNoLWpresetBanks( tVoid) const          { return m_rNewKdsGenericTunerParameter.u8NumberOfBanksHmiLW; }
  tU8  u8GetNoOfLWpresetPerBank( tVoid) const      { return m_rNewKdsGenericTunerParameter.u8NumberOfPresetsPerBankLW; }
  tU8  u8GetNoDRMpresetBanks( tVoid) const           { return m_rNewKdsGenericTunerParameter.u8NumberOfBanksHmiDRM; }
  tU8  u8GetNoOfDRMpresetPerBank( tVoid) const       { return m_rNewKdsGenericTunerParameter.u8NumberOfPresetsPerBankDRM; }
#endif
  tU8  u8GetNoMIXEDpresetBanks( tVoid) const           { return m_rNewKdsGenericTunerParameter.u8NumberOfBanksHmiForMixedAllBands; }
  tU8  u8GetNoOfMIXEDpresetPerBank( tVoid) const       { return m_rNewKdsGenericTunerParameter.u8NumberOfPresetsPerBankForMixedAllBands; }

  tU32 u32GetPreset( tU32 u32Index) const
  {
    if( u32Index < TUN_CONFIG_NUMBER_OF_PRESETS)
    {
      return m_rNewKdsGenericTunerParameter.au32Preset[ u32Index];
    }
    else
    {
      return 0;
    }
  }
 
  /*==========================================================================
   * Stage 1:
   * Tuning behaviour (not audio tuner dependent):
   *------------------------------------------------------------------------*/
  tU8  u8GetNumOfSeekPeriods( tVoid) const          { return m_rNewKdsFmAmTunerParameter.u8NumOfSeekPeriods; }
  tU8 u8GetScanTime( tVoid) const                   { return m_rNewKdsFmAmTunerParameter.u8ScanTime; }
  tU32 u32GetPtyFilterForSeek( tVoid) const          { return m_rNewKdsFmAmTunerParameter.u32PtyFilterForSeek; }
  //tU8  u8GetAfBestDelayTimeAfterSeek( tVoid) const  { return m_rKdsFmAmTunerParameter.u8AfBestDelayTimeAfterSeek; }
  tU8  u8GetTpSeekStrategy( tVoid) const            { return m_rNewKdsGenericTunerParameter.u8FM_TPseekStrategy; }
  tU16 u16GetAutoSeekStepDelay( tVoid) const        { return m_rNewKdsFmAmTunerParameter.u16AutoSeekStepDelay; }
  tU16 u16GetPemanentManualStepDelay( tVoid) const  { return m_rNewKdsFmAmTunerParameter.u16PemanentManualStepDelay; }

  //tU8  u8GetWrongPiAcceptDelay( tVoid) const        { return m_rKdsFmAmTunerParameter.u8WrongPiAcceptDelay; }
  //tU8  u8GetTuningSwitches( tVoid) const            { return m_rKdsFmAmTunerParameter.u8TuningSwitches; }
  tU16 u16GetMUTE_RAMP_1( tVoid) const              { return m_rNewKdsFmAmTunerParameter.u16MUTE_RAMP_1; }
  tU16 u16GetDEMUTE_RAMP_1( tVoid) const            { return m_rNewKdsFmAmTunerParameter.u16DEMUTE_RAMP_1; }
  tU16 u16GetMUTE_RAMP_2( tVoid) const              { return m_rNewKdsFmAmTunerParameter.u16MUTE_RAMP_2; }
  tU16 u16GetDEMUTE_RAMP_2( tVoid) const            { return m_rNewKdsFmAmTunerParameter.u16DEMUTE_RAMP_2; }
 
   /*==========================================================================
   * Stage 2:
   * Behaviour of  ATstation info auto notification (not audio tuner dependent):
   *------------------------------------------------------------------------*/
   tU8  u8GetATStationInfoSwitches( tVoid);

   /*==========================================================================
   * Stage 3:
   * Timer Configuration (not audio tuner dependent):
   *------------------------------------------------------------------------*/

  tU8 u8GetTimerScanRadio( tVoid) const					{ return m_rNewKdsFmAmTunerParameter.u8TimerScanRadio; }
  tU8 u8GetTimerInhibitLstSortAfterSkip( tVoid) const	{ return m_rNewKdsFmAmTunerParameter.u8TimerInhibitLstSortAfterSkip; }
  tU8 u8GetTimerDelayTPstnSelAfterVol0( tVoid) const	{ return m_rNewKdsGenericTunerParameter.u8TimeDelayFM_TPStnAftVolZero; }
  tU8 u8GetTimerDelayNoTPstnFound( tVoid) const			{ return m_rNewKdsGenericTunerParameter.u8TimeDelayNoFM_TPStnFound; }
  tU8 u8GetTimerNonRDS_Detect ( tVoid) const			{ return m_rNewKdsFmAmTunerParameter.u8TimerNonRDS_Detect; }
  tU8 u8GetTimerPS_DelAfterRDSLost( tVoid) const		{ return m_rNewKdsFmAmTunerParameter.u8TimerPS_DelAfterRDSLost; }
  tU8 u8GetTimerWrongPIAccept( tVoid) const				{ return m_rNewKdsFmAmTunerParameter.u8WrongPIAcceptAftStnlost; } 
  tU8 u8GetTimerScrollPSDetect( tVoid) const			{ return m_rNewKdsFmAmTunerParameter.u8TimerScrollPSDetect; }
  tU8 u8GetTimerDelayTPstnSelAfterRDSLost( tVoid) const { return m_rNewKdsGenericTunerParameter.u8TimeDelayFM_TPStnSelAftRDSLost; }
  tU8 u8GetTimerTAendAfterRDSLost( tVoid) const			{ return m_rNewKdsGenericTunerParameter.u8TimeFM_TAEndAftRDSLost; }
  tU8 u8GetTimerAM_MinTunTime( tVoid) const				{ return m_rNewKdsFmAmTunerParameter.u8TimerAM_MinTunTime; }
  tU8 u8GetTimerCursorDelayAfterStnSel( tVoid) const	{ return m_rNewKdsFmAmTunerParameter.u8TimerCursorDelayAfterStnSel; }
  tU8 u8GetTimerAF_InhibitAfterManTune( tVoid) const	{ return m_rNewKdsFmAmTunerParameter.u8TimerAF_InhibitAfterManTune; }
  tU8 u8GetTimerAcceptPIAfterRecall( tVoid) const		{ return m_rNewKdsFmAmTunerParameter.u8WrongPIAcceptDelayAftStnSelection; }
  tU8 u8GetTimerPTY_Hold( tVoid) const					{ return m_rNewKdsFmAmTunerParameter.u8TimerPTY_Hold; }

  
  /*==========================================================================
   * Stage 4:
   * Parameters that influence the reception behaviour (not audio tuner dependent):
   *------------------------------------------------------------------------*/


  tU8 u8GetFieldStrengthSeekLevel_FM_RDS(tVoid) const		{ return m_rNewKdsFmAmTunerParameter.u8FieldStrengthSeekLevel_FM_RDS;}
  tU8 u8GetQualitySeekLevel_FM_RDS(tVoid) const;
  tU8 u8GetFieldStrengthSeekLevel_FM_NONRDS(tVoid) const	{ return m_rNewKdsFmAmTunerParameter.u8FieldStrengthSeekLevel_FM_NONRDS;}
  tU8 u8GetQualitySeekLevel_FM_NONRDS(tVoid) const;
  tU8 u8GetFieldStrengthSeekLevel_LW(tVoid) const			{ return m_rNewKdsFmAmTunerParameter.u8FieldStrengthSeekLevel_LW; }
  tU8 u8GetQualitySeekLevel_LW(tVoid) const					{ return m_rNewKdsFmAmTunerParameter.u8QualitySeekLevel_LW; }
  tU8 u8GetFieldStrengthSeekLevel_MW(tVoid) const			{ return m_rNewKdsFmAmTunerParameter.u8FieldStrengthSeekLevel_MW; }
  tU8 u8GetQualitySeekLevel_MW(tVoid) const;
  tU8 u8GetFieldStrengthSeekLevel_SW(tVoid) const			{ return m_rNewKdsFmAmTunerParameter.u8FieldStrengthSeekLevel_SW; }
  tU8 u8GetQualitySeekLevel_SW(tVoid) const					{ return m_rNewKdsFmAmTunerParameter.u8QualitySeekLevel_SW; }
  tU8 u8GetFieldStrengthSeekLevel_TRF(tVoid) const			{ return m_rNewKdsFmAmTunerParameter.u8FieldStrengthSeekLevel_TRF; }
  tU8 u8GetQualitySeekLevel_TRF(tVoid) const				{ return m_rNewKdsFmAmTunerParameter.u8QualitySeekLevel_TRF; }
  tU8 u8GetFieldStrengthSeekLevel_WB(tVoid) const			{ return m_rNewKdsFmAmTunerParameter.u8FieldStrengthSeekLevel_WB; }
  tU8 u8GetQualitySeekLevel_WB(tVoid) const					{ return m_rNewKdsFmAmTunerParameter.u8QualitySeekLevel_WB; }

  tU8 u8GetFieldStrengthListLevel_FM_RDS(tVoid) const		{ return m_rNewKdsFmAmTunerParameter.u8FieldStrengthListLevel_FM_RDS;  }
  tU8 u8GetQualityListLevel_FM_RDS(tVoid) const;
  tU8 u8GetFieldStrengthListLevel_FM_NONRDS(tVoid) const	{ return m_rNewKdsFmAmTunerParameter.u8FieldStrengthListLevel_FM_NONRDS; }
  tU8 u8GetQualityListLevel_FM_NONRDS(tVoid) const;
  tU8 u8GetFieldStrengthListLevel_LW(tVoid) const			{ return m_rNewKdsFmAmTunerParameter.u8FieldStrengthListLevel_LW; }
  tU8 u8GetQualityListLevel_LW(tVoid) const					{ return m_rNewKdsFmAmTunerParameter.u8QualityListLevel_LW; }
  tU8 u8GetFieldStrengthListLevel_MW(tVoid) const			{ return m_rNewKdsFmAmTunerParameter.u8FieldStrengthListLevel_MW; }
  tU8 u8GetQualityListLevel_MW(tVoid) const;
  tU8 u8GetFieldStrengthListLevel_SW(tVoid) const			{ return m_rNewKdsFmAmTunerParameter.u8FieldStrengthListLevel_SW; }
  tU8 u8GetQualityListLevel_SW(tVoid) const					{ return m_rNewKdsFmAmTunerParameter.u8QualityListLevel_SW; }
  tU8 u8GetFieldStrengthListLevel_TRF(tVoid) const			{ return m_rNewKdsFmAmTunerParameter.u8FieldStrengthListLevel_TRF; }
  tU8 u8GetQualityListLevel_TRF(tVoid) const				{ return m_rNewKdsFmAmTunerParameter.u8QualityListLevel_TRF; }
  tU8 u8GetFieldStrengthListLevel_WB(tVoid) const			{ return m_rNewKdsFmAmTunerParameter.u8FieldStrengthListLevel_WB; }
  tU8 u8GetQualityListLevel_WB(tVoid) const					{ return m_rNewKdsFmAmTunerParameter.u8QualityListLevel_WB; }

  tU8  u8GetFM_SHARX( tVoid) const                  { return m_rNewKdsFmAmTunerParameter.u8SHARX & 0x0F; }
  tU8  u8GetAM_SHARX( tVoid) const                  { return (tU8)(m_rNewKdsFmAmTunerParameter.u8SHARX >> 4); }
  tU8  u8GetFM_HICUT( tVoid) const                  { return m_rNewKdsFmAmTunerParameter.u8HICUT & 0x0F; }
  tU8  u8GetAM_HICUT( tVoid) const                  { return (tU8)(m_rNewKdsFmAmTunerParameter.u8HICUT >> 4); }

  /*==========================================================================
   * Stage 5:
   * Area setup for CONFIG area (no CONFIG arear needed) (not audio tuner dependent):
   *------------------------------------------------------------------------*/

  tU32 u32GetAvailableBands( tVoid) const          { return m_rNewKdsFmAmTunerParameter.u32AvailableBands; }
  tU32 u32GetCombiBands( tVoid) const			   { return m_rNewKdsFmAmTunerParameter.u32CombiBands; }
  /** FM */
  tU32 u32GetMinFrqFM( tVoid) const                 { return m_rNewKdsFmAmTunerParameter.u32LowFM; }
  tU32 u32GetMaxFrqFM( tVoid) const                 { return m_rNewKdsFmAmTunerParameter.u32HighFM; }
  tU16  u16GetStepSizeFM( tVoid) const                { return m_rNewKdsFmAmTunerParameter.u8StepSizeFM; }
  tU16  u16GetStepSizeManualFM( tVoid) const          { return m_rNewKdsFmAmTunerParameter.u8StepSizeManualFM; }
  tU32 u32GetFM_LSM( tVoid) const                   { return m_rNewKdsFmAmTunerParameter.u32FM_LSM; }
  /** LW */
  tU32 u32GetMinFrqLW( tVoid) const                 { return m_rNewKdsFmAmTunerParameter.u32LowLW; }
  tU32 u32GetMaxFrqLW( tVoid) const                 { return m_rNewKdsFmAmTunerParameter.u32HighLW; }
  tU16  u16GetStepSizeLW( tVoid) const                { return m_rNewKdsFmAmTunerParameter.u8StepSizeLW; }
  tU16  u16GetStepSizeManualLW( tVoid) const          { return m_rNewKdsFmAmTunerParameter.u8StepSizeManualLW; }
  tU32 u32GetLW_LSM( tVoid) const                   { return m_rNewKdsFmAmTunerParameter.u32LW_LSM; }
  /** MW */
  tU32 u32GetMinFrqMW( tVoid) const                 { return m_rNewKdsFmAmTunerParameter.u32LowMW; }
  tU32 u32GetMaxFrqMW( tVoid) const                 { return m_rNewKdsFmAmTunerParameter.u32HighMW; }
  tU16  u16GetStepSizeMW( tVoid) const                { return m_rNewKdsFmAmTunerParameter.u8StepSizeMW; }
  tU16  u16GetStepSizeManualMW( tVoid) const          { return m_rNewKdsFmAmTunerParameter.u8StepSizeManualMW; }
  tU32 u32GetMW_LSM( tVoid) const                   { return m_rNewKdsFmAmTunerParameter.u32MW_LSM; }
  /*==========================================================================
   * Stage 6:
   * User settings (audio tuner dependent):
   *------------------------------------------------------------------------*/

  tU8  u8TP_Switch_AudioTuner1( tVoid) const			{ return (( m_rNewKdsFmAmTunerParameter.u8ADRSwitches1 & 0x01) != 0) ? 0x01 : 0x00; }
  tU8  u8TP_Switch_AudioTuner2( tVoid) const			{ return (( m_rNewKdsFmAmTunerParameter.u8ADRSwitches1 & 0x02) != 0) ? 0x01 : 0x00; }
  tU8  u8AF_Switch_AudioTuner1( tVoid) const;
  tU8  u8AF_Switch_AudioTuner2( tVoid) const            { return (( m_rNewKdsFmAmTunerParameter.u8ADRSwitches1 & 0x08) != 0) ? 0x01 : 0x00; }
  tenDDA_Switch enDDA_Switch_AudioTuner1( tVoid) const; 
  tenDDA_Switch enDDA_Switch_AudioTuner2( tVoid) const  { return (tenDDA_Switch)(((m_rNewKdsFmAmTunerParameter.u8ADRSwitches1 >> 6) & 0x03) ? 0x01 : 0x00); }
  tU8  u8RDS_Switch_AudioTuner1( tVoid) const;
  tU8  u8RDS_Switch_AudioTuner2( tVoid) const			{ return (( m_rNewKdsFmAmTunerParameter.u8ADRSwitches2 & 0x02) != 0) ? 0x01 : 0x00; }
  tU8  u8PS_FIX_Switch_AudioTuner1( tVoid) const		{ return (( m_rNewKdsFmAmTunerParameter.u8ADRSwitches2 & 0x04) != 0) ? 0x01 : 0x00; }
  tU8  u8PS_FIX_Switch_AudioTuner2( tVoid) const		{ return (( m_rNewKdsFmAmTunerParameter.u8ADRSwitches2 & 0x08) != 0) ? 0x01 : 0x00; }
  tU8  u8Stereo_Switch_AudioTuner1( tVoid) const		{ return (( m_rNewKdsFmAmTunerParameter.u8ADRSwitches2 & 0x10) != 0) ? 0x01 : 0x00; }
  tU8  u8Stereo_Switch_AudioTuner2( tVoid) const		{ return (( m_rNewKdsFmAmTunerParameter.u8ADRSwitches2 & 0x20) != 0) ? 0x01 : 0x00; }
  tenREG_Switch enREG_Switch_AudioTuner1( tVoid) const	{ return (tenREG_Switch)(m_rNewKdsFmAmTunerParameter.u8ADRSwitches3 & 0x07); }
  tenREG_Switch enREG_Switch_AudioTuner2( tVoid) const	{ return (tenREG_Switch)((m_rNewKdsFmAmTunerParameter.u8ADRSwitches3 >> 3) & 0x07); }
 
  tU8  u8PTY31_Switch_AudioTuner1( tVoid) const;
  tU8  u8PTY31_Switch_AudioTuner2( tVoid) const			{ return (( m_rNewKdsFmAmTunerParameter.u8ADRSwitches4 & 0x02) != 0) ? 0x01 : 0x00; }
  tU8  u8DDS_Switch_AudioTuner1( tVoid) const			{ return (( m_rNewKdsFmAmTunerParameter.u8ADRSwitches4 & 0x04) != 0) ? 0x01 : 0x00; }
  tU8  u8DDS_Switch_AudioTuner2( tVoid) const			{ return (( m_rNewKdsFmAmTunerParameter.u8ADRSwitches4 & 0x08) != 0) ? 0x01 : 0x00; }
  tU8  u8DMQ_Switch_AudioTuner1( tVoid) const			{ return (( m_rNewKdsFmAmTunerParameter.u8ADRSwitches4 & 0x10) != 0) ? 0x01 : 0x00; }
  tU8  u8DMQ_Switch_AudioTuner2( tVoid) const			{ return (( m_rNewKdsFmAmTunerParameter.u8ADRSwitches4 & 0x20) != 0) ? 0x01 : 0x00; }

   /*==========================================================================
   * Stage 7:
   * HF Parameter (not physical tuner dependent):
   *------------------------------------------------------------------------*/
 
  tU8  u8GetFmChannelSeparation( tVoid) const       { return m_rNewKdsFmAmTunerParameter.u8ADRFmChannelSeparation; }
  tU8  u8GetFmLimiting( tVoid) const                { return m_rNewKdsFmAmTunerParameter.u8ADRFmLimiting; }
  tU8  u8GetFmMultipath( tVoid) const               { return m_rNewKdsFmAmTunerParameter.u8ADRFmMultipath; }
  tU8  u8GetFmSoftMute( tVoid) const                { return m_rNewKdsFmAmTunerParameter.u8ADRFmSoftMute; }
  tU8  u8GetAmSoftMute( tVoid) const                { return m_rNewKdsFmAmTunerParameter.u8ADRAmSoftMute; }
  tU8  u8GetFmHighCut( tVoid) const                 { return m_rNewKdsFmAmTunerParameter.u8ADRFmHighCut; }
  tU8  u8GetAM_Hicut_Configuration( tVoid) const    { return m_rNewKdsFmAmTunerParameter.u8ADRAM_Hicut_Configuration; }
  tU8  u8GetFmDeemphasis( tVoid) const              { return m_rNewKdsFmAmTunerParameter.u8ADRFmDeemphasis; }
  tU8  u8GetAM_Deemphasis( tVoid) const             { return m_rNewKdsFmAmTunerParameter.u8ADRAM_Deemphasis; }
  tU8  u8GetFmFrequencyResponse( tVoid) const       { return m_rNewKdsFmAmTunerParameter.u8ADRFmFrequencyResponse; }
  tU8  u8GetAmFrequencyResponse( tVoid) const       { return m_rNewKdsFmAmTunerParameter.u8ADRAmFrequencyResponse; }
  tU8  u8GetFmMisc( tVoid) const                    { return m_rNewKdsFmAmTunerParameter.u8ADRFmMisc; }
  tU8  u8GetAmMisc( tVoid) const                    { return m_rNewKdsFmAmTunerParameter.u8ADRAmMisc; }
  tU8  u8GetFmAmplificationLevel( tVoid) const      { return m_rNewKdsFmAmTunerParameter.u8ADRFmAmplificationLevel; }
  tU8  u8GetAmAmplificationLevel( tVoid) const      { return m_rNewKdsFmAmTunerParameter.u8ADRAmAmplificationLevel; }
  tenFmDabHdTunerMode enGetPhysicalTunerConfigId( tVoid) const     { return m_rKdsCMVariantCoding.enFmDabHdTunerMode; }
  tU8  u8GetAntennaConfigId( tVoid) const           { return m_rNewKdsFmAmTunerParameter.u8AntennaConfigId; }
#ifdef VARIANT_S_FTR_ENABLE_ADR_API_3_20
  tU8  u8GetAMFMMisc2( tVoid) const           { return m_rNewKdsFmAmTunerParameter.u8AMFMMisc2; }
#endif

  /*==========================================================================
  * Stage 8:
  * RDS Parameter
  *------------------------------------------------------------------------*/

  tU8  u8GetExchangeAntennaLevel( tVoid) const      { return m_rNewKdsFmAmTunerParameter.u8ExchangeAntennaLevel; }
  tU8  u8GetPitestLevel( tVoid) const               { return m_rNewKdsFmAmTunerParameter.u8PitestLevel; }
  tU8  u8GetNrdsLevel( tVoid) const                 { return m_rNewKdsFmAmTunerParameter.u8NrdsLevel; }
  tU8  u8GetEonLevel( tVoid) const                  { return m_rNewKdsFmAmTunerParameter.u8EonLevel; }
  tU8  u8GetFMMemoLevel( tVoid) const               { return m_rNewKdsFmAmTunerParameter.u8FMMemoLevel; }
  tU8  u8GetPanicFieldstLevel( tVoid) const         { return m_rNewKdsFmAmTunerParameter.u8PanicFieldstLevel; }
  tU8  u8GetParamRDSLevel( tVoid) const             { return m_rNewKdsFmAmTunerParameter.u8ParamRDSLevel; }
  tU8  u8GetRDSDataAcceptLevel( tVoid) const        { return m_rNewKdsFmAmTunerParameter.u8RDSDataAcceptLevel; }
  tU8  u8GetPIStepLevel( tVoid) const               { return m_rNewKdsFmAmTunerParameter.u8PIStepLevel; }
  tU8  u8GetDDAAntennaDistLevel( tVoid) const       { return m_rNewKdsFmAmTunerParameter.u8DDAAntennaDistLevel; }
  tU8  u8GetRDSSynDelayCnt( tVoid) const            { return m_rNewKdsFmAmTunerParameter.u8RDSSynDelayCnt; }
  tU8  u8GetMpMeasureLevel( tVoid) const            { return m_rNewKdsFmAmTunerParameter.u8MpMeasureLevel; }
  tU8  u8GetMpLevel2Level( tVoid) const             { return m_rNewKdsFmAmTunerParameter.u8MpLevel2Level; }
  tU8  u8GetMpVeryBadLevel( tVoid) const            { return m_rNewKdsFmAmTunerParameter.u8MpVeryBadLevel; }
  tU8  u8GetMpMediumLevel( tVoid) const             { return m_rNewKdsFmAmTunerParameter.u8MpMediumLevel; }
  tU8  u8GetRDSBadLevel( tVoid) const               { return m_rNewKdsFmAmTunerParameter.u8RDSBadLevel; }
  tU8  u8GetIllegalAf( tVoid) const                 { return m_rNewKdsFmAmTunerParameter.u8IllegalAf; }
  tU8  u8GetAfSaveLevel( tVoid) const               { return m_rNewKdsFmAmTunerParameter.u8AfSaveLevel; }
  tU8  u8GetFirstRundum( tVoid) const               { return m_rNewKdsFmAmTunerParameter.u8FirstRundum; }
  tU8  u8GetSecondRundum( tVoid) const              { return m_rNewKdsFmAmTunerParameter.u8SecondRundum; }
  tU8  u8GetThirdRundum( tVoid) const               { return m_rNewKdsFmAmTunerParameter.u8ThirdRundum; }
  tU8  u8PIStepMode( tVoid) const                   { return m_rNewKdsFmAmTunerParameter.u8PIStepMode; }
  tU8  u8GetAudioOffScanMode( tVoid) const          { return m_rNewKdsFmAmTunerParameter.u8AudioOffScanMode; }
  tU8  u8GetAMLearnCycle( tVoid) const              { return m_rNewKdsFmAmTunerParameter.u8AMLearnCycle; }
  tU8  u8GetPIIgnore( tVoid) const;
  /** Free1 Byte */
  tU8  u8GetPISearchTime( tVoid) const        		{ return m_rNewKdsFmAmTunerParameter.u8PISearchTime; }
  /** Free2 Byte */
  tU8  u8GetDisableSeamlessBlending( tVoid) const   { return m_rNewKdsFmAmTunerParameter.u8DisableSeamlessblending; }
  tU8  u8GetSlAdjThreshold( tVoid) const            { return m_rNewKdsFmAmTunerParameter.u8SlAdjThreshold; }
  tU8  u8GetSlMpThreshold( tVoid) const             { return m_rNewKdsFmAmTunerParameter.u8SlMpThreshold; }
  tU8  u8GetSpecialFMStartFrequency( tVoid) const;
  tU8  u8GetFMSharxLimit( tVoid) const				{ return m_rNewKdsFmAmTunerParameter.u8FMSharxLimit; }
  tU8  u8GetHDScanEveryFrequency( tVoid) const		{ return m_rNewKdsFmAmTunerParameter.u8HDScanEveryFrequency; }
  tU8  u8GetHWSubConfig( tVoid) const				{ return m_rNewKdsFmAmTunerParameter.u8HWSubConfig; }  
  tU8  u8GetFM2DABDelayTime( tVoid) const				{ return m_rNewKdsFmAmTunerParameter.u8FM2DABDelayTime; }

  tU8  u8GetAMFMHicutExtender( tVoid) const								{ return m_rNewKdsFmAmTunerParameter.u8AMFMHicutExtender							;}
  tU8  u8Getu8FMSoftwareMuteTiming( tVoid) const						{ return m_rNewKdsFmAmTunerParameter.u8FMSoftwareMuteTiming							;}
  tU8  u8Getu8AMFMMisscellaneous2( tVoid) const							{ return m_rNewKdsFmAmTunerParameter.u8AMFMMisscellaneous2							;}
  tU8  u8Getu8FMHicutExtenderFieldstrengthThreshold( tVoid) const		{ return m_rNewKdsFmAmTunerParameter.u8FMHicutExtenderFieldstrengthThreshold		;}
  tU8  u8Getu8FMHicutExtenderFieldstrengthFilterOffset( tVoid) const	{ return m_rNewKdsFmAmTunerParameter.u8FMHicutExtenderFieldstrengthFilterOffset		;}
  tU8  u8Getu8FMChannelSeparationMaximum( tVoid) const					{ return m_rNewKdsFmAmTunerParameter.u8FMChannelSeparationMaximum					;}
  tU8  u8Getu8FMChannelSeparationTiming	( tVoid) const					{ return m_rNewKdsFmAmTunerParameter.u8FMChannelSeparationTiming					;}
  tU8  u8Getu8FMHicutFieldstrengthThreshold	( tVoid) const				{ return m_rNewKdsFmAmTunerParameter.u8FMHicutFieldstrengthThreshold				;}
  tU8  u8Getu8FMHicutTiming( tVoid) const								{ return m_rNewKdsFmAmTunerParameter.u8FMHicutTiming								;}
  tU8  u8Getu8FMDMQSensitivity	( tVoid) const							{ return m_rNewKdsFmAmTunerParameter.u8FMDMQSensitivity								;}
  tU8  u8Getu8FMDeviationAdaption( tVoid) const							{ return m_rNewKdsFmAmTunerParameter.u8FMDeviationAdaption							;}
  tU8  u8Getu8AMNoiseBlanker228kHzSensitivity( tVoid) const				{ return m_rNewKdsFmAmTunerParameter.u8AMNoiseBlanker228kHzSensitivity				;}
  tU8  u8Getu8AMNoiseBlanker45_6kHzSensitivity( tVoid) const			{ return m_rNewKdsFmAmTunerParameter.u8AMNoiseBlanker45_6kHzSensitivity				;}
  tU8  u8Getu8AMHicutExtenderFieldstrengthThreshold	( tVoid) const		{ return m_rNewKdsFmAmTunerParameter.u8AMHicutExtenderFieldstrengthThreshold		;}
  tU8  u8Getu8AMHicutExtenderFieldstrengthFilterOffset( tVoid) const	{ return m_rNewKdsFmAmTunerParameter.u8AMHicutExtenderFieldstrengthFilterOffset		;}
   
  
  tU8  u8GetDemodulationSwitchAudioTuner1( tVoid) const				{ return m_rNewKdsFmAmTunerParameter.u8DemodulationSwitchAudioTuner1				;}
  tU8  u8GetDemodulationSwitchAudioTuner2( tVoid) const				{ return m_rNewKdsFmAmTunerParameter.u8DemodulationSwitchAudioTuner2				;}
   /*==========================================================================
   * Stage 9:
   * HD Parameter configuration:
   *------------------------------------------------------------------------*/

  tU8	u8GetBlendThresholdFMHybrid( tVoid) const				{ return m_rKdsHDRadioParameterSet.u8BlendThresholdFMHybrid;				}
  tU16	u16GetDigiAudioScalingFMHybrid( tVoid) const			{ return m_rKdsHDRadioParameterSet.u16DigiAudioScalingFMHybrid;				}
  tU16	u16GetDigiAudioDelayFMHybridPrimaryBBRate( tVoid) const { return m_rKdsHDRadioParameterSet.u16DigiAudioDelayFMHybridPrimaryBBRate;	}
  tU8	u8GetBlendRateFMHybrid( tVoid) const					{ return m_rKdsHDRadioParameterSet.u8BlendRateFMHybrid;						}			
  tU8	u8GetBlendThresholdAMHybrid( tVoid) const				{ return m_rKdsHDRadioParameterSet.u8BlendThresholdAMHybrid;				}				
  tU16	u16GetAM_MPSAudioScaling( tVoid) const					{ return m_rKdsHDRadioParameterSet.u16AM_MPSAudioScaling;					}					
  tU8	u8GetAM_MPSBlendRate( tVoid) const						{ return m_rKdsHDRadioParameterSet.u8AM_MPSBlendRate;						}						
  tU16	u16GetDigiAudioDelayAM_MPSPrimaryBBRate( tVoid) const   { return m_rKdsHDRadioParameterSet.u16DigiAudioDelayAM_MPSPrimaryBBRate;	}	   
  tU8	u8GetDigiAudioPhaseInversion( tVoid) const				{ return m_rKdsHDRadioParameterSet.u8DigiAudioPhaseInversion;				}				
  tU8	u8GetBlendDecision( tVoid) const						{ return m_rKdsHDRadioParameterSet.u8BlendDecision;							}						
  tU8	u8GetFM_CDNO_BlendDecision( tVoid) const				{ return m_rKdsHDRadioParameterSet.u8FM_CDNO_BlendDecision;					}				
  tU8	u8GetAM_CDNO_BlendDecision( tVoid) const				{ return m_rKdsHDRadioParameterSet.u8AM_CDNO_BlendDecision;					}						
  tU8	u8GetFMSPSBlendThreshold( tVoid) const					{ return m_rKdsHDRadioParameterSet.u8FMSPSBlendThreshold;					}								
  tU16	u8GetFMSPSAudioScaling( tVoid) const					{ return m_rKdsHDRadioParameterSet.u16FMSPSAudioScaling;					}								
  tU8	u8GetFMSPSBlendRate( tVoid) const						{ return m_rKdsHDRadioParameterSet.u8FMSPSBlendRate;						}								
  tU8	u8GetBallgameModeAudioOutput( tVoid) const				{ return m_rKdsHDRadioParameterSet.u8BallgameModeAudioOutput;				}								
  tU8	u8GetFilteredDSQMTimeConstant( tVoid) const				{ return m_rKdsHDRadioParameterSet.u8FilteredDSQMTimeConstant;				}								
  tU8	u8GetEZBlendEnable( tVoid) const						{ return m_rKdsHDRadioParameterSet.u8EZBlendEnable;							}								
  tU16	u16GetDSQMSeekThreshold( tVoid) const					{ return m_rKdsHDRadioParameterSet.u16DSQMSeekThreshold;					}	
  tU8	u8GetHDImageType( tVoid) const							{ return m_rKdsHDRadioParameterSet.u8HDImageType;							}								
  tU8	u8GetHDPSDFieldBitMask( tVoid) const						{ return m_rKdsHDRadioParameterSet.u8HDPSDFieldBitMask;						}								
  tU8	u8GetHDPSDCommentsBitMask( tVoid) const					{ return m_rKdsHDRadioParameterSet.u8HDPSDCommentsBitMask;					}								
  tU8	u8GetHDPSDCommercialBitMask( tVoid) const				{ return m_rKdsHDRadioParameterSet.u8HDPSDCommercialBitMask;				}								
  tU8	u8GetHDPSD_UFIDBitMask( tVoid) const					{ return m_rKdsHDRadioParameterSet.u8HDPSD_UFIDBitMask;						}		
  tU8	u8GetTimer1( tVoid) const								{ return m_rKdsHDRadioParameterSet.u8Timer1;								}								
  tU8	u8GetTimer2( tVoid) const								{ return m_rKdsHDRadioParameterSet.u8Timer2;								}								
  tU8	u8GetTimer3( tVoid) const								{ return m_rKdsHDRadioParameterSet.u8Timer3;								}								
  tU8	u8GetTimer4( tVoid) const								{ return m_rKdsHDRadioParameterSet.u8Timer4;								}								
  tU8	u8GetTimer5( tVoid) const								{ return m_rKdsHDRadioParameterSet.u8Timer5;								}		

   
   
   /*==========================================================================
   * Stage 9:
   * DRM Tuner Parameter configuration:
   *------------------------------------------------------------------------*/

  tU8	u8GetDRMBandEnable( tVoid) const							{ return m_rKdsDRMTunerParameterSet.u8DRMBandEnable;									}
  tU8	u8GetDRMMWTunningStepSizeForSearch( tVoid) const			{ return m_rKdsDRMTunerParameterSet.u8DRMMWTunningStepSizeForSearch;				}
  tU8	u8GetDRMMWTunningStepSizeForManualStep( tVoid) const		{ return m_rKdsDRMTunerParameterSet.u8DRMMWTunningStepSizeForManualStep;			}
  tU8	u8GetDRMServiceFollowing( tVoid) const						{ return m_rKdsDRMTunerParameterSet.u8DRMServiceFollowing;							}			
  tU8	u8GetDRM_DRMAFSearchDelayTime( tVoid) const					{ return m_rKdsDRMTunerParameterSet.u8DRM_DRMAFSearchDelayTime;						}				
  tU8	u8GetDRMServiceListUnlearnTimeThreshold( tVoid) const		{ return m_rKdsDRMTunerParameterSet.u8DRMServiceListUnlearnTimeThreshold;			}				
  tU8	u8GetDRMServiceListLearnAcceptanceTime( tVoid) const		{ return m_rKdsDRMTunerParameterSet.u8DRMServiceListLearnAcceptanceTime;				}						
  tU8	u8GetDRMServiceListLearnAcceptanceThreshold( tVoid) const	{ return m_rKdsDRMTunerParameterSet.u8DRMServiceListLearnAcceptanceThreshold;		}				
  tU8	u8GetDRMQualityThresholdForAudioMute( tVoid) const			{ return m_rKdsDRMTunerParameterSet.u8DRMQualityThresholdForAudioMute;				}				
  tU8	u8GetDRMAudioErrorConcealmentLevel( tVoid) const			{ return m_rKdsDRMTunerParameterSet.u8DRMAudioErrorConcealmentLevel;					}								
  tS8 s8GetDRMMWSeekStopFieldStrengthThreshold(tVoid)const                    {return m_rKdsDRMTunerParameterSet.s8DRMMWSeekStopFieldStrengthThreshold;                     }
  tU8 u8GetDRMSwitches(tVoid)const                                      { return m_rKdsDRMTunerParameterSet.u8DRMSwitches; }

   
  /*==========================================================================
   * Stage 10:
   * Testmode and AF-List configuration:
   *------------------------------------------------------------------------*/
  tU8 u8GetTestModeDataUpdateInterval(tVoid) const	{ return m_rNewKdsFmAmTunerParameter.u8TestModeDataUpdateInterval; }
  tU8 u8GetAFListUpdateInterval(tVoid) const		{ return m_rNewKdsFmAmTunerParameter.u8AFListUpdateInterval; }
  tU8 u8GetAFListReleaseBehaviour(tVoid) const		{ return m_rNewKdsFmAmTunerParameter.u8AFListReleaseBehaviour; }

  /*==========================================================================
  * Stage 11:
  * Station list configuration:
  *------------------------------------------------------------------------*/
  tU8  u8GetMaxNumberofStationsInStnList(tVoid) const { return m_rNewKdsFmAmTunerParameter.u8MaxNumberofStationsInStnList; }


  tBool bIsHDSupported() const;

  tU8 bIsDRMSupported() const;

  tBool fGetLandscapeSeekNeeded( tVoid) const;

  tU8  u8GetPresetListType(tVoid) const { return m_rNewKdsGenericTunerParameter.u8PresetListType; }
  tU32 u32GetLSMFrequency(midw_fi_tcl_e8_Tun_TunerBand::tenType midwTunerBand);
  tVoid vGetFrequencyRangeToDeleteElement(tU32 &u32StartFrequency, tU32 &u32EndFrequency, midw_fi_tcl_e8_Tun_TunerBand::tenType enBand);

  //Configuration function for various KDS parameters
  tVoid vReadKDSConfig( tenKDSParameter enKDSParameter);




  //tBool fGetTunerMasterAvailability( tVoid) const { return (( m_rKdsFmAmTunerParameter.u8TunerMaster & DIA_TUNER_MASTER_AVAILABLE) != 0)     ? TRUE : FALSE; }
  tBool fGetTunerMasterAvailability( tVoid) const { return FALSE; }
  
  tU8 fGetTunerMode() const;
  tVoid vTraceCommand( tPU8 pu8Msg, tU32 u32Len );

  /** Maping of HMI List ID with ADR List Id */
  typedef std::map<tU8, tU16> mapListIDs;
  mapListIDs m_mapHMIListIDtoADRListID;

  /** Maping of HMI List ID with ADR List Id */
  typedef std::map<tU8, tU8> mapListIDBands;
  mapListIDBands m_mapHMIListIDtoBand;

  /** Get HMI list Id by giving ADR List ID */
  tU8 u8GetHMIListIDFromMap(const tU16 u16ADRListID);

  /** Get ADR list Id by giving HMI List ID */
  tU16 u16GetADRListIDFromMap(const tU8 u8HMIListID);
  
  /** Get Band associated with HMI list Id */
  tU8 u8GetBandForHMIListID(const tU8 u8HMIListID);

  /** Get Default HMIListId for Band */
  tU8 u8GetDefaultHMIListIDForBand(const tU8 u8Band);

  /** Get MaxNoOfStations for autostore*/
  tU8 u8GetMaxNoOfStationsForAutostore(tVoid);
  /** Get Behavioural switch for autostore*/
  tU8 u8GetBehaviouralSwitchForAutostore(tVoid);
  /** Get unused preset value for autostore*/
  tU8 u8GetUnusedPresetForAutostore(tVoid);
  /** Get sorting criteria value for autostore*/
  tU8 u8GetSortingCriteriaForAutostore(tU16 u16FunctionIdOfList);
  tU8 u8GetNoOfPresetsPerBank(tU16 U16PresetListID);

  tU8 u8GetTuningBehaviourSwitches();

#if FEATURE_VW_MIBE

  tVoid vUpdateFreqAccording2EOLFMTunerBand();
  tVoid vUpdateFreqAccording2EOLAMTunerBand();
#endif

#if FEATURE_PSA_RCC
  tVoid vReadKDSEOLConfigAudio();
  tVoid vUpdateAvailableBandsAccording2EOL();
  tVoid vUpdateFrequency_StepSizeFromEOL();
  tVoid vUpdateDeemphasisAccording2Country();
  tVoid vUpdateFieldStrengthSeekLevel();
#endif 

  tVoid vSet_tun_ConfigInterface_Ptr(clConfigInterface* poclConfigInterface);

  tVoid vGetAutostoreSortingCriteriaFM(tU8 &AutostoreSortingCriteriaFM);
  tVoid vGetAutostoreSortingCriteriaAM(tU8 &AutostoreSortingCriteriaAM);
  tVoid vGetTunerSwitches(tU8 &TunerSwitches);
  tVoid vGetATStationInfoSwitches(tU8 &ATStationInfoSwitches);
  tVoid vGetBehaviourSwitches(tU8 &BehaviourSwitches);
  tVoid vGetBackgroundFMTA(tU32 &BackgroundFMTA);
  tU8 u8GetPresetListType(tVoid);

private:

  tun_trKdsCMVariantCoding   m_rKdsCMVariantCoding;

  clConfigInterface* 	m_poclConfigInterface;

    /** --------------- New AMFMParam Set ----------------------- */
  tun_trNewKdsFmAmTunerParameter      m_rNewKdsFmAmTunerParameter;
  tun_trNewKdsGenericTunerParameter   m_rNewKdsGenericTunerParameter;
  /** --------------------------------------------------------- */
  
  /** HD Parameter Set */
  tun_trKdsHDRadioParameterSet   m_rKdsHDRadioParameterSet;

  /** DRM Parameter Set */
  tun_trKdsDRMTunerParameterSet   m_rKdsDRMTunerParameterSet;

  //KDS read and print trace function
  tS32 s32ReadNewKDS(OSAL_tIODescriptor IOKdsHandle, tU8* u8Buf, tU16 u16KdsKey, tU16 u16Len) const;
  


#if FEATURE_PSA_RCC

  tun_trKdsEOLConfigRadioParameter m_rKdsEOLConfigRadioParameter;
  tun_trKdsEOLConfigAudioParameter m_rKdsEOLConfigAudioParameter;

#endif
};

#endif /* TUN_CONFIG_H */
/*** END OF FILE *************************************************************/
