/************************************************************************
 * FILE:          clAars_DRMAudioStreamDecoderStatus.cpp
 * PROJECT:        g3g
 * SW-COMPONENT:   fc_tmctuner
 *----------------------------------------------------------------------
 *
 * DESCRIPTION:   Implementation of the Class clAars_DRMAudioStreamDecoderStatus
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
 * HISTORY:
 * Date      		 | Author                       | Modification
  1-jul-2015   
				
 *************************************************************************/


#include "clAars_DRMAudioStreamDecoderStatus.h"
#include "AarsApp/clAarsHandler.h"
#include "ADRMsg/ADRMsg_if.h"
#include "tuner_trace.h"
#include "drmPlayer/drmPlayerInterface.h"

unsigned  int clAars_DRMAudioStreamDecoderStatus::m_AudioFrameCRCFailCount_PreviousValue = 0;
unsigned  int clAars_DRMAudioStreamDecoderStatus::m_SuperFrameCRCFailCount_PreviousValue = 0;
unsigned int clAars_DRMAudioStreamDecoderStatus::m_AudioFrameCounter = 0;
unsigned int clAars_DRMAudioStreamDecoderStatus::m_SuperFrameCounter = 0;

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TUN_TRACE_CLASS_TUNAPP
#include "trcGenProj/Header/clAars_DRMAudioStreamDecoderStatus.cpp.trc.h"
#endif

#define MaxCounterValue 255

using namespace tuner::drm;

clAars_DRMAudioStreamDecoderStatus::clAars_DRMAudioStreamDecoderStatus(){

}


clAars_DRMAudioStreamDecoderStatus::~clAars_DRMAudioStreamDecoderStatus(){

}


clAars_DRMAudioStreamDecoderStatus::clAars_DRMAudioStreamDecoderStatus(tuner::AarsApp::clAarsHandler* poAarsHandler):
  tuner::AarsApp::clClientProperty(poAarsHandler, ADR_MSG_FID_CM_DRMAUDIOSTREAMDEOCDERSTATUS, ADR_MSG_CLIENT_AMFMTUNER, ADR_MSG_FBLOCK_AMFMTUNER, ADR_MSG_INSTANCE_ALL){

}


void clAars_DRMAudioStreamDecoderStatus::vOnNewStatus(clCMDRMAudioStreamDecoderStatusSet oData)
{
   ETG_TRACE_USR4(("clAars_DRMAudioStreamDecoderStatus::vSetDRMAudioDecoderValues"));
   vSet(oData);
}


void clAars_DRMAudioStreamDecoderStatus::vStatus(tunerVector<unsigned char> &/*bitStream*/){

}


void clAars_DRMAudioStreamDecoderStatus::vUpdateDatatoAudioStreamDecoderStatus(clCMDRMAudioLogicalFrameStatus* m_poFrame)
{
   drmPlayerInterface* player = drmPlayerInterface::getInstance();

    clCMDRMAudioStreamDecoderStatusSet oData;
    oData.AudioStreamType.enType = static_cast<clCMEnum_AudioStreamType::enAudioStreamType>(m_poFrame->AudioStreamType.enType);
    oData.FrameCntr = m_poFrame->FrameCntr;
    oData.NonReliabilityIndication.enType = static_cast<clCMEnum_NonReliabilityIndication::enNonReliabilityIndication>(player->bGetNonReliabilityStatus());
    if (m_poFrame->LenAudioInfoEntity >=3)
    {
       oData.AudioInfoEntityShort.assign(m_poFrame->AudioInfoEntity.begin(), m_poFrame->AudioInfoEntity.begin()+3);
    }
    else
    {
      ETG_TRACE_USR4(("clAars_DRMLogicalFrame::vUpdateDatatoAudioStreamDecoderStatus AudioInfoEntity is less than 3 bytes"));
      return;
    }
    // decoder: last frame OK = 1 and NOT_OK = 0.
    //ADR: audio not muted = 0. muted = 1. So negate the value and send to ADR.
    oData.MuteStatus.enType = static_cast<clCMEnum_MuteStatus::enMuteStatus>(!(player->bGetLastFrameStatus()));
    vCalculateCRCFailCount();
    if(oData.AudioStreamType.enType  == clCMEnum_AudioStreamType::DRM_xHE_AAC)
    {
       oData.SuperFrameCRCFailCounter = u32GetSuperFrameCRCFailCount();
    }
    else
    {
       //not applicable for AAC audio.
       oData.SuperFrameCRCFailCounter = 0;
    }
    oData.AudioFrameCRCFailCounter = u32GetAudioFrameCRCFailCount();
    ETG_TRACE_USR4(("vUpdateDatatoAudioStreamDecoderStatus MuteStatus=%d", oData.MuteStatus.enType));
    ETG_TRACE_USR4(("vUpdateDatatoAudioStreamDecoderStatus SuperFrameCRCFailCounter=%d", oData.SuperFrameCRCFailCounter));
    ETG_TRACE_USR4(("vUpdateDatatoAudioStreamDecoderStatus AudioFrameCRCFailCounter=%d", oData.AudioFrameCRCFailCounter));
    vOnNewStatus(oData);
}


void clAars_DRMAudioStreamDecoderStatus::vCalculateCRCFailCount()
{
   vCalculateSuperFrameCRCFailCount();
   vCalculateAudioCRCFailCount();
}


void clAars_DRMAudioStreamDecoderStatus::vCalculateSuperFrameCRCFailCount()
{
   drmPlayerInterface* player = drmPlayerInterface::getInstance();
   if(bIsSuperFrameCorruptCountSame() && bIsAudioNotMuted())
    {
      vResetSuperFrameCounterToZero();
    }
    else
    {
       if(u32GetSuperFrameCRCFailCount() < MaxCounterValue)
       {
          vSetSuperFrameCRCFailCount(u32GetSuperFrameCRCFailCount() +u32GetDifference_PreviousAndCurrentSuperFrameCRCFail());
          ETG_TRACE_USR4(("SuperFrameCounter = %d  SuperFrameCRCFailCount_PreviousValue=%d",u32GetSuperFrameCRCFailCount(), u32GetPreviousValueOfSuperFrameCRCFailCount()));
          if(u32GetSuperFrameCRCFailCount() >= MaxCounterValue)
           {
             vSetSuperFrameCRCFailCount(MaxCounterValue);
           }
        }
        else
        {
           vSetSuperFrameCRCFailCount(MaxCounterValue);
        }
       vSetPreviousValueOfSuperFrameCRCFailCount( player->u32GetSuperFrameCorruptCount());
    }
}

void clAars_DRMAudioStreamDecoderStatus::vCalculateAudioCRCFailCount()
{
   drmPlayerInterface* player = drmPlayerInterface::getInstance();
   if(bIsAudioFrameCorruptCountSame() && bIsAudioNotMuted())
     {
         vResetAudioFrameCounterToZero();
     }
     else
     {
        if(u32GetAudioFrameCRCFailCount() < MaxCounterValue)
        {
           vSetAudioFrameCRCFailCount(u32GetAudioFrameCRCFailCount() +u32GetDifference_PreviousAndCurrentAudioFrameCRCFail());
           ETG_TRACE_USR4(("AudioFrameCounter = %d  AudioFrameCRCFailCount_PreviousValue=%d",u32GetAudioFrameCRCFailCount(), u32GetPreviousValueOfAudioFrameCRCFailCount()));
           if(u32GetAudioFrameCRCFailCount()  >= MaxCounterValue)
           {
              vSetAudioFrameCRCFailCount( MaxCounterValue);
           }
        }
        else
        {
           vSetAudioFrameCRCFailCount( MaxCounterValue);
        }
        vSetPreviousValueOfAudioFrameCRCFailCount( player->u32GetAudioFrameCorruptCount());
     }
}


unsigned int clAars_DRMAudioStreamDecoderStatus::u32GetAudioFrameCRCFailCount()
{
   return m_AudioFrameCounter;
}


unsigned int clAars_DRMAudioStreamDecoderStatus::u32GetSuperFrameCRCFailCount()
{
   return m_SuperFrameCounter;
}


unsigned int clAars_DRMAudioStreamDecoderStatus::u32GetPreviousValueOfSuperFrameCRCFailCount()
{
   return m_SuperFrameCRCFailCount_PreviousValue;
}


unsigned int clAars_DRMAudioStreamDecoderStatus::u32GetPreviousValueOfAudioFrameCRCFailCount()
{
   return m_AudioFrameCRCFailCount_PreviousValue;
}


void clAars_DRMAudioStreamDecoderStatus::vSetSuperFrameCRCFailCount(unsigned int u32newValue)
{
   m_SuperFrameCounter = u32newValue;
}


void clAars_DRMAudioStreamDecoderStatus::vSetAudioFrameCRCFailCount(unsigned int u32NewValue)
{
   m_AudioFrameCounter = u32NewValue;
}


void clAars_DRMAudioStreamDecoderStatus::vSetPreviousValueOfAudioFrameCRCFailCount(unsigned int u32Newvalue)
{
   m_AudioFrameCRCFailCount_PreviousValue = u32Newvalue;
}


void clAars_DRMAudioStreamDecoderStatus::vSetPreviousValueOfSuperFrameCRCFailCount(unsigned int u32Newvalue)
{
   m_SuperFrameCRCFailCount_PreviousValue = u32Newvalue;
}


bool clAars_DRMAudioStreamDecoderStatus:: bIsSuperFrameCorruptCountSame()
{
   drmPlayerInterface* player = drmPlayerInterface::getInstance();
   if (player->u32GetSuperFrameCorruptCount() == u32GetPreviousValueOfSuperFrameCRCFailCount())
      return true;
   else
      return false;
}


bool clAars_DRMAudioStreamDecoderStatus:: bIsAudioFrameCorruptCountSame()
{
   drmPlayerInterface* player = drmPlayerInterface::getInstance();
   if (player->u32GetAudioFrameCorruptCount() == u32GetPreviousValueOfAudioFrameCRCFailCount())
      return true;
   else
      return false;
}


void clAars_DRMAudioStreamDecoderStatus::vResetSuperFrameCounterToZero()
{
   ETG_TRACE_USR4(("Reset the super frame counter to 0"));
   vSetSuperFrameCRCFailCount(0);
}


void clAars_DRMAudioStreamDecoderStatus::vResetAudioFrameCounterToZero()
{
   ETG_TRACE_USR4(("Reset the audio frame counter to 0"));
   vSetAudioFrameCRCFailCount(0);
}


unsigned int clAars_DRMAudioStreamDecoderStatus::u32GetDifference_PreviousAndCurrentSuperFrameCRCFail()
{
   drmPlayerInterface* player = drmPlayerInterface::getInstance();
   unsigned int u32DiffValue = player->u32GetSuperFrameCorruptCount() - u32GetPreviousValueOfSuperFrameCRCFailCount();
   return u32DiffValue;
}


unsigned int clAars_DRMAudioStreamDecoderStatus::u32GetDifference_PreviousAndCurrentAudioFrameCRCFail()
{
   drmPlayerInterface* player = drmPlayerInterface::getInstance();
   unsigned int u32DiffValue = player->u32GetAudioFrameCorruptCount() - u32GetPreviousValueOfAudioFrameCRCFailCount();
   return u32DiffValue;
}


bool clAars_DRMAudioStreamDecoderStatus::bIsAudioNotMuted()
{
   drmPlayerInterface* player = drmPlayerInterface::getInstance();
   if (player->bGetLastFrameStatus())
   {
      // last frame is OK. i.e audio is not muted
      return true;
   }
   else
   {

      ETG_TRACE_USR4(("Last frame is NOK. (audio is muted)"));
      // last frame is NOK. i.e audio is muted
      return false;
   }
}
