/************************************************************************
 * FILE:           clHDImageMemoryManagement.cpp
 * PROJECT:        g3g
 * SW-COMPONENT:   fc_tmctuner
 *----------------------------------------------------------------------
 *
 * DESCRIPTION:  Implementation of clHDImageMemoryManagement.cpp
 *----------------------------------------------------------------------
* COPYRIGHT:   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
*----------------------------------------------------------------------
 * HISTORY:
 * Date      		 | Author                       | Modification

 *************************************************************************/

#include "clHDImageMemoryManagement.h"
#include <dirent.h>		//parse directory
#include <sys/stat.h>	//get filename
#include <iterator>		//std::back_inserter
#include <algorithm>	//std::copy
#include <sstream>
#include <errno.h>
#include <sys/wait.h>
#include <unistd.h>
#include <time.h>

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#include "tun_trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TUN_TRACE_CLASS_TUNAPP
#include "trcGenProj/Header/clHDImageMemoryManagement.cpp.trc.h"
#endif

#define CURRENTDATESTRINGLENGTH		12
#define YEARBYTES					4
#define MMDDBYTES					3
#define DDMMBYTES					2
#define SPACE						0x20
#define STATIONLOGOFILENAMELENGTH	24
#define STATIONLOGODIRECTORYPATH	"/var/opt/bosch/dynamic/HDStationLogo/"
#define STATIONLOGOPARENTDIRECTORYPATH "/var/opt/bosch/dynamic/"
#define MAXSTATIONLOGOSINFLASH		100
#define MAXHDCALLSIGNLENGTH			6
#define MAXHDSTATIONIDLENGTH		9
#define SPACESTRING					" "
#define DATEOFFSET					10
#define BASE_10						10
#define JPG							".jpg"
#define PNG							".png"
#define STRINGEQUAL					0
#define JPEG						0
#define MAXIMUMQUEUESIZE			2
#define STATIONLOGODIRECTORYMODE	0755
#define RAMDIRECTORY				"/tmp/HDStationLogo/"
#define STATIONLOGOLENGTHCOMPARE	12

clHDImageMemoryManagement::clHDImageMemoryManagement()
{

}

clHDImageMemoryManagement::~clHDImageMemoryManagement()
{

}

/**
* Below function check image for given station ID is existing in MAP.
*/
bool clHDImageMemoryManagement::bIsImageExistsForGivenStationID(unsigned int u32StationID, unsigned char u8AudioProgram, unsigned short u16LotID)
{
	bool bImageExist = false;

	ETG_TRACE_USR4((" clHDImageMemoryManagement::bIsImageExistsForGivenStationID mStationIDQueue size = %u", mStationIDQueue.size()));
	for(StationIDQueue::iterator queueIt = mStationIDQueue.begin(); queueIt != mStationIDQueue.end(); queueIt++)
	{
		StationIDImageFileMap objStationIDImageFileMap = *queueIt;
		for (StationIDImageFileMapIterator loopIt = objStationIDImageFileMap.begin(); loopIt != objStationIDImageFileMap.end(); loopIt++)
		{
			if (loopIt->first == u32StationID)
			{
				bImageExist = bIsImageExistForGivenAudioProgram(u8AudioProgram, u16LotID, loopIt->second);
				break;
			}
		}
	}

	return bImageExist;
}

/**
* Below function check image for given AudioProgram is existing in MAP.
*/
bool clHDImageMemoryManagement::bIsImageExistForGivenAudioProgram(unsigned char u8AudioProgram, unsigned short u16LotID, AudioProgramImageDataMap objAudioProgramImageDataMap)
{
	bool bImageExist = false;

	ETG_TRACE_USR4((" clHDImageMemoryManagement::bIsImageExistForGivenAudioProgram AudioProgramImageDataMap size = %u", objAudioProgramImageDataMap.size()));
	for (AudioProgramImageDataMapIterator loopIt = objAudioProgramImageDataMap.begin(); loopIt != objAudioProgramImageDataMap.end(); loopIt++)
	{
		if (loopIt->first == u8AudioProgram)
		{
			bImageExist = bIsImageExistForGivenLotID(u16LotID, loopIt->second);
			break;
		}
	}

	return bImageExist;

}

/**
* Below function check image for given LOT ID is existing in Queue.
*/
bool clHDImageMemoryManagement::bIsImageExistForGivenLotID(unsigned short u16LotID, CoverArtImageDataQueue objCoverArtImageDataQueue)
{
	bool bImageExist = false;

	ETG_TRACE_USR4((" clHDImageMemoryManagement::bIsImageExistForGivenLotID CoverArtImageDataQueue size = %u", objCoverArtImageDataQueue.size()));
	for (CoverArtImageDataQueue::iterator loopIt = objCoverArtImageDataQueue.begin(); loopIt != objCoverArtImageDataQueue.end(); loopIt++)
	{
		if (loopIt->getLOTID() == u16LotID)
		{
			bImageExist = true;
			break;
		}
	}

	return bImageExist;
}

/**
* Below function check image for given station ID, AudioProgram and LOTID is existing in MAP.
*/
bool clHDImageMemoryManagement::bIsImageExist(unsigned int u32StationID, unsigned char u8AudioProgram, unsigned short u16LotID)
{
	return bIsImageExistsForGivenStationID(u32StationID, u8AudioProgram, u16LotID);
}

/**
* Get image for given station ID, AudioProgram and LOTID from MAP.
*/
clCMATHDImageFileUpdateStatus clHDImageMemoryManagement::objGetImageFromMap(unsigned int u32StationID, unsigned char u8AudioProgram, unsigned short u16LotID)
{
	clCMATHDImageFileUpdateStatus oData;
	bool bImageFound = false;
	ETG_TRACE_USR4((" clHDImageMemoryManagement::objGetImageFromMap queue size = %u", mStationIDQueue.size()));
	for(StationIDQueue::iterator loopIt = mStationIDQueue.begin(); loopIt != mStationIDQueue.end(); loopIt++)
	{
		StationIDImageFileMap range = *loopIt;
		for (StationIDImageFileMapIterator rangeIt = range.begin(); rangeIt != range.end(); rangeIt++)
		{
			if (rangeIt->first == u32StationID)
			{
				bImageFound = true;
				oData = oGetImageFileForGivenAudioProgram(u8AudioProgram, u16LotID, rangeIt->second);
				break;
			}
		}
		if(bImageFound == true)
		{
			break;
		}
	}
	return oData;
}

/**
* Add given image to MAP.
*/
void clHDImageMemoryManagement::vAddImageToMap(clCMATHDImageFileUpdateStatus& objHDImageFileStatus)
{
	ETG_TRACE_USR4(( "clHDImageMemoryManagement::vAddImageToMap() Outer Queue size = %u", mStationIDQueue.size()));
	if(mStationIDQueue.empty())
	{
		mStationIDQueue.push_back(vGetStationIDImageFileMap(objHDImageFileStatus));
	}
	else
	{
		bool bExist = false;
		for(StationIDQueue::iterator loopIt = mStationIDQueue.begin(); loopIt != mStationIDQueue.end(); loopIt++)
		{
			StationIDImageFileMap * pRange = &(*loopIt);
			for(StationIDImageFileMapIterator rangeIt = pRange->begin(); rangeIt != pRange->end(); rangeIt++)
			{
				if(rangeIt->first == objHDImageFileStatus.getHDStationId())
				{
					bExist = true;
					vAddImageToMap(&rangeIt->second, objHDImageFileStatus);
					break;
				}
			}
			if(bExist == true)
			{
				ETG_TRACE_USR4(( "clHDImageMemoryManagement::vAddImageToMap() added image to existing station ID = 0x%08x", objHDImageFileStatus.getHDStationId()));
				break;
			}
		}
		if(bExist == false)
		{
			ETG_TRACE_USR4(( "clHDImageMemoryManagement::vAddImageToMap() received image for station ID = 0x%08x", objHDImageFileStatus.getHDStationId()));
			if(mStationIDQueue.size() == MAXIMUMQUEUESIZE)
			{
				ETG_TRACE_USR4(( "clHDImageMemoryManagement::vAddImageToMap() deleted images of Station ID = %p", (void *)mStationIDQueue.begin()->begin()->first));
				mStationIDQueue.pop_front();
			}
			mStationIDQueue.push_back(vGetStationIDImageFileMap(objHDImageFileStatus));
		}
	}

}

/**
* Get StationID Map by inserting given image.
*/
StationIDImageFileMap clHDImageMemoryManagement::vGetStationIDImageFileMap(clCMATHDImageFileUpdateStatus& objHDImageFileStatus)
{
	StationIDImageFileMap obj;
	ETG_TRACE_USR4(( "clHDImageMemoryManagement::vGetStationIDImageFileMap() inserting image into queue StationID = 0x%08x", objHDImageFileStatus.getHDStationId()));
	obj.insert(StationIDImageFileMapPair(objHDImageFileStatus.getHDStationId(),vGetAudioProgramImageFileMap(objHDImageFileStatus)));
	return obj;
}

/**
* Get AudioProgram Map by inserting given image.
*/
AudioProgramImageDataMap clHDImageMemoryManagement::vGetAudioProgramImageFileMap(clCMATHDImageFileUpdateStatus& objHDImageFileStatus)
{
	AudioProgramImageDataMap obj;
	ETG_TRACE_USR4(( "clHDImageMemoryManagement::vGetAudioProgramImageFileMap() inserting into map Audio Program = %d", (int)objHDImageFileStatus.getHDAudioService()));
	obj.insert(AudioProgramImageDataMapPair((unsigned char)objHDImageFileStatus.getHDAudioService(), vGetImageDataQueue(objHDImageFileStatus)));
	return obj;
}

/**
* Get CoverArt Queue by inserting given image.
*/
CoverArtImageDataQueue clHDImageMemoryManagement::vGetImageDataQueue(clCMATHDImageFileUpdateStatus& objHDImageFileStatus)
{
	CoverArtImageDataQueue obj;
	ETG_TRACE_USR4(( "clHDImageMemoryManagement::vGetImageDataQueue() inserting into inner queue LOT ID = %d", objHDImageFileStatus.getLOTID()));
	obj.push_back(objHDImageFileStatus);
	return obj;
}

/**
* Add given image to AudioProgram MAP.
*/
void clHDImageMemoryManagement::vAddImageToMap(AudioProgramImageDataMap* objAudioProgramImageDataMap, clCMATHDImageFileUpdateStatus& objHDImageFileStatus)
{
	ETG_TRACE_USR4(( "clHDImageMemoryManagement::vAddImageToMap() add it to inner map size = %u", objAudioProgramImageDataMap->size()));
	bool bExist = false;
	for (AudioProgramImageDataMapIterator loopIt = objAudioProgramImageDataMap->begin(); loopIt != objAudioProgramImageDataMap->end(); loopIt++)
	{
		if (loopIt->first == objHDImageFileStatus.getHDAudioService())
		{
			bExist = true;
			vAddImageToMap(&loopIt->second, objHDImageFileStatus);
			break;
		}
	}
	if(bExist == false)
	{
		ETG_TRACE_USR4(( "clHDImageMemoryManagement::vAddImageToMap() received image for audio program = %d", (int)objHDImageFileStatus.getHDAudioService()));
		objAudioProgramImageDataMap->insert(AudioProgramImageDataMapPair((unsigned char)objHDImageFileStatus.getHDAudioService(), vGetImageDataQueue(objHDImageFileStatus)));
	}
}

/**
* Add given image to CoverArt Queue.
*/
void clHDImageMemoryManagement::vAddImageToMap(CoverArtImageDataQueue* pObjQueue, clCMATHDImageFileUpdateStatus& objHDImageFileStatus)
{
	if(pObjQueue->empty())
	{
		pObjQueue->push_back(objHDImageFileStatus);
		ETG_TRACE_USR4(( "clHDImageMemoryManagement::vAddImageToMap() inner queue empty (AudioProgram)%d enqueue LOTID = %d", objHDImageFileStatus.getHDAudioService(), objHDImageFileStatus.getLOTID()));
	}
	else
	{
		bool bImageExist = false;
		ETG_TRACE_USR4(( "clHDImageMemoryManagement::vAddImageToMap() inner queue not empty (AudioProgram)%d and size = %u", objHDImageFileStatus.getHDAudioService(), pObjQueue->size()));
		for(CoverArtImageDataQueue::iterator loopIt = pObjQueue->begin(); loopIt != pObjQueue->end(); loopIt++)
		{
			if(loopIt->getLOTID() == objHDImageFileStatus.getLOTID() && loopIt->getHDImageCoding() == objHDImageFileStatus.getHDImageCoding()
					&& loopIt->getHDImageType() == objHDImageFileStatus.getHDImageType())
			{
				bImageExist = true;
				if(bIsImageFileContentEqual(objHDImageFileStatus.getImageFileBytes(), loopIt->getImageFileBytes()) == false)
				{
					ETG_TRACE_USR4(( "clHDImageMemoryManagement::vAddImageToMap() replace image file LOTID = %d", objHDImageFileStatus.getLOTID()));
					loopIt->ImageFileBytes.clear();
					loopIt->ImageFileBytes.assign(objHDImageFileStatus.ImageFileBytes.begin(), objHDImageFileStatus.ImageFileBytes.end());
				}
				break;
			}
		}
		if(bImageExist == false)
		{
			ETG_TRACE_USR4(( "clHDImageMemoryManagement::vAddImageToMap() enqueue image file LOT ID = %d", objHDImageFileStatus.getLOTID()));
			if(pObjQueue->size() == MAXIMUMQUEUESIZE)
			{
				ETG_TRACE_USR4(( "clHDImageMemoryManagement::vAddImageToMap() dequeue image file LOT ID = %d", pObjQueue->front().getLOTID()));
				pObjQueue->pop_front();
			}
			pObjQueue->push_back(objHDImageFileStatus);
			ETG_TRACE_USR4(( "clHDImageMemoryManagement::vAddImageToMap() inner queue size = %u", pObjQueue->size()));
		}
	}

}

/**
* Check whether existing and received image contents are same.
*/
bool clHDImageMemoryManagement::bIsImageFileContentEqual(tunerVector<unsigned char> vReceivedImage, tunerVector<unsigned char> vExistingImage)
{
	return (vReceivedImage == vExistingImage);
}

/**
* Get image for given AudioProgram.
*/
clCMATHDImageFileUpdateStatus clHDImageMemoryManagement::oGetImageFileForGivenAudioProgram(unsigned char u8AudioProgram, unsigned short u16LotID, AudioProgramImageDataMap objAudioProgramImageDataMap)
{
	clCMATHDImageFileUpdateStatus oData;
	ETG_TRACE_USR4((" clHDImageMemoryManagement::oGetImageFileForGivenAudioProgram queue size = %u", objAudioProgramImageDataMap.size()));
	for (AudioProgramImageDataMapIterator loopIt = objAudioProgramImageDataMap.begin(); loopIt != objAudioProgramImageDataMap.end(); loopIt++)
	{
		if (loopIt->first == u8AudioProgram)
		{
			oData = oGetImageFileForGivenLotID(u16LotID, loopIt->second);
			break;
		}
	}
	return oData;
}

/**
* Get image for given LOTID.
*/
clCMATHDImageFileUpdateStatus clHDImageMemoryManagement::oGetImageFileForGivenLotID(unsigned short u16LotID, CoverArtImageDataQueue objCoverArtImageDataQueue)
{
	clCMATHDImageFileUpdateStatus oData;
	ETG_TRACE_USR4((" clHDImageMemoryManagement::oGetImageFileForGivenLotID queue size = %u", objCoverArtImageDataQueue.size()));
	for (CoverArtImageDataQueue::iterator loopIt = objCoverArtImageDataQueue.begin(); loopIt != objCoverArtImageDataQueue.end(); loopIt++)
	{
		if (loopIt->getLOTID() == u16LotID)
		{
			oData.HDApplicationMimeType = loopIt->HDApplicationMimeType;
			oData.HDAudioService = loopIt->HDAudioService;
			oData.HDImageCoding = loopIt->HDImageCoding;
			oData.HDImageType = loopIt->HDImageType;
			oData.HDStationId = loopIt->HDStationId;
			oData.ImageFileBytes.assign(loopIt->ImageFileBytes.begin(), loopIt->ImageFileBytes.end());
			oData.LOTID = loopIt->LOTID;
			oData.NumberOfImageFileBytes = loopIt->NumberOfImageFileBytes;
			break;
		}
	}
	return oData;
}

/**
* Flush images for given StationID and AudioProgram.
*/
void clHDImageMemoryManagement::vFlushImagesFromMap(unsigned int u32StationID, unsigned char u8AudioProgram)
{
	ETG_TRACE_USR4((" clHDImageMemoryManagement::vFlushImagesFromMap mStationIDQueue size = %u", mStationIDQueue.size()));
	for(StationIDQueue::iterator queueIt = mStationIDQueue.begin(); queueIt != mStationIDQueue.end(); queueIt++)
	{
		StationIDImageFileMap objStationIDImageFileMap = *queueIt;
		for (StationIDImageFileMapIterator loopIt = objStationIDImageFileMap.begin(); loopIt != objStationIDImageFileMap.end(); loopIt++)
		{
			if (loopIt->first == u32StationID)
			{
				vFlushImagesFromMap(u8AudioProgram, loopIt->second);
				break;
			}
		}
	}
}

/**
* Get Flush images for given AudioProgram.
*/
void clHDImageMemoryManagement::vFlushImagesFromMap(unsigned char u8AudioProgram, AudioProgramImageDataMap objAudioProgramImageDataMap)
{
	ETG_TRACE_USR4((" clHDImageMemoryManagement::vFlushImagesFromMap AudioProgramImageDataMap size = %u", objAudioProgramImageDataMap.size()));
	for (AudioProgramImageDataMapIterator loopIt = objAudioProgramImageDataMap.begin(); loopIt != objAudioProgramImageDataMap.end(); loopIt++)
	{
		if (loopIt->first == u8AudioProgram)
		{
			loopIt->second.clear();
			ETG_TRACE_USR4((" clHDImageMemoryManagement::vFlushImagesFromMap deleted images of AudioProgram = %d", u8AudioProgram));
			break;
		}
	}
}

clHDStationLogoManagement::clHDStationLogoManagement() : mNumberOfStationLogosInFlash(0)
{

}

/**
* Handle station logo operation, it invloves creating of HDStationLogo directory in both flash as well as RAM
* If directory already exists in flash, copy existing logos to RAM
* Avoid this operation in constructor as it consumes time during startup. It cause delaying of tuner service availablity.
*/
void clHDStationLogoManagement::vHandleHDStationLogo()
{
	DIR *stationLogoParentDirectory = NULL;
	struct dirent *DiretoryDetails = NULL;
	const char* strStationLogoParentDirectoryName = STATIONLOGOPARENTDIRECTORYPATH;
	const char* strStationLogoDirectoryName = STATIONLOGODIRECTORYPATH;
	bool bIsStationLogoPathExists = false;
	bool bIsStationLogoRAMPathExists = false;

	stationLogoParentDirectory = opendir(strStationLogoParentDirectoryName);

	int retValue = mkdir(RAMDIRECTORY, STATIONLOGODIRECTORYMODE);
	if(retValue < 0)
	{
		ETG_TRACE_USR1(( "clHDStationLogoManagement::clHDStationLogoManagement() Error creating directory "));
	}
	else
	{
		bIsStationLogoRAMPathExists = true;
	}

	if(!stationLogoParentDirectory)
	{
		ETG_TRACE_USR1(( "clHDStationLogoManagement::vGetNumberOfStationLogos() directory not found "));
	}
	else
	{
		while( (DiretoryDetails = readdir(stationLogoParentDirectory)) != NULL)
		{
			if( ! strcmp(DiretoryDetails->d_name, "HDStationLogo") )
			{
				bIsStationLogoPathExists = true;
			}
		}
		if(!bIsStationLogoPathExists)
		{
			retValue = mkdir(STATIONLOGODIRECTORYPATH, STATIONLOGODIRECTORYMODE);
			if(retValue < 0)
			{
				ETG_TRACE_USR1(( "clHDStationLogoManagement::clHDStationLogoManagement() Error creating directory "));
			}
		}
	}
	vGetNumberOfStationLogos(strStationLogoDirectoryName, bIsStationLogoRAMPathExists);
	if(stationLogoParentDirectory)
	{
		closedir(stationLogoParentDirectory);
	}
}

clHDStationLogoManagement::~clHDStationLogoManagement()
{
	mNumberOfStationLogosInFlash = 0;
	vStationLogoFiles.clear();
}

/**
* Do differential copy from Flash to RAM
*/
void clHDStationLogoManagement::vDoDifferentialCopyLogosFromRAMToFlash()
{
	ETG_TRACE_USR1(( "clHDStationLogoManagement::vDoDifferentialCopyLogosFromRAMToFlash()" ));

	char  *copy_argv[8];
	copy_argv[0] = (char *)"rsync";
	copy_argv[1] = (char *)"--no-whole-file";	// delta transfer, do not update complete file for a small change
	copy_argv[2] = (char *)"-c";				// skip based on checksum, not mod-time & size
	copy_argv[3] = (char *)"-t";				// preserve modification times
	copy_argv[4] = (char *)"--recursive";		// do update for entire folder
	copy_argv[5] = (char *)RAMDIRECTORY; //source - RAM path
	copy_argv[6] = (char *)STATIONLOGODIRECTORYPATH; //destination - flash memory
	copy_argv[7] = NULL;

	pid_t  pid;
	int   status;

	int u8Result = 0;
	pid = fork();

	if (pid < 0)
	{     /* fork a child process */
		  ETG_TRACE_ERR(("clHDStationLogoManagement::vDoDifferentialCopyLogosFromRAMToFlash() ERROR: forking child process failed"));
	}
	else if (pid == 0)
	{ /* child process execution*/
	  u8Result = execvp(*copy_argv, copy_argv);
	  if (u8Result < 0)
	  {
		  ETG_TRACE_ERR(("clHDStationLogoManagement::vDoDifferentialCopyLogosFromRAMToFlash() ERROR: execvp failed"));
		  exit(1);
	  }
	}
	else
	{
		while (wait(&status) != pid) // parent process wait for child process to complete
			;
	}
	ETG_TRACE_USR4(("clHDStationLogoManagement::vDoDifferentialCopyLogosFromRAMToFlash() Result : %d",u8Result));

}
/**
* Read number of Station Logo's from Flash memory and save station logo file name in vector
*/
void clHDStationLogoManagement::vGetNumberOfStationLogos(const char * strStationLogoDirectoryName, bool bIsStationLogoRAMPathExists)
{
	DIR *stationLogoDirectory = NULL;
	struct dirent *DiretoryDetails = NULL;


	stationLogoDirectory = opendir(strStationLogoDirectoryName);
	if(!stationLogoDirectory)
	{
		ETG_TRACE_USR1(( "clHDStationLogoManagement::vGetNumberOfStationLogos() directory not found "));
		return;
	}
	while( (DiretoryDetails = readdir(stationLogoDirectory)) != NULL)
	{
		if( (strncmp(DiretoryDetails->d_name, "SL", 2) == STRINGEQUAL) && (strlen(DiretoryDetails->d_name) >= STATIONLOGOFILENAMELENGTH))
		{
			mNumberOfStationLogosInFlash++;
			try
			{
				vStationLogoFiles.push_back(std::string(DiretoryDetails->d_name));
			}
			catch(std::bad_alloc&)
			{
				closedir(stationLogoDirectory);
				return;
			}
		}
	}
	if(stationLogoDirectory)
	{
		closedir(stationLogoDirectory);
	}
	ETG_TRACE_USR4(( "clHDStationLogoManagement::vGetNumberOfStationLogos() %d, MaximumAllowed - %d", mNumberOfStationLogosInFlash, MAXSTATIONLOGOSINFLASH));
	if(bIsStationLogoRAMPathExists && mNumberOfStationLogosInFlash)
	{
		vCopyLogoFromFlashToRAM();
	}
}

/**
* Copy logos from flash to RAM
*/
void clHDStationLogoManagement::vCopyLogoFromFlashToRAM()
{
	ETG_TRACE_USR1(( "clHDStationLogoManagement::vCopyLogoFromFlashToRAM()" ));
	char  *copy_argv[6];
	copy_argv[0] = (char *)"rsync";
	copy_argv[1] = (char *)"--recursive";		// do update for entire folder
	copy_argv[2] = (char *)"-t";				// preserve modification times
	copy_argv[3] = (char *)STATIONLOGODIRECTORYPATH; //source - flash memory
	copy_argv[4] = (char *)RAMDIRECTORY; //destination - RAM path
	copy_argv[5] = NULL;

	pid_t  pid;
	int   status;

	int u8Result = 0;
	pid = fork();

	if (pid < 0)
	{     /* fork a child process */
		  ETG_TRACE_ERR(("clHDStationLogoManagement::vCopyLogoFromFlashToRAM() ERROR: forking child process failed"));
	}
	else if (pid == 0)
	{ /* child process execution*/
	  u8Result = execvp(*copy_argv, copy_argv);
	  if (u8Result < 0)
	  {
		  ETG_TRACE_ERR(("clHDStationLogoManagement::vCopyLogoFromFlashToRAM() ERROR: execvp failed"));
		  exit(1);
	  }
	}
	else
	{
		while (wait(&status) != pid) // parent process wait for child process to complete
			;
	}
	ETG_TRACE_USR4(("clHDStationLogoManagement::vCopyLogoFromFlashToRAM() Result : %d",u8Result));
}

/**
* Handling of actions to be taken on station logo's whenver date is changed
*/
#ifdef VARIANT_S_FTR_ENABLE_VDCLOCK
void clHDStationLogoManagement::vOnVdClockStatus(clock_mainfi_tclMsgLocalTimeDate_MinuteUpdateStatus oMsg)
{
	m_oclVDClockData.vUpdateClockData(oMsg);
	ETG_TRACE_USR4(( "clHDStationLogoManagement::vOnVdClockStatus() Time %04d-%02d-%02d",
			m_oclVDClockData.s16Year, m_oclVDClockData.u8Month, m_oclVDClockData.u8Day));
	if(bIsDeletionCheckToBeRun() == true)
	{
		vHandleDeletionOfExpiredLogo();
	}
	else
	{
		ETG_TRACE_USR4(( "clHDStationLogoManagement::vOnVdClockStatus() Deletion check not required"));
	}
}
#endif

/**
* Is deletion check to be run for expired station logo's.
*/
bool clHDStationLogoManagement::bIsDeletionCheckToBeRun()
{
	bool bIsCheckDeletionRun = false;
	char sCurrentDate[CURRENTDATESTRINGLENGTH] = {0};
	clVDClockData oclVDClockData;

#ifndef VARIANT_S_FTR_ENABLE_DP_CLIENT_MOCK
	moDPCurrentDate.u32GetData(sCurrentDate, CURRENTDATESTRINGLENGTH);
#endif
	if(strlen(sCurrentDate) == 0)
	{
		ETG_TRACE_USR2(( "clHDStationLogoManagement::bIsDeletionCheckToBeRun() , datapoolRead error = %s ",sCurrentDate));
		return bIsCheckDeletionRun;
	}
	else
	{
		ETG_TRACE_USR4(( "clHDStationLogoManagement::bIsDeletionCheckToBeRun() , datapoolRead success = %s ",sCurrentDate));
		if(strcmp(sCurrentDate, "DEFAULT") == STRINGEQUAL)
		{
			vUpdateDateInDp();
			return bIsCheckDeletionRun;
		}
		vGetFormattedDate(oclVDClockData, sCurrentDate);
		ETG_TRACE_USR4(( "clHDStationLogoManagement::bIsDeletionCheckToBeRun() Time %04d-%02d-%02d",
				oclVDClockData.s16Year, oclVDClockData.u8Month, oclVDClockData.u8Day));
		if(bIsDateOlderThanSystemDate(oclVDClockData) == true)
		{
			bIsCheckDeletionRun = true;
		}
	}

	return bIsCheckDeletionRun;
}

/**
* Delete expired station logo's and update current date to datapool.
*/
void clHDStationLogoManagement::vHandleDeletionOfExpiredLogo()
{
	vDeleteExpiredStationLogo();
	vUpdateDateInDp();
}

/**
* Delete expired station logo's.
*/
void clHDStationLogoManagement::vDeleteExpiredStationLogo()
{
	DIR *stationLogoDirectory = NULL;
	struct dirent *DiretoryDetails = NULL;
	std::string strStationLogoFileName;

	stationLogoDirectory = opendir(RAMDIRECTORY);
	if(!stationLogoDirectory)
	{
		ETG_TRACE_USR1(( "clHDStationLogoManagement::vDeleteExpiredStationLogo() directory %s not found ", (char *)RAMDIRECTORY));
		return;
	}
	while( (DiretoryDetails = readdir(stationLogoDirectory)) != NULL)
	{
		if(bIsStationLogoExpired(DiretoryDetails->d_name) == true)
		{
			std::string strFileName = std::string(STATIONLOGODIRECTORYPATH) + std::string(DiretoryDetails->d_name);
			std::string strRAMFileName = std::string(RAMDIRECTORY) + std::string(DiretoryDetails->d_name);
			if(remove(strRAMFileName.c_str()) != 0)
			{
				ETG_TRACE_USR4(( "clHDStationLogoManagement::vDeleteExpiredStationLogo() %s File not deleted", strRAMFileName.c_str()));
			}
			else
			{
				vRemoveStationLogoFileNameFromVector(std::string(DiretoryDetails->d_name));
				ETG_TRACE_USR4(( "clHDStationLogoManagement::vDeleteExpiredStationLogo() %s File deleted", strRAMFileName.c_str()));
			}
			if((remove(strFileName.c_str()) != 0) && (errno != ENOENT))
			{
				ETG_TRACE_USR4(( "clHDStationLogoManagement::vDeleteExpiredStationLogo() %s File not deleted", strFileName.c_str()));
			}
		}
	}
	if(stationLogoDirectory)
	{
		closedir(stationLogoDirectory);
	}
}

/**
* Delete all logos
*/
void clHDStationLogoManagement::vDeleteAllLogosInPath(const char* strLogoPath)
{
	DIR *stationLogoDirectory = NULL;
	struct dirent *DiretoryDetails = NULL;
	std::string strStationLogoFileName;

	stationLogoDirectory = opendir(strLogoPath);
	if(!stationLogoDirectory)
	{
		ETG_TRACE_USR1(( "clHDStationLogoManagement::vDeleteAllLogosInPath() directory not found "));
		return;
	}
	while( (DiretoryDetails = readdir(stationLogoDirectory)) != NULL)
	{
		if(!strcmp((const char *)DiretoryDetails->d_name, ".") || !strcmp((const char *)DiretoryDetails->d_name, ".."))
		{
			continue;
		}
		std::string strFileName = std::string(strLogoPath) + std::string(DiretoryDetails->d_name);
		if(remove(strFileName.c_str()) != 0)
		{
			ETG_TRACE_USR4(( "clHDStationLogoManagement::vDeleteAllLogosInPath() %s File not deleted", strFileName.c_str()));
		}
		else
		{
			vRemoveStationLogoFileNameFromVector(std::string(DiretoryDetails->d_name));
			ETG_TRACE_USR4(( "clHDStationLogoManagement::vDeleteAllLogosInPath() %s File deleted", strFileName.c_str()));
		}
	}
	if(stationLogoDirectory)
	{
		closedir(stationLogoDirectory);
	}
}

/**
* Delete all station logos
*/
void clHDStationLogoManagement::vDeleteAllStationLogos()
{
	vDeleteAllLogosInPath(RAMDIRECTORY); //delete all logos from RAM
	vDeleteAllLogosInPath(STATIONLOGODIRECTORYPATH); // delete all logos from flash
}

/**
* Remove deleted filename from staion logo vector.
*/
void clHDStationLogoManagement::vRemoveStationLogoFileNameFromVector(std::string strFileName)
{
	for(std::vector<std::string>::iterator it = vStationLogoFiles.begin(); it != vStationLogoFiles.end(); it++)
	{
		if(strFileName == *it)
		{
			vStationLogoFiles.erase(it);
			ETG_TRACE_USR4(( "clHDStationLogoManagement::vRemoveStationLogoFileNameFromVector() %s FileName deleted", strFileName.c_str()));
			mNumberOfStationLogosInFlash--;
			break;
		}
	}
}

/**
* Check whether station logo expired.
*/
bool clHDStationLogoManagement::bIsStationLogoExpired(char *strStationLogoFileName)
{
	bool bExpired = false;
	char strTempFileName[CURRENTDATESTRINGLENGTH] = {0};
	int iSourceOffset = DATEOFFSET;
	int iDestOffset = 0;

	if( strncmp(strStationLogoFileName, "SL", 2) == STRINGEQUAL)
	{
		if(strlen(strStationLogoFileName) >= STATIONLOGOFILENAMELENGTH)
		{
			vExtractSubString(strTempFileName, iDestOffset, strStationLogoFileName, iSourceOffset, YEARBYTES);
			vExtractSubString(strTempFileName, iDestOffset, strStationLogoFileName, iSourceOffset, DDMMBYTES);
			vExtractSubString(strTempFileName, iDestOffset, strStationLogoFileName, iSourceOffset, DDMMBYTES);
			strTempFileName[iDestOffset] = '\0';

			clVDClockData oData;
			vGetFormattedDate(oData, strTempFileName);
			if(bIsDateOlderThanSystemDate(oData) == true)
			{
				bExpired = true;
			}
		}
	}

	return bExpired;
}

/**
* Extract sub string from given String with provided start and end offset.
*/
void clHDStationLogoManagement::vExtractSubString(char *dest, int& iDestOffset, const char *source, int& iSrcOffset, int iNumberOfBytesToBeCopied)
{
	strncpy(dest + iDestOffset, source + iSrcOffset, iNumberOfBytesToBeCopied);
	dest[iDestOffset + iNumberOfBytesToBeCopied] = SPACE;
	iDestOffset++;
	iDestOffset += iNumberOfBytesToBeCopied;
	iSrcOffset += iNumberOfBytesToBeCopied;
}

/**
* Update current date to datapool.
*/
void clHDStationLogoManagement::vUpdateDateInDp()
{
	std::string strSystemDate = strGetFormattedDate(m_oclVDClockData, true);
#ifndef VARIANT_S_FTR_ENABLE_DP_CLIENT_MOCK
	moDPCurrentDate.vSetData(const_cast<char*>(strSystemDate.c_str()));
#endif
	ETG_TRACE_USR2(( "clHDStationLogoManagement::vUpdateDateInDp() , sSystemDate = %s updated to DP",strSystemDate.c_str()));
}

/**
* Convert integer to string in required format.
*/
void clHDStationLogoManagement::vIntegerToString(char *strDestination, int& iStart, tU8 u8Source, int iLengthToBeCopied, bool bIsSpaceNeeded)
{
	if(((CURRENTDATESTRINGLENGTH)>iStart) && (CURRENTDATESTRINGLENGTH) > (iLengthToBeCopied+iStart))
{
	char str[4] = {0};
	if(bIsSpaceNeeded == true)
	{
		sprintf(str, " %02d", u8Source);
	}
	else
	{
		sprintf(str, "%02d", u8Source);
	}
	strncat(strDestination+iStart, str, iLengthToBeCopied);
	iStart += iLengthToBeCopied;
	}
}

/**
* Is datapool date older than current system date.
*/
bool clHDStationLogoManagement::bIsDateOlderThanSystemDate(const clVDClockData& oclVDClockData)
{
	bool bIsOlder = false;

	if(oclVDClockData.s16Year <= m_oclVDClockData.s16Year)
	{
		if(oclVDClockData.s16Year < m_oclVDClockData.s16Year)
		{
			bIsOlder = true;
		}
		else if(oclVDClockData.u8Month <= m_oclVDClockData.u8Month)
		{
			if(oclVDClockData.u8Month < m_oclVDClockData.u8Month)
			{
				bIsOlder = true;
			}
			else if(oclVDClockData.u8Day <= m_oclVDClockData.u8Day)
			{
				if(oclVDClockData.u8Day < m_oclVDClockData.u8Day)
				{
					bIsOlder = true;
				}
			}
		}
	}

	return bIsOlder;
}

/**
 *
 */
bool clHDStationLogoManagement::bIsRcvdExpiredDatelatest(const clVDClockData& oExistingExpireDate, const clVDClockData& oReceivedExpireDate)
{
	bool bIsNewer = false;
	ETG_TRACE_USR4((" clHDStationLogoManagement:bIsRcvdExpiredDatelatest() (YYYY-MM-DD) Existing Date :%d-%d-%d Received Date :%d-%d-%d",
			oExistingExpireDate.s16Year, oExistingExpireDate.u8Month, oExistingExpireDate.u8Day,
			oReceivedExpireDate.s16Year, oReceivedExpireDate.u8Month, oReceivedExpireDate.u8Day));
	if(oExistingExpireDate.s16Year <= oReceivedExpireDate.s16Year)
	{
		if(oExistingExpireDate.s16Year < oReceivedExpireDate.s16Year)
		{
			bIsNewer = true;
		}
		else if(oExistingExpireDate.u8Month <= oReceivedExpireDate.u8Month)
		{
			if(oExistingExpireDate.u8Month < oReceivedExpireDate.u8Month)
			{
				bIsNewer = true;
			}
			else if(oExistingExpireDate.u8Day <= oReceivedExpireDate.u8Day)
			{
				if(oExistingExpireDate.u8Day < oReceivedExpireDate.u8Day)
				{
					bIsNewer = true;
				}
			}
		}
	}
	return bIsNewer;
}
/**
* Get date in formatted way (string to int).
*/
void clHDStationLogoManagement::vGetFormattedDate(clVDClockData &oclVDClockData, const char * sCurrentDate)
{
    char * pEnd=NULL;
    oclVDClockData.s16Year   = (tS16)strtol (sCurrentDate,&pEnd,BASE_10);
    oclVDClockData.u8Month   = (tU8)strtol (pEnd,&pEnd,BASE_10);
    oclVDClockData.u8Day     = (tU8)strtol (pEnd,&pEnd,BASE_10);
}

clVDClockData::clVDClockData():
		s16Year(0), u8Month(0), u8Day(0)
{

}

clVDClockData::~clVDClockData()
{
	s16Year = 0;
	u8Month = 0;
	u8Day = 0;
}

/**
* update current date locally.
*/
#ifdef VARIANT_S_FTR_ENABLE_VDCLOCK
void clVDClockData::vUpdateClockData(clock_mainfi_tclMsgLocalTimeDate_MinuteUpdateStatus oMsg)
{
	s16Year   = oMsg.s16Year;
	u8Month   = oMsg.u8Month;
	u8Day     = oMsg.u8Day;
}
#endif

/**
* Format HD Call sign having size 6.
*/
std::string clHDStationLogoManagement::strFormattedHDStationId(unsigned int u32HDStationId)
{
	char strHDStationId[MAXHDSTATIONIDLENGTH] = {0};
	sprintf(strHDStationId, "%08x", u32HDStationId);
	return strHDStationId;
}

/**
* Handler for new station logo received from ADR.
*/
bool clHDStationLogoManagement::vOnNewStationLogo(clCMATHDImageFileUpdateStatus& oImageFileData, unsigned int u32HDStationId)
{
	if(!u32HDStationId)
	{
		ETG_TRACE_USR2((" clHDStationLogoManagement:vOnNewStationLogo() invalid station id - %u", u32HDStationId));
		return false;
	}
	if(mNumberOfStationLogosInFlash < (short)MAXSTATIONLOGOSINFLASH)
	{
		return vCreateStationLogoFile(oImageFileData, u32HDStationId);
	}
	else if(bDeleteOldestAccessedStationLogo())
	{
		return vCreateStationLogoFile(oImageFileData, u32HDStationId);
	}
	else
	{
		ETG_TRACE_USR2((" clHDStationLogoManagement:vOnNewStationLogo() unable to create station logo"));
		return false;
	}
}

/**
 *
 */
bool clHDStationLogoManagement::bDeleteOldestAccessedStationLogo()
{
	if(vStationLogoFiles.empty())
	{
		ETG_TRACE_USR1(("clHDStationLogoManagement::bDeleteOldestAccessedStationLogo() vStationLogoFiles is empty"));
		return false;
	}

	std::string strStationLogoAbsolutePathRAM(RAMDIRECTORY);
	std::string strStationLogoFileRAM = strStationLogoAbsolutePathRAM + vStationLogoFiles[0];
	struct stat stStationLogo;

	const char *pchStationLogoFileRAM = strStationLogoFileRAM.c_str();
	if(stat(pchStationLogoFileRAM, &stStationLogo) == -1)
	{
		int errsv = errno;
		ETG_TRACE_ERR(("clHDStationLogoManagement::bDeleteOldestAccessedStationLogo() stat failed error %d for file -%s", errsv, strStationLogoFileRAM.c_str()));
		ETG_TRACE_ERR(("clHDStationLogoManagement::bDeleteOldestAccessedStationLogo() stat failed error %s", strerror(errsv)));
		return false;
	}

	time_t ulOldestAccessed = stStationLogo.st_atim.tv_sec;
	int iOldestLogoIndex = 0, iIndex=0;
	strStationLogoFileRAM.clear();

	for(std::vector<std::string>::iterator it = vStationLogoFiles.begin()+1; it != vStationLogoFiles.end(); it++, iIndex++)
	{
		strStationLogoFileRAM = strStationLogoAbsolutePathRAM + *it;
		struct stat stStationLogo;
		pchStationLogoFileRAM = strStationLogoFileRAM.c_str();
		if(stat(pchStationLogoFileRAM, &stStationLogo) == -1)
		{
			int errsv = errno;
			ETG_TRACE_ERR(("clHDStationLogoManagement::bDeleteOldestAccessedStationLogo() stat failed %d for file -%s", errsv, strStationLogoFileRAM.c_str()));
			ETG_TRACE_ERR(("clHDStationLogoManagement::bDeleteOldestAccessedStationLogo() stat failed %s", strerror(errsv)));
			return false;
		}
		time_t ulAccessTime = stStationLogo.st_atim.tv_sec;
		ETG_TRACE_USR4(("clHDStationLogoManagement::bDeleteOldestAccessedStationLogo() ulOldestAccessed - %d, ulAccessTime - %d", ulOldestAccessed, ulAccessTime));
		if(ulAccessTime < ulOldestAccessed)
		{
			ulOldestAccessed = ulAccessTime;
			iOldestLogoIndex = iIndex;
		}
		strStationLogoFileRAM.clear();
	}
	if(iOldestLogoIndex <= (int)vStationLogoFiles.size())
	{
		strStationLogoFileRAM = strStationLogoAbsolutePathRAM + *(vStationLogoFiles.begin() + iOldestLogoIndex);
		if(remove(strStationLogoFileRAM.c_str()) == -1)
		{
			int errsv = errno;
			ETG_TRACE_ERR(("clHDStationLogoManagement::bDeleteOldestAccessedStationLogo() remove failed %d for file -%s", errsv, strStationLogoFileRAM.c_str()));
			ETG_TRACE_ERR(("clHDStationLogoManagement::bDeleteOldestAccessedStationLogo() remove failed %s", strerror(errsv)));
			return false;
		}
		std::string strStationLogoFileFlash = std::string(STATIONLOGODIRECTORYPATH) + *(vStationLogoFiles.begin() + iOldestLogoIndex);
		if(remove(strStationLogoFileFlash.c_str()) && errno != ENOENT)
		{
			int errsv = errno;
			ETG_TRACE_ERR(("clHDStationLogoManagement::bDeleteOldestAccessedStationLogo() remove failed %d for file -%s", errsv, strStationLogoFileRAM.c_str()));
			ETG_TRACE_ERR(("clHDStationLogoManagement::bDeleteOldestAccessedStationLogo() remove failed %s", strerror(errsv)));
			return false;
		}
		vStationLogoFiles.erase(vStationLogoFiles.begin() + iOldestLogoIndex);
		ETG_TRACE_USR4(("clHDStationLogoManagement::bDeleteOldestAccessedStationLogo() removed %s", strStationLogoFileRAM.c_str()));
		ETG_TRACE_USR4(("clHDStationLogoManagement::bDeleteOldestAccessedStationLogo() removed %s", strStationLogoFileFlash.c_str()));
		mNumberOfStationLogosInFlash--;
		return true;
	}
	return false;
}

/**
 *
 */
bool clHDStationLogoManagement::vCreateStationLogoFile(clCMATHDImageFileUpdateStatus& oImageFileData, unsigned int u32HDStationId)
{
	clVDClockData oclVDClockData;
	oclVDClockData.s16Year = oImageFileData.Year;
	oclVDClockData.u8Month = oImageFileData.Month;
	oclVDClockData.u8Day   = oImageFileData.Day;

	if(bIsDateOlderThanSystemDate(oclVDClockData) == true)
	{
		ETG_TRACE_USR2((" clHDStationLogoManagement:vCreateStationLogoFile() failed, system date %04d-%02d-%02d received logo expire date %04d-%02d-%02d",
			m_oclVDClockData.s16Year, m_oclVDClockData.u8Month, m_oclVDClockData.u8Day, oclVDClockData.s16Year, oclVDClockData.u8Month, oclVDClockData.u8Day));
		return false;
	}

	char strTemp[MMDDBYTES] = {0};
	std::string strExistingStationLogo;
	std::string strFileName("SL");
	strFileName += strFormattedHDStationId(u32HDStationId);

	sprintf(strTemp, "%02d", oImageFileData.getHDAudioService());
	strFileName += std::string(strTemp);

	strFileName += strGetFormattedDate(oclVDClockData, false);
	strFileName += strGetImageCodingType(oImageFileData.getHDImageCoding());

	if(bIsStationLogoExist(strFileName))
	{
		return bReplaceStationLogo(strFileName, oImageFileData);
	}
	else
	{
		std::string strStationLogoAbsolutePath(RAMDIRECTORY);
		strStationLogoAbsolutePath += strFileName;
		if(vCreateStationLogoFile(strStationLogoAbsolutePath, oImageFileData.getImageFileBytes()))
		{
			vStationLogoFiles.push_back(strFileName);
			mNumberOfStationLogosInFlash++;
			return true;
		}
		return false;
	}
}
/**
 *
 */
bool clHDStationLogoManagement::bReplaceStationLogo(const std::string& strFileName, clCMATHDImageFileUpdateStatus& oImageFileData)
{
	std::string strExistingFileName;
	std::string strStationLogoAbsolutePath(RAMDIRECTORY);
	strStationLogoAbsolutePath += strFileName;
	bool bIsFileNotExist = false;
	ETG_TRACE_USR4((" clHDStationLogoManagement:bReplaceStationLogo() %s exist", strStationLogoAbsolutePath.c_str()));
	if(bIsExpiryDateUpdateNeeded(strFileName, strExistingFileName) == true)
	{
		ETG_TRACE_USR4((" clHDStationLogoManagement:bReplaceStationLogo() %s expire date updated", strStationLogoAbsolutePath.c_str()));
		if(!strExistingFileName.empty())
		{
			std::string strLogoAbsolutePathInRAM(RAMDIRECTORY);
			strLogoAbsolutePathInRAM += strExistingFileName;
			if(remove(strLogoAbsolutePathInRAM.c_str()) != 0)
			{
				ETG_TRACE_ERR((" clHDStationLogoManagement:bReplaceStationLogo() Unable to remove %s", strLogoAbsolutePathInRAM.c_str()));
			}
			std::string strLogoAbsolutePathInFlash(STATIONLOGODIRECTORYPATH);
			strLogoAbsolutePathInFlash += strExistingFileName;
			if(remove(strLogoAbsolutePathInFlash.c_str()) != 0)
			{
				ETG_TRACE_ERR((" clHDStationLogoManagement:bReplaceStationLogo() Unable to remove %s", strLogoAbsolutePathInFlash.c_str()));
			}
			vRemoveStationLogoFileNameFromVector(strExistingFileName);

			if(vCreateStationLogoFile(strStationLogoAbsolutePath, oImageFileData.getImageFileBytes()))
			{
				vStationLogoFiles.push_back(strFileName);
				mNumberOfStationLogosInFlash++;
				return true;
			}
		}
		return false;
	}
	else if(bIsReceivedStationLogoContentSame(strStationLogoAbsolutePath, oImageFileData.getImageFileBytes(), bIsFileNotExist) == false)
	{
		if(bIsFileNotExist)
		{
			return false;
		}
		else
		{
			ETG_TRACE_USR4((" clHDStationLogoManagement:bReplaceStationLogo() %s received contents are different", strStationLogoAbsolutePath.c_str()));
			vCreateStationLogoFile(strStationLogoAbsolutePath, oImageFileData.getImageFileBytes());
			return true;
		}
	}
	else
	{
		ETG_TRACE_USR4((" clHDStationLogoManagement:bReplaceStationLogo() %s received logo has old/same expire date than existing", strExistingFileName.c_str()));
		return false;
	}
}

/**
 *
 */
void clHDStationLogoManagement::vGetFormattedString(std::string &strStationLogoName)
{
	const char *cptrSpace = SPACESTRING;
	strStationLogoName.insert((size_t)YEARBYTES, cptrSpace);
	strStationLogoName.insert((size_t)(YEARBYTES+MMDDBYTES), cptrSpace);
	strStationLogoName.insert((size_t)(YEARBYTES+MMDDBYTES+MMDDBYTES), cptrSpace);
}

/**
 *
 */
bool clHDStationLogoManagement::bIsExpiryDateUpdateNeeded(const std::string& strReceivedStationLogo, std::string& strExistingFileName)
{
	clVDClockData oReceivedDate, oExistingDate;
	std::string strRecvLogo(strReceivedStationLogo.c_str() + STATIONLOGOLENGTHCOMPARE);
	vGetFormattedString(strRecvLogo);
	vGetFormattedDate(oReceivedDate, strRecvLogo.c_str());
	for(std::vector<std::string>::iterator stationLogoFilesIterator = vStationLogoFiles.begin(); stationLogoFilesIterator != vStationLogoFiles.end(); stationLogoFilesIterator++)
	{
		if(strncmp(strReceivedStationLogo.c_str(), stationLogoFilesIterator->data(), (size_t)STATIONLOGOLENGTHCOMPARE) == STRINGEQUAL)
		{
			strExistingFileName = *stationLogoFilesIterator;
			std::string strExistingLogo((*stationLogoFilesIterator).c_str() + STATIONLOGOLENGTHCOMPARE);
			vGetFormattedString(strExistingLogo);
			vGetFormattedDate(oExistingDate, strExistingLogo.c_str());
			return bIsRcvdExpiredDatelatest(oExistingDate, oReceivedDate);
		}
	}
	return false;
}

/**
* Get Image coding type (JPEG or PNG format).
*/
std::string clHDStationLogoManagement::strGetImageCodingType(unsigned char u8ImageCoding)
{
	std::string strImageCodingType;
	if(u8ImageCoding == JPEG)
	{
		strImageCodingType = JPG;
	}
	else
	{
		strImageCodingType = PNG;
	}
	return strImageCodingType;
}

/**
* Create and save station logo file.
*/
bool clHDStationLogoManagement::vCreateStationLogoFile(std::string strStationLogoAbsolutePath, std::vector<unsigned char> vImageFileBytes)
{
	std::ofstream fStationLogoFile(strStationLogoAbsolutePath.c_str(), std::ios::out | std::ifstream::binary);
	ETG_TRACE_USR2((" clHDStationLogoManagement:vCreateStationLogoFile() %s", strStationLogoAbsolutePath.c_str()));
	if(fStationLogoFile.is_open())
	{
	    std::ostream_iterator<unsigned char> output_iterator(fStationLogoFile);
	    std::copy(vImageFileBytes.begin(), vImageFileBytes.end(), output_iterator);
	    fStationLogoFile.close();
	    return true;
	}
	else
	{
		ETG_TRACE_USR2(( "clHDStationLogoManagement::vCreateStationLogoFile() unable to create file (%s)",strStationLogoAbsolutePath.c_str()));
		return false;
	}
}

/**
* Check wheter received station logo already exist in flash.
*/
bool clHDStationLogoManagement::bIsStationLogoExist(const std::string& strReceivedStationLogo)
{
	bool bExist = false;
	for(std::vector<std::string>::iterator stationLogoFilesIterator = vStationLogoFiles.begin(); stationLogoFilesIterator != vStationLogoFiles.end(); stationLogoFilesIterator++)
	{
		if(strncmp(strReceivedStationLogo.c_str(), stationLogoFilesIterator->data(), (size_t)STATIONLOGOLENGTHCOMPARE) == STRINGEQUAL)
		{
			bExist = true;
		}
	}
	return bExist;
}

/**
* Check whether received station logo contents are same as existing station logo.
*/
bool clHDStationLogoManagement::bIsReceivedStationLogoContentSame(std::string strReceivedStationLogo, std::vector<unsigned char> vReceivedLogoContent, bool& bIsFileNotExist)
{
	bool bEqual = false;
	std::ifstream fStationLogoFile(strReceivedStationLogo.c_str(), std::ios::in | std::ifstream::binary);
	if(fStationLogoFile.is_open())
	{
		bIsFileNotExist = false;
		std::ostringstream oExistingLogoStream;
		oExistingLogoStream << fStationLogoFile.rdbuf();
		const std::string& strExistingLogoStream = oExistingLogoStream.str();
		std::vector<unsigned char> vExistingLogoContent(strExistingLogoStream.begin(), strExistingLogoStream.end());

		bEqual = (vReceivedLogoContent == vExistingLogoContent);
		ETG_TRACE_USR2(( "clHDStationLogoManagement::bIsReceivedStationLogoContentSame() bEqual %d", bEqual));
		if(!bEqual)
		{
			ETG_TRACE_USR2(( "bIsReceivedStationLogoContentSame existing logo size = %u, ",vExistingLogoContent.size() ));
			ETG_TRACE_USR2(( "bIsReceivedStationLogoContentSame received logo size = %u, ",vReceivedLogoContent.size() ));
		}
		fStationLogoFile.close();
	}
	else
	{
		bIsFileNotExist = true;
		ETG_TRACE_USR2(( "clHDStationLogoManagement::bIsReceivedStationLogoContentSame() unable to open file (%s)",strReceivedStationLogo.c_str()));
	}
	return bEqual;
}

/**
* Get Date in formatted way (int to string).
*/
std::string clHDStationLogoManagement::strGetFormattedDate(const clVDClockData &oclVDClockData, bool bIsSpaceNeeded)
{
	char strTemp[YEARBYTES+1] = {0};
	char sSystemDate[CURRENTDATESTRINGLENGTH] = {0};
	int iStart = 0;
	int iLengthToBeCopied = YEARBYTES;

	sprintf(strTemp, "%04hd", oclVDClockData.s16Year);
	strncat(sSystemDate, strTemp, iLengthToBeCopied);
	iStart += iLengthToBeCopied;
	if(bIsSpaceNeeded == true)
	{
		iLengthToBeCopied = MMDDBYTES;
	}
	else
	{
		iLengthToBeCopied = DDMMBYTES;
	}

	vIntegerToString(sSystemDate, iStart, oclVDClockData.u8Month, iLengthToBeCopied, bIsSpaceNeeded);
	vIntegerToString(sSystemDate, iStart, oclVDClockData.u8Day, iLengthToBeCopied, bIsSpaceNeeded);

	ETG_TRACE_USR2(( "clHDStationLogoManagement::strGetFormattedDate() sSystemDate = %s",sSystemDate));

	return std::string(sSystemDate);
}

/**
* Get Active station logo path.
*/
std::string clHDStationLogoManagement::strGetActiveStationLogoPath(unsigned int u32HDStationId, unsigned char u8AudioProgram)
{
	std::string strActiveStationLogoPath;
	char strTemp[MMDDBYTES] = {0};
	std::string strFileName("SL");
	strFileName += strFormattedHDStationId(u32HDStationId);
	
	if(u8AudioProgram > 7)
    {
		 return strActiveStationLogoPath;
	} 
	sprintf(strTemp, "%02d", u8AudioProgram);
	strFileName += std::string(strTemp);

	for(std::vector<std::string>::iterator it = vStationLogoFiles.begin(); it != vStationLogoFiles.end(); it++)
	{
		if(strncmp(strFileName.c_str(), it->c_str(), strlen(strFileName.c_str())) == STRINGEQUAL)
		{
			strActiveStationLogoPath += std::string(RAMDIRECTORY);
			strActiveStationLogoPath += *it;
			break;
		}
	}

	ETG_TRACE_USR2(( "clHDStationLogoManagement::strGetActiveStationLogoPath() strActiveStationLogoPath = %s",strActiveStationLogoPath.c_str()));
	return strActiveStationLogoPath;
}

/**
* Date validation.
*/
bool clHDStationLogoManagement::bIsValidDate(const clVDClockData& oclVDClockData)
{

	bool bIsItCurrectFormat = true;

	if(( oclVDClockData.s16Year > 0 )&& (oclVDClockData.u8Month >0) && (oclVDClockData.u8Month < 13) && (oclVDClockData.u8Day > 0) && (oclVDClockData.u8Day < 32 ))
	{
		bIsItCurrectFormat =((oclVDClockData.u8Month == 2)?
							 (oclVDClockData.s16Year % 4 == 0)?
							 (oclVDClockData.u8Day > 29)?false:true:
							 (oclVDClockData.u8Day > 28)?false:true:
							 (oclVDClockData.u8Month == 1 || oclVDClockData.u8Month == 3 || oclVDClockData.u8Month  == 5 || oclVDClockData.u8Month == 7||
							  oclVDClockData.u8Month == 8 || oclVDClockData.u8Month == 10 || oclVDClockData.u8Month == 12 )?
							 (oclVDClockData.u8Day > 32 )? false:true:( oclVDClockData.u8Day > 31 )? false:true);
	}
	else
	{
		bIsItCurrectFormat = false;
	}

	return bIsItCurrectFormat;
}

