/*
//====================================================================
// MWT_SiriusReceiverFrameDecoder.h:
//
// Copyright � 2005 WSI Corporation.  All rights reserved.
// Author: Damon M. Hill
//
// This ADT implements decoding of the Sirius receiver frames
// as output by the Sirius receiver external data interface.
//
// It can operate in one of 2 modes:
//		Remove Transparency Only,
//		Complete Deframing.
//
// In the "Complete Deframing" mode, the frames are received, checked
// for integrity, and then the DPU's therein are passed to the
// MWT_SiriusDPUAssembler ADT for creation of an output product.
//
// In the "Remove Transparency Only" mode, the transparency encoding
// is removed but the framing structure is left.  The DPU content
// is ignored and written to the output buffer as is after transparency
// removal. Note that this is for manual inspection only because the
// transparency encoding is required for 100% frame border detection
// accuracy.
//
//
//====================================================================
*/
#ifndef MWT_SIRIUSRECEIVERFRAMEDECODER_HEADER
#define MWT_SIRIUSRECEIVERFRAMEDECODER_HEADER

/*
//====================================================================
// Required includes
//====================================================================
*/
#include "MW_DataFieldTypes.h"
#include <stdio.h>

/*
//====================================================================
// Public methods
//====================================================================
*/

/*
//====================================================================
// MWTSiriusFrameDecoder_reset:
//
// This method resets the frame parsing state to a desynchronized one
// and resets the DPU processor to not be tracking any product.
//
// It is useful when reading products in from a non-continuous link
// (from seperate files, for example)
//====================================================================
*/
void MWTSiriusFrameDecoder_reset( void );

/*
//====================================================================
// MWTSiriusFrameDecoder_extractProduct:
//
// Call this method to pass bytes to the decoder in order to undo
// the framing scheme and extract the next correctly transmitted
// product into a byte buffer.
//
//
// PARAMETERS:
//
//	numBytes:
//		The number of bytes in the input buffer
//
//	p_bytes:
//		The pointer to the first byte of the input buffer
//
//	inout_p_numBytesUsed:
//		A pointer to a counter of the number of bytes from
//		the input buffer that have already been consumed.
//		On input, this should be the offset of the next byte
//		to be read, and on return, it will be updated to
//		account for any bytes read during this call.
//
//	numProductStreamBufferBytes:
//		The number of bytes of length of the product stream buffer.
//
//	p_productStreamBuffer:
//		Pointer to the first byte of the product stream buffer.
//
//	p_inout_numProductStreamBufferBytesUsed:
//		Pointer to a variable which tracks the number of bytes
//		of the product stream buffer filled.  This will be
//		modified with each call.
//		This will automatically be reset to 0 whenever a new
//		product instance is found in the frame stream.
//
//	p_headerInfoFile:
//		Pointer to a file handle to which translations of DPU
//		and product headers should be written. Pass in NULL
//		if you don not wish this information to be recorded.
//
// RETURNS:
//
//	MWRes_Success
//		If a product instance has been finished.
//
//	MWRes_InputBufferExhausted:
//		If all of the input buffer has been exhausted without
//		finishing a product isntance.  Resupply the buffer
//		and call us again if you would like.
//
//	MWRes_DataStructureFull:
//		If the numProductStreamBufferBytes is not large enough
//		to hold the product instance stream being decoded.
//
//
//
//====================================================================
*/
TMW_ResultCode MWTSiriusFrameDecoder_extractProduct
(
	TFourByteInteger_Unsigned numBytes,
	TByte_Unsigned* p_bytes,
	TFourByteInteger_Unsigned* p_inout_numBytesUsed,
	TFourByteInteger_Unsigned numProductStreamBufferBytes,
	TByte_Unsigned* p_productStreamBuffer,
	TFourByteInteger_Unsigned* p_inout_numProductStreamBufferBytesUsed,
	FILE* p_headerInfoFile
);

/*
//====================================================================
// MWTSiriusFrameDecoder_deframeProductFile:
//
// This method applies MWTSiriusFrameDecoder_extractProduct to
// a given input file to create an output file which consists of the
// product held in the frames of the input file.
//
// PARAMETERS:
//
//	pstr_inputFrameFilePath:
//		This should be the path of a file which contains the frames of
//		a single product as output by a Sirius Receiver.
//
//	pstr_outputProductFilePath:
//		This should be the path of a file which will be created to
//		hold the product without any framing.
//
//	bool_outputHeaderInfoToFile:
//		If set to non-zero, (TRUE) then the DPU and product headers will
//		be translated into text in an output file
//
//	pstr_headerInfoFilePath:
//		This is the file path to which the header info should be written.
//		The file will be replaced if it already exists.
//		This can be NULL only if bool_outputHeaderInfoToFile is 0 (FALSE)
//
// RETURNS:
//
//	MWRes_Success
//
//	MWRes_NullPointer:
//		If a path string pointer is NULL
//
//	MWRes_CommunicationsError:
//		On failure to open or create a file.
//
//	MWRes_TransportProtocolError:
//		If the frame structure was invalid.
//
//====================================================================
*/
TMW_ResultCode MWTSiriusFrameDecoder_deframeProductFile
(
	const TASCIIChar* pstr_inputFrameFilePath,
	const TASCIIChar* pstr_outputProductFilePath,
	TByte_Unsigned bool_outputHeaderInfoToFile,
	const TASCIIChar* pstr_headerInfoFilePath
);

/*
//====================================================================
// MWTSiriusFrameDecoder_undoTransparencyEncoding:
//
// Call this method to pass bytes to the decoder in order to remove
// the transparency encoding and place all the resulting bytes
// of the frames (including the borders) into the output buffer.
// Note that this is for manual inspection of the output only.
// Once tranpsarency is removed, a data byte that happens to be
// the frame border flag or the transparency flag cannot be
// distinguished from the flags.
//
// PARAMETERS:
//
//	numBytes:
//		The number of bytes in the input buffer
//
//	p_bytes:
//		The pointer to the first byte of the input buffer
//
//	inout_p_numBytesUsed:
//		A pointer to a counter of the number of bytes from
//		the input buffer that have already been consumed.
//		On input, this should be the offset of the next byte
//		to be read, and on return, it will be updated to
//		account for any bytes read during this call.
//
//	numOutputBufferBytes:
//		The number of bytes of length of the output buffer.
//
//	p_outputBuffer:
//		Pointer to the first byte of the output buffer.
//
//	p_inout_numProductStreamBufferBytesUsed:
//		Pointer to a variable which tracks the number of bytes
//		of the output buffer filled.  This will be
//		modified with each call.
//
//
// RETURNS:
//
//	MWRes_Success
//		If all the bytes were transparency decoded without error.
//
//	MWRes_DataStructureFull:
//		If the numProductStreamBufferBytes was not large enough
//		to hold all the transparency decoded bytes.
//
//====================================================================
*/
TMW_ResultCode MWTSiriusFrameDecoder_undoTransparencyEncoding
(
	TFourByteInteger_Unsigned numBytes,
	TByte_Unsigned* p_bytes,
	TFourByteInteger_Unsigned* p_inout_numBytesUsed,
	TFourByteInteger_Unsigned numOutputBufferBytes,
	TByte_Unsigned* p_outputBuffer,
	TFourByteInteger_Unsigned* p_inout_numOutputBufferBytesUsed
);

/*
//====================================================================
// MWTSiriusFrameDecoder_removeTransparencyFromFrameFile:
//
// This method applies MWTSiriusFrameDecoder_undoTransparencyEncoding to
// a given input file to create an output file which consists of the
// frames with transparency encoding removed.  Note that this is for
// manual inspection of the result only.  Once tranpsarency is removed,
// a data byte that happens to be the frame border flag or the
// transparency flag cannot be distinguished from the flags.
//
// PARAMETERS:
//
//	pstr_inputFrameFilePath:
//		This should be the path of a file which contains the frames of
//		a single product as output by a Sirius Receiver.
//
//	pstr_outputProductFilePath:
//		This should be the path of a file which will be created to
//		hold the product without any framing.
//
// RETURNS:
//
//	MWRes_Success
//
//	MWRes_NullPointer:
//		If a path string pointer is NULL
//
//	MWRes_CommunicationsError:
//		On failure to open or create a file.
//
//	MWRes_TransportProtocolError:
//		If the frame structure was invalid.
//
//====================================================================
*/
TMW_ResultCode MWTSiriusFrameDecoder_removeTransparencyFromFrameFile
(
	const TASCIIChar* pstr_inputFrameFilePath,
	const TASCIIChar* pstr_outputProductFilePath
);


#endif

