/*
//====================================================================
// MWT_SiriusDPUAssembler.c:
//
// Copyright � 2005 WSI Corporation.  All rights reserved.
// Author: Damon M. Hill
//
// Implementation of the Sirius DPU Assembler ADT.
//
//====================================================================
*/

/*
//====================================================================
// Includes
//====================================================================
*/
#include "MWT_SiriusDPUAssembler.h"
#include "MW_ProductHeaders.h"


/*
//====================================================================
// ADT data
//====================================================================
*/

/* The current product ID and product instance ID being assembled.*/
static TFourByteInteger_Unsigned MWTSiriusFrameDecoder_currentProductID = 0;
static TFourByteInteger_Unsigned MWTSiriusFrameDecoder_currentInstanceID = 0;

/* The DPU sequence number of the previous DPU of the assembling product */
static TFourByteInteger_Signed MWTSiriusFrameDecoder_previousDPUNumber = -1;

/*
//====================================================================
// Public methods
//====================================================================
*/

void MWTSiriusDPUAssembler_reset()
{
	/* Not assembling any product */
	MWTSiriusFrameDecoder_currentProductID = 0;
	MWTSiriusFrameDecoder_currentInstanceID = 0;
	MWTSiriusFrameDecoder_previousDPUNumber = -1;
}

/*--------------------------------------------------------------------*/

void MWTSiriusDPUAssembler_Internal_LogMessage
(
	FILE* p_headerInfoFile,
	const char* pstr_Message
)
{
	if (p_headerInfoFile != NULL)
	{
		fprintf (p_headerInfoFile, "%s\n", pstr_Message);
	}
}


/*--------------------------------------------------------------------*/

TMW_ResultCode MWTSiriusDPUAssembler_Internal_outputProductHeaderLog
(
	TFourByteInteger_Unsigned numBytesInProductStream,
	TByte_Unsigned* p_productStreamBuffer,
	FILE* p_headerInfoFile
)
{
	TASCIIChar MessageBuffer[1024];


	/*
	//##############################################################
	// Test if any bytes
	//##############################################################
	*/

	if (numBytesInProductStream < 1)
	{
		MWTSiriusDPUAssembler_Internal_LogMessage (p_headerInfoFile, "Cannot determine header type as there are 0 bytes in the product");
		return MWRes_Success;
	}

	/*
	//##############################################################
	// Output header type message
	//##############################################################
	*/
	switch ((TMW_ProductHeaderType) p_productStreamBuffer[0])
	{
	case Generic_ProductHeaderType:
		MWTSiriusDPUAssembler_Internal_LogMessage (p_headerInfoFile, "Generic Product Header:");
		break;

	case GenericRaster_ProductHeaderType:
		MWTSiriusDPUAssembler_Internal_LogMessage (p_headerInfoFile, "Generic Raster Product Header:");
		break;

	default:
		sprintf (MessageBuffer, "Unknown Product Header Type '%d'", p_productStreamBuffer[0]);
		MWTSiriusDPUAssembler_Internal_LogMessage (p_headerInfoFile, MessageBuffer);
		break;

	}

	/*
	//##############################################################
	// Output generic product header fields
	//##############################################################
	*/
	if
	(
		(p_productStreamBuffer[0] == Generic_ProductHeaderType) ||
		(p_productStreamBuffer[0] == GenericRaster_ProductHeaderType)
	)
	{
		/* Get header */
		TMW_GenericProductHeader_Output* p_header;

		p_header = (TMW_GenericProductHeader_Output*) p_productStreamBuffer;
		if (numBytesInProductStream < sizeof (TMW_GenericProductHeader_Output))
		{
			sprintf (MessageBuffer, "\tHeader is too short, only '%u' bytes ",
					 (unsigned int)numBytesInProductStream);
			MWTSiriusDPUAssembler_Internal_LogMessage (p_headerInfoFile, MessageBuffer);
			return MWRes_Success;
		}

		/* Output PID */
		sprintf (MessageBuffer, "\tProduct ID: %d", p_header->productID);
		MWTSiriusDPUAssembler_Internal_LogMessage (p_headerInfoFile, MessageBuffer);

		/* Issue date*/
		sprintf (MessageBuffer, "\tIssue month: %.2d", p_header->issueMonth);
		MWTSiriusDPUAssembler_Internal_LogMessage (p_headerInfoFile, MessageBuffer);

		sprintf (MessageBuffer, "\tIssue day of month: %.2d", p_header->issueDayOfMonth);
		MWTSiriusDPUAssembler_Internal_LogMessage (p_headerInfoFile, MessageBuffer);

		sprintf (MessageBuffer, "\tIssue hour: %.2d", p_header->issueHour);
		MWTSiriusDPUAssembler_Internal_LogMessage (p_headerInfoFile, MessageBuffer);

		sprintf (MessageBuffer, "\tIssue minute: %.2d", p_header->issueMinute);
		MWTSiriusDPUAssembler_Internal_LogMessage (p_headerInfoFile, MessageBuffer);

		/* Valid date*/
		sprintf (MessageBuffer, "\tValid month: %.2d", p_header->validMonth);
		MWTSiriusDPUAssembler_Internal_LogMessage (p_headerInfoFile, MessageBuffer);

		sprintf (MessageBuffer, "\tValid day of month: %.2d", p_header->validDayOfMonth);
		MWTSiriusDPUAssembler_Internal_LogMessage (p_headerInfoFile, MessageBuffer);

		sprintf (MessageBuffer, "\tValid hour: %.2d", p_header->validHour);
		MWTSiriusDPUAssembler_Internal_LogMessage (p_headerInfoFile, MessageBuffer);

		sprintf (MessageBuffer, "\tValid minute: %.2d", p_header->validMinute);
		MWTSiriusDPUAssembler_Internal_LogMessage (p_headerInfoFile, MessageBuffer);

		/* Georference */

		sprintf (MessageBuffer, "\tUpper latitude: %.2f", p_header->maxLatitude / 100.0);
		MWTSiriusDPUAssembler_Internal_LogMessage (p_headerInfoFile, MessageBuffer);

		sprintf (MessageBuffer, "\tLeft longitude: %.2f", p_header->minLongitude / 100.0);
		MWTSiriusDPUAssembler_Internal_LogMessage (p_headerInfoFile, MessageBuffer);

		sprintf (MessageBuffer, "\tLower latitude: %.2f", p_header->minLatitude / 100.0);
		MWTSiriusDPUAssembler_Internal_LogMessage (p_headerInfoFile, MessageBuffer);

		sprintf (MessageBuffer, "\tRight longitude: %.2f", p_header->maxLongitude / 100.0);
		MWTSiriusDPUAssembler_Internal_LogMessage (p_headerInfoFile, MessageBuffer);

	}

	/*
	//##############################################################
	// Output generic raster specific fields
	//##############################################################
	*/

	if (p_productStreamBuffer[0] == GenericRaster_ProductHeaderType)
	{
		TMW_GenericRasterProductHeader_Output* p_header;
		TMW_GenericRasterProductHeader_FixedScaleSpecifier* p_fixedScaleSpecifiers;
		TByte_Unsigned planeIndex;

		/*
		//##############################################################
		// Raster Header specific fields
		//##############################################################
		*/


		p_header = (TMW_GenericRasterProductHeader_Output*) p_productStreamBuffer;
		if (numBytesInProductStream < sizeof(TMW_GenericRasterProductHeader_Output) )
		{
			sprintf (MessageBuffer, "\tHeader is too short, only '%u' bytes ",
					 (unsigned int)numBytesInProductStream);
			MWTSiriusDPUAssembler_Internal_LogMessage (p_headerInfoFile, MessageBuffer);
			return MWRes_Success;
		}


		sprintf (MessageBuffer, "\tNumber of rows: %d", p_header->numRows);
		MWTSiriusDPUAssembler_Internal_LogMessage (p_headerInfoFile, MessageBuffer);

		sprintf (MessageBuffer, "\tNumber of columns: %d", p_header->numColumns);
		MWTSiriusDPUAssembler_Internal_LogMessage (p_headerInfoFile, MessageBuffer);

		sprintf (MessageBuffer, "\tbits Per Pixel: %d", p_header->bitsPerPixel);
		MWTSiriusDPUAssembler_Internal_LogMessage (p_headerInfoFile, MessageBuffer);

		sprintf (MessageBuffer, "\tnumPlanes: %d", p_header->numPlanes);
		MWTSiriusDPUAssembler_Internal_LogMessage (p_headerInfoFile, MessageBuffer);

		/*
		//##############################################################
		// Output the fixed scale specifiers
		//##############################################################
		*/

		if
		(
			numBytesInProductStream <
			(
				sizeof(TMW_GenericRasterProductHeader_Output) +
				(
					p_header->numPlanes *
					  sizeof(TMW_GenericRasterProductHeader_FixedScaleSpecifier)
				)
			)
		)
		{
			sprintf (MessageBuffer, "\tHeader is too short only '%u' bytes ",
					 (unsigned int)numBytesInProductStream);
			MWTSiriusDPUAssembler_Internal_LogMessage (p_headerInfoFile, MessageBuffer);
			return MWRes_Success;
		}

		p_fixedScaleSpecifiers = (TMW_GenericRasterProductHeader_FixedScaleSpecifier*) (p_header + 1);


		for (planeIndex = 0; planeIndex < p_header->numPlanes; planeIndex ++)
		{
			sprintf (MessageBuffer, "\tfixed scale specifier struct for plane %d\n", planeIndex);
			MWTSiriusDPUAssembler_Internal_LogMessage (p_headerInfoFile, MessageBuffer);

			sprintf (MessageBuffer, "\t\tprecisionSpecifier: %d", p_fixedScaleSpecifiers[planeIndex].precisionSpecifier);
			MWTSiriusDPUAssembler_Internal_LogMessage (p_headerInfoFile, MessageBuffer);

			sprintf (MessageBuffer, "\t\toffsetSpecifier: %d", p_fixedScaleSpecifiers[planeIndex].offsetSpecifier);
			MWTSiriusDPUAssembler_Internal_LogMessage (p_headerInfoFile, MessageBuffer);
		}
	}

	/*
	//##############################################################
	// Done
	//##############################################################
	*/
	return MWRes_Success;


}


/*--------------------------------------------------------------------*/

TMW_ResultCode MWTSiriusDPUAssembler_processDPU
(
	TWSI_Output_DPUHeader* p_DPUHeader,
	TFourByteInteger_Unsigned numPayloadBytes,
	TByte_Unsigned* p_payloadBytes,
	TFourByteInteger_Unsigned numProductStreamBufferBytes,
	TByte_Unsigned* p_productStreamBuffer,
	TFourByteInteger_Unsigned* p_inout_numProductStreamBufferBytesUsed,
	FILE* p_headerInfoFile
)
{
	TTwoByteInteger_Unsigned DPUProductID;
	TTwoByteInteger_Unsigned DPUInstanceID;
	TTwoByteInteger_Unsigned DPUSequenceNumber;
	char MessageBuffer[1024];

#ifndef HAS_MEMCPY
	TFourByteInteger_Unsigned payloadIndex;
#endif

	/*
	//##############################################################
	// Get the DPU header fields
	//##############################################################
	*/
	DPUSequenceNumber = p_DPUHeader->DPUSequenceNumber_LOWByte +
		(((TTwoByteInteger_Unsigned) p_DPUHeader->DPUSequenceNumber_HIGHByte) * 256);

	DPUProductID = p_DPUHeader->productID_LOWByte +
		(((TTwoByteInteger_Unsigned) p_DPUHeader->productID_HIGHByte) * 256);

	DPUInstanceID = p_DPUHeader->instanceID_LOWByte +
		(((TTwoByteInteger_Unsigned) p_DPUHeader->instanceID_HIGHByte) * 256);

	//##############################################################
	// Log the dpu header information
	//##############################################################

	MWTSiriusDPUAssembler_Internal_LogMessage
	(
		p_headerInfoFile,
		"DPU Header:"
	);

	// DPU Product ID
	sprintf (MessageBuffer, "\tProduct ID: %d", DPUProductID);
	MWTSiriusDPUAssembler_Internal_LogMessage (p_headerInfoFile, MessageBuffer);

	// DPU Product Instance ID
	sprintf (MessageBuffer, "\tProduct Instance ID: %d", DPUInstanceID);
	MWTSiriusDPUAssembler_Internal_LogMessage (p_headerInfoFile, MessageBuffer);

	// DPU Sequence Number
	sprintf (MessageBuffer, "\tSequence Number: %d", DPUSequenceNumber);
	MWTSiriusDPUAssembler_Internal_LogMessage (p_headerInfoFile, MessageBuffer);

	// DPU Is Last DPU
	sprintf (MessageBuffer, "\tIs last DPU: %s", p_DPUHeader->finalDPUThisProductInstance ? "YES" : "NO");
	MWTSiriusDPUAssembler_Internal_LogMessage (p_headerInfoFile, MessageBuffer);

	/*
	//##############################################################
	// Is it the first in a series? If so, then starting new
	// product instance assembly.
	//
	// If it is not, and it is not the next in the expected sequence,
	// throw it away and ignore it.
	//##############################################################
	*/
	if (DPUSequenceNumber == 0)
	{
		/* Reset assembly */
		MWTSiriusFrameDecoder_currentProductID = DPUProductID;
		MWTSiriusFrameDecoder_currentInstanceID = DPUInstanceID;
		MWTSiriusFrameDecoder_previousDPUNumber = 0;
		*p_inout_numProductStreamBufferBytesUsed = 0;

	}
	else if
	(
		(DPUProductID != MWTSiriusFrameDecoder_currentProductID) &&
		(DPUInstanceID != MWTSiriusFrameDecoder_currentInstanceID) &&
		(DPUSequenceNumber != (TTwoByteInteger_Unsigned) (MWTSiriusFrameDecoder_previousDPUNumber + 1))
	)
	{
		/* Is not in expected order, ignore it */
		MWTSiriusDPUAssembler_Internal_LogMessage (p_headerInfoFile, "DPU received out of sequence");
		return MWRes_Success;
	}

	/*
	//##############################################################
	// Append this frame payload into the product output buffer
	//##############################################################
	*/

	/*
	// Will the payload fit?
	*/
	if (numPayloadBytes > (numProductStreamBufferBytes - *p_inout_numProductStreamBufferBytesUsed))
	{
		// Not enough room
		MWTSiriusDPUAssembler_Internal_LogMessage (p_headerInfoFile, "Product too long to fit in the buffer supplied");
		return MWRes_DataStructureFull;
	}

	/* Do copy */
#ifdef HAS_MEMCPY
	memcpy
	(
		(p_productStreamBuffer + *p_inout_numProductStreamBufferBytesUsed),
		MWTSiriusFrameDecoder_frameBuffer + sizeof(TWSI_Output_DPUHeader),
		numPayloadBytes
	);
#else

	for (payloadIndex = 0; payloadIndex < numPayloadBytes; payloadIndex ++)
	{
		p_productStreamBuffer[(*p_inout_numProductStreamBufferBytesUsed) + payloadIndex]
			= p_payloadBytes[payloadIndex];
	}
#endif

	/*
	//##############################################################
	// Log contents of product header
	//##############################################################
	*/
	if (DPUSequenceNumber == 0)
	{
		MWTSiriusDPUAssembler_Internal_outputProductHeaderLog
		(
			numPayloadBytes,
			p_productStreamBuffer + *p_inout_numProductStreamBufferBytesUsed,
			p_headerInfoFile
		);
	}


	/*
	//##############################################################
	// More bytes in caller's product assembly buffer
	//##############################################################
	*/
	*p_inout_numProductStreamBufferBytesUsed = (*p_inout_numProductStreamBufferBytesUsed) + numPayloadBytes;


	//##############################################################
	// Remember this as last DPU processed
	//##############################################################
	MWTSiriusFrameDecoder_previousDPUNumber = DPUSequenceNumber;

	/*
	//##############################################################
	// Was it the last DPU of a product?
	// If so, then a product assembly is complete.
	// If not, then we must wait for next DPU.
	//##############################################################
	*/
	if (p_DPUHeader->finalDPUThisProductInstance != 0)
	{
		/* Product was completed with this DPU !*/
		MWTSiriusDPUAssembler_Internal_LogMessage (p_headerInfoFile, "Product completed");
		return MWRes_EndOfIteration;
	}
	else
	{
		/* Normal processing*/
		return MWRes_Success;
	}


}
