/*
====================================================================
// MW_WxHuffmanProcessor.h:
//
// This defines the public interface for an ADT which can
// process Weather Huffman compressed rasters into other
// output formats.
//
// All source code was built from inspection of the DOT document
// describing Weather Huffman.  It may not be the same as
// other Weather Huffman implementations. Therefore the
// format is referred to as WSI Weather Huffman.
//
// Author: Damon M. Hill
// Copyright � 2005 WSI Corporation.  All rights reserved.
//
====================================================================
*/
#ifndef MW_WXHUFFMANPROCESSOR_HEADER
#define MW_WXHUFFMANPROCESSOR_HEADER

/*
====================================================================
// Required includes
====================================================================
*/
#include "MW_ProductHeaders.h"

/*
====================================================================
// MWWxHuffmanProcessor_transformFormat:
//
// This method is used to transform a raster body from the
// WSI Weather Huffman format to the specified format.
// The output result is sent through the MWOutputDPUEncoder.
//
// PARAMETERS:
//
//	p_WxHuffmanBody:
//		The pointer to the first byte in the array of bytes describing
//		the raster body in WSI Weather Huffman format.  This must be
//		at least 4 bytes long, as the first 4 bytes define the
//		length of the WxHuffmanBody in LOHI order.
//
//	bool_productEndsWithThisBody:
//
//		If set to 0, then the last frame required to output this raster body
//		is not marked as the end of the product.
//
//		If set to !0, then the last frame required to output this raster body
//		is marked as the end of the product.
//
// numRasterColumns:
//		The number of columns in the raster body
//
// numRasterRows:
//		The number of rows in the raster body
//
// numBitsPerPixel:
//		The number of bits per pixel in the raster body
//		Valid values are currently 1-8
//
// p_out_numBitsConsumed:
//		This returns the number of bits consumed by the weather huffman processor
//
// RETURNS:
//
//	MWRes_Success
//		If this is returned, the body has been written to the output
//
//	MWRes_UnrecognizedTransmogrificationType:
//		If the outputFormat is not one of the supported types
//		(see PARAMETERS section)
//
//		If the numBitsPerPixel is not in then inclusive range 1 to 8
//
//	MWRes_OutOfMemory:
//		If the raster product is too large to work with given our
//		fixed memory constraints.
//
//	MWRes_ProductNotExpectedSize:
//		If the number of bytes consumed in the weather huffman processing
//		is not the same as the number of bytes indicated in the weather
//		huffman block header (read internally)
//
//	MWRes_NullPointer:
//		If a pointer parameter is NULL
====================================================================
*/
TMW_ResultCode MWWxHuffmanProcessor_transformFormat
(
	TByte_Unsigned* p_WxHuffmanBody,
	TByte_Unsigned bool_productEndsWithThisBody,
	TTwoByteInteger_Unsigned numRasterColumns,
	TTwoByteInteger_Unsigned numRasterRows,
	TByte_Unsigned numBitsPerPixel,
	TFourByteInteger_Unsigned* p_out_numBitsConsumed
);


/* End header wrapper */
#endif

