/*
//====================================================================
// MW_Zlib.c:
//
// Implementation of a simplified interface for dealing with Zlib.
// This is designed to shield the rest of the application with concerns about
// handling the z_alloc and z_free functions.
//
// This implementation uses a compile-time sized block of memory for zlib,
// and allocates and deallocates it.  At a call to MWZLib_InflateEnd, the
// entire memory structure is considered deallocated, and hence, any fragmentation
// in the structure is removed.
//
// Copyright � 2005 WSI Corporation.  All rights reserved.
// Author: Damon M. Hill
//====================================================================
*/

/*
//====================================================================
// Includes
//====================================================================
*/
#include "MW_ZLib.h"

/*
//====================================================================
// Defines
//====================================================================
*/
#define INTERNAL_INFLATION_BUFFER_LENGTH 1024

/*
//====================================================================
// The implementations of z_alloc and z_free that use a given memory
// space manager
//====================================================================
*/

void* MW_ZAlloc (voidpf opaque, uInt items, uInt size)
{
	TMW_ResultCode MWRes;
	TMW_MemorySpaceManager* p_memorySpaceManager;
	TFourByteInteger_Unsigned numReqBytes;
	TByte_Unsigned* outPtr;

	/*
	//########################################
	// Opaque should be pointer to a
	// memory space manager
	//########################################
	*/
	p_memorySpaceManager = (TMW_MemorySpaceManager*) opaque;
	if (p_memorySpaceManager == NULL)
	{
		return NULL;
	}

	/*
	//########################################
	// Compute total size
	//########################################
	*/

	/* TODO: Protect against overflow */
	numReqBytes = items * size;

	/*
	//########################################
	// Allocate the memory
	//########################################
	*/

	outPtr = NULL;

	MWRes = MemorySpaceManager_allocate
	(
		p_memorySpaceManager,
		numReqBytes,
		&outPtr
	);
	if (MWRes != MWRes_Success)
	{
		return NULL;
	}

	/*
	//########################################
	// Done
	//########################################
	*/

	return outPtr;

}

/*-----------------------------------------------------------------------------*/

void MW_ZFree (voidpf opaque, voidpf address)
{
	TMW_ResultCode MWRes;
	TMW_MemorySpaceManager* p_memorySpaceManager;

	/*
	//########################################
	// Opaque should be pointer to a memory
	// space manager
	//########################################
	*/
	p_memorySpaceManager = (TMW_MemorySpaceManager*) opaque;
	if (p_memorySpaceManager == NULL)
	{
		/* TODO: Trigger global error trapping */
		return;
	}

	/*
	//########################################
	// Deallocate
	//########################################
	*/

	MWRes = MemorySpaceManager_deallocate
	(
		p_memorySpaceManager,
		address
	);
	if (MWRes != MWRes_Success)
	{
		/* TODO: Handle result by setting global error trapping */
	}

	/*
	//########################################
	// Done
	//########################################
	*/

}

/*
//====================================================================
// Initialization
//====================================================================
*/

TMW_ResultCode MWZLib_inflateStart
(
	TMWZLib_SessionStructure* p_zlibSession,
	TFourByteInteger_Unsigned numBytes,
	TByte_Unsigned* p_sourceByteArray,
	TMW_MemorySpaceManager* p_memorySpaceManager
)
{

	/*
	//########################################
	// Test parameters
	//########################################
	*/
	if
	(
		(p_sourceByteArray == NULL) ||
		(p_memorySpaceManager == NULL) ||
		(p_zlibSession == NULL)
	)
	{
		return MWRes_NullPointer;
	}

	/*
	//########################################
	// Initialize our structure
	//########################################
	*/
	p_zlibSession->numInflatedBytesInternal = 0;
	p_zlibSession->numInflatedBytesInternalConsumed = 0;
	p_zlibSession->p_memorySpaceManager = p_memorySpaceManager;

	/*
	//########################################
	// Setup zstream
	//########################################
	*/
	/*
	// Opqaue is a parameter that Zlib passes to our
	// ZAlloc and ZFree methods as is. It is intended for our
	// usage and Zlib never touches it otherwise.
	*/
	p_zlibSession->zStream.opaque = p_memorySpaceManager;

	p_zlibSession->zStream.next_in = p_sourceByteArray;
	p_zlibSession->zStream.avail_in = numBytes;
	p_zlibSession->zStream.total_in = 0;

	p_zlibSession->zStream.next_out = p_zlibSession->inflatedBytesInternal;
	p_zlibSession->zStream.avail_out = INTERNAL_INFLATION_BUFFER_LENGTH;
	p_zlibSession->zStream.total_out = 0;

	p_zlibSession->zStream.zalloc = &MW_ZAlloc;
	p_zlibSession->zStream.zfree = &MW_ZFree;

	/*
	//########################################
	// Call inflateInit
	//########################################
	*/
	p_zlibSession->ZRes = inflateInit
	(
		&(p_zlibSession->zStream)
	);
	if (p_zlibSession->ZRes != Z_OK)
	{
		return MWRes_InflationError;
	}

	/*
	//########################################
	// Success
	//########################################
	*/
	return MWRes_Success;

}

/*
//====================================================================
// End session
//====================================================================
*/

TMW_ResultCode MWZLib_inflateEnd
(
	TMWZLib_SessionStructure* p_zlibSession
)
{

	/*
	//########################################
	// Test parameters
	//########################################
	*/
	if
	(
		(p_zlibSession == NULL)
	)
	{
		return MWRes_NullPointer;
	}

	/*
	//########################################
	// Call inflateEnd
	//########################################
	*/
	p_zlibSession->ZRes = inflateEnd
	(
		&(p_zlibSession->zStream)
	);

	/*
	//########################################
	// Session is over, memory no longer in
	// use
	//########################################
	*/
	MemorySpaceManager_reset
	(
		p_zlibSession->p_memorySpaceManager
	);

	/*
	//########################################
	// Handle result of inflateEnd
	//########################################
	*/
	if (p_zlibSession->ZRes != Z_OK)
	{
		return MWRes_InflationError;
	}
	else
	{
		return MWRes_Success;
	}

}

/*
//====================================================================
// Resupplying the input buffer
//====================================================================
*/

TMW_ResultCode MWZLib_resupplyInputBuffer
(
	TMWZLib_SessionStructure* p_zlibSession,
	TFourByteInteger_Unsigned numBytes,
	TByte_Unsigned* p_sourceByteArray
)
{
	/*
	//########################################
	// Test parameters
	//########################################
	*/
	if
	(
		(p_zlibSession == NULL) ||
		(p_sourceByteArray == NULL)
	)
	{
		return MWRes_NullPointer;
	}

	if (numBytes == 0)
	{
		return MWRes_BadParamValue;
	}

	/*
	//########################################
	// Test state
	//########################################
	*/
	if ( p_zlibSession->zStream.avail_in != 0 )
	{
		return MWRes_BadCallingOrder;
	}

	/*
	//########################################
	// Set new input
	//########################################
	*/
	p_zlibSession->zStream.next_in = p_sourceByteArray;
	p_zlibSession->zStream.avail_in = numBytes;

	/*
	//########################################
	// Success
	//########################################
	*/
	return MWRes_Success;

}

/*
//====================================================================
// Inflate data
//====================================================================
*/

TMW_ResultCode MWZLib_inflateBytes
(
	TMWZLib_SessionStructure* p_zlibSession,
	TFourByteInteger_Unsigned numBytesToInflate,
	TByte_Unsigned* p_destinationArray,
	TFourByteInteger_Unsigned* p_out_numBytesInflated,
	TFourByteInteger_Unsigned* p_out_numBytesConsumed
)
{

	TFourByteInteger_Unsigned numBytesGotten = 0;
	TFourByteInteger_Unsigned memCopyIndex;
	TFourByteInteger_Unsigned numAvailableInputBytesBeforeInflate;


	/*
	//########################################
	// Test parameters
	//########################################
	*/
	if
	(
		(p_zlibSession == NULL) ||
		(p_destinationArray == NULL) ||
		(p_out_numBytesInflated == NULL) ||
		(p_out_numBytesConsumed == NULL)
	)
	{
		return MWRes_NullPointer;
	}

	/*
	//########################################
	// So far no additional input bytes
	// consumed. Remember number of input bytes
	// available for later determining how
	// many were consumed.
	//########################################
	*/
	numAvailableInputBytesBeforeInflate = p_zlibSession->zStream.avail_in;
	*p_out_numBytesConsumed = 0;


	/*
	//########################################
	// Loop until all requested bytes gotten
	// for caller.  The loop exits
	// the function if a terminal condition
	// is reached due to end of stream, error
	// etc...
	//########################################
	*/
	while (numBytesGotten < numBytesToInflate)
	{
		/*
		//########################################
		// If the internal buffer is empty,
		// inflate a block of data into the
		// internal buffer.
		//########################################
		*/
		if (p_zlibSession->numInflatedBytesInternalConsumed >= p_zlibSession->numInflatedBytesInternal)
		{

			/* All the internal buffer was consumed, or it is empty. */

			/* Reset internal buffer counters */
			p_zlibSession->numInflatedBytesInternalConsumed = 0;
			p_zlibSession->numInflatedBytesInternal = 0;

			/* Reset zlib buffer write position */
			p_zlibSession->zStream.next_out = p_zlibSession->inflatedBytesInternal;;
			p_zlibSession->zStream.avail_out = INTERNAL_INFLATION_BUFFER_LENGTH;

			/* If we already reached the end of the stream, we have a problem, as there is no more data to INFLATE */
			if (p_zlibSession->ZRes == Z_STREAM_END)
			{
				*p_out_numBytesInflated = numBytesGotten;
				return MWRes_EndOfDeflatedData;
			}

			/* If we need more input, then return a code indicating that more input is required */
			if (p_zlibSession->zStream.avail_in == 0)
			{
				/* We need more input given by a call to MWZLib_ResupplyInputBuffer */
				*p_out_numBytesInflated = numBytesGotten;
				return MWRes_InputBufferExhausted;
			}

			/* Perform zlib INFLATION into the internal buffer */
			p_zlibSession->ZRes = inflate (&(p_zlibSession->zStream), Z_SYNC_FLUSH);

			/* How many bytes were consumed from the input*/
			*p_out_numBytesConsumed = numAvailableInputBytesBeforeInflate - p_zlibSession->zStream.avail_in;

			if (p_zlibSession->ZRes == Z_STREAM_END)
			{
				/*
				// End of inflated stream reached
				// Continue to give to caller whatever was INFLATED until the end pint
				*/
			}
			else if (p_zlibSession->ZRes != Z_OK)
			{
				/* Inflation error */
				*p_out_numBytesInflated = numBytesGotten;
				return MWRes_InflationError;
			}

			/*
			// Update number of bytes available in the internal buffer from the zstream state
			// zStream.avail_out gives the number of bytes still available to zlib in the buffer we gave it.
			// It starts at INTERNAL_INFLATION_BUFFER_LENGTH and counts down to 0
			*/
			p_zlibSession->numInflatedBytesInternal =
				INTERNAL_INFLATION_BUFFER_LENGTH - p_zlibSession->zStream.avail_out;

		}

		/*
		//########################################
		// Get all the output we already have for
		// the caller.
		//########################################
		*/
		if (p_zlibSession->numInflatedBytesInternalConsumed < p_zlibSession->numInflatedBytesInternal)
		{
			/* How many bytes can we transfer bytes from the internal buffer to the caller */
			TFourByteInteger_Unsigned numToTransfer = p_zlibSession->numInflatedBytesInternal -
				p_zlibSession->numInflatedBytesInternalConsumed;

			/* If we have more bytes than needed, limit transfer to just those needed.*/
			if (numToTransfer > (numBytesToInflate - numBytesGotten) )
			{
				numToTransfer = numBytesToInflate - numBytesGotten;
			}

			/* Transfer the bytes */
#ifdef HAS_MEMCPY
	memcpy
	(
		p_destinationArray + numBytesGotten ,
		p_zlibSession->inflatedBytesInternal + p_zlibSession->numInflatedBytesInternalConsumed,
		numToTransfer
	);
#else
	for (memCopyIndex = 0; memCopyIndex < numToTransfer; memCopyIndex ++)
	{
		*(p_destinationArray + numBytesGotten + memCopyIndex) =
			*(p_zlibSession->inflatedBytesInternal + p_zlibSession->numInflatedBytesInternalConsumed + memCopyIndex);

	}

#endif
			/* More of internal buffer consumed */
			p_zlibSession->numInflatedBytesInternalConsumed += numToTransfer;

			/* More bytes given to caller */
			numBytesGotten += numToTransfer;

		} /* end internal buffer transfer to caller */

	} /* end loop that INFLATES/transfers until all requested bytes are transferred to caller */

	/*
	//########################################
	// All of the requested bytes have been
	// retrieved
	//########################################
	*/

	*p_out_numBytesInflated = numBytesGotten;
	return MWRes_Success;


}
